
;-------------------------------------------------------------------------------
;
;  Unit Name : l_NumFluxRtns.pro
;
;  Purpose   : All routines related to converting data to number flux.
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;   ELH             04/23/03        v1.0.11           Original implementation
;
;
;  File Revision Number:   %I%
;  File Last Modified  :   %E%  %T%
;
;-------------------------------------------------------------------------------
;
;-------------------------------------------------------------------------------
;  Procedure:l_CalcEffciency
;
;  Description: Determine the efficiency based on the instrument state at that
;        time. 
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  dtime                 long[]        I        data start time
;  level                 string        O        instrument state
;  efficiency            double        O        new effciency adjusted for state 
;  nominal_eff           double        I        nominal efficiency
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------

PRO l_CalcEffciency, dtime, level, efficiency, nominal_eff 

    if (dtime[0] eq 2000) then begin
       if (dtime[1] ge 126) AND (dtime[1] le 154) then begin
          level = 'IOC_Level'
          efficiency = nominal_eff/4.0D0
       endif else begin
           if (dtime[1] ge 155) AND (dtime[1] le 310) then begin
              level = 'Nominal_Ops_Level'
              efficiency = nominal_eff 
           endif else begin
              level = 'PPSP_Modes_Level'
              efficiency = 3.0D0 * nominal_eff 
           endelse
       endelse
    endif else begin
        level = 'PPSP_Modes_Level'
        efficiency = 3.0D0 * nominal_eff 
    endelse


RETURN
END

;-------------------------------------------------------------------------------
;  Procedure: l_GenEffTable
;
;  Description: Generate a table of nominal efficiencies for oxygen and hydrogen.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  O_Eff_Table          struct[]       O        oxygen nominal efficiencies
;  H_Eff_Table          struct[]       O        hydrogen nominal efficiencies
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------

PRO l_GenEffTable, O_Eff_Table, H_Eff_Table


   O_Eff_Table = replicate ({EFF_PARMS}, 12)        
   H_Eff_Table = replicate ({EFF_PARMS}, 9)        

   O_Eff_Table[0].energy = 30
   O_Eff_Table[0].eff = 0.0000015 

   O_Eff_Table[1].energy = 60
   O_Eff_Table[1].eff = 0.0000017 
   
   O_Eff_Table[2].energy = 60
   O_Eff_Table[2].eff = 0.0000020

   O_Eff_Table[3].energy = 100
   O_Eff_Table[3].eff = 0.0000074

   O_Eff_Table[4].energy = 200
   O_Eff_Table[4].eff = 0.000053

   O_Eff_Table[5].energy = 200
   O_Eff_Table[5].eff = 0.000024

   O_Eff_Table[6].energy = 300
   O_Eff_Table[6].eff = 0.000021

   O_Eff_Table[7].energy = 300
   O_Eff_Table[7].eff = 0.000040

   O_Eff_Table[8].energy = 300
   O_Eff_Table[8].eff = 0.000056

   O_Eff_Table[9].energy = 400
   O_Eff_Table[9].eff = 0.000073

   O_Eff_Table[10].energy = 500
   O_Eff_Table[10].eff = 0.000058

   O_Eff_Table[11].energy = 1000
   O_Eff_Table[11].eff = 0.00011


   H_Eff_Table[0].energy = 20
   H_Eff_Table[0].eff = 0.00000043

   H_Eff_Table[1].energy = 20
   H_Eff_Table[1].eff = 0.00000099

   H_Eff_Table[2].energy = 30
   H_Eff_Table[2].eff = 0.0000053

   H_Eff_Table[3].energy = 30
   H_Eff_Table[3].eff = 0.0000016

   H_Eff_Table[4].energy = 100
   H_Eff_Table[4].eff = 0.000015

   H_Eff_Table[5].energy = 200
   H_Eff_Table[5].eff = 0.000016

   H_Eff_Table[6].energy = 300
   H_Eff_Table[6].eff = 0.0000084

   H_Eff_Table[7].energy = 300
   H_Eff_Table[7].eff = 0.000014

   H_Eff_Table[8].energy = 1000
   H_Eff_Table[8].eff = 0.000016


RETURN
END

;-------------------------------------------------------------------------------
;  Function: l_FindNominalEff
;
;  Description: Find the nominal efficiency given mass, energy, & instrument state.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  SUCCESS                       successful completion
;  FAILURE                       failure to complete
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  O_Eff_Table          struct[]       I        oxygen efficiencies
;  H_Eff_Table          struct[]       I        hydrogen efficiencies
;  species              string         I        hydrogen or oxygen
;  eV                   long           I        energy range
;  nominal_eff          double         O        nominal efficiency
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------

FUNCTION l_FindNominalEff, O_Eff_Table, H_Eff_Table, species, eV , nominal_eff

    if (species eq 'oxygen') then begin
       if (eV lt O_Eff_Table[0].energy) OR (eV gt O_Eff_Table[11].energy) then begin
          RETURN, -2
       endif
       oindx = where (O_Eff_Table[*].energy eq eV, ocnt)
       if (ocnt gt 0) then begin
          if (ocnt eq 1) then begin
             nominal_eff = O_Eff_Table[oindx[0]].eff
          endif else begin 
             nominal_eff = MEAN(O_Eff_Table[oindx].eff)
          endelse
       endif else begin
           x0indx = where (O_Eff_Table[*].energy lt eV, x0cnt) 
           if (x0cnt gt 0) then begin
              x0 = O_Eff_Table[x0indx[n_elements(x0indx)-1]].energy  
              y0 = O_Eff_Table[x0indx[n_elements(x0indx)-1]].eff
           endif else begin
              x0 = eV - 10
              y0 = 0.000001
           endelse
              
           x1 = eV

           x2indx = where (O_Eff_Table[*].energy gt eV, x2cnt) 
           if (x2cnt gt 0) then begin
              x2 = O_Eff_Table[x2indx[0]].energy  
              y2 = O_Eff_Table[x2indx[0]].eff
           endif else begin
              x2 = O_Eff_Table[0].energy  
              y2 = O_Eff_Table[0].eff
           endelse

           nominal_eff = l_Interpolate (x0, x1, x2, y0, y2)
           ;interpolate
       endelse
    endif else begin
       if (eV lt H_Eff_Table[0].energy) OR (eV gt H_Eff_Table[8].energy) then begin
          RETURN, -2
       endif
       hindx = where (H_Eff_Table[*].energy eq eV, hcnt)
       if (hcnt gt 0) then begin
          if (hcnt eq 1) then begin
             nominal_eff = H_Eff_Table[hindx[0]].eff
          endif else begin
             nominal_eff = MEAN (H_Eff_Table[hindx].eff) 
          endelse
       endif else begin
         ;interpolate
         x0indx = where (H_Eff_Table[*].energy lt eV, x0cnt) 
         x0 = H_Eff_Table[x0indx[n_elements(x0indx)-1]].energy  
         y0 = H_Eff_Table[x0indx[n_elements(x0indx)-1]].eff
         x1 = eV
         x2indx = where (H_Eff_Table[*].energy gt eV, x2cnt) 
         x2 = H_Eff_Table[x2indx[0]].energy  
         y2 = H_Eff_Table[x2indx[0]].eff
         nominal_eff = l_Interpolate (x0, x1, x2, y0, y2)
       endelse
    endelse

    nominal_eff = nominal_eff * 4.0

RETURN, 0
END

;-------------------------------------------------------------------------------
;  Procedure: l_WriteEff2File
;
;  Description:  Write the efficiency values to a file.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  flux_ofd               long         I        output file descriptor
;  mass                   string       I        hydrogen or oxygen
;  eV                     long         I        energy range
;  state                  string       I        instrument state
;  nom_eff                double       I        nominal efficiency
;  eff                    double       I        nominal eff. adjusted for instrument
;                                               state
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------

PRO l_WriteEff2File, flux_ofd, mass, eV, state, nom_eff, eff

    printf, flux_ofd, ' '
    printf, flux_ofd, 'Mass               : ', strcompress (mass, /remove_all)
    printf, flux_ofd, 'eV                 : ', strcompress (eV, /remove_all)
    printf, flux_ofd, 'State              : ', strcompress (state, /remove_all)
    printf, flux_ofd, 'Nominal efficiency : ', strcompress (nom_eff, /remove_all)
    printf, flux_ofd, 'Efficiency         : ', strcompress (eff, /remove_all)
    printf, flux_ofd, ' '

RETURN
END

