;-------------------------------------------------------------------------------
;
;  Unit Name : l_StateTableRtns.pro
;
;  Purpose   : Generate a state table including changes on a sub-spin level.
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;   ELH              07/02          v1.0.9           Original implementation
;
;
;  File Revision Number:   %I%
;  File Last Modified  :   %E%  %T%
;
;-------------------------------------------------------------------------------

;-------------------------------------------------------------------------------
;  Function/Procedure: l_StateTableDocumentation
;
;  Description:
;     Provides a high level description of file contents, a header, to
;  each state log file generated.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  ofd                    int          I        file handler
;  echo_msg               string       I        data related information 
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                06/01          v1.0             Original implementation
;-------------------------------------------------------------------------------


PRO l_StateTableDocumentation, ofd, echo_msg

   printf, ofd, 'd = disabled'
   printf, ofd, ' '
   printf, ofd, ' '
   printf, ofd, '------------------------- State Table Description -------------------------'
   printf, ofd, ' '
   printf, ofd, '  Column Heading Definitions: '
   printf, ofd, ' '
   printf, ofd, '    UTC             : Universal Time Coordinate (s/c time corrected)'
   printf, ofd, '    MCP Start       : HVPS MCP Start Supply Commanded Level'
   printf, ofd, '    MCP Stop        : HVPS MCP Stop Supply Commanded Level'
   printf, ofd, '    Coll Pos        : HVPS Collimator Positive Commanded Level'
   printf, ofd, '    Coll Neg        : HVPS Collimator Negative Commanded Level'
   printf, ofd, '    Optics          : HVPS Optics Commanded Level'
   printf, ofd, '    Steering        : HVPS Optics Steering Controller Commanded Level'
   printf, ofd, '    Start CFD       : TOF Start CFD Level'
   printf, ofd, '    Stop CFD        : TOF Stop CFD Level'
   printf, ofd, '    Mass 1 Start    : Mass 1 Bin Start'
   printf, ofd, '    Mass 1 Stop     : Mass 1 Bin Stop'
   printf, ofd, '    Mass 2 Start    : Mass 2 Bin Start'
   printf, ofd, '    Mass 2 Stop     : Mass 2 Bin Stop'
   printf, ofd, '    FSW Ver         : Flight Software Version Number'
   printf, ofd, '    ROI Description : ROI 0-3 status (1 = on, 0 = off)'
   printf, ofd, ' '
   printf, ofd, '---------------------------------------------------------------------------'
   printf, ofd, ' '
   printf, ofd, ' '
 

   if (echo_msg ne ' ') then printf, ofd, echo_msg
   printf, ofd, ' '
   printf, ofd, ' '

   printf, ofd, 'UTC', '      MCP      ', '     Collimator     ', $
                'Optics', 'Steering', '      CFD      ', '      Mass 1      ', $
                '      Mass2      ', 'FSW Ver', ' ROI Description', $
                format='((A8,TR8), (A15,TR5), (A20,TR4), (A6,TR3), (A8,TR4), (A15,TR5), (A18,TR5), (A17,TR5), (A7,TR4), (A16))'


   printf, ofd, '   ', 'Start     Stop ', '  Pos          Neg  ', $
                '      ', '        ', 'Start     Stop ', 'Start       Stop  ', $
                'Start      Stop  ', '       ', ' 0    1    2    3', $
                format='((A8,TR8), (A15,TR5), (A20,TR4), (A6,TR3), (A8,TR4), (A15,TR5), (A18,TR5), (A17,TR5), (A7,TR4), (A17))'


   printf, ofd, '---', '------    ------','-----       ----- ', $
                '------', '--------', '------   ------', '------      ------', $
                '-----     -------', '-------', '---  ---  ---  ---', $
                format='((A8,TR8), (A15,TR5), (A20,TR4), (A6,TR3), (A8,TR4), (A15,TR5), (A18,TR5), (A17,TR5), (A7,TR4), (A18))'


RETURN
END



;-------------------------------------------------------------------------------
;  Function/Procedure: l_StateTableDumpAll
;
;  Description:
;      Generates the the state table which includes commands on a spin level
;  and on a sub-spin level.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  ofd                   int           O        file handler
;  l_setup               struct        I        general setup parameters
;  l_hskp                struct *      I        housekeeping data pointers
;  l_pltpos              struct        I        general plot setup parameters
;  filename              string        I        file name
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                06/01          v1.0             Original implementation
;-------------------------------------------------------------------------------

PRO l_StateTableDumpAll, ofd, l_setup, l_hskp, l_pltpos, filename, l_echo


   l_DumpHdrInfo, ofd, l_setup, l_pltpos, filename

   l_StateTableDocumentation, ofd, ' '

   nitems = n_elements ((*l_hskp.startUdfTimePtr)[*])

   prev_mcp_start = 0L
   prev_mcp_stop  = 0L
   prev_collp     = 0L
   prev_colln     = 0L
   prev_optics    = 0L
   prev_steering  = 0L
   prev_start_cfd = 0L
   prev_stop_cfd  = 0L
   prev_m1_start  = 0L
   prev_m1_stop   = 0L
   prev_m2_start  = 0L
   prev_m2_stop   = 0L
   prev_sw_ver    = 0L
   prev_roi0      = 0L
   prev_roi1      = 0L
   prev_roi2      = 0L
   prev_roi3      = 0L

   sub_prev_mcp_start = 0L
   sub_prev_mcp_stop  = 0L
   sub_prev_collp     = 0L
   sub_prev_colln     = 0L
   sub_prev_optics    = 0L
   sub_prev_steering  = 0L
   sub_prev_start_cfd = 0L
   sub_prev_stop_cfd  = 0L
   sub_prev_m1_start  = 0L 
   sub_prev_m1_stop   = 0L
   sub_prev_m2_start  = 0L
   sub_prev_m2_stop   = 0L
   sub_prev_sw_ver    = 0L
   sub_prev_roi0      = 0L
   sub_prev_roi1      = 0L
   sub_prev_roi2      = 0L
   sub_prev_roi3      = 0L


   new_j = 0L
   count_cmd = 0L
   prev_time = (*l_echo.sub_spinTime)[0]

   for i = 0L, nitems-1L do begin

     count_cmd = 0L
     for j = new_j, l_echo.nitems - 1L do begin 

         ncmds = (*l_echo.nevents_cmded)[j]
         curr_time = (*l_echo.sub_spinTime)[j]
         if ((*l_echo.sub_spinTime)[j] eq (*l_hskp.timePtr)[i]) AND $ 
             (ncmds gt 1) then begin

            count_cmd = count_cmd + 1L 
            if (count_cmd lt ncmds) then begin
               sub_change = 0L
               dmpstr = ' '
               tail = ' '

               new_j = j

               ;-------------------------------------------------------
               ; write the data to the text file
               ;-------------------------------------------------------
               tmestr = STRING ((*l_hskp.startUdfTimePtr)[i].doy, $
                                (*l_hskp.startUdfTimePtr)[i].hour,$
                                (*l_hskp.startUdfTimePtr)[i].min, $
                                (*l_hskp.startUdfTimePtr)[i].sec, $
                                FORMAT="(i3,' ',i2.2,':',i2.2,':',i2.2)")


               dmpstr = 'printf, ofd, tmestr, '                           ;-- utc --
               dmpstr = dmpstr + 'mcp_start, '                            ;-- mcp start --
               dmpstr = dmpstr + 'mcp_stop, '                             ;-- mcp stop --
               dmpstr = dmpstr + 'coll_p, '                               ;-- coll positive --  
               dmpstr = dmpstr + 'coll_n, '                               ;-- coll negative --
               dmpstr = dmpstr + 'optics, '                               ;-- optics --
               dmpstr = dmpstr + '(*l_hskp.hvps_configPtr)[i, 13], '      ;-- optics steering --
               dmpstr = dmpstr + '(*l_hskp.tof_subsys_configPtr)[i, 0], ' ;-- start cfd --
               dmpstr = dmpstr + '(*l_hskp.tof_subsys_configPtr)[i, 1], ' ;-- stop cfd --
               dmpstr = dmpstr + '(*l_hskp.mass_binn_configPtr)[i,0], '   ;-- mass 1 bin start --
               dmpstr = dmpstr + '(*l_hskp.mass_binn_configPtr)[i,1], '   ;-- mass 1 bin stop --
               dmpstr = dmpstr + '(*l_hskp.mass_binn_configPtr)[i,2], '   ;-- mass 2 bin start --
               dmpstr = dmpstr + '(*l_hskp.mass_binn_configPtr)[i,3], '   ;-- mass 2 bin stop --
               dmpstr = dmpstr + '(*l_hskp.sys_lev_statsPtr)[i, 0], '     ;-- sw version --
               dmpstr = dmpstr + '(*l_hskp.data_pkt_sel_statPtr)[i,3], '  ;-- ROI 0 packet status --
               dmpstr = dmpstr + '(*l_hskp.data_pkt_sel_statPtr)[i,4], '  ;-- ROI 1 packet status --
               dmpstr = dmpstr + '(*l_hskp.data_pkt_sel_statPtr)[i,5], '  ;-- ROI 2 packet status --
               dmpstr = dmpstr + '(*l_hskp.data_pkt_sel_statPtr)[i,6], '  ;-- ROI 3 packet status --  
               dmpstr = dmpstr + 'format="((A12,TR1), '

               tail = '(e9.2,TR1), 2(e9.2,TR2), 2(e9.2,TR2), (e9.2,TR2), (e9.2,TR6), (I2,TR8), 4(I1,TR4))"'

               ;-------------------------------------------------------
               ; check if flags are enabled, if not write out a -9
               ; for no data or disabled
               ;-------------------------------------------------------
               ;--- mcp start enable flag ---
               if ((*l_hskp.hvps_configPtr)[i,2] ne 0) then begin
                   if ((*l_echo.l_hvp_mstr_cmd)[j] eq 0) then begin
                      mcp_start = (*l_hskp.hvps_configPtr)[i,8]
                   endif else begin
                      mcp_start = (*l_echo.l_hvp_mstr_cmd)[j]
                   endelse
                   sub_curr_mcp_start = mcp_start
                   dmpstr = dmpstr + '(e9.2,TR2)'
               endif else begin
                   mcp_start = 'd' 
                   sub_curr_mcp_start = -99999 
                   dmpstr = dmpstr + '(A3,8X)'
               endelse
               if (sub_prev_mcp_start ne sub_curr_mcp_start) then sub_change = 1L

               ;--- mcp stop enable flag ---
               if ((*l_hskp.hvps_configPtr)[i,3] ne 0) then begin
                   if ((*l_echo.l_hvp_mstp_cmd)[j] eq 0) then begin
                      mcp_stop = (*l_hskp.hvps_configPtr)[i,9]
                   endif else begin
                      mcp_stop = (*l_echo.l_hvp_mstp_cmd)[j]
                   endelse
                   sub_curr_mcp_stop = mcp_stop
                   dmpstr = dmpstr + '(e9.2,TR2)'
               endif else begin
                   mcp_stop =  'd'
                   sub_curr_mcp_stop = -99999 
                   dmpstr = dmpstr + '(A3,8X)'
               endelse
               if (sub_prev_mcp_stop ne sub_curr_mcp_stop) then sub_change = 1L

               ;--- coll positive enable flag ---
               if ((*l_hskp.hvps_configPtr)[i,4] ne 0) then begin
                   if ((*l_echo.l_hvp_colp_cmd)[j] eq 0) then begin
                      coll_p = (*l_hskp.hvps_configPtr)[i,10] 
                   endif else begin
                      coll_p = (*l_echo.l_hvp_colp_cmd)[j]
                   endelse
                   sub_curr_collp = coll_p
                   dmpstr = dmpstr + '(e9.2,TR2)'
               endif else begin
                   coll_p = 'd' 
                   sub_curr_collp = -99999 
                   dmpstr = dmpstr + '(A3,8X)'
               endelse
               if (sub_prev_collp ne sub_curr_collp) then sub_change = 1L

               ;--- coll negative enable flag ---
               if ((*l_hskp.hvps_configPtr)[i,5] ne 0) then begin
                   if ((*l_echo.l_hvp_coln_cmd)[j] eq 0) then begin
                      coll_n = (*l_hskp.hvps_configPtr)[i,11]
                   endif else begin
                      coll_n = (*l_echo.l_hvp_coln_cmd)[j]
                   endelse
                   sub_curr_colln = coll_n
                   dmpstr = dmpstr + '(e9.2,TR2)'
               endif else begin
                   coll_n =  'd'
                   sub_curr_colln = -99999 
                   dmpstr = dmpstr + '(A3,8X)'
               endelse
               if (sub_prev_colln ne sub_curr_colln) then sub_change = 1L
 
               ;--- optics enable flag ---
               if ((*l_hskp.hvps_configPtr)[i,6] ne 0) then begin
                   if ((*l_echo.l_hvp_opt_cmd)[j] eq 0) then begin
                      optics = (*l_hskp.hvps_configPtr)[i,12]
                   endif else begin
                      optics = (*l_echo.l_hvp_opt_cmd)[j]
                   endelse
                   sub_curr_optics = optics
                   dmpstr = dmpstr + '(e9.2,TR2)'
               endif else begin
                   optics = 'd' 
                   sub_curr_optics = -99999 
                   dmpstr = dmpstr + '(A3,8X)'
               endelse
               if (sub_prev_optics ne sub_curr_optics) then sub_change = 1L


               ;--- optics steering ---
               if ((*l_echo.l_hvp_str_cmd)[j] eq 0) then begin
                  sub_curr_steering = (*l_hskp.hvps_configPtr)[i,13]
               endif else begin
                  sub_curr_steering = (*l_echo.l_hvp_str_cmd)[j]
               endelse
               if (sub_prev_steering ne sub_curr_steering) then sub_change = 1L

               ;--- start cfd ---
               sub_curr_start_cfd = (*l_hskp.tof_subsys_configPtr)[i,0]
               if (sub_prev_start_cfd ne sub_curr_start_cfd) then sub_change = 1L

               ;--- stop cfd ---
               sub_curr_stop_cfd = (*l_hskp.tof_subsys_configPtr)[i, 1]
               if (sub_prev_stop_cfd ne sub_curr_stop_cfd) then sub_change = 1L

               ;--- mass 1 bin start ---
               sub_curr_m1_start =  (*l_hskp.mass_binn_configPtr)[i,0]
               if (sub_prev_m1_start ne sub_curr_m1_start) then sub_change = 1L

               ;--- mass 1 bin stop ---
               sub_curr_m1_stop = (*l_hskp.mass_binn_configPtr)[i,1]
               if (sub_prev_m1_stop ne sub_curr_m1_stop) then sub_change = 1L

               ;--- mass 2 bin start ---
               sub_curr_m2_start = (*l_hskp.mass_binn_configPtr)[i,2]
               if (sub_prev_m2_start ne sub_curr_m2_start) then sub_change = 1L
   
               ;--- mass 2 bin stop ---
               sub_curr_m2_stop = (*l_hskp.mass_binn_configPtr)[i,3]
               if (sub_prev_m2_stop ne sub_curr_m2_stop) then sub_change = 1L
 
               ;--- sw version ---
               sub_curr_sw_ver = (*l_hskp.sys_lev_statsPtr)[i, 0]
               if (sub_prev_sw_ver ne sub_curr_sw_ver) then sub_change = 1L

               ;--- ROI 0 packet status ---
               sub_curr_roi0 = (*l_hskp.data_pkt_sel_statPtr)[i,3]
               if (sub_prev_roi0 ne sub_curr_roi0) then sub_change = 1L
    
               ;--- ROI 1 packet status ---
               sub_curr_roi1 = (*l_hskp.data_pkt_sel_statPtr)[i,4]
               if (sub_prev_roi1 ne sub_curr_roi1) then sub_change = 1L

               ;--- ROI 2 packet status ---
               sub_curr_roi2 = (*l_hskp.data_pkt_sel_statPtr)[i,5]
               if (sub_prev_roi2 ne sub_curr_roi2) then sub_change = 1L

               ;--- ROI 3 packet status ---
               sub_curr_roi3 = (*l_hskp.data_pkt_sel_statPtr)[i,6]
               if (sub_prev_roi3 ne sub_curr_roi3) then sub_change = 1L

               dmpstr = dmpstr + tail


               if (prev_time ne curr_time) AND (sub_change eq 0L) then sub_change = 1L

               if (sub_change eq 1L) then begin
                  result = execute (dmpstr)
                  sub_prev_mcp_start = sub_curr_mcp_start
                  sub_prev_mcp_stop  = sub_curr_mcp_stop
                  sub_prev_collp     = sub_curr_collp
                  sub_prev_colln     = sub_curr_colln
                  sub_prev_optics    = sub_curr_optics
                  sub_prev_steering  = sub_curr_steering
                  sub_prev_start_cfd = sub_curr_start_cfd
                  sub_prev_stop_cfd  = sub_curr_stop_cfd
                  sub_prev_m1_start  = sub_curr_m1_start
                  sub_prev_m1_stop   = sub_curr_m1_stop
                  sub_prev_m2_start  = sub_curr_m2_start
                  sub_prev_m2_stop   = sub_curr_m2_stop
                  sub_prev_sw_ver    = sub_curr_sw_ver
                  sub_prev_roi0      = sub_curr_roi0
                  sub_prev_roi1      = sub_curr_roi1
                  sub_prev_roi2      = sub_curr_roi2
                  sub_prev_roi3      = sub_curr_roi3
                  prev_time          = curr_time
               endif

          endif
               if (prev_time ne curr_time) then begin
                  sub_change = 0L
                  sub_prev_mcp_start = 0L
                  sub_prev_mcp_stop  = 0L
                  sub_prev_collp     = 0L
                  sub_prev_colln     = 0L
                  sub_prev_optics    = 0L
                  sub_prev_steering  = 0L
                  sub_prev_start_cfd = 0L
                  sub_prev_stop_cfd  = 0L
                  sub_prev_m1_start  = 0L
                  sub_prev_m1_stop   = 0L
                  sub_prev_m2_start  = 0L
                  sub_prev_m2_stop   = 0L
                  sub_prev_sw_ver    = 0L
                  sub_prev_roi0      = 0L
                  sub_prev_roi1      = 0L
                  sub_prev_roi2      = 0L
                  count_cmd          = 0L
               endif
      endif else begin
         count_cmd = 0L
         new_j = j 
         if ((*l_echo.sub_spinTime)[j] eq (*l_hskp.timePtr)[i]) then new_j = new_j + 1L 
         GOTO, BREAK
      endelse

     endfor

BREAK:

      change = 0L
      dmpstr = ' '
      tail = ' '

      ;-------------------------------------------------------
      ; write the data to the text file
      ;-------------------------------------------------------
      tmestr = STRING ((*l_hskp.startUdfTimePtr)[i].doy, $
                       (*l_hskp.startUdfTimePtr)[i].hour,$
                       (*l_hskp.startUdfTimePtr)[i].min, $
                       (*l_hskp.startUdfTimePtr)[i].sec, $
                       FORMAT="(i3,' ',i2.2,':',i2.2,':',i2.2)")


      dmpstr = 'printf, ofd, tmestr, '                           ;-- utc --
      dmpstr = dmpstr + 'mcp_start, '                            ;-- mcp start --
      dmpstr = dmpstr + 'mcp_stop, '                             ;-- mcp stop --
      dmpstr = dmpstr + 'coll_p, '                               ;-- coll positive --  
      dmpstr = dmpstr + 'coll_n, '                               ;-- coll negative --
      dmpstr = dmpstr + 'optics, '                               ;-- optics --
      dmpstr = dmpstr + '(*l_hskp.hvps_configPtr)[i, 13], '      ;-- optics steering --
      dmpstr = dmpstr + '(*l_hskp.tof_subsys_configPtr)[i, 0], ' ;-- start cfd --
      dmpstr = dmpstr + '(*l_hskp.tof_subsys_configPtr)[i, 1], ' ;-- stop cfd --
      dmpstr = dmpstr + '(*l_hskp.mass_binn_configPtr)[i,0], '   ;-- mass 1 bin start --
      dmpstr = dmpstr + '(*l_hskp.mass_binn_configPtr)[i,1], '   ;-- mass 1 bin stop --
      dmpstr = dmpstr + '(*l_hskp.mass_binn_configPtr)[i,2], '   ;-- mass 2 bin start --
      dmpstr = dmpstr + '(*l_hskp.mass_binn_configPtr)[i,3], '   ;-- mass 2 bin stop --
      dmpstr = dmpstr + '(*l_hskp.sys_lev_statsPtr)[i, 0], '     ;-- sw version --
      dmpstr = dmpstr + '(*l_hskp.data_pkt_sel_statPtr)[i,3], '  ;-- ROI 0 packet status --
      dmpstr = dmpstr + '(*l_hskp.data_pkt_sel_statPtr)[i,4], '  ;-- ROI 1 packet status --
      dmpstr = dmpstr + '(*l_hskp.data_pkt_sel_statPtr)[i,5], '  ;-- ROI 2 packet status --
      dmpstr = dmpstr + '(*l_hskp.data_pkt_sel_statPtr)[i,6], '  ;-- ROI 3 packet status --  
      dmpstr = dmpstr + 'format="((A12,TR1), '

      tail = '(e9.2,TR1), 2(e9.2,TR2), 2(e9.2,TR2), (e9.2,TR2), (e9.2,TR6), (I2,TR8), 4(I1,TR4))"'

      ;-------------------------------------------------------
      ; check if flags are enabled, if not write out a -9
      ; for no data or disabled
      ;-------------------------------------------------------
      ;--- mcp start enable flag ---
      if ((*l_hskp.hvps_configPtr)[i,2] ne 0) then begin
          mcp_start = (*l_hskp.hvps_configPtr)[i, 8]
          curr_mcp_start = mcp_start
          dmpstr = dmpstr + '(e9.2,TR2)'
      endif else begin
          mcp_start = 'd' 
          curr_mcp_start = -99999 
          dmpstr = dmpstr + '(A3,8X)'
      endelse
      if (prev_mcp_start ne curr_mcp_start) then change = 1L

      ;--- mcp stop enable flag ---
      if ((*l_hskp.hvps_configPtr)[i,3] ne 0) then begin
          mcp_stop = (*l_hskp.hvps_configPtr)[i, 9]
          curr_mcp_stop = mcp_stop
          dmpstr = dmpstr + '(e9.2,TR2)'
      endif else begin
          mcp_stop =  'd'
          curr_mcp_stop = -99999 
          dmpstr = dmpstr + '(A3,8X)'
      endelse
      if (prev_mcp_stop ne curr_mcp_stop) then change = 1L

      ;--- coll positive enable flag ---
      if ((*l_hskp.hvps_configPtr)[i,4] ne 0) then begin
          coll_p = (*l_hskp.hvps_configPtr)[i, 10]
          curr_collp = coll_p
          dmpstr = dmpstr + '(e9.2,TR2)'
      endif else begin
          coll_p = 'd' 
          curr_collp = -99999 
          dmpstr = dmpstr + '(A3,8X)'
      endelse
      if (prev_collp ne curr_collp) then change = 1L

      ;--- coll negative enable flag ---
      if ((*l_hskp.hvps_configPtr)[i,5] ne 0) then begin
          coll_n = (*l_hskp.hvps_configPtr)[i, 11]
          curr_colln = coll_n
          dmpstr = dmpstr + '(e9.2,TR2)'
      endif else begin
          coll_n =  'd'
          curr_colln = -99999 
          dmpstr = dmpstr + '(A3,8X)'
      endelse
      if (prev_colln ne curr_colln) then change = 1L
 
      ;--- optics enable flag ---
      if ((*l_hskp.hvps_configPtr)[i,6] ne 0) then begin
          optics = (*l_hskp.hvps_configPtr)[i, 12]
          curr_optics = optics
          dmpstr = dmpstr + '(e9.2,TR2)'
      endif else begin
          optics = 'd' 
          curr_optics = -99999 
          dmpstr = dmpstr + '(A3,8X)'
      endelse
      if (prev_optics ne curr_optics) then change = 1L


      ;--- optics steering ---
      curr_steering = (*l_hskp.hvps_configPtr)[i,13]
      if (prev_steering ne curr_steering) then change = 1L

      ;--- start cfd ---
      curr_start_cfd = (*l_hskp.tof_subsys_configPtr)[i,0]
      if (prev_start_cfd ne curr_start_cfd) then change = 1L

      ;--- stop cfd ---
      curr_stop_cfd = (*l_hskp.tof_subsys_configPtr)[i, 1]
      if (prev_stop_cfd ne curr_stop_cfd) then change = 1L

      ;--- mass 1 bin start ---
      curr_m1_start =  (*l_hskp.mass_binn_configPtr)[i,0]
      if (prev_m1_start ne curr_m1_start) then change = 1L

      ;--- mass 1 bin stop ---
      curr_m1_stop = (*l_hskp.mass_binn_configPtr)[i,1]
      if (prev_m1_stop ne curr_m1_stop) then change = 1L

      ;--- mass 2 bin start ---
      curr_m2_start = (*l_hskp.mass_binn_configPtr)[i,2]
      if (prev_m2_start ne curr_m2_start) then change = 1L
   
      ;--- mass 2 bin stop ---
      curr_m2_stop = (*l_hskp.mass_binn_configPtr)[i,3]
      if (prev_m2_stop ne curr_m2_stop) then change = 1L
 
      ;--- sw version ---
      curr_sw_ver = (*l_hskp.sys_lev_statsPtr)[i, 0]
      if (prev_sw_ver ne curr_sw_ver) then change = 1L

      ;--- ROI 0 packet status ---
      curr_roi0 = (*l_hskp.data_pkt_sel_statPtr)[i,3]
      if (prev_roi0 ne curr_roi0) then change = 1L
    
      ;--- ROI 1 packet status ---
      curr_roi1 = (*l_hskp.data_pkt_sel_statPtr)[i,4]
      if (prev_roi1 ne curr_roi1) then change = 1L

      ;--- ROI 2 packet status ---
      curr_roi2 = (*l_hskp.data_pkt_sel_statPtr)[i,5]
      if (prev_roi2 ne curr_roi2) then change = 1L

      ;--- ROI 3 packet status ---
      curr_roi3 = (*l_hskp.data_pkt_sel_statPtr)[i,6]
      if (prev_roi3 ne curr_roi3) then change = 1L

      dmpstr = dmpstr + tail

      if (change eq 1L) then begin
         result = execute (dmpstr)
         prev_mcp_start = curr_mcp_start
         prev_mcp_stop  = curr_mcp_stop
         prev_collp     = curr_collp
         prev_colln     = curr_colln
         prev_optics    = curr_optics
         prev_steering  = curr_steering
         prev_start_cfd = curr_start_cfd
         prev_stop_cfd  = curr_stop_cfd
         prev_m1_start  = curr_m1_start
         prev_m1_stop   = curr_m1_stop
         prev_m2_start  = curr_m2_start
         prev_m2_stop   = curr_m2_stop
         prev_sw_ver    = curr_sw_ver
         prev_roi0      = curr_roi0
         prev_roi1      = curr_roi1
         prev_roi2      = curr_roi2
      endif
 endfor

RETURN
END


;-------------------------------------------------------------------------------
;  Procedure: l_StateTableDumpHskpOnly
;
;  Description:
;      Generates the the state table. Only HSKP command data available, no sub-spin
;  level data from ECHO available.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  ofd                   int           O        file handler
;  l_setup               struct        I        general setup parameters
;  l_hskp                struct *      I        housekeeping data pointers
;  l_pltpos              struct        I        general plot setup parameters
;  filename              string        I        file name
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                06/01          v1.0             Original implementation
;-------------------------------------------------------------------------------

PRO l_StateTableDumpHskpOnly, ofd, l_setup, l_hskp, l_pltpos, filename, echo_msg


   l_DumpHdrInfo, ofd, l_setup, l_pltpos, filename

   l_StateTableDocumentation, ofd, echo_msg 

   nitems = n_elements ((*l_hskp.startUdfTimePtr)[*])

   prev_mcp_start = 0L
   prev_mcp_stop  = 0L
   prev_collp     = 0L
   prev_colln     = 0L
   prev_optics    = 0L
   prev_steering  = 0L
   prev_start_cfd = 0L
   prev_stop_cfd  = 0L
   prev_m1_start  = 0L
   prev_m1_stop   = 0L
   prev_m2_start  = 0L
   prev_m2_stop   = 0L
   prev_sw_ver    = 0L
   prev_roi0      = 0L
   prev_roi1      = 0L
   prev_roi2      = 0L
   prev_roi3      = 0L

   for i=0L, nitems-1L do begin

      change = 0L
      dmpstr = ' '
      tail = ' '

      ;-------------------------------------------------------
      ; write the data to the text file
      ;-------------------------------------------------------
      tmestr = STRING ((*l_hskp.startUdfTimePtr)[i].doy, $
                       (*l_hskp.startUdfTimePtr)[i].hour,$
                       (*l_hskp.startUdfTimePtr)[i].min, $
                       (*l_hskp.startUdfTimePtr)[i].sec, $
                       FORMAT="(i3,' ',i2.2,':',i2.2,':',i2.2)")

      dmpstr = 'printf, ofd, tmestr, '                           ;-- utc --
      dmpstr = dmpstr + 'mcp_start, '                            ;-- mcp start --
      dmpstr = dmpstr + 'mcp_stop, '                             ;-- mcp stop --
      dmpstr = dmpstr + 'coll_p, '                               ;-- coll positive --
      dmpstr = dmpstr + 'coll_n, '                               ;-- coll negative --
      dmpstr = dmpstr + 'optics, '                               ;-- optics --
      dmpstr = dmpstr + '(*l_hskp.hvps_configPtr)[i, 13], '      ;-- optics steering --
      dmpstr = dmpstr + '(*l_hskp.tof_subsys_configPtr)[i, 0], ' ;-- start cfd --
      dmpstr = dmpstr + '(*l_hskp.tof_subsys_configPtr)[i, 1], ' ;-- stop cfd --
      dmpstr = dmpstr + '(*l_hskp.mass_binn_configPtr)[i,0], '   ;-- mass 1 bin start --
      dmpstr = dmpstr + '(*l_hskp.mass_binn_configPtr)[i,1], '   ;-- mass 1 bin stop --
      dmpstr = dmpstr + '(*l_hskp.mass_binn_configPtr)[i,2], '   ;-- mass 2 bin start --
      dmpstr = dmpstr + '(*l_hskp.mass_binn_configPtr)[i,3], '   ;-- mass 2 bin stop --
      dmpstr = dmpstr + '(*l_hskp.sys_lev_statsPtr)[i, 0], '     ;-- sw version --
      dmpstr = dmpstr + '(*l_hskp.data_pkt_sel_statPtr)[i,3], '  ;-- ROI 0 packet status --
      dmpstr = dmpstr + '(*l_hskp.data_pkt_sel_statPtr)[i,4], '  ;-- ROI 1 packet status --
      dmpstr = dmpstr + '(*l_hskp.data_pkt_sel_statPtr)[i,5], '  ;-- ROI 2 packet status --
      dmpstr = dmpstr + '(*l_hskp.data_pkt_sel_statPtr)[i,6], '  ;-- ROI 3 packet status --
      dmpstr = dmpstr + 'format="((A12,TR1), '

      tail = '(e9.2,TR1), 2(e9.2,TR2), 2(e9.2,TR2), (e9.2,TR2), (e9.2,TR6), (I2,TR8), 4(I1,TR4))"'

      ;-------------------------------------------------------
      ; check if flags are enabled, if not write out a -9
      ; for no data or disabled
      ;-------------------------------------------------------
      ;--- mcp start enable flag ---
      if ((*l_hskp.hvps_configPtr)[i,2] ne 0) then begin
          mcp_start = (*l_hskp.hvps_configPtr)[i, 8]
          curr_mcp_start = mcp_start
          dmpstr = dmpstr + '(e9.2,TR2)'
      endif else begin
          mcp_start = 'd'
          curr_mcp_start = -99999
          dmpstr = dmpstr + '(A3,8X)'
      endelse
      if (prev_mcp_start ne curr_mcp_start) then change = 1L

      ;--- mcp stop enable flag ---
      if ((*l_hskp.hvps_configPtr)[i,3] ne 0) then begin
          mcp_stop = (*l_hskp.hvps_configPtr)[i, 9]
          curr_mcp_stop = mcp_stop
          dmpstr = dmpstr + '(e9.2,TR2)'
      endif else begin
          mcp_stop =  'd'
          curr_mcp_stop = -99999
          dmpstr = dmpstr + '(A3,8X)'
      endelse
      if (prev_mcp_stop ne curr_mcp_stop) then change = 1L

      ;--- coll positive enable flag ---
      if ((*l_hskp.hvps_configPtr)[i,4] ne 0) then begin
          coll_p = (*l_hskp.hvps_configPtr)[i, 10]
          curr_collp = coll_p
          dmpstr = dmpstr + '(e9.2,TR2)'
      endif else begin
          coll_p = 'd'
          curr_collp = -99999
         dmpstr = dmpstr + '(A3,8X)'
      endelse
      if (prev_collp ne curr_collp) then change = 1L

      ;--- coll negative enable flag ---
      if ((*l_hskp.hvps_configPtr)[i,5] ne 0) then begin
          coll_n = (*l_hskp.hvps_configPtr)[i, 11]
          curr_colln = coll_n
          dmpstr = dmpstr + '(e9.2,TR2)'
      endif else begin
          coll_n =  'd'
          curr_colln = -99999
          dmpstr = dmpstr + '(A3,8X)'
      endelse
      if (prev_colln ne curr_colln) then change = 1L

      ;--- optics enable flag ---
      if ((*l_hskp.hvps_configPtr)[i,6] ne 0) then begin
          optics = (*l_hskp.hvps_configPtr)[i, 12]
          curr_optics = optics
          dmpstr = dmpstr + '(e9.2,TR2)'
      endif else begin
          optics = 'd'
          curr_optics = -99999
          dmpstr = dmpstr + '(A3,8X)'
      endelse
      if (prev_optics ne curr_optics) then change = 1L


      ;--- optics steering ---
      curr_steering = (*l_hskp.hvps_configPtr)[i,13]
      if (prev_steering ne curr_steering) then change = 1L

      ;--- start cfd ---
      curr_start_cfd = (*l_hskp.tof_subsys_configPtr)[i,0]
      if (prev_start_cfd ne curr_start_cfd) then change = 1L

      ;--- stop cfd ---
      curr_stop_cfd = (*l_hskp.tof_subsys_configPtr)[i, 1]
      if (prev_stop_cfd ne curr_stop_cfd) then change = 1L

      ;--- mass 1 bin start ---
      curr_m1_start =  (*l_hskp.mass_binn_configPtr)[i,0]
      if (prev_m1_start ne curr_m1_start) then change = 1L

      ;--- mass 1 bin stop ---
      curr_m1_stop = (*l_hskp.mass_binn_configPtr)[i,1]
      if (prev_m1_stop ne curr_m1_stop) then change = 1L

      ;--- mass 2 bin start ---
      curr_m2_start = (*l_hskp.mass_binn_configPtr)[i,2]
      if (prev_m2_start ne curr_m2_start) then change = 1L

      ;--- mass 2 bin stop ---
      curr_m2_stop = (*l_hskp.mass_binn_configPtr)[i,3]
      if (prev_m2_stop ne curr_m2_stop) then change = 1L

      ;--- sw version ---
      curr_sw_ver = (*l_hskp.sys_lev_statsPtr)[i, 0]
      if (prev_sw_ver ne curr_sw_ver) then change = 1L

      ;--- ROI 0 packet status ---
      curr_roi0 = (*l_hskp.data_pkt_sel_statPtr)[i,3]
      if (prev_roi0 ne curr_roi0) then change = 1L

      ;--- ROI 1 packet status ---
      curr_roi1 = (*l_hskp.data_pkt_sel_statPtr)[i,4]
      if (prev_roi1 ne curr_roi1) then change = 1L

      ;--- ROI 2 packet status ---
      curr_roi2 = (*l_hskp.data_pkt_sel_statPtr)[i,5]
      if (prev_roi2 ne curr_roi2) then change = 1L

      ;--- ROI 3 packet status ---
      curr_roi3 = (*l_hskp.data_pkt_sel_statPtr)[i,6]
      if (prev_roi3 ne curr_roi3) then change = 1L

      dmpstr = dmpstr + tail

      if (change eq 1L) then begin
         result = execute (dmpstr)
         prev_mcp_start = curr_mcp_start
         prev_mcp_stop  = curr_mcp_stop
         prev_collp     = curr_collp
         prev_colln     = curr_colln
         prev_optics    = curr_optics
         prev_steering  = curr_steering
         prev_start_cfd = curr_start_cfd
         prev_stop_cfd  = curr_stop_cfd
         prev_m1_start  = curr_m1_start
         prev_m1_stop   = curr_m1_stop
         prev_m2_start  = curr_m2_start
         prev_m2_stop   = curr_m2_stop
         prev_sw_ver    = curr_sw_ver
         prev_roi0      = curr_roi0
         prev_roi1      = curr_roi1
         prev_roi2      = curr_roi2
         prev_roi3      = curr_roi3
      endif

   endfor

RETURN
END

;-------------------------------------------------------------------------------
;  Function: l_StateTableProc
;
;  Description:
;     Main controlling routine used to call external routines to read the 
;  housekeeping data and to generate the state table.
;
;  Return Value:  type = int
;  Value                         Description
;  -------------------------     ------------------------------
;  SUCCESS                       successful completion
;  FAILURE                       failure to complete
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_setup               struct        I        general setup parameters
;  l_setup_file          string        I        setup file name
;  l_pltpos              struct        I        general plot parameters
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro 
;      ERROR_TYPES     SUCCESS         int         G          successful completion
;                      FAILURE         int         G          unsuccessful completion
;                      WARNING         int         G          warning message
;                      INFO            int         G          informational message
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                06/01          v1.0             Original implementation
;-------------------------------------------------------------------------------


FUNCTION l_NewStateTableProc, l_setup, l_setup_file, l_pltpos 
COMMON ERROR_TYPES, SUCCESS, FAILURE, WARNING, INFO


   l_pltpos.plot_hdr_title = 'State Table'
   l_setup.l_filedesc = 'st'

   ;-----------------------------------------
   ; generate the state table log file name 
   ;-----------------------------------------
   l_GenFileName, l_setup, '0'
   outfile = l_setup.l_prod_dst + '/' + $
             l_setup.l_filename + '.dmp'

   ;-----------------------------------------
   ; open state table log file
   ;-----------------------------------------
   openw, ofd, outfile, /get_lun


   l_GenHvpsCmdTable, l_hvps_cmd

   echo_status = SUCCESS
   hskp_status = SUCCESS

   echo_msg = ' ' 
   echo_status = l_EchoGetData (l_setup, $
                                l_setup_file, $
                                l_pltpos, $
                                l_echo)

   if (echo_status eq FAILURE) then begin
       echo_msg = 'No Echo (sub-spin level) command data available.'
   endif else begin
       l_EchoProcData, l_echo, l_hvps_cmd
   endelse

   hskp_status = l_HskpGetData (l_setup, $
                                l_setup_file, $
                                l_pltpos, $
                                l_hskp_setup, $
                                l_hskp)
   if (hskp_status eq FAILURE) then hskp_msg = 'No HSKP command data available.'


   if (echo_status eq SUCCESS) AND (hskp_status eq SUCCESS) then begin 
      l_StateTableDumpAll, ofd, l_setup, l_hskp, l_pltpos, outfile, l_echo
   endif else begin
      if (echo_status eq FAILURE) AND (hskp_status eq SUCCESS) then begin
          l_StateTableDumpHskpOnly, ofd, l_setup, l_hskp, l_pltpos, outfile, echo_msg 
      endif else begin 
         start_time = l_setup.l_start_date[0] + ' ' + $
                      l_setup.l_start_date[1] + ' ' + $
                      l_setup.l_start_time[0] + ':' + $
                      l_setup.l_start_time[1] + ':' + $
                      l_setup.l_start_time[2] 
         stop_time  = l_setup.l_stop_date[0] + ' ' + $
                      l_setup.l_stop_date[1] + ' ' + $
                      l_setup.l_stop_time[0] + ':' + $
                      l_setup.l_stop_time[1] + ':' + $
                      l_setup.l_stop_time[2] 
         printf, ofd, 'Unable to generate State Table for ', start_time, '--', stop_time 
         printf, ofd, 'No Housekeeping Data for the specified time period is currently available.'
      endelse
   endelse


   close, ofd
   free_lun, ofd

   if (echo_status eq SUCCESS) then l_EchoTerm, l_echo
   if (hskp_status eq SUCCESS) then l_HskpTerm, l_hskp

   RETURN, SUCCESS
 
END
