;-------------------------------------------------------------------------------
;
;  Unit Name : l_MagFieldRtns.pro
;
;  Purpose   : Calculate and draw the magnetic field direction on summary plots.
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;   ELH              02/02           v1.0.8           Original implementation
;
;
;  File Revision Number:   %I%
;  File Last Modified  :   %E%  %T%
;
;-------------------------------------------------------------------------------

;-------------------------------------------------------------------------------
;  Procedure: l_DrawBDirection
;  Description:
;     Draw the B direction.      
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;     PLOTDEFS         MAX_PLOTS        int         G         max # plots per page
;                      MAX_WEB_SPINS    int         G         max # spins per plot
;                                                             when using web interface
;                      NO_CNTS          double      G         no counts indicator
;                      NO_DATA          double      G         no data indicator
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;   ELH              02/02           v1.0.8           Original implementation
;-------------------------------------------------------------------------------

PRO l_DrawBDirection, X, Y, img_pos, l_setup
COMMON PLOTDEFS, MAX_PLOTS, MAX_WEB_SPINS, NO_CNTS, NO_DATA

   blankticks = strarr (30)
   blankticks(*) = ' '

   ntck = 15
  
   ;-------------------------------------------
   ; determine the distance between points
   ; used to plot
   ;-------------------------------------------
   if (n_elements (X) ge 60) AND (n_elements (X) lt 240)  then begin
      ntck = 8
   endif else begin
      if (n_elements (X) lt 60) then begin
         ntck = 1
      endif
   endelse

   nVsum = fix(n_elements(X)/ntck)
   ;-------------------------------------------
   ; reduce the number of points to plot
   ;-------------------------------------------
   x_tmp = X
   y_tmp = Y

   X = fltarr (n_elements(x_tmp))
   Y = dblarr (n_elements(y_tmp))

   for ii = 0L, nVsum -1L do begin 
      X[ii] = x_tmp[ii*ntck]    
      Y[ii] = y_tmp[ii*ntck]
   endfor

   
   X = X[0:ii-1L]
   Y = Y[0:ii-1L]

   gindx = where (Y gt l_setup.l_max_spin_angle, gcnt)
   if (gcnt gt 0) then begin
      Y[gindx] = Y[gindx] - l_setup.l_max_spin_angle
      Y[gindx] = Y[gindx] + l_setup.l_min_spin_angle
   endif

   lindx = where (Y lt l_setup.l_min_spin_angle, lcnt)
   if (lcnt gt 0) then begin
      Y[lindx] = Y[lindx] - l_setup.l_min_spin_angle
      Y[lindx] = Y[lindx] + l_setup.l_max_spin_angle
   endif

   ;-------------------------------------------
   ; plot the positive B direction
   ;-------------------------------------------

   PLOT, X, Y, /normal, /noerase, $
         xstyle=1, $
         ystyle=1, $
         xticks = 1, $
         yticks = 1, $
         xrange = [min(X), max(X)], $
         yrange = [l_setup.l_min_spin_angle, l_setup.l_max_spin_angle], $
         ticklen = -0.01, $
         xtickname = blankticks, $
         ytickname = blankticks, $
         position = img_pos, $
         PSYM = 1,  $                  ; + 
         SYMSIZE = 1., $
         color = 2                     ; white 


   ;-------------------------------------------
   ; determine the negative B direction
   ;-------------------------------------------
   Z = Y

   mid_angle = float((l_setup.l_max_spin_angle + l_setup.l_min_spin_angle)/2.)
   gindx = where (Y ge mid_angle, gcnt)
   if (gcnt gt 0) then Z[gindx] = Y[gindx] - 180.
   lindx = where (Y lt mid_angle, lcnt)
   if (lcnt gt 0) then Z[lindx] = Y[lindx] + 180.


   ;-------------------------------------------
   ; create user symbol '-'
   ;-------------------------------------------
   x_sym = [-0.75, 0, 0.75]
   y_sym = [0, 0, 0]

   usersym, x_sym, y_sym

   ;-------------------------------------------
   ; plot the negative B direction
   ;-------------------------------------------
   OPLOT, X, Z, $
          color = 2, $
          psym = 8


RETURN
END


;-------------------------------------------------------------------------------
;  Procedure: l_DetermineBDir
;  Description:
;     Determine the B direction if the data has been rolled to center sun, earth, or
;  the moon at spin angle zero.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;   ELH              02/02           v1.0.8           Original implementation
;-------------------------------------------------------------------------------

PRO l_DetermineBDir, B_markers, B_dir, l_setup, l_oa

      B_dir = B_markers
      sectors = l_setup.l_num_spinsec - 1L

      if (l_setup.l_angle_zero eq 'E') then begin
         ;---------------------------------------------------------
         ; find the distance the earth was moved to center position
         ;---------------------------------------------------------
         diff = (l_setup.l_earth_center - l_setup.l_spin_ang_zero)

         ;---------------------------------------------------------
         ; subtract from the sun position the amount earth moved
         ;---------------------------------------------------------
         gt_indx = where (B_markers ge diff, gtcnt)

         if (gtcnt gt 0) then begin
            B_dir[gt_indx] = B_markers[gt_indx] - diff
         endif

         ;---------------------------------------------------------
         ; position of sun when readjusted, crosses over the
         ; 44, 0 boundary
         ;---------------------------------------------------------
         lt_indx = where (B_markers lt diff, ltcnt)
         if (ltcnt gt 0) then begin
            B_dir[lt_indx] = sectors - (diff - B_markers[lt_indx] - 1)
         endif

         indx = where (B_dir ge 45, cnt)
         if (cnt gt 0) then begin
           B_dir[indx] = B_dir[indx] - l_setup.l_num_spinsec
         endif

     endif else begin
          if (l_setup.l_angle_zero eq 'S') then begin

             sun = (*l_oa.lena_sun_sectPtr)
             ;---------------------------------------------------------
             ; subtract from the B position the amount sun moved
             ; to be centered at zero spin angle
             ;---------------------------------------------------------
             gt_indx = where (sun ge l_setup.l_spin_ang_zero, gtcnt)

             if (gtcnt gt 0) then begin
                sun_dist = sun[gt_indx] - l_setup.l_spin_ang_zero
                B_dir[gt_indx] = B_markers[gt_indx] - sun_dist
             endif

             ;---------------------------------------------------------
             ; position of s/c when readjusted, if s/c sector is at
             ; 33 will cross the 44, 0 boundary
             ;---------------------------------------------------------
             lt_indx = where (sun lt l_setup.l_spin_ang_zero, ltcnt)
             if (ltcnt gt 0) then begin
                sun_dist = l_setup.l_spin_ang_zero - sun[lt_indx]
                indx_gt33 = where (B_dir[lt_indx] ge $
                          (sectors - l_setup.l_spin_ang_zero), gt33cnt)
                B_dir[lt_indx] = B_markers[lt_indx] + sun_dist

                if (gt33cnt gt 0) then begin
                   B_dir[lt_indx[indx_gt33]] = (B_dir[lt_indx[indx_gt33]] - $
                              l_setup.l_num_spinsec)
                endif
              endif

          endif else begin
             if (l_setup.l_angle_zero eq 'M') then begin
                moon = (*l_oa.lena_moon_sectPtr)
                ;---------------------------------------------------------
                ; subtract from the s/c position the amount moon moved
                ; to be centered at zero spin angle
                ;---------------------------------------------------------
                gt_indx = where (moon ge l_setup.l_spin_ang_zero, gtcnt)

                if (gtcnt gt 0) then begin
                   moon_dist = moon[gt_indx] - l_setup.l_spin_ang_zero
                   B_dir[gt_indx] = B_markers[gt_indx] - moon_dist
                endif

                ;---------------------------------------------------------
                ; position of sun when readjusted, if sun sector is at
                ; 33 will cross the 44, 0 boundary
                ;---------------------------------------------------------
                lt_indx = where (moon lt l_setup.l_spin_ang_zero, ltcnt)
                if (ltcnt gt 0) then begin
                   moon_dist = l_setup.l_spin_ang_zero - moon[lt_indx]
                   indx_gt33 = where (B_dir[lt_indx] ge $
                          (sectors - l_setup.l_spin_ang_zero), gt33cnt)
                   B_dir[lt_indx] = B_markers[lt_indx] + moon_dist

                   if (gt33cnt gt 0) then begin
                      B_dir[lt_indx[indx_gt33]] = (B_dir[lt_indx[indx_gt33]] - $
                                  l_setup.l_num_spinsec)
                   endif
                endif
             endif
          endelse
     endelse


RETURN
END


;-------------------------------------------------------------------------------
;  Procedure: l_CalcBDirection
;  Description:
;     Calls the routines needed to fill in missing data, calculate and draw
;  the B direction to the plots.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;   ELH              02/02           v1.0.8           Original implementation
;-------------------------------------------------------------------------------


PRO l_CalcBDirection, l_oa, l_setup, img_pos

    nspins = n_elements ((*l_oa.tme_ptr))
    l_HndlMissOA, (*l_oa.tme_ptr), $
                  hmd_oa_tme, $
                  nspins, $
                  (*l_oa.lena_mag_fieldPtr), $
                  B_markers, $
                  l_setup.l_start_dtime, $
                  l_setup.l_stop_dtime, $
                  1

    l_DetermineBDir, B_markers, B_dir, l_setup, l_oa

    ;---------------------------------
    ; convert from sector to degrees
    ;---------------------------------
    B_dir = B_dir * 8.

    l_DrawBDirection, hmd_oa_tme, $
                      B_dir, $
                      img_pos,$ 
                      l_setup


RETURN
END

