;------------------------------------------------------------------------------
; Procedure  :  l_LenaPlot.pro
; Description:  The main routine of the LENA data processing software
;
; Change History:
; Developer          Date       Release         Description
; ---------          ----       -------         ---------------------
; John Park                                     Original implemenation
; ELH                           prod1.1         initial plots             
; ELH                           prod1.2         new plot format
;                                                .added no counts/no data 
;                                                 legend
;                                                .new plot documentation
;                                                .new plot x/y sizes
;                                                .color table implementation
;                                                .earth markers
;                                                .treatment of zeros for log
;                                                .extra decades in linear 
;                                                 plots
; ELH                10/00      prod1.3         fixed l_HndlMissData loop 
;                                               control variable needs to be 
;                                               a long var
;                                               fixed line in valid events plot
;                                               added some documentation
;                                               to plots 
; ELH                11/00      prod1.4         added perigee plots, 
;                                               postscript, & sun pulse
;                                               monitoring
; ELH                07/01      v1.0.0          NO_DATA is -9999. and NO_CNT is -8888.
;                                               to match no data for event dumps
; ELH                08/01      v1.0.7          .Added l_BinnedEvent, option to select
;                                               to plot binned or event data.
;                                               .Added background correction algorithm
;                                               .Interpolated UDF times
; RLW                08/01      v1.0.7          . added l_image_id option to l_setup
;                                               . replace l_pltpos.xin and l_pltpos.yin
;                                               to handle sizes for movies, in l_zc3 option
; ELH               09/18/01    v1.0.7          Added l_sun_direction
; ELH               10/29/01    v1.0.7          Added l_LttProc, long term trending of
;                                               sun pulse data using the singles start data
; ELH               11/21/01    v1.0.7          Added msec field to udf time struct
; ELH               12/14/01    v1.0.8          Moved the l_HndlMissData routine in every
;                                               file outside of the l_xxxPlot routines.
;                                               Eliminates redundant processing if PS option
;                                               selected.
; ELH               01/02       v1.0.8          Added l_max_spin_angle and l_min_spin_angle to
;                                               l_setup, Added the routine FindZeroDegBin, set
;                                               min/max spin angle in l_pltpos
;                                               . add l_ram_direction and l_B_direction to
;                                                 l_setup structure
;                                               . removed hdr_title from all calls
;
;
;
;  File Revision Number:   %I%
;  File Last Modified  :   %E%  %T%
;
;-------------------------------------------------------------------------------

;-------------------------------------------------------------------------------
;
;  Return Value:  type = error status
;  Value                         Description
;  -------------------------     ------------------------------
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;     GLOBALS          G_REPORT         int         G         report file handle
;                      G_LOG            int         G         log file handle
;     PLOTDEFS         MAX_PLOTS        int         G         max # plots per page
;                      MAX_WEB_SPINS    int         G         max # spins per plot
;                                                             when using web interface
;                      NO_CNTS          double      G         no counts indicator
;                      NO_DATA          double      G         no data indicator
;     MYCOLORS         N_COLOR_TABLE    int         G         # of colors in color table
;                      N_GRAYS          int         G         # of grays in color table
;     ERROR_TYPES      SUCCESS          int         G         routine completed 
;                                                             successfully
;                      FAILURE          int         G         routine completed
;                                                             unsuccessfully
;     EFFTABLES        O_Eff_Table      double[]    G         oxygen efficiency values 
;                      H_Eff_Table      double[]    G         hydrogen efficiency values
;     EPOCH_DATA       EPOCH_YR         int         G         launch year
;     
;-------------------------------------------------------------------------------

PRO l_LenaPlot
COMMON GLOBALS, G_REPORT, G_LOG
COMMON PLOTDEFS, MAX_PLOTS, MAX_WEB_SPINS, NO_CNTS, NO_DATA
COMMON MYCOLORS, N_COLOR_TABLE, N_GRAYS
COMMON ERROR_TYPES, SUCCESS, FAILURE, WARNING, INFO
COMMON EFFTABLES, O_Eff_Table, H_Eff_Table
COMMON EPOCH_DATA, EPOCH_YR


   !EXCEPT=2

   ;----------------------------------------
   ; global variable declarations
   ;----------------------------------------

   MAX_PLOTS     =  6
   MAX_WEB_SPINS =  50
   NO_DATA       = -9999.
   NO_CNTS       = -8888.
   N_COLOR_TABLE = 226
   N_GRAYS       = 5 

   SUCCESS       = 0
   FAILURE       = -1
   WARNING       = 1 
   INFO          = 2 
 
   EPOCH_YR      = 2000

   status = SUCCESS 

   l_setup_file = ''

   print, 'Enter name of file containing input parameters.'
   read, 'setup file name > ', l_setup_file


   ;----------------------------------------
   ; Hydrogen/Oxygen Efficiency Structure
   ;----------------------------------------
   l_eff_parms = { EFF_PARMS,  $
      energy : 0L, $                ;--- energy bin ---
      eff    : 0.0D $               ;--- efficiency ---
   }

   ;----------------------------------------
   ; UDF Time Structure
   ;----------------------------------------
   l_udf_times = { UDF_TIMES_STRUCT, $
                  year: 0, $ 
                  doy : 0, $
                  hour: 0, $ 
                  min : 0, $
                  sec : 0, $
                  msec: 0  $
                 }

   ;----------------------------------------
   ; begin structure definitions
   ;----------------------------------------
   l_axis_struct = {AXIS_DATA,            $
                    char_size : 1.0,       $
                    ytitle    : '',        $
                    yformat   : '',        $
                    yrange    : [0.0,0.0]  }


   ;----------------------------------------
   ; setup file parameters
   ; values are defaults
   ;----------------------------------------
   l_setup = {                        $
      l_via_web        : 'N',         $    ;--- use via web ---
      l_process_id     : '',          $    ;--- process id ---
      l_gen_sum        : 'N',         $    ;--- generate daily summary plot ---
      l_start_date     : [0,0,0],     $    ;--- user start date ---
      l_start_time     : [0,0,0],     $    ;--- user start time ---
      l_stop_date      : [0,0,0],     $    ;--- user stop date ---
      l_stop_time      : [0,0,0],     $    ;--- user stop time ---
      l_start_dtime    : [0,0,0,0,0,0], $    ;--- user start date/time 
                                             ;    [yy,doy,hh,mm,ss,ms] ---
      l_stop_dtime     : [0,0,0,0,0,0], $    ;--- user stop date/time ---
                                             ;    [yy,doy,hh,mm,ss,ms] ---
      l_d_start_dtime  : [0,0,0,0,0,0], $    ;--- data start date/time ---
                                             ;    [yy,doy,hh,mm,ss,ms] ---
      l_d_stop_dtime   : [0,0,0,0,0,0], $    ;--- data start date/time ---
                                             ;    [yy,doy,hh,mm,ss,ms] ---
      l_data_type      : '',          $    ;--- equivalent to data packet type ---
      l_plot_type      : 'X',          $    ;--- plot type ---
      l_data_src       : '',          $    ;--- data source ---
      l_prod_dst       : '',          $    ;--- product destination ---
      l_num_spin       : 0.0,        $    ;--- # of spins to process ---
      l_num_energylev  : 3,           $    ;--- # of energy bins ---
      l_num_polarzn    : 12,          $    ;--- # of polar zones ---
      l_num_spinsec    : 45,          $    ;--- # of spin sectors ---
      l_num_roibins    : 32,          $    ;--- # of ROI bins ---
      l_gen_report     : 'N',         $    ;--- generate report file ---
      l_report_file    : '',          $    ;--- report filename ---
      l_gen_movie      : 'Y',         $    ;--- generate movie ---
      l_image_id       : 0L,          $    ;--- gen movie of this species ---
      l_image_correction: 'N',        $    ;--- correct distorted image ---
      l_color_tbl      : 1,           $    ;--- color table selection ---
      l_filename       : '',          $    ;--- filename to be created ---
      l_abspathfn      : '',          $    ;--- absolute path of file location ---
      l_filedesc       : '',          $    ;--- file type --- 
      l_earth_center   : 17,          $    ;--- spin sector where earth is ---
      l_spin_ang_zero  : 11,          $    ;--- plot bin # 11 is spin angle zero ---
      l_linear_log     : 0L,          $
      l_color_min      : 0.0,         $    ;--- min color scale ---
      l_color_max      : 0.0,         $    ;--- max color scale ---
      l_auto_scale     : '',          $    ;--- automatically color scale ---
      l_sep_plot_scale : '',          $    ;--- separately color scale each plot ---
      l_avg_sum        : '',          $    ;--- average and sum data ---
      l_line_plots     : 'N',         $    ;--- generate line plots ---
      l_earth_markers  : 'N',         $    ;--- draw earth markers ---
      l_output_type    : 0L,          $    
      l_angle_zero     : '',          $    ;--- angle zero is earth, sun, moon ---
      l_sun_direction  : 'N',         $    ;--- draw the sun direction ---
      l_postscript     : 'N',         $    ;--- generate a postscript ---
      l_dumps          : 'N',         $    ;--- generate text dumps of data ---
      l_dumps_bkdata   : 'N',         $    ;--- generate text dumps of background data ---
      l_pwrsup_bars    : 'N',         $
      l_singles_sum    : 1  ,         $ 
      l_count_rate     : 'Y',         $    ;--- convert data into count rate ---
      l_log            : 'Y',         $    ;--- convert data into log ---
      l_flux           : 'Y',         $    ;--- convert data into flux ---
      l_background     : 'N',         $    ;--- background subtraction ---
      l_max_spin_angle : 270,         $    ;--- max spin angle ---
      l_min_spin_angle : -90,         $    ;--- min spin angle ---
      l_ram_direction  : 'N',         $    ;--- ram direction ---
      l_B_direction    : 'N',         $    ;--- magnetic field directions ---
      l_Xaxis_Label    : '' ,         $    ;--- for long term plots, X axis label ---
      l_Xaxis_ymd_label: 'N',         $    ;--- year/month/day labels for LTT ONLY!!! 
      l_color_values   : 'N',         $    ;--- dump color values from plot to text dump ---
      l_smooth_data    : 'N',         $    ;--- smooth spectrogram data ---
      l_output_plot_format : 'GIF'    $    ;--- format of plot output, PNG or GIF --- 
   }

   ;----------------------------------------
   ;  color table
   ;----------------------------------------
   l_rgb_table = {                $
     red     : intarr (N_COLOR_TABLE),     $
     blue    : intarr (N_COLOR_TABLE),     $
     green   : intarr (N_COLOR_TABLE)      $
   }


   ;---------------------------------------------
   ; plot attributes
   ;---------------------------------------------


   l_pltpos = {             $
      window_width : 0L,    $           ;--- plot window width ---
      window_height: 0L,    $           ;--- plot window height ---
      yin          : 9.9,   $           ;--- height in inches ---
      xin          : 8.2,   $           ;--- width in inches ---
      ps_xoffset   : 0.50,  $           ;--- postscript x offset ---
      ps_yoffset   : 1.30,  $           ;--- postscript y offset ---
      ps_xsize     : 7.70,  $           ;--- postscript width size ---
      ps_ysize     : 9.00,  $           ;--- postscript height size ---
      axis_size    : 6.35,  $           ;--- width of plot ---
      xdel         : 0.0,   $           ;--- width (along x-axis) of plots ---
      xpltmn       : 0.0,   $           ;--- start x location of plot ---  
      xpltmx       : 0.0,   $           ;--- end x location of plot ---
      ypltmn       : 0.0,   $           ;--- start y location of plot ---
      ypltmx       : 0.0,   $           ;--- stop y location of plot ---
      top_ypltmn   : 0.0,   $           ;--- first original y start location of plot ---
      top_ypltmx   : 0.0,   $           ;--- first original y stop location of plot ---
      doc_ymn      : 0.0,   $           ;--- current document y start location ---
      doc_ymx      : 0.0,   $           ;--- current document y stop location ---
      ydel         : 0.0,   $           ;--- height of plot --- 
      yspace       : 0.0,   $           ;--- spacing (along y-axis) between plots --- 
      lg_font      : 1.1,   $           ;--- default large font size ---
      md_font      : 1.0,   $           ;--- default medium font size ---
      sm_font      : 0.9,   $           ;--- default small font size ---
      xsm_font     : 0.5,   $           ;--- default extra small font size ---
      plot_font    : 1.2,   $           ;--- default font size selected for plot text ---
      xmn_tick     : 1,     $           ;--- minimum # of x-axis tick marks ---
      ymn_tick     : 1,     $           ;--- minimum # of y-axis tick marks ---
      nticks       : 6,     $           ;--- default number of major tick marks ---
      data_source  : '',    $           ;--- definitive/quicklook data indicator ---
      filename     : '',    $           ;--- current filename ---
      ps           : 'N',   $           ;--- postscript format to be used ---
      linePlots    : 'N',   $           ;--- plot line plots ---
      xtick_labels : strarr (20), $     ;--- string array of labels for x-axis ---
      xtick_ymd_labels : strarr (20), $ ;--- string array of labels for x-axis LTT ONLY!! ---
      xtick_locs   : dblarr (20), $     ;--- location of x-axis labels
      xtick_delta  : 0, $               ;--- spacing between x-axis tick labels/marks ---
      mn_tick_delta: 0, $               ;--- minor tick label spacing ---
      min_time_axis_len : 0.0D0, $       
      max_time_axis_len : 0.0D0, $
      first_time_label : 0.0D0, $
      xtitle       : ' ',   $
      sw_ver       : 'lena_dps_v1.0.8', $ ;--- LENA DPS software version ---
      plot_hdr_title: ' ', $            ;--- Plot title ---
      dmp_hdr_title: ' ', $             ;--- text dump title ---
      color_table  : 1, $               ;--- color table option ---
      ang_zero     : '', $              ;--- center data (sun, earth,moon) at 0 spin angle --- 
      min_spin_ang : -90, $             ;--- minimum spin angle ---
      max_spin_ang : 270, $             ;--- maximum spin angle ---
      X_axis_eq_time: 'Y', $            ;--- x-axis labelling is time ---
      X_axis_label : ' ', $             ;--- x-axis type: time or other ---
      X_axis_ymd_label : ' ', $         ;--- this option only available for LTT ONLY !! ---
      linestyle    : 'N', $           
      data_type    : ' ', $
      tokar_display: 'N',  $             ;--- indicates if tokar plots are selected to be plotted ---
      start_plt_time : [0,0,0,0,0,0], $
      stop_plt_time : [0,0,0,0,0,0] $
    }

   ; screen pixels per inch
   SCREEN_PIXELS = 75
   l_pltpos.window_width  = fix(l_pltpos.xin * SCREEN_PIXELS) 
   l_pltpos.window_height = fix(l_pltpos.yin * SCREEN_PIXELS) 

   ;----------------------------------------
   ; end structure definitions
   ;----------------------------------------

   ;----------------------------------------
   ; Error handling routine
   ;----------------------------------------
   ;error_report = 0
   ;catch, error_status
   ;if error_status ne 0 then begin
   ;   print, !ERR_STRING
   ;   goto, ERROR_FOUND
   ;
   ;   ;openw, error_report, /get_lun, 'l_error_log' 
   ;   ;close, error_report
   ;   ;free_lun, error_report
   ;endif


   ;----------------------------------------
   ; generate H/O efficiency tables 
   ;----------------------------------------
   l_GenEffTable, O_Eff_Table, H_Eff_Table

   ;----------------------------------------
   ; read the setup file
   ;----------------------------------------
   error_msg = ' '
   status = l_ReadSetup (l_setup, l_setup_file, error_msg)
   if (status eq FAILURE) then begin
      print, error_msg
      goto, ERROR_FOUND
   endif
   if (l_setup.l_pwrsup_bars eq 'Y') then begin
      MAX_PLOTS = 5
   endif


   l_pltpos.linePlots = l_setup.l_line_plots
   l_pltpos.ang_zero  = l_setup.l_angle_zero
   l_pltpos.color_table = l_setup.l_color_tbl
   l_pltpos.X_axis_label = l_setup.l_Xaxis_Label
   l_pltpos.X_axis_ymd_label = l_setup.l_Xaxis_ymd_label 

   ;----------------------------------------
   ; this checks the user's setup file
   ; if user does not want the earth drawn
   ; at spin angle zero (spin angle zero is
   ; either the sun or the earth), then
   ; earth markers should not be drawn
   ;----------------------------------------
   range = abs(l_setup.l_min_spin_angle) + abs(l_setup.l_max_spin_angle)
   if (range ne 360) then begin
      print, 'Min/Max spin angle is not 360 degrees. Please re-enter angles.'
      return
   endif else begin
      if (l_setup.l_angle_zero ne 'E') then begin
        if (l_setup.l_angle_zero eq 'N') then begin
            l_setup.l_min_spin_angle = -140
            l_setup.l_max_spin_angle = 220 
        endif
      endif
      l_pltpos.min_spin_ang = l_setup.l_min_spin_angle
      l_pltpos.max_spin_ang = l_setup.l_max_spin_angle
      l_setup.l_spin_ang_zero = FindZeroDegBinNo (l_pltpos.max_spin_ang, l_setup.l_num_spinsec)
   endelse



   ;----------------------------------------
   ; LENA software allows the input of date
   ; and time to be either day of year or
   ; month/day. This routine will convert 
   ; the start date and time to day of year,
   ; if necessary, for UDF processing
   ;----------------------------------------
   l_setup.l_start_dtime = l_InputTimeConv (l_setup.l_start_date, $
                                            l_setup.l_start_time)

   l_setup.l_stop_dtime  = l_InputTimeConv (l_setup.l_stop_date, $
                                            l_setup.l_stop_time)


   print, 'l_setup.l_start_dtime = ', l_setup.l_start_dtime
   print, 'l_setup.l_stop_dtime  = ', l_setup.l_stop_dtime

   ;-----------------------------------------
   ; values are based on post script output 
   ; (inches)
   ; set values for plot output
   ;-----------------------------------------

   ;-----------------------------------------
   ; starting location of x
   ;-----------------------------------------
   l_pltpos.xpltmn = 0.92 / l_pltpos.xin

   ;-----------------------------------------
   ;sets the width (x axis)
   ;-----------------------------------------
   l_pltpos.xpltmx = l_pltpos.xpltmn + l_pltpos.axis_size/ l_pltpos.xin

   ;-----------------------------------------
   ; starting location at y
   ;-----------------------------------------
   l_pltpos.ypltmx = 8.9/l_pltpos.yin

;print, 'l_LENAPLOT l_pltpos.ypltmx = ', l_pltpos.ypltmx
   ;-----------------------------------------
   ;the Y (height) of the plot, by default
   ;-----------------------------------------
   plot_size = floor(float(l_pltpos.yin/MAX_PLOTS))
   l_pltpos.ydel   = plot_size/l_pltpos.yin

   ;-----------------------------------------
   ; the amount of spacing in between plots
   ;-----------------------------------------
   l_pltpos.yspace = 0.15/ l_pltpos.yin

   l_pltpos.ypltmn = l_pltpos.ypltmx - l_pltpos.ydel

;print, 'l_LENAPLOT l_pltpos.ypltmn = ', l_pltpos.ypltmn

   l_pltpos.top_ypltmx = l_pltpos.ypltmx
   l_pltpos.top_ypltmn = l_pltpos.ypltmn


   ;----------------------------------------
   ; init report
   ;----------------------------------------
   if (l_setup.l_gen_report eq 'Y') then begin
       filename = l_setup.l_prod_dst+ '/' + l_setup.l_report_file
       openw, G_REPORT, /get_lun, filename
       printf, G_REPORT, '--------------------------------------------------------------------------------'
       printf, G_REPORT, 'LENAPLOT REPORT : ', systime()
       printf, G_REPORT, '--------------------------------------------------------------------------------'
   endif

   if (l_setup.l_via_web eq 'Y') then begin
       filename = l_setup.l_prod_dst + "/"
       filename = filename + "lenaplot" + l_setup.l_process_id + ".log"
       openw, G_LOG, /get_lun, filename
       printf, G_LOG, '--------------------------------------------------------------------------------'
       printf, G_LOG, 'LENAPLOT LOG : ', systime()
       printf, G_LOG, '--------------------------------------------------------------------------------'
   endif else begin
       G_LOG = -1
       printf, G_LOG, "G_LOG is stdout"
   endelse


   l_pltpos.data_type = l_setup.l_data_type



   ;----------------------------------------
   ; process data and generate plot
   ;----------------------------------------
   case l_setup.l_data_type of
       'l_hob': begin   ;--- hydrogen, oxygen, roi ---
           status = l_BinnedEventProc (l_setup, l_setup_file, l_pltpos, l_rgb_table)
        end
       'l_zss'	: begin ;--- singles ---
           status = l_SnglsProc (l_setup, l_setup_file, l_pltpos, l_rgb_table)
        end
       'l_zse'	: begin ;--- event ---
           status = l_EventProc (l_setup, l_setup_file, l_pltpos, l_rgb_table)
        end
       'l_mspec'	: begin ;--- mass spectrum ---
           status = l_MassSpectProc (l_setup, l_setup_file, l_pltpos, l_rgb_table)
        end
       'l_zr0': begin   ;--- roi0 ---
           status = l_RoiProc (l_setup, l_setup_file, l_pltpos, l_rgb_table)
        end
       'l_zhd'	: begin
           status = l_DiagProc (l_setup, l_setup_file, l_pltpos)
        end
       'l_zhk' : begin  ;--- normal housekeeping ---
           status = l_HskpProc (l_setup, l_setup_file, l_pltpos, l_rgb_table)
        end
       'l_zc1': begin   ;--- comb 1, singles & housekeeping ---
           status = l_ZC1Proc (l_setup, l_setup_file, l_pltpos, l_rgb_table)
        end
       'l_zc2': begin   ;--- comb 2, singles & coincidences ---
           status = l_ZC2Proc (l_setup, l_setup_file, l_pltpos, l_rgb_table)
        end
       'l_zc3': begin   ;--- comb3, image (browse) plots ---
           if (l_setup.l_gen_movie eq 'Y') then begin
              if (l_setup.l_output_type eq 1) then begin
                 l_pltpos.xin = 3.4
                 l_pltpos.yin = 7.3
              endif else begin
                 l_pltpos.xin = 4.5
                 l_pltpos.yin = 4.0
              endelse
           endif else begin
              l_pltpos.xin = 9.0
              l_pltpos.yin = 8.0
           endelse
           l_pltpos.window_width = fix(l_pltpos.xin * SCREEN_PIXELS)
           l_pltpos.window_height = fix(l_pltpos.yin * SCREEN_PIXELS)
           l_pltpos.ps_xoffset = 0.10
           l_pltpos.ps_yoffset = 1.50
           l_pltpos.ps_xsize  = 9.00
           l_pltpos.ps_ysize  = 8.00
           l_ZC3Proc, l_setup, l_setup_file, l_pltpos, l_rgb_table
        end
       'l_iod': begin   ;--- comb3, image (browse) plots ---
           if (l_setup.l_gen_movie eq 'Y') then begin
              if (l_setup.l_output_type eq 1) then begin
                 l_pltpos.xin = 3.4
                 l_pltpos.yin = 7.3
              endif else begin
                 l_pltpos.xin = 4.5
                 l_pltpos.yin = 4.0
              endelse
           endif else begin
              l_pltpos.xin = 9.0
              l_pltpos.yin = 8.0
           endelse
           l_pltpos.window_width = fix(l_pltpos.xin * SCREEN_PIXELS)
           l_pltpos.window_height = fix(l_pltpos.yin * SCREEN_PIXELS)
           l_pltpos.ps_xoffset = 0.10
           l_pltpos.ps_yoffset = 1.50
           l_pltpos.ps_xsize  = 9.00
           l_pltpos.ps_ysize  = 8.00
           l_IODProc, l_setup, l_setup_file, l_pltpos, l_rgb_table
        end
       'l_per': begin  ;--- perigee ---
           status = l_PerigeeProc (l_setup, l_setup_file, l_pltpos, l_rgb_table)
        end
       'l_apo': begin  ;--- apogee ---
           status = l_ApogeeProc (l_setup, l_setup_file, l_pltpos, l_rgb_table)
        end
       'l_psc': begin ;--- power supply commanded levels ---
          status = l_PwSupCmdProc (l_setup, l_pltpos)
        end
       'l_psm': begin ;--- power supply monitored levels ---
          status = l_PwSupMonProc (l_setup, l_pltpos)
        end
       'l_stab': begin ;--- state table log file ---
           ;l_StateTableProc, l_setup, l_setup_file, l_pltpos
           status = l_NewStateTableProc (l_setup, l_setup_file, l_pltpos)
        end
       'l_cmd': begin ;--- state table log file ---
            status = l_EchoProc (l_setup, l_setup_file, l_pltpos)
        end
       'l_ltt': begin ;--- long term trending of start singles ---
           status = l_LttProc (l_setup, l_setup_file, l_pltpos, l_rgb_table)
        end
   else: begin
        printf, G_LOG, 'option unknown'
       end
   endcase


   ERROR_FOUND: 

   if (l_setup.l_gen_report eq 'Y') then begin
       close, G_REPORT
       free_lun, G_REPORT
   endif

   if (l_setup.l_via_web eq 'Y') then begin
      close, G_LOG
      free_lun, G_LOG
   endif

   ;exit, status=error_status


END
