
;-------------------------------------------------------------------------------
;
;  Unit Name : l_EventDumpRtns.pro
;
;  Purpose   : Contains routines which generates a text dump of the event data.
;             Special routines used for Michael Collier's studies on the back
;             ground corrected data.  Not part of the normal production runs.
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;   ELH              04/25/03       v1.0.11           Original implementation
;
;
;  File Revision Number:   %I%
;  File Last Modified  :   %E%  %T%
;
;-------------------------------------------------------------------------------

;-------------------------------------------------------------------------------
;  Procedure: EventBKHeader
;
;  Description: Writes a column description.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  type                  string        I        hydrogen or oxygen
;  filedes               long          I        output file descriptor
;  title                 string        I        title 
;  zero_title            string        I        additional title data
;  data_type             string        I       
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------

PRO EventBKHeader, type, filedes, title, zero_title, data_type


      printf, filedes, ' '
      printf, filedes, title
      printf, filedes, zero_title
      printf, filedes, ' '

      if (STRUPCASE(data_type) eq 'HYDROGEN') then begin 
         printf, filedes,'UTC','Azi', 'H Raw', 'Events', 'Binned', 'Scale', $
                         '  Corr/Scl',  $
                         format='((A8,TR4), (A6, TR7) , 5(A10, TR5), (A8,TR1), 2(A10, TR4))'
      endif

      if (STRUPCASE(data_type) eq 'OXYGEN') then begin 
         printf, filedes,'UTC','Azi', 'O Raw', 'Events', 'Binned', 'Scale', $
                         '  Corr/Scl',  $
                         format='((A8,TR4), (A6, TR7) , 5(A10, TR5), (A8,TR1), 2(A10, TR4))'
      endif

      printf, filedes,'   ','   ', '     ',  '     ', 'Events', 'Factor', $
                      ' Value',  $
                      format='((A8,TR4), (A6,TR7) , 4(A12, TR3), (A10, TR4))'

      printf, filedes,'---','---', '------', '-----', '------', '--------', $
                      ' ----------', $
                      format='((A8,TR4), (A6, TR7) , 4(A12, TR3), (A10, TR4))'

RETURN
END


;-------------------------------------------------------------------------------
;  Procedure: EventWriteBKFile
;
;  Description: Write the event background data to a file.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  nspins                long          I        number of spins to write
;  bkcUdfTimePtr         int[]         I        UDF time corresponding to background
;                                               corrected data
;  raw_cnts              long[]        I        raw data counts 
;  events_n_sec          long[]        I        # events in sector 
;  binn_event            long[]        I        # of binned events 
;  scale_factor          long[]        I        scaled data 
;  corr_cnt              long[]        I        background corrected data 
;  spinsec               long[]        I        spin sector 
;  filede                int           I        output file descriptor
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------

PRO EventWriteBKFile, nspins, background_flag, bkcUdfTimePtr, raw_cnts, events_n_sect, $
binn_events, scale_factor, corr_cnts, spinsec, filedes


          for i = 0L, nspins - 1L do begin
              tmestr = STRING ((bkcUdfTimePtr)[i].doy, $
                               (bkcUdfTimePtr)[i].hour,$
                               (bkcUdfTimePtr)[i].min, $
                               (bkcUdfTimePtr)[i].sec, $
                               FORMAT="(i3,' ',i2.2,':',i2.2,':',i2.2)")

              for j = 0L, spinsec - 1L do begin
                 printf, filedes, tmestr, $
                                  j, $
                                  raw_cnts[j,i], $
                                  events_n_sect[j,i], $
                                  binn_events[j,i], $
                                  scale_factor[j,i], $
                                  corr_cnts[j, i], $
                                  format='((A12,TR2),(I4,TR8), 5(e12.3,TR2))'

              endfor
         endfor

RETURN
END



;-------------------------------------------------------------------------------

PRO SyncBinnedEvent, l_binned, l_event, binn_data 


   nevents = n_elements ((*l_event.bkcUdfTimePtr))
   nbinns = n_elements ((*l_binned.startUdfTimePtr))
 
   new_data = fltarr (45L, nevents)
   prev_bin_idx = 0L

   for e = 0L, nevents - 1L do begin
      start_event_time = Convert((*l_event.bkcUdfTimePtr)[e])
      for b = prev_bin_idx, nbinns - 1L do begin
          start_binn_time = Convert((*l_binned.startUdfTimePtr)[b])

          if (start_binn_time eq start_event_time) then begin
               new_data [*,e] = binn_data[*,b]
         endif else begin
             if (start_binn_time lt start_event_time) then begin
                ; go to next binn spin
             endif else begin
                prev_bin_idx = b
                goto, NEXT 
             endelse
         endelse
      endfor
NEXT:
   endfor

   binn_data = new_data
RETURN
END


;-------------------------------------------------------------------------------
;  Procedure: BECmpDmp
;
;  Description: Generates a text dump of the background corrected data.  File contains
;        data from binned and event.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  filedes                int          I        output file descriptor
;  l_binned               struct *     I        pointer to binned data structures
;  l_event                struct *     I        pointer to event data structures
;  l_event_setup          struct       I        event processing & plotting parameters
;  l_setup                struct       I        general setup parameters
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;     PLOTDEFS         MAX_PLOTS        int         G         max # plots per page
;                      MAX_WEB_SPINS    int         G         max # spins per plot
;                                                             when using web interface
;                      NO_CNTS          double      G         no counts indicator
;                      NO_DATA          double      G         no data indicator
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------

PRO BECmpDmp, filedes, l_binned, l_event, l_event_setup, l_setup
COMMON PLOTDEFS, MAX_PLOTS, MAX_WEB_SPINS, NO_CNTS, NO_DATA


   nspins = n_elements ((*l_event.bkcUdfTimePtr)[*])
  
   zero_title = 'Spin Angle (-140 - 220)'
   if (l_setup.l_angle_zero eq 'E') then begin
      zero_title = 'Earth Centered Azimuth'
   endif else begin
      if (l_setup.l_angle_zero eq 'S') then begin
         zero_title = 'Sun Centered Azimuth'
      endif else begin
         if (l_setup.l_angle_zero eq 'M') then begin
            zero_title = 'Moon Centered Azimuth'
         endif
      endelse
   endelse

   if (l_event_setup.hydrogen eq 'Y') then begin 
      printf, filedes, '----------------------- Column Descriptions -----------------------------'
      printf, filedes, '                        EVENT DATA '
      printf, filedes, ' '
      printf, filedes, '      Azi             : Azimuthal spin sector (0-44)'
      printf, filedes, '      summed H Events : non scaled summed O events'
      printf, filedes, '      summed H scaled : scaled summed H events'
      printf, filedes, '      summed H binned events : H Binned data events'
      printf, filedes, ' '
      printf, filedes, '-------------------------------------------------------------------------'
      printf, filedes, ' '
      printf, filedes, ' '
   endif else begin
      printf, filedes, '----------------------- Column Descriptions -----------------------------'
      printf, filedes, '                        EVENT DATA '
      printf, filedes, ' '
      printf, filedes, '      Azi             : Azimuthal spin sector (0-44)'
      printf, filedes, '      summed O Events : non scaled summed O events'
      printf, filedes, '      summed O scaled : scaled summed O events'
      printf, filedes, '      summed O binned events : O Binned data events'
      printf, filedes, ' '
      printf, filedes, '-------------------------------------------------------------------------'
      printf, filedes, ' '
      printf, filedes, ' '
   endelse


   n_binn = n_elements ((*l_binned.startUdfTimePtr)[*])
   n_event = l_event.nitems


   if (l_event_setup.hydrogen eq 'Y') then begin
   binn_event = (*l_binned.hydrogen_raw_cnts)
   SyncBinnedEvent, l_binned, l_event, binn_event

   sum_H_events = fltarr (45)
   sum_H_scaled = fltarr (45)
   sum_H_binned_events = fltarr (45)

   Hpk_indx = where (*l_event.H_n_in_peak eq NO_DATA, hpk_cnt)
   if (hpk_cnt gt 0) then begin
      (*l_event.H_n_in_peak)[Hpk_indx] = 0.0
   endif

   Hscl_indx = where (*l_event.scaled_H eq NO_DATA, hscl_cnt)
   if (hscl_cnt gt 0) then begin
      (*l_event.scaled_H)[Hscl_indx] = 0.0
   endif

   be_indx = where (binn_event eq NO_DATA, be_cnt)
   if (be_cnt gt 0) then begin
      binn_event[be_indx] = 0.0
   endif


   for sp = 0L, nspins - 1L do begin
      for sec = 0L, 44L do begin
          sum_H_events[sec] = sum_H_events[sec] + (*l_event.H_n_in_peak)[sec,sp] 
          sum_H_scaled[sec] = sum_H_scaled[sec] + (*l_event.scaled_H)[sec,sp] 
          sum_H_binned_events[sec] = sum_H_binned_events[sec] + binn_event[sec,sp]
      endfor
   endfor


   printf, filedes,'Azi', 'summed H events', 'summed H scaled', 'summed H binned events', $
                    format='((A6, TR7) , 3(A25, TR4))'
   printf, filedes,'---', '---------------', '---------------', '----------------------', $
                    format='((A6, TR7) , 3(A25, TR4))'


   for sec = 0L, 44L do begin
     printf, filedes, sec, $
                      sum_H_events[sec], $
                      sum_H_scaled[sec], $
                      sum_H_binned_events[sec], $
                      format='((I4,TR10), 3(e20.3,TR4))'
   endfor
   endif else begin
      if (l_event_setup.oxygen eq 'Y') then begin
         binn_event = (*l_binned.oxygen_raw_cnts)
         SyncBinnedEvent, l_binned, l_event, binn_event

         sum_O_events = fltarr (45)
         sum_O_scaled = fltarr (45)
         sum_O_binned_events = fltarr (45)

         Opk_indx = where (*l_event.O_n_in_peak eq NO_DATA, opk_cnt)
         if (opk_cnt gt 0) then begin
            (*l_event.O_n_in_peak)[Opk_indx] = 0.0
         endif

         Oscl_indx = where (*l_event.scaled_O eq NO_DATA, oscl_cnt)
         if (oscl_cnt gt 0) then begin
            (*l_event.scaled_O)[Oscl_indx] = 0.0
         endif
      
         be_indx = where (binn_event eq NO_DATA, be_cnt)
         if (be_cnt gt 0) then begin
            binn_event[be_indx] = 0.0
         endif

         for sp = 0L, nspins - 1L do begin
            for sec = 0L, 44L do begin
                sum_O_events[sec] = sum_O_events[sec] + (*l_event.O_n_in_peak)[sec,sp] 
                sum_O_scaled[sec] = sum_O_scaled[sec] + (*l_event.scaled_O)[sec,sp] 
                sum_O_binned_events[sec] = sum_O_binned_events[sec] + binn_event[sec,sp]
            endfor
         endfor

         printf, filedes,'Azi', 'summed O events', 'summed O scaled', 'summed O binned events', $
                          format='((A6, TR7) , 3(A25, TR4))'
         printf, filedes,'---', '---------------', '---------------', '----------------------', $
                          format='((A6, TR7) , 3(A25, TR4))'


         for sec = 0L, 44L do begin
           printf, filedes, sec, $
                            sum_O_events[sec], $
                            sum_O_scaled[sec], $
                            sum_O_binned_events[sec], $
                            format='((I4,TR10), 3(e20.3,TR4))'
         endfor
      endif
   endelse





   printf, filedes, ' '
   printf, filedes, ' '
   printf, filedes, ' '


   printf, filedes, '----------------------- Column Descriptions -----------------------------'
   printf, filedes, '                        EVENT DATA '
   printf, filedes, ' '
   printf, filedes, '      UTC             : Universal Time Coordinate (corrected)'
   printf, filedes, '      Azi             : Azimuthal spin sector (0-44)'
   printf, filedes, '      Raw Counts      : the number of H/O events per sector'
   printf, filedes, '      Events in Sector: events in the sector ( <= 64)'
   printf, filedes, '      Binned Events   : Binned counts '
   printf, filedes, '      Scale Factor    : scale factor  (>64 events/events in sector)'
   printf, filedes, '      Corr/Scl Val    : background corrected scaled counts'
   printf, filedes, ' '
   printf, filedes, '-------------------------------------------------------------------------'
   printf, filedes, ' '
   printf, filedes, ' '
   printf, filedes, ' '

   if (l_event_setup.hydrogen eq 'Y') then begin

         title = 'Hydrogen'
         type  = 'H'
         if (l_event_setup.background eq 'Y') then begin
            title = title + ' Scaled Background Corrected'
         endif

         EventBKHeader, type, filedes, title, zero_title, 'HYDROGEN' 

         raw_cnts  = (*l_event.H_n_in_peak)
         events_n_sect = (*l_event.H_sec_events)
         binn_events = (*l_binned.hydrogen_raw_cnts)
         SyncBinnedEvent, l_binned, l_event, binn_events 
         scale_factor = (*l_event.H_scale_factor)
         ;corr_cnts = (*l_event.corrected_H)
         corr_cnts = (*l_event.scaled_H)

         EventWriteBKFile, nspins, (*l_event.bkcUdfTimePtr), $
         raw_cnts, events_n_sect, binn_events, $
         scale_factor, corr_cnts, l_setup.l_num_spinsec, filedes


         if (l_event_setup.energy0 eq 'Y') then begin
            title = 'Hydrogen Energy 0'
            type  = 'H E0'
            if (l_event_setup.background eq 'Y') then begin
               title = title + ' Scaled Background Corrected'
            endif

            EventBKHeader, type, filedes, title, zero_title, 'HYDROGEN'

            raw_cnts  = (*l_event.hydrogen_raw_e0)

            corr_cnts = (*l_event.corrected_H_e0)
            events_n_sect = (*l_event.H_sec_events_e0)
            binn_events = (*l_binned.hydrogen_raw_cnts_e0)
            SyncBinnedEvent, l_binned, l_event, binn_events 
            scale_factor = (*l_event.H_scale_factor_e0)

            EventWriteBKFile, nspins, (*l_event.bkcUdfTimePtr), $
            raw_cnts, events_n_sect, binn_events, $
            scale_factor, corr_cnts, l_setup.l_num_spinsec, filedes
         endif

         if (l_event_setup.energy1 eq 'Y') then begin
            title = 'Hydrogen Energy 1'
            type  = 'H E1'
            if (l_event_setup.background eq 'Y') then begin
               title = title + ' Scaled Background Corrected'
            endif

            EventBKHeader, type, filedes, title, zero_title, 'HYDROGEN' 

            raw_cnts  = (*l_event.hydrogen_raw_e1)

            corr_cnts = (*l_event.corrected_H_e1)
            events_n_sect = (*l_event.H_sec_events_e1)
            binn_events = (*l_binned.hydrogen_raw_cnts_e1)
            SyncBinnedEvent, l_binned, l_event, binn_events 
            scale_factor = (*l_event.H_scale_factor_e1)

            EventWriteBKFile, nspins, (*l_event.bkcUdfTimePtr), $
            raw_cnts, events_n_sect, binn_events, $
            scale_factor, corr_cnts, l_setup.l_num_spinsec, filedes

         endif

         if (l_event_setup.energy2 eq 'Y') then begin
            title = 'Hydrogen Energy 2'
            type  = 'H E2'
            if (l_event_setup.background eq 'Y') then begin
               title = title + ' Scaled Background Corrected'
            endif

            EventBKHeader, type, filedes, title, zero_title, 'HYDROGEN'

            raw_cnts  = (*l_event.hydrogen_raw_e2)

            corr_cnts = (*l_event.corrected_H_e2)
            events_n_sect = (*l_event.H_sec_events_e2)
            binn_events = (*l_binned.hydrogen_raw_cnts_e2)
            SyncBinnedEvent, l_binned, l_event, binn_events 
            scale_factor = (*l_event.H_scale_factor_e2)

            EventWriteBKFile, nspins, (*l_event.bkcUdfTimePtr), $
            raw_cnts, events_n_sect, binn_events, $
            scale_factor, corr_cnts, l_setup.l_num_spinsec, filedes
         endif
   endif


   if (l_event_setup.oxygen eq 'Y') then begin

         title = 'Oxygen'
         type  = 'O'
         if (l_event_setup.background eq 'Y') then begin
            title = title + ' Scaled Background Corrected'
         endif

         EventBKHeader, type, filedes, title, zero_title, 'OXYGEN' 

         raw_cnts  = (*l_event.O_n_in_peak)
         events_n_sect = (*l_event.O_sec_events)
         binn_events = (*l_binned.oxygen_raw_cnts)
         SyncBinnedEvent, l_binned, l_event, binn_events 
         scale_factor = (*l_event.O_scale_factor)
         corr_cnts = (*l_event.scaled_O)

         EventWriteBKFile, nspins, (*l_event.bkcUdfTimePtr), $
         raw_cnts, events_n_sect, binn_events, $
         scale_factor, corr_cnts, l_setup.l_num_spinsec, filedes



      if (l_event_setup.energy0 eq 'Y') then begin
         title = 'Oxygen Energy0'
         type = 'O E0'
         if (l_event_setup.background eq 'Y') then begin
            title = title + ' Scaled Background Corrected'
         endif

         EventBKHeader, type, filedes, title, zero_title,  'OXYGEN'

         raw_cnts = (*l_event.oxygen_raw_e0)

         if (l_event_setup.background eq 'Y') then begin
            bk_cnts = (*l_event.background_O_e0)
            bk_raw  = (*l_event.background_raw_O_e0)
         endif

         corr_cnts = (*l_event.corrected_O_e0)
         events_n_sect = (*l_event.O_sec_events_e0)
         nevents_n_peak = (*l_event.O_n_in_peak_e0)
         scale_factor = (*l_event.O_scale_factor_e0)
         corr_scl_val = (*l_event.corr_scl_O_e0)

         EventWriteBKFile, nspins, (*l_event.bkcUdfTimePtr), $
         raw_cnts, corr_cnts, events_n_sect, binn_events, $
         scale_factor, corr_scl_val, l_setup.l_num_spinsec, filedes

      endif

      if (l_event_setup.energy1 eq 'Y') then begin
         title = 'Oxygen Energy1'
         type = 'O E1'
         if (l_event_setup.background eq 'Y') then begin
            title = title + ' Scaled Background Corrected'
         endif

         EventBKHeader, type, filedes, title, zero_title, 'OXYGEN' 

         raw_cnts = (*l_event.oxygen_raw_e1)

         if (l_event_setup.background eq 'Y') then begin
            bk_cnts = (*l_event.background_O_e1)
            bk_raw  = (*l_event.background_raw_O_e1)
         endif

         corr_cnts = (*l_event.corrected_O_e1)
         events_n_sect = (*l_event.O_sec_events_e1)
         nevents_n_peak = (*l_event.O_n_in_peak_e1)
         scale_factor = (*l_event.O_scale_factor_e1)
         corr_scl_val = (*l_event.corr_scl_O_e1)

         EventWriteBKFile, nspins, (*l_event.bkcUdfTimePtr), $
         raw_cnts, corr_cnts, events_n_sect, binn_events, $
         scale_factor, corr_scl_val, l_setup.l_num_spinsec, filedes

      endif

      if (l_event_setup.energy2 eq 'Y') then begin
         title = 'Oxygen Energy2'
         type = 'O E2'
         if (l_event_setup.background eq 'Y') then begin
            title = title + ' Scaled Background Corrected'
         endif

         EventBKHeader, type, filedes, title, zero_title, 'OXYGEN'

         raw_cnts = (*l_event.oxygen_raw_e2)

         if (l_event_setup.background eq 'Y') then begin
            bk_cnts = (*l_event.background_O_e2)
            bk_raw  = (*l_event.background_raw_O_e2)
         endif

         corr_cnts = (*l_event.corrected_O_e2)
         events_n_sect = (*l_event.O_sec_events_e2)
         nevents_n_peak = (*l_event.O_n_in_peak_e2)
         scale_factor = (*l_event.O_scale_factor_e2)
         corr_scl_val = (*l_event.corr_scl_O_e2)

         EventWriteBKFile, nspins, (*l_event.bkcUdfTimePtr), $
         raw_cnts, corr_cnts, events_n_sect, binn_events, $
         scale_factor, corr_scl_val, l_setup.l_num_spinsec, filedes

      endif
endif


   printf, filedes, ' '
   printf, filedes, ' '
   printf, filedes, '-----------------   Eliminated Spins ---------------------------------'
   printf, filedes, '-----------------  due to NO_DATA (-9) -------------------------------'
   printf, filedes, ' '
   
   if (l_event_setup.hydrogen eq 'Y') then begin
   if (l_event.num_H_elim_spins gt 0) then begin 
      for i = 0L, l_event.num_H_elim_spins - 1L do begin
         struct_time =  ConvSec2Struct ((*l_event.H_elim_spins_time)[i], l_setup.l_start_dtime[0])
         tmestr = STRING ((struct_time)[i].doy, $
                          (struct_time)[i].hour,$
                          (struct_time)[i].min, $
                          (struct_time)[i].sec, $
                          FORMAT="(i3,' ',i2.2,':',i2.2,':',i2.2)")

         printf, filedes, 'Time : ', tmestr, '      Total number of events in spin: ', $
                (*l_event.H_elim_spins_nevents)[i]
         printf, filedes, ' '
      endfor
      printf, filedes, ' '
      printf, filedes, ' '
      printf, filedes, 'Total number of spins eliminated: ', l_event.num_H_elim_spins
      printf, filedes, ' '
    endif else begin
      printf, filedes, 'No spins were eliminated.'
    endelse
    endif else begin
       if (l_event.num_O_elim_spins gt 0) then begin 
         for i = 0L, l_event.num_O_elim_spins - 1L do begin
         struct_time =  ConvSec2Struct ((*l_event.O_elim_spins_time)[i], l_setup.l_start_dtime[0])
         tmestr = STRING ((struct_time)[i].doy, $
                          (struct_time)[i].hour,$
                          (struct_time)[i].min, $
                          (struct_time)[i].sec, $
                          FORMAT="(i3,' ',i2.2,':',i2.2,':',i2.2)")

         printf, filedes, 'Time : ', tmestr, '      Total number of events in spin: ', $
                (*l_event.O_elim_spins_nevents)[i]
         printf, filedes, ' '
      endfor
      printf, filedes, ' '
      printf, filedes, ' '
      printf, filedes, 'Total number of spins eliminated: ', l_event.num_O_elim_spins
      printf, filedes, ' '
    endif else begin
      printf, filedes, 'No spins were eliminated.'
    endelse
    endelse
 

RETURN
END
