
;-------------------------------------------------------------------------------
;
;  Unit Name : l_ConversionRtns.pro
;
;  Purpose   : Conversions of measurements to physical units.
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               11/00      v1.0        Original Implementation
;  ELH              07/05/01    v1.0        Modified l_Conv2CntRate and
;                                           l_Conv2Log to handle NO_DATA
;  ELH               01/02      v1.0.8      Preliminary flux routine
;
;
;  File Revision Number:   %I%
;  File Last Modified  :   %E%  %T%
;
;-------------------------------------------------------------------------------

;-------------------------------------------------------------------------------
;  Procedure:  l_Conv2CntRate
;
;  Description:
;       Converts the array of values into count rate [Hz].
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  input                  array        I        raw counts
;  output                 array        O        count rates
;  data_type              string       I        data in spin or spin sector format
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;     PLOTDEFS         MAX_PLOTS        int         G         max # plots per page
;                      MAX_WEB_SPINS    int         G         max # spins per plot
;                                                             when using web interface
;                      NO_CNTS          double      G         no counts indicator
;                      NO_DATA          double      G         no data indicator
 
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;
;
;  NOTES:
;   data_type = sector (2.67 seconds)
;   data_type = spin (120 seconds, ~2 minutes, or 2.67 * 45 sectors)
;-------------------------------------------------------------------------------

PRO l_Conv2CntRate, input, output, data_type
COMMON PLOTDEFS, MAX_PLOTS, MAX_WEB_SPINS, NO_CNTS, NO_DATA


  if (STRUPCASE(data_type) eq 'SPIN') then begin
     sec_per_sample = 120.0
  endif else begin
     ;----------------------------
     ; 2.67 seconds, per spin sector
     ;----------------------------
     sec_per_sample = 120.0/45.0
  endelse

  tmp = input
  output = input

  nele = n_elements (tmp)

  nd_indx = where (tmp eq NO_DATA, ndcnt)
  nc_indx = where (tmp eq NO_CNTS, nccnt)

  if (ndcnt eq nele) OR (nccnt eq nele) then return

  if (ndcnt gt 0) OR (nccnt gt 0) then begin
     d_indx = where (tmp ne NO_DATA, dcnt)
     if (dcnt gt 0) then begin
        c_indx = where (tmp[d_indx] ne NO_CNTS, ccnt)
        if (ccnt gt 0) then begin
           output[d_indx[c_indx]] =  tmp[d_indx[c_indx]]/sec_per_sample
        endif else begin
           output[d_indx] =  tmp[d_indx]/sec_per_sample
        endelse
     endif else begin
        c_indx = where (tmp ne NO_CNTS, ccnt)
        if (ccnt gt 0) then begin
           output[c_indx] =  tmp[c_indx]/sec_per_sample
        endif
     endelse 
  endif else begin
     output = input/ sec_per_sample
  endelse 

 
  RETURN

END


;-------------------------------------------------------------------------------
;  Procedure: l_ConvFlux
;
;  Description:  Convert to flux units.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  input                 array         I        input counts
;  output                array         O        output of number flux values
;  species               string        I        hydrogen or oxygen
;  eV                    long          I        energy range
;  dtime                 array         I        data time
;  level                 string        I        instrument state 
;  nominal_eff           double        I        nominal efficiency from tables
;  eff                   double        I        efficiency adjusted based on instrument
;                                               state
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;     PLOTDEFS         MAX_PLOTS        int         G         max # plots per page
;                      MAX_WEB_SPINS    int         G         max # spins per plot
;                                                             when using web interface
;                      NO_CNTS          double      G         no counts indicator
;                      NO_DATA          double      G         no data indicator
;
;     EFFTABLES        O_Eff_table      struct      G         oxygen efficiency values
;                      H_Eff_table      struct      G         hydrogen efficiency values
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------
PRO l_ConvFlux, input, output, species, eV, dtime, level, nominal_eff, eff 
COMMON PLOTDEFS, MAX_PLOTS, MAX_WEB_SPINS, NO_CNTS, NO_DATA
COMMON EFFTABLES, O_Eff_Table, H_Eff_Table

    status = 0L
    status = l_FindNominalEff (O_Eff_Table, H_Eff_Table, species, eV , nominal_eff)
    if (status lt 0) then begin
       print, 'eV outside of range: ', species, eV
       RETURN
    endif
    
    l_CalcEffciency, dtime, level, eff, nominal_eff

    output = input

    nd_indx = where (input eq NO_DATA, ndcnt)
    nc_indx = where (input eq NO_CNTS, nccnt)

    ;--------------------------
    ; 1.0 cm^2, area
    ;--------------------------

    if (ndcnt gt 0 OR nccnt gt 0) then begin
       d_indx = where (input ne NO_DATA, dcnt)
       if (dcnt gt 0) then begin
          c_indx = where (input [d_indx] ne NO_CNTS, ccnt)
          if (ccnt gt 0) then begin
             output[d_indx[c_indx]] = input[d_indx[c_indx]]/eff/1.0D0
          endif else begin
             output[d_indx] = input[d_indx]/eff/1.0D0
          endelse
       endif else begin
          c_indx = where (input ne NO_CNTS, ccnt)
          if (ccnt gt 0) then begin
             output[c_indx] = input[c_indx]/eff/1.0D0
          endif else begin
             output = input/eff/1.0D0
          endelse
       endelse 
    endif else begin
       output = input/eff/1.0D0
    endelse 


RETURN
END

;-------------------------------------------------------------------------------
;  Procedure: l_Conv2Log
;
;  Description:
;      Converts the array of values into log.
;      A zero count is the log of 1
;      NO_CNTS is when the original input data is a 0.0 and 
;      when converting to log zero is a problem;therefore, store as 
;      NO_CNTS
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  input                 array         I        raw counts
;  output                array         O        log counts
;  min_val               long          O        minimum count after log
;  max_val               long          O        maximum count after log
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;     PLOTDEFS         MAX_PLOTS        int         G         max # plots per page
;                      MAX_WEB_SPINS    int         G         max # spins per plot
;                                                             when using web interface
;                      NO_CNTS          double      G         no counts indicator
;                      NO_DATA          double      G         no data indicator
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------


PRO l_Conv2Log, input, output, min_val, max_val
COMMON PLOTDEFS, MAX_PLOTS, MAX_WEB_SPINS, NO_CNTS, NO_DATA

  min_val = 0.0
  max_val = 0.0

  nele = n_elements (input)

  str = 'orig_min = min(input'

  ;----------------------------------------------
  ; determine where no data, zeros, and no counts
  ; are in the array
  ;----------------------------------------------
  d_indx = where (input eq NO_DATA, dcnt)
  c_indx = where (input eq NO_CNTS, ccnt)
  z_indx = where (input eq 0.0, zcnt)

  less_zero = where (input lt 0.0, lzcnt)
  if (lzcnt gt 0) then input[less_zero] = NO_CNTS
  output = input


  if (dcnt eq nele) or (ccnt eq nele) then return
  if (zcnt eq nele) then begin
     output[*] = NO_CNTS
     return
  endif

  nd_indx = where (input ne NO_DATA, ndcnt)
  nc_indx = where (input[nd_indx] ne NO_CNTS, nccnt)
  nz_indx = where (input[nd_indx[nc_indx]] ne 0.0, nzcnt)


  tot = dcnt + ccnt + zcnt

  ;----------------------------------------------
  ; build string to eliminate indexes with the
  ; no data, no counts, and zeros
  ;----------------------------------------------
  if (tot lt nele) then begin

     if (ndcnt gt 0) then str = str + '[nd_indx'
     if (nccnt gt 0) then str = str + '[nc_indx'  
     if (nzcnt gt 0) then str = str + '[nz_indx'  
     if (ndcnt gt 0) then str = str + ']'
     if (nccnt gt 0) then str = str + ']' 
     if (nzcnt gt 0) then str = str + ']'


     str = str + ')'


     result = execute(str) 

     ;print, 'orig_min = ', orig_min

     ;----------------------------------------------
     ; find the original minimum value
     ;----------------------------------------------
     if (orig_min eq 0.0) then begin
        if (dcnt gt 0) then input[d_indx] = 0.0
        if (ccnt gt 0) then input[c_indx] = 0.0
        zero_indx = where (input gt 0.0, zerocnt)
        if (zerocnt gt 0) then begin 
           orig_min = min (input[zero_indx]) 
        endif else begin
           orig_min = min (input) 
        endelse
     endif    

     ;print, 'orig_min = ', orig_min

     ;----------------------------------------------
     ; convert to log values all values that are
     ; greater than the original minimum value
     ;----------------------------------------------
     output = alog10 (input > orig_min)

     min_val = min (output)
     max_val = max (output)


     ;----------------------------------------------
     ; put the no data, no counts, and zeros back
     ; into array
     ;----------------------------------------------

     if (dcnt gt 0) then output[d_indx] = NO_DATA 
     if (ccnt gt 0) then output[c_indx] = NO_CNTS
     if (zcnt gt 0) then output[z_indx] = NO_CNTS

  endif else begin

     output = input
     min_val = min (output)
     max_val = max (output)
     if (dcnt gt 0) then output[d_indx] = NO_DATA 
     if (ccnt gt 0) then output[c_indx] = NO_CNTS
     if (zcnt gt 0) then output[z_indx] = NO_CNTS

  endelse


  RETURN

END

;-------------------------------------------------------------------------------
;  Function/Procedure: l_ConvHkTemp
;
;  Description:
;     Converts the housekeeping temperatures into Celsius.
;
;  Return Value:  type = long
;  Value                         Description
;  -------------------------     ------------------------------
;  conv_temp                     temperature in Celsius
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  hk_temp               long or 
;                        array         I        raw housekeeping temperature
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------

FUNCTION l_ConvHkTemp, hk_temp

   A = 2.5802e-7
   B = -0.0001622D
   C = 0.0344D
   D = -2.952D
   E = 74.406D

   conv_temp = A*hk_temp^4 + B*hk_temp^3 + C*hk_temp^2 + D*hk_temp + e

   RETURN, conv_temp

END


;-------------------------------------------------------------------------------
;  Function:  l_ConvErrVal2Str
;
;  Description:  Determine the error value.
;
;  Return Value:  type = string
;  Value                         Description
;  -------------------------     ------------------------------
;  error_arr                     error string
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  error_val             long          I        error value
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;      ERROR_TYPES     SUCCESS         int         G          successful completion
;                      FAILURE         int         G          unsuccessful completion
;                      WARNING         int         G          warning message
;                      INFO            int         G          informational message
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------

FUNCTION l_ConvErrVal2Str, error_val
COMMON ERROR_TYPES, SUCCESS, FAILURE, WARNING, INFO

  error_arr = strarr ('SUCCESS', 'WARNING', 'INFO')

  if (error_val lt 0) then begin
      return, 'FAILURE'
  endif else begin
      return, error_arr[error_val]
  endelse

END
    
