;============================================================
; FUNCTION: l_BrowseRtns
; Description: Contains all of the coincidence routines 
;
; Developer    Date     Build       Description
; ---------    ----     -----       -------------------------
; E. Lee       5/00      0.0        original implementation
;
;============================================================

;============================================================
; Function:  l_BrowseProc
; Description: read the binned data and sum over all mass,
;              energy, and polar zones
;
;============================================================
PRO l_BrowseProc, bd, mass0_bd, mass1_bd, l_setup, nrec
COMMON GLOBALS, g_report, g_log

  ;------------------------
  ;  binned data structure
  ;------------------------
  ptr=ptr_new()       ; NULL ptr
  bd = {timePtr        : ptr, $    ; begin time for each spin sector
        spinTimePtr    : ptr, $    ; begin time for each spin - 1 time/spin
        filledTimePtr  : ptr, $    ; missing data included
        startUdfTimePtr: ptr, $    ; udf start time
        filledStartUdfTimePtr  : ptr, $    ; missing data included
        stopUdfTimePtr : ptr, $    ; udf start time
        m0RawPtr       : ptr, $    ; m0 telemetry, 3 energy levels
        filledM0RawPtr : ptr, $   ; missing data included, m0 telemetry, 3 energy levels
        m0RawExtraPtr  : ptr, $    ; m0 telemetry, 3 energy levels
        m1RawPtr       : ptr, $    ; m1 telemetry, 3 energy levels
        filledM1RawPtr : ptr, $    ; missing data included, m1 telemetry, 3 energy levels
        m1RawExtraPtr  : ptr, $    ; m1 telemetry, 3 energy levels
        rawPtr         : ptr, $    ; --- collapsed telemetry data
        m0CntPtr       : ptr, $    ; m0 count rate, 3 energy levels
        filledM0CntPtr : ptr, $    ; missing data included, m0 count rate, 3 energy levels
        m0CntExtraPtr  : ptr, $    ; m0 count rate, 3 energy levels
        m1CntPtr       : ptr, $    ; m1 count rate, 3 energy levels
        filledM1CntPtr : ptr, $    ; missing data included, m1 count rate, 3 energy levels
        m1CntExtraPtr  : ptr, $    ; m1 count rate, 3 energy levels
        combCntPtr     : ptr, $    ; m1 count rate, 3 energy levels
        cntPtr         : ptr, $    ; --- collapsed count rate data
        m0FlxPtr       : ptr, $    ; m0 number flux, 3 energy levels
        m1FlxPtr       : ptr, $    ; m0 number flux, 3 energy levels
        flxPtr         : ptr, $    ; collapsed number flux
        vePtr          : ptr, $    ; all masses (valid events)
        filledVePtr    : ptr, $    ; missing data included, all masses (valid events)
        veExtraPtr     : ptr, $    ; all masses (valid events)
        veDmpPtr       : ptr, $    ; all masses (valid events) Dump
        m0Ptr          : ptr, $    ; m0 (H) all energies, polar zone
                                   ; [sector, spin]
        filledM0Ptr    : ptr, $    ; missing data included, m0 (H) all energies, polar zone
                                   ; [sector, spin]
        m0ExtraPtr     : ptr, $    ; m0 (H) all energies, polar zone
                                   ; [sector, spin]
        m0DmpPtr       : ptr, $    ; m0 (H) all energies, polar zone
                                   ; [sector, spin], text dump version
        m0lePtr        : ptr, $    ; m0 (H) lower energy
        filledM0lePtr  : ptr, $    ; missing data included, m0 (H) lower energy
        m0leExtraPtr   : ptr, $    ; m0 (H) lower energy
        m0leDmpPtr     : ptr, $    ; m0 (H) lower energy, text dump version
        m0cePtr        : ptr, $    ; m0 (H) center energy
        filledM0cePtr  : ptr, $    ; missing data included, m0 (H) center energy
        m0ceExtraPtr   : ptr, $    ; m0 (H) center energy
        m0ceDmpPtr     : ptr, $    ; m0 (H) center energy, text dump version
        m0uePtr        : ptr, $    ; m0 (H) upper energy
        filledM0uePtr  : ptr, $    ; missing data included, m0 (H) upper energy
        m0ueExtraPtr   : ptr, $    ; m0 (H) upper energy
        m0ueDmpPtr     : ptr, $    ; m0 (H) upper energy , text dump version
        m1Ptr          : ptr, $    ; m1 (O)  all energies, polar zone
                                   ; [sector, spin]
        filledM1Ptr    : ptr, $    ; missing data included, m1 (O)  all energies, polar zone
                                   ; [sector, spin]
        m1ExtraPtr     : ptr, $    ; m1 (O)  all energies, polar zone
                                   ; [sector, spin]
        m1DmpPtr       : ptr, $    ; m1 (O)  all energies, polar zone
                                   ; [sector, spin], text dump version
        m1lePtr        : ptr, $    ; m1 (O) lower energy
        filledM1lePtr  : ptr, $    ; missing data included, m1 (O) lower energy
        m1leExtraPtr   : ptr, $    ; m1 (O) lower energy
        m1leDmpPtr     : ptr, $    ; m1 (O) lower energy, text dump version
        m1cePtr        : ptr, $    ; m1 (O) center energy
        filledM1cePtr  : ptr, $    ; missing data included, m1 (O) center energy
        m1ceExtraPtr   : ptr, $    ; m1 (O) center energy
        m1ceDmpPtr     : ptr, $    ; m1 (O) center energy, text dump version
        m1uePtr        : ptr, $    ; m1 (O) upper energy
        filledM1uePtr  : ptr, $    ; missing data included, m1 (O) upper energy
        m1ueExtraPtr   : ptr, $    ; m1 (O) upper energy
        m1ueDmpPtr     : ptr, $    ; m1 (O) upper energy, text dump version
        strStUdfTimePtr:ptr,  $    ; steering start UDF time
        filledStrStUdfTimePtr:ptr,  $    ; missing data included, steering start UDF time
        strSpUdfTimePtr:ptr,  $    ; steering stop UDF time
        filledStrSpUdfTimePtr:ptr,  $    ; missing data included, steering stop UDF time
        steervPtr      : ptr, $    ; steering voltage levels / spin
        filledSteervPtr: ptr, $    ; missing data included, steering voltage levels / spin
       bkcUdfTimePtr  : ptr, $
       hydrogen_raw_cnts      : ptr, $
       hydrogen_raw_cnts_e0   : ptr, $
       hydrogen_raw_cnts_e1   : ptr, $
       hydrogen_raw_cnts_e2   : ptr, $
       corrected_hydrogen     : ptr, $
       corrected_hydrogen_e0  : ptr, $
       corrected_hydrogen_e1  : ptr, $
       corrected_hydrogen_e2  : ptr, $
       corr_scl_H             : ptr, $
       corr_scl_H_e0          : ptr, $
       corr_scl_H_e1          : ptr, $
       corr_scl_H_e2          : ptr, $
       oxygen_raw_cnts        : ptr, $
       oxygen_raw_cnts_e0     : ptr, $
       oxygen_raw_cnts_e1     : ptr, $
       oxygen_raw_cnts_e2     : ptr, $
       corrected_oxygen       : ptr, $
       corrected_oxygen_e0    : ptr, $
       corrected_oxygen_e1    : ptr, $
       corrected_oxygen_e2    : ptr, $
       corr_scl_O             : ptr, $
       corr_scl_O_e0          : ptr, $
       corr_scl_O_e1          : ptr, $
       corr_scl_O_e2          : ptr  $
}

;print,'created the binn data pointer struct'

  ;--------------------
  ;  get num image rec
  ;--------------------
  inst = 'LENA'
  expr = 'LENASCI'
  vinst = 'IMLIMAGE'
  nrec = 0L
  data_source = ' '

  nrec = l_UdfGetNumRec(l_setup.l_start_dtime, $
                        l_setup.l_stop_dtime,  $
                        inst, expr, vinst, data_source)

  ;----------------------------------------------------
  ; determine the number of spins 
  ;----------------------------------------------------
  nspins = nrec/l_setup.l_num_spinsec
  l_setup.l_num_spin = nspins


  ;----------------------------------------------------
  ; read the udf data
  ;----------------------------------------------------
  if (nspins eq 1) then begin
     l_BinnedReadUdfDataBySpin, l_setup, nrec, bd
  endif else begin
     start_time = l_setup.l_start_dtime
     stop_time = l_setup.l_stop_dtime
;     status = l_BinnedReadUdfDataByTime (l_setup, nrec, bd)
     status = l_BinnedReadUdfDataByTime (l_setup, nrec, bd, $
                                         start_time, stop_time)
  endelse

  l_BinnedConv2CountRate, l_setup, l_nplot_setup, rec, bd

  ;----------------------------------------------------
  ; sum binned data over energy, mass, and polar zone
  ; to obtain the sum value for each spin sector
  ; these are the coincidences  
  ;----------------------------------------------------
  spin_period = 120./45.


  m0cnt     = (*bd.m0CntPtr)
  m1cnt     = (*bd.m1CntPtr)

  ;----------------------------------------------------
  ;  generating plots on a spin basis
  ;----------------------------------------------------
  if (l_setup.l_singles_sum eq 1) then begin

     mass0_bd  = fltarr (l_setup.l_num_polarzn, $
                         l_setup.l_num_spinsec, nspins)

     for sp = 0, nspins-1 do begin
       for sec = 0, 44 do begin
         for p = 0, 11 do begin
           for e = 0, 2 do begin
              mass0_bd[p,sec,sp] = mass0_bd[p,sec,sp] + m0cnt[e,p,sec,sp]
           endfor
         endfor
       endfor
     endfor

    mass1_bd  = fltarr (l_setup.l_num_polarzn, $
                        l_setup.l_num_spinsec, nspins)
    for sp = 0, nspins-1 do begin
       for sec = 0, 44 do begin
         for p = 0, 11 do begin
           for e = 0, 2 do begin
               mass1_bd[p,sec,sp] = mass1_bd[p,sec,sp] + m1cnt[e,p,sec,sp]
           endfor
         endfor
       endfor
    endfor

  endif else begin

     ;----------------------------------------------------
     ; plots are all spins summed over spins and energy to
     ; get one 45x12 image
     ;----------------------------------------------------
     nmass0_bd  = fltarr (l_setup.l_num_energylev, $
                          l_setup.l_num_polarzn, $
                          l_setup.l_num_spinsec)


     for sp = 0, nspins-1 do begin
       for sec = 0, 44 do begin
         for p = 0, 11 do begin
           for e = 0, 2 do begin
               nmass0_bd[e,p,sec] = nmass0_bd[e,p,sec] + m0cnt[e,p,sec,sp]
           endfor
         endfor
       endfor
     endfor

     mass0_bd  = fltarr (l_setup.l_num_polarzn, l_setup.l_num_spinsec)

     for e = 0, 2 do begin
       for p = 0, 11 do begin
        for sec = 0, 44 do begin
              mass0_bd[p,sec] = mass0_bd[p,sec] + nmass0_bd[e,p,sec]
           endfor
        endfor
     endfor

     nmass1_bd  = fltarr (l_setup.l_num_energylev, $
                          l_setup.l_num_polarzn, $
                          l_setup.l_num_spinsec)


     for sp = 0, nspins-1 do begin
       for sec = 0, 44 do begin
         for p = 0, 11 do begin
           for e = 0, 2 do begin
               nmass1_bd[e,p,sec] = nmass1_bd[e,p,sec] + m1cnt[e,p,sec,sp]
             endfor
         endfor
       endfor
     endfor


     mass1_bd  = fltarr (l_setup.l_num_polarzn, l_setup.l_num_spinsec)

     for e = 0, 2 do begin
       for p = 0, 11 do begin
        for sec = 0, 44 do begin
              mass1_bd[p,sec] = mass1_bd[p,sec] + nmass1_bd[e,p,sec]
           endfor
        endfor
     endfor

  endelse

RETURN
END

;============================================================
; FUNCTION: l_BrowsePlot
; Description:  Plot the coincidence data
;------------------------------------------------------------------------------
; Richard L. West      2001/02/08
; -----
; Modified so that when 1 spin time period is selected the same OA information
; is used if l_singles_sum is set to either 1 or 2
;============================================================
   
PRO l_BrowsePlot, nrec, btme, mass0_bd, mass1_bd, l_oa, l_setup, $
                      l_pltpos, l_rgb_table

;
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;
common rlw_plot_parm, char_size, char_size_1, $
                      window_width, window_height, $
                      x_inches, y_inches

;
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;

cross_chan_calib = fltarr(12)
cross_chan_calib(0)  = 0.00
cross_chan_calib(1)  = 0.81
cross_chan_calib(2)  = 0.88
cross_chan_calib(3)  = 0.94
cross_chan_calib(4)  = 0.98
cross_chan_calib(5)  = 1.00
cross_chan_calib(6)  = 1.00
cross_chan_calib(7)  = 0.98
cross_chan_calib(8)  = 0.94
cross_chan_calib(9)  = 0.88
cross_chan_calib(10) = 0.81
cross_chan_calib(11) = 0.00

;
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;

first0 = 1
first1 = 1

nspins = nrec / l_setup.l_num_spinsec

nadir = 1                  ; 1=earthward view, -1=anti-earthward view
spin_mode = -1              ; 1=cort wheel, -1=opposite cart wheel

re = 6371.0     ; earth radius in km

; polar angle arrays
nsector = 45
npolar = 12
pix_size = 8                               ; pixel size in deg

center1 = (1 - npolar) * pix_size / 2            ; center of the first pixel
polar_angle = intarr(npolar)
for i = 0,npolar-1 do begin
   polar_angle(i) = center1 + i * pix_size     ; centers of pixels
endfor

spin_size = 360 / nsector
if (spin_size ne pix_size) then print,' Error: non-square pixel '

gci_satellite_position=fltarr(3)
gci_satellite_velocity=fltarr(3)
geo_satellite_position=fltarr(3)
gsm_satellite_position=fltarr(3)
gsm_satellite_velocity=fltarr(3)
sm_satellite_position=fltarr(3)
sm_satellite_velocity=fltarr(3)

sm_sat_pos_next = fltarr(3)
sm_sat_pos_curr = fltarr(3)
geo_N_sm = fltarr(3)

;
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;

start = 0
stop  = 44 

if (l_setup.l_dumps eq 'Y') then begin
   mo = 0
   dy = 0
   l_ConvDOY2YMD, l_setup.l_start_dtime[1], l_setup.l_start_dtime[0], mo, dy
   year = string(strcompress(l_setup.l_start_dtime[0], /REMOVE_ALL))
   yr = strmid(year, 2, 2)
   l_setup.l_filename = strcompress(('l' + $
                        string(format='(A2)', yr) +  $
                        string(format='(I2.2)',mo) + $
                        string(format='(I2.2)',dy) + $
                        string(format='(I2.2)',l_setup.l_start_dtime[2]) + $
                        string(format='(I2.2)',l_setup.l_start_dtime[3]) + $
                        string(format='(I2.2)',l_setup.l_stop_dtime[2])  + $
                        string(format='(I2.2)',l_setup.l_stop_dtime[3]))  + $
                        '.image', /REMOVE_ALL)

   dump_filename = l_setup.l_prod_dst + '/' + l_setup.l_filename + $
                   l_setup.l_process_id + '.dmp'
   openw, fp1, dump_filename, /get_lun
endif

if (l_setup.l_singles_sum eq 1) then begin
for nsp = 0, nspins - 1 do begin

   oaindx = nsp
   plot_title_start = ConvToUtc(btme[start], 2)
   plot_title_stop = ConvToUtc(btme[stop], 2)
   l_ConvSec2DOYTime, btme[start], start_doy, hr, min, sec
   l_ConvSec2DOYTime, btme[stop], stop_doy, hr, min, sec
   start_year = l_setup.l_start_date[0]
   start_year_2 = start_year - (fix(start_year / 100) * 100)
   stop_year = l_setup.l_stop_date[0]
   if (stop_doy lt start_doy) then start_year = start_year - 1
   l_ConvDOY2YMD, fix(start_doy), start_year, start_mon, start_dom
   l_ConvDOY2YMD, fix(stop_doy), stop_year, stop_mon, stop_dom
   start = stop + 1
   stop =  stop + 45

   time_array = ConvSec2Arr ((*l_oa.tme_ptr)[oaindx], $
                               l_setup.l_start_dtime[0])

   RECALC, time_array[0], time_array[1], time_array[2], $
           time_array[3], time_array[4]
 
;
;------------------------------------------------------------------------------
; position in re : gci is the same as gei : convert gei to geo 
;------------------------------------------------------------------------------
;

   xgei = ((*l_oa.gci_x_pos_ptr)[oaindx]) / re
   ygei = ((*l_oa.gci_y_pos_ptr)[oaindx]) / re
   zgei = ((*l_oa.gci_z_pos_ptr)[oaindx]) / re

   sat_re = sqrt(xgei*xgei + ygei*ygei + zgei*zgei)
   lshell = (*l_oa.lsh_ptr)[oaindx]
   invlat = acos(sqrt(1.0/lshell)) * 57.295779

   ; gci satellite position and velocity values 

   gci_satellite_position[0] = xgei
   gci_satellite_position[1] = ygei
   gci_satellite_position[2] = zgei

   gci_satellite_velocity[0] = ((*l_oa.gci_x_vel_ptr)[oaindx]) / re 
   gci_satellite_velocity[1] = ((*l_oa.gci_y_vel_ptr)[oaindx]) / re 
   gci_satellite_velocity[2] = ((*l_oa.gci_z_vel_ptr)[oaindx]) / re 

   ; calculate the gci satellite spin 

   gci_sat_spin = fltarr(3)
   gci_sat_spin[0] = gci_satellite_position[1] * gci_satellite_velocity[2] - $
                     gci_satellite_position[2] * gci_satellite_velocity[1]
   gci_sat_spin[1] = gci_satellite_position[2] * gci_satellite_velocity[0] - $
                     gci_satellite_position[0] * gci_satellite_velocity[2]
   gci_sat_spin[2] = gci_satellite_position[0] * gci_satellite_velocity[1] - $
                     gci_satellite_position[1] * gci_satellite_velocity[0]
   gci_sat_spin[*] = spin_mode * gci_sat_spin[*]

   geigeo, xgei, ygei, zgei, xgeo, ygeo, zgeo, 1, time_array

   geo_satellite_position[0] = xgeo
   geo_satellite_position[1] = ygeo
   geo_satellite_position[2] = zgeo

   GEOGSM, 0.0, 0.0, 1.0 ,xgsm, ygsm, zgsm, 1
   SMGSM, xsm, ysm, zsm, xgsm, ygsm, zgsm, -1

   mlt = ((atan(ysm, xsm) * 12.0) / !pi ) + 12.0
   mlat = (asin(zsm/sat_re) * 180.0) / !pi

   geo_N_sm[0] = xsm
   geo_N_sm[1] = ysm
   geo_N_sm[2] = zsm

   gsm_satellite_position[0] = ((*l_oa.gsm_x_pos_ptr)[oaindx]) / re
   gsm_satellite_position[1] = ((*l_oa.gsm_y_pos_ptr)[oaindx]) / re
   gsm_satellite_position[2] = ((*l_oa.gsm_z_pos_ptr)[oaindx]) / re

   ; velocity in re/sec

   gsm_satellite_velocity[0] = ((*l_oa.gsm_x_vel_ptr)[oaindx]) / re 
   gsm_satellite_velocity[1] = ((*l_oa.gsm_y_vel_ptr)[oaindx]) / re 
   gsm_satellite_velocity[2] = ((*l_oa.gsm_z_vel_ptr)[oaindx]) / re 

   ; convert the gsm position coordinates to sm

   SMGSM, xsm_pos, ysm_pos, zsm_pos, $
          (*l_oa.gsm_x_pos_ptr)[oaindx] / re, $
          (*l_oa.gsm_y_pos_ptr)[oaindx] / re, $
          (*l_oa.gsm_z_pos_ptr)[oaindx] / re, $
          -1

   rsm = sqrt(xsm_pos*xsm_pos + ysm_pos*ysm_pos + zsm_pos*zsm_pos)

   sm_sat_pos_curr[0] = xsm_pos
   sm_sat_pos_curr[1] = ysm_pos
   sm_sat_pos_curr[2] = zsm_pos

   sm_satellite_position[*] = sm_sat_pos_curr[*]

   SMGSM, xsm_pos, ysm_pos, zsm_pos, $
          (*l_oa.gsm_x_pos_ptr)[oaindx+1]/re, $
          (*l_oa.gsm_y_pos_ptr)[oaindx+1]/re, $
          (*l_oa.gsm_z_pos_ptr)[oaindx+1]/re, $
          -1

   sm_sat_pos_next[0] = xsm_pos
   sm_sat_pos_next[1] = ysm_pos
   sm_sat_pos_next[2] = zsm_pos

   ; convert the gsm velocity coordinates to sm

   SMGSM, xsm_vel, ysm_vel, zsm_vel, $
          (*l_oa.gsm_x_vel_ptr)[oaindx] / re, $
          (*l_oa.gsm_y_vel_ptr)[oaindx] / re, $
          (*l_oa.gsm_z_vel_ptr)[oaindx] / re, $
          -1

   sm_satellite_velocity[0] = xsm_vel 
   sm_satellite_velocity[1] = ysm_vel 
   sm_satellite_velocity[2] = zsm_vel 

   ; find the direction of spin axis in sm coordinates

   sm_satellite_spin = fltarr(3)

   xgci = gci_sat_spin[0]
   ygci = gci_sat_spin[1]
   zgci = gci_sat_spin[2]
   epoch = 0
   geigse, xgci, ygci, zgci, xgse, ygse, zgse, 1, epoch
   gsmgse, xgsm, ygsm, zgsm, xgse, ygse, zgse, -1
   smgsm, xsm, ysm, zsm, xgsm, ygsm, zgsm, -1
   sm_satellite_spin[0] = xsm
   sm_satellite_spin[1] = ysm
   sm_satellite_spin[2] = zsm
;   sm_satellite_spin[*] = spin_mode*sm_satellite_spin[*]

;
;------------------------------------------------------------------------------
; this is hard coded for now, but there is a routine which does this
;------------------------------------------------------------------------------

   image0_dmp=fltarr(npolar,45)
   image0_dmp[*,0:10]  = mass0_bd[*,6:16, nsp]
   image0_dmp[*,11]    = mass0_bd[*,17, nsp]
   image0_dmp[*,12:38] = mass0_bd[*,18:44, nsp]
   image0_dmp[*,39:44] = mass0_bd[*,0:5, nsp]
   image1_dmp=fltarr(npolar,45)
   image1_dmp[*,0:10]  = mass1_bd[*,6:16, nsp]
   image1_dmp[*,11]    = mass1_bd[*,17, nsp]
   image1_dmp[*,12:38] = mass1_bd[*,18:44, nsp]
   image1_dmp[*,39:44] = mass1_bd[*,0:5, nsp]
   if (l_setup.l_output_type eq 1) then begin
      naz_bins = 44
      image0=fltarr(npolar,45)
      image0[*,0:10]  = mass0_bd[*,6:16, nsp]
      image0[*,11]    = mass0_bd[*,17, nsp]
      image0[*,12:38] = mass0_bd[*,18:44, nsp]
      image0[*,39:44] = mass0_bd[*,0:5, nsp]
      image1=fltarr(npolar,45)
      image1[*,0:10]  = mass1_bd[*,6:16, nsp]
      image1[*,11]    = mass1_bd[*,17, nsp]
      image1[*,12:38] = mass1_bd[*,18:44, nsp]
      image1[*,39:44] = mass1_bd[*,0:5, nsp]
   endif else if(l_setup.l_output_type eq 2) then begin
      naz_bins = 44
      image0=fltarr(npolar,45)
      image0[*,0:5]  = mass0_bd[*,39:44,nsp]
      image0[*,6:44] = mass0_bd[*,0:38,nsp]
      image1=fltarr(npolar,45)
      image1[*,0:5]  = mass1_bd[*,39:44,nsp]
      image1[*,6:44] = mass1_bd[*,0:38,nsp]
   endif else if(l_setup.l_output_type eq 3) then begin
      naz_bins = 11
      image0=fltarr(npolar,12)
      image0[*,0:11] = mass0_bd[*,11:22, nsp]
      image1=fltarr(npolar,12)
      image1[*,0:11] = mass1_bd[*,11:22, nsp]
   endif
   for np = 0,11 do begin
      for na = 0,naz_bins do begin
         if (cross_chan_calib(np) eq 0.0) then begin
            image0(np,na) = -99.0
            image1(np,na) = -99.0
         endif else begin
            image0(np,na) = image0(np,na) / cross_chan_calib(np)
            image1(np,na) = image1(np,na) / cross_chan_calib(np)
         endelse
      endfor
   endfor

;
;------------------------------------------------------------------------------
; build the plotting file name
;------------------------------------------------------------------------------
;

   date_start = strcompress(( $
      string(format='(I4.4)', start_year) + '/' + $
      string(format='(I2.2)', start_mon)  + '/' + $
      string(format='(I2.2)', start_dom)),  /REMOVE_ALL)
   date_stop = strcompress(( $
      string(format='(I4.4)', stop_year) + '/' + $
      string(format='(I2.2)', stop_mon)  + '/' + $
      string(format='(I2.2)', stop_dom)),  /REMOVE_ALL)
;   date_start = strcompress(( $
;      string(format='(I4.4)', l_setup.l_start_date[0]) + '/' + $
;      string(format='(I2.2)', l_setup.l_start_date[1]) + '/' + $
;      string(format='(I2.2)', l_setup.l_start_date[2])), /REMOVE_ALL)
;   date_stop = strcompress(( $
;      string(format='(I4.4)', l_setup.l_stop_date[0]) + '/' + $
;      string(format='(I2.2)', l_setup.l_stop_date[1]) + '/' + $
;      string(format='(I2.2)', l_setup.l_stop_date[2])), /REMOVE_ALL)
   hr_str = strmid(plot_title_start, 4, 2)
   mn_str = strmid(plot_title_start, 7, 2)
   start_time_str = hr_str + mn_str
   hr_str = strmid(plot_title_stop, 4, 2)
   mn_str = strmid(plot_title_stop, 7, 2)
   stop_time_str = hr_str + mn_str
   if (l_setup.l_output_type eq 1) then begin
      l_setup.l_filedesc = 'image_3full'
      l_setup.l_filename = strcompress('l' + $
            string(format='(I2.2)', start_year_2) + $
            string(format='(I2.2)', start_mon) + $
            string(format='(I2.2)', start_dom) + $
            start_time_str + stop_time_str + '_3fl', $
            /REMOVE_ALL)
   endif else if (l_setup.l_output_type eq 2) then begin
      l_setup.l_filedesc = 'image_3cnst'
      l_setup.l_filename = strcompress('l' + $
            string(format='(I2.2)', start_year_2) + $
            string(format='(I2.2)', start_mon) + $
            string(format='(I2.2)', start_dom) + $
            start_time_str + stop_time_str + '_3cn', $
            /REMOVE_ALL)
   endif else begin
      l_setup.l_filedesc = 'image_3sq'
      l_setup.l_filename = strcompress('l' + $
            string(format='(I2.2)', start_year_2) + $
            string(format='(I2.2)', start_mon) + $
            string(format='(I2.2)', start_dom) + $
            start_time_str + stop_time_str + '_3sq', $
            /REMOVE_ALL)
   endelse

;
;------------------------------------------------------------------------------
; loop through and do each plot type
;------------------------------------------------------------------------------
;

   nplots = 0
   if (strupcase(l_setup.l_via_web) eq 'N') then $
      do_x_1 = 0 else $
      do_x_1 = 1
   if (l_setup.l_gen_movie eq 'Y') then $
      do_x_2 = 1 else $
      do_x_2 = 2
   for output_plot_type = do_x_1,do_x_2 do begin
      if (output_plot_type eq 0) then begin
         l_SetUpDisplay, 'X', l_pltpos, l_rgb_table, $
                         l_setup.l_color_tbl, colorbar, nplots
      endif else if (output_plot_type eq 1) then begin
         l_SetUpDisplay, 'Z', l_pltpos, l_rgb_table, $
                         l_setup.l_color_tbl, colorbar, nplots
      endif else begin
         ps_filename = l_setup.l_prod_dst + '/' + l_setup.l_filename
         ps_filename = ps_filename + l_setup.l_process_id
         l_pltpos.filename = ps_filename + '_1sp.ps'
         l_SetUpDisplay, 'P', l_pltpos, l_rgb_table, $
                         l_setup.l_color_tbl, colorbar, nplots
      endelse

;     
;------------------------------------------------------------------------------
; plot the 3 images and the global overlay and the color bar
;------------------------------------------------------------------------------
;

   if (l_setup.l_gen_movie eq 'Y') then begin
      plot_ImageMovie, image0, image1, npolar, polar_angle, l_setup, $
                       geo_satellite_position, $
                       sm_satellite_position, sm_satellite_spin, $
                       nadir, sm_satellite_velocity, $
                       gci_satellite_position, gci_sat_spin, geo_N_sm
   endif else begin
      plot_ENAflux, image0, image1, npolar, polar_angle, l_setup, $
                    geo_satellite_position, $
                    sm_satellite_position, sm_satellite_spin, $
                    nadir, sm_satellite_velocity, $
                    gci_satellite_position, gci_sat_spin, geo_N_sm
   endelse

;
;------------------------------------------------------------------------------
; put on the top left annotation
;------------------------------------------------------------------------------
;

   plot_start_time = strmid(plot_title_start, 4, 8)
   plot_stop_time = strmid(plot_title_stop, 4, 8)
   if (l_setup.l_gen_movie eq 'Y') then begin
      if (l_setup.l_output_type eq 1) then begin
         xpos=0.06
         ypos=0.96
      endif else begin
         xpos=0.11
         ypos=0.90
      endelse
      label = date_start + ' ' + plot_start_time + ' -> ' + plot_stop_time
      xyouts, xpos, ypos, /normal, label, charsize=char_size_1, align = 0.0
   endif else begin
      xpos=0.10
      ypos=0.97
      xyouts, 0.0, 0.0, '!5', /normal

      ydelta=0.022

      label = 'IMAGE LENA'
      xyouts, xpos, ypos, /normal, label, charsize=char_size_1, align = 0.0
      label = 'start time: ' + date_start + ' ' + plot_start_time
      xyouts, xpos, ypos-ydelta, /normal, label, $
              charsize=char_size_1, align = 0.0
      label = 'stop time: ' + date_stop + ' ' + plot_stop_time
      xyouts, xpos, ypos-2*ydelta, /normal, label, $
              charsize=char_size_1, align = 0.0
      xyouts, xpos, ypos-3*ydelta, /normal, '1 spin', $
              charsize=char_size_1, align = 0.0
   endelse

;
;------------------------------------------------------------------------------
; put on the orbital annotation
;------------------------------------------------------------------------------
;

   if (l_setup.l_gen_movie eq 'Y') then begin
   endif else begin
      xpos = 0.50
      ypos = 0.97

      label = string (format = '(''Re = '', f7.4)', sat_re)
      xyouts, xpos, ypos, /normal, label, charsize=char_size_1, align = 0.0
      label = string (format = '(''Lshell = '', f7.3)', lshell)
      xyouts, xpos, ypos-ydelta, /normal, label, charsize=char_size_1, $
              align = 0.0
      label = string (format = '(''mlt = '', f4.1,''hr'')', mlt)
      xyouts, xpos, ypos-2*ydelta, /normal, label, charsize=char_size_1, $
              align = 0.0
      label = string (format = '(''mlat = '', f7.3,''!m%!5'')', mlat)
      xyouts, xpos, ypos-3*ydelta, /normal, label, charsize=char_size_1, $
              align = 0.0
      label = string (format = '(''invlat = '', f6.3,''!m%!5'')', invlat)
      xyouts, xpos, ypos-4*ydelta, /normal, label, charsize=char_size_1, $
              align = 0.0
   endelse

;
;------------------------------------------------------------------------------
; put on version number and date
;------------------------------------------------------------------------------
;

   if (l_setup.l_gen_movie eq 'Y') then begin
   endif else begin
      xpos = 0.80
      ypos = 0.10

      label = 'Version 1.0.1'
      xyouts, xpos, ypos, /normal, label, charsize=char_size, align = 0.0

      time_string = systime(0)
      time = strmid(time_string, 11, 8)
      dom = strmid(time_string, 8, 2)
      mon = strmid(time_string, 4, 3)
      year = strmid(time_string, 20, 4)
      label = mon + '-' + dom + '-' + year
      xyouts, xpos, ypos-ydelta, /normal, label, charsize=char_size, align=0.0
      label = time
      xyouts, xpos, ypos-(2.0*ydelta), /normal, label, charsize=char_size, $
              align=0.0
   endelse

;
;------------------------------------------------------------------------------
; write the gif file or close the postscript file
;------------------------------------------------------------------------------
;

     if (output_plot_type eq 1) then begin
        plot_image = TVRD()
        gif_filename = l_setup.l_prod_dst + '/' + l_setup.l_filename
        gif_filename = gif_filename + l_setup.l_process_id
        gif_filename = gif_filename + '_1sp.gif'
        write_gif, gif_filename, plot_image, $
                   l_rgb_table.red(0:225), l_rgb_table.green(0:225), $
                   l_rgb_table.blue(0:225)
     endif else if (output_plot_type eq 2) then begin
        device, /close_file
     endif

;
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;

   endfor     ; for output_plot_type = 0,2 do begin

;
;------------------------------------------------------------------------------
; output to the dump file
;------------------------------------------------------------------------------
;

   if (l_setup.l_dumps eq 'Y') then begin
      num_spins = 1
      l_BrowsePrintDump, fp1, dump_filename, l_pltpos, num_spins, $
                         date_start, start_doy, plot_start_time, $
                         date_stop, stop_doy, plot_stop_time, $
                         sat_re, sat_re, lshell, lshell, $
                         mlt, mlt, mlat, mlat, invlat, invlat, $
                         image0_dmp, image1_dmp
      if (nsp le (nspins-2)) then begin
         printf, fp1, format = '("")'
         printf, fp1, format = '("")'
      endif
   endif

;
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;

endfor     ; for nsp = 0, nspins - 1 do begin
endif else begin     ; if (l_setup.l_singles_sum eq 2) then begin

   start = 0
   stop  = (nspins * 44) + (nspins - 1) 

   if (nspins eq 1) then begin
      oaindx = 0
   endif else begin
      oaindx = ceil(n_elements((*l_oa.gsm_x_pos_ptr)[*])/2)
   endelse
   plot_title_start = ConvToUtc(btme[start], 2)
   plot_title_stop = ConvToUtc(btme[stop], 2)
   l_ConvSec2DOYTime, btme[start], start_doy, hr, min, sec
   l_ConvSec2DOYTime, btme[stop], stop_doy, hr, min, sec
   start_year = l_setup.l_start_date[0]
   start_year_2 = start_year - (fix(start_year / 100) * 100)
   stop_year = l_setup.l_stop_date[0]
   if (stop_doy lt start_doy) then start_year = start_year - 1
   l_ConvDOY2YMD, fix(start_doy), start_year, start_mon, start_dom
   l_ConvDOY2YMD, fix(stop_doy), stop_year, stop_mon, stop_dom
   start = stop + 1
   stop =  stop + 45

   time_array = ConvSec2Arr ((*l_oa.tme_ptr)[oaindx], $
                               l_setup.l_start_dtime[0])

   RECALC, time_array[0], time_array[1], time_array[2], $
           time_array[3], time_array[4]

   ; position in re : gci is the same as gei : convert gei to geo

   xgei = ((*l_oa.gci_x_pos_ptr)[oaindx]) / re
   ygei = ((*l_oa.gci_y_pos_ptr)[oaindx]) / re
   zgei = ((*l_oa.gci_z_pos_ptr)[oaindx]) / re

   ; gci satellite position and velocity values

   gci_satellite_position[0] = xgei
   gci_satellite_position[1] = ygei
   gci_satellite_position[2] = zgei

   gci_satellite_velocity[0] = ((*l_oa.gci_x_vel_ptr)[oaindx]) / re
   gci_satellite_velocity[1] = ((*l_oa.gci_y_vel_ptr)[oaindx]) / re
   gci_satellite_velocity[2] = ((*l_oa.gci_z_vel_ptr)[oaindx]) / re

   ; calculate the gci satellite spin

   gci_sat_spin = fltarr(3)
   gci_sat_spin[0] = gci_satellite_position[1]*gci_satellite_velocity[2]- $
                     gci_satellite_position[2]*gci_satellite_velocity[1]
   gci_sat_spin[1] = gci_satellite_position[2]*gci_satellite_velocity[0]- $
                     gci_satellite_position[0]*gci_satellite_velocity[2]
   gci_sat_spin[2] = gci_satellite_position[0]*gci_satellite_velocity[1]- $
                     gci_satellite_position[1]*gci_satellite_velocity[0]
   gci_sat_spin[*] = spin_mode*gci_sat_spin[*]

   geigeo, xgei, ygei, zgei, xgeo, ygeo, zgeo, 1, time_array

   geo_satellite_position[0] = xgeo
   geo_satellite_position[1] = ygeo
   geo_satellite_position[2] = zgeo

   GEOGSM, 0.0 , 0.0 ,1.0 ,xgsm, ygsm, zgsm, 1
   SMGSM, xsm, ysm, zsm, xgsm, ygsm, zgsm, -1

   geo_N_sm[0] = xsm
   geo_N_sm[1] = ysm
   geo_N_sm[2] = zsm

  gsm_satellite_position[0] = ((*l_oa.gsm_x_pos_ptr)[oaindx]) / re
  gsm_satellite_position[1] = ((*l_oa.gsm_y_pos_ptr)[oaindx]) / re
  gsm_satellite_position[2] = ((*l_oa.gsm_z_pos_ptr)[oaindx]) / re

   ; velocity in re/sec

   gsm_satellite_velocity[0] = ((*l_oa.gsm_x_vel_ptr)[oaindx]) / re
   gsm_satellite_velocity[1] = ((*l_oa.gsm_y_vel_ptr)[oaindx]) / re
   gsm_satellite_velocity[2] = ((*l_oa.gsm_z_vel_ptr)[oaindx]) / re

   ; convert the gsm position coordinates to sm

   SMGSM, xsm_pos, ysm_pos, zsm_pos, $
          (*l_oa.gsm_x_pos_ptr)[oaindx] / re, $
          (*l_oa.gsm_y_pos_ptr)[oaindx] / re, $
          (*l_oa.gsm_z_pos_ptr)[oaindx] / re, $
          -1

   rsm = sqrt(xsm_pos*xsm_pos + ysm_pos*ysm_pos + zsm_pos*zsm_pos)

   sm_sat_pos_curr[0] = xsm_pos
   sm_sat_pos_curr[1] = ysm_pos
   sm_sat_pos_curr[2] = zsm_pos

   sm_satellite_position[*] = sm_sat_pos_curr[*]

   SMGSM, xsm_pos, ysm_pos, zsm_pos, $
          (*l_oa.gsm_x_pos_ptr)[oaindx+1] / re, $
          (*l_oa.gsm_y_pos_ptr)[oaindx+1] / re, $
          (*l_oa.gsm_z_pos_ptr)[oaindx+1] / re, $
          -1

   sm_sat_pos_next[0] = xsm_pos
   sm_sat_pos_next[1] = ysm_pos
   sm_sat_pos_next[2] = zsm_pos

   ; convert the gsm velocity coordinates to sm

   SMGSM, xsm_vel, ysm_vel, zsm_vel, $
          (*l_oa.gsm_x_vel_ptr)[oaindx] / re, $
          (*l_oa.gsm_y_vel_ptr)[oaindx] / re, $
          (*l_oa.gsm_z_vel_ptr)[oaindx] / re, $
          -1

   sm_satellite_velocity[0] = xsm_vel
   sm_satellite_velocity[1] = ysm_vel
   sm_satellite_velocity[2] = zsm_vel


   ;Find the direction of spin axis in sm coordinates

   sm_satellite_spin = fltarr(3)

   xgci = gci_sat_spin[0]
   ygci = gci_sat_spin[1]
   zgci = gci_sat_spin[2]
   epoch = 0
   geigse, xgci, ygci, zgci, xgse, ygse, zgse, 1, epoch
   gsmgse, xgsm, ygsm, zgsm, xgse, ygse, zgse, -1
   smgsm, xsm, ysm, zsm, xgsm, ygsm, zgsm, -1
   sm_satellite_spin[0] = xsm
   sm_satellite_spin[1] = ysm
   sm_satellite_spin[2] = zsm
;   sm_satellite_spin[*] = spin_mode*sm_satellite_spin[*]

;
;------------------------------------------------------------------------------
; this is hard coded for now, but there is a routine which does this
;------------------------------------------------------------------------------

   image0_dmp=fltarr(npolar,45)
   image0_dmp[*,0:10]  = mass0_bd[*,6:16]
   image0_dmp[*,11]    = mass0_bd[*,17]
   image0_dmp[*,12:38] = mass0_bd[*,18:44]
   image0_dmp[*,39:44] = mass0_bd[*,0:5]
   image1_dmp=fltarr(npolar,45)
   image1_dmp[*,0:10]  = mass1_bd[*,6:16]
   image1_dmp[*,11]    = mass1_bd[*,17]
   image1_dmp[*,12:38] = mass1_bd[*,18:44]
   image1_dmp[*,39:44] = mass1_bd[*,0:5]
   if (l_setup.l_output_type eq 1) then begin
      naz_bins = 44
      image0=fltarr(npolar,45)
      image0[*,0:10]  = mass0_bd[*,6:16]
      image0[*,11]    = mass0_bd[*,17]
      image0[*,12:38] = mass0_bd[*,18:44]
      image0[*,39:44] = mass0_bd[*,0:5]
      image1=fltarr(npolar,45)
      image1[*,0:10]  = mass1_bd[*,6:16]
      image1[*,11]    = mass1_bd[*,17]
      image1[*,12:38] = mass1_bd[*,18:44]
      image1[*,39:44] = mass1_bd[*,0:5]
   endif else if(l_setup.l_output_type eq 2) then begin
      naz_bins = 44
      image0=fltarr(npolar,45)
      image0[*,0:5]  = mass0_bd[*,39:44]
      image0[*,6:44] = mass0_bd[*,0:38]
      image1=fltarr(npolar,45)
      image1[*,0:5]  = mass1_bd[*,39:44]
      image1[*,6:44] = mass1_bd[*,0:38]
   endif else if (l_setup.l_output_type eq 3) then begin
      naz_bins = 11
      image0=fltarr(npolar,12)
      image0[*,0:11] = mass0_bd[*,11:22]
      image1=fltarr(npolar,12)
      image1[*,0:11] = mass1_bd[*,11:22]
   endif
   if (l_setup.l_avg_sum eq 1) then begin
      image0 = image0 / (2.7 * float(nspins))
      image1 = image1 / (2.7 * float(nspins))
   endif
   for np = 0,11 do begin
      for na = 0,naz_bins do begin
         if (cross_chan_calib(np) eq 0.0) then begin
            image0(np,na) = -99.0
            image1(np,na) = -99.0
         endif else begin
            image0(np,na) = image0(np,na) / cross_chan_calib(np)
            image1(np,na) = image1(np,na) / cross_chan_calib(np)
         endelse
      endfor
   endfor

;
;------------------------------------------------------------------------------
; build the plotting file name
;------------------------------------------------------------------------------
;

   date_start = strcompress(( $
      string(format='(I4.4)', start_year) + '/' + $
      string(format='(I2.2)', start_mon)  + '/' + $
      string(format='(I2.2)', start_dom)),  /REMOVE_ALL)
   date_stop = strcompress(( $
      string(format='(I4.4)', stop_year) + '/' + $
      string(format='(I2.2)', stop_mon)  + '/' + $
      string(format='(I2.2)', stop_dom)),  /REMOVE_ALL)
;   date_start = strcompress(( $
;      string(format='(I4.4)', l_setup.l_start_date[0]) + '/' + $
;      string(format='(I2.2)', l_setup.l_start_date[1]) + '/' + $
;      string(format='(I2.2)', l_setup.l_start_date[2])), /REMOVE_ALL)
;   date_stop = strcompress(( $
;      string(format='(I4.4)', l_setup.l_stop_date[0]) + '/' + $
;      string(format='(I2.2)', l_setup.l_stop_date[1]) + '/' + $
;      string(format='(I2.2)', l_setup.l_stop_date[2])), /REMOVE_ALL)
   hr_str = strmid(plot_title_start, 4, 2)
   mn_str = strmid(plot_title_start, 7, 2)
   start_time_str = hr_str + mn_str
   hr_str = strmid(plot_title_stop, 4, 2)
   mn_str = strmid(plot_title_stop, 7, 2)
   stop_time_str = hr_str + mn_str
   if (l_setup.l_output_type eq 1) then begin
      l_setup.l_filedesc = 'image_3full'
      l_setup.l_filename = strcompress('l' + $
            string(format='(I2.2)', start_year_2) + $
            string(format='(I2.2)', start_mon) + $
            string(format='(I2.2)', start_dom) + $
            start_time_str + stop_time_str + '_3fl', $
            /REMOVE_ALL)
   endif else if (l_setup.l_output_type eq 2) then begin
      l_setup.l_filedesc = 'image_3cnst'
      l_setup.l_filename = strcompress('l' + $
            string(format='(I2.2)', start_year_2) + $
            string(format='(I2.2)', start_mon) + $
            string(format='(I2.2)', start_dom) + $
            start_time_str + stop_time_str + '_3cn', $
            /REMOVE_ALL)
   endif else begin
      l_setup.l_filedesc = 'image_3sq'
      l_setup.l_filename = strcompress('l' + $
            string(format='(I2.2)', start_year_2) + $
            string(format='(I2.2)', start_mon) + $
            string(format='(I2.2)', start_dom) + $
            start_time_str + stop_time_str + '_3sq', $
            /REMOVE_ALL)
   endelse

   if (l_setup.l_avg_sum eq 1) then begin
      spinavgsum = string (format = '(i2.2,''spa'')', nspins)
;      if (nspins lt 10) then begin
;         spinavgsum = string (format = '(i2.2,''spa'')', nspins)
;      endif else begin
;         spinavgsum = string (format = '(i2.2,''spa'')', nspins)
;      endelse
   endif else begin
      spinavgsum = string (format = '(i2.2,''sps'')', nspins)
;      if (nspins lt 10) then begin
;         spinavgsum = string (format = '(i2.2,''sps'')', nspins)
;      endif else begin
;         spinavgsum = string (format = '(i2.2,''sps'')', nspins)
;      endelse
   endelse

;
;------------------------------------------------------------------------------
; loop through and do each plot type
;------------------------------------------------------------------------------
;

   nplots = 0
   if (strupcase(l_setup.l_via_web) eq 'N') then $
      do_x_1 = 0 else $
      do_x_1 = 1
   if (l_setup.l_gen_movie eq 'Y') then $
      do_x_2 = 1 else $
      do_x_2 = 2
   for output_plot_type = do_x_1,do_x_2 do begin
      if (output_plot_type eq 0) then begin
         l_SetUpDisplay, 'X', l_pltpos, l_rgb_table, $
                         l_setup.l_color_tbl, colorbar, nplots
      endif else if (output_plot_type eq 1) then begin
         l_SetUpDisplay, 'Z', l_pltpos, l_rgb_table, $
                         l_setup.l_color_tbl, colorbar, nplots
      endif else begin
         ps_filename = l_setup.l_prod_dst + '/' + l_setup.l_filename
         ps_filename = ps_filename + l_setup.l_process_id
         l_pltpos.filename = ps_filename + '_' + spinavgsum + '.ps'
         l_SetUpDisplay, 'P', l_pltpos, l_rgb_table, $
                         l_setup.l_color_tbl, colorbar, nplots
      endelse
;     
;------------------------------------------------------------------------------
; plot the 3 images and the global overlay and the color bar
;------------------------------------------------------------------------------
;

      if (l_setup.l_gen_movie eq 'Y') then begin
         plot_ImageMovie, image0, image1, npolar, polar_angle, l_setup, $
                          geo_satellite_position, $
                          sm_satellite_position, sm_satellite_spin, $
                          nadir, sm_satellite_velocity, $
                          gci_satellite_position, gci_sat_spin, geo_N_sm
      endif else begin
         plot_ENAflux, image0, image1, npolar, polar_angle, l_setup, $
                       geo_satellite_position, $
                       sm_satellite_position, sm_satellite_spin, $
                       nadir, sm_satellite_velocity, $
                       gci_satellite_position, gci_sat_spin, geo_N_sm
      endelse

;------------------------------------------------------------------------------
; put on the top label
;------------------------------------------------------------------------------
;

   plot_start_time = strmid(plot_title_start, 4, 8)
   plot_stop_time = strmid(plot_title_stop, 4, 8)
   if (l_setup.l_gen_movie eq 'Y') then begin
      if (l_setup.l_output_type eq 1) then begin
         xpos=0.06
         ypos=0.96
      endif else begin
         xpos=0.11
         ypos=0.90
      endelse
      label = date_start + ' ' + plot_start_time + ' -> ' + plot_stop_time
      xyouts, xpos, ypos, /normal, label, charsize=char_size_1, align = 0.0
   endif else begin
      xpos=0.10
      ypos=0.97
      xyouts, 0.0, 0.0, '!5', /normal

      ydelta=0.022

      label = 'IMAGE LENA'
      xyouts, xpos, ypos, /normal, label, charsize=char_size_1, align = 0.0
      label = 'start time: ' + date_start + ' ' + plot_start_time
      xyouts, xpos, ypos-ydelta, /normal, label, $
              charsize=char_size_1, align = 0.0
      label = 'stop time: ' + date_stop + ' ' + plot_stop_time
      xyouts, xpos, ypos-2*ydelta, /normal, label, $
              charsize=char_size_1, align = 0.0
      if (l_setup.l_avg_sum eq 1) then begin
         if (nspins lt 10) then begin
            label = string (format = '(i1,'' spin average'')', nspins)
         endif else begin
            label = string (format = '(i2,'' spin average'')', nspins)
         endelse
      endif else begin
         if (nspins lt 10) then begin
            label = string (format = '(i1,'' spin sum'')', nspins)
         endif else begin
            label = string (format = '(i2,'' spin sum'')', nspins)
         endelse
      endelse
      xyouts, xpos, ypos-3*ydelta, /normal, label, $
              charsize=char_size_1, align = 0.0
   endelse

;
;------------------------------------------------------------------------------
; put on the orbital annotation
;------------------------------------------------------------------------------
;

   xgei = ((*l_oa.gci_x_pos_ptr)[0]) / re
   ygei = ((*l_oa.gci_y_pos_ptr)[0]) / re
   zgei = ((*l_oa.gci_z_pos_ptr)[0]) / re

   sat_re_1 = sqrt(xgei*xgei + ygei*ygei + zgei*zgei)
   lshell_1 = (*l_oa.lsh_ptr)[0]
   invlat_1 = acos(sqrt(1.0/lshell_1)) * 57.295779

   xgei = ((*l_oa.gci_x_pos_ptr)[nspins-1]) / re
   ygei = ((*l_oa.gci_y_pos_ptr)[nspins-1]) / re
   zgei = ((*l_oa.gci_z_pos_ptr)[nspins-1]) / re

   sat_re_2 = sqrt(xgei*xgei + ygei*ygei + zgei*zgei)
   lshell_2 = (*l_oa.lsh_ptr)[nspins-1]
   invlat_2 = acos(sqrt(1.0/lshell_2)) * 57.295779

   time_array = ConvSec2Arr ((*l_oa.tme_ptr)[0], $
                               l_setup.l_start_dtime[0])

   RECALC, time_array[0], time_array[1], time_array[2], $
           time_array[3], time_array[4]
 
   GEOGSM, 0.0, 0.0, 1.0 ,xgsm, ygsm, zgsm, 1
   SMGSM, xsm, ysm, zsm, xgsm, ygsm, zgsm, -1

   mlt_1 = ((atan(ysm, xsm) * 12.0) / !pi ) + 12.0
   mlat_1 = (asin(zsm/sat_re_1) * 180.0) / !pi


   time_array = ConvSec2Arr ((*l_oa.tme_ptr)[nspins-1], $
                               l_setup.l_start_dtime[0])

   RECALC, time_array[0], time_array[1], time_array[2], $
           time_array[3], time_array[4]
 
   GEOGSM, 0.0, 0.0, 1.0 ,xgsm, ygsm, zgsm, 1
   SMGSM, xsm, ysm, zsm, xgsm, ygsm, zgsm, -1

   mlt_2 = ((atan(ysm, xsm) * 12.0) / !pi ) + 12.0
   mlat_2 = (asin(zsm/sat_re_2) * 180.0) / !pi

   if (l_setup.l_gen_movie eq 'Y') then begin
   endif else begin
      xpos = 0.50
      ypos = 0.97

      label = string (format = '(''Re = '', f7.4, '' -> '', f7.4)', $
                      sat_re_1, sat_re_2)
      xyouts, xpos, ypos, /normal, label, charsize=char_size_1, align = 0.0
      label = string (format = '(''Lshell = '', f7.3, '' -> '', f7.3)', $
                      lshell_1, lshell_2)
      xyouts, xpos, ypos-ydelta, /normal, label, charsize=char_size_1, $
              align = 0.0
      label = string (format = '(''mlt = '', f4.1, '' -> '', f4.1, ''hr'')', $
              mlt_1, mlt_2)
      xyouts, xpos, ypos-2*ydelta, /normal, label, charsize=char_size_1, $
              align = 0.0
      label = string ( $
              format = '(''mlat = '', f7.3, '' -> '', f7.3,''!m%!5'')', $
              mlat_1, mlat_2)
      xyouts, xpos, ypos-3*ydelta, /normal, label, charsize=char_size_1, $
              align = 0.0
      label = string ( $
              format = '(''invlat = '', f6.3, '' -> '', f6.3,''!m%!5'')', $
              invlat_1, invlat_2)
      xyouts, xpos, ypos-4*ydelta, /normal, label, charsize=char_size_1, $
              align = 0.0
   endelse

;
;------------------------------------------------------------------------------
; put on version number and date
;------------------------------------------------------------------------------
;

   if (l_setup.l_gen_movie eq 'Y') then begin
   endif else begin
      xpos = 0.80
      ypos = 0.10

      label = 'Version 1.0.1'
      xyouts, xpos, ypos, /normal, label, charsize=char_size, align = 0.0

      time_string = systime(0)
      time = strmid(time_string, 11, 8)
      dom = strmid(time_string, 8, 2)
      mon = strmid(time_string, 4, 3)
      year = strmid(time_string, 20, 4)
      label = mon + '-' + dom + '-' + year
      xyouts, xpos, ypos-ydelta, /normal, label, charsize=char_size, align=0.0
      label = time
      xyouts, xpos, ypos-(2.0*ydelta), /normal, label, charsize=char_size, $
              align=0.0
   endelse

;
;------------------------------------------------------------------------------
; write the gif file or close the postscript file
;------------------------------------------------------------------------------
;

        if (output_plot_type eq 1) then begin
           plot_image = TVRD()
           gif_filename = l_setup.l_prod_dst + '/' + l_setup.l_filename
           gif_filename = gif_filename + l_setup.l_process_id
           gif_filename = gif_filename + '_' + spinavgsum + '.gif'
           write_gif, gif_filename, plot_image, $
                      l_rgb_table.red(0:225), l_rgb_table.green(0:225), $
                      l_rgb_table.blue(0:225)
        endif else if (output_plot_type eq 2) then begin
           device, /close_file
        endif

;
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;

   endfor     ; for output_plot_type = 0,2 do begin
;
;------------------------------------------------------------------------------
; output to the dump file
;------------------------------------------------------------------------------
;

if (l_setup.l_dumps eq 'Y') then begin
   l_BrowsePrintDump, fp1, dump_filename, l_pltpos, nspins, $
                      date_start, start_doy, plot_start_time, $
                      date_stop, stop_doy, plot_stop_time, $
                      sat_re_1, sat_re_2, lshell_1, lshell_2, $
                      mlt_1, mlt_2, mlat_1, mlat_2, invlat_1, invlat_2, $
                      image0_dmp, image1_dmp
endif

;
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;

endelse
if (l_setup.l_dumps eq 'Y') then close, fp1

;
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;

RETURN
END

;============================================================
;============================================================
PRO l_BrowsePrintDump, fp1, dump_filename, l_pltpos, num_spins, $
                       date_start, start_doy, plot_start_time, $
                       date_stop, stop_doy, plot_stop_time, $
                       sat_re_1, sat_re_2, lshell_1, lshell_2, $
                       mlt_1, mlt_2, mlat_1, mlat_2, invlat_1, invlat_2, $
                       image0_dmp, image1_dmp
;
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;
no_cnts = 0.0
no_data = -99.0
;
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;
printf, fp1, '============================================================'
header = 'IMAGE/LENA Dump File: Images Text Dump'
printf, fp1, header
btime = string (format='(a10," (",i3.3,") ",a8)', date_start, start_doy, $
                plot_start_time)
printf, fp1, 'User specified data Start Time: ' + btime
etime = string (format='(a10," (",i3.3,") ",a8)', date_stop, stop_doy, $
                plot_stop_time)
printf, fp1, 'User specified data Stop Time:  ' + etime
temp = string (format = '(i5)', num_spins)
printf, fp1, 'Number of spins = ' + temp
printf, fp1, ' '
sw_ver= 'Software Version: ' + l_pltpos.sw_ver
printf, fp1, sw_ver
tme = 'Text Dump Creation Date: ' + systime()
printf, fp1, tme
if (l_pltpos.data_source eq 'D') then begin
   ds = 'UDF Definitive'
endif else begin
   ds = 'UDF Quicklook'
endelse
printf, fp1, 'Data Source: ' + ds
printf, fp1, 'Filename : ' + dump_filename
printf, fp1, '============================================================'
;
;------------------------------------------------------------------------------
; print the orbital information
;------------------------------------------------------------------------------
;
printf, fp1, format = '("Re         = ", f7.4, " -> ", f7.4)', $
                      sat_re_1, sat_re_2
printf, fp1, format = '("Lshell     = ", f7.3, " -> ", f7.3)', $
                      lshell_1, lshell_2
printf, fp1, format = '("mlt        = ", f4.1, " -> ", f4.1, " hr")', $
                      mlt_1, mlt_2
printf, fp1, format = '("mlat       = ", f7.3, " -> ", f7.3, " deg")', $
                      mlat_1, mlat_2
printf, fp1, format = '("invlat     = ", f6.3, " -> ", f6.3, " deg")', $
                      invlat_1, invlat_2
printf, fp1, '============================================================'
;
;------------------------------------------------------------------------------
; print mass data
;------------------------------------------------------------------------------
;
printf, fp1, ' '
printf, fp1, 'No Data Value   = ', no_data
printf, fp1, 'No Counts Value = ', no_cnts
printf, fp1, ' '
printf, fp1, ' '
printf, fp1, '****************** Hydrogen ******************'
printf, fp1, 'Values = Raw Data'
printf, fp1, ' '
printf, fp1, format='(a22, 4x, (A40))', 'Spin Sectors (degrees)', $
             'Polar Sectors 0->11 (-48 to +48 degrees)'
printf, fp1, format='(a22, 4x, (A40))', '----------------------', $
             '----------------------------------------'
spin_angle = -90
for ispin = 0,44 do begin
   printf, fp1, format = '(i4, ":", 1x, 12(e11.3))', $
                spin_angle, image0_dmp(*, ispin)
   spin_angle = spin_angle + 8
endfor
printf, fp1, ' '
printf, fp1, '******************* Oxygen *******************'
printf, fp1, 'Values = Raw Data'
printf, fp1, ' '
printf, fp1, format='(a22, 4x, (A39))', 'Spin Sectors (degrees)', $
             'Polar Sectors 0->11 (-48 to 48 degrees)'
printf, fp1, format='(a22, 4x, (A39))', '----------------------', $
             '---------------------------------------'
spin_angle = -90
for ispin = 0,44 do begin
   printf, fp1, format = '(i4, ":", 1x, 12(e11.3))', $
                spin_angle, image1_dmp(*, ispin)
   spin_angle = spin_angle + 8
endfor
printf, fp1, ' '
printf, fp1, '***************** All Masses *****************'
printf, fp1, 'Values = Raw Data'
printf, fp1, ' '
printf, fp1, format='(a22, 4x, (A39))', 'Spin Sectors (degrees)', $
             'Polar Sectors 0->11 (-48 to 48 degrees)'
printf, fp1, format='(a22, 4x, (A39))', '----------------------', $
             '---------------------------------------'
spin_angle = -90
image2_dmp = image0_dmp + image1_dmp
lt_zero = where (image2_dmp lt 0.0, lt_zero_counts)
if (lt_zero_counts gt 0) then image2_dmp(lt_zero) = no_data
for ispin = 0,44 do begin
   printf, fp1, format = '(i4, ":", 1x, 12(e11.3))', $
                spin_angle, image2_dmp(*, ispin)
   spin_angle = spin_angle + 8
endfor
;
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;

RETURN
END
