
;-------------------------------------------------------------------------------
;
;  Unit Name : l_BinnedRtns.pro
;
;  Purpose   : Provides calling procedures to read and process the binned/image
;              data.
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;   ELH                               v1.0            Original implementation
;   ELH               04/01           v1.0            Added H/O, all energies, 
;                                                     background (ROI0) 
;   ELH               08/01           v1.0.7          Handle UDF interpolated times 
;   RLW               08/01           v1.0.7          Removed MAX_WEB_SPINS checks
;   ELH              08/31/01         v1.0.7          Added reading of extra spins
;                                                     to do adjusting S,M,E
;   ELH              09/26/01         v1.0.7          Added l_DetermineSunDir
;   ELH               01/02           v1.0.8          Added parm in l_HndlMissData to
;                                                     handle rolling data based on
;                                                     spin angle input
;                                                     Added SyncBinnedData
;   ELH              01/27/2004       v1.0.11         Changed add extra spins to end
;                                                     time from 4 to 2 minutes
;   ELH               05/04           v1.0.11         Check for valid ptr in 
;                                                     l_Binned_AdjExData
;   ELH              05/27/04         v1.0.11         Replaced use of multiple for loops
;                                                     to use IDL total () in
;                                                     l_Binned_AdjExData
;
;
;
;  File Revision Number:   %I%
;  File Last Modified  :   %E%  %T%
;
;-------------------------------------------------------------------------------

;-------------------------------------------------------------------------------
;  Procedure: SyncBinnedData
;  Description: Sync the binned data to whatever data packet
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_binned              struct *      I        binned data ptr structure
;  l_X                   struct *      I        X data ptr struct
;  n_binn                long          I        # of binned records
;  n_X                   long          I        # of X records
;  startUdfTime          struct[]      O        synced binned udf times 
;  stopUdfTime           struct[]      O        synced binned udf times 
;  steering              float[]       O        synced steering data
;  time                  float[]       O        synced binned time in seconds
;  sptime                double[]      I        spin times
;  m0                    float[]       I        hydrogen data
;  m1                    float[]       I        oxygen data
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               01/02           v1.0.8           Original implementation
;-------------------------------------------------------------------------------

PRO SyncBinnedData, l_binned, l_X, n_binn, n_X, startUdfTime, stopUdfTime, $
steering, time, sptime, m0, m1, count_rate

   ;--------------------------------------------------------------------
   ; sync the event data to the BINNED SUM COUNTS and STEERING
   ;   . steering from housekeeping is retrieved synced to binned data
   ;   . match each item in the event packet to each time
   ;     in the binned time
   ;
   ; NOTE: UDF time of binned data are not 2 minute spins; therefore,
   ;       must use the start bin time of the current record as the
   ;       start time and the start bin time of the next record as the
   ;       stop bin time
   ;--------------------------------------------------------------------

   m0data = (*l_binned.m0RawPtr)
   m1data = (*l_binned.m1RawPtr)

   last_ii = 0L
   last_jj = 0L
   indx    = 0L
   status  = 0L
   ii      = 0L
   jj      = 0L
   st_indx = 0L
   sp_indx = 44L

   for jj = 0L, n_binn - 1L do begin

       start_binn_time = Convert((*l_binned.startUdfTimePtr)[jj])
       stop_binn_time = Convert((*l_binned.stopUdfTimePtr)[jj])

       if (jj lt n_binn - 1L) then begin
          next_start_binn_time = Convert((*l_binned.startUdfTimePtr)[jj+1])
       endif else begin
          next_start_binn_time = Convert((*l_binned.stopUdfTimePtr)[jj])
       endelse

       for ii = last_ii, n_X - 1L do begin
           start_time = Convert((*l_X.startUdfTimePtr)[ii])
           stop_time = Convert((*l_X.stopUdfTimePtr)[ii])

           ;--------------------------------------------------------------------
           ; determine if event data falls within this spin
           ;--------------------------------------------------------------------
           if (start_time ge start_binn_time) and $
              (start_time le next_start_binn_time) then begin

              ;--------------------------------------------------------------------
              ; if the time is equal to the boundary end time
              ;--------------------------------------------------------------------
              if (start_time eq next_start_binn_time) then begin
                 ;--------------------------------------------------------------------
                 ; check that the data belongs in the current bin spin, if it does
                 ; then extract the data, else need to go to the next spin
                 ;--------------------------------------------------------------------
                 if (start_time ge start_binn_time) and $
                    (start_time le stop_binn_time) then begin
                       time[st_indx:sp_indx] = (*l_X.timePtr)[ii*45L:(ii*45L+44L)]
                       sptime[ii] = (*l_binned.spinTimePtr)[jj] 
                       st_indx = sp_indx + 1L
                       sp_indx = sp_indx + 45L
                       startUdfTime[ii] = (*l_X.startUdfTimePtr)[ii]
                       stopUdfTime[ii] = (*l_X.stopUdfTimePtr)[ii]
                       m0[*,*,*,ii] = m0data[*,*,*,jj]
                       m1[*,*,*,ii] = m1data[*,*,*,jj]
                       steering[ii] = (*l_binned.steervPtr)[jj]
                       last_ii = ii
                   endif else begin
                       last_ii = ii
                       goto, NEXT_PKT2
                   endelse
              endif else begin
                   time[st_indx:sp_indx] = (*l_X.timePtr)[ii*45L:(ii*45L+44L)]
                   sptime[ii] = (*l_binned.spinTimePtr)[jj] 
                   st_indx = sp_indx + 1L
                   sp_indx = sp_indx + 45L
                   startUdfTime[ii] = (*l_X.startUdfTimePtr)[ii]
                   stopUdfTime[ii] = (*l_X.stopUdfTimePtr)[ii]
                   m0[*,*,*,ii] = m0data[*,*,*,jj]
                   m1[*,*,*,ii] = m1data[*,*,*,jj]
                   steering[ii] = (*l_binned.steervPtr)[jj]
                   last_ii = ii
              endelse
            endif else begin
              ;--------------------------------------------------------------------
              ; does not fall within that particular spin period, check if there
              ; is a data gap and if the current data belongs in the next
              ; spin, if status = 0, no data gap and data belongs in the next spin
              ; then get the next packet
              ;--------------------------------------------------------------------
              status = l_DataGap (next_start_binn_time, stop_binn_time, $
                                  start_time)
              if (status eq 0L) then begin
                 last_ii = ii
                 goto, NEXT_PKT2
              endif
            endelse
       endfor
NEXT_PKT2:
   endfor


RETURN
END



;-------------------------------------------------------------------------------
;  Procedure:  l_BinnedPlot
;  Description:
;        Determines which user combinations chosen and plots the data.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_binned_setup        struct        I        binned data user specific options
;                                               read from the setup file
;  l_binned              struct *      I        pointers to the different binned
;                                               data
;  l_pltpos              struct        I        general plot parameters
;  nplots                long          I        total # of plots to be plotted
;  windx                 long[]        I        window index
;  pages                 long          I        total number of pages needed to hold 
;                                               all plots
;  plotted               long          I        total number of plots currently plotted
;  l_setup               struct        I        general setup parameters (date, time, etc)
;  l_oa                  struct *      I        OA parameters
;  l_rgb_table           struct        I        color table parameters selected by user
;  colorbar              float[]       I        color bar values
;  l_hskp                struct *      I        pointers housekeeeping parameters
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;     PLOTDEFS         MAX_PLOTS        int         G         max # plots per page
;                      MAX_WEB_SPINS    int         G         max # spins per plot
;                                                             when using web interface
;                      NO_CNTS          double      G         no counts indicator
;                      NO_DATA          double      G         no data indicator
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11          Original implementation
;-------------------------------------------------------------------------------

PRO l_BinnedPlot, l_binned_setup, l_binned, l_pltpos, nplots, windx, pages, plotted, $
l_setup, l_oa, l_rgb_table, colorbar, l_hskp
COMMON PLOTDEFS, MAX_PLOTS, MAX_WEB_SPINS, NO_CNTS, NO_DATA


   extra_data = make_array (l_setup.l_num_spinsec, 2L, /FLOAT, Value = NO_DATA)

   ;---------------------------------------------------
   ; determine which trace to center to zero degree
   ; spin angle 
   ;---------------------------------------------------
   ;---------------------------------------------------
   ; calculate the sun pulse sectors
   ;---------------------------------------------------
   if (l_setup.l_angle_zero eq 'S') then begin
      l_CalcSunPulse, l_oa, lena_sector, 'S'
      center_sector = fix(lena_sector)
      cs = lena_sector/8.
      if NOT (ptr_valid(l_oa.lena_sun_sectPtr)) then begin
         ptr = ptr_new (cs, /NO_COPY)
         l_oa.lena_sun_sectPtr = ptr
      endif
      ytitle_frame = '!5Sun Cntrd Azi'
   endif else begin
      ;---------------------------------------------------
      ; calculate the moon sectors 
      ;---------------------------------------------------
      if (l_setup.l_angle_zero eq 'M') then begin
         l_CalcSunPulse, l_oa, lena_sector, 'M'
         center_sector = fix(lena_sector)
         cs = center_sector/8.
         if NOT (ptr_valid(l_oa.lena_moon_sectPtr)) then begin
            ptr = ptr_new (cs, /NO_COPY)
            l_oa.lena_moon_sectPtr = ptr
         endif
         ytitle_frame = '!5Moon Cntrd Azi'
      endif else begin
         ;---------------------------------------------------
         ; calculate the earth sectors 
         ;---------------------------------------------------
         if (l_setup.l_angle_zero eq 'E') then begin
            center_sector = l_setup.l_earth_center
            ytitle_frame = '!5Earth Cntrd Azi'
         endif else begin
             ytitle_frame = '!5Spin Angle'
             center_sector = 0 
         endelse
      endelse
   endelse

   ;---------------------------------------------------
   ; store the sun direction data in the OA struct
   ;---------------------------------------------------
   if (l_setup.l_sun_direction eq 'Y') then begin
      l_CalcSunPulse, l_oa, lena_sector, 'S'
      sun_direction = lena_sector/8
      if NOT (ptr_valid(l_oa.lena_sun_sectPtr)) then begin
         ptr = ptr_new (sun_direction, /NO_COPY)
         l_oa.lena_sun_sectPtr = ptr
      endif
   endif


   ;---------------------------------------------------
   ; determine which masses and energies to plot
   ;---------------------------------------------------
    if (l_binned_setup.merge eq 'Y') then begin
      title = 'Binned'

      ;---------------------------------------------------
      ; plot hydrogen and oxygen
      ;---------------------------------------------------
      if (l_binned_setup.hydrogen eq 'Y') AND (l_binned_setup.oxygen eq 'Y') then begin 
         ;---------------------------------------------------
         ; plot hydrogen and oxygen and all energies
         ;---------------------------------------------------
         if (l_binned_setup.energy0 eq 'Y') AND (l_binned_setup.energy1 eq 'Y') AND $
            (l_binned_setup.energy2 eq 'Y') then begin
               title1 = ' '        
               plot_data =  (*l_binned.vePtr)
               extra_data = (*l_binned.veExtraPtr)
         endif else begin
            ;---------------------------------------------------
            ; plot hydrogen and oxygen and energies 0 and 1
            ;---------------------------------------------------
            if (l_binned_setup.energy0 eq 'Y') AND (l_binned_setup.energy1 eq 'Y') then begin
               title1 = 'E0,E1'
               plot_data = (*l_binned.m0lePtr) + (*l_binned.m1lePtr) + $
                           (*l_binned.m0cePtr) + (*l_binned.m1cePtr)
               extra_data[*,0] = (*l_binned.m0leExtraPtr)[*,0] + (*l_binned.m1leExtraPtr)[*,0] + $
                            (*l_binned.m0ceExtraPtr)[*,0] + (*l_binned.m1ceExtraPtr)[*,0]
               extra_data[*,1] = (*l_binned.m0leExtraPtr)[*,1] + (*l_binned.m1leExtraPtr)[*,1] + $
                            (*l_binned.m0ceExtraPtr)[*,1] + (*l_binned.m1ceExtraPtr)[*,1]
            endif else begin
               ;---------------------------------------------------
               ; plot hydrogen and oxygen and energies 0 and 2
               ;---------------------------------------------------
               if (l_binned_setup.energy0 eq 'Y') AND (l_binned_setup.energy2 eq 'Y') then begin
                  title1 = 'E0,E2'
                  plot_data = (*l_binned.m0lePtr) + (*l_binned.m0uePtr) + $
                              (*l_binned.m1lePtr) + (*l_binned.m1uePtr)
                  extra_data[*,0] = (*l_binned.m0leExtraPtr)[*,0] + (*l_binned.m0ueExtraPtr)[*,0] + $
                               (*l_binned.m1leExtraPtr)[*,0] + (*l_binned.m1ueExtraPtr)[*,0]
                  extra_data[*,1] = (*l_binned.m0leExtraPtr)[*,1] + (*l_binned.m0ueExtraPtr)[*,1] + $
                               (*l_binned.m1leExtraPtr)[*,1] + (*l_binned.m1ueExtraPtr)[*,1]
               endif else begin
                  ;---------------------------------------------------
                  ; plot hydrogen and oxygen and energies 1 and 2
                  ;---------------------------------------------------
                  if (l_binned_setup.energy1 eq 'Y') AND (l_binned_setup.energy2 eq 'Y') then begin
                      title1 = 'E1,E2'
                      plot_data = (*l_binned.m0cePtr) + (*l_binned.m0uePtr) + $
                                  (*l_binned.m1cePtr) + (*l_binned.m1uePtr)
                      extra_data[*,0] = (*l_binned.m0ceExtraPtr)[*,0] + (*l_binned.m0ueExtraPtr)[*,0] + $
                                   (*l_binned.m1ceExtraPtr)[*,0] + (*l_binned.m1ueExtraPtr)[*,0]
                      extra_data[*,1] = (*l_binned.m0ceExtraPtr)[*,1] + (*l_binned.m0ueExtraPtr)[*,1] + $
                                   (*l_binned.m1ceExtraPtr)[*,1] + (*l_binned.m1ueExtraPtr)[*,1]
                  endif else begin
                      ;---------------------------------------------------
                      ; plot hydrogen and oxygen and just energies 0 
                      ;---------------------------------------------------
                      if (l_binned_setup.energy0) then begin
                         title1 = 'E0'
                         plot_data = (*l_binned.m0lePtr) + (*l_binned.m1lePtr)
                         extra_data[*,0] = (*l_binned.m0leExtraPtr)[*,0] + (*l_binned.m1leExtraPtr)[*,0]
                         extra_data[*,1] = (*l_binned.m0leExtraPtr)[*,1] + (*l_binned.m1leExtraPtr)[*,1]
                      endif else begin
                         ;---------------------------------------------------
                         ; plot hydrogen and oxygen and just energies 1 
                         ;---------------------------------------------------
                         if (l_binned_setup.energy1) then begin
                            title1 = 'E1'
                            plot_data = (*l_binned.m0cePtr) + (*l_binned.m1cePtr)
                            extra_data[*,0] = (*l_binned.m0ceExtraPtr)[*,0] + (*l_binned.m1ceExtraPtr)[*,0]
                            extra_data[*,1] = (*l_binned.m0ceExtraPtr)[*,1] + (*l_binned.m1ceExtraPtr)[*,1]
                         endif else begin
                            ;---------------------------------------------------
                            ; plot hydrogen and oxygen and just energies 2 
                            ;---------------------------------------------------
                            if (l_binned_setup.energy2) then begin
                               title1 = 'E2'
                               plot_data = (*l_binned.m0uePtr) + (*l_binned.m1uePtr)
                               extra_data[*,0] = (*l_binned.m0ueExtraPtr)[*,0] + (*l_binned.m1ueExtraPtr)[*,0]
                               extra_data[*,1] = (*l_binned.m0ueExtraPtr)[*,1] + (*l_binned.m1ueExtraPtr)[*,1]
                            endif
                         endelse ;--- E1 ---
                      endelse ;--- E0 ---
                  endelse ;--- E1,E2 ---
               endelse ;--- E0, E2 ---
            endelse ;--- E0,E1 ---
           endelse ;--- all energies ---
        endif else begin ;--- end H/O ---
           ;---------------------------------------------------
           ; plot just hydrogen 
           ;---------------------------------------------------
           if (l_binned_setup.hydrogen eq 'Y') then begin
             title = 'H'
             ;---------------------------------------------------
             ; plot just hydrogen all energies
             ;---------------------------------------------------
             if (l_binned_setup.energy0 eq 'Y') AND (l_binned_setup.energy1 eq 'Y') AND $
                (l_binned_setup.energy2 eq 'Y') then begin
                 title1 = ' '
                 plot_data = (*l_binned.m0lePtr) + (*l_binned.m0cePtr) + (*l_binned.m0uePtr)
                 extra_data[*,0] = (*l_binned.m0leExtraPtr)[*,0] + (*l_binned.m0ceExtraPtr)[*,0] + $
                              (*l_binned.m0ueExtraPtr)[*,0]
                 extra_data[*,1] = (*l_binned.m0leExtraPtr)[*,1] + (*l_binned.m0ceExtraPtr)[*,1] + $
                              (*l_binned.m0ueExtraPtr)[*,1]
             endif else begin
             ;---------------------------------------------------
             ; plot just hydrogen and energies 0 and 1
             ;---------------------------------------------------
             if (l_binned_setup.energy0 eq 'Y') AND (l_binned_setup.energy1 eq 'Y') then begin
                title1 = 'E0, E1'
                plot_data = (*l_binned.m0lePtr) + (*l_binned.m0cePtr)
                extra_data[*,0] = (*l_binned.m0leExtraPtr)[*,0] + (*l_binned.m0ceExtraPtr)[*,0]
                extra_data[*,1] = (*l_binned.m0leExtraPtr)[*,1] + (*l_binned.m0ceExtraPtr)[*,1]
             endif else begin
                ;---------------------------------------------------
                ; plot just hydrogen and energies 0 and 2 
                ;---------------------------------------------------
                if (l_binned_setup.energy0 eq 'Y') AND (l_binned_setup.energy2 eq 'Y') then begin
                   title1 = 'E0, E2'
                   plot_data = (*l_binned.m0lePtr) + (*l_binned.m0uePtr)
                   extra_data[*,0] = (*l_binned.m0leExtraPtr)[*,0] + (*l_binned.m0ueExtraPtr)[*,0]
                   extra_data[*,1] = (*l_binned.m0leExtraPtr)[*,1] + (*l_binned.m0ueExtraPtr)[*,1]
                endif else begin
                   ;---------------------------------------------------
                   ; plot just hydrogen and energies 1 and 2 
                   ;---------------------------------------------------
                  if (l_binned_setup.energy1 eq 'Y') AND $
                     (l_binned_setup.energy2 eq 'Y') then begin 
                      title1 = 'E1, E2'
                      plot_data = (*l_binned.m0cePtr) + (*l_binned.m0uePtr)
                      extra_data[*,0] = (*l_binned.m0ceExtraPtr)[*,0] + (*l_binned.m0ueExtraPtr)[*,0]
                      extra_data[*,1] = (*l_binned.m0ceExtraPtr)[*,1] + (*l_binned.m0ueExtraPtr)[*,1]
                  endif else begin
                      ;---------------------------------------------------
                      ; plot just hydrogen and energy 0
                      ;---------------------------------------------------
                      if (l_binned_setup.energy0 eq 'Y') then begin
                         title1 = 'E0'
                         plot_data = (*l_binned.m0lePtr)
                         extra_data = (*l_binned.m0leExtraPtr)
                      endif else begin
                         ;---------------------------------------------------
                         ; plot just hydrogen and energy 1
                         ;---------------------------------------------------
                         if (l_binned_setup.energy1 eq 'Y') then begin
                           title1 = 'E1'
                           plot_data = (*l_binned.m0cePtr)
                           extra_data = (*l_binned.m0ceExtraPtr)
                         endif else begin
                             ;---------------------------------------------------
                             ; plot just hydrogen and energy 2
                             ;---------------------------------------------------
                             if (l_binned_setup.energy2 eq 'Y') then begin
                                title1 = 'E2'
                                plot_data = (*l_binned.m0uePtr)
                                extra_data = (*l_binned.m0ueExtraPtr)
                             endif
                         endelse ;--- E2 ---
                      endelse ;--- E1 ---
                  endelse ;--- E0 --- 
                 endelse ;--- E1,E2 --- 
               endelse
             endelse
           endif else begin
              ;---------------------------------------------------
              ; plot just oxygen 
              ;---------------------------------------------------
              if (l_binned_setup.oxygen eq 'Y') then begin
                title = 'O'
                ;---------------------------------------------------
                ; plot just oxygen  and all energies
                ;---------------------------------------------------
                if (l_binned_setup.energy0 eq 'Y') AND (l_binned_setup.energy1 eq 'Y') AND $
                   (l_binned_setup.energy2 eq 'Y') then begin
                    title1 = ' '
                    plot_data = (*l_binned.m1lePtr) + (*l_binned.m1cePtr) + (*l_binned.m1uePtr)
                    extra_data[*,0] = (*l_binned.m1leExtraPtr)[*,0] + (*l_binned.m1ceExtraPtr)[*,0] + $
                                 (*l_binned.m1ueExtraPtr)[*,0]
                    extra_data[*,1] = (*l_binned.m1leExtraPtr)[*,1] + (*l_binned.m1ceExtraPtr)[*,1] + $
                                 (*l_binned.m1ueExtraPtr)[*,1]
                endif else begin
                ;---------------------------------------------------
                ; plot just oxygen and energies 0 and 1
                ;---------------------------------------------------
                if (l_binned_setup.energy0 eq 'Y') AND (l_binned_setup.energy1 eq 'Y') then begin
                   title1 = 'E0,E1'
                   plot_data = (*l_binned.m1lePtr) + (*l_binned.m1cePtr)
                   extra_data[*,0] = (*l_binned.m1leExtraPtr)[*,0] + (*l_binned.m1ceExtraPtr)[*,0]
                   extra_data[*,1] = (*l_binned.m1leExtraPtr)[*,1] + (*l_binned.m1ceExtraPtr)[*,1]
                endif else begin
                  ;---------------------------------------------------
                  ; plot just oxygen and energies 0 and 2 
                  ;---------------------------------------------------
                  if (l_binned_setup.energy0 eq 'Y') AND (l_binned_setup.energy2 eq 'Y') then begin
                     title1 = 'E0,E2'
                     plot_data = (*l_binned.m1lePtr) + (*l_binned.m1uePtr)
                     extra_data[*,0] = (*l_binned.m1leExtraPtr)[*,0] + (*l_binned.m1ueExtraPtr)[*,0]
                     extra_data[*,1] = (*l_binned.m1leExtraPtr)[*,1] + (*l_binned.m1ueExtraPtr)[*,1]
                  endif else begin
                     ;---------------------------------------------------
                     ; plot just oxygen and energies 1 and 2 
                     ;---------------------------------------------------
                     if (l_binned_setup.energy1 eq 'Y') AND $
                        (l_binned_setup.energy2 eq 'Y') then begin
                          title1 = 'E1,E2'
                          plot_data = (*l_binned.m1cePtr) + (*l_binned.m1uePtr)
                          extra_data[*,0] = (*l_binned.m1ceExtraPtr)[*,0] + (*l_binned.m1ueExtraPtr)[*,0]
                          extra_data[*,1] = (*l_binned.m1ceExtraPtr)[*,1] + (*l_binned.m1ueExtraPtr)[*,1]
                     endif else begin
                         ;---------------------------------------------------
                         ; plot just oxygen and energy 0 
                         ;---------------------------------------------------
                         if (l_binned_setup.energy0 eq 'Y') then begin
                           title1 = 'E0'
                           plot_data = (*l_binned.m1lePtr)
                           extra_data = (*l_binned.m1leExtraPtr)
                         endif else begin
                            ;---------------------------------------------------
                            ; plot just oxygen and energy 1 
                            ;---------------------------------------------------
                            if (l_binned_setup.energy1 eq 'Y') then begin
                               title1 = 'E1'
                               plot_data = (*l_binned.m1cePtr)
                               extra_data = (*l_binned.m1ceExtraPtr)
                            endif else begin
                               ;---------------------------------------------------
                               ; plot just oxygen and energy 2 
                               ;---------------------------------------------------
                               if (l_binned_setup.energy2 eq 'Y') then begin
                                  title1 = 'E2'
                                  plot_data = (*l_binned.m1uePtr)
                                  extra_data = (*l_binned.m1ueExtraPtr)
                               endif
                            endelse ;--- E2 ---
                         endelse ;--- E1 ---
                     endelse ;--- E0 ---
                  endelse ;--- OE0, OE2 ---
                endelse ;--- OE0, OE1 ---
               endelse ;--- all O Energy ---
              endif ;--- oxygen ---
           endelse ;--- hydrogen ---
        endelse ;--- all energies ---


        index = reform (plot_data, n_elements(plot_data), /OVERWRITE)

;nitems = n_elements ((*l_binned.startUdfTimePtr)) - 1L
;for i=0, nitems do begin
;   print, 'binned udf time = ', (*l_binned.startUdfTimePtr)[i]
;   print, 'time in seconds = ', (*l_binned.timePtr)[i*45]
;endfor
help, (*l_binned.startUdfTimePtr)
help, (*l_binned.timePtr)
        ;----------------------------------------------------
        ; call routine to handle any data gaps and plot
        ;----------------------------------------------------
        l_BinnedProcData, plot_data, $
                          (*l_binned.timePtr), $
                          filled_time, $
                          (*l_binned.startUdfTimePtr), $
                          filled_start_udf_time, $
                          l_setup, $
                          center_sector, l_pltpos, $
                          plotted, l_oa, l_rgb_table, $
                          colorbar, l_hskp, $
                          title, title1, nplots, windx, pages, $
                          ytitle_frame, $
                          l_binned_setup.color_min, $
                          l_binned_setup.color_max, $
                          dmpPtr, extra_data

;help, filled_start_udf_time

        if (ptr_valid (l_binned.veDmpPtr)) then ptr_free, l_binned.veDmpPtr
        ptr = ptr_new (dmpPtr, /NO_COPY)
        l_binned.veDmpPtr = ptr

        if (ptr_valid (l_binned.filledStartUdfTimePtr)) then ptr_free, l_binned.filledStartUdfTimePtr
        ptr = ptr_new (filled_start_udf_time, /NO_COPY)
        l_binned.filledStartUdfTimePtr = ptr

   endif else begin ;--- not merge ---
         ;----------------------------------------------------
         ; plot only hydrogen
         ;----------------------------------------------------
         if (l_binned_setup.hydrogen eq 'Y') then begin
            title = 'H'
            ;----------------------------------------------------
            ; plot only hydrogen energy 0
            ;----------------------------------------------------
            if (l_binned_setup.energy0 eq 'Y') then begin
                title1 = 'E0'
                l_BinnedProcData, (*l_binned.m0lePtr), $
                                  (*l_binned.timePtr)[*], $
                                  filled_time, $
                                  (*l_binned.startUdfTimePtr), $
                                  filled_start_udf_time, $
                                  l_setup, $
                                  center_sector, l_pltpos, $
                                  plotted, l_oa, l_rgb_table, $
                                  colorbar, l_hskp, $
                                  title, title1, nplots, windx, pages, $
                                  ytitle_frame, $
                                  l_binned_setup.color_min, $
                                  l_binned_setup.color_max, $
                                  dmpPtr, (*l_binned.m0leExtraPtr)

                if (ptr_valid (l_binned.m0leDmpPtr)) then ptr_free, l_binned.m0leDmpPtr
                ptr = ptr_new (dmpPtr, /NO_COPY)
                l_binned.m0leDmpPtr = ptr
            endif 

            ;----------------------------------------------------
            ; plot only hydrogen energy 1 
            ;----------------------------------------------------
            if (l_binned_setup.energy1 eq 'Y') then begin
               title1 = 'E1'
               l_BinnedProcData, (*l_binned.m0cePtr), $
                                 (*l_binned.timePtr)[*], $
                                 filled_time, $
                                 (*l_binned.startUdfTimePtr), $
                                 filled_start_udf_time, $
                                 l_setup, $
                                 center_sector, l_pltpos, $
                                 plotted, l_oa, l_rgb_table, $
                                 colorbar, l_hskp, $
                                 title, title1, nplots, windx, pages, $
                                 ytitle_frame, $
                                 l_binned_setup.color_min, $
                                 l_binned_setup.color_max, $
                                 dmpPtr, (*l_binned.m0ceExtraPtr)

               if (ptr_valid (l_binned.m0ceDmpPtr)) then ptr_free, l_binned.m0ceDmpPtr
               ptr = ptr_new (dmpPtr, /NO_COPY)
               l_binned.m0ceDmpPtr = ptr

            endif

            ;----------------------------------------------------
            ; plot only hydrogen energy 2 
            ;----------------------------------------------------
            if (l_binned_setup.energy2 eq 'Y') then begin
               title1 = 'E2'
               l_BinnedProcData, (*l_binned.m0uePtr), $
                                 (*l_binned.timePtr)[*], $
                                 filled_time, $
                                 (*l_binned.startUdfTimePtr), $
                                 filled_start_udf_time, $
                                 l_setup, $
                                 center_sector, l_pltpos, $
                                 plotted, l_oa, l_rgb_table, $
                                 colorbar, l_hskp, $
                                 title, title1, nplots, windx, pages, $
                                 ytitle_frame, $
                                 l_binned_setup.color_min, $
                                 l_binned_setup.color_max, $
                                 dmpPtr, (*l_binned.m0ueExtraPtr)

                if (ptr_valid (l_binned.m0ueDmpPtr)) then ptr_free, l_binned.m0ueDmpPtr
                ptr = ptr_new (dmpPtr, /NO_COPY)
                l_binned.m0ueDmpPtr = ptr

            endif ;--- E2 ---

            if (ptr_valid (l_binned.filledStartUdfTimePtr)) then ptr_free, l_binned.filledStartUdfTimePtr
            ptr = ptr_new (filled_start_udf_time, /NO_COPY)
            l_binned.filledStartUdfTimePtr = ptr

         endif ;--- Hydrogen ---

         ;----------------------------------------------------
         ; plot only oxygen
         ;----------------------------------------------------
         if (l_binned_setup.oxygen eq 'Y') then begin
            title = 'O'
            ;----------------------------------------------------
            ; plot only oxygen energy 0
            ;----------------------------------------------------
            if (l_binned_setup.energy0 eq 'Y') then begin
               title1 = 'E0'
               l_BinnedProcData, (*l_binned.m1lePtr), $
                                 (*l_binned.timePtr)[*], $
                                 filled_time, $
                                 (*l_binned.startUdfTimePtr), $
                                 filled_start_udf_time, $
                                 l_setup, $
                                 center_sector, l_pltpos, $
                                 plotted, l_oa, l_rgb_table, $
                                 colorbar, l_hskp, $
                                 title, title1, nplots, windx, pages, $
                                 ytitle_frame, $
                                 l_binned_setup.color_min, $
                                 l_binned_setup.color_max, $
                                 dmpPtr, (*l_binned.m1leExtraPtr)

               if (ptr_valid (l_binned.m1leDmpPtr)) then ptr_free, l_binned.m1leDmpPtr 
               ptr = ptr_new (dmpPtr, /NO_COPY)
               l_binned.m1leDmpPtr = ptr

            endif 

            ;----------------------------------------------------
            ; plot only oxygen energy 1 
            ;----------------------------------------------------
            if (l_binned_setup.energy1 eq 'Y') then begin
                title1 = 'E1'
                l_BinnedProcData, (*l_binned.m1cePtr), $
                                  (*l_binned.timePtr)[*], $
                                  filled_time, $
                                  (*l_binned.startUdfTimePtr), $
                                  filled_start_udf_time, $
                                  l_setup, $
                                  center_sector, l_pltpos, $
                                  plotted, l_oa, l_rgb_table, $
                                  colorbar, l_hskp, $
                                  title, title1, nplots, windx, pages, $
                                  ytitle_frame, $
                                  l_binned_setup.color_min, $
                                  l_binned_setup.color_max, $
                                  dmpPtr, (*l_binned.m1ceExtraPtr)

                if (ptr_valid (l_binned.m1ceDmpPtr)) then ptr_free, l_binned.m1ceDmpPtr
                ptr = ptr_new (dmpPtr, /NO_COPY)
                l_binned.m1ceDmpPtr = ptr

            endif 

            ;----------------------------------------------------
            ; plot only oxygen energy 2 
            ;----------------------------------------------------
            if (l_binned_setup.energy2 eq 'Y') then begin
                title1 = 'E2'
                l_BinnedProcData, (*l_binned.m1uePtr), $
                                  (*l_binned.timePtr)[*], $
                                  filled_time, $
                                  (*l_binned.startUdfTimePtr), $
                                  filled_start_udf_time, $
                                  l_setup, $
                                  center_sector, l_pltpos, $
                                  plotted, l_oa, l_rgb_table, $
                                  colorbar, l_hskp, $
                                  title, title1, nplots, windx, pages, $
                                  ytitle_frame, $
                                  l_binned_setup.color_min, $
                                  l_binned_setup.color_max, $
                                  dmpPtr, (*l_binned.m1ueExtraPtr)

                 if (ptr_valid (l_binned.m1ueDmpPtr)) then ptr_free, l_binned.m1ueDmpPtr
                 ptr = ptr_new (dmpPtr, /NO_COPY)
                 l_binned.m1ueDmpPtr = ptr

            endif ;--- E2 ---

            if (ptr_vaild (l_binned.filledStartUdfTimePtr)) then ptr_free, l_binned.filledStartUdfTimePtr
            ptr = ptr_new (filled_start_udf_time, /NO_COPY)
            l_binned.filledStartUdfTimePtr = ptr

         endif  ;--- oxygen ---
    endelse ;--- merge ---


RETURN
END

;-------------------------------------------------------------------------------
;  Procedure: SyncBinn2EventBK
;
;  Description: Sync the event background data to the binned data.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None 
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_binned              struct *      I        pointers to binned data structures
;  l_event               struct *      I        pointers to event data structures
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------

PRO SyncBinn2EventBK, l_binned, l_event

   n_binn = n_elements (*l_binned.startUdfTimePtr) 
   n_event = n_elements (*l_event.filledStartUdfTimePtr) 

   m0 = fltarr (3,12,45,n_binn)
   m1 = fltarr (3,12,45,n_binn)

   last_ii = 0L
   last_jj = 0L
   indx    = 0L
   status  = 0L
   ii      = 0L
   jj      = 0L
   idx     = 0L

   for jj = 0L, n_binn - 1L do begin

       start_binn_time = Convert((*l_binned.startUdfTimePtr)[jj])
       stop_binn_time = Convert((*l_binned.stopUdfTimePtr)[jj])

       if (jj lt n_binn - 1L) then begin
          next_start_binn_time = Convert((*l_binned.startUdfTimePtr)[jj+1])
       endif else begin
          next_start_binn_time = Convert((*l_binned.stopUdfTimePtr)[jj])
       endelse

       for ii = last_ii, n_event - 1L do begin
           start_event_time = Convert((*l_event.filledStartUdfTimePtr)[ii])
           stop_event_time = Convert((*l_event.filledStopUdfTimePtr)[ii])

           ;--------------------------------------------------------------------
           ; determine if event data falls within this spin
           ;--------------------------------------------------------------------
           if (start_event_time ge start_binn_time) and $
              (start_event_time le next_start_binn_time) then begin

              ;--------------------------------------------------------------------
              ; if the event time is equal to the boundary end time
              ;--------------------------------------------------------------------
              if (start_event_time eq next_start_binn_time) then begin
                 ;--------------------------------------------------------------------
                 ; check that the event data belongs in the current bin spin, if it does
                 ; then extract the data, else need to go to the next spin
                 ;--------------------------------------------------------------------
                 if (start_event_time ge start_binn_time) and $
                    (start_event_time le stop_binn_time) then begin
                       m0[*,*,*,idx] = (*l_binned.m0RawPtr)[*,*,*,jj]
                       m1[*,*,*,idx] = (*l_binned.m1RawPtr)[*,*,*,jj]
                       last_ii = ii
                       idx = idx + 1L
                   endif else begin
                       last_ii = ii
                       goto, NEXT
                   endelse
              endif else begin
                   m0[*,*,*,idx] = (*l_binned.m0RawPtr)[*,*,*,jj]
                   m1[*,*,*,idx] = (*l_binned.m1RawPtr)[*,*,*,jj]
                   last_ii = ii
                   idx = idx + 1L
              endelse
            endif else begin
              ;--------------------------------------------------------------------
              ; does not fall within that particular spin period, check if there
              ; is a data gap and if the current event data belongs in the next
              ; spin, if status = 0, no data gap and data belongs in the next spin
              ; then get the next packet
              ;--------------------------------------------------------------------
              status = l_DataGap (next_start_binn_time, stop_binn_time, $
                                  start_event_time)
              if (status eq 0L) then begin
                 last_ii = ii
                 goto, NEXT
              endif
            endelse
       endfor
NEXT:
   endfor


   (*l_binned.m0RawPtr) = m0
   (*l_binned.m1RawPtr) = m1

RETURN
END

;-------------------------------------------------------------------------------
;  Procedure: l_Binned_AdjExData
;
;  Description: Adjust the arrays when extra spins are added to the begin and end
;           of specified time range.  This is needed for when traces are centered
;           at zero degree spin angle and data from either a previous or next spin
;           is needed to fill the current spin. 
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_setup               struct        I        general setup parameters
;  l_pltpos              struct        I        general plotting parameters
;  l_binned              struct *      I        pointer to binned data structures
;  l_binned_setup        struct        I        binned processing & plotting parameters 
;  l_event               struct *      I        poiner to event data structures
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;     PLOTDEFS         MAX_PLOTS        int         G         max # plots per page
;                      MAX_WEB_SPINS    int         G         max # spins per plot
;                                                             when using web interface
;                      NO_CNTS          double      G         no counts indicator
;                      NO_DATA          double      G         no data indicator
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;  ELH                 05/04        v1.0.11           Added ptr check for vePtr
;                                                     if valid, free before new
;                                                     assignment
;  ELH               05/27/04       v1.0.11           replaced use of multiple
;                                                     for loops to use IDL total()
;-------------------------------------------------------------------------------
PRO l_Binned_AdjExData, l_setup, l_pltpos, l_binned, l_binned_setup, l_event 
COMMON PLOTDEFS, MAX_PLOTS, MAX_WEB_SPINS, NO_CNTS, NO_DATA


  ;----------------------------------------------------
  ; adjust the time arrays, if necessary
  ;----------------------------------------------------

  if (l_setup.l_flux eq 'Y') AND (l_setup.l_dumps eq 'Y') then begin
     l_setup.l_filedesc = 'eff'
     l_GenFileName, l_setup, '0'
     flux_filename = l_setup.l_prod_dst + '/' + $
             l_setup.l_filename + '.dmp'
     openw, flux_ofd, flux_filename, /get_lun
     l_DumpHdrInfo, flux_ofd, l_setup, l_pltpos, flux_filename
     printf, flux_ofd, 'Data :   Binned'
  endif


  ;nrec = n_elements ((*l_binned.timePtr)[*])
  nrec = n_elements (*l_binned.timePtr) - 1L
  ;--------------------------------------------------------
  ; subtract 1 to account for array beginning at zero
  ;--------------------------------------------------------
  tme_nrec = n_elements (*l_binned.startUdfTimePtr) - 1L
  ;tme_nrec = n_elements ((*l_binned.startUdfTimePtr)[*])
  nspins =  n_elements (*l_binned.startUdfTimePtr)

print, 'nrec = ', nrec
print, 'tme_nrec  = ', tme_nrec

  m0DataPtr = (*l_binned.m0RawPtr)
  m1DataPtr = (*l_binned.m1RawPtr)

;openw, ifd, 'inside_before_binned_time.txt', /get_lun
;for i = 0, tme_nrec -1L do begin
;   printf, ifd, (*l_binned.startUdfTimePtr)[i], (*l_binned.timePtr)[i*45]
;endfor
;close, ifd
;free_lun, ifd


  ;------------------------------------------------
  ; if generating an event file text dump
  ; calculate the valid events 
  ;------------------------------------------------
  if (l_setup.l_data_type eq 'l_zse') then begin
     comb  = fltarr (l_setup.l_num_spinsec, nspins)
     HO    = (*l_binned.m0RawPtr) + (*l_binned.m1RawPtr)

     ;---------------------------------------------------------
     ;for sp = 0L, nspins-1L do begin
     ;  for sec = 0L, l_setup.l_num_spinsec-1L do begin
     ;     for p = 0L, 11L do begin
     ;         for e = 0L, 2L do begin
     ;           comb[sec, sp] = comb[sec, sp] + HO[e,p,sec,sp]
     ;         endfor
     ;     endfor
     ;  endfor
     ;endfor
     ;---------------------------------------------------------
     ; use of total is equivalent to use of 4 for loops
     ;---------------------------------------------------------
     temp = total (HO,1)
     comb = total (temp, 1)

     if (ptr_valid(l_binned.vePtr)) then ptr_free,l_binned.vePtr
     ptr = ptr_new (comb, /NO_COPY)
     l_binned.vePtr = ptr


  endif else begin
    if (l_binned_setup.addedStartFlag eq 1L) AND (l_binned_setup.addedStopFlag eq 1L) then begin
;print, (*l_binned.spinTimePtr)
       new_end = nrec - l_setup.l_num_spinsec
;print, 'new_end = ', new_end
       (*l_binned.timePtr) = (*l_binned.timePtr)[45:new_end]
       (*l_binned.spinTimePtr)     =  (*l_binned.spinTimePtr)[1L:tme_nrec-1L]
       (*l_binned.startUdfTimePtr) =  (*l_binned.startUdfTimePtr)[1L:tme_nrec-1L] 
       (*l_binned.stopUdfTimePtr)  =  (*l_binned.stopUdfTimePtr)[1L:tme_nrec-1L] 
    endif else begin
       if (l_binned_setup.addedStartFlag eq 1L) AND (l_binned_setup.addedStopFlag eq 0L) then begin
         (*l_binned.timePtr) = (*l_binned.timePtr)[l_setup.l_num_spinsec:nrec]
         (*l_binned.spinTimePtr)     =  (*l_binned.spinTimePtr)[1L:tme_nrec]
         (*l_binned.startUdfTimePtr) =  (*l_binned.startUdfTimePtr)[1L:tme_nrec] 
         (*l_binned.stopUdfTimePtr)  =  (*l_binned.stopUdfTimePtr)[1L:tme_nrec] 
       endif else begin
         if (l_binned_setup.addedStartFlag eq 0L) AND $
            (l_binned_setup.addedStopFlag eq 1L) then begin
            ; --- BAD ??  (*l_binned.timePtr) = (*l_binned.timePtr)[0:(nrec-1L-l_setup.l_num_spinsec)] ---
            (*l_binned.timePtr) = (*l_binned.timePtr)[0:(nrec-l_setup.l_num_spinsec)]

print, 'BINNED ADJ timePtr = '
help, (*l_binned.timePtr)

            (*l_binned.spinTimePtr)     =  (*l_binned.spinTimePtr)[0L:tme_nrec-1L]
print, 'BINNED ADJ spinTime = '
help, (*l_binned.spinTimePtr)

            (*l_binned.startUdfTimePtr) =  (*l_binned.startUdfTimePtr)[0L:tme_nrec-1L] 
            (*l_binned.stopUdfTimePtr)  =  (*l_binned.stopUdfTimePtr)[0L:tme_nrec-1L] 
         endif
       endelse
    endelse

;print, (*l_binned.spinTimePtr)

;print, 'l_setup.l_num_spinsec = ', l_setup.l_num_spinsec
;print, 'nrec - 45 = ', nrec-l_setup.l_num_spinsec
;new_tme_nrec = n_elements ((*l_binned.spinTimePtr))
;print, 'new_tme_nrec = ', new_tme_nrec
;help, (*l_binned.spinTimePtr)
;help, (*l_binned.timePtr)
;openw, ifd, 'inside_after_binned_time.txt', /get_lun
;for i = 0, new_tme_nrec-1L do begin
;   printf, ifd, (*l_binned.startUdfTimePtr)[i], (*l_binned.timePtr)[i*45]
;endfor
;close, ifd
;free_lun, ifd

;help, (*l_binned.timePtr)
;help, (*l_binned.startUdfTimePtr)
;print, (*l_binned.startUdfTimePtr)


  nsp_rec = nspins - 1L

  ;----------------------------------------------------
  ;  Hydrogen, Mass 0
  ;----------------------------------------------------
  if (l_binned_setup.hydrogen eq 'Y') then begin

     ;----------------------------------------------------
     ;  Hydrogen, Mass 0, lower energy bin
     ;----------------------------------------------------
     if (l_binned_setup.energy0 eq 'Y') then begin
        mass0_le  = fltarr (l_setup.l_num_spinsec, nspins)
        m0        = m0DataPtr

        ;-------------------------------------------------------------
        ;for sp = 0L, nspins-1L do begin
        ;   for sec = 0L, l_setup.l_num_spinsec-1L do begin
        ;      for p = 0L, 11 do begin
        ;          mass0_le[sec, sp] = mass0_le[sec, sp] + m0[0,p,sec,sp]
        ;      endfor
        ;   endfor
        ;endfor
        ;-------------------------------------------------------------
        ; use of total is equivalent to the 3 for loops
        ;-------------------------------------------------------------
        temp = total (m0[0,*,*,*],1)
        mass0_le = total(temp, 1)


        ptr = ptr_new (mass0_le)
        if (ptr_valid (l_binned.hydrogen_raw_cnts_e0)) then ptr_free, l_binned.hydrogen_raw_cnts_e0
        l_binned.hydrogen_raw_cnts_e0 = ptr

        if (l_setup.l_flux eq 'Y') OR (l_binned_setup.background eq 'Y') then begin

           ;--------------------------------------------------
           ; if the event background data is less than the
           ; available binned data, then increase the size of
           ; the event data to subtract a background of zero,
           ; this maintains the number of spins in the binned
           ; data
           ;--------------------------------------------------
           nele_bk = n_elements ((*l_event.background_e0))
           nele_dt = n_elements (mass0_le)
           if (nele_bk lt nele_dt) then begin
              sp = nele_bk/l_setup.l_num_spinsec
              tmp_bk = fltarr (l_setup.l_num_spinsec, nspins)
              tmp_bk[*,0:sp-1] = (*l_event.background_e0)
              (*l_event.background_e0) = tmp_bk
           endif

           ;--------------------------------------------------
           ; subtract background data
           ;--------------------------------------------------
           mass0_le = TEMPORARY(mass0_le) - ((*l_event.background_e0) * 0.4)
           mass0_le = mass0_le > 0.0
           if (ptr_valid (l_binned.corrected_hydrogen_e0)) then ptr_free, l_binned.corrected_hydrogen_e0
           ptr = ptr_new (mass0_le)
           l_binned.corrected_hydrogen_e0 = ptr

        endif

        ;--------------------------------------------------
        ; convert to count rate
        ;--------------------------------------------------
        if (l_setup.l_count_rate eq 'Y') then begin
           l_Conv2CntRate, mass0_le, mass0_le, 'SECTOR'
           if (ptr_valid (l_binned.corr_scl_H_e0)) then ptr_free, l_binned.corr_scl_H_e0
           ptr = ptr_new (mass0_le)
           l_binned.corr_scl_H_e0 = ptr
        endif

        ;--------------------------------------------------
        ; convert to flux
        ;--------------------------------------------------
        if (l_setup.l_flux eq 'Y') then begin
           l_ConvFlux, mass0_le, mass0_le, 'hydrogen', $
                       l_binned_setup.he0_eV, l_setup.l_start_dtime, $
                       state, nom_eff, eff
           if (l_setup.l_dumps eq 'Y') then begin
              l_WriteEff2File, flux_ofd, 'hydrogen', l_binned_setup.he0_eV, $
                           state, nom_eff, eff
           endif

        endif

        if (ptr_valid (l_binned.m0lePtr)) then ptr_free, l_binned.m0lePtr
        ptr = ptr_new (mass0_le, /NO_COPY)
        l_binned.m0lePtr = ptr

        ;--------------------------------------------------
        ; adjust arrays depending on time add flag
        ;--------------------------------------------------
        extra_data = make_array (l_setup.l_num_spinsec, 2L, /FLOAT, Value = NO_DATA)
        if (l_binned_setup.addedStartFlag eq 1L) AND (l_binned_setup.addedStopFlag eq 1L) then begin
           extra_data[*,0] = (*l_binned.m0lePtr)[*, 0]
           extra_data[*,1] = (*l_binned.m0lePtr)[*, nsp_rec]
           (*l_binned.m0lePtr) = (*l_binned.m0lePtr)[*, 1:nsp_rec-1L] 
           (*l_binned.hydrogen_raw_cnts_e0) = (*l_binned.hydrogen_raw_cnts_e0)[*,1:nsp_rec-1L]
           if (l_setup.l_flux eq 'Y') then begin
              (*l_binned.corrected_hydrogen_e0) =  (*l_binned.corrected_hydrogen_e0)[*,1:nsp_rec-1L]
           endif
           if (l_setup.l_count_rate eq 'Y') then begin
              (*l_binned.corr_scl_H_e0) = (*l_binned.corr_scl_H_e0)[*,1:nsp_rec-1L]
           endif
        endif else begin
           if (l_binned_setup.addedStartFlag eq 1L) AND $
           (l_binned_setup.addedStopFlag eq 0L) then begin
              extra_data[*,0] = (*l_binned.m0lePtr)[*, 0]
              extra_data[*,1] = NO_DATA 
              (*l_binned.m0lePtr) = (*l_binned.m0lePtr)[*, 1:nsp_rec] 
              (*l_binned.hydrogen_raw_cnts_e0) = (*l_binned.hydrogen_raw_cnts_e0)[*,1:nsp_rec]
              if (l_setup.l_flux eq 'Y') then begin
                 (*l_binned.corrected_hydrogen_e0) =  (*l_binned.corrected_hydrogen_e0)[*,1:nsp_rec]
              endif
              if (l_setup.l_count_rate eq 'Y') then begin
                 (*l_binned.corr_scl_H_e0) = (*l_binned.corr_scl_H_e0)[*,1:nsp_rec]
              endif
           endif else begin
              if (l_binned_setup.addedStartFlag eq 0L) AND $
                 (l_binned_setup.addedStopFlag eq 1L) then begin
                 extra_data[*,0] = NO_DATA 
                 extra_data[*,1] = (*l_binned.m0lePtr)[*, nsp_rec]
                 (*l_binned.m0lePtr) = (*l_binned.m0lePtr)[*, 0:nsp_rec-1L] 
                 (*l_binned.hydrogen_raw_cnts_e0) = (*l_binned.hydrogen_raw_cnts_e0)[*,0:nsp_rec-1L]
                 if (l_setup.l_flux eq 'Y') then begin
                    (*l_binned.corrected_hydrogen_e0) =  (*l_binned.corrected_hydrogen_e0)[*,0:nsp_rec-1L]
                 endif
                 if (l_setup.l_count_rate eq 'Y') then begin
                    (*l_binned.corr_scl_H_e0) = (*l_binned.corr_scl_H_e0)[*,0:nsp_rec-1L]
                 endif
              endif
           endelse
        endelse

        if (ptr_valid (l_binned.m0leExtraPtr)) then ptr_free, l_binned.m0leExtraPtr
        ptr = ptr_new (extra_data, /NO_COPY)
        l_binned.m0leExtraPtr = ptr

     endif


     ;----------------------------------------------------
     ;  Hydrogen, Mass 0, center energy bin
     ;----------------------------------------------------

     if (l_binned_setup.energy1 eq 'Y') then begin
        mass0_ce  = fltarr (l_setup.l_num_spinsec, nspins)
        m0        = m0DataPtr

        ;-----------------------------------------------------------------
        ;for sp = 0, nspins-1 do begin
        ;   for sec = 0, l_setup.l_num_spinsec-1L do begin
        ;      for p = 0, 11 do begin
        ;          mass0_ce[sec, sp] = mass0_ce[sec, sp] + m0[1,p,sec,sp]
        ;      endfor
        ;   endfor
        ;endfor
        ;-----------------------------------------------------------------
        ; use of total() is equivalent to the 3 for loops
        ;-----------------------------------------------------------------
        temp = total (m0[1,*,*,*], 1)
        mass0_ce = total (temp, 1)

        if (ptr_valid (l_binned.hydrogen_raw_cnts_e1)) then ptr_free, l_binned.hydrogen_raw_cnts_e1
        ptr = ptr_new (mass0_ce)
        l_binned.hydrogen_raw_cnts_e1 = ptr

        if (l_setup.l_flux eq 'Y') OR  (l_binned_setup.background eq 'Y') then begin

           ;--------------------------------------------------
           ; if the event background data is less than the
           ; available binned data, then increase the size of
           ; the event data to subtract a background of zero,
           ; this maintains the number of spins in the binned
           ; data
           ;--------------------------------------------------
           nele_bk = n_elements ((*l_event.background_e1))
           nele_dt = n_elements (mass0_ce)
           if (nele_bk lt nele_dt) then begin
              sp = nele_bk/l_setup.l_num_spinsec
              tmp_bk = fltarr (l_setup.l_num_spinsec, nspins)
              tmp_bk[*,0:sp-1] = (*l_event.background_e1)
              (*l_event.background_e1) = tmp_bk
           endif

           mass0_ce = TEMPORARY (mass0_ce) - ((*l_event.background_e1) * 0.4)
           mass0_ce = mass0_ce > 0.0
           if (ptr_valid (l_binned.corrected_hydrogen_e1)) then ptr_free, l_binned.corrected_hydrogen_e1
           ptr = ptr_new (mass0_ce)
           l_binned.corrected_hydrogen_e1 = ptr

        endif

        if (l_setup.l_count_rate eq 'Y') then begin
           l_Conv2CntRate, mass0_ce, mass0_ce, 'SECTOR'
           if (ptr_valid (l_binned.corr_scl_H_e1)) then ptr_free, l_binned.corr_scl_H_e1
           ptr = ptr_new (mass0_ce)
           l_binned.corr_scl_H_e1 = ptr
        endif

        if (l_setup.l_flux eq 'Y') then begin
           l_ConvFlux, mass0_ce, mass0_ce, 'hydrogen', $
                       l_binned_setup.he1_eV, l_setup.l_start_dtime, $
                       state, nom_eff, eff
           if (l_setup.l_dumps eq 'Y') then begin
              l_WriteEff2File, flux_ofd, 'hydrogen', l_binned_setup.he1_eV, $
                        state, nom_eff, eff
           endif
        endif

        if (ptr_valid (l_binned.m0cePtr)) then ptr_free, l_binned.m0cePtr
        ptr = ptr_new (mass0_ce, /NO_COPY)
        l_binned.m0cePtr = ptr

        extra_data = make_array (l_setup.l_num_spinsec, 2L, /FLOAT, Value = NO_DATA)
        if (l_binned_setup.addedStartFlag eq 1L) AND (l_binned_setup.addedStopFlag eq 1L) then begin
           extra_data[*,0] = (*l_binned.m0cePtr)[*, 0]
           extra_data[*,1] = (*l_binned.m0cePtr)[*, nsp_rec]
           (*l_binned.m0cePtr) = (*l_binned.m0cePtr)[*, 1:nsp_rec-1L] 
           (*l_binned.hydrogen_raw_cnts_e1) = (*l_binned.hydrogen_raw_cnts_e1)[*,1:nsp_rec-1L]
           if (l_setup.l_flux eq 'Y') then begin
              (*l_binned.corrected_hydrogen_e1) = (*l_binned.corrected_hydrogen_e1)[*, 1:nsp_rec-1L] 
           endif
           if (l_setup.l_count_rate eq 'Y') then begin
              (*l_binned.corr_scl_H_e1) = (*l_binned.corr_scl_H_e1)[*, 1:nsp_rec-1L]
           endif
        endif else begin
           if (l_binned_setup.addedStartFlag eq 1L) AND $
              (l_binned_setup.addedStopFlag eq 0L) then begin
              extra_data[*,0] = (*l_binned.m0cePtr)[*, 0]
              extra_data[*,1] = NO_DATA 
              (*l_binned.m0cePtr) = (*l_binned.m0cePtr)[*, 1:nsp_rec] 
              (*l_binned.hydrogen_raw_cnts_e1) = (*l_binned.hydrogen_raw_cnts_e1)[*,1:nsp_rec]
              if (l_setup.l_flux eq 'Y') then begin
                 (*l_binned.corrected_hydrogen_e1) = (*l_binned.corrected_hydrogen_e1)[*, 1:nsp_rec] 
              endif
              if (l_setup.l_count_rate eq 'Y') then begin
                 (*l_binned.corr_scl_H_e1) = (*l_binned.corr_scl_H_e1)[*, 1:nsp_rec]
              endif
           endif else begin
              if (l_binned_setup.addedStartFlag eq 0L) AND $
                 (l_binned_setup.addedStopFlag eq 1L) then begin
                 extra_data[*,0] = NO_DATA 
                 extra_data[*,1] = (*l_binned.m0cePtr)[*, nsp_rec]
                 (*l_binned.m0cePtr) = (*l_binned.m0cePtr)[*, 0:nsp_rec-1L] 
                 (*l_binned.hydrogen_raw_cnts_e1) = (*l_binned.hydrogen_raw_cnts_e1)[*,0:nsp_rec-1L]
                 if (l_setup.l_flux eq 'Y') then begin
                    (*l_binned.corrected_hydrogen_e1) = (*l_binned.corrected_hydrogen_e1)[*, 0:nsp_rec-1L] 
                 endif
                 if (l_setup.l_count_rate eq 'Y') then begin
                    (*l_binned.corr_scl_H_e1) = (*l_binned.corr_scl_H_e1)[*, 0:nsp_rec-1L]
                 endif
              endif
           endelse
        endelse

        if (ptr_valid (l_binned.m0ceExtraPtr)) then ptr_free, l_binned.m0ceExtraPtr
        ptr = ptr_new (extra_data, /NO_COPY)
        l_binned.m0ceExtraPtr = ptr

     endif


     ;----------------------------------------------------
     ;  Hydrogen, Mass 0, upper energy bin
     ;----------------------------------------------------

     if (l_binned_setup.energy2 eq 'Y') then begin
        mass0_ue  = fltarr (l_setup.l_num_spinsec, nspins)
        m0        = m0DataPtr

        ;----------------------------------------------------------------
        ;for sp = 0, nspins-1 do begin
        ;   for sec = 0, l_setup.l_num_spinsec-1L do begin
        ;      for p = 0, 11 do begin
        ;          mass0_ue[sec, sp] = mass0_ue[sec, sp] + m0[2,p,sec,sp]
        ;      endfor
        ;   endfor
        ;endfor
        ;----------------------------------------------------------------
        ; use of total() is equivalent to the 3 for loops
        ;----------------------------------------------------------------
        temp = total (m0[2,*,*,*], 1)
        mass0_ue = total (temp, 1)
        

        if (ptr_valid (l_binned.hydrogen_raw_cnts_e2)) then ptr_free, l_binned.hydrogen_raw_cnts_e2
        ptr = ptr_new (mass0_ue)
        l_binned.hydrogen_raw_cnts_e2 = ptr

        if (l_setup.l_flux eq 'Y') OR  (l_binned_setup.background eq 'Y') then begin

           ;--------------------------------------------------
           ; if the event background data is less than the
           ; available binned data, then increase the size of
           ; the event data to subtract a background of zero,
           ; this maintains the number of spins in the binned
           ; data
           ;--------------------------------------------------
           nele_bk = n_elements ((*l_event.background_e2))
           nele_dt = n_elements (mass0_ue)
           if (nele_bk lt nele_dt) then begin
              sp = nele_bk/l_setup.l_num_spinsec
              tmp_bk = fltarr (l_setup.l_num_spinsec, nspins)
              tmp_bk[*,0:sp-1] = (*l_event.background_e2)
              (*l_event.background_e2) = tmp_bk
           endif

           mass0_ue = TEMPORARY (mass0_ue) - ((*l_event.background_e2) * 0.4)
           mass0_ue = mass0_ue > 0.0
           if (ptr_valid (l_binned.corrected_hydrogen_e2)) then ptr_free, l_binned.corrected_hydrogen_e2
           ptr = ptr_new (mass0_ue)
           l_binned.corrected_hydrogen_e2 = ptr

        endif

        if (l_setup.l_count_rate eq 'Y') then begin
           l_Conv2CntRate, mass0_ue, mass0_ue, 'SECTOR'
           if (ptr_valid (l_binned.corr_scl_H_e2)) then ptr_free, l_binned.corr_scl_H_e2
           ptr = ptr_new (mass0_ue)
           l_binned.corr_scl_H_e2 = ptr
        endif

        if (l_setup.l_flux eq 'Y') then begin
           l_ConvFlux, mass0_ue, mass0_ue, 'hydrogen', $
                       l_binned_setup.he2_eV, l_setup.l_start_dtime, $
                       state, nom_eff, eff
           if (l_setup.l_dumps eq 'Y') then begin
              l_WriteEff2File, flux_ofd, 'hydrogen', l_binned_setup.he2_eV, $
                           state, nom_eff, eff
           endif
        endif

        if (ptr_valid (l_binned.m0uePtr)) then ptr_free, l_binned.m0uePtr
        ptr = ptr_new (mass0_ue, /NO_COPY)
        l_binned.m0uePtr = ptr


        extra_data = make_array (l_setup.l_num_spinsec, 2L, /FLOAT, Value = NO_DATA)
        if (l_binned_setup.addedStartFlag eq 1L) AND (l_binned_setup.addedStopFlag eq 1L) then begin
           extra_data[*,0] = (*l_binned.m0uePtr)[*, 0]
           extra_data[*,1] = (*l_binned.m0uePtr)[*, nsp_rec]
           (*l_binned.m0uePtr) = (*l_binned.m0uePtr)[*, 1:nsp_rec-1L] 
           (*l_binned.hydrogen_raw_cnts_e2) = (*l_binned.hydrogen_raw_cnts_e2)[*,1:nsp_rec-1L]
           if (l_setup.l_flux eq 'Y') then begin
              (*l_binned.corrected_hydrogen_e2) = (*l_binned.corrected_hydrogen_e2)[*, 1:nsp_rec-1L] 
           endif
           if (l_setup.l_count_rate eq 'Y') then begin
              (*l_binned.corr_scl_H_e2) = (*l_binned.corr_scl_H_e2)[*, 1:nsp_rec-1L]
           endif
        endif else begin
           if (l_binned_setup.addedStartFlag eq 1L) AND $
              (l_binned_setup.addedStopFlag eq 0L) then begin
              extra_data[*,0] = (*l_binned.m0uePtr)[*, 0]
              extra_data[*,1] = NO_DATA 
              (*l_binned.m0uePtr) = (*l_binned.m0uePtr)[*, 1:nsp_rec] 
              (*l_binned.hydrogen_raw_cnts_e2) = (*l_binned.hydrogen_raw_cnts_e2)[*,1:nsp_rec]
              if (l_setup.l_flux eq 'Y') then begin
                 (*l_binned.corrected_hydrogen_e2) = (*l_binned.corrected_hydrogen_e2)[*, 1:nsp_rec] 
              endif
              if (l_setup.l_count_rate eq 'Y') then begin
                 (*l_binned.corr_scl_H_e2) = (*l_binned.corr_scl_H_e2)[*,1:nsp_rec]
              endif
           endif else begin
              if (l_binned_setup.addedStartFlag eq 0L) AND $
                 (l_binned_setup.addedStopFlag eq 1L) then begin
                 extra_data[*,0] = NO_DATA 
                 extra_data[*,1] = (*l_binned.m0uePtr)[*, nsp_rec]
                 (*l_binned.m0uePtr) = (*l_binned.m0uePtr)[*, 0:nsp_rec-1L] 
                 (*l_binned.hydrogen_raw_cnts_e2) = (*l_binned.hydrogen_raw_cnts_e2)[*,0:nsp_rec-1L]
                 if (l_setup.l_flux eq 'Y') then begin
                    (*l_binned.corrected_hydrogen_e2) = (*l_binned.corrected_hydrogen_e2)[*,0:nsp_rec-1L] 
                 endif
                 if (l_setup.l_count_rate eq 'Y') then begin
                    (*l_binned.corr_scl_H_e2) = (*l_binned.corr_scl_H_e2)[*,0:nsp_rec-1L]
                 endif
              endif
           endelse
        endelse

        if (ptr_valid (l_binned.m0ueExtraPtr)) then ptr_free, l_binned.m0ueExtraPtr
        ptr = ptr_new (extra_data, /NO_COPY)
        l_binned.m0ueExtraPtr = ptr

     endif

     if (l_binned_setup.energy0 eq 'Y') AND (l_binned_setup.energy1 eq 'Y') AND $
        (l_binned_setup.energy2 eq 'Y') then begin

        mass0_bd  = fltarr (l_setup.l_num_spinsec, nspins)
        m0 = m0DataPtr

        ;-------------------------------------------------------------------
        ;for sp = 0, nspins-1 do begin
        ;   for sec = 0, l_setup.l_num_spinsec-1L do begin
        ;      for p = 0, 11 do begin
        ;         for e = 0, 2 do begin
        ;          mass0_bd[sec, sp] = mass0_bd[sec, sp] + m0[e,p,sec,sp]
        ;         endfor
        ;      endfor
        ;   endfor
        ;endfor
        ;-------------------------------------------------------------------
        ; use of total() is equivalent to the 4 for loops
        ;-------------------------------------------------------------------
        temp = total (m0, 1)
        mass0_bd = total (temp, 1)



        if (ptr_valid (l_binned.hydrogen_raw_cnts)) then ptr_free, l_binned.hydrogen_raw_cnts
        ptr = ptr_new (mass0_bd)
        l_binned.hydrogen_raw_cnts = ptr

        if (l_setup.l_flux eq 'Y') OR  (l_binned_setup.background eq 'Y') then begin

           ;--------------------------------------------------
           ; if the event background data is less than the
           ; available binned data, then increase the size of
           ; the event data to subtract a background of zero,
           ; this maintains the number of spins in the binned
           ; data
           ;--------------------------------------------------
           nele_bk = n_elements ((*l_event.background))
           nele_dt = n_elements (mass0_bd)
           if (nele_bk lt nele_dt) then begin
              sp = nele_bk/l_setup.l_num_spinsec
              tmp_bk = fltarr (l_setup.l_num_spinsec, nspins)
              tmp_bk[*,0:sp-1] = (*l_event.background)
              (*l_event.background) = tmp_bk
           endif

           mass0_bd = TEMPORARY (mass0_bd) - ((*l_event.background) * 0.4)
           mass0_bd = mass0_bd > 0.0
           if (ptr_valid (l_binned.corrected_hydrogen)) then ptr_free, l_binned.corrected_hydrogen
           ptr = ptr_new (mass0_bd)
           l_binned.corrected_hydrogen = ptr

        endif


        if (l_setup.l_count_rate eq 'Y') then begin
           l_Conv2CntRate, mass0_bd, mass0_bd, 'SECTOR'
           if (ptr_valid (l_binned.corr_scl_H)) then ptr_free, l_binned.corr_scl_H
           ptr = ptr_new (mass0_bd)
           l_binned.corr_scl_H = ptr
        endif

        if (l_setup.l_flux eq 'Y') then begin
           mass0_bd = (*l_binned.corr_scl_H_e0) + (*l_binned.corr_scl_H_e1) + $
                            (*l_binned.corr_scl_H_e2)
        endif

        if (ptr_valid (l_binned.m0Ptr)) then ptr_free, l_binned.m0Ptr
        ptr = ptr_new (mass0_bd, /NO_COPY)
        l_binned.m0Ptr = ptr


        extra_data = make_array (l_setup.l_num_spinsec, 2L, /FLOAT, Value = NO_DATA)
        if (l_binned_setup.addedStartFlag eq 1L) AND (l_binned_setup.addedStopFlag eq 1L) then begin
           extra_data[*,0] = (*l_binned.m0leExtraPtr)[*,0] + (*l_binned.m0ceExtraPtr)[*,0] + $
                             (*l_binned.m0ueExtraPtr)[*,0]
           extra_data[*,1] = (*l_binned.m0leExtraPtr)[*,1] + (*l_binned.m0ceExtraPtr)[*,1] + $
                             (*l_binned.m0ueExtraPtr)[*,1]
           (*l_binned.hydrogen_raw_cnts) = (*l_binned.hydrogen_raw_cnts)[*,1:nsp_rec-1]
           if (l_setup.l_flux eq 'Y') then begin
              (*l_binned.corrected_hydrogen) = (*l_binned.corrected_hydrogen)[*,1:nsp_rec-1] 
           endif
           if (l_setup.l_count_rate eq 'Y') then begin
              (*l_binned.corr_scl_H) = (*l_binned.corr_scl_H)[*,1:nsp_rec-1]
           endif
        endif else begin
           if (l_binned_setup.addedStartFlag eq 1L) AND $
              (l_binned_setup.addedStopFlag eq 0L) then begin
              extra_data[*,0] = (*l_binned.m0leExtraPtr)[*,0] + (*l_binned.m0ceExtraPtr)[*,0] + $
                             (*l_binned.m0ueExtraPtr)[*,0]
              extra_data[*,1] = NO_DATA 
              (*l_binned.hydrogen_raw_cnts) = (*l_binned.hydrogen_raw_cnts)[*,1:nsp_rec]
              if (l_setup.l_flux eq 'Y') then begin
                 (*l_binned.corrected_hydrogen) = (*l_binned.corrected_hydrogen)[*,1:nsp_rec] 
              endif
              if (l_setup.l_count_rate eq 'Y') then begin
                 (*l_binned.corr_scl_H) = (*l_binned.corr_scl_H)[*,1:nsp_rec]
              endif
           endif else begin
              if (l_binned_setup.addedStartFlag eq 0L) AND $
                 (l_binned_setup.addedStopFlag eq 1L) then begin
                 extra_data[*,0] = NO_DATA 
                 extra_data[*,1] = (*l_binned.m0leExtraPtr)[*,1] + (*l_binned.m0ceExtraPtr)[*,1] + $
                             (*l_binned.m0ueExtraPtr)[*,1]
                 (*l_binned.hydrogen_raw_cnts) = (*l_binned.hydrogen_raw_cnts)[*,0:nsp_rec-1L]
                 if (l_setup.l_flux eq 'Y') then begin
                    (*l_binned.corrected_hydrogen) = (*l_binned.corrected_hydrogen)[*,0:nsp_rec-1L] 
                 endif
                 if (l_setup.l_count_rate eq 'Y') then begin
                    (*l_binned.corr_scl_H) = (*l_binned.corr_scl_H)[*,0:nsp_rec-1L]
                 endif
              endif
           endelse
        endelse

        if (ptr_valid (l_binned.m0ExtraPtr)) then ptr_free, l_binned.m0ExtraPtr
        ptr = ptr_new (extra_data, /NO_COPY)
        l_binned.m0ExtraPtr = ptr

      endif

  endif ;--- end hydrogen ---


  ;----------------------------------------------------
  ;  Oxygen, Mass 1 
  ;----------------------------------------------------

  if (l_binned_setup.oxygen eq 'Y') then begin

     ;----------------------------------------------------
     ;  Oxygen, Mass 1, lower energy bin
     ;----------------------------------------------------

     if (l_binned_setup.energy0 eq 'Y') then begin
        mass1_le  = fltarr (l_setup.l_num_spinsec, nspins)
        m1        = m1DataPtr

        ;-----------------------------------------------------------------
        ;for sp = 0, nspins-1 do begin
        ;   for sec = 0, l_setup.l_num_spinsec-1L do begin
        ;      for p = 0, 11 do begin
        ;          mass1_le[sec, sp] = mass1_le[sec, sp] + m1[0,p,sec,sp]
        ;      endfor
        ;   endfor
        ;endfor
        ;-----------------------------------------------------------------
        ; use of total() is equivalent to use of 3 for loops
        ;-----------------------------------------------------------------
        temp = total (m1[0,*,*,*], 1)
        mass1_le = total (temp, 1)

        if (ptr_valid (l_binned.oxygen_raw_cnts_e0)) then ptr_free, l_binned.oxygen_raw_cnts_e0
        ptr = ptr_new (mass1_le)
        l_binned.oxygen_raw_cnts_e0 = ptr

        if (l_setup.l_flux eq 'Y') OR  (l_binned_setup.background eq 'Y') then begin

           ;--------------------------------------------------
           ; if the event background data is less than the
           ; available binned data, then increase the size of
           ; the event data to subtract a background of zero,
           ; this maintains the number of spins in the binned
           ; data
           ;--------------------------------------------------
           nele_bk = n_elements ((*l_event.background_e0))
           nele_dt = n_elements (mass1_le)
           if (nele_bk lt nele_dt) then begin
              sp = nele_bk/l_setup.l_num_spinsec
              tmp_bk = fltarr (l_setup.l_num_spinsec, nspins)
              tmp_bk[*,0:sp-1] = (*l_event.background_e0)
              (*l_event.background_e0) = tmp_bk
           endif

           mass1_le = TEMPORARY (mass1_le) - ((*l_event.background_e0) * 6.0)
           mass1_le = mass1_le > 0.0
           if (ptr_valid (l_binned.corrected_oxygen_e0)) then ptr_free, l_binned.corrected_oxygen_e0
           ptr = ptr_new (mass1_le)
           l_binned.corrected_oxygen_e0 = ptr

        endif

        if (l_setup.l_count_rate eq 'Y') then begin
           l_Conv2CntRate, mass1_le, mass1_le, 'SECTOR'
           if (ptr_valid (l_binned.corr_scl_O_e0)) then ptr_free, l_binned.corr_scl_O_e0
           ptr = ptr_new (mass1_le)
           l_binned.corr_scl_O_e0 = ptr
        endif

        if (l_setup.l_flux eq 'Y') then begin
           l_ConvFlux, mass1_le, mass1_le, 'oxygen', $
                       l_binned_setup.oe0_eV, l_setup.l_start_dtime, $
                       state, nom_eff, eff
           if (l_setup.l_dumps eq 'Y') then begin
              l_WriteEff2File, flux_ofd, 'oxygen', l_binned_setup.oe0_eV, $
                           state, nom_eff, eff
           endif
        endif

        if (ptr_valid (l_binned.m1lePtr)) then ptr_free, l_binned.m1lePtr
        ptr = ptr_new (mass1_le, /NO_COPY)
        l_binned.m1lePtr = ptr


        extra_data = make_array (l_setup.l_num_spinsec, 2L, /FLOAT, Value = NO_DATA)
        if (l_binned_setup.addedStartFlag eq 1L) AND (l_binned_setup.addedStopFlag eq 1L) then begin
           extra_data[*,0] = (*l_binned.m1lePtr)[*, 0]
           extra_data[*,1] = (*l_binned.m1lePtr)[*, nsp_rec]
           (*l_binned.m1lePtr) = (*l_binned.m1lePtr)[*, 1:nsp_rec-1L] 
           (*l_binned.oxygen_raw_cnts_e0) = (*l_binned.oxygen_raw_cnts_e0)[*,1:nsp_rec-1] 
           if (l_setup.l_flux eq 'Y') then begin
              (*l_binned.corrected_oxygen_e0) = (*l_binned.corrected_oxygen_e0)[*,1:nsp_rec-1] 
           endif
           if (l_setup.l_count_rate eq 'Y') then begin
              (*l_binned.corr_scl_O_e0) = (*l_binned.corr_scl_O_e0)[*,1:nsp_rec-1] 
           endif
        endif else begin
           if (l_binned_setup.addedStartFlag eq 1L) AND $
              (l_binned_setup.addedStopFlag eq 0L) then begin
              extra_data[*,0] = (*l_binned.m1lePtr)[*, 0]
              extra_data[*,1] = NO_DATA 
              (*l_binned.m1lePtr) = (*l_binned.m1lePtr)[*, 1:nsp_rec] 
              (*l_binned.oxygen_raw_cnts_e0) = (*l_binned.oxygen_raw_cnts_e0)[*,1:nsp_rec] 
              if (l_setup.l_flux eq 'Y') then begin
                 (*l_binned.corrected_oxygen_e0) = (*l_binned.corrected_oxygen_e0)[*,1:nsp_rec] 
              endif
              if (l_setup.l_count_rate eq 'Y') then begin
                 (*l_binned.corr_scl_O_e0) = (*l_binned.corr_scl_O_e0)[*,1:nsp_rec] 
              endif
           endif else begin
              if (l_binned_setup.addedStartFlag eq 0L) AND $
                 (l_binned_setup.addedStopFlag eq 1L) then begin
                 extra_data[*,0] = NO_DATA 
                 extra_data[*,1] = (*l_binned.m1lePtr)[*, nsp_rec]
                 (*l_binned.m1lePtr) = (*l_binned.m1lePtr)[*, 0:nsp_rec-1L] 
                 (*l_binned.oxygen_raw_cnts_e0) = (*l_binned.oxygen_raw_cnts_e0)[*,0:nsp_rec-1L] 
                 if (l_setup.l_flux eq 'Y') then begin
                    (*l_binned.corrected_oxygen_e0) = (*l_binned.corrected_oxygen_e0)[*,0:nsp_rec-1L] 
                 endif
                 if (l_setup.l_count_rate eq 'Y') then begin
                    (*l_binned.corr_scl_O_e0) = (*l_binned.corr_scl_O_e0)[*,0:nsp_rec-1L] 
                 endif
              endif
           endelse
        endelse

        if (ptr_valid (l_binned.m1leExtraPtr)) then ptr_free, l_binned.m1leExtraPtr
        ptr = ptr_new (extra_data, /NO_COPY)
        l_binned.m1leExtraPtr = ptr


     endif


     ;----------------------------------------------------
     ;  Oxygen, Mass 1, center energy bin
     ;----------------------------------------------------

     if (l_binned_setup.energy1 eq 'Y') then begin
        mass1_ce  = fltarr (l_setup.l_num_spinsec, nspins)
        m1        = m1DataPtr

        ;---------------------------------------------------------------
        ;for sp = 0, nspins-1 do begin
        ;   for sec = 0, l_setup.l_num_spinsec-1L do begin
        ;      for p = 0, 11 do begin
        ;          mass1_ce[sec, sp] = mass1_ce[sec, sp] + m1[1,p,sec,sp]
        ;      endfor
        ;   endfor
        ;endfor
        ;---------------------------------------------------------------
        ; use of total() is equivalent to use of 3 for loops
        ;---------------------------------------------------------------
        temp = total (m1[1,*,*,*], 1)
        mass1_ce = total (temp, 1)

        if (ptr_valid (l_binned.oxygen_raw_cnts_e1)) then ptr_free, l_binned.oxygen_raw_cnts_e1
        ptr = ptr_new (mass1_ce)
        l_binned.oxygen_raw_cnts_e1 = ptr
       
        if (l_setup.l_flux eq 'Y') OR  (l_binned_setup.background eq 'Y') then begin

           ;--------------------------------------------------
           ; if the event background data is less than the
           ; available binned data, then increase the size of
           ; the event data to subtract a background of zero,
           ; this maintains the number of spins in the binned
           ; data
           ;--------------------------------------------------
           nele_bk = n_elements ((*l_event.background_e1))
           nele_dt = n_elements (mass1_ce)
           if (nele_bk lt nele_dt) then begin
              sp = nele_bk/l_setup.l_num_spinsec
              tmp_bk = fltarr (l_setup.l_num_spinsec, nspins)
              tmp_bk[*,0:sp-1] = (*l_event.background_e1)
              (*l_event.background_e1) = tmp_bk
           endif

           mass1_ce = TEMPORARY (mass1_ce) - ((*l_event.background_e1) * 6.0)
           mass1_ce = mass1_ce > 0.0
           if (ptr_valid (l_binned.corrected_oxygen_e1)) then ptr_free, l_binned.corrected_oxygen_e1
           ptr = ptr_new (mass1_ce)
           l_binned.corrected_oxygen_e1 = ptr

        endif

        if (l_setup.l_count_rate eq 'Y') then begin
           l_Conv2CntRate, mass1_ce, mass1_ce, 'SECTOR'
           if (ptr_valid (l_binned.corr_scl_O_e1)) then ptr_free, l_binned.corr_scl_O_e1
           ptr = ptr_new (mass1_ce)
           l_binned.corr_scl_O_e1 = ptr
        endif

        if (l_setup.l_flux eq 'Y') then begin
           l_ConvFlux, mass1_ce, mass1_ce, 'oxygen', $
                       l_binned_setup.oe1_eV, l_setup.l_start_dtime, $
                       state, nom_eff, eff 
           if (l_setup.l_dumps eq 'Y') then begin
              l_WriteEff2File, flux_ofd, 'oxygen', l_binned_setup.oe1_eV, $
                           state, nom_eff, eff
           endif
 
        endif

        if (ptr_valid (l_binned.m1cePtr)) then ptr_free, l_binned.m1cePtr
        ptr = ptr_new (mass1_ce, /NO_COPY)
        l_binned.m1cePtr = ptr


        extra_data = make_array (l_setup.l_num_spinsec, 2L, /FLOAT, Value = NO_DATA)
        if (l_binned_setup.addedStartFlag eq 1L) AND (l_binned_setup.addedStopFlag eq 1L) then begin
           extra_data[*,0] = (*l_binned.m1cePtr)[*, 0]
           extra_data[*,1] = (*l_binned.m1cePtr)[*, nsp_rec]
           (*l_binned.m1cePtr) = (*l_binned.m1cePtr)[*, 1:nsp_rec-1L] 
           (*l_binned.oxygen_raw_cnts_e1) = (*l_binned.oxygen_raw_cnts_e1)[*,1:nsp_rec-1] 
           if (l_setup.l_flux eq 'Y') then begin
              (*l_binned.corrected_oxygen_e1) = (*l_binned.corrected_oxygen_e1)[*,1:nsp_rec-1] 
           endif
           if (l_setup.l_count_rate eq 'Y') then begin
              (*l_binned.corr_scl_O_e1) = (*l_binned.corr_scl_O_e1)[*,1:nsp_rec-1] 
           endif
        endif else begin
           if (l_binned_setup.addedStartFlag eq 1L) AND $
              (l_binned_setup.addedStopFlag eq 0L) then begin
              extra_data[*,0] = (*l_binned.m1cePtr)[*, 0]
              extra_data[*,1] = NO_DATA 
              (*l_binned.m1cePtr) = (*l_binned.m1cePtr)[*, 1:nsp_rec] 
              (*l_binned.oxygen_raw_cnts_e1) = (*l_binned.oxygen_raw_cnts_e1)[*,1:nsp_rec] 
              if (l_setup.l_flux eq 'Y') then begin
                 (*l_binned.corrected_oxygen_e1) = (*l_binned.corrected_oxygen_e1)[*,1:nsp_rec] 
              endif
              if (l_setup.l_count_rate eq 'Y') then begin
                 (*l_binned.corr_scl_O_e1) = (*l_binned.corr_scl_O_e1)[*,1:nsp_rec] 
              endif
           endif else begin
              if (l_binned_setup.addedStartFlag eq 0L) AND $
                 (l_binned_setup.addedStopFlag eq 1L) then begin
                 extra_data[*,0] = NO_DATA 
                 extra_data[*,1] = (*l_binned.m1cePtr)[*, nsp_rec]
                 (*l_binned.m1cePtr) = (*l_binned.m1cePtr)[*, 0:nsp_rec-1L] 
                 (*l_binned.oxygen_raw_cnts_e1) = (*l_binned.oxygen_raw_cnts_e1)[*,0:nsp_rec-1L] 
                 if (l_setup.l_flux eq 'Y') then begin
                    (*l_binned.corrected_oxygen_e1) = (*l_binned.corrected_oxygen_e1)[*,0:nsp_rec-1L] 
                 endif
                 if (l_setup.l_count_rate eq 'Y') then begin
                    (*l_binned.corr_scl_O_e1) = (*l_binned.corr_scl_O_e1)[*,0:nsp_rec-1L] 
                 endif
              endif
           endelse
        endelse

        if (ptr_valid (l_binned.m1ceExtraPtr)) then ptr_free, l_binned.m1ceExtraPtr
        ptr = ptr_new (extra_data, /NO_COPY)
        l_binned.m1ceExtraPtr = ptr

     endif


     ;----------------------------------------------------
     ;  Oxygen, Mass 1, upper energy bin
     ;----------------------------------------------------

     if (l_binned_setup.energy2 eq 'Y') then begin
        mass1_ue  = fltarr (l_setup.l_num_spinsec, nspins)
        m1        = m1DataPtr

        ;----------------------------------------------------------------
        ;for sp = 0, nspins-1 do begin
        ;   for sec = 0, l_setup.l_num_spinsec-1L do begin
        ;      for p = 0, 11 do begin
        ;          mass1_ue[sec, sp] = mass1_ue[sec, sp] + m1[2,p,sec,sp]
        ;      endfor
        ;   endfor
        ;endfor
        ;----------------------------------------------------------------
        ; use of total() is equivalent to te 3 for loops
        ;----------------------------------------------------------------
        temp = total (m1[2,*,*,*], 1)
        mass1_ue = total (temp, 1)

        if (ptr_valid (l_binned.oxygen_raw_cnts_e2)) then ptr_free, l_binned.oxygen_raw_cnts_e2
        ptr = ptr_new (mass1_ue)
        l_binned.oxygen_raw_cnts_e2 = ptr

        if (l_setup.l_flux eq 'Y') OR (l_binned_setup.background eq 'Y') then begin

           ;--------------------------------------------------
           ; if the event background data is less than the
           ; available binned data, then increase the size of
           ; the event data to subtract a background of zero,
           ; this maintains the number of spins in the binned
           ; data
           ;--------------------------------------------------
           nele_bk = n_elements ((*l_event.background_e2))
           nele_dt = n_elements (mass1_ue)
           if (nele_bk lt nele_dt) then begin
              sp = nele_bk/l_setup.l_num_spinsec
              tmp_bk = fltarr (l_setup.l_num_spinsec, nspins)
              tmp_bk[*,0:sp-1] = (*l_event.background_e2)
              (*l_event.background_e2) = tmp_bk
           endif

           mass1_ue = TEMPORARY (mass1_ue) - ((*l_event.background_e2) * 6.0)
           mass1_ue = mass1_ue > 0.0
           if (ptr_valid (l_binned.corrected_oxygen_e2)) then ptr_free, l_binned.corrected_oxygen_e2
           ptr = ptr_new (mass1_ue)
           l_binned.corrected_oxygen_e2 = ptr

        endif

        if (l_setup.l_count_rate eq 'Y') then begin
           l_Conv2CntRate, mass1_ue, mass1_ue, 'SECTOR'
           if (ptr_valid (l_binned.corr_scl_O_e2)) then ptr_free, l_binned.corr_scl_O_e2
           ptr = ptr_new (mass1_ue)
           l_binned.corr_scl_O_e2 = ptr
        endif

        if (l_setup.l_flux eq 'Y') then begin
           l_ConvFlux, mass1_ue, mass1_ue, 'oxygen', $
                       l_binned_setup.oe2_eV, l_setup.l_start_dtime, $
                       state, nom_eff, eff
           if (l_setup.l_dumps eq 'Y') then begin
              l_WriteEff2File, flux_ofd, 'oxygen', l_binned_setup.oe2_eV, $
                           state, nom_eff, eff
           endif
        endif

        if (ptr_valid (l_binned.m1uePtr)) then ptr_free, l_binned.m1uePtr
        ptr = ptr_new (mass1_ue, /NO_COPY)
        l_binned.m1uePtr = ptr


        extra_data = make_array (l_setup.l_num_spinsec, 2L, /FLOAT, Value = NO_DATA)
        if (l_binned_setup.addedStartFlag eq 1L) AND (l_binned_setup.addedStopFlag eq 1L) then begin
           extra_data[*,0] = (*l_binned.m1uePtr)[*, 0]
           extra_data[*,1] = (*l_binned.m1uePtr)[*, nsp_rec]
           (*l_binned.m1uePtr) = (*l_binned.m1uePtr)[*, 1:nsp_rec-1L] 
           (*l_binned.oxygen_raw_cnts_e2) = (*l_binned.oxygen_raw_cnts_e2)[*,1:nsp_rec-1] 
           if (l_setup.l_flux eq 'Y') then begin
              (*l_binned.corrected_oxygen_e2) = (*l_binned.corrected_oxygen_e2)[*,1:nsp_rec-1] 
           endif
           if (l_setup.l_count_rate eq 'Y') then begin
              (*l_binned.corr_scl_O_e2) = (*l_binned.corr_scl_O_e2)[*,1:nsp_rec-1] 
           endif
        endif else begin
           if (l_binned_setup.addedStartFlag eq 1L) AND $
              (l_binned_setup.addedStopFlag eq 0L) then begin
              extra_data[*,0] = (*l_binned.m1uePtr)[*, 0]
              extra_data[*,1] = NO_DATA 
              (*l_binned.m1uePtr) = (*l_binned.m1uePtr)[*, 1:nsp_rec] 
              (*l_binned.oxygen_raw_cnts_e2) = (*l_binned.oxygen_raw_cnts_e2)[*,1:nsp_rec] 
              if (l_setup.l_flux eq 'Y') then begin
                 (*l_binned.corrected_oxygen_e2) = (*l_binned.corrected_oxygen_e2)[*,1:nsp_rec] 
              endif
              if (l_setup.l_count_rate eq 'Y') then begin
                 (*l_binned.corr_scl_O_e2) = (*l_binned.corr_scl_O_e2)[*,1:nsp_rec] 
              endif
           endif else begin
              if (l_binned_setup.addedStartFlag eq 0L) AND $
                 (l_binned_setup.addedStopFlag eq 1L) then begin
                 extra_data[*,0] = NO_DATA 
                 extra_data[*,1] = (*l_binned.m1uePtr)[*, nsp_rec]
                 (*l_binned.m1uePtr) = (*l_binned.m1uePtr)[*, 0:nsp_rec-1L] 
                 (*l_binned.oxygen_raw_cnts_e2) = (*l_binned.oxygen_raw_cnts_e2)[*,0:nsp_rec-1L] 
                 if (l_setup.l_flux eq 'Y') then begin
                    (*l_binned.corrected_oxygen_e2) = (*l_binned.corrected_oxygen_e2)[*,0:nsp_rec-1L] 
                 endif
                 if (l_setup.l_count_rate eq 'Y') then begin
                    (*l_binned.corr_scl_O_e2) = (*l_binned.corr_scl_O_e2)[*,0:nsp_rec-1L] 
                 endif
              endif 
           endelse
        endelse

        if (ptr_valid (l_binned.m1ueExtraPtr)) then ptr_free, l_binned.m1ueExtraPtr
        ptr = ptr_new (extra_data, /NO_COPY)
        l_binned.m1ueExtraPtr = ptr

     endif


     if (l_binned_setup.energy0 eq 'Y') AND (l_binned_setup.energy1 eq 'Y') AND $
        (l_binned_setup.energy2 eq 'Y') then begin
        mass1_bd  = fltarr (l_setup.l_num_spinsec, nspins)
        m1 = m1DataPtr


        ;------------------------------------------------------------------------
        ;for sp = 0, nspins-1 do begin
        ;   for sec = 0, l_setup.l_num_spinsec-1L do begin
        ;      for p = 0, 11 do begin
        ;         for e = 0, 2 do begin
        ;          mass1_bd[sec, sp] = mass1_bd[sec, sp] + m1[e,p,sec,sp]
        ;         endfor
        ;      endfor
        ;   endfor
        ;endfor
        ;------------------------------------------------------------------------
        ; use of total() is equivalent to the 3 for loops
        ;------------------------------------------------------------------------
        temp = total (m1, 1)
        mass1_bd = total (temp, 1)
    

        if (ptr_valid (l_binned.oxygen_raw_cnts)) then ptr_free, l_binned.oxygen_raw_cnts
        ptr = ptr_new (mass1_bd)
        l_binned.oxygen_raw_cnts = ptr

        if (l_setup.l_flux eq 'Y') OR (l_binned_setup.background eq 'Y') then begin

           ;--------------------------------------------------
           ; if the event background data is less than the
           ; available binned data, then increase the size of
           ; the event data to subtract a background of zero,
           ; this maintains the number of spins in the binned
           ; data
           ;--------------------------------------------------
           nele_bk = n_elements ((*l_event.background))
           nele_dt = n_elements (mass1_bd)
           if (nele_bk lt nele_dt) then begin
              sp = nele_bk/l_setup.l_num_spinsec
              tmp_bk = fltarr (l_setup.l_num_spinsec, nspins)
              tmp_bk[*,0:sp-1] = (*l_event.background)
              (*l_event.background) = tmp_bk
           endif

           mass1_bd = TEMPORARY (mass1_bd) - ((*l_event.background) * 6.0)
           mass1_bd = mass1_bd > 0.0
           if (ptr_valid (l_binned.corrected_oxygen)) then ptr_free, l_binned.corrected_oxygen
           ptr = ptr_new (mass1_bd)
           l_binned.corrected_oxygen = ptr

        endif

        if (l_setup.l_count_rate eq 'Y') then begin
           l_Conv2CntRate, mass1_bd, mass1_bd, 'SECTOR'
           if (ptr_valid (l_binned.corr_scl_O)) then ptr_free, l_binned.corr_scl_O
           ptr = ptr_new (mass1_bd)
           l_binned.corr_scl_O = ptr
        endif

        if (l_setup.l_flux eq 'Y') then begin
           mass1_bd = (*l_binned.corr_scl_O_e0) + (*l_binned.corr_scl_O_e1) + $
                      (*l_binned.corr_scl_O_e2)
        endif

        if (ptr_valid (l_binned.m1Ptr)) then ptr_free, l_binned.m1Ptr
        ptr = ptr_new (mass1_bd, /NO_COPY)
        l_binned.m1Ptr = ptr

        extra_data = make_array (l_setup.l_num_spinsec, 2L, /FLOAT, Value = NO_DATA)
        if (l_binned_setup.addedStartFlag eq 1L) AND (l_binned_setup.addedStopFlag eq 1L) then begin
           extra_data[*,0] = (*l_binned.m1leExtraPtr)[*,0] + (*l_binned.m1ceExtraPtr)[*,0] + $
                             (*l_binned.m1ueExtraPtr)[*,0]
           extra_data[*,1] = (*l_binned.m1leExtraPtr)[*,1] + (*l_binned.m1ceExtraPtr)[*,1] + $
                             (*l_binned.m1ueExtraPtr)[*,1]
           (*l_binned.oxygen_raw_cnts) = (*l_binned.oxygen_raw_cnts)[*,1:nsp_rec-1] 
           if (l_setup.l_flux eq 'Y') then begin
              (*l_binned.corrected_oxygen) = (*l_binned.corrected_oxygen)[*,1:nsp_rec-1] 
           endif
           if (l_setup.l_count_rate eq 'Y') then begin
              (*l_binned.corr_scl_O) = (*l_binned.corr_scl_O)[*,1:nsp_rec-1] 
           endif
        endif else begin
           if (l_binned_setup.addedStartFlag eq 1L) AND $
              (l_binned_setup.addedStopFlag eq 0L) then begin
              extra_data[*,0] = (*l_binned.m1leExtraPtr)[*,0] + (*l_binned.m1ceExtraPtr)[*,0] + $
                             (*l_binned.m1ueExtraPtr)[*,0]
              extra_data[*,1] = NO_DATA 
              (*l_binned.oxygen_raw_cnts) = (*l_binned.oxygen_raw_cnts)[*,1:nsp_rec] 
              if (l_setup.l_flux eq 'Y') then begin
                 (*l_binned.corrected_oxygen) = (*l_binned.corrected_oxygen)[*,1:nsp_rec] 
              endif
              if (l_setup.l_count_rate eq 'Y') then begin
                 (*l_binned.corr_scl_O) = (*l_binned.corr_scl_O)[*,1:nsp_rec] 
              endif
           endif else begin
              if (l_binned_setup.addedStartFlag eq 0L) AND $
                 (l_binned_setup.addedStopFlag eq 1L) then begin
                 extra_data[*,0] = NO_DATA 
                 extra_data[*,1] = (*l_binned.m1leExtraPtr)[*,1] + (*l_binned.m1ceExtraPtr)[*,1] + $
                             (*l_binned.m1ueExtraPtr)[*,1]
                 (*l_binned.oxygen_raw_cnts) = (*l_binned.oxygen_raw_cnts)[*,0:nsp_rec-1L] 
                 if (l_setup.l_flux eq 'Y') then begin
                    (*l_binned.corrected_oxygen) = (*l_binned.corrected_oxygen)[*,0:nsp_rec-1L] 
                 endif
                 if (l_setup.l_count_rate eq 'Y') then begin
                    (*l_binned.corr_scl_O) = (*l_binned.corr_scl_O)[*,0:nsp_rec-1L] 
                 endif
              endif
           endelse
        endelse

        if (ptr_valid (l_binned.m1ExtraPtr)) then ptr_free, l_binned.m1ExtraPtr
        ptr = ptr_new (extra_data, /NO_COPY)
        l_binned.m1ExtraPtr = ptr

     endif
  endif ;--- end oxygen ---


  ;----------------------------------------------------
  ; sum binned data over energy, mass, and polar zone
  ; to obtain the sum value for each spin sector
  ; these are the coincidences
  ;----------------------------------------------------
  if (l_binned_setup.hydrogen eq 'Y') AND (l_binned_setup.oxygen eq 'Y') then begin
     if (l_binned_setup.energy0 eq 'Y') AND (l_binned_setup.energy1 eq 'Y') AND $
        (l_binned_setup.energy2 eq 'Y') then begin

        ;all = fltarr (l_setup.l_num_spinsec, nspins)
        ;mass = m0DataPtr + m1DataPtr
        ;for sp = 0, nspins-1 do begin
        ;   for sec = 0, l_setup.l_num_spinsec-1L do begin
        ;      for p = 0, 11 do begin
        ;         for e = 0, 2 do begin
        ;          all [sec, sp] = all [sec, sp] + mass[e,p,sec,sp]
        ;         endfor
        ;      endfor
        ;   endfor
        ;endfor

        comb_bd   = (*l_binned.m0lePtr) + (*l_binned.m0cePtr) + (*l_binned.m0uePtr) + $
                    (*l_binned.m1lePtr) + (*l_binned.m1cePtr) + (*l_binned.m1uePtr)


        ptr = ptr_new (comb_bd, /NO_COPY)
        if (ptr_valid(l_binned.vePtr)) then ptr_free, l_binned.vePtr
        l_binned.vePtr = ptr

        extra_data = make_array (l_setup.l_num_spinsec, 2L, /FLOAT, Value = NO_DATA)
        if (l_binned_setup.addedStartFlag eq 1L) AND (l_binned_setup.addedStopFlag eq 1L) then begin
            extra_data[*,0] = (*l_binned.m0leExtraPtr)[*,0] + (*l_binned.m0ceExtraPtr)[*,0] + $
                              (*l_binned.m0ueExtraPtr)[*,0] + (*l_binned.m1leExtraPtr)[*,0] + $
                              (*l_binned.m1ceExtraptr)[*,0] + (*l_binned.m1ueExtraPtr)[*,0]
            extra_data[*,1] = (*l_binned.m0leExtraPtr)[*,1] + (*l_binned.m0ceExtraPtr)[*,1] + $
                              (*l_binned.m0ueExtraPtr)[*,1] + (*l_binned.m1leExtraPtr)[*,1] + $
                              (*l_binned.m1ceExtraPtr)[*,1] + (*l_binned.m1ueExtraPtr)[*,1]
        endif else begin
            if (l_binned_setup.addedStartFlag eq 1L) AND $
               (l_binned_setup.addedStopFlag eq 0L) then begin
               extra_data[*,0] = (*l_binned.m0leExtraPtr)[*,0] + (*l_binned.m0ceExtraPtr)[*,0] + $
                              (*l_binned.m0ueExtraPtr)[*,0] + (*l_binned.m1leExtraPtr)[*,0] + $
                              (*l_binned.m1ceExtraPtr)[*,0] + (*l_binned.m1ueExtraPtr)[*,0]
               extra_data[*,1] = NO_DATA
            endif else begin
               if (l_binned_setup.addedStartFlag eq 0L) AND $
                  (l_binned_setup.addedStopFlag eq 1L) then begin
                  extra_data[*,0] = NO_DATA
                  extra_data[*,1] = (*l_binned.m0leExtraPtr)[*,1] + (*l_binned.m0ceExtraPtr)[*,1] + $
                              (*l_binned.m0ueExtraPtr)[*,1] + (*l_binned.m1leExtraPtr)[*,1] + $
                              (*l_binned.m1ceExtraPtr)[*,1] + (*l_binned.m1ueExtraPtr)[*,1]
               endif
            endelse
        endelse

        if (ptr_valid (l_binned.veExtraPtr)) then ptr_free, l_binned.veExtraPtr
        ptr = ptr_new (extra_data, /NO_COPY)
        l_binned.veExtraPtr = ptr


     endif
  endif

     if (l_setup.l_flux eq 'Y') then begin
     nele = (n_elements ((*l_event.background))/45) - 1
     if (l_binned_setup.addedStartFlag eq 1L) AND (l_binned_setup.addedStopFlag eq 1L) then begin
       (*l_event.background)    = (*l_event.background)[*,1:nele-1L]
       (*l_event.background_e0) = (*l_event.background_e0)[*,1:nele-1L]
       (*l_event.background_e1) = (*l_event.background_e1)[*,1:nele-1L] 
       (*l_event.background_e2) = (*l_event.background_e2)[*,1:nele-1L] 
     endif else begin
       if (l_binned_setup.addedStartFlag eq 1L) AND (l_binned_setup.addedStopFlag eq 0L) then begin
          (*l_event.background)    = (*l_event.background)[*,1:nele]
          (*l_event.background_e0) = (*l_event.background_e0)[*,1:nele]
          (*l_event.background_e1) = (*l_event.background_e1)[*,1:nele] 
          (*l_event.background_e2) = (*l_event.background_e2)[*,1:nele] 
       endif else begin
         if (l_binned_setup.addedStartFlag eq 0L) AND $
            (l_binned_setup.addedStopFlag eq 1L) then begin
            (*l_event.background)    = (*l_event.background)[*,0:nele-1]
            (*l_event.background_e0) = (*l_event.background_e0)[*,0:nele-1]
            (*l_event.background_e1) = (*l_event.background_e1)[*,0:nele-1] 
            (*l_event.background_e2) = (*l_event.background_e2)[*,0:nele-1] 
         endif
       endelse
    endelse
    endif


  endelse

  if (l_setup.l_flux eq 'Y') AND (l_setup.l_dumps eq 'Y') then begin
     free_lun, flux_ofd
     close, flux_ofd
  endif


RETURN
END

;-------------------------------------------------------------------------------
;  Function:  l_BinnedGetData
;
;  Description: 
;     Read the binned/image data packets given time range and set up the 
;     data to point to the correct pointers in the structure.
;
;  Return Value:  type = int
;  Value                         Description
;  -------------------------     ------------------------------
;  SUCCESS                       success completion
;  FAILURE                       failure to complete
; 
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_setup               struct        I        setup parameters
;  l_setup_file          string        I        setup filename
;  l_pltpos              struct        I        plot specific parameters
;  l_binned_setup        struct        I/O      setup parameters specific
;                                               to binned/image data options
;  l_binned              struct *      I/O      structure of pointers to the
;                                               binned data
;  nrec                  long          I        number of records 
; 
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;     PLOTDEFS         MAX_PLOTS        int         G         max # plots per page
;                      MAX_WEB_SPINS    int         G         max # spins per plot
;                                                             when using web interface
;                      NO_CNTS          double      G         no counts indicator
;                      NO_DATA          double      G         no data indicator
;
;      ERROR_TYPES     SUCCESS         int         G          successful completion
;                      FAILURE         int         G          unsuccessful completion
;                      WARNING         int         G          warning message
;                      INFO            int         G          informational message
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;  ELH               08/31/01        v1.0.7           Add extra time to read and
;                                                     correctly adjust arrays 
;  ELH               11/02/01        v1.0.7           Removed l_BinnedReadSVL,
;                                                     redundant here since it is
;                                                     called in l_BinnedReadData
;-------------------------------------------------------------------------------

FUNCTION l_BinnedGetData, l_setup, l_setup_file, l_pltpos, $
l_binned_setup, l_binned, nrec
COMMON PLOTDEFS, MAX_PLOTS, MAX_WEB_SPINS, NO_CNTS, NO_DATA
COMMON ERROR_TYPES, SUCCESS, FAILURE, WARNING, INFO


  ;----------------------------------------------------
  ; read the udf data
  ;----------------------------------------------------
  start_tme = l_setup.l_start_dtime
  stop_tme = l_setup.l_stop_dtime

  ;----------------------------------------------------
  ; if the start time is not 00:00:00, then add extra
  ; minutes to the begin time
  ;----------------------------------------------------
  l_binned_setup.addedStartFlag = 0L
  if (ChckStartTime(start_tme) eq FAILURE) then begin
      AddExtraTime2Start, start_tme, 2
      l_binned_setup.addedStartFlag = 1L
  endif

  ;----------------------------------------------------
  ; if the stop time is not 23:59:59, then add extra
  ; minutes to the end time
  ;----------------------------------------------------
  l_binned_setup.addedStopFlag = 0L
  if (ChckStopTime(stop_tme) eq FAILURE) then begin
      ;AddExtraTime2Stop, stop_tme, 6 
      ;AddExtraTime2Stop, stop_tme, 4 
      AddExtraTime2Stop, stop_tme, 2 
      l_binned_setup.addedStopFlag = 1L
  endif


  ;----------------------------------------------------
  ; read data in time format
  ;----------------------------------------------------
  l_binned_setup.l_binned_plot_type = 'T'
  status = l_BinnedReadData (l_setup, l_binned_setup, nrec, $
                             l_binned, l_pltpos, start_tme, stop_tme)


  if (status eq 0) then begin
     RETURN, SUCCESS 
  endif else begin
     RETURN, FAILURE
  endelse


END


;--------------------------------------------------------------------------------
;  Procedure: l_BinnedDmpSpectData
;
;  Description:
;      Generates an ASCII dump of the binned data. The contents of the file 
;   depends on the user's options of which data to see.  The dumps are of
;   the "raw" data.  No conversions or processing done to the data.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  filedes               long          I        file descriptor
;  l_binned              struct *      I        structure of pointers to the
;                                               binned data
;  l_binned_setup        struct        I        structure of binned specific
;                                               parameters
;  l_roi                 struct *      I        structure of pointers to the
;                                               ROI0 data
;  l_setup               struct        I        setup parameters
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;--------------------------------------------------------------------------------

PRO l_BinnedDmpSpectData, filedes, l_binned, l_binned_setup, l_roi, l_setup


   nitems = n_elements (*l_binned.filledStartUdfTimePtr)

   if (l_binned_setup.merge eq 'Y') then begin
       if (l_binned_setup.hydrogen eq 'Y') AND (l_binned_setup.oxygen eq 'Y') then begin
          title = 'Binned'
       endif else begin
         if (l_binned_setup.hydrogen eq 'Y') then begin
            title = 'Binned H'
            if (l_binned_setup.energy0 eq 'Y') then title = title + ' E0'
            if (l_binned_setup.energy1 eq 'Y') then title = title + ' E1'
            if (l_binned_setup.energy2 eq 'Y') then title = title + ' E2'
         endif else begin 
             if (l_binned_setup.oxygen eq 'Y') then begin
                title = 'Binned O'
                if (l_binned_setup.energy0 eq 'Y') then title = title + ' E0'
                if (l_binned_setup.energy1 eq 'Y') then title = title + ' E1'
                if (l_binned_setup.energy2 eq 'Y') then title = title + ' E2'
             endif
         endelse
       endelse
       
       l_DumpData, filedes, title, (*l_binned.veDmpPtr), (*l_binned.filledStartUdfTimePtr), $
                   nitems, l_setup.l_num_spinsec, l_setup
   endif else begin
       if (l_binned_setup.hydrogen eq 'Y') then begin
          if (l_binned_setup.energy0 eq 'Y') then begin
             title = 'Binned H E0' 
             l_DumpData, filedes, title, (*l_binned.m0leDmpPtr), (*l_binned.filledStartUdfTimePtr), $
                   nitems, l_setup.l_num_spinsec, l_setup
          endif
          if (l_binned_setup.energy1 eq 'Y') then begin
             title = 'Binned H E1' 
             l_DumpData, filedes, title, (*l_binned.m0ceDmpPtr), (*l_binned.filledStartUdfTimePtr), $
                   nitems, l_setup.l_num_spinsec, l_setup
          endif
          if (l_binned_setup.energy2 eq 'Y') then begin
             title = 'Binned H E2' 
             l_DumpData, filedes, title, (*l_binned.m0ueDmpPtr), (*l_binned.filledStartUdfTimePtr), $
                   nitems, l_setup.l_num_spinsec, l_setup
          endif
       endif
       if (l_binned_setup.oxygen eq 'Y') then begin
          if (l_binned_setup.energy0 eq 'Y') then begin
             title = 'Binned O E0' 
             l_DumpData, filedes, title, (*l_binned.m1leDmpPtr), (*l_binned.filledStartUdfTimePtr), $
                   nitems, l_setup.l_num_spinsec, l_setup
          endif
          if (l_binned_setup.energy1 eq 'Y') then begin
             title = 'Binned O E1' 
             l_DumpData, filedes, title, (*l_binned.m1ceDmpPtr), (*l_binned.filledStartUdfTimePtr), $
                   nitems, l_setup.l_num_spinsec, l_setup
          endif
          if (l_binned_setup.energy2 eq 'Y') then begin
             title = 'Binned O E2' 
             l_DumpData, filedes, title, (*l_binned.m1ueDmpPtr), (*l_binned.filledStartUdfTimePtr), $
                   nitems, l_setup.l_num_spinsec, l_setup
          endif
       endif
   endelse ;--- merge ---


RETURN
END

;-------------------------------------------------------------------------------
;  Function/Procedure: l_BinnedBKHeader
;
;  Description: Generate a generic header for the background corrected binned data
;        for text dumps.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  type                  string        I        hydrogen or oxygen
;  filedes               long          I        output file descriptor 
;  title                 string        I        text dump title
;  zero_title            string        I        text dump title
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------
PRO l_BinnedBKHeader, type, filedes, title, zero_title


      printf, filedes, ' '
      printf, filedes, title
      printf, filedes, zero_title
      printf, filedes, ' '

      printf, filedes,'UTC','Azi', 'Raw', 'Bkgrnd', 'Bkgrnd', $
                      '  Corr/Scl', '  Corr/Scl' , $
                      format='((A8,TR4), (A6, TR7) , 3(A10, TR4), 2(A10, TR4))'

      type_str = 'in ' + type + ' Peak'
      printf, filedes,'   ','   ', 'Counts',  ' Raw ',  'Cnts', $
                      ' Value', ' [Hz] ', $
                      format='((A8,TR4), (A6,TR7) , 3(A12, TR2), 2(A10, TR4))'

      printf, filedes,'---','---', '------',  '-----','------', $
                      ' ------','------', $
                      format='((A8,TR4), (A6, TR7) , 3(A12, TR2), 2(A10, TR4))'


RETURN
END


;-------------------------------------------------------------------------------
;  Function/Procedure: l_BinnedWriteBK2File
;
;  Description:  Write the binned background data to a text file.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  nspins                long          I        number of spins in time period
;  background_flag       int           I        do background correction?
;  bkcUdfTimePtr         array         I        background corrected UDF times
;  raw_cnts              array         I        binned raw counts
;  bk_raw                array         I        background raw data
;  bk_cnts               array         I        # of background counts
;  corr_cnts             array         I        binned background corrected counts
;  corr_scl_val          array         I        binned background corrected and scaled
;  spinsec               long          I        spin sector
;  filedes               long          I        output file descriptor
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------
PRO l_BinnedWriteBK2File, nspins, background_flag, bkcUdfTimePtr, $
raw_cnts, bk_raw, bk_cnts, corr_cnts, corr_scl_val, spinsec, filedes


      diff_nspins = n_elements (raw_cnts)/spinsec
      if (diff_nspins lt nspins) then nspins = diff_nspins


      if (background_flag eq 'Y') then begin

          for i = 0L, nspins - 1L do begin
              tmestr = STRING ((bkcUdfTimePtr)[i].doy, $
                               (bkcUdfTimePtr)[i].hour,$
                               (bkcUdfTimePtr)[i].min, $
                               (bkcUdfTimePtr)[i].sec, $
                               FORMAT="(i3,' ',i2.2,':',i2.2,':',i2.2)")

              for j = 0L, spinsec - 1L do begin
                 printf, filedes, tmestr, $
                                  j, $
                                  raw_cnts[j, i], $
                                  bk_raw[j, i], $
                                  bk_cnts[j, i], $
                                  corr_cnts[j, i], $
                                  corr_scl_val[j,i], $
                                  format='((A12,TR2),(I4,TR8), 5(e12.3,TR2))'

              endfor
         endfor
    endif else begin
          for i = 0L, nspins - 1L do begin
              tmestr = STRING ((bkcUdfTimePtr)[i].doy, $
                               (bkcUdfTimePtr)[i].hour,$
                               (bkcUdfTimePtr)[i].min, $
                               (bkcUdfTimePtr)[i].sec, $
                               FORMAT="(i3,' ',i2.2,':',i2.2,':',i2.2)")

              for j = 0L, spinsec - 1L do begin
                 printf, filedes, tmestr, $
                                  j, $
                                  raw_cnts[j, i], $
                                  corr_cnts[j, i], $
                                  corr_scl_val[j,i], $
                                  format='((A12,TR2),(I4,TR8), 3(e12.3,TR2))'

              endfor
         endfor

    endelse


RETURN
END



;-------------------------------------------------------------------------------
;  Procedure: l_BinnedDumpBKData
;
;  Description: Generate a text dump of the background corrected binned data.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  filedes               long          I        output file descriptor
;  l_binned              struct *      I        pointer to binned data structures
;  l_binned_setup        struct        I        binned processing & plotting parameters
;  l_setup               struct        I        general setup paramters
;  l_event               struct *      I        pointers to event data structures
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------

PRO l_BinnedDumpBKData, filedes, l_binned, l_binned_setup, l_setup, l_event


   nspins = n_elements ((*l_binned.filledStartUdfTimePtr)[*])


   zero_title = 'Spin Angle (-140 - 220)'
   if (l_setup.l_angle_zero eq 'E') then begin
      zero_title = 'Earth Centered Azimuth'
   endif else begin
      if (l_setup.l_angle_zero eq 'S') then begin
         zero_title = 'Sun Centered Azimuth'
      endif else begin
         if (l_setup.l_angle_zero eq 'M') then begin
            zero_title = 'Moon Centered Azimuth'
         endif
      endelse
   endelse


   printf, filedes, '----------------------- Column Descriptions -----------------------------'
   printf, filedes, '                      BINNED DATA         '
   printf, filedes, ' '
   printf, filedes, '      UTC             : Universal Time Coordinate (corrected)'
   printf, filedes, '      Azi             : Azimuthal spin sector (0-44)'
   printf, filedes, '      Raw Counts      : binned counts'
   printf, filedes, '      Bkgrnd Raw      : counts in each spin sector falling in TOF '
   printf, filedes, '                        background range'
   printf, filedes, '      Bkgrnd Cnts     : scaled background values '
   printf, filedes, '      Corr/Scl [Hz]   : background corrected scaled counts in '
   printf, filedes, '                        count rate [Hz]'
   printf, filedes, ' '
   printf, filedes, '-------------------------------------------------------------------------'
   printf, filedes, ' '
   printf, filedes, ' '
   printf, filedes, ' '



   if (l_binned_setup.hydrogen eq 'Y') then begin
      title = 'Hydrogen'
      type = 'H'
      if (l_binned_setup.background eq 'Y') then begin
         title = title + ' Background Corrected'
      endif


      l_BinnedBKHeader, type, filedes, title, zero_title

      raw_cnts  = (*l_binned.hydrogen_raw_cnts)

      if (l_binned_setup.background eq 'Y') then begin
         bk_raw  = (*l_event.background_e0) + (*l_event.background_e1) + (*l_event.background_e2)
         bk_cnts = bk_raw * 0.4
      endif

      corr_cnts = (*l_binned.corrected_hydrogen)
      corr_scl_val = (*l_binned.corr_scl_H)

      l_BinnedWriteBK2File, nspins, l_binned_setup.background, (*l_binned.filledStartUdfTimePtr), $
      raw_cnts, bk_raw, bk_cnts, corr_cnts, corr_scl_val, l_setup.l_num_spinsec, filedes

      if (l_binned_setup.energy0 eq 'Y') then begin
         title = 'Hydrogen Energy 0'
         type = 'H E0'
         if (l_binned_setup.background eq 'Y') then begin
            title = title + ' Background Corrected'
         endif

         l_BinnedBKHeader, type, filedes, title, zero_title

         raw_cnts  = (*l_binned.hydrogen_raw_cnts_e0)
   
         if (l_binned_setup.background eq 'Y') then begin
            bk_raw  = (*l_event.background_e0) 
            bk_cnts = bk_raw * 0.4 
         endif

         corr_cnts = (*l_binned.corrected_hydrogen_e0)
         corr_scl_val = (*l_binned.corr_scl_H_e0)

         l_BinnedWriteBK2File, nspins, l_binned_setup.background, (*l_binned.filledStartUdfTimePtr), $
         raw_cnts, bk_raw, bk_cnts, corr_cnts, corr_scl_val, l_setup.l_num_spinsec, filedes
      endif

      if (l_binned_setup.energy1 eq 'Y') then begin
         title = 'Hydrogen Energy 1'
         type = 'H E1'
         if (l_binned_setup.background eq 'Y') then begin
            title = title + ' Background Corrected'
         endif

         l_BinnedBKHeader, type, filedes, title, zero_title

         raw_cnts  = (*l_binned.hydrogen_raw_cnts_e1)
   
         if (l_binned_setup.background eq 'Y') then begin
            bk_raw  = (*l_event.background_e1) 
            bk_cnts = bk_raw * 0.4 
         endif

         corr_cnts = (*l_binned.corrected_hydrogen_e1)
         corr_scl_val = (*l_binned.corr_scl_H_e1)

         l_BinnedWriteBK2File, nspins, l_binned_setup.background, (*l_binned.filledStartUdfTimePtr), $
         raw_cnts, bk_raw, bk_cnts, corr_cnts, corr_scl_val, l_setup.l_num_spinsec, filedes

      endif

      if (l_binned_setup.energy2 eq 'Y') then begin
         title = 'Hydrogen Energy 2'
         type = 'H E2'
         if (l_binned_setup.background eq 'Y') then begin
            title = title + ' Background Corrected'
         endif

         l_BinnedBKHeader, type, filedes, title, zero_title

         raw_cnts  = (*l_binned.hydrogen_raw_cnts_e2)
   
         if (l_binned_setup.background eq 'Y') then begin
            bk_raw  = (*l_event.background_e2) 
            bk_cnts = bk_raw * 0.4
         endif

         corr_cnts = (*l_binned.corrected_hydrogen_e2)
         corr_scl_val = (*l_binned.corr_scl_H_e2)

         l_BinnedWriteBK2File, nspins, l_binned_setup.background, (*l_binned.filledStartUdfTimePtr), $
         raw_cnts, bk_raw, bk_cnts, corr_cnts, corr_scl_val, l_setup.l_num_spinsec, filedes

      endif


   endif


   if (l_binned_setup.oxygen eq 'Y') then begin
      title = 'Oxygen'
      type = 'O'
      if (l_binned_setup.background eq 'Y') then begin
         title = title + ' Background Corrected'
      endif

      l_BinnedBKHeader, type, filedes, title, zero_title
      raw_cnts = (*l_binned.oxygen_raw_cnts)

      if (l_binned_setup.background eq 'Y') then begin
         bk_raw  = (*l_event.background_e0) + (*l_event.background_e1) + (*l_event.background_e2)
         bk_cnts = bk_raw * 6.0
      endif

      corr_cnts = (*l_binned.corrected_oxygen)
      corr_scl_val = (*l_binned.corr_scl_O)

      l_BinnedWriteBK2File, nspins, l_binned_setup.background, (*l_binned.filledStartUdfTimePtr), $
      raw_cnts, bk_raw, bk_cnts, corr_cnts, corr_scl_val, l_setup.l_num_spinsec, filedes

      if (l_binned_setup.energy0 eq 'Y') then begin
         title = 'Oxygen Energy0'
         type = 'O E0'
         if (l_binned_setup.background eq 'Y') then begin
            title = title + ' Background Corrected'
         endif

         l_BinnedBKHeader, type, filedes, title, zero_title
         raw_cnts = (*l_binned.oxygen_raw_cnts_e0)

         if (l_binned_setup.background eq 'Y') then begin
            bk_raw = (*l_event.background_e0) 
            bk_cnts = bk_raw * 6.0
         endif

         corr_cnts = (*l_binned.corrected_oxygen_e0)
         corr_scl_val = (*l_binned.corr_scl_O_e0)

         l_BinnedWriteBK2File, nspins, l_binned_setup.background, (*l_binned.filledStartUdfTimePtr), $
         raw_cnts, bk_raw, bk_cnts, corr_cnts, corr_scl_val, l_setup.l_num_spinsec, filedes
      endif

      if (l_binned_setup.energy1 eq 'Y') then begin
         title = 'Oxygen Energy1'
         type = 'O E1'
         if (l_binned_setup.background eq 'Y') then begin
            title = title + ' Background Corrected'
         endif

         l_BinnedBKHeader, type, filedes, title, zero_title
         raw_cnts = (*l_binned.oxygen_raw_cnts_e1)

         if (l_binned_setup.background eq 'Y') then begin
            bk_raw = (*l_event.background_e1) 
            bk_cnts = bk_raw * 6.0
         endif

         corr_cnts = (*l_binned.corrected_oxygen_e1)
         corr_scl_val = (*l_binned.corr_scl_O_e1)

         l_BinnedWriteBK2File, nspins, l_binned_setup.background, (*l_binned.filledStartUdfTimePtr), $
         raw_cnts, bk_raw, bk_cnts, corr_cnts, corr_scl_val, l_setup.l_num_spinsec, filedes
      endif

      if (l_binned_setup.energy2 eq 'Y') then begin
         title = 'Oxygen Energy2'
         type = 'O E2'
         if (l_binned_setup.background eq 'Y') then begin
            title = title + ' Background Corrected'
         endif

         l_BinnedBKHeader, type, filedes, title, zero_title
         raw_cnts = (*l_binned.oxygen_raw_cnts_e2)

         if (l_binned_setup.background eq 'Y') then begin
            bk_raw = (*l_event.background_e2) 
            bk_cnts = bk_raw * 6.0
         endif

         corr_cnts = (*l_binned.corrected_oxygen_e2)
         corr_scl_val = (*l_binned.corr_scl_O_e2)

         l_BinnedWriteBK2File, nspins, l_binned_setup.background, (*l_binned.filledStartUdfTimePtr), $
         raw_cnts, bk_raw, bk_cnts, corr_cnts, corr_scl_val, l_setup.l_num_spinsec, filedes
      endif

   endif


RETURN
END


;-------------------------------------------------------------------------------
;  Procedure: l_BinnedProcData
;  Description: 
;      Calls the routines necessary in creating the image to be plotted.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  dataPtr               float*        I        pointer to data
;  timePtr               double*       I        pointer to time data
;  filled_time           double*       I        filled time data
;  orig_udf_time         struct[]      I        UDF times 
;  filled_start_udf_time struct[]      I        UDF times & interpolated UDF times
;                                               for missing data
;  l_setup               struct        I        setup parameters
;  center_sector         long          I        align this sector to the
;                                               zero degree spin angle
;  l_pltpos              struct        I        general plot parameters
;  plotted               long          I        # of plots plotted
;  l_oa                  struct        I        OA fields
;  l_rgb_table           struct        I        color options/parameters
;  colorbar              float[]       I        color bar read from color table file
;  l_hskp                struct*       I        housekeeping parameters
;  nplots                long          I        total # plots to be plotted
;  windx                 long[]        I        window index
;  pages                 long          I        # of plot pages total
;  ytitle_frame          string        I        y-axis unit label
;  min_color             double        I        minimum color scale limit 
;  max_color             double        I        maximum color scale limit 
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;     PLOTDEFS         MAX_PLOTS        int         G         max # plots per page
;                      MAX_WEB_SPINS    int         G         max # spins per plot
;                                                             when using web interface
;                      NO_CNTS          double      G         no counts indicator
;                      NO_DATA          double      G         no data indicator
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;  ELH                               v1.0.7           Added necessary parameters to 
;                                                     access interpolated UDF times 
;  ELH               09/26/01        v1.0.7           Added l_DetermineSunDir                
;-------------------------------------------------------------------------------

PRO l_BinnedProcData, dataPtr, timePtr, filled_time, orig_udf_time, filled_start_udf_time, $
l_setup, center_sector, l_pltpos, plotted, l_oa, l_rgb_table, colorbar, $
l_hskp, ytitle, ytitle1, nplots, windx, pages, ytitle_frame, min_color, max_color, $
dmpPtr, extra_data 
COMMON PLOTDEFS, MAX_PLOTS, MAX_WEB_SPINS, NO_CNTS, NO_DATA

 
      raw_data  = dataPtr 
      orig_tme  = timePtr
      dum       = 0
      nspins    = n_elements(orig_udf_time)
      data_format = l_setup.l_num_spinsec

      ;---------------------------------------------------
      ; flag indicating whether to adjust a feature (sun,
      ; earth, moon) to zero degree spin angle
      ;---------------------------------------------------
      adjFlag   = 'YES'
      if (l_setup.l_angle_zero eq 'N') then adjFlag = 'NO'

      ;---------------------------------------------------
      ; determine if there are data gaps
      ;---------------------------------------------------
      l_HndlMissData, orig_tme,              $
                      filled_time,           $
                      nspins,                $
                      raw_data,              $
                      filled_data,           $
                      data_format,           $
                      center_sector,         $
                      l_setup,               $
                      adjFlag,               $
                      l_oa,                  $
                      orig_udf_time,         $
                      filled_start_udf_time, $
                      extra_data,            $
                      l_setup.l_spin_ang_zero 


    binn_data = filled_data


    ;SyncData2Time, binn_data, filled_time, filled_start_udf_time, l_pltpos

    dmpPtr    = binn_data 


   ;---------------------------------------------------
   ; determine the number of major and minor ticks
   ; on the x-axis
   ;---------------------------------------------------


   l_setup.l_d_start_dtime[0] = filled_start_udf_time[0].year
   l_setup.l_d_start_dtime[1] = filled_start_udf_time[0].doy
   l_setup.l_d_start_dtime[2] = filled_start_udf_time[0].hour
   l_setup.l_d_start_dtime[3] = filled_start_udf_time[0].min
   l_setup.l_d_start_dtime[4] = filled_start_udf_time[0].sec
   l_setup.l_d_start_dtime[5] = filled_start_udf_time[0].msec

   max_ele = n_elements (filled_start_udf_time) - 1

   l_setup.l_d_stop_dtime[0] = filled_start_udf_time[max_ele].year
   l_setup.l_d_stop_dtime[1] = filled_start_udf_time[max_ele].doy
   l_setup.l_d_stop_dtime[2] = filled_start_udf_time[max_ele].hour
   l_setup.l_d_stop_dtime[3] = filled_start_udf_time[max_ele].min
   l_setup.l_d_stop_dtime[4] = filled_start_udf_time[max_ele].sec
   l_setup.l_d_stop_dtime[5] = filled_start_udf_time[max_ele].msec


   l_DetMjMnTicks, l_setup.l_start_dtime, $
                   l_setup.l_stop_dtime, $
                   l_setup.l_d_start_dtime, $
                   l_setup.l_d_stop_dtime, $
                   l_pltpos


      ztitle = ' '
      cnt_ztitle = ' '
      flx_ztitle = ' '
      log_ztitle = ' '

      ;---------------------------------------------------
      ; convert non zero values to log
      ;---------------------------------------------------
      if (l_setup.l_log eq 'Y') then begin
          l_Conv2Log, binn_data, log_data, min_value, max_value
          binn_data = log_data 
          zaxis_title = ' '
          log_ztitle = '!5log '
      endif else begin
          NO_CNTS = 0.0
          non_zero = where (binn_data ne NO_CNTS, wcnt)
          if (wcnt gt 0) then begin
              min_value = min (binn_data[non_zero])
          endif else begin
              min_value = min (binn_data)
          endelse
          max_value = max (binn_data) 
      endelse

      if (l_setup.l_count_rate eq 'Y') then begin
         cnt_ztitle = 'cnt Rate[Hz]'
         if (l_setup.l_flux eq 'Y') then begin
            cnt_ztitle = ' '
            flx_ztitle = 'atoms/cm2/s'
         endif
       endif else begin
         if (l_setup.l_flux eq 'Y') then begin
            cnt_ztitle = ' '
            flx_ztitle = 'atoms/cm2/s'
         endif else begin
            ztitle = 'linear cnts'
         endelse
      endelse

      zaxis_title = ztitle + log_ztitle + flx_ztitle + cnt_ztitle


      ;---------------------------------------------------
      ; set upper and lower color bounds 
      ;---------------------------------------------------
      if (l_setup.l_auto_scale eq 'Y') then begin
          l_setup.l_color_min = min_value
          l_setup.l_color_max = max_value
      endif else begin
          if (l_setup.l_sep_plot_scale eq 'Y') then begin
             l_setup.l_color_min = min_color
             l_setup.l_color_max = max_color
          endif
      endelse


      if (l_setup.l_line_plots eq 'Y') then begin

      ;---------------------------------------------------
      ; generate a line plot
      ;---------------------------------------------------

          diff = max_value - min_value
          if (diff gt 1) then begin
             new_max = (max_value/4.) + max_value
             max_value = new_max
          endif else begin
             max_value = max_value + diff
          endelse

          l_PlotStuff, nplots,                   $
                       filled_time,              $
                       binn_data,                $
                       min_value,                $
                       max_value,                $
                       zaxis_title,              $
                       l_pltpos,                 $
                       plotted,                  $
                       l_oa,                     $
                       l_setup.l_start_dtime


      endif else begin  ; a spectrogram plot

      ;---------------------------------------------------
      ; generate a spectrogram
      ;---------------------------------------------------

          ;---------------------------------------------------
          ; scale the data to the selected color table
          ;---------------------------------------------------
          l_ScaleColor, binn_data, scaled_data, $
                        l_setup.l_color_min, $
                        l_setup.l_color_max , $
                        l_setup.l_color_tbl


          ;---------------------------------------------------
          ; dump out values from color bar and NOT data values
          ;---------------------------------------------------
          if (l_setup.l_color_values eq 'Y') then dmpPtr = scaled_data 

          ;------------------------------------------------------------
          ; correct the image positioning, rotate 240 degrees
          ;------------------------------------------------------------
          rot_data = rotate(scaled_data, 3)

          yrng=[l_pltpos.min_spin_ang,l_pltpos.max_spin_ang]

          img_pos = [l_pltpos.xpltmn,l_pltpos.ypltmn,$
                     l_pltpos.xpltmx,l_pltpos.ypltmx]

          position = float(img_pos)

          ;---------------------------------------------------
          ; image coordinates
          ;---------------------------------------------------
          xstart = position[0]
          ystart = position[1]
          ximage = l_pltpos.axis_size/ l_pltpos.xin
          yimage = l_pltpos.ypltmx - l_pltpos.ypltmn
          xsize  = round(l_pltpos.window_width * ximage)
          ysize  = round(l_pltpos.window_height * yimage)

          spect_data = congrid (rot_data, $
                                xsize, $
                                ysize)


          ;---------------------------------------------------
          ; plot the data
          ;---------------------------------------------------
          if (l_setup.l_smooth_data eq 'Y') then begin
             TV, smooth (spect_data, 3, /edge_truncate), xstart, ystart, $
                 XSIZE=ximage, YSIZE=yimage, /ORDER, $
                 /NORMAL
          endif else begin
             TV, spect_data, xstart, ystart, $
                 XSIZE=ximage, YSIZE=yimage, /ORDER, $
                 /NORMAL

          endelse

          ;---------------------------------------------------
          ; draw the earth lines 
          ;---------------------------------------------------
          if (l_setup.l_earth_markers eq 'Y') then begin
             l_CalcEarthDirection, l_oa, l_setup, img_pos
          endif

          ;---------------------------------------------------
          ; draw the sun direction
          ;---------------------------------------------------
          if (l_setup.l_sun_direction eq 'Y') then begin
              l_CalcSunDirection, l_oa, l_setup, img_pos
          endif

          ;---------------------------------------------------
          ; draw the ram direction
          ;---------------------------------------------------
          if (l_setup.l_ram_direction eq 'Y') then begin
             l_CalcSunPulse, l_oa, ram_sector, 'R'
             ram_sector = ram_sector/8.
             if NOT (ptr_valid(l_oa.lena_ram_sectPtr)) then begin
                ptr = ptr_new (ram_sector, /NO_COPY)
                l_oa.lena_ram_sectPtr = ptr
             endif
             l_CalcRamDirection, l_oa, l_setup, img_pos
             ram_sector = 0
          endif

          ;---------------------------------------------------
          ; draw the B direction
          ;---------------------------------------------------
          if (l_setup.l_B_direction eq 'Y') then begin
             l_CallT96Model, l_oa, l_setup
             l_CalcSunPulse, l_oa, B_sector, 'B'
             B_sector = B_sector/8.
             if NOT (ptr_valid(l_oa.lena_mag_fieldPtr)) then begin
                ptr = ptr_new (B_sector, /NO_COPY)
                l_oa.lena_mag_fieldPtr = ptr
             endif
             l_CalcBDirection, l_oa, l_setup, img_pos
             B_sector = 0
          endif

          ;---------------------------------------------------
          ; draw the color bar
          ;---------------------------------------------------
          bar_pos = [l_pltpos.xpltmx+0.015, $
                    l_pltpos.ypltmn, $
                    l_pltpos.xpltmx+0.025, $
                    l_pltpos.ypltmx]

          bar_info = {AXIS_DATA, char_size: l_pltpos.plot_font, $
                      ytitle:zaxis_title, $
                      yformat:'(F6.1)', $
                      yrange:[l_setup.l_color_min, $
                              l_setup.l_color_max]}

          l_DrawColorbar, colorbar, l_setup, bar_pos, l_pltpos, $
                          RAXIS=bar_info

          ;---------------------------------------------------
          ; draw the axis around the spectrogram
          ;---------------------------------------------------
          l_PlotFrame, nplots, filled_time, yrng, img_pos, l_pltpos, $
                       plotted, l_oa, ytitle_frame, l_setup.l_start_dtime, $
                       l_hskp


      endelse

      ;---------------------------------------------------
      ; label y-axis
      ;---------------------------------------------------
      yctr = (l_pltpos.ypltmn + l_pltpos.ypltmx)/2.0
      xyouts, 0.015, yctr, ytitle, size=l_pltpos.sm_font, $
              /NORMAL, alignment = 0.5, orientation=90.0

      xyouts, 0.035, yctr, ytitle1, size=l_pltpos.sm_font, $
              /NORMAL, alignment = 0.5, orientation=90.0

      ;---------------------------------------------------
      ; check if the last possible plot for the page
      ;---------------------------------------------------
      l_pltpos.ypltmx = l_pltpos.ypltmn - l_pltpos.yspace
      l_pltpos.ypltmn = l_pltpos.ypltmx - l_pltpos.ydel
      plotted = plotted + 1L
      l_ChckCompPg, plotted, nplots, pages, $
                    l_setup, windx, l_pltpos, l_rgb_table

      ;---------------------------------------------------
      ; DO NOT REMOVE, need to reset this
      ;---------------------------------------------------
      NO_CNTS       = -8888.



return
end

;-------------------------------------------------------------------------------
;  Function :  l_BinnedProc
;  Description:
;          Main
;
;  Return Value:  type = int
;  Value                         Description
;  -------------------------     ------------------------------
;  SUCCESS                       successful completion
;  FAILURE                       unsuccessful completion
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_setup               struct        I        general setup options/parameters
;  l_setup_file          string        I        setup filename
;  l_pltpos              struct        I        general plot parameters
;  l_rgb_table           struct        I        color table parameters
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;     PLOTDEFS         MAX_PLOTS        int         G         max # plots per page
;                      MAX_WEB_SPINS    int         G         max # spins per plot
;                                                             when using web interface
;                      NO_CNTS          double      G         no counts indicator
;                      NO_DATA          double      G         no data indicator
;
;      ERROR_TYPES     SUCCESS         int         G          successful completion
;                      FAILURE         int         G          unsuccessful completion
;                      WARNING         int         G          warning message
;                      INFO            int         G          informational message
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;  RLW                08/01          v1.0.7           Removed web check constraint 
;-------------------------------------------------------------------------------

FUNCTION l_BinnedProc, l_setup, l_setup_file, l_pltpos, l_rgb_table
COMMON PLOTDEFS, MAX_PLOTS, MAX_WEB_SPINS, NO_CNTS, NO_DATA
COMMON ERROR_TYPES, SUCCESS, FAILURE, WARNING, INFO


   ;-------------------------------------------------
   ; the maximum number of spins to plot if plotting
   ; via the web
   ;-------------------------------------------------

   nele      = 0L
   year      = strmid(systime(0), 20, 23)

   ;-------------------------------------------------
   ; if via the web, need to determine the number
   ; of spins desired.  if the number of spins is
   ; greater than MAX_WEB_SPINS, then plot only
   ; MAX_WEB_SPINS
   ;-------------------------------------------------
   sp = ConvArrTimeToSecs (l_setup.l_stop_dtime)
   st = ConvArrTimeToSecs (l_setup.l_start_dtime)
   est_nspins = round((sp - st)/120.0)


   ;-------------------------------------------------
   ; read setup paramters specific to the binned plots
   ;-------------------------------------------------
   l_BinnedReadSetup, l_binned_setup, l_setup_file

   ;-------------------------------------------------
   ; get the singles data based on user start and
   ; stop time
   ;-------------------------------------------------
   l_pltpos.plot_hdr_title = '!5Binned Data'
   status = l_BinnedGetData  (l_setup, $
                              l_setup_file, $
                              l_pltpos, $
                              l_binned_setup, $
                              l_binned, $
                              nele)
   l_Binned_AdjExData, l_setup, l_pltpos, l_binned, l_binned_setup

   ;-------------------------------------------------
   ; read the corresponding OA data
   ;-------------------------------------------------
   status = l_OARead (l_oa, $
                      l_setup.l_start_dtime, $
                      l_setup.l_stop_dtime, $
                      l_pltpos)
   if (status eq FAILURE) then return, FAILURE




   ;---------------------------------------------------
   ; read the housekeeping data
   ;---------------------------------------------------
   status = l_HskpGetData (l_setup, $
                  l_setup_file, $
                  l_pltpos, $
                  l_hskp_setup, $
                  l_hskp)
   if (status eq FAILURE) then return, FAILURE




   nloops = 1L
   if (l_setup.l_postscript eq 'Y') then nloops = 2L

   l_setup.l_line_plots = 'N'
   if (l_setup.l_line_plots eq 'N') then begin
      for i=0L, nloops-1L do begin

         l_pltpos.ps = 'N'
         pages     = 0L
         nplots    = 0L
         plotted   = 0L

         if (l_setup.l_pwrsup_bars eq 'Y') then begin
            nplots    = 1L
         endif

         if (i eq 1) and (l_setup.l_postscript eq 'Y') then begin
            l_pltpos.plot_hdr_title = '!5Binned Data'
            l_pltpos.ps = 'Y'
            l_GenFileName, l_setup, '0'
            l_pltpos.filename = l_setup.l_prod_dst + '/' + $
                                l_setup.l_filename + '.ps'
         endif

         nplots    = nplots + l_binned_setup.nplots

         ;---------------------------------------------------
         ; setup display and plotting environment
         ;---------------------------------------------------
         l_SetUpZBuffer, l_pltpos, $
                         l_setup, $
                         l_rgb_table, $
                         colorbar, $
                         nplots


         l_pltpos.linePlots = l_setup.l_line_plots


         npages = CEIL(nplots/MAX_PLOTS)
         if (nplots - max_plots gt 0) then npages = npages + 1
         if (npages eq 0) then npages = 1
         if (npages gt 0) then windx = intarr (npages)


         ;-------------------------------------------------
         ; do a spectrogram plot
         ; determine the number of plots and calculate the
         ; size of each plot to fit the page
         ;-------------------------------------------------
         if (npages gt 1) then begin
            plots_per_page = 5
            if (l_setup.l_pwrsup_bars eq 'Y') then plots_per_page = 6 
         endif else begin
            plots_per_page = nplots 
         endelse

         l_AdjPlotSize, l_pltpos, l_setup, plots_per_page 

         if (l_setup.l_pwrsup_bars eq 'Y') then begin
            l_DrawOpBars, l_hskp, l_setup, l_pltpos, l_rgb_table, l_oa
            nplots = nplots - 1L
            plotted   = 1L
         endif


         l_setup.l_filedesc = 'bin'

         l_setup.l_filedesc = l_setup.l_filedesc + 'sp'

         l_BinnedPlot, l_binned_setup, $
                       l_binned, $
                       l_pltpos, $
                       nplots, $
                       windx, $
                       pages, $
                       plotted, $
                       l_setup, $
                       l_oa , $
                       l_rgb_table, $
                       colorbar, $
                       l_hskp


          if (i eq 1) and (l_setup.l_postscript eq 'Y') then begin
             device, /close_file
          endif

          if (!D.NAME eq 'Z') then begin
            if (l_setup.l_via_web eq 'Y') or (l_setup.l_gen_sum eq 'Y') then begin
               device, /close
            endif
          endif

      endfor
   endif


   ;-------------------------------------------------
   ; create an ascii dump of the binned data
   ;-------------------------------------------------
   if (l_setup.l_dumps eq 'Y') and (l_binned_setup.use_binned eq 'Y') then begin

      ;-------------------------------
      ; create text dump filename
      ;-------------------------------
      l_setup.l_filedesc = 'bin'
      l_GenFileName, l_setup, '0'
      filename = l_setup.l_prod_dst + '/' + l_setup.l_filename + '.dmp'

      ;-------------------------------
      ; open text dump file
      ;-------------------------------
      openw, ofd, filename, /get_lun
      l_pltpos.dmp_hdr_title = 'Binned Dumps'
      l_DumpHdrInfo, ofd, l_setup, l_pltpos, filename
      l_BinnedDmpSpectData, ofd, l_binned, l_binned_setup, l_roi0, l_setup
      close, ofd
      free_lun, ofd

   endif


   ;-------------------------------------------------
   ; free pointers
   ;-------------------------------------------------
   ;if (l_binned_setup.l_binned_bkgrnd eq 'Y') then begin
   ;   l_RoiTerm, l_roi0
   ;endif
  
   l_BinnedTerm, l_binned
   l_HskpTerm, l_hskp
   l_OATerm, l_oa


   RETURN, SUCCESS

END
