
;--------------------------------------------------------------------------------
;
;  Unit Name : l_BinnedReadData.pro
;
;  Purpose   : Read the binned/image data packets.
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  John Park         1999             v1.0            Original Implementation
;  ELH               06/28/01                         Corrected syncing of
;                                                     binned data to steering
;
;  ELH               08/01           v1.0.7           Fixed indexing problem in
;                                                     steering value, l_BinnedReadSVL
;                                                     # of items in steering did not
;                                                     match # of items in binned data
;  RLW               08/01           v1.0.7           handle mettime file to work on
;                                                     the web
;  ELH               08/31/01        v1.0.7           removed unused fields from 
;                                                     binned data structure in
;                                                     l_BinnedReadData
;  ELH               01/27/2004      v1.0.11          Fixed sector drop out problem
;                                                     Deleted l_BinnedReadUdfDataBySpin.
;  ELH               05/27/2004      v1.0.11          Replaced use of for loops with
;                                                     IDL total() in l_BinnedReadDataByTime
;  ELH               05/24/2005      v1.0.11          Fixed indexing problem when sector changes
;                                                     from 44 to 1, skipping sector 0
;                                                     Added a condition to check for duplicate
;                                                     sector 0
;                                                     
;
;  File Revision Number:   %I%
;  File Last Modified  :   %E%  %T%
;
;--------------------------------------------------------------------------------


;-------------------------------------------------------------------------------
;  Function: l_BinnedReadUdfDataByTime
;
;  Description:  Read and the binned data from the UDF data base on a time basis,
;          store as in UDF format, record by record.
;
;  Return Value:  type = int
;  Value                         Description
;  -------------------------     ------------------------------
;   SUCCESS                      successful completion
;   FAILURE                      unsuccessful completion
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_setup               struct        I        general setup parameters
;  nspins                int           I        total number of spins in 
;                                               specified time range
;  binndata              struct *      I        pointer to binned data structures
;  start_tme             array         I        user specified start time
;  stop_tme              array         I        user specified stop time
;  data_source           string        O        data from definitive or QL  
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;       ERROR_TYPES    SUCCESS         int          G         successful completion
;                      FAILURE         int          G         unsuccessful completion
;                      WARNING         int          G         warning message
;                      INFO            int          G         informational message
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;  ELH               01/09/04       v1.0.11           Fixed a sector drop out
;                                                     problem 
;  ELH               04/12/04       v1.0.11           Under if new spin, fixed
;                                                     problem with when a new spin
;                                                     should begin, removed the
;                                                     time_diff condition check for
;                                                     120 seconds since a spin
;                                                     could be less
;  ELH               04/19/04       v1.0.11           readded the time_diff condition
;                                                     to check for sectors
;  ELH               05/27/04       v1.0.11           replaced multiple use of for
;                                                     loops to use IDL total()
;  ELH               05/24/2005      v1.0.11          Fixed indexing problem when sector changes
;                                                     from 44 to 1, skipping sector 0
;                                                     Added a condition to check for duplicate
;                                                     sector 0
;-------------------------------------------------------------------------------

FUNCTION l_BinnedReadUdfDataByTime, l_setup, nspins, binndata, start_tme, stop_tme, $
data_source
COMMON ERROR_TYPES, SUCCESS, FAILURE, WARNING, INFO


   nrec = (nspins * 45L)

   l_setup.l_num_spin = round(nspins)

   mettime_file = l_setup.l_prod_dst + "/" + "mettime" + l_setup.l_process_id

   ;--------------------
   ;  data storage 
   ;--------------------
   ttime = dblarr(nrec)
   stime = dblarr(l_setup.l_num_spin)
   start_array = replicate ({UDF_TIMES_STRUCT}, l_setup.l_num_spin)
   stop_array = replicate ({UDF_TIMES_STRUCT}, l_setup.l_num_spin)
   vedata = fltarr (l_setup.l_num_spinsec, l_setup.l_num_spin)

   m0data = fltarr(l_setup.l_num_energylev, $
                    l_setup.l_num_polarzn,   $
                    l_setup.l_num_spinsec,   $
                    l_setup.l_num_spin)

   m1data = fltarr(l_setup.l_num_energylev, $
                    l_setup.l_num_polarzn,   $
                    l_setup.l_num_spinsec,   $
                    l_setup.l_num_spin)


   ;--------------------
   ;  data retrieval 
   ;--------------------
   data_source = ' '


   uh = l_UdfGetDataHandle('LENA', 'LENASCI', 'IMLIMAGE', $
                           start_tme, stop_tme, data_source, /GROUP)

   if (uh eq -1 ) then begin
      udf_close, uh
      RETURN, FAILURE 
   endif

   ii=0L
   ss=0L   ; spin sector
   prev_ss = 0L
   sp=0L   ; spin 
   new_spin = 1L

   time_diff = 0L
   prev_time = 0.0D
   curr_time = 0.0D


;openw, ofd, 'time_data', /get_lun

   ;------------------------------------------------------
   ; read data on a record by record basis and store
   ; the data based on time
   ;------------------------------------------------------
   while (not udf_eof(uh)) do begin

       ud = udf_read(uh)

       curr_time = l_GetTime(ud.btime, doy, hr, min, sec)
       ;ttime[ii] = l_GetTime(ud.btime, doy, hr, min, sec)

       ;--------------------------------------------------
       ; determine the sector number using the start and
       ; stop azimuth from UDF
       ;--------------------------------------------------

       ;--------------------------------------------------
       ; determine the center phase angle
       ; algorithm from Chris Gurgiolo
       ;--------------------------------------------------

       delta_angle = ud.stop_az[0] - ud.start_az[0]
       udf_angle = ud.stop_az[0] + ud.start_az[0]

       if (delta_angle gt 0.0) then begin
           udf_phase_angle = udf_angle/2.0
       endif else begin
           udf_phase_angle = (360.0 + udf_angle)/2.0
           udf_phase_angle = udf_phase_angle mod 360.0
       endelse

       ;--------------------------------------------------
       ; convert the UDF phase angle into spin sector
       ; algorithm from Michael Collier
       ;--------------------------------------------------
       if (udf_phase_angle ge 225.1) then begin
          ss  = fix((udf_phase_angle - 225.1)/8.)
       endif else begin
          if (udf_phase_angle lt 225.1) then begin
             ss  = fix((udf_phase_angle + 134.9)/8.)
          endif
       endelse


       ;--------------------------------------------------
       ; check to see if one spin, >= 120 seconds
       ;--------------------------------------------------
       time_diff = 0L
       ;if (ii gt (sp*44)) then begin
       ;   time_diff = ttime[ii] - ttime[ii-45]
       ;endif
       time_diff = curr_time - prev_time

       ;--------------------------------------------------
       ; if a new spin
       ;   ELH 2004/04/12 - changed IF condition, removed 
       ;           time_diff.  Only start a new 
       ;           spin based on change in sector number
       ;   ELH 2004/04/19 - readded the time_diff condition
       ;           to check for time difference between
       ;           sectors.  UDF may return the last sector,
       ;           sector 44, of a previous spin as the first
       ;           piece of data, this creates an initial
       ;           spin 0 with one sector and two spin
       ;           times that are only ~2 seconds apart.
       ;--------------------------------------------------

       ;if ((ss ge 0) AND (prev_ss gt ss)) AND (time_diff gt 3) then begin
       ;   new_spin = 1L
       ;endif
       if ((ss ge 0) AND (prev_ss gt ss)) then begin
;printf, ofd, ss, prev_ss, time_diff
          if (time_diff gt 3) then begin
             new_spin = 1L
          endif else begin
             ;-------------------------------------------
             ; double check to see if sector change is a
             ; new spin based on the time difference
             ;-------------------------------------------
             ;if (ii ge 45 ) then begin
;printf, ofd, 'ii = ', ii, time_diff
                ;time_diff = ttime[ii] - ttime[ii-45]
                time_diff = curr_time - prev_time
                if (time_diff gt 3) then begin
                   new_spin = 1L
                endif else begin
                    ;-------------------------------------------
                    ; duplicate spin sector 
                    ;-------------------------------------------
                    if (time_diff eq 0) then begin
;printf, ofd, 'duplicate sector read next record'
                       ud = udf_read(uh)
                       new_spin = 1L
                    endif
                endelse
             ;endif else begin
             ;       ;-------------------------------------------
             ;       ; duplicate spin sector 
             ;       ;-------------------------------------------
             ;       if (time_diff eq 0) then begin
;printf, ofd, 'duplicate sector read next record'
;                       ud = udf_read(uh)
;                       new_spin = 1L
;                    endif
;             endelse
          endelse
       endif


       ;--------------------------------------------------
       ; store the data 
       ;--------------------------------------------------

       m0data[*,0,ss,sp]=ud.NEUTRAL_HYDROGEN_IMAGE[*,0]
       m0data[*,1,ss,sp]=ud.NEUTRAL_HYDROGEN_IMAGE[*,1]
       m0data[*,2,ss,sp]=ud.NEUTRAL_HYDROGEN_IMAGE[*,2]
       m0data[*,3,ss,sp]=ud.NEUTRAL_HYDROGEN_IMAGE[*,3]
       m0data[*,4,ss,sp]=ud.NEUTRAL_HYDROGEN_IMAGE[*,4]
       m0data[*,5,ss,sp]=ud.NEUTRAL_HYDROGEN_IMAGE[*,5]
       m0data[*,6,ss,sp]=ud.NEUTRAL_HYDROGEN_IMAGE[*,6]
       m0data[*,7,ss,sp]=ud.NEUTRAL_HYDROGEN_IMAGE[*,7]
       m0data[*,8,ss,sp]=ud.NEUTRAL_HYDROGEN_IMAGE[*,8]
       m0data[*,9,ss,sp]=ud.NEUTRAL_HYDROGEN_IMAGE[*,9]
       m0data[*,10,ss,sp]=ud.NEUTRAL_HYDROGEN_IMAGE[*,10]
       m0data[*,11,ss,sp]=ud.NEUTRAL_HYDROGEN_IMAGE[*,11]

       m1data[*,0,ss,sp]=ud.NEUTRAL_OXYGEN_IMAGE[*,0]
       m1data[*,1,ss,sp]=ud.NEUTRAL_OXYGEN_IMAGE[*,1]
       m1data[*,2,ss,sp]=ud.NEUTRAL_OXYGEN_IMAGE[*,2]
       m1data[*,3,ss,sp]=ud.NEUTRAL_OXYGEN_IMAGE[*,3]
       m1data[*,4,ss,sp]=ud.NEUTRAL_OXYGEN_IMAGE[*,4]
       m1data[*,5,ss,sp]=ud.NEUTRAL_OXYGEN_IMAGE[*,5]
       m1data[*,6,ss,sp]=ud.NEUTRAL_OXYGEN_IMAGE[*,6]
       m1data[*,7,ss,sp]=ud.NEUTRAL_OXYGEN_IMAGE[*,7]
       m1data[*,8,ss,sp]=ud.NEUTRAL_OXYGEN_IMAGE[*,8]
       m1data[*,9,ss,sp]=ud.NEUTRAL_OXYGEN_IMAGE[*,9]
       m1data[*,10,ss,sp]=ud.NEUTRAL_OXYGEN_IMAGE[*,10]
       m1data[*,11,ss,sp]=ud.NEUTRAL_OXYGEN_IMAGE[*,11]

;printf, ofd, ud.btime, ss, m0data[0,0,ss,sp]

       if (new_spin eq 1L) then begin

          ;stime[sp] = ttime[ii]
          stime[sp] = curr_time
          prev_time = curr_time 
          start_array[sp].year = ud.btime.year
          start_array[sp].doy  = ud.btime.doy
          start_array[sp].hour = ud.btime.hour
          start_array[sp].min  = ud.btime.min
          start_array[sp].sec  = ud.btime.sec
          start_array[sp].msec = ud.btime.msec

;printf, ofd, sp, start_array[sp], curr_time 

          stop_array[sp].year = ud.etime.year
          stop_array[sp].doy  = ud.etime.doy
          stop_array[sp].hour = ud.etime.hour
          stop_array[sp].min  = ud.etime.min
          stop_array[sp].sec  = ud.etime.sec
          stop_array[sp].msec = ud.etime.msec
       
          new_spin = 0L
          sp = sp + 1L

       endif



       ii = ii + 1L
       prev_ss = ss

   endwhile

;close, ofd
;free_lun, ofd

   udf_close, uh

;openw, ofd, 'new_time_data', /get_lun

   ;--------------------------------------------------
   ;  fill the time in seconds array
   ;--------------------------------------------------
   for i = 0, sp - 1L do begin
      tmp = l_GetTime (start_array[i], doy, hr, min, sec)
      for j = 0, 44 do begin
         ttime[(i*45)+j] = tmp
      endfor
      ;printf, ofd, start_array[i], ttime[i*45]
   endfor

;close, ofd
;free_lun, ofd


   ;------------------------------------------------------
   ; valid events data, sum over all species, energies, 
   ; and  polar bins
   ;------------------------------------------------------
   l_setup.l_num_spin = sp 

   ;new_nrec = (sp+1L) * 45L
   new_nrec = sp * 45L
;print, 'new_nrec = ', new_nrec
;print, 'sp = ', sp

   mass = m0data + m1data

   ;------------------------------------------------------
   ;for spin = 0L, sp - 1L do begin
   ;   for ss = 0L, 44L do begin
   ;       for p = 0L, 11L do begin
   ;           for e = 0L, 2 do begin
   ;              vedata[ss,spin] = vedata[ss,spin] + mass[e,p,ss,spin]
   ;           endfor
   ;       endfor
   ;   endfor
   ;endfor
   ;------------------------------------------------------
   ; use of total is equivalent to above 4 for loops
   ;------------------------------------------------------
   temp = total (mass, 1)
   vedata = total (temp, 1)

;help, vedata

   ;--------------------
   ;  setup data ptr
   ;--------------------
   ptr = ptr_new (vedata[*,0:sp-1L], /NO_COPY)
   ;ptr = ptr_new (vedata, /NO_COPY)
   binndata.vePtr = ptr

help, ttime
help, stime
help, start_array
;openw, ifd, 'inside_BEFORE_read_binned_data.txt', /get_lun
;nitems = n_elements (start_array)
;for i = 0, nitems - 1L do begin
;printf, ifd, start_array[i], ttime[i*45]
;endfor
;free_lun, ifd
;close, ifd

   ptr=ptr_new(ttime[0:new_nrec-1L], /NO_COPY)
   ;ptr=ptr_new(ttime, /NO_COPY)
   binndata.timePtr=ptr
print, 'timePtr'
help, (*binndata.timePtr)

   ptr=ptr_new(stime[0:sp-1L], /NO_COPY)
   ;ptr=ptr_new(stime, /NO_COPY)
   binndata.spinTimePtr=ptr
print, 'spinTimePtr'
help, (*binndata.spinTimePtr)

   ptr=ptr_new(m0data[*,*,*,0:sp-1L], /NO_COPY)
   ;ptr=ptr_new(m0data, /NO_COPY)
   binndata.m0RawPtr=ptr

   ptr=ptr_new(m1data[*,*,*,0:sp-1L], /NO_COPY)
   ;ptr=ptr_new(m1data, /NO_COPY)
   binndata.m1RawPtr=ptr

   ptr=ptr_new(start_array[0:sp-1L], /NO_COPY)
   ;ptr=ptr_new(start_array, /NO_COPY)
   binndata.startUdfTimePtr=ptr

   ptr=ptr_new(stop_array[0:sp-1L], /NO_COPY)
   ;ptr=ptr_new(stop_array, /NO_COPY)
   binndata.stopUdfTimePtr=ptr

;help, (*binndata.timePtr)
;help, (*binndata.spinTimePtr)

;openw, ifd, 'inside_read_binned_data.txt', /get_lun
;nitems = n_elements ((*binndata.startUdfTimePtr))
;for i = 0, nitems - 1L do begin
;printf, ifd, (*binndata.startUdfTimePtr)[i], (*binndata.timePtr)[i*45]
;endfor
;free_lun, ifd
;close, ifd

   RETURN, SUCCESS 

END


;-------------------------------------------------------------------------------
;  Function: l_BinnedReadSVL
;
;  Description: Read the steering voltage level from the housekeeping data and
;      sync it to the binned data using time.
;
;  Return Value:  type = int
;  Value                         Description
;  -------------------------     ------------------------------
;   SUCCESS                      successful completion
;   FAILURE                      unsuccessful completion
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_setup               struct        I        general setup parameters
;  l_plot_setup          struct        I        binned plotting parameters
;  binndata              struct *      I        pointers to binned data structures
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;      ERROR_TYPES     SUCCESS         int         G          successful completion
;                      FAILURE         int         G          unsuccessful completion 
;                      WARNING         int         G          warning message
;                      INFO            int         G          informational message
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11             Original implementation
;-------------------------------------------------------------------------------

FUNCTION l_BinnedReadSVL, l_setup, l_plot_setup, binndata
COMMON ERROR_TYPES, SUCCESS, FAILURE, WARNING, INFO




   inst = 'LENA'
   expr = 'LENAHSKP'
   vinst = 'IMLHSKP'

   data_source = ' '
   start_tme = l_setup.l_start_dtime
   stop_tme = l_setup.l_stop_dtime

   nspins = 0L


   ;------------------------------------------------
   ; do pass 1, determine the number of records 
   ; that need to be read
   ;------------------------------------------------
   nspins = l_UdfGetNumRec (start_tme, $
                          stop_tme,  $
                          inst, expr, vinst, data_source )
   if (nspins le 0) then return, FAILURE



   ;------------------------------------------------
   ; set up data structures for storage
   ;------------------------------------------------
   btime = (*binndata.spinTimePtr)
   bnspins = n_elements (btime)

   nspins = nspins + 15L

   ttime = make_array (nspins, /DOUBLE, Value = -9)
   steervdata = make_array (nspins, /UINT, value = -9)
   start_array = replicate ({UDF_TIMES_STRUCT}, nspins)
   stop_array = replicate ({UDF_TIMES_STRUCT}, nspins)
   ;--- BAD --- steering   = make_array(nspins, /INT, value = -9) ---


   ;------------------------------------------------
   ; open the UDF files
   ;------------------------------------------------
   uh = l_UdfGetDataHandle(inst, expr, vinst, $
                           start_tme, stop_tme, /GROUP)


   if (uh eq -1 ) then begin
      udf_close, uh
      RETURN, FAILURE
   endif


   ii=0L
   ;------------------------------------------------
   ; read the UDF data a record at a time
   ;------------------------------------------------
   while (not udf_eof(uh)) do begin


       ud = udf_read(uh)
       ttime[ii] = l_GetTime(ud.btime, doy, hr, min, sec)
       start_array[ii].year = ud.btime.year
       start_array[ii].doy  = ud.btime.doy
       start_array[ii].hour = ud.btime.hour
       start_array[ii].min  = ud.btime.min
       start_array[ii].sec  = ud.btime.sec
       start_array[ii].msec = ud.btime.msec

       stop_array[ii].year = ud.etime.year
       stop_array[ii].doy  = ud.etime.doy
       stop_array[ii].hour = ud.etime.hour
       stop_array[ii].min  = ud.etime.min
       stop_array[ii].sec  = ud.etime.sec
       stop_array[ii].msec = ud.etime.msec
 
       ;-----------------------------------------
       ; using HVPS_OPTICS_STEERING_CMD
       ;-----------------------------------------
       steervdata[ii]=ud.HVPS_CONFIGURATION[0,13]
       ii = ii + 1L
   endwhile

   udf_close, uh


   ;----------------------------------------
   ; sync hskp data time to image data time
   ;----------------------------------------


   last_ii = 0L
   bitems = n_elements (btime)
   hitems = n_elements (ttime)

   steering   = make_array(bitems, /INT, value = -9)

help, steering
print, 'bitems = ', bitems

   cnt = 0L
   JUMP3:
   for ii = last_ii, bitems - 1L do begin
       for jj = 0L, hitems - 1L do begin
           if btime[ii] eq ttime[jj] then begin
              steering[ii] = steervdata[jj]
              cnt = cnt + 1L
              last_ii = ii + 1L
              goto, JUMP3
            endif else begin
              steering[ii] = -9
            endelse
        endfor
   endfor


   ;--------------------
   ;  setup data ptr
   ;--------------------
   ptr=ptr_new(steering, /NO_COPY)
   binndata.steervPtr=ptr

   ptr = ptr_new(start_array, /NO_COPY)
   binndata.strStUdfTimePtr = ptr

   ptr = ptr_new(stop_array, /NO_COPY)
   binndata.strSpUdfTimePtr = ptr


   RETURN, SUCCESS

END


;-------------------------------------------------------------------------------
;  Function: l_BinnedReadData
;
;  Description: Define and initialize data structures to read the binned data
;        from the UDF database.
;
;  Return Value:  type = int
;  Value                         Description
;  -------------------------     ------------------------------
;   SUCCESS                      successful completion
;   FAILURE                      unsuccessful completion
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_setup               struct        I        general plotting parameters
;  l_plot_setup          struct        I        binned plotting parameters
;  est_nspins            int           I        estimated number of spins
;  binndata              struct *      I        pointers to binned data structures
;  l_pltpos              struct        I        general plotting parameters
;  start_tme             array         I        user specified start time
;  stop_tme              array         I        user specified stop time
;  
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;      ERROR_TYPES     SUCCESS         int         G          successful completion
;                      FAILURE         int         G          unsuccessful completion 
;                      WARNING         int         G          warning message
;                      INFO            int         G          informational message
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------

FUNCTION l_BinnedReadData, l_setup, l_plot_setup, est_nspins, binndata, l_pltpos, start_tme, $
stop_tme
COMMON ERROR_TYPES, SUCCESS, FAILURE, WARNING, INFO

   ;--------------------
   ;  define data structures
   ;--------------------
   ptr=ptr_new()       ; NULL ptr
   binndata = {              $
       timePtr        : ptr, $   ;begin time for each spin sector
       filledTimePtr  : ptr, $   ;missing data included 
       spinTimePtr    : ptr, $   ;begin time for each spin - 1 time/spin
       startUdfTimePtr: ptr, $   ;udf start time
       filledStartUdfTimePtr : ptr, $ ;missing data included
       stopUdfTimePtr : ptr, $   ;udf start time
       m0RawPtr       : ptr, $   ;m0 telemetry, 3 energy levels
       filledM0RawPtr : ptr, $   ;missing data included, m0 telemetry, 3 energy levels
       m0RawExtraPtr  : ptr, $   ;m0 telemetry, 3 energy levels, 2 extra spins
       m1RawPtr       : ptr, $   ;m1 telemetry, 3 energy levels
       filledM1RawPtr : ptr, $   ;missing data included, m1 telemetry, 3 energy levels
       m1RawExtraPtr  : ptr, $   ;m1 telemetry, 3 energy levels, 2 extra spins
       m0CntPtr       : ptr, $   ;m0 count rate, 3 energy levels
       filledM0CntPtr : ptr, $   ;missing data included, m0 count rate, 3 energy levels
       m0CntExtraPtr  : ptr, $   ;m0 count rate, 3 energy levels, 2 extra spins
       m1CntPtr       : ptr, $   ;m1 count rate, 3 energy levels
       filledM1CntPtr : ptr, $   ;missing data included, m1 count rate, 3 energy levels
       m1CntExtraPtr  : ptr, $   ;m1 count rate, 3 energy levels, 2 extra spins
       m0FlxPtr       : ptr, $   ;m0 number flux, 3 energy levels
       m1FlxPtr       : ptr, $   ;m1 number flux, 3 energy levels
       vePtr          : ptr, $   ;all masses (valid events) 
       filledVePtr    : ptr, $   ;missing data included, all masses (valid events) 
       veExtraPtr     : ptr, $   ;all masses (valid events) 
       veDmpPtr       : ptr, $   ;all masses (valid events) Dump
       m0Ptr          : ptr, $   ;m0 (H) all energies, polar zone [sector, spin]
       filledM0Ptr    : ptr, $   ;missing data included, m0 (H) all energies, polar zone [sector, spin]
       m0ExtraPtr     : ptr, $   ;m0 (H) all energies, polar zone [sector, spin]
       m0DmpPtr       : ptr, $   ;m0 (H) all energies, polar zone [sector, spin], 
                                 ;text dump version
       m0lePtr        : ptr, $   ;m0 (H) lower energy 
       filledM0lePtr  : ptr, $   ;missing data included, m0 (H) lower energy 
       m0leExtraPtr   : ptr, $   ;m0 (H) lower energy 
       m0leDmpPtr     : ptr, $   ;m0 (H) lower energy, text dump version
       m0cePtr        : ptr, $   ;m0 (H) center energy 
       filledM0cePtr  : ptr, $   ;missing data included, m0 (H) center energy 
       m0ceExtraPtr   : ptr, $   ;m0 (H) center energy 
       m0ceDmpPtr     : ptr, $   ;m0 (H) center energy, text dump version
       m0uePtr        : ptr, $   ;m0 (H) upper energy 
       filledM0uePtr  : ptr, $   ;missing data included, m0 (H) upper energy 
       m0ueExtraPtr   : ptr, $   ;m0 (H) upper energy 
       m0ueDmpPtr     : ptr, $   ;m0 (H) upper energy , text dump version
       m1Ptr          : ptr, $   ;m1 (O)  all energies, polar zone [sector, spin]
       filledM1Ptr    : ptr, $   ;missing data included, m1 (O)  all energies, polar zone [sector, spin]
       m1ExtraPtr     : ptr, $   ;m1 (O)  all energies, polar zone [sector, spin]
       m1DmpPtr       : ptr, $   ;m1 (O)  all energies, polar zone [sector, spin], 
                                 ;text dump version
       m1lePtr        : ptr, $   ;m1 (O) lower energy
       filledM1lePtr  : ptr, $   ;missing data included, m1 (O) lower energy
       m1leExtraPtr   : ptr, $   ;m1 (O) lower energy
       m1leDmpPtr     : ptr, $   ;m1 (O) lower energy, text dump version
       m1cePtr        : ptr, $   ;m1 (O) center energy 
       filledM1cePtr  : ptr, $   ;missing data included, m1 (O) center energy 
       m1ceExtraPtr   : ptr, $   ;m1 (O) center energy 
       m1ceDmpPtr     : ptr, $   ;m1 (O) center energy, text dump version
       m1uePtr        : ptr, $   ;m1 (O) upper energy
       filledM1uePtr  : ptr, $   ;missing data included, m1 (O) upper energy
       m1ueExtraPtr   : ptr, $   ;m1 (O) upper energy
       m1ueDmpPtr     : ptr, $   ;m1 (O) upper energy, text dump version
       strStUdfTimePtr:ptr,  $   ;steering start UDF time
       filledStrStUdfTimePtr:ptr,  $   ;missing data included, steering start UDF time
       strSpUdfTimePtr:ptr,  $   ;steering stop UDF time
       filledStrSpUdfTimePtr:ptr,  $   ;missing data included, steering stop UDF time
       steervPtr      : ptr, $   ;steering voltage levels / spin
       filledSteervPtr: ptr,  $   ;missing data included, steering voltage levels / spin
       bkcUdfTimePtr  : ptr, $    ;background corrected UDF times
       hydrogen_raw_cnts      : ptr, $  ;hydrogen raw counts
       hydrogen_raw_cnts_e0   : ptr, $  ;hydrogen raw counts for energy 0
       hydrogen_raw_cnts_e1   : ptr, $  ;hydrogen raw counts for energy 1
       hydrogen_raw_cnts_e2   : ptr, $  ;hydrogen raw counts for energy 2
       corrected_hydrogen     : ptr, $  ;background corrected hydrogen counts
       corrected_hydrogen_e0  : ptr, $  ;background corrected hydrogen counts for energy 0
       corrected_hydrogen_e1  : ptr, $  ;background corrected hydrogen counts for energy 1 
       corrected_hydrogen_e2  : ptr, $  ;background corrected hydrogen counts for energy 2 
       corr_scl_H             : ptr, $  ;background corrected scaled hydrogen counts
       corr_scl_H_e0          : ptr, $  ;background corrected scaled hydrogen counts for E0
       corr_scl_H_e1          : ptr, $  ;background corrected scaled hydrogen counts for E1
       corr_scl_H_e2          : ptr, $  ;background corrected scaled hydrogen counts for E2
       oxygen_raw_cnts        : ptr, $  ;oxygen raw counts
       oxygen_raw_cnts_e0     : ptr, $  ;oxygen raw counts for energy 0
       oxygen_raw_cnts_e1     : ptr, $  ;oxygen raw counts for energy 1
       oxygen_raw_cnts_e2     : ptr, $  ;oxygen raw counts for energy 2
       corrected_oxygen       : ptr, $  ;background corrected oxygen counts
       corrected_oxygen_e0    : ptr, $  ;background corrected oxygen counts for energy 0
       corrected_oxygen_e1    : ptr, $  ;background corrected oxygen counts for energy 1
       corrected_oxygen_e2    : ptr, $  ;background corrected oxygen counts for energy 2
       corr_scl_O             : ptr, $  ;background corrected scaled oxygen counts 
       corr_scl_O_e0          : ptr, $  ;background corrected scaled oxygen counts for E0
       corr_scl_O_e1          : ptr, $  ;background corrected scaled oxygen counts for E1
       corr_scl_O_e2          : ptr  $  ;background corrected scaled oxygen counts for E2
   }  


   ;--------------------
   ;  get num image rec
   ;--------------------
   inst = 'LENA'
   expr = 'LENASCI'
   vinst = 'IMLIMAGE'
   data_source = ' '

   ;---------------------------------------------------------------
   ; estimate the number of spins in the user specified time range
   ;---------------------------------------------------------------
   est_nspins = 0L
   ;est_nspins = round((ConvArrTimeToSecs(stop_tme) - $
   ;             ConvArrTimeToSecs(start_tme))/120L) + 15L


   est_nspins = l_UdfGetNumRec (start_tme, $
                                stop_tme,  $
                                inst, expr, vinst, data_source )
   if (est_nspins le 0) then return, FAILURE
   est_nspins = (est_nspins / 45L) + 3L


   ;--------------------
   ;  get image data
   ;--------------------
   status = l_BinnedReadUdfDataByTime (l_setup, est_nspins, $
                        binndata, start_tme, stop_tme, data_source) 

   if (status eq FAILURE) then return, FAILURE


   ;---------------------------------------
   ;  get hskp steering voltage levels
   ;---------------------------------------

   l_pltpos.data_source = data_source
   status = l_BinnedReadSVL (l_setup, l_plot_setup, binndata)


   RETURN, status 

END

