
;-------------------------------------------------------------------------------
;
;  Unit Name : l_BEDump.pro
;
;  Purpose   : Dump to a text file the binned and event data summed counts.
;              Counts are summed over all masses, energies, & polar zones.
;              This is a special routine written for Michael Collier.  These 
;              text dumps are NOT a part of the standard production items.
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;   ELH              04/23/03       v1.0.11           Original implementation
;
;
;  File Revision Number:   %I%
;  File Last Modified  :   %E%  %T%
;
;-------------------------------------------------------------------------------

;-------------------------------------------------------------------------------
;  Procedure: BinDump
; 
;  Description: Dump the hydrogen and oxygen summed over all energies and polar
;               zone counts into a text file. 
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_binned              struct *      I        pointers to binned data types
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03        v1.0.11         Original implementation
;-------------------------------------------------------------------------------


PRO BinDump, l_binned


    openw, ofd, /get_lun, 'sumBinn.dmp'

    printf, ofd, '----------------------- Column Descriptions -----------------------------'
    printf, ofd, '                      BINNED DATA         '
    printf, ofd, ' '
    printf, ofd, '      UTC             : Universal Time Coordinate (corrected)'
    printf, ofd, '      Azi             : Azimuthal spin sector (0-44)'
    printf, ofd, '      Sum Counts      : Sum of Hydrogen and Oxygen counts
    printf, ofd, '                        summed over energy, polar, and mass '
    printf, ofd, ' '
    printf, ofd, '-------------------------------------------------------------------------'
    printf, ofd , ' '
    printf, ofd , ' '
    printf, ofd , ' '


    ;-----------------------------
    ;  Time, sector, Sum H/O
    ;-----------------------------

    printf, ofd ,'UTC','Azi', 'Sum H/O Counts', $
                     format='((A8,TR4), (A6, TR7) , (A16, TR4))'

    printf, ofd ,'   ','   ', ' ',   $
                    format='((A8,TR4), (A6,TR7) , (A12, TR2))'

    printf, ofd,'---','---', '---------------',  $
                    format='((A8,TR4), (A6, TR7) , (A18, TR2))'


    sum = (*l_binned.hydrogen_raw_cnts) + (*l_binned.oxygen_raw_cnts)
    nspins = n_elements (*l_binned.startUdfTimePtr)

    for i = 0L, nspins - 1L do begin
        tmestr = STRING ((*l_binned.startUdfTimePtr)[i].doy, $
                         (*l_binned.startUdfTimePtr)[i].hour,$
                         (*l_binned.startUdfTimePtr)[i].min, $
                         (*l_binned.startUdfTimePtr)[i].sec, $
                         FORMAT="(i3,' ',i2.2,':',i2.2,':',i2.2)")

        for j = 0L, 44L do begin
           printf, ofd, tmestr, $
                            j, $
                            sum[j, i], $
                            format='((A12,TR2), (I4,TR8), (e12.3,TR2))'

        endfor
    endfor


    close, ofd
    free_lun, ofd


RETURN
END


;-------------------------------------------------------------------------------
;  Procedure: EventDump
; 
;  Description: Dump the hydrogen and oxygen summed over all energies and polar
;               zone counts into a text file. 
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_event               struct *      I        pointers to event data types
;  l_event_setup         struct        I        event processing & plotting parameters
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03        v1.0.11         Original implementation
;-------------------------------------------------------------------------------


PRO EventDump, l_event, l_event_setup

    openw, ofd, /get_lun, 'sumEvent.dmp'

    ;---------------------------------
    ; Time, sector, Sum H/O
    ;---------------------------------

    printf, ofd, '----------------------- Column Descriptions -----------------------------'
    printf, ofd, '                      EVENT DATA         '
    printf, ofd, ' '
    printf, ofd, '      UTC             : Universal Time Coordinate (corrected)'
    printf, ofd, '      Azi             : Azimuthal spin sector (0-44)'
    printf, ofd, '      Sum Counts      : Sum of Hydrogen and Oxygen counts
    printf, ofd, ' '
    printf, ofd, '-------------------------------------------------------------------------'
    printf, ofd , ' '
    printf, ofd , ' '
    printf, ofd , ' '


    printf, ofd, 'Hydrogen TOF bin range : ' , string(l_event_setup.hydrogen_beg_chan) + '-' + $
                   string(l_event_setup.hydrogen_end_chan) 

    printf, ofd, 'Oxygen TOF bin range   : ', string(l_event_setup.oxygen_beg_chan) + '-' + $
                   string(l_event_setup.oxygen_end_chan)

   
    printf, ofd, ' '
    printf, ofd, ' '

    printf, ofd ,'UTC','Azi', 'Sum H/O Counts', $
                     format='((A8,TR4), (A6, TR7) , (A16, TR4))'

    printf, ofd ,'   ','   ', ' ',   $
                    format='((A8,TR4), (A6,TR7) , (A12, TR2))'

    printf, ofd,'---','---', '---------------',  $
                    format='((A8,TR4), (A6, TR7) , (A18, TR2))'


    sum = (*l_event.H_n_in_peak) + (*l_event.O_n_in_peak)

    nspins = n_elements (*l_event.bkcUdfTimePtr)

    for i = 0L, nspins - 1L do begin
        tmestr = STRING ((*l_event.bkcUdfTimePtr)[i].doy, $
                         (*l_event.bkcUdfTimePtr)[i].hour,$
                         (*l_event.bkcUdfTimePtr)[i].min, $
                         (*l_event.bkcUdfTimePtr)[i].sec, $
                         FORMAT="(i3,' ',i2.2,':',i2.2,':',i2.2)")

        for j = 0L, 44L do begin
           printf, ofd, tmestr, $
                            j, $
                            sum[j, i], $
                            format='((A12,TR2), (I4,TR8), (e12.3,TR2))'

         endfor
    endfor


    close, ofd
    free_lun, ofd


RETURN
END
