
;--------------------------------------------------------------------------------
;
;  Unit Name : l_ApogeeRtns.pro
;
;  Purpose   : Plot apogees for a given orbit. This is used for all other LENA 
;              plots and not for the long term trending plots.
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               11/00           v1.0            Original Implementation
;  ELH               05/01           v1.0            Updated read routines to return
;                                                      status
;                                                    Added l_AdjPlotSize 
;
;
;  File Revision Number:   %I%
;  File Last Modified  :   %E%  %T%
;
;--------------------------------------------------------------------------------


;--------------------------------------------------------------------------------
;  Procedure: l_ApogeeFind 
;  Description:  Find the apogee times in the orbit
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_oa                  struct *      I        pointer to OA data fields
;  apogee_times          double        O        array of apogee times
;  flag                  int           I        indicates if for a day or more 
;  ndays                 int           I        the number of days
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               11/00           v1.0            Original Implementation
;--------------------------------------------------------------------------------

PRO l_ApogeeFind, l_oa, apogee_times, flag, ndays

   nspins = n_elements ((*l_oa.gsm_x_pos_ptr)[*])

   if (flag ne 'ltt') then begin
       NUMRECS = 5L
   endif else begin
       NUMRECS = ndays * 3L 
   endelse
      
   apogee  = dblarr (NUMRECS)

   a_idx = 0L

   ;------------------------------------------------
   ; convert the gsm coordinates to sm
   ;------------------------------------------------

   dist = dblarr (nspins)


   for i = 0L, nspins-1L do begin

      RECALC, (*l_oa.startUdfTimePtr)[i].year, (*l_oa.startUdfTimePtr)[i].doy, $
              (*l_oa.startUdfTimePtr)[i].hour, (*l_oa.startUdfTimePtr)[i].min, $
              (*l_oa.startUdfTimePtr)[i].sec

      SMGSM, xsm, ysm, zsm, $
             (*l_oa.gsm_x_pos_ptr)[i], $
             (*l_oa.gsm_y_pos_ptr)[i], $
             (*l_oa.gsm_z_pos_ptr)[i], $
             -1

       dist[i] = sqrt(xsm*xsm + ysm*ysm + zsm*zsm)

  endfor


  ;------------------------------------------------
  ; find the apogee times 
  ;------------------------------------------------
  for i = 1L, nspins-2L do begin
      if (dist[i] gt dist[i-1]) and (dist[i] gt dist[i+1]) then begin
         apogee[a_idx] =  (*l_oa.tme_ptr)[i]
         a_idx = a_idx + 1L
      endif
  endfor

  if (a_idx eq 0) then begin
     print, 'No apogees.'
     return
  endif

  apogee_times = apogee[0:a_idx-1L]


RETURN
END


;--------------------------------------------------------------------------------
;  Procedure:  l_ApogeeProc
;  Description:
;    Calls the necessary routines to read the data to generate
;    the apogee plots.
;
;  Return Value:  type = int
;  Value                         Description
;  -------------------------     ------------------------------
;  SUCCESS                       successful completion
;  FAILURE                       unsuccessful completion
;  
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_setup               struct        I        setup parameters
;  l_setup_file          string        I        setup filename
;  l_pltpos              struct        I        plot parameters
;  l_rgb_table           struct        I        RGB color table specifications
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;     PLOTDEFS         MAX_PLOTS        int         G         max # plots per page
;                      MAX_WEB_SPINS    int         G         max # spins per plot
;                                                             when using web interface
;                      NO_CNTS          double      G         no counts indicator
;                      NO_DATA          double      G         no data indicator     
;     ERROR_TYPES      SUCCESS          int         G         successful completion
;                      FAILURE          int         G         unsuccessful completion
;                      WARNING          int         G         warning status
;                      INFO             int         G         informational message
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;--------------------------------------------------------------------------------

FUNCTION l_ApogeeProc, l_setup, l_setup_file, l_pltpos, $
l_rgb_table
COMMON PLOTDEFS, MAX_PLOTS, MAX_WEB_SPINS, NO_CNTS, NO_DATA
COMMON ERROR_TYPES, SUCCESS, FAILURE, WARNING, INFO

   hr_n_secs     = 3600. 
   start_apogee = 0.0D
   stop_apogee  = 0.0D

   l_pltpos.plot_hdr_title = '!5Apogee'
   l_pltpos.X_axis_eq_time = 'Y'

   ;-------------------------------------------------------
   ; .setup buffer
   ; .read the oa data
   ; .find the apogee times 
   ; .for the number of apogee times do
   ; .add and subtract one hour to the time
   ; .reconvert the start/stop time back to UDF time
   ; .check the user start and stop time 
   ; .read the singles and binned data
   ; .plot the apogee 
   ; .terminate 
   ;-------------------------------------------------------

   year      = strmid(systime(0), 20, 23)
   l_setup.l_line_plots = 'N'

   ;-------------------------------------------------------
   ; MUST BE A 24 HOUR PERIOD
   ;-------------------------------------------------------
   start_time = [l_setup.l_start_dtime[0], $
                 l_setup.l_start_dtime[1], $
                 0, 0, 0]

   stop_time  = [l_setup.l_start_dtime[0], $
                 l_setup.l_start_dtime[1], $
                 23, 59, 59]

   status = l_OARead (l_oa, $
             start_time, $
             stop_time, $
             l_pltpos)
   if (status  eq FAILURE) then return, FAILURE

   start_ndays = l_ConvYMD2DOY (l_setup.l_start_date[0], $
                                l_setup.l_start_date[1], $
                                l_setup.l_start_date[2])
   stop_ndays = l_ConvYMD2DOY (l_setup.l_stop_date[0], $
                               l_setup.l_stop_date[1], $
                               l_setup.l_stop_date[2])
   ndays = stop_ndays - start_ndays

   l_ApogeeFind, l_oa, $
                 apogee_times, $
                 'day', ndays

   l_SnglsReadSetup, l_sngls_setup, l_setup_file

   l_CoincReadSetup, l_coinc_setup, l_setup_file

   l_BinnedReadSetup, l_binned_setup, l_setup_file

   n_apogee = n_elements (apogee_times)


   for i = 0L, n_apogee-1L do begin
    
      status = 0L
      start_apogee = apogee_times[i] - hr_n_secs
      stop_apogee  = apogee_times[i] + hr_n_secs

      l_setup.l_start_dtime = ConvSec2Arr_CalDOY (start_apogee, $
                                  l_setup.l_start_date[0])
      l_setup.l_stop_dtime  = ConvSec2Arr_CalDOY (stop_apogee, $
                                  l_setup.l_start_date[0])

      status = l_OARead (l_oa, $
                l_setup.l_start_dtime, $
                l_setup.l_stop_dtime, $
                l_pltpos)
      if (status eq FAILURE) then return, FAILURE


      status = l_HskpGetData (l_setup, $
                     l_setup_file, $
                     l_pltpos, $
                     l_hskp_setup, $
                     l_hskp)
      if (status eq FAILURE) then return, FAILURE


      if (l_sngls_setup.start eq 'Y') OR (l_sngls_setup.stop eq 'Y') then begin
         status = l_SnglsGetData (l_setup, $
                                  l_setup_file, $
                                  l_pltpos, $
                                  l_sngls_setup, $
                                  l_sngls)
         if (status eq FAILURE) then return, FAILURE

         l_SnglsAdjExData, l_sngls_setup, l_setup.l_num_spinsec, l_sngls
         l_SnglsHndlMissData, l_sngls_setup, l_sngls, l_setup, l_oa
      endif


      status = l_BinnedEventSelect (l_setup, l_setup_file, l_pltpos, $
                                    l_roi0, l_binned, l_binned_setup, $
                                    l_event, l_event_setup, l_memd, l_memd_setup, $
                                    l_evsp, l_evsp_setup, l_hskp, l_oa, davail)
      if (status eq FAILURE) then return, FAILURE


      if (status ge 0L) then begin
         ;----------------------------------------
         ; generate a spectrogram plot
         ;----------------------------------------
         nloops = 1L
         if (l_setup.l_postscript eq 'Y') then nloops = 2L
         for j = 0L, nloops-1L do begin
            nplots = 0L
            nrows = 0L
            npages = 0L
            nrows = l_sngls_setup.nplots

            if (l_binned_setup.use_binned eq 'Y') then begin
                nrows = nrows + l_binned_setup.nplots
            endif

            if (l_event_setup.use_event eq 'Y') then begin
                nrows = nrows + l_event_setup.nplots
            endif


            nplots = nrows
            npages = CEIL(nrows / MAX_PLOTS)

            if (nrows - MAX_PLOTS gt 0L) then npages = npages + 1L
            if (npages eq 0L) then npages = 1L
            if (npages gt 0L) then windx = intarr (npages)

            l_setup.l_filedesc = 'apo'

            plotted = 0L
            pages   = 0L
            web     = l_setup.l_via_web

            l_AdjPlotSize, l_pltpos, l_setup, nplots

            if (j eq 1) and (l_setup.l_postscript eq 'Y') then begin
               l_pltpos.ps = 'Y'
               l_GenFileName, l_setup, '0'
               l_pltpos.filename = l_setup.l_prod_dst + '/' + $
                                l_setup.l_filename + '.ps'
            endif

            l_pltpos.linePlots = l_setup.l_line_plots

            l_SetUpZBuffer, l_pltpos, $
                            l_setup, $
                            l_rgb_table, $
                            colorbar, $
                            nplots


             if (l_setup.l_pwrsup_bars eq 'Y') then begin
               l_DrawOpBars, l_hskp, l_setup, l_pltpos, l_rgb_table, l_oa
             endif


            if (l_sngls_setup.start eq 'Y') OR (l_sngls_setup.stop eq 'Y') then begin
                l_SnglsPlot, l_sngls_setup, $
                             l_sngls, $
                             l_pltpos, $
                             nrows, $
                             windx, $
                             pages, $
                             plotted,$
                             l_setup, $
                             l_oa, $
                             l_rgb_table, $
                             colorbar, $
                             l_hskp
            endif

            if (l_binned_setup.use_binned eq 'Y') OR (l_event_setup.use_event eq 'Y') then begin

                 l_BinnedEventPlot, l_setup, $
                                    l_binned_setup, $
                                    l_event_setup, $
                                    l_binned, $
                                    l_event, $
                                    l_pltpos, $
                                    l_roi0, $
                                    l_oa, $
                                    l_hskp, $
                                    l_rgb_table, $
                                    nrows, windx, plotted, $
                                    pages, colorbar
             endif


            if (j eq 1) and (l_setup.l_postscript eq 'Y') then begin
                device, /close_file
            endif

            if (!D.NAME eq 'Z') then begin
               if (l_setup.l_via_web eq 'Y') or (l_setup.l_gen_sum eq 'Y') then begin
                  device, /close
               endif
            endif

           endfor ;---- nloops ----
        endif


        if (l_sngls_setup.start eq 'Y') OR (l_sngls_setup.stop eq 'Y') then l_SnglsTerm, l_sngls
        if (l_coinc_setup.coinc eq 'Y') then l_BinnedTerm, l_binned 
        if (l_binned_setup.use_binned eq 'Y') OR (l_event_setup.use_event eq 'Y') then begin
            l_BinnedEventTerm, l_roi0, l_binned, l_binned_setup, $
                               l_event, l_event_setup, l_memd, $
                               l_evsp, l_hskp, l_oa, l_setup, davail
     endif

     l_OATerm, l_oa


   endfor

   RETURN, SUCCESS

END
