
;-------------------------------------------------------------------------------
;
;  Unit Name : l_SnglsRtns.pro
;
;  Purpose   : Contains all routines related to processing the Singles data.
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0            Original implementation
;  ELH               05/01           v1.0            Added l_AdjPlotSize 
;  ELH               07/05/01        v1.0            Modified to generate a dump
;                                                    before conversions, but after
;                                                    handling missing data 
;  ELH               09/18/01        v1.0.7          Draw sun direction
;  ELH               09/26/01        v1.0.7          Added l_DetermineSunDir 
;  ELH               01/02           v1.0.8          Added parm in l_HndlMissData to
;                                                    handle rolling data based on
;                                                    spin angle input
;  ELH               01/27/2004      v1.0.11         Deleted check for < 4 spins
;  ELH                 05/04         v1.0.11         Modified l_SnglsReadSetup to
;                                                    convert input string to lowercase
;
;
;  File Revision Number:   %I%
;  File Last Modified  :   %E%  %T%
;
;-------------------------------------------------------------------------------

;-------------------------------------------------------------------------------
;  Procedure: l_SnglsAdjExData
;
;  Description: If extra spins read at the beginning and end of user specified
;     time period, adjust the arrays for these extra spins.  This is needed when
;     adjusting the various traces to zero degree spin angle.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_sngls_setup         struct        I        singles processing & plotting parameters
;  SPIN_SECTORS          long          I        number of spin sectors
;  l_sngls               struct *      I        pointers to the singles data
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;     PLOTDEFS         MAX_PLOTS        int         G         max # plots per page
;                      MAX_WEB_SPINS    int         G         max # spins per plot
;                                                             when using web interface
;                      NO_CNTS          double      G         no counts indicator
;                      NO_DATA          double      G         no data indicator
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;  ELH               01/27/04       v1.0.11           Deleted spin check
;-------------------------------------------------------------------------------


PRO l_SnglsAdjExData, l_sngls_setup, SPIN_SECTORS, l_sngls
COMMON PLOTDEFS, MAX_PLOTS, MAX_WEB_SPINS, NO_CNTS, NO_DATA



    extra_start  = make_array (SPIN_SECTORS, 2, /FLOAT, value = NO_DATA)
    extra_stop  = make_array (SPIN_SECTORS, 2, /FLOAT, value = NO_DATA)

    nele = n_elements ((*l_sngls.timePtr)) - 1
    numRecs = n_elements ((*l_sngls.startUdfTimePtr)) - 1L

;    if (nele lt (SPIN_SECTORS * 4)) then begin
;       ptr = ptr_new (extra_start)
;       l_sngls.extraStartPtr = ptr
;       ptr = ptr_new (extra_stop)
;       l_sngls.extraStopPtr = ptr
;       return
;    endif


    ;---------------------------------------------------------------------------
    ; if extra time was added to get an extra spin in the beginning and end
    ;---------------------------------------------------------------------------
    if (l_sngls_setup.addedStartFlag eq 1L) AND (l_sngls_setup.addedStopFlag eq 1L) then begin

       extra_start[*,0] = (*l_sngls.startPtr)[0:SPIN_SECTORS-1] 
       extra_start[*,1] = (*l_sngls.startPtr)[nele-SPIN_SECTORS:nele-1L] 

       if (ptr_valid (l_sngls.extraStartPtr)) then ptr_free, l_sngls.extraStartPtr
       ptr = ptr_new (extra_start, /NO_COPY)
       l_sngls.extraStartPtr = ptr

       extra_stop[*,0] = (*l_sngls.stopPtr)[0:SPIN_SECTORS-1L] 
       extra_stop[*,1] = (*l_sngls.stopPtr)[nele-SPIN_SECTORS:nele-1L] 

       if (ptr_valid (l_sngls.extraStopPtr)) then ptr_free, l_sngls.extraStopPtr
       ptr = ptr_new (extra_stop, /NO_COPY)
       l_sngls.extraStopPtr  = ptr

       (*l_sngls.startPtr) =  (*l_sngls.startPtr)[SPIN_SECTORS:nele-SPIN_SECTORS] 
       (*l_sngls.stopPtr) = (*l_sngls.stopPtr)[SPIN_SECTORS:nele-SPIN_SECTORS]
       (*l_sngls.timePtr) = (*l_sngls.timePtr)[SPIN_SECTORS:nele-SPIN_SECTORS]
       (*l_sngls.startUdfTimePtr) = (*l_sngls.startUdfTimePtr)[1:numRecs-1]
       (*l_sngls.stopUdfTimePtr) = (*l_sngls.stopUdfTimePtr)[1:numRecs-1]

    endif else begin

       ;---------------------------------------------------------------------------
       ; if extra time was added to get an extra spin in the beginning 
       ;---------------------------------------------------------------------------

       if (l_sngls_setup.addedStartFlag eq 1L) AND (l_sngls_setup.addedStopFlag eq 0L) then begin

          extra_start[*,0] = (*l_sngls.startPtr)[0:SPIN_SECTORS-1L] 
          extra_start[*,1] = NO_DATA 
          if (ptr_valid (l_sngls.extraStartPtr)) then ptr_free, l_sngls.extraStartPtr
          ptr = ptr_new (extra_start, /NO_COPY)
          l_sngls.extraStartPtr = ptr

          extra_stop[*,0] = (*l_sngls.stopPtr)[0:SPIN_SECTORS-1L] 
          extra_stop[*,1] = NO_DATA
          if (ptr_valid (l_sngls.extraStopPtr)) then ptr_free, l_sngls.extraStopPtr
          ptr = ptr_new (extra_stop, /NO_COPY)
          l_sngls.extraStopPtr  = ptr

          (*l_sngls.startPtr) = (*l_sngls.startPtr)[SPIN_SECTORS:nele] 
          (*l_sngls.stopPtr)  = (*l_sngls.stopPtr)[SPIN_SECTORS:nele]
          (*l_sngls.timePtr)  = (*l_sngls.timePtr)[SPIN_SECTORS:nele]
          (*l_sngls.startUdfTimePtr) = (*l_sngls.startUdfTimePtr)[1:numRecs] 
          (*l_sngls.stopUdfTimePtr) = (*l_sngls.stopUdfTimePtr)[1:numRecs]

       endif else begin

           ;---------------------------------------------------------------------------
           ; if extra time was added to get an extra spin in the  end
           ;---------------------------------------------------------------------------
           if (l_sngls_setup.addedStartFlag eq 0L) AND $
              (l_sngls_setup.addedStopFlag eq 1L) then begin


               extra_start[*,0] = NO_DATA 
               extra_start[*,1] = (*l_sngls.startPtr)[nele-SPIN_SECTORS:nele-1L] 
               if (ptr_valid (l_sngls.extraStartPtr)) then ptr_free, l_sngls.extraStartPtr
               ptr = ptr_new (extra_start, /NO_COPY)
               l_sngls.extraStartPtr = ptr

               extra_stop[*,0] = NO_DATA
               extra_stop[*,1] = (*l_sngls.stopPtr)[nele-SPIN_SECTORS:nele-1L] 
               if (ptr_valid (l_sngls.extraStopPtr)) then ptr_free, l_sngls.extraStopPtr
               ptr = ptr_new (extra_stop, /NO_COPY)
               l_sngls.extraStopPtr  = ptr

               (*l_sngls.startPtr) = (*l_sngls.startPtr)[0:nele-SPIN_SECTORS] 
               (*l_sngls.stopPtr)  = (*l_sngls.stopPtr)[0:nele-SPIN_SECTORS] 
               (*l_sngls.timePtr)  = (*l_sngls.timePtr)[0:nele-SPIN_SECTORS]
               (*l_sngls.startUdfTimePtr) = (*l_sngls.startUdfTimePtr)[0:numRecs-1L] 
               (*l_sngls.stopUdfTimePtr) = (*l_sngls.stopUdfTimePtr)[0:numRecs-1L]


           endif else begin

               ;--------------------------------------------
               ; no extra time was added 
               ;--------------------------------------------
               extra_start[*,*] =  NO_DATA 
               if (ptr_valid (l_sngls.extraStartPtr)) then ptr_free, l_sngls.extraStartPtr
               ptr = ptr_new (extra_start, /NO_COPY)
               l_sngls.extraStartPtr = ptr

               extra_stop[*,*] = NO_DATA 
               if (ptr_valid (l_sngls.extraStopPtr)) then ptr_free, l_sngls.extraStopPtr
               ptr = ptr_new (extra_stop, /NO_COPY)
               l_sngls.extraStopPtr  = ptr

           endelse
        endelse
    endelse


    return

END

;-------------------------------------------------------------------------------
;  Function : l_SnglsReadData
;
;  Description: Read the singles data a record at a time.
;
;  Return Value:  type = int
;  Value                         Description
;  -------------------------     ------------------------------
;  SUCCESS                       successful completion 
;  FAILURE                       failure to complete
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  inst                  string        I        instrument (LENA)
;  expr                  string        I        experiment (LENASCI or LENAHSKP)
;  vinst                 string        I        virtual instrument (name found
;                                               in $UDF_DATA in the experiment
;                                               directory under IMAGE1 or IMAGE1Q
;                                               data)
;  btime                 long[]        I        UDF begin time [yr,doy,hh,mm,ss]
;  etime                 long[]        I        UDF end time [yr,doy,hh,mm,ss]
;  btme                  double[]      O        UDF begin time converted to seconcds
;  SPIN_SECTORS          long          I        number of spins sectors, 45
;  start                 float[]       O        singles start data
;  stop                  float[]       O        singles stop data
;  start_array           struct        O        UDF time structure array, s/c corrected
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;      ERROR_TYPES     SUCCESS         int         G          successful completion
;                      FAILURE         int         G          unsuccessful completion
;                      WARNING         int         G          warning message
;                      INFO            int         G          informational message
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;-------------------------------------------------------------------------------

FUNCTION l_SnglsReadData, inst, expr, vinst, btime, etime, numRecs, l_sngls, $
data_source
COMMON ERROR_TYPES, SUCCESS, FAILURE, WARNING, INFO


   ;----------------------------------------------
   ; allocate space for to temporarily store the
   ; data 
   ;----------------------------------------------

   SPIN_SECTORS = 45L

   nele   = SPIN_SECTORS * numRecs
   start  = fltarr (nele)
   stop   = fltarr (nele)
   sctime = dblarr (nele)
   start_array = replicate ({UDF_TIMES_STRUCT}, numRecs)
   stop_array  = replicate ({UDF_TIMES_STRUCT}, numRecs)


   data_source = ''

   fh = l_UdfGetDataHandle (inst, expr, vinst, btime, etime, data_source, /GROUP)
   if (fh eq -1 ) then begin
       udf_close, fh
       RETURN, FAILURE
   endif

   indx = 0L
   ;last_i = 0L
 
   beg_idx = 0L
   end_idx = SPIN_SECTORS - 1L 

   while not udf_eof (fh) do begin

       d = udf_read (fh)

       bt = l_GetTime (d.btime, doy, hr, min, sec)

       start_array[indx].year = d.btime.year
       start_array[indx].doy  = d.btime.doy
       start_array[indx].hour = d.btime.hour
       start_array[indx].min  = d.btime.min
       start_array[indx].sec  = d.btime.sec
       start_array[indx].msec = d.btime.msec

       stop_array[indx].year = d.etime.year
       stop_array[indx].doy  = d.etime.doy
       stop_array[indx].hour = d.etime.hour
       stop_array[indx].min  = d.etime.min
       stop_array[indx].sec  = d.etime.sec
       stop_array[indx].msec = d.etime.msec

       sctime[beg_idx:end_idx] = bt
       start[beg_idx:end_idx] = d.singles_data[*,0]
       stop[beg_idx:end_idx] = d.singles_data[*,1]
       beg_idx = end_idx + 1L
       end_idx = beg_idx + SPIN_SECTORS - 1L

       indx = indx + 1L

   endwhile 


   udf_close, fh


   ptr = ptr_new (start[0:end_idx - SPIN_SECTORS], /NO_COPY)
   l_sngls.startPtr = ptr

   ptr = ptr_new (stop[0:end_idx - SPIN_SECTORS], /NO_COPY)
   l_sngls.stopPtr = ptr

   ptr = ptr_new (sctime[0:end_idx - SPIN_SECTORS], /NO_COPY)
   l_sngls.timePtr = ptr

   ptr = ptr_new (start_array[0:indx-1L], /NO_COPY)
   l_sngls.startUdfTimePtr = ptr

   ptr = ptr_new (stop_array[0:indx-1L], /NO_COPY)
   l_sngls.stopUdfTimePtr = ptr


   RETURN, SUCCESS
END


;-------------------------------------------------------------------------------
;  Function :  l_SnglsGetData
;
;  Description:
;      Determins the number of records and calls routine to read the data.
;
;  Return Value:  type = int
;  Value                         Description
;  -------------------------     ------------------------------
;  SUCCESS                       successful completion
;  FAILURE                       failure to complete 
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_setup               struct        I        generic setup parameters
;  l_setup_file          string        I        setup file name
;  l_pltpos              struct        I        generic plot parameters
;  l_sngls_setup         struct        O        singles plot parameters
;  l_sngls               struct*       O        singles data parameters
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;     PLOTDEFS         MAX_PLOTS        int         G         max # plots per page
;                      MAX_WEB_SPINS    int         G         max # spins per plot
;                                                             when using web interface
;                      NO_CNTS          double      G         no counts indicator
;                      NO_DATA          double      G         no data indicator
;
;      ERROR_TYPES     SUCCESS         int         G          successful completion
;                      FAILURE         int         G          unsuccessful completion
;                      WARNING         int         G          warning message
;                      INFO            int         G          informational message
;
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;  ELH               08/01           v1.0.7           read in extra spins for
;                                                     the adjustment of the sun,
;                                                     earth, moon
;-------------------------------------------------------------------------------


FUNCTION l_SnglsGetData, l_setup, l_setup_file, l_pltpos, l_sngls_setup, l_sngls
COMMON PLOTDEFS, MAX_PLOTS, MAX_WEB_SPINS, NO_CNTS, NO_DATA
COMMON ERROR_TYPES, SUCCESS, FAILURE, WARNING, INFO

   inst    = 'LENA' 
   expr    = 'LENASCI'
   vinst   = 'IMLSNGLS' 

   data_source = ' '
   numRecs     = 0L

   ptr= ptr_new()
   l_sngls = {                              $
      startPtr               :ptr,          $
      filledStartPtr         :ptr,          $
      extraStartPtr          :ptr,          $
      startDmpPtr            :ptr,          $
      stopPtr                :ptr,          $
      filledStopPtr          :ptr,          $
      extraStopPtr           :ptr,          $
      stopDmpPtr             :ptr,          $
      timePtr                :ptr,          $
      filledTimePtr          :ptr,          $
      startUdfTimePtr        :ptr,          $
      filledStartUdfTimePtr  :ptr,          $
      stopUdfTimePtr         :ptr           $
   }


   start_tme = l_setup.l_start_dtime
   stop_tme = l_setup.l_stop_dtime


   ;----------------------------------------------------
   ; if the start time is not 00:00:00, then add extra 
   ; minutes to the begin time
   ;----------------------------------------------------
   l_sngls_setup.addedStartFlag = 0L
   if (ChckStartTime(start_tme) eq FAILURE) then begin
      AddExtraTime2Start, start_tme, 2
      l_sngls_setup.addedStartFlag = 1L
   endif

   ;----------------------------------------------------
   ; if the stop time is not 23:59:59, then add extra 
   ; minutes to the end time
   ;----------------------------------------------------
   l_sngls_setup.addedStopFlag = 0L 
   if (ChckStopTime(stop_tme) eq FAILURE) then begin 
      AddExtraTime2Stop, stop_tme, 2
      l_sngls_setup.addedStopFlag = 1L
   endif



   start_sngls_time      = ConvArrTimeToSecs(start_tme)
   stop_sngls_time       = ConvArrTimeToSecs(stop_tme)

   ;est_spins             = round((stop_sngls_time - start_sngls_time) / 120.) + 20L
   ;if (est_spins le 0) then return, FAILURE

   est_nspins = l_UdfGetNumRec (start_tme, $
                                stop_tme,  $
                                inst, expr, vinst, data_source )
   if (est_nspins le 0) then return, FAILURE


   status = l_SnglsReadData (inst, expr, vinst,     $
                    start_tme, $
                    stop_tme,  $
                    est_nspins, $
                    l_sngls, $
                    data_source)

   l_pltpos.data_source = data_source



   ;------------- calculate start/stop totals ------------ 

   plotted = 0L
   pages   = 1L


   RETURN, status 

END

;-------------------------------------------------------------------------------
;  Procedure :  l_SnglsReadSetup
;
;  Description:
;     Reads options from setup file specific to singles data.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_plot_setup          struct        O        singles plot parameters
;  l_setup_file          string        I        setup file name
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;  ELH                05/04          v1.0.11          Convert input string to
;                                                     lowercase
;-------------------------------------------------------------------------------

PRO l_SnglsReadSetup, l_plot_setup, l_setup_file


  l_plot_setup = {               $
       addedStartFlag  : 0L,   $
       addedStopFlag   : 0L,   $
       plot_type       :  'S', $
       start           :  'N', $
       stop            :  'N', $
       start_color_min :  0.0, $
       start_color_max :  0.0, $
       stop_color_min  :  0.0, $
       stop_color_max  :  0.0, $
       nplots          :  0L   $
   }

   openr, inlun, /get_lun, l_setup_file
   ;----------------------------------------
   ; read singles data setup parameters
   ;----------------------------------------

   l_plot_setup.nplots = 0L

   line = ' '

   ;----------------------------------------
   ;While there is text left, output it.
   ;----------------------------------------
   while not EOF(inlun) do begin		
     readf, inlun, line
     IF (STRMID(line, 0, 1) ne ';') and $
          (STRLEN(line) gt 0) then begin

       parts = STR_SEP(line, ' ')
       if (n_elements (parts) gt 2) then begin
       case STRLOWCASE(parts[0]) of
	     'l_singles_plot_type' : begin
                                   l_plot_setup.plot_type=STRUPCASE(parts[2])
                                 end
	     'l_singles_start'     : begin
		                           l_plot_setup.start =STRUPCASE(parts[2])
		                            if l_plot_setup.start eq 'Y' then begin 
		                              l_plot_setup.nplots = $
                                           l_plot_setup.nplots + 1
                                    endif
		                         end
	     'l_singles_stop'      : begin
		                           l_plot_setup.stop = STRUPCASE(parts[2])
		                            if l_plot_setup.stop eq 'Y' then begin 
		                              l_plot_setup.nplots = $
                                           l_plot_setup.nplots + 1
                                    endif
		                         end
	     'l_singles_start_color_min' : begin
                        l_plot_setup.start_color_min=parts[2]
                                      end
	     'l_singles_start_color_max' : begin
                        l_plot_setup.start_color_max=parts[2]
                                      end
	     'l_singles_stop_color_min' : begin
                        l_plot_setup.stop_color_min=parts[2]
                                      end
	     'l_singles_stop_color_max' : begin
                        l_plot_setup.stop_color_max=parts[2]
                                      end

         else: begin
                 ;print, 'Unknown option: ', parts[0]
	           end
       endcase				;End of CASE statement.
       endif
    endif
  endwhile

  close, inlun
  free_lun, inlun

RETURN
END


;-------------------------------------------------------------------------------
; Procedure :  l_SnglsShowSetup
;   Used for debug purposes.
;-------------------------------------------------------------------------------

PRO l_SnglsShowSetup, l_plot_setup

print, '____LENA Singles Data Setup Parameters'
print, 'L_SINGLES_PLOT_TYPE             : ', l_plot_setup.plot_type
print, 'L_SINGLES_START                 : ', l_plot_setup.start
print, 'L_SINGLES_STOP                  : ', l_plot_setup.stop

RETURN
END



;-------------------------------------------------------------------------------
;  Procedure: l_SnglsHndlMissData
;
;  Description: Fill data gaps with fill data.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_sngls_setup         struct        I        singles processing & plotting parameters
;  l_sngls               struct *      I        pointers to singles data 
;  l_setup               struct        I        general setup parameters
;  l_oa                  struct *      I        pointers to OA data
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------

PRO l_SnglsHndlMissData, l_sngls_setup, l_sngls, l_setup, l_oa


   data_format = 45L

   adjFlag = 'YES'
   if (l_setup.l_angle_zero eq 'N') then adjFlag = 'NO'

   if (l_setup.l_angle_zero eq 'S') then begin
      l_CalcSunPulse, l_oa, lena_sector, 'S'
      center_sector = fix(lena_sector)
      cs = lena_sector/8.
      if NOT (ptr_valid(l_oa.lena_sun_sectPtr)) then begin
         ptr = ptr_new (cs, /NO_COPY)
         l_oa.lena_sun_sectPtr = ptr
      endif
   endif else begin
      if (l_setup.l_angle_zero eq 'M') then begin
         l_CalcSunPulse, l_oa, lena_sector, 'M'
         center_sector = fix(lena_sector)
         cs = center_sector/8.
         if NOT (ptr_valid (l_oa.lena_moon_sectPtr)) then begin
            ptr = ptr_new(cs, /NO_COPY)
            l_oa.lena_moon_sectPtr = ptr
         endif
      endif else begin
         if (l_setup.l_angle_zero eq 'E') then begin
            center_sector = l_setup.l_earth_center
         endif else begin
            center_sector = 0
         endelse
      endelse
   endelse


   if (l_setup.l_sun_direction eq 'Y') then begin
      l_CalcSunPulse, l_oa, lena_sector, 'S'
      sun_direction = lena_sector/8.
      if NOT (ptr_valid(l_oa.lena_sun_sectPtr)) then begin
         ptr = ptr_new (sun_direction, /NO_COPY)
         l_oa.lena_sun_sectPtr = ptr
      endif
   endif


   nspins = n_elements ((*l_sngls.startUdfTimePtr))

   if (l_sngls_setup.start eq 'Y') then begin

      l_HndlMissData, (*l_sngls.timePtr),    $
                      filled_time,         $  
                      nspins,                $
                      (*l_sngls.startPtr),   $
                      filled_data,             $
                      data_format ,          $
                      center_sector, $
                      l_setup, $
                      adjFlag, $
                      l_oa, $
                      (*l_sngls.startUdfTimePtr), $
                      filled_start_udf_time, $
                      (*l_sngls.extraStartPtr), $
                      l_setup.l_spin_ang_zero 

      if (ptr_valid(l_sngls.filledTimePtr)) then ptr_free, l_sngls.filledTimePtr 
      ptr = ptr_new (filled_time, /NO_COPY)
      l_sngls.filledTimePtr = ptr
 
      if (ptr_valid(l_sngls.filledStartPtr)) then ptr_free, l_sngls.filledStartPtr 
      ptr = ptr_new (filled_data, /NO_COPY)
      l_sngls.filledStartPtr = ptr

      if (ptr_valid(l_sngls.filledStartUdfTimePtr)) then ptr_free, l_sngls.filledStartUdfTimePtr 
      ptr = ptr_new (filled_start_udf_time, /NO_COPY)
      l_sngls.filledStartUdfTimePtr = ptr 


  endif 

  if (l_sngls_setup.stop eq 'Y') then begin

      l_HndlMissData, (*l_sngls.timePtr),    $
                      filled_time,         $  
                      nspins,                $
                      (*l_sngls.stopPtr),   $
                      filled_data,             $
                      data_format ,          $
                      center_sector, $
                      l_setup, $
                      adjFlag, $
                      l_oa, $
                      (*l_sngls.startUdfTimePtr), $
                      filled_start_udf_time, $
                      (*l_sngls.extraStopPtr) , $
                      l_setup.l_spin_ang_zero 



      if (ptr_valid(l_sngls.filledTimePtr)) then ptr_free, l_sngls.filledTimePtr 
      ptr = ptr_new (filled_time, /NO_COPY)
      l_sngls.filledTimePtr = ptr

      if (ptr_valid(l_sngls.filledStopPtr)) then ptr_free, l_sngls.filledStopPtr 
      ptr = ptr_new (filled_data, /NO_COPY)
      l_sngls.filledStopPtr = ptr
 
      if (ptr_valid(l_sngls.filledStartUdfTimePtr)) then ptr_free, l_sngls.filledStartUdfTimePtr 
      ptr = ptr_new (filled_start_udf_time, /NO_COPY)
      l_sngls.filledStartUdfTimePtr = ptr 


  endif




RETURN
END

;-------------------------------------------------------------------------------
;  Procedure :  l_SnglsPlot
;
;  Description: Plot the data.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_sngls_setup         struct        I        singles plot parameters
;  l_sngls               struct *      I        singles data parameters
;  l_pltpos              struct        I        generic plot parameters
;  nplots                long          I        total # of plots to be plotted
;  windx                 long[]        I        window index
;  pages                 long         I/O       page number
;  plotted               long         I/O       # of plots already plotted
;  l_setup               struct        I        generic setup parameters
;  l_oa                  struct *      I        OA data parameters
;  data_format           long          I        define a single scalar array or
;                                               an array with 45 elements
;  l_rgb_table           struct        I        generic color table values
;  colorbar              float[]       I        color bar values
;  l_hskp                struct *      I        HSKP data parameters 
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;     PLOTDEFS         MAX_PLOTS        int         G         max # plots per page
;                      MAX_WEB_SPINS    int         G         max # spins per plot
;                                                             when using web interface
;                      NO_CNTS          double      G         no counts indicator
;                      NO_DATA          double      G         no data indicator
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0            Original implementation
;  ELH              07/05/01         v1.0            Modified for dumps to match
;                                                    spectrograms except no conversions 
;                                                    Handle y-axis title for earth, sun,
;                                                    moon, none
;  ELH              09/18/01         v1.0.7          Added capability to draw sun
;                                                    direction on plots if selected
;  ELH              09/26/01         v1.0.7          Added l_DetermineSunDir 
;-------------------------------------------------------------------------------

PRO l_SnglsPlot, l_sngls_setup, l_sngls, l_pltpos, $
nplots, windx, pages, plotted, l_setup , l_oa, $
l_rgb_table, colorbar, l_hskp
COMMON PLOTDEFS, MAX_PLOTS, MAX_WEB_SPINS, NO_CNTS, NO_DATA


   if (l_sngls_setup.plot_type eq 'S') then begin
      st_ytitle = '!5Starts' 
      sp_ytitle = '!5Stops' 
   endif else begin
      st_ytitle = '!5Starts (Total)' 
      sp_ytitle = '!5Stops  (Total)' 
   endelse

   if (l_setup.l_angle_zero eq 'S') then begin
      ytitle_frame = '!5Sun Cntrd Azi'
   endif else begin
      if (l_setup.l_angle_zero eq 'M') then begin
         ytitle_frame = '!5Moon Cntrd Azi'
      endif else begin
         if (l_setup.l_angle_zero eq 'E') then begin
            ytitle_frame = '!5Earth Cntrd Azi'
         endif else begin
            ytitle_frame  = '!5Spin Angle'
         endelse
      endelse
   endelse


   orig_filled_time = (*l_sngls.filledStartUdfTimePtr)

   if (l_sngls_setup.start eq 'Y') then begin
      orig_filled_time = (*l_sngls.filledStartUdfTimePtr)
      ;SyncData2Time, (*l_sngls.filledStartPtr), orig_filled_time, l_setup
   endif 

   if (l_sngls_setup.stop eq 'Y') then begin
      orig_filled_time = (*l_sngls.filledStartUdfTimePtr)
      ;SyncData2Time, (*l_sngls.filledStopPtr), orig_filled_time, l_setup
   endif

   (*l_sngls.filledStartUdfTimePtr) = orig_filled_time

   l_setup.l_d_start_dtime[0] = (*l_sngls.filledStartUdfTimePtr)[0].year
   l_setup.l_d_start_dtime[1] = (*l_sngls.filledStartUdfTimePtr)[0].doy
   l_setup.l_d_start_dtime[2] = (*l_sngls.filledStartUdfTimePtr)[0].hour
   l_setup.l_d_start_dtime[3] = (*l_sngls.filledStartUdfTimePtr)[0].min
   l_setup.l_d_start_dtime[4] = (*l_sngls.filledStartUdfTimePtr)[0].sec
   l_setup.l_d_start_dtime[5] = (*l_sngls.filledStartUdfTimePtr)[0].msec

   max_ele = n_elements((*l_sngls.filledStartUdfTimePtr)[*])-1L

   l_setup.l_d_stop_dtime[0] = (*l_sngls.filledStartUdfTimePtr)[max_ele].year
   l_setup.l_d_stop_dtime[1] = (*l_sngls.filledStartUdfTimePtr)[max_ele].doy
   l_setup.l_d_stop_dtime[2] = (*l_sngls.filledStartUdfTimePtr)[max_ele].hour
   l_setup.l_d_stop_dtime[3] = (*l_sngls.filledStartUdfTimePtr)[max_ele].min
   l_setup.l_d_stop_dtime[4] = (*l_sngls.filledStartUdfTimePtr)[max_ele].sec
   l_setup.l_d_stop_dtime[5] = (*l_sngls.filledStartUdfTimePtr)[max_ele].msec


   l_DetMjMnTicks, l_setup.l_start_dtime, $
                   l_setup.l_stop_dtime, $
                   l_setup.l_d_start_dtime, $
                   l_setup.l_d_stop_dtime, $
                   l_pltpos


   if (l_sngls_setup.start eq 'Y') then begin

      start_data = (*l_sngls.filledStartPtr)

      ztitle = 'linear cnts'
      cnt_ztitle = ' '
      log_ztitle = ' '

      if (l_setup.l_dumps eq 'Y') then begin
         if (ptr_valid (l_sngls.startDmpPtr)) then ptr_free, l_sngls.startDmpPtr
         ptr = ptr_new (start_data, /NO_COPY)
         l_sngls.startDmpPtr = ptr
         start_data = (*l_sngls.filledStartPtr)
      endif

      if (l_setup.l_count_rate eq 'Y') then begin
          l_Conv2CntRate, start_data, start_cntRate, 'SECTOR'
          start_data = start_cntRate
          ztitle = ' '
          cnt_ztitle = '!5Cnt Rate [Hz]'
      endif

      if (l_setup.l_log eq 'Y') then begin
         l_Conv2Log, start_data, start_log, start_min, start_max
         start_data = start_log
         log_ztitle  = '!5log '
      endif


      zaxis_title = ztitle + log_ztitle + cnt_ztitle


      if (l_setup.l_auto_scale eq 'Y') then begin
          l_setup.l_color_min = start_min
          l_setup.l_color_max = start_max
      endif else begin
          if (l_setup.l_sep_plot_scale eq 'Y') then begin
              l_setup.l_color_min = l_sngls_setup.start_color_min
              l_setup.l_color_max = l_sngls_setup.start_color_max
         endif
      endelse
     
      if (l_setup.l_line_plots eq 'Y') then begin
 
          diff = start_max - start_min
          if (diff gt 1) then begin
             new_max = (start_max/4.) + start_max
             start_max = new_max
          endif else begin
             start_max = start_max + diff
          endelse

          l_PlotStuff, nplots,                   $
                       (*l_sngls.filledTimePtr), $
                       start_data,               $
                       start_min,                $
                       start_max,                $
                       zaxis_title,              $
                       l_pltpos,                 $
                       plotted,                  $
                       l_oa,                     $
                       l_setup.l_start_dtime


      endif else begin  ; a spectrogram plot

          l_ScaleColor, start_data, start_scaled, $
                        l_setup.l_color_min, $
                        l_setup.l_color_max , $
                        l_setup.l_color_tbl


          ;------------------------------------------------------------
          ; correct the image positioning, rotate 240 degrees
          ;------------------------------------------------------------
          start_rot = rotate(start_scaled, 3)

          yrng=[l_pltpos.min_spin_ang,l_pltpos.max_spin_ang]

          img_pos = [l_pltpos.xpltmn,l_pltpos.ypltmn,$
                     l_pltpos.xpltmx,l_pltpos.ypltmx]

          position = float(img_pos)

          xstart = position[0]
          ystart = position[1]
          ximage = l_pltpos.axis_size/ l_pltpos.xin
          yimage = l_pltpos.ypltmx - l_pltpos.ypltmn
          xsize  = round(l_pltpos.window_width * ximage)
          ysize  = round(l_pltpos.window_height * yimage)

          start_spect = congrid (start_rot, $
                                 xsize, $
                                 ysize) 

          if (l_setup.l_smooth_data eq 'Y') then begin
             TV, smooth (start_spect, 3, /edge_truncate), xstart, ystart, $
                 XSIZE=ximage, YSIZE=yimage, /ORDER, $
                 /NORMAL 
          endif else begin
             TV, start_spect, xstart, ystart, $
                 XSIZE=ximage, YSIZE=yimage, /ORDER, $
                 /NORMAL 

          endelse

          ;---------------------------------------------------
          ; draw the earth direction
          ;---------------------------------------------------
          if (l_setup.l_earth_markers eq 'Y') then begin
              l_CalcEarthDirection, l_oa, l_setup, img_pos
          endif

          ;---------------------------------------------------
          ; draw the sun direction
          ;---------------------------------------------------
          if (l_setup.l_sun_direction eq 'Y')  then begin
              l_CalcSunDirection, l_oa, l_setup, img_pos
          endif

          ;---------------------------------------------------
          ; draw the ram direction
          ;---------------------------------------------------
          if (l_setup.l_ram_direction eq 'Y') then begin
             l_CalcSunPulse, l_oa, ram_sector, 'R'
             ram_sector = ram_sector/8.
             if NOT (ptr_valid(l_oa.lena_ram_sectPtr)) then begin
                ptr = ptr_new (ram_sector, /NO_COPY)
                l_oa.lena_ram_sectPtr = ptr
             endif
             l_CalcRamDirection, l_oa, l_setup, img_pos 
             ram_sector = 0
          endif

          ;---------------------------------------------------
          ; draw the B direction
          ;---------------------------------------------------
          if (l_setup.l_B_direction eq 'Y') then begin
             l_CallT96Model, l_oa, l_setup
             l_CalcSunPulse, l_oa, B_sector, 'B'
             B_sector = B_sector/8.
             if NOT (ptr_valid(l_oa.lena_mag_fieldPtr)) then begin
                ptr = ptr_new (B_sector, /NO_COPY)
                l_oa.lena_mag_fieldPtr = ptr
             endif
             l_CalcBDirection, l_oa, l_setup, img_pos
             B_sector = 0
          endif

          ;---------------------------------------------------
          ; draw the z-axis, color bar
          ;---------------------------------------------------

          bar_pos = [l_pltpos.xpltmx+0.015, $
                    l_pltpos.ypltmn, $
                    l_pltpos.xpltmx+0.025, $
                    l_pltpos.ypltmx]

          bar_info = {AXIS_DATA, char_size: l_pltpos.plot_font, $
                      ytitle:zaxis_title, $
                      yformat:'(F4.1)', $
                      yrange:[l_setup.l_color_min, $
                              l_setup.l_color_max]}

          l_DrawColorbar, colorbar, l_setup, bar_pos, l_pltpos, $
                          RAXIS=bar_info

          l_PlotFrame, nplots, (*l_sngls.filledTimePtr), yrng, img_pos, l_pltpos, $
                       plotted, l_oa, ytitle_frame, l_setup.l_start_dtime, $
                       l_hskp


      endelse

      yctr = (l_pltpos.ypltmn + l_pltpos.ypltmx)/2.0
      xyouts, 0.02, yctr, st_ytitle, size=l_pltpos.md_font, $
              /NORMAL, alignment = 0.5, orientation=90.0

      l_pltpos.ypltmx = l_pltpos.ypltmn - l_pltpos.yspace
      l_pltpos.ypltmn = l_pltpos.ypltmx - l_pltpos.ydel
      plotted = plotted + 1


      l_ChckCompPg, plotted, nplots, pages, $
                    l_setup, windx, l_pltpos, l_rgb_table

   endif


   if (l_sngls_setup.stop eq 'Y') then begin

      stop_data = (*l_sngls.filledStopPtr)
 
      ztitle = 'linear cnts'
      cnt_ztitle = ' '
      log_ztitle = ' '


      if (l_setup.l_dumps eq 'Y') then begin
         if (ptr_valid (l_sngls.stopDmpPtr)) then ptr_free, l_sngls.stopDmpPtr
         ptr = ptr_new (stop_data, /NO_COPY)
         l_sngls.stopDmpPtr = ptr
         stop_data = (*l_sngls.filledStopPtr)
      endif

      if (l_setup.l_count_rate eq 'Y') then begin
         l_Conv2CntRate, stop_data, stop_cntRate, 'SECTOR'
         stop_data = stop_cntRate
         ztitle = ' '
         cnt_ztitle = '!5Cnt Rate [Hz]'
      endif

      if (l_setup.l_log eq 'Y') then begin
         l_Conv2Log, stop_data, stop_log, stop_min, stop_max
         stop_data = stop_log
         log_ztitle  = '!5log '
      endif


      zaxis_title = ztitle + log_ztitle + cnt_ztitle

      if (l_setup.l_auto_scale eq 'Y') then begin
          l_setup.l_color_min = stop_min
          l_setup.l_color_max = stop_max
      endif else begin
          if (l_setup.l_sep_plot_scale eq 'Y') then begin
             l_setup.l_color_min = l_sngls_setup.stop_color_min
             l_setup.l_color_max = l_sngls_setup.stop_color_max
         endif
      endelse

      if (l_setup.l_line_plots eq 'Y') then begin

          diff = stop_max - stop_min
          if (diff gt 1) then begin
             new_max = (stop_max/4.) + stop_max
             stop_max = new_max
          endif else begin
             stop_max = stop_max + diff
          endelse

         l_PlotStuff, nplots, $
                      (*l_sngls.filledTimePtr), $
                      stop_data, $
                      stop_min, $
                      stop_max,  $
                      zaxis_title,   $
                      l_pltpos, $
                      plotted,  $
                      l_oa, $
                      l_setup.l_start_dtime


      endif else begin  ; a spectrogram plot

         l_ScaleColor, stop_data, stop_scaled, $
                       l_setup.l_color_min, $
                       l_setup.l_color_max , $
                       l_setup.l_color_tbl

         ;------------------------------------------------------------
         ; correct the image positioning, rotate 240 degrees
         ;------------------------------------------------------------
         stop_rot = rotate(stop_scaled, 3)

         yrng=[l_pltpos.min_spin_ang,l_pltpos.max_spin_ang]

         img_pos = [l_pltpos.xpltmn,l_pltpos.ypltmn,$
                    l_pltpos.xpltmx,l_pltpos.ypltmx]

         position = float(img_pos)

         xstart = position[0]
         ystart = position[1]
         ximage = l_pltpos.axis_size/ l_pltpos.xin
         yimage = l_pltpos.ypltmx - l_pltpos.ypltmn
         xsize  = round(l_pltpos.window_width * ximage)
         ysize  = round(l_pltpos.window_height * yimage)
 
         stop_spect = congrid (stop_rot, $
                               xsize, $
                               ysize)


         TV, stop_spect, xstart, ystart, $
             XSIZE=ximage, YSIZE=yimage, /ORDER, $
             /NORMAL


         ;---------------------------------------------------
         ; draw the earth direction
         ;---------------------------------------------------
         if (l_setup.l_earth_markers eq 'Y') then begin
             l_CalcEarthDirection, l_oa, l_setup, img_pos
         endif

         ;---------------------------------------------------
         ; draw the sun direction
         ;---------------------------------------------------
         if (l_setup.l_sun_direction eq 'Y') then begin
             l_CalcSunDirection, l_oa, l_setup, img_pos
         endif

         ;---------------------------------------------------
         ; draw the ram direction
         ;---------------------------------------------------
         if (l_setup.l_ram_direction eq 'Y') then begin
             l_CalcSunPulse, l_oa, ram_sector, 'R'
             ram_sector = ram_sector/8.
             if NOT (ptr_valid(l_oa.lena_ram_sectPtr)) then begin
                ptr = ptr_new (ram_sector, /NO_COPY)
                l_oa.lena_ram_sectPtr = ptr
             endif
             l_CalcRamDirection, l_oa, l_setup, img_pos 
         endif

          ;---------------------------------------------------
          ; draw the B direction
          ;---------------------------------------------------
          if (l_setup.l_B_direction eq 'Y') then begin
             l_CallT96Model, l_oa, l_setup
             l_CalcSunPulse, l_oa, B_sector, 'B'
             B_sector = B_sector/8.
             if NOT (ptr_valid(l_oa.lena_mag_fieldPtr)) then begin
                ptr = ptr_new (B_sector, /NO_COPY)
                l_oa.lena_mag_fieldPtr = ptr
             endif
             l_CalcBDirection, l_oa, l_setup, img_pos
          endif


         ;---------------------------------------------------
         ; draw z-axis, color bar
         ;---------------------------------------------------
         bar_pos = [l_pltpos.xpltmx+0.015, $
                    l_pltpos.ypltmn, $
                    l_pltpos.xpltmx+0.025, $
                    l_pltpos.ypltmx]

         bar_info = {AXIS_DATA, char_size:l_pltpos.plot_font, $
                     ytitle:zaxis_title, $
                     yformat:'(F4.1)', $
                     yrange:[l_setup.l_color_min, $
                             l_setup.l_color_max] }

         l_DrawColorbar, colorbar, l_setup, bar_pos, l_pltpos, $
                         RAXIS=bar_info

         l_PlotFrame, nplots, (*l_sngls.filledTimePtr), yrng, img_pos, l_pltpos, $
                      plotted, l_oa, ytitle_frame, l_setup.l_start_dtime, $
                      l_hskp

      endelse

      yctr = (l_pltpos.ypltmn + l_pltpos.ypltmx)/2.0
      xyouts, 0.02, yctr, sp_ytitle, size=l_pltpos.md_font, $
              /NORMAL, alignment = 0.5, orientation=90.0

      l_pltpos.ypltmx = l_pltpos.ypltmn - l_pltpos.yspace
      l_pltpos.ypltmn = l_pltpos.ypltmx - l_pltpos.ydel
      plotted = plotted + 1


      l_ChckCompPg, plotted, nplots, pages, $
                    l_setup, windx, l_pltpos, l_rgb_table
   endif


RETURN
END


;-------------------------------------------------------------------------------
;  Procedure :  l_SnglsDumpData
;
;  Description: Dump data to a text file.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_sngls               struct        I        singles data parameters
;  l_sngls_setup         struct        I        singles plot parameters
;  l_setup               struct        I        generic setup parameters
;  nele                  long          I        # records this pass
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;-------------------------------------------------------------------------------

PRO l_SnglsDumpData, l_sngls, l_sngls_setup, l_setup, nele 


   ;------------------------------------------------------------
   ; open g_dump file
   ;------------------------------------------------------------

   filename = l_setup.l_prod_dst + '/' + l_setup.l_data_type + '.dmp'
   openw, g_dump, /get_lun, filename
   printf, g_dump, '================================================================================'
   printf, g_dump, 'LENAPLOT REPORT : ', + filename + ' : ' + systime()
   printf, g_dump, '================================================================================'

   ;------------------------------------------------------------
   ; build string to execute based on options selected from menu
   ;------------------------------------------------------------

   if (l_sngls_setup.plot_type eq 'S') then begin
      str='Spin' 
   endif else begin 
      str='Totals'
   endelse

   printf, g_dump, 'Plot type: ', str
   printf, g_dump, 'Data     : Singles'
   printf, g_dump, ' '

   ;------------------------------------------------------------
   ; add time column to string
   ;------------------------------------------------------------
   ncnt = 1 
   header1 = 'printf, g_dump,  ''Time'' '
   header2 = 'printf, g_dump,  ''----'' '
   dumpstr = 'printf, g_dump, ConvToUtc ((*l_sngls.timePtr)[i], 2)'

   ;------------------------------------------------------------
   ; add start data to string
   ;------------------------------------------------------------
   if (l_sngls_setup.start eq 'Y') then begin
      header1 = header1 + ', ''Start'' '
      header2 = header2 + ', ''-----'' '
      dumpstr = dumpstr + ', (*l_sngls.startPtr)[i]'
      ncnt = ncnt + 1
   endif

   ;------------------------------------------------------------
   ; add stop data to string
   ;------------------------------------------------------------
   if (l_sngls_setup.stop eq 'Y') then begin
      header1 = header1 + ', ''Stop'' '
      header2 = header2 + ', ''----'' '
      dumpstr = dumpstr + ', (*l_sngls.stopPtr)[i]'
      ncnt = ncnt + 1
   endif
 

   ;------------------------------------------------------------
   ; print header data to dump file
   ;------------------------------------------------------------
   hdrform  = ', format=''(' + strcompress(string(ncnt),/remove_all) $
               + '(A8,TR12))'''
   header1 = header1 + hdrform
   header2 = header2 + hdrform
   result = execute (header1)
   if (result ne 1) then print, 'Error writing Singles Data Ascii Dump Header'

   result = execute (header2)
   if (result ne 1) then print, 'Error writing Singles Data Ascii Dump Header'

   printf, g_dump, ''

   ;------------------------------------------------------------
   ; print singles data to dump file
   ;------------------------------------------------------------
   dumpform = ', format = ''((A14,TR10),' $
               + strcompress(string(ncnt),/remove_all) $
               + '(I6,TR12))'''
   dumpstr = dumpstr + dumpform

   for i=0, nele-1 do begin
      result = execute (dumpstr) 
   endfor

   if (result ne 1) then print, 'Error writing Singles Data (Ascii Dump)'

   ;------------------------------------------------------------
   ; close g_dump file
   ;------------------------------------------------------------
   close, g_dump 
   free_lun, g_dump

RETURN
END


;-------------------------------------------------------------------------------
;  Procedure :  l_SnglsTerm
;
;  Description: Free allocated memory.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_sngls               struct        I        singles data parameters
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;-------------------------------------------------------------------------------

PRO l_SnglsTerm, l_sngls


   if (ptr_valid (l_sngls.startPtr)) then begin
       ptr_free, l_sngls.startPtr
   endif

   if (ptr_valid (l_sngls.filledStartPtr)) then begin
       ptr_free, l_sngls.filledStartPtr
   endif

   if (ptr_valid (l_sngls.extraStartPtr)) then begin
       ptr_free, l_sngls.extraStartPtr
   endif

   if (ptr_valid (l_sngls.startDmpPtr)) then begin
       ptr_free, l_sngls.startDmpPtr
   endif

   if (ptr_valid (l_sngls.stopPtr)) then begin
       ptr_free, l_sngls.stopPtr
   endif

   if (ptr_valid (l_sngls.filledStopPtr)) then begin
       ptr_free, l_sngls.filledStopPtr
   endif

   if (ptr_valid (l_sngls.extraStopPtr)) then begin
       ptr_free, l_sngls.extraStopPtr
   endif

   if (ptr_valid (l_sngls.stopDmpPtr)) then begin
       ptr_free, l_sngls.stopDmpPtr
   endif

   if (ptr_valid (l_sngls.timePtr)) then begin
       ptr_free, l_sngls.timePtr
   endif

   if (ptr_valid (l_sngls.filledTimePtr)) then begin
       ptr_free, l_sngls.filledTimePtr
   endif

   if (ptr_valid (l_sngls.startUdfTimePtr)) then begin
       ptr_free, l_sngls.startUdfTimePtr
   endif

   if (ptr_valid (l_sngls.filledStartUdfTimePtr)) then begin
       ptr_free, l_sngls.filledStartUdfTimePtr
   endif

   if (ptr_valid (l_sngls.stopUdfTimePtr)) then begin
       ptr_free, l_sngls.stopUdfTimePtr
   endif

END

;-------------------------------------------------------------------------------
;  Function:  l_SnglsProc
;
;  Description: Main.
;
;  Return Value:  type = int
;  Value                         Description
;  -------------------------     ------------------------------
;  SUCCESS                       successful completion
;  FAILURE                       unsuccessful completion
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_setup               struct        I        generic setup parameters
;  l_setup_file          string        I        setup file name
;  l_pltpos              struct        I        generic plot parameters
;  l_rgb_table           struct        I        generic color table parameters
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;     PLOTDEFS         MAX_PLOTS        int         G         max # plots per page
;                      MAX_WEB_SPINS    int         G         max # spins per plot
;                                                             when using web interface
;                      NO_CNTS          double      G         no counts indicator
;                      NO_DATA          double      G         no data indicator
;
;      ERROR_TYPES     SUCCESS         int         G          successful completion
;                      FAILURE         int         G          unsuccessful completion
;                      WARNING         int         G          warning message
;                      INFO            int         G          informational message
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;-------------------------------------------------------------------------------

FUNCTION l_SnglsProc, l_setup, l_setup_file, l_pltpos, l_rgb_table
COMMON PLOTDEFS, MAX_PLOTS, MAX_WEB_SPINS, NO_CNTS, NO_DATA
COMMON ERROR_TYPES, SUCCESS, FAILURE, WARNING, INFO


   ;-------------------------------------------------
   ; the maximum number of spins to plot if plotting
   ; via the web
   ;-------------------------------------------------

   nele      = 0L
   year      = strmid(systime(0), 20, 23)
   l_pltpos.plot_hdr_title = '!5Singles'


   ;-------------------------------------------------
   ; read the singles data setup file
   ;-------------------------------------------------

   l_SnglsReadSetup, l_sngls_setup, l_setup_file
   
   ;-------------------------------------------------
   ; get the singles data based on user start and 
   ; stop time
   ;-------------------------------------------------
   status = l_SnglsGetData  (l_setup, $
                             l_setup_file, $
                             l_pltpos, $
                             l_sngls_setup, $
                             l_sngls)

   if (status eq FAILURE) then return, FAILURE

   ;-------------------------------------------------
   ; read the corresponding OA data
   ;-------------------------------------------------
   status = l_OARead (l_oa, $ 
                      l_setup.l_start_dtime, $
                      l_setup.l_stop_dtime, $
                      l_pltpos)
   if (status eq FAILURE) then return, FAILURE



   status = l_HskpGetData (l_setup, $
                  l_setup_file, $
                  l_pltpos, $
                  l_hskp_setup, $
                  l_hskp)
   if (status eq FAILURE) then return, FAILURE



   l_SnglsAdjExData, l_sngls_setup, l_setup.l_num_spinsec, l_sngls
   l_SnglsHndlMissData, l_sngls_setup, l_sngls, l_setup, l_oa


   if (status eq 0) then begin
      nloops = 1L
      if (l_setup.l_postscript eq 'Y') then nloops = 2L

      l_setup.l_line_plots = 'N'

      if (l_setup.l_line_plots eq 'N') then begin

         for i=0, nloops-1 do begin

            plotted   = 0L
            pages     = 0L
            nplots    = 0L

            if (i eq 1L) and (l_setup.l_postscript eq 'Y') then begin
               l_pltpos.ps = 'Y'
               l_GenFileName, l_setup, '0'
               l_pltpos.filename = l_setup.l_prod_dst + '/' + $
                                   l_setup.l_filename + '.ps'
            endif

            nplots    = l_sngls_setup.nplots

            l_SetUpZBuffer, l_pltpos, $
                            l_setup, $
                            l_rgb_table, $
                            colorbar, $
                            nplots


            l_pltpos.linePlots = l_setup.l_line_plots

            npages = CEIL(l_sngls_setup.nplots/MAX_PLOTS)
            if (npages eq 0L) then npages = 1L
            if (npages gt 0L) then windx = intarr (npages)

            ;-------------------------------------------------
            ; do a spectrogram plot
            ; determine the number of plots and calculate the 
            ; size of each plot to fit the page
            ;-------------------------------------------------
            plotted   = 0L

            l_AdjPlotSize, l_pltpos, l_setup, nplots

            l_setup.l_filedesc = 'sng'

            l_setup.l_filedesc = l_setup.l_filedesc + 'sp'


            l_SnglsPlot, l_sngls_setup, $
                         l_sngls, $
                         l_pltpos, $
                         l_sngls_setup.nplots, $
                         windx, $
                         pages, $
                         plotted, $
                         l_setup, $
                         l_oa , $
                         l_rgb_table, $    
                         colorbar, $
                         l_hskp

         endfor ;endfor nloops
      endif 


      l_pltpos.ps = 'N'
      ;-------------------------------------------------
      ; do the log plot
      ;-------------------------------------------------
      l_setup.l_line_plots = 'N'
      if (l_setup.l_line_plots eq 'Y') then begin

         for i=0L, nloops-1L do begin

            plotted   = 0L
            pages     = 0L
            nplots    = 0L
   
            if (i eq 1) and (l_setup.l_postscript eq 'Y') then begin
               l_pltpos.ps = 'Y'
               l_GenFileName, l_setup, '0'
               l_pltpos.filename = l_setup.l_prod_dst + '/' + $
                                   l_setup.l_filename + '.ps'
            endif

            nplots    = l_sngls_setup.nplots
            l_SetUpZBuffer, l_pltpos, $
                            l_setup, $
                            l_rgb_table, $
                            colorbar, $
                            nplots

            l_pltpos.linePlots = l_setup.l_line_plots
            erase
            npages = CEIL(l_sngls_setup.nplots/MAX_PLOTS)
            if (npages eq 0L) then npages = 1L
            if (npages gt 0L) then windx = intarr (npages)

            plotted   = 0L
            l_AdjPlotSize, l_pltpos, l_setup, nplots

            l_setup.l_filedesc = 'sng'
            l_setup.l_filedesc = l_setup.l_filedesc + 'ln'

            l_SnglsPlot, l_sngls_setup, $
                         l_sngls, $
                         l_pltpos, $
                         l_sngls_setup.nplots, $
                         windx, $
                         pages, $
                         plotted, $
                         l_setup, $
                         l_oa , $
                         l_rgb_table, $
                         colorbar, $
                         l_hskp

          endfor
      endif
   endif


   ;-------------------------------------------------
   ; free pointers
   ;-------------------------------------------------
   l_SnglsTerm, l_sngls
   l_HskpTerm, l_hskp
   l_OATerm, l_oa

   RETURN, SUCCESS

END
