
;-------------------------------------------------------------------------------
;
;  Unit Name : l_ReadSetup.pro
;
;  Purpose   : Read the generic setup parameters in the setup files.
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  John Park         06/00           v1.0            Original Implementation
;  RLW                08/01          v1.0.7          added l_image_id 
;  ELH                05/04          v1.0.11         Modified case statement to
;                                                    convert all input strings
;                                                    to lower case
;
;
;  File Revision Number:   %I%
;  File Last Modified  :   %E%  %T%
;
;-------------------------------------------------------------------------------

;-------------------------------------------------------------------------------
;  Procedure:   l_ReadSetup
;  Description:
;        Reads the generic setup parameters in the setup files (i.e. date, time,
;        color table, etc...).
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_setup               struct        O        generic setup parameters
;  l_setup_file          string        I        setup file name
;  error_msg             string        O        error message
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;      ERROR_TYPES     SUCCESS         int         G          successful completion
;                      FAILURE         int         G          unsuccessful completion
;                      WARNING         int         G          warning message
;                      INFO            int         G          informational message
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;  RLW                08/01          v1.0.7           added l_image_id 
;  ELH               09/18/01        v1.0.7           added l_sun_direction
;  ELH                05/04          v1.0.11         Modified case statement to
;                                                    convert all input strings
;                                                    to lower case
;-------------------------------------------------------------------------------

FUNCTION l_ReadSetup, l_setup, l_setup_file, error_msg
COMMON ERROR_TYPES, SUCCESS, FAILURE, WARNING, INFO



  error_msg = ' '

  openr, inlun, /get_lun, l_setup_file

  ;----------------------------------------
  ; read general setup parameters
  ;----------------------------------------
  line = ' '

  ;----------------------------------------
  ; while there is text left, output it.
  ;----------------------------------------
  while not EOF(inlun) do begin           
    readf, inlun, line
    IF (STRMID(line, 0, 1) ne ';') and (STRLEN(line) gt 0) then begin
       parts = STR_SEP(line, ' ')

       if (n_elements (parts) gt 2) then begin
       case STRLOWCASE(parts[0]) of
         'l_via_web'      : begin  ;--- process via web only --
                              l_setup.l_via_web= STRUPCASE(parts[2])
                            end
         'l_process_id'   : begin  ;--- process id for web processing only --
                              str_len = strlen(parts[2])
                              if (str_len gt 0) then begin
                                 l_setup.l_process_id= '_' + parts[2]
                              endif else begin
                                 l_setup.l_process_id= ''
                              endelse
                            end
         'l_gen_sum'      : begin  ;-- generate summary plots only --
                              l_setup.l_gen_sum=STRUPCASE(parts[2])
                            end
         'l_start_date'   : begin  ;-- start date (YYYY|MM|DD) or (YYYY|DOY)--
                              aa=parts[2]
                              bb=STR_SEP(aa,'|')
                              if (n_elements(bb) eq 3) then begin
                                 if (bb[0] lt 2000) then begin
                                    error_msg = 'Invalid year entered.'
                                    close, inlun
                                    free_lun, inlun
                                    return, FAILURE
                                 endif
                                 if (bb[1] lt 1) or (bb[1] gt 12) then begin
                                    error_msg = 'Invalid month entered.'
                                    close, inlun
                                    free_lun, inlun
                                    return, FAILURE
                                 endif
                                 if (bb[2] lt 1) or (bb[2] gt 31) then begin
                                    error_msg = 'Invalid day entered.'
                                    close, inlun
                                    free_lun, inlun
                                    return, FAILURE
                                 endif
                              endif else begin
                                 if (n_elements(bb) eq 2) then begin
                                    if (bb[0] lt 2000) then begin
                                       error_msg = 'Invalid year entered.'
                                       close, inlun
                                       free_lun, inlun
                                       return, FAILURE
                                    endif
                                    if (bb[1] lt 1) or (bb[1] gt 366) then begin
                                       error_msg = 'Invalid day of year entered.'
                                       close, inlun
                                       free_lun, inlun
                                       return, FAILURE
                                    endif
                                 endif else begin
                                     error_msg = 'No start date specified.'
                                     close, inlun
                                     free_lun, inlun
                                     return, FAILURE
                                 endelse
                              endelse
                              l_setup.l_start_date=uint(bb(*))
                              
                            end
         'l_start_time'   : begin  ;-- start time (HH:MM:SS) --
                              aa=parts[2]
                              bb=STR_SEP(aa,':')
                              if (bb[0] lt 0) or (bb[0] gt 23) then begin
                                 error_msg = 'Invalid start hour entered (00 - 23).'
                                 close, inlun
                                 free_lun, inlun
                                 return, FAILURE
                              endif
                              if (bb[1] lt 0) or (bb[1] gt 59) then begin
                                 error_msg = 'Invalid start minutes entered (00 - 59).'
                                 close, inlun
                                 free_lun, inlun
                                 return, FAILURE
                              endif
                              if (bb[2] lt 0) or (bb[2] gt 59) then begin
                                 error_msg = 'Invalid start seconds entered (00 - 59).'
                                 close, inlun
                                 free_lun, inlun
                                 return, FAILURE
                              endif
                              l_setup.l_start_time=uint(bb(*))
                            end
         'l_stop_date'    : begin  ;-- stop date (YYYY|MM|DD) or (YYYY|DOY) --
                              aa=parts[2]
                              bb=STR_SEP(aa,'|')
                              if (n_elements(bb) eq 3) then begin
                                 if (bb[0] lt 2000) then begin
                                    error_msg = 'Invalid year entered.'
                                    close, inlun
                                    free_lun, inlun
                                    return, FAILURE
                                 endif
                                 if (bb[1] lt 1) or (bb[1] gt 12) then begin
                                    error_msg = 'Invalid month entered.'
                                    close, inlun
                                    free_lun, inlun
                                    return, FAILURE
                                 endif
                                 if (bb[2] lt 1) or (bb[2] gt 31) then begin
                                    error_msg = 'Invalid day entered.'
                                    close, inlun
                                    free_lun, inlun
                                    return, FAILURE
                                 endif
                              endif else begin
                                 if (n_elements(bb) eq 2) then begin
                                    if (bb[0] lt 2000) then begin
                                       error_msg = 'Invalid year entered.'
                                       close, inlun
                                       free_lun, inlun
                                       return, FAILURE
                                    endif
                                    if (bb[1] lt 1) or (bb[1] gt 366) then begin
                                       error_msg = 'Invalid day of year entered.'
                                       close, inlun
                                       free_lun, inlun
                                       return, FAILURE
                                    endif
                                 endif else begin
                                     error_msg = 'No stop date specified.'
                                     close, inlun
                                     free_lun, inlun
                                     return, FAILURE
                                 endelse
                              endelse
                              l_setup.l_stop_date=uint(bb(*))
                            end
         'l_stop_time'    : begin  ;-- stop time (HH:MM:SS) -- 
                              aa=parts[2]
                              bb=STR_SEP(aa,':')
                              if (bb[0] lt 0) or (bb[0] gt 23) then begin
                                 error_msg = 'Invalid stop hour entered (00 - 23).'
                                 close, inlun
                                 free_lun, inlun
                                 return, FAILURE
                              endif
                              if (bb[1] lt 0) or (bb[1] gt 59) then begin
                                 error_msg = 'Invalid stop minutes entered (00 - 59).'
                                 close, inlun
                                 free_lun, inlun
                                 return, FAILURE
                              endif
                              if (bb[2] lt 0) or (bb[2] gt 59) then begin
                                 error_msg = 'Invalid stop seconds entered (00 - 59).'
                                 close, inlun
                                 free_lun, inlun
                                 return, FAILURE
                              endif
                              l_setup.l_stop_time=uint(bb(*))
                            end
         'l_data_type'    : begin  ;-- indicates science/hk packets --
                              l_setup.l_data_type=STRLOWCASE(parts[2])
                            end
         'l_plot_type'    : begin  ;-- type of plot, P(PS), X, G(GIF) --
                              l_setup.l_plot_type=STRUPCASE(parts[2])
                            end
         'l_data_src'     : begin  ;-- data source --
                              l_setup.l_data_src=parts[2]
                            end
         'l_prod_dst'     : begin  ;-- path to store product --
                              temp = parts[2]
                              str_len = strlen(temp)
                              if (str_len eq 0) then begin
                                 l_setup.l_prod_dst = '.'
                              endif else begin
                                 lchrn = str_len - 1
                                 lchr = strmid(temp, lchrn, lchrn)
                                 if (lchr eq '/') then begin
                                    l_setup.l_prod_dst = strmid(temp, 0, lchrn)
                                 endif else begin
                                    l_setup.l_prod_dst=temp
                                 endelse
                              endelse
                            end
         'l_num_spin'     : begin  ;-- nspins to process --
                              l_setup.l_num_spin=parts[2]
                            end
         'l_num_energylev': begin  ;-- energy levels --
                              l_setup.l_num_energylev=parts[2]
                            end
         'l_num_polarzn'  : begin  ;-- polar zones --
                              l_setup.l_num_polarzn=parts[2]
                            end
         'l_num_spinsec'  : begin  ;-- spin sectors --
                              l_setup.l_num_spinsec=parts[2]
                            end
         'l_gen_report'   : begin  ;-- generate report --
                              l_setup.l_gen_report=STRUPCASE(parts[2])
                            end
         'l_report_file'  : begin  ;-- report file loc/name --
                              l_setup.l_report_file=STRUPCASE(parts[2])
                            end
         'l_gen_movie'    : begin  ;-- generate movie --
                              l_setup.l_gen_movie=STRUPCASE(parts[2])
                            end
         'l_image_id'     : begin  ;-- image mass to plot --
                              l_setup.l_image_id=STRUPCASE(parts[2])
                            end
         'l_image_correction': begin  ;-- correct distorted image --
                              l_setup.l_image_correction=STRUPCASE(parts[2])
                            end
         'l_color_tbl'    : begin  ;-- color table selection --
                              l_setup.l_color_tbl=parts[2]
                            end
         'l_linear_log'   : begin  ;-- linear or log --
                              l_setup.l_linear_log=parts[2]
                            end
         'l_color_min'    : begin  ;-- min plotting range --
                              l_setup.l_color_min=parts[2]
                            end
         'l_color_max'    : begin  ;-- max plotting range --
                              l_setup.l_color_max=parts[2]
                            end
         'l_auto_scale'   : begin  ;-- max color to scale to --
                              l_setup.l_auto_scale=STRUPCASE(parts[2])
                            end
         'l_sep_plot_scale': begin  ;-- scale each plot separately --
                              l_setup.l_sep_plot_scale=STRUPCASE(parts[2])
                            end
         'l_avg_sum'      : begin  ;-- image avg(A) or summed (S) --
                              l_setup.l_avg_sum=STRUPCASE(parts[2])
                            end
         'l_line_plots'   : begin  ;-- image Y/N --
                              l_setup.l_line_plots = STRUPCASE(parts[2])
                            end
         'l_earth_markers': begin  ;-- image Y/N --
                              l_setup.l_earth_markers= STRUPCASE(parts[2])
                            end
         'l_angle_zero'   : begin  ;-- S(sun), E(Earth), M(Moon) --
                              l_setup.l_angle_zero= STRUPCASE(parts[2])
                            end
         'l_sun_direction': begin  ;-- draw the sun direction  --
                              l_setup.l_sun_direction = STRUPCASE(parts[2])
                            end
         'l_output_type'  : begin  ;-- image plotted as 12x45 or 12x12 --
                              l_setup.l_output_type=parts[2]
                            end
         'l_dumps'        : begin  ;-- image avg(A) or summed (S) --
                              l_setup.l_dumps=STRUPCASE(parts[2])
                            end
         'l_dumps_bkdata' : begin  ;-- image avg(A) or summed (S) --
                              l_setup.l_dumps_bkdata=STRUPCASE(parts[2])
                            end
         'l_pwrsup_bars'  : begin  ;-- power supply bars  --
                              l_setup.l_pwrsup_bars=STRUPCASE(parts[2])
                            end
         'l_postscript'   : begin  ;-- gen poscript files --
                              l_setup.l_postscript=STRUPCASE(parts[2])
                            end
         'l_singles_sum'  : begin  ;-- summed or single spins --
                              l_setup.l_singles_sum=STRUPCASE(parts[2])
                            end
         'l_count_rate'   : begin  ;-- count rate conversion --
                              l_setup.l_count_rate=STRUPCASE(parts[2])
                            end
         'l_log'          : begin  ;-- log conversion --
                              l_setup.l_log=STRUPCASE(parts[2])
                            end
         'l_flux'          : begin  ;-- log conversion --
                              l_setup.l_flux=STRUPCASE(parts[2])
                            end
         'l_max_spin_angle': begin  ;-- max spin angle in degrees --
                              l_setup.l_max_spin_angle=parts[2]
                            end
         'l_min_spin_angle': begin  ;-- min spin angle in degrees --
                              l_setup.l_min_spin_angle=parts[2]
                            end
         'l_ram_direction': begin  ;-- ram direction --
                              l_setup.l_ram_direction=STRUPCASE(parts[2])
                            end
         'l_b_direction'  : begin  ;-- magnetic field direction --
                              l_setup.l_B_direction=STRUPCASE(parts[2])
                            end
         'l_xaxis_Label'  : begin  ;-- X axis labelling type --
                              l_setup.l_Xaxis_Label=STRUPCASE(parts[2])
                            end
         'l_xaxis_ymd_label': begin  ;-- X axis labelling type --
                              l_setup.l_Xaxis_ymd_label=STRUPCASE(parts[2])
                            end
         'l_tokardisplay' : begin  ;-- LTT Tokar display option --
                              l_setup.l_TokarDisplay=STRUPCASE(parts[2])
                            end
         'l_color_values' : begin  ;-- dump color plot values to text dump --
                              l_setup.l_color_values=STRUPCASE(parts[2])
                            end
         'l_smooth_data' : begin  ;-- smooth the spectrogram data --
                              l_setup.l_smooth_data =STRUPCASE(parts[2])
                            end
         'l_output_plot_format' : begin  ;-- plot output format --
                              l_setup.l_output_plot_format=STRUPCASE(parts[2])
                            end

         else             : begin
                             ;print, 'Unknown option : ', parts[0]
                            end
       endcase                         ;End of CASE statement.
       endif
    endif
  endwhile

  close, inlun
  free_lun, inlun


  RETURN, SUCCESS

END


;=========================================================================
; NAME:  l_ShowSetup
;=========================================================================
PRO l_ShowSetup, l_setup
COMMON GLOBALS, G_REPORT, G_LOG

  printf, G_LOG, '____LenaPlot General Setup Parameters'
  printf, G_LOG, 'L_VIA_WEB                : ', l_setup.l_via_web
  printf, G_LOG, 'L_START_DTIME            : ', l_setup.l_start_dtime
  printf, G_LOG, 'L_STOP_DTIME             : ', l_setup.l_stop_dtime
  printf, G_LOG, 'L_DATA_SRC               : ', l_setup.l_data_src
  printf, G_LOG, 'L_PROD_DST               : ', l_setup.l_prod_dst
  printf, G_LOG, 'L_NUM_SPIN               : ', l_setup.l_num_spin
  printf, G_LOG, 'L_NUM_ENERGYLEV          : ', l_setup.l_num_energylev
  printf, G_LOG, 'L_NUM_POLARZN            : ', l_setup.l_num_polarzn
  printf, G_LOG, 'L_NUM_SPINSEC            : ', l_setup.l_num_spinsec
  printf, G_LOG, 'L_PLOT_XSIZE             : ', l_setup.l_plot_xsize
  printf, G_LOG, 'L_PLOT_YSIZE             : ', l_setup.l_plot_ysize
  printf, G_LOG, 'L_GEN_REPORT             : ', l_setup.l_gen_report
  printf, G_LOG, 'L_REPORT_FILENAME        : ', l_setup.l_report_file
  printf, G_LOG, 'L_GEN_GIF                : ', l_setup.l_gen_gif
  printf, G_LOG, 'L_GEN_MOVIE              : ', l_setup.l_gen_movie
  printf, G_LOG, 'L_COLOR_TBL              : ', l_setup.l_color_tbl

  RETURN

END
