
;-------------------------------------------------------------------------------
;
;  Unit Name : l_PwSupMonRtns.pro
;
;  Purpose   : A file containing all of the routines related to reading, writing,
;              and plotting of the instrument state data.  Power supply data for
;              the monitored levels, plus temperatures.
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;   ELH               06/00           v1.0            Original Implementation
;
;
;  File Revision Number:   %I%
;  File Last Modified  :   %E%  %T%
;
;-------------------------------------------------------------------------------

;-------------------------------------------------------------------------------
;  Function :  l_PwSupMonReadData
;
;  Description: Read the Instrument data (hskp commands), monitored levels.
;
;  Return Value:  type = int
;  Value                         Description
;  -------------------------     ------------------------------
;  SUCCESS                       successful completion
;  FAILURE                       unsuccessful completion
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_setup               struct        I        generic setup parameters
;  l_psm                 struct *      I        instrument data parameters
;  l_pltpos              struct        I        generic plot parameters
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;      ERROR_TYPES     SUCCESS         int         G          successful completion
;                      FAILURE         int         G          unsuccessful completion
;                      WARNING         int         G          warning message
;                      INFO            int         G          informational message
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------

FUNCTION l_PwSupMonReadData, l_setup, l_psm, l_pltpos
COMMON ERROR_TYPES, SUCCESS, FAILURE, WARNING, INFO

   inst    = 'LENA'
   expr    = 'LENAHSKP'
   vinst   = 'IMLHSKP'
   data_source = ''

   ;-------------------------------------------------  
   ; get the number of records from T1 to T2
   ;-------------------------------------------------  
   ; get the number of records from T1 to T2
   numRecs = l_UdfGetNumRec (l_setup.l_start_dtime, $
                             l_setup.l_stop_dtime,  $
                             inst, expr, vinst, data_source )
   if (numRecs le 0) then return, FAILURE


   l_pltpos.data_source = data_source

   array_struct = replicate ({UDF_TIMES_STRUCT}, numRecs) 
   ptr = ptr_new ()
   l_psm = {                       $
      hvps_vmonsPtr          : ptr, $
      hvps_vmonsNParms       : 6L,  $
      temp_monsPtr           : ptr, $
      temp_monsNParms        : 8L,  $
      startUdfTimePtr             : ptr $
   }

   ;-------------------------------------------------  
   ; read the data from the UDF DB
   ;-------------------------------------------------  
   fh = l_UdfGetDataHandle (inst, expr, vinst, $
                            l_setup.l_start_dtime, $
                            l_setup.l_stop_dtime, data_source, /GROUP)
   if (fh eq -1) then begin
       udf_close, fh
       RETURN, FAILURE
   endif

   indx = 0L

   temp_mons    = fltarr (numRecs, l_psm.temp_monsNParms)
   hvps_vmons   = fltarr (numRecs, l_psm.hvps_vmonsNParms)


   while not udf_eof (fh) do begin
      d = udf_read (fh)
      array_struct[indx].year = d.btime.year
      array_struct[indx].doy  = d.btime.doy
      array_struct[indx].hour = d.btime.hour
      array_struct[indx].min  = d.btime.min
      array_struct[indx].sec  = d.btime.sec
      array_struct[indx].msec = d.btime.msec
      temp_mons[indx, *]      = d.temperature_monitors[*]
      hvps_vmons[indx, *]     = d.hvps_voltage_monitors[*]
      indx = indx + 1
 

   endwhile

   udf_close, fh


   ptr = ptr_new (temp_mons, /NO_COPY)
   l_psm.temp_monsPtr = ptr

   ptr = ptr_new (hvps_vmons, /NO_COPY)
   l_psm.hvps_vmonsPtr = ptr

   ptr = ptr_new (array_struct, /NO_COPY)
   l_psm.startUdfTimePtr = ptr


   RETURN, SUCCESS
 
end

;-------------------------------------------------------------------------------
;  Procedure: l_PwSupMonTerm
;
;  Description: Free memory allocated for the power supply monitor data. 
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_psm                struct*        I        instrument state data parameters
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;-------------------------------------------------------------------------------

PRO l_PwSupMonTerm, l_psm

   if (ptr_valid (l_psm.temp_monsPtr)) then begin
       ptr_free, l_psm.temp_monsPtr
   endif

   if (ptr_valid (l_psm.hvps_vmonsPtr)) then begin
       ptr_free, l_psm.hvps_vmonsPtr
   endif

   if (ptr_valid (l_psm.startUdfTimePtr)) then begin
       ptr_free, l_psm.startUdfTimePtr
   endif

return
end


;-------------------------------------------------------------------------------
;  Procedure: l_PwSupMonDump
;
;  Description: Dump instrument data to a text file
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_psm                 struct*       I        hskp data parameters
;  l_setup               struct        I        generic plot setup parameters
;  l_pltpos              struct        I        generic plot parameters
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;-------------------------------------------------------------------------------

PRO l_PwSupMonDump, l_psm, l_setup, l_pltpos

   ;-------------------------------
   ; create text dump filename
   ;-------------------------------
   l_setup.l_filedesc = 'psm'
   l_GenFileName, l_setup, '0'
   filename = l_setup.l_prod_dst + '/' + l_setup.l_filename + '.dmp'

   ;-------------------------------
   ; open text dump file
   ;-------------------------------
   openw, ofd, filename, /get_lun

   ;-------------------------------
   ; write standard header info
   ; to text file
   ;-------------------------------
   l_DumpHdrInfo, ofd, l_setup, l_pltpos, filename

   ;-------------------------------
   ; column definitions
   ;-------------------------------
   printf, ofd, '--------------- Power Supply Text Dumps -------------------------------'
   printf, ofd, ' '
   printf, ofd, '   Column Definitions:  '
   printf, ofd, ' '
   printf, ofd, '      UTC        : Universal Time Coordinate (corrected)'
   printf, ofd, '      MCP Start  : HVPS MCP Start Voltage Monitor'
   printf, ofd, '      MCP Stop   : HVPS MCP Stop Voltage Monitor'
   printf, ofd, '      Coll +     : HVPS COLLP Voltage Monitor'
   printf, ofd, '      Coll -     : HVPS COLLN Voltage Monitor'
   printf, ofd, '      Optics     : HVPS Optics Voltage Monitor'
   printf, ofd, '      Str Mon    : Steering controller level, HVPS Steering Voltage Monitor'
   printf, ofd, '      Temp 0     : Temperature 0, Hex'
   printf, ofd, '      Temp 0 (C) : Temperature 0, Celsius'
   printf, ofd, '      Temp 1     : Temperature 1, Hex'
   printf, ofd, '      Temp 1 (C) : Temperature 1, Celsius'
   printf, ofd, ' '
   printf, ofd, '-------------------------------------------------------------------------'
   printf, ofd, ' '
   printf, ofd, ' '
   printf, ofd, ' '


   printf, ofd, 'Values are in HEX, except for 2 of the temperatures (noted in Celsius).'
   printf, ofd, 'd = disabled'
   printf, ofd, ' '

   ;-------------------------------
   ; column headings
   ;-------------------------------
   printf, ofd,'  ',' MCP ','  MCP ', $
                format='((A8,TR1),2(A10))'

   printf, ofd,'UTC','Start','Stop', 'Coll +','Coll -', 'Optics', $
               'Str Mon', 'Temp0', 'Temp 0 (C)', 'Temp1', 'Temp 1 (C)', $
                format='((A8,TR1),7(A10), (A12), (A10), (A12))'

   printf, ofd,'---','-----','----', '------','------','------', $
               '--------', '-----', '----------', '-----', '---------', $
               format='((A8,TR1),7(A10), (A12), (A10), (A12))'



   ;-------------------------------------------------
   ; a 1 indicates enabled, 0 indicates disabled
   ;-------------------------------------------------
   prev_mcp_start_flag = '00'X 
   prev_mcp_stop_flag  = '00'X
   prev_optics_flag    = '00'X
   prev_coll_n_flag    = '00'X
   prev_coll_p_flag    = '00'X
   prev_str_cmd_flag   = '00'X
   prev_str_mon_flag   = '00'X
   prev_temp0_flag     = '00'X
   prev_temp1_flag     = '00'X

   ;-----------------------------
   ;  **** NEED TO HANDLE MISSING
   ;       OA DATA *********
   ;-----------------------------
   nele = n_elements((*l_psm.hvps_vmonsPtr)[*,0])


   for i = 0L, nele-1L do begin

     change = 0L
     tmestr = STRING ((*l_psm.startUdfTimePtr)[i].doy, $
                      (*l_psm.startUdfTimePtr)[i].hour,$
                      (*l_psm.startUdfTimePtr)[i].min, $
                      (*l_psm.startUdfTimePtr)[i].sec, $
                      format="(i3,' ',i2.2,':',i2.2,':',i2.2)")

     ;-------------------------------
     ; build printf formatted output
     ; string based on data type
     ;-------------------------------

     dumpStr = 'printf, ofd, tmestr, mcp_start_flag, mcp_stop_flag, '
     dumpStr = dumpStr + 'coll_p_flag, coll_n_flag, optics_flag, '
     dumpStr = dumpStr + 'str_mon_flag, temp0_flag, temp0_celsius, '
     dumpStr = dumpStr + 'temp1_flag, temp1_celsius, format=''((A12,2X), '

     ;----------------------------------------
     ;** MCP start, e/d flag **
     ;----------------------------------------

     mcp_start_flag = (*l_psm.hvps_vmonsPtr)[i,0]
     dumpStr = dumpStr + '(Z3.2, 8X),'
     if (prev_mcp_start_flag ne mcp_start_flag) then change = 1L


     ;----------------------------------------
     ;** MCP stop, e/d flag **
     ;----------------------------------------
     mcp_stop_flag = (*l_psm.hvps_vmonsPtr)[i,1]
     dumpStr = dumpStr + '(Z3.2, 8X),'
     if (prev_mcp_stop_flag ne mcp_stop_flag) then change = 1L


     ;----------------------------------------
     ;** collimator +, e/d flag **
     ;----------------------------------------
     coll_p_flag = (*l_psm.hvps_vmonsPtr)[i,2]
     dumpStr = dumpStr + '(Z3.2, 6X),'
     if (prev_coll_p_flag ne coll_p_flag) then change = 1L


     ;----------------------------------------
     ;** collimator -, e/d flag**
     ;----------------------------------------
     coll_n_flag = (*l_psm.hvps_vmonsPtr)[i,3]
     dumpStr = dumpStr + '(Z3.2, 6X), '
     if (prev_coll_n_flag ne coll_n_flag) then change = 1L


     ;----------------------------------------
     ;** optics, e/d flag **
     ;----------------------------------------
     optics_flag = (*l_psm.hvps_vmonsPtr)[i,4]
     dumpStr = dumpStr + '(Z3.2, 6X),'
     if (prev_optics_flag ne optics_flag) then change = 1L


     ;----------------------------------------
     ;** Steering voltage monitor level **
     ;----------------------------------------
     str_mon_flag = (*l_psm.hvps_vmonsPtr)[i,5]
     if (prev_str_mon_flag ne str_mon_flag) then change = 1L
     dumpStr = dumpStr + '(Z3.2,8X), '


     ;----------------------------------------
     ;** Temperature 0 **
     ;----------------------------------------
     temp0_flag = (*l_psm.temp_monsPtr)[i,0]
     dumpStr = dumpStr + '(Z3.2,6X), '
     temp0_celsius = l_ConvHKTemp((*l_psm.temp_monsPtr)[i,0])
     dumpStr = dumpStr + '(F6.2,8X), '
     if (prev_temp0_flag ne temp0_flag) then change = 1L


     ;----------------------------------------
     ;** Temperature 1 **
     ;----------------------------------------
     temp1_flag = (*l_psm.temp_monsPtr)[i,1]
     dumpStr = dumpStr + '(Z3.2,6X), '
     temp1_celsius = l_ConvHKTemp ((*l_psm.temp_monsPtr)[i,1])
     dumpStr = dumpStr + '(F6.2,1X)'
     if (prev_temp1_flag ne temp1_flag) then change = 1L


     dumpStr = dumpStr + ')'''

     ;-------------------------------
     ; execute the printf
     ;-------------------------------
     if (change eq 1L) then begin
        result = execute (dumpStr)
        prev_mcp_start_flag = mcp_start_flag
        prev_mcp_stop_flag = mcp_stop_flag
        prev_optics_flag = optics_flag 
        prev_coll_n_flag = coll_n_flag
        prev_coll_p_flag = coll_p_flag
        prev_str_mon_flag = str_mon_flag
        prev_temp0_flag = temp0_flag
        prev_temp1_flag = temp1_flag
     endif

     dumpStr = ' '

   endfor

   ;-------------------------------
   ; close text file
   ;-------------------------------
   close, ofd
   free_lun, ofd

return
end


;-------------------------------------------------------------------------------
;  Function : l_PwSupMonProc
;
;  Description: Calls necessary procedures to generate Instrument State
;               information
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  SUCCESS                       successful completion
;  FAILURE                       failure to complete
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_setup               struct        I        general setup parameters
;  l_pltpos              struct        I        general plotting parameters
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;      ERROR_TYPES     SUCCESS         int         G          successful completion
;                      FAILURE         int         G          unsuccessful completion
;                      WARNING         int         G          warning message
;                      INFO            int         G          informational message
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------

FUNCTION l_PwSupMonProc, l_setup, l_pltpos
COMMON ERROR_TYPES, SUCCESS, FAILURE, WARNING, INFO

    status = l_PwSupMonReadData (l_setup, l_psm, l_pltpos)
    if (status eq FAILURE) then return, FAILURE

    l_PwSupMonDump, l_psm, l_setup, l_pltpos
    l_PwSupMonTerm, l_psm

    RETURN, SUCCESS
end

