
;-------------------------------------------------------------------------------
;
;  Unit Name : l_PwSupCmdRtns.pro
;
;  Purpose   : A file containing all of the routines related to reading, writing,
;              and plotting of the instrument state data.
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;   ELH               06/00           v1.0            Original Implementation
;   ELH               01/02           v1.0.8          Added parm in l_HndlMissData to
;                                                     handle rolling data based on
;                                                     spin angle input
;
;
;  File Revision Number:   %I%
;  File Last Modified  :   %E%  %T%
;
;-------------------------------------------------------------------------------

;-------------------------------------------------------------------------------
;  Function :  l_PwSupCmdReadData
;
;  Description:  Read the PwSupCmdrument data (hskp commands)
;
;  Return Value:  type = int
;  Value                         Description
;  -------------------------     ------------------------------
;  SUCCESS                       successful completion
;  FAILURE                       unsuccessful completion
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_setup               struct        I        generic setup parameters
;  l_psc                 struct        I        instrument data parameters
;  l_pltpos              struct        I        generic plot parameters
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;      ERROR_TYPES     SUCCESS         int         G          successful completion
;                      FAILURE         int         G          unsuccessful completion
;                      WARNING         int         G          warning message
;                      INFO            int         G          informational message
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11            Original implementation
;-------------------------------------------------------------------------------

FUNCTION l_PwSupCmdReadData, l_setup, l_psc, l_pltpos
COMMON ERROR_TYPES, SUCCESS, FAILURE, WARNING, INFO

   inst    = 'LENA'
   expr    = 'LENAHSKP'
   vinst   = 'IMLHSKP'
   data_source = ''

   ;------------------------------------------------
   ; get the number of records from T1 to T2
   ;------------------------------------------------
   numRecs = l_UdfGetNumRec (l_setup.l_start_dtime, $
                             l_setup.l_stop_dtime,  $
                             inst, expr, vinst, data_source )

   if (numRecs le 0) then return, FAILURE

   l_pltpos.data_source = data_source

   ;------------------------------------------------
   ; define data types
   ;------------------------------------------------
   array_struct = replicate ({UDF_TIMES_STRUCT}, numRecs) 
   ptr = ptr_new ()
   l_psc = {                       $
      hvps_configPtr         : ptr, $
      hvps_configNParms      : 14L, $
      hvps_vmonsPtr          : ptr, $
      hvps_vmonsNParms       : 6L,  $
      startUdfTimePtr             : ptr $
   }


   ;------------------------------------------------
   ; read data from UDF DB
   ;------------------------------------------------
   fh = l_UdfGetDataHandle (inst, expr, vinst, $
                            l_setup.l_start_dtime, $
                            l_setup.l_stop_dtime, data_source, /GROUP)
   if (fh eq -1) then begin
      udf_close, fh
      RETURN, FAILURE
   endif

   indx = 0L

   hvps_config       = fltarr (numRecs, l_psc.hvps_configNParms)
   hvps_vmons        = fltarr (numRecs, l_psc.hvps_vmonsNParms)


   while not udf_eof (fh) do begin
      d = udf_read (fh)
      array_struct[indx].year = d.btime.year
      array_struct[indx].doy  = d.btime.doy
      array_struct[indx].hour = d.btime.hour
      array_struct[indx].min  = d.btime.min
      array_struct[indx].sec  = d.btime.sec
      array_struct[indx].msec = d.btime.msec
      hvps_config[indx, *]       = d.hvps_configuration[*]
      hvps_vmons[indx, *]       = d.hvps_voltage_monitors[*]
      indx = indx + 1
 

   endwhile

   udf_close, fh


   ptr = ptr_new (hvps_config, /NO_COPY)
   l_psc.hvps_configPtr = ptr

   ptr = ptr_new (hvps_vmons, /NO_COPY)
   l_psc.hvps_vmonsPtr = ptr

   ptr = ptr_new (array_struct, /NO_COPY)
   l_psc.startUdfTimePtr = ptr

   RETURN, SUCCESS
 
end

;-------------------------------------------------------------------------------
;  Procedure: l_PwSupCmdTerm
;
;  Description: Free memory allocated to power supply commands. 
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_psc                struct        I        instrument state data parameters
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11          Original implementation
;-------------------------------------------------------------------------------

PRO l_PwSupCmdTerm, l_psc

   if (ptr_valid (l_psc.hvps_configPtr)) then begin
       ptr_free, l_psc.hvps_configPtr
   endif

   if (ptr_valid (l_psc.hvps_vmonsPtr)) then begin
       ptr_free, l_psc.hvps_vmonsPtr
   endif

   if (ptr_valid (l_psc.startUdfTimePtr)) then begin
       ptr_free, l_psc.startUdfTimePtr
   endif

return
end


;-------------------------------------------------------------------------------
;  Procedure: l_PwSupCmdDump
;
;  Description: Dump instrument data to a text file
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_psc                 struct*       I        hskp data parameters
;  l_setup               struct        I        generic plot setup parameters
;  l_pltpos              struct        I        generic plot parameters
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11          Original implementation
;-------------------------------------------------------------------------------

PRO l_PwSupCmdDump, l_psc, l_setup, l_pltpos

   ;-------------------------------
   ; create text dump filename
   ;-------------------------------
   l_setup.l_filedesc = 'psc'
   l_GenFileName, l_setup, '0'
   filename = l_setup.l_prod_dst + '/' + l_setup.l_filename + '.dmp'

   ;-------------------------------
   ; open text dump file
   ;-------------------------------
   openw, ofd, filename, /get_lun

   ;-------------------------------
   ; write standard header info
   ; to text file
   ;-------------------------------
   l_pltpos.dmp_hdr_title = 'Power Supply & Steering Controller'
   l_DumpHdrInfo, ofd, l_setup, l_pltpos, filename

   ;-------------------------------
   ; column definitions
   ;-------------------------------
   printf, ofd, '--------------- Power Supply Text Dumps -------------------------------'
   printf, ofd, ' '
   printf, ofd, '   Column Definitions:  '
   printf, ofd, ' '
   printf, ofd, '      UTC        : Universal Time Coordinate (corrected)'
   printf, ofd, '      MCP Start  : HVPS MCP Start Command'
   printf, ofd, '      MCP Stop   : HVPS MCP Stop Command'
   printf, ofd, '      Optics     : HVPS Optics Command'
   printf, ofd, '      Coll -     : HVPS COLLN Command'
   printf, ofd, '      Coll +     : HVPS COLLP Command'
   printf, ofd, '      Str Cmd    : Steering controller level, HVPS Optics Steering Command'
   printf, ofd, '      Str Mon    : Steering controller level, HVPS Steering Voltage Monitor'
   printf, ofd, ' '
   printf, ofd, '-------------------------------------------------------------------------'
   printf, ofd, ' '
   printf, ofd, ' '
   printf, ofd, ' '


   printf, ofd, 'Values are in HEX.'
   printf, ofd, 'd = disabled'
   printf, ofd, ' '

   ;-------------------------------
   ; column headings
   ;-------------------------------
   printf, ofd,'  ','MCP','MCP', $
                format='((A8,TR1),2(A10))'

   printf, ofd,'UTC','Start','Stop', 'Optics','Coll -','Coll +', $
               'Str Cmd', 'Str Mon', $
                format='((A8,TR1),7(A10))'

   printf, ofd,'---','-----','----', '------','------','------', $
               '--------', '--------', $
               format='((A8,TR1),7(A10))'



   ;-------------------------------------------------
   ; a 1 indicates enabled, 0 indicates disabled
   ;-------------------------------------------------
   prev_mcp_start_flag = 0L 
   prev_mcp_stop_flag  = 0L
   prev_optics_flag    = 0L
   prev_coll_n_flag    = 0L
   prev_coll_p_flag    = 0L
   prev_str_cmd_flag   = 0L
   prev_str_mon_flag   = 0L

   ;-----------------------------
   ;  **** NEED TO HANDLE MISSING
   ;       OA DATA *********
   ;-----------------------------
   nele = n_elements((*l_psc.hvps_configPtr)[*,2])


   for i = 0L, nele-1L do begin

     change = 0L
     tmestr = STRING ((*l_psc.startUdfTimePtr)[i].doy, $
                      (*l_psc.startUdfTimePtr)[i].hour,$
                      (*l_psc.startUdfTimePtr)[i].min, $
                      (*l_psc.startUdfTimePtr)[i].sec, $
                      format="(i3,' ',i2.2,':',i2.2,':',i2.2)")

     ;-------------------------------
     ; build printf formatted output
     ; string based on data type
     ;-------------------------------

     dumpStr = 'printf, ofd, tmestr, mcp_start_flag, mcp_stop_flag, '
     dumpStr = dumpStr + 'optics_flag, coll_n_flag, coll_p_flag, '
     dumpStr = dumpStr + 'str_cmd_flag, str_mon_flag, format=''((A12,1X), '

     ;** MCP start, e/d flag **

     if ((*l_psc.hvps_configPtr)[i,2] eq 0) then begin
         mcp_start_flag = 'd'
         curr_mcp_start_flag = -99999
         dumpStr = dumpStr + '(A3, 8X),'
     endif else begin
         mcp_start_flag = (*l_psc.hvps_configPtr)[i,8]
         curr_mcp_start_flag = mcp_start_flag 
         dumpStr = dumpStr + '(Z3.2, 8X),'
     endelse
     if (prev_mcp_start_flag ne curr_mcp_start_flag) then change = 1L

     ;** MCP stop, e/d flag **
     if ((*l_psc.hvps_configPtr)[i,3] eq 0) then begin
         mcp_stop_flag = 'd'
         curr_mcp_stop_flag = -99999
         dumpStr = dumpStr + '(A3, 8X),'
     endif else begin
         mcp_stop_flag = (*l_psc.hvps_configPtr)[i,9]
         curr_mcp_stop_flag = mcp_stop_flag
         dumpStr = dumpStr + '(Z3.2, 8X),'
     endelse
     if (prev_mcp_stop_flag ne curr_mcp_stop_flag) then change = 1L


     ;** optics, e/d flag **
     if ((*l_psc.hvps_configPtr)[i,6] eq 0) then begin
         optics_flag = 'd'
         curr_optics_flag = -99999
         dumpStr = dumpStr + '(A3, 6X),'
     endif else begin
         optics_flag = (*l_psc.hvps_configPtr)[i,12]
         curr_optics_flag = optics_flag
         dumpStr = dumpStr + '(Z3.2, 6X),'
     endelse
     if (prev_optics_flag ne curr_optics_flag) then change = 1L

     ;** collimator -, e/d flag**
     if ((*l_psc.hvps_configPtr)[i,5] eq 0) then begin
         coll_n_flag = 'd'
         curr_coll_n_flag = -99999
         dumpStr = dumpStr + '(A3, 6X), '
     endif else begin
         coll_n_flag = (*l_psc.hvps_configPtr)[i,11]
         curr_coll_n_flag = coll_n_flag
         dumpStr = dumpStr + '(Z3.2, 6X), '
     endelse
     if (prev_coll_n_flag ne curr_coll_n_flag) then change = 1L

     ;** collimator +, e/d flag **
     if ((*l_psc.hvps_configPtr)[i,4] eq 0) then begin
         coll_p_flag = 'd'
         curr_coll_p_flag = -99999
         dumpStr = dumpStr + '(A3, 6X),'
     endif else begin
         coll_p_flag = (*l_psc.hvps_configPtr)[i,10]
         curr_coll_p_flag = coll_p_flag
         dumpStr = dumpStr + '(Z3.2, 6X),'
     endelse
     if (prev_coll_p_flag ne curr_coll_p_flag) then change = 1L



     ;** Steering commanded level **
     str_cmd_flag = (*l_psc.hvps_configPtr)[i,13]
     if (prev_str_cmd_flag ne str_cmd_flag) then change = 1L

     ;** Steering voltage monitor level **
     str_mon_flag = (*l_psc.hvps_vmonsPtr)[i,5]
     if (prev_str_mon_flag ne str_mon_flag) then change = 1L

     dumpStr = dumpStr + '2(Z3.2,8X)'
     dumpStr = dumpStr + ')'''


     ;-------------------------------
     ; execute the printf
     ;-------------------------------
     if (change eq 1L) then begin
        result = execute (dumpStr)
        prev_mcp_start_flag = curr_mcp_start_flag
        prev_mcp_stop_flag = curr_mcp_stop_flag
        prev_optics_flag = curr_optics_flag 
        prev_coll_n_flag = curr_coll_n_flag
        prev_coll_p_flag = curr_coll_p_flag
        prev_str_cmd_flag = str_cmd_flag
        prev_str_mon_flag = str_mon_flag
     endif

   endfor

   ;-------------------------------
   ; close text file
   ;-------------------------------
   close, ofd
   free_lun, ofd

return
end


;-------------------------------------------------------------------------------
;  Procedure : l_CreateOpBars
;
;  Description: Create the operational bars for the selected parameters
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  theField              long[]        I        instrument state data 
;  xsize                 long          I        # times pixel replicated on X-axis
;  ximage                float         I        width of image 
;  yimage                float         I        height of image
;  ypos                  float         I        y position of image
;  pos                   float[]       I        xyz plot coordinates
;  label                 string        I        data text label
;  font                  long          I        font size
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH              04/23/03         v1.0.11         Original implementation
;-------------------------------------------------------------------------------

PRO l_CreateOpBars, theField, xsize, ximage, yimage, ypos, $
pos, label, font

   parm = theField
   tmp = bytarr (n_elements(parm), 3)

   tmp[*,0] = parm[*]
   tmp[*,1] = parm[*]
   tmp[*,2] = parm[*]

   opbar = congrid (tmp,  xsize, 6)


   TV, opbar, pos[0], ypos, $
       XSIZE=ximage, YSIZE=yimage, $
       /NORMAL

   plots, [pos[0], pos[0], pos[2], pos[2], pos[0]], $
          [ypos, ypos+.006, ypos+.006, ypos, ypos], $
          /normal

   xyouts, pos[2]+.006, ypos, /normal, label , charsize=font


   RETURN

END

;-------------------------------------------------------------------------------
;  Procedure : l_ChckDEBars
;
;  Description:
;      Check to see if the instrument voltage is disabled
;      or enabled.  If disabled, then fill with white to indicate
;      that the power supply is off.  Scale the colors in the
;      power supply arrays to the current color table.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  inData                float[]       I        input data values
;  outData               float[]       O        color scaled data values
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;     PLOTDEFS         MAX_PLOTS        int         G         max # plots per page
;                      MAX_WEB_SPINS    int         G         max # spins per plot
;                                                             when using web interface
;                      NO_CNTS          double      G         no counts indicator
;                      NO_DATA          double      G         no data indicator
;
;      MYCOLORS        N_COLOR_TABLE   int         G          # colors in color table
;                      N_GRAYS         int         G          # grays in color table
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------

PRO l_ChckDEBars, inData, outData
COMMON PLOTDEFS, MAX_PLOTS, MAX_WEB_SPINS, NO_CNTS, NO_DATA
COMMON MYCOLORS, N_COLOR_TABLE, N_GRAYS


   color_min = 0
   color_max = 255

   nitems = n_elements (inData)
   outData = bytarr (nitems)

   ;--------------------------------------
   ; scale the byte values from commands
   ; to color bar
   ;--------------------------------------
   ncolors = float(N_COLOR_TABLE - N_GRAYS)
   outData = fix(((inData-color_min)/(color_max-color_min))* ncolors) + N_GRAYS 

   ;--------------------------------------
   ; value is below the requested minimum
   ; make the lowest color value in table
   ;--------------------------------------
   less_min = where(inData le color_min, wcnt)
   if (wcnt gt 0) then outData[less_min] = 5

   ;--------------------------------------
   ; value is above requested maximum, make
   ; highest color value in table
   ;--------------------------------------
   over_max = where(inData gt color_max, wcnt)
   if (wcnt gt 0) then outData[over_max] = 225

   ;--------------------------------------
   ; no data, make gray
   ;--------------------------------------
   nodata = where(inData eq NO_DATA, wcnt)
   if (wcnt gt 0) then outData[nodata] = 3

   ;--------------------------------------
   ; flag was disabled, make white 
   ;--------------------------------------
   nodata = where(inData eq NO_CNTS, wcnt)
   if (wcnt gt 0) then outData[nodata] = 0 

   RETURN
END


;-------------------------------------------------------------------------------
;  Procedure: SyncOpData
;
;  Description: Synchronize the the power supply data with time that has been
;         filled if there are data gaps.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  theData               float[]      I         input data 
;  theDataTime           struct[]     I         input data time
;  l_setup               struct       I         general setup parameters
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------

PRO SyncOpData, theData, theDataTime, l_setup

   nitems = n_elements (theData)
   new_theData = theData
   cnt = 0L
   tme_arr = dblarr(6)


   for i = 0L, nitems - 1L do begin
      tme_arr[0] = theDataTime[i].year
      tme_arr[1] = theDataTime[i].doy
      tme_arr[2] = theDataTime[i].hour
      tme_arr[3] = theDataTime[i].min
      tme_arr[4] = theDataTime[i].sec
      tme_arr[5] = theDataTime[i].msec

      if (ConvArrTimeToSecs(tme_arr) ge ConvArrTimeToSecs(l_setup.l_start_dtime)) AND $
         (ConvArrTimeToSecs(tme_arr) le ConvArrTimeToSecs(l_setup.l_stop_dtime)) then begin
         new_theData[cnt] = theData[i]
         cnt = cnt + 1L
      endif
   endfor

   theData = new_theData[0:cnt-1L]


RETURN
END

;-------------------------------------------------------------------------------
;  Procedure : l_DrawOpBars
;
;  Description:
;     Display bars on plot indicating operational status
;     of selected parameters
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_psc                 struct *      I        hskp data parameters
;  l_setup               struct        I        generic setup parameters
;  l_pltpos              struct        I        generic plot parameters
;  l_rgb_table           struct        I        generic color table parameters
;  l_oa                  struct *      I        OA data parameters
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;     PLOTDEFS         MAX_PLOTS        int         G         max # plots per page
;                      MAX_WEB_SPINS    int         G         max # spins per plot
;                                                             when using web interface
;                      NO_CNTS          double      G         no counts indicator
;                      NO_DATA          double      G         no data indicator
;
;      MYCOLORS        N_COLOR_TABLE   int         G          # colors in color table
;                      N_GRAYS         int         G          # grays in color table
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;-------------------------------------------------------------------------------

PRO l_DrawOpBars, l_psc, l_setup, l_pltpos, l_rgb_table, l_oa
COMMON PLOTDEFS, MAX_PLOTS, MAX_WEB_SPINS, NO_CNTS, NO_DATA
COMMON MYCOLORS, N_COLOR_TABLE, N_GRAYS


   linear = 1L
   center_sector = 0L

   adjFlag = 'NO'


   ypltmx = 8.8/l_pltpos.yin
   img_pos = [l_pltpos.xpltmn,l_pltpos.ypltmn,$
              l_pltpos.xpltmx,ypltmx]

   position = float(img_pos)

   xstart = position[0]
   ystart = position[3] + .005
   ydelta = 0.015
   ximage = l_pltpos.axis_size/ l_pltpos.xin
   yimage = 0.005
   xsize  = round(l_pltpos.window_width * ximage)
   ysize  = round(l_pltpos.window_height * yimage)

   xpos = xstart
   ypos = ystart

   ;--------------------------
   ;** Off Box **
   ;--------------------------
   off_box_size = 10
   off = bytarr(off_box_size,off_box_size)
   off (*,*) = 0

   bar_w = off_box_size / float(l_pltpos.window_width)
   bar_h = off_box_size / float(l_pltpos.window_height)
   bar_x = 0.03
   bar_y = ypos - (ydelta*3)
   tv, off , bar_x, bar_y, xsize=bar_w, ysize=bar_h, /normal

   box_x = [bar_x, bar_x, bar_x+.016, bar_x+.016, bar_x]
   box_y = [bar_y, bar_y+.015, bar_y+.015, bar_y, bar_y]

   plots, box_x, box_y, /normal
   xyouts, bar_x+.035, bar_y, 'off', $
           alignment=0.5, charsize=l_pltpos.sm_font, /normal


   ;--------------------------
   ;** OP Color Bar Legend**
   ;--------------------------
   rgb = findgen (N_COLOR_TABLE - N_GRAYS)
   rgb = rgb + N_GRAYS 

   l_CreateOpBars, rgb, $
                   xsize, ximage, yimage, ypos, position, $
                   'Bar Legend ', l_pltpos.xsm_font

  ymax = 9.0/l_pltpos.yin - 0.0072
  ymin = ypos
  pos = [l_pltpos.xpltmn, $
         ymin,  $
         l_pltpos.xpltmx, $
         ymax]
  lgd = float(pos)

  ;-------------------------------------------------------- 
  ; specifies location to where to draw X axis labels
  ;-------------------------------------------------------- 
  PLOT, [0,255],              $
        [0,1],                $
        /NODATA,              $
        /NOERASE,             $
        XTICKS = 4,           $
        YTICKS = 1,           $
        XSTYLE = 1,           $
        YSTYLE = 1,           $
        POSITION = lgd,       $
        COLOR = 1,            $
        YTICKFORMAT = '(A1)', $
        XTICKFORMAT = '(A1)', $
        XTICKLEN = -0.2,      $
        FONT=-1,              $
        XMINOR=8,             $
        CHARSIZE = l_pltpos.md_font

   ;---------------------------------------------------
   ;Draw the top x-axis, supplying labels, etc.
   ;---------------------------------------------------
   AXIS, XRANGE=[0, 255], $
         XTICKLEN = -0.2, $
         XAXIS=1,         $
         XTICKS=4,        $
         XMINOR = 8,       $
         XSTYLE = 1,       $
         CHARSIZE = l_pltpos.md_font 


   ypos = ypos - ydelta


   orig_udf_tme = (*l_psc.startUdfTimePtr)[*]
   no_extra = 0L

   ;-------------------------------------------------
   ;** MCP start **
   ; a 1 indicates enabled, 0 indicates disabled
   ; if a flag for a power supply is disabled,
   ; fill with zeros to print out a white bar
   ;-------------------------------------------------
   new_data = 0L
   data = (*l_psc.hvps_configPtr)[*,8]
   flag = (*l_psc.hvps_configPtr)[*,2]
   nitems = n_elements(data)
   for ii = 0L, nitems - 1L do begin
       if (flag[ii] eq 0) then begin
          data[ii] = NO_CNTS 
       endif
   endfor


   l_HndlMissData, (*l_psc.timePtr)[*], $
                   new_tme, $
                   nitems, $
                   data, $
                   new_data , $
                   linear, $
                   center_sector, $
                   l_setup , $
                   adjFlag, $
                   l_oa, $
                   orig_udf_tme, $
                   new_udf_tme, $
                   no_extra, $
                   l_setup.l_spin_ang_zero



   l_ChckDEBars,   new_data, $
                   mcp_start_bar

   SyncOpData, mcp_start_bar, new_udf_tme, l_setup

   l_CreateOpBars, mcp_start_bar, $
                   xsize, ximage, yimage, ypos, position, $
                   'MCP Start Cmd' , l_pltpos.xsm_font


   ypos = ypos - ydelta

   ;------------------------------------
   ;** MCP stop **
   ;------------------------------------
   new_data = 0L
   data = (*l_psc.hvps_configPtr)[*,9]
   flag = (*l_psc.hvps_configPtr)[*,3]
   nitems = n_elements(data)
   for ii = 0L, nitems - 1L do begin
       if (flag[ii] eq 0) then begin
          data[ii] = NO_CNTS 
       endif
   endfor
   l_HndlMissData, (*l_psc.timePtr)[*], $
                   new_tme, $
                   nitems, $
                   data, $
                   new_data, $
                   linear, $
                   center_sector, $
                   l_setup , $
                   adjFlag, $
                   l_oa, $
                   orig_udf_tme, $
                   new_udf_tme, $
                   no_extra, $
                   l_setup.l_spin_ang_zero


   l_ChckDEBars,   new_data, $
                   mcp_stop_bar

   SyncOpData, mcp_stop_bar, new_udf_tme, l_setup

   l_CreateOpBars, mcp_stop_bar, $
                   xsize, ximage, yimage, ypos, position, $
                   'MCP Stop Cmd' , l_pltpos.xsm_font
   ypos = ypos - ydelta



   ;------------------------------------
   ;** optics **
   ;------------------------------------
   new_data = 0L
   data = (*l_psc.hvps_configPtr)[*,12]
   flag = (*l_psc.hvps_configPtr)[*,6]
   nitems = n_elements(data)
   for ii = 0L, nitems - 1L do begin
       if (flag[ii] eq 0) then begin
          data[ii] =  NO_CNTS
       endif
   endfor

   l_HndlMissData, (*l_psc.timePtr)[*], $
                   new_tme, $
                   nitems, $
                   data, $
                   new_data, $
                   linear, $
                   center_sector, $
                   l_setup , $
                   adjFlag, $
                   l_oa, $
                   orig_udf_tme, $
                   new_udf_tme, $
                   no_extra, $
                   l_setup.l_spin_ang_zero


   l_ChckDEBars,   new_data, $
                   optics_bar 

   SyncOpData, optics_bar, new_udf_tme, l_setup

   l_CreateOpBars, optics_bar, $
                   xsize, ximage, yimage, ypos, position, $
                   'Optics Cmd' , l_pltpos.xsm_font
   ypos = ypos - ydelta

   ;------------------------------------
   ;** collimator - **
   ;------------------------------------
   new_data = 0L
   data = (*l_psc.hvps_configPtr)[*,11]
   flag = (*l_psc.hvps_configPtr)[*,5]
   nitems = n_elements(data)
   for ii = 0L, nitems - 1L do begin
       if (flag[ii] eq 0) then begin
          data[ii] = NO_CNTS 
       endif
   endfor


   l_HndlMissData, (*l_psc.timePtr)[*], $
                   new_tme, $
                   nitems, $
                   data, $
                   new_data, $
                   linear, $
                   center_sector, $
                   l_setup , $
                   adjFlag, $
                   l_oa, $
                   orig_udf_tme, $ 
                   new_udf_tme, $
                   no_extra, $
                   l_setup.l_spin_ang_zero

   l_ChckDEBars,   new_data, $
                   coll_n_bar 

   SyncOpData, coll_n_bar, new_udf_tme, l_setup

   l_CreateOpBars, coll_n_bar, $
                   xsize, ximage, yimage, ypos, position, $
                   'Coll - Cmd' , l_pltpos.xsm_font
   ypos = ypos - ydelta


   ;------------------------------------
   ;** collimator + **
   ;------------------------------------
   new_data = 0L
   data = (*l_psc.hvps_configPtr)[*,10]
   flag = (*l_psc.hvps_configPtr)[*,4]
   nitems = n_elements(data)
   for ii = 0L, nitems - 1L do begin
       if (flag[ii] eq 0) then begin
          data[ii] = NO_CNTS
       endif
   endfor

   l_HndlMissData, (*l_psc.timePtr)[*], $
                   new_tme, $
                   nitems, $
                   data, $
                   new_data, $
                   linear, $
                   center_sector, $
                   l_setup , $
                   adjFlag, $
                   l_oa, $
                   orig_udf_tme, $ 
                   new_udf_tme, $
                   no_extra, $
                   l_setup.l_spin_ang_zero


   l_ChckDEBars,   new_data, $
                   coll_p_bar 

   SyncOpData, coll_p_bar, new_udf_tme, l_setup

   l_CreateOpBars, coll_p_bar, $
                   xsize, ximage, yimage, ypos, position, $
                   'Coll + Cmd' , l_pltpos.xsm_font
   ypos = ypos - ydelta



   ;------------------------------------
   ;** steering controller **
   ;-------------------------------------------
   ; scale to color table, skip the first 5 gray
   ; values, begin at lowest color 
   ;-------------------------------------------
   new_data = 0L
   data = (*l_psc.hvps_configPtr)[*,13]
   nitems = n_elements (data)
   l_HndlMissData, (*l_psc.timePtr)[*], $
                   new_tme, $
                   nitems, $
                   data, $
                   new_data, $
                   linear, $
                   center_sector, $
                   l_setup , $
                   adjFlag, $
                   l_oa, $
                   orig_udf_tme, $ 
                   new_udf_tme, $
                   no_extra, $
                   l_setup.l_spin_ang_zero


   l_ChckDEBars,   new_data, $
                   steering

   SyncOpData, steering, new_udf_tme, l_setup

   l_CreateOpBars, steering, $
                   xsize, ximage, yimage, ypos, position, $
                   'Steering Cmd', l_pltpos.xsm_font
   ypos = ypos - ydelta


RETURN
END


;-------------------------------------------------------------------------------
;  Function : l_PwSupCmdProc
;
;  Description: Calls necessary procedures to generate PwSupCmdrument State
;               information
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_setup               struct        I        general setup parameters
;  l_pltpos              struct        I        general plotting parameters
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;      ERROR_TYPES     SUCCESS         int         G          successful completion
;                      FAILURE         int         G          unsuccessful completion
;                      WARNING         int         G          warning message
;                      INFO            int         G          informational message
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------

FUNCTION l_PwSupCmdProc, l_setup, l_pltpos
COMMON ERROR_TYPES, SUCCESS, FAILURE, WARNING, INFO

    status = l_PwSupCmdReadData (l_setup, l_psc, l_pltpos)
    if (status eq FAILURE) then return, FAILURE

    l_PwSupCmdDump, l_psc, l_setup, l_pltpos
    l_PwSupCmdTerm, l_psc

    RETURN, SUCCESS
end

