;============================================================
; Routine : plot_ImageMovie.pro
; Description: Creates the earth image/browse products
;
; Developer       Date     Build       Description
; ---------       ----     -----       -------------------------
; Mei-Ching Fok             0.0        original implementation
; Richard L. West (CSC)                extensive plotting modifications
;
;============================================================

pro plot_ImageMovie, image0, image1, image2, npolar, polar_angle, l_setup, $
                     nadir
;
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;
common rlw_plot_parm, char_size, char_size_1, $
                      window_width, window_height, $
                      x_inches, y_inches
common sm_coord, rs, rs2, e_rad, sm_im_x, sm_im_y, sm_im_z
common earth, npt, earth_x, earth_y
common re_circles, re_circles_x, re_circles_y, x_connect, y_connect, $
                   circle_pts, lt_lab
common dipole, dipole_x, dipole_y, dipole_pts
common auroral, auroral_x, auroral_y, auroral_pts
common color_table, black, white, magenta, nc_color, md_gray, $
                     color_min, color_max, del_color_minmax
;
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;
linear_log = l_setup.l_linear_log     ; linear = 1   log = 2
;
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;
x_min = -48.0
x_max = 48.0
l_clip = x_min
r_clip = x_max
if (l_setup.l_output_type eq 1) then begin
   y_min = -90
   y_max = 270
   b_clip = -90.0
   t_clip = 90.0
endif else if (l_setup.l_output_type eq 2) then begin
   y_min = -180
   y_max = 180
endif else if (l_setup.l_output_type eq 3) then begin
   y_min = x_min
   y_max = x_max
   b_clip = y_min
   t_clip = y_max
endif
;
;------------------------------------------------------------------------------
; get the minimum and maximin scaling values from the setup file or
; determine minimum and maximum scaling in log of units based on data values
;------------------------------------------------------------------------------
;
if (l_setup.l_color_min lt l_setup.l_color_max) then begin
   data_min = l_setup.l_color_min
   data_max = l_setup.l_color_max
   del_data_minmax = data_max - data_min
endif else begin
   if (l_setup.l_output_type eq 2) then begin
      temp = long((3.0 * e_rad) / 8.0)
      xbin1 = 5 - temp
      xbin2 = 6 + temp
      xbin1 = xbin1 > 0
      xbin2 = xbin2 < 11
      ybin1 = 23 - temp
      ybin2 = 22 + temp
      ybin1 = ybin1 > 0
      ybin2 = ybin2 < 44
   endif else begin
      xbin1 = 0
      ybin1 = 0
      size_info = size(image0)
      xbin2 = size_info(1) - 1
      ybin2 = size_info(2) - 1
   endelse
;   get the minimum value greater than zero from the 3 image arrays
   gt0 = where(image0(xbin1:xbin2,ybin1:ybin2) gt 0, gt0cnt)
   if (gt0cnt gt 0) then min0 = min(image0(gt0)) $
   else                  min0 = 0.01
   gt0 = where(image1(xbin1:xbin2,ybin1:ybin2) gt 0, gt0cnt)
   if (gt0cnt gt 0) then min1 = min(image1(gt0)) $
   else                  min1 = 0.01
   gt0 = where(image2(xbin1:xbin2,ybin1:ybin2) gt 0, gt0cnt)
   if (gt0cnt gt 0) then min2 = min(image2(gt0)) $
   else                  min2 = 0.01
   if (l_setup.l_image_id eq.1) then begin
      flx_min = min1
   endif else if (l_setup.l_image_id eq.2) then begin
      flx_min = min2
   endif else begin
      flx_min = min0
   endelse
   if (flx_min le 0.0) then flx_min = 0.01
;  get the maximum value from the 3 image arrays
   max0 = max(image0(xbin1:xbin2,ybin1:ybin2))
   max1 = max(image1(xbin1:xbin2,ybin1:ybin2))
   max2 = max(image2(xbin1:xbin2,ybin1:ybin2))
   if (l_setup.l_image_id eq.1) then begin
      flx_max = max1
   endif else if (l_setup.l_image_id eq.2) then begin
      flx_max = max2
   endif else begin
      flx_max = max0
   endelse
   data_min = alog10(flx_min)
   data_max = alog10(flx_max)
;  round off minimum and maximum values to nearest .5
   sign = 1.0
   if (data_min lt 0.0) then sign = -1.0
   temp = abs(data_min)
   temp1 = long(temp)
   temp2 = temp - temp1
   if (temp2 gt 0.5) then begin
      if (temp2 ge 0.75) then temp2 = 1.0 else temp2 = 0.5
   endif else begin
      if (temp2 ge 0.25) then temp2 = 0.5 else temp2 = 0.0
   endelse
   data_min = sign * (float(temp1) + temp2)

   sign = 1.0
   if (data_max lt 0.0) then sign = -1.0
   temp = abs(data_max)
   temp1 = long(temp)
   temp2 = temp - temp1
   if (temp2 gt 0.5) then begin
      if (temp2 ge 0.75) then temp2 = 1.0 else temp2 = 0.5
   endif else begin
      if (temp2 ge 0.25) then temp2 = 0.5 else temp2 = 0.0
   endelse
   data_max = sign * (float(temp1) + temp2)
   if (data_max le data_min) then data_max = data_min + 0.5
   if (linear_log eq 1) then begin
   endif else begin
   endelse

   del_data_minmax = data_max - data_min
endelse
;
;------------------------------------------------------------------------------
; load TIDE color table
;------------------------------------------------------------------------------
;
load_color_table
;
;------------------------------------------------------------------------------
; determine constants that are image display type dependent
; l_setup.l_output_type = 1  => 3 12x45 images
; l_setup.l_output_type = 2  => 3 images with earth-sc distance held constant
; l_setup.l_output_type = 3  => 3 12x12 images
;------------------------------------------------------------------------------
;
if (l_setup.l_output_type eq 1) then begin
   image_x_pos_start = 0.18
   image_y_pos_start = 0.05
   x_pos = image_x_pos_start
   y_pos = image_y_pos_start
   delx_image = 1.68 / x_inches   ; image size in normalized coord 48 degrees
   dely_image = 6.300 / y_inches   ; image size in normalized coord
   x_newimg_size = window_width * delx_image   ; image size in pixels
   y_newimg_size = window_height * dely_image   ; image size in pixels
   x_newimg_size = round(x_newimg_size)
   y_newimg_size = round(y_newimg_size)
   image_x_space = delx_image + 0.05
   bar_x = (image_x_pos_start +delx_image) + 0.10
   left_axis_label_x = x_pos - 0.14
   bottom_axis_label_y = y_pos - 0.045
   no_counts_bar_y = 0.11
   no_counts_y = 0.015
   no_data_bar_y = 0.16
   no_bar_y = 0.015
   mass_label_y = 0.015
   y_tklng_fctr = 0.015
   x_tklng_fctr = 0.075
endif else begin
   image_x_pos_start = 0.13
   image_y_pos_start = 0.11
   x_pos = image_x_pos_start
   y_pos = image_y_pos_start
   delx_image = 2.8 / x_inches   ; image size in normalized coord
   dely_image = 2.8 / y_inches   ; image size in normalized coord
   x_newimg_size = window_width * delx_image   ; image size in pixels
   y_newimg_size = window_height * dely_image   ; image size in pixels
   x_newimg_size = round(x_newimg_size)
   y_newimg_size = round(y_newimg_size)
   image_x_space = delx_image + 0.04
   image_y_space = dely_image + 0.04
   bar_x = (image_x_pos_start +delx_image) + 0.10
   left_axis_label_x = x_pos - 0.10
   bottom_axis_label_y = y_pos - 0.09
   no_counts_bar_y = 0.32
   no_counts_y = 0.030
   no_data_bar_y = 0.425
   no_bar_y = 0.030
   mass_label_y = 0.03
   y_tklng_fctr = 0.03
   x_tklng_fctr = 0.05
endelse
;    
;------------------------------------------------------------------------------
; put on color bar
;------------------------------------------------------------------------------
;
temp_cb = bytarr (2, 221)
for i = 0, 220 do begin
   j = i + 5
   temp_cb (0,i) = j
   temp_cb(1,i) = j
endfor
bar_x_size = 15
bar_y_size = 200
colorbar=congrid(temp_cb, bar_x_size, bar_y_size)
bar_w = float(bar_x_size) / float(window_width)
bar_h = float(bar_y_size) / float(window_height)
bar_y = 0.50 - (bar_h / 2.0) + 0.10
tv, colorbar, bar_x, bar_y, xsize=bar_w, ysize=bar_h, /normal
box_x = [bar_x, bar_x+bar_w, bar_x+bar_w, bar_x, bar_x]
box_y = [bar_y, bar_y, bar_y+bar_h, bar_y+bar_h, bar_y]
plot, [0.0,1.0,1.0,0.0,0.0], [0.0,0.0,1.0,1.0,0.0], xstyle=4, ystyle=4, $
      /noerase, color=0
oplot, box_x, box_y, /noclip
; label color bar
if (linear_log eq 1) then begin
   xyouts, bar_x-0.02, bar_y+(bar_h/2.0), 'linear counts/sec', $
           alignment=0.5, charsize=char_size, color=black, $
           /normal, orientation = 90
endif else begin
   xyouts, bar_x-0.02, bar_y+(bar_h/2.0), 'log counts/sec', $
           alignment=0.5, charsize=char_size, color=black, $
           /normal, orientation = 90
endelse
; tick marks & tick label
tck_x1 = bar_x + bar_w
tck_x2 = tck_x1 + 0.01
str_x = tck_x2 + 0.005
if (linear_log eq 1) then begin
   num_ticks = (data_max - data_min) / 0.20
   dely_tick = bar_h / num_ticks
   for iticks = 0,num_ticks do begin
      tick_value = data_min + (iticks * 0.20)
      tick_y = bar_y + (iticks * dely_tick)
      oplot, [tck_x1, tck_x2], [tick_y, tick_y], /noclip
      xyouts, str_x, tick_y-0.005, string(tick_value,'(f4.1)'), $
              alignment=0.0, charsize=char_size, color=black, $
              /normal
   endfor
endif else begin
   num_ticks = (data_max - data_min) / 0.50
   dely_tick = bar_h / num_ticks
   for iticks = 0,num_ticks do begin
      tick_value = data_min + (iticks * 0.50)
      tick_y = bar_y + (iticks * dely_tick)
      oplot, [tck_x1, tck_x2], [tick_y, tick_y], /noclip
      xyouts, str_x, tick_y-0.005, string(tick_value,'(f4.1)'), $
              alignment=0.0, charsize=char_size, color=black, $
              /normal
   endfor
endelse
;    
;------------------------------------------------------------------------------
; no counts bar, no data bar, and labels
;------------------------------------------------------------------------------
;
nocounts = bytarr(15,15)
nocounts(*,*) = nc_color
bar_w = 15 / float(window_width)
bar_h = 15 / float(window_height)
bar_y = (0.50 - (bar_h / 2.0)) - no_counts_bar_y
tv, nocounts, bar_x, bar_y, xsize=bar_w, ysize=bar_h, /normal
box_x = [bar_x, bar_x+bar_w, bar_x+bar_w, bar_x, bar_x]
box_y = [bar_y, bar_y, bar_y+bar_h, bar_y+bar_h, bar_y]
oplot, box_x, box_y, /noclip
xyouts, bar_x+.02, bar_y-no_counts_y, 'no counts', $
        alignment=0.5, charsize=char_size, /normal

nodata = bytarr(15,15)
nodata(*,*) = md_gray
bar_w = 15 / float(window_width)
bar_h = 15 / float(window_height)
bar_y = (0.50 - (bar_h / 2.0)) - no_data_bar_y
tv, nodata, bar_x, bar_y, xsize=bar_w, ysize=bar_h, /normal
box_x = [bar_x, bar_x+bar_w, bar_x+bar_w, bar_x, bar_x]
box_y = [bar_y, bar_y, bar_y+bar_h, bar_y+bar_h, bar_y]
oplot, box_x, box_y, /noclip
xyouts, bar_x+.02, bar_y-no_bar_y, 'no data', $
        alignment=0.5, charsize=char_size, /normal
;
;------------------------------------------------------------------------------
; determine which images to plot
; 0 = H, 1 = O, 2 = H + O
;------------------------------------------------------------------------------
;
if (l_setup.l_image_id eq 1) then begin
   start_image = 1
   stop_image = 1
endif else if (l_setup.l_image_id eq 2) then begin
   start_image = 2
   stop_image = 2
endif else begin
   start_image = 0
   stop_image = 0
endelse
;
;------------------------------------------------------------------------------
; loop thru the images
;------------------------------------------------------------------------------
;
;for image_num = start_image,stop_image do begin
;
;------------------------------------------------------------------------------
; take the log of the image and calculate image placement
;------------------------------------------------------------------------------
;

;if (image_num eq 0) then begin
if (l_setup.l_image_id eq 0) then begin
   mass_label = 'H'
   original_image = image0
   log_image = image0
;endif else if (image_num eq 1) then begin
endif else if (l_setup.l_image_id eq 1) then begin
   mass_label = 'O'
   original_image = image1
   log_image = image1
endif else begin
   mass_label = 'All Masses'
   original_image = image2
   log_image = image2
endelse
no_data = where (original_image lt 0.0, no_data_counts)
no_counts = where (original_image eq 0.0, no_counts_counts)
have_data = where (original_image gt 0.0, have_data_count)
have_data_in_spin = 1
if ((no_counts_counts eq 0) and (have_data_count eq 0)) then $
   have_data_in_spin = 0
if (have_data_count gt 0) then begin
   if (linear_log eq 1) then begin
      log_image[have_data] = original_image[have_data]
   endif else begin
      log_image[have_data] = alog10(original_image[have_data])
   endelse
endif
;------------------------------------------------------------------
; scale the image data to the color range
;------------------------------------------------------------------
new_image = log_image
temp = log_image

if (have_data_count gt 0) then begin
   temp[have_data] = (new_image[have_data] - data_min) / del_data_minmax
   new_image[have_data] = round (temp[have_data] * del_color_minmax)
   new_image[have_data] = new_image[have_data] + color_min
   lt_dmin = where (log_image lt data_min, lt_dmin_count)
   if (lt_dmin_count gt 0) then new_image(lt_dmin) = color_min
   gt_dmax = where (log_image gt data_max, gt_dmax_count)
   if (gt_dmax_count gt 0) then new_image(gt_dmax) = color_max
endif

if (no_counts_counts gt 0) then $
   new_image[no_counts] = nc_color     ; give no counts this color

if (no_data_counts gt 0) then $
   new_image[no_data] = md_gray     ; give no data this color
;
;------------------------------------------------------------------------------
; Re-set x_max, x_min, y_max and y_min when l_setup.l_output_type=2
;   the e_rad changes the size of the earth, 
;   at 90.0 it makes the earth smaller
;------------------------------------------------------------------------------
;

if (l_setup.l_output_type eq 2) then begin
   new_image = congrid(new_image, 960, 3600)   ; create one degree bins
; save old plotting min,max values in degrees
   x_min_old = x_min
   x_max_old = x_max
   y_min_old = y_min
   y_max_old = y_max
; earth diameter = factor * image size
; calculate new plotting min,max values in degrees based on earth size
   x_max = 3.0 * e_rad
   x_min     = -x_max
   y_max     = x_max
   y_min     = x_min
; define clipping values when plotting earth and field lines
   l_clip = x_min
   r_clip = x_max
   b_clip = y_min
   t_clip = y_max
; determine the size of the plot image array
   temp = round(x_max * 10)
   plot_image_x = 2 * temp
   plot_image_y = 2 * temp
; determine bin ranges to extract from data image (+/-48, +/-180)
   bin_xmin = round((x_min + 48.0) * 10)
   bin_xmax = round((x_max + 48.0) * 10)
   bin_ymin = 1840 + round(y_min * 10.0)
   bin_ymax = 1839 + round(y_max * 10.0)
   bin_xmin = bin_xmin > 0
   bin_xmax = bin_xmax < 959
   bin_ymin = bin_ymin > 0
   bin_ymax = bin_ymax < 3599
   nx_bins = (bin_xmax - bin_xmin) + 1
   ny_bins = (bin_ymax - bin_ymin) + 1
; fill plotting array with extracted data array
   if (x_min lt -48.0) then begin   ; plotting array larger than data array
      plot_image = bytarr(plot_image_x, plot_image_y)
      plot_image(*,*)=black                  ; black background
      start_bin = long((abs(x_min) - 48.0) * 10.0) + 1
      stop_bin = start_bin + (nx_bins -1)
      plot_image(start_bin:stop_bin, 0:plot_image_x-1) = $
                               new_image(bin_xmin:bin_xmax,bin_ymin:bin_ymax)
   endif else begin     ; plotting array equal or smaller than data array
      plot_image = bytarr(nx_bins, ny_bins)
      plot_image = new_image(bin_xmin:bin_xmax,bin_ymin:bin_ymax)
   endelse
; congrid plotting to proper plotting size
   x_degrees = (x_max - x_min)
   y_degrees = (y_max - y_min)
   new_x_size = (x_degrees / (x_max-x_min)) * x_newimg_size
   new_y_size = (y_degrees / (y_max-y_min)) * y_newimg_size
   plot_image = congrid(plot_image, x_newimg_size, y_newimg_size)
; smooth the image, put back in set values
   mask = plot_image
   plot_image = smooth(plot_image, 11, /EDGE_TRUNCATE)
   nc_mask = where(mask eq nc_color, nc_mask_cnts)
   if (nc_mask_cnts gt 0) then plot_image(nc_mask) = mask(nc_mask)
   nd_mask = where(mask eq md_gray, nd_mask_cnts)
   if (nd_mask_cnts gt 0) then plot_image(nd_mask) = mask(nd_mask)
   bl_mask = where(mask eq black, bl_mask_cnts)
   if (bl_mask_cnts gt 0) then plot_image(bl_mask) = mask(bl_mask)
; plot the image
   tv, plot_image, x_pos, y_pos, $
       xsize=delx_image, ysize=dely_image, $
       /normal
   str_x = x_pos + (delx_image / 2.0)
   str_y = (y_pos + dely_image) + mass_label_y
   xyouts, str_x, str_y, mass_label, $
           charsize = char_size, $
           align = 0.5, /normal
endif else begin  ; l_setup.l_output_type = 1,3
   new_image_1 = new_image
   new_image_1_min = min(new_image_1) > color_min
   le0 = where(original_image le 0.0, le0_counts)
   if (le0_counts gt 0) then new_image_1(le0) = new_image_1_min
   c_image=congrid(new_image,x_newimg_size,y_newimg_size)
   c_image = smooth(c_image, 11, /EDGE_TRUNCATE)
   mask=congrid(new_image,x_newimg_size,y_newimg_size)
   nc_mask = where(mask eq nc_color, nc_mask_cnts)
   if (nc_mask_cnts gt 0) then c_image(nc_mask) = mask(nc_mask)
   nd_mask = where(mask eq md_gray, nd_mask_cnts)
   if (nd_mask_cnts gt 0) then c_image(nd_mask) = mask(nd_mask)
   tv, c_image, x_pos, y_pos, $
       xsize=delx_image, ysize=dely_image, $
       /normal
   str_x = x_pos + (delx_image / 2.0)
   str_y = (y_pos + dely_image) + mass_label_y
   xyouts, str_x, str_y, mass_label, $
           charsize = char_size, $
           align = 0.5, /normal
endelse
;
;------------------------------------------------------------------------------
; scale the image size to 0 - 1
;------------------------------------------------------------------------------
;
x0 = x_pos
y0 = y_pos
x1 = x_pos + delx_image
y1 = y_pos + dely_image
;
;------------------------------------------------------------------------------
; add tickmarks to image
;------------------------------------------------------------------------------
;
if (nadir eq 1) then begin
   ; draw box around image
   plot, [x_min, x_max, x_max, x_min, x_min], $
          [y_min, y_min, y_max, y_max, y_min], $
         position=[x0,y0,x1,y1], $
         xrange=[x_min,x_max], yrange=[y_min,y_max], $
         xstyle=5, ystyle=5, $
          /noclip, /noerase
   ; draw top x tickmarks
   deltck = (x_max - x_min) / 4.0
   y_tklng = (y_max - y_min) * y_tklng_fctr
   for itk = 0,4 do begin
      tkx = x_min + (itk * deltck)
      tky1 = y_max
      tky2 = y_max + (y_tklng / 2.0)
      oplot, [tkx, tkx], [tky1, tky2], $
             /noclip
   endfor
   ; draw and label bottom x tickmarks
   deltck = (x_max - x_min) / 4.0
   for itk = 0,4 do begin
      tkx = x_min + (itk * deltck)
      tky1 = y_min
      tky2 = y_min - y_tklng
      oplot, [tkx, tkx], [tky1, tky2], /noclip
      stry = (tky2 - y_tklng) - 1.5
      if (l_setup.l_output_type eq 3) then begin
         label = string(format = '(f5.1)', tkx)
         xyouts, tkx, stry, label, charsize = char_size, align = 0.5
      endif else if(l_setup.l_output_type eq 2) then begin
         label = string(format = '(f6.1)', tkx)
         xyouts, tkx, stry, label, charsize = char_size, align = 0.5
      endif else if (l_setup.l_output_type eq 1) then begin
         if ((itk mod 2) eq 0) then begin
            label = string(format = '(f5.1)', tkx)
            xyouts, tkx, stry, label, charsize = char_size, align = 0.5
         endif
      endif
   endfor
   str_x = x_pos + (delx_image / 2.0)
   str_y = bottom_axis_label_y
   xyouts, str_x, str_y, 'degrees', charsize = char_size, align = 0.5, $
           /normal
   ; draw and label left y tickmarks
   deltck = (y_max - y_min) / 4.0
   x_tklng = (x_max - x_min) * x_tklng_fctr
   for itk = 0,4 do begin
      tky = y_min + (itk * deltck)
      tkx1 = x_min
      tkx2 = x_min - x_tklng
      oplot, [tkx1, tkx2], [tky, tky], /noclip
      if (l_setup.l_output_type eq 1) then begin
         label = string(format = '(f5.1)', tky)
      endif else if (l_setup.l_output_type eq 2) then begin
         label = string(format = '(f6.1)', tky)
      endif else if (l_setup.l_output_type eq 3) then begin
         label = string(format = '(f5.1)', tky)
      endif
      xyouts, tkx2, tky-1.5, label, charsize = char_size, align = 1.0
   endfor
   str_x = left_axis_label_x
   str_y = y_pos + (dely_image / 2.0)
   xyouts, str_x, str_y, 'degrees', charsize = char_size, $
           align = 0.5, orientation = 90, /normal
   ; draw right y tickmarks
   deltck = (y_max - y_min) / 4.0
   for itk = 0,4 do begin
      tky = y_min + (itk * deltck)
      tkx1 = x_max
      tkx2 = x_max + (x_tklng / 2.0)
      oplot, [tkx1, tkx2], [tky, tky], $
             /noclip
   endfor
endif
;
;------------------------------------------------------------------------------
; if there is data for this spin put on the earth etc.
;------------------------------------------------------------------------------
;
if (have_data_in_spin eq 1) then begin

   ;----- draw the earth

   if (nadir eq 1) then oplot, earth_x, earth_y, color = white

   ;----- draw circles at 3 and 6.6 Re at the equator
   ;----- start drawing from opposite LT

   for ii1 = 0,1 do begin
      for ii2 = 0,1 do begin
         n_pts = circle_pts(ii1,ii2)
         oplot, re_circles_x(ii1,ii2,0:n_pts), re_circles_y(ii1,ii2,0:n_pts), $
                color=white, clip=[l_clip, b_clip, r_clip, t_clip]
      endfor
   endfor
   ;  draw connections between circles and label them
   for i=0,14,2 do begin
      if (abs(y_connect(i)) le 90.0 and abs(y_connect(i+1)) le 90.0) then begin
         ; connection ./. L=3.3 & 6
         oplot, x_connect(i:i+1), y_connect(i:i+1), color=white
         labx=1.1*x_connect(i+1)
         laby=1.1*y_connect(i+1)
         if (labx ge x_min and labx le x_max) then  $
            if (laby ge y_min and laby le y_max) then  $
               xyouts, labx, laby, lt_lab(i+1), color=white, charsize=char_size
      endif
   endfor

   ;----- draw dipole fieldlines at L=3, 6.6, MLT=0, 6, 12, 18

   for ii1=0,1 do begin
      for jj2=0,3 do begin
         for nn1=0,1 do begin
           n_pts = dipole_pts(ii1,jj2,nn1)
           if (n_pts gt 0) then begin
              oplot, dipole_x(ii1,jj2,nn1,0:n_pts), $
                     dipole_y(ii1,jj2,nn1,0:n_pts), $
                     color=white
           endif
         endfor
      endfor
   endfor

   ;----- draw the -75, -65, 65 and 75 deg latitudes on the surface
   ;----- This section of stuff draws the auroral lines.
   ;----- start drawing from opp. LT

   for ii1 = 0,3 do begin
      n_pts = auroral_pts(ii1)
      if (n_pts gt 0) then oplot, auroral_x(ii1,0:n_pts), $
                                  auroral_y(ii1,0:n_pts), $
                                  color=magenta
   endfor   ; for i=1,4 do begin

   ;----- Add the continents when nadir = 1

   if (nadir eq 1) then $
      draw_continents, x1, x0, x_max, x_min, y1, y0, y_max, y_min, white

endif   ; have_data_in_spin

;
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;

;endfor     ; image_num = 0,2

;
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;

return
end
