
;-------------------------------------------------------------------------------
;
;  Unit Name : l_PerigeeRtns.pro
;
;  Purpose   : File contains routines to read, write, and plot perigees.
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;   ELH               11/00           v1.0            Original Implementation
;   ELH               05/01           v1.0            Updated read routines to return
;                                                     status value
;                                                       Added l_AdjPlotSize
;
;
;  File Revision Number:   %I%
;  File Last Modified  :   %E%  %T%
;
;-------------------------------------------------------------------------------

;-------------------------------------------------------------------------------
;  Procedure: l_PerigeeFind 
;
;  Description: Find the perigee times.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_oa                  struct *      I        OA data parameters
;  perigee_times         double[]      O        perigee times
;  flag                  string        I        determine if regular or LTT plots
;  ndays                 long          I        total number of perigeen times 
;                                               between specified time period
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;-------------------------------------------------------------------------------

PRO l_PerigeeFind, l_oa, perigee_times, flag, ndays

   nspins = n_elements ((*l_oa.gsm_x_pos_ptr)[*])
   if (flag ne 'ltt') then begin
       NUMRECS = 5L
   endif else begin
       NUMRECS = ndays * 3L
   endelse

   perigee = dblarr (NUMRECS) 
 
   p_idx = 0L
   a_idx = 0L

   ;------------------------------------------------
   ; convert the gsm coordinates to sm
   ;------------------------------------------------

   dist = fltarr (nspins)

   for i = 0L, nspins-1L do begin

      RECALC, (*l_oa.startUdfTimePtr)[i].year, (*l_oa.startUdfTimePtr)[i].doy, $
              (*l_oa.startUdfTimePtr)[i].hour, (*l_oa.startUdfTimePtr)[i].min, $
              (*l_oa.startUdfTimePtr)[i].sec

      SMGSM, xsm, ysm, zsm, $
             (*l_oa.gsm_x_pos_ptr)[i], $
             (*l_oa.gsm_y_pos_ptr)[i], $
             (*l_oa.gsm_z_pos_ptr)[i], $
             -1

       dist[i] = sqrt(xsm*xsm + ysm*ysm + zsm*zsm)

   endfor

   for i = 1L, nspins-2L do begin
      if (dist[i] lt dist[i-1]) and (dist[i] lt dist[i+1]) then begin
         perigee[p_idx] = (*l_oa.tme_ptr)[i] 
         p_idx = p_idx + 1L
      endif
   endfor  


   if (p_idx eq 0) then begin
      print, 'No perigees.'
      return
   endif

   perigee_times = perigee[0:p_idx-1L]
      

RETURN
END

;-------------------------------------------------------------------------------
;  Function : l_PerigeeProc
;
;  Description:
;     Calls the neccessary routines to read the data to create
;  the perigee plots.
;
;  Return Value:  type = int
;  Value                         Description
;  -------------------------     ------------------------------
;  SUCCESS                       successful completion
;  FAILURE                       unsuccessful completion
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_setup               struct        I        generic setup parameters
;  l_setup_file          string        I        setup file name
;  l_pltpos              struct        I        generic plot parameters
;  l_rgb_table           struct        I        generic color table parameters
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;     PLOTDEFS         MAX_PLOTS        int         G         max # plots per page
;                      MAX_WEB_SPINS    int         G         max # spins per plot
;                                                             when using web interface
;                      NO_CNTS          double      G         no counts indicator
;                      NO_DATA          double      G         no data indicator
;
;      ERROR_TYPES     SUCCESS         int         G          successful completion
;                      FAILURE         int         G          unsuccessful completion
;                      WARNING         int         G          warning message
;                      INFO            int         G          informational message
;
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;-------------------------------------------------------------------------------

FUNCTION l_PerigeeProc, l_setup, l_setup_file, l_pltpos, $
l_rgb_table
COMMON PLOTDEFS, MAX_PLOTS, MAX_WEB_SPINS, NO_CNTS, NO_DATA
COMMON ERROR_TYPES, SUCCESS, FAILURE, WARNING, INFO

   hr_n_secs     = 3600. 
   start_perigee = 0.0D
   stop_perigee  = 0.0D

   l_pltpos.plot_hdr_title = '!5Perigee'
   l_pltpos.X_axis_eq_time = 'Y'

   ;--------------------------------------------------------
   ; .setup buffer
   ; .read the oa data
   ; .find the perigee times 
   ; .for the number of perigee times do
   ; .add and subtract one hour to the time
   ; .reconvert the start/stop time back to UDF time
   ; .check the user start and stop time 
   ; .read the singles and binned data
   ; .plot the perigee
   ; .terminate 
   ;--------------------------------------------------------

   year      = strmid(systime(0), 20, 23)
   l_setup.l_line_plots = 'N'

   ;--------------------------------------------------------
   ; MUST BE A 24 HOUR PERIOD
   ;--------------------------------------------------------
   start_time = [l_setup.l_start_dtime[0], $
                 l_setup.l_start_dtime[1], $
                 0, 0, 0]

   stop_time  = [l_setup.l_start_dtime[0], $
                 l_setup.l_start_dtime[1], $
                 23, 59, 59]

   status = l_OARead (l_oa, $
             start_time, $
             stop_time, $
             l_pltpos)
   if (status eq FAILURE) then return, FAILURE

   start_ndays = l_ConvYMD2DOY (l_setup.l_start_date[0], l_setup.l_start_date[1], l_setup.l_start_date[2])
   stop_ndays = l_ConvYMD2DOY (l_setup.l_stop_date[0], l_setup.l_stop_date[1], l_setup.l_stop_date[2])
   ndays = stop_ndays - start_ndays 

   l_PerigeeFind, l_oa, $
                  perigee_times, $ 
                  'day', ndays

   l_SnglsReadSetup, l_sngls_setup, l_setup_file

   l_CoincReadSetup, l_coinc_setup, l_setup_file

   l_BinnedReadSetup, l_binned_setup, l_setup_file

   l_OATerm, l_oa

   n_perigees = n_elements (perigee_times)


   nloops = 1L
   if (l_setup.l_postscript eq 'Y') then nloops = 2L

   for i = 0L, n_perigees-1L do begin

      status = 0L
      start_perigee = perigee_times[i] - hr_n_secs
      stop_perigee  = perigee_times[i] + hr_n_secs

      l_setup.l_start_dtime = ConvSec2Arr_CalDOY (start_perigee, $
                                  l_setup.l_start_date[0])
      l_setup.l_stop_dtime  = ConvSec2Arr_CalDOY (stop_perigee, $
                                  l_setup.l_start_date[0])

      status = l_OARead (l_oa, $
                         l_setup.l_start_dtime, $
                         l_setup.l_stop_dtime, $
                         l_pltpos)
      if (status eq FAILURE) then return, FAILURE

      status = l_HskpGetData ( l_setup, $
                     l_setup_file, $
                     l_pltpos, $
                     l_hskp_setup, $
                     l_hskp)
      if (status eq FAILURE) then return, FAILURE


      if (l_sngls_setup.start eq 'Y') OR (l_sngls_setup.stop eq 'Y') then begin
         status = l_SnglsGetData ( l_setup, $
                                   l_setup_file, $
                                   l_pltpos, $
                                   l_sngls_setup, $
                                   l_sngls)
         if (status eq FAILURE) then return, FAILURE
       

         l_SnglsAdjExData, l_sngls_setup, l_setup.l_num_spinsec, l_sngls

         l_SnglsHndlMissData, l_sngls_setup, l_sngls, l_setup, l_oa

      endif

      status = l_BinnedEventSelect (l_setup, l_setup_file, l_pltpos, $
                                    l_roi0, l_binned, l_binned_setup, $
                                    l_event, l_event_setup, l_memd, l_memd_setup, $
                                    l_evsp, l_evsp_setup, l_hskp, l_oa, davail)
      if (status eq FAILURE) then return, FAILURE


      if (status ge 0L) then begin
      ;----------------------------------------
      ; generate a spectrogram plot
      ;----------------------------------------
      for j = 0L, nloops-1L do begin
          nplots = 0L
          nrows = 0L
          npages = 0L
          nrows = l_sngls_setup.nplots 

          if (l_binned_setup.use_binned eq 'Y') then begin
             nrows = nrows + l_binned_setup.nplots
          endif

          if (l_event_setup.use_event eq 'Y') then begin
              nrows = nrows + l_event_setup.nplots
          endif

          npages  = CEIL(nrows/ MAX_PLOTS)

          if (nrows - MAX_PLOTS gt 0L) then npages = npages + 1L
          if (npages eq 0L) then npages = 1L
          if (npages gt 0L) then windx = intarr (npages)

          nplots = nrows

          l_setup.l_filedesc = 'per'

          plotted = 0L
          pages   = 0L
          web     = l_setup.l_via_web


          l_AdjPlotSize, l_pltpos, l_setup, nplots

          if (j eq 1) and (l_setup.l_postscript eq 'Y') then begin
             l_pltpos.ps = 'Y'
             l_GenFileName, l_setup, '0'
             l_pltpos.filename = l_setup.l_prod_dst + '/' + $
                                 l_setup.l_filename + '.ps'
          endif

          l_pltpos.linePlots = l_setup.l_line_plots

          l_SetUpZBuffer, l_pltpos, $
                          l_setup, $
                          l_rgb_table, $
                          colorbar, $
                          nplots

          if (l_setup.l_pwrsup_bars eq 'Y') then begin
            l_DrawOpBars, l_hskp, l_setup, l_pltpos, l_rgb_table, l_oa
          endif


          if (l_sngls_setup.start eq 'Y') OR (l_sngls_setup.stop eq 'Y') then begin
          l_SnglsPlot, l_sngls_setup, $
                        l_sngls, $
                        l_pltpos, $
                        nrows, $
                        windx, $
                        pages, $
                        plotted,$
                        l_setup, $
                        l_oa, $
                        l_rgb_table, $
                        colorbar, $
                        l_hskp
         endif


         if (l_binned_setup.use_binned eq 'Y') OR (l_event_setup.use_event eq 'Y') then begin

              l_BinnedEventPlot, l_setup, $
                                 l_binned_setup, $
                                 l_event_setup, $
                                 l_binned, $
                                 l_event, $
                                 l_pltpos, $
                                 l_roi0, $
                                 l_oa, $
                                 l_hskp, $
                                 l_rgb_table, $
                                 nrows, windx, plotted, $
                                 pages, colorbar
          endif


          if (j eq 1L) and (l_setup.l_postscript eq 'Y') then begin
             l_pltpos.ps = 'N'
             device, /close_file
          endif

          if (!D.NAME eq 'Z') then begin
            if (l_setup.l_via_web eq 'Y') or (l_setup.l_gen_sum eq 'Y') then begin
               device, /close
            endif
          endif

      endfor ;---- nloops ----
    endif

    if (l_sngls_setup.start eq 'Y') OR (l_sngls_setup.stop eq 'Y') then l_SnglsTerm, l_sngls
    if (l_binned_setup.use_binned eq 'Y') OR (l_event_setup.use_event eq 'Y') then begin
        l_BinnedEventTerm, l_roi0, l_binned, l_binned_setup, $
                           l_event, l_event_setup, l_memd, $
                            l_evsp, l_hskp, l_oa, l_setup, davail
     endif

    l_OATerm, l_oa

   endfor


   RETURN, SUCCESS

END

