
;-------------------------------------------------------------------------------
;
;  Unit Name : l_MemdRtns.pro
;
;  Purpose   : File containing routines to read, write, and plot the memory
;              dump data.
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;   ELH               04/01           v1.0            Original Implementation
;
;
;  File Revision Number:   %I%
;  File Last Modified  :   %E%  %T%
;
;-------------------------------------------------------------------------------

;-------------------------------------------------------------------------------
;  Function: l_MemdGetHskpFlag
;
;  Description:
;    Reads the temperature 07 from the Normal Housekeeping packet.  This flag
;    indicates whether or not the new packets: ROI summation, sectored events/sun
;    pulse and HK monitor are included in the memory dump package 
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  SUCCESS                       success completion
;  FAILURE                       failure to complete
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_memd                struct *      I        memory dump data parameters
;  l_setup               struct        I        generic setup parameters
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;-------------------------------------------------------------------------------

FUNCTION l_MemdGetHskpFlag, l_memd, l_setup

   ;----------------------------------------
   ; define the data to retrieve from UDF
   ;----------------------------------------
   inst    = 'LENA'
   expr    = 'LENAHSKP'
   vinst   = 'IMLHSKP'

   status = 0L
   numRecs = 0L
   data_source = ' '


   ;----------------------------------------
   ; pass 1, determine the number of records
   ; to read between user specified time
   ;----------------------------------------
   numRecs = l_UdfGetNumRec (l_setup.l_start_dtime, $
                             l_setup.l_stop_dtime,  $
                             inst, expr, vinst, data_source)


   if (numRecs eq -1) or (numRecs eq 0) then RETURN, -1

   memd_flag = fltarr (numRecs)

   ;----------------------------------------
   ; read the data from the UDF DB
   ;----------------------------------------
   fh = l_UdfGetDataHandle (inst, expr, vinst, $
                            l_setup.l_start_dtime, $
                            l_setup.l_stop_dtime, data_source, /GROUP)

   idx = 0L 
   while not udf_eof (fh) do begin
       d = udf_read (fh)
       memd_flag [idx] = d.temperature_monitors[7] 
       idx = idx + 1L
   endwhile

   udf_close, fh

   ptr = ptr_new (memd_flag)
   l_memd.flagPtr = ptr

   RETURN, status

END



;-------------------------------------------------------------------------------
;  Function : l_MemdReadData
;
;  Description:
;    Read the memory dump records. 
;
;  Return Value:  type = int 
;  Value                         Description
;  -------------------------     ------------------------------
;  SUCCESS                       successful completion 
;  FAILURE                       unsuccessful completion 
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  btime                 long[]        I        UDF begin time [yr,doy,hh,mm,ss]
;  etime                 long[]        I        UDF end time [yr,doy,hh,mm,ss]
;  numRecs               long          I        total number of records
;  l_memd                struct       I/O       memory dump data parameters
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;      ERROR_TYPES     SUCCESS         int         G          successful completion
;                      FAILURE         int         G          unsuccessful completion
;                      WARNING         int         G          warning message
;                      INFO            int         G          inform
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;-------------------------------------------------------------------------------

FUNCTION l_MemdReadData, btime, etime, numRecs, l_memd
COMMON ERROR_TYPES, SUCCESS, FAILURE, WARNING, INFO


   ;-------------------------------------------------
   ; define data type to read
   ;-------------------------------------------------
   inst    = 'LENA'
   expr    = 'LENAHSKP'
   vinst   = 'IMLMEMD'
   data_source = ''


   ;-------------------------------------------------
   ; allocate space for the necessary structures
   ;-------------------------------------------------
   data   = fltarr (256, numRecs)
   xferBytes = fltarr (numRecs)
   dumpLength = fltarr (numRecs)

   memory_address = fltarr (numRecs)
   total_records = fltarr (numRecs)
   dump_seq = fltarr (numRecs)


   sctime = dblarr (numRecs)
   start_array = replicate ({UDF_TIMES_STRUCT}, numRecs)
   stop_array  = replicate ({UDF_TIMES_STRUCT}, numRecs)

 

   ;-------------------------------------------------
   ; read the data from the UDF DB
   ;-------------------------------------------------
   fh = l_UdfGetDataHandle (inst, expr, vinst, btime, etime, data_source, /GROUP)
   if (fh eq -1) then begin
      udf_close, fh
      RETURN, FAILURE
   endif

   indx = 0L

   while (not udf_eof (fh)) do begin

       d = udf_read (fh)

       sctime[indx] = l_GetTime (d.btime, doy, hr, min, sec)
       start_array[indx].year = d.btime.year
       start_array[indx].doy  = d.btime.doy
       start_array[indx].hour = d.btime.hour
       start_array[indx].min  = d.btime.min
       start_array[indx].sec  = d.btime.sec
       start_array[indx].msec = d.btime.msec

       stop_array[indx].year = d.etime.year
       stop_array[indx].doy  = d.etime.doy
       stop_array[indx].hour = d.etime.hour
       stop_array[indx].min  = d.etime.min
       stop_array[indx].sec  = d.etime.sec
       stop_array[indx].msec = d.etime.msec

       ;------ memory dump data ------
       data[*,indx] = d.memory_dump[*]
  
       ;------ transfer byte/ byte count ------
       xferBytes[indx] = d.transfer_mode[0]

       ;------ dump length of memory dump data ------ 
       dumpLength[indx] = d.dump_length[0]

       ;------ the total number of buffers containing one record ------ 
       total_records[indx] = d.total_records_in_mem_dump[0]

       ;------ the buffer sequence number ------
       dump_seq[indx] = d.dump_sequence_number[0]
       

       indx = indx + 1L

   endwhile

   udf_close, fh


   ptr = ptr_new (data)
   l_memd.dataPtr = ptr

   ptr = ptr_new (xferBytes)
   l_memd.xferModePtr = ptr

   ptr = ptr_new (dumpLength)
   l_memd.dmpLenPtr = ptr

   ptr = ptr_new (total_records)
   l_memd.totRecPtr = ptr

   ptr = ptr_new (dump_seq)
   l_memd.seqNumPtr = ptr

   ptr = ptr_new (sctime)
   l_memd.timePtr = ptr

   ptr = ptr_new (start_array)
   l_memd.startUdfTimePtr = ptr

   ptr = ptr_new (stop_array)
   l_memd.stopUdfTimePtr = ptr


   RETURN, SUCCESS

END

;-------------------------------------------------------------------------------
;  Procedure: l_MemdParseData
;
;  Description:
;     Separate the ROI summation, sectored events, HK monitor packets from
;     the memory dump.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_memd                struct       I/O       memory dump data parameters
;  numRecs               long          I        total number of records
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;-------------------------------------------------------------------------------

PRO l_MemdParseData, l_memd, numRecs

    ntime  = dblarr (numRecs)
    data   = fltarr (256, numRecs)
    roi    = fltarr (20, numRecs)
    se     = fltarr (45, numRecs)
    sunp   = fltarr (45, numRecs)
    hk     = fltarr (41, numRecs)
    totRec = fltarr (numRecs)
    seqNum = fltarr (numRecs)
    xfer   = fltarr (numRecs)
    dmpLen = fltarr (numRecs)
    start_array = replicate ({UDF_TIMES_STRUCT}, numRecs)
    stop_array  = replicate ({UDF_TIMES_STRUCT}, numRecs)

    indx = 0L

    for i = 0L, numRecs - 1L do begin
      if ((*l_memd.totRecPtr)[i] eq 2) then begin
         if ((*l_memd.seqNumPtr)[i] eq 1) then begin
            ntime[indx]    = (*l_memd.timePtr)[i]
           
            start_array[indx] = (*l_memd.startUdfTimePtr)[i]
            stop_array[indx]  = (*l_memd.stopUdfTimePtr)[i]

            ;--------------------------------------------------- 
            ;  memory dump data
            ;--------------------------------------------------- 
            data[*,indx]   = (*l_memd.dataPtr)[*, i]

            ;--------------------------------------------------- 
            ;  ROI summation data
            ;--------------------------------------------------- 
            roi[0:19,indx] = (*l_memd.dataPtr)[0:19, i]

            ;--------------------------------------------------- 
            ;   sectored events data  
            ;     data stored as a single byte, but actual
            ;     sectored event data is 2 bytes
            ;--------------------------------------------------- 

            new_j = 0L
            j = 20L
            for new_j = 0L, 44L do begin
               tmp1 = fix((*l_memd.dataPtr)[j,i])
               j = j + 1L
               tmp2 = fix((*l_memd.dataPtr)[j,i])
               tmp3 = ishft(tmp1, 8)
               tmp4 = tmp3 or tmp2
               se [new_j, indx] = tmp4 
               j = j + 1L
            endfor

            ;--------------------------------------------------- 
            ; calculate the sun pulse sector 
            ;--------------------------------------------------- 
            tmp1       = fix((*l_memd.dataPtr)[110,i])
            tmp2       = fix((*l_memd.dataPtr)[111,i])
            tmp3       = ishft (tmp1, 8)
            tmp4       = tmp3 or tmp2
            sp         = tmp4 + 1350
            sp1        = sp mod 3600
            sp2        = sp1/3600.
            sun_pulse  = sp2 * 45
            sunp[*, indx] = sun_pulse[*]

            ;--------------------------------------------------- 
            ; HK Monitor
            ;--------------------------------------------------- 
            hk [0:40,indx] = (*l_memd.dataPtr)[112:152, i]

            ;--------------------------------------------------- 
            ; buffer data 
            ;--------------------------------------------------- 
            totRec[indx]   = (*l_memd.totRecPtr)[i]
            seqNum[indx]   = (*l_memd.seqNumPtr)[i]
            xfer[indx]     = (*l_memd.xferModePtr)[i]
            dmpLen[indx]   = (*l_memd.dmpLenPtr)[i]
            indx = indx + 1L
         endif
      endif
    endfor

    ptr = ptr_new (ntime[0:indx])
    l_memd.timePtr = ptr

    ptr = ptr_new (start_array[0:indx])
    l_memd.startUdfTimePtr = ptr

    ptr = ptr_new (stop_array[0:indx])
    l_memd.stopUdfTimePtr = ptr

    ptr = ptr_new (data[*,0:indx])
    l_memd.dataPtr = ptr

    ptr = ptr_new (roi[*,0:indx])
    l_memd.roiSumPtr = ptr

    ptr = ptr_new (se[*,0:indx])
    l_memd.eventCntsPtr = ptr

    ptr = ptr_new (hk[*,0:indx])
    l_memd.hkMonPtr = ptr

    ptr = ptr_new (totRec[0:indx])
    l_memd.totRecPtr = ptr

    ptr = ptr_new (seqNum[0:indx])
    l_memd.seqNumPtr = ptr

    ptr = ptr_new (xfer[0:indx])
    l_memd.xferModePtr = ptr

    ptr = ptr_new (sunp[*, 0:indx])
    l_memd.sunPulsePtr = ptr

    ptr = ptr_new (dmpLen[0:indx])
    l_memd.dmpLenPtr = ptr


    return
END


;-------------------------------------------------------------------------------
;  Function: l_MemdGetData
;
;  Description:
;    Calls the necessary routines to read and parse the memory dump data
;
;  Return Value:  type = int
;  Value                         Description
;  -------------------------     ------------------------------
;  SUCCESS                       successful completion
;  FAILURE                       failure to complete
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_setup               struct        I        generic setup parameters
;  l_pltpos              struct        I        generic plot parameters
;  l_memd                struct *     I/O       memory dump data parameters
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;     PLOTDEFS         MAX_PLOTS        int         G         max # plots per page
;                      MAX_WEB_SPINS    int         G         max # spins per plot
;                                                             when using web interface
;                      NO_CNTS          double      G         no counts indicator
;                      NO_DATA          double      G         no data indicator
;
;      ERROR_TYPES     SUCCESS         int         G          successful completion
;                      FAILURE         int         G          unsuccessful completion
;                      WARNING         int         G          warning message
;                      INFO            int         G          informational message
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;-------------------------------------------------------------------------------

FUNCTION l_MemdGetData, l_setup, l_pltpos, l_memd 
COMMON PLOTDEFS, MAX_PLOTS, MAX_WEB_SPINS, NO_CNTS, NO_DATA
COMMON ERROR_TYPES, SUCCESS, FAILURE, WARNING, INFO


   data_source = ' '
   numRecs     = 0L


   ptr= ptr_new()
   l_memd = {             $
      timePtr           :ptr,  $
      startUdfTimePtr   :ptr,  $
      stopUdfTimePtr    :ptr,  $
      dataPtr           :ptr,  $
      xferModePtr       :ptr,  $
      dmpLenPtr         :ptr,  $
      totRecPtr         :ptr,  $
      seqNumPtr         :ptr,  $
      flagPtr           :ptr,  $
      roiSumPtr         :ptr,  $
      eventCntsPtr      :ptr,  $
      hkMonPtr          :ptr,  $
      sunPulsePtr       :ptr   $
   }


   status = l_MemdGetHskpFlag (l_memd, l_setup)


   inst    = 'LENA'
   expr    = 'LENAHSKP'
   vinst   = 'IMLMEMD'

   numRecs = l_UdfGetNumRec (l_setup.l_start_dtime, $
                             l_setup.l_stop_dtime,  $
                             inst, expr, vinst, data_source)


   if (numRecs le 0) then RETURN, FAILURE 

   l_pltpos.data_source = data_source


   status = l_MemdReadData (l_setup.l_start_dtime, $
                            l_setup.l_stop_dtime,  $
                            numRecs, l_memd)

   if (status eq FAILURE) then return, FAILURE

   l_MemdParseData, l_memd, numRecs


   RETURN, status 

END

;-------------------------------------------------------------------------------
;  Procedure: l_MemdTerm
;
;  Description:
;   Free allocated pointers
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_memd                struct *      I        memory dump parameters
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;-------------------------------------------------------------------------------

PRO l_MemdTerm, l_memd


   if (ptr_valid (l_memd.timePtr)) then begin
       ptr_free, l_memd.timePtr
   endif

   if (ptr_valid (l_memd.startUdfTimePtr)) then begin
       ptr_free, l_memd.startUdfTimePtr
   endif

   if (ptr_valid (l_memd.stopUdfTimePtr)) then begin
       ptr_free, l_memd.stopUdfTimePtr
   endif

   if (ptr_valid (l_memd.dataPtr)) then begin
       ptr_free, l_memd.dataPtr
   endif

   if (ptr_valid (l_memd.xferModePtr)) then begin
       ptr_free, l_memd.xferModePtr
   endif

   if (ptr_valid (l_memd.totRecPtr)) then begin
       ptr_free, l_memd.totRecPtr
   endif

   if (ptr_valid (l_memd.seqNumPtr)) then begin
       ptr_free, l_memd.seqNumPtr
   endif

   if (ptr_valid (l_memd.dmpLenPtr)) then begin
       ptr_free, l_memd.dmpLenPtr
   endif

   if (ptr_valid (l_memd.flagPtr)) then begin
       ptr_free, l_memd.flagPtr
   endif

   if (ptr_valid (l_memd.roiSumPtr)) then begin
       ptr_free, l_memd.roiSumPtr
   endif

   if (ptr_valid (l_memd.eventCntsPtr)) then begin
       ptr_free, l_memd.eventCntsPtr
   endif

   if (ptr_valid (l_memd.hkMonPtr)) then begin
       ptr_free, l_memd.hkMonPtr
   endif

   if (ptr_valid (l_memd.sunPulsePtr)) then begin
       ptr_free, l_memd.sunPulsePtr
   endif


RETURN
END

;-------------------------------------------------------------------------------
;  Procedure: l_MemdProc
;
;  Description: Main
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_setup               struct        I        generic setup parameter
;  l_pltpos              struct        I        generic plot parameter
;  l_rgb_table           struct        I        generic color table parameter 
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;-------------------------------------------------------------------------------


PRO l_MemdProc, l_setup, l_pltpos, l_rgb_table

   status = l_MemdGetData (l_setup, l_pltpos, l_memd)
   print, 'data = ', (*l_memd.dataPtr)[*, 0]
   print, 'xfer bytes = ', (*l_memd.xferModePtr)[0]
   print, 'dump length = ', (*l_memd.dmpLenPtr)[0]

RETURN
END
