
;-------------------------------------------------------------------------------
;
;  Unit Name : l_LttRtns.pro
;
;  Purpose   : Generates plots for long term trending of data.
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;   ELH              04/22/03        v1.0.11         Original implementation
;
;
;  File Revision Number:   %I%
;  File Last Modified  :   %E%  %T%
;
;-------------------------------------------------------------------------------

;-------------------------------------------------------------------------------
;  Function/Procedure: l_LttReadSetup
;
;  Description: Reads the setup file for long term trending specific processing
;               and plot options
;
;  Return Value:  type = int
;  Value                         Description
;  -------------------------     ------------------------------
;  SUCCESS                       Succesful completion
;  FAILURE                       Unsuccessful completion
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;   err_msg              string        O        error message 
;   l_plot_setup         struct        O        plot parameters
;   l_setup_file         string        I        setup file name
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;     ERROR_TYPES      SUCCESS          int         G         successful completion
;                      FAILURE          int         G         failure to complete
;                      WARNING          int         G         warning of problem
;                      INFO             int         G         informational message
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH              04/22/03         v1.0.11         Original implementation
;-------------------------------------------------------------------------------


FUNCTION l_LttReadSetup, err_msg, l_plot_setup, l_setup_file
COMMON ERROR_TYPES, SUCCESS, FAILURE, WARNING, INFO

   ;----------------------------------------------------------
   ; initialize plot setup parameters to default values
   ;----------------------------------------------------------

   l_plot_setup = {          $
      l_sngls_start        : 'N', $       ; use singles start data
      l_sngls_stop         : 'N', $       ; use singles stop data
      l_sngls_y_axis_measurement : 'A', $ ; A = daily average, I = integral counts 
      l_binned             : 'N', $       ; use binned data
      l_event              : 'N', $       ; use event data
      l_hskp_mcp_hv_start  : 'N', $       ; use HSKP HV MCP start
      l_hskp_mcp_hv_stop   : 'N', $       ; use HSKP HV MCP stop
      l_hskp_cfd_start     : 'N', $       ; use HSKP CFD start
      l_hskp_cfd_stop      : 'N', $       ; use HSKP CFD stop
      bk_correction        : 'N', $       ; back background correction
      sun_threshold        : 1L , $       ; +/- N sectors of sun sector
      azi_begin            : 0L , $       ; begin sector range
      azi_end              : 44L, $       ; end sector range
      azi_threshold        : 2L,  $       ; allow sectors whose azi is within this number of bins
      plr_begin            : 0L,  $       ; begin polar range
      plr_end              : 11L, $       ; end polar range
      plr_threshold        : 2L,  $       ; allow sectors whose polar bin is within 
                                          ; this number of bins
      eng_begin            : 0L,  $       ; begin energy range
      eng_end              : 2L,  $       ; end energy range
      eng_threshold        : 2L,  $       ; allow sectors whose energy bin is within 
                                          ; this number of bins
      tof                  : 'H', $       ; TOF range, Hydrogen or Oxygen
      time_filter          : ' ', $       ; filter data w/in apogee or perigee
      trace_filter         : 'S', $       ; filter of different traces, i.e. Sun  
      nplots               : 0L , $       ; number of plots to be plotted
      TokarDisplay         : 'N', $       ; LTT as a Tokar display 
      y_func               : 'H', $       ; Tokar y-axis, by hour or by spin
      time_range           : 2,   $       ; +/- # of hours between apogee/perigee 
      sngls_Y_axis_min     : '',  $       ; min Y-axis, blank = data min 
      sngls_Y_axis_max     : ''  $        ; max Y-axis, blank = data max

   }


   ;----------------------------------------------------------
   ; open the long term trending runtime setup file
   ;----------------------------------------------------------
   error_msg = ' '

   openr, inlun, /get_lun, l_setup_file, ERROR = err
   if (err NE 0) then begin
      err_msg = 'Failure to open l_ltt.setup file.' 
      close, inlun
      free_lun, inlun
      RETURN, FAILURE
   endif


   ;----------------------------------------------------------
   ; read the parameters in the setup file
   ;----------------------------------------------------------
   line = ' '

   while not EOF (inlun) do begin
     readf, inlun, line
     if (strmid(line, 0, 1) ne ':') and (strlen(line) gt 0) then begin
        parts = str_sep (line, ' ')

        if (n_elements (parts) gt 2) then begin 
        case parts[0] of 
          'l_singles_start'     : begin
                                      l_plot_setup.l_sngls_start = STRUPCASE(parts[2])
                                      if (l_plot_setup.l_sngls_start eq 'Y') then begin
                                         l_plot_setup.nplots = l_plot_setup.nplots + 1L
                                      endif
                                    end
          'l_singles_stop'      : begin
                                      l_plot_setup.l_sngls_stop = STRUPCASE(parts[2])
                                      if (l_plot_setup.l_sngls_stop eq 'Y') then begin
                                         l_plot_setup.nplots = l_plot_setup.nplots + 1L
                                      endif
                                  end
          'l_singles_y_axis_measurement' : begin
                                      l_plot_setup.l_sngls_y_axis_measurement= STRUPCASE(parts[2])
                                  end
          'l_binned'            : begin
                                      l_plot_setup.l_binned = STRUPCASE(parts[2])
                                      if (l_plot_setup.l_binned eq 'Y') then begin
                                         l_plot_setup.nplots = l_plot_setup.nplots + 1L
                                      endif
                                  end
          'l_event'             : begin
                                      l_plot_setup.l_event = STRUPCASE(parts[2])
                                      if (l_plot_setup.l_event eq 'Y') then begin
                                         l_plot_setup.nplots = l_plot_setup.nplots + 1L
                                      endif
                                  end

          'l_hskp_mcp_hv_start'  : begin
                                      l_plot_setup.l_hskp_mcp_hv_start = STRUPCASE(parts[2])
                                      if (l_plot_setup.l_hskp_mcp_hv_start eq 'Y') then begin
                                         l_plot_setup.nplots = l_plot_setup.nplots + 1L
                                      endif
                                  end
          'l_hskp_mcp_hv_stop'  : begin
                                      l_plot_setup.l_hskp_mcp_hv_stop = STRUPCASE(parts[2])
                                      if (l_plot_setup.l_hskp_mcp_hv_stop eq 'Y') then begin
                                         l_plot_setup.nplots = l_plot_setup.nplots + 1L
                                      endif
                                  end
          'l_hskp_cfd_start'    : begin
                                      l_plot_setup.l_hskp_cfd_start = STRUPCASE(parts[2])
                                      if (l_plot_setup.l_hskp_cfd_start eq 'Y') then begin
                                         l_plot_setup.nplots = l_plot_setup.nplots + 1L
                                      endif
                                  end
          'l_hskp_cfd_stop'    : begin
                                      l_plot_setup.l_hskp_cfd_stop = STRUPCASE(parts[2])
                                      if (l_plot_setup.l_hskp_cfd_stop eq 'Y') then begin
                                         l_plot_setup.nplots = l_plot_setup.nplots + 1L
                                      endif
                                  end
          'background_correction' : begin
                                      l_plot_setup.bk_correction = STRUPCASE(parts[2])
                                    end
          'azimuthal_begin'       : begin
                                      l_plot_setup.azi_begin = STRUPCASE(parts[2])
                                    end
          'azimuthal_end'         : begin
                                      l_plot_setup.azi_end = STRUPCASE(parts[2])
                                    end
          'azimuthal_threshold'   : begin
                                      l_plot_setup.azi_threshold= STRUPCASE(parts[2])
                                    end
          'polar_begin'           : begin
                                      l_plot_setup.plr_begin = STRUPCASE(parts[2])
                                    end
          'polar_end'             : begin
                                      l_plot_setup.plr_end = STRUPCASE(parts[2])
                                    end
          'polar_threshold'       : begin
                                      l_plot_setup.plr_threshold= STRUPCASE(parts[2])
                                    end
          'energy_begin'          : begin 
                                      l_plot_setup.eng_begin = STRUPCASE(parts[2])
                                    end
          'energy_end'            : begin
                                      l_plot_setup.eng_end = STRUPCASE(parts[2])
                                    end
          'energy_threshold'      : begin
                                      l_plot_setup.eng_threshold = STRUPCASE(parts[2])
                                    end
          'tof'                   : begin
                                      l_plot_setup.tof = STRUPCASE(parts[2])
                                    end
          'time_filter'                : begin
                                      l_plot_setup.time_filter = STRUPCASE(parts[2])
                                    end
          'trace_filter'          : begin
                                      l_plot_setup.trace_filter = STRUPCASE(parts[2])
                                    end
          'sun_threshold'         : begin 
                                      l_plot_setup.sun_threshold = STRUPCASE(parts[2])
                                    end
          'TokarDisplay'          : begin
                                      l_plot_setup.TokarDisplay= STRUPCASE(parts[2])
                                    end
          'tokar_y_func'          : begin
                                      l_plot_setup.y_func= STRUPCASE(parts[2])
                                    end
          'time_range'            : begin
                                      l_plot_setup.time_range= STRUPCASE(parts[2])
                                    end
          'l_sngls_Y_axis_min'    : begin
                                      l_plot_setup.sngls_Y_axis_min = STRUPCASE(parts[2])
                                    end
          'l_sngls_Y_axis_max'    : begin
                                      l_plot_setup.sngls_Y_axis_max = STRUPCASE(parts[2])
                                    end
          else   : begin
                   end
        endcase
        endif
     endif

   endwhile
    
   ;----------------------------------------------------------
   ; close the setup file
   ;----------------------------------------------------------
   line = ' '
   close, inlun
   free_lun, inlun

   ;----------------------------------------------------------
   ; set the default coincidences to used binned if user
   ; accidentally selects to use both event and binned data
   ;----------------------------------------------------------
   if (l_plot_setup.l_event eq 'Y') AND (l_plot_setup.l_binned eq 'Y') then begin
      l_plot_setup.l_event = 'N'
      l_plot_setup.nplots = l_plot_setup.nplots - 1L
      err_msg = 'Either Event OR Binned may be selected.  Defaulting to use Binned data.'
      RETURN, INFO 
   endif


   RETURN, SUCCESS

END


;-------------------------------------------------------------------------------
;  Procedure: l_DetTicks4LinePlots
;  Description:  Determine the number and location of major and minor tick
;            marks for the long term trending line plots
;
;  Return Value:  type = None
;  Value                         Description
;  -------------------------     ------------------------------
;   None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  start_year             int          I        user specified beginning of year
;  stop_year              int          I        user specified ending of year
;  start_doy              int          I        user specified beginning day of year
;  stop_doy               int          I        user specified ending day of year
;  act_start_doy          int          I        data start day of year
;  act_stop_doy           int          I        data stop day of year
;  l_pltpos               struct       I        plot parameters
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;     EPOCH_DATA       EPOCH_YR         int         G         launch year 
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH              04/22/03         v1.0.11          Original implementation
;-------------------------------------------------------------------------------

PRO l_DetTicks4LinePlots, start_year, stop_year, start_doy, stop_doy, $
act_start_doy, act_stop_doy, l_pltpos
COMMON EPOCH_DATA, EPOCH_YR

    MAX_TICKS = 12L

    lyear = { LEAP_YR_STRUCT, $
        flag : 0L, $
        year : 0L, $
        days : 0L  $
    }

    diff_year = stop_year - start_year + 1L
    if (start_year eq stop_year) then diff_year = 1L
    leap_year = replicate ({LEAP_YR_STRUCT}, diff_year)
    for ii = 0L, diff_year - 1L do begin
       leap_year[ii].flag = l_LeapYear (EPOCH_YR+ii, yr_n_secs, n_days)
       leap_year[ii].year = EPOCH_YR+ii
       leap_year[ii].days = n_days
    endfor
    
    l_pltpos.min_time_axis_len = start_doy 
    l_pltpos.max_time_axis_len = stop_doy 

    start_label = 0L
    delta = 90L
    mn_tick_delta = 30L
    mn_ticks = 2L
    nticks = (diff_year * 4) + 1L 
    diff = act_stop_doy - act_start_doy + 1L


    if (diff lt delta) then begin

       ;----------------------------------------------------
       ; if less than 90 days
       ;----------------------------------------------------
       nontime_deltas = [1, 2, 5, 10, 15, 20, 30, 100, 1000]
       num_ntds = n_elements (nontime_deltas)

       mtest = -1
       for i = 0, num_ntds - 1 do begin
          if (i eq 7) or (i eq 8) then begin
             for j = 1, 9 do begin
                 nt_delta = diff/nontime_deltas[i] * j
                 if (mtest lt 0) AND (nt_delta lt MAX_TICKS) then begin
                    mtest = i
                 endif
             endfor
          endif else begin
             nt_delta = diff/nontime_deltas[i]
             if (mtest lt 0) AND (nt_delta lt MAX_TICKS) then begin
                 mtest = i
             endif
          endelse
       endfor

      delta = nontime_deltas[mtest]
      start_label = start_doy
      nticks = (diff/delta) + 1L 
      maj_xtick_loc = dblarr (nticks)
      xtick_labels = strarr (nticks)

      label = 0L
      for i = 0L, nticks - 1L do begin
         label = start_label + i * delta
         xtick_labels[i] = strcompress (string (label), /remove_all)
         maj_xtick_loc[i] = i * 24. * delta
      endfor

      XaxisLen = diff * 24.
      l_pltpos.min_time_axis_len = 0
      l_pltpos.max_time_axis_len = XaxisLen 

      tmin = start_label
      tmax = start_label + delta 

      min_time_deltas = [1, 5, 10, 15, 20, 30, 60, 120, 180, 240]
      nmtime_deltas = n_elements (min_time_deltas)

      if (delta gt 1439) then begin
         min_diff = fix(tmax - tmin)
         case min_diff of
            5:  mn_tick = 5
            10:  mn_tick = 5
            30:  mn_tick = 3
            180:  mn_tick = 3
            360:  mn_tick = 3
            1080:  mn_tick = 3
            else:  mn_tick = 4
         endcase
      endif else begin

         MN_TICKS = 6
         mn_diff = tmax - tmin
         mtest = -1
         for i = 0, nmtime_deltas - 1 do begin
             mn_delta = mn_diff/min_time_deltas[i]
             if (mtest lt 0) AND (mn_delta lt MN_TICKS) then begin
                mtest = i
             endif
         endfor

         mn_delta = min_time_deltas[mtest]
         mn_tick = mn_diff/mn_delta

      endelse

         mn_tick_delta = ((tmax - tmin)/mn_tick) * 24L

    endif else begin

       ;----------------------------------------------------
       ; if more than 90 days
       ;----------------------------------------------------

       maj_xtick_loc = dblarr (nticks)
       xtick_labels = strarr (nticks)

       curr_yr = start_year - EPOCH_YR
       left_over_days = 0L

       label = 0L
       new_doy = 360L

       for i = 0L, nticks - 1L do begin
   
           if (label ge new_doy) then begin
              left_over_days = left_over_days + (leap_year[curr_yr].days - 360)
              curr_yr = curr_yr + 1L
              new_doy = 360L * (curr_yr + 1L)
           endif 

           label = i * delta + left_over_days

           maj_xtick_loc[i] = label
           xtick_labels[i] = strcompress (string (label), /remove_all)
       endfor

   endelse

    l_pltpos.xtick_labels = xtick_labels
    l_pltpos.xtick_locs = maj_xtick_loc
    l_pltpos.mn_tick_delta = mn_tick_delta
    l_pltpos.xtick_delta = delta
    l_pltpos.nticks = nticks
    l_pltpos.xmn_tick = mn_ticks

RETURN
END

;-------------------------------------------------------------------------------
;  Procedure: l_LttTokarPlotByHr
;
;  Description: Generates the tokar y-axis on an hourly basis
;            does one day at a time, this routine does avg of 14 hours separately
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  Y                     long[]        O        Y-axis values
;  tokar_data            float[]       O        tokar data
;  event_data            float[]       I        event data
;  event_time            double[]      I        event times
;  start_time            long[]        I        user start time
;  stop_time             long[]        I        user stop time
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------

PRO l_LttTokarPlotByHr, Y, tokar_data, event_data, event_time, start_time, stop_time

   ;-------------------------------
   ; one hour in minutes, the event
   ; time is in minutes, not seconds
   ;-------------------------------
   one_hour = 3600.
   local_start_time = event_time[0] 
   local_stop_time = event_time[n_elements(event_time)-1L]
   sub_stop_time = local_start_time + one_hour


   ntimes = n_elements (event_time)

   indx = 0L
   cnt = 0L
   tmp = 0L


   prev_event_time = event_time[0]
   for i = 0L, ntimes - 1L do begin
      if (event_time[i] ge local_start_time) AND (event_time[i] le sub_stop_time) then begin
         tmp = tmp + event_data[i]
         if (prev_event_time ne event_time[i]) then cnt = cnt + 1L
      endif else begin
         if (event_time[i] gt sub_stop_time) then begin
            local_start_time = sub_stop_time
            sub_stop_time = local_start_time + one_hour
            Y[indx] = indx + 1L
            tokar_data[indx] = tmp/cnt
            cnt = 0L
            tmp = 0L
            tmp = tmp + event_data[i]
            cnt = cnt + 1L
            indx = indx + 1L
         endif
       endelse
       prev_event_time = event_time[i]
   endfor

   RETURN

END


;-------------------------------------------------------------------------------
;  Procedure: l_LttTokarPlotBySpin
;
;  Description:  Generates the tokar y-axis on a spin basis
;      does one day at a time, this routine does avg of 14 hours separately
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  Y                     long[]        O        Y-axis values
;  tokar_data            float[]       O        tokar data
;  event_data            float[]       I        event data
;  event_time            double[]      I        event times
;  start_time            long[]        I        user start time
;  stop_time             long[]        I        user stop time
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------

PRO l_LttTokarPlotBySpin, Y, tokar_data, event_data, event_time, start_time, stop_time

   ;-------------------------------
   ; one hour in minutes, the event
   ; time is in minutes, not seconds
   ;-------------------------------
   one_hour = 3600.
   local_start_time = event_time[0] 
   local_stop_time = event_time[n_elements(event_time)-1L]
   sub_stop_time = local_start_time + one_hour


   ntimes = n_elements (event_time)

   indx = 0L
   cnt = 0L
   tmp = 0L


   prev_event_time = event_time[0]
   for i = 0L, ntimes - 1L do begin
      if (event_time[i] ge local_start_time) AND (event_time[i] le sub_stop_time) then begin
         tmp = tmp + event_data[i]
         if (prev_event_time ne event_time[i]) then cnt = cnt + 1L
      endif else begin
         if (event_time[i] gt sub_stop_time) then begin
            local_start_time = sub_stop_time
            sub_stop_time = local_start_time + one_hour
            Y[indx] = indx + 1L
            tokar_data[indx] = tmp/cnt
            cnt = 0L
            tmp = 0L
            tmp = tmp + event_data[i]
            cnt = cnt + 1L
            indx = indx + 1L
         endif
       endelse
       prev_event_time = event_time[i]
   endfor

   RETURN

END


;-------------------------------------------------------------------------------
;  Procedure: l_LttSyncMarkers2Event
;
;  Description: Sync the trace (sun, earth, etc..) times to the event times. 
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_event               struct *      I        pointers to event data structures
;  hmd_tme               double[]      I        data gap filled times
;  markers               long []       I        trace sectors 
;  synced_tme            double []     O        synced times
;  synced_markers        long []       O        synced traces 
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------

PRO l_LttSyncMarkers2Event, l_event, hmd_tme, markers, $
                            synced_tme, synced_markers


   n_event  = n_elements ((*l_event.filledStartUdfTimePtr))
   n_marker = n_elements (hmd_tme)

   synced_tme = fltarr (n_elements (hmd_tme))
   synced_markers = fltarr (n_elements (markers))

   last_ii = 0L
   last_jj = 0L
   indx    = 0L
   status  = 0L
   ii      = 0L
   jj      = 0L

   for ii = 0L, n_event - 1L do begin

       start_event_time = Convert((*l_event.filledStartUdfTimePtr)[ii])
       ;stop_event_time = Convert((*l_event.filledStopUdfTimePtr)[ii])
       stop_event_time = start_event_time + 120. 

       for jj = last_jj, n_marker - 1L do begin

          start_oa_time = hmd_tme[jj]
          stop_oa_time = start_oa_time + 120.
          next_start_oa_time = stop_oa_time

          ;----------------------------------------------------------
          ; if the start event time falls between the OA spin period
          ;----------------------------------------------------------
          if (start_event_time ge start_oa_time) and $
             (start_event_time lt stop_oa_time) then begin
               sector = (*l_event.azimuthPtr)[ii]
               ;-----------------------------------------------------
               ; sun_pulse is in degrees, need to convert to sector
               ;-----------------------------------------------------
               synced_markers[ii] = markers[jj]
               synced_tme[ii] = hmd_tme[jj]
               last_jj = jj
               goto, FSW_NEXT_PKT1
          endif else begin
               ;----------------------------------------------------------
               ; if the stop event time falls between the OA spin period
               ;----------------------------------------------------------
               if ((stop_event_time ge start_oa_time) AND $
                   (stop_event_time lt stop_oa_time)) then begin
                   sector = (*l_event.azimuthPtr)[ii]
                   ;-----------------------------------------------------
                   ; sun_pulse is in degrees, need to convert to sector
                   ;-----------------------------------------------------
                   synced_markers[ii] = markers[jj]
                   synced_tme[ii] = hmd_tme[jj]
                   last_jj = jj
                   goto, FSW_NEXT_PKT1
               endif
          endelse
          ;----------------------------------------------------------
          ; if event times do not fall in between a start/stop OA
          ; period, then no OA data corresponds that the event
          ; array filled with -9 for no data
          ;----------------------------------------------------------

       endfor
FSW_NEXT_PKT1:
   endfor


   synced_markers = synced_markers[0:ii-1L]
   synced_tme     = synced_tme[0:ii-1L]

RETURN
END


;-------------------------------------------------------------------------------
;  Procedure: l_LttReformatSunData
;
;  Description: Reformat the sun sector data previously synced to the event data and
;          store the data on a spin/sector basis instead of on an event basis 
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_event               struct *      I        pointers to event structures
;  sunp                  float [,]     I        sunc sectors, 2 dim
;  reformatted_sunp      float []      O        reformatted sun sector array
;                                               one dim 
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------

PRO l_LttReformatSunData, l_event, sunp, reformatted_sunp


   spin      = 0L
   curr_sect = 0L
   prev_sect = 0L

   time_nele             = n_elements((*l_event.orig_startUdfTimePtr)[*])
   estart_udf_time       = (*l_event.orig_startUdfTimePtr)[0]
   estop_udf_time        = (*l_event.orig_startUdfTimePtr)[time_nele-1L]
   est_start_event_time  = Convert (estart_udf_time)
   est_stop_event_time   = Convert (estop_udf_time)
   est_nspins            = round((est_stop_event_time - est_start_event_time) / 120.) + 1L

   prev_event_time = (*l_event.timePtr)[0]

   nele = n_elements (*l_event.timePtr)

   reformatted_sunp = fltarr (est_nspins) 

   for i = 0L, nele - 1L do begin

      start_event_time = (*l_event.timePtr)[i]
      curr_sect = (*l_event.azimuthPtr)[i]

      reformatted_sunp[spin] = sunp[i]
      if (curr_sect lt prev_sect) AND (start_event_time ge prev_event_time) then begin
         spin = spin + 1L
      endif

      prev_sect = curr_sect
      prev_event_time = start_event_time

   endfor

   reformatted_sunp = reformatted_sunp[0:spin-1L]


RETURN
END

;-------------------------------------------------------------------------------
;  Procedure: l_LttReadSnglsData
;
;  Description: Singles start or stop data
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_setup               struct        I        general setup parameters
;  l_setup_file          string        I        run time setup file name
;  l_pltpos              struct        I        general plotting parameters
;  l_sngls_setup         struct        I        singles processing & plotting parameters
;  l_sngls               struct *      I        pointers to singles data structures
;  l_ltt_setup           struct        I        LTT procesing & plotting parameters
;  days_in_yr_struct     long []       I        the number of days in a year
;  sngls_start           float []      O        singles start data
;  sngls_stop            float []      O        singles stop data
;  sngls_x_axis_day      long []       O        x-axis values 
;  ndays                 long          I        total number of days between specified
;                                               start/stop times
;  indx                  long          I        index into sngls start/stop arrays
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;      ERROR_TYPES     SUCCESS         int         G          successful completion
;                      FAILURE         int         G          unsuccessful completion
;                      WARNING         int         G          warning message
;                      INFO            int         G          informational message
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------

PRO l_LttReadSnglsData, l_setup, l_setup_file, l_pltpos, l_sngls_setup, l_sngls, $
l_ltt_setup, days_in_yr_struct, sngls_start, sngls_stop, sngls_x_axis_day, ndays, indx
COMMON ERROR_TYPES, SUCCESS, FAILURE, WARNING, INFO


           status = SUCCESS
         
;          print, 'singles start_dtime = ', l_setup.l_start_dtime
;          print, 'singles stop_dtime = ', l_setup.l_stop_dtime

           status = l_SnglsGetData ( l_setup, $
                                     l_setup_file, $
                                     l_pltpos, $
                                     l_sngls_setup, $
                                     l_sngls)

           if (status ne FAILURE) then begin

              if (l_ltt_setup.l_sngls_start eq 'Y') then begin

                 sumrow = 0L
                 sumidx = 0L
                 sidxcnt = 0L
                 sumrow = total ((*l_sngls.startPtr))
                 nitems = n_elements ((*l_sngls.startPtr))/l_setup.l_num_spinsec

                 sngls_start[indx] = total(sumrow)
                 if (l_ltt_setup.l_sngls_y_axis_measurement eq 'I') then begin
                 ;-------------------------------------------------
                 ; integral counts of singles data
                 ;-------------------------------------------------
                    if (indx gt 0) then begin
                       sngls_start[indx] = sngls_start[indx-1] + sngls_start[indx]
                    endif 
                 endif else begin
                 ;-------------------------------------------------
                 ; daily average of singles data
                 ;-------------------------------------------------
                    sngls_start[indx] = sngls_start[indx]/nitems
                 endelse
                

              endif

              if (l_ltt_setup.l_sngls_stop eq 'Y') then begin

                 sumrow = 0L
                 sumidx = 0L
                 sidxcnt = 0L
                 sumrow = total ((*l_sngls.stopPtr))
                 nitems = n_elements ((*l_sngls.stopPtr))/l_setup.l_num_spinsec 

                 sngls_stop[indx] = total(sumrow)
                 if (l_ltt_setup.l_sngls_y_axis_measurement eq 'I') then begin
                 ;-------------------------------------------------
                 ; integral counts of singles data
                 ;-------------------------------------------------
                    if (indx gt 0) then begin
                       sngls_stop[indx] = sngls_stop[indx-1] + sngls_stop[indx]
                    endif
                 endif else begin
                 ;-------------------------------------------------
                 ; daily average of singles data
                 ;-------------------------------------------------
                    sngls_stop[indx] = sngls_stop[indx]/nitems
                 endelse

              endif

           sngls_x_axis_day[indx] = ndays
           indx = indx + 1L

          endif


       l_SnglsTerm, l_sngls


RETURN
END

;-------------------------------------------------------------------------------
;  Procedure: l_LttReadHskpData
;
;  Description: Read the HSKP data.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_setup               struct        I        general setup parameters
;  l_setup_file          string        I        run time setup filename
;  l_pltpos              struct        I        general plotting parameters
;  l_hskp_setup          struct        I        HSKP processing & plotting parameters
;  l_hskp                struct *      I        pointers to HSKP data structures
;  l_ltt_setup           struct        I        LTT processing and plotting parameters
;  days_in_yr_struct     long []       I        the number of days in a year
;  hskp_mcp_hv_start     float[]       O        HSKP MCP high voltage start
;  hskp_mcp_hv_stop      float[]       O        HSKP MCP high voltage stop
;  hskp_cfd_start        float[]       O        HSKP CFD start
;  hskp_cfd_stop         float[]       O        HSKP CFD stop
;  hskp_x_axis_day       float[]       O        HSKP x-axis values 
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;      ERROR_TYPES     SUCCESS         int         G          successful completion
;                      FAILURE         int         G          unsuccessful completion
;                      WARNING         int         G          warning message
;                      INFO            int         G          informational message
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------

PRO l_LttReadHskpData, l_setup, l_setup_file, l_pltpos, l_hskp_setup, l_hskp, $
l_ltt_setup, days_in_yr_struct, hskp_mcp_hv_start, hskp_mcp_hv_stop, hskp_cfd_start, $
hskp_cfd_stop, hskp_x_axis_day
COMMON ERROR_TYPES, SUCCESS, FAILURE, WARNING, INFO

;       print, 'HSKP start_dtime = ', l_setup.l_start_dtime
;       print, 'HSKP stop_dtime = ', l_setup.l_stop_dtime

        status = SUCCESS
        ;-------------------------------------------------------
        ; convert the given year, month, day input start/stop
        ; time into day of year to determine the number of days
        ; to process
        ;-------------------------------------------------------
        start_doy = l_ConvYMD2DOY (l_setup.l_start_date[0], l_setup.l_start_date[1], $
                                   l_setup.l_start_date[2])

        stop_doy = l_ConvYMD2DOY (l_setup.l_stop_date[0], l_setup.l_stop_date[1], $
                                   l_setup.l_stop_date[2])

        ;print, 'hskp start_doy = ', start_doy
        ;print, 'hskp stop_doy = ', stop_doy

        norbits = stop_doy - start_doy + 1L

        ;print, 'hskp norbits = ', norbits 
          
        hskp_mcp_hv_start = fltarr (norbits)
        hskp_mcp_hv_stop = fltarr (norbits)
        hskp_cfd_start = fltarr (norbits)
        hskp_cfd_stop = fltarr (norbits)
        hskp_x_axis_day  = fltarr (norbits)

        indx = 0L
        month = 0L
        day = 0L

        year = l_setup.l_start_dtime[0]

        for ndays = start_doy, stop_doy do begin

           ;-------------------------------------------------------
           ; from the day of year determine the correct year that
           ; day falls into
           ;-------------------------------------------------------
           stindx = where (ndays ge days_in_yr_struct[*].start_doy, stcnt)
           if (stcnt gt 0) then begin
              spindx = where (ndays le days_in_yr_struct[stindx].stop_doy, spcnt)
              if (spcnt gt 0) then begin
                 year = days_in_yr_struct[stindx[spindx]].year
              endif
           endif

           ;print, 'hskp year = ', year
           ;-------------------------------------------------------
           ; set the start and stop time to read that day's data
           ;-------------------------------------------------------
           cal_doy = FindCalDOY (year, ndays)

           ;-------------------------------------------------------
           ; need to adjust for LENA mission start day of year, day
           ; 85
           ;-------------------------------------------------------
           cal_doy = cal_doy - 85L

           ;print, 'hskp cal_doy = ', cal_doy
           l_ConvDOY2YMD, cal_doy, year, month, day

           ;print, 'hskp day = ', day
           ;print, 'hskp month = ', month

           l_setup.l_start_dtime[0] = year
           l_setup.l_start_dtime[1] = cal_doy 
           l_setup.l_start_dtime[2] = 0L
           l_setup.l_start_dtime[3] = 0L
           l_setup.l_start_dtime[4] = 0L
           l_setup.l_start_dtime[5] = 0L

           l_setup.l_stop_dtime[0] = year
           l_setup.l_stop_dtime[1] = cal_doy
           l_setup.l_stop_dtime[2] = 23L
           l_setup.l_stop_dtime[3] = 59L
           l_setup.l_stop_dtime[4] = 59L
           l_setup.l_stop_dtime[5] =  0L
 
;          print, 'hskp start_dtime = ', l_setup.l_start_dtime
;          print, 'hskp stop_dtime = ', l_setup.l_stop_dtime

           status = l_HskpGetData (l_setup, l_setup_file, l_pltpos, $
                                   l_hskp_setup, l_hskp)
           if (status ne FAILURE) then begin

           if (l_ltt_setup.l_hskp_mcp_hv_start eq 'Y') then begin

              sumrow = total ((*l_hskp.hvps_vmonsPtr)[*,0])
              nitems = n_elements ((*l_hskp.hvps_vmonsPtr)[*,0])
              hskp_mcp_hv_start[indx] = sumrow/nitems

           endif

           if (l_ltt_setup.l_hskp_mcp_hv_stop eq 'Y') then begin

              sumrow = total ((*l_hskp.hvps_vmonsPtr)[*,1])
              nitems = n_elements ((*l_hskp.hvps_vmonsPtr)[*,1]) 
              hskp_mcp_hv_stop[indx] = sumrow/nitems

           endif

           if (l_ltt_setup.l_hskp_cfd_start eq 'Y') then begin

              sumrow = total ((*l_hskp.tof_subsys_configPtr)[*,0])
              nitems = n_elements ((*l_hskp.tof_subsys_configPtr)[*,0])
              hskp_cfd_start [indx] = sumrow/nitems

           endif

           if (l_ltt_setup.l_hskp_cfd_stop eq 'Y') then begin

              sumrow = total ((*l_hskp.tof_subsys_configPtr)[*,1])
              nitems = n_elements ((*l_hskp.tof_subsys_configPtr)[*,1]) 
              hskp_cfd_stop [indx] = sumrow/nitems

           endif

           hskp_x_axis_day[indx] = ndays
           indx = indx + 1L

          endif ;--- not failure ---
       endfor ;--- ndays ---


       hskp_x_axis_day = hskp_x_axis_day[0:indx-1L]

       if (l_ltt_setup.l_hskp_mcp_hv_start eq 'Y') then begin
          hskp_mcp_hv_start = hskp_mcp_hv_start[0:indx-1L]
          ;if (l_setup.l_count_rate eq 'Y') then begin
          ;   l_Conv2CntRate, hskp_mcp_hv_start, tmp, 'SECTOR'
          ;   hskp_mcp_hv_start = tmp
          ;endif
          ;if (l_setup.l_log eq 'Y') then begin
          ;   l_Conv2Log, hskp_mcp_hv_start, tmp, hskp_mcp_hv_start_min, hskp_mcp_hv_start_max
          ;   hskp_mcp_hv_start = tmp
          ;endif

       endif

       if (l_ltt_setup.l_hskp_mcp_hv_stop eq 'Y') then begin
          hskp_mcp_hv_stop = hskp_mcp_hv_stop[0:indx-1L]
          ;if (l_setup.l_count_rate eq 'Y') then begin
          ;   l_Conv2CntRate, hskp_mcp_hv_stop, tmp, 'SECTOR'
          ;   hskp_mcp_hv_stop = tmp
          ;endif
          ;if (l_setup.l_flux eq 'Y') then begin
          ;endif
          ;if (l_setup.l_log eq 'Y') then begin
          ;   l_Conv2Log, hskp_mcp_hv_stop, tmp, hskp_mcp_hv_stop_min, hskp_mcp_hv_stop_max
          ;   hskp_mcp_hv_stop = tmp
          ;endif
       endif

       if (l_ltt_setup.l_hskp_cfd_start eq 'Y') then begin
          hskp_cfd_start = hskp_cfd_start [0:indx-1L]
          ;if (l_setup.l_count_rate eq 'Y') then begin
          ;   l_Conv2CntRate, hskp_cfd_start, tmp, 'SECTOR'
          ;   hskp_cfd_start = tmp
          ;endif
          ;if (l_setup.l_flux eq 'Y') then begin
          ;endif
          ;if (l_setup.l_log eq 'Y') then begin
          ;   l_Conv2Log, hskp_cfd_start, tmp, hskp_cfd_start_min, hskp_cfd_start_max
          ;   hskp_cfd_start = tmp
          ;endif
       endif

       if (l_ltt_setup.l_hskp_cfd_stop eq 'Y') then begin
          hskp_cfd_stop = hskp_cfd_stop [0:indx-1L]
          ;if (l_setup.l_count_rate eq 'Y') then begin
          ;   l_Conv2CntRate, hskp_cfd_stop, tmp, 'SECTOR'
          ;   hskp_cfd_stop = tmp
          ;endif
          ;if (l_setup.l_flux eq 'Y') then begin
          ;endif
          ;if (l_setup.l_log eq 'Y') then begin
          ;   l_Conv2Log, hskp_cfd_stop, tmp, hskp_cfd_stop_min, hskp_cfd_stop_max
          ;   hskp_cfd_stop = tmp
          ;endif
       endif

       l_HskpTerm, l_hskp


RETURN
END


;-------------------------------------------------------------------------------
;  Procedure: l_LttDrawOpBar
;
;  Description: Operation status bars.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_pltpos              struct        I        general plotting parameters
;  days_in_yr_struct     long []       I        the number of days in a year
;  min_leg_val           long          I        minimum status bar value
;  max_leg_val           long          I        maximum status bar value
;  color_table           float []      I        color table values
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;      MYCOLORS        N_COLOR_TABLE   long         G         # colors in color table
;                      N_GRAYS         long         G         # grays in color table
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------

PRO l_LttDrawOpBar, l_pltpos, daily_spin_avg, min_leg_val, max_leg_val, color_table
COMMON MYCOLORS, N_COLOR_TABLE, N_GRAYS


   ypltmx = 8.8/l_pltpos.yin
   bar_pos = [l_pltpos.xpltmn,l_pltpos.ypltmn,$
              l_pltpos.xpltmx,ypltmx]

   ximage = l_pltpos.axis_size/ l_pltpos.xin
   yimage = 0.005

   xsize  = round(l_pltpos.window_width * ximage)

   ypos = ypltmx + 0.005
   ydelta = 0.015
   
   ;-----------------------------
   ; Color Bar Legend based on 
   ; selected color table
   ;-----------------------------
   rgb = findgen (N_COLOR_TABLE - N_GRAYS)
   rgb = rgb + N_GRAYS

   l_CreateOpBars, rgb, $
                   xsize, ximage, yimage, ypos, bar_pos, $
                   'Bar Legend ', l_pltpos.xsm_font

   ymax = 9.0/l_pltpos.yin - 0.0072
   ymin = ypos
   pos = [l_pltpos.xpltmn, $
          ymin,  $
          l_pltpos.xpltmx, $
          ymax]
   lgd = float (pos)

  ;--------------------------------------------------------
  ; specifies location to where to draw X axis labels
  ;--------------------------------------------------------
  PLOT, [min_leg_val,max_leg_val], $
        [0,1],                $
        /NODATA,              $
        /NOERASE,             $
        XTICKS = 4,           $
        YTICKS = 1,           $
        XSTYLE = 1,           $
        YSTYLE = 1,           $
        POSITION = lgd,       $
        COLOR = 1,            $
        YTICKFORMAT = '(A1)', $
        XTICKFORMAT = '(A1)', $
        XTICKLEN = -0.2,      $
        FONT=-1,              $
        XMINOR=2,             $
        CHARSIZE = l_pltpos.md_font

   ;-----------------------------------------------
   ;Draw the top x-axis, supplying labels, etc.
   ;-----------------------------------------------
   AXIS, XRANGE=[min_leg_val, max_leg_val], $
         XTICKLEN = -0.2, $
         XTICKFORMAT = '(F4.1)', $
         XAXIS=1,         $
         XTICKS=4,        $
         XMINOR = 2,       $
         XSTYLE = 1,       $
         CHARSIZE = l_pltpos.md_font


   ypos = ypos - ydelta

   ;-----------------------------
   ; create the average rate over
   ; all spins bar
   ;-----------------------------


   l_ScaleColor, daily_spin_avg, $
                 avg_rate, $
                 min_leg_val, $
                 max_leg_val, $
                 color_table
                 

   l_CreateOpBars, avg_rate, $
                   xsize, ximage, yimage, ypos, bar_pos, $
                   'Avg Rate', l_pltpos.xsm_font


RETURN
END

;-------------------------------------------------------------------------------
;  Procedure: l_LttReadEventData
;
;  Description: Read the event data.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_setup               struct        I        general setup parameters
;  l_setup_file          string        I        runtime setup filename
;  l_pltpos              struct        I        general plotting parameters
;  l_event_setup         struct        I        event processing & plotting parameters
;  l_event               struct *      I        pointers to event data structures
;  l_ltt_setup           struct        I        LTT processing & plotting parameters
;  days_in_yr_struct     long []       I        the number of days in a year
;  l_evsp                struct *      I        pointers to sectored events data structs
;  time                  struct[]      I        struct array of apogee/perigee times
;  hr_n_secs             double        I        hours in seconds
;  event_data            float[]       O        event data
;  event_x_axis_day      float []      O        x-axis values      
;  procEvYAxis           float []      O        y-axis values
;  event_min_val         float         O        log minimum
;  event_max_val         float         O        log maximum
;  daily_spin_avg        float []      O        daily average 
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;     PLOTDEFS         MAX_PLOTS        int         G         max # plots per page
;                      MAX_WEB_SPINS    int         G         max # spins per plot
;                                                             when using web interface
;                      NO_CNTS          double      G         no counts indicator
;                      NO_DATA          double      G         no data indicator
;
;      ERROR_TYPES     SUCCESS         int         G          successful completion
;                      FAILURE         int         G          unsuccessful completion
;                      WARNING         int         G          warning message
;                      INFO            int         G          informational message
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------


PRO l_LttReadEventData, l_setup, l_setup_file, l_pltpos, l_event_setup, l_event, $
l_ltt_setup, days_in_yr_struct, l_evsp, time, hr_n_secs, event_data, $
event_x_axis_day, procEvYAxis, event_min_val, event_max_val, daily_spin_avg 
COMMON ERROR_TYPES, SUCCESS, FAILURE, WARNING, INFO
COMMON PLOTDEFS, MAX_PLOTS, MAX_WEB_SPINS, NO_CNTS, NO_DATA

    event_min_val = 0L
    event_max_val = 0L

    y_title = ' '

    nitems = n_elements (time)

    if (l_ltt_setup.TokarDisplay eq 'Y') or (l_ltt_setup.trace_filter ne ' ') then begin 
    ;--------------------------------------------
    ; define data structure to store data on an
    ; hourly basis
    ;--------------------------------------------
        if (l_ltt_setup.y_func eq 'H') then begin
           norbit_phases = l_ltt_setup.time_range * 2
           procEvData = fltarr (norbit_phases)
           event_data = make_array(norbit_phases, nitems, /FLOAT, Value = NO_DATA)
           procEvYAxis    = fltarr (norbit_phases)
        endif else begin
           ;--------------------------------------------
           ; define data structure to store data on a
           ; spin basis 
           ;--------------------------------------------
           if (l_ltt_setup.trace_filter ne ' ') then begin
              est_nspins = l_ltt_setup.time_range * 2 * 60/2 + 10
              event_data = make_array (est_nspins, nitems, /FLOAT, Value = NO_DATA)
              procEvYAxis = indgen (est_nspins) + 1L
           endif else begin
              event_data = fltarr (730, nitems)
              procEvYAxis = indgen (730) + 1L
           endelse
  
        endelse
    endif else begin
       event_data = fltarr (nitems)
    endelse 

    event_x_axis_day         = fltarr (nitems)
    daily_spin_avg           = fltarr (nitems)
    
    indx = 0L
    chngYrFlag = 0L
    start_year = time[0].year

    prev_start_time = time[0]
    day_count = time[0].doy 
   
    real_nspins = 0L
    old_nfe = 0L

    for i = 0L, nitems - 1L do begin


        start_time = time[i].time_n_secs - hr_n_secs
        stop_time = time[i].time_n_secs + hr_n_secs
        curr_year = time[i].year
 
        orig_start_time = time[i]

        if (orig_start_time.doy ne prev_start_time.doy) then day_count = day_count + 1L
        prev_start_time = orig_start_time

        l_setup.l_start_dtime = ConvSec2Arr_CalDOY (start_time, $
                                         curr_year)
        l_setup.l_stop_dtime  = ConvSec2Arr_CalDOY (stop_time, $
                                         curr_year)


        ;------------------------------------------------------
        ; change the seconds in the start time structure to 
        ; zero, else UDF does not know how to read the data.
        ; UDF reads the OA data on a record basis 
        ;------------------------------------------------------
        l_setup.l_start_dtime[4] = 0

        status = l_EventGetData  (l_setup, $
                                  l_setup_file, $
                                  l_pltpos, $
                                  l_event_setup, $
                                  l_event)

        if (status ne FAILURE) then begin 
           
        event_cnts = fltarr (l_event.nitems)
        sunp       = fltarr (l_event.nitems)
        filtered_events = 0L
 
        if (l_ltt_setup.bk_correction eq 'Y') then l_event_setup.background = 'Y'

                ;-------------------------------------------------
                ; read sectored events/sun pulse packet
                ; !!!  need to handle case where EVSP data may 
                ;      not be available, for sun pulse information
                ;      look at using OA data !!!
                ;-------------------------------------------------
                if (l_ltt_setup.trace_filter eq 'S') then begin
                   ;-------------------------------------------------
                   ; orbit and attitude data to determine the sun
                   ; sectors
                   ;-------------------------------------------------
                   status = l_OARead (l_oa, $
                                      l_setup.l_start_dtime, $
                                      l_setup.l_stop_dtime, $
                                      l_pltpos)

                   if (status eq FAILURE) then return

                   l_CalcSunPulse, l_oa, lena_sector, 'S'
                   center_sector = fix(lena_sector)
                   cs = lena_sector/8.

                   if NOT (ptr_valid(l_oa.lena_sun_sectPtr)) then begin
                      ptr = ptr_new (cs, /NO_COPY)
                      l_oa.lena_sun_sectPtr = ptr
                   endif

                   l_HndlMissOA, (*l_oa.tme_ptr), $
                                 sun_hmd_oa_tme, $
                                 n_elements(*l_oa.tme_ptr), $
                                 (*l_oa.lena_sun_sectPtr), $
                                 sun_markers, $
                                 l_setup.l_start_dtime, $
                                 l_setup.l_stop_dtime, $
                                 1

                endif

            davail = CheckAvailData (l_setup.l_start_dtime)
            if (davail eq 2) then begin
               ;-------------------------------------------------
               ; sectored events packets available in memory dump
               ; packets
               ;-------------------------------------------------
               status = l_MemdGetData (l_setup, l_pltpos, l_memd)
               if (status ne FAILURE) then begin
                  l_Xstruct = l_memd
                  n_Xitems = n_elements ((*l_memd.startUdfTimePtr)[*])
               endif
            endif else begin
               if (davail eq 4) then begin
                  nele = 0L
                  EvDmpFlg = 1L   ;--- read an extra 4 spins at end ---
                  status = l_EvSpGetData (l_setup, l_pltpos, l_evsp, nele, EvDmpFlg)
                  if (status ne FAILURE) then begin
                     l_Xstruct = l_evsp
                     n_Xitems = n_elements ((*l_evsp.startUdfTimePtr)[*])
                  endif
               endif else begin
                  print, 'No sectored events data available for this time period!!'
                  status = FAILURE
               endelse
            endelse

            if (status ne FAILURE) then begin 
               SyncEvent2SectEvnts, l_XStruct, n_Xitems, l_event, $
                                    l_event.nitems, event_cnts, sunp

               if (l_ltt_setup.tof eq 'H') then l_event_setup.hydrogen  = 'Y'
               if (l_ltt_setup.tof eq 'O') then l_event_setup.oxygen = 'Y'
               if (l_ltt_setup.tof eq 'BK') then l_event_setup.background = 'Y'
               for ee = l_ltt_setup.eng_begin, l_ltt_setup.eng_end do begin
                   if (ee eq 0) then l_event_setup.energy0 = 'Y'
                   if (ee eq 1) then l_event_setup.energy1 = 'Y'
                   if (ee eq 2) then l_event_setup.energy2 = 'Y'
               endfor


               l_Event_CorrBkgrnd, l_event_setup, l_event, l_setup, $
                                   event_cnts
               ;--------------------------------------------------------
               ; handle extra spins read for adjusting traces (sun, earth,
               ; ...) to zero degree spin angle 
               ;--------------------------------------------------------
               l_Event_Adj4ExtraData, l_event, l_event_setup, l_setup

               ;--------------------------------------------------------
               ; handle missing data from event data
               ;--------------------------------------------------------
               l_EventHndlMissData, l_setup, l_event, l_event_setup, l_oa, $
                                    l_pltpos

               ;print, 'filtered events BEFORE filter = ', filtered_events
               ;l_LttReformatSunData, l_event, l_setup.l_num_spinsec, $
               ;                      sunp, reformatted_sunp

               if (l_ltt_setup.trace_filter eq 'S') then begin
                  l_LttSyncMarkers2Event, l_event, sun_hmd_oa_tme, sun_markers, $
                                          synced_sun_tme, synced_sun_markers
               endif

               l_FilterEvent, l_ltt_setup, l_event_setup, l_event, $
                              filtered_events, synced_sun_markers 

               l_OATerm, l_oa

               if (l_ltt_setup.TokarDisplay eq 'Y') then begin
                  if (l_ltt_setup.y_func eq 'H') then begin
                  ;------------------------------------
                  ; tokar plots are on an hourly basis
                  ;------------------------------------
                     procEvData [*] = 0.0
                     ntimes = n_elements (*l_event.spTimePtr)
                     l_LttTokarPlotByHr, procEvYAxis, procEvData, $
                                         filtered_events, $
                                         (*l_event.spTimePtr), $
                                         start_time, stop_time
                     event_data[*,indx] = procEvData 
                     ;print, 'procEvData = ', procEvData 
                  endif else begin
                  ;------------------------------------
                  ; tokar plots are on a spin basis
                  ;------------------------------------
                      nfe = n_elements (filtered_events) - 1L
                      if (nfe gt old_nfe) then real_nspins = nfe
                      old_nfe = nfe
                      event_data[0:nfe,indx] = filtered_events[*]

                      nd_idx = where (filtered_events eq NO_DATA, nd_cnt)
                      if (nd_cnt gt 0) then begin
                         filtered_events[nd_idx] = 0.0
                         div = n_elements (filtered_events) - nd_cnt
                         daily_spin_avg[indx] = total(filtered_events)/div
                      endif else begin
                         nc_idx = where (filtered_events eq NO_CNTS, nc_cnt)
                         if (nc_cnt gt 0) then begin
                            filtered_events[nc_idx] = 0.0
                         endif 

                         daily_spin_avg[indx] = mean(filtered_events)

                      endelse
                          
                      filtered_events [*] = 0.0
                   endelse

                endif else begin
                  ;------------------------------------
                  ; just a regular line plot
                  ;------------------------------------

                   if (l_ltt_setup.trace_filter eq 'S') then begin
                      ;nitems = (l_ltt_setup.sun_threshold * 2L) + 1L
                      event_data[indx] = filtered_events
                   endif else begin
                      sumrow = 0L
                      sumidx = 0L
                      sidxcnt = 0L
                      sumrow = total (filtered_events, 1)
                      sumidx = where (sumrow gt 0, sidxcnt)
                      nitems = 1L
                      if (sidxcnt gt 0) then begin
                         nitems = sidxcnt
                      endif 

                      event_data[indx] = total(sumrow)/nitems
                      ;print, 'avg filtered events = ', total(sumrow)/nitems 
                    endelse ;--- trace filter ---

                endelse ;--- tokar display ---


                ;-------------------------------------------
                ;-------------------------------------------
                event_x_axis_day[indx] = day_count 
                indx = indx + 1L

            endif

            l_EventTerm, l_event

            if (davail eq 4) then begin
               l_EvSpTerm, l_evsp
            endif else begin
               if (davail eq 2) then begin
                  l_MemdTerm, l_memd
               endif
            endelse

        endif


     endfor


     nc_idx = where (event_data ne NO_CNTS, nc_cnt)
     nd_idx = where (event_data ne NO_DATA, nd_cnt)
 
     event_min_val = min (event_data) 
     if (nc_cnt gt 0) AND (nd_cnt gt 0) then begin
         event_min_val = min (event_data[nc_idx[nd_idx]]) 
     endif else begin
         if (nc_cnt gt 0) then begin
             event_min_val = min (event_data[nc_idx]) 
         endif else begin
             if (nd_cnt gt 0) then begin
                event_min_val = min (event_data[nd_idx]) 
             endif 
         endelse
     endelse

     event_max_val = max (event_data)

     if (l_setup.l_count_rate eq 'Y') then begin
        input = event_data
        l_Conv2CntRate, input, event_data, 'SECTOR'
     endif 

     ;-------------------------------------------
     ; convert to flux, assume nominal solar wind
     ; energy of 1keV with an efficiency of 
     ; 1.9 x 10^-4
     ;-------------------------------------------

     if (l_setup.l_flux eq 'Y') then begin
        event_data = event_data/0.00019
     endif

     ;-------------------------------------------
     ; Converting to log
     ;-------------------------------------------
     if (l_setup.l_log eq 'Y') then begin
        input = event_data
        l_Conv2Log, input, event_data, event_min_val, event_max_val
     endif


     if (l_ltt_setup.TokarDisplay eq 'Y') then begin
        event_data = event_data[*, 0:indx-1L]
        event_data = event_data[0:real_nspins-1L,*]
        procEvYAxis = procEvYAxis[0:real_nspins-1L]

     endif else begin
        event_data = event_data[0:indx-1L]
     endelse
     event_x_axis_day = event_x_axis_day[0:indx-1L]

     l_EventTerm, l_event


RETURN
END


;-------------------------------------------------------------------------------
;  Procedure: l_LttSnglsPlot
; 
;  Description: Plot the long term singles data.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_ltt_setup           struct        I        LTT procesing & plotting parameters
;                                               start/stop times
;  plotted               long          I        # plots plotted
;  nplots                long          I        # plots to plot
;  pages                 long          I        # pages needed for plots
;  l_setup               struct        I        general setup parameters
;  windx                 long []       I        window indexes
;  l_pltpos              struct        I        general plotting parameters
;  l_rgb_table           struct        I        red, green, blue color table values
;  sngls_start           float []      I        singles start data
;  sngls_stop            float []      I        singles stop data
;  sngls_x_axis_day      long []       I        x-axis values 
;  zaxis_title           string        I        plot title 
;  start_min             float         I        start log min value 
;  start_max             float         I        start log max value
;  stop_min              float         I        stop log min value
;  stop_max              float         I        stop log max value
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;     PLOTDEFS         MAX_PLOTS        int         G         max # plots per page
;                      MAX_WEB_SPINS    int         G         max # spins per plot
;                                                             when using web interface
;                      NO_CNTS          double      G         no counts indicator
;                      NO_DATA          double      G         no data indicator
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------

PRO l_LttSnglsPlot, l_ltt_setup, plotted, nplots, pages, l_setup, windx, l_pltpos, $
l_rgb_table, sngls_start, sngls_stop, sngls_x_axis_day, zaxis_title, start_min, $
start_max, stop_min, stop_max
COMMON PLOTDEFS, MAX_PLOTS, MAX_WEB_SPINS, NO_CNTS, NO_DATA


    if (l_ltt_setup.l_sngls_start eq 'Y') then begin

       stindx = where (sngls_start eq NO_CNTS, stcnt)
       if (stcnt gt 0) then sngls_start[stindx] = 0.0

       if (l_ltt_setup.sngls_Y_axis_min eq '') AND (l_ltt_setup.sngls_Y_axis_max eq '') then begin
          minV = min (sngls_start) - min(sngls_start)/2
          maxV = max (sngls_start) + max(sngls_start)/2

          if (l_setup.l_log eq 'Y') then begin
             minV = start_min
             maxV = start_max
          endif 
       endif else begin
          ;-------------------------
          ; special request
          ;-------------------------
          minV = l_ltt_setup.sngls_Y_axis_min 
          maxV = l_ltt_setup.sngls_Y_axis_max
       endelse

       l_PlotLttStuff, nplots, $
                       sngls_x_axis_day, $
                       sngls_start, $
                       minV, $
                       maxV, $
                       zaxis_title, $
                       l_pltpos, $
                       plotted, $
                       l_setup

       y_title = 'Singles Start'
       yctr = (l_pltpos.ypltmn + l_pltpos.ypltmx)/2.0
       xyouts, 0.01, yctr, y_title, size=l_pltpos.sm_font, $
               /NORMAL, alignment = 0.5, orientation=90.0

       y_title = ' '
       newstr = ' '
       if (l_ltt_setup.time_filter eq 'P') then begin
          newstr = strcompress ('Perigee ' + '!M+!5 '+ string(l_ltt_setup.time_range) + $
                                ' hrs', /remove_all)
       endif else begin
          if (l_ltt_setup.time_filter eq 'A') then begin
             newstr = strcompress ('Apogee ' + '!M+!5 ' + string(l_ltt_setup.time_range) + $
                                   ' hrs', /remove_all)
          endif
       endelse
       y_title = y_title + newstr


       xctr = l_pltpos.xpltmx + 0.03
       xyouts, xctr, yctr, y_title, size=l_pltpos.sm_font, $
               /NORMAL, alignment = 0.5, orientation=90.0

       l_pltpos.ypltmx = l_pltpos.ypltmn - l_pltpos.yspace
       l_pltpos.ypltmn = l_pltpos.ypltmx - l_pltpos.ydel
       plotted = plotted + 1

       l_pltpos.doc_ymn = l_pltpos.ypltmx
       l_ChckCompPg, plotted, nplots, pages, $
                     l_setup, windx, l_pltpos, l_rgb_table


    endif

    if (l_ltt_setup.l_sngls_stop eq 'Y') then begin

       spindx = where (sngls_stop eq NO_CNTS, spcnt)
       if (spcnt gt 0) then sngls_stop[spindx] = 0.0

       if (l_ltt_setup.sngls_Y_axis_min eq '') AND (l_ltt_setup.sngls_Y_axis_max eq '') then begin
           minV = min (sngls_stop) - min (sngls_stop)/2 
           maxV = max (sngls_stop) + max (sngls_stop)/2 

          if (l_setup.l_log eq 'Y') then begin
             minV = stop_min
             maxV = stop_max
          endif 
       endif else begin
          ;-------------------------
          ; special request
          ;-------------------------
          minV = l_ltt_setup.sngls_X_axis_min 
          maxV = l_ltt_setup.sngls_X_axis_max
       endelse
       
       l_PlotLttStuff, nplots, $
                       sngls_x_axis_day, $
                       sngls_stop, $
                       minV, $
                       maxV, $
                       zaxis_title, $
                       l_pltpos, $
                       plotted, $
                       l_setup

       y_title = 'Singles Stop'
       yctr = (l_pltpos.ypltmn + l_pltpos.ypltmx)/2.0
       xyouts, 0.01, yctr, y_title, size=l_pltpos.sm_font, $
              /NORMAL, alignment = 0.5, orientation=90.0


       y_title = ' '
       if (l_ltt_setup.time_filter eq 'P') then begin
          newstr = strcompress ('Perigee ' + '!M+!5 ' + string(l_ltt_setup.time_range) + $
                                ' hrs', /remove_all)
       endif else begin
          if (l_ltt_setup.time_filter eq 'A') then begin
             newstr = strcompress ('Apogee ' + '!M+!5 ' + string(l_ltt_setup.time_range) + $
                                   ' hrs', /remove_all)
          endif
       endelse
       y_title = y_title + newstr


       xctr = l_pltpos.xpltmx + 0.03
       xyouts, xctr, yctr, y_title, size=l_pltpos.sm_font, $
               /NORMAL, alignment = 0.5, orientation=90.0

       l_pltpos.ypltmx = l_pltpos.ypltmn - l_pltpos.yspace
       l_pltpos.ypltmn = l_pltpos.ypltmx - l_pltpos.ydel
       plotted = plotted + 1

       l_pltpos.doc_ymn = l_pltpos.ypltmx
       l_ChckCompPg, plotted, nplots, pages, $
                     l_setup, windx, l_pltpos, l_rgb_table

    endif 


RETURN
END


;-------------------------------------------------------------------------------
;  Procedure: l_LttHskpPlot
;
;  Description: Plot the long term HSKP data.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_ltt_setup           struct        I        LTT processing and plotting parameters
;  plotted               long          I        # plots plotted
;  nplots                long          I        # plots to be plotted
;  pages                 long          I        # pages needed for plots
;  l_setup               struct        I        general setup parameters
;  windx                 long []       I        window indexes
;  l_pltpos              struct        I        general plotting parameters
;  l_rgb_table           struct        I        red, green, blue color table values
;  hskp_mcp_hv_start     float[]       I        HSKP MCP high voltage start
;  hskp_mcp_hv_stop      float[]       I        HSKP MCP high voltage stop
;  hskp_cfd_start        float[]       I        HSKP CFD start
;  hskp_cfd_stop         float[]       I        HSKP CFD stop
;  hskp_x_axis_day       float[]       I        HSKP x-axis values 
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;     PLOTDEFS         MAX_PLOTS        int         G         max # plots per page
;                      MAX_WEB_SPINS    int         G         max # spins per plot
;                                                             when using web interface
;                      NO_CNTS          double      G         no counts indicator
;                      NO_DATA          double      G         no data indicator
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------

PRO l_LttHskpPlot, l_ltt_setup, plotted, nplots, pages, l_setup, windx, l_pltpos, $
l_rgb_table, hskp_mcp_hv_start, hskp_mcp_hv_stop, hskp_cfd_start, hskp_cfd_stop, $
hskp_x_axis_day
COMMON PLOTDEFS, MAX_PLOTS, MAX_WEB_SPINS, NO_CNTS, NO_DATA


     zaxis_title = 'total cnts'

     if (l_ltt_setup.l_hskp_mcp_hv_start eq 'Y') then begin
        hvcnt = 0L
        hvindx = where (hskp_mcp_hv_start eq NO_CNTS, hvcnt)
        if (hvcnt gt 0) then hskp_mcp_hv_start[hvindx] = 0.0
        minV = min (hskp_mcp_hv_start) - 1
        maxV = max (hskp_mcp_hv_start) + 1

        l_PlotLttStuff, nplots, $
                        hskp_x_axis_day, $
                        hskp_mcp_hv_start, $
                        minV, $
                        maxV, $
                        zaxis_title, $
                        l_pltpos, $
                        plotted, $
                        l_setup

         y_title = 'MCP VMON Start'
         yctr = (l_pltpos.ypltmn + l_pltpos.ypltmx)/2.0
         xyouts, 0.02, yctr, y_title, size=l_pltpos.sm_font, $
                 /NORMAL, alignment = 0.5, orientation=90.0

         l_pltpos.ypltmx = l_pltpos.ypltmn - l_pltpos.yspace
         l_pltpos.ypltmn = l_pltpos.ypltmx - l_pltpos.ydel
         plotted = plotted + 1

         l_pltpos.doc_ymn = l_pltpos.ypltmx
         l_ChckCompPg, plotted, nplots, pages, $
                             l_setup, windx, l_pltpos, l_rgb_table

      endif ;--- hskp_mcp_hv_start ---

      if (l_ltt_setup.l_hskp_mcp_hv_stop eq 'Y') then begin
         hvcnt = 0L
         hvindx = where (hskp_mcp_hv_stop eq NO_CNTS, spcnt)
         if (spcnt gt 0) then hskp_mcp_hv_stop[hvindx] = 0.0
         minV = min (hskp_mcp_hv_stop) - 1
         maxV = max (hskp_mcp_hv_stop) + 1

         l_PlotLttStuff, nplots, $
                         hskp_x_axis_day, $
                         hskp_mcp_hv_stop, $
                         minV, $
                         maxV, $
                         zaxis_title, $
                         l_pltpos, $
                         plotted, $
                         l_setup

         y_title = 'MCP VMON Stop'
         yctr = (l_pltpos.ypltmn + l_pltpos.ypltmx)/2.0
         xyouts, 0.02, yctr, y_title, size=l_pltpos.sm_font, $
                 /NORMAL, alignment = 0.5, orientation=90.0

         l_pltpos.ypltmx = l_pltpos.ypltmn - l_pltpos.yspace
         l_pltpos.ypltmn = l_pltpos.ypltmx - l_pltpos.ydel
         plotted = plotted + 1

         l_pltpos.doc_ymn = l_pltpos.ypltmx
         l_ChckCompPg, plotted, nplots, pages, $
                             l_setup, windx, l_pltpos, l_rgb_table
      endif ;--- hskp_mcp_hv_stop ---

      if (l_ltt_setup.l_hskp_cfd_start eq 'Y') then begin
         hvcnt = 0L
         hvindx = where (hskp_cfd_start eq NO_CNTS, spcnt)
         if (spcnt gt 0) then hskp_cfd_start [hvindx] = 0.0
         minV = min (hskp_cfd_start) - 1
         maxV = max (hskp_cfd_start) + 1

         l_PlotLttStuff, nplots, $
                         hskp_x_axis_day, $
                         hskp_cfd_start, $
                         minV, $
                         maxV, $
                         zaxis_title, $ 
                         l_pltpos, $
                         plotted, $
                         l_setup

         y_title = 'CFD Start'
         yctr = (l_pltpos.ypltmn + l_pltpos.ypltmx)/2.0
         xyouts, 0.02, yctr, y_title, size=l_pltpos.sm_font, $
                  /NORMAL, alignment = 0.5, orientation=90.0

         l_pltpos.ypltmx = l_pltpos.ypltmn - l_pltpos.yspace
         l_pltpos.ypltmn = l_pltpos.ypltmx - l_pltpos.ydel
         plotted = plotted + 1

         l_pltpos.doc_ymn = l_pltpos.ypltmx
         l_ChckCompPg, plotted, nplots, pages, $
                       l_setup, windx, l_pltpos, l_rgb_table
      endif ;--- hskp_cfd_start ---

      if (l_ltt_setup.l_hskp_cfd_stop eq 'Y') then begin
         hvcnt = 0L
         hvindx = where (hskp_cfd_stop eq NO_CNTS, spcnt)
         if (spcnt gt 0) then hskp_cfd_stop[hvindx] = 0.0
         minV = min (hskp_cfd_stop) - 1
         maxV = max (hskp_cfd_stop) + 1

         l_PlotLttStuff, nplots, $
                         hskp_x_axis_day, $
                         hskp_cfd_stop, $
                         minV, $
                         maxV, $
                         zaxis_title, $
                         l_pltpos, $
                         plotted, $
                         l_setup

         y_title = 'CFD Stop'
         yctr = (l_pltpos.ypltmn + l_pltpos.ypltmx)/2.0
         xyouts, 0.02, yctr, y_title, size=l_pltpos.sm_font, $
                 /NORMAL, alignment = 0.5, orientation=90.0

         l_pltpos.ypltmx = l_pltpos.ypltmn - l_pltpos.yspace
         l_pltpos.ypltmn = l_pltpos.ypltmx - l_pltpos.ydel
         plotted = plotted + 1

         l_pltpos.doc_ymn = l_pltpos.ypltmx
         l_ChckCompPg, plotted, nplots, pages, $
                             l_setup, windx, l_pltpos, l_rgb_table
      endif ;--- hskp_cdf_stop ---


RETURN
END

;-------------------------------------------------------------------------------
;  Procedure: l_LttEventPlot
;
;  Description:  Plot the long term event data
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_ltt_setup           struct        I        LTT processing & plotting parameters
;  plotted               long          I        # plots plotted
;  nplots                long          I        # plots to be plotted
;  pages                 long          I        # pages needed for plots
;  l_setup               struct        I        general setup parameters
;  windx                 long[]        I        window indexes
;  l_pltpos              struct        I        general plotting parameters
;  l_rgb_table           struct        I        red, green, blue color table values
;  colorbar              float[]       I        color bar values
;  event_data            float[]       I        event data
;  event_x_axis_day      float []      I        x-axis values      
;  procEvYAxis           float []      I        y-axis values
;  zaxis_title           string        I        plot title
;  bar_min_val           float         I        operational bar log minimum
;  bar_max_val           float         I        operational bar log maximum
;  daily_spin_avg        float []      I        daily average 
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;     PLOTDEFS         MAX_PLOTS        int         G         max # plots per page
;                      MAX_WEB_SPINS    int         G         max # spins per plot
;                                                             when using web interface
;                      NO_CNTS          double      G         no counts indicator
;                      NO_DATA          double      G         no data indicator
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------

PRO l_LttEventPlot, l_ltt_setup, plotted, nplots, pages, l_setup, windx, l_pltpos, $
l_rgb_table, colorbar, event_data, event_x_axis_day, procEvYAxis, zaxis_title, $
bar_min_val, bar_max_val, daily_spin_avg
COMMON PLOTDEFS, MAX_PLOTS, MAX_WEB_SPINS, NO_CNTS, NO_DATA


    if (l_ltt_setup.TokarDisplay eq 'Y') then begin

       ;------------------------------------------
       ; determine min/max color ranges
       ;------------------------------------------
       if (l_setup.l_auto_scale eq 'Y') then begin
          l_setup.l_color_min = bar_min_val 
          l_setup.l_color_max = bar_max_val
       endif else begin
          if (l_setup.l_sep_plot_scale eq 'Y') then begin
              l_setup.l_color_min = l_sngls_setup.start_color_min
              l_setup.l_color_max = l_sngls_setup.start_color_max
          endif
       endelse

       l_ScaleColor, event_data, tokar_scaled, $
                     l_setup.l_color_min, $
                     l_setup.l_color_max , $
                     l_setup.l_color_tbl

       ;------------------------------------------------------------
       ; correct the image positioning, rotate 240 degrees
       ;------------------------------------------------------------
       ;tokar_rot = rotate(tokar_scaled, 3)
       tokar_rot = rotate(tokar_scaled, 3)

       img_pos = [l_pltpos.xpltmn,l_pltpos.ypltmn,$
                  l_pltpos.xpltmx,l_pltpos.ypltmx]

       position = float(img_pos)

       xstart = position[0]
       ystart = position[1]
       ximage = l_pltpos.axis_size/ l_pltpos.xin
       yimage = l_pltpos.ypltmx - l_pltpos.ypltmn
       xsize  = round(l_pltpos.window_width * ximage)
       ysize  = round(l_pltpos.window_height * yimage)

       tokar_spect = congrid (tokar_rot, $
                              xsize, $
                              ysize)

       TV, tokar_spect, xstart, ystart, $
           XSIZE=ximage, YSIZE=yimage, /ORDER, $
           /NORMAL

       ;---------------------------------------------------
       ; if average spin bar selected to be displayed
       ;---------------------------------------------------
       if (l_setup.l_pwrsup_bars eq 'Y') then begin
          l_LttDrawOpBar, l_pltpos , daily_spin_avg, bar_min_val, bar_max_val, $
                          l_setup.l_color_tbl
       endif

       ;---------------------------------------------------
       ; draw the z-axis, color bar
       ;---------------------------------------------------

       bar_pos = [l_pltpos.xpltmx+0.015, $
                  l_pltpos.ypltmn, $
                  l_pltpos.xpltmx+0.025, $
                  l_pltpos.ypltmx]

       bar_info = {AXIS_DATA, char_size: l_pltpos.plot_font, $
                   ytitle:zaxis_title, $
                   yformat:'(F4.1)', $
                   yrange:[l_setup.l_color_min, $
                   l_setup.l_color_max]}

       l_DrawColorbar, colorbar, l_setup, bar_pos, l_pltpos, $
                       RAXIS=bar_info

            
       if (l_ltt_setup.time_filter eq 'P') then begin
          if (l_ltt_setup.y_func eq 'H') then begin 
             ytitle_frame = $
             strcompress ('Perigee +/- ' + string(l_ltt_setup.time_range) + ' hours', /remove_all)
          endif else begin
             if (l_ltt_setup.y_func eq 'S') then begin
                ytitle_frame = ' Spin '
             endif
          endelse
       endif else begin
          if (l_ltt_setup.time_filter eq 'A') then begin
             if (l_ltt_setup.y_func eq 'H') then begin 
                ytitle_frame =    $
                strcompress('Apogee +/- ' + string(l_ltt_setup.time_range) + ' hours', /remove_all)
             endif else begin
                if (l_ltt_setup.y_func eq 'S') then begin
                   ytitle_frame = ' Spin '
                endif
             endelse
          endif else begin
             if (l_ltt_setup.y_func eq 'S') then begin
                ytitle_frame = strcompress('Spin  ', /remove_all)
             endif
          endelse
       endelse


       l_PlotTokarDisplay, nplots, event_x_axis_day, procEvYAxis, img_pos, l_pltpos, $
                           plotted, ytitle_frame, l_setup.l_start_dtime, $
                           l_ltt_setup.time_range, l_ltt_setup.y_func


    endif else begin
       ;-------------------------------------------
       ; make sure the minimum value for Y-axis is
       ; greater than zero (log zero or negative
       ; is invalid)
       ;-------------------------------------------
       minV = min (event_data)
       gt_zero_idx = where (event_data gt 0, gzcnt)
       if (gzcnt gt 0) then begin
          minV = min (event_data[gt_zero_idx]) 
       endif
       l_PlotLttStuff, nplots, $
                       event_x_axis_day, $
                       event_data, $
                       minV, $
                       max (event_data), $
                       zaxis_title, $
                       l_pltpos, $
                       plotted, $
                       l_setup
    endelse

    y_title = 'Event - '
    if (l_ltt_setup.bk_correction eq 'Y') then y_title = y_title + 'Bk Corr'
    if (l_ltt_setup.tof eq 'H') then y_title = y_title + ' H'
    if (l_ltt_setup.tof eq 'O') then y_title = y_title + ' O' 
    if (l_ltt_setup.tof eq 'BK') then y_title = 'Background' 
    if (l_ltt_setup.trace_filter eq 'S') then y_title = y_title + ' (Sun Sectors)'

    if (l_ltt_setup.y_func eq 'S') then begin
       if (l_ltt_setup.time_filter eq 'P') then begin
          newstr = strcompress ('Perigee +/- ' + string(l_ltt_setup.time_range) + ' hours', /remove_all)
       endif else begin
          newstr = strcompress ('Apogee +/- ' + string(l_ltt_setup.time_range) + ' hours', /remove_all)
       endelse
       y_title = y_title + newstr 
    endif 

    ;yctr = (l_pltpos.ypltmn + l_pltpos.ypltmx)/2.0
    yctr = l_pltpos.ypltmx + 0.01
    xctr = (l_pltpos.xpltmn + l_pltpos.xpltmx)/2.0
    xyouts, xctr, yctr, y_title, size=l_pltpos.sm_font, $
            /NORMAL, alignment = 0.5


    l_pltpos.ypltmx = l_pltpos.ypltmn - l_pltpos.yspace
    l_pltpos.ypltmn = l_pltpos.ypltmx - l_pltpos.ydel
    plotted = plotted + 1

    l_pltpos.doc_ymn = l_pltpos.ypltmx
    l_ChckCompPg, plotted, nplots, pages, $
                  l_setup, windx, l_pltpos, l_rgb_table



RETURN
END

;-------------------------------------------------------------------------------
;  Procedure: l_LttSnglsDump
;
;  Description: Generate a text dump of the LTT singles data.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  ofd                   long          I        output file descriptor
;  sngls_x_axis          float[]       I        x axis values
;  sngls_start           float[]       I        start data
;  sngls_stop            float[]       I        stop data
;  l_ltt_setup           struct        I        LTT processing & plotting parameters
;  col2title             string        I        measurement title
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------

PRO l_LttSnglsDump, ofd, sngls_x_axis,  sngls_start,  sngls_stop, l_ltt_setup, col2title


    nitems = n_elements (sngls_x_axis)

    time_filter_str = 'Day of Year'
    time_filter = 0L 
    newstr = ' '
    if (l_ltt_setup.time_filter eq 'P') then begin
       newstr = strcompress ('Perigee ' + '+/-'+ string(l_ltt_setup.time_range) + $
                             ' hrs', /remove_all)
       time_filter_str = 'Perigee'
       time_filter = 1L 
    endif else begin
       if (l_ltt_setup.time_filter eq 'A') then begin
          newstr = strcompress ('Apogee ' + '+/-' + string(l_ltt_setup.time_range) + $
                                ' hrs', /remove_all)
          time_filter_str = 'Apogee'
          time_filter = 1L 
       endif
    endelse

    printf, ofd, ' '

    if (l_ltt_setup.l_sngls_y_axis_measurement eq 'I') then begin
       measurement = ' -- Integral Counts'
    endif else begin
       measurement = ' -- Daily Average'
    endelse

    col2title = col2title + measurement 

    if (l_ltt_setup.l_sngls_start eq 'Y') then begin
       printf, ofd, 'Singles Start Data:  ' + newstr 
       printf, ofd, ' '

       printf, ofd, time_filter_str, col2title, format='((A14, TR2), (A42))'

       if (time_filter gt 0L) then begin 
          for i = 0, nitems - 1L do begin
             time_str = STRING (sngls_x_axis[i].doy, $ 
                                sngls_x_axis[i].hour, $
                                sngls_x_axis[i].min, $
                                sngls_x_axis[i].sec, $
                                format="(i4, ' ', i2.2,':',i2.2,':',i2.2)")

             printf, ofd, time_str, sngls_start[i], format='((A18, TR2), (I16))'
          endfor  
       endif else begin
          for i = 0, nitems - 1L do begin
             printf, ofd, sngls_x_axis[i], sngls_start[i], format='((I14, TR2), (I16))'
          endfor  
       endelse

    endif

    printf, ofd, ' '
    printf, ofd, ' '

    if (l_ltt_setup.l_sngls_stop eq 'Y') then begin

       printf, ofd, 'Singles Stop Data:  ' + newstr
       printf, ofd, ' '

       printf, ofd, time_filter_str, col2title, format='((A14, TR1), (A42))'

       if (time_filter gt 0L) then begin
          for i = 0, nitems - 1L do begin
             time_str = STRING (sngls_x_axis[i].doy, $ 
                                sngls_x_axis[i].hour, $
                                sngls_x_axis[i].min, $
                                sngls_x_axis[i].sec, $
                                format="(i4, ' ', i2.2,':',i2.2,':',i2.2)")
             printf, ofd, time_str, sngls_stop[i], format='((A18, TR2), (I16))'
          endfor  
       endif else begin
          for i = 0, nitems - 1L do begin
             printf, ofd, sngls_x_axis[i], sngls_stop[i], format='((I14, TR2), (I16))'
          endfor  
       endelse
    endif

    printf, ofd, ' '
    printf, ofd, ' '
   

RETURN
END

;-------------------------------------------------------------------------------
;  Procedure: l_LttHskpDump
;
;  Description: Generate a text dump  of the LTT HSKP data.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  ofd                   long          I        output file descriptor
;  hskp_x_axis           long[]        I        x-axis values
;  hskp_mcp_hv_start     float[]       I        HSKP MCP high voltage start
;  hskp_mcp_hv_stop      float[]       I        HSKP MCP high voltage stop
;  hskp_cfd_start        float[]       I        HSKP CFD start
;  hskp_cfd_stop         float[]       I        HSKP CFD stop
;  l_ltt_setup           struct        I        LTT processing & plotting paramters
;  col2title             string        I        measurement title
;  days_in_yr_struct     struct[]      I        # of days in that year
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------

PRO l_LttHskpDump, ofd, hskp_x_axis, hskp_mcp_hv_start, hskp_mcp_hv_stop, $
hskp_cfd_start, hskp_cfd_stop, l_ltt_setup, col2title, days_in_yr_struct


    nitems = n_elements (hskp_x_axis)
    cal_doy = intarr(nitems)
    for i = 0L, nitems - 1L do begin
       stindx = where (hskp_x_axis[i] ge days_in_yr_struct[*].start_doy, stcnt)
       if (stcnt gt 0) then begin
          spindx = where (hskp_x_axis[i] le days_in_yr_struct[stindx].stop_doy, spcnt)
          if (spcnt gt 0) then begin
             year = days_in_yr_struct[stindx[spindx]].year
          endif
       endif
       cal_doy[i] = FindCalDOY (year, hskp_x_axis[i])
       cal_doy[i] = cal_doy[i] - 85L
    endfor

    printf, ofd, ' '

    if (l_ltt_setup.l_hskp_mcp_hv_start eq 'Y') then begin
       printf, ofd, 'HSKP MCP HV Start ' 
       printf, ofd, '------------------' 
       printf, ofd, ' '

       printf, ofd, 'Day of Year', col2title, format='((A14, TR1), (A20))'

       for i = 0L, nitems - 1L do begin
           printf, ofd, cal_doy[i], hskp_mcp_hv_start[i], format='((I14, TR1), (I16))'
       endfor  

    endif

    printf, ofd, ' '
    printf, ofd, ' '

    if (l_ltt_setup.l_hskp_mcp_hv_stop eq 'Y') then begin

       printf, ofd, 'HSKP MCP HV Stop '
       printf, ofd, '-----------------'
       printf, ofd, ' '

       printf, ofd, 'Day of Year', col2title, format='((A14, TR1), (A20))'

       for i = 0L, nitems - 1L do begin
          printf, ofd, cal_doy[i], hskp_mcp_hv_stop[i], format='((I14, TR1), (I16))'
       endfor  
    endif

    printf, ofd, ' '
    printf, ofd, ' '

    if (l_ltt_setup.l_hskp_cfd_start eq 'Y') then begin 

       printf, ofd, 'HSKP CFD Start '
       printf, ofd, '---------------'
       printf, ofd, ' '

       printf, ofd, 'Day of Year', col2title, format='((A14, TR1), (A20))'

       for i = 0L, nitems - 1L do begin
          printf, ofd, cal_doy[i], hskp_cfd_start[i], format='((I14, TR1), (I16))'
       endfor

    endif

    printf, ofd, ' '
    printf, ofd, ' '

    if (l_ltt_setup.l_hskp_cfd_stop eq 'Y') then begin

       printf, ofd, 'HSKP CFD Start '
       printf, ofd, '---------------'
       printf, ofd, ' '

       printf, ofd, 'Day of Year', col2title, format='((A14, TR1), (A20))'

       for i = 0L, nitems - 1L do begin
          printf, ofd, cal_doy[i], hskp_cfd_stop[i], format='((I14, TR1), (I16))'
       endfor

    endif
   
    printf, ofd, ' '
    printf, ofd, ' '

RETURN
END


;-------------------------------------------------------------------------------
;  Procedure: l_LttEventDump
;
;  Description: Generate a text dump of the LTT event data.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  ofd                   long          I        output file descriptor
;  event_x_axis          long[]        I        x-axis values
;  event_y_axis          long[]        I        y-axis values
;  event_data            float[]       I        event data 
;  daily_spin_avg        float[]       I        daily averages
;  l_ltt_setup           struct        I        LTT processing & plotting parameters
;  col2title             string        I        measurement title
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------

PRO l_LttEventDump, ofd, event_x_axis, event_y_axis, event_data, $
daily_spin_avg, l_ltt_setup, col2title


    nitems = n_elements (event_x_axis)
    yitems = n_elements (event_y_axis)

    printf, ofd, ' '
    printf, ofd, ' '

    printf, ofd, 'Event Data       '
    printf, ofd, '-----------------'
    printf, ofd, ' '

    printf, ofd, 'Day of Year', col2title, format='((A14, TR1), (A20))'

    if (l_ltt_setup.TokarDisplay eq 'Y') or (l_ltt_setup.trace_filter ne ' ') then begin 

       for i = 0L, nitems - 1L do begin
          time_str = STRING (event_x_axis[i].doy, $ 
                             event_x_axis[i].hour, $
                             event_x_axis[i].min, $
                             event_x_axis[i].sec, $
                             format="(i4, ' ', i2.2,':',i2.2,':',i2.2)")
          printf, ofd, time_str, event_data[*,i], format='((A18, TR2), TR2, (10e12.3))'
          printf, ofd, ' '
          printf, ofd, ' '
       endfor  
    endif

    printf, ofd, ' '
    printf, ofd, ' '


RETURN
END


;-------------------------------------------------------------------------------
;  Function: l_LttProc
;
;  Description:  Main calling routine which calls procedures to process the LTT
;       plots based on user selections.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_setup               struct        I        general setup parameters
;  l_setup_file          string        I        runtime setup filename
;  l_pltpos              struct        I        general plot parameters
;  l_rgb_table           struct        I        color table values 
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;     PLOTDEFS         MAX_PLOTS        int         G         max # plots per page
;                      MAX_WEB_SPINS    int         G         max # spins per plot
;                                                             when using web interface
;                      NO_CNTS          double      G         no counts indicator
;                      NO_DATA          double      G         no data indicator
;
;      ERROR_TYPES     SUCCESS         int         G          successful completion
;                      FAILURE         int         G          unsuccessful completion
;                      WARNING         int         G          warning message
;                      INFO            int         G          informational message
;
;      EPOCH_DATA      EPOCH_YR        int         G          launch year
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------


FUNCTION l_LttProc, l_setup, l_setup_file, l_pltpos, l_rgb_table
COMMON PLOTDEFS, MAX_PLOTS, MAX_WEB_SPINS, NO_CNTS, NO_DATA
COMMON ERROR_TYPES, SUCCESS, FAILURE, WARNING, INFO
COMMON EPOCH_DATA, EPOCH_YR


    year      = strmid(systime(0), 20, 23)
   

    l_dtime_struct = { DATE_TIME_STRUCT, $
                       year  : 0, $
                       doy   : 0, $
                       month : 0, $
                       day   : 0, $
                       hour  : 0, $
                       min   : 0, $
                       sec   : 0, $
                       time_n_secs : 0.0D $
                     }


    status = l_LttReadSetup (err_msg, l_ltt_setup, l_setup_file)
    if (status lt 0) then begin
       print, err_msg 
       RETURN, FAILURE 
    endif else begin
       if (status ne SUCCESS) then begin
          print, l_ConvErrVal2Str(status), err_msg
       endif
    endelse
    
    l_SnglsReadSetup, l_sngls_setup, l_setup_file
    l_BinnedReadSetup, l_binned_setup, l_setup_file
    l_EventReadSetup, l_event_setup, l_setup_file
    
    if (l_ltt_setup.l_event eq 'Y') AND (l_ltt_setup.TokarDisplay eq 'Y') then begin
       if (l_ltt_setup.time_filter eq ' ') then l_ltt_setup.time_filter = 'A'
    endif


    l_pltpos.plot_hdr_title = l_pltpos.plot_hdr_title + '!5Long Term Trending'


    if (l_ltt_setup.time_filter eq 'P') then begin
       l_ReadPerigeeTimes, l_setup.l_start_dtime, l_setup.l_stop_dtime, perigee_times
       time = perigee_times
    endif else begin
       if (l_ltt_setup.time_filter eq 'A') then begin
          l_ReadApogeeTimes, l_setup.l_start_dtime, l_setup.l_stop_dtime, apogee_times
          time = apogee_times
       endif
    endelse


    nitems = n_elements (time)

    ;-------------------------------
    ; store original times will be used
    ; later in plot header 
    ;-------------------------------
    orig_start_dtime = l_setup.l_start_dtime
    orig_stop_dtime = l_setup.l_stop_dtime


    if (l_ltt_setup.TokarDisplay eq 'Y') then begin
       ;-------------------------------
       ; seven hours in seconds
       ;------------------------------
       hr_n_secs = 3600. * l_ltt_setup.time_range 
       l_pltpos.tokar_display = 'Y'
    endif else begin
       ;-------------------------------
       ; one hour in seconds
       ;------------------------------
       hr_n_secs = 3600.
    endelse



    ;-------------------------------
    ; not using hh:mm for time label
    ;-------------------------------
    l_pltpos.X_axis_eq_time = 'Y'

    ;-------------------------------
    ;-------------------------------
    DetYrByDOY, days_in_yr_struct

    ;-------------------------------------------------------------------
    ;  Read the Singles data, if selected
    ;-------------------------------------------------------------------
    if (l_ltt_setup.l_sngls_start eq 'Y') OR (l_ltt_setup.l_sngls_stop eq 'Y') then begin

       l_ltt_setup.TokarDisplay = 'N'
       if (l_ltt_setup.time_filter eq ' ') then begin


           ;-------------------------------------------------------
           ; convert the given year, month, day input start/stop
           ; time into day of year to determine the number of days
           ; to process
           ;-------------------------------------------------------
           start_doy = l_ConvYMD2DOY (l_setup.l_start_date[0], l_setup.l_start_date[1], $
                                     l_setup.l_start_date[2])

           stop_doy = l_ConvYMD2DOY (l_setup.l_stop_date[0], l_setup.l_stop_date[1], $
                                      l_setup.l_stop_date[2])


   
           norbits = stop_doy - start_doy + 1L

          
           sngls_start = fltarr (norbits)
           sngls_stop  = fltarr (norbits)
           sngls_x_axis_day  = fltarr (norbits)


           year = l_setup.l_start_dtime[0]
           indx = 0L

           for ndays = start_doy, stop_doy do begin

              month  = 0L
              day    = 0L

              ;-------------------------------------------------------
              ; from the day of year determine the correct year that
              ; day falls into
              ;-------------------------------------------------------
              stindx = where (ndays ge days_in_yr_struct[*].start_doy, stcnt)
              if (stcnt gt 0) then begin
                 spindx = where (ndays le days_in_yr_struct[stindx].stop_doy, spcnt)
                 if (spcnt gt 0) then begin
                    year = days_in_yr_struct[stindx[spindx]].year
                 endif
              endif

              ;-------------------------------------------------------
              ; set the start and stop time to read that day's data
              ;-------------------------------------------------------
              cal_doy = FindCalDOY (year, ndays)

              ;-------------------------------------------------------
              ; need to adjust for LENA mission start day of year, day
              ; 85
              ;-------------------------------------------------------
              cal_doy = cal_doy - 85L

              l_ConvDOY2YMD, cal_doy, year, month, day


              l_setup.l_start_dtime[0] = year
              l_setup.l_start_dtime[1] = cal_doy
              l_setup.l_start_dtime[2] = 0L
              l_setup.l_start_dtime[3] = 0L
              l_setup.l_start_dtime[4] = 0L
              l_setup.l_start_dtime[5] = 0L

              l_setup.l_stop_dtime[0] = year
              l_setup.l_stop_dtime[1] = cal_doy
              l_setup.l_stop_dtime[2] = 23L
              l_setup.l_stop_dtime[3] = 59L
              l_setup.l_stop_dtime[4] = 59L
              l_setup.l_stop_dtime[5] =  0L 

              l_LttReadSnglsData, l_setup, $
                                  l_setup_file, $
                                  l_pltpos, $ 
                                  l_sngls_setup, $
                                  l_sngls, $
                                  l_ltt_setup, $
                                  days_in_yr_struct, $
                                  sngls_start, $
                                  sngls_stop, $
                                  sngls_x_axis_day, $
                                  ndays, indx
          endfor

          sngls_x_axis = sngls_x_axis_day[0:indx-1L]

       endif else begin


          ntf_items = n_elements (time)
          sngls_start = fltarr (ntf_items)
          sngls_stop  = fltarr (ntf_items)
          sngls_x_axis_day  = fltarr (ntf_items)
          prev_start_time = time[0]
          day_count = time[0].doy 
          indx = 0L

          for i = 0L, ntf_items - 1L do begin
              start_time = time[i].time_n_secs - hr_n_secs
              stop_time = time[i].time_n_secs + hr_n_secs
              curr_year = time[i].year

              orig_start_time = time[i]

              if (orig_start_time.doy ne prev_start_time.doy) then day_count = day_count + 1L
              prev_start_time = orig_start_time

              l_setup.l_start_dtime = ConvSec2Arr_CalDOY (start_time, $
                                               curr_year)
              l_setup.l_stop_dtime  = ConvSec2Arr_CalDOY (stop_time, $
                                               curr_year)


  
              l_LttReadSnglsData, l_setup, $
                                  l_setup_file, $
                                  l_pltpos, $ 
                                  l_sngls_setup, $
                                  l_sngls, $
                                  l_ltt_setup, $
                                  days_in_yr_struct, $
                                  sngls_start, $
                                  sngls_stop, $
                                  sngls_x_axis_day, $
                                  i, indx
          endfor

           sngls_x_axis = time[0:indx-1L]

       endelse


       sngls_x_axis_day = sngls_x_axis_day [0:indx-1L] 

       if (l_ltt_setup.l_sngls_start eq 'Y') then begin
          sngls_start = sngls_start[0:indx-1L]
          ;print, 'sngls start = ', sngls_start
          if (l_setup.l_count_rate eq 'Y') then begin
             l_Conv2CntRate, sngls_start, tmp, 'SECTOR'
             sngls_start = tmp
             ;print, 'singles count rate = ', sngls_start
          endif 
          if (l_setup.l_flux eq 'Y') then begin
          endif
          if (l_setup.l_log eq 'Y') then begin
             l_Conv2Log, sngls_start, tmp, start_min, start_max
             sngls_start = tmp
             ;print, 'singles log = ', sngls_start
          endif

       endif

       if (l_ltt_setup.l_sngls_stop eq 'Y') then begin
          sngls_stop = sngls_stop[0:indx-1L]

          if (l_setup.l_count_rate eq 'Y') then begin
             l_Conv2CntRate, sngls_stop, tmp, 'SECTOR'
             sngls_stop = tmp
          endif
          if (l_setup.l_flux eq 'Y') then begin
          endif
          if (l_setup.l_log eq 'Y') then begin
             l_Conv2Log, sngls_stop, tmp, stop_min, stop_max
             sngls_stop = tmp
          endif
       endif

    endif ;--- end if singles data ---- 

    ;-------------------------------------------------------------------
    ;  Read the Housekeeping data, if selected
    ;-------------------------------------------------------------------
    if (l_ltt_setup.l_hskp_mcp_hv_start eq 'Y') OR (l_ltt_setup.l_hskp_mcp_hv_stop eq 'Y') OR $
       (l_ltt_setup.l_hskp_cfd_start eq 'Y') OR (l_ltt_setup.l_hskp_cfd_stop eq 'Y') then begin

       l_ltt_setup.TokarDisplay = 'N'
 
       l_LttReadHskpData, l_setup, $
                          l_setup_file, $ 
                          l_pltpos, $
                          l_hskp_setup, $ 
                          l_hskp, $
                          l_ltt_setup, $
                          days_in_yr_struct, $
                          hskp_mcp_hv_start, $
                          hskp_mcp_hv_stop, $
                          hskp_cfd_start, $
                          hskp_cfd_stop, $
                          hskp_x_axis_day

    endif ;--- end if hskp data ---- 



    ;-------------------------------------------------------------------
    ;  Read the Event data, if selected
    ;-------------------------------------------------------------------
    if (l_ltt_setup.l_event eq 'Y') then begin

       l_LttReadEventData, l_setup, $
                           l_setup_file, $
                           l_pltpos, $
                           l_event_setup, $
                           l_event, $
                           l_ltt_setup, $
                           days_in_yr_struct, $
                           l_evsp, $
                           time, $
                           hr_n_secs, $
                           event_data, $
                           event_x_axis_day, $
                           procEvYAxis, $
                           event_min_val, event_max_val, $
                           daily_spin_avg


    endif ;--- end if Event data ---


    ;-------------------------------------------------------------------
    ; Read the Binned/Image data, if selected
    ;-------------------------------------------------------------------


    ;-------------------------------------------------------------------
    ; Plot the data
    ;-------------------------------------------------------------------

    ;----------------------------------------
    ; replacing with original user start/stop
    ; time for use in plot header
    ;----------------------------------------
    l_setup.l_start_dtime = orig_start_dtime
    l_setup.l_stop_dtime = orig_stop_dtime

    nloops = 1L
    if (l_setup.l_postscript eq 'Y') then nloops = 2L

    ;----------------------------------------
    ; create output text dump file name
    ; open the file for writing
    ;----------------------------------------
    if (l_setup.l_dumps eq 'Y') then begin
       l_setup.l_filedesc = 'ltt'
       l_GenFileName, l_setup, '0'
       filename = l_setup.l_prod_dst + '/' + l_setup.l_filename + '.dmp'
       openw, ofd, filename, /get_lun 
       l_pltpos.dmp_hdr_title = 'Long Term Trending Text Dump'
       l_DumpHdrInfo, ofd, l_setup, l_pltpos, filename
    endif

    for i = 0L, nloops - 1L do begin
       nrows =  l_ltt_setup.nplots
       nplots = nrows
       npages = CEIL (nrows/MAX_PLOTS) - 1L
       pages = 0L

       if (nrows - MAX_PLOTS gt 0L) then npages = npages + 1L
       if (npages eq 0L) then npages = 1L
       if (npages gt 0L) then windx = intarr (npages)


       if (i eq 1L) AND (l_setup.l_postscript eq 'Y') then begin
          plot_ps = 'Y'
          ;generate filename
       endif

       ;------------------------------------------------------
       ; Setup X window display
       ;------------------------------------------------------
       l_SetUpZBuffer, l_pltpos, l_setup, l_rgb_table, colorbar, nplots

 
       plotted = 0L
       pages = 0L

       ;------------------------------------------------------
       ; adjust plot sizes
       ;------------------------------------------------------
       if (l_ltt_setup.TokarDisplay ne 'Y') OR $
           (l_ltt_setup.y_func ne 'S') then l_setup.l_pwrsup_bars = 'N'

       l_AdjPlotSize, l_pltpos, l_setup, nplots


       ;------------------------------------------------------
       ; convert to real cal doy
       ;------------------------------------------------------

       if (l_ltt_setup.TokarDisplay ne 'Y') then begin
 
       if (l_setup.l_start_date[2] ne 0 and l_setup.l_start_date[1] le 12) then begin
          start_doy = l_ConvYMD2DOY (l_setup.l_start_date[0], l_setup.l_start_date[1], $
                             l_setup.l_start_date[2])
       endif else begin
          l_ConvDOY2YMD, l_setup.l_start_date[1], l_setup.l_start_date[0], mon, dy
          start_doy = l_ConvYMD2DOY (l_setup.l_start_date[0], mon, dy)
       endelse
       time_array = [l_setup.l_start_date[0], start_doy]
       DaysSinceLaunch, time_array, new_start_doy
       
       if (l_setup.l_stop_date[2] ne 0 and l_setup.l_stop_date[1] le 12) then begin
          stop_doy = l_ConvYMD2DOY (l_setup.l_stop_date[0], l_setup.l_stop_date[1], $
                               l_setup.l_stop_date[2])
       endif else begin
          l_ConvDOY2YMD, l_setup.l_stop_date[1], l_setup.l_stop_date[0], mon, dy
          stop_doy = l_ConvYMD2DOY (l_setup.l_stop_date[0], mon, dy)
       endelse
       time_array = [l_setup.l_stop_date[0], stop_doy]
       DaysSinceLaunch, time_array, new_stop_doy
        
          start_year = l_setup.l_start_date[0]
          stop_year = l_setup.l_stop_date[0]

          l_DetTicks4LinePlots, start_year, stop_year, start_doy, stop_doy, $
                                new_start_doy, new_stop_doy, l_pltpos

       endif else begin
          l_DetXTicks4Tokar, l_ltt_setup, time, hr_n_secs, l_pltpos
       endelse

       ;-----------------------------------------------------------------------
       ; convert the x axis labels's string from day of year value since
       ; launch to the calander day of year.
       ;-----------------------------------------------------------------------
       xindx = where (l_pltpos.xtick_labels ne 0, xcnt)
       nlabels = n_elements (l_pltpos.xtick_labels)
       if (xcnt gt 0) then begin
          nlabels = n_elements (xindx)
       endif

       ;-----------------------------------------------------------------------
       ;  ADD CONVERSION OF YY/MM/DD STRING IN THE FOR LOOP
       ;-----------------------------------------------------------------------

       for i = 0L, nlabels - 1L do begin
          doy = long (l_pltpos.xtick_labels[i])
          stindx = where (doy ge days_in_yr_struct[*].start_doy, stcnt)
          indexes = stindx
          if (stcnt gt 0) then begin
            spindx = where (doy le days_in_yr_struct[stindx].stop_doy, spcnt)
            if (spcnt lt 0) then begin
               print, 'day of year is out of range: ', doy
            endif else begin
               year = days_in_yr_struct[indexes[spindx]].year
               month = 0L
               day = 0L
               new_label = FindDOYSinceLaunch (days_in_yr_struct[indexes[spindx]].year, doy)
               l_pltpos.xtick_labels[i] = strcompress(string(new_label), /remove_all)
               l_ConvDOY2YMD, new_label, year, month, day
               year = string (strcompress (year, /remove_all))
               yr_tmp = strmid(year, 2, 2)
               l_pltpos.xtick_ymd_labels[i] = strcompress ( $
                                              string (format='(A2)', yr_tmp) + '/' + $
                                              string (format='(i2.2)', month) + '/' + $
                                              string (format='(i2.2)', day), /remove_all)
            endelse
          endif else begin
               print, 'day of year is out of range: ', doy
               spindx = stindx
          endelse
       
       endfor  ;---- nlabels ----


       ztitle = ' '
       cnt_ztitle = ' '
       flx_ztitle = ' '
       log_ztitle = ' '

       if (l_setup.l_log eq 'Y') then log_ztitle = 'log '

       if (l_setup.l_count_rate eq 'Y') then begin
          cnt_ztitle = 'cnt Rate[Hz]'
          if (l_setup.l_flux eq 'Y') then begin
             cnt_ztitle = ' '
             flx_ztitle = '!5atoms/!Ncm!E2/!5s'
          endif
       endif else begin
          if (l_setup.l_flux eq 'Y') then begin
             cnt_ztitle = ' '
             flx_ztitle = '!5atoms/!Ncm!E2/!5s'
          endif else begin
             ztitle = 'total cnts '
             if (l_ltt_setup.l_sngls_start eq 'Y') or (l_ltt_setup.l_sngls_stop eq 'Y') then begin
                if (l_ltt_setup.l_sngls_y_axis_measurement eq 'A') then begin
                   ztitle = 'daily avg '
                endif else begin
                   ztitle = 'integral cnts '
                endelse
             endif
          endelse
       endelse

       zaxis_title = ztitle + log_ztitle + flx_ztitle + cnt_ztitle



       ;-----------------------------------------------------------
       ; Plot the singles data, if selected
       ;-----------------------------------------------------------
       if (l_ltt_setup.l_sngls_start eq 'Y') OR (l_ltt_setup.l_sngls_stop eq 'Y') then begin

          l_LttSnglsPlot, l_ltt_setup, $
                          plotted, $
                          nplots, $
                          pages, $
                          l_setup, $
                          windx, $
                          l_pltpos, $
                          l_rgb_table, $
                          sngls_start, $
                          sngls_stop, $
                          sngls_x_axis_day, $
                          zaxis_title, $
                          start_min, start_max, $
                          stop_min, stop_max

          if (l_setup.l_dumps eq 'Y') then begin
             l_LttSnglsDump, ofd, $ 
                             sngls_x_axis, $
                             sngls_start, $
                             sngls_stop, $
                             l_ltt_setup, $
                             zaxis_title
          endif 

       endif ;--- end plot singles data ---

       ;-----------------------------------------------------------
       ; Plot the HSKP data, if selected
       ; High Voltage Monitor start/stop
       ; CFD start/stop
       ;-----------------------------------------------------------

       if (l_ltt_setup.l_hskp_mcp_hv_start eq 'Y') OR $
          (l_ltt_setup.l_hskp_mcp_hv_stop eq 'Y') OR $
          (l_ltt_setup.l_hskp_cfd_start eq 'Y') OR $
          (l_ltt_setup.l_hskp_cfd_stop eq 'Y') then begin

          l_LttHskpPlot, l_ltt_setup, $
                         plotted, $
                         nplots, $ 
                         pages, $ 
                         l_setup, $
                         windx, $
                         l_pltpos, $
                         l_rgb_table, $
                         hskp_mcp_hv_start, $
                         hskp_mcp_hv_stop, $
                         hskp_cfd_start, $
                         hskp_cfd_stop, $
                         hskp_x_axis_day

         if (l_setup.l_dumps eq 'Y') then begin
            l_LttHskpDump, ofd, $
                           hskp_x_axis_day, $
                           hskp_mcp_hv_start, $
                           hskp_mcp_hv_stop, $
                           hskp_cfd_start, $
                           hskp_cfd_stop, $
                           l_ltt_setup, $
                           zaxis_title , $
                           days_in_yr_struct
         endif

       endif ;--- end plot HSKP data ---

       ;-----------------------------------------------------------
       ; Plot the Event data, if selected
       ;-----------------------------------------------------------
       if (l_ltt_setup.l_event eq 'Y') then begin

       
          l_LttEventPlot, l_ltt_setup, $
                          plotted, $
                          nplots, $
                          pages, $
                          l_setup, $
                          windx, $
                          l_pltpos, $
                          l_rgb_table, $
                          colorbar, $
                          event_data, $
                          event_x_axis_day, $
                          procEvYAxis, $
                          zaxis_title, $
                          event_min_val, event_max_val, $
                          daily_spin_avg

          if (l_setup.l_dumps eq 'Y') then begin
             l_LttEventDump, ofd, $
                             time, $
                             procEvYAxis, $
                             event_data, $
                             daily_spin_avg, $
                             l_ltt_setup, $
                             zaxis_title
          endif

       endif ;--- end plot Event data ---


       ;-----------------------------------------------------------
       ; Plot the Binned/Image data, if selected
       ;-----------------------------------------------------------



       if (i eq 1L) AND (l_setup.l_postscript eq 'Y') then begin
          plot_ps = 'N'
          device, /close_file
       endif

    endfor ;--- nloops, postscript? ---

    if (l_setup.l_dumps eq 'Y') then begin
       close, ofd
       free_lun, ofd
    endif

    RETURN, SUCCESS

END

