
;-------------------------------------------------------------------------------
;
;  Unit Name : l_HskpRtns.pro
;
;  Purpose   : Contains routines to read, write, and plot housekeeping
;              data.
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               06/00           v1.0            Original Implementation
;  ELH               05/01           v1.0            Added l_AdjPlotSize 
;  ELH               01/02           v1.0.8          Added parm in l_HndlMissData to
;                                                    handle rolling data based on
;                                                    spin angle input
;
;
;  File Revision Number:   %I%
;  File Last Modified  :   %E%  %T%
;
;-------------------------------------------------------------------------------

;-------------------------------------------------------------------------------
;  Function :  l_HskpReadData
;
;  Description:
;    Read the housekeeping data record by record.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  inst                  string        I        instrument (LENA)
;  expr                  string        I        experiment (LENASCI or LENAHSKP)
;  vinst                 string        I        virtual instrument (name found in
;                                               $UDF_DATA in LENASCI or LENAHSKP
;                                               in IMAGE1 or IMAGE1Q) 
;  btime                 long[]        I        UDF begin time [yr, doy, hh, mm, ss]
;  etime                 long[]        I        UDF end time [yr, doy, hh, mm, ss]
;  l_hskp                struct*      I/O       hskp data specific parameters
;  numRecs               long          O        total number of records
;  data_format           long          I        a one dim array of [spins * spin sector]
;                                               or a two dim array of [spins, spin sector]
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;      ERROR_TYPES     SUCCESS         int         G          successful completion
;                      FAILURE         int         G          unsuccessful completion
;                      WARNING         int         G          warning message
;                      INFO            int         G          informational message
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;-------------------------------------------------------------------------------

FUNCTION l_HskpReadData, inst, expr, vinst, btime, etime, l_hskp, numRecs, data_source
COMMON ERROR_TYPES, SUCCESS, FAILURE, WARNING, INFO


   data_source = ' '
   fh = l_UdfGetDataHandle (inst, expr, vinst, btime, etime, data_source, /GROUP)
   if (fh eq -1) then begin
      ;print, 'CLOSE l_HskpReadData FN: ', btime
      udf_close, fh
      RETURN, FAILURE
   endif


   indx = 0L


   hvps_sys_errs     = fltarr (numRecs, l_hskp.hvps_sys_errsNParms)
   lvps_sys_errs     = fltarr (numRecs, l_hskp.lvps_sys_errsNParms)
   sys_lev_errs      = fltarr (numRecs, l_hskp.sys_lev_errsNParms)
   sys_lev_stats     = fltarr (numRecs, l_hskp.sys_lev_statsNParms)
   data_pkt_sel_stat = fltarr (numRecs, l_hskp.data_pkt_sel_statNParms)
   mass_binn_config  = fltarr (numRecs, l_hskp.mass_binn_configNParms)
   tof_subsys_config = fltarr (numRecs, l_hskp.tof_subsys_configNParms)
   hvps_config       = fltarr (numRecs, l_hskp.hvps_configNParms)
   cmd_n_pkt_stats   = fltarr (numRecs, l_hskp.cmd_n_pkt_statsNParms)
   hvps_vmons        = fltarr (numRecs, l_hskp.hvps_vmonsNParms)
   hvps_imons        = fltarr (numRecs, l_hskp.hvps_imonsNParms)
   lvps_vmons        = fltarr (numRecs, l_hskp.lvps_vmonsNParms)
   lvps_imons        = fltarr (numRecs, l_hskp.lvps_imonsNParms)
   temp_mons         = fltarr (numRecs, l_hskp.temp_monsNParms)
   sctime            = dblarr (numRecs)

   start_array = replicate ({UDF_TIMES_STRUCT}, numRecs)
   stop_array  = replicate ({UDF_TIMES_STRUCT}, numRecs)


   while not udf_eof (fh) do begin

      d = udf_read (fh)

      bt = l_GetTime (d.btime, doy, hr, min, sec)

      start_array[indx].year = d.btime.year
      start_array[indx].doy  = d.btime.doy
      start_array[indx].hour = d.btime.hour
      start_array[indx].min  = d.btime.min
      start_array[indx].sec  = d.btime.sec
      start_array[indx].msec = d.btime.msec

      stop_array[indx].year = d.etime.year
      stop_array[indx].doy  = d.etime.doy
      stop_array[indx].hour = d.etime.hour
      stop_array[indx].min  = d.etime.min
      stop_array[indx].sec  = d.etime.sec
      stop_array[indx].msec = d.etime.msec

      hvps_sys_errs[indx, *]     = d.hvps_system_errors[*]
      lvps_sys_errs[indx, *]     = d.lvps_system_errors[*]
      sys_lev_errs[indx, *]      = d.system_level_errors[*] 
      sys_lev_stats[indx, *]     = d.system_level_status[*] 
      data_pkt_sel_stat[indx, *] = d.data_packet_selection_status[*]
      mass_binn_config[indx, *]  = d.mass_binning_configuration[*] 
      tof_subsys_config[indx, *] = d.tof_subsystem_configuration[*]
      hvps_config[indx, *]       = d.hvps_configuration[*] 
      cmd_n_pkt_stats[indx, *]   = d.command_and_packet_status[*] 
      hvps_vmons[indx, *]        = d.hvps_voltage_monitors[*]
      hvps_imons[indx, *]        = d.hvps_current_monitors[*]
      lvps_vmons[indx, *]        = d.lvps_voltage_monitors[*]
      lvps_imons[indx, *]        = d.lvps_current_monitors[*]
      temp_mons[indx, *]         = d.temperature_monitors[*]
      sctime[indx]               = bt

      indx = indx + 1L

   endwhile 

   udf_close, fh

   ptr = ptr_new (sctime[0:indx-1], /NO_COPY)
   l_hskp.timePtr = ptr

   ptr = ptr_new (start_array[0:indx-1], /NO_COPY)
   l_hskp.startUdfTimePtr = ptr

   ptr = ptr_new (stop_array[0:indx-1], /NO_COPY)
   l_hskp.stopUdfTimePtr = ptr

   ptr = ptr_new (hvps_sys_errs[0:indx-1, *], /NO_COPY)
   l_hskp.hvps_sys_errsPtr = ptr

   ptr = ptr_new (lvps_sys_errs[0:indx-1, *], /NO_COPY)
   l_hskp.lvps_sys_errsPtr = ptr

   ptr = ptr_new (sys_lev_errs[0:indx-1, *], /NO_COPY)
   l_hskp.sys_lev_errsPtr = ptr

   ptr = ptr_new (sys_lev_stats[0:indx-1, *], /NO_COPY)
   l_hskp.sys_lev_statsPtr = ptr

   ptr = ptr_new (data_pkt_sel_stat[0:indx-1, *], /NO_COPY)
   l_hskp.data_pkt_sel_statPtr = ptr

   ptr = ptr_new (mass_binn_config[0:indx-1, *], /NO_COPY)
   l_hskp.mass_binn_configPtr = ptr

   ptr = ptr_new (tof_subsys_config[0:indx-1, *], /NO_COPY)
   l_hskp.tof_subsys_configPtr = ptr

   ptr = ptr_new (hvps_config[0:indx-1, *], /NO_COPY)
   l_hskp.hvps_configPtr = ptr

   ptr = ptr_new (cmd_n_pkt_stats[0:indx-1, *], /NO_COPY)
   l_hskp.cmd_n_pkt_statsPtr = ptr

   ptr = ptr_new (hvps_vmons[0:indx-1, *], /NO_COPY)
   l_hskp.hvps_vmonsPtr = ptr

   ptr = ptr_new (hvps_imons[0:indx-1, *], /NO_COPY)
   l_hskp.hvps_imonsPtr = ptr

   ptr = ptr_new (lvps_vmons[0:indx-1, *], /NO_COPY)
   l_hskp.lvps_vmonsPtr = ptr
 
   ptr = ptr_new (lvps_imons[0:indx-1, *], /NO_COPY)
   l_hskp.lvps_imonsPtr = ptr

   ptr = ptr_new (temp_mons[0:indx-1, *], /NO_COPY)
   l_hskp.temp_monsPtr = ptr
  

   RETURN, SUCCESS


END



;-------------------------------------------------------------------------------
;  Function: l_HskpGetData
;  Description:
;    Determines the number of reads and calls routine to read HSKP data.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  SUCCESS                       successful completion
;  FAILURE                       failure to complete
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_setup               struct        I        generic setup parameters
;  l_setup_file          string        I        setup filename
;  l_pltpos              struct        I        generic plot parameters
;  l_hskp_setup          struct        I        hskp plot parameters
;  l_hskp                struct*       I        hskp data parameters
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;     PLOTDEFS         MAX_PLOTS        int         G         max # plots per page
;                      MAX_WEB_SPINS    int         G         max # spins per plot
;                                                             when using web interface
;                      NO_CNTS          double      G         no counts indicator
;                      NO_DATA          double      G         no data indicator
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;-------------------------------------------------------------------------------

FUNCTION l_HskpGetData, l_setup, l_setup_file, l_pltpos, $
l_hskp_setup, l_hskp 
COMMON PLOTDEFS, MAX_PLOTS, MAX_WEB_SPINS, NO_CNTS, NO_DATA
COMMON ERROR_TYPES, SUCCESS, FAILURE, WARNING, INFO

 
   inst    = 'LENA'
   expr    = 'LENAHSKP' 
   vinst   = 'IMLHSKP'

   l_HskpReadSetup, l_hskp_setup, l_setup_file

   data_source = ' '
   numRecs = 0L


   ;---------------------------------------------
   ; create a null pointer
   ; define housekeeping storage arrays
   ;---------------------------------------------
   ptr = ptr_new ()

   l_hskp = {                       $
      timePtr                      : ptr,  $
      filledTimePtr                : ptr,  $
      startUdfTimePtr              : ptr, $
      filledStartUdfTimePtr        : ptr, $
      stopUdfTimePtr               : ptr, $
      hvps_sys_errsPtr             : ptr, $
      filled_hvps_sys_errsPtr      : ptrarr(11), $
      hvps_sys_errsNParms          : 11L, $
      lvps_sys_errsPtr             : ptr, $
      filled_lvps_sys_errsPtr      : ptrarr(10), $
      lvps_sys_errsNParms          : 10L, $
      sys_lev_errsPtr              : ptr, $
      filled_sys_lev_errsPtr       : ptrarr(8), $
      sys_lev_errsNParms           : 8L,  $
      sys_lev_statsPtr             : ptr, $
      filled_sys_lev_statsPtr      : ptrarr(4), $
      sys_lev_statsNParms          : 4L,  $
      data_pkt_sel_statPtr         : ptr, $
      filled_data_pkt_sel_statPtr  : ptrarr(15), $
      data_pkt_sel_statNParms      : 15L, $
      mass_binn_configPtr          : ptr, $
      filled_mass_binn_configPtr   : ptrarr(4), $
      mass_binn_configNParms       : 4L,  $
      tof_subsys_configPtr         : ptr, $
      filled_tof_subsys_configPtr  : ptrarr(3), $
      tof_subsys_configNParms      : 3L,  $
      hvps_configPtr               : ptr, $
      filled_hvps_configPtr        : ptrarr(14), $
      hvps_configNParms            : 14L, $
      cmd_n_pkt_statsPtr           : ptr, $
      filled_cmd_n_pkt_statsPtr    : ptrarr(6), $
      cmd_n_pkt_statsNParms        : 6L,  $
      hvps_vmonsPtr                : ptr, $
      filled_hvps_vmonsPtr         : ptrarr(6), $
      hvps_vmonsNParms             : 6L,  $
      hvps_imonsPtr                : ptr, $
      filled_hvps_imonsPtr         : ptrarr(5), $
      hvps_imonsNParms             : 5L,  $
      lvps_vmonsPtr                : ptr, $
      filled_lvps_vmonsPtr         : ptrarr(5), $
      lvps_vmonsNParms             : 5L,  $
      lvps_imonsPtr                : ptr, $
      filled_lvps_imonsPtr         : ptrarr(5), $
      lvps_imonsNParms             : 5L,  $
      temp_monsPtr                 : ptr, $
      filled_temp_monsPtr          : ptrarr(8), $
      temp_monsNParms              : 8L  $
   }


   start_tme = l_setup.l_start_dtime
   stop_tme = l_setup.l_stop_dtime

   ;--------------------------------------------------------------
   ; estimate the number of records to read between user start
   ; and stop time
   ;--------------------------------------------------------------
   start_hskp_time      = ConvArrTimeToSecs(start_tme)
   stop_hskp_time       = ConvArrTimeToSecs(stop_tme)
   numRecs = round((stop_hskp_time - start_hskp_time) / 120.) + 20L


   status = l_HskpReadData (inst, expr, vinst,     $
                            start_tme, $
                            stop_tme,  $
                            l_hskp, numRecs, data_source)

   l_pltpos.data_source = data_source


   RETURN, status 

END


;-------------------------------------------------------------------------------
;  Procedure :  l_HskpReadSetup
;  Description:
;   Reads options from setup file specific to housekeeping data.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_hskp_setup          struct        I        hskp plot parameters
;  l_setup_file          string        I        setup filename
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;-------------------------------------------------------------------------------

PRO l_HskpReadSetup, l_hskp_setup, l_setup_file


   openr, inlun, /get_lun, l_setup_file

   l_hskp_setup = {                      $
       sw_version                : 'N'  ,$  
       sci_roi0_packet_status    : 'N'  ,$
       sci_roi1_packet_status    : 'N'  ,$
       sci_roi2_packet_status    : 'N'  ,$
       sci_roi3_packet_status    : 'N'  ,$  
       tof_bin1_start            : 'N'  ,$
       tof_bin1_stop             : 'N'  ,$
       tof_bin2_start            : 'N'  ,$
       tof_bin2_stop             : 'N'  ,$
       tof_start_cfd_level       : 'N'  ,$
       tof_stop_cfd_level        : 'N'  ,$
       hvps_mcp_start_cmd        : 'N'  ,$
       hvps_mcp_stop_cmd         : 'N'  ,$
       hvps_collp_cmd            : 'N'  ,$
       hvps_colln_cmd            : 'N'  ,$
       hvps_optics_cmd           : 'N'  ,$
       hvps_optics_steering_cmd  : 'N'  ,$
       hvps_mcp_start_vmon       : 'N'  ,$
       hvps_mcp_stop_vmon        : 'N'  ,$
       hvps_collp_vmon           : 'N'  ,$
       hvps_colln_vmon           : 'N'  ,$
       hvps_optics_vmon          : 'N'  ,$
       hvps_steering_vmon        : 'N'  ,$
       hvps_mcp_start_imon       : 'N'  ,$
       hvps_mcp_stop_imon        : 'N'  ,$
       hvps_collp_imon           : 'N'  ,$
       hvps_colln_imon           : 'N'  ,$
       hvps_optics_imon          : 'N'  ,$
       lvps_15vp_vmon            : 'N'  ,$
       lvps_15vn_vmon            : 'N'  ,$
       lvps_5vp_vmon             : 'N'  ,$
       lvps_5vn_vmon             : 'N'  ,$
       temp0                     : 'N'  ,$
       temp1                     : 'N'  ,$
       nplots                    :  0L   $
   }




   ;----------------------------------------
   ; read hskp data setup parameters
   ;----------------------------------------

   l_hskp_setup.nplots = 0L

   line = ' '

   while not EOF(inlun) do begin		;While there is text left, output it.
   
     readf, inlun, line
     ;line = STRUPCASE(line)

     IF (STRMID(line, 0, 1) ne ';') and (STRLEN(line) gt 0) then begin
        parts = STR_SEP(line, ' ')
     if (n_elements (parts) gt 2) then begin
     case parts[0] of
        'sw_version'                : begin
          l_hskp_setup.sw_version = STRUPCASE (parts[2])
		  if STRUPCASE(parts[2]) eq 'Y' then $
		    l_hskp_setup.nplots = l_hskp_setup.nplots + 1L
        end

        'sci_roi0_packet_status'    : begin
          l_hskp_setup.sci_roi0_packet_status = STRUPCASE (parts[2])
		  if STRUPCASE(parts[2]) eq 'Y' then $
		    l_hskp_setup.nplots = l_hskp_setup.nplots + 1L
        end

        'sci_roi1_packet_status'    : begin
          l_hskp_setup.sci_roi1_packet_status = STRUPCASE (parts[2])
		  if STRUPCASE(parts[2]) eq 'Y' then $
		    l_hskp_setup.nplots = l_hskp_setup.nplots + 1L
         end

        'sci_roi2_packet_status'    : begin
          l_hskp_setup.sci_roi2_packet_status = STRUPCASE (parts[2])
		  if STRUPCASE(parts[2]) eq 'Y' then $
		    l_hskp_setup.nplots = l_hskp_setup.nplots + 1L
         end

        'sci_roi3_packet_status'    : begin
          l_hskp_setup.sci_roi3_packet_status = STRUPCASE (parts[2])
		  if STRUPCASE(parts[2]) eq 'Y' then $
		    l_hskp_setup.nplots = l_hskp_setup.nplots + 1L
         end

        'tof_bin1_start'            : begin 
          l_hskp_setup.tof_bin1_start = STRUPCASE (parts[2])
		  if STRUPCASE(parts[2]) eq 'Y' then $
		    l_hskp_setup.nplots = l_hskp_setup.nplots + 1L
         end

        'tof_bin1_stop'             : begin
          l_hskp_setup.tof_bin1_stop = STRUPCASE (parts[2])
		  if STRUPCASE(parts[2]) eq 'Y' then $
		    l_hskp_setup.nplots = l_hskp_setup.nplots + 1L
         end

        'tof_bin2_start'            : begin
          l_hskp_setup.tof_bin2_start = STRUPCASE (parts[2])
		  if STRUPCASE(parts[2]) eq 'Y' then $
		    l_hskp_setup.nplots = l_hskp_setup.nplots + 1L
         end

        'tof_bin2_stop'             : begin
          l_hskp_setup.tof_bin2_stop = STRUPCASE (parts[2])
		  if STRUPCASE(parts[2]) eq 'Y' then $
		    l_hskp_setup.nplots = l_hskp_setup.nplots + 1L
         end

        'tof_start_cfd_level'       : begin
          l_hskp_setup.tof_start_cfd_level = STRUPCASE (parts[2])
		  if STRUPCASE(parts[2]) eq 'Y' then $
		    l_hskp_setup.nplots = l_hskp_setup.nplots + 1L
         end

        'tof_stop_cfd_level'        : begin
          l_hskp_setup.tof_stop_cfd_level = STRUPCASE (parts[2])
		  if STRUPCASE(parts[2]) eq 'Y' then $
		    l_hskp_setup.nplots = l_hskp_setup.nplots + 1L
         end

        'hvps_mcp_start_cmd'        : begin
          l_hskp_setup.hvps_mcp_start_cmd = STRUPCASE (parts[2])
		  if STRUPCASE(parts[2]) eq 'Y' then $
		    l_hskp_setup.nplots = l_hskp_setup.nplots + 1L
         end

        'hvps_mcp_stop_cmd'         : begin
          l_hskp_setup.hvps_mcp_stop_cmd = STRUPCASE (parts[2])
		  if STRUPCASE(parts[2]) eq 'Y' then $
		    l_hskp_setup.nplots = l_hskp_setup.nplots + 1L
         end

        'hvps_collp_cmd'            : begin   
          l_hskp_setup.hvps_collp_cmd = STRUPCASE (parts[2])
		  if STRUPCASE(parts[2]) eq 'Y' then $
		    l_hskp_setup.nplots = l_hskp_setup.nplots + 1L
         end

        'hvps_colln_cmd'            : begin 
          l_hskp_setup.hvps_colln_cmd = STRUPCASE (parts[2])
		  if STRUPCASE(parts[2]) eq 'Y' then $
		    l_hskp_setup.nplots = l_hskp_setup.nplots + 1L
         end

        'hvps_optics_cmd'           : begin 
          l_hskp_setup.hvps_optics_cmd = STRUPCASE (parts[2])
		  if STRUPCASE(parts[2]) eq 'Y' then $
		    l_hskp_setup.nplots = l_hskp_setup.nplots + 1L
         end

        'hvps_optics_steering_cmd' : begin
          l_hskp_setup.hvps_optics_steering_cmd = STRUPCASE (parts[2])
		  if STRUPCASE(parts[2]) eq 'Y' then $
		    l_hskp_setup.nplots = l_hskp_setup.nplots + 1L
         end

        'hvps_mcp_start_vmon' : begin
          l_hskp_setup.hvps_mcp_start_vmon = STRUPCASE(parts[2])
		  if STRUPCASE(parts[2]) eq 'Y' then $
		    l_hskp_setup.nplots = l_hskp_setup.nplots + 1L
        end

        'hvps_mcp_stop_vmon'  : begin 
          l_hskp_setup.hvps_mcp_stop_vmon = STRUPCASE(parts[2])
		  if STRUPCASE(parts[2]) eq 'Y' then $
		    l_hskp_setup.nplots = l_hskp_setup.nplots + 1L
        end

        'hvps_collp_vmon'     : begin 
          l_hskp_setup.hvps_collp_vmon = STRUPCASE(parts[2])
		  if STRUPCASE(parts[2]) eq 'Y' then $
		    l_hskp_setup.nplots = l_hskp_setup.nplots + 1L
        end

        'hvps_colln_vmon'     : begin
          l_hskp_setup.hvps_colln_vmon = STRUPCASE(parts[2])
		  if STRUPCASE(parts[2]) eq 'Y' then $
		    l_hskp_setup.nplots = l_hskp_setup.nplots + 1L
        end

        'hvps_optics_vmon'    : begin 
          l_hskp_setup.hvps_optics_vmon = STRUPCASE(parts[2])
		  if STRUPCASE(parts[2]) eq 'Y' then $
		    l_hskp_setup.nplots = l_hskp_setup.nplots + 1L
        end

        'hvps_steering_vmon' : begin 
          l_hskp_setup.hvps_steering_vmon = STRUPCASE(parts[2])
		  if STRUPCASE(parts[2]) eq 'Y' then $
		    l_hskp_setup.nplots = l_hskp_setup.nplots + 1L
        end

        'hvps_mcp_start_imon': begin
          l_hskp_setup.hvps_mcp_start_imon = STRUPCASE(parts[2])
          if STRUPCASE(parts[2]) eq 'Y' then $
            l_hskp_setup.nplots = l_hskp_setup.nplots + 1L
         end

        'hvps_mcp_stop_imon': begin
          l_hskp_setup.hvps_mcp_stop_imon = STRUPCASE(parts[2])
          if STRUPCASE(parts[2]) eq 'Y' then $
            l_hskp_setup.nplots = l_hskp_setup.nplots + 1L
         end

        'hvps_collp_imon': begin
          l_hskp_setup.hvps_collp_imon = STRUPCASE(parts[2])
          if STRUPCASE(parts[2]) eq 'Y' then $
            l_hskp_setup.nplots = l_hskp_setup.nplots + 1L
         end

        'hvps_colln_imon': begin
          l_hskp_setup.hvps_colln_imon = STRUPCASE(parts[2])
          if STRUPCASE(parts[2]) eq 'Y' then $
            l_hskp_setup.nplots = l_hskp_setup.nplots + 1L
         end

        'hvps_optics_imon': begin
          l_hskp_setup.hvps_optics_imon = STRUPCASE(parts[2])
          if STRUPCASE(parts[2]) eq 'Y' then $
            l_hskp_setup.nplots = l_hskp_setup.nplots + 1L
         end

        'lvps_15vp_vmon' : begin
          l_hskp_setup.lvps_15vp_vmon= STRUPCASE(parts[2])
		  if STRUPCASE(parts[2]) eq 'Y' then $
		    l_hskp_setup.nplots = l_hskp_setup.nplots + 1L
        end

        'lvps_15vn_vmon' : begin
          l_hskp_setup.lvps_15vn_vmon= STRUPCASE(parts[2])
		  if STRUPCASE(parts[2]) eq 'Y' then $
		    l_hskp_setup.nplots = l_hskp_setup.nplots + 1L
        end

        'lvps_5vp_vmon' : begin
          l_hskp_setup.lvps_5vp_vmon= STRUPCASE(parts[2])
		  if STRUPCASE(parts[2]) eq 'Y' then $
		    l_hskp_setup.nplots = l_hskp_setup.nplots + 1L
        end

        'lvps_5vn_vmon' : begin
          l_hskp_setup.lvps_5vn_vmon= STRUPCASE(parts[2])
		  if STRUPCASE(parts[2]) eq 'Y' then $
		    l_hskp_setup.nplots = l_hskp_setup.nplots + 1L
        end

        'temp0'                 : begin
          l_hskp_setup.temp0 = STRUPCASE(parts[2])
		  if STRUPCASE(parts[2]) eq 'Y' then $
		    l_hskp_setup.nplots = l_hskp_setup.nplots + 1L
        end

        'temp1'                 : begin
          l_hskp_setup.temp1 = STRUPCASE(parts[2])
		  if STRUPCASE(parts[2]) eq 'Y' then $
		    l_hskp_setup.nplots = l_hskp_setup.nplots + 1L
        end

       else: begin
;	   printf, G_LOG, parts[0], ' : Unknown option', strlen(line) 
	   end
       endcase				;End of CASE statement.
       endif
    endif
  endwhile



  close, inlun
  free_lun, inlun

RETURN
END


;-------------------------------------------------------------------------------
;  Procedure :  l_HskpShowSetup
;  Description:
;    Used for debug purposes.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_hskp_setup          struct        I        hskp plot setup parameters
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;-------------------------------------------------------------------------------

PRO l_HskpShowSetup, l_hskp_setup

   print, '____LENA Hskp Data Setup Parameters'
   print, 'hvps_optics_steering_verr : ', l_hskp_setup.hvps_optics_steering_verr
   print, 'hvps_optics_steering_cmd  : ', l_hskp_setup.hvps_optics_steering_cmd
   print, 'hvps_mcp_start_vmon       : ', l_hskp_setup.hvps_mcp_start_vmon
   print, 'hvps_mcp_stop_vmon        : ', l_hskp_setup.hvps_mcp_stop_vmon
   print, 'hvps_collp_vmon           : ', l_hskp_setup.hvps_collp_vmon
   print, 'hvps_colln_vmon           : ', l_hskp_setup.hvps_colln_vmon
   print, 'hvps_optics_vmon          : ', l_hskp_setup.hvps_optics_vmon
   print, 'hvps_steering_vmon        : ', l_hskp_setup.hvps_steering_vmon
   print, 'temp0                     : ', l_hskp_setup.temp0
   print, 'temp1                     : ', l_hskp_setup.temp1
   print, 'nplots                    : ', l_hskp_setup.nplots

   RETURN

END



;-------------------------------------------------------------------------------
;  Procedure: l_HskpDumpData
;  Description:
;    Generate a text dump of the data.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_setup               struct        I        generic setup parameters
;  l_pltpos              struct        I        generic plot parameters
;  l_hskp                struct        I        hskp data parm/attributes
;  l_hskp_setup          struct        I        hskp setup/plot parameters
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;     PLOTDEFS         MAX_PLOTS        int         G         max # plots per page
;                      MAX_WEB_SPINS    int         G         max # spins per plot
;                                                             when using web interface
;                      NO_CNTS          double      G         no counts indicator
;                      NO_DATA          double      G         no data indicator
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;-------------------------------------------------------------------------------

PRO l_HskpDumpData, l_setup, l_pltpos, l_hskp, l_hskp_setup
COMMON PLOTDEFS, MAX_PLOTS, MAX_WEB_SPINS, NO_CNTS, NO_DATA

   ;-------------------------------
   ; create text dump filename
   ;-------------------------------
   l_setup.l_filedesc = 'hsk'
   l_GenFileName, l_setup, '0'
   filename = l_setup.l_prod_dst + '/' + l_setup.l_filename + '.dmp'

   ;-------------------------------
   ; open text dump file
   ;-------------------------------
   openw, ofd, filename, /get_lun

   ;-------------------------------
   ; write standard header info
   ; to text file
   ;-------------------------------
   l_pltpos.dmp_hdr_title = 'Housekeeping Text Dump'
   l_DumpHdrInfo, ofd, l_setup, l_pltpos, filename

   printf, ofd, 'No Data Value   = ', NO_DATA
   printf, ofd, 'No Counts Value = ', NO_CNTS
   printf, ofd, ' '

   if (l_hskp_setup.hvps_mcp_start_vmon eq 'Y') then begin
      printf, ofd, 'Hvps MCP Start Vmon'
      printf, ofd, (*l_hskp.filled_hvps_vmonsPtr[0]) 
      printf, ofd, ' '
   endif

   if (l_hskp_setup.hvps_mcp_stop_vmon eq 'Y') then begin
      printf, ofd, 'Hvps MCP Stop Vmon'
      printf, ofd, (*l_hskp.filled_hvps_vmonsPtr[1])
      printf, ofd, ' '
   endif

   if (l_hskp_setup.hvps_collp_vmon eq 'Y') then begin
      printf, ofd, 'Hvps CollP Vmon'
      printf, ofd, (*l_hskp.filled_hvps_vmonsPtr[2]) 
      printf, ofd, ' '
   endif

   if (l_hskp_setup.hvps_colln_vmon eq 'Y') then begin
      printf, ofd, 'Hvps CollN Vmon'
      printf, ofd, (*l_hskp.filled_hvps_vmonsPtr[3]) 
      printf, ofd, ' '
   endif

   if (l_hskp_setup.hvps_optics_vmon eq 'Y') then begin
      printf, ofd, 'Hvps Optics Vmon'
      printf, ofd, (*l_hskp.filled_hvps_vmonsPtr[4])
      printf, ofd, ' '
   endif

   if (l_hskp_setup.hvps_steering_vmon eq 'Y') then begin
      printf, ofd, 'Hvps Steering Vmon'
      printf, ofd, (*l_hskp.filled_hvps_vmonsPtr[5]) 
      printf, ofd, ' '
   endif

   if (l_hskp_setup.hvps_mcp_start_imon eq 'Y') then begin
      printf, ofd, 'Hvps MCP Start Imon'
      printf, ofd, (*l_hskp.filled_hvps_imonsPtr[0]) 
      printf, ofd, ' '
   endif

   if (l_hskp_setup.hvps_mcp_stop_imon eq 'Y') then begin
      printf, ofd, 'Hvps MCP Stop Imon'
      printf, ofd, (*l_hskp.filled_hvps_imonsPtr[1]) 
      printf, ofd, ' '
   endif

   if (l_hskp_setup.hvps_collp_imon eq 'Y') then begin
      printf, ofd, 'Hvps CollP Imon'
      printf, ofd, (*l_hskp.filled_hvps_imonsPtr[2])
      printf, ofd, ' '
   endif

   if (l_hskp_setup.hvps_colln_imon eq 'Y') then begin
      printf, ofd, 'Hvps CollN Imon'
      printf, ofd, (*l_hskp.filled_hvps_imonsPtr[3]) 
      printf, ofd, ' '
   endif

   if (l_hskp_setup.hvps_optics_imon eq 'Y') then begin
      printf, ofd, 'Hvps Optics Imon'
      printf, ofd, (*l_hskp.filled_hvps_imonsPtr[4]) 
      printf, ofd, ' '
   endif

   if (l_hskp_setup.lvps_15vp_vmon eq 'Y') then begin
      printf, ofd, 'Lvps 15vp Vmon'
      printf, ofd, (*l_hskp.filled_lvps_vmonsPtr[1]) 
      printf, ofd, ' '
   endif

   if (l_hskp_setup.lvps_15vn_vmon eq 'Y') then begin
      printf, ofd, 'Lvps 15vn Vmon'
      printf, ofd, (*l_hskp.filled_lvps_vmonsPtr[2])
      printf, ofd, ' '
   endif

   if (l_hskp_setup.lvps_5vp_vmon eq 'Y') then begin
      printf, ofd, 'Lvps 5vp Vmon'
      printf, ofd, (*l_hskp.filled_lvps_vmonsPtr[3])
      printf, ofd, ' '
   endif

   if (l_hskp_setup.lvps_5vn_vmon eq 'Y') then begin
      printf, ofd, 'Lvps 5vn Vmon'
      printf, ofd, (*l_hskp.filled_lvps_vmonsPtr[4]) 
      printf, ofd, ' '
   endif


   if (l_hskp_setup.temp0 eq 'Y') then begin
      printf, ofd, 'Temperature 0'
      printf, ofd, (*l_hskp.filled_temp_monsPtr[0]) 
      printf, ofd, ' '
   endif

   if (l_hskp_setup.temp1 eq 'Y') then begin
      printf, ofd, 'Temperature 1'
      printf, ofd, (*l_hskp.filled_temp_monsPtr[1])
      printf, ofd, ' '
   endif

print, 'CLOSE l_HskpDumpData FN: ', filename
   close, ofd
   free_lun, ofd

RETURN
END


;-------------------------------------------------------------------------------
;  Procedure: l_HskpHndlMissData
;
;  Description: Fill data gaps in the HSKP data. 
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_hskp_setup          struct        I        housekeeping processing & plotting parameters
;  l_hskp                struct *      I        pointers to HSKP data structures
;  l_setup               struct        I        general setup parameters
;  l_oa                  struct *      I        OA processing & plotting parameters
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------

PRO l_HskpHndlMissData, l_hskp_setup, l_hskp, l_setup, l_oa


   linear = 1L   ; data_format
   no_extra_data = 0
   adjFlag = 'NO'


   ;-------------------------------------------------------------------
   ; HVPS Voltage Monitors  (IMON)
   ;-------------------------------------------------------------------
   if (l_hskp_setup.hvps_collp_imon eq 'Y') then begin

       nspins = n_elements ((*l_hskp.hvps_imonsPtr)[*,2])

       l_HndlMissData, (*l_hskp.timePtr), $
                       filled_time, $
                       nspins, $
                       (*l_hskp.hvps_imonsPtr)[*,2], $
                       filled_data, $
                       linear,  $
                       center_sector, $
                       l_setup , $
                       adjFlag, $
                       l_oa, $
                       (*l_hskp.startUdfTimePtr), $
                       filled_start_udf_time, $
                       no_extra_data, $
                       l_setup.l_spin_ang_zero

       ptr = ptr_new (filled_time, /NO_COPY)
       l_hskp.filledTimePtr = ptr
   
       ptr = ptr_new (filled_start_udf_time, /NO_COPY)
       l_hskp.filledStartUdfTimePtr = ptr

       filled_data = reform (filled_data, n_elements(filled_data), /overwrite)
       ptr = ptr_new (filled_data, /NO_COPY)
       l_hskp.filled_hvps_imonsPtr[2] = ptr 

       
   endif



   if (l_hskp_setup.hvps_colln_imon eq 'Y') then begin

       nspins = n_elements ((*l_hskp.hvps_imonsPtr)[*,3])

       l_HndlMissData, (*l_hskp.timePtr), $ 
                       filled_time, $
                       nspins, $
                       (*l_hskp.hvps_imonsPtr)[*,3], $
                       filled_data, $
                       linear, $
                       center_sector, $
                       l_setup, $
                       adjFlag, $
                       l_oa, $
                       (*l_hskp.startUdfTimePtr), $
                       filled_start_udf_time, $
                       no_extra_data, $
                       l_setup.l_spin_ang_zero

       ptr = ptr_new (filled_time, /NO_COPY)
       l_hskp.filledTimePtr = ptr
   
       ptr = ptr_new (filled_start_udf_time, /NO_COPY)
       l_hskp.filledStartUdfTimePtr = ptr


       filled_data = reform (filled_data, n_elements(filled_data), /overwrite)
       ptr = ptr_new (filled_data, /NO_COPY)
       l_hskp.filled_hvps_imonsPtr[3] = ptr

   endif



   ;-------------------------------------------------------------------
   ;  Status - Temperatures
   ;  (-20, 60), (10, 30)
   ;-------------------------------------------------------------------


   if (l_hskp_setup.temp0 eq 'Y') then begin

      nspins = n_elements ((*l_hskp.temp_monsPtr)[*,0])

      ohskp = (*l_hskp.temp_monsPtr)[*,0] 
      hskp  =  l_ConvHkTemp (ohskp)

      l_HndlMissData, (*l_hskp.timePtr), $
                      filled_time, $
                      nspins, $
                      hskp, $
                      filled_data, $
                      linear, $
                      center_sector, $
                      l_setup, $
                      adjFlag, $
                      l_oa, $
                      (*l_hskp.startUdfTimePtr), $
                      filled_start_udf_time, $
                      no_extra_data, $
                      l_setup.l_spin_ang_zero


       ptr = ptr_new (filled_time, /NO_COPY)
       l_hskp.filledTimePtr = ptr
   
       ptr = ptr_new (filled_start_udf_time, /NO_COPY)
       l_hskp.filledStartUdfTimePtr = ptr


       filled_data = reform (filled_data, n_elements(filled_data), /overwrite)
       ptr = ptr_new (filled_data, /NO_COPY)
       l_hskp.filled_temp_monsPtr[0] = ptr 

   endif


   if (l_hskp_setup.temp1 eq 'Y') then begin

      nspins = n_elements ((*l_hskp.temp_monsPtr)[*,1])

      ohskp = (*l_hskp.temp_monsPtr)[*,1] 
      hskp  =  l_ConvHkTemp (ohskp) 

      l_HndlMissData, (*l_hskp.timePtr), $
                      filled_time, $
                      nspins, $
                      hskp,  $
                      filled_data, $
                      linear, $
                      center_sector, $
                      l_setup, $
                      adjFlag, $
                      l_oa, $
                      (*l_hskp.startUdfTimePtr), $
                      filled_start_udf_time, $
                      no_extra_data, $
                      l_setup.l_spin_ang_zero

       ptr = ptr_new (filled_time, /NO_COPY)
       l_hskp.filledTimePtr = ptr
   
       ptr = ptr_new (filled_start_udf_time, /NO_COPY)
       l_hskp.filledStartUdfTimePtr = ptr


       filled_data = reform (filled_data, n_elements(filled_data), /overwrite)
       ptr = ptr_new (filled_data, /NO_COPY)
       l_hskp.filled_temp_monsPtr[1] = ptr 


   endif

   ;-------------------------------------------------------------------
   ; HVPS Voltage Monitors  (VMON)
   ;-------------------------------------------------------------------

   if (l_hskp_setup.hvps_mcp_start_vmon eq 'Y') then begin

       nspins = n_elements ((*l_hskp.hvps_vmonsPtr)[*,0])

       l_HndlMissData, (*l_hskp.timePtr), $
                       filled_time,$
                       nspins, $
                       (*l_hskp.hvps_vmonsPtr)[*,0], $
                       filled_data, $
                       linear, $
                       center_sector, $
                       l_setup, $
                       adjFlag, $
                       l_oa, $
                       (*l_hskp.startUdfTimePtr), $
                       filled_start_udf_time, $
                       no_extra_data, $
                       l_setup.l_spin_ang_zero

       ptr = ptr_new (filled_time, /NO_COPY)
       l_hskp.filledTimePtr = ptr
   
       ptr = ptr_new (filled_start_udf_time, /NO_COPY)
       l_hskp.filledStartUdfTimePtr = ptr

       filled_data = reform (filled_data, n_elements(filled_data), /overwrite)
       ptr = ptr_new (filled_data, /NO_COPY)
       l_hskp.filled_hvps_vmonsPtr[0] =  ptr

   endif


   if (l_hskp_setup.hvps_mcp_stop_vmon eq 'Y') then begin

       nspins = n_elements ((*l_hskp.hvps_vmonsPtr)[*,1])

       l_HndlMissData, (*l_hskp.timePtr), $
                       filled_time,$
                       nspins, $
                       (*l_hskp.hvps_vmonsPtr)[*,1], $
                       filled_data, $
                       linear, $
                       center_sector, $
                       l_setup, $
                       adjFlag, $
                       l_oa, $
                       (*l_hskp.startUdfTimePtr), $
                       filled_start_udf_time, $
                       no_extra_data, $
                       l_setup.l_spin_ang_zero

       ptr = ptr_new (filled_time, /NO_COPY)
       l_hskp.filledTimePtr = ptr
   
       ptr = ptr_new (filled_start_udf_time, /NO_COPY)
       l_hskp.filledStartUdfTimePtr = ptr

       filled_data = reform (filled_data, n_elements(filled_data), /overwrite)
       ptr = ptr_new (filled_data, /NO_COPY)
       l_hskp.filled_hvps_vmonsPtr[1] =  ptr

   endif


   if (l_hskp_setup.hvps_optics_vmon eq 'Y') then begin

       nspins = n_elements ((*l_hskp.hvps_vmonsPtr)[*,4])

       l_HndlMissData, (*l_hskp.timePtr), $
                       filled_time,$
                       nspins, $
                       (*l_hskp.hvps_vmonsPtr)[*,4], $
                       filled_data, $
                       linear, $
                       center_sector, $
                       l_setup, $
                       adjFlag, $
                       l_oa, $
                       (*l_hskp.startUdfTimePtr), $
                       filled_start_udf_time, $
                       no_extra_data, $
                       l_setup.l_spin_ang_zero

       ptr = ptr_new (filled_time, /NO_COPY)
       l_hskp.filledTimePtr = ptr
   
       ptr = ptr_new (filled_start_udf_time, /NO_COPY)
       l_hskp.filledStartUdfTimePtr = ptr

       filled_data = reform (filled_data, n_elements(filled_data), /overwrite)
       ptr = ptr_new (filled_data, /NO_COPY)
       l_hskp.filled_hvps_vmonsPtr[4] =  ptr

   endif


   if (l_hskp_setup.hvps_steering_vmon eq 'Y') then begin

       nspins = n_elements ((*l_hskp.hvps_vmonsPtr)[*,5])

       l_HndlMissData, (*l_hskp.timePtr), $
                       filled_time,$
                       nspins, $
                       (*l_hskp.hvps_vmonsPtr)[*,5], $
                       filled_data, $
                       linear, $
                       center_sector, $
                       l_setup, $
                       adjFlag, $
                       l_oa, $
                       (*l_hskp.startUdfTimePtr), $
                       filled_start_udf_time, $
                       no_extra_data, $
                       l_setup.l_spin_ang_zero

       ptr = ptr_new (filled_time, /NO_COPY)
       l_hskp.filledTimePtr = ptr
   
       ptr = ptr_new (filled_start_udf_time, /NO_COPY)
       l_hskp.filledStartUdfTimePtr = ptr

       filled_data = reform (filled_data, n_elements(filled_data), /overwrite)
       ptr = ptr_new (filled_data, /NO_COPY)
       l_hskp.filled_hvps_vmonsPtr[5] = ptr

   endif


   if (l_hskp_setup.hvps_collp_vmon eq 'Y') then begin

       nspins = n_elements ((*l_hskp.hvps_vmonsPtr)[*,2])

       l_HndlMissData, (*l_hskp.timePtr), $
                       filled_time,$
                       nspins, $
                       (*l_hskp.hvps_vmonsPtr)[*,2], $
                       filled_data, $
                       linear, $
                       center_sector, $
                       l_setup, $
                       adjFlag, $
                       l_oa, $
                       (*l_hskp.startUdfTimePtr), $
                       filled_start_udf_time, $
                       no_extra_data, $
                       l_setup.l_spin_ang_zero

       ptr = ptr_new (filled_time, /NO_COPY)
       l_hskp.filledTimePtr = ptr
   
       ptr = ptr_new (filled_start_udf_time, /NO_COPY)
       l_hskp.filledStartUdfTimePtr = ptr

       filled_data = reform (filled_data, n_elements(filled_data), /overwrite)
       ptr = ptr_new (filled_data, /NO_COPY)
       l_hskp.filled_hvps_vmonsPtr[2] = ptr 

   endif


   if (l_hskp_setup.hvps_colln_vmon eq 'Y') then begin

       nspins = n_elements ((*l_hskp.hvps_vmonsPtr)[*,3])

       l_HndlMissData, (*l_hskp.timePtr), $
                       filled_time,$
                       nspins, $
                       (*l_hskp.hvps_vmonsPtr)[*,3], $
                       filled_data, $
                       linear, $
                       center_sector, $
                       l_setup, $
                       adjFlag, $
                       l_oa, $
                       (*l_hskp.startUdfTimePtr), $
                       filled_start_udf_time, $
                       no_extra_data, $
                       l_setup.l_spin_ang_zero

       ptr = ptr_new (filled_time, /NO_COPY)
       l_hskp.filledTimePtr = ptr
   
       ptr = ptr_new (filled_start_udf_time, /NO_COPY)
       l_hskp.filledStartUdfTimePtr = ptr

       filled_data = reform (filled_data, n_elements(filled_data), /overwrite)
       ptr = ptr_new (filled_data, /NO_COPY)
       l_hskp.filled_hvps_vmonsPtr[3] = ptr

   endif

   ;-------------------------------------------------------------------
   ; HVPS Current Monitors (IMON)
   ;-------------------------------------------------------------------
   if (l_hskp_setup.hvps_mcp_start_imon eq 'Y') then begin

       nspins = n_elements ((*l_hskp.hvps_imonsPtr)[*,0])

       l_HndlMissData, (*l_hskp.timePtr), $
                       filled_time,$
                       nspins, $
                       (*l_hskp.hvps_imonsPtr)[*,0], $
                       filled_data, $
                       linear, $
                       center_sector, $
                       l_setup, $
                       adjFlag, $
                       l_oa, $
                       (*l_hskp.startUdfTimePtr), $
                       filled_start_udf_time, $
                       no_extra_data, $
                       l_setup.l_spin_ang_zero

       ptr = ptr_new (filled_time, /NO_COPY)
       l_hskp.filledTimePtr = ptr
   
       ptr = ptr_new (filled_start_udf_time, /NO_COPY)
       l_hskp.filledStartUdfTimePtr = ptr

       filled_data = reform (filled_data, n_elements(filled_data), /overwrite)
       ptr = ptr_new (filled_data, /NO_COPY)
       l_hskp.filled_hvps_imonsPtr[0] = ptr

   endif

   if (l_hskp_setup.hvps_mcp_stop_imon eq 'Y') then begin

       nspins = n_elements ((*l_hskp.hvps_imonsPtr)[*,1])

       l_HndlMissData, (*l_hskp.timePtr), $
                       filled_time,$
                       nspins, $
                       (*l_hskp.hvps_imonsPtr)[*,1], $
                       filled_data, $
                       linear, $
                       center_sector, $
                       l_setup, $
                       adjFlag, $
                       l_oa, $
                       (*l_hskp.startUdfTimePtr), $
                       filled_start_udf_time, $
                       no_extra_data, $
                       l_setup.l_spin_ang_zero

       ptr = ptr_new (filled_time, /NO_COPY)
       l_hskp.filledTimePtr = ptr
   
       ptr = ptr_new (filled_start_udf_time, /NO_COPY)
       l_hskp.filledStartUdfTimePtr = ptr

       filled_data = reform (filled_data, n_elements(filled_data), /overwrite)
       ptr = ptr_new (filled_data, /NO_COPY)
       l_hskp.filled_hvps_imonsPtr[1] = ptr 

 
   endif

   if (l_hskp_setup.hvps_optics_imon eq 'Y') then begin

       nspins = n_elements ((*l_hskp.hvps_imonsPtr)[*,4])

       l_HndlMissData, (*l_hskp.timePtr), $
                       filled_time,$
                       nspins, $
                       (*l_hskp.hvps_imonsPtr)[*,4], $
                       filled_data, $
                       linear, $
                       center_sector, $
                       l_setup, $
                       adjFlag, $
                       l_oa, $
                       (*l_hskp.startUdfTimePtr), $
                       filled_start_udf_time, $
                       no_extra_data, $
                       l_setup.l_spin_ang_zero

       ptr = ptr_new (filled_time, /NO_COPY)
       l_hskp.filledTimePtr = ptr
   
       ptr = ptr_new (filled_start_udf_time, /NO_COPY)
       l_hskp.filledStartUdfTimePtr = ptr

       filled_data = reform (filled_data, n_elements(filled_data), /overwrite)
       ptr = ptr_new (filled_data, /NO_COPY)
       l_hskp.filled_hvps_imonsPtr[4] = ptr 

   endif

   ;-------------------------------------------------------------------
   ; LVPS Voltage Monitors 
   ;-------------------------------------------------------------------
   if (l_hskp_setup.lvps_15vp_vmon eq 'Y') then begin

       nspins = n_elements ((*l_hskp.lvps_vmonsPtr)[*,1])

       l_HndlMissData, (*l_hskp.timePtr), $
                       filled_time,$
                       nspins, $
                       (*l_hskp.lvps_vmonsPtr)[*,1], $
                       filled_data, $
                       linear, $
                       center_sector, $
                       l_setup, $
                       adjFlag, $
                       l_oa, $
                       (*l_hskp.startUdfTimePtr), $
                       filled_start_udf_time, $
                       no_extra_data, $
                       l_setup.l_spin_ang_zero

       ptr = ptr_new (filled_time, /NO_COPY)
       l_hskp.filledTimePtr = ptr
   
       ptr = ptr_new (filled_start_udf_time, /NO_COPY)
       l_hskp.filledStartUdfTimePtr = ptr

       filled_data = reform (filled_data, n_elements(filled_data), /overwrite)
       ptr = ptr_new (filled_data, /NO_COPY)
       l_hskp.filled_lvps_vmonsPtr[1] = ptr 

   endif

   if (l_hskp_setup.lvps_15vn_vmon eq 'Y') then begin

       nspins = n_elements ((*l_hskp.lvps_vmonsPtr)[*,2])

       l_HndlMissData, (*l_hskp.timePtr), $
                       filled_time,$
                       nspins, $
                       (*l_hskp.lvps_vmonsPtr)[*,2], $
                       filled_data, $
                       linear, $
                       center_sector, $
                       l_setup, $
                       adjFlag, $
                       l_oa, $
                       (*l_hskp.startUdfTimePtr), $
                       filled_start_udf_time, $
                       no_extra_data, $
                       l_setup.l_spin_ang_zero

       ptr = ptr_new (filled_time, /NO_COPY)
       l_hskp.filledTimePtr = ptr
   
       ptr = ptr_new (filled_start_udf_time, /NO_COPY)
       l_hskp.filledStartUdfTimePtr = ptr

       filled_data = reform (filled_data, n_elements(filled_data), /overwrite)
       ptr = ptr_new (filled_data, /NO_COPY)
       l_hskp.filled_lvps_vmonsPtr[2] = ptr

   endif

   if (l_hskp_setup.lvps_5vp_vmon eq 'Y') then begin

       nspins = n_elements ((*l_hskp.lvps_vmonsPtr)[*,3])

       l_HndlMissData, (*l_hskp.timePtr), $
                       filled_time,$
                       nspins, $
                       (*l_hskp.lvps_vmonsPtr)[*,3], $
                       filled_data, $
                       linear, $
                       center_sector, $
                       l_setup, $
                       adjFlag, $
                       l_oa, $
                       (*l_hskp.startUdfTimePtr), $
                       filled_start_udf_time, $
                       no_extra_data,  $
                       l_setup.l_spin_ang_zero

       ptr = ptr_new (filled_time, /NO_COPY)
       l_hskp.filledTimePtr = ptr
   
       ptr = ptr_new (filled_start_udf_time, /NO_COPY)
       l_hskp.filledStartUdfTimePtr = ptr

       filled_data = reform (filled_data, n_elements(filled_data), /overwrite)
       ptr = ptr_new (filled_data, /NO_COPY)
       l_hskp.filled_lvps_vmonsPtr[3] = ptr 


   endif

   if (l_hskp_setup.lvps_5vn_vmon eq 'Y') then begin

       nspins = n_elements ((*l_hskp.lvps_vmonsPtr)[*,4])

       tme = (*l_hskp.timePtr)[*]
       hskp = (*l_hskp.lvps_vmonsPtr)[*,4] 

       l_HndlMissData, (*l_hskp.timePtr), $
                       filled_time,$
                       nspins, $
                       (*l_hskp.lvps_vmonsPtr)[*,4], $
                       filled_data, $
                       linear, $
                       center_sector, $
                       l_setup, $
                       adjFlag, $
                       l_oa, $
                       (*l_hskp.startUdfTimePtr), $
                       filled_start_udf_time, $
                       no_extra_data, $
                       l_setup.l_spin_ang_zero

       ptr = ptr_new (filled_time, /NO_COPY)
       l_hskp.filledTimePtr = ptr
   
       ptr = ptr_new (filled_start_udf_time, /NO_COPY)
       l_hskp.filledStartUdfTimePtr = ptr

       filled_data = reform (filled_data, n_elements(filled_data), /overwrite)
       ptr = ptr_new (filled_data, /NO_COPY)
       l_hskp.filled_lvps_vmonsPtr[4] = ptr

   endif

RETURN
END

;-------------------------------------------------------------------------------
;  Procedure :  l_HskpPlot
;
;  Description: Plot the data.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_hskp_setup          struct        I        hskp setup/plot parameters
;  l_hskp                struct        I        hskp data parameters
;  l_pltpos              struct        I        generic plot parameters
;  nplots                long          I        total number of plots to be 
;                                               plotted 
;  windx                 long[]        I        window index
;  pages                 long          I        total number of pages needed
;  plotted               long          I        number of plots plotted
;  l_setup               struct        I        generic setup parameters
;  l_oa                  struct        I        OA data parameters
;  l_rgb_table           struct        I        generic color table parameters 
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;     PLOTDEFS         MAX_PLOTS        int         G         max # plots per page
;                      MAX_WEB_SPINS    int         G         max # spins per plot
;                                                             when using web interface
;                      NO_CNTS          double      G         no counts indicator
;                      NO_DATA          double      G         no data indicator
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;-------------------------------------------------------------------------------

PRO l_HskpPlot, l_hskp_setup, l_hskp, l_pltpos, nplots, $
                windx, pages, plotted, l_setup, l_oa, $
                l_rgb_table 
COMMON PLOTDEFS, MAX_PLOTS, MAX_WEB_SPINS, NO_CNTS, NO_DATA


   l_setup.l_d_start_dtime[0] = (*l_hskp.startUdfTimePtr)[0].year
   l_setup.l_d_start_dtime[1] = (*l_hskp.startUdfTimePtr)[0].doy
   l_setup.l_d_start_dtime[2] = (*l_hskp.startUdfTimePtr)[0].hour
   l_setup.l_d_start_dtime[3] = (*l_hskp.startUdfTimePtr)[0].min
   l_setup.l_d_start_dtime[4] = (*l_hskp.startUdfTimePtr)[0].sec
   l_setup.l_d_start_dtime[5] = (*l_hskp.startUdfTimePtr)[0].msec

   max_ele = n_elements((*l_hskp.stopUdfTimePtr)[*])-1L

   l_setup.l_d_stop_dtime[0] = (*l_hskp.stopUdfTimePtr)[max_ele].year
   l_setup.l_d_stop_dtime[1] = (*l_hskp.stopUdfTimePtr)[max_ele].doy
   l_setup.l_d_stop_dtime[2] = (*l_hskp.stopUdfTimePtr)[max_ele].hour
   l_setup.l_d_stop_dtime[3] = (*l_hskp.stopUdfTimePtr)[max_ele].min
   l_setup.l_d_stop_dtime[4] = (*l_hskp.stopUdfTimePtr)[max_ele].sec
   l_setup.l_d_stop_dtime[5] = (*l_hskp.stopUdfTimePtr)[max_ele].msec


   l_DetMjMnTicks, l_setup.l_start_dtime, $
                   l_setup.l_stop_dtime, $
                   l_setup.l_d_start_dtime, $
                   l_setup.l_d_stop_dtime, $
                   l_pltpos


   ;-------------------------------------------------------------------
   ; HVPS Voltage Monitors  (IMON)
   ;-------------------------------------------------------------------
   if (l_hskp_setup.hvps_collp_imon eq 'Y') then begin

       l_PlotStuff, nplots, $
                    (*l_hskp.filledTimePtr), $
                    (*l_hskp.filled_hvps_imonsPtr[2]), $
                    min((*l_hskp.hvps_imonsPtr)[*,2])-1, $
                    max((*l_hskp.hvps_imonsPtr)[*,2])+1, $
                    'Imon', $
                    l_pltpos, $
                    plotted, $
                    l_oa, $
                    l_setup.l_start_dtime

      yctr = (l_pltpos.ypltmn + l_pltpos.ypltmx)/2.0
      xyouts, 0.02, yctr, 'HVPS CollP', size=l_pltpos.sm_font, $
              /NORMAL, alignment = 0.5, orientation=90.0

       l_pltpos.ypltmx = l_pltpos.ypltmn - l_pltpos.yspace
       l_pltpos.ypltmn = l_pltpos.ypltmx - l_pltpos.ydel
       plotted = plotted + 1

       l_ChckCompPg, plotted, nplots, pages, $
                     l_setup, windx, l_pltpos, l_rgb_table
   endif


   if (l_hskp_setup.hvps_colln_imon eq 'Y') then begin
       
       l_PlotStuff, nplots, $
                    (*l_hskp.filledTimePtr), $
                    (*l_hskp.filled_hvps_imonsPtr[3]), $
                    min((*l_hskp.hvps_imonsPtr)[*,3])-1, $
                    max((*l_hskp.hvps_imonsPtr)[*,3])+1, $
                    'Imon', $
                    l_pltpos, $
                    plotted, $
                    l_oa, $ 
                    l_setup.l_start_dtime

      yctr = (l_pltpos.ypltmn + l_pltpos.ypltmx)/2.0
      xyouts, 0.02, yctr, 'HVPS CollN', size=l_pltpos.sm_font, $
              /NORMAL, alignment = 0.5, orientation=90.0

      l_pltpos.ypltmx = l_pltpos.ypltmn - l_pltpos.yspace
      l_pltpos.ypltmn = l_pltpos.ypltmx - l_pltpos.ydel
      plotted = plotted + 1

      l_ChckCompPg, plotted, nplots, pages, $
                    l_setup, windx, l_pltpos, l_rgb_table
   endif



   ;-------------------------------------------------------------------
   ;  Status - Temperatures
   ;-------------------------------------------------------------------

   if (l_hskp_setup.temp0 eq 'Y') then begin

      conv_temp = (*l_hskp.filled_temp_monsPtr[0])
      min_indx = where (conv_temp ne NO_DATA, cnt)
      if (cnt gt 0) then begin
         min_Y_val = min (conv_temp[min_indx]) - 2
         max_Y_val = max (conv_temp[min_indx]) + 2
      endif else begin
         min_Y_val = min (conv_temp) - 2
         max_Y_val = max (conv_temp) + 2
      endelse

      l_PlotStuff, nplots, $
                   (*l_hskp.filledTimePtr), $
                   (*l_hskp.filled_temp_monsPtr[0]), $
                   min_Y_val, $
                   max_Y_val, $
                   'Temp0', $
                   l_pltpos, $
                   plotted, $
                   l_oa, $
                   l_setup.l_start_dtime

       l_pltpos.ypltmx = l_pltpos.ypltmn - l_pltpos.yspace
       l_pltpos.ypltmn = l_pltpos.ypltmx - l_pltpos.ydel
       plotted = plotted + 1

       l_ChckCompPg, plotted, nplots, pages, $
                     l_setup, windx, l_pltpos, l_rgb_table
   endif


   if (l_hskp_setup.temp1 eq 'Y') then begin

      conv_temp = (*l_hskp.filled_temp_monsPtr[1])
      min_indx = where (conv_temp ne NO_DATA, cnt)
      if (cnt gt 0) then begin
         min_Y_val = min (conv_temp[min_indx]) - 2
         max_Y_val = max (conv_temp[min_indx]) + 2
      endif else begin
         min_Y_val = min (conv_temp) - 2
         max_Y_val = max (conv_temp) + 2
      endelse

      l_PlotStuff, nplots, $
                   (*l_hskp.filledTimePtr), $
                   (*l_hskp.filled_temp_monsPtr[1]), $
                   min_Y_val, $
                   max_Y_val, $
                   'Temp1', $
                   l_pltpos, $
                   plotted, $
                   l_oa, $
                   l_setup.l_start_dtime


       l_pltpos.ypltmx = l_pltpos.ypltmn - l_pltpos.yspace
       l_pltpos.ypltmn = l_pltpos.ypltmx - l_pltpos.ydel
       plotted = plotted + 1

       l_ChckCompPg, plotted, nplots, pages, $
                     l_setup, windx, l_pltpos, l_rgb_table

   endif

   ;-------------------------------------------------------------------
   ; HVPS Voltage Monitors  (VMON)
   ;-------------------------------------------------------------------

   if (l_hskp_setup.hvps_mcp_start_vmon eq 'Y') then begin

      l_PlotStuff, nplots, $
                   (*l_hskp.filledTimePtr), $
                   (*l_hskp.filled_hvps_vmonsPtr[0]), $
                   min((*l_hskp.hvps_vmonsPtr)[*,0])-1, $
                   max((*l_hskp.hvps_vmonsPtr)[*,0])+1, $
                   'Start Vmon', $
                   l_pltpos, $
                   plotted, $
                   l_oa, $
                   l_setup.l_start_dtime

      yctr = (l_pltpos.ypltmn + l_pltpos.ypltmx)/2.0
      xyouts, 0.02, yctr, 'HVPS MCP', size=l_pltpos.sm_font, $
              /NORMAL, alignment = 0.5, orientation=90.0

       l_pltpos.ypltmx = l_pltpos.ypltmn - l_pltpos.yspace
       l_pltpos.ypltmn = l_pltpos.ypltmx - l_pltpos.ydel
       plotted = plotted + 1

       l_ChckCompPg, plotted, nplots,  pages, $
                     l_setup, windx, l_pltpos, l_rgb_table

   endif

   if (l_hskp_setup.hvps_mcp_stop_vmon eq 'Y') then begin

      l_PlotStuff, nplots, $
                   (*l_hskp.filledTimePtr), $
                   (*l_hskp.filled_hvps_vmonsPtr[1]), $
                   min((*l_hskp.hvps_vmonsPtr)[*,1])-1,$
                   max((*l_hskp.hvps_vmonsPtr)[*,1])+1,$
                   'Stop Vmon', $
                   l_pltpos, $
                   plotted, $
                   l_oa, $
                   l_setup.l_start_dtime
 
      yctr = (l_pltpos.ypltmn + l_pltpos.ypltmx)/2.0
      xyouts, 0.02, yctr, 'HVPS MCP', size=l_pltpos.sm_font, $
              /NORMAL, alignment = 0.5, orientation=90.0

       l_pltpos.ypltmx = l_pltpos.ypltmn - l_pltpos.yspace
       l_pltpos.ypltmn = l_pltpos.ypltmx - l_pltpos.ydel
       plotted = plotted + 1

       l_ChckCompPg, plotted, nplots, pages, $
                     l_setup, windx, l_pltpos, l_rgb_table

   endif


   if (l_hskp_setup.hvps_optics_vmon eq 'Y') then begin
 
      l_PlotStuff, nplots, $
                   (*l_hskp.filledTimePtr), $
                   (*l_hskp.filled_hvps_vmonsPtr[4]), $
                   min((*l_hskp.hvps_vmonsPtr)[*,4])-1,$
                   max((*l_hskp.hvps_vmonsPtr)[*,4])+1,$
                   'Vmon',     $
                   l_pltpos, $
                   plotted, $
                   l_oa, $
                   l_setup.l_start_dtime

      yctr = (l_pltpos.ypltmn + l_pltpos.ypltmx)/2.0
      xyouts, 0.02, yctr, 'HVPS Optics', size=l_pltpos.sm_font, $
              /NORMAL, alignment = 0.5, orientation=90.0

       l_pltpos.ypltmx = l_pltpos.ypltmn - l_pltpos.yspace
       l_pltpos.ypltmn = l_pltpos.ypltmx - l_pltpos.ydel
       plotted = plotted + 1

       l_ChckCompPg, plotted, nplots, pages, $
                     l_setup, windx, l_pltpos, l_rgb_table
   endif


   if (l_hskp_setup.hvps_steering_vmon eq 'Y') then begin

       l_PlotStuff, nplots, $
                    (*l_hskp.filledTimePtr), $
                    (*l_hskp.filled_hvps_vmonsPtr[5]), $
                    min((*l_hskp.hvps_vmonsPtr)[*,5])-1, $
                    max((*l_hskp.hvps_vmonsPtr)[*,5])+1, $
                    'Vmon',     $
                    l_pltpos, $
                    plotted, $
                    l_oa, $
                    l_setup.l_start_dtime

       yctr = (l_pltpos.ypltmn + l_pltpos.ypltmx)/2.0
       xyouts, 0.02, yctr, 'HVPS Steering', size=l_pltpos.sm_font, $
               /NORMAL, alignment = 0.5, orientation=90.0

       l_pltpos.ypltmx = l_pltpos.ypltmn - l_pltpos.yspace
       l_pltpos.ypltmn = l_pltpos.ypltmx - l_pltpos.ydel
       plotted = plotted + 1

       l_ChckCompPg, plotted, nplots, pages, $
                     l_setup, windx, l_pltpos, l_rgb_table
   endif

   if (l_hskp_setup.hvps_collp_vmon eq 'Y') then begin

       l_PlotStuff, nplots, $
                    (*l_hskp.filledTimePtr), $
                    (*l_hskp.filled_hvps_vmonsPtr[2]), $
                    min((*l_hskp.hvps_vmonsPtr)[*,2])-1, $
                    max((*l_hskp.hvps_vmonsPtr)[*,2])+1, $
                    'Vmon',     $
                    l_pltpos, $
                    plotted, $
                    l_oa, $
                    l_setup.l_start_dtime

       yctr = (l_pltpos.ypltmn + l_pltpos.ypltmx)/2.0
       xyouts, 0.02, yctr, 'HVPS CollP', size=l_pltpos.sm_font, $
               /NORMAL, alignment = 0.5, orientation=90.0

       l_pltpos.ypltmx = l_pltpos.ypltmn - l_pltpos.yspace
       l_pltpos.ypltmn = l_pltpos.ypltmx - l_pltpos.ydel
       plotted = plotted + 1

       l_ChckCompPg, plotted, nplots, pages, $
                     l_setup, windx, l_pltpos, l_rgb_table
   endif


   if (l_hskp_setup.hvps_colln_vmon eq 'Y') then begin

       l_PlotStuff, nplots, $
                    (*l_hskp.filledTimePtr), $
                    (*l_hskp.filled_hvps_vmonsPtr[3]), $
                    min((*l_hskp.hvps_vmonsPtr)[*,3])-1, $
                    max((*l_hskp.hvps_vmonsPtr)[*,3])+1, $
                    'Vmon',     $
                    l_pltpos, $
                    plotted, $
                    l_oa, $
                    l_setup.l_start_dtime


       yctr = (l_pltpos.ypltmn + l_pltpos.ypltmx)/2.0
       xyouts, 0.02, yctr, 'HVPS CollN', size=l_pltpos.sm_font, $
               /NORMAL, alignment = 0.5, orientation=90.0

       l_pltpos.ypltmx = l_pltpos.ypltmn - l_pltpos.yspace
       l_pltpos.ypltmn = l_pltpos.ypltmx - l_pltpos.ydel
       plotted = plotted + 1

       l_ChckCompPg, plotted, nplots, pages, $
                     l_setup, windx, l_pltpos, l_rgb_table
   endif

   ;-------------------------------------------------------------------
   ; HVPS Current Monitors (IMON)
   ;-------------------------------------------------------------------
   if (l_hskp_setup.hvps_mcp_start_imon eq 'Y') then begin

       l_PlotStuff, nplots, $
                    (*l_hskp.filledTimePtr), $
                    (*l_hskp.filled_hvps_imonsPtr[0]), $
                    min((*l_hskp.hvps_imonsPtr)[*,0])-1, $
                    max((*l_hskp.hvps_imonsPtr)[*,0])+1, $
                    'Start Imon', $
                    l_pltpos, $
                    plotted, $
                    l_oa, $
                    l_setup.l_start_dtime

       yctr = (l_pltpos.ypltmn + l_pltpos.ypltmx)/2.0
       xyouts, 0.02, yctr, 'HVPS MCP', size=l_pltpos.sm_font, $
               /NORMAL, alignment = 0.5, orientation=90.0

       l_pltpos.ypltmx = l_pltpos.ypltmn - l_pltpos.yspace
       l_pltpos.ypltmn = l_pltpos.ypltmx - l_pltpos.ydel
       plotted = plotted + 1

       l_ChckCompPg, plotted, nplots,  pages, $
                     l_setup, windx, l_pltpos, l_rgb_table
   endif

   if (l_hskp_setup.hvps_mcp_stop_imon eq 'Y') then begin

       l_PlotStuff, nplots, $
                    (*l_hskp.filledTimePtr), $
                    (*l_hskp.filled_hvps_imonsPtr[1]), $
                    min((*l_hskp.hvps_imonsPtr)[*,1])-1, $
                    max((*l_hskp.hvps_imonsPtr)[*,1])+1, $
                    'Stop Imon', $
                    l_pltpos, $
                    plotted, $
                    l_oa, $
                    l_setup.l_start_dtime


       yctr = (l_pltpos.ypltmn + l_pltpos.ypltmx)/2.0
       xyouts, 0.02, yctr, 'HVPS MCP', size=l_pltpos.sm_font, $
               /NORMAL, alignment = 0.5, orientation=90.0

       l_pltpos.ypltmx = l_pltpos.ypltmn - l_pltpos.yspace
       l_pltpos.ypltmn = l_pltpos.ypltmx - l_pltpos.ydel
       plotted = plotted + 1

       l_ChckCompPg, plotted, nplots, pages, $
                     l_setup, windx, l_pltpos, l_rgb_table
   endif

   if (l_hskp_setup.hvps_optics_imon eq 'Y') then begin

       l_PlotStuff, nplots, $
                    (*l_hskp.filledTimePtr), $
                    (*l_hskp.filled_hvps_imonsPtr[4]), $
                    min((*l_hskp.hvps_imonsPtr)[*,4])-1, $
                    max((*l_hskp.hvps_imonsPtr)[*,4])+1, $
                    'Imon', $
                    l_pltpos, $
                    plotted, $
                    l_oa, $
                    l_setup.l_start_dtime
 
       yctr = (l_pltpos.ypltmn + l_pltpos.ypltmx)/2.0
       xyouts, 0.02, yctr, 'HVPS Optics', size=l_pltpos.sm_font, $
               /NORMAL, alignment = 0.5, orientation=90.0

       l_pltpos.ypltmx = l_pltpos.ypltmn - l_pltpos.yspace
       l_pltpos.ypltmn = l_pltpos.ypltmx - l_pltpos.ydel
       plotted = plotted + 1

       l_ChckCompPg, plotted, nplots, pages, $
                     l_setup, windx, l_pltpos, l_rgb_table
   endif

   ;-------------------------------------------------------------------
   ; LVPS Voltage Monitors 
   ;-------------------------------------------------------------------
   if (l_hskp_setup.lvps_15vp_vmon eq 'Y') then begin
 
       l_PlotStuff, nplots, $
                    (*l_hskp.filledTimePtr), $
                    (*l_hskp.filled_lvps_vmonsPtr[1]), $
                    min((*l_hskp.lvps_vmonsPtr)[*,1])-1, $
                    max((*l_hskp.lvps_vmonsPtr)[*,1])+1, $
                    'Vmon', $
                    l_pltpos, $
                    plotted, $
                    l_oa, $
                    l_setup.l_start_dtime

       yctr = (l_pltpos.ypltmn + l_pltpos.ypltmx)/2.0
       xyouts, 0.02, yctr, 'LVPS 15vp', size=l_pltpos.sm_font, $
               /NORMAL, alignment = 0.5, orientation=90.0

       l_pltpos.ypltmx = l_pltpos.ypltmn - l_pltpos.yspace
       l_pltpos.ypltmn = l_pltpos.ypltmx - l_pltpos.ydel
       plotted = plotted + 1

       l_ChckCompPg, plotted, nplots, pages, $
                     l_setup, windx, l_pltpos, l_rgb_table
   endif

   if (l_hskp_setup.lvps_15vn_vmon eq 'Y') then begin

       l_PlotStuff, nplots, $
                    (*l_hskp.filledTimePtr), $
                    (*l_hskp.filled_lvps_vmonsPtr[2]), $
                    min((*l_hskp.lvps_vmonsPtr)[*,2])-1, $
                    max((*l_hskp.lvps_vmonsPtr)[*,2])+1, $
                    'Vmon', $
                    l_pltpos, $
                    plotted, $
                    l_oa, $
                    l_setup.l_start_dtime

       yctr = (l_pltpos.ypltmn + l_pltpos.ypltmx)/2.0
       xyouts, 0.02, yctr, 'LVPS 15vn', size=l_pltpos.sm_font, $
               /NORMAL, alignment = 0.5, orientation=90.0

       l_pltpos.ypltmx = l_pltpos.ypltmn - l_pltpos.yspace
       l_pltpos.ypltmn = l_pltpos.ypltmx - l_pltpos.ydel
       plotted = plotted + 1

       l_ChckCompPg, plotted, nplots, pages, $
                     l_setup, windx, l_pltpos, l_rgb_table
   endif

   if (l_hskp_setup.lvps_5vp_vmon eq 'Y') then begin

       l_PlotStuff, nplots, $
                    (*l_hskp.filledTimePtr), $
                    (*l_hskp.filled_lvps_vmonsPtr[3]), $
                    min((*l_hskp.lvps_vmonsPtr)[*,3])-1, $
                    max((*l_hskp.lvps_vmonsPtr)[*,3])+1, $
                    'Vmon', $
                    l_pltpos, $
                    plotted, $
                    l_oa, $
                    l_setup.l_start_dtime

       yctr = (l_pltpos.ypltmn + l_pltpos.ypltmx)/2.0
       xyouts, 0.02, yctr, 'LVPS 5vp', size=l_pltpos.sm_font, $
               /NORMAL, alignment = 0.5, orientation=90.0

       l_pltpos.ypltmx = l_pltpos.ypltmn - l_pltpos.yspace
       l_pltpos.ypltmn = l_pltpos.ypltmx - l_pltpos.ydel
       plotted = plotted + 1

       l_ChckCompPg, plotted, nplots, pages, $
                     l_setup, windx, l_pltpos, l_rgb_table
   endif

   if (l_hskp_setup.lvps_5vn_vmon eq 'Y') then begin

       l_PlotStuff, nplots, $
                    (*l_hskp.filledTimePtr), $
                    (*l_hskp.filled_lvps_vmonsPtr[4]), $
                    min((*l_hskp.lvps_vmonsPtr)[*,4])-1, $
                    max((*l_hskp.lvps_vmonsPtr)[*,4])+1, $
                    'Vmon', $
                    l_pltpos, $
                    plotted, $
                    l_oa, $
                    l_setup.l_start_dtime


       yctr = (l_pltpos.ypltmn + l_pltpos.ypltmx)/2.0
       xyouts, 0.02, yctr, 'LVPS 5vn', size=l_pltpos.sm_font, $
               /NORMAL, alignment = 0.5, orientation=90.0

       l_pltpos.ypltmx = l_pltpos.ypltmn - l_pltpos.yspace
       l_pltpos.ypltmn = l_pltpos.ypltmx - l_pltpos.ydel
       plotted = plotted + 1

       l_ChckCompPg, plotted, nplots, pages, $
                     l_setup, windx, l_pltpos, l_rgb_table
   endif



RETURN
END

;-------------------------------------------------------------------------------
;  Procedure :  l_HskpTerm
;  Description:
;    Free pointers allocated.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_hskp                struct        I        hskp data parameters 
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;-------------------------------------------------------------------------------

PRO l_HskpTerm, l_hskp


   if (ptr_valid (l_hskp.timePtr)) then begin
      ptr_free, l_hskp.timePtr
   endif

   if (ptr_valid (l_hskp.filledTimePtr)) then begin
      ptr_free, l_hskp.filledTimePtr
   endif

   if (ptr_valid (l_hskp.startUdfTimePtr)) then begin
      ptr_free, l_hskp.startUdfTimePtr
   endif

   if (ptr_valid (l_hskp.filledStartUdfTimePtr)) then begin
      ptr_free, l_hskp.filledStartUdfTimePtr
   endif

   if (ptr_valid (l_hskp.stopUdfTimePtr)) then begin
      ptr_free, l_hskp.stopUdfTimePtr
   endif

   if (ptr_valid (l_hskp.hvps_sys_errsPtr)) then begin
      ptr_free, l_hskp.hvps_sys_errsPtr
   endif

   for i = 0, l_hskp.hvps_sys_errsNParms - 1 do begin
      if (ptr_valid (l_hskp.filled_hvps_sys_errsPtr[i])) then begin
          ptr_free, (l_hskp.filled_hvps_sys_errsPtr[i])
      endif
   endfor

   if (ptr_valid (l_hskp.lvps_sys_errsPtr)) then begin
      ptr_free, l_hskp.lvps_sys_errsPtr
   endif

   for i = 0, l_hskp.lvps_sys_errsNParms - 1 do begin
      if (ptr_valid (l_hskp.filled_lvps_sys_errsPtr[i])) then begin
         ptr_free, (l_hskp.filled_lvps_sys_errsPtr[i])
      endif
   endfor

   if (ptr_valid (l_hskp.sys_lev_errsPtr)) then begin
      ptr_free, l_hskp.sys_lev_errsPtr
   endif

   for i = 0, l_hskp.sys_lev_errsNParms - 1 do begin
      if (ptr_valid (l_hskp.filled_sys_lev_errsPtr[i])) then begin
         ptr_free, (l_hskp.filled_sys_lev_errsPtr[i])
      endif
   endfor

   if (ptr_valid (l_hskp.sys_lev_statsPtr)) then begin
      ptr_free, l_hskp.sys_lev_statsPtr
   endif

   for i = 0, l_hskp.sys_lev_statsNParms - 1 do begin
      if (ptr_valid (l_hskp.filled_sys_lev_statsPtr[i])) then begin
         ptr_free, (l_hskp.filled_sys_lev_statsPtr[i])
      endif
   endfor

   if (ptr_valid (l_hskp.data_pkt_sel_statPtr)) then begin
      ptr_free, l_hskp.data_pkt_sel_statPtr
   endif

   for i = 0, l_hskp.data_pkt_sel_statNParms - 1 do begin
      if (ptr_valid (l_hskp.filled_data_pkt_sel_statPtr[i])) then begin
         ptr_free, (l_hskp.filled_data_pkt_sel_statPtr[i])
      endif
   endfor

   if (ptr_valid (l_hskp.mass_binn_configPtr)) then begin
      ptr_free, l_hskp.mass_binn_configPtr
   endif

   for i = 0, l_hskp.mass_binn_configNParms - 1 do begin
      if (ptr_valid (l_hskp.filled_mass_binn_configPtr[i])) then begin
         ptr_free, (l_hskp.filled_mass_binn_configPtr[i])
      endif
   endfor

   if (ptr_valid (l_hskp.tof_subsys_configPtr)) then begin
      ptr_free, l_hskp.tof_subsys_configPtr
   endif

   for i = 0, l_hskp.tof_subsys_configNParms - 1 do begin
      if (ptr_valid (l_hskp.filled_tof_subsys_configPtr[i])) then begin
         ptr_free, (l_hskp.filled_tof_subsys_configPtr[i])
      endif
   endfor

   if (ptr_valid (l_hskp.hvps_configPtr)) then begin
      ptr_free, l_hskp.hvps_configPtr
   endif

   for i = 0, l_hskp.hvps_configNParms - 1 do begin
      if (ptr_valid (l_hskp.filled_hvps_configPtr[i])) then begin
         ptr_free, (l_hskp.filled_hvps_configPtr[i])
      endif
   endfor

   if (ptr_valid (l_hskp.cmd_n_pkt_statsPtr)) then begin
      ptr_free, l_hskp.cmd_n_pkt_statsPtr
   endif

   for i = 0, l_hskp.cmd_n_pkt_statsNParms - 1 do begin
      if (ptr_valid (l_hskp.filled_cmd_n_pkt_statsPtr[i])) then begin
         ptr_free, (l_hskp.filled_cmd_n_pkt_statsPtr[i])
      endif
   endfor

   if (ptr_valid (l_hskp.hvps_vmonsPtr)) then begin
      ptr_free, l_hskp.hvps_vmonsPtr
   endif

   for i = 0, l_hskp.hvps_vmonsNParms - 1 do begin
      if (ptr_valid (l_hskp.filled_hvps_vmonsPtr[i])) then begin
         ptr_free, (l_hskp.filled_hvps_vmonsPtr[i])
      endif
   endfor

   if (ptr_valid (l_hskp.hvps_imonsPtr)) then begin
      ptr_free, l_hskp.hvps_imonsPtr
   endif

   for i = 0, l_hskp.hvps_imonsNParms - 1 do begin
      if (ptr_valid (l_hskp.filled_hvps_imonsPtr[i])) then begin
         ptr_free, (l_hskp.filled_hvps_imonsPtr[i])
      endif
   endfor

   if (ptr_valid (l_hskp.lvps_vmonsPtr)) then begin
      ptr_free, l_hskp.lvps_vmonsPtr
   endif

   for i = 0, l_hskp.lvps_vmonsNParms - 1 do begin
      if (ptr_valid (l_hskp.filled_lvps_vmonsPtr[i])) then begin
         ptr_free, (l_hskp.filled_lvps_vmonsPtr[i])
      endif
   endfor
 
   if (ptr_valid (l_hskp.lvps_imonsPtr)) then begin
      ptr_free, l_hskp.lvps_imonsPtr
   endif

   for i = 0, l_hskp.lvps_imonsNParms - 1 do begin
      if (ptr_valid (l_hskp.filled_lvps_imonsPtr[i])) then begin
         ptr_free, (l_hskp.filled_lvps_imonsPtr[i])
      endif
   endfor

   if (ptr_valid (l_hskp.temp_monsPtr)) then begin
      ptr_free, l_hskp.temp_monsPtr
   endif

   for i = 0, l_hskp.temp_monsNParms - 1 do begin
      if (ptr_valid (l_hskp.filled_temp_monsPtr[i])) then begin
         ptr_free, (l_hskp.filled_temp_monsPtr[i])
      endif
   endfor
  


RETURN
END

;-------------------------------------------------------------------------------
;  Function:  l_HskpProc
;  Description:
;    Main.
;
;  Return Value:  type = int
;  Value                         Description
;  -------------------------     ------------------------------
;  SUCCESS                       successful completion
;  FAILURE                       failure to complete
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_setup               struct        I        generic setup parameters
;  l_setup_file          string        I        setup file name 
;  l_pltpos              struct        I        generic plot parameters
;  l_rgb_table           struct        I        color table parameters
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;     PLOTDEFS         MAX_PLOTS        int         G         max # plots per page
;                      MAX_WEB_SPINS    int         G         max # spins per plot
;                                                             when using web interface
;                      NO_CNTS          double      G         no counts indicator
;                      NO_DATA          double      G         no data indicator
;
;      ERROR_TYPES     SUCCESS         int         G          successful completion
;                      FAILURE         int         G          unsuccessful completion
;                      WARNING         int         G          warning message
;                      INFO            int         G          informational message
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;-------------------------------------------------------------------------------

FUNCTION l_HskpProc, l_setup, l_setup_file, l_pltpos, l_rgb_table
COMMON PLOTDEFS, MAX_PLOTS, MAX_WEB_SPINS, NO_CNTS, NO_DATA
COMMON ERROR_TYPES, SUCCESS, FAILURE, WARNING, INFO

 
   nele = 0L
   year  = strmid (systime(0), 20, 23)
   l_pltpos.plot_hdr_title = '!5Housekeeping'


   status = l_OARead (l_oa, $ 
                      l_setup.l_start_dtime, $
                      l_setup.l_stop_dtime, $
                      l_pltpos)
   if (status eq FAILURE) then return, FAILURE


   status = l_HskpGetData (l_setup, $
                  l_setup_file, $
                  l_pltpos, $
                  l_hskp_setup, $ 
                  l_hskp)
   if (status eq FAILURE) then return, FAILURE


   l_HskpHndlMissData, l_hskp_setup, l_hskp, l_setup, l_oa


   nloops = 1L
   if (l_setup.l_postscript eq 'Y') then nloops = 2L


   for i = 0L, nloops - 1L do begin

      plotted = 0L
      pages   = 0L

      if (i eq 1L) and (l_setup.l_postscript eq 'Y') then begin
         l_pltpos.ps = 'Y'
         l_GenFileName, l_setup, '0'
         l_pltpos.filename = l_setup.l_prod_dst + '/' + $
                                l_setup.l_filename + '.ps'
      endif

      nplots = l_hskp_setup.nplots
      nrows  = nplots

      npages = CEIL (nrows/MAX_PLOTS)
      if (nrows - MAX_PLOTS gt 0L) then npages = npages + 1L 
      if (npages eq 0L) then npages = 1L
      if (npages gt 0L) then windx = intarr (npages)

      l_SetUpZBuffer, l_pltpos, $
                      l_setup,  $
                      l_rgb_table, $ 
                      colorbar, $
                      nplots

      l_setup.l_filedesc = 'hsk'
      l_setup.l_filedesc = l_setup.l_filedesc + 'ln'
      l_setup.l_line_plots = 'Y'

      l_AdjPlotSize, l_pltpos, l_setup, nplots 


      l_HskpPlot, l_hskp_setup, $
                  l_hskp, $
                  l_pltpos, $
                  nplots, $
                  windx, $
                  pages, $
                  plotted, $
                  l_setup, $
                  l_oa, $
                  l_rgb_table

    endfor
   

   l_HskpTerm, l_hskp
   l_OATerm, l_oa

   RETURN, SUCCESS

END
