
;-------------------------------------------------------------------------------
;
;  Unit Name : l_HskpLtt.pro
;
;  Purpose   : Housekeeping Trending plots.
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;   ELH              06/02           v1.0.9           Original implementation
;
;
;  File Revision Number:   %I%
;  File Last Modified  :   %E%  %T%
;
;-------------------------------------------------------------------------------

;-----------------------------------------------------------------
;   Determine the number of days in the month and the string
;   name.
;-----------------------------------------------------------------

FUNCTION DaysNMonth, yr, mon , month_str

    str_mon = strarr (13);
    str_mon[0] = ' '
    str_mon[1] = 'Jan'
    str_mon[2] = 'Feb'
    str_mon[3] = 'Mar'
    str_mon[4] = 'Apr'
    str_mon[5] = 'May'
    str_mon[6] = 'Jun'
    str_mon[7] = 'Jul'
    str_mon[8] = 'Aug'
    str_mon[9] = 'Sep'
    str_mon[10] = 'Oct'
    str_mon[11] = 'Nov'
    str_mon[12] = 'Dec'

    numdays = intarr(13);
    numdays[0] = 0
    numdays[1] = 31
    numdays[2] = 28
    numdays[3] = 31
    numdays[4] = 30
    numdays[5] = 31
    numdays[6] = 30
    numdays[7] = 31
    numdays[8] = 31
    numdays[9] = 30
    numdays[10] = 31
    numdays[11] = 30
    numdays[12] = 31

    if ( (((yr mod 4) eq 0) and ((yr mod 100) ne 0)) or ((yr mod 400) eq 0) ) then begin
       numdays[2] = 29;
    endif

    month_str = str_mon[mon]

    RETURN, numdays[mon]

END 


;-----------------------------------------------------------------
;  Draw the X-axis tick marks.
;-----------------------------------------------------------------
PRO l_DrawXTick, plot_parms


   ntcks   = plot_parms.nticks
   lb_xpmn = plot_parms.xpltmn
   lb_xpmx = plot_parms.xpltmx
   lb_ypmn = plot_parms.ypltmn
   lb_ypmx = plot_parms.ypltmx

   lb_xdel = 0.03
   lb_ydel = 0.018

   lb_xpos_base = plot_parms.xpltmn
   lb_ypos_base = plot_parms.ypltmn - 0.02


   ;------------------------------------------------------------
   ;  add trailer
   ;------------------------------------------------------------
   lb_xpos = 0.01
   lb_ypos = lb_ypos_base


   ;------------------------------------------------------------
   ; the begin/end time of actual data on plot in hours
   ;------------------------------------------------------------
   tmin = plot_parms.min_time_axis_len
   tmax = plot_parms.max_time_axis_len

   ;------------------------------------------------------------
   ; find the length of the time based on begin/end time of data
   ;------------------------------------------------------------
   tdel = tmax - tmin

   ;------------------------------------------------------------
   ; find the size of the plot based on the x axis size and
   ; size of the drawable area
   ;------------------------------------------------------------
   xdel = plot_parms.axis_size/plot_parms.xin
   xpltmn = 0.92/plot_parms.xin

   ;------------------------------------------------------------
   ; draw the x axis time labels on plot
   ;------------------------------------------------------------
   min_x_pos = lb_xpmn - 0.001
   max_x_pos = lb_xpmx + 0.001


   for i = 0, ntcks-1 do begin

      lb_ypos = lb_ypos_base

      ;------------------------------------------------------------
      ; calculate the position of the major tick mark based on the
      ; start time of actual data and time of label
      ;------------------------------------------------------------
      curr = plot_parms.xtick_locs[i]
      lb_xpos = double((curr - tmin)/ tdel * xdel + xpltmn)


      if (lb_xpos ge min_x_pos) AND (lb_xpos le max_x_pos) then begin
         plots, [lb_xpos, lb_xpos], [lb_ypmn-0.006, lb_ypmn], /normal
         plots, [lb_xpos, lb_xpos], [lb_ypmx+0.006, lb_ypmx], /normal
         lb_ypos = lb_ypos - lb_ydel
      endif

      ;------------------------------------------------------------
      ; determine the position of the minor tick marks
      ;------------------------------------------------------------
      for j = 1, plot_parms.xmn_tick do begin
          xmn_xpos = double(((curr+j*plot_parms.mn_tick_delta) - tmin)/ tdel * xdel + xpltmn)
          if (xmn_xpos ge min_x_pos) AND (xmn_xpos le max_x_pos) then begin
             plots, [xmn_xpos, xmn_xpos], [lb_ypmn-0.003, lb_ypmn], /normal
             plots, [xmn_xpos, xmn_xpos], [lb_ypmx+0.003, lb_ypmx], /normal
          endif
      endfor

   endfor


RETURN
END


;-----------------------------------------------------------------
;  Draw the X-axis tick labels.
;-----------------------------------------------------------------

PRO l_DrawXTickNLabel, plot_parms



   ntcks   = plot_parms.nticks
   lb_xpmn = plot_parms.xpltmn
   lb_xpmx = plot_parms.xpltmx
   lb_ypmn = plot_parms.ypltmn
   lb_ypmx = plot_parms.ypltmx

   lb_xdel = 0.03
   lb_ydel = 0.018

   lb_xpos_base = plot_parms.xpltmn
   lb_ypos_base = plot_parms.ypltmn - 0.02

   x_axis_len = lb_xpmx-lb_xpmn
   xtck_delta = x_axis_len/ntcks


   ;------------------------------------------------------------
   ;  add trailer
   ;------------------------------------------------------------
   lb_xpos = 0.01
   lb_ypos = lb_ypos_base

   xyouts, lb_xpos, lb_ypos, '!5UT', size=plot_parms.sm_font, /NORMAL
   lb_ypos = lb_ypos - lb_ydel


   ;------------------------------------------------------------
   ; the begin/end time of actual data on plot in hours
   ;------------------------------------------------------------
   tmin = plot_parms.min_time_axis_len
   tmax = plot_parms.max_time_axis_len

   ;------------------------------------------------------------
   ; find the length of the time based on begin/end time of data
   ;------------------------------------------------------------
   tdel = tmax - tmin

   ;------------------------------------------------------------
   ; find the size of the plot based on the x axis size and
   ; size of the drawable area
   ;------------------------------------------------------------
   xdel = plot_parms.axis_size/plot_parms.xin
   xpltmn = 0.92/plot_parms.xin


   ;------------------------------------------------------------
   ; draw the x axis time labels on plot
   ;------------------------------------------------------------
   first = 0
   min_x_pos = lb_xpmn - 0.001
   max_x_pos = lb_xpmx + 0.001


   for i = 0L, ntcks-1L do begin

      lb_ypos = lb_ypos_base

      ;------------------------------------------------------------
      ; calculate the position of the major tick mark based on the
      ; start time of actual data and time of label
      ;------------------------------------------------------------
      curr = plot_parms.xtick_locs[i]
      lb_xpos = double((curr - tmin)/ tdel * xdel + xpltmn)


      if (lb_xpos ge min_x_pos) AND (lb_xpos le max_x_pos) then begin
         plots, [lb_xpos, lb_xpos], [lb_ypmn-0.006, lb_ypmn], /normal
         plots, [lb_xpos, lb_xpos], [lb_ypmx+0.006, lb_ypmx], /normal
         if (first eq 0) then begin
            plot_parms.first_time_label = plot_parms.xtick_locs[i]
            first = 1
         endif

         label=plot_parms.xtick_labels[i]
         xyouts, lb_xpos, lb_ypos, label, /normal, $
                 size=plot_parms.sm_font, alignment=0.5
         lb_ypos = lb_ypos - lb_ydel

      endif
      ;------------------------------------------------------------
      ; determine the position of the minor tick marks
      ;------------------------------------------------------------
      for j = 1, plot_parms.xmn_tick do begin
          xmn_xpos = double(((curr+j*plot_parms.mn_tick_delta) - tmin)/ tdel * xdel + xpltmn)
          if (xmn_xpos ge min_x_pos) AND (xmn_xpos le max_x_pos) then begin
             plots, [xmn_xpos, xmn_xpos], [lb_ypmn-0.003, lb_ypmn], /normal
             plots, [xmn_xpos, xmn_xpos], [lb_ypmx+0.003, lb_ypmx], /normal
          endif
      endfor

   endfor


   ;------------------------------------------------------------
   ; calculate the position for the text label on the x axis
   ;------------------------------------------------------------
   lb_xpos = lb_xpos_base + (ntcks*xtck_delta) + 0.05
   lb_ypos = lb_ypos_base


   ndays = fix(tdel/3600./24.)


   if (plot_parms.X_axis_eq_time eq 'Y') then begin
      if (ndays le 1) then begin
         xyouts, lb_xpos, lb_ypos, 'hh:mm', size=plot_parms.sm_font, /NORMAL
         lb_ypos = lb_ypos - lb_ydel
      endif else begin
         xyouts, lb_xpos, lb_ypos, 'doy', size=plot_parms.sm_font, /NORMAL
         lb_ypos = lb_ypos - lb_ydel
      endelse
   endif


   plot_parms.ypltmn = lb_ypos


RETURN
END


;-----------------------------------------------------------------
;  Generate a GIF file.
;-----------------------------------------------------------------
PRO l_GenGif, data, ii, setup_parms, rgb_table


   gif_filename = setup_parms.prod_dst + '/' + setup_parms.filename+'.gif'
   write_gif, gif_filename, data, rgb_table.red[0:225], $
              rgb_table.green[0:225], rgb_table.blue[0:225]


RETURN
END

;-----------------------------------------------------------------
;  Generate the GIF filename.
;-----------------------------------------------------------------
PRO l_GenFileName, setup_parms, sn2
COMMON EPOCH_DATA, EPOCH_YR

   yr = setup_parms.start_date[0] - EPOCH_YR

   fn = strcompress(('l' + $
                     string(format='(I2.2)', yr) +  $
                     string(format='(I2.2)', setup_parms.start_date[1]) + $
                     string(format='(I2.2)', setup_parms.start_date[2]) + $
                     '00002359_hsk' + string(format='(I1)', sn2)),  $
                     /REMOVE_ALL)

   setup_parms.filename = fn


RETURN
END


;-----------------------------------------------------------------
;  Load the specified color table.
;-----------------------------------------------------------------
PRO l_LoadColorTable, color_table, rgb_table
COMMON MYCOLORS, N_COLOR_TABLE, N_GRAYS


  case color_table of
    1: begin
         openr, ifd, 'color.tbl', /get_lun
         rgb = fltarr(3, N_COLOR_TABLE)
       end
    2: begin
         openr, ifd, 'bproy.tbl', /get_lun
         rgb = fltarr(3,N_COLOR_TABLE)
       end
    3: begin
         openr, ifd, 'black_white.tbl', /get_lun
         rgb = fltarr(3,N_COLOR_TABLE)
       end
    4: begin
         openr, ifd, 'bpbcgyr.tbl', /get_lun
         rgb = fltarr(3,N_COLOR_TABLE)
       end
    else: begin
       end
  endcase

  if (color_table ge 1) and (color_table le 4) then begin
     readf, ifd, rgb

     rgb_table.red   = rgb[0,*]
     rgb_table.green = rgb[1,*]
     rgb_table.blue  = rgb[2,*]

     tvlct, rgb_table.red, rgb_table.green, rgb_table.blue

  endif

  close, ifd
  free_lun, ifd

RETURN
END


;-----------------------------------------------------------------
;  Determine the number of plots to plot per page and calculate
;  the size of each plot.
;-----------------------------------------------------------------
PRO l_AdjPlotSize, plot_parms, setup_parms, nplots
COMMON PLOTDEFS, MAX_PLOTS, MAX_WEB_SPINS, NO_CNTS, NO_DATA

    if (nplots lt MAX_PLOTS) then begin
       if (nplots eq 1) then begin
           new_plot_size = 3.55
           plot_parms.ydel = (new_plot_size / plot_parms.yin)
        endif else begin
           new_plot_size = floor(plot_parms.yin/(nplots+1))
           plot_parms.ydel = (new_plot_size / plot_parms.yin)
           if (nplots eq 4) then begin
              plot_parms.ydel = (new_plot_size / plot_parms.yin) + 0.05
           endif else begin
              if (nplots ge 5) then begin
                 plot_parms.ydel = (new_plot_size / plot_parms.yin) + 0.005
              endif
           endelse
        endelse
        plot_parms.ypltmn = plot_parms.ypltmx - plot_parms.ydel
    endif 

RETURN
END


;-----------------------------------------------------------------
;  Set up the display environment.
;-----------------------------------------------------------------
PRO SetUpDisplay, plot_parms, nplots, rgb_table, setup_parms


    !P.Multi = [0, 1, nplots, 0, 0]

    set_plot, 'X'
    device, PSEUDO_COLOR = 8


    window, title='LENA HSKP Long Term Trending Plots', $
            xsize=plot_parms.window_width, $
            ysize=plot_parms.window_height, $
            retain=2, colors = N_COLOR_TABLE

    char_size = 1.0
    char_size_1 = 1.2
    !p.position = [0.0,0.0, 1.0, 1.0]
    erase

    ;-----------------------------------------------------------
    ; if device fonts are less than 12, need to use larger
    ; font sizes
    ;-----------------------------------------------------------
    if (!D.X_CH_SIZE lt 12) AND (!D.Y_CH_SIZE lt 12) then begin
       plot_parms.lg_font = 1.8D
       plot_parms.md_font = 1.1D
       plot_parms.sm_font = 1.0D
       plot_parms.xsm_font= .85D
       if (nplots le 2) then begin
          plot_parms.plot_font = plot_parms.sm_font
       endif else begin
          if (nplots ge 3) then begin
             plot_parms.plot_font = 1.35D
          endif
       endelse
    endif else begin
       plot_parms.lg_font = 1.0D
       plot_parms.md_font = 0.9D
       plot_parms.sm_font = 0.6D
       plot_parms.xsm_font= 0.45D
       if (nplots le 2) then begin
          plot_parms.plot_font = plot_parms.md_font
       endif else begin
          if (nplots ge 3) then begin
             plot_parms.plot_font = 1.1D
          endif
       endelse
    endelse


    l_LoadColorTable, setup_parms.color_tbl, rgb_table

    !p.color = 1
    !p.background = 0




RETURN
END



;---------------------------------------------------------------------------
;  Read the parameters in the setup file.
;---------------------------------------------------------------------------


FUNCTION ReadSetupFile, setup_filename, setup_parms 
COMMON ERROR_TYPES, SUCCESS, FAILURE, WARNING, INFO
COMMON EPOCH_DATA, EPOCH_YR

   error_msg = ' '
   openr, inlun, /get_lun, setup_filename 

   line = ' '

   setup_parms.nplots = 0L

   while not EOF (inlun) do begin
      readf, inlun, line
      if ((strmid (line, 0, 1) ne ';') OR (strmid (line, 0, 1) ne '#')) AND (strlen(line) gt 0) then begin

         parts = str_sep (line, ' ')

         case parts[0] of
         'start_date'   : begin  ;-- start date (YYYY|MM|DD) or (YYYY|DOY)--
                              aa=parts[2]
                              bb=STR_SEP(aa,'|')
                              if (n_elements(bb) eq 3) then begin
                                 if (bb[0] lt EPOCH_YR) then begin
                                    error_msg = 'Invalid year entered.'
                                    close, inlun
                                    free_lun, inlun
                                    return, FAILURE
                                 endif
                                 if (bb[1] lt 1) or (bb[1] gt 12) then begin
                                    error_msg = 'Invalid month entered.'
                                    close, inlun
                                    free_lun, inlun
                                    return, FAILURE
                                 endif
                                 if (bb[2] lt 1) or (bb[2] gt 31) then begin
                                    error_msg = 'Invalid day entered.'
                                    close, inlun
                                    free_lun, inlun
                                    return, FAILURE
                                 endif
                              endif else begin
                                 if (n_elements(bb) eq 2) then begin
                                    if (bb[0] lt EPOCH_YR) then begin
                                       error_msg = 'Invalid year entered.'
                                       close, inlun
                                       free_lun, inlun
                                       return, FAILURE
                                    endif
                                    if (bb[1] lt 1) or (bb[1] gt 366) then begin
                                       error_msg = 'Invalid day of year entered.'
                                       close, inlun
                                       free_lun, inlun
                                       return, FAILURE
                                    endif
                                 endif else begin
                                     error_msg = 'No start date specified.'
                                     close, inlun
                                     free_lun, inlun
                                     return, FAILURE
                                 endelse
                              endelse
                              setup_parms.start_date=uint(bb(*))

                            end
         'stop_date'    : begin  ;-- stop date (YYYY|MM|DD) or (YYYY|DOY) --
                              aa=parts[2]
                              bb=STR_SEP(aa,'|')
                              if (n_elements(bb) eq 3) then begin
                                 if (bb[0] lt EPOCH_YR) then begin
                                    error_msg = 'Invalid year entered.'
                                    close, inlun
                                    free_lun, inlun
                                    return, FAILURE
                                 endif
                                 if (bb[1] lt 1) or (bb[1] gt 12) then begin
                                    error_msg = 'Invalid month entered.'
                                    close, inlun
                                    free_lun, inlun
                                    return, FAILURE
                                 endif
                                 if (bb[2] lt 1) or (bb[2] gt 31) then begin
                                    error_msg = 'Invalid day entered.'
                                    close, inlun
                                    free_lun, inlun
                                    return, FAILURE
                                 endif
                              endif else begin
                                 if (n_elements(bb) eq 2) then begin
                                    if (bb[0] lt EPOCH_YR) then begin
                                       error_msg = 'Invalid year entered.'
                                       close, inlun
                                       free_lun, inlun
                                       return, FAILURE
                                    endif
                                    if (bb[1] lt 1) or (bb[1] gt 366) then begin
                                       error_msg = 'Invalid day of year entered.'
                                       close, inlun
                                       free_lun, inlun
                                       return, FAILURE
                                    endif
                                 endif else begin
                                     error_msg = 'No stop date specified.'
                                     close, inlun
                                     free_lun, inlun
                                     return, FAILURE
                                 endelse
                              endelse
                              setup_parms.stop_date=uint(bb(*))
                            end
         'dumps'        : begin  ;-- indicates science/hk packets --
                            setup_parms.gen_dumps=STRUPCASE(parts[2])
                          end
         'data_type'    : begin  ;-- indicates science/hk packets --
                            setup_parms.data_type=STRUPCASE(parts[2])
                          end
         'plot_type'    : begin  ;-- type of plot, P(PS), X, G(GIF) --
                            setup_parms.plot_type=STRUPCASE(parts[2])
                          end
         'prod_dst'     : begin  ;-- path to store product --
                              temp = parts[2]
                              str_len = strlen(temp)
                              if (str_len eq 0) then begin
                                 setup_parms.prod_dst = '.'
                              endif else begin
                                 lchrn = str_len - 1
                                 lchr = strmid(temp, lchrn, lchrn)
                                 if (lchr eq '/') then begin
                                    setup_parms.prod_dst = strmid(temp, 0, lchrn)
                                 endif else begin
                                    setup_parms.prod_dst=temp
                                 endelse
                              endelse
                            end
         'avg_type'     : begin  ;-- path to store product --
                              setup_parms.avg_type=STRUPCASE(parts[2])
                          end
         'postscript'   : begin  ;-- path to store product --
                              setup_parms.postscript=STRUPCASE(parts[2])
                          end
         'starts' : begin
                      setup_parms.starts=STRUPCASE(parts[2])
                      if (setup_parms.starts eq 'Y') then $ 
                      setup_parms.nplots = setup_parms.nplots + 1L
                    end
         'stops' : begin
                      setup_parms.stops=STRUPCASE(parts[2])
                      if (setup_parms.stops eq 'Y') then $ 
                      setup_parms.nplots = setup_parms.nplots + 1L
                    end
         'hvps_mcp_start_vmon' : begin
                                   setup_parms.hvps_mcp_start_vmon=STRUPCASE(parts[2])
                                   if (setup_parms.hvps_mcp_start_vmon eq 'Y') then $ 
                                   setup_parms.nplots = setup_parms.nplots + 1L
                                 end
         'hvps_mcp_stop_vmon' : begin
                                   setup_parms.hvps_mcp_stop_vmon=STRUPCASE(parts[2])
                                   if (setup_parms.hvps_mcp_stop_vmon eq 'Y') then $ 
                                   setup_parms.nplots = setup_parms.nplots + 1L
                                 end
         'hvps_collp_vmon' : begin
                                   setup_parms.hvps_collp_vmon=STRUPCASE(parts[2])
                                   if (setup_parms.hvps_collp_vmon eq 'Y') then $ 
                                   setup_parms.nplots = setup_parms.nplots + 1L
                                 end
         'hvps_colln_vmon' : begin
                                   setup_parms.hvps_colln_vmon=STRUPCASE(parts[2])
                                   if (setup_parms.hvps_colln_vmon eq 'Y') then $ 
                                   setup_parms.nplots = setup_parms.nplots + 1L
                                 end
         'hvps_optics_vmon' : begin
                                   setup_parms.hvps_optics_vmon=STRUPCASE(parts[2])
                                   if (setup_parms.hvps_optics_vmon eq 'Y') then $ 
                                   setup_parms.nplots = setup_parms.nplots + 1L
                                 end
         'hvps_steering_vmon' : begin
                                   setup_parms.hvps_steering_vmon=STRUPCASE(parts[2])
                                   if (setup_parms.hvps_steering_vmon eq 'Y') then $ 
                                   setup_parms.nplots = setup_parms.nplots + 1L
                                 end
         'hvps_mcp_start_imon' : begin
                                   setup_parms.hvps_mcp_start_imon=STRUPCASE(parts[2])
                                   if (setup_parms.hvps_mcp_start_imon eq 'Y') then $ 
                                   setup_parms.nplots = setup_parms.nplots + 1L
                                 end
         'hvps_mcp_stop_imon' : begin
                                   setup_parms.hvps_mcp_stop_imon=STRUPCASE(parts[2])
                                   if (setup_parms.hvps_mcp_stop_imon eq 'Y') then $ 
                                   setup_parms.nplots = setup_parms.nplots + 1L
                                 end
         'hvps_collp_imon' : begin
                                   setup_parms.hvps_collp_imon=STRUPCASE(parts[2])
                                   if (setup_parms.hvps_collp_imon eq 'Y') then $ 
                                   setup_parms.nplots = setup_parms.nplots + 1L
                                 end
         'hvps_colln_imon' : begin
                                   setup_parms.hvps_colln_imon=STRUPCASE(parts[2])
                                   if (setup_parms.hvps_colln_imon eq 'Y') then $ 
                                   setup_parms.nplots = setup_parms.nplots + 1L
                                 end
         'hvps_optics_imon' : begin
                                   setup_parms.hvps_optics_imon=STRUPCASE(parts[2])
                                   if (setup_parms.hvps_optics_imon eq 'Y') then $ 
                                   setup_parms.nplots = setup_parms.nplots + 1L
                                 end
         'lvps_15vp_vmon' : begin
                               setup_parms.lvps_15vp_vmon=STRUPCASE(parts[2])
                               if (setup_parms.lvps_15vp_vmon eq 'Y') then $ 
                               setup_parms.nplots = setup_parms.nplots + 1L
                            end
         'lvps_15vn_vmon' : begin
                               setup_parms.lvps_15vn_vmon=STRUPCASE(parts[2])
                               if (setup_parms.lvps_15vn_vmon eq 'Y') then $ 
                               setup_parms.nplots = setup_parms.nplots + 1L
                            end
         'lvps_5vp_vmon' : begin
                               setup_parms.lvps_5vp_vmon=STRUPCASE(parts[2])
                               if (setup_parms.lvps_5vp_vmon eq 'Y') then $ 
                               setup_parms.nplots = setup_parms.nplots + 1L
                            end
         'lvps_5vn_vmon' : begin
                               setup_parms.lvps_5vn_vmon=STRUPCASE(parts[2])
                               if (setup_parms.lvps_5vn_vmon eq 'Y') then $ 
                               setup_parms.nplots = setup_parms.nplots + 1L
                            end
         'temp0' : begin
                      setup_parms.temp0=STRUPCASE(parts[2])
                      if (setup_parms.temp0 eq 'Y') then $ 
                      setup_parms.nplots = setup_parms.nplots + 1L
                    end
         'temp1' : begin
                      setup_parms.temp1=STRUPCASE(parts[2])
                      if (setup_parms.temp1 eq 'Y') then $ 
                      setup_parms.nplots = setup_parms.nplots + 1L
                    end
          else          : begin
                          end
         endcase
      endif
   endwhile

   close, inlun
   free_lun, inlun

RETURN, SUCCESS
END


;---------------------------------------------------------------------------
;  Title, date, etc... general plot documentation.
;---------------------------------------------------------------------------
PRO l_CreatePlotHdr, plot_parms, setup_parms
COMMON EPOCH_DATA, EPOCH_YR

   ;--------------------
   ; plot info
   ;--------------------
   xpos=0.01
   ypos=0.97
   ydelta=0.022

   header = '!5IMAGE/LENA ' + plot_parms.plot_hdr_title
   if (setup_parms.avg_type eq 'D') then begin
       header = header + ' Daily Avg'
   endif else begin
       header = header + ' Monthly Avg'
   endelse

   xyouts, xpos, ypos, /normal, header, charsize=plot_parms.lg_font
   ypos=ypos-ydelta

   mo = 0
   dy = 0


   doy = l_ConvYMD2DOY (setup_parms.start_date[0], setup_parms.start_date[1], $
                        setup_parms.start_date[2])
   new_doy = FindDOYSinceLaunch (setup_parms.start_date[0], doy)

   btime = STRING(setup_parms.start_date[0], $
                  setup_parms.start_date[1], $
                  setup_parms.start_date[2], $
                  new_doy,      $
                  setup_parms.start_time[0], $
                  setup_parms.start_time[1], $
                  setup_parms.start_time[2], $
   FORMAT="(i4,'/',i2.2,'/',i2.2,'  ', '(',i3,')','  ',i2.2,':',i2.2,':',i2.2)")

   plot_txt = 'Start Time: ' + btime
   xyouts, xpos, ypos, /normal, plot_txt, charsize=plot_parms.lg_font


   ypos=ypos-ydelta

   mo = 0
   dy = 0

   doy = l_ConvYMD2DOY (setup_parms.stop_date[0], setup_parms.stop_date[1], $
                        setup_parms.stop_date[2])
   new_doy = FindDOYSinceLaunch (setup_parms.stop_date[0], doy)

   etime = STRING(setup_parms.stop_date[0], $
                  setup_parms.stop_date[1], $
                  setup_parms.stop_date[2], $
                  new_doy,      $
                  setup_parms.stop_time[0], $
                  setup_parms.stop_time[1], $
                  setup_parms.stop_time[2], $
   FORMAT="(i4,'/',i2.2,'/',i2.2,'  ','(',i3,')','  ',i2.2,':',i2.2,':',i2.2)")

   plot_txt = 'Stop Time: ' + etime
   xyouts, xpos, ypos, /normal, plot_txt, charsize=plot_parms.lg_font
   ypos=ypos-ydelta

END


;---------------------------------------------------------------------------
;   Check if the last plot to be drawn on the page, if so, add documentation
;   and generate a postscript file, if selected.  If not the last plot, 
;   continue ...
;---------------------------------------------------------------------------
PRO l_ChckCompPg, plotted, nplots, pages, setup_parms, $
windx, plot_parms, rgb_table
COMMON PLOTDEFS, MAX_PLOTS, MAX_WEB_SPINS, NO_CNTS, NO_DATA


   if (plotted ge MAX_PLOTS) OR (nplots le 0) then begin
  
       if (plot_parms.ps ne 'Y') then begin
          !p.multi = 0

          sn1 = strcompress(strtrim(string(pages), 1))
          sn2 = string(format='(I4.4)', sn1)

          l_GenFileName, setup_parms, sn2
          plot_parms.filename = setup_parms.prod_dst + '/' + setup_parms.filename+'.gif'

          l_CreatePlotHdr, plot_parms, setup_parms
          l_TrailerLttDoc,  plot_parms
          if (setup_parms.plot_type ne 'P') then begin
             img_plot = TVRD ()
          endif

          ;------------- write plots to GIF file ------------
          if (setup_parms.gen_gif eq 'Y') then begin
              l_GenGif, img_plot, pages, setup_parms, rgb_table
          endif


          pages = pages + 1L

          plotted = 0L
          if (plotted lt nplots) AND (setup_parms.via_web eq 'N') AND $
             (setup_parms.gen_sum eq 'N') then begin
              erase
              wset, windx[pages]
          endif

          if (!D.NAME eq 'Z') then begin
            if (setup_parms.via_web eq 'Y') or (setup_parms.gen_sum eq 'Y') then begin
               device, /close
            endif
          endif


          plot_parms.ypltmx = plot_parms.top_ypltmx
          plot_parms.ypltmn = plot_parms.top_ypltmn
          !P.Multi = [0, 1, MAX_PLOTS, 0, 0]

       endif else begin


           !p.multi = 0
           l_CreatePlotHdr, plot_parms, setup_parms
           plot_parms.filename = setup_parms.prod_dst + '/' + $
                               setup_parms.filename + '.ps'

           l_TrailerLttDoc, plot_parms

           ;---------------------------
           ; close the postscript file
           ;---------------------------
           device, /close_file

           if (!D.NAME eq 'Z') then begin
             if (setup_parms.via_web eq 'Y') or (setup_parms.gen_sum eq 'Y') then begin
                device, /close
             endif
           endif


          pages = pages + 1L
           if (plotted le nplots) or (nplots gt 0) then begin
              plot_parms.ps = 'Y'
              l_GenFileName, setup_parms, string(pages)
              plot_parms.filename = setup_parms.prod_dst + '/' + $
                                  setup_parms.filename + '.ps'
              SetUpDisplay, plot_parms, nplots , rgb_table, setup_parms
                   
           endif

           plotted = 0L
           plot_parms.ypltmx = plot_parms.top_ypltmx
           plot_parms.ypltmn = plot_parms.top_ypltmn
           !P.Multi = [0, 1, MAX_PLOTS, 0, 0]

       endelse

   endif

RETURN
END



;---------------------------------------------------------------------------
;   Routine which determines which selected options to plot.
;---------------------------------------------------------------------------
PRO PlotData, l_hskpltt, setup_parms, plot_parms, nplots, plotted, pages, $
windx, rgb_table
COMMON PLOTDEFS, MAX_PLOTS, MAX_WEB_SPINS, NO_CNTS, NO_DATA
COMMON EPOCH_DATA, EPOCH_YR



   ; convert to real cal doy

   start_doy = l_ConvYMD2DOY (setup_parms.start_date[0], setup_parms.start_date[1], $
                        setup_parms.start_date[2])
   new_start_doy = FindDOYSinceLaunch (setup_parms.start_date[0], start_doy)

   stop_doy = l_ConvYMD2DOY (setup_parms.stop_date[0], setup_parms.stop_date[1], $
                        setup_parms.stop_date[2])
   new_stop_doy = FindDOYSinceLaunch (setup_parms.stop_date[0], stop_doy)

   DetYrByDOY, days_in_yr_struct

   l_DetMjMnTicks, new_start_doy, $
                   new_stop_doy, $
                   start_doy, $
                   stop_doy, $
                   plot_parms 

   ;-----------------------------------------------------------------------
   ; convert the x axis labels's string from day of year value since 
   ; launch to the calander day of year.
   ;-----------------------------------------------------------------------
   xindx = where (plot_parms.xtick_labels ne 0, xcnt)
   nlabels = n_elements (plot_parms.xtick_labels)
   if (xcnt gt 0) then begin
      nlabels = n_elements (xindx)
   endif
   for i = 0L, nlabels - 1L do begin
      doy = long (plot_parms.xtick_labels[i])   
      stindx = where (doy ge days_in_yr_struct[*].start_doy, stcnt)
      indexes = stindx
      if (stcnt gt 0) then begin 
        spindx = where (doy le days_in_yr_struct[stindx].stop_doy, spcnt)
        if (spcnt lt 0) then begin
           print, 'day of year is out of range: ', doy
        endif else begin
           new_label = FindDOYSinceLaunch (days_in_yr_struct[indexes[spindx]].year, doy)
           plot_parms.xtick_labels[i] = strcompress(string(new_label), /remove_all)
        endelse
      endif else begin
           print, 'day of year is out of range: ', doy
           ;spindx = where (doy le days_in_yr_struct[*].stop_doy, spcnt)
           spindx = stindx
      endelse

   endfor


   if (setup_parms.starts eq 'Y') then begin

       if (setup_parms.avg_type eq 'D') then begin
          data = (*l_hskpltt.starts_daily)
          time = (*l_hskpltt.time_daily)
       endif else begin
          data = (*l_hskpltt.starts_monthly)
          time = (*l_hskpltt.time_monthly)
       endelse

       l_Conv2CntRate, data, cntRate, 'SECTOR'
       l_Conv2Log, cntRate, data, start_min, start_max

       l_PlotLttStuff, nplots, $
                    time, $
                    data, $
                    start_min - 1, $
                    start_max + 1, $
                    'log Cnt Rate [Hz]', $
                    plot_parms, $
                    plotted


      yctr = (plot_parms.ypltmn + plot_parms.ypltmx)/2.0
      xyouts, 0.02, yctr, 'Starts', size=plot_parms.sm_font, $
              /NORMAL, alignment = 0.5, orientation=90.0

      plot_parms.ypltmx = plot_parms.ypltmn - plot_parms.yspace
      plot_parms.ypltmn = plot_parms.ypltmx - plot_parms.ydel
      plotted = plotted + 1

      l_ChckCompPg, plotted, nplots, pages, $
                    setup_parms, windx, plot_parms, rgb_table

   endif

   if (setup_parms.stops eq 'Y') then begin
       if (setup_parms.avg_type eq 'D') then begin
          data = (*l_hskpltt.stops_daily)
          time = (*l_hskpltt.time_daily)
       endif else begin
          data = (*l_hskpltt.stops_monthly)
          time = (*l_hskpltt.time_monthly)
       endelse

       l_Conv2CntRate, data, cntRate, 'SECTOR'
       l_Conv2Log, cntRate, data, stop_min, stop_max

       l_PlotLttStuff, nplots, $
                    time, $
                    data, $
                    stop_min - 1, $
                    stop_max + 1, $
                    'log Cnt Rate [Hz]', $
                    plot_parms, $
                    plotted


      yctr = (plot_parms.ypltmn + plot_parms.ypltmx)/2.0
      xyouts, 0.02, yctr, 'Stops', size=plot_parms.sm_font, $
              /NORMAL, alignment = 0.5, orientation=90.0

      plot_parms.ypltmx = plot_parms.ypltmn - plot_parms.yspace
      plot_parms.ypltmn = plot_parms.ypltmx - plot_parms.ydel
      plotted = plotted + 1

      l_ChckCompPg, plotted, nplots, pages, $
                    setup_parms, windx, plot_parms, rgb_table
   endif


   if (setup_parms.hvps_collp_imon eq 'Y') then begin

       if (setup_parms.avg_type eq 'D') then begin
          data = (*l_hskpltt.hvps_collp_imon_daily)
          time = (*l_hskpltt.time_daily)
       endif else begin
          data = (*l_hskpltt.hvps_collp_imon_monthly)
          time = (*l_hskpltt.time_monthly)
       endelse


       l_PlotLttStuff, nplots, $
                    time, $
                    data, $
                    min(data)-1, $
                    max(data)+1, $
                    'Imon', $
                    plot_parms, $
                    plotted


      yctr = (plot_parms.ypltmn + plot_parms.ypltmx)/2.0
      xyouts, 0.02, yctr, 'HVPS CollP', size=plot_parms.sm_font, $
              /NORMAL, alignment = 0.5, orientation=90.0

      plot_parms.ypltmx = plot_parms.ypltmn - plot_parms.yspace
      plot_parms.ypltmn = plot_parms.ypltmx - plot_parms.ydel
      plotted = plotted + 1

      l_ChckCompPg, plotted, nplots, pages, $
                    setup_parms, windx, plot_parms, rgb_table
   endif


   if (setup_parms.hvps_colln_imon eq 'Y') then begin

       if (setup_parms.avg_type eq 'D') then begin
          data = (*l_hskpltt.hvps_colln_imon_daily)
          time = (*l_hskpltt.time_daily)
       endif else begin
          data = (*l_hskpltt.hvps_colln_imon_monthly)
          time = (*l_hskpltt.time_monthly)
       endelse

       l_PlotLttStuff, nplots, $
                    time, $
                    data, $
                    min(data)-1, $
                    max(data)+1, $
                    'Imon', $
                    plot_parms, $
                    plotted

      yctr = (plot_parms.ypltmn + plot_parms.ypltmx)/2.0
      xyouts, 0.02, yctr, 'HVPS CollN', size=plot_parms.sm_font, $
              /NORMAL, alignment = 0.5, orientation=90.0

      plot_parms.ypltmx = plot_parms.ypltmn - plot_parms.yspace
      plot_parms.ypltmn = plot_parms.ypltmx - plot_parms.ydel
      plotted = plotted + 1

      l_ChckCompPg, plotted, nplots, pages, $
                    setup_parms, windx, plot_parms, rgb_table
   endif


   if (setup_parms.temp0 eq 'Y') then begin

      if (setup_parms.avg_type eq 'D') then begin
         data = (*l_hskpltt.temp0_daily)
         time = (*l_hskpltt.time_daily)
      endif else begin
         data = (*l_hskpltt.temp0_monthly)
         time = (*l_hskpltt.time_monthly)
      endelse


      conv_temp = data
      min_indx = where (conv_temp ne NO_DATA, cnt)
      if (cnt gt 0) then begin
         min_Y_val = min (conv_temp[min_indx]) - 2
         max_Y_val = max (conv_temp[min_indx]) + 2
      endif else begin
         min_Y_val = min (conv_temp) - 2
         max_Y_val = max (conv_temp) + 2
      endelse

      l_PlotLttStuff, nplots, $
                   time, $
                   data, $
                   min_Y_val, $
                   max_Y_val, $
                   ' ', $
                   plot_parms, $
                   plotted

      yctr = (plot_parms.ypltmn + plot_parms.ypltmx)/2.0
      xyouts, 0.02, yctr, 'Temp 0', size=plot_parms.sm_font, $
              /NORMAL, alignment = 0.5, orientation=90.0

       plot_parms.ypltmx = plot_parms.ypltmn - plot_parms.yspace
       plot_parms.ypltmn = plot_parms.ypltmx - plot_parms.ydel
       plotted = plotted + 1

       l_ChckCompPg, plotted, nplots, pages, $
                     setup_parms, windx, plot_parms, rgb_table
   endif

   if (setup_parms.temp1 eq 'Y') then begin

      if (setup_parms.avg_type eq 'D') then begin
         data = (*l_hskpltt.temp1_daily)
         time = (*l_hskpltt.time_daily)
      endif else begin
         data = (*l_hskpltt.temp1_monthly)
         time = (*l_hskpltt.time_monthly)
      endelse


      conv_temp = data
      min_indx = where (conv_temp ne NO_DATA, cnt)
      if (cnt gt 0) then begin
         min_Y_val = min (conv_temp[min_indx]) - 2
         max_Y_val = max (conv_temp[min_indx]) + 2
      endif else begin
         min_Y_val = min (conv_temp) - 2
         max_Y_val = max (conv_temp) + 2
      endelse

      l_PlotLttStuff, nplots, $
                   time, $
                   data, $
                   min_Y_val, $
                   max_Y_val, $
                   ' ', $
                   plot_parms, $
                   plotted


       yctr = (plot_parms.ypltmn + plot_parms.ypltmx)/2.0
       xyouts, 0.02, yctr, 'Temp 1', size=plot_parms.sm_font, $
              /NORMAL, alignment = 0.5, orientation=90.0

       plot_parms.ypltmx = plot_parms.ypltmn - plot_parms.yspace
       plot_parms.ypltmn = plot_parms.ypltmx - plot_parms.ydel
       plotted = plotted + 1

       l_ChckCompPg, plotted, nplots, pages, $
                     setup_parms, windx, plot_parms, rgb_table

   endif

   if (setup_parms.hvps_mcp_start_vmon eq 'Y') then begin

      if (setup_parms.avg_type eq 'D') then begin
         data = (*l_hskpltt.hvps_mcp_start_vmon_daily)
         time = (*l_hskpltt.time_daily)
      endif else begin
         data = (*l_hskpltt.hvps_mcp_start_vmon_monthly)
         time = (*l_hskpltt.time_monthly)
      endelse

      l_PlotLttStuff, nplots, $
                   time, $
                   data, $
                   min(data)-1, $
                   max(data)+1, $
                   'Start Vmon', $
                   plot_parms, $
                   plotted

      yctr = (plot_parms.ypltmn + plot_parms.ypltmx)/2.0
      xyouts, 0.02, yctr, 'HVPS MCP', size=plot_parms.sm_font, $
              /NORMAL, alignment = 0.5, orientation=90.0

       plot_parms.ypltmx = plot_parms.ypltmn - plot_parms.yspace
       plot_parms.ypltmn = plot_parms.ypltmx - plot_parms.ydel
       plotted = plotted + 1

       l_ChckCompPg, plotted, nplots,  pages, $
                     setup_parms, windx, plot_parms, rgb_table

   endif

   if (setup_parms.hvps_mcp_stop_vmon eq 'Y') then begin

      if (setup_parms.avg_type eq 'D') then begin
         data = (*l_hskpltt.hvps_mcp_stop_vmon_daily)
         time = (*l_hskpltt.time_daily)
      endif else begin
         data = (*l_hskpltt.hvps_mcp_stop_vmon_monthly)
         time = (*l_hskpltt.time_monthly)
      endelse

      l_PlotLttStuff, nplots, $
                   time, $
                   data, $
                   min(data)-1,$
                   max(data)+1,$
                   'Stop Vmon', $
                   plot_parms, $
                   plotted

      yctr = (plot_parms.ypltmn + plot_parms.ypltmx)/2.0
      xyouts, 0.02, yctr, 'HVPS MCP', size=plot_parms.sm_font, $
              /NORMAL, alignment = 0.5, orientation=90.0

      plot_parms.ypltmx = plot_parms.ypltmn - plot_parms.yspace
      plot_parms.ypltmn = plot_parms.ypltmx - plot_parms.ydel
      plotted = plotted + 1

      l_ChckCompPg, plotted, nplots, pages, $
                    setup_parms, windx, plot_parms, rgb_table

   endif


   if (setup_parms.hvps_optics_vmon eq 'Y') then begin

      if (setup_parms.avg_type eq 'D') then begin
         data = (*l_hskpltt.hvps_optics_vmon_daily)
         time = (*l_hskpltt.time_daily)
      endif else begin
         data = (*l_hskpltt.hvps_optics_vmon_monthly)
         time = (*l_hskpltt.time_monthly)
      endelse

      l_PlotLttStuff, nplots, $
                   time, $
                   data, $
                   min(data)-1,$
                   max(data)+1,$
                   'Vmon',     $
                   plot_parms, $
                   plotted

      yctr = (plot_parms.ypltmn + plot_parms.ypltmx)/2.0
      xyouts, 0.02, yctr, 'HVPS Optics', size=plot_parms.sm_font, $
              /NORMAL, alignment = 0.5, orientation=90.0

      plot_parms.ypltmx = plot_parms.ypltmn - plot_parms.yspace
      plot_parms.ypltmn = plot_parms.ypltmx - plot_parms.ydel
      plotted = plotted + 1

      l_ChckCompPg, plotted, nplots, pages, $
                    setup_parms, windx, plot_parms, rgb_table
   endif


   if (setup_parms.hvps_steering_vmon eq 'Y') then begin

      if (setup_parms.avg_type eq 'D') then begin
         data = (*l_hskpltt.hvps_steering_vmon_daily)
         time = (*l_hskpltt.time_daily)
      endif else begin
         data = (*l_hskpltt.hvps_steering_vmon_monthly)
         time = (*l_hskpltt.time_monthly)
      endelse

      l_PlotLttStuff, nplots, $
                   time, $
                   data, $
                   min(data)-1, $
                   max(data)+1, $
                   'Vmon',     $
                   plot_parms, $
                   plotted

      yctr = (plot_parms.ypltmn + plot_parms.ypltmx)/2.0
      xyouts, 0.02, yctr, 'HVPS Steering', size=plot_parms.sm_font, $
               /NORMAL, alignment = 0.5, orientation=90.0

      plot_parms.ypltmx = plot_parms.ypltmn - plot_parms.yspace
      plot_parms.ypltmn = plot_parms.ypltmx - plot_parms.ydel
      plotted = plotted + 1

      l_ChckCompPg, plotted, nplots, pages, $
                    setup_parms, windx, plot_parms, rgb_table
   endif



   if (setup_parms.hvps_collp_vmon eq 'Y') then begin

      if (setup_parms.avg_type eq 'D') then begin
         data = (*l_hskpltt.hvps_collp_vmon_daily)
         time = (*l_hskpltt.time_daily)
      endif else begin
         data = (*l_hskpltt.hvps_collp_vmon_monthly)
         time = (*l_hskpltt.time_monthly)
      endelse

      l_PlotLttStuff, nplots, $
                   time, $
                   data, $
                   min(data)-1, $
                   max(data)+1, $
                   'Vmon',     $
                   plot_parms, $
                   plotted

      yctr = (plot_parms.ypltmn + plot_parms.ypltmx)/2.0
      xyouts, 0.02, yctr, 'HVPS CollP', size=plot_parms.sm_font, $
               /NORMAL, alignment = 0.5, orientation=90.0

      plot_parms.ypltmx = plot_parms.ypltmn - plot_parms.yspace
      plot_parms.ypltmn = plot_parms.ypltmx - plot_parms.ydel
      plotted = plotted + 1

      l_ChckCompPg, plotted, nplots, pages, $
                    setup_parms, windx, plot_parms, rgb_table
   endif


   if (setup_parms.hvps_colln_vmon eq 'Y') then begin

      if (setup_parms.avg_type eq 'D') then begin
         data = (*l_hskpltt.hvps_colln_vmon_daily)
         time = (*l_hskpltt.time_daily)
      endif else begin
         data = (*l_hskpltt.hvps_colln_vmon_monthly)
         time = (*l_hskpltt.time_monthly)
      endelse

      l_PlotLttStuff, nplots, $
                   time, $
                   data, $
                   min(data)-1, $
                   max(data)+1, $
                   'Vmon',     $
                   plot_parms, $
                   plotted


      yctr = (plot_parms.ypltmn + plot_parms.ypltmx)/2.0
      xyouts, 0.02, yctr, 'HVPS CollN', size=plot_parms.sm_font, $
              /NORMAL, alignment = 0.5, orientation=90.0

      plot_parms.ypltmx = plot_parms.ypltmn - plot_parms.yspace
      plot_parms.ypltmn = plot_parms.ypltmx - plot_parms.ydel
      plotted = plotted + 1

      l_ChckCompPg, plotted, nplots, pages, $
                    setup_parms, windx, plot_parms, rgb_table
   endif

   if (setup_parms.hvps_mcp_start_imon eq 'Y') then begin

      if (setup_parms.avg_type eq 'D') then begin
         data = (*l_hskpltt.hvps_mcp_start_imon_daily)
         time = (*l_hskpltt.time_daily)
      endif else begin
         data = (*l_hskpltt.hvps_mcp_start_imon_monthly)
         time = (*l_hskpltt.time_monthly)
      endelse

      l_PlotLttStuff, nplots, $
                   time, $
                   data, $
                   min(data)-1, $
                   max(data)+1, $
                   'Start Imon', $
                   plot_parms, $
                   plotted

      yctr = (plot_parms.ypltmn + plot_parms.ypltmx)/2.0
      xyouts, 0.02, yctr, 'HVPS MCP', size=plot_parms.sm_font, $
               /NORMAL, alignment = 0.5, orientation=90.0

      plot_parms.ypltmx = plot_parms.ypltmn - plot_parms.yspace
      plot_parms.ypltmn = plot_parms.ypltmx - plot_parms.ydel
      plotted = plotted + 1

      l_ChckCompPg, plotted, nplots,  pages, $
                    setup_parms, windx, plot_parms, rgb_table
   endif

   if (setup_parms.hvps_mcp_stop_imon eq 'Y') then begin

      if (setup_parms.avg_type eq 'D') then begin
         data = (*l_hskpltt.hvps_mcp_stop_imon_daily)
         time = (*l_hskpltt.time_daily)
      endif else begin
         data = (*l_hskpltt.hvps_mcp_stop_imon_monthly)
         time = (*l_hskpltt.time_monthly)
      endelse


      l_PlotLttStuff, nplots, $
                   time, $
                   data,  $
                   min(data)-1, $
                   max(data)+1, $
                   'Stop Imon', $
                   plot_parms, $
                   plotted


      yctr = (plot_parms.ypltmn + plot_parms.ypltmx)/2.0
      xyouts, 0.02, yctr, 'HVPS MCP', size=plot_parms.sm_font, $
              /NORMAL, alignment = 0.5, orientation=90.0

      plot_parms.ypltmx = plot_parms.ypltmn - plot_parms.yspace
      plot_parms.ypltmn = plot_parms.ypltmx - plot_parms.ydel
      plotted = plotted + 1

      l_ChckCompPg, plotted, nplots, pages, $
                    setup_parms, windx, plot_parms, rgb_table
   endif

   if (setup_parms.hvps_optics_imon eq 'Y') then begin

      if (setup_parms.avg_type eq 'D') then begin
         data = (*l_hskpltt.hvps_optics_imon_daily)
         time = (*l_hskpltt.time_daily)
      endif else begin
         data = (*l_hskpltt.hvps_optics_imon_monthly)
         time = (*l_hskpltt.time_monthly)
      endelse

      l_PlotLttStuff, nplots, $
                   time, $
                   data, $
                   min(data)-1, $
                   max(data)+1, $
                   'Imon', $
                   plot_parms, $
                   plotted

      yctr = (plot_parms.ypltmn + plot_parms.ypltmx)/2.0
      xyouts, 0.02, yctr, 'HVPS Optics', size=plot_parms.sm_font, $
              /NORMAL, alignment = 0.5, orientation=90.0

      plot_parms.ypltmx = plot_parms.ypltmn - plot_parms.yspace
      plot_parms.ypltmn = plot_parms.ypltmx - plot_parms.ydel
      plotted = plotted + 1

      l_ChckCompPg, plotted, nplots, pages, $
                    setup_parms, windx, plot_parms, rgb_table
   endif

   if (setup_parms.lvps_15vp_vmon eq 'Y') then begin

      if (setup_parms.avg_type eq 'D') then begin
         data = (*l_hskpltt.lvps_15vp_vmon_daily)
         time = (*l_hskpltt.time_daily)
      endif else begin
         data = (*l_hskpltt.lvps_15vp_vmon_monthly)
         time = (*l_hskpltt.time_monthly)
      endelse

      l_PlotLttStuff, nplots, $
                   time, $
                   data, $
                   min(data)-1, $
                   max(data)+1, $
                   'Vmon', $
                   plot_parms, $
                   plotted

      yctr = (plot_parms.ypltmn + plot_parms.ypltmx)/2.0
      xyouts, 0.02, yctr, 'LVPS 15vp', size=plot_parms.sm_font, $
              /NORMAL, alignment = 0.5, orientation=90.0

      plot_parms.ypltmx = plot_parms.ypltmn - plot_parms.yspace
      plot_parms.ypltmn = plot_parms.ypltmx - plot_parms.ydel
      plotted = plotted + 1

      l_ChckCompPg, plotted, nplots, pages, $
                    setup_parms, windx, plot_parms, rgb_table
   endif


   if (setup_parms.lvps_15vn_vmon eq 'Y') then begin

      if (setup_parms.avg_type eq 'D') then begin
         data = (*l_hskpltt.lvps_15vn_vmon_daily)
         time = (*l_hskpltt.time_daily)
      endif else begin
         data = (*l_hskpltt.lvps_15vn_vmon_monthly)
         time = (*l_hskpltt.time_monthly)
      endelse

      l_PlotLttStuff, nplots, $
                   time, $
                   data, $
                   min(data)-1, $
                   max(data)+1, $
                   'Vmon', $
                   plot_parms, $
                   plotted

      yctr = (plot_parms.ypltmn + plot_parms.ypltmx)/2.0
      xyouts, 0.02, yctr, 'LVPS 15vn', size=plot_parms.sm_font, $
              /NORMAL, alignment = 0.5, orientation=90.0

      plot_parms.ypltmx = plot_parms.ypltmn - plot_parms.yspace
      plot_parms.ypltmn = plot_parms.ypltmx - plot_parms.ydel
      plotted = plotted + 1

      l_ChckCompPg, plotted, nplots, pages, $
                    setup_parms, windx, plot_parms, rgb_table
   endif


   if (setup_parms.lvps_5vp_vmon eq 'Y') then begin

      if (setup_parms.avg_type eq 'D') then begin
         data = (*l_hskpltt.lvps_5vp_vmon_daily)
         time = (*l_hskpltt.time_daily)
      endif else begin
         data = (*l_hskpltt.lvps_5vp_vmon_monthly)
         time = (*l_hskpltt.time_monthly)
      endelse

      l_PlotLttStuff, nplots, $
                   time, $
                   data, $
                   min(data)-1, $
                   max(data)+1, $
                   'Vmon', $
                   plot_parms, $
                   plotted

      yctr = (plot_parms.ypltmn + plot_parms.ypltmx)/2.0
      xyouts, 0.02, yctr, 'LVPS 5vp', size=plot_parms.sm_font, $
              /NORMAL, alignment = 0.5, orientation=90.0

      plot_parms.ypltmx = plot_parms.ypltmn - plot_parms.yspace
      plot_parms.ypltmn = plot_parms.ypltmx - plot_parms.ydel
      plotted = plotted + 1

      l_ChckCompPg, plotted, nplots, pages, $
                    setup_parms, windx, plot_parms, rgb_table
   endif


   if (setup_parms.lvps_5vn_vmon eq 'Y') then begin

      if (setup_parms.avg_type eq 'D') then begin
         data = (*l_hskpltt.lvps_5vn_vmon_daily)
         time = (*l_hskpltt.time_daily)
      endif else begin
         data = (*l_hskpltt.lvps_5vn_vmon_monthly)
         time = (*l_hskpltt.time_monthly)
      endelse

      l_PlotLttStuff, nplots, $
                   time, $
                   data, $
                   min(data)-1, $
                   max(data)+1, $
                   'Vmon', $
                   plot_parms, $
                   plotted


      yctr = (plot_parms.ypltmn + plot_parms.ypltmx)/2.0

      xyouts, 0.02, yctr, 'LVPS 5vn', size=plot_parms.sm_font, $
              /NORMAL, alignment = 0.5, orientation=90.0

      plot_parms.ypltmx = plot_parms.ypltmn - plot_parms.yspace
      plot_parms.ypltmn = plot_parms.ypltmx - plot_parms.ydel
      plotted = plotted + 1

      l_ChckCompPg, plotted, nplots, pages, $
                    setup_parms, windx, plot_parms, rgb_table
   endif



RETURN
END


;---------------------------------------------------------------------------
;  Read the housekeeping ascii text files.
;---------------------------------------------------------------------------
FUNCTION ReadHskpFile, setup_parms, l_hskpltt, ndays, est_nspins, est_days, est_months
COMMON ERROR_TYPES, SUCCESS, FAILURE, WARNING, INFO
COMMON EPOCH_DATA, EPOCH_YR
COMMON PLOTDEFS, MAX_PLOTS, MAX_WEB_SPINS, NO_CNTS, NO_DATA

    ;----------------------------------------
    ; path to RAIDS where housekeeping text
    ; dumps reside
    ; Hskp_FilePath = '/export/home/elee/lena/udf/udf_dlm/T/src/HSKP'
    ;----------------------------------------
    Hskp_FilePath = '/archive4/PRODUCTS/PRODUCTS/dev_Daily/hk_dmps'


    ;------------------------------------------------------
    ; X-axis
    ;------------------------------------------------------
    X_daily = intarr (est_days)
    X_monthly = intarr (est_months)

    ;------------------------------------------------------
    ; setup/allocate storage for HSKP parameters
    ;------------------------------------------------------

    starts                      = fltarr (45, est_nspins)
    starts_daily                = fltarr (est_days)
    starts_monthly              = fltarr (est_months)
 
    stops                       = fltarr (45, est_nspins) 
    stops_daily                 = fltarr (est_days) 
    stops_monthly               = fltarr (est_months) 

    hvps_collp_imon             = fltarr (est_nspins)
    hvps_collp_imon_daily       = fltarr (est_days) 
    hvps_collp_imon_monthly     = fltarr (est_months) 

    hvps_colln_imon             = fltarr (est_nspins)
    hvps_colln_imon_daily       = fltarr (est_days) 
    hvps_colln_imon_monthly     = fltarr (est_months) 

    temp0                       = fltarr (est_nspins)
    temp0_daily                 = fltarr (est_days) 
    temp0_monthly               = fltarr (est_months) 

    temp1                       = fltarr (est_nspins) 
    temp1_daily                 = fltarr (est_days) 
    temp1_monthly               = fltarr (est_months) 

    hvps_mcp_start_vmon         = fltarr (est_nspins)
    hvps_mcp_start_vmon_daily   = fltarr (est_days) 
    hvps_mcp_start_vmon_monthly = fltarr (est_months) 

    hvps_mcp_stop_vmon          = fltarr (est_nspins)
    hvps_mcp_stop_vmon_daily    = fltarr (est_days) 
    hvps_mcp_stop_vmon_monthly  = fltarr (est_months) 

    hvps_optics_vmon            = fltarr (est_nspins)
    hvps_optics_vmon_daily      = fltarr (est_days) 
    hvps_optics_vmon_monthly    = fltarr (est_months) 

    hvps_steering_vmon          = fltarr (est_nspins)
    hvps_steering_vmon_daily    = fltarr (est_days) 
    hvps_steering_vmon_monthly  = fltarr (est_months) 

    hvps_collp_vmon             = fltarr (est_nspins)
    hvps_collp_vmon_daily       = fltarr (est_days) 
    hvps_collp_vmon_monthly     = fltarr (est_months) 

    hvps_colln_vmon             = fltarr (est_nspins) 
    hvps_colln_vmon_daily       = fltarr (est_days) 
    hvps_colln_vmon_monthly     = fltarr (est_months) 

    hvps_mcp_start_imon         = fltarr (est_nspins)
    hvps_mcp_start_imon_daily   = fltarr (est_days) 
    hvps_mcp_start_imon_monthly = fltarr (est_months) 

    hvps_mcp_stop_imon          = fltarr (est_nspins)
    hvps_mcp_stop_imon_daily    = fltarr (est_days) 
    hvps_mcp_stop_imon_monthly  = fltarr (est_months) 

    hvps_optics_imon            = fltarr (est_nspins)
    hvps_optics_imon_daily      = fltarr (est_days) 
    hvps_optics_imon_monthly    = fltarr (est_months) 

    lvps_15vp_vmon              = fltarr (est_nspins)
    lvps_15vp_vmon_daily        = fltarr (est_days) 
    lvps_15vp_vmon_monthly      = fltarr (est_months) 

    lvps_15vn_vmon              = fltarr (est_nspins)
    lvps_15vn_vmon_daily        = fltarr (est_days) 
    lvps_15vn_vmon_monthly      = fltarr (est_months) 

    lvps_5vp_vmon               = fltarr (est_nspins)
    lvps_5vp_vmon_daily         = fltarr (est_days) 
    lvps_5vp_vmon_monthly       = fltarr (est_months) 

    lvps_5vn_vmon               = fltarr (est_nspins)
    lvps_5vn_vmon_daily         = fltarr (est_days) 
    lvps_5vn_vmon_monthly       = fltarr (est_months) 

    header                      = strarr (20)
    time                        = strarr(1)


    ;------------------------------------------------------
    ; Determine which files are available, between specified
    ; start and stop time period
    ;------------------------------------------------------
    ;files = FindFile ()

    ;------------------------------------------------------
    ; Determine the format of the X-axis tick labels and
    ; the number of tick marks
    ;------------------------------------------------------

    ;------------------------------------------------------
    ; initialize flags, determines which parameter currently
    ; reading from file 
    ;------------------------------------------------------
    start_flag               = 0L
    stop_flag                = 0L
    hvps_mcp_start_vmon_flag = 0L
    hvps_mcp_stop_vmon_flag  = 0L
    hvps_collp_vmon_flag     = 0L
    hvps_colln_vmon_flag     = 0L
    hvps_optics_vmon_flag    = 0L
    hvps_steering_vmon_flag  = 0L
    hvps_mcp_start_imon_flag = 0L
    hvps_mcp_stop_imon_flag  = 0L
    hvps_collp_imon_flag     = 0L
    hvps_colln_imon_flag     = 0L
    hvps_optics_imon_flag    = 0L
    lvps_15vp_vmon_flag      = 0L
    lvps_15vn_vmon_flag      = 0L
    lvps_5vp_vmon_flag       = 0L
    lvps_5vn_vmon_flag       = 0L
    temp0_flag               = 0L
    temp1_flag               = 0L

print, 'start hskp ltt ... ', systime ()
    
    day = 0L
    month = 0L

    doy = l_ConvYMD2DOY (setup_parms.start_date[0], setup_parms.start_date[1], $
                        setup_parms.start_date[2])
    cal_doy = FindDOYSinceLaunch (setup_parms.start_date[0], doy)


    cal_month = setup_parms.start_date[1]

    nyears = setup_parms.stop_date[0] - setup_parms.start_date[0]
    nmonths = setup_parms.stop_date[1] - setup_parms.start_date[1]


    yrs = setup_parms.start_date[0]
    while (yrs le setup_parms.stop_date[0]) do begin
       yr_path = strcompress(('/' + string (yrs)), /remove_all)

       mloops = 1L
       if (setup_parms.start_date[1] gt setup_parms.stop_date[1]) then begin 
          mloops = 2L
       endif else begin
          if (setup_parms.start_date[1] ge setup_parms.stop_date[1]) AND $
             (setup_parms.start_date[0] ne setup_parms.stop_date[0]) then begin 
             mloops = 2L
          endif
       endelse

       for m = 0L, mloops - 1L do begin

          if (m eq 0L) AND (mloops gt 1L) then begin
             start_mon_idx = setup_parms.start_date[1]
             stop_mon_idx = 12 
          endif else begin
              if (m eq 1L) AND (mloops gt 1L) then begin
                 start_mon_idx = 1 
                 stop_mon_idx =  setup_parms.stop_date[1] 
                 yrs = yrs + 1L
              endif else begin
                 start_mon_idx = setup_parms.start_date[1]
                 stop_mon_idx = setup_parms.stop_date[1]
              endelse
          endelse


       for mon = start_mon_idx, stop_mon_idx do begin
          ;find month string and ndays in the month
          ndays = DaysNMonth(yrs, mon, mon_str)
          mon_path = "/" + mon_str

          if (mon eq setup_parms.start_date[1]) AND (nyears ge 1L) AND (m eq 0L) then begin
             days_start_idx = setup_parms.start_date[2]
             ndays = DaysNMonth(yrs, mon, mon_str)
             days_stop_idx = ndays
          endif else begin
             if (nyears ge 1L) AND (mloops ge 1L) AND (m eq 1L) AND (mon eq setup_parms.stop_date[1]) then begin
                days_start_idx = 1
                days_stop_idx = setup_parms.stop_date[2]
             endif else begin
                if (nyears ge 1L) AND (mloops ge 1L) AND (m eq 1L) then begin
                   days_start_idx = 1 
                   ndays = DaysNMonth(yrs, mon, mon_str)
                   days_stop_idx = ndays 
                endif else begin
                   days_start_idx = setup_parms.start_date[2]
                   days_stop_idx = setup_parms.stop_date[2]
                endelse
             endelse
          endelse

 
          for days = days_start_idx, days_stop_idx do begin


              start_indx       = 0L
              stop_indx        = 0L
              hskp_mstv_indx   = 0L
              hskp_mspv_indx   = 0L
              hskp_cpv_indx    = 0L
              hskp_cnv_indx    = 0L
              hskp_ov_indx     = 0L
              hskp_sv_indx     = 0L
              hskp_msti_indx   = 0L
              hskp_mspi_indx   = 0L
              hskp_cpi_indx    = 0L
              hskp_cni_indx    = 0L
              hskp_oi_indx     = 0L
              hskp_l15vpv_indx = 0L
              hskp_l15vnv_indx = 0L
              hskp_l5vpv_indx  = 0L
              hskp_l5vnv_indx  = 0L
              hskp_temp0_indx  = 0L
              hskp_temp1_indx  = 0L

              starts[*, *] = 0
              stops[*, *] = 0
              hvps_collp_imon[*] = 0
              hvps_colln_imon[*] = 0
              temp0[*] = 0
              temp1[*] = 0
              hvps_mcp_start_vmon[*] = 0
              hvps_mcp_stop_vmon[*] = 0
              hvps_optics_vmon[*] = 0
              hvps_steering_vmon[*] = 0
              hvps_collp_vmon[*] = 0
              hvps_colln_vmon[*] = 0
              hvps_mcp_start_imon[*] = 0
              hvps_mcp_stop_imon[*] = 0
              hvps_optics_imon[*] = 0
              lvps_15vp_vmon[*] = 0
              lvps_15vn_vmon[*] = 0
              lvps_5vp_vmon[*] = 0
              lvps_5vn_vmon[*] = 0 

              ;--------------------------------------------
              ; build filename
              ; '00002359_hks0.dmp'),  $
              ;--------------------------------------------
              yr = (yrs - EPOCH_YR) 


              fn = strcompress(('l' + $
                                string(format='(I2.2)', yr) +  $
                                string(format='(I2.2)',mon) + $
                                string(format='(I2.2)',days) + $
                                '00002359_hsk0.dmp'),  $
                                /REMOVE_ALL)


              filename = strcompress((Hskp_FilePath + yr_path + mon_path + '/' + fn) , /remove_all)
              ;filename = 'HSKP/'+fn 


              print, 'Processing ', filename , '......'
              setup_parms.filename = filename

              ;--------------------------------------------
              ; check for file existence
              ;--------------------------------------------
              openr, ifd, filename, /get_lun, ERROR=err

              if (err ne 0) then begin
                 print, !Err_String 
                 RETURN, FAILURE
              endif

              if (err eq 0) then begin
                 line = ' '

                 readf, ifd, header
                 snglsspins = 0L
                 hskpspins  = 0L


                 while (NOT EOF (ifd)) do begin
   
                    readf, ifd, line
   
                    if (line eq '****************** Start Singles ******************') OR $
                       (start_flag eq 1) then begin
 
                       temp1_flag = 0L

                       line = strtrim (line, 2)
                       parts = str_sep (line, ' ')
                       time_match = strpos(parts[0], ':')
                       if (time_match gt -1) then begin 
                          line = strcompress (line)
                          parts = str_sep(line, ' ')
                          starts[*, start_indx] = parts[1:45]
                          start_indx = start_indx + 1L
                          snglsspins = snglsspins + 1
                       endif
             
                       start_flag = 1
                 
                    endif

                    if (line eq '****************** Stop Singles ******************') OR $
                       (stop_flag eq 1) then begin
   
                       start_flag = 0
   
                       line = strtrim (line, 2)
                       parts = str_sep (line, ' ')
                       time_match = strpos(parts[0], ':')
                       if (time_match gt -1) then begin 
                          line = strcompress (line)
                          parts = str_sep(line, ' ')
                          stops[*, stop_indx] = parts[1:45]
                          stop_indx = stop_indx + 1L
                       endif
   
                       stop_flag = 1 
   
                    endif
   
                    if (line eq '****************** Hvps MCP Start Vmon ******************') OR $
                       (hvps_mcp_start_vmon_flag eq 1) then begin
   
                       stop_flag = 0 
   
                       line = strtrim (line, 2)
                       parts = str_sep (line, ' ')
                       time_match = strpos(parts[0], ':')
                       if (time_match gt -1) then begin 
                          line = strcompress (line)
                          parts = str_sep(line, ' ')
                          hvps_mcp_start_vmon[hskp_mstv_indx] = parts[1]
                          hskp_mstv_indx = hskp_mstv_indx + 1L
                       endif
   
                       hvps_mcp_start_vmon_flag = 1 
   
                    endif
   
                    if (line eq '****************** Hvps MCP Stop Vmon ******************') OR $
                       (hvps_mcp_stop_vmon_flag eq 1) then begin
   
                       hvps_mcp_start_vmon_flag = 0 
   
                       line = strtrim (line, 2)
                       parts = str_sep (line, ' ')
                       time_match = strpos(parts[0], ':')
                       if (time_match gt -1) then begin 
                          line = strcompress (line)
                          parts = str_sep(line, ' ')
                          hvps_mcp_stop_vmon[hskp_mspv_indx] = parts[1]
                          hskp_mspv_indx = hskp_mspv_indx + 1L
                       endif
   
                       hvps_mcp_stop_vmon_flag = 1 
   
                    endif
   
                    if (line eq '****************** Hvps CollP Vmon ******************') OR $
                       (hvps_collp_vmon_flag eq 1) then begin
   
                       hvps_mcp_stop_vmon_flag = 0
   
                       line = strtrim (line, 2)
                    parts = str_sep (line, ' ')
                       time_match = strpos(parts[0], ':')
                       if (time_match gt -1) then begin 
                          line = strcompress (line)
                          parts = str_sep(line, ' ')
                          hvps_collp_vmon[hskp_cpv_indx] = parts[1]
                          hskp_cpv_indx = hskp_cpv_indx + 1L
                       endif
   
                       hvps_collp_vmon_flag = 1 
   
                    endif
   
                    if (line eq '****************** Hvps CollN Vmon ******************') OR $
                       (hvps_colln_vmon_flag eq 1) then begin
   
                       hvps_collp_vmon_flag = 0
   
                       line = strtrim (line, 2)
                       parts = str_sep (line, ' ')
                       time_match = strpos(parts[0], ':')
                       if (time_match gt -1) then begin 
                          line = strcompress (line)
                          parts = str_sep(line, ' ')
                          hvps_colln_vmon[hskp_cnv_indx] = parts[1]
                          hskp_cnv_indx = hskp_cnv_indx + 1L
                       endif
      
                       hvps_colln_vmon_flag = 1 
   
                    endif
   
                    if (line eq '****************** Hvps Optics Vmon ******************') OR $
                       (hvps_optics_vmon_flag eq 1) then begin
   
                       hvps_colln_vmon_flag = 0 
   
                       line = strtrim (line, 2)
                       parts = str_sep (line, ' ')
                       time_match = strpos(parts[0], ':')
                       if (time_match gt -1) then begin 
                          line = strcompress (line)
                          parts = str_sep(line, ' ')
                          hvps_optics_vmon[hskp_ov_indx] = parts[1]
                          hskp_ov_indx = hskp_ov_indx + 1L
                       endif
   
                       hvps_optics_vmon_flag = 1 
   
                    endif
   
                    if (line eq '****************** Hvps Steering Vmon ******************') OR $
                       (hvps_steering_vmon_flag eq 1) then begin
   
                       hvps_optics_vmon_flag = 0
   
                       line = strtrim (line, 2)
                       parts = str_sep (line, ' ')
                       time_match = strpos(parts[0], ':')
                       if (time_match gt -1) then begin 
                          line = strcompress (line)
                          parts = str_sep(line, ' ')
                          hvps_steering_vmon[hskp_sv_indx] = parts[1]
                          hskp_sv_indx = hskp_sv_indx + 1L
                       endif
   
                       hvps_steering_vmon_flag = 1 
   
                    endif
      
                    if (line eq '****************** Hvps MCP Start Imon ******************') OR $
                       (hvps_mcp_start_imon_flag eq 1) then begin
   
                       hvps_steering_vmon_flag = 0
   
                       line = strtrim (line, 2)
                       parts = str_sep (line, ' ')
                       time_match = strpos(parts[0], ':')
                       if (time_match gt -1) then begin 
                          line = strcompress (line)
                          parts = str_sep(line, ' ')
                          hvps_mcp_start_imon[hskp_msti_indx] = parts[1]
                          hskp_msti_indx = hskp_msti_indx + 1L
                          endif
   
                       hvps_mcp_start_imon_flag = 1 
   
                    endif
   
                    if (line eq '****************** Hvps MCP Stop Imon ******************') OR $
                       (hvps_mcp_stop_imon_flag eq 1) then begin
   
                       hvps_mcp_start_imon_flag = 0
   
                       line = strtrim (line, 2)
                       parts = str_sep (line, ' ')
                       time_match = strpos(parts[0], ':')
                       if (time_match gt -1) then begin 
                          line = strcompress (line)
                          parts = str_sep(line, ' ')
                          hvps_mcp_stop_imon[hskp_mspi_indx] = parts[1]
                          hskp_mspi_indx = hskp_mspi_indx + 1L
                       endif
   
                       hvps_mcp_stop_imon_flag = 1 
   
                    endif
          
                    if (line eq '****************** Hvps CollP Imon ******************') OR $
                       (hvps_collp_imon_flag eq 1) then begin
   
                       hvps_mcp_stop_imon_flag = 0
   
                       line = strtrim (line, 2)
                       parts = str_sep (line, ' ')
                       time_match = strpos(parts[0], ':')
                       if (time_match gt -1) then begin 
                          line = strcompress (line)
                          parts = str_sep(line, ' ')
                          hvps_collp_imon[hskp_cpi_indx] = parts[1]
                          hskp_cpi_indx = hskp_cpi_indx + 1L
                       endif
   
                       hvps_collp_imon_flag = 1 
   
                    endif
   
                    if (line eq '****************** Hvps CollN Imon ******************') OR $
                       (hvps_colln_imon_flag eq 1) then begin
   
                       hvps_collp_imon_flag = 0
   
                       line = strtrim (line, 2)
                       parts = str_sep (line, ' ')
                       time_match = strpos(parts[0], ':')
                       if (time_match gt -1) then begin 
                          line = strcompress (line)
                          parts = str_sep(line, ' ')
                          hvps_colln_imon[hskp_cni_indx] = parts[1]
                          hskp_cni_indx = hskp_cni_indx + 1L
                       endif
   
                       hvps_colln_imon_flag = 1 
   
                    endif
   
                    if (line eq '****************** Hvps Optics Imon ******************') OR $
                       (hvps_optics_imon_flag eq 1) then begin
   
                       hvps_colln_imon_flag = 0
   
                       line = strtrim (line, 2)
                       parts = str_sep (line, ' ')
                       time_match = strpos(parts[0], ':')
                       if (time_match gt -1) then begin 
                          line = strcompress (line)
                          parts = str_sep(line, ' ')
                          hvps_optics_imon[hskp_oi_indx] = parts[1]
                          hskp_cni_indx = hskp_oi_indx + 1L
                       endif
   
                       hvps_optics_imon_flag = 1 
   
                    endif
   
                    if (line eq '****************** Lvps 15vp Vmon ******************') OR $
                       (lvps_15vp_vmon_flag eq 1) then begin
   
                       hvps_optics_imon_flag = 0
   
                       line = strtrim (line, 2)
                       parts = str_sep (line, ' ')
                       time_match = strpos(parts[0], ':')
                       if (time_match gt -1) then begin 
                          line = strcompress (line)
                          parts = str_sep(line, ' ')
                          lvps_15vp_vmon[hskp_l15vpv_indx] = parts[1]
                          hskp_l15vpv_indx = hskp_l15vpv_indx + 1L
                       endif
   
                       lvps_15vp_vmon_flag = 1 
   
                    endif
   
                    if (line eq '****************** Lvps 15vn Vmon ******************') OR $
                       (lvps_15vn_vmon_flag eq 1) then begin
   
                       lvps_15vp_vmon_flag = 0
   
                       line = strtrim (line, 2)
                       parts = str_sep (line, ' ')
                       time_match = strpos(parts[0], ':')
                       if (time_match gt -1) then begin 
                          line = strcompress (line)
                          parts = str_sep(line, ' ')
                          lvps_15vn_vmon[hskp_l15vnv_indx] = parts[1]
                          hskp_l15vnv_indx = hskp_l15vnv_indx + 1L
                       endif
   
                       lvps_15vn_vmon_flag = 1 
   
                    endif
   
                    if (line eq '****************** Lvps 5vp Vmon ******************') OR $
                       (lvps_5vp_vmon_flag eq 1) then begin
   
                       lvps_15vn_vmon_flag = 0
   
                       line = strtrim (line, 2)
                       parts = str_sep (line, ' ')
                       time_match = strpos(parts[0], ':')
                       if (time_match gt -1) then begin 
                          line = strcompress (line)
                          parts = str_sep(line, ' ')
                          lvps_5vp_vmon[hskp_l5vpv_indx] = parts[1]
                          hskp_l5vpv_indx = hskp_l5vpv_indx + 1L
                       endif
   
                       lvps_5vp_vmon_flag = 1 
   
                    endif
   
                    if (line eq '****************** Lvps 5vn Vmon ******************') OR $
                       (lvps_5vn_vmon_flag eq 1) then begin

                       lvps_5vp_vmon_flag = 0
   
                       line = strtrim (line, 2)
                       parts = str_sep (line, ' ')
                       time_match = strpos(parts[0], ':')
                       if (time_match gt -1) then begin 
                          line = strcompress (line)
                          parts = str_sep(line, ' ')
                          lvps_5vn_vmon[hskp_l5vnv_indx] = parts[1]
                          hskp_l5vnv_indx = hskp_l5vnv_indx + 1L
                       endif
   
                       lvps_5vn_vmon_flag = 1 
   
                    endif
   
                    if (line eq '****************** Temperature 0 ******************') OR $
                       (temp0_flag eq 1) then begin
      
                       lvps_5vn_vmon_flag = 0
   
                       line = strtrim (line, 2)
                       parts = str_sep (line, ' ')
                       time_match = strpos(parts[0], ':')
                       if (time_match gt -1) then begin 
                          line = strcompress (line)
                          parts = str_sep(line, ' ')
                          temp0[hskp_temp0_indx] = parts[1]
                          hskp_temp0_indx = hskp_temp0_indx + 1L
                       endif
   
                       temp0_flag = 1 
   
                    endif
   
                    if (line eq '****************** Temperature 1 ******************') OR $
                       (temp1_flag eq 1) then begin
   
                       temp0_flag = 0
   
                       line = strtrim (line, 2)
                       parts = str_sep (line, ' ')
                       time_match = strpos(parts[0], ':')
                       if (time_match gt -1) then begin 
                          line = strcompress (line)
                          parts = str_sep(line, ' ')
                          temp1[hskp_temp1_indx] = parts[1]
                          hskp_temp1_indx = hskp_temp1_indx + 1L
                          hskpspins = hskpspins + 1L
                       endif
   
                       temp1_flag = 1 
   
                    endif 
    
                 endwhile
          
                 close, ifd
                 free_lun, ifd
   
                 ;------------------------------------------------------
                 ; daily averaging
                 ;------------------------------------------------------
                 startspins = snglsspins
                 nd_indx = where (starts eq NO_DATA, ndcnt)
                 if (ndcnt gt 0) then begin
                    starts[nd_indx] = 0  
                    realspins = ndcnt/45
                    startspins = snglsspins - realspins 
                 endif    

                 starts_daily[day] = total(starts)/startspins

                 ;------------------------------------------------------
                 ;------------------------------------------------------
   
                 stopspins = snglsspins
                 nd_indx = where (stops eq NO_DATA, ndcnt)
                 if (ndcnt gt 0) then begin
                    stops[nd_indx] = 0  
                    realspins = ndcnt/45
                    stopspins = snglsspins - realspins
                 endif 

                 stops_daily[day] = total(stops)/stopspins
      
                 ;------------------------------------------------------
                 ;------------------------------------------------------
   
                 nspins = hskpspins
                 nd_indx = where (hvps_mcp_start_vmon eq NO_DATA, ndcnt)
                 if (ndcnt gt 0) then begin
                    hvps_mcp_start_vmon[nd_indx] = 0  
                    nspins = hskpspins - ndcnt 
                 endif 

                 hvps_mcp_start_vmon_daily[day] = total(hvps_mcp_start_vmon)/nspins

                 ;------------------------------------------------------
                 ;------------------------------------------------------
                 nspins = hskpspins
                 nd_indx = where (hvps_mcp_stop_vmon eq NO_DATA, ndcnt)
                 if (ndcnt gt 0) then begin
                    hvps_mcp_stop_vmon[nd_indx] = 0  
                    nspins = hskpspins - ndcnt
                 endif 

                 hvps_mcp_stop_vmon_daily[day] = total(hvps_mcp_stop_vmon)/nspins

                 ;------------------------------------------------------
                 ;------------------------------------------------------
                 nspins = hskpspins
                 nd_indx = where (hvps_collp_vmon eq NO_DATA, ndcnt)
                 if (ndcnt gt 0) then begin
                    hvps_collp_vmon[nd_indx] =  0  
                    nspins = hskpspins - ndcnt 
                 endif 
   
                 hvps_collp_vmon_daily[day] = total(hvps_collp_vmon)/nspins
   
                 ;------------------------------------------------------
                 ;------------------------------------------------------
                 nspins = hskpspins
                 nd_indx = where (hvps_colln_vmon eq NO_DATA, ndcnt)
                 if (ndcnt gt 0) then begin
                    hvps_colln_vmon[nd_indx] = 0  
                    nspins = hskpspins - ndcnt
                 endif 

                 hvps_colln_vmon_daily[day] = total(hvps_colln_vmon)/nspins

                 ;------------------------------------------------------
                 ;------------------------------------------------------
                 nspins = hskpspins
                 nd_indx = where (hvps_optics_vmon eq NO_DATA, ndcnt)
                 if (ndcnt gt 0) then begin
                    hvps_optics_vmon[nd_indx] = 0  
                    nspins = hskpspins - ndcnt 
                 endif 
   
                 hvps_optics_vmon_daily[day] = total(hvps_optics_vmon)/nspins
   
                 ;------------------------------------------------------
                 ;------------------------------------------------------
                 nspins = hskpspins
                 nd_indx = where (hvps_steering_vmon eq NO_DATA, ndcnt)
                 if (ndcnt gt 0) then begin
                    hvps_steering_vmon[nd_indx] = 0  
                    nspins = hskpspins - ndcnt 
                 endif 
   
                 hvps_steering_vmon_daily[day] = total(hvps_steering_vmon)/nspins
   
                 ;------------------------------------------------------
                 ;------------------------------------------------------
                 nspins = hskpspins
                 nd_indx = where (hvps_mcp_start_imon eq NO_DATA, ndcnt)
                 if (ndcnt gt 0) then begin
                    hvps_mcp_start_imon[nd_indx] = 0  
                    nspins = hskpspins - ndcnt 
                 endif 
   
                 hvps_mcp_start_imon_daily[day] = total(hvps_mcp_start_imon)/nspins
   
                 ;------------------------------------------------------
                 ;------------------------------------------------------
                 nspins = hskpspins
                 nd_indx = where (hvps_mcp_stop_imon eq NO_DATA, ndcnt)
                 if (ndcnt gt 0) then begin
                    hvps_mcp_stop_imon[nd_indx] = 0  
                    nspins = hskpspins - ndcnt
                 endif 
   
                 hvps_mcp_stop_imon_daily[day] = total(hvps_mcp_stop_imon)/nspins

      
                 ;------------------------------------------------------
                 ;------------------------------------------------------
                 nspins = hskpspins
                 nd_indx = where (hvps_collp_imon eq NO_DATA, ndcnt)
                 if (ndcnt gt 0) then begin
                    hvps_collp_imon[nd_indx] = 0  
                    nspins = hskpspins - ndcnt 
                 endif 
   
                 hvps_collp_imon_daily[day] = total(hvps_collp_imon)/nspins
   
                 ;------------------------------------------------------
                 ;------------------------------------------------------
                 nspins = hskpspins
                 nd_indx = where (hvps_colln_imon eq NO_DATA, ndcnt)
                 if (ndcnt gt 0) then begin
                    hvps_colln_imon[nd_indx] = 0  
                    nspins = hskpspins - ndcnt
                 endif 
   
                 hvps_colln_imon_daily[day] = total(hvps_colln_imon)/nspins


                 ;------------------------------------------------------
                 ;------------------------------------------------------
                 nspins = hskpspins
                 nd_indx = where (hvps_optics_imon eq NO_DATA, ndcnt)
                 if (ndcnt gt 0) then begin
                    hvps_optics_imon[nd_indx] = 0  
                    nspins = hskpspins - ndcnt 
                 endif 
   
                 hvps_optics_imon_daily[day] = total(hvps_optics_imon)/nspins
   
   
                 ;------------------------------------------------------
                 ;------------------------------------------------------
                 nspins = hskpspins
                 nd_indx = where (lvps_15vp_vmon eq NO_DATA, ndcnt)
                 if (ndcnt gt 0) then begin
                    lvps_15vp_vmon[nd_indx] = 0  
                    nspins = hskpspins - ndcnt
                 endif 
   
                 lvps_15vp_vmon_daily[day] = total(lvps_15vp_vmon)/nspins
   
   
                 ;------------------------------------------------------
                 ;------------------------------------------------------
                 nspins = hskpspins
                 nd_indx = where (lvps_15vn_vmon eq NO_DATA, ndcnt)
                 if (ndcnt gt 0) then begin
                    lvps_15vn_vmon[nd_indx] = 0  
                    nspins = hskpspins - ndcnt 
                 endif 
   
                 lvps_15vn_vmon_daily[day] = total(lvps_15vn_vmon)/nspins
   
   
                 ;------------------------------------------------------
                 ;------------------------------------------------------
                 nspins = hskpspins
                 nd_indx = where (lvps_5vn_vmon eq NO_DATA, ndcnt)
                 if (ndcnt gt 0) then begin
                    lvps_5vn_vmon[nd_indx] = 0  
                    nspins = hskpspins - ndcnt 
                 endif 
   
                 lvps_5vn_vmon_daily[day] = total(lvps_5vn_vmon)/nspins
   
   
                 ;------------------------------------------------------
                 ;------------------------------------------------------
                 nspins = hskpspins
                 nd_indx = where (temp0 eq NO_DATA, ndcnt)
                 if (ndcnt gt 0) then begin
                    temp0[nd_indx] = 0  
                    nspins = hskpspins - ndcnt
                 endif 

                 temp0_daily[day] = total(temp0)/nspins

                 ;------------------------------------------------------
                 ;------------------------------------------------------
                 nspins = hskpspins
                 nd_indx = where (temp1 eq NO_DATA, ndcnt)
                 if (ndcnt gt 0) then begin
                    temp1[nd_indx] = 0  
                    nspins = hskpspins - ndcnt 
                 endif 
   
                 temp1_daily[day] = total(temp1)/nspins

                 ;------------------------------------------------------
                 ;------------------------------------------------------
                 endif
   
                 X_daily[day] = cal_doy 
                 cal_doy = cal_doy + 1L
                 day    = day + 1L

           endfor  ;--- daily ---

         ;------------------------------------------------------
         ; MONTHLY averaging
         ;------------------------------------------------------
         starts_monthly[month] = total(starts_daily)/day

         ;------------------------------------------------------
         ;------------------------------------------------------
         stops_monthly[month] = total(stops_daily)/day
      
         ;------------------------------------------------------
         ;------------------------------------------------------
         hvps_mcp_start_vmon_monthly[month] = total(hvps_mcp_start_vmon_daily)/day

         ;------------------------------------------------------
         ;------------------------------------------------------
         hvps_mcp_stop_vmon_monthly[month] = total(hvps_mcp_stop_vmon_daily)/day

         ;------------------------------------------------------
         ;------------------------------------------------------
         hvps_collp_vmon_monthly[month] = total(hvps_collp_vmon_daily)/day
   
         ;------------------------------------------------------
         ;------------------------------------------------------
         hvps_colln_vmon_monthly[month] = total(hvps_colln_vmon_daily)/day

         ;------------------------------------------------------
         ;------------------------------------------------------
         hvps_optics_vmon_monthly[month] = total(hvps_optics_vmon_daily)/day
   
         ;------------------------------------------------------
         ;------------------------------------------------------
         hvps_steering_vmon_monthly[month] = total(hvps_steering_vmon_daily)/day
   
         ;------------------------------------------------------
         ;------------------------------------------------------
         hvps_mcp_start_imon_monthly[month] = total(hvps_mcp_start_imon_daily)/day
   
         ;------------------------------------------------------
         ;------------------------------------------------------
         hvps_mcp_stop_imon_monthly[month] = total(hvps_mcp_stop_imon_daily)/day

   
         ;------------------------------------------------------
         ;------------------------------------------------------
         hvps_collp_imon_monthly[month] = total(hvps_collp_imon_daily)/day
   
         ;------------------------------------------------------
         ;------------------------------------------------------
         hvps_colln_imon_monthly[month] = total(hvps_colln_imon_daily)/day

         ;------------------------------------------------------
         ;------------------------------------------------------
         hvps_optics_imon_monthly[month] = total(hvps_optics_imon_daily)/day
   
         ;------------------------------------------------------
         ;------------------------------------------------------
         lvps_15vp_vmon_monthly[month] = total(lvps_15vp_vmon_daily)/day
   
         ;------------------------------------------------------
         ;------------------------------------------------------
         lvps_15vn_vmon_monthly[month] = total(lvps_15vn_vmon_daily)/day
   
         ;------------------------------------------------------
         ;------------------------------------------------------
         lvps_5vn_vmon_monthly[month] = total(lvps_5vn_vmon_daily)/day
   
         ;------------------------------------------------------
         ;------------------------------------------------------
         temp0_monthly[month] = total(temp0_daily)/day

         ;------------------------------------------------------
         ;------------------------------------------------------
         temp1_monthly[month] = total(temp1_daily)/day


         X_monthly[month] = cal_month
         cal_month = cal_month + 1L
         month = month + 1L

         endfor ;--- monthly ---
        endfor ;--- nloops ---
        yrs = yrs + 1L
    endwhile  ;--- years ---

    ptr = ptr_new (X_daily, /NO_COPY)
    l_hskpltt.time_daily = ptr 
    ptr = ptr_new (X_monthly, /NO_COPY)
    l_hskpltt.time_monthly = ptr 

    ptr = ptr_new (starts_daily, /NO_COPY)
    l_hskpltt.starts_daily = ptr 
    ptr = ptr_new (starts_monthly, /NO_COPY)
    l_hskpltt.starts_monthly = ptr 

    ptr = ptr_new (stops_daily, /NO_COPY)
    l_hskpltt.stops_daily = ptr 
    ptr = ptr_new (stops_monthly, /NO_COPY)
    l_hskpltt.stops_monthly = ptr 

    ptr = ptr_new (hvps_collp_imon_daily, /NO_COPY)
    l_hskpltt.hvps_collp_imon_daily = ptr 
    ptr = ptr_new (hvps_collp_imon_monthly, /NO_COPY)
    l_hskpltt.hvps_collp_imon_monthly = ptr 

    ptr = ptr_new (hvps_colln_imon_daily, /NO_COPY)
    l_hskpltt.hvps_colln_imon_daily = ptr 
    ptr = ptr_new (hvps_colln_imon_monthly, /NO_COPY)
    l_hskpltt.hvps_colln_imon_monthly = ptr 

    ptr = ptr_new (temp0_daily, /NO_COPY)
    l_hskpltt.temp0_daily = ptr 
    ptr = ptr_new (temp0_monthly, /NO_COPY)
    l_hskpltt.temp0_monthly = ptr 

    ptr = ptr_new (temp1_daily, /NO_COPY)
    l_hskpltt.temp1_daily = ptr 
    ptr = ptr_new (temp1_monthly, /NO_COPY)
    l_hskpltt.temp1_monthly = ptr 

    ptr = ptr_new (hvps_mcp_start_vmon_daily, /NO_COPY)
    l_hskpltt.hvps_mcp_start_vmon_daily = ptr 
    ptr = ptr_new (hvps_mcp_start_vmon_monthly, /NO_COPY)
    l_hskpltt.hvps_mcp_start_vmon_monthly = ptr 

    ptr = ptr_new (hvps_mcp_stop_vmon_daily, /NO_COPY)
    l_hskpltt.hvps_mcp_stop_vmon_daily = ptr 
    ptr = ptr_new (hvps_mcp_stop_vmon_monthly, /NO_COPY)
    l_hskpltt.hvps_mcp_stop_vmon_monthly = ptr 

    ptr = ptr_new (hvps_optics_vmon_daily, /NO_COPY)
    l_hskpltt.hvps_optics_vmon_daily = ptr 
    ptr = ptr_new (hvps_optics_vmon_monthly, /NO_COPY)
    l_hskpltt.hvps_optics_vmon_monthly = ptr 

    ptr = ptr_new (hvps_steering_vmon_daily, /NO_COPY)
    l_hskpltt.hvps_steering_vmon_daily = ptr 
    ptr = ptr_new (hvps_steering_vmon_monthly, /NO_COPY)
    l_hskpltt.hvps_steering_vmon_monthly = ptr 

    ptr = ptr_new (hvps_collp_vmon_daily, /NO_COPY)
    l_hskpltt.hvps_collp_vmon_daily = ptr 
    ptr = ptr_new (hvps_collp_vmon_monthly, /NO_COPY)
    l_hskpltt.hvps_collp_vmon_monthly = ptr 

    ptr = ptr_new (hvps_colln_vmon_daily, /NO_COPY)
    l_hskpltt.hvps_colln_vmon_daily = ptr 
    ptr = ptr_new (hvps_colln_vmon_monthly, /NO_COPY)
    l_hskpltt.hvps_colln_vmon_monthly = ptr 

    ptr = ptr_new (hvps_mcp_start_imon_daily, /NO_COPY)
    l_hskpltt.hvps_mcp_start_imon_daily = ptr 
    ptr = ptr_new (hvps_mcp_start_imon_monthly, /NO_COPY)
    l_hskpltt.hvps_mcp_start_imon_monthly = ptr 

    ptr = ptr_new (hvps_mcp_stop_imon_daily, /NO_COPY)
    l_hskpltt.hvps_mcp_stop_imon_daily = ptr 
    ptr = ptr_new (hvps_mcp_stop_imon_monthly, /NO_COPY)
    l_hskpltt.hvps_mcp_stop_imon_monthly = ptr 

    ptr = ptr_new (hvps_optics_imon_daily, /NO_COPY)
    l_hskpltt.hvps_optics_imon_daily = ptr 
    ptr = ptr_new (hvps_optics_imon_monthly, /NO_COPY)
    l_hskpltt.hvps_optics_imon_monthly = ptr 

    ptr = ptr_new (lvps_15vp_vmon_daily, /NO_COPY)
    l_hskpltt.lvps_15vp_vmon_daily = ptr 
    ptr = ptr_new (lvps_15vp_vmon_monthly, /NO_COPY)
    l_hskpltt.lvps_15vp_vmon_monthly = ptr 

    ptr = ptr_new (lvps_15vn_vmon_daily, /NO_COPY)
    l_hskpltt.lvps_15vn_vmon_daily = ptr 
    ptr = ptr_new (lvps_15vn_vmon_monthly, /NO_COPY)
    l_hskpltt.lvps_15vn_vmon_monthly = ptr 

    ptr = ptr_new (lvps_5vp_vmon_daily, /NO_COPY)
    l_hskpltt.lvps_5vp_vmon_daily = ptr 
    ptr = ptr_new (lvps_5vp_vmon_monthly, /NO_COPY)
    l_hskpltt.lvps_5vp_vmon_monthly = ptr 

    ptr = ptr_new (lvps_5vn_vmon_daily, /NO_COPY)
    l_hskpltt.lvps_5vn_vmon_daily = ptr 
    ptr = ptr_new (lvps_5vn_vmon_monthly, /NO_COPY)
    l_hskpltt.lvps_5vn_vmon_monthly = ptr 

    setup_parms.filename = ' '


RETURN, SUCCESS
END

;---------------------------------------------------------------------------
;  Free allocated memory.
;---------------------------------------------------------------------------
PRO CleanUp, l_hskpltt

    if (ptr_valid(l_hskpltt.time_daily)) then ptr_free, l_hskpltt.time_daily
    if (ptr_valid(l_hskpltt.time_monthly)) then ptr_free, l_hskpltt.time_monthly 

    if (ptr_valid(l_hskpltt.starts_daily)) then ptr_free, l_hskpltt.starts_daily
    if (ptr_valid(l_hskpltt.starts_monthly)) then ptr_free, l_hskpltt.starts_monthly

    if (ptr_valid(l_hskpltt.stops_daily)) then ptr_free, l_hskpltt.stops_daily
    if (ptr_valid(l_hskpltt.stops_monthly)) then ptr_free, l_hskpltt.stops_monthly

    if (ptr_valid(l_hskpltt.hvps_collp_imon_daily)) then ptr_free, l_hskpltt.hvps_collp_imon_daily
    if (ptr_valid(l_hskpltt.hvps_collp_imon_monthly)) then ptr_free, l_hskpltt.hvps_collp_imon_monthly

    if (ptr_valid(l_hskpltt.hvps_colln_imon_daily)) then ptr_free, l_hskpltt.hvps_colln_imon_daily
    if (ptr_valid(l_hskpltt.hvps_colln_imon_monthly)) then ptr_free, l_hskpltt.hvps_colln_imon_monthly

    if (ptr_valid(l_hskpltt.temp0_daily)) then ptr_free, l_hskpltt.temp0_daily
    if (ptr_valid(l_hskpltt.temp0_monthly)) then ptr_free, l_hskpltt.temp0_monthly

    if (ptr_valid(l_hskpltt.temp1_daily)) then ptr_free, l_hskpltt.temp1_daily
    if (ptr_valid(l_hskpltt.temp1_monthly)) then ptr_free, l_hskpltt.temp1_monthly

    if (ptr_valid(l_hskpltt.hvps_mcp_start_vmon_daily)) then $
       ptr_free,l_hskpltt.hvps_mcp_start_vmon_daily  
    if (ptr_valid(l_hskpltt.hvps_mcp_start_vmon_monthly)) then $
       ptr_free,  l_hskpltt.hvps_mcp_start_vmon_monthly

    if (ptr_valid(l_hskpltt.hvps_mcp_stop_vmon_daily)) then $
       ptr_free, l_hskpltt.hvps_mcp_stop_vmon_daily
    if (ptr_valid(l_hskpltt.hvps_mcp_stop_vmon_monthly)) then $
       ptr_free, l_hskpltt.hvps_mcp_stop_vmon_monthly

    if (ptr_valid(l_hskpltt.hvps_optics_vmon_daily)) then $
       ptr_free, l_hskpltt.hvps_optics_vmon_daily 
    if (ptr_valid(l_hskpltt.hvps_optics_vmon_monthly)) then $
       ptr_free , l_hskpltt.hvps_optics_vmon_monthly

    if (ptr_valid(l_hskpltt.hvps_steering_vmon_daily)) then $
       ptr_free, l_hskpltt.hvps_steering_vmon_daily 
    if (ptr_valid(l_hskpltt.hvps_steering_vmon_monthly)) then $
       ptr_free, l_hskpltt.hvps_steering_vmon_monthly 

    if (ptr_valid(l_hskpltt.hvps_collp_vmon_daily)) then $ 
       ptr_free, l_hskpltt.hvps_collp_vmon_daily 
    if (ptr_valid(l_hskpltt.hvps_collp_vmon_monthly)) then $
       ptr_free, l_hskpltt.hvps_collp_vmon_monthly

    if (ptr_valid(l_hskpltt.hvps_colln_vmon_daily)) then $ 
       ptr_free, l_hskpltt.hvps_colln_vmon_daily  
    if (ptr_valid(l_hskpltt.hvps_colln_vmon_monthly)) then $
       ptr_free, l_hskpltt.hvps_colln_vmon_monthly  

    if (ptr_valid(l_hskpltt.hvps_mcp_start_imon_daily)) then $ 
       ptr_free, l_hskpltt.hvps_mcp_start_imon_daily  
    if (ptr_valid(l_hskpltt.hvps_mcp_start_imon_monthly)) then $
       ptr_free, l_hskpltt.hvps_mcp_start_imon_monthly  

    if (ptr_valid(l_hskpltt.hvps_mcp_stop_imon_daily)) then $ 
       ptr_free, l_hskpltt.hvps_mcp_stop_imon_daily 
    if (ptr_valid(l_hskpltt.hvps_mcp_stop_imon_monthly)) then $
       ptr_free, l_hskpltt.hvps_mcp_stop_imon_monthly  

    if (ptr_valid(l_hskpltt.hvps_optics_imon_daily)) then $ 
       ptr_free, l_hskpltt.hvps_optics_imon_daily  
    if (ptr_valid(l_hskpltt.hvps_optics_imon_monthly)) then $
       ptr_free, l_hskpltt.hvps_optics_imon_monthly 

    if (ptr_valid(l_hskpltt.lvps_15vp_vmon_daily))  then $
       ptr_free, l_hskpltt.lvps_15vp_vmon_daily 
    if (ptr_valid(l_hskpltt.lvps_15vp_vmon_monthly)) then $
       ptr_free, l_hskpltt.lvps_15vp_vmon_monthly 

    if (ptr_valid(l_hskpltt.lvps_15vn_vmon_daily))  then $ 
       ptr_free, l_hskpltt.lvps_15vn_vmon_daily 
    if (ptr_valid(l_hskpltt.lvps_15vn_vmon_monthly)) then $
       ptr_free, l_hskpltt.lvps_15vn_vmon_monthly 

    if (ptr_valid(l_hskpltt.lvps_5vp_vmon_daily)) then $
       ptr_free, l_hskpltt.lvps_5vp_vmon_daily 
    if (ptr_valid(l_hskpltt.lvps_5vp_vmon_monthly)) then $
       ptr_free, l_hskpltt.lvps_5vp_vmon_monthly 

    if (ptr_valid(l_hskpltt.lvps_5vn_vmon_daily))  then $
       ptr_free, l_hskpltt.lvps_5vn_vmon_daily 
    if (ptr_valid(l_hskpltt.lvps_5vn_vmon_monthly)) then $
       ptr_free, l_hskpltt.lvps_5vn_vmon_monthly

RETURN
END

;---------------------------------------------------------------------------
;  Generate text dumps.
;---------------------------------------------------------------------------
PRO GenTextDumps, setup_parms, plot_parms, l_hskpltt

    l_GenFileName, setup_parms, 0 
    plot_parms.filename = setup_parms.prod_dst + '/' + setup_parms.filename + 'ltt.dmp'
   
    openw, ofd, plot_parms.filename, /get_lun

    
    plot_parms.dmp_hdr_title = 'HSKP Trending Dumps'

    l_DumpHdrInfo, ofd, setup_parms, plot_parms, plot_parms.filename
    
    if (setup_parms.avg_type eq 'D') then begin

       printf, ofd, 'Average :  Daily'
       printf, ofd, ' '

       starts              = (*l_hskpltt.starts_daily)
       stops               = (*l_hskpltt.stops_daily)
       hvps_collp_imon     = (*l_hskpltt.hvps_collp_imon_daily)
       hvps_colln_imon     = (*l_hskpltt.hvps_colln_imon_daily) 
       temp0               = (*l_hskpltt.temp0_daily)  
       temp1               = (*l_hskpltt.temp1_daily)  
       hvps_mcp_start_vmon = (*l_hskpltt.hvps_mcp_start_vmon_daily)
       hvps_mcp_stop_vmon  = (*l_hskpltt.hvps_mcp_stop_vmon_daily)
       hvps_optics_vmon    = (*l_hskpltt.hvps_optics_vmon_daily)
       hvps_steering_vmon  = (*l_hskpltt.hvps_steering_vmon_daily)
       hvps_collp_vmon     = (*l_hskpltt.hvps_collp_vmon_daily)  
       hvps_colln_vmon     = (*l_hskpltt.hvps_colln_vmon_daily) 
       hvps_mcp_start_imon = (*l_hskpltt.hvps_mcp_start_imon_daily) 
       hvps_mcp_stop_imon  = (*l_hskpltt.hvps_mcp_stop_imon_daily) 
       hvps_optics_imon    = (*l_hskpltt.hvps_optics_imon_daily) 
       lvps_15vp_vmon      = (*l_hskpltt.lvps_15vp_vmon_daily)  
       lvps_15vn_vmon      = (*l_hskpltt.lvps_15vn_vmon_daily)  
       lvps_5vp_vmon       = (*l_hskpltt.lvps_5vp_vmon_daily)  
       lvps_5vn_vmon       = (*l_hskpltt.lvps_5vn_vmon_daily)  
       timestr             = (*l_hskpltt.time_daily)


    endif else begin

       printf, ofd, 'Average :  Monthly'

       starts              = (*l_hskpltt.starts_monthly)
       stops               = (*l_hskpltt.stops_monthly)  
       hvps_collp_imon     = (*l_hskpltt.hvps_collp_imon_monthly)
       hvps_colln_imon     = (*l_hskpltt.hvps_colln_imon_monthly) 
       temp0               = (*l_hskpltt.temp0_monthly)
       temp1               = (*l_hskpltt.temp1_monthly)
       hvps_mcp_start_vmon = (*l_hskpltt.hvps_mcp_start_vmon_monthly)
       hvps_mcp_stop_vmon  = (*l_hskpltt.hvps_mcp_stop_vmon_monthly)
       hvps_optics_vmon    = (*l_hskpltt.hvps_optics_vmon_monthly)
       hvps_steering_vmon  = (*l_hskpltt.hvps_steering_vmon_monthly)
       hvps_collp_vmon     = (*l_hskpltt.hvps_collp_vmon_monthly)
       hvps_colln_vmon     = (*l_hskpltt.hvps_colln_vmon_monthly)
       hvps_mcp_start_imon = (*l_hskpltt.hvps_mcp_start_imon_monthly)
       hvps_mcp_stop_imon  = (*l_hskpltt.hvps_mcp_stop_imon_monthly)
       hvps_optics_imon    = (*l_hskpltt.hvps_optics_imon_monthly)
       lvps_15vp_vmon      = (*l_hskpltt.lvps_15vp_vmon_monthly) 
       lvps_15vn_vmon      = (*l_hskpltt.lvps_15vn_vmon_monthly) 
       lvps_5vp_vmon       = (*l_hskpltt.lvps_5vp_vmon_monthly) 
       lvps_5vn_vmon       = (*l_hskpltt.lvps_5vn_vmon_monthly) 
       timestr             = (*l_hskpltt.time_monthly)

    endelse
  

    if (setup_parms.starts eq 'Y') then begin
       nele = n_elements (starts)
       printf, ofd, 'DOY', 'Singles Starts', format='((A10,TR1), (A16))'
       printf, ofd, '----------', '--------------', format='((A10,TR1), (A16))'
       for i = 0L , nele - 1L do begin
           printf, ofd, timestr[i], starts[i], format = '((A10,TR1), (e12.3))'
       endfor
       printf, ofd, ' '
       printf, ofd, ' '
    endif

    if (setup_parms.stops eq 'Y') then begin
       nele = n_elements (stops)
       printf, ofd, 'DOY', 'Singles Stops', format='((A10,TR1), (A16))'
       printf, ofd, '----------', '--------------', format='((A10,TR1), (A16))'
       for i = 0L , nele - 1L do begin
           printf, ofd, timestr[i], stops[i], format = '((A10,TR1), (e12.3))'
       endfor
       printf, ofd, ' '
       printf, ofd, ' '
    endif

    if (setup_parms.hvps_collp_imon eq 'Y') then begin
       nele = n_elements (hvps_collp_imon)
       printf, ofd, 'DOY', 'HVPS CollP IMON', format='((A10,TR1), (A16))'
       printf, ofd, '----------', '---------------', format='((A10,TR1), (A16))'
       for i = 0L, nele - 1L do begin
           printf, ofd, timestr[i], hvps_collp_imon[i], $
               format='((A10,TR1), (e12.3))' 
       endfor
       printf, ofd, ' '
       printf, ofd, ' '
    endif

    if (setup_parms.hvps_colln_imon eq 'Y') then begin
       nele = n_elements (hvps_colln_imon)
       printf, ofd, 'DOY', 'HVPS CollN IMON', format='((A10,TR1), (A16))'
       printf, ofd, '----------', '---------------', format='((A10,TR1), (A16))'
       for i = 0L, nele - 1L do begin
           printf, ofd, timestr[i], hvps_colln_imon[i], $
               format='((A10,TR1), (e12.3))' 
       endfor
       printf, ofd, ' '
       printf, ofd, ' '
    endif

    if (setup_parms.temp0 eq 'Y') then begin
       nele = n_elements (temp0)
       printf, ofd, 'DOY', 'Temperature 0', format='((A10,TR1), (A16))'
       printf, ofd, '----------', '---------------', format='((A10,TR1), (A16))'
       for i = 0L, nele - 1L do begin
           printf, ofd, timestr[i], temp0[i], $
               format='((A10,TR1), (e12.3))' 
       endfor
       printf, ofd, ' '
       printf, ofd, ' '
    endif

    if (setup_parms.temp1 eq 'Y') then begin
       nele = n_elements (temp1)
       printf, ofd, 'DOY', 'Temperature 1', format='((A10,TR1), (A16))'
       printf, ofd, '----------', '---------------', format='((A10,TR1), (A16))'
       for i = 0L, nele - 1L do begin
           printf, ofd, timestr[i], temp1[i], $
               format='((A10,TR1), (e12.3))' 
       endfor
       printf, ofd, ' '
       printf, ofd, ' '
    endif

    if (setup_parms.hvps_mcp_start_vmon eq 'Y') then begin
       nele = n_elements (hvps_mcp_start_vmon)
       printf, ofd, 'DOY', 'HVPS MCP Start Vmon', format='((A10,TR1), (A16))'
       printf, ofd, '----------', '-------------------', format='((A10,TR1), (A16))'
       for i = 0L, nele - 1L do begin
           printf, ofd, timestr[i], hvps_mcp_start_vmon[i], $
               format='((A10,TR1), (e12.3))' 
       endfor
       printf, ofd, ' '
       printf, ofd, ' '
    endif

    if (setup_parms.hvps_mcp_stop_vmon eq 'Y') then begin
       nele = n_elements (hvps_mcp_stop_vmon)
       printf, ofd, 'DOY', 'HVPS MCP Stop Vmon', format='((A10,TR1), (A16))'
       printf, ofd, '----------', '-------------------', format='((A10,TR1), (A16))'
       for i = 0L, nele - 1L do begin
           printf, ofd, timestr[i], hvps_mcp_stop_vmon[i], $
               format='((A10,TR1), (e12.3))' 
       endfor
       printf, ofd, ' '
       printf, ofd, ' '
    endif

    if (setup_parms.hvps_optics_vmon eq 'Y') then begin
       nele = n_elements (hvps_optics_vmon)
       printf, ofd, 'DOY', 'HVPS Optics Vmon', format='((A10,TR1), (A16))'
       printf, ofd, '----------', '-------------------', format='((A10,TR1), (A16))'
       for i = 0L, nele - 1L do begin
           printf, ofd, timestr[i], hvps_optics_vmon[i], $
               format='((A10,TR1), (e12.3))' 
       endfor
       printf, ofd, ' '
       printf, ofd, ' '
    endif

    if (setup_parms.hvps_steering_vmon eq 'Y') then begin
       nele = n_elements (hvps_steering_vmon)
       printf, ofd, 'DOY', 'HVPS Steering Vmon', format='((A10,TR1), (A16))'
       printf, ofd, '----------', '-------------------', format='((A10,TR1), (A16))'
       for i = 0L, nele - 1L do begin
           printf, ofd, timestr[i], hvps_steering_vmon[i], $
               format='((A10,TR1), (e12.3))' 
       endfor
       printf, ofd, ' '
       printf, ofd, ' '
    endif

    if (setup_parms.hvps_collp_vmon eq 'Y') then begin
       nele = n_elements (hvps_collp_vmon)
       printf, ofd, 'DOY', 'HVPS CollP Vmon', format='((A10,TR1), (A16))'
       printf, ofd, '----------', '-------------------', format='((A10,TR1), (A16))'
       for i = 0L, nele - 1L do begin
           printf, ofd, timestr[i], hvps_collp_vmon[i], $
               format='((A10,TR1), (e12.3))' 
       endfor
       printf, ofd, ' '
       printf, ofd, ' '
    endif

    if (setup_parms.hvps_colln_vmon eq 'Y') then begin
       nele = n_elements (hvps_colln_vmon)
       printf, ofd, 'DOY', 'HVPS CollN Vmon', format='((A10,TR1), (A16))'
       printf, ofd, '----------', '-------------------', format='((A10,TR1), (A16))'
       for i = 0L, nele - 1L do begin
           printf, ofd, timestr[i], hvps_colln_vmon[i], $
               format='((A10,TR1), (e12.3))' 
       endfor
       printf, ofd, ' '
       printf, ofd, ' '
    endif

    if (setup_parms.hvps_mcp_start_imon eq 'Y') then begin
       nele = n_elements (hvps_mcp_start_imon)
       printf, ofd, 'DOY', 'HVPS Start Imon', format='((A10,TR1), (A16))'
       printf, ofd, '----------', '-------------------', format='((A10,TR1), (A16))'
       for i = 0L, nele - 1L do begin
           printf, ofd, timestr[i], hvps_mcp_start_imon[i], $
               format='((A10,TR1), (e12.3))' 
       endfor
       printf, ofd, ' '
       printf, ofd, ' '
    endif

    if (setup_parms.hvps_mcp_stop_imon eq 'Y') then begin
       nele = n_elements (hvps_mcp_stop_imon)
       printf, ofd, 'DOY', 'HVPS Stop Imon', format='((A10,TR1), (A16))'
       printf, ofd, '----------', '-------------------', format='((A10,TR1), (A16))'
       for i = 0L, nele - 1L do begin
           printf, ofd, timestr[i], hvps_mcp_stop_imon[i], $
               format='((A10,TR1), (e12.3))' 
       endfor
       printf, ofd, ' '
       printf, ofd, ' '
    endif

    if (setup_parms.hvps_optics_imon eq 'Y') then begin
       nele = n_elements (hvps_optics_imon)
       printf, ofd, 'DOY', 'HVPS Optics Imon', format='((A10,TR1), (A16))'
       printf, ofd, '----------', '-------------------', format='((A10,TR1), (A16))'
       for i = 0L, nele - 1L do begin
           printf, ofd, timestr[i], hvps_optics_imon[i], $
               format='((A10,TR1), (e12.3))' 
       endfor
       printf, ofd, ' '
       printf, ofd, ' '
    endif

    if (setup_parms.lvps_15vp_vmon eq 'Y') then begin
       nele = n_elements (lvps_15vp_vmon)
       printf, ofd, 'DOY', 'LVPS 15vp Vmon', format='((A10,TR1), (A16))'
       printf, ofd, '----------', '-------------------', format='((A10,TR1), (A16))'
       for i = 0L, nele - 1L do begin
           printf, ofd, timestr[i], lvps_15vp_vmon[i], $
               format='((A10,TR1), (e12.3))' 
       endfor
       printf, ofd, ' '
       printf, ofd, ' '
    endif

    if (setup_parms.lvps_15vn_vmon eq 'Y') then begin
       nele = n_elements (lvps_15vn_vmon)
       printf, ofd, 'DOY', 'LVPS 15vn Vmon', format='((A10,TR1), (A16))'
       printf, ofd, '----------', '-------------------', format='((A10,TR1), (A16))'
       for i = 0L, nele - 1L do begin
           printf, ofd, timestr[i], lvps_15vn_vmon[i], $
               format='((A10,TR1), (e12.3))' 
       endfor
       printf, ofd, ' '
       printf, ofd, ' '
    endif

    if (setup_parms.lvps_5vp_vmon eq 'Y') then begin
       nele = n_elements (lvps_5vp_vmon)
       printf, ofd, 'DOY', 'LVPS 5vp Vmon', format='((A10,TR1), (A16))'
       printf, ofd, '----------', '-------------------', format='((A10,TR1), (A16))'
       for i = 0L, nele - 1L do begin
           printf, ofd, timestr[i], lvps_5vp_vmon[i], $
               format='((A10,TR1), (e12.3))' 
       endfor
       printf, ofd, ' '
       printf, ofd, ' '
    endif

    if (setup_parms.lvps_5vn_vmon eq 'Y') then begin
       nele = n_elements (lvps_5vn_vmon)
       printf, ofd, 'DOY', 'LVPS 5vn Vmon', format='((A10,TR1), (A16))'
       printf, ofd, '----------', '-------------------', format='((A10,TR1), (A16))'
       for i = 0L, nele - 1L do begin
           printf, ofd, timestr[i], lvps_5vn_vmon[i], $
               format='((A10,TR1), (e12.3))' 
       endfor
       printf, ofd, ' '
       printf, ofd, ' '
    endif

    close, ofd
    free_lun, ofd


RETURN
END


;---------------------------------------------------------------------------
;  MAIN
;---------------------------------------------------------------------------

PRO l_HskpLtt, setup_filename 
COMMON ERROR_TYPES, SUCCESS, FAILURE, WARNING, INFO
COMMON PLOTDEFS, MAX_PLOTS, MAX_WEB_SPINS, NO_CNTS, NO_DATA
COMMON MYCOLORS, N_COLOR_TABLE, N_GRAYS
COMMON EPOCH_DATA, EPOCH_YR


    !EXCEPT=2

    ;----------------------------------------
    ; global variable declarations
    ;----------------------------------------
    NO_DATA       = -9999.
    NO_CNTS       = -8888.
    MAX_PLOTS     = 6
    EPOCH_YR      = 2000
    FAILURE       = -1
    SUCCESS       = 0
    N_COLOR_TABLE = 226
    N_GRAYS       = 5
    MAX_WEB_SPINS = 1





    times_struct = { TIMES_STRUCT, $
                  year: 0, $
                  doy : 0, $
                  hour: 0, $
                  min : 0, $
                  sec : 0, $
                  msec: 0  $
                 }


    ;------------------------------------------------------
    ; setup file parameters, used at run-time
    ;------------------------------------------------------
    setup_parms = { $
      start_date           : [0,0,0],          $    ;--- user start date ---
      start_time           : [0,0,0],          $    ;--- user start time ---
      stop_date            : [0,0,0],          $    ;--- user stop date ---
      stop_time            : [23,59,59],       $    ;--- user stop time ---
      l_start_dtime        : [0,0,0,0,0,0],    $    ;--- user start date/time ---
      l_stop_dtime         : [0,0,0,23,59,59], $    ;--- user stop date/time ---
      data_type            : '',               $
      prod_dst             : '',               $    ;--- product destination ---
      nplots               : 0L,               $    
      avg_type             : 'D',              $
      postscript           : 'N',        $
      starts               : 'Y',        $
      stops                : 'Y',        $
      hvps_mcp_start_vmon  : 'Y',        $
      hvps_mcp_stop_vmon   : 'Y',        $
      hvps_collp_vmon      : 'Y',        $ 
      hvps_colln_vmon      : 'Y',        $
      hvps_optics_vmon     : 'Y',        $
      hvps_steering_vmon   : 'Y',        $
      hvps_mcp_start_imon  : 'Y',        $
      hvps_mcp_stop_imon   : 'Y',        $
      hvps_collp_imon      : 'Y',        $
      hvps_colln_imon      : 'Y',        $
      hvps_optics_imon     : 'Y',        $
      lvps_15vp_vmon       : 'Y',        $
      lvps_15vn_vmon       : 'Y',        $
      lvps_5vp_vmon        : 'Y',        $
      lvps_5vn_vmon        : 'Y',        $
      temp0                : 'Y',        $
      temp1                : 'Y',        $
      filename             : '' ,        $
      gen_gif              : 'Y',        $
      via_web              : 'N',        $
      gen_sum              : 'N',        $
      gen_dumps            : 'N',        $
      plot_type            : 'G',        $
      color_tbl            : 1L          $

    }

    ;----------------------------------------
    ;  color table
    ;----------------------------------------
    rgb_table = {                $
      red     : fltarr (N_COLOR_TABLE),     $
      blue    : fltarr (N_COLOR_TABLE),     $
      green   : fltarr (N_COLOR_TABLE)      $
    }


    ;------------------------------------------------------
    ; plot parameters
    ;------------------------------------------------------
    plot_parms = { $
      xin                : 8.2, $
      yin                : 9.9, $
      lg_font            : 1.1, $
      md_font            : 1.0, $
      sm_font            : 0.9, $
      xsm_font           : 0.5, $
      plot_font          : 1.2, $
      window_width       : 0L,  $
      window_height      : 0L,  $
      ydel               : 0.0, $
      xdel               : 0.0, $
      ypltmx             : 0.0, $
      ypltmn             : 0.0, $
      xpltmx             : 0.0, $
      xpltmn             : 0.0, $
      top_ypltmx         : 0.0, $
      top_ypltmn         : 0.0, $
      axis_size          : 6.35,$
      yspace             : 0.0, $
      X_axis_eq_time     : 'N', $
      linestyle          : 'N', $
      ps                 : 'N', $
      filename           : '',  $
      plot_hdr_title     : '',  $
      start_doy          : 0,   $
      stop_doy           : 0,   $
      xmn_tick           : 1,   $
      ymn_tick           : 1,   $
      nticks             : 6,   $
      xtick_labels       : strarr (12), $
      xtick_locs         : dblarr (12), $
      xtick_delta        : 0,   $
      mn_tick_delta      : 0.0, $
      min_time_axis_len  : 0.0, $
      max_time_axis_len  : 0.0, $
      first_time_label   : 0.0, $
      xtitle             : ' ', $
      dmp_hdr_title      : ' ', $
      data_source        : '',               $    ;--- data source ---
      sw_ver             : 'lena_hskp_ltt_v0.0' $
    }

    plot_parms.xtitle = 'Day of Year'

    ; starting location of x
    plot_parms.xpltmn = 0.92 / plot_parms.xin

    ;sets the width (x axis)
    plot_parms.xpltmx = plot_parms.xpltmn + plot_parms.axis_size/ plot_parms.xin

    ; starting location at y
    plot_parms.ypltmx = 8.9/plot_parms.yin

    ;the Y (height) of the plot, by default
    plot_size = floor(float(plot_parms.yin/MAX_PLOTS))
    plot_parms.ydel   = plot_size/plot_parms.yin

    ; the amount of spacing in between plots
    plot_parms.yspace = 0.15/ plot_parms.yin

    plot_parms.ypltmn = plot_parms.ypltmx - plot_parms.ydel

    plot_parms.top_ypltmx = plot_parms.ypltmx
    plot_parms.top_ypltmn = plot_parms.ypltmn

    plot_parms.window_width  = fix(plot_parms.xin * 75)
    plot_parms.window_height = fix(plot_parms.yin * 75)



    ;------------------------------------------------------
    ; pointers to housekeeping parameters
    ;------------------------------------------------------
    ptr = ptr_new ()
    l_hskpltt = { $
      time_daily                   : ptr,  $
      time_monthly                 : ptr,  $
      starts_daily                 : ptr, $
      starts_monthly               : ptr, $
      stops_daily                  : ptr, $
      stops_monthly                : ptr, $
      hvps_collp_imon_daily        : ptr, $
      hvps_collp_imon_monthly      : ptr, $
      hvps_colln_imon_daily        : ptr, $
      hvps_colln_imon_monthly      : ptr, $
      temp0_daily                  : ptr, $
      temp0_monthly                : ptr, $
      temp1_daily                  : ptr, $
      temp1_monthly                : ptr, $
      hvps_mcp_start_vmon_daily    : ptr, $
      hvps_mcp_start_vmon_monthly  : ptr, $
      hvps_mcp_stop_vmon_daily     : ptr, $
      hvps_mcp_stop_vmon_monthly   : ptr, $
      hvps_optics_vmon_daily       : ptr, $
      hvps_optics_vmon_monthly     : ptr, $
      hvps_steering_vmon_daily     : ptr, $
      hvps_steering_vmon_monthly   : ptr, $
      hvps_collp_vmon_daily        : ptr, $
      hvps_collp_vmon_monthly      : ptr, $
      hvps_colln_vmon_daily        : ptr, $
      hvps_colln_vmon_monthly      : ptr, $
      hvps_mcp_start_imon_daily    : ptr, $
      hvps_mcp_start_imon_monthly  : ptr, $
      hvps_mcp_stop_imon_daily     : ptr, $
      hvps_mcp_stop_imon_monthly   : ptr, $
      hvps_optics_imon_daily       : ptr, $
      hvps_optics_imon_monthly     : ptr, $
      lvps_15vp_vmon_daily         : ptr, $
      lvps_15vp_vmon_monthly       : ptr, $
      lvps_15vn_vmon_daily         : ptr, $
      lvps_15vn_vmon_monthly       : ptr, $
      lvps_5vp_vmon_daily          : ptr, $
      lvps_5vp_vmon_monthly        : ptr, $
      lvps_5vn_vmon_daily          : ptr, $
      lvps_5vn_vmon_monthly        : ptr  $
    }


    ;------------------------------------------------------
    ; read setup parameters from setup file, l_hskp_ltt
    ;------------------------------------------------------
    status = ReadSetupFile (setup_filename, setup_parms)

    plot_parms.data_source = 'D' 


    ;------------------------------------------------------
    ; convert start/stop time in YYYY/MM/DD format into
    ; seconds to estimate the number of spins 
    ;------------------------------------------------------
    time_array = replicate ({TIMES_STRUCT}, 1)

    start_doy = l_ConvYMD2DOY (setup_parms.start_date[0], $
                               setup_parms.start_date[1], $
                               setup_parms.start_date[2])

    new_start_doy = FindDOYSinceLaunch (setup_parms.start_date[0], start_doy)

    setup_parms.l_start_dtime[0] = setup_parms.start_date[0]
    setup_parms.l_start_dtime[1] = new_start_doy
    setup_parms.l_start_dtime[2] = setup_parms.start_time[0]
    setup_parms.l_start_dtime[3] = setup_parms.start_time[1]
    setup_parms.l_start_dtime[4] = setup_parms.start_time[2]
    setup_parms.l_start_dtime[5] = 0



    time_array.year = setup_parms.start_date[0]
    time_array.doy  = start_doy 
    time_array.hour = setup_parms.start_time[0]
    time_array.min  = setup_parms.start_time[1]
    time_array.sec  = setup_parms.start_time[2]
    time_array.msec = 0
    start_time_secs = Convert (time_array)


    stop_doy = l_ConvYMD2DOY (setup_parms.stop_date[0], $
                              setup_parms.stop_date[1], $
                              setup_parms.stop_date[2])

    new_stop_doy = FindDOYSinceLaunch (setup_parms.stop_date[0], stop_doy)

    setup_parms.l_stop_dtime[0] = setup_parms.stop_date[0]
    setup_parms.l_stop_dtime[1] = new_stop_doy
    setup_parms.l_stop_dtime[2] = setup_parms.stop_time[0]
    setup_parms.l_stop_dtime[3] = setup_parms.stop_time[1]
    setup_parms.l_stop_dtime[4] = setup_parms.stop_time[2]
    setup_parms.l_stop_dtime[5] = 0


    time_array.year = setup_parms.stop_date[0]
    time_array.doy  = stop_doy 
    time_array.hour = setup_parms.stop_time[0]
    time_array.min  = setup_parms.stop_time[1]
    time_array.sec  = setup_parms.stop_time[2]
    time_array.msec = 0
    stop_time_secs  = Convert (time_array)


    plot_parms.start_doy = start_doy
    plot_parms.stop_doy = stop_doy

    ;------------------------------------------------------
    ; estimate the number of spins, days, and months to
    ; process based on user start/stop date/time
    ;------------------------------------------------------
    est_nspins = 730
    est_days = (stop_doy - start_doy) + 1
    if (est_days le 0) then est_days = 1
    est_months = 12  
    ndays = stop_doy - start_doy

    ;------------------------------------------------------
    ; Read the HSKP ASCII Text dump file
    ;------------------------------------------------------
    status = ReadHskpFile (setup_parms, l_hskpltt, ndays, est_nspins, est_days, est_months)
    if (status eq FAILURE) then begin
      print, 'Failure to open file: ', setup_parms.filename
      RETURN
    endif


    ;------------------------------------------------------
    ; plot the data
    ;------------------------------------------------------
    nloops = 1L
    if (setup_parms.postscript eq 'Y') then nloops = 2L


    for i = 0L, nloops - 1L do begin
       nrows =  setup_parms.nplots 
       nplots = nrows
       npages = CEIL (nrows/MAX_PLOTS) - 1L
       pages = 0L

       if (nrows - MAX_PLOTS gt 0L) then npages = npages + 1L
       if (npages eq 0L) then npages = 1L
       if (npages gt 0L) then windx = intarr (npages)


       if (i eq 1L) AND (setup_parms.postscript eq 'Y') then begin
          plot_ps = 'Y'
          ;generate filename
       endif

       plot_parms.plot_hdr_title = 'HSKP LTT'
       ;------------------------------------------------------
       ; Setup X window display
       ;------------------------------------------------------
       SetUpDisplay, plot_parms, nplots, rgb_table, setup_parms

  
       plotted = 0L
       pages = 0L
      
       ;------------------------------------------------------
       ; adjust plot sizes 
       ;------------------------------------------------------
       l_AdjPlotSize, plot_parms, setup_parms, nplots
      
       ;------------------------------------------------------
       ; call plot routine
       ;------------------------------------------------------
       PlotData, l_hskpltt, setup_parms, plot_parms, nplots, plotted, pages, $
            windx, rgb_table


       if (i eq 1L) AND (setup_parms.postscript eq 'Y') then begin
          plot_ps = 'N'
          device, /close_file
       endif

    endfor

    if (setup_parms.gen_dumps eq 'Y') then begin
       GenTextDumps, setup_parms, plot_parms, l_hskpltt
    endif


    CleanUp, l_hskpltt

print, 'end hskp ltt ... ', systime ()

RETURN
END

