;-------------------------------------------------------------------------------
;
;  Unit Name : l_HndlMissOA
;
;  Purpose   : Fill in missing packets with a fill value. 
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;   ELH             04/23/03        v1.0.11           Original implementation
;
;
;  File Revision Number:   %I%
;  File Last Modified  :   %E%  %T%
;
;-------------------------------------------------------------------------------

;-------------------------------------------------------------------------------
;  Function/Procedure: l_HndlMissOA
;
;  Description:
;      Fill in missing packets with a fill value. 
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  tme                   double[]      I        original data time in seconds
;  new_time              double[]      O        interpolated times for data gaps
;  nspin                 long          I        total number of spins of data
;  data                  float[]       I        original data, no data gaps filled
;  the_data              float[]       O        data with data gaps filled
;  ustart_time           double        I        user start time in seconds
;  ustop_time            double        I        user stop time in seconds
;  data_format           long          I        single scalar array or 45 spin
;                                               sectors
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;     PLOTDEFS         MAX_PLOTS        int         G         max # plots per page
;                      MAX_WEB_SPINS    int         G         max # spins per plot
;                                                             when using web interface
;                      NO_CNTS          double      G         no counts indicator
;                      NO_DATA          double      G         no data indicator
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  E. Lee            5/00            v1.0             Original implementation
;-------------------------------------------------------------------------------


PRO l_HndlMissOA, tme, new_time, nspin, data, the_data, $
ustart_time, ustop_time, data_format
COMMON PLOTDEFS, MAX_PLOTS, MAX_WEB_SPINS, NO_CNTS, NO_DATA


      ;------------------------------------------------------------
      ; the fill value used for data gaps 
      ;------------------------------------------------------------
      dg_value = NO_DATA
   
      ;------------------------------------------------------------
      ; the spin interval of 2 minutes per spin
      ;------------------------------------------------------------
      spin_interval = 120.0

      ;------------------------------------------------------------
      ; the spin interval of 2 minutes + threshold of 5 seconds
      ;------------------------------------------------------------
      ;spin_threshold = 120.0 + 5.0
      spin_threshold = 120.0 + 20.0

      ;------------------------------------------------------------
      ; get the user's start and stop time requested
      ; get the data  start and stop time for the requested user 
      ; time
      ;------------------------------------------------------------
      indx = where (tme gt 0.0, wcnt)
      if (wcnt gt 0) then begin
         start_time = min(tme[indx])
      endif else begin
         start_time = min(tme)
      endelse
      stop_time = max(tme)

      user_st_time = ConvArrTimeToSecs (ustart_time)
      user_sp_time = ConvArrTimeToSecs (ustop_time)

      ;------------------------------------------------------------
      ; the number of spin sectors
      ;------------------------------------------------------------
      spin_sector = data_format 

      ;------------------------------------------------------------
      ; the number of seconds between spin sectors
      ;------------------------------------------------------------
      spin_sec_interval = 120.0/ float(data_format)


      time_diff = stop_time - start_time
      spins = round (float(time_diff)/spin_interval)


      ;------------------------------------------------------------
      ; determine the number of missing packets within the user
      ; request time
      ;------------------------------------------------------------
      total_missing = 0L
      for ii = 1, nspin-1  do begin
         spin_diff = tme[ii*data_format] - $
                     tme[(ii*data_format)-(data_format-1)]
         if spin_diff gt spin_threshold then begin
            spin_diff = spin_diff - 120.0
            missing = floor(spin_diff/spin_interval)
            total_missing = total_missing + missing
         endif
      endfor

      ;------------------------------------------------------------
      ; set up the correct size array filling it with gray for
      ; no data
      ;------------------------------------------------------------
      ele = nspin + total_missing
      spin_avg  = fltarr (ele)
      spin_tme  = dblarr(ele)
      nd = n_elements (data)
      nt = n_elements (tme)
      spin_avg[0:nd-1]  = data[*]
      spin_tme[0:nt-1]  = tme [*]

      ;------------------------------------------------------------
      ;  if data is available fill new array with the original
      ;  data else if there are missing spins, fill with fill
      ;  data.  Extrapolate the time.
      ;------------------------------------------------------------
      debug = 0
      if (debug) then begin

      spin_avg[0] = data[0:(data_format-1)]
      spin_tme[0] = tme [0:(data_format-1)]
      idx = 1

      for ii = 1, nspin-1  do begin
         ;-----------------------------------
         ; determine the time difference between
         ; current spin and last spin
         ;-----------------------------------
         spin_diff = tme[ii*data_format] - $
                     tme[(ii*data_format)-(data_format-1)]

         ;-----------------------------------
         ; a data gap is found
         ;-----------------------------------
         if spin_diff gt spin_threshold then begin

            ;-----------------------------------
            ; determine the number of missing spins
            ;-----------------------------------
 
            spin_diff = spin_diff - 120.0
            missing = floor(spin_diff/spin_interval)
            idx = idx + missing

            ;-----------------------------------
            ; skip the number of missing spins and
            ; store data in correct spin location 
            ;-----------------------------------
            spin_avg[idx] =  $
                data[(ii*data_format):((ii*data_format)+(data_format-1))]


            ;-----------------------------------
            ; fill in the missing spin times
            ;-----------------------------------
            if (missing gt 0) then begin
               miss_idx = idx - missing - 1
               mt_adj = spin_diff/(missing+1)
               miss_tme = tme [((ii-1)*data_format):$ 
                      (((ii-1)*data_format)+(data_format-1))] + mt_adj 
               miss_data= data[((ii-1)*data_format):$
                      (((ii-1)*data_format)+(data_format-1))]
               for jj=0, missing do begin
                 spin_tme[miss_idx] =  miss_tme
                 spin_avg[miss_idx] =  miss_data 
                 miss_tme = miss_tme + mt_adj 
                 miss_idx = miss_idx + 1
               endfor
               idx = miss_idx
            endif


            ;-----------------------------------
            ; fill in the spin time from original to
            ; correct spin location 
            ;-----------------------------------
            spin_tme[idx] =  $
                 tme[(ii*data_format):((ii*data_format)+(data_format-1))]


         endif else begin

            ;-----------------------------------
            ; no missing data, fill with original
            ;-----------------------------------

            spin_avg[idx] =  $
                data[(ii*data_format):((ii*data_format)+(data_format-1))]

            spin_tme[idx] =  $
                 tme[(ii*data_format):((ii*data_format)+(data_format-1))]

         endelse
         idx = idx + 1
      endfor

      ;----------------------------------------------------------
      ; change the spin_tme format back to a 1 dimensional array
      ;----------------------------------------------------------
      ;index = reform (spin_tme, n_elements(spin_tme), /OVERWRITE)
      endif

      ;------------------------------------------------------------
      ; if there is data missing for both the beginning and the end 
      ; of the user requested time
      ;------------------------------------------------------------
         missing_data = 0
         if (start_time gt user_st_time) and  $
            (user_sp_time gt stop_time) then begin
              missing_data = 1
              b_miss_pkts = ceil((start_time - user_st_time)/spin_interval)
              e_miss_pkts = ceil((user_sp_time - stop_time)/spin_interval)
              total_mps = b_miss_pkts + e_miss_pkts + ele

              new_data = fltarr (total_mps)
              new_data[0:b_miss_pkts-1] = spin_avg[0]
              new_data[b_miss_pkts:((b_miss_pkts+ele)-1)] = spin_avg[*]
              n = n_elements(spin_avg)
              new_data[(b_miss_pkts+ele):total_mps-1] = spin_avg[n-1]

              tele = n_elements (spin_tme) 
              tot_time_ele = (data_format*(b_miss_pkts+e_miss_pkts))+tele
              new_time = fltarr(tot_time_ele)
              new_time[*] = user_st_time
              fill_end = tot_time_ele - e_miss_pkts*data_format

              new_time[b_miss_pkts*data_format:fill_end-1] = spin_tme[*]
              new_time[fill_end:tot_time_ele-1] = user_sp_time

         endif else begin
            ;------------------------------------------------------------
            ; fill the beginning of the user data set
            ; there is no data for the time the user requested 
            ;------------------------------------------------------------

            if (start_time gt user_st_time) then begin
               missing_data = 1
               miss_pkts = ceil((start_time - user_st_time)/spin_interval)
               mpkts = miss_pkts + ele
               new_data = fltarr (mpkts)
               new_data[*] = spin_avg[0] 
               new_data[miss_pkts:mpkts-1] = spin_avg[*]

               tele = n_elements (spin_tme) 
               tot_time_ele = (data_format*miss_pkts)+tele
               new_time = fltarr(tot_time_ele)
               new_time[*] = user_st_time
               fill_end = tot_time_ele - tele
               new_time[miss_pkts*data_format] = spin_tme[*]

            endif

            ;------------------------------------------------------------
            ; fill the end of the user data set
            ; there is no data for the time the user requested 
            ;------------------------------------------------------------
            if (stop_time lt user_sp_time)  then begin
               missing_data = 1
               miss_pkts = ceil((user_sp_time - stop_time)/spin_interval)
               mpkts = miss_pkts + ele
               new_data = fltarr (mpkts)
               n = n_elements(spin_avg)
               new_data[*] = spin_avg[n-1]
               new_data[0:ele-1] = spin_avg[*]

               tele = n_elements (spin_tme) 
               tot_time_ele = (data_format*miss_pkts)+tele
               new_time = fltarr(tot_time_ele)
               new_time[*] = user_sp_time
               new_time[0:tele-1] = spin_tme[*]
            endif

        endelse

     if (missing_data) then begin
         the_data = new_data
         new_time = new_time
     endif else begin
         the_data = spin_avg
         new_time = spin_tme
     endelse


RETURN
END
