
;-------------------------------------------------------------------------------
;
;  Unit Name : l_EventRtns.pro
;
;  Purpose   : All routines related to reading, writing, and plotting the
;              Event data.
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;   ELH                               v1.0           Original implementation
;   ELH               08/01          v1.0.7          Background scaled/correction 
;                                                     . select to generate text
;                                                       dumps to match plot or
;                                                       as untouched data
;                                                     . sun direction 
;   ELH               09/26/01       v1.0.7          Added l_DetermineSunDir 
;   ELH               12/03/01       v1.0.7          Separated handling of missing
;                                                    data from l_EventPlot into
;                                                    a separate routine 
;   ELH               01/02          v1.0.8          Added parm in l_HndlMissData to
;                                                    handle rolling data based on
;                                                    spin angle input
;
;
;  File Revision Number:   %I%
;  File Last Modified  :   %E%  %T%
;
;-------------------------------------------------------------------------------

;-------------------------------------------------------------------------------
;  Procedure :  l_EventReadSetup
;
;  Description: Reads the options from the setup file specific to event data.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_plot_setup          struct       I/O       event data specific parameters
;                                               and attributes
;  l_setup_file          string        I        setup filename
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;  ELH               05/04           v1.0.11          Convert input string to
;                                                     lowercase
;-------------------------------------------------------------------------------

PRO l_EventReadSetup, l_plot_setup, l_setup_file


   ;---------------------------------------------
   ; event specific processing & plot parameters
   ;---------------------------------------------
   l_plot_setup = {                $
       use_event          : 'N',   $
       hydrogen           : 'N',   $
       he0_eV             : 20L,   $
       he1_eV             : 65L,   $
       he2_eV             : 200L,  $
       H_scale_factor     : 0.4,   $
       oxygen             : 'N',   $
       oe0_eV             : 30L,   $
       oe1_eV             : 65L,   $
       oe2_eV             : 200L,  $
       O_scale_factor     : 6.0,   $
       energy0            : 'N',   $
       energy1            : 'N',   $
       energy2            : 'N',   $
       background         : 'N',   $
       hydrogen_beg_chan  : 151.0, $
       hydrogen_end_chan  : 190.0, $
       oxygen_beg_chan    : 401.0, $
       oxygen_end_chan    : 1000.0,$
       bkgrnd_beg_chan    : 251.0, $
       bkgrnd_end_chan    : 350.0, $
       color_min          : 0.0,   $
       color_max          : 0.0,   $
       tof_beg            : 1L,    $
       tof_end            : 1024L, $
       polar_bin          : 0L,    $
       energy_bin         : 0L,    $
       mass_spect         : 'N',   $
       merge              : 'N',   $
       nplots             : 0L     $
   }

   ;----------------------------------------
   ; read event data setup parameters
   ;----------------------------------------

   l_plot_setup.nplots = 0L

   openr, inlun, /get_lun, l_setup_file
   line = ' '
   while not EOF(inlun) do begin		;While there is text left, output it.
       readf, inlun, line
       IF (STRMID(line, 0, 1) ne ';') and (STRLEN(line) gt 0) then begin
           parts = STR_SEP(line, ' ')
           if (n_elements(parts) gt 2) then begin
           case STRLOWCASE(parts[0]) of

           'l_event' : begin
		       l_plot_setup.use_event=STRUPCASE(parts[2])
            end

           'l_event_hydrogen' : begin
		   l_plot_setup.hydrogen=STRUPCASE(parts[2])
           end

           'l_event_hydrogen_energy0_eV' : begin
		   l_plot_setup.he0_eV=LONG(parts[2])
           end

           'l_event_hydrogen_energy1_eV' : begin
		   l_plot_setup.he1_eV=LONG(parts[2])
           end

           'l_event_hydrogen_energy2_eV' : begin
		   l_plot_setup.he2_eV=LONG(parts[2])
           end

           'l_event_oxygen' : begin
		   l_plot_setup.oxygen=STRUPCASE(parts[2])
           end

           'l_event_oxygen_energy0_eV' : begin
		   l_plot_setup.oe0_eV=LONG(parts[2])
           end

           'l_event_oxygen_energy1_eV' : begin
		   l_plot_setup.oe1_eV=LONG(parts[2])
           end

           'l_event_oxygen_energy2_eV' : begin
		   l_plot_setup.oe2_eV=LONG(parts[2])
           end

           'l_event_energy0' : begin
		   l_plot_setup.energy0=STRUPCASE(parts[2])
           end

           'l_event_energy1' : begin
		   l_plot_setup.energy1=STRUPCASE(parts[2])
           end

           'l_event_energy2' : begin
		   l_plot_setup.energy2=STRUPCASE(parts[2])
           end

           'l_event_background' : begin
		   l_plot_setup.background=STRUPCASE(parts[2])
           end

	       'hydrogen_beg_chan'    : begin
		   l_plot_setup.hydrogen_beg_chan=double(parts[2])
		   end

	       'hydrogen_end_chan'    : begin
		   l_plot_setup.hydrogen_end_chan=double(parts[2])
		   end

	       'oxygen_beg_chan'    : begin
		   l_plot_setup.oxygen_beg_chan=double(parts[2])
		   end

	       'oxygen_end_chan'    : begin
		   l_plot_setup.oxygen_end_chan=double(parts[2])
		   end

	       'bkgrnd_beg_chan'    : begin
		   l_plot_setup.bkgrnd_beg_chan=double(parts[2])
		   end

	       'bkgrnd_end_chan'    : begin
		   l_plot_setup.bkgrnd_end_chan=double(parts[2])
		   end

	       'l_event_color_min'    : begin
		   l_plot_setup.color_min = double(parts[2])
		   end

	       'l_event_color_max'    : begin
		   l_plot_setup.color_max = double(parts[2])
		   end
       

           'l_mass_spectrum_plots' : begin
		       l_plot_setup.mass_spect=STRUPCASE(parts[2])
               if (l_plot_setup.mass_spect eq 'Y') then begin
                  l_plot_setup.nplots = l_plot_setup.nplots + 1L
               endif
            end

	       'l_event_tof_beg' : begin
		       l_plot_setup.tof_beg=long(parts[2])
		   end

	       'l_event_tof_end' : begin
		       l_plot_setup.tof_end=long(parts[2])
		   end

	       'l_event_polar_bin'   : begin
		   l_plot_setup.polar_bin= long(parts[2])
		   end

	       'l_event_energy_bin'   : begin
		   l_plot_setup.energy_bin= long(parts[2])
	       end

	       'l_event_merge'   : begin
		   l_plot_setup.merge=STRUPCASE(parts[2])
	       end

           else: begin
               ;print, 'Unknown option : ', parts[0]
	       end

           endcase				;End of CASE statement.
           endif
       endif
    endwhile


    close, inlun
    free_lun, inlun

    np = l_plot_setup.nplots

    if (l_plot_setup.use_event eq 'Y') then begin
       if (l_plot_setup.merge eq 'Y') then begin
           np = 1L
       endif else begin
          if (l_plot_setup.hydrogen eq 'Y') then begin
              if (l_plot_setup.energy0 eq 'Y') then np = np + 1L
              if (l_plot_setup.energy1 eq 'Y') then np = np + 1L
              if (l_plot_setup.energy2 eq 'Y') then np = np + 1L
          endif

          if (l_plot_setup.oxygen eq 'Y') then begin
              if (l_plot_setup.energy0 eq 'Y') then np = np + 1L
              if (l_plot_setup.energy1 eq 'Y') then np = np + 1L
              if (l_plot_setup.energy2 eq 'Y') then np = np + 1L
          endif
       endelse
    endif

    l_plot_setup.nplots = np 

    if (l_plot_setup.use_event eq 'N') then l_plot_setup.nplots = 0L

RETURN
END


;-------------------------------------------------------------------------------
;  Procedure :  l_EventShowSetup
;  Description:
;     Show the setup information read.  Used for debug purposes.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_plot_setup          struct        I        event specific parameters
;                                               and attributes
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;-------------------------------------------------------------------------------

PRO l_EventShowSetup, l_plot_setup

   print, '____LENA Event Data Setup Parameters'
   print, 'L_EVENT_TOF      : ', l_plot_setup.tof
   print, 'L_EVENT_POLAR    : ', l_plot_setup.polar
   print, 'L_EVENT_ENERGY   : ', l_plot_setup.energy

RETURN
END



;-------------------------------------------------------------------------------
;  Function: CheckFiller
;  Description:
;    Check if the tof, polar, & energy values are 0, 12, 3 which is a
;    UDF indicated that the sector has no events.
;
;  Return Value:  type = long
;  Value                         Description
;  -------------------------     ------------------------------
;  0                             data is filler data
;  1                             data is NOT filler data  
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  tof                   long          I        time of flight value
;  polar                 long          I        polar zone value
;  energy                long          I        energy bin value
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;
;  NOTES:
;    Definition of filler data: 
;      The event sector does not contain any events.  UDF fills with the following
;      values:
;        tof    = 3
;        polar  = 12
;        energy = 0
;-------------------------------------------------------------------------------

FUNCTION CheckFiller, tof, polar, energy

  status = 1
  filler = 0

  if (tof eq 3) then filler = filler + 1
  if (polar eq 12) then filler = filler + 1
  if (energy eq 0) then filler = filler + 1

  if (filler eq 3L) then begin
     status = 0
  endif

  RETURN, status

END


;-------------------------------------------------------------------------------
;  Function:  l_EventReadData
;  Description:
;    Read the event data.
;
;  Return Value:  type = long
;  Value                         Description
;  -------------------------     ------------------------------
;  0                             success
;  -1                            failure
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  inst                  string        I        instrument (LENA)
;  expr                  string        I        experiment (LENASCI or LENAHSKP)
;  vinst                 string        I        virtual instrument (name can be
;                                               found in $UDF_DATA in either the
;                                               IMAGE1 or IMAGE1Q of the LENASCI or
;                                               LENAHSKP directories) 
;  btime                 long[]        O        UDF begin time [yr, doy, hh, mm, ss]    
;  etime                 long[]        O        UDF end time [yr, doy, hh, mm, ss]    
;  btme                  double[]      O        UDF time converted into seconds
;  l_event               struct        O        event data specific parameters and
;                                               attributes
;  nele                  long          O        total number of records for this pass
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;-------------------------------------------------------------------------------

FUNCTION l_EventReadData, inst, expr, vinst, btime, etime, $
btme, l_event, nele
COMMON ERROR_TYPES, SUCCESS, FAILURE, WARNING, INFO


   indx = 0L
   first = 0L
   st_indx = 0L
   sec_indx = 0L
   event_per_sect_cnt = 0L
   udf_phase_angle = 0.0
   prev_time = 0.0D0
   curr_time = 0.0D0

   ;filler  = intarr (nele)
   tof     = intarr (nele)
   plr     = intarr (nele)
   engy    = intarr (nele)
   az      = fltarr (nele)
   sectCnts= fltarr (nele)
   sctime  = dblarr (nele)
   start_array = replicate ({UDF_TIMES_STRUCT}, nele)
   stop_array = replicate ({UDF_TIMES_STRUCT}, nele)


   data_source = ' '
   fh = l_UdfGetDataHandle (inst, expr, vinst, btime, etime, data_source, /GROUP)
   if (fh eq -1) then begin
       udf_close, fh
       RETURN, FAILURE 
   endif

   while not udf_eof (fh) do begin

       d = udf_read (fh)
       bt = l_GetTime (d.btime, doy, hr, min, sec)
       curr_time = bt
       if (first eq 0) then begin
          prev_time = bt
          first = 1
       endif

       start_array[indx].year = d.btime.year
       start_array[indx].doy  = d.btime.doy
       start_array[indx].hour = d.btime.hour
       start_array[indx].min  = d.btime.min
       start_array[indx].sec  = d.btime.sec
       start_array[indx].msec = d.btime.msec

       stop_array[indx].year = d.etime.year
       stop_array[indx].doy  = d.etime.doy
       stop_array[indx].hour = d.etime.hour
       stop_array[indx].min  = d.etime.min
       stop_array[indx].sec  = d.etime.sec
       stop_array[indx].msec = d.etime.msec

       ;--------------------------------------------------
       ; determine the center phase angle
       ; algorithm from Chris Gurgiolo
       ;--------------------------------------------------

       delta_angle = d.stop_az[0] - d.start_az[0] 
       udf_angle = d.stop_az[0] + d.start_az[0]

       if (delta_angle gt 0.0) then begin
           udf_phase_angle = udf_angle/2.0
       endif else begin
           udf_phase_angle = (360.0 + udf_angle)/2.0
           udf_phase_angle = udf_phase_angle mod 360.0
       endelse

       ;--------------------------------------------------
       ; convert the UDF phase angle into spin sector
       ; algorithm from Michael Collier
       ;--------------------------------------------------
       if (udf_phase_angle ge 225.1) then begin
          az[indx]  = fix((udf_phase_angle - 225.1)/8.)
       endif else begin
          if (udf_phase_angle lt 225.1) then begin
             az[indx]  = fix((udf_phase_angle + 134.9)/8.)
          endif
       endelse

       if (az[indx] eq 45) then az[indx] = 0

       ;------------------------------
       ; energy bins
       ;------------------------------
       engy[indx]  = d.event_data[0,0]

       ;------------------------------
       ; polar bins 
       ;------------------------------
       plr[indx]   = d.event_data[0,1]

       ;------------------------------
       ; tof bins
       ;------------------------------
       tof[indx]   = d.event_data[0,2]

       sctime[indx]= bt

       ;--------------------------------------------------
       ; count the number of events in each sector
       ;--------------------------------------------------
       if (prev_time eq curr_time) then begin
          stat = CheckFiller (tof[indx], plr[indx], engy[indx])
          ;--------------------------------------------------
          ; if stat = 1 not filler, good data
          ; if stat = 0 filler data
          ;--------------------------------------------------
          if (stat eq 1L) then begin
             event_per_sect_cnt = event_per_sect_cnt + 1L
          endif
       endif else begin
          sectCnts[st_indx:indx-1L] = event_per_sect_cnt
          st_indx = indx
          event_per_sect_cnt = 1L
       endelse
       prev_time = curr_time
         
       indx = indx + 1L

   endwhile 

   ;--------------------------------------------------
   ; store last counter
   ;--------------------------------------------------
   sectCnts[st_indx:indx-1L] = event_per_sect_cnt
   
   udf_close, fh

   l_event.nitems = indx


   ;----------------------------------------------------------
   ; need to adjust indexing since originally the number of
   ; records is estimated, this will store the actual
   ;----------------------------------------------------------
   ptr = ptr_new (az[0:indx-1L], /NO_COPY)
   l_event.azimuthPtr = ptr

   ptr = ptr_new (tof[0:indx-1L], /NO_COPY)
   l_event.tofPtr = ptr

   ptr = ptr_new (plr[0:indx-1L], /NO_COPY)
   l_event.polarPtr = ptr

   ptr = ptr_new (engy[0:indx-1L], /NO_COPY)
   l_event.energyPtr = ptr

   ptr = ptr_new (sctime[0:indx-1L], /NO_COPY)
   l_event.timePtr = ptr

   ptr = ptr_new (sctime[0:indx-1L], /NO_COPY)
   l_event.filledTimePtr = ptr

   ptr = ptr_new (start_array[0:indx-1L], /NO_COPY)
   l_event.startUdfTimePtr = ptr

   ptr = ptr_new (start_array[0:indx-1L], /NO_COPY)
   l_event.orig_startUdfTimePtr = ptr

   ptr = ptr_new (stop_array[0:indx-1L], /NO_COPY)
   l_event.stopUdfTimePtr = ptr

   ptr = ptr_new (stop_array[0:indx-1L], /NO_COPY)
   l_event.orig_stopUdfTimePtr = ptr

   ptr = ptr_new (sectCnts[0:indx-1L], /NO_COPY)
   l_event.sectCntsPtr = ptr


  RETURN, SUCCESS 

END


;-------------------------------------------------------------------------------
;  Procedure: l_EventImages
;
;  Description: Do image correction on the event hydrogen and oxygen data.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_event               struct *      I        pointers to event data structures
;  l_event_setup         struct        I        event processing & plotting parameters
;  l_setup               struct        I        general setup parameters
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------

PRO l_EventImages, l_event, l_event_setup, l_setup

   nspins = n_elements ((*l_event.startUdfTimePtr))

   nsp = 0L
   prev_sect = 0L
   curr_sect = 0L
   Oimg = fltarr (12, 45, nspins)
   Himg = fltarr (12, 45, nspins)
   sp_tme = dblarr (nspins) 

   prev_event_time = (*l_event.timePtr)[0] 

   for i = 0L, l_event.nitems - 1L do begin

     if (((*l_event.tofPtr)[i] ge l_event_setup.oxygen_beg_chan) AND $
         ((*l_event.tofPtr)[i] le l_event_setup.oxygen_end_chan)) then begin
         Oimg[(*l_event.polarPtr)[i], (*l_event.azimuthPtr)[i], nsp] =  $ 
             Oimg[(*l_event.polarPtr)[i], (*l_event.azimuthPtr)[i], nsp] + 1L 
     endif

     if (((*l_event.tofPtr)[i] ge l_event_setup.hydrogen_beg_chan) AND $
         ((*l_event.tofPtr)[i] le l_event_setup.hydrogen_end_chan)) then begin
         Himg[(*l_event.polarPtr)[i], (*l_event.azimuthPtr)[i], nsp] =  $
            Himg[(*l_event.polarPtr)[i], (*l_event.azimuthPtr)[i], nsp]  + 1L  
     endif

     start_event_time = (*l_event.timePtr)[i]
     curr_sect = (*l_event.azimuthPtr)[i]

     if (curr_sect lt prev_sect) AND (start_event_time ge prev_event_time) then begin
        sp_tme[nsp] = (*l_event.timePtr)[i]
        nsp = nsp + 1L
     endif

     prev_sect = curr_sect
     prev_event_time = start_event_time

   endfor


   if (l_setup.l_image_correction eq 'Y') then begin
      l_CorrectImages, Oimg, nspins
      l_CorrectImages, Himg, nspins
   endif

   ptr = ptr_new (sp_tme, /NO_COPY)
   l_event.imgSpTme = ptr

   ptr = ptr_new (Oimg, /NO_COPY)
   l_event.imgOcorr = ptr

   ptr = ptr_new (Himg, /NO_COPY)
   l_event.imgHcorr = ptr



RETURN
END


;-------------------------------------------------------------------------------
;  Procedure: l_EventJustBkgrndData
;
;  Description: Extract just the background data. Used for background
;      correction of the binned data.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_event               struct *      I        pointer to event data structures
;  l_event_setup         struct        I        event processing & plotting parameters
;  l_setup               struct        I        general setup parameters
;  event_cnts            float []      I        count of # of events per sector
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;     PLOTDEFS         MAX_PLOTS        int         G         max # plots per page
;                      MAX_WEB_SPINS    int         G         max # spins per plot
;                                                             when using web interface
;                      NO_CNTS          double      G         no counts indicator
;                      NO_DATA          double      G         no data indicator
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------

PRO l_EventJustBkgrndData, l_event, l_event_setup, l_setup, event_cnts
COMMON PLOTDEFS, MAX_PLOTS, MAX_WEB_SPINS, NO_CNTS, NO_DATA


   nele             = n_elements ((*l_event.tofPtr))
   time_nele        = n_elements((*l_event.orig_startUdfTimePtr)[*])
   start_event_time = Convert((*l_event.orig_startUdfTimePtr)[0])
   stop_event_time  = Convert((*l_event.orig_startUdfTimePtr)[time_nele-1L])
   est_nspins       = round((stop_event_time - start_event_time) / 120.) + 1L

   spin      = 0L
   curr_sect = 0L
   prev_sect = 0L

   background               = fltarr (l_setup.l_num_spinsec, est_nspins)
   background_e0            = fltarr (l_setup.l_num_spinsec, est_nspins)
   background_e1            = fltarr (l_setup.l_num_spinsec, est_nspins)
   background_e2            = fltarr (l_setup.l_num_spinsec, est_nspins)
   background_nevents       = fltarr (l_setup.l_num_spinsec, est_nspins)
   background_nevents_e0    = fltarr (l_setup.l_num_spinsec, est_nspins)
   background_nevents_e1    = fltarr (l_setup.l_num_spinsec, est_nspins)
   background_nevents_e2    = fltarr (l_setup.l_num_spinsec, est_nspins)
   background_sec_events    = fltarr (l_setup.l_num_spinsec, est_nspins)
   background_sec_events_e0 = fltarr (l_setup.l_num_spinsec, est_nspins)
   background_sec_events_e1 = fltarr (l_setup.l_num_spinsec, est_nspins)
   background_sec_events_e2 = fltarr (l_setup.l_num_spinsec, est_nspins)

   event_start_udf_time  = replicate ({UDF_TIMES_STRUCT}, est_nspins)
   event_stop_udf_time   = replicate ({UDF_TIMES_STRUCT}, est_nspins)
   event_time            = dblarr (l_setup.l_num_spinsec, est_nspins)


   prev_event_time = (*l_event.timePtr)[0]

   event_start_udf_time[0] = (*l_event.orig_startUdfTimePtr)[0]
   event_stop_udf_time[0] = (*l_event.orig_stopUdfTimePtr)[0]
   event_time[*,0] = (*l_event.timePtr)[0]


   for i = 0L, nele - 1L do begin

      start_event_time = (*l_event.timePtr)[i]
      curr_sect = (*l_event.azimuthPtr)[i]

      if (curr_sect lt prev_sect) AND (start_event_time ge prev_event_time) then begin
         spin = spin + 1L
         event_start_udf_time[spin] = (*l_event.orig_startUdfTimePtr)[i]
         event_stop_udf_time[spin] = (*l_event.orig_stopUdfTimePtr)[i]
         event_time[*,spin] = (*l_event.timePtr)[i]
      endif

      prev_sect = curr_sect
      prev_event_time = start_event_time

      ;-----------------------------------------------------
      ; determine scaled background number
      ;-----------------------------------------------------
      if (((*l_event.tofPtr)[i] ge l_event_setup.bkgrnd_beg_chan) AND $
          ((*l_event.tofPtr)[i] le l_event_setup.bkgrnd_end_chan)) then begin

          background[curr_sect, spin] = background[curr_sect, spin] + 1

          if (event_cnts[i] ne -9) then begin
             ;-----------------------------------------------------
             ; sector events counter, does not saturate at 64
             ;-----------------------------------------------------
             background_nevents[curr_sect, spin] = event_cnts[i]
          endif else begin
             background_nevents[curr_sect, spin] = 0.0
          endelse

          ;-----------------------------------------------------
          ; counter that saturates at 64
          ;-----------------------------------------------------
          background_sec_events[curr_sect, spin] = (*l_event.sectCntsPtr)[i]

          ;-----------------------------------------------------
          ; energy 0
          ;-----------------------------------------------------
          if ((*l_event.energyPtr)[i] eq 0) then begin
             background_e0 [curr_sect, spin] = background_e0[curr_sect, spin] + 1

             if (event_cnts[i] ne -9) then begin
                ;-----------------------------------------------------
                ; sector events counter, does not saturate at 64
                ;-----------------------------------------------------
                background_nevents_e0[curr_sect, spin] = event_cnts[i]
             endif else begin
                background_nevents_e0[curr_sect, spin] = 0.0
             endelse

             ;-----------------------------------------------------
             ; counter that saturates at 64
             ;-----------------------------------------------------
             background_sec_events_e0[curr_sect, spin] = (*l_event.sectCntsPtr)[i]

          endif else begin

             if ((*l_event.energyPtr)[i] eq 1) then begin
                background_e1 [curr_sect, spin] = background_e1[curr_sect, spin] + 1

                if (event_cnts[i] ne -9) then begin
                ;-----------------------------------------------------
                ; sector events counter, does not saturate at 64
                ;-----------------------------------------------------
                   background_nevents_e1[curr_sect, spin] = event_cnts[i]
                endif else begin
                   background_nevents_e1[curr_sect, spin] = 0.0
                endelse

          
                ;-----------------------------------------------------
                ; counter that saturates at 64
                ;-----------------------------------------------------
                background_sec_events_e1[curr_sect, spin] = (*l_event.sectCntsPtr)[i]

             endif else begin

                 if ((*l_event.energyPtr)[i] eq 2) then begin
                    background_e2 [curr_sect, spin] = background_e2[curr_sect, spin] + 1

                    if (event_cnts[i] ne -9) then begin
                       ;-----------------------------------------------------
                       ; sector events counter, does not saturate at 64
                       ;-----------------------------------------------------
                       background_nevents_e2[curr_sect, spin] = event_cnts[i]
                    endif else begin
                       background_nevents_e2[curr_sect, spin] = 0.0
                    endelse

          
                    ;-----------------------------------------------------
                    ; counter that saturates at 64
                    ;-----------------------------------------------------
                    background_sec_events_e2[curr_sect, spin] = (*l_event.sectCntsPtr)[i]
                endif
             endelse
          endelse
      endif


   endfor


   ;spin = spin + 1L


   ;-----------------------------------------------------
   ; take the total number of events per spin sector and
   ; divide by the sector event count (possible number of
   ; events per sector)
   ;-----------------------------------------------------
   scaled_background    = fltarr(l_setup.l_num_spinsec, spin)
   scaled_background_e0 = fltarr(l_setup.l_num_spinsec, spin)
   scaled_background_e1 = fltarr(l_setup.l_num_spinsec, spin)
   scaled_background_e2 = fltarr(l_setup.l_num_spinsec, spin)
   nsectors = l_setup.l_num_spinsec

   for i = 0L, spin - 1L do begin
      for j = 0L,  nsectors - 1L do begin
         if (background_sec_events[j,i] ne 0.0) then begin
            scaled_background[j,i]  = background[j,i] * $
                      background_nevents[j,i]/background_sec_events[j,i]
         endif
         if (background_sec_events_e0[j,i] ne 0.0) then begin
            scaled_background_e0[j,i]  = background_e0[j,i] * $
                      background_nevents_e0[j,i]/background_sec_events_e0[j,i]
         endif
         if (background_sec_events_e1[j,i] ne 0.0) then begin
            scaled_background_e1[j,i]  = background_e1[j,i] * $
                      background_nevents_e1[j,i]/background_sec_events_e1[j,i]
         endif
         if (background_sec_events_e2[j,i] ne 0.0) then begin
            scaled_background_e2[j,i]  = background_e2[j,i] * $
                      background_nevents_e2[j,i]/background_sec_events_e2[j,i]
         endif
      endfor
   endfor


   if (ptr_valid (l_event.filledStartUdfTimePtr)) then ptr_free, l_event.filledStartUdfTimePtr
   ptr = ptr_new (event_start_udf_time[0:spin-1], /NO_COPY)
   l_event.filledStartUdfTimePtr = ptr

   if (ptr_valid (l_event.filledStopUdfTimePtr)) then ptr_free, l_event.filledStopUdfTimePtr
   ptr = ptr_new (event_stop_udf_time[0:spin-1], /NO_COPY)
   l_event.filledStopUdfTimePtr = ptr

   if (ptr_valid (l_event.background)) then ptr_free, l_event.background
   ptr = ptr_new (scaled_background[*,0:spin-1], /NO_COPY)
   l_event.background = ptr

   if (ptr_valid (l_event.background_e0)) then ptr_free, l_event.background_e0
   ptr = ptr_new (scaled_background_e0[*,0:spin-1], /NO_COPY)
   l_event.background_e0 = ptr

   if (ptr_valid (l_event.background_e1)) then ptr_free, l_event.background_e1
   ptr = ptr_new (scaled_background_e1[*,0:spin-1], /NO_COPY)
   l_event.background_e1 = ptr

   if (ptr_valid (l_event.background_e2)) then ptr_free, l_event.background_e2
   ptr = ptr_new (scaled_background_e2[*,0:spin-1], /NO_COPY)
   l_event.background_e2 = ptr


RETURN
END



;-------------------------------------------------------------------------------
;  Function : l_EventGetData
;
;  Description: Determines the number of records and reads the event data.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  SUCCESS                       successful completion
;  FAILURE                       failure to complete
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_setup               struct        I        generic setup parameters
;  l_setup_file          string        I        setup file name
;  l_pltpos              struct        I        generic plot parameters/attributes
;  l_event_setup         struct        I        event specific plot setup parameters
;  l_event               struct        I        event specific data plot setup 
;                                               parameters
;  
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;     PLOTDEFS         MAX_PLOTS        int         G         max # plots per page
;                      MAX_WEB_SPINS    int         G         max # spins per plot
;                                                             when using web interface
;                      NO_CNTS          double      G         no counts indicator
;                      NO_DATA          double      G         no data indicator
;
;      ERROR_TYPES     SUCCESS         int         G          successful completion
;                      FAILURE         int         G          unsuccessful completion
;                      WARNING         int         G          warning message
;                      INFO            int         G          informational message
;
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;  ELH                               v1.0.7           Added parameters to l_event
;                                                     struct needed for H/O background
;                                                     correction 
;                                                     Eliminated call to l_UdfGetNumRec
;                                                     and estimated number of records
;                                                     based on user start/stop times
;-------------------------------------------------------------------------------

FUNCTION l_EventGetData, l_setup, l_setup_file, l_pltpos, $
l_event_setup, l_event
COMMON PLOTDEFS, MAX_PLOTS, MAX_WEB_SPINS, NO_CNTS, NO_DATA
COMMON ERROR_TYPES, SUCCESS, FAILURE, WARNING, INFO

   inst    = 'LENA' 
   expr    = 'LENASCI'
   vinst   = 'IMLEVENT' 

   numRecs = 0L
   data_source = ' '

   ptr= ptr_new()
   l_event = {          $
      timePtr                  :ptr, $
      spTimePtr                :ptr, $
      filledTimePtr            :ptr, $
      bkcUdfTimePtr            :ptr, $
      azimuthPtr               :ptr, $
      tofPtr                   :ptr, $
      polarPtr                 :ptr, $
      energyPtr                :ptr, $
      sectCntsPtr              :ptr, $
      startUdfTimePtr          :ptr, $
      filledStartUdfTimePtr    :ptr, $
      orig_startUdfTimePtr     :ptr, $
      stopUdfTimePtr           :ptr, $
      filledStopUdfTimePtr     :ptr, $
      orig_stopUdfTimePtr      :ptr, $
      background               :ptr, $
      background_e0            :ptr, $
      background_e1            :ptr, $
      background_e2            :ptr, $
      background_raw_H         :ptr, $
      background_raw_H_e0      :ptr, $
      background_raw_H_e1      :ptr, $
      background_raw_H_e2      :ptr, $
      background_raw_O         :ptr, $
      background_raw_O_e0      :ptr, $
      background_raw_O_e1      :ptr, $
      background_raw_O_e2      :ptr, $
      background_raw_extra     :ptr, $
      background_raw_extra_e0  :ptr, $
      background_raw_extra_e1  :ptr, $
      background_raw_extra_e2  :ptr, $
      hydrogen_raw             :ptr, $
      hydrogen_raw_e0          :ptr, $
      hydrogen_raw_e1          :ptr, $
      hydrogen_raw_e2          :ptr, $
      hydrogen_raw_extra       :ptr, $
      hydrogen_raw_extra_e0    :ptr, $
      hydrogen_raw_extra_e1    :ptr, $
      hydrogen_raw_extra_e2    :ptr, $
      hydrogen_e0              :ptr, $
      hydrogen_e0_extra        :ptr, $
      hydrogen_e0_flux         :ptr, $
      hydrogen_e0_log_flux     :ptr, $
      hydrogen_e1              :ptr, $
      hydrogen_e1_extra        :ptr, $
      hydrogen_e1_flux         :ptr, $
      hydrogen_e1_log_flux     :ptr, $
      hydrogen_e2              :ptr, $
      hydrogen_e2_extra        :ptr, $
      hydrogen_e2_flux         :ptr, $
      hydrogen_e2_log_flux     :ptr, $
      H_sec_events             :ptr, $
      H_sec_events_e0          :ptr, $
      H_sec_events_e1          :ptr, $
      H_sec_events_e2          :ptr, $
      H_sec_events_extra       :ptr, $
      H_sec_events_extra_e0    :ptr, $
      H_sec_events_extra_e1    :ptr, $
      H_sec_events_extra_e2    :ptr, $
      H_n_in_peak              :ptr, $
      H_n_in_peak_e0           :ptr, $
      H_n_in_peak_e1           :ptr, $
      H_n_in_peak_e2           :ptr, $
      H_n_in_peak_extra        :ptr, $
      H_n_in_peak_extra_e0     :ptr, $
      H_n_in_peak_extra_e1     :ptr, $
      H_n_in_peak_extra_e2     :ptr, $
      H_scale_factor           :ptr, $
      H_scale_factor_e0        :ptr, $
      H_scale_factor_e1        :ptr, $
      H_scale_factor_e2        :ptr, $
      H_scale_factor_extra     :ptr, $
      H_scale_factor_extra_e0  :ptr, $
      H_scale_factor_extra_e1  :ptr, $
      H_scale_factor_extra_e2  :ptr, $
      scaled_H                 :ptr, $
      scaled_H_e0              :ptr, $
      scaled_H_e1              :ptr, $
      scaled_H_e2              :ptr, $
      corrected_H              :ptr, $
      corrected_H_e0           :ptr, $
      corrected_H_e1           :ptr, $
      corrected_H_e2           :ptr, $
      corrected_H_extra        :ptr, $
      corrected_H_extra_e0     :ptr, $
      corrected_H_extra_e1     :ptr, $
      corrected_H_extra_e2     :ptr, $
      corr_scl_H               :ptr, $
      corr_scl_H_e0            :ptr, $
      corr_scl_H_e1            :ptr, $
      corr_scl_H_e2            :ptr, $
      background_H             :ptr, $
      background_H_e0          :ptr, $
      background_H_e1          :ptr, $
      background_H_e2          :ptr, $
      background_H_extra       :ptr, $
      background_H_extra_e0    :ptr, $
      background_H_extra_e1    :ptr, $
      background_H_extra_e2    :ptr, $
      oxygen_raw               :ptr, $
      oxygen_raw_e0            :ptr, $
      oxygen_raw_e1            :ptr, $
      oxygen_raw_e2            :ptr, $
      oxygen_raw_extra         :ptr, $
      oxygen_raw_extra_e0      :ptr, $
      oxygen_raw_extra_e1      :ptr, $
      oxygen_raw_extra_e2      :ptr, $
      oxygen_e0                :ptr, $
      oxygen_e0_extra          :ptr, $
      oxygen_e0_flux           :ptr, $
      oxygen_e0_log_flux       :ptr, $
      oxygen_e1                :ptr, $
      oxygen_e1_extra          :ptr, $
      oxygen_e1_flux           :ptr, $
      oxygen_e1_log_flux       :ptr, $
      oxygen_e2                :ptr, $
      oxygen_e2_extra          :ptr, $
      oxygen_e2_flux           :ptr, $
      oxygen_e2_log_flux       :ptr, $
      O_sec_events             :ptr, $
      O_sec_events_e0          :ptr, $
      O_sec_events_e1          :ptr, $
      O_sec_events_e2          :ptr, $
      O_sec_events_extra       :ptr, $
      O_sec_events_extra_e0    :ptr, $
      O_sec_events_extra_e1    :ptr, $
      O_sec_events_extra_e2    :ptr, $
      O_n_in_peak              :ptr, $
      O_n_in_peak_e0           :ptr, $
      O_n_in_peak_e1           :ptr, $
      O_n_in_peak_e2           :ptr, $
      O_n_in_peak_extra        :ptr, $
      O_n_in_peak_extra_e0     :ptr, $
      O_n_in_peak_extra_e1     :ptr, $
      O_n_in_peak_extra_e2     :ptr, $
      O_scale_factor           :ptr, $
      O_scale_factor_e0        :ptr, $
      O_scale_factor_e1        :ptr, $
      O_scale_factor_e2        :ptr, $
      O_scale_factor_extra     :ptr, $
      O_scale_factor_extra_e0  :ptr, $
      O_scale_factor_extra_e1  :ptr, $
      O_scale_factor_extra_e2  :ptr, $
      corrected_O              :ptr, $
      corrected_O_e0           :ptr, $
      corrected_O_e1           :ptr, $
      corrected_O_e2           :ptr, $
      corrected_O_extra        :ptr, $
      corrected_O_extra_e0     :ptr, $
      corrected_O_extra_e1     :ptr, $
      corrected_O_extra_e2     :ptr, $
      corr_scl_O               :ptr, $
      corr_scl_O_e0            :ptr, $
      corr_scl_O_e1            :ptr, $
      corr_scl_O_e2            :ptr, $
      scaled_O                 :ptr, $
      scaled_O_e0              :ptr, $
      scaled_O_e1              :ptr, $
      scaled_O_e2              :ptr, $
      background_O             :ptr, $
      background_O_e0          :ptr, $
      background_O_e1          :ptr, $
      background_O_e2          :ptr, $
      background_O_extra       :ptr, $
      background_O_extra_e0    :ptr, $
      background_O_extra_e1    :ptr, $
      background_O_extra_e2    :ptr, $
      all_masses               :ptr, $
      amDmpPtr                 :ptr, $
      nitems                   :0L,  $
      addedStartFlag           :0L,  $
      addedStopFlag            :0L,  $
      imgHcorr                 :ptr, $
      imgOcorr                 :ptr, $
      imgSpTme                 :ptr, $
      num_bk_elim_spins        :0L,  $
      num_H_elim_spins         :0L,  $
      num_O_elim_spins         :0L,  $
      H_elim_spins_time        :ptr, $
      H_elim_spins_nevents     :ptr, $
      O_elim_spins_time        :ptr, $
      O_elim_spins_nevents     :ptr  $
   }



   ;----------------------------------------------------
   ; Add extra time to the user specified stop time
   ; to correct for the fact that UDF does not read up
   ; to and including the last spin for the binned data 
   ; as it does for the other data, this is so that the 
   ; time and data will be sync with the other data 
   ;----------------------------------------------------

   start_tme = l_setup.l_start_dtime
   stop_tme = l_setup.l_stop_dtime

   ;----------------------------------------------------
   ; if the start time is not 00:00:00, then add extra
   ; minutes to the begin time
   ;----------------------------------------------------
   l_event.addedStartFlag = 0L
   if (ChckStartTime(start_tme) eq FAILURE) then begin
      AddExtraTime2Start, start_tme, 2
      l_event.addedStartFlag = 1L
   endif

   ;----------------------------------------------------
   ; if the stop time is not 23:59:59, then add extra
   ; minutes to the end time
   ;----------------------------------------------------
   l_event.addedStopFlag = 0L
   if (ChckStopTime(stop_tme) eq FAILURE) then begin
      ;AddExtraTime2Stop, stop_tme, 6 
      AddExtraTime2Stop, stop_tme, 2 
      l_event.addedStopFlag = 1L
   endif


   ;----------------------------------------------------
   ; want to find exact number of records and not 
   ; estimate, event data requires too much resources
   ; if estimating the size of a data structure
   ; (64*45*nspins)
   ;
   ; get the number of records from T1 to T2
   ;----------------------------------------------------
   numRecs = l_UdfGetNumRec (start_tme, $
                             stop_tme,  $
                             inst, expr, vinst, data_source )

   if (numRecs le 0) then return, FAILURE



   l_pltpos.data_source = data_source

   status = l_EventReadData (inst, expr, vinst,     $
                    start_tme, $
                    stop_tme,  $
                    sctime,                $
                    l_event, numRecs) 
   if (status eq FAILURE) then return, FAILURE


   if (l_setup.l_data_type eq 'l_zc3') then begin
      l_EventImages, l_event, l_event_setup, l_setup
   endif


   RETURN, status 

END

;-------------------------------------------------------------------------------
;  Procedure :  l_Event_Adj4ExtraData
;
;  Description:
;     Adjust the array sizes if extra data was read for the adjusting of
;  sun, earth, moon, routine.  This is only used for the H/O/Background plots!
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_event               struct *      I        pointers to event data structures
;  l_event_setup         struct        I        event processing & plotting parameters
;  l_setup               struct        I        general setup parameters
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;     PLOTDEFS         MAX_PLOTS        int         G         max # plots per page
;                      MAX_WEB_SPINS    int         G         max # spins per plot
;                                                             when using web interface
;                      NO_CNTS          double      G         no counts indicator
;                      NO_DATA          double      G         no data indicator
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               09/04/01        v1.0.7          Original implementation
;-------------------------------------------------------------------------------


PRO l_Event_Adj4ExtraData, l_event, l_event_setup, l_setup
COMMON PLOTDEFS, MAX_PLOTS, MAX_WEB_SPINS, NO_CNTS, NO_DATA


   extra_data = make_array (l_setup.l_num_spinsec, 2L, /FLOAT, Value = NO_DATA)

   ptr = ptr_new (extra_data)

   if (l_event_setup.background eq 'Y') then begin
      l_event.background_H_extra = ptr
      l_event.background_H_extra_e0 = ptr
      l_event.background_H_extra_e1 = ptr
      l_event.background_H_extra_e2 = ptr
      l_event.background_O_extra = ptr
      l_event.background_O_extra_e0 = ptr
      l_event.background_O_extra_e1 = ptr
      l_event.background_O_extra_e2 = ptr
      l_event.background_raw_extra = ptr
      l_event.background_raw_extra_e0 = ptr
      l_event.background_raw_extra_e1 = ptr
      l_event.background_raw_extra_e2 = ptr
   endif

   if (l_event_setup.hydrogen eq 'Y') then begin 
      l_event.hydrogen_raw_extra = ptr
      l_event.hydrogen_raw_extra_e0 = ptr
      l_event.hydrogen_raw_extra_e1 = ptr
      l_event.hydrogen_raw_extra_e2 = ptr
      l_event.hydrogen_e0_extra = ptr
      l_event.hydrogen_e1_extra = ptr
      l_event.hydrogen_e2_extra = ptr
      l_event.H_sec_events_extra = ptr
      l_event.H_sec_events_extra_e0 = ptr
      l_event.H_sec_events_extra_e1 = ptr
      l_event.H_sec_events_extra_e2 = ptr
      l_event.H_n_in_peak_extra = ptr
      l_event.H_n_in_peak_extra_e0 = ptr
      l_event.H_n_in_peak_extra_e1 = ptr
      l_event.H_n_in_peak_extra_e2 = ptr
      l_event.H_scale_factor_extra = ptr
      l_event.H_scale_factor_extra_e0 = ptr
      l_event.H_scale_factor_extra_e1 = ptr
      l_event.H_scale_factor_extra_e2 = ptr
      l_event.corrected_H_extra = ptr 
      l_event.corrected_H_extra_e0 = ptr 
      l_event.corrected_H_extra_e1 = ptr 
      l_event.corrected_H_extra_e2 = ptr 
    
   endif

   if (l_event_setup.oxygen eq 'Y') then begin 
      l_event.oxygen_raw_extra = ptr
      l_event.oxygen_raw_extra_e0 = ptr
      l_event.oxygen_raw_extra_e1 = ptr
      l_event.oxygen_raw_extra_e2 = ptr
      l_event.oxygen_e0_extra = ptr
      l_event.oxygen_e1_extra = ptr
      l_event.oxygen_e2_extra = ptr
      l_event.O_sec_events_extra = ptr
      l_event.O_sec_events_extra_e0 = ptr
      l_event.O_sec_events_extra_e1 = ptr
      l_event.O_sec_events_extra_e2 = ptr
      l_event.O_n_in_peak_extra = ptr
      l_event.O_n_in_peak_extra_e0 = ptr
      l_event.O_n_in_peak_extra_e1 = ptr
      l_event.O_n_in_peak_extra_e2 = ptr
      l_event.O_scale_factor_extra = ptr
      l_event.O_scale_factor_extra_e0 = ptr
      l_event.O_scale_factor_extra_e1 = ptr
      l_event.O_scale_factor_extra_e2 = ptr
      l_event.corrected_O_extra = ptr 
      l_event.corrected_O_extra_e0 = ptr 
      l_event.corrected_O_extra_e1 = ptr 
      l_event.corrected_O_extra_e2 = ptr 

   endif

   tme_nrec = n_elements ((*l_event.orig_startUdfTimePtr))


   if (l_event.addedStartFlag eq 1L) AND (l_event.addedStopFlag eq 1L) then begin

      (*l_event.startUdfTimePtr) = (*l_event.orig_startUdfTimePtr)[1:tme_nrec-2L]
      (*l_event.stopUdfTimePtr) = (*l_event.orig_stopUdfTimePtr)[1:tme_nrec-2L]

      e_nrec = n_elements((*l_event.spTimePtr))/l_setup.l_num_spinsec
      (*l_event.spTimePtr) = (*l_event.spTimePtr)[*, 1:e_nrec-2L]

      eu_nrec = n_elements((*l_event.filledStartUdfTimePtr))
      (*l_event.filledStartUdfTimePtr) = (*l_event.filledStartUdfTimePtr) [1:eu_nrec-2L]
      (*l_event.filledStopUdfTimePtr) = (*l_event.filledStopUdfTimePtr) [1:eu_nrec-2L]


      if (l_event_setup.hydrogen eq 'Y') then begin

         if (l_event_setup.background eq 'Y') then begin

            bk_nrec = n_elements((*l_event.background_H))/l_setup.l_num_spinsec 
            extra_data[*,0] = (*l_event.background_H)[*, 0]
            extra_data[*,1] = (*l_event.background_H)[*, bk_nrec-1L]
            (*l_event.background_H_extra) = extra_data
            (*l_event.background_H) = (*l_event.background_H)[*, 1:bk_nrec-2L]
            bkr_nrec = n_elements((*l_event.background_raw_H))/l_setup.l_num_spinsec 
            (*l_event.background_raw_H) = (*l_event.background_raw_H)[*, 1:bkr_nrec-2L]

            bk_nrec = n_elements((*l_event.background_H_e0))/l_setup.l_num_spinsec 
            extra_data[*,0] = (*l_event.background_H_e0)[*, 0]
            extra_data[*,1] = (*l_event.background_H_e0)[*, bk_nrec-1L]
            (*l_event.background_H_extra_e0) = extra_data
            (*l_event.background_H_e0) = (*l_event.background_H_e0)[*, 1:bk_nrec-2L]
            bkr_nrec = n_elements((*l_event.background_raw_H_e0))/l_setup.l_num_spinsec 
            (*l_event.background_raw_H_e0) = (*l_event.background_raw_H_e0)[*, 1:bkr_nrec-2L]

            bk_nrec = n_elements((*l_event.background_H_e1))/l_setup.l_num_spinsec 
            extra_data[*,0] = (*l_event.background_H_e1)[*, 0]
            extra_data[*,1] = (*l_event.background_H_e1)[*, bk_nrec-1L]
            (*l_event.background_H_extra_e1) = extra_data
            (*l_event.background_H_e1) = (*l_event.background_H_e1)[*, 1:bk_nrec-2L]
            bkr_nrec = n_elements((*l_event.background_raw_H_e1))/l_setup.l_num_spinsec 
            (*l_event.background_raw_H_e1) = (*l_event.background_raw_H_e1)[*, 1:bkr_nrec-2L]

            bk_nrec = n_elements((*l_event.background_H_e2))/l_setup.l_num_spinsec 
            extra_data[*,0] = (*l_event.background_H_e2)[*, 0]
            extra_data[*,1] = (*l_event.background_H_e2)[*, bk_nrec-1L]
            (*l_event.background_H_extra_e2) = extra_data
            (*l_event.background_H_e2) = (*l_event.background_H_e2)[*, 1:bk_nrec-2L]
            bkr_nrec = n_elements((*l_event.background_raw_H_e2))/l_setup.l_num_spinsec 
            (*l_event.background_raw_H_e2) = (*l_event.background_raw_H_e2)[*, 1:bkr_nrec-2L]

         endif

         H_nrec = n_elements((*l_event.hydrogen_raw))/l_setup.l_num_spinsec 
         extra_data[*,0] = (*l_event.hydrogen_raw)[*, 0]
         extra_data[*,1] = (*l_event.hydrogen_raw)[*, H_nrec-1L]
         (*l_event.hydrogen_raw_extra) = extra_data 
         (*l_event.hydrogen_raw) = (*l_event.hydrogen_raw)[*, 1:H_nrec-2L]

         H_nrec = n_elements((*l_event.hydrogen_raw_e0))/l_setup.l_num_spinsec 
         extra_data[*,0] = (*l_event.hydrogen_raw_e0)[*, 0]
         extra_data[*,1] = (*l_event.hydrogen_raw_e0)[*, H_nrec-1L]
         (*l_event.hydrogen_raw_extra_e0) = extra_data 
         (*l_event.hydrogen_raw_e0) = (*l_event.hydrogen_raw_e0)[*, 1:H_nrec-2L]

         H_nrec = n_elements((*l_event.hydrogen_raw_e1))/l_setup.l_num_spinsec 
         extra_data[*,0] = (*l_event.hydrogen_raw_e1)[*, 0]
         extra_data[*,1] = (*l_event.hydrogen_raw_e1)[*, H_nrec-1L]
         (*l_event.hydrogen_raw_extra_e1) = extra_data 
         (*l_event.hydrogen_raw_e1) = (*l_event.hydrogen_raw_e1)[*, 1:H_nrec-2L]

         H_nrec = n_elements((*l_event.hydrogen_raw_e2))/l_setup.l_num_spinsec 
         extra_data[*,0] = (*l_event.hydrogen_raw_e2)[*, 0]
         extra_data[*,1] = (*l_event.hydrogen_raw_e2)[*, H_nrec-1L]
         (*l_event.hydrogen_raw_extra_e2) = extra_data 
         (*l_event.hydrogen_raw_e2) = (*l_event.hydrogen_raw_e2)[*, 1:H_nrec-2L]

         H_nrec = n_elements((*l_event.hydrogen_e0))/l_setup.l_num_spinsec 
         extra_data[*,0] = (*l_event.hydrogen_e0)[*, 0]
         extra_data[*,1] = (*l_event.hydrogen_e0)[*, H_nrec-1L]
         (*l_event.hydrogen_e0_extra) = extra_data 
         (*l_event.hydrogen_e0) = (*l_event.hydrogen_e0)[*, 1:H_nrec-2L]

         H_nrec = n_elements((*l_event.hydrogen_e1))/l_setup.l_num_spinsec 
         extra_data[*,0] = (*l_event.hydrogen_e1)[*, 0]
         extra_data[*,1] = (*l_event.hydrogen_e1)[*, H_nrec-1L]
         (*l_event.hydrogen_e1_extra) = extra_data 
         (*l_event.hydrogen_e1) = (*l_event.hydrogen_e1)[*, 1:H_nrec-2L]

         H_nrec = n_elements((*l_event.hydrogen_e2))/l_setup.l_num_spinsec 
         extra_data[*,0] = (*l_event.hydrogen_e2)[*, 0]
         extra_data[*,1] = (*l_event.hydrogen_e2)[*, H_nrec-1L]
         (*l_event.hydrogen_e2_extra) = extra_data 
         (*l_event.hydrogen_e2) = (*l_event.hydrogen_e2)[*, 1:H_nrec-2L]

         H_nrec = n_elements((*l_event.H_sec_events))/l_setup.l_num_spinsec 
         extra_data[*,0] = (*l_event.H_sec_events)[*, 0]
         extra_data[*,1] = (*l_event.H_sec_events)[*, H_nrec-1L]
         (*l_event.H_sec_events_extra) = extra_data
         (*l_event.H_sec_events) = (*l_event.H_sec_events)[*, 1:H_nrec-2L]

         H_nrec = n_elements((*l_event.H_sec_events_e0))/l_setup.l_num_spinsec 
         extra_data[*,0] = (*l_event.H_sec_events_e0)[*, 0]
         extra_data[*,1] = (*l_event.H_sec_events_e0)[*, H_nrec-1L]
         (*l_event.H_sec_events_extra_e0) = extra_data
         (*l_event.H_sec_events_e0) = (*l_event.H_sec_events_e0)[*, 1:H_nrec-2L]

         H_nrec = n_elements((*l_event.H_sec_events_e1))/l_setup.l_num_spinsec 
         extra_data[*,0] = (*l_event.H_sec_events_e1)[*, 0]
         extra_data[*,1] = (*l_event.H_sec_events_e1)[*, H_nrec-1L]
         (*l_event.H_sec_events_extra_e1) = extra_data
         (*l_event.H_sec_events_e1) = (*l_event.H_sec_events_e1)[*, 1:H_nrec-2L]

         H_nrec = n_elements((*l_event.H_sec_events_e2))/l_setup.l_num_spinsec 
         extra_data[*,0] = (*l_event.H_sec_events_e2)[*, 0]
         extra_data[*,1] = (*l_event.H_sec_events_e2)[*, H_nrec-1L]
         (*l_event.H_sec_events_extra_e2) = extra_data
         (*l_event.H_sec_events_e2) = (*l_event.H_sec_events_e2)[*, 1:H_nrec-2L]

         H_nrec = n_elements((*l_event.H_n_in_peak))/l_setup.l_num_spinsec 
         extra_data[*,0] = (*l_event.H_n_in_peak)[*, 0]
         extra_data[*,1] = (*l_event.H_n_in_peak)[*, H_nrec-1L]
         (*l_event.H_n_in_peak_extra) = extra_data
         (*l_event.H_n_in_peak) = (*l_event.H_n_in_peak)[*, 1:H_nrec-2L]

         H_nrec = n_elements((*l_event.H_n_in_peak_e0))/l_setup.l_num_spinsec 
         extra_data[*,0] = (*l_event.H_n_in_peak_e0)[*, 0]
         extra_data[*,1] = (*l_event.H_n_in_peak_e0)[*, H_nrec-1L]
         (*l_event.H_n_in_peak_extra_e0) = extra_data
         (*l_event.H_n_in_peak_e0) = (*l_event.H_n_in_peak_e0)[*, 1:H_nrec-2L]

         H_nrec = n_elements((*l_event.H_n_in_peak_e1))/l_setup.l_num_spinsec 
         extra_data[*,0] = (*l_event.H_n_in_peak_e1)[*, 0]
         extra_data[*,1] = (*l_event.H_n_in_peak_e1)[*, H_nrec-1L]
         (*l_event.H_n_in_peak_extra_e1) = extra_data
         (*l_event.H_n_in_peak_e1) = (*l_event.H_n_in_peak_e1)[*, 1:H_nrec-2L]

         H_nrec = n_elements((*l_event.H_n_in_peak_e2))/l_setup.l_num_spinsec 
         extra_data[*,0] = (*l_event.H_n_in_peak_e2)[*, 0]
         extra_data[*,1] = (*l_event.H_n_in_peak_e2)[*, H_nrec-1L]
         (*l_event.H_n_in_peak_extra_e2) = extra_data
         (*l_event.H_n_in_peak_e2) = (*l_event.H_n_in_peak_e2)[*, 1:H_nrec-2L]

         H_nrec = n_elements((*l_event.H_scale_factor))/l_setup.l_num_spinsec 
         extra_data[*,0] = (*l_event.H_scale_factor)[*, 0]
         extra_data[*,1] = (*l_event.H_scale_factor)[*, H_nrec-1L]
         (*l_event.H_scale_factor_extra) = extra_data
         (*l_event.H_scale_factor) = (*l_event.H_scale_factor)[*, 1:H_nrec-2L]

         H_nrec = n_elements((*l_event.H_scale_factor_e0))/l_setup.l_num_spinsec 
         extra_data[*,0] = (*l_event.H_scale_factor_e0)[*, 0]
         extra_data[*,1] = (*l_event.H_scale_factor_e0)[*, H_nrec-1L]
         (*l_event.H_scale_factor_extra_e0) = extra_data
         (*l_event.H_scale_factor_e0) = (*l_event.H_scale_factor_e0)[*, 1:H_nrec-2L]

         H_nrec = n_elements((*l_event.H_scale_factor_e1))/l_setup.l_num_spinsec 
         extra_data[*,0] = (*l_event.H_scale_factor_e1)[*, 0]
         extra_data[*,1] = (*l_event.H_scale_factor_e1)[*, H_nrec-1L]
         (*l_event.H_scale_factor_extra_e1) = extra_data
         (*l_event.H_scale_factor_e1) = (*l_event.H_scale_factor_e1)[*, 1:H_nrec-2L]

         H_nrec = n_elements((*l_event.H_scale_factor_e2))/l_setup.l_num_spinsec 
         extra_data[*,0] = (*l_event.H_scale_factor_e2)[*, 0]
         extra_data[*,1] = (*l_event.H_scale_factor_e2)[*, H_nrec-1L]
         (*l_event.H_scale_factor_extra_e2) = extra_data
         (*l_event.H_scale_factor_e2) = (*l_event.H_scale_factor_e2)[*, 1:H_nrec-2L]

         cH_nrec = n_elements((*l_event.corrected_H))/l_setup.l_num_spinsec 
         extra_data[*,0] = (*l_event.corrected_H)[*, 0]
         extra_data[*,1] = (*l_event.corrected_H)[*, cH_nrec-1L]
         (*l_event.corrected_H_extra) = extra_data
         (*l_event.corrected_H) = (*l_event.corrected_H)[*, 1:cH_nrec-2L]

         cH_nrec = n_elements((*l_event.corrected_H_e0))/l_setup.l_num_spinsec 
         extra_data[*,0] = (*l_event.corrected_H_e0)[*, 0]
         extra_data[*,1] = (*l_event.corrected_H_e0)[*, cH_nrec-1L]
         (*l_event.corrected_H_extra_e0) = extra_data
         (*l_event.corrected_H_e0) = (*l_event.corrected_H_e0)[*, 1:cH_nrec-2L]

         cH_nrec = n_elements((*l_event.corrected_H_e1))/l_setup.l_num_spinsec 
         extra_data[*,0] = (*l_event.corrected_H_e1)[*, 0]
         extra_data[*,1] = (*l_event.corrected_H_e1)[*, cH_nrec-1L]
         (*l_event.corrected_H_extra_e1) = extra_data
         (*l_event.corrected_H_e1) = (*l_event.corrected_H_e1)[*, 1:cH_nrec-2L]

         cH_nrec = n_elements((*l_event.corrected_H_e2))/l_setup.l_num_spinsec 
         extra_data[*,0] = (*l_event.corrected_H_e2)[*, 0]
         extra_data[*,1] = (*l_event.corrected_H_e2)[*, cH_nrec-1L]
         (*l_event.corrected_H_extra_e2) = extra_data
         (*l_event.corrected_H_e2) = (*l_event.corrected_H_e2)[*, 1:cH_nrec-2L]

         cH_nrec = n_elements((*l_event.scaled_H))/l_setup.l_num_spinsec 
         (*l_event.scaled_H) = (*l_event.scaled_H)[*, 1:cH_nrec-2L]

         cH_nrec = n_elements((*l_event.scaled_H_e0))/l_setup.l_num_spinsec 
         (*l_event.scaled_H_e0) = (*l_event.scaled_H_e0)[*, 1:cH_nrec-2L]

         cH_nrec = n_elements((*l_event.scaled_H_e1))/l_setup.l_num_spinsec 
         (*l_event.scaled_H_e1) = (*l_event.scaled_H_e1)[*, 1:cH_nrec-2L]

         cH_nrec = n_elements((*l_event.scaled_H_e1))/l_setup.l_num_spinsec 
         (*l_event.scaled_H_e1) = (*l_event.scaled_H_e1)[*, 1:cH_nrec-2L]

      endif



      if (l_event_setup.oxygen eq 'Y') then begin

         if (l_event_setup.background eq 'Y') then begin

            bk_nrec = n_elements((*l_event.background_O))/l_setup.l_num_spinsec 
            extra_data[*,0] = (*l_event.background_O)[*, 0]
            extra_data[*,1] = (*l_event.background_O)[*, bk_nrec-1L]
            (*l_event.background_O_extra) = extra_data
            (*l_event.background_O) = (*l_event.background_O)[*, 1:bk_nrec-2L]
            bkr_nrec = n_elements((*l_event.background_raw_O))/l_setup.l_num_spinsec 
            (*l_event.background_raw_O) = (*l_event.background_raw_O)[*, 1:bkr_nrec-2L]

            bk_nrec = n_elements((*l_event.background_O_e0))/l_setup.l_num_spinsec 
            extra_data[*,0] = (*l_event.background_O_e0)[*, 0]
            extra_data[*,1] = (*l_event.background_O_e0)[*, bk_nrec-1L]
            (*l_event.background_O_extra_e0) = extra_data
            (*l_event.background_O_e0) = (*l_event.background_O_e0)[*, 1:bk_nrec-2L]
            bkr_nrec = n_elements((*l_event.background_raw_O_e0))/l_setup.l_num_spinsec 
            (*l_event.background_raw_O_e0) = (*l_event.background_raw_O_e0)[*, 1:bkr_nrec-2L]

            bk_nrec = n_elements((*l_event.background_O_e1))/l_setup.l_num_spinsec 
            extra_data[*,0] = (*l_event.background_O_e1)[*, 0]
            extra_data[*,1] = (*l_event.background_O_e1)[*, bk_nrec-1L]
            (*l_event.background_O_extra_e1) = extra_data
            (*l_event.background_O_e1) = (*l_event.background_O_e1)[*, 1:bk_nrec-2L]
            bkr_nrec = n_elements((*l_event.background_raw_O_e1))/l_setup.l_num_spinsec 
            (*l_event.background_raw_O_e1) = (*l_event.background_raw_O_e1)[*, 1:bkr_nrec-2L]

            bk_nrec = n_elements((*l_event.background_O_e2))/l_setup.l_num_spinsec 
            extra_data[*,0] = (*l_event.background_O_e2)[*, 0]
            extra_data[*,1] = (*l_event.background_O_e2)[*, bk_nrec-1L]
            (*l_event.background_O_extra_e2) = extra_data
            (*l_event.background_O_e2) = (*l_event.background_O_e2)[*, 1:bk_nrec-2L]
            bkr_nrec = n_elements((*l_event.background_raw_O_e2))/l_setup.l_num_spinsec 
            (*l_event.background_raw_O_e2) = (*l_event.background_raw_O_e2)[*, 1:bkr_nrec-2L]
         endif

         O_nrec = n_elements((*l_event.oxygen_raw))/l_setup.l_num_spinsec 
         extra_data[*,0] = (*l_event.oxygen_raw)[*, 0]
         extra_data[*,1] = (*l_event.oxygen_raw)[*, O_nrec-1L]
         (*l_event.oxygen_raw_extra) = extra_data 
         (*l_event.oxygen_raw) = (*l_event.oxygen_raw)[*, 1:O_nrec-2L]

         O_nrec = n_elements((*l_event.oxygen_raw_e0))/l_setup.l_num_spinsec 
         extra_data[*,0] = (*l_event.oxygen_raw_e0)[*, 0]
         extra_data[*,1] = (*l_event.oxygen_raw_e0)[*, O_nrec-1L]
         (*l_event.oxygen_raw_extra_e0) = extra_data 
         (*l_event.oxygen_raw_e0) = (*l_event.oxygen_raw_e0)[*, 1:O_nrec-2L]

         O_nrec = n_elements((*l_event.oxygen_raw_e1))/l_setup.l_num_spinsec 
         extra_data[*,0] = (*l_event.oxygen_raw_e1)[*, 0]
         extra_data[*,1] = (*l_event.oxygen_raw_e1)[*, O_nrec-1L]
         (*l_event.oxygen_raw_extra_e1) = extra_data 
         (*l_event.oxygen_raw_e1) = (*l_event.oxygen_raw_e1)[*, 1:O_nrec-2L]

         O_nrec = n_elements((*l_event.oxygen_raw_e2))/l_setup.l_num_spinsec 
         extra_data[*,0] = (*l_event.oxygen_raw_e2)[*, 0]
         extra_data[*,1] = (*l_event.oxygen_raw_e2)[*, O_nrec-1L]
         (*l_event.oxygen_raw_extra_e2) = extra_data 
         (*l_event.oxygen_raw_e2) = (*l_event.oxygen_raw_e2)[*, 1:O_nrec-2L]

         O_nrec = n_elements((*l_event.oxygen_e0))/l_setup.l_num_spinsec 
         extra_data[*,0] = (*l_event.oxygen_e0)[*, 0]
         extra_data[*,1] = (*l_event.oxygen_e0)[*, O_nrec-1L]
         (*l_event.oxygen_e0_extra) = extra_data 
         (*l_event.oxygen_e0) = (*l_event.oxygen_e0)[*, 1:O_nrec-2L]

         O_nrec = n_elements((*l_event.oxygen_e1))/l_setup.l_num_spinsec 
         extra_data[*,0] = (*l_event.oxygen_e1)[*, 0]
         extra_data[*,1] = (*l_event.oxygen_e1)[*, O_nrec-1L]
         (*l_event.oxygen_e1_extra) = extra_data 
         (*l_event.oxygen_e1) = (*l_event.oxygen_e1)[*, 1:O_nrec-2L]

         O_nrec = n_elements((*l_event.oxygen_e2))/l_setup.l_num_spinsec 
         extra_data[*,0] = (*l_event.oxygen_e2)[*, 0]
         extra_data[*,1] = (*l_event.oxygen_e2)[*, O_nrec-1L]
         (*l_event.oxygen_e2_extra) = extra_data 
         (*l_event.oxygen_e2) = (*l_event.oxygen_e2)[*, 1:O_nrec-2L]

         O_nrec = n_elements((*l_event.O_sec_events))/l_setup.l_num_spinsec 
         extra_data[*,0] = (*l_event.O_sec_events)[*, 0]
         extra_data[*,1] = (*l_event.O_sec_events)[*, O_nrec-1L]
         (*l_event.O_sec_events_extra) = extra_data
         (*l_event.O_sec_events) = (*l_event.O_sec_events)[*, 1:O_nrec-2L]

         O_nrec = n_elements((*l_event.O_sec_events_e0))/l_setup.l_num_spinsec 
         extra_data[*,0] = (*l_event.O_sec_events_e0)[*, 0]
         extra_data[*,1] = (*l_event.O_sec_events_e0)[*, O_nrec-1L]
         (*l_event.O_sec_events_extra_e0) = extra_data
         (*l_event.O_sec_events_e0) = (*l_event.O_sec_events_e0)[*, 1:O_nrec-2L]

         O_nrec = n_elements((*l_event.O_sec_events_e1))/l_setup.l_num_spinsec 
         extra_data[*,0] = (*l_event.O_sec_events_e1)[*, 0]
         extra_data[*,1] = (*l_event.O_sec_events_e1)[*, O_nrec-1L]
         (*l_event.O_sec_events_extra_e1) = extra_data
         (*l_event.O_sec_events_e1) = (*l_event.O_sec_events_e1)[*, 1:O_nrec-2L]

         O_nrec = n_elements((*l_event.O_sec_events_e2))/l_setup.l_num_spinsec 
         extra_data[*,0] = (*l_event.O_sec_events_e2)[*, 0]
         extra_data[*,1] = (*l_event.O_sec_events_e2)[*, O_nrec-1L]
         (*l_event.O_sec_events_extra_e2) = extra_data
         (*l_event.O_sec_events_e2) = (*l_event.O_sec_events_e2)[*, 1:O_nrec-2L]

         O_nrec = n_elements((*l_event.O_n_in_peak))/l_setup.l_num_spinsec 
         extra_data[*,0] = (*l_event.O_n_in_peak)[*, 0]
         extra_data[*,1] = (*l_event.O_n_in_peak)[*, O_nrec-1L]
         (*l_event.O_n_in_peak_extra) = extra_data
         (*l_event.O_n_in_peak) = (*l_event.O_n_in_peak)[*, 1:O_nrec-2L]

         O_nrec = n_elements((*l_event.O_n_in_peak_e0))/l_setup.l_num_spinsec 
         extra_data[*,0] = (*l_event.O_n_in_peak_e0)[*, 0]
         extra_data[*,1] = (*l_event.O_n_in_peak_e0)[*, O_nrec-1L]
         (*l_event.O_n_in_peak_extra_e0) = extra_data
         (*l_event.O_n_in_peak_e0) = (*l_event.O_n_in_peak_e0)[*, 1:O_nrec-2L]

         O_nrec = n_elements((*l_event.O_n_in_peak_e1))/l_setup.l_num_spinsec 
         extra_data[*,0] = (*l_event.O_n_in_peak_e1)[*, 0]
         extra_data[*,1] = (*l_event.O_n_in_peak_e1)[*, O_nrec-1L]
         (*l_event.O_n_in_peak_extra_e1) = extra_data
         (*l_event.O_n_in_peak_e1) = (*l_event.O_n_in_peak_e1)[*, 1:O_nrec-2L]

         O_nrec = n_elements((*l_event.O_n_in_peak_e2))/l_setup.l_num_spinsec 
         extra_data[*,0] = (*l_event.O_n_in_peak_e2)[*, 0]
         extra_data[*,1] = (*l_event.O_n_in_peak_e2)[*, O_nrec-1L]
         (*l_event.O_n_in_peak_extra_e2) = extra_data
         (*l_event.O_n_in_peak_e2) = (*l_event.O_n_in_peak_e2)[*, 1:O_nrec-2L]

         O_nrec = n_elements((*l_event.O_scale_factor))/l_setup.l_num_spinsec 
         extra_data[*,0] = (*l_event.O_scale_factor)[*, 0]
         extra_data[*,1] = (*l_event.O_scale_factor)[*, O_nrec-1L]
         (*l_event.O_scale_factor_extra) = extra_data
         (*l_event.O_scale_factor) = (*l_event.O_scale_factor)[*, 1:O_nrec-2L]

         O_nrec = n_elements((*l_event.O_scale_factor_e0))/l_setup.l_num_spinsec 
         extra_data[*,0] = (*l_event.O_scale_factor_e0)[*, 0]
         extra_data[*,1] = (*l_event.O_scale_factor_e0)[*, O_nrec-1L]
         (*l_event.O_scale_factor_extra_e0) = extra_data
         (*l_event.O_scale_factor_e0) = (*l_event.O_scale_factor_e0)[*, 1:O_nrec-2L]

         O_nrec = n_elements((*l_event.O_scale_factor_e1))/l_setup.l_num_spinsec 
         extra_data[*,0] = (*l_event.O_scale_factor_e1)[*, 0]
         extra_data[*,1] = (*l_event.O_scale_factor_e1)[*, O_nrec-1L]
         (*l_event.O_scale_factor_extra_e1) = extra_data
         (*l_event.O_scale_factor_e1) = (*l_event.O_scale_factor_e1)[*, 1:O_nrec-2L]

         O_nrec = n_elements((*l_event.O_scale_factor_e2))/l_setup.l_num_spinsec 
         extra_data[*,0] = (*l_event.O_scale_factor_e2)[*, 0]
         extra_data[*,1] = (*l_event.O_scale_factor_e2)[*, O_nrec-1L]
         (*l_event.O_scale_factor_extra_e2) = extra_data
         (*l_event.O_scale_factor_e2) = (*l_event.O_scale_factor_e2)[*, 1:O_nrec-2L]

         cO_nrec = n_elements((*l_event.corrected_O))/l_setup.l_num_spinsec 
         extra_data[*,0] = (*l_event.corrected_O)[*, 0]
         extra_data[*,1] = (*l_event.corrected_O)[*, cO_nrec-1L]
         (*l_event.corrected_O_extra) = extra_data
         (*l_event.corrected_O) = (*l_event.corrected_O)[*, 1:cO_nrec-2L]

         cO_nrec = n_elements((*l_event.corrected_O_e0))/l_setup.l_num_spinsec 
         extra_data[*,0] = (*l_event.corrected_O_e0)[*, 0]
         extra_data[*,1] = (*l_event.corrected_O_e0)[*, cO_nrec-1L]
         (*l_event.corrected_O_extra_e0) = extra_data
         (*l_event.corrected_O_e0) = (*l_event.corrected_O_e0)[*, 1:cO_nrec-2L]

         cO_nrec = n_elements((*l_event.corrected_O_e1))/l_setup.l_num_spinsec 
         extra_data[*,0] = (*l_event.corrected_O_e1)[*, 0]
         extra_data[*,1] = (*l_event.corrected_O_e1)[*, cO_nrec-1L]
         (*l_event.corrected_O_extra_e1) = extra_data
         (*l_event.corrected_O_e1) = (*l_event.corrected_O_e1)[*, 1:cO_nrec-2L]

         cO_nrec = n_elements((*l_event.corrected_O_e2))/l_setup.l_num_spinsec 
         extra_data[*,0] = (*l_event.corrected_O_e2)[*, 0]
         extra_data[*,1] = (*l_event.corrected_O_e2)[*, cO_nrec-1L]
         (*l_event.corrected_O_extra_e2) = extra_data
         (*l_event.corrected_O_e2) = (*l_event.corrected_O_e2)[*, 1:cO_nrec-2L]

         cO_nrec = n_elements((*l_event.scaled_O))/l_setup.l_num_spinsec 
         (*l_event.scaled_O) = (*l_event.scaled_O)[*, 1:cO_nrec-2L]

         cO_nrec = n_elements((*l_event.scaled_O_e0))/l_setup.l_num_spinsec 
         (*l_event.scaled_O_e0) = (*l_event.scaled_O_e0)[*, 1:cO_nrec-2L]

         cO_nrec = n_elements((*l_event.scaled_O_e1))/l_setup.l_num_spinsec 
         (*l_event.scaled_O_e1) = (*l_event.scaled_O_e1)[*, 1:cO_nrec-2L]

         cO_nrec = n_elements((*l_event.scaled_O_e1))/l_setup.l_num_spinsec 
         (*l_event.scaled_O_e1) = (*l_event.scaled_O_e1)[*, 1:cO_nrec-2L]

      endif

   endif else begin
      if (l_event.addedStartFlag eq 1L) AND (l_event.addedStopFlag eq 0L) then begin

         (*l_event.startUdfTimePtr) = (*l_event.orig_startUdfTimePtr)[1:tme_nrec-1L]
         (*l_event.stopUdfTimePtr) = (*l_event.orig_stopUdfTimePtr)[1:tme_nrec-1L]

         e_nrec = n_elements((*l_event.spTimePtr))/l_setup.l_num_spinsec
         (*l_event.spTimePtr) = (*l_event.spTimePtr)[*, 1:e_nrec-1L]

         eu_nrec = n_elements((*l_event.filledStartUdfTimePtr))
         (*l_event.filledStartUdfTimePtr) = (*l_event.filledStartUdfTimePtr)[1:eu_nrec-1L]
         (*l_event.filledStopUdfTimePtr) = (*l_event.filledStopUdfTimePtr)[1:eu_nrec-1L]


         if (l_event_setup.hydrogen eq 'Y') then begin

            if (l_event_setup.background eq 'Y') then begin

               bk_nrec = n_elements((*l_event.background_H))/l_setup.l_num_spinsec 
               extra_data[*,0] = (*l_event.background_H)[*, 0]
               extra_data[*,1] = NO_DATA 
               (*l_event.background_H_extra) = extra_data
               (*l_event.background_H) = (*l_event.background_H)[*, 1:bk_nrec-1L]
               bkr_nrec = n_elements((*l_event.background_raw_H))/l_setup.l_num_spinsec 
               (*l_event.background_raw_H) = (*l_event.background_raw_H)[*, 1:bkr_nrec-1L]

               bk_nrec = n_elements((*l_event.background_H_e0))/l_setup.l_num_spinsec 
               extra_data[*,0] = (*l_event.background_H_e0)[*, 0]
               extra_data[*,1] = NO_DATA 
               (*l_event.background_H_extra_e0) = extra_data
               (*l_event.background_H_e0) = (*l_event.background_H_e0)[*, 1:bk_nrec-1L]
               bkr_nrec = n_elements((*l_event.background_raw_H_e0))/l_setup.l_num_spinsec 
               (*l_event.background_raw_H_e0) = (*l_event.background_raw_H_e0)[*, 1:bkr_nrec-1L]

               bk_nrec = n_elements((*l_event.background_H_e1))/l_setup.l_num_spinsec 
               extra_data[*,0] = (*l_event.background_H_e1)[*, 0]
               extra_data[*,1] = NO_DATA 
               (*l_event.background_H_extra_e1) = extra_data
               (*l_event.background_H_e1) = (*l_event.background_H_e1)[*, 1:bk_nrec-1L]
               bkr_nrec = n_elements((*l_event.background_raw_H_e1))/l_setup.l_num_spinsec 
               (*l_event.background_raw_H_e1) = (*l_event.background_raw_H_e1)[*, 1:bkr_nrec-1L]

               bk_nrec = n_elements((*l_event.background_H_e2))/l_setup.l_num_spinsec 
               extra_data[*,0] = (*l_event.background_H_e2)[*, 0]
               extra_data[*,1] = NO_DATA 
               (*l_event.background_H_extra_e2) = extra_data
               (*l_event.background_H_e2) = (*l_event.background_H_e2)[*, 1:bk_nrec-1L]
               bkr_nrec = n_elements((*l_event.background_raw_H_e2))/l_setup.l_num_spinsec 
               (*l_event.background_raw_H_e2) = (*l_event.background_raw_H_e2)[*, 1:bkr_nrec-1L]
            endif

            H_nrec = n_elements((*l_event.hydrogen_raw))/l_setup.l_num_spinsec 
            extra_data[*,0] = (*l_event.hydrogen_raw)[*, 0]
            extra_data[*,1] = NO_DATA 
            (*l_event.hydrogen_raw_extra) =  extra_data
            (*l_event.hydrogen_raw) = (*l_event.hydrogen_raw)[*, 1:H_nrec-1L]

            H_nrec = n_elements((*l_event.hydrogen_raw_e0))/l_setup.l_num_spinsec 
            extra_data[*,0] = (*l_event.hydrogen_raw_e0)[*, 0]
            extra_data[*,1] = NO_DATA 
            (*l_event.hydrogen_raw_extra_e0) =  extra_data
            (*l_event.hydrogen_raw_e0) = (*l_event.hydrogen_raw_e0)[*, 1:H_nrec-1L]

            H_nrec = n_elements((*l_event.hydrogen_raw_e1))/l_setup.l_num_spinsec 
            extra_data[*,0] = (*l_event.hydrogen_raw_e1)[*, 0]
            extra_data[*,1] = NO_DATA 
            (*l_event.hydrogen_raw_extra_e1) =  extra_data
            (*l_event.hydrogen_raw_e1) = (*l_event.hydrogen_raw_e1)[*, 1:H_nrec-1L]

            H_nrec = n_elements((*l_event.hydrogen_raw_e2))/l_setup.l_num_spinsec 
            extra_data[*,0] = (*l_event.hydrogen_raw_e2)[*, 0]
            extra_data[*,1] = NO_DATA 
            (*l_event.hydrogen_raw_extra_e2) =  extra_data
            (*l_event.hydrogen_raw_e2) = (*l_event.hydrogen_raw_e2)[*, 1:H_nrec-1L]

            H_nrec = n_elements((*l_event.hydrogen_e0))/l_setup.l_num_spinsec 
            extra_data[*,0] = (*l_event.hydrogen_e0)[*, 0]
            extra_data[*,1] = NO_DATA 
            (*l_event.hydrogen_e0_extra) = extra_data 
            (*l_event.hydrogen_e0) = (*l_event.hydrogen_e0)[*, 1:H_nrec-1L]

            H_nrec = n_elements((*l_event.hydrogen_e1))/l_setup.l_num_spinsec 
            extra_data[*,0] = (*l_event.hydrogen_e1)[*, 0]
            extra_data[*,1] =  NO_DATA
            (*l_event.hydrogen_e1_extra) = extra_data 
            (*l_event.hydrogen_e1) = (*l_event.hydrogen_e1)[*, 1:H_nrec-1L]
   
            H_nrec = n_elements((*l_event.hydrogen_e2))/l_setup.l_num_spinsec 
            extra_data[*,0] = (*l_event.hydrogen_e2)[*, 0]
            extra_data[*,1] = NO_DATA 
            (*l_event.hydrogen_e2_extra) = extra_data 
            (*l_event.hydrogen_e2) = (*l_event.hydrogen_e2)[*, 1:H_nrec-1L]

            H_nrec = n_elements((*l_event.H_sec_events))/l_setup.l_num_spinsec 
            extra_data[*,0] = (*l_event.H_sec_events)[*, 0]
            extra_data[*,1] = NO_DATA 
            (*l_event.H_sec_events_extra) =  extra_data
            (*l_event.H_sec_events) = (*l_event.H_sec_events)[*, 1:H_nrec-1L]

            H_nrec = n_elements((*l_event.H_sec_events_e0))/l_setup.l_num_spinsec 
            extra_data[*,0] = (*l_event.H_sec_events_e0)[*, 0]
            extra_data[*,1] = NO_DATA 
            (*l_event.H_sec_events_extra_e0) =  extra_data
            (*l_event.H_sec_events_e0) = (*l_event.H_sec_events_e0)[*, 1:H_nrec-1L]

            H_nrec = n_elements((*l_event.H_sec_events_e1))/l_setup.l_num_spinsec 
            extra_data[*,0] = (*l_event.H_sec_events_e1)[*, 0]
            extra_data[*,1] = NO_DATA 
            (*l_event.H_sec_events_extra_e1) =  extra_data
            (*l_event.H_sec_events_e1) = (*l_event.H_sec_events_e1)[*, 1:H_nrec-1L]

            H_nrec = n_elements((*l_event.H_sec_events_e2))/l_setup.l_num_spinsec 
            extra_data[*,0] = (*l_event.H_sec_events_e2)[*, 0]
            extra_data[*,1] = NO_DATA 
            (*l_event.H_sec_events_extra_e2) =  extra_data
            (*l_event.H_sec_events_e2) = (*l_event.H_sec_events_e2)[*, 1:H_nrec-1L]

            H_nrec = n_elements((*l_event.H_n_in_peak))/l_setup.l_num_spinsec 
            extra_data[*,0] = (*l_event.H_n_in_peak)[*, 0]
            extra_data[*,1] = NO_DATA 
            (*l_event.H_n_in_peak_extra) =  extra_data
            (*l_event.H_n_in_peak) = (*l_event.H_n_in_peak)[*, 1:H_nrec-1L]

            H_nrec = n_elements((*l_event.H_n_in_peak_e0))/l_setup.l_num_spinsec 
            extra_data[*,0] = (*l_event.H_n_in_peak_e0)[*, 0]
            extra_data[*,1] = NO_DATA 
            (*l_event.H_n_in_peak_extra_e0) =  extra_data
            (*l_event.H_n_in_peak_e0) = (*l_event.H_n_in_peak_e0)[*, 1:H_nrec-1L]

            H_nrec = n_elements((*l_event.H_n_in_peak_e1))/l_setup.l_num_spinsec 
            extra_data[*,0] = (*l_event.H_n_in_peak_e1)[*, 0]
            extra_data[*,1] = NO_DATA 
            (*l_event.H_n_in_peak_extra_e1) =  extra_data
            (*l_event.H_n_in_peak_e1) = (*l_event.H_n_in_peak_e1)[*, 1:H_nrec-1L]

            H_nrec = n_elements((*l_event.H_n_in_peak_e2))/l_setup.l_num_spinsec 
            extra_data[*,0] = (*l_event.H_n_in_peak_e2)[*, 0]
            extra_data[*,1] = NO_DATA 
            (*l_event.H_n_in_peak_extra_e2) =  extra_data
            (*l_event.H_n_in_peak_e2) = (*l_event.H_n_in_peak_e2)[*, 1:H_nrec-1L]

            H_nrec = n_elements((*l_event.H_scale_factor))/l_setup.l_num_spinsec 
            extra_data[*,0] = (*l_event.H_scale_factor)[*, 0]
            extra_data[*,1] = NO_DATA 
            (*l_event.H_scale_factor_extra) =  extra_data
            (*l_event.H_scale_factor) = (*l_event.H_scale_factor)[*, 1:H_nrec-1L]

            H_nrec = n_elements((*l_event.H_scale_factor_e0))/l_setup.l_num_spinsec 
            extra_data[*,0] = (*l_event.H_scale_factor_e0)[*, 0]
            extra_data[*,1] = NO_DATA 
            (*l_event.H_scale_factor_extra_e0) =  extra_data
            (*l_event.H_scale_factor_e0) = (*l_event.H_scale_factor_e0)[*, 1:H_nrec-1L]

            H_nrec = n_elements((*l_event.H_scale_factor_e1))/l_setup.l_num_spinsec 
            extra_data[*,0] = (*l_event.H_scale_factor_e1)[*, 0]
            extra_data[*,1] = NO_DATA 
            (*l_event.H_scale_factor_extra_e1) =  extra_data
            (*l_event.H_scale_factor_e1) = (*l_event.H_scale_factor_e1)[*, 1:H_nrec-1L]

            H_nrec = n_elements((*l_event.H_scale_factor_e2))/l_setup.l_num_spinsec 
            extra_data[*,0] = (*l_event.H_scale_factor_e2)[*, 0]
            extra_data[*,1] = NO_DATA 
            (*l_event.H_scale_factor_extra_e2) =  extra_data
            (*l_event.H_scale_factor_e2) = (*l_event.H_scale_factor_e2)[*, 1:H_nrec-1L]

            cH_nrec = n_elements((*l_event.corrected_H))/l_setup.l_num_spinsec 
            extra_data[*,0] = (*l_event.corrected_H)[*, 0]
            extra_data[*,1] = NO_DATA 
            (*l_event.corrected_H_extra) = extra_data
            (*l_event.corrected_H) = (*l_event.corrected_H)[*, 1:cH_nrec-1L]

            cH_nrec = n_elements((*l_event.corrected_H_e0))/l_setup.l_num_spinsec 
            extra_data[*,0] = (*l_event.corrected_H_e0)[*, 0]
            extra_data[*,1] = NO_DATA 
            (*l_event.corrected_H_extra_e0) = extra_data
            (*l_event.corrected_H_e0) = (*l_event.corrected_H_e0)[*, 1:cH_nrec-1L]

            cH_nrec = n_elements((*l_event.corrected_H_e1))/l_setup.l_num_spinsec 
            extra_data[*,0] = (*l_event.corrected_H_e1)[*, 0]
            extra_data[*,1] = NO_DATA 
            (*l_event.corrected_H_extra_e1) = extra_data
            (*l_event.corrected_H_e1) = (*l_event.corrected_H_e1)[*, 1:cH_nrec-1L]

            cH_nrec = n_elements((*l_event.corrected_H_e2))/l_setup.l_num_spinsec 
            extra_data[*,0] = (*l_event.corrected_H_e2)[*, 0]
            extra_data[*,1] = NO_DATA 
            (*l_event.corrected_H_extra_e2) = extra_data
            (*l_event.corrected_H_e2) = (*l_event.corrected_H_e2)[*, 1:cH_nrec-1L]

            cH_nrec = n_elements((*l_event.scaled_H))/l_setup.l_num_spinsec 
            (*l_event.scaled_H) = (*l_event.scaled_H)[*, 1:cH_nrec-1L]

            cH_nrec = n_elements((*l_event.scaled_H_e0))/l_setup.l_num_spinsec 
            (*l_event.scaled_H_e0) = (*l_event.scaled_H_e0)[*, 1:cH_nrec-1L]

            cH_nrec = n_elements((*l_event.scaled_H_e1))/l_setup.l_num_spinsec 
            (*l_event.scaled_H_e1) = (*l_event.scaled_H_e1)[*, 1:cH_nrec-1L]

            cH_nrec = n_elements((*l_event.scaled_H_e2))/l_setup.l_num_spinsec 
            (*l_event.scaled_H_e2) = (*l_event.scaled_H_e2)[*, 1:cH_nrec-1L]

         endif


         if (l_event_setup.oxygen eq 'Y') then begin
            if (l_event_setup.background eq 'Y') then begin

               bk_nrec = n_elements((*l_event.background_O))/l_setup.l_num_spinsec 
               extra_data[*,0] = (*l_event.background_O)[*, 0]
               extra_data[*,1] = NO_DATA 
               (*l_event.background_O_extra) = extra_data
               (*l_event.background_O) = (*l_event.background_O)[*, 1:bk_nrec-1L]
               bkr_nrec = n_elements((*l_event.background_raw_O))/l_setup.l_num_spinsec 
               (*l_event.background_raw_O) = (*l_event.background_raw_O)[*, 1:bkr_nrec-1L]

               bk_nrec = n_elements((*l_event.background_O_e0))/l_setup.l_num_spinsec 
               extra_data[*,0] = (*l_event.background_O_e0)[*, 0]
               extra_data[*,1] = NO_DATA 
               (*l_event.background_O_extra_e0) = extra_data
               (*l_event.background_O_e0) = (*l_event.background_O_e0)[*, 1:bk_nrec-1L]
               bkr_nrec = n_elements((*l_event.background_raw_O_e0))/l_setup.l_num_spinsec 
               (*l_event.background_raw_O_e0) = (*l_event.background_raw_O_e0)[*, 1:bkr_nrec-1L]

               bk_nrec = n_elements((*l_event.background_O_e1))/l_setup.l_num_spinsec 
               extra_data[*,0] = (*l_event.background_O_e1)[*, 0]
               extra_data[*,1] = NO_DATA 
               (*l_event.background_O_extra_e1) = extra_data
               (*l_event.background_O_e1) = (*l_event.background_O_e1)[*, 1:bk_nrec-1L]
               bkr_nrec = n_elements((*l_event.background_raw_O_e1))/l_setup.l_num_spinsec 
               (*l_event.background_raw_O_e1) = (*l_event.background_raw_O_e1)[*, 1:bkr_nrec-1L]

               bk_nrec = n_elements((*l_event.background_O_e2))/l_setup.l_num_spinsec 
               extra_data[*,0] = (*l_event.background_O_e2)[*, 0]
               extra_data[*,1] = NO_DATA 
               (*l_event.background_O_extra_e2) = extra_data
               (*l_event.background_O_e2) = (*l_event.background_O_e2)[*, 1:bk_nrec-1L]
               bkr_nrec = n_elements((*l_event.background_raw_O_e2))/l_setup.l_num_spinsec 
               (*l_event.background_raw_O_e2) = (*l_event.background_raw_O_e2)[*, 1:bkr_nrec-1L]
            endif

            O_nrec = n_elements((*l_event.oxygen_raw))/l_setup.l_num_spinsec 
            extra_data[*,0] = (*l_event.oxygen_raw)[*, 0]
            extra_data[*,1] = NO_DATA 
            (*l_event.oxygen_raw_extra) = extra_data
            (*l_event.oxygen_raw) = (*l_event.oxygen_raw)[*, 1:O_nrec-1L]

            O_nrec = n_elements((*l_event.oxygen_raw_e0))/l_setup.l_num_spinsec 
            extra_data[*,0] = (*l_event.oxygen_raw_e0)[*, 0]
            extra_data[*,1] = NO_DATA 
            (*l_event.oxygen_raw_extra_e0) = extra_data
            (*l_event.oxygen_raw_e0) = (*l_event.oxygen_raw_e0)[*, 1:O_nrec-1L]

            O_nrec = n_elements((*l_event.oxygen_raw_e1))/l_setup.l_num_spinsec 
            extra_data[*,0] = (*l_event.oxygen_raw_e1)[*, 0]
            extra_data[*,1] = NO_DATA 
            (*l_event.oxygen_raw_extra_e1) = extra_data
            (*l_event.oxygen_raw_e1) = (*l_event.oxygen_raw_e1)[*, 1:O_nrec-1L]

            O_nrec = n_elements((*l_event.oxygen_raw_e2))/l_setup.l_num_spinsec 
            extra_data[*,0] = (*l_event.oxygen_raw_e2)[*, 0]
            extra_data[*,1] = NO_DATA 
            (*l_event.oxygen_raw_extra_e2) = extra_data
            (*l_event.oxygen_raw_e2) = (*l_event.oxygen_raw_e2)[*, 1:O_nrec-1L]

            O_nrec = n_elements((*l_event.oxygen_e0))/l_setup.l_num_spinsec 
            extra_data[*,0] = (*l_event.oxygen_e0)[*, 0]
            extra_data[*,1] = NO_DATA 
            (*l_event.oxygen_e0_extra) = extra_data 
            (*l_event.oxygen_e0) = (*l_event.oxygen_e0)[*, 1:O_nrec-1L]

            O_nrec = n_elements((*l_event.oxygen_e1))/l_setup.l_num_spinsec 
            extra_data[*,0] = (*l_event.oxygen_e1)[*, 0]
            extra_data[*,1] = NO_DATA 
            (*l_event.oxygen_e1_extra) = extra_data 
            (*l_event.oxygen_e1) = (*l_event.oxygen_e1)[*, 1:O_nrec-1L]

            O_nrec = n_elements((*l_event.oxygen_e2))/l_setup.l_num_spinsec 
            extra_data[*,0] = (*l_event.oxygen_e2)[*, 0]
            extra_data[*,1] = NO_DATA 
            (*l_event.oxygen_e2_extra) = extra_data 
            (*l_event.oxygen_e2) = (*l_event.oxygen_e2)[*, 1:O_nrec-1L]

            O_nrec = n_elements((*l_event.O_sec_events))/l_setup.l_num_spinsec 
            extra_data[*,0] = (*l_event.O_sec_events)[*, 0]
            extra_data[*,1] = NO_DATA 
            (*l_event.O_sec_events_extra) = extra_data
            (*l_event.O_sec_events) = (*l_event.O_sec_events)[*, 1:O_nrec-1L]

            O_nrec = n_elements((*l_event.O_sec_events_e0))/l_setup.l_num_spinsec 
            extra_data[*,0] = (*l_event.O_sec_events_e0)[*, 0]
            extra_data[*,1] = NO_DATA 
            (*l_event.O_sec_events_extra_e0) = extra_data
            (*l_event.O_sec_events_e0) = (*l_event.O_sec_events_e0)[*, 1:O_nrec-1L]

            O_nrec = n_elements((*l_event.O_sec_events_e1))/l_setup.l_num_spinsec 
            extra_data[*,0] = (*l_event.O_sec_events_e1)[*, 0]
            extra_data[*,1] = NO_DATA 
            (*l_event.O_sec_events_extra_e1) = extra_data
            (*l_event.O_sec_events_e1) = (*l_event.O_sec_events_e1)[*, 1:O_nrec-1L]

            O_nrec = n_elements((*l_event.O_sec_events_e2))/l_setup.l_num_spinsec 
            extra_data[*,0] = (*l_event.O_sec_events_e2)[*, 0]
            extra_data[*,1] = NO_DATA 
            (*l_event.O_sec_events_extra_e2) = extra_data
            (*l_event.O_sec_events_e2) = (*l_event.O_sec_events_e2)[*, 1:O_nrec-1L]

            O_nrec = n_elements((*l_event.O_n_in_peak))/l_setup.l_num_spinsec 
            extra_data[*,0] = (*l_event.O_n_in_peak)[*, 0]
            extra_data[*,1] = NO_DATA 
            (*l_event.O_n_in_peak_extra) = extra_data
            (*l_event.O_n_in_peak) = (*l_event.O_n_in_peak)[*, 1:O_nrec-1L]

            O_nrec = n_elements((*l_event.O_n_in_peak_e0))/l_setup.l_num_spinsec 
            extra_data[*,0] = (*l_event.O_n_in_peak_e0)[*, 0]
            extra_data[*,1] = NO_DATA 
            (*l_event.O_n_in_peak_extra_e0) = extra_data
            (*l_event.O_n_in_peak_e0) = (*l_event.O_n_in_peak_e0)[*, 1:O_nrec-1L]

            O_nrec = n_elements((*l_event.O_n_in_peak_e1))/l_setup.l_num_spinsec 
            extra_data[*,0] = (*l_event.O_n_in_peak_e1)[*, 0]
            extra_data[*,1] = NO_DATA 
            (*l_event.O_n_in_peak_extra_e1) = extra_data
            (*l_event.O_n_in_peak_e1) = (*l_event.O_n_in_peak_e1)[*, 1:O_nrec-1L]

            O_nrec = n_elements((*l_event.O_n_in_peak_e2))/l_setup.l_num_spinsec 
            extra_data[*,0] = (*l_event.O_n_in_peak_e2)[*, 0]
            extra_data[*,1] = NO_DATA 
            (*l_event.O_n_in_peak_extra_e2) = extra_data
            (*l_event.O_n_in_peak_e2) = (*l_event.O_n_in_peak_e2)[*, 1:O_nrec-1L]

            O_nrec = n_elements((*l_event.O_scale_factor))/l_setup.l_num_spinsec 
            extra_data[*,0] = (*l_event.O_scale_factor)[*, 0]
            extra_data[*,1] = NO_DATA 
            (*l_event.O_scale_factor_extra) = extra_data
            (*l_event.O_scale_factor) = (*l_event.O_scale_factor)[*, 1:O_nrec-1L]

            O_nrec = n_elements((*l_event.O_scale_factor_e0))/l_setup.l_num_spinsec 
            extra_data[*,0] = (*l_event.O_scale_factor_e0)[*, 0]
            extra_data[*,1] = NO_DATA 
            (*l_event.O_scale_factor_extra_e0) = extra_data
            (*l_event.O_scale_factor_e0) = (*l_event.O_scale_factor_e0)[*, 1:O_nrec-1L]

            O_nrec = n_elements((*l_event.O_scale_factor_e1))/l_setup.l_num_spinsec 
            extra_data[*,0] = (*l_event.O_scale_factor_e1)[*, 0]
            extra_data[*,1] = NO_DATA 
            (*l_event.O_scale_factor_extra_e1) = extra_data
            (*l_event.O_scale_factor_e1) = (*l_event.O_scale_factor_e1)[*, 1:O_nrec-1L]

            O_nrec = n_elements((*l_event.O_scale_factor_e2))/l_setup.l_num_spinsec 
            extra_data[*,0] = (*l_event.O_scale_factor_e2)[*, 0]
            extra_data[*,1] = NO_DATA 
            (*l_event.O_scale_factor_extra_e2) = extra_data
            (*l_event.O_scale_factor_e2) = (*l_event.O_scale_factor_e2)[*, 1:O_nrec-1L]

            cO_nrec = n_elements((*l_event.corrected_O))/l_setup.l_num_spinsec 
            extra_data[*,0] = (*l_event.corrected_O)[*, 0]
            extra_data[*,1] = NO_DATA 
            (*l_event.corrected_O_extra) = extra_data
            (*l_event.corrected_O) = (*l_event.corrected_O)[*, 1:cO_nrec-1L]

            cO_nrec = n_elements((*l_event.corrected_O_e0))/l_setup.l_num_spinsec 
            extra_data[*,0] = (*l_event.corrected_O_e0)[*, 0]
            extra_data[*,1] = NO_DATA 
            (*l_event.corrected_O_extra_e0) = extra_data
            (*l_event.corrected_O_e0) = (*l_event.corrected_O_e0)[*, 1:cO_nrec-1L]

            cO_nrec = n_elements((*l_event.corrected_O_e1))/l_setup.l_num_spinsec 
            extra_data[*,0] = (*l_event.corrected_O_e1)[*, 0]
            extra_data[*,1] = NO_DATA 
            (*l_event.corrected_O_extra_e1) = extra_data
            (*l_event.corrected_O_e1) = (*l_event.corrected_O_e1)[*, 1:cO_nrec-1L]

            cO_nrec = n_elements((*l_event.corrected_O_e2))/l_setup.l_num_spinsec 
            extra_data[*,0] = (*l_event.corrected_O_e2)[*, 0]
            extra_data[*,1] = NO_DATA 
            (*l_event.corrected_O_extra_e2) = extra_data
            (*l_event.corrected_O_e2) = (*l_event.corrected_O_e2)[*, 1:cO_nrec-1L]

            cO_nrec = n_elements((*l_event.scaled_O))/l_setup.l_num_spinsec 
            (*l_event.scaled_O) = (*l_event.scaled_O)[*, 1:cO_nrec-1L]

            cO_nrec = n_elements((*l_event.scaled_O_e0))/l_setup.l_num_spinsec 
            (*l_event.scaled_O_e0) = (*l_event.scaled_O_e0)[*, 1:cO_nrec-1L]

            cO_nrec = n_elements((*l_event.scaled_O_e1))/l_setup.l_num_spinsec 
            (*l_event.scaled_O_e1) = (*l_event.scaled_O_e1)[*, 1:cO_nrec-1L]

            cO_nrec = n_elements((*l_event.scaled_O_e2))/l_setup.l_num_spinsec 
            (*l_event.scaled_O_e2) = (*l_event.scaled_O_e2)[*, 1:cO_nrec-1L]

         endif

      endif else begin

         if (l_event.addedStartFlag eq 0L) AND (l_event.addedStopFlag eq 1L) then begin

            (*l_event.startUdfTimePtr) = (*l_event.orig_startUdfTimePtr)[0:tme_nrec-2L]
            (*l_event.stopUdfTimePtr) = (*l_event.orig_stopUdfTimePtr)[0:tme_nrec-2L]

            e_nrec = n_elements((*l_event.spTimePtr))/l_setup.l_num_spinsec
            (*l_event.spTimePtr)= (*l_event.spTimePtr)[*, 0:e_nrec-2L]

            eu_nrec = n_elements((*l_event.filledStartUdfTimePtr))
            (*l_event.filledStartUdfTimePtr) = (*l_event.filledStartUdfTimePtr)[0:eu_nrec-2L]
            (*l_event.filledStopUdfTimePtr) = (*l_event.filledStopUdfTimePtr)[0:eu_nrec-2L]



            if (l_event_setup.hydrogen eq 'Y') then begin
               if (l_event_setup.background eq 'Y') then begin

                  bk_nrec = n_elements((*l_event.background_H))/l_setup.l_num_spinsec 
                  extra_data[*,0] = NO_DATA 
                  extra_data[*,1] = (*l_event.background_H)[*, bk_nrec-1L]
                  (*l_event.background_H_extra) = extra_data
                  (*l_event.background_H) = (*l_event.background_H)[*, 0:bk_nrec-2L]
                  bkr_nrec = n_elements((*l_event.background_raw_H))/l_setup.l_num_spinsec 
                  (*l_event.background_raw_H) = (*l_event.background_raw_H)[*, 0:bkr_nrec-2L]

                  bk_nrec = n_elements((*l_event.background_H_e0))/l_setup.l_num_spinsec 
                  extra_data[*,0] = NO_DATA 
                  extra_data[*,1] = (*l_event.background_H_e0)[*, bk_nrec-1L]
                  (*l_event.background_H_extra_e0) = extra_data
                  (*l_event.background_H_e0) = (*l_event.background_H_e0)[*, 0:bk_nrec-2L]
                  bkr_nrec = n_elements((*l_event.background_raw_H_e0))/l_setup.l_num_spinsec 
                  (*l_event.background_raw_H_e0) = (*l_event.background_raw_H_e0)[*, 0:bkr_nrec-2L]

                  bk_nrec = n_elements((*l_event.background_H_e1))/l_setup.l_num_spinsec 
                  extra_data[*,0] = NO_DATA 
                  extra_data[*,1] = (*l_event.background_H_e1)[*, bk_nrec-1L]
                  (*l_event.background_H_extra_e1) = extra_data
                  (*l_event.background_H_e1) = (*l_event.background_H_e1)[*, 0:bk_nrec-2L]
                  bkr_nrec = n_elements((*l_event.background_raw_H_e1))/l_setup.l_num_spinsec 
                  (*l_event.background_raw_H_e1) = (*l_event.background_raw_H_e1)[*, 0:bkr_nrec-2L]

                  bk_nrec = n_elements((*l_event.background_H_e2))/l_setup.l_num_spinsec 
                  extra_data[*,0] = NO_DATA 
                  extra_data[*,1] = (*l_event.background_H_e2)[*, bk_nrec-1L]
                  (*l_event.background_H_extra_e2) = extra_data
                  (*l_event.background_H_e2) = (*l_event.background_H_e2)[*, 0:bk_nrec-2L]
                  bkr_nrec = n_elements((*l_event.background_raw_H_e2))/l_setup.l_num_spinsec 
                  (*l_event.background_raw_H_e2) = (*l_event.background_raw_H_e2)[*, 0:bkr_nrec-2L]
               endif

               H_nrec = n_elements((*l_event.hydrogen_raw))/l_setup.l_num_spinsec 
               extra_data[*,0] = NO_DATA 
               extra_data[*,1] = (*l_event.hydrogen_raw)[*, H_nrec-1L]
               (*l_event.hydrogen_raw_extra) =  extra_data
               (*l_event.hydrogen_raw) = (*l_event.hydrogen_raw)[*, 0:H_nrec-2L]

               H_nrec = n_elements((*l_event.hydrogen_raw_e0))/l_setup.l_num_spinsec 
               extra_data[*,0] = NO_DATA 
               extra_data[*,1] = (*l_event.hydrogen_raw_e0)[*, H_nrec-1L]
               (*l_event.hydrogen_raw_extra_e0) =  extra_data
               (*l_event.hydrogen_raw_e0) = (*l_event.hydrogen_raw_e0)[*, 0:H_nrec-2L]

               H_nrec = n_elements((*l_event.hydrogen_raw_e1))/l_setup.l_num_spinsec 
               extra_data[*,0] = NO_DATA 
               extra_data[*,1] = (*l_event.hydrogen_raw_e1)[*, H_nrec-1L]
               (*l_event.hydrogen_raw_extra_e1) =  extra_data
               (*l_event.hydrogen_raw_e1) = (*l_event.hydrogen_raw_e1)[*, 0:H_nrec-2L]

               H_nrec = n_elements((*l_event.hydrogen_raw_e2))/l_setup.l_num_spinsec 
               extra_data[*,0] = NO_DATA 
               extra_data[*,1] = (*l_event.hydrogen_raw_e2)[*, H_nrec-1L]
               (*l_event.hydrogen_raw_extra_e2) =  extra_data
               (*l_event.hydrogen_raw_e2) = (*l_event.hydrogen_raw_e2)[*, 0:H_nrec-2L]

               H_nrec = n_elements((*l_event.hydrogen_e0))/l_setup.l_num_spinsec 
               extra_data[*,0] = NO_DATA 
               extra_data[*,1] = (*l_event.hydrogen_e0)[*, H_nrec-1L]
               (*l_event.hydrogen_e0_extra) = extra_data 
               (*l_event.hydrogen_e0) = (*l_event.hydrogen_e0)[*, 0:H_nrec-2L]

               H_nrec = n_elements((*l_event.hydrogen_e1))/l_setup.l_num_spinsec 
               extra_data[*,0] = NO_DATA 
               extra_data[*,1] = (*l_event.hydrogen_e1)[*, H_nrec-1L]
               (*l_event.hydrogen_e1_extra) = extra_data 
               (*l_event.hydrogen_e1) = (*l_event.hydrogen_e1)[*, 0:H_nrec-2L]

               H_nrec = n_elements((*l_event.hydrogen_e2))/l_setup.l_num_spinsec 
               extra_data[*,0] = NO_DATA 
               extra_data[*,1] = (*l_event.hydrogen_e2)[*, H_nrec-1L]
               (*l_event.hydrogen_e2_extra) = extra_data 
               (*l_event.hydrogen_e2) = (*l_event.hydrogen_e2)[*, 0:H_nrec-2L]

               H_nrec = n_elements((*l_event.H_sec_events))/l_setup.l_num_spinsec 
               extra_data[*,0] = NO_DATA 
               extra_data[*,1] = (*l_event.H_sec_events)[*, H_nrec-1L]
               (*l_event.H_sec_events_extra) =  extra_data
               (*l_event.H_sec_events) = (*l_event.H_sec_events)[*, 0:H_nrec-2L]

               H_nrec = n_elements((*l_event.H_sec_events_e0))/l_setup.l_num_spinsec 
               extra_data[*,0] = NO_DATA 
               extra_data[*,1] = (*l_event.H_sec_events_e0)[*, H_nrec-1L]
               (*l_event.H_sec_events_extra_e0) =  extra_data
               (*l_event.H_sec_events_e0) = (*l_event.H_sec_events_e0)[*, 0:H_nrec-2L]

               H_nrec = n_elements((*l_event.H_sec_events_e1))/l_setup.l_num_spinsec 
               extra_data[*,0] = NO_DATA 
               extra_data[*,1] = (*l_event.H_sec_events_e1)[*, H_nrec-1L]
               (*l_event.H_sec_events_extra_e1) =  extra_data
               (*l_event.H_sec_events_e1) = (*l_event.H_sec_events_e1)[*, 0:H_nrec-2L]

               H_nrec = n_elements((*l_event.H_sec_events_e2))/l_setup.l_num_spinsec 
               extra_data[*,0] = NO_DATA 
               extra_data[*,1] = (*l_event.H_sec_events_e2)[*, H_nrec-1L]
               (*l_event.H_sec_events_extra_e2) =  extra_data
               (*l_event.H_sec_events_e2) = (*l_event.H_sec_events_e2)[*, 0:H_nrec-2L]

               H_nrec = n_elements((*l_event.H_n_in_peak))/l_setup.l_num_spinsec 
               extra_data[*,0] = NO_DATA 
               extra_data[*,1] = (*l_event.H_n_in_peak)[*, H_nrec-1L]
               (*l_event.H_n_in_peak_extra) =  extra_data
               (*l_event.H_n_in_peak) = (*l_event.H_n_in_peak)[*, 0:H_nrec-2L]

               H_nrec = n_elements((*l_event.H_n_in_peak_e0))/l_setup.l_num_spinsec 
               extra_data[*,0] = NO_DATA 
               extra_data[*,1] = (*l_event.H_n_in_peak_e0)[*, H_nrec-1L]
               (*l_event.H_n_in_peak_extra_e0) =  extra_data
               (*l_event.H_n_in_peak_e0) = (*l_event.H_n_in_peak_e0)[*, 0:H_nrec-2L]

               H_nrec = n_elements((*l_event.H_n_in_peak_e1))/l_setup.l_num_spinsec 
               extra_data[*,0] = NO_DATA 
               extra_data[*,1] = (*l_event.H_n_in_peak_e1)[*, H_nrec-1L]
               (*l_event.H_n_in_peak_extra_e1) =  extra_data
               (*l_event.H_n_in_peak_e1) = (*l_event.H_n_in_peak_e1)[*, 0:H_nrec-2L]

               H_nrec = n_elements((*l_event.H_n_in_peak_e2))/l_setup.l_num_spinsec 
               extra_data[*,0] = NO_DATA 
               extra_data[*,1] = (*l_event.H_n_in_peak_e2)[*, H_nrec-1L]
               (*l_event.H_n_in_peak_extra_e2) =  extra_data
               (*l_event.H_n_in_peak_e2) = (*l_event.H_n_in_peak_e2)[*, 0:H_nrec-2L]

               H_nrec = n_elements((*l_event.H_scale_factor))/l_setup.l_num_spinsec 
               extra_data[*,0] = NO_DATA 
               extra_data[*,1] = (*l_event.H_scale_factor)[*, H_nrec-1L]
               (*l_event.H_scale_factor_extra) =  extra_data
               (*l_event.H_scale_factor) = (*l_event.H_scale_factor)[*, 0:H_nrec-2L]

               H_nrec = n_elements((*l_event.H_scale_factor_e0))/l_setup.l_num_spinsec 
               extra_data[*,0] = NO_DATA 
               extra_data[*,1] = (*l_event.H_scale_factor_e0)[*, H_nrec-1L]
               (*l_event.H_scale_factor_extra_e0) =  extra_data
               (*l_event.H_scale_factor_e0) = (*l_event.H_scale_factor_e0)[*, 0:H_nrec-2L]

               H_nrec = n_elements((*l_event.H_scale_factor_e1))/l_setup.l_num_spinsec 
               extra_data[*,0] = NO_DATA 
               extra_data[*,1] = (*l_event.H_scale_factor_e1)[*, H_nrec-1L]
               (*l_event.H_scale_factor_extra_e1) =  extra_data
               (*l_event.H_scale_factor_e1) = (*l_event.H_scale_factor_e1)[*, 0:H_nrec-2L]

               H_nrec = n_elements((*l_event.H_scale_factor_e2))/l_setup.l_num_spinsec 
               extra_data[*,0] = NO_DATA 
               extra_data[*,1] = (*l_event.H_scale_factor_e2)[*, H_nrec-1L]
               (*l_event.H_scale_factor_extra_e2) =  extra_data
               (*l_event.H_scale_factor_e2) = (*l_event.H_scale_factor_e2)[*, 0:H_nrec-2L]

               cH_nrec = n_elements((*l_event.corrected_H))/l_setup.l_num_spinsec 
               extra_data[*,0] = NO_DATA 
               extra_data[*,1] = (*l_event.corrected_H)[*, cH_nrec-1L]
               (*l_event.corrected_H_extra) = extra_data
               (*l_event.corrected_H) = (*l_event.corrected_H)[*, 0:cH_nrec-2L]

               cH_nrec = n_elements((*l_event.corrected_H_e0))/l_setup.l_num_spinsec 
               extra_data[*,0] = NO_DATA 
               extra_data[*,1] = (*l_event.corrected_H_e0)[*, cH_nrec-1L]
               (*l_event.corrected_H_extra_e0) = extra_data
               (*l_event.corrected_H_e0) = (*l_event.corrected_H_e0)[*, 0:cH_nrec-2L]

               cH_nrec = n_elements((*l_event.corrected_H_e1))/l_setup.l_num_spinsec 
               extra_data[*,0] = NO_DATA 
               extra_data[*,1] = (*l_event.corrected_H_e1)[*, cH_nrec-1L]
               (*l_event.corrected_H_extra_e1) = extra_data
               (*l_event.corrected_H_e1) = (*l_event.corrected_H_e1)[*, 0:cH_nrec-2L]

               cH_nrec = n_elements((*l_event.corrected_H_e2))/l_setup.l_num_spinsec 
               extra_data[*,0] = NO_DATA 
               extra_data[*,1] = (*l_event.corrected_H_e2)[*, cH_nrec-1L]
               (*l_event.corrected_H_extra_e2) = extra_data
               (*l_event.corrected_H_e2) = (*l_event.corrected_H_e2)[*, 0:cH_nrec-2L]

               cH_nrec = n_elements((*l_event.scaled_H))/l_setup.l_num_spinsec 
               (*l_event.scaled_H) = (*l_event.scaled_H)[*, 0:cH_nrec-2L]

               cH_nrec = n_elements((*l_event.scaled_H_e0))/l_setup.l_num_spinsec 
               (*l_event.scaled_H_e0) = (*l_event.scaled_H_e0)[*, 0:cH_nrec-2L]

               cH_nrec = n_elements((*l_event.scaled_H_e1))/l_setup.l_num_spinsec 
               (*l_event.scaled_H_e1) = (*l_event.scaled_H_e1)[*, 0:cH_nrec-2L]

               cH_nrec = n_elements((*l_event.scaled_H_e2))/l_setup.l_num_spinsec 
               (*l_event.scaled_H_e2) = (*l_event.scaled_H_e2)[*, 0:cH_nrec-2L]
   
            endif

            if (l_event_setup.oxygen eq 'Y') then begin
               if (l_event_setup.background eq 'Y') then begin
                  bk_nrec = n_elements((*l_event.background_O))/l_setup.l_num_spinsec 
                  extra_data[*,0] = NO_DATA 
                  extra_data[*,1] = (*l_event.background_O)[*, bk_nrec-1L]
                  (*l_event.background_O_extra) =  extra_data
                  (*l_event.background_O) = (*l_event.background_O)[*, 0:bk_nrec-2L]
                  bkr_nrec = n_elements((*l_event.background_raw_O))/l_setup.l_num_spinsec 
                  (*l_event.background_raw_O) = (*l_event.background_raw_O)[*, 0:bkr_nrec-2L]

                  bk_nrec = n_elements((*l_event.background_O_e0))/l_setup.l_num_spinsec 
                  extra_data[*,0] = NO_DATA 
                  extra_data[*,1] = (*l_event.background_O_e0)[*, bk_nrec-1L]
                  (*l_event.background_O_extra_e0) =  extra_data
                  (*l_event.background_O_e0) = (*l_event.background_O_e0)[*, 0:bk_nrec-2L]
                  bkr_nrec = n_elements((*l_event.background_raw_O_e0))/l_setup.l_num_spinsec 
                  (*l_event.background_raw_O_e0) = (*l_event.background_raw_O_e0)[*, 0:bkr_nrec-2L]

                  bk_nrec = n_elements((*l_event.background_O_e1))/l_setup.l_num_spinsec 
                  extra_data[*,0] = NO_DATA 
                  extra_data[*,1] = (*l_event.background_O_e1)[*, bk_nrec-1L]
                  (*l_event.background_O_extra_e1) =  extra_data
                  (*l_event.background_O_e1) = (*l_event.background_O_e1)[*, 0:bk_nrec-2L]
                  bkr_nrec = n_elements((*l_event.background_raw_O_e1))/l_setup.l_num_spinsec 
                  (*l_event.background_raw_O_e1) = (*l_event.background_raw_O_e1)[*, 0:bkr_nrec-2L]

                  bk_nrec = n_elements((*l_event.background_O_e2))/l_setup.l_num_spinsec 
                  extra_data[*,0] = NO_DATA 
                  extra_data[*,1] = (*l_event.background_O_e2)[*, bk_nrec-1L]
                  (*l_event.background_O_extra_e2) =  extra_data
                  (*l_event.background_O_e2) = (*l_event.background_O_e2)[*, 0:bk_nrec-2L]
                  bkr_nrec = n_elements((*l_event.background_raw_O_e2))/l_setup.l_num_spinsec 
                  (*l_event.background_raw_O_e2) = (*l_event.background_raw_O_e2)[*, 0:bkr_nrec-2L]
               endif

               O_nrec = n_elements((*l_event.oxygen_raw))/l_setup.l_num_spinsec 
               extra_data[*,0] = NO_DATA 
               extra_data[*,1] = (*l_event.oxygen_raw)[*, O_nrec-1L]
               (*l_event.oxygen_raw_extra) = extra_data
               (*l_event.oxygen_raw) = (*l_event.oxygen_raw)[*, 0:O_nrec-2L]

               O_nrec = n_elements((*l_event.oxygen_raw_e0))/l_setup.l_num_spinsec 
               extra_data[*,0] = NO_DATA 
               extra_data[*,1] = (*l_event.oxygen_raw_e0)[*, O_nrec-1L]
               (*l_event.oxygen_raw_extra_e0) = extra_data
               (*l_event.oxygen_raw_e0) = (*l_event.oxygen_raw_e0)[*, 0:O_nrec-2L]

               O_nrec = n_elements((*l_event.oxygen_raw_e1))/l_setup.l_num_spinsec 
               extra_data[*,0] = NO_DATA 
               extra_data[*,1] = (*l_event.oxygen_raw_e1)[*, O_nrec-1L]
               (*l_event.oxygen_raw_extra_e1) = extra_data
               (*l_event.oxygen_raw_e1) = (*l_event.oxygen_raw_e1)[*, 0:O_nrec-2L]

               O_nrec = n_elements((*l_event.oxygen_raw_e2))/l_setup.l_num_spinsec 
               extra_data[*,0] = NO_DATA 
               extra_data[*,1] = (*l_event.oxygen_raw_e2)[*, O_nrec-1L]
               (*l_event.oxygen_raw_extra_e2) = extra_data
               (*l_event.oxygen_raw_e2) = (*l_event.oxygen_raw_e2)[*, 0:O_nrec-2L]

               O_nrec = n_elements((*l_event.oxygen_e0))/l_setup.l_num_spinsec 
               extra_data[*,0] =  NO_DATA
               extra_data[*,1] = (*l_event.oxygen_e0)[*, O_nrec-1L]
               (*l_event.oxygen_e0_extra) = extra_data 
               (*l_event.oxygen_e0) = (*l_event.oxygen_e0)[*, 0:O_nrec-2L]

               O_nrec = n_elements((*l_event.oxygen_e1))/l_setup.l_num_spinsec 
               extra_data[*,0] = NO_DATA 
               extra_data[*,1] = (*l_event.oxygen_e1)[*, O_nrec-1L]
               (*l_event.oxygen_e1_extra) = extra_data 
               (*l_event.oxygen_e1) = (*l_event.oxygen_e1)[*, 0:O_nrec-2L]

               O_nrec = n_elements((*l_event.oxygen_e2))/l_setup.l_num_spinsec 
               extra_data[*,0] =  NO_DATA
               extra_data[*,1] = (*l_event.oxygen_e2)[*, O_nrec-1L]
               (*l_event.oxygen_e2_extra) = extra_data 
               (*l_event.oxygen_e2) = (*l_event.oxygen_e2)[*, 0:O_nrec-2L]

               O_nrec = n_elements((*l_event.O_sec_events))/l_setup.l_num_spinsec 
               extra_data[*,0] = NO_DATA 
               extra_data[*,1] = (*l_event.O_sec_events)[*, O_nrec-1L]
               (*l_event.O_sec_events_extra) = extra_data
               (*l_event.O_sec_events) = (*l_event.O_sec_events)[*, 0:O_nrec-2L]

               O_nrec = n_elements((*l_event.O_sec_events_e0))/l_setup.l_num_spinsec 
               extra_data[*,0] = NO_DATA 
               extra_data[*,1] = (*l_event.O_sec_events_e0)[*, O_nrec-1L]
               (*l_event.O_sec_events_extra_e0) = extra_data
               (*l_event.O_sec_events_e0) = (*l_event.O_sec_events_e0)[*, 0:O_nrec-2L]

               O_nrec = n_elements((*l_event.O_sec_events_e1))/l_setup.l_num_spinsec 
               extra_data[*,0] = NO_DATA 
               extra_data[*,1] = (*l_event.O_sec_events_e1)[*, O_nrec-1L]
               (*l_event.O_sec_events_extra_e1) = extra_data
               (*l_event.O_sec_events_e1) = (*l_event.O_sec_events_e1)[*, 0:O_nrec-2L]

               O_nrec = n_elements((*l_event.O_sec_events_e2))/l_setup.l_num_spinsec 
               extra_data[*,0] = NO_DATA 
               extra_data[*,1] = (*l_event.O_sec_events_e2)[*, O_nrec-1L]
               (*l_event.O_sec_events_extra_e2) = extra_data
               (*l_event.O_sec_events_e2) = (*l_event.O_sec_events_e2)[*, 0:O_nrec-2L]

               O_nrec = n_elements((*l_event.O_n_in_peak))/l_setup.l_num_spinsec 
               extra_data[*,0] = NO_DATA 
               extra_data[*,1] = (*l_event.O_n_in_peak)[*, O_nrec-1L]
               (*l_event.O_n_in_peak_extra) = extra_data
               (*l_event.O_n_in_peak) = (*l_event.O_n_in_peak)[*, 0:O_nrec-2L]

               O_nrec = n_elements((*l_event.O_n_in_peak_e0))/l_setup.l_num_spinsec 
               extra_data[*,0] = NO_DATA 
               extra_data[*,1] = (*l_event.O_n_in_peak_e0)[*, O_nrec-1L]
               (*l_event.O_n_in_peak_extra_e0) = extra_data
               (*l_event.O_n_in_peak_e0) = (*l_event.O_n_in_peak_e0)[*, 0:O_nrec-2L]

               O_nrec = n_elements((*l_event.O_n_in_peak_e1))/l_setup.l_num_spinsec 
               extra_data[*,0] = NO_DATA 
               extra_data[*,1] = (*l_event.O_n_in_peak_e1)[*, O_nrec-1L]
               (*l_event.O_n_in_peak_extra_e1) = extra_data
               (*l_event.O_n_in_peak_e1) = (*l_event.O_n_in_peak_e1)[*, 0:O_nrec-2L]

               O_nrec = n_elements((*l_event.O_n_in_peak_e2))/l_setup.l_num_spinsec 
               extra_data[*,0] = NO_DATA 
               extra_data[*,1] = (*l_event.O_n_in_peak_e2)[*, O_nrec-1L]
               (*l_event.O_n_in_peak_extra_e2) = extra_data
               (*l_event.O_n_in_peak_e2) = (*l_event.O_n_in_peak_e2)[*, 0:O_nrec-2L]

               O_nrec = n_elements((*l_event.O_scale_factor))/l_setup.l_num_spinsec 
               extra_data[*,0] = NO_DATA 
               extra_data[*,1] = (*l_event.O_scale_factor)[*, O_nrec-1L]
               (*l_event.O_scale_factor_extra) = extra_data
               (*l_event.O_scale_factor) = (*l_event.O_scale_factor)[*, 0:O_nrec-2L]

               O_nrec = n_elements((*l_event.O_scale_factor_e0))/l_setup.l_num_spinsec 
               extra_data[*,0] = NO_DATA 
               extra_data[*,1] = (*l_event.O_scale_factor_e0)[*, O_nrec-1L]
               (*l_event.O_scale_factor_extra_e0) = extra_data
               (*l_event.O_scale_factor_e0) = (*l_event.O_scale_factor_e0)[*, 0:O_nrec-2L]

               O_nrec = n_elements((*l_event.O_scale_factor_e1))/l_setup.l_num_spinsec 
               extra_data[*,0] = NO_DATA 
               extra_data[*,1] = (*l_event.O_scale_factor_e1)[*, O_nrec-1L]
               (*l_event.O_scale_factor_extra_e1) = extra_data
               (*l_event.O_scale_factor_e1) = (*l_event.O_scale_factor_e1)[*, 0:O_nrec-2L]

               O_nrec = n_elements((*l_event.O_scale_factor_e2))/l_setup.l_num_spinsec 
               extra_data[*,0] = NO_DATA 
               extra_data[*,1] = (*l_event.O_scale_factor_e2)[*, O_nrec-1L]
               (*l_event.O_scale_factor_extra_e2) = extra_data
               (*l_event.O_scale_factor_e2) = (*l_event.O_scale_factor_e2)[*, 0:O_nrec-2L]

               cO_nrec = n_elements((*l_event.corrected_O))/l_setup.l_num_spinsec 
               extra_data[*,0] = NO_DATA 
               extra_data[*,1] = (*l_event.corrected_O)[*, cO_nrec-1L]
               (*l_event.corrected_O_extra) = extra_data
               (*l_event.corrected_O) = (*l_event.corrected_O)[*, 0:cO_nrec-2L]

               cO_nrec = n_elements((*l_event.corrected_O_e0))/l_setup.l_num_spinsec 
               extra_data[*,0] = NO_DATA 
               extra_data[*,1] = (*l_event.corrected_O_e0)[*, cO_nrec-1L]
               (*l_event.corrected_O_extra_e0) = extra_data
               (*l_event.corrected_O_e0) = (*l_event.corrected_O_e0)[*, 0:cO_nrec-2L]

               cO_nrec = n_elements((*l_event.corrected_O_e1))/l_setup.l_num_spinsec 
               extra_data[*,0] = NO_DATA 
               extra_data[*,1] = (*l_event.corrected_O_e1)[*, cO_nrec-1L]
               (*l_event.corrected_O_extra_e1) = extra_data
               (*l_event.corrected_O_e1) = (*l_event.corrected_O_e1)[*, 0:cO_nrec-2L]

               cO_nrec = n_elements((*l_event.corrected_O_e2))/l_setup.l_num_spinsec 
               extra_data[*,0] = NO_DATA 
               extra_data[*,1] = (*l_event.corrected_O_e2)[*, cO_nrec-1L]
               (*l_event.corrected_O_extra_e2) = extra_data
               (*l_event.corrected_O_e2) = (*l_event.corrected_O_e2)[*, 0:cO_nrec-2L]

               cO_nrec = n_elements((*l_event.scaled_O))/l_setup.l_num_spinsec 
               (*l_event.scaled_O) = (*l_event.scaled_O)[*, 0:cO_nrec-2L]

               cO_nrec = n_elements((*l_event.scaled_O_e0))/l_setup.l_num_spinsec 
               (*l_event.scaled_O_e0) = (*l_event.scaled_O_e0)[*, 0:cO_nrec-2L]

               cO_nrec = n_elements((*l_event.scaled_O_e1))/l_setup.l_num_spinsec 
               (*l_event.scaled_O_e1) = (*l_event.scaled_O_e1)[*, 0:cO_nrec-2L]

               cO_nrec = n_elements((*l_event.scaled_O_e2))/l_setup.l_num_spinsec 
               (*l_event.scaled_O_e2) = (*l_event.scaled_O_e2)[*, 0:cO_nrec-2L]

            endif
         endif
      endelse
   endelse 


RETURN
END

;-------------------------------------------------------------------------------
;  Procedure : l_Event_CorrBkgrnd
;
;  Description:
;    Apply scaling parameter to correct for the ratio of total binned
;  coincidences to the total event coincidences. Extract hydrogen, oxygen,
;  and background data from the event data based on the TOF ranges.  Apply
;  the scaling factor.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_event_setup         struct        I        general event data setup parameters
;  l_event               struct       I/O       event data pointers
;  l_setup               struct        I        general setup parameters
;  event_cnts            fltarr        I        event data sectored event counts
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;     PLOTDEFS         MAX_PLOTS        int         G         max # plots per page
;                      MAX_WEB_SPINS    int         G         max # spins per plot
;                                                             when using web interface
;                      NO_CNTS          double      G         no counts indicator
;                      NO_DATA          double      G         no data indicator
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                08/01          v1.0.7             Original implementation
;-------------------------------------------------------------------------------


PRO l_Event_CorrBkgrnd, l_event_setup, l_event, l_setup, event_cnts
COMMON PLOTDEFS, MAX_PLOTS, MAX_WEB_SPINS, NO_CNTS, NO_DATA


   ;----------------------------------------------------------------
   ; initialize data structures
   ;----------------------------------------------------------------

   nele                  = n_elements ((*l_event.tofPtr)[*])
   time_nele             = n_elements((*l_event.orig_startUdfTimePtr)[*])
   estart_udf_time       = (*l_event.orig_startUdfTimePtr)[0]
   estop_udf_time        = (*l_event.orig_startUdfTimePtr)[time_nele-1L] 
   start_event_time      = Convert (estart_udf_time)
   stop_event_time       = Convert (estop_udf_time)

   est_nspins            = round((stop_event_time - start_event_time) / 120.) + 10L

   spin      = 1L
   curr_sect = 0L
   prev_sect = 0L

   event_time            = dblarr (l_setup.l_num_spinsec, est_nspins)
   event_start_udf_time  = replicate ({UDF_TIMES_STRUCT}, est_nspins)
   event_stop_udf_time   = replicate ({UDF_TIMES_STRUCT}, est_nspins)

   prev_event_time = (*l_event.timePtr)[0]

   background               = fltarr (l_setup.l_num_spinsec, est_nspins)
   background_e0            = fltarr (l_setup.l_num_spinsec, est_nspins)
   background_e1            = fltarr (l_setup.l_num_spinsec, est_nspins)
   background_e2            = fltarr (l_setup.l_num_spinsec, est_nspins)
   background_nevents       = fltarr (l_setup.l_num_spinsec, est_nspins)
   background_nevents_e0    = fltarr (l_setup.l_num_spinsec, est_nspins)
   background_nevents_e1    = fltarr (l_setup.l_num_spinsec, est_nspins)
   background_nevents_e2    = fltarr (l_setup.l_num_spinsec, est_nspins)
   background_sec_events    = fltarr (l_setup.l_num_spinsec, est_nspins)
   background_sec_events_e0 = fltarr (l_setup.l_num_spinsec, est_nspins)
   background_sec_events_e1 = fltarr (l_setup.l_num_spinsec, est_nspins)
   background_sec_events_e2 = fltarr (l_setup.l_num_spinsec, est_nspins)

   hydrogen              = fltarr (l_setup.l_num_spinsec, est_nspins)
   hydrogen_e0           = fltarr (l_setup.l_num_spinsec, est_nspins)
   hydrogen_e1           = fltarr (l_setup.l_num_spinsec, est_nspins)
   hydrogen_e2           = fltarr (l_setup.l_num_spinsec, est_nspins)
   hydrogen_nevents      = fltarr (l_setup.l_num_spinsec, est_nspins)
   hydrogen_nevents_e0   = fltarr (l_setup.l_num_spinsec, est_nspins)
   hydrogen_nevents_e1   = fltarr (l_setup.l_num_spinsec, est_nspins)
   hydrogen_nevents_e2   = fltarr (l_setup.l_num_spinsec, est_nspins)
   hydrogen_sec_events   = fltarr (l_setup.l_num_spinsec, est_nspins)
   hydrogen_sec_events_e0= fltarr (l_setup.l_num_spinsec, est_nspins)
   hydrogen_sec_events_e1= fltarr (l_setup.l_num_spinsec, est_nspins)
   hydrogen_sec_events_e2= fltarr (l_setup.l_num_spinsec, est_nspins)
   H_no_data_flag        = intarr (l_setup.l_num_spinsec, est_nspins)

   oxygen                = fltarr (l_setup.l_num_spinsec, est_nspins)
   oxygen_e0             = fltarr (l_setup.l_num_spinsec, est_nspins)
   oxygen_e1             = fltarr (l_setup.l_num_spinsec, est_nspins)
   oxygen_e2             = fltarr (l_setup.l_num_spinsec, est_nspins)
   oxygen_nevents        = fltarr (l_setup.l_num_spinsec, est_nspins)
   oxygen_nevents_e0     = fltarr (l_setup.l_num_spinsec, est_nspins)
   oxygen_nevents_e1     = fltarr (l_setup.l_num_spinsec, est_nspins)
   oxygen_nevents_e2     = fltarr (l_setup.l_num_spinsec, est_nspins)
   oxygen_sec_events     = fltarr (l_setup.l_num_spinsec, est_nspins)
   oxygen_sec_events_e0  = fltarr (l_setup.l_num_spinsec, est_nspins)
   oxygen_sec_events_e1  = fltarr (l_setup.l_num_spinsec, est_nspins)
   oxygen_sec_events_e2  = fltarr (l_setup.l_num_spinsec, est_nspins)
   O_no_data_flag        = intarr (l_setup.l_num_spinsec, est_nspins)
      
   event_start_udf_time[0] = (*l_event.orig_startUdfTimePtr)[0]
   event_stop_udf_time[0] = (*l_event.orig_stopUdfTimePtr)[0]
   event_time[*,0] = (*l_event.timePtr)[0]


   for i = 0L, nele - 1L do begin

      start_event_time = (*l_event.timePtr)[i]
      curr_sect = (*l_event.azimuthPtr)[i]

      ;-------------------------------------------------------
      ; determine the begin and end of a spin
      ;-------------------------------------------------------
      if (curr_sect lt prev_sect) AND (start_event_time ge prev_event_time) then begin
         event_start_udf_time[spin] = (*l_event.orig_startUdfTimePtr)[i]
         event_stop_udf_time[spin] = (*l_event.orig_stopUdfTimePtr)[i]
         event_time[*,spin] = (*l_event.timePtr)[i]
         spin = spin + 1L
      endif

      prev_sect = curr_sect
      prev_event_time = start_event_time

      ;-----------------------------------------------------
      ; determine the number of background occurrences 
      ;-----------------------------------------------------
      if (((*l_event.tofPtr)[i] ge l_event_setup.bkgrnd_beg_chan) AND $
          ((*l_event.tofPtr)[i] le l_event_setup.bkgrnd_end_chan)) then begin

          background [curr_sect, spin] = background[curr_sect, spin] + 1
          background_nevents[curr_sect, spin] = event_cnts[i]
          background_sec_events[curr_sect, spin] = (*l_event.sectCntsPtr)[i]

          ;-----------------------------------------------------
          ; energy 0
          ;-----------------------------------------------------
          if ((*l_event.energyPtr)[i] eq 0) then begin
             background_e0 [curr_sect, spin] = background_e0[curr_sect, spin] + 1
             background_nevents_e0[curr_sect, spin] = event_cnts[i]

             ;-----------------------------------------------------
             ; counter that saturates at 64
             ;-----------------------------------------------------
             background_sec_events_e0[curr_sect, spin] = (*l_event.sectCntsPtr)[i]

          endif else begin

             if ((*l_event.energyPtr)[i] eq 1) then begin
                background_e1 [curr_sect, spin] = background_e1[curr_sect, spin] + 1
                background_nevents_e1[curr_sect, spin] = event_cnts[i]

          
                ;-----------------------------------------------------
                ; counter that saturates at 64
                ;-----------------------------------------------------
                background_sec_events_e1[curr_sect, spin] = (*l_event.sectCntsPtr)[i]

             endif else begin

                 if ((*l_event.energyPtr)[i] eq 2) then begin
                    background_e2 [curr_sect, spin] = background_e2[curr_sect, spin] + 1
                    background_nevents_e2[curr_sect, spin] = event_cnts[i]
          
                    ;-----------------------------------------------------
                    ; counter that saturates at 64
                    ;-----------------------------------------------------
                    background_sec_events_e2[curr_sect, spin] = (*l_event.sectCntsPtr)[i]
                endif
             endelse
          endelse
      endif

      ;-----------------------------------------------------
      ; determine number of hydrogen occurrences per sector
      ; in each spin 
      ; hydrogen counts
      ; hydrogen_sec_events = saturates at 64
      ; hydrogen_nevents = > 64 events
      ;-----------------------------------------------------

      if (((*l_event.tofPtr)[i] ge l_event_setup.hydrogen_beg_chan) AND $
          ((*l_event.tofPtr)[i] le l_event_setup.hydrogen_end_chan)) then begin

          hydrogen [curr_sect, spin] = hydrogen[curr_sect, spin] + 1
          hydrogen_nevents [curr_sect, spin] = event_cnts[i]
          hydrogen_sec_events [curr_sect, spin] = (*l_event.sectCntsPtr)[i]

          if (*l_event.energyPtr)[i] eq 0 then begin
             hydrogen_e0 [curr_sect, spin] = hydrogen_e0[curr_sect, spin] + 1
             hydrogen_nevents_e0 [curr_sect, spin] = event_cnts[i]
             hydrogen_sec_events_e0 [curr_sect, spin] = (*l_event.sectCntsPtr)[i]
          endif else begin

             if (*l_event.energyPtr)[i] eq 1 then begin
                hydrogen_e1 [curr_sect, spin] = hydrogen_e1[curr_sect, spin] + 1
                hydrogen_nevents_e1 [curr_sect, spin] = event_cnts[i]
                hydrogen_sec_events_e1 [curr_sect, spin] = (*l_event.sectCntsPtr)[i]
             endif else begin

                if (*l_event.energyPtr)[i] eq 2 then begin
                   hydrogen_e2 [curr_sect, spin] = hydrogen_e2[curr_sect, spin] + 1
                   hydrogen_nevents_e2 [curr_sect, spin] = event_cnts[i]
                   hydrogen_sec_events_e2 [curr_sect, spin] = (*l_event.sectCntsPtr)[i]
                endif
             endelse
          endelse

      endif

      ;-----------------------------------------------------
      ; determine number of oxygen occurrences per sector
      ; in each spin 
      ;-----------------------------------------------------
      if (((*l_event.tofPtr)[i] ge l_event_setup.oxygen_beg_chan) AND $
          ((*l_event.tofPtr)[i] le l_event_setup.oxygen_end_chan)) then begin

          oxygen [curr_sect, spin] = oxygen [curr_sect, spin] + 1
          oxygen_nevents[curr_sect, spin] = event_cnts[i]
          oxygen_sec_events [curr_sect, spin] = (*l_event.sectCntsPtr)[i]

          if (*l_event.energyPtr)[i] eq 0 then begin
             oxygen_e0 [curr_sect, spin] = oxygen_e0 [curr_sect, spin] + 1
             oxygen_nevents_e0[curr_sect, spin] = event_cnts[i]
             oxygen_sec_events_e0 [curr_sect, spin] = (*l_event.sectCntsPtr)[i]
          endif else begin

             if (*l_event.energyPtr)[i] eq 1 then begin
                oxygen_e1 [curr_sect, spin] = oxygen_e1 [curr_sect, spin] + 1
                oxygen_nevents_e1[curr_sect, spin] = event_cnts[i]
                oxygen_sec_events_e1 [curr_sect, spin] = (*l_event.sectCntsPtr)[i]
             endif else begin

                if (*l_event.energyPtr)[i] eq 2 then begin
                   oxygen_e2 [curr_sect, spin] = oxygen_e2 [curr_sect, spin] + 1
                   oxygen_nevents_e2[curr_sect, spin] = event_cnts[i]
                   oxygen_sec_events_e2 [curr_sect, spin] = (*l_event.sectCntsPtr)[i]
                endif
             endelse
           endelse

     endif 
   
   endfor


   ptr = ptr_new (event_time[*,0:spin-1], /NO_COPY)
   l_event.spTimePtr = ptr 

   if (l_event_setup.background eq 'Y') then begin
      ;-----------------------------------------------------
      ; scale the background counts
      ; n_background * total_numevents_gt64/ numevents_le64
      ;-----------------------------------------------------
      scaled_background    = fltarr(l_setup.l_num_spinsec, spin)
      scaled_background_e0 = fltarr(l_setup.l_num_spinsec, spin)
      scaled_background_e1 = fltarr(l_setup.l_num_spinsec, spin)
      scaled_background_e2 = fltarr(l_setup.l_num_spinsec, spin)
      nsectors = l_setup.l_num_spinsec

      for i = 0L, spin - 1L do begin
         neg_9_flag = 0L
         for j = 0L,  nsectors - 1L do begin

            ;--------------------------------------------------
            ; if the total number of events (> 64) is not a
            ; -9 (no data) then do the background scaling
            ;--------------------------------------------------
            if (background_nevents[j,i] ne NO_DATA) then begin
               scaled_background[j,i]  = background[j,i] * $
                         background_nevents[j,i]/(background_sec_events[j,i] > 1.0)
            endif else begin
               ;--------------------------------------------------
               ; if the total number of events in the spin is less 
               ; than 64, then assume scale factor is one, otherwise
               ; get rid of the spin
               ;--------------------------------------------------
               number_of_events = total (background_sec_events) 
               if (number_of_events lt 64) then begin
                  scaled_background[*,i]  = 1.0 
               endif else begin
                  scaled_background[*,i]  = 0.0 
                  background[*,i] = 0.0 
                  background_nevents[*,i] = 0.0 
                  background_sec_events[*,i] = 0.0 
                  neg_9_flag = 1L
               endelse
            endelse

            if (background_nevents_e0[j,i] ne NO_DATA) then begin
               scaled_background_e0[j,i]  = background_e0[j,i] * $
                         background_nevents_e0[j,i]/(background_sec_events_e0[j,i] > 1.0)
            endif else begin
               number_of_events = total (background_sec_events_e0)
               if (number_of_events lt 64) then begin
                  scaled_background_e0[*,i]  = 1.0 
               endif else begin
                  scaled_background_e0[*,i]  = 0.0 
                  background_e0[*,i] = 0.0 
                  background_nevents_e0[*,i] = 0.0 
                  background_sec_events_e0[*,i] = 0.0 
               endelse
            endelse


            if (background_nevents_e1[j,i] ne NO_DATA) then begin
               scaled_background_e1[j,i]  = background_e1[j,i] * $
                         background_nevents_e1[j,i]/(background_sec_events_e1[j,i] > 1.0)
            endif else begin
               number_of_events = total (background_sec_events_e1)
               if (number_of_events lt 64) then begin
                  scaled_background_e1[*,i]  = 1.0 
               endif else begin
                  scaled_background_e1[*,i]  = 0.0 
                  background_e1[*,i] = 0.0 
                  background_nevents_e1[*,i] = 0.0 
                  background_sec_events_e1[*,i] = 0.0 
               endelse
            endelse

            if (background_nevents_e2[j,i] ne NO_DATA) then begin
               scaled_background_e2[j,i]  = background_e2[j,i] * $
                         background_nevents_e2[j,i]/(background_sec_events_e2[j,i] > 1.0)
            endif else begin
               number_of_events = total (background_sec_events_e2)
               if (number_of_events lt 64) then begin
                  scaled_background_e2[*,i]  = 1.0 
               endif else begin
                  scaled_background_e2[*,i]  = 0.0 
                  background_e2[*,i] = 0.0 
                  background_nevents_e2[*,i] = 0.0 
                  background_sec_events_e2[*,i] = 0.0 
               endelse
            endelse

         endfor
         if (neg_9_flag) then l_event.num_bk_elim_spins = l_event.num_bk_elim_spins + 1L
      endfor

      ptr = ptr_new (background[*,0:spin-1L])
      l_event.background_raw_H = ptr

      ptr = ptr_new (background_e0[*,0:spin-1L])
      l_event.background_raw_H_e0 = ptr

      ptr = ptr_new (background_e1[*,0:spin-1L])
      l_event.background_raw_H_e1 = ptr

      ptr = ptr_new (background_e2[*,0:spin-1L])
      l_event.background_raw_H_e2 = ptr

      ptr = ptr_new (background[*,0:spin-1L], /NO_COPY)
      l_event.background_raw_O = ptr

      ptr = ptr_new (background_e0[*,0:spin-1L], /NO_COPY)
      l_event.background_raw_O_e0 = ptr

      ptr = ptr_new (background_e1[*,0:spin-1L], /NO_COPY)
      l_event.background_raw_O_e1 = ptr

      ptr = ptr_new (background_e2[*,0:spin-1L], /NO_COPY)
      l_event.background_raw_O_e2 = ptr

      ptr = ptr_new (scaled_background_e0[*,0:spin-1L], /NO_COPY)
      l_event.background_e0 = ptr

      ptr = ptr_new (scaled_background_e1[*,0:spin-1L], /NO_COPY)
      l_event.background_e1 = ptr

      ptr = ptr_new (scaled_background_e2[*,0:spin-1L], /NO_COPY)
      l_event.background_e2 = ptr

   endif


   if (l_event_setup.hydrogen eq 'Y') then begin 

      ;-----------------------------------------------------
      ; take the total number of events per spin sector and
      ; divide by the sector event count (possible number of
      ; events per sector)
      ;-----------------------------------------------------
      scaled_hydrogen    = fltarr (l_setup.l_num_spinsec, spin)
      scaled_hydrogen_e0 = fltarr (l_setup.l_num_spinsec, spin)
      scaled_hydrogen_e1 = fltarr (l_setup.l_num_spinsec, spin)
      scaled_hydrogen_e2 = fltarr (l_setup.l_num_spinsec, spin)
      scaled_H           = fltarr (l_setup.l_num_spinsec, spin)
      scaled_H_e0        = fltarr (l_setup.l_num_spinsec, spin)
      scaled_H_e1        = fltarr (l_setup.l_num_spinsec, spin)
      scaled_H_e2        = fltarr (l_setup.l_num_spinsec, spin)


      if (l_event_setup.background eq 'N') then begin
         scaled_background = make_array (l_setup.l_num_spinsec, spin, /FLOAT, value = 0.0)
         background = make_array (l_setup.l_num_spinsec, spin, /FLOAT, value = 0.0)
         ptr = ptr_new (background, /NO_COPY)
         l_event.background_e0 = ptr
         l_event.background_e1 = ptr
         l_event.background_e2 = ptr
         l_event.background_raw_H = ptr
         l_event.background_raw_H_e0 = ptr
         l_event.background_raw_H_e1 = ptr
         l_event.background_raw_H_e2 = ptr
      endif

      ;-----------------------------------------------------
      ; scale hydrogen data 
      ; n_hydrogens * total_events_gt64/events_le64
      ;-----------------------------------------------------
      H_background      = scaled_background * l_event_setup.H_scale_factor 
      corrected_H       = fltarr (l_setup.l_num_spinsec, spin)
      nsectors          = l_setup.l_num_spinsec
      H_elim_spins_time  = dblarr (720)
      H_elim_spins_nevents = fltarr (720)

      for i = 0L, spin - 1L do begin
         for j = 0L, nsectors - 1L do begin

             current_time = event_time[j,i]

             if (hydrogen_nevents[j,i] ne NO_DATA) then begin 
                scaled_hydrogen[j,i] = hydrogen[j,i] * $
                      hydrogen_nevents[j,i]/ (hydrogen_sec_events[j,i] > 1.0) 
             endif else begin
                number_of_events = total (hydrogen_sec_events[*,i])
                if (number_of_events lt 64) then begin
                   scaled_hydrogen[*,i] = 1.0
                endif else begin
                   scaled_hydrogen[*,i] = 0.0 
                   hydrogen[*,i] = 0.0
                   hydrogen_nevents[*,i] = 0.0 
                   hydrogen_sec_events[*,i] = 0.0 
                   start_udf_tme = ConvArrTimeToSecs(l_setup.l_start_dtime)
                   stop_udf_tme = ConvArrTimeToSecs(l_setup.l_stop_dtime)

                   if (current_time ge start_udf_tme) AND (current_time le stop_udf_tme) then begin
                      H_elim_spins_time[l_event.num_H_elim_spins] = current_time 
                      H_elim_spins_nevents[l_event.num_H_elim_spins] = number_of_events
                      l_event.num_H_elim_spins = l_event.num_H_elim_spins + 1L
                   endif
                endelse
             endelse


             if (hydrogen_nevents_e0[j,i] ne NO_DATA) then begin 
                scaled_hydrogen_e0[j,i] = hydrogen_e0[j,i] * $
                      hydrogen_nevents_e0[j,i]/ (hydrogen_sec_events_e0[j,i] > 1.0) 
             endif else begin
                number_of_events = total (hydrogen_sec_events_e0[*,i])
                if (number_of_events lt 64) then begin
                   scaled_hydrogen_e0[*,i] = 1.0
                endif else begin
                   scaled_hydrogen_e0[*,i] = 0.0 
                   hydrogen_e0[*,i] = 0.0 
                   hydrogen_nevents_e0[*,i] = 0.0 
                   hydrogen_sec_events_e0[*,i] = 0.0 
                endelse
             endelse

             if (hydrogen_nevents_e1[j,i] ne NO_DATA) then begin 
                scaled_hydrogen_e1[j,i] = hydrogen_e1[j,i] * $
                      hydrogen_nevents_e1[j,i]/ (hydrogen_sec_events_e1[j,i] > 1.0) 
             endif else begin
                number_of_events = total (hydrogen_sec_events_e1[*,i])
                if (number_of_events lt 64) then begin
                   scaled_hydrogen_e1[*,i] = 1.0
                endif else begin
                   scaled_hydrogen_e1[*,i] = 0.0 
                   hydrogen_e1[*,i] = 0.0 
                   hydrogen_nevents_e1[*,i] = 0.0 
                   hydrogen_sec_events_e1[*,i] = 0.0 
                endelse
             endelse

             if (hydrogen_nevents_e2[j,i] ne NO_DATA) then begin 
                scaled_hydrogen_e2[j,i] = hydrogen_e2[j,i] * $
                      hydrogen_nevents_e2[j,i]/ (hydrogen_sec_events_e2[j,i] > 1.0) 
             endif else begin
                number_of_events = total (hydrogen_sec_events_e2[*,i])
                if (number_of_events lt 64) then begin
                   scaled_hydrogen_e2[*,i] = 1.0
                endif else begin
                   scaled_hydrogen_e2[*,i] = 0.0 
                   hydrogen_e2[*,i] = 0.0 
                   hydrogen_nevents_e2[*,i] = 0.0 
                   hydrogen_sec_events_e2[*,i] = 0.0 
                endelse
             endelse
         endfor
      endfor

      ptr = ptr_new (H_elim_spins_time[0:l_event.num_H_elim_spins], /NO_COPY)
      l_event.H_elim_spins_time = ptr
      ptr = ptr_new (H_elim_spins_nevents[0:l_event.num_H_elim_spins], /NO_COPY)
      l_event.H_elim_spins_nevents = ptr


      zero_indx = where (hydrogen_sec_events eq 0.0, zcnt)
      nzero_indx = where (hydrogen_sec_events ne 0.0, nzcnt)
      if (nzcnt gt 0) then begin
         scaled_H[nzero_indx] = hydrogen_nevents[nzero_indx]/hydrogen_sec_events[nzero_indx]
         scaled_H[zero_indx] = 0.0
      endif else begin
          if (zcnt ge n_elements (hydrogen_sec_events)) then begin
            scaled_H = hydrogen_nevents
            scaled_H[*,*] = 0.0
          endif else begin
            scaled_H = hydrogen_nevents/hydrogen_sec_events
          endelse
      endelse

      zero_indx = where (hydrogen_sec_events_e0 eq 0.0, zcnt)
      nzero_indx = where (hydrogen_sec_events_e0 ne 0.0, nzcnt)
      if (nzcnt gt 0) then begin
         scaled_H_e0[nzero_indx] = hydrogen_nevents_e0[nzero_indx]/hydrogen_sec_events_e0[nzero_indx]
         scaled_H_e0[zero_indx] = 0.0
      endif else begin
         if (zcnt ge n_elements (hydrogen_sec_events_e0)) then begin
            scaled_H_e0 = hydrogen_nevents_e0
            scaled_H_e0[*,*] = 0.0
         endif else begin 
            scaled_H_e0 = hydrogen_nevents_e0/hydrogen_sec_events_e0
         endelse
      endelse

      zero_indx = where (hydrogen_sec_events_e1 eq 0.0, zcnt)
      nzero_indx = where (hydrogen_sec_events_e1 ne 0.0, nzcnt)
      if (nzcnt gt 0) then begin
         scaled_H_e1[nzero_indx] = hydrogen_nevents_e1[nzero_indx]/hydrogen_sec_events_e1[nzero_indx]
         scaled_H_e1[zero_indx] = 0.0
      endif else begin
         if (zcnt ge n_elements (hydrogen_sec_events_e1)) then begin
            scaled_H_e1 = hydrogen_nevents_e1
            scaled_H_e1[*,*] = 0.0
         endif else begin
            scaled_H_e1 = hydrogen_nevents_e1/hydrogen_sec_events_e1
         endelse
      endelse

      zero_indx = where (hydrogen_sec_events_e2 eq 0.0, zcnt)
      nzero_indx = where (hydrogen_sec_events_e2 ne 0.0, nzcnt)
      if (nzcnt gt 0) then begin
         scaled_H_e2[nzero_indx] = hydrogen_nevents_e2[nzero_indx]/hydrogen_sec_events_e2[nzero_indx]
         scaled_H_e2[zero_indx] = 0.0
      endif else begin
         if (zcnt ge n_elements (hydrogen_sec_events_e2)) then begin
            scaled_H_e2 = hydrogen_nevents_e2
            scaled_H_e2[*,*] = 0.0
         endif else begin
            scaled_H_e2 = hydrogen_nevents_e2/hydrogen_sec_events_e2
         endelse
      endelse

      ;-----------------------------------------------------
      ; do background subtraction 
      ;-----------------------------------------------------

      n9_eq_idx = where (scaled_hydrogen eq NO_DATA, n9_eq_cnt)
      n9_ne_idx = where (scaled_hydrogen ne NO_DATA, n9_ne_cnt)
      if (n9_eq_cnt gt 0) then begin
         corrected_H[n9_ne_idx]  = scaled_hydrogen[n9_ne_idx] - H_background[n9_ne_idx] 
      endif else begin
         corrected_H  = scaled_hydrogen - H_background 
      endelse

      neg_values = where (corrected_H lt 0.0, ncnt)
      corrected_H = (corrected_H > 0.0)

      h_indx = where (hydrogen_nevents eq NO_DATA, h_cnt)
      if (h_cnt gt 0) then begin
          corrected_H[h_indx] = NO_DATA 
      endif

      ;-----------------------------------------------------
      ; separate into different energy bins
      ;-----------------------------------------------------
      corrected_hydrogen_e0 = fltarr (l_setup.l_num_spinsec, spin)
      corrected_hydrogen_e0 = scaled_hydrogen_e0 - ((*l_event.background_e0) * 0.4)
      corrected_hydrogen_e0 = corrected_hydrogen_e0 > 0.0
      h_indx = where (hydrogen_nevents_e0 eq NO_DATA, h_cnt)
      if (h_cnt gt 0) then begin
          corrected_hydrogen_e0[h_indx] = NO_DATA 
          scaled_hydrogen_e0[h_indx] = NO_DATA
      endif


      corrected_hydrogen_e1 = fltarr (l_setup.l_num_spinsec, spin)
      corrected_hydrogen_e1 = scaled_hydrogen_e1 - ((*l_event.background_e1) * 0.4)
      corrected_hydrogen_e1 = corrected_hydrogen_e1 > 0.0
      h_indx = where (hydrogen_nevents_e1 eq NO_DATA, h_cnt)
      if (h_cnt gt 0) then begin
          corrected_hydrogen_e1[h_indx] = NO_DATA 
          scaled_hydrogen_e1[h_indx] = NO_DATA
      endif


      corrected_hydrogen_e2 = fltarr (l_setup.l_num_spinsec, spin)
      corrected_hydrogen_e2 = scaled_hydrogen_e2 - ((*l_event.background_e2) * 0.4)
      corrected_hydrogen_e2 = corrected_hydrogen_e2 > 0.0
      h_indx = where (hydrogen_nevents_e2 eq NO_DATA, h_cnt)
      if (h_cnt gt 0) then begin
          corrected_hydrogen_e2[h_indx] = NO_DATA 
          scaled_hydrogen_e2[h_indx] = NO_DATA
      endif


      ptr = ptr_new (hydrogen_e0)
      l_event.hydrogen_e0 = ptr

      ptr = ptr_new (hydrogen_e1)
      l_event.hydrogen_e1 = ptr

      ptr = ptr_new (hydrogen_e2)
      l_event.hydrogen_e2 = ptr


      ptr = ptr_new (event_start_udf_time[0:spin-1], /NO_COPY)
      l_event.filledStartUdfTimePtr = ptr

      ptr = ptr_new (event_stop_udf_time[0:spin-1], /NO_COPY)
      l_event.filledStopUdfTimePtr = ptr

      ptr = ptr_new (hydrogen_nevents[*,0:spin-1], /NO_COPY)
      l_event.hydrogen_raw = ptr

      ptr = ptr_new (hydrogen_nevents_e0[*,0:spin-1], /NO_COPY)
      l_event.hydrogen_raw_e0 = ptr

      ptr = ptr_new (hydrogen_nevents_e1[*,0:spin-1], /NO_COPY)
      l_event.hydrogen_raw_e1 = ptr

      ptr = ptr_new (hydrogen_nevents_e2[*,0:spin-1], /NO_COPY)
      l_event.hydrogen_raw_e2 = ptr

      tmp =  (*l_event.background_raw_H) * l_event_setup.H_scale_factor
      ptr = ptr_new (tmp, /NO_COPY)
      l_event.background_H = ptr

      tmp =  (*l_event.background_raw_H_e0) * l_event_setup.H_scale_factor
      ptr = ptr_new (tmp, /NO_COPY)
      l_event.background_H_e0 = ptr

      tmp =  (*l_event.background_raw_H_e1) * l_event_setup.H_scale_factor
      ptr = ptr_new (tmp, /NO_COPY)
      l_event.background_H_e1 = ptr

      tmp =  (*l_event.background_raw_H_e2) * l_event_setup.H_scale_factor
      ptr = ptr_new (tmp, /NO_COPY)
      l_event.background_H_e2 = ptr

      ptr = ptr_new (corrected_H, /NO_COPY)
      l_event.corrected_H = ptr

      ptr = ptr_new (corrected_hydrogen_e0, /NO_COPY)
      l_event.corrected_H_e0 = ptr

      ptr = ptr_new (corrected_hydrogen_e1, /NO_COPY)
      l_event.corrected_H_e1 = ptr

      ptr = ptr_new (corrected_hydrogen_e2, /NO_COPY)
      l_event.corrected_H_e2 = ptr

      ptr = ptr_new (hydrogen_sec_events[*,0:spin-1], /NO_COPY)
      l_event.H_sec_events = ptr

      ptr = ptr_new (hydrogen_sec_events_e0[*,0:spin-1], /NO_COPY)
      l_event.H_sec_events_e0 = ptr

      ptr = ptr_new (hydrogen_sec_events_e1[*,0:spin-1], /NO_COPY)
      l_event.H_sec_events_e1 = ptr

      ptr = ptr_new (hydrogen_sec_events_e2[*,0:spin-1], /NO_COPY)
      l_event.H_sec_events_e2 = ptr

      ptr = ptr_new (hydrogen[*,0:spin-1], /NO_COPY)
      l_event.H_n_in_peak = ptr

      ptr = ptr_new (hydrogen_e0[*,0:spin-1], /NO_COPY)
      l_event.H_n_in_peak_e0 = ptr

      ptr = ptr_new (hydrogen_e1[*,0:spin-1], /NO_COPY)
      l_event.H_n_in_peak_e1 = ptr

      ptr = ptr_new (hydrogen_e2[*,0:spin-1], /NO_COPY)
      l_event.H_n_in_peak_e2 = ptr

      ptr = ptr_new (scaled_H[*,0:spin-1], /NO_COPY)
      l_event.H_scale_factor = ptr

      ptr = ptr_new (scaled_H_e0[*,0:spin-1], /NO_COPY)
      l_event.H_scale_factor_e0 = ptr

      ptr = ptr_new (scaled_H_e1[*,0:spin-1], /NO_COPY)
      l_event.H_scale_factor_e1 = ptr

      ptr = ptr_new (scaled_H_e2[*,0:spin-1], /NO_COPY)
      l_event.H_scale_factor_e2 = ptr

      ptr = ptr_new (scaled_hydrogen, /NO_COPY)
      l_event.scaled_H = ptr

      ptr = ptr_new (scaled_hydrogen_e0, /NO_COPY)
      l_event.scaled_H_e0 = ptr

      ptr = ptr_new (scaled_hydrogen_e1, /NO_COPY)
      l_event.scaled_H_e1 = ptr

      ptr = ptr_new (scaled_hydrogen_e2, /NO_COPY)
      l_event.scaled_H_e2 = ptr
   endif


   if (l_event_setup.oxygen eq 'Y') then begin 

      ;-----------------------------------------------------
      ; take the total number of events per spin sector and
      ; divide by the sector event count (possible number of
      ; events per sector)
      ;-----------------------------------------------------
      scaled_oxygen     = fltarr (l_setup.l_num_spinsec, spin)
      scaled_oxygen_e0  = fltarr (l_setup.l_num_spinsec, spin)
      scaled_oxygen_e1  = fltarr (l_setup.l_num_spinsec, spin)
      scaled_oxygen_e2  = fltarr (l_setup.l_num_spinsec, spin)
      scaled_O          = fltarr (l_setup.l_num_spinsec, spin)
      scaled_O_e0       = fltarr (l_setup.l_num_spinsec, spin)
      scaled_O_e1       = fltarr (l_setup.l_num_spinsec, spin)
      scaled_O_e2       = fltarr (l_setup.l_num_spinsec, spin)
      if (l_event_setup.background eq 'N') then begin
         scaled_background = make_array (l_setup.l_num_spinsec, spin, /FLOAT, value = 0.0)
         background = make_array (l_setup.l_num_spinsec, spin, /FLOAT, value = 0.0)
         ptr = ptr_new (background[*, 0:spin-1L], /NO_COPY)
         l_event.background_e0 = ptr
         l_event.background_e1 = ptr
         l_event.background_e2 = ptr
         l_event.background_raw_O = ptr
         l_event.background_raw_O_e0 = ptr
         l_event.background_raw_O_e1 = ptr
         l_event.background_raw_O_e2 = ptr
      endif

      ;-----------------------------------------------------
      ; scaled background, oxygen has only 60 channels
      ;-----------------------------------------------------
      O_background      = scaled_background * l_event_setup.O_scale_factor 
      corrected_O       = fltarr (l_setup.l_num_spinsec, spin)
      nsectors          = l_setup.l_num_spinsec
      O_elim_spins_time = dblarr (720)
      O_elim_spins_nevents = fltarr (720)

      for i = 0L, spin - 1L do begin
         for j = 0L,  nsectors - 1L do begin
             current_time = event_time[j,i]

             if (oxygen_nevents[j,i] ne NO_DATA) then begin 
                scaled_oxygen[j,i] = oxygen[j,i] * $
                      oxygen_nevents[j,i]/(oxygen_sec_events[j,i] > 1.0)
             endif else begin
                number_of_events = total (oxygen_sec_events[*,i])
                if (number_of_events lt 64) then begin
                   scaled_oxygen[*,i] = 1.0
                endif else begin
                   scaled_oxygen[*,i] = 0.0
                   oxygen[*,i] = 0.0
                   oxygen_nevents[*,i] = 0.0
                   oxygen_sec_events[*,i] = 0.0
                   start_udf_tme = ConvArrTimeToSecs (l_setup.l_start_dtime)
                   stop_udf_tme = ConvArrTimeToSecs (l_setup.l_stop_dtime)

                   if (current_time gt start_udf_tme) AND (current_time le stop_udf_tme) then begin
                       O_elim_spins_time[l_event.num_O_elim_spins] = current_time
                       O_elim_spins_nevents[l_event.num_O_elim_spins] = number_of_events
                       l_event.num_O_elim_spins = l_event.num_O_elim_spins + 1L
                   endif
 
                endelse
             endelse


             if (oxygen_nevents_e0[j,i] ne NO_DATA) then begin 
                scaled_oxygen_e0[j,i] = oxygen_e0[j,i] * $
                      oxygen_nevents_e0[j,i]/(oxygen_sec_events_e0[j,i] > 1.0)
             endif else begin
                number_of_events = total (oxygen_sec_events_e0[*,i])
                if (number_of_events lt 64) then begin
                   scaled_oxygen_e0[*,i] = 1.0
                endif else begin
                   scaled_oxygen_e0[*,i] = 0.0
                   oxygen_e0[*,i] = 0.0
                   oxygen_nevents_e0[*,i] = 0.0
                   oxygen_sec_events_e0[*,i] = 0.0
                endelse
             endelse


             if (oxygen_nevents_e1[j,i] ne NO_DATA) then begin 
                scaled_oxygen_e1[j,i] = oxygen_e1[j,i] * $
                      oxygen_nevents_e1[j,i]/(oxygen_sec_events_e1[j,i] > 1.0)
             endif else begin
                number_of_events = total (oxygen_sec_events_e1[*,i])
                if (number_of_events lt 64) then begin
                   scaled_oxygen_e1[*,i] = 1.0
                endif else begin
                   scaled_oxygen_e1[*,i] = 0.0
                   oxygen_e1[*,i] = 0.0
                   oxygen_nevents_e1[*,i] = 0.0
                   oxygen_sec_events_e1[*,i] = 0.0
                endelse
             endelse

             if (oxygen_nevents_e2[j,i] ne NO_DATA) then begin 
                scaled_oxygen_e2[j,i] = oxygen_e2[j,i] * $
                      oxygen_nevents_e2[j,i]/(oxygen_sec_events_e2[j,i] > 1.0)
             endif else begin
                number_of_events = total (oxygen_sec_events_e2[*,i])
                if (number_of_events lt 64) then begin
                   scaled_oxygen_e2[*,i] = 1.0
                endif else begin
                   scaled_oxygen_e2[*,i] = 0.0
                   oxygen_e2[*,i] = 0.0
                   oxygen_nevents_e2[*,i] = 0.0
                   oxygen_sec_events_e2[*,i] = 0.0
                endelse
             endelse
         endfor
      endfor

      ptr = ptr_new (O_elim_spins_time[0:l_event.num_O_elim_spins], /NO_COPY)
      l_event.O_elim_spins_time = ptr
      ptr = ptr_new (O_elim_spins_nevents[0:l_event.num_O_elim_spins], /NO_COPY)
      l_event.O_elim_spins_nevents = ptr


      zero_indx = where (oxygen_sec_events eq 0.0, zcnt)
      nzero_indx = where (oxygen_sec_events ne 0.0, nzcnt)
      if (nzcnt gt 0) then begin
         scaled_O[nzero_indx] = oxygen_nevents[nzero_indx]/oxygen_sec_events[nzero_indx]
         scaled_O[zero_indx] = 0.0
      endif else begin
         if (zcnt ge n_elements (oxygen_sec_events)) then begin
            scaled_O = oxygen_nevents
            scaled_O[*,*] = 0.0
         endif else begin
            scaled_O = oxygen_nevents/oxygen_sec_events
         endelse
      endelse

      zero_indx = where (oxygen_sec_events_e0 eq 0.0, zcnt)
      nzero_indx = where (oxygen_sec_events_e0 ne 0.0, nzcnt)
      if (nzcnt gt 0) then begin
         scaled_O_e0[nzero_indx] = oxygen_nevents_e0[nzero_indx]/oxygen_sec_events_e0[nzero_indx]
         scaled_O_e0[zero_indx] = 0.0
      endif else begin
         if (zcnt ge n_elements (oxygen_sec_events_e0)) then begin
            scaled_O_e0 = oxygen_nevents_e0
            scaled_O_e0[*,*] = 0.0
         endif else begin
            scaled_O_e0 = oxygen_nevents_e0/oxygen_sec_events_e0
         endelse
      endelse

      zero_indx = where (oxygen_sec_events_e1 eq 0.0, zcnt)
      nzero_indx = where (oxygen_sec_events_e1 ne 0.0, nzcnt)
      if (nzcnt gt 0) then begin
         scaled_O_e1[nzero_indx] = oxygen_nevents_e1[nzero_indx]/oxygen_sec_events_e1[nzero_indx]
         scaled_O_e1[zero_indx] = 0.0
      endif else begin
         if (zcnt ge n_elements (oxygen_sec_events_e1)) then begin
            scaled_O_e1 = oxygen_nevents_e1
            scaled_O_e1[*,*] = 0.0
         endif else begin
            scaled_O_e1 = oxygen_nevents_e1/oxygen_sec_events_e1
         endelse
      endelse

      zero_indx = where (oxygen_sec_events_e2 eq 0.0, zcnt)
      nzero_indx = where (oxygen_sec_events_e2 ne 0.0, nzcnt)
      if (nzcnt gt 0) then begin
         scaled_O_e2[nzero_indx] = oxygen_nevents_e2[nzero_indx]/oxygen_sec_events_e2[nzero_indx]
         scaled_O_e2[zero_indx] = 0.0
      endif else begin
         if (zcnt ge n_elements (oxygen_sec_events_e2)) then begin
            scaled_O_e2 = oxygen_nevents_e2
            scaled_O_e2[*,*] = 0.0
         endif else begin
            scaled_O_e2 = oxygen_nevents_e2/oxygen_sec_events_e2
         endelse
      endelse

      ;-----------------------------------------------------
      ; do background subtraction
      ;-----------------------------------------------------
      n9_eq_idx = where (scaled_oxygen eq NO_DATA, n9_eq_cnt)
      n9_ne_idx = where (scaled_oxygen ne NO_DATA, n9_ne_cnt)
      if (n9_eq_cnt gt 0) then begin
          corrected_O[n9_ne_idx] = scaled_oxygen[n9_ne_idx] - O_background[n9_ne_idx] 
      endif else begin
          corrected_O = scaled_oxygen - O_background 
      endelse

      neg_values = where (corrected_O lt 0.0, ncnt)
      corrected_O = (corrected_O > 0.0)

      ;-----------------------------------------------------
      ; if the corrected/scaled value goes negative, flag 
      ; as a no count value 
      ;-----------------------------------------------------
      ;if (ncnt gt 0) then begin
      ;   corrected_O[neg_values] = NO_CNTS
      ;endif

      o_indx = where (oxygen_nevents eq NO_DATA, o_cnt)
      if (o_cnt gt 0) then begin
         corrected_O[o_indx] = NO_DATA 
         scaled_oxygen[o_indx] = NO_DATA
      endif

      ;-----------------------------------------------------
      ; separate into different energy bins
      ;-----------------------------------------------------
      corrected_oxygen_e0 = fltarr (l_setup.l_num_spinsec, spin)
      corrected_oxygen_e0 = scaled_oxygen_e0 - ((*l_event.background_e0) * 6.0)
      corrected_oxygen_e0 = corrected_oxygen_e0 > 0.0
      ;neg_values = where (corrected_oxygen_e0 lt 0.0, ncnt)
      ;if (ncnt gt 0) then corrected_oxygen_e0[neg_values] = NO_CNTS
      o_indx = where (oxygen_nevents_e0 eq NO_DATA, o_cnt)
      if (o_cnt gt 0) then begin
         corrected_oxygen_e0[o_indx] = NO_DATA
         scaled_oxygen_e0[o_indx] = NO_DATA
      endif

      corrected_oxygen_e1 = fltarr (l_setup.l_num_spinsec, spin)
      corrected_oxygen_e1 = scaled_oxygen_e1 - ((*l_event.background_e1) * 6.0) 
      corrected_oxygen_e1 = corrected_oxygen_e1 > 0.0
      ;neg_values = where (corrected_oxygen_e1 lt 0.0, ncnt)
      ;if (ncnt gt 0) then corrected_oxygen_e1[neg_values] = NO_CNTS
      o_indx = where (oxygen_nevents_e1 eq NO_DATA, o_cnt)
      if (o_cnt gt 0) then begin
         corrected_oxygen_e1[o_indx] = NO_DATA
         scaled_oxygen_e1[o_indx] = NO_DATA
      endif

      corrected_oxygen_e2 = fltarr (l_setup.l_num_spinsec, spin)
      corrected_oxygen_e2 = scaled_oxygen_e2 - ((*l_event.background_e2) * 6.0)
      corrected_oxygen_e2 = corrected_oxygen_e2 > 0.0
      ;neg_values = where (corrected_oxygen_e2 lt 0.0, ncnt)
      ;if (ncnt gt 0) then corrected_oxygen_e2[neg_values] = NO_CNTS
      o_indx = where (oxygen_nevents_e2 eq NO_DATA, o_cnt)
      if (o_cnt gt 0) then begin
         corrected_oxygen_e2[o_indx] = NO_DATA
         ;scaled_oxygen_e2[o_indx] = NO_DATA
      endif

      ptr = ptr_new (oxygen_e0)
      l_event.oxygen_e0 = ptr

      ptr = ptr_new (oxygen_e1)
      l_event.oxygen_e1 = ptr

      ptr = ptr_new (oxygen_e2)
      l_event.oxygen_e2 = ptr

      ptr = ptr_new (event_start_udf_time[0:spin-1], /NO_COPY)
      l_event.filledStartUdfTimePtr = ptr

      ptr = ptr_new (event_stop_udf_time[0:spin-1], /NO_COPY)
      l_event.filledStopUdfTimePtr = ptr

      ptr = ptr_new (oxygen_nevents[*,0:spin-1], /NO_COPY)
      l_event.oxygen_raw = ptr

      ptr = ptr_new (oxygen_nevents_e0[*,0:spin-1], /NO_COPY)
      l_event.oxygen_raw_e0 = ptr

      ptr = ptr_new (oxygen_nevents_e1[*,0:spin-1], /NO_COPY)
      l_event.oxygen_raw_e1 = ptr

      ptr = ptr_new (oxygen_nevents_e2[*,0:spin-1], /NO_COPY)
      l_event.oxygen_raw_e2 = ptr

      tmp = (*l_event.background_raw_O) * l_event_setup.O_scale_factor
      ptr = ptr_new (tmp, /NO_COPY)
      l_event.background_O = ptr

      tmp = (*l_event.background_raw_O_e0) * l_event_setup.O_scale_factor
      ptr = ptr_new (tmp, /NO_COPY)
      l_event.background_O_e0 = ptr

      tmp = (*l_event.background_raw_O_e1) * l_event_setup.O_scale_factor
      ptr = ptr_new (tmp, /NO_COPY)
      l_event.background_O_e1 = ptr

      tmp = (*l_event.background_raw_O_e2) * l_event_setup.O_scale_factor
      ptr = ptr_new (tmp, /NO_COPY)
      l_event.background_O_e2 = ptr

      ptr = ptr_new (corrected_O, /NO_COPY)
      l_event.corrected_O = ptr

      ptr = ptr_new (corrected_oxygen_e0, /NO_COPY)
      l_event.corrected_O_e0 = ptr

      ptr = ptr_new (corrected_oxygen_e1, /NO_COPY)
      l_event.corrected_O_e1 = ptr

      ptr = ptr_new (corrected_oxygen_e2, /NO_COPY)
      l_event.corrected_O_e2 = ptr

      ptr = ptr_new (oxygen_sec_events[*,0:spin-1], /NO_COPY)
      l_event.O_sec_events = ptr

      ptr = ptr_new (oxygen_sec_events_e0[*,0:spin-1], /NO_COPY)
      l_event.O_sec_events_e0 = ptr

      ptr = ptr_new (oxygen_sec_events_e1[*,0:spin-1], /NO_COPY)
      l_event.O_sec_events_e1 = ptr

      ptr = ptr_new (oxygen_sec_events_e2[*,0:spin-1], /NO_COPY)
      l_event.O_sec_events_e2 = ptr

      ptr = ptr_new (oxygen[*,0:spin-1], /NO_COPY)
      l_event.O_n_in_peak = ptr

      ptr = ptr_new (oxygen_e0[*,0:spin-1], /NO_COPY)
      l_event.O_n_in_peak_e0 = ptr

      ptr = ptr_new (oxygen_e1[*,0:spin-1], /NO_COPY)
      l_event.O_n_in_peak_e1 = ptr

      ptr = ptr_new (oxygen_e2[*,0:spin-1], /NO_COPY)
      l_event.O_n_in_peak_e2 = ptr

      ptr = ptr_new (scaled_O[*,0:spin-1], /NO_COPY)
      l_event.O_scale_factor = ptr

      ptr = ptr_new (scaled_O_e0[*,0:spin-1], /NO_COPY)
      l_event.O_scale_factor_e0 = ptr

      ptr = ptr_new (scaled_O_e1[*,0:spin-1], /NO_COPY)
      l_event.O_scale_factor_e1 = ptr

      ptr = ptr_new (scaled_O_e2[*,0:spin-1], /NO_COPY)
      l_event.O_scale_factor_e2 = ptr

      ptr = ptr_new (scaled_oxygen, /NO_COPY)
      l_event.scaled_O = ptr

      ptr = ptr_new (scaled_oxygen_e0, /NO_COPY)
      l_event.scaled_O_e0 = ptr

      ptr = ptr_new (scaled_oxygen_e1, /NO_COPY)
      l_event.scaled_O_e1 = ptr

      ptr = ptr_new (scaled_oxygen_e2, /NO_COPY)
      l_event.scaled_O_e2 = ptr

   endif


   RETURN

END

;-------------------------------------------------------------------------------
;  Function : l_EventHndlMissData
;
;  Description: Handle missing data in the event packets.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_setup               struct        I        generic setup parameters
;  l_event               struct *      I        event specific data plot parameters
;  l_event_setup         struct        I        event plot specific parameters
;  l_oa                  struct *      I        OA specific parameters/attributes
;  l_pltpos              struct        I        generic plot parameters
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;     PLOTDEFS         MAX_PLOTS        int         G         max # plots per page
;                      MAX_WEB_SPINS    int         G         max # spins per plot
;                                                             when using web interface
;                      NO_CNTS          double      G         no counts indicator
;                      NO_DATA          double      G         no data indicator
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                12/04/01       v1.0.7          Original implementation
;-------------------------------------------------------------------------------
PRO l_EventHndlMissData, l_setup, l_event, l_event_setup, l_oa, l_pltpos
COMMON PLOTDEFS, MAX_PLOTS, MAX_WEB_SPINS, NO_CNTS, NO_DATA


   data_format = l_setup.l_num_spinsec


   ;---------------------------------------------
   ; get sector locations for the specified trace
   ;---------------------------------------------
   if (l_setup.l_angle_zero eq 'E') then begin
      center_sector = l_setup.l_earth_center
      ytitle_frame = '!5Earth Cntrd Azi'
   endif else begin
      if (l_setup.l_angle_zero eq 'S') then begin
          l_CalcSunPulse, l_oa, lena_sector, 'S'
          center_sector = fix(lena_sector)
          cs = lena_sector/8.
          if NOT (ptr_valid(l_oa.lena_sun_sectPtr)) then begin
             ptr = ptr_new (cs, /NO_COPY)
             l_oa.lena_sun_sectPtr = ptr
          endif
          ytitle_frame = '!5Sun Cntrd Azi'
       endif else begin
          if (l_setup.l_angle_zero eq 'M') then begin
             l_CalcSunPulse, l_oa, lena_sector, 'M'
             center_sector = fix(lena_sector)
             cs = fix (center_sector/8.)
             if NOT (ptr_valid(l_oa.lena_moon_sectPtr)) then begin
                ptr = ptr_new(cs, /NO_COPY)
                l_oa.lena_moon_sectPtr = ptr
             endif
             ytitle_frame = '!5Moon Cntrd Azi'
          endif else begin
            ytitle_frame  = '!5Spin Angle'
            center_sector = 0
         endelse
      endelse
   endelse

   ;---------------------------------------------
   ; determine if adjusting zero angle to earth,
   ; sun or moon
   ;---------------------------------------------
   adjFlag = 'YES'

   if (l_setup.l_angle_zero eq 'N') then adjFlag = 'NO'

   if (l_setup.l_sun_direction eq 'Y') then begin
      l_CalcSunPulse, l_oa, lena_sector, 'S'
      sun_direction = lena_sector/8.
      if NOT (ptr_valid(l_oa.lena_sun_sectPtr)) then begin
         ptr = ptr_new (sun_direction, /NO_COPY)
         l_oa.lena_sun_sectPtr = ptr
      endif
   endif


   orig_tme = reform ((*l_event.spTimePtr), n_elements((*l_event.spTimePtr)))

   orig_start_udf_tme = (*l_event.filledStartUdfTimePtr)
   orig_stop_udf_tme =  (*l_event.filledStopUdfTimePtr)


   nspins = n_elements (orig_start_udf_tme)
  
   ;---------------------------------------------
   ; determine the number of major and minor ticks
   ;---------------------------------------------
   l_setup.l_d_start_dtime[0] = orig_start_udf_tme[0].year
   l_setup.l_d_start_dtime[1] = orig_start_udf_tme[0].doy
   l_setup.l_d_start_dtime[2] = orig_start_udf_tme[0].hour
   l_setup.l_d_start_dtime[3] = orig_start_udf_tme[0].min
   l_setup.l_d_start_dtime[4] = orig_start_udf_tme[0].sec
   l_setup.l_d_start_dtime[5] = orig_start_udf_tme[0].msec

   max_ele = n_elements ((orig_stop_udf_tme)) - 1

   l_setup.l_d_stop_dtime[0] = orig_stop_udf_tme[max_ele].year
   l_setup.l_d_stop_dtime[1] = orig_stop_udf_tme[max_ele].doy
   l_setup.l_d_stop_dtime[2] = orig_stop_udf_tme[max_ele].hour
   l_setup.l_d_stop_dtime[3] = orig_stop_udf_tme[max_ele].min
   l_setup.l_d_stop_dtime[4] = orig_stop_udf_tme[max_ele].sec
   l_setup.l_d_stop_dtime[5] = orig_stop_udf_tme[max_ele].msec


   l_DetMjMnTicks, l_setup.l_start_dtime, $
                   l_setup.l_stop_dtime, $
                   l_setup.l_d_start_dtime, $
                   l_setup.l_d_stop_dtime, $
                   l_pltpos




   ;---------------------------------------------------
   ; handle missing data in hydrogen data
   ;---------------------------------------------------
   if (l_event_setup.hydrogen eq 'Y') then begin 

      ;---------------------------------------------------
      ; background data 
      ;---------------------------------------------------
      if (l_event_setup.background eq 'Y') then begin
          background_raw = (*l_event.background_raw_H)
          ;nspins = n_elements (background_raw)/l_setup.l_num_spinsec

          l_HndlMissData, orig_tme,              $
                          hmd_tme,               $
                          nspins,                $
                          background_raw,        $
                          event_hmd,             $
                          data_format ,          $
                          center_sector, $
                          l_setup, $
                          adjFlag, $
                          l_oa, $
                          orig_start_udf_tme, $ 
                          new_udf_tme, $
                          (*l_event.background_raw_extra), $
                          l_setup.l_spin_ang_zero

          (*l_event.background_raw_H) = event_hmd

          event_background = (*l_event.background_H)

          ;nspins = n_elements (event_background)/l_setup.l_num_spinsec
          l_HndlMissData, orig_tme,              $
                          hmd_tme,               $
                          nspins,                $
                          event_background,      $
                          event_hmd,             $
                          data_format ,          $
                          center_sector, $
                          l_setup, $
                          adjFlag, $
                          l_oa, $
                          orig_start_udf_tme, $ 
                          new_udf_tme, $
                          (*l_event.background_H_extra), $
                          l_setup.l_spin_ang_zero

          (*l_event.background_H) = event_hmd

          if (l_event_setup.energy0 eq 'Y') then begin
             background_raw = (*l_event.background_raw_H_e0)

             l_HndlMissData, orig_tme,              $
                             hmd_tme,               $
                             nspins,                $
                             background_raw,        $
                             event_hmd,             $
                             data_format ,          $
                             center_sector, $
                             l_setup, $
                             adjFlag, $
                             l_oa, $
                             orig_start_udf_tme, $ 
                             new_udf_tme, $
                             (*l_event.background_raw_extra_e0), $
                             l_setup.l_spin_ang_zero

             (*l_event.background_raw_H_e0) = event_hmd

             event_background = (*l_event.background_H_e0)

             l_HndlMissData, orig_tme,              $
                             hmd_tme,               $
                             nspins,                $
                             event_background,      $
                             event_hmd,             $
                             data_format ,          $
                             center_sector, $
                             l_setup, $
                             adjFlag, $
                             l_oa, $
                             orig_start_udf_tme, $ 
                             new_udf_tme, $
                             (*l_event.background_H_extra_e0), $
                             l_setup.l_spin_ang_zero

             (*l_event.background_H_e0) = event_hmd

          endif

          if (l_event_setup.energy1 eq 'Y') then begin
             background_raw = (*l_event.background_raw_H_e1)

             l_HndlMissData, orig_tme,              $
                             hmd_tme,               $
                             nspins,                $
                             background_raw,        $
                             event_hmd,             $
                             data_format ,          $
                             center_sector, $
                             l_setup, $
                             adjFlag, $
                             l_oa, $
                             orig_start_udf_tme, $ 
                             new_udf_tme, $
                             (*l_event.background_raw_extra_e1), $
                             l_setup.l_spin_ang_zero

             (*l_event.background_raw_H_e1) = event_hmd

             event_background = (*l_event.background_H_e1)

             l_HndlMissData, orig_tme,              $
                             hmd_tme,               $
                             nspins,                $
                             event_background,      $
                             event_hmd,             $
                             data_format ,          $
                             center_sector, $
                             l_setup, $
                             adjFlag, $
                             l_oa, $
                             orig_start_udf_tme, $ 
                             new_udf_tme, $
                             (*l_event.background_H_extra_e1), $
                             l_setup.l_spin_ang_zero

             (*l_event.background_H_e1) = event_hmd

          endif

          if (l_event_setup.energy2 eq 'Y') then begin
             background_raw = (*l_event.background_raw_H_e2)

             l_HndlMissData, orig_tme,              $
                             hmd_tme,               $
                             nspins,                $
                             background_raw,        $
                             event_hmd,             $
                             data_format ,          $
                             center_sector, $
                             l_setup, $
                             adjFlag, $
                             l_oa, $
                             orig_start_udf_tme, $ 
                             new_udf_tme, $
                             (*l_event.background_raw_extra_e2), $
                             l_setup.l_spin_ang_zero

             (*l_event.background_raw_H_e2) = event_hmd
   
             event_background = (*l_event.background_H_e2)

             l_HndlMissData, orig_tme,              $
                             hmd_tme,               $
                             nspins,                $
                             event_background,      $
                             event_hmd,             $
                             data_format ,          $
                             center_sector, $
                             l_setup, $
                             adjFlag, $
                             l_oa, $
                             orig_start_udf_tme, $ 
                             new_udf_tme, $
                             (*l_event.background_H_extra_e2), $
                             l_setup.l_spin_ang_zero

             (*l_event.background_H_e2) = event_hmd

          endif

      endif

      ;---------------------------------------------------
      ; handle missing data/center hydrogen data 
      ;---------------------------------------------------
      event_raw = (*l_event.hydrogen_raw) 
      l_HndlMissData, orig_tme,              $
                      hmd_tme,               $
                      nspins,                $
                      event_raw,             $
                      event_hmd,             $
                      data_format ,          $
                      center_sector, $
                      l_setup, $
                      adjFlag, $
                      l_oa, $
                      orig_start_udf_tme, $ 
                      new_udf_tme, $
                      (*l_event.hydrogen_raw_extra), $
                      l_setup.l_spin_ang_zero

      (*l_event.hydrogen_raw) = event_hmd
   
    

      event_corr = (*l_event.corrected_H) 
      l_HndlMissData, orig_tme,              $
                      hmd_tme,               $
                      nspins,                $
                      event_corr,            $
                      event_hmd,             $
                      data_format ,          $
                      center_sector, $
                      l_setup, $
                      adjFlag, $
                      l_oa, $
                      orig_start_udf_tme, $ 
                      new_udf_tme, $
                      (*l_event.corrected_H_extra), $
                      l_setup.l_spin_ang_zero

      (*l_event.corrected_H) = event_hmd


      events_n_sect = (*l_event.H_sec_events)
      l_HndlMissData, orig_tme,              $
                      hmd_tme,               $
                      nspins,                $
                      events_n_sect,         $
                      event_hmd_1,           $
                      data_format ,          $
                      center_sector, $
                      l_setup, $
                      adjFlag, $
                      l_oa, $
                      orig_start_udf_tme, $ 
                      new_udf_tme, $
                      (*l_event.H_sec_events_extra), $
                      l_setup.l_spin_ang_zero

      (*l_event.H_sec_events) = event_hmd_1



      nevents_n_peak = (*l_event.H_n_in_peak)
      l_HndlMissData, orig_tme,              $
                      hmd_tme,               $
                      nspins,                $
                      nevents_n_peak,        $
                      event_hmd_1,             $
                      data_format ,          $
                      center_sector, $
                      l_setup, $
                      adjFlag, $
                      l_oa, $
                      orig_start_udf_tme, $ 
                      new_udf_tme, $
                      (*l_event.H_n_in_peak_extra), $
                      l_setup.l_spin_ang_zero

      (*l_event.H_n_in_peak) = event_hmd_1



      scale_factor = (*l_event.H_scale_factor)
      l_HndlMissData, orig_tme,              $
                      hmd_tme,               $
                      nspins,                $
                      scale_factor,          $
                      event_hmd_1,             $
                      data_format ,          $
                      center_sector, $
                      l_setup, $
                      adjFlag, $
                      l_oa, $
                      orig_start_udf_tme, $ 
                      new_udf_tme, $
                      (*l_event.H_scale_factor_extra), $
                      l_setup.l_spin_ang_zero

      (*l_event.H_scale_factor) = event_hmd_1


      scale_factor = (*l_event.scaled_H)
      dummy_extra = fltarr(l_setup.l_num_spinsec, 2L)
      l_HndlMissData, orig_tme,              $
                      hmd_tme,               $
                      nspins,                $
                      scale_factor,          $
                      event_hmd_1,             $
                      data_format ,          $
                      center_sector, $
                      l_setup, $
                      adjFlag, $
                      l_oa, $
                      orig_start_udf_tme, $ 
                      new_udf_tme, $
                      dummy_extra, $
                      l_setup.l_spin_ang_zero

      (*l_event.scaled_H) = event_hmd_1




      ;---------------------------------------------------
      ; hydrogen energy 0
      ;---------------------------------------------------
      if (l_event_setup.energy0 eq 'Y') then begin
         event_raw = (*l_event.hydrogen_raw_e0) 
         l_HndlMissData, orig_tme,              $
                         hmd_tme,               $
                         nspins,                $
                         event_raw,             $
                         event_hmd,             $
                         data_format ,          $
                         center_sector, $
                         l_setup, $
                         adjFlag, $
                         l_oa, $
                         orig_start_udf_tme, $ 
                         new_udf_tme, $
                         (*l_event.hydrogen_raw_extra_e0), $
                         l_setup.l_spin_ang_zero

         (*l_event.hydrogen_raw_e0) = event_hmd
   
         event_corr = (*l_event.corrected_H_e0) 
         l_HndlMissData, orig_tme,              $
                         hmd_tme,               $
                         nspins,                $
                         event_corr,            $
                         event_hmd,             $
                         data_format ,          $
                         center_sector, $
                         l_setup, $
                         adjFlag, $
                         l_oa, $
                         orig_start_udf_tme, $ 
                         new_udf_tme, $
                         (*l_event.corrected_H_extra_e0), $
                         l_setup.l_spin_ang_zero

         (*l_event.corrected_H_e0) = event_hmd


         events_n_sect = (*l_event.H_sec_events_e0)
         l_HndlMissData, orig_tme,              $
                         hmd_tme,               $
                         nspins,                $
                         events_n_sect,         $
                         event_hmd_1,           $
                         data_format ,          $
                         center_sector, $
                         l_setup, $
                         adjFlag, $
                         l_oa, $
                         orig_start_udf_tme, $ 
                         new_udf_tme, $
                         (*l_event.H_sec_events_extra_e0), $
                         l_setup.l_spin_ang_zero

         (*l_event.H_sec_events_e0) = event_hmd_1


         nevents_n_peak = (*l_event.H_n_in_peak_e0)
         l_HndlMissData, orig_tme,              $
                         hmd_tme,               $
                         nspins,                $
                         nevents_n_peak,        $
                         event_hmd_1,             $
                         data_format ,          $
                         center_sector, $
                         l_setup, $
                         adjFlag, $
                         l_oa, $
                         orig_start_udf_tme, $ 
                         new_udf_tme, $
                         (*l_event.H_n_in_peak_extra_e0), $
                         l_setup.l_spin_ang_zero

         (*l_event.H_n_in_peak_e0) = event_hmd_1


         scale_factor = (*l_event.H_scale_factor_e0)
         l_HndlMissData, orig_tme,              $
                         hmd_tme,               $
                         nspins,                $
                         scale_factor,          $
                         event_hmd_1,             $
                         data_format ,          $
                         center_sector, $
                         l_setup, $
                         adjFlag, $
                         l_oa, $
                         orig_start_udf_tme, $ 
                         new_udf_tme, $
                         (*l_event.H_scale_factor_extra_e0), $
                         l_setup.l_spin_ang_zero
   
         (*l_event.H_scale_factor_e0) = event_hmd_1

      endif


      ;---------------------------------------------------
      ; hydrogen energy 1  
      ;---------------------------------------------------
      if (l_event_setup.energy1 eq 'Y') then begin
         event_raw = (*l_event.hydrogen_raw_e1) 
         l_HndlMissData, orig_tme,              $
                         hmd_tme,               $
                         nspins,                $
                         event_raw,             $
                         event_hmd,             $
                         data_format ,          $
                         center_sector, $
                         l_setup, $
                         adjFlag, $
                         l_oa, $
                         orig_start_udf_tme, $ 
                         new_udf_tme, $
                         (*l_event.hydrogen_raw_extra_e1), $
                         l_setup.l_spin_ang_zero

         (*l_event.hydrogen_raw_e1) = event_hmd
   
         event_corr = (*l_event.corrected_H_e1) 
         l_HndlMissData, orig_tme,              $
                         hmd_tme,               $
                         nspins,                $
                         event_corr,            $
                         event_hmd,             $
                         data_format ,          $
                         center_sector, $
                         l_setup, $
                         adjFlag, $
                         l_oa, $
                         orig_start_udf_tme, $ 
                         new_udf_tme, $
                         (*l_event.corrected_H_extra_e1), $
                         l_setup.l_spin_ang_zero

         (*l_event.corrected_H_e1) = event_hmd


         events_n_sect = (*l_event.H_sec_events_e1)
         l_HndlMissData, orig_tme,              $
                         hmd_tme,               $
                         nspins,                $
                         events_n_sect,         $
                         event_hmd_1,           $
                         data_format ,          $
                         center_sector, $
                         l_setup, $
                         adjFlag, $
                         l_oa, $
                         orig_start_udf_tme, $ 
                         new_udf_tme, $
                         (*l_event.H_sec_events_extra_e1), $
                         l_setup.l_spin_ang_zero

         (*l_event.H_sec_events_e1) = event_hmd_1


         nevents_n_peak = (*l_event.H_n_in_peak_e1)
         l_HndlMissData, orig_tme,              $
                         hmd_tme,               $
                         nspins,                $
                         nevents_n_peak,        $
                         event_hmd_1,             $
                         data_format ,          $
                         center_sector, $
                         l_setup, $
                         adjFlag, $
                         l_oa, $
                         orig_start_udf_tme, $ 
                         new_udf_tme, $
                         (*l_event.H_n_in_peak_extra_e1), $
                         l_setup.l_spin_ang_zero

         (*l_event.H_n_in_peak_e1) = event_hmd_1


         scale_factor = (*l_event.H_scale_factor_e1)
         l_HndlMissData, orig_tme,              $
                         hmd_tme,               $
                         nspins,                $
                         scale_factor,          $
                         event_hmd_1,             $
                         data_format ,          $
                         center_sector, $
                         l_setup, $
                         adjFlag, $
                         l_oa, $
                         orig_start_udf_tme, $ 
                         new_udf_tme, $
                         (*l_event.H_scale_factor_extra_e1), $
                         l_setup.l_spin_ang_zero
   
         (*l_event.H_scale_factor_e1) = event_hmd_1
      endif

      ;---------------------------------------------------
      ; hydrogen energy 2  
      ;---------------------------------------------------
      if (l_event_setup.energy2 eq 'Y') then begin
         event_raw = (*l_event.hydrogen_raw_e2) 
         l_HndlMissData, orig_tme,              $
                         hmd_tme,               $
                         nspins,                $
                         event_raw,             $
                         event_hmd,             $
                         data_format ,          $
                         center_sector, $
                         l_setup, $
                         adjFlag, $
                         l_oa, $
                         orig_start_udf_tme, $ 
                         new_udf_tme, $
                         (*l_event.hydrogen_raw_extra_e2), $
                         l_setup.l_spin_ang_zero

         (*l_event.hydrogen_raw_e2) = event_hmd
   
         event_corr = (*l_event.corrected_H_e2) 
         l_HndlMissData, orig_tme,              $
                         hmd_tme,               $
                         nspins,                $
                         event_corr,            $
                         event_hmd,             $
                         data_format ,          $
                         center_sector, $
                         l_setup, $
                         adjFlag, $
                         l_oa, $
                         orig_start_udf_tme, $ 
                         new_udf_tme, $
                         (*l_event.corrected_H_extra_e2), $
                         l_setup.l_spin_ang_zero

         (*l_event.corrected_H_e2) = event_hmd


         events_n_sect = (*l_event.H_sec_events_e2)
         l_HndlMissData, orig_tme,              $
                         hmd_tme,               $
                         nspins,                $
                         events_n_sect,         $
                         event_hmd_1,           $
                         data_format ,          $
                         center_sector, $
                         l_setup, $
                         adjFlag, $
                         l_oa, $
                         orig_start_udf_tme, $ 
                         new_udf_tme, $
                         (*l_event.H_sec_events_extra_e2), $
                         l_setup.l_spin_ang_zero

         (*l_event.H_sec_events_e2) = event_hmd_1


         nevents_n_peak = (*l_event.H_n_in_peak_e2)
         l_HndlMissData, orig_tme,              $
                         hmd_tme,               $
                         nspins,                $
                         nevents_n_peak,        $
                         event_hmd_1,             $
                         data_format ,          $
                         center_sector, $
                         l_setup, $
                         adjFlag, $
                         l_oa, $
                         orig_start_udf_tme, $ 
                         new_udf_tme, $
                         (*l_event.H_n_in_peak_extra_e2), $
                         l_setup.l_spin_ang_zero

         (*l_event.H_n_in_peak_e2) = event_hmd_1


         scale_factor = (*l_event.H_scale_factor_e2)
         l_HndlMissData, orig_tme,              $
                         hmd_tme,               $
                         nspins,                $
                         scale_factor,          $
                         event_hmd_1,             $
                         data_format ,          $
                         center_sector, $
                         l_setup, $
                         adjFlag, $
                         l_oa, $
                         orig_start_udf_tme, $ 
                         new_udf_tme, $
                         (*l_event.H_scale_factor_extra_e2), $
                         l_setup.l_spin_ang_zero
   
         (*l_event.H_scale_factor_e2) = event_hmd_1
      endif

      he0 = (*l_event.hydrogen_e0)
      l_HndlMissData, orig_tme,              $
                      hmd_tme,               $
                      nspins,                $
                      he0,          $
                      event_hmd_1,             $
                      data_format ,          $
                      center_sector, $
                      l_setup, $
                      adjFlag, $
                      l_oa, $
                      orig_start_udf_tme, $ 
                      new_udf_tme, $
                      (*l_event.hydrogen_e0_extra), $
                      l_setup.l_spin_ang_zero

      (*l_event.hydrogen_e0) = event_hmd_1


      he1 = (*l_event.hydrogen_e1)
      l_HndlMissData, orig_tme,              $
                      hmd_tme,               $
                      nspins,                $
                      he1,          $
                      event_hmd_1,             $
                      data_format ,          $
                      center_sector, $
                      l_setup, $
                      adjFlag, $
                      l_oa, $
                      orig_start_udf_tme, $ 
                      new_udf_tme, $
                      (*l_event.hydrogen_e1_extra), $
                      l_setup.l_spin_ang_zero

      (*l_event.hydrogen_e1) = event_hmd_1


      he2 = (*l_event.hydrogen_e2)
      l_HndlMissData, orig_tme,              $
                      hmd_tme,               $
                      nspins,                $
                      he2,          $
                      event_hmd_1,             $
                      data_format ,          $
                      center_sector, $
                      l_setup, $
                      adjFlag, $
                      l_oa, $
                      orig_start_udf_tme, $ 
                      new_udf_tme, $
                      (*l_event.hydrogen_e2_extra), $
                      l_setup.l_spin_ang_zero

      (*l_event.hydrogen_e2) = event_hmd_1


   endif



   ;---------------------------------------------------
   ; process oxygen data
   ;---------------------------------------------------

   if (l_event_setup.oxygen eq 'Y') then begin 

      ;---------------------------------------------------
      ; check for missing background data 
      ;---------------------------------------------------
      if (l_event_setup.background eq 'Y') then begin
         background_raw = (*l_event.background_raw_O)
         l_HndlMissData, orig_tme,              $
                         hmd_tme,               $
                         nspins,                $
                         background_raw,        $
                         event_hmd,             $
                         data_format ,          $
                         center_sector, $
                         l_setup, $
                         adjFlag, $
                         l_oa, $
                         orig_start_udf_tme, $ 
                         new_udf_tme, $
                         (*l_event.background_raw_extra), $
                         l_setup.l_spin_ang_zero

         (*l_event.background_raw_O) = event_hmd

         event_background = (*l_event.background_O)
         l_HndlMissData, orig_tme,              $
                         hmd_tme,               $
                         nspins,                $
                         event_background,      $
                         event_hmd,             $
                         data_format ,          $
                         center_sector, $
                         l_setup, $
                         adjFlag, $
                         l_oa, $
                         orig_start_udf_tme, $ 
                         new_udf_tme, $
                         (*l_event.background_O_extra), $
                         l_setup.l_spin_ang_zero

         (*l_event.background_O) = event_hmd


         if (l_event_setup.energy0 eq 'Y') then begin
            background_raw = (*l_event.background_raw_O_e0)
            l_HndlMissData, orig_tme,              $
                            hmd_tme,               $
                            nspins,                $
                            background_raw,        $
                            event_hmd,             $
                            data_format ,          $
                            center_sector, $
                            l_setup, $
                            adjFlag, $
                            l_oa, $
                            orig_start_udf_tme, $ 
                            new_udf_tme, $
                            (*l_event.background_raw_extra_e0), $
                            l_setup.l_spin_ang_zero

            (*l_event.background_raw_O_e0) = event_hmd

            event_background = (*l_event.background_O_e0)
            l_HndlMissData, orig_tme,              $
                            hmd_tme,               $
                            nspins,                $
                            event_background,      $
                            event_hmd,             $
                            data_format ,          $
                            center_sector, $
                            l_setup, $
                            adjFlag, $
                            l_oa, $
                            orig_start_udf_tme, $ 
                            new_udf_tme, $
                            (*l_event.background_O_extra_e0), $
                            l_setup.l_spin_ang_zero
   
            (*l_event.background_O_e0) = event_hmd
         endif

         if (l_event_setup.energy1 eq 'Y') then begin
            background_raw = (*l_event.background_raw_O_e1)
            l_HndlMissData, orig_tme,              $
                            hmd_tme,               $
                            nspins,                $
                            background_raw,        $
                            event_hmd,             $
                            data_format ,          $
                            center_sector, $
                            l_setup, $
                            adjFlag, $
                            l_oa, $
                            orig_start_udf_tme, $ 
                            new_udf_tme, $
                            (*l_event.background_raw_extra_e1), $
                            l_setup.l_spin_ang_zero

            (*l_event.background_raw_O_e1) = event_hmd

            event_background = (*l_event.background_O_e1)
            l_HndlMissData, orig_tme,              $
                            hmd_tme,               $
                            nspins,                $
                            event_background,      $
                            event_hmd,             $
                            data_format ,          $
                            center_sector, $
                            l_setup, $
                            adjFlag, $
                            l_oa, $
                            orig_start_udf_tme, $ 
                            new_udf_tme, $
                            (*l_event.background_O_extra_e1), $
                            l_setup.l_spin_ang_zero
   
            (*l_event.background_O_e1) = event_hmd
         endif


         if (l_event_setup.energy2 eq 'Y') then begin
            background_raw = (*l_event.background_raw_O_e2)
            l_HndlMissData, orig_tme,              $
                            hmd_tme,               $
                            nspins,                $
                            background_raw,        $
                            event_hmd,             $
                            data_format ,          $
                            center_sector, $
                            l_setup, $
                            adjFlag, $
                            l_oa, $
                            orig_start_udf_tme, $ 
                            new_udf_tme, $
                            (*l_event.background_raw_extra_e2), $
                            l_setup.l_spin_ang_zero

            (*l_event.background_raw_O_e2) = event_hmd

            event_background = (*l_event.background_O_e2)
            l_HndlMissData, orig_tme,              $
                            hmd_tme,               $
                            nspins,                $
                            event_background,      $
                            event_hmd,             $
                            data_format ,          $
                            center_sector, $
                            l_setup, $
                            adjFlag, $
                            l_oa, $
                            orig_start_udf_tme, $ 
                            new_udf_tme, $
                            (*l_event.background_O_extra_e2), $
                            l_setup.l_spin_ang_zero
   
            (*l_event.background_O_e2) = event_hmd
         endif
      endif

      event_raw = (*l_event.oxygen_raw) 
      l_HndlMissData, orig_tme,              $
                      hmd_tme,               $
                      nspins,                $
                      event_raw,             $
                      event_hmd,             $
                      data_format ,          $
                      center_sector, $
                      l_setup, $
                      adjFlag, $
                      l_oa, $
                      orig_start_udf_tme, $ 
                      new_udf_tme, $
                      (*l_event.oxygen_raw_extra), $
                      l_setup.l_spin_ang_zero
 

      (*l_event.oxygen_raw) = event_hmd


      event_corr = (*l_event.corrected_O) 
      l_HndlMissData, orig_tme,              $
                      hmd_tme,               $
                      nspins,                $
                      event_corr,            $
                      event_hmd,             $
                      data_format ,          $
                      center_sector, $
                      l_setup, $
                      adjFlag, $
                      l_oa, $
                      orig_start_udf_tme, $ 
                      new_udf_tme, $
                      (*l_event.corrected_O_extra), $
                      l_setup.l_spin_ang_zero

      (*l_event.corrected_O) =  event_hmd

      events_n_sect = (*l_event.O_sec_events)
      l_HndlMissData, orig_tme,              $
                      hmd_tme,               $
                      nspins,                $
                      events_n_sect,         $
                      event_hmd_1,             $
                      data_format ,          $
                      center_sector, $
                      l_setup, $
                      adjFlag, $
                      l_oa, $
                      orig_start_udf_tme, $ 
                      new_udf_tme, $
                      (*l_event.O_sec_events_extra), $
                      l_setup.l_spin_ang_zero

      (*l_event.O_sec_events) =  event_hmd_1


      nevents_n_peak = (*l_event.O_n_in_peak)
      l_HndlMissData, orig_tme,              $
                      hmd_tme,               $
                      nspins,                $
                      nevents_n_peak,        $
                      event_hmd_1,             $
                      data_format ,          $
                      center_sector, $
                      l_setup, $
                      adjFlag, $
                      l_oa, $
                      orig_start_udf_tme, $ 
                      new_udf_tme, $
                      (*l_event.O_n_in_peak_extra), $
                      l_setup.l_spin_ang_zero

      (*l_event.O_n_in_peak) =  event_hmd_1

      scale_factor = (*l_event.O_scale_factor)
      l_HndlMissData, orig_tme,              $
                      hmd_tme,               $
                      nspins,                $
                      scale_factor,        $
                      event_hmd_1,             $
                      data_format ,          $
                      center_sector, $
                      l_setup, $
                      adjFlag, $
                      l_oa, $
                      orig_start_udf_tme, $ 
                      new_udf_tme, $
                      (*l_event.O_scale_factor_extra), $
                      l_setup.l_spin_ang_zero

      (*l_event.O_scale_factor) =  event_hmd_1


      scale_factor = (*l_event.scaled_O)
      dummy_extra = fltarr(l_setup.l_num_spinsec, 2L)
      l_HndlMissData, orig_tme,              $
                      hmd_tme,               $
                      nspins,                $
                      scale_factor,        $
                      event_hmd_1,             $
                      data_format ,          $
                      center_sector, $
                      l_setup, $
                      adjFlag, $
                      l_oa, $
                      orig_start_udf_tme, $ 
                      new_udf_tme, $
                      dummy_extra, $
                      l_setup.l_spin_ang_zero
      (*l_event.scaled_O) =  event_hmd_1


      ;----------------------------------------------------
      ; oxygen energy0
      ;----------------------------------------------------
      if (l_event_setup.energy0 eq 'Y') then begin
         event_raw = (*l_event.oxygen_raw_e0) 
         l_HndlMissData, orig_tme,              $
                         hmd_tme,               $
                         nspins,                $
                         event_raw,             $
                         event_hmd,             $
                         data_format ,          $
                         center_sector, $
                         l_setup, $
                         adjFlag, $
                         l_oa, $
                         orig_start_udf_tme, $ 
                         new_udf_tme, $
                         (*l_event.oxygen_raw_extra_e0), $
                         l_setup.l_spin_ang_zero
 

         (*l_event.oxygen_raw_e0) = event_hmd


         event_corr = (*l_event.corrected_O_e0) 
            l_HndlMissData, orig_tme,              $
                         hmd_tme,               $
                         nspins,                $
                         event_corr,            $
                         event_hmd,             $
                         data_format ,          $
                         center_sector, $
                         l_setup, $
                         adjFlag, $
                         l_oa, $
                         orig_start_udf_tme, $ 
                         new_udf_tme, $
                         (*l_event.corrected_O_extra_e0), $
                         l_setup.l_spin_ang_zero

         (*l_event.corrected_O_e0) =  event_hmd

         events_n_sect = (*l_event.O_sec_events_e0)
         l_HndlMissData, orig_tme,              $
                         hmd_tme,               $
                         nspins,                $
                         events_n_sect,         $
                         event_hmd_1,             $
                         data_format ,          $
                         center_sector, $
                         l_setup, $
                         adjFlag, $
                         l_oa, $
                         orig_start_udf_tme, $ 
                         new_udf_tme, $
                         (*l_event.O_sec_events_extra_e0), $
                         l_setup.l_spin_ang_zero
   
         (*l_event.O_sec_events_e0) =  event_hmd_1


         nevents_n_peak = (*l_event.O_n_in_peak_e0)
         l_HndlMissData, orig_tme,              $
                         hmd_tme,               $
                         nspins,                $
                         nevents_n_peak,        $
                         event_hmd_1,             $
                         data_format ,          $
                         center_sector, $
                         l_setup, $
                         adjFlag, $
                         l_oa, $
                         orig_start_udf_tme, $ 
                         new_udf_tme, $
                         (*l_event.O_n_in_peak_extra_e0), $
                         l_setup.l_spin_ang_zero
   
         (*l_event.O_n_in_peak_e0) =  event_hmd_1

         scale_factor = (*l_event.O_scale_factor_e0)
         l_HndlMissData, orig_tme,              $
                         hmd_tme,               $
                         nspins,                $
                         scale_factor,        $
                         event_hmd_1,             $
                         data_format ,          $
                         center_sector, $
                         l_setup, $
                         adjFlag, $
                         l_oa, $
                         orig_start_udf_tme, $ 
                         new_udf_tme, $
                         (*l_event.O_scale_factor_extra_e0), $
                         l_setup.l_spin_ang_zero

         (*l_event.O_scale_factor_e0) =  event_hmd_1

      endif


      ;----------------------------------------------------
      ; oxygen energy1
      ;----------------------------------------------------
      if (l_event_setup.energy1 eq 'Y') then begin
         event_raw = (*l_event.oxygen_raw_e1) 
         l_HndlMissData, orig_tme,              $
                         hmd_tme,               $
                         nspins,                $
                         event_raw,             $
                         event_hmd,             $
                         data_format ,          $
                         center_sector, $
                         l_setup, $
                         adjFlag, $
                         l_oa, $
                         orig_start_udf_tme, $ 
                         new_udf_tme, $
                         (*l_event.oxygen_raw_extra_e1), $
                         l_setup.l_spin_ang_zero
 

         (*l_event.oxygen_raw_e1) = event_hmd


         event_corr = (*l_event.corrected_O_e1) 
            l_HndlMissData, orig_tme,              $
                         hmd_tme,               $
                         nspins,                $
                         event_corr,            $
                         event_hmd,             $
                         data_format ,          $
                         center_sector, $
                         l_setup, $
                         adjFlag, $
                         l_oa, $
                         orig_start_udf_tme, $ 
                         new_udf_tme, $
                         (*l_event.corrected_O_extra_e1), $
                         l_setup.l_spin_ang_zero

         (*l_event.corrected_O_e1) =  event_hmd

         events_n_sect = (*l_event.O_sec_events_e1)
         l_HndlMissData, orig_tme,              $
                         hmd_tme,               $
                         nspins,                $
                         events_n_sect,         $
                         event_hmd_1,             $
                         data_format ,          $
                         center_sector, $
                         l_setup, $
                         adjFlag, $
                         l_oa, $
                         orig_start_udf_tme, $ 
                         new_udf_tme, $
                         (*l_event.O_sec_events_extra_e1), $
                         l_setup.l_spin_ang_zero
   
         (*l_event.O_sec_events_e1) =  event_hmd_1


         nevents_n_peak = (*l_event.O_n_in_peak_e1)
         l_HndlMissData, orig_tme,              $
                         hmd_tme,               $
                         nspins,                $
                         nevents_n_peak,        $
                         event_hmd_1,             $
                         data_format ,          $
                         center_sector, $
                         l_setup, $
                         adjFlag, $
                         l_oa, $
                         orig_start_udf_tme, $ 
                         new_udf_tme, $
                         (*l_event.O_n_in_peak_extra_e1), $
                         l_setup.l_spin_ang_zero
   
         (*l_event.O_n_in_peak_e1) =  event_hmd_1

         scale_factor = (*l_event.O_scale_factor_e1)
         l_HndlMissData, orig_tme,              $
                         hmd_tme,               $
                         nspins,                $
                         scale_factor,        $
                         event_hmd_1,             $
                         data_format ,          $
                         center_sector, $
                         l_setup, $
                         adjFlag, $
                         l_oa, $
                         orig_start_udf_tme, $ 
                         new_udf_tme, $
                         (*l_event.O_scale_factor_extra_e1), $
                         l_setup.l_spin_ang_zero

         (*l_event.O_scale_factor_e1) =  event_hmd_1
      endif


      ;----------------------------------------------------
      ; oxygen energy2
      ;----------------------------------------------------
      if (l_event_setup.energy2 eq 'Y') then begin
         event_raw = (*l_event.oxygen_raw_e2) 
         l_HndlMissData, orig_tme,              $
                         hmd_tme,               $
                         nspins,                $
                         event_raw,             $
                         event_hmd,             $
                         data_format ,          $
                         center_sector, $
                         l_setup, $
                         adjFlag, $
                         l_oa, $
                         orig_start_udf_tme, $ 
                         new_udf_tme, $
                         (*l_event.oxygen_raw_extra_e2), $
                         l_setup.l_spin_ang_zero
 

         (*l_event.oxygen_raw_e2) = event_hmd


         event_corr = (*l_event.corrected_O_e2) 
            l_HndlMissData, orig_tme,              $
                         hmd_tme,               $
                         nspins,                $
                         event_corr,            $
                         event_hmd,             $
                         data_format ,          $
                         center_sector, $
                         l_setup, $
                         adjFlag, $
                         l_oa, $
                         orig_start_udf_tme, $ 
                         new_udf_tme, $
                         (*l_event.corrected_O_extra_e2), $
                         l_setup.l_spin_ang_zero

         (*l_event.corrected_O_e2) =  event_hmd

         events_n_sect = (*l_event.O_sec_events_e2)
         l_HndlMissData, orig_tme,              $
                         hmd_tme,               $
                         nspins,                $
                         events_n_sect,         $
                         event_hmd_1,             $
                         data_format ,          $
                         center_sector, $
                         l_setup, $
                         adjFlag, $
                         l_oa, $
                         orig_start_udf_tme, $ 
                         new_udf_tme, $
                         (*l_event.O_sec_events_extra_e2), $
                         l_setup.l_spin_ang_zero
   
         (*l_event.O_sec_events_e2) =  event_hmd_1


         nevents_n_peak = (*l_event.O_n_in_peak_e2)
         l_HndlMissData, orig_tme,              $
                         hmd_tme,               $
                         nspins,                $
                         nevents_n_peak,        $
                         event_hmd_1,             $
                         data_format ,          $
                         center_sector, $
                         l_setup, $
                         adjFlag, $
                         l_oa, $
                         orig_start_udf_tme, $ 
                         new_udf_tme, $
                         (*l_event.O_n_in_peak_extra_e2), $
                         l_setup.l_spin_ang_zero
   
         (*l_event.O_n_in_peak_e2) =  event_hmd_1

         scale_factor = (*l_event.O_scale_factor_e2)
         l_HndlMissData, orig_tme,              $
                         hmd_tme,               $
                         nspins,                $
                         scale_factor,        $
                         event_hmd_1,             $
                         data_format ,          $
                         center_sector, $
                         l_setup, $
                         adjFlag, $
                         l_oa, $
                         orig_start_udf_tme, $ 
                         new_udf_tme, $
                         (*l_event.O_scale_factor_extra_e2), $
                         l_setup.l_spin_ang_zero

         (*l_event.O_scale_factor_e2) =  event_hmd_1
      endif


      Oe0 = (*l_event.oxygen_e0)
      l_HndlMissData, orig_tme,              $
                      hmd_tme,               $
                      nspins,                $
                      Oe0,        $
                      event_hmd_1,             $
                      data_format ,          $
                      center_sector, $
                      l_setup, $
                      adjFlag, $
                      l_oa, $
                      orig_start_udf_tme, $ 
                      new_udf_tme, $
                      (*l_event.oxygen_e0_extra), $
                      l_setup.l_spin_ang_zero

      (*l_event.oxygen_e0) =  event_hmd_1


      Oe1 = (*l_event.oxygen_e1)
      l_HndlMissData, orig_tme,              $
                      hmd_tme,               $
                      nspins,                $
                      Oe1,        $
                      event_hmd_1,             $
                      data_format ,          $
                      center_sector, $
                      l_setup, $
                      adjFlag, $
                      l_oa, $
                      orig_start_udf_tme, $ 
                      new_udf_tme, $
                      (*l_event.oxygen_e1_extra), $
                      l_setup.l_spin_ang_zero

      (*l_event.oxygen_e1) =  event_hmd_1



      Oe2 = (*l_event.oxygen_e2)
      l_HndlMissData, orig_tme,              $
                      hmd_tme,               $
                      nspins,                $
                      Oe2,        $
                      event_hmd_1,             $
                      data_format ,          $
                      center_sector, $
                      l_setup, $
                      adjFlag, $
                      l_oa, $
                      orig_start_udf_tme, $ 
                      new_udf_tme, $
                      (*l_event.oxygen_e2_extra), $
                      l_setup.l_spin_ang_zero

      (*l_event.oxygen_e2) =  event_hmd_1


   endif 
 

   ptr = ptr_new (new_udf_tme)
   l_event.bkcUdfTimePtr = ptr 

   (*l_event.filledStartUdfTimePtr) = new_udf_tme 
   (*l_event.filledTimePtr) = hmd_tme


return
END


;-------------------------------------------------------------------------------
;  Procedure: l_EventConvPU
;
;  Description: Determines which conversion units were selected to be applied.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_setup               struct        I        general setup parameters
;  input                 float []      I        input data arary
;  output                float []      O        output data array
;  mass                  string        I        hydrogen or oxygen
;  eV                    long          I        energy range
;  cntRate               float []      O        data in count rate 
;  flux_ofd              long          I        output file descriptor
;  EXTRA                 long          I        if extra spins were read
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------

PRO l_EventConvPU, l_setup, input, output, mass, eV, cntRate, flux_ofd, EXTRA

   tmp = input
   output = input

   if (l_setup.l_count_rate eq 'Y') then begin
       l_Conv2CntRate, tmp, output, 'SECTOR'
       tmp = output
       cntRate = output
   endif

   if (l_setup.l_flux eq 'Y') then begin
       l_ConvFlux, tmp, output, mass, eV, l_setup.l_start_dtime, state, nom_eff, eff  
       if (EXTRA eq 'NO') then begin
          if (l_setup.l_dumps eq 'Y') then begin
             l_WriteEff2File, flux_ofd, mass, eV, state, nom_eff, eff
          endif
       endif
   endif
   


RETURN
END

;-------------------------------------------------------------------------------
;  Procedure: l_Event_Conv2Units
;
;  Description:  Applies the conversion algorithm to the data.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;   None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_setup               struct        I        general setup parameters
;  l_event_setup         struct        I        event processing & plotting parameters
;  l_event               struct *      I        pointers to event data structures
;  l_pltpos              struct        I        general plotting parameters
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;     PLOTDEFS         MAX_PLOTS        int         G         max # plots per page
;                      MAX_WEB_SPINS    int         G         max # spins per plot
;                                                             when using web interface
;                      NO_CNTS          double      G         no counts indicator
;                      NO_DATA          double      G         no data indicator
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------

PRO l_Event_Conv2Units, l_setup, l_event_setup, l_event, l_pltpos
COMMON PLOTDEFS, MAX_PLOTS, MAX_WEB_SPINS, NO_CNTS, NO_DATA

    flux_ofd = 0L
    flux_filename = ''

    if (l_setup.l_flux eq 'Y') then begin
       l_setup.l_filedesc = 'eff'
       l_GenFileName, l_setup, '0'
       flux_filename = l_setup.l_prod_dst + '/' + $
             l_setup.l_filename + '.dmp'
        openw, flux_ofd, flux_filename, /get_lun 
        l_DumpHdrInfo, flux_ofd, l_setup, l_pltpos, flux_filename
        printf, flux_ofd, 'Data    :   Event'
    endif

    if (l_event_setup.hydrogen eq 'Y') then begin
    ;-----------------------------------------------
    ; HYDROGEN
    ;-----------------------------------------------
    mass = 'hydrogen'

    ;-----------------------------------------------
    ; energy 0
    ;-----------------------------------------------
    l_EventConvPU, l_setup, (*l_event.corrected_H_e0), $
                     output, mass, $
                     l_event_setup.he0_eV, cntRate, flux_ofd, 'NO'

    if (ptr_valid (l_event.hydrogen_e0_flux)) then ptr_free, l_event.hydrogen_e0_flux
    ptr = ptr_new (output, /NO_COPY)
    l_event.hydrogen_e0_flux = ptr

    if (ptr_valid (l_event.corr_scl_H_e0)) then ptr_free, l_event.corr_scl_H_e0
    ptr = ptr_new (cntRate, /NO_COPY)
    l_event.corr_scl_H_e0 = ptr

    if (ptr_valid (l_event.hydrogen_e0_extra)) then begin
       l_EventConvPU, l_setup, (*l_event.hydrogen_e0_extra), $
                        output, mass, $
                        l_event_setup.he0_eV, cntRate, flux_ofd, 'YES'

       ptr_free, l_event.hydrogen_e0_extra
       ptr = ptr_new (output, /NO_COPY)
       l_event.hydrogen_e0_extra = ptr
    endif

    ;-----------------------------------------------
    ; energy 1
    ;-----------------------------------------------
    l_EventConvPU, l_setup, (*l_event.corrected_H_e1), $
                     output, mass, $
                     l_event_setup.he1_eV, cntRate, flux_ofd, 'NO'

    if (ptr_valid ( l_event.hydrogen_e1_flux)) then ptr_free, l_event.hydrogen_e1_flux
    ptr = ptr_new (output, /NO_COPY)
    l_event.hydrogen_e1_flux = ptr

    if (ptr_valid ( l_event.corr_scl_H_e1)) then ptr_free, l_event.corr_scl_H_e1
    ptr = ptr_new (cntRate, /NO_COPY)
    l_event.corr_scl_H_e1 = ptr

    if (ptr_valid (l_event.hydrogen_e1_extra)) then begin
       l_EventConvPU, l_setup, (*l_event.hydrogen_e1_extra), $
                        output, mass, $
                        l_event_setup.he1_eV, cntRate, flux_ofd, 'YES'

       ptr_free, l_event.hydrogen_e1_extra
       ptr = ptr_new (output, /NO_COPY)
       l_event.hydrogen_e1_extra= ptr
    endif

    ;-----------------------------------------------
    ; energy 2
    ;-----------------------------------------------
    l_EventConvPU, l_setup, (*l_event.corrected_H_e2), $
                   output, mass, $
                   l_event_setup.he2_eV, cntRate, flux_ofd, 'NO'

    if (ptr_valid (l_event.hydrogen_e2_flux)) then ptr_free, l_event.hydrogen_e2_flux
    ptr = ptr_new (output, /NO_COPY)
    l_event.hydrogen_e2_flux = ptr

    if (ptr_valid (l_event.corr_scl_H_e2)) then ptr_free, l_event.corr_scl_H_e2
    ptr = ptr_new (cntRate, /NO_COPY)
    l_event.corr_scl_H_e2 = ptr

    if (ptr_valid (l_event.hydrogen_e2_extra)) then begin
       l_EventConvPU, l_setup, (*l_event.hydrogen_e2_extra), $
                      output, mass, $
                      l_event_setup.he2_eV, cntRate, flux_ofd, 'YES'

       ptr_free, l_event.hydrogen_e2_extra
       ptr = ptr_new (output, /NO_COPY)
       l_event.hydrogen_e2_extra = ptr
    endif


    l_Conv2CntRate, (*l_event.corrected_H), output, 'SECTOR'
    if (ptr_valid (l_event.corr_scl_H)) then ptr_free, l_event.corr_scl_H
    ptr = ptr_new (output, /NO_COPY)
    l_event.corr_scl_H = ptr


    endif

    if (l_event_setup.oxygen eq 'Y') then begin

    ;-----------------------------------------------
    ; OXYGEN
    ;-----------------------------------------------
    mass = 'oxygen'

    ;-----------------------------------------------
    ; energy 0
    ;-----------------------------------------------
    l_EventConvPU, l_setup, (*l_event.corrected_O_e0), $
                     output, mass, $
                     l_event_setup.oe0_eV, cntRate, flux_ofd, 'NO'

    if (ptr_valid (l_event.oxygen_e0_flux)) then ptr_free,l_event.oxygen_e0_flux
    ptr = ptr_new (output, /NO_COPY)
    l_event.oxygen_e0_flux = ptr

    if (ptr_valid (l_event.corr_scl_O_e0)) then ptr_free,l_event.corr_scl_O_e0
    ptr = ptr_new (cntRate, /NO_COPY)
    l_event.corr_scl_O_e0 = ptr

    if (ptr_valid (l_event.oxygen_e0_extra)) then begin
       l_EventConvPU, l_setup, (*l_event.oxygen_e0_extra), $
                        output, mass, $
                        l_event_setup.oe0_eV, cntRate, flux_ofd, 'YES'

       ptr_free, l_event.oxygen_e0_extra
       ptr = ptr_new (output, /NO_COPY)
       l_event.oxygen_e0_extra = ptr
    endif

    ;-----------------------------------------------
    ; energy 1
    ;-----------------------------------------------
    l_EventConvPU, l_setup, (*l_event.corrected_O_e1), $
                   output, mass, $
                   l_event_setup.oe1_eV, cntRate, flux_ofd, 'NO'

    if (ptr_valid (l_event.oxygen_e1_flux)) then ptr_free, l_event.oxygen_e1_flux
    ptr = ptr_new (output, /NO_COPY)
    l_event.oxygen_e1_flux = ptr

    if (ptr_valid (l_event.corr_scl_O_e1)) then ptr_free, l_event.corr_scl_O_e1
    ptr = ptr_new (cntRate, /NO_COPY)
    l_event.corr_scl_O_e1 = ptr

    if (ptr_valid (l_event.oxygen_e1_extra)) then begin
       l_EventConvPU, l_setup, (*l_event.oxygen_e1_extra), $
                      output, mass, $
                      l_event_setup.oe1_eV, cntRate, flux_ofd, 'YES'

       ptr_free, l_event.oxygen_e1_extra
       ptr = ptr_new (output, /NO_COPY)
       l_event.oxygen_e1_extra = ptr
    endif


    ;-----------------------------------------------
    ; energy 2
    ;-----------------------------------------------
    l_EventConvPU, l_setup, (*l_event.corrected_O_e2), $
                   output, mass, $
                   l_event_setup.oe2_eV, cntRate, flux_ofd, 'NO'

    if (ptr_valid (l_event.oxygen_e2_flux)) then ptr_free, l_event.oxygen_e2_flux
    ptr = ptr_new (output, /NO_COPY)
    l_event.oxygen_e2_flux = ptr

    if (ptr_valid (l_event.corr_scl_O_e2)) then ptr_free, l_event.corr_scl_O_e2
    ptr = ptr_new (cntRate, /NO_COPY)
    l_event.corr_scl_O_e2 = ptr

    if (ptr_valid (l_event.oxygen_e2_extra)) then begin
       l_EventConvPU, l_setup, (*l_event.oxygen_e2_extra), $
                      output, mass, $
                      l_event_setup.oe2_eV, cntRate, flux_ofd, 'YES'

       ptr_free, l_event.oxygen_e2_extra
       ptr = ptr_new (output, /NO_COPY)
       l_event.oxygen_e2_extra = ptr
    endif


    l_Conv2CntRate, (*l_event.corrected_O), output, 'SECTOR'
    if (ptr_valid (l_event.corr_scl_O)) then ptr_free, l_event.corr_scl_O
    ptr = ptr_new (output, /NO_COPY)
    l_event.corr_scl_O = ptr

  endif

  if (l_setup.l_flux eq 'Y') then begin
     free_lun, flux_ofd
     close, flux_ofd
  endif


RETURN
END

;-------------------------------------------------------------------------------
;  Procedure : l_EventPlot
;
;  Description:  Plot the event data.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_event_setup         struct        I        event plot specific parameters
;  l_event               struct        I        event specific data plot parameters
;  l_pltpos              struct        I        generic plot parameters
;  nplots                long          I        total number of plots to be plotted
;  windx                 long[]        I        window index
;  pages                 long          I        total number of pages needed to
;                                               fit plots       
;  plotted               long          I        total number of plots currently plotted
;  l_setup               struct        I        generic setup parameters
;  l_oa                  struct        I        OA specific parameters/attributes
;  data_format           long          I        one dim array of all data [spin * spin sector]
;                                               or 2 dim array of [spin, spin sector]
;  l_rgb_table           struct        I        red, green, blue color table values
;  colorbar              float[]       I        color bar values
;  l_hskp                struct *      I        pointers to HSKP data structures
;  hmd_tme               double[]      I        time corresponding to data gaps
;  event_min             float         I        minimum log value
;  event_max             float         I        maximum log value
;  title, title1         string        I        plot titles
;  
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;     PLOTDEFS         MAX_PLOTS        int         G         max # plots per page
;                      MAX_WEB_SPINS    int         G         max # spins per plot
;                                                             when using web interface
;                      NO_CNTS          double      G         no counts indicator
;                      NO_DATA          double      G         no data indicator
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                08/01          v1.0.7          Original implementation
;  ELH                09/26/01       v1.0.7          Added l_DetermineSunDir 
;  ELH                12/03/01       v1.0.7          Removed all l_HndlMissData 
;-------------------------------------------------------------------------------

PRO l_EventPlot, l_event_setup, event_data, l_pltpos, $
nplots, windx, pages, plotted, l_setup , l_oa, $
data_format, l_rgb_table, colorbar, l_hskp, hmd_tme, event_min, event_max, $
title, title1
COMMON PLOTDEFS, MAX_PLOTS, MAX_WEB_SPINS, NO_CNTS, NO_DATA



   if (l_setup.l_angle_zero eq 'E') then begin
      ytitle_frame = '!5Earth Cntrd Azi'
   endif else begin
      if (l_setup.l_angle_zero eq 'S') then begin
          ytitle_frame = '!5Sun Cntrd Azi'
       endif else begin
          if (l_setup.l_angle_zero eq 'M') then begin
             ytitle_frame = '!5Moon Cntrd Azi'
          endif else begin
            ytitle_frame  = '!5Spin Angle'
         endelse
      endelse
   endelse


   ztitle = ' '
   cnt_ztitle = ' '
   flx_ztitle = ' '
   log_ztitle = ' '

   if (l_setup.l_log eq 'Y') then log_ztitle = 'log '
   if (l_setup.l_count_rate eq 'Y') then begin
      cnt_ztitle = 'cnt Rate[Hz]'
      if (l_setup.l_flux eq 'Y') then begin
         cnt_ztitle = ' '
         flx_ztitle = 'atoms/cm2/s'
      endif
    endif else begin
      if (l_setup.l_flux eq 'Y') then begin
         cnt_ztitle = ' '
         flx_ztitle = 'atoms/cm2/s'
      endif else begin
         ztitle = 'linear cnts'
      endelse
   endelse

   zaxis_title = ztitle + log_ztitle + flx_ztitle + cnt_ztitle


   if (l_setup.l_auto_scale eq 'Y') then begin
       l_setup.l_color_min = event_min
       l_setup.l_color_max = event_max
   endif else begin
      if (l_setup.l_sep_plot_scale eq 'Y') then begin
       l_setup.l_color_min = l_event_setup.color_min
       l_setup.l_color_max = l_event_setup.color_max
      endif
   endelse


   if (l_setup.l_line_plots eq 'Y') then begin

      ;---------------------------------------------------
      ; generate a line plot
      ;---------------------------------------------------

      diff = event_max - event_min
      if (diff gt 1) then begin
         new_max = (event_max/4.) + event_max
         event_max= new_max
      endif else begin
         event_max = event_max + diff
      endelse

      l_PlotStuff, nplots,                   $
                   hmd_tme,                  $
                   event_data,               $
                   event_min,                $
                   event_max,                $
                   zaxis_title,              $
                   l_pltpos,                 $
                   plotted,                  $
                   l_oa,                     $
                   l_setup.l_start_dtime


   endif else begin  ; a spectrogram plot


   ;---------------------------------------------------
   ; generate a spectrogram
   ;---------------------------------------------------

   l_ScaleColor, event_data, event_scaled, $
                 l_setup.l_color_min, $
                 l_setup.l_color_max , $
                 l_setup.l_color_tbl

   ;------------------------------------------------------------
   ; correct the image positioning, rotate 240 degrees
   ;------------------------------------------------------------
   event_rot = rotate(event_scaled, 3)

   yrng=[l_pltpos.min_spin_ang,l_pltpos.max_spin_ang]

   img_pos = [l_pltpos.xpltmn,l_pltpos.ypltmn,$
              l_pltpos.xpltmx,l_pltpos.ypltmx]

   position = double(img_pos)

   xstart = position[0]
   ystart = position[1]
   ximage = l_pltpos.axis_size/ l_pltpos.xin
   yimage = l_pltpos.ypltmx - l_pltpos.ypltmn
   xsize  = round(l_pltpos.window_width * ximage)
   ysize  = round(l_pltpos.window_height * yimage)

   event_spect = congrid (event_rot, $
                          xsize, $
                          ysize)

   ;---------------------------------------------------
   ; smooth the image
   ;---------------------------------------------------
   if (l_setup.l_data_type eq 'l_zc3') or (l_setup.l_smooth_data eq 'Y') then begin 
      TV, smooth(event_spect, 3, /edge_truncate), xstart, ystart, $
          XSIZE=ximage, YSIZE=yimage, /ORDER, $
          /NORMAL
   endif else begin
      TV, event_spect, xstart, ystart, $
          XSIZE=ximage, YSIZE=yimage, /ORDER, $
          /NORMAL
   endelse

   ;---------------------------------------------------
   ; draw the earth lines
   ;---------------------------------------------------
   if (l_setup.l_earth_markers eq 'Y') then begin
      l_CalcEarthDirection, l_oa, l_setup, img_pos
   endif

   ;---------------------------------------------------
   ; draw the sun direction
   ;---------------------------------------------------
   if (l_setup.l_sun_direction eq 'Y') then begin
      l_CalcSunDirection, l_oa, l_setup, img_pos
   endif

   ;---------------------------------------------------
   ; draw the ram direction
   ;---------------------------------------------------
   if (l_setup.l_ram_direction eq 'Y') then begin
      l_CalcSunPulse, l_oa, ram_sector, 'R'
      ram_sector = ram_sector/8.
      if NOT (ptr_valid(l_oa.lena_ram_sectPtr)) then begin
         ptr = ptr_new (ram_sector, /NO_COPY)
         l_oa.lena_ram_sectPtr = ptr
      endif
      l_CalcRamDirection, l_oa, l_setup, img_pos
   endif

   ;---------------------------------------------------
   ; draw the B direction
   ;---------------------------------------------------
   if (l_setup.l_B_direction eq 'Y') then begin
      l_CallT96Model, l_oa, l_setup
      l_CalcSunPulse, l_oa, B_sector, 'B'
      B_sector = B_sector/8.
      if NOT (ptr_valid(l_oa.lena_mag_fieldPtr)) then begin
         ptr = ptr_new (B_sector, /NO_COPY)
         l_oa.lena_mag_fieldPtr = ptr
      endif
      l_CalcBDirection, l_oa, l_setup, img_pos
   endif



    bar_pos = [l_pltpos.xpltmx+0.015, $
               l_pltpos.ypltmn, $
               l_pltpos.xpltmx+0.025, $
               l_pltpos.ypltmx]

    bar_info = {AXIS_DATA, char_size: l_pltpos.plot_font, $
                ytitle:zaxis_title, $
                yformat:'(F6.1)', $
                yrange:[l_setup.l_color_min, $
                l_setup.l_color_max]}

    l_DrawColorbar, colorbar, l_setup, bar_pos, l_pltpos, $
                    RAXIS=bar_info


    l_PlotFrame, nplots, hmd_tme, yrng, img_pos, l_pltpos, $
                 plotted, l_oa, ytitle_frame, l_setup.l_start_dtime, $
                 l_hskp

    endelse


    yctr = ((l_pltpos.ypltmn + l_pltpos.ypltmx)/2.0) + 0.015
    xyouts, 0.015, yctr, title, size=l_pltpos.sm_font, $
            /NORMAL, alignment = 0.5, orientation=90.0

    xyouts, 0.035, yctr, title1, size=l_pltpos.sm_font, $
            /NORMAL, alignment = 0.5, orientation=90.0


    l_pltpos.ypltmx = l_pltpos.ypltmn - l_pltpos.yspace
    l_pltpos.ypltmn = l_pltpos.ypltmx - l_pltpos.ydel
    plotted = plotted + 1
    l_ChckCompPg, plotted, nplots, pages, $
                  l_setup, windx, l_pltpos, l_rgb_table


RETURN
END


;-------------------------------------------------------------------------------
;  Procedure: SyncEvent2SectEvnts
;
;  Description: Time synchronize the event data to the sectored events data.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_evsp                struct        I        sectored event/sun pulse
;                                               specific data fields
;  n_evsp                long          I        total # of evsp records
;  l_event               struct        I        event specific data fields
;  n_event               long          I        total # of event records
;  event_cnts            double[]       O        # of events per sector synced to 
;                                               event data
;  sunp                  double[]       O        sun pulse sectors synced 
;                                               to event data
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;-------------------------------------------------------------------------------

PRO SyncEvent2SectEvnts, l_evsp, n_evsp, l_event, n_event, event_cnts, sunp


   ;-----------------------------------------------------------
   ; sync the EVENT data to the SECTORED EVENTS/SUN PULSE data
   ; packet
   ;   . match the event packet time to each 
   ;     of the sesp time 
   ;
   ;   for the number of sesp items do
   ;     for the number of event items do
   ;        if the event time falls between the start &
   ;          stop time of the sesp then 
   ;             event data synced to sesp packet
   ;        else
   ;          check if there is a data gap
   ;          if there is a data gap then check the next event 
   ;              time to the current sesp time
   ;          else if there is no data gap then get the next
   ;              sesp item  
   ;          endelse
   ;        endelse
   ;     endfor
   ;   endfor
   ; 
   ;  NOTE: the sesp start and stop times are the entire 2 minute
   ;        spins 
   ;-----------------------------------------------------------

   last_ii = 0L
   last_jj = 0L
   indx    = 0L
   status  = 0L


   for jj = 0L, n_evsp - 1L do begin

      start_evsp_time = Convert((*l_evsp.startUdfTimePtr)[jj])
      stop_evsp_time = Convert((*l_evsp.stopUdfTimePtr)[jj])

      if (jj lt n_evsp - 1L) then begin
         next_start_evsp_time = Convert((*l_evsp.startUdfTimePtr)[jj+1])
      endif else begin
         next_start_evsp_time = Convert((*l_evsp.stopUdfTimePtr)[jj])
      endelse


       for ii = last_ii, n_event - 1L do begin

          start_event_time = Convert((*l_event.startUdfTimePtr)[ii])
          stop_event_time = Convert((*l_event.stopUdfTimePtr)[ii])

           if (start_event_time ge start_evsp_time) and $
              (start_event_time le stop_evsp_time) then begin


                if (start_event_time eq next_start_evsp_time) then begin
                 if (start_event_time ge start_evsp_time) and $
                    (start_event_time le stop_evsp_time) then begin
                       sector = (*l_event.azimuthPtr)[ii]
                       event_cnts[ii] = (*l_evsp.eventCntsPtr)[sector,jj]
                       sunp[ii] = (*l_evsp.sunPulsePtr)[sector,jj]
                       last_ii = ii
                   endif else begin 
                       last_ii = ii
                       goto, NEXT_PKT1
                   endelse
              endif else begin
                   sector = (*l_event.azimuthPtr)[ii]
                   event_cnts[ii] = (*l_evsp.eventCntsPtr)[sector,jj]
                   sunp[ii] = (*l_evsp.sunPulsePtr)[sector,jj]
                   last_ii = ii
              endelse

            endif else begin
              status = l_DataGap (next_start_evsp_time, stop_evsp_time, $
                                  start_event_time)
              if (status eq 0L) then begin
                 last_ii = ii
                 goto, NEXT_PKT1
              endif else begin
                sector = (*l_event.azimuthPtr)[ii]
                event_cnts[ii] = (*l_evsp.eventCntsPtr)[sector,jj]
                sunp[ii] = (*l_evsp.sunPulsePtr)[sector,jj]
                last_ii = ii
              endelse
            endelse

       endfor
NEXT_PKT1:
   endfor


RETURN
END


;-------------------------------------------------------------------------------
;  Procedure: SyncEvent2OA
;
;  Description: Time synchronize the event data to OA data.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_oa                  struct        I        OA data specific parameters
;  n_oa                  long          I        total # OA records
;  l_event               struct        I        event data specific parameters
;  n_event               long          I        total # event records
;  sun_pulse             long[]        I        sun pulse sectors calculated from
;                                               the OA data
;  sunp                  long[]        O        sun pulse sectors synced to event
;                                               data
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;-------------------------------------------------------------------------------

PRO SyncEvent2OA, l_oa, n_oa, l_event, n_event, sun_pulse, sunp


   ;-----------------------------------------------------------
   ; sync the EVENT data to the OA SUN PULSE data
   ; packet
   ;   . match each of the times in the event packet to each time 
   ;     in the oa data
   ;
   ;   for the number of oa items do
   ;     for the number of event items do
   ;        if the event time falls between the start &
   ;          stop time of the oa then 
   ;             event data synced to oa packet
   ;        else
   ;          check if there is a data gap
   ;          if there is a data gap then check the next event 
   ;              time to the current oa time
   ;          else if there is no data gap then get the next
   ;              oa item  
   ;          endelse
   ;        endelse
   ;     endfor
   ;   endfor
   ; 
   ;-----------------------------------------------------------

   last_ii = 0L
   last_jj = 0L
   indx    = 0L
   status  = 0L
   ii      = 0L
   jj      = 0L


   for ii = 0L, n_event - 1L do begin

       start_event_time = Convert((*l_event.startUdfTimePtr)[ii])
       stop_event_time = Convert((*l_event.stopUdfTimePtr)[ii])

       for jj = last_jj, n_oa - 1L do begin

          start_oa_time = Convert((*l_oa.startUdfTimePtr)[jj])
          stop_oa_time = Convert((*l_oa.stopUdfTimePtr)[jj])
          next_start_oa_time = stop_oa_time

          ;----------------------------------------------------------
          ; if the start event time falls between the OA spin period
          ;----------------------------------------------------------
          if (start_event_time ge start_oa_time) and $
             (start_event_time lt stop_oa_time) then begin
               sector = (*l_event.azimuthPtr)[ii]
               ;-----------------------------------------------------
               ; sun_pulse is in degrees, need to convert to sector
               ;-----------------------------------------------------
               sunp[ii] = sun_pulse[jj]/8.0
               last_jj = jj
               goto, FSW_NEXT_PKT1
          endif else begin
               ;----------------------------------------------------------
               ; if the stop event time falls between the OA spin period
               ;----------------------------------------------------------
               if ((stop_event_time ge start_oa_time) AND $
                   (stop_event_time lt stop_oa_time)) then begin 
                   sector = (*l_event.azimuthPtr)[ii]
                   ;-----------------------------------------------------
                   ; sun_pulse is in degrees, need to convert to sector
                   ;-----------------------------------------------------
                   sunp[ii] = sun_pulse[jj]/8.0
                   last_jj = jj 
                   goto, FSW_NEXT_PKT1
               endif
          endelse
          ;----------------------------------------------------------
          ; if event times do not fall in between a start/stop OA
          ; period, then no OA data corresponds that the event
          ; array filled with -9 for no data
          ;----------------------------------------------------------

       endfor
FSW_NEXT_PKT1:
   endfor


RETURN
END


;-------------------------------------------------------------------------------
;  Procedure:  SyncEvent2Binned
;
;  Description: Time synchronize the event data to the binned data.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_binned              struct        I        image/binned data parameters
;  l_event               struct        I        event data parameters
;  n_binn                long          I        total # of image records
;  n_event               long          I        total # of event records
;  binn_cnts             double[]       O        summed of binned counts synced
;                                               to the event data
;  steering              double[]       O        steering from HSKP synced to
;                                               the event data
;  
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;-------------------------------------------------------------------------------

PRO SyncEvent2Binned, l_binned, l_event, n_binn, n_event, binn_cnts, steering

   ;--------------------------------------------------------------------
   ; sync the event data to the BINNED SUM COUNTS and STEERING
   ;   . steering from housekeeping is retrieved synced to binned data
   ;   . match each item in the event packet to each time 
   ;     in the binned time 
   ;
   ; NOTE: UDF time of binned data are not 2 minute spins; therefore,
   ;       must use the start bin time of the current record as the
   ;       start time and the start bin time of the next record as the
   ;       stop bin time
   ;--------------------------------------------------------------------

   last_ii = 0L
   last_jj = 0L
   indx    = 0L
   status  = 0L
   ii      = 0L
   jj      = 0L


   for jj = 0L, n_binn - 1L do begin

       start_binn_time = Convert((*l_binned.startUdfTimePtr)[jj])
       stop_binn_time = Convert((*l_binned.stopUdfTimePtr)[jj])

       if (jj lt n_binn - 1L) then begin
          next_start_binn_time = Convert((*l_binned.startUdfTimePtr)[jj+1])
       endif else begin
          next_start_binn_time = Convert((*l_binned.stopUdfTimePtr)[jj]) 
       endelse

       for ii = last_ii, n_event - 1L do begin
           start_event_time = Convert((*l_event.startUdfTimePtr)[ii])
           stop_event_time = Convert((*l_event.stopUdfTimePtr)[ii])

           ;--------------------------------------------------------------------
           ; determine if event data falls within this spin
           ;--------------------------------------------------------------------
           if (start_event_time ge start_binn_time) and $
              (start_event_time le next_start_binn_time) then begin

              ;--------------------------------------------------------------------
              ; if the event time is equal to the boundary end time  
              ;--------------------------------------------------------------------
              if (start_event_time eq next_start_binn_time) then begin
                 ;--------------------------------------------------------------------
                 ; check that the event data belongs in the current bin spin, if it does
                 ; then extract the data, else need to go to the next spin
                 ;--------------------------------------------------------------------
                 if (start_event_time ge start_binn_time) and $
                    (start_event_time le stop_binn_time) then begin
                       sector = (*l_event.azimuthPtr)[ii]
                       binn_cnts[ii] = (*l_binned.vePtr)[sector,jj]
                       steering[ii] = (*l_binned.steervPtr)[jj] 
                       last_ii = ii
                   endif else begin 
                       last_ii = ii
                       goto, NEXT_PKT2
                   endelse
              endif else begin
                   sector = (*l_event.azimuthPtr)[ii]
                   binn_cnts[ii] = (*l_binned.vePtr)[sector,jj]
                   steering[ii] = (*l_binned.steervPtr)[jj] 
                   last_ii = ii
              endelse
            endif else begin
              ;--------------------------------------------------------------------
              ; does not fall within that particular spin period, check if there
              ; is a data gap and if the current event data belongs in the next
              ; spin, if status = 0, no data gap and data belongs in the next spin
              ; then get the next packet 
              ;--------------------------------------------------------------------
              status = l_DataGap (next_start_binn_time, stop_binn_time, $
                                  start_event_time)
              if (status eq 0L) then begin
                 last_ii = ii
                 goto, NEXT_PKT2
              endif else begin
                 sector = (*l_event.azimuthPtr)[ii]
                 binn_cnts[ii] = (*l_binned.vePtr)[sector,jj]
                 steering[ii] = (*l_binned.steervPtr)[jj] 
                 last_ii = ii
              endelse
            endelse
       endfor
NEXT_PKT2:
   endfor


RETURN
END


;-------------------------------------------------------------------------------
;  Procedure : l_EventDump_Bef0731 
;
;  Description: 
;     The event data before July 31, 2000.  The FSW upload occurred.
;     Before this date, there were no SECTORED EVENTS/SUN PULSE packets
;     available.  The sec_events will be filled with the sum_cnts column and
;     the sun_pulse column is calculated using the OA data. 
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_setup               struct        I        generic setup parameters
;  l_event               struct        I        event data parameters
;  l_oa                  struct        I        OA data parameters
;  sun_pulse             long[]        I        sun pulse sector values
;  l_binned              struct        I        image data parameters
;  l_pltpos              struct        I        generic plot parameters
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;     PLOTDEFS         MAX_PLOTS        int         G         max # plots per page
;                      MAX_WEB_SPINS    int         G         max # spins per plot
;                                                             when using web interface
;                      NO_CNTS          double      G         no counts indicator
;                      NO_DATA          double      G         no data indicator
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;-------------------------------------------------------------------------------


PRO l_EventDump_Bef0731, l_setup , l_event, l_oa, sun_pulse, l_binned, l_pltpos
COMMON PLOTDEFS, MAX_PLOTS, MAX_WEB_SPINS, NO_CNTS, NO_DATA

   ;-----------------------------------------
   ; generate the event file name
   ;-----------------------------------------
   l_GenFileName, l_setup, '0'
   outfile = l_setup.l_prod_dst + '/' + $
             l_setup.l_filename + '.dmp'

   ;-----------------------------------------
   ; open event file
   ;-----------------------------------------
   openw, ofd, outfile, /get_lun

   ;-----------------------------------------
   ; write standard dump header info
   ;-----------------------------------------
   l_pltpos.dmp_hdr_title = 'Event Dumps'
   l_DumpHdrInfo, ofd, l_setup, l_pltpos, outfile

   ;-----------------------------------------
   ; text documentation specific to data
   ; before FSW (7/31/00) 
   ;-----------------------------------------
   printf, ofd, '--------------- Data before Flight Software (FSW) Upload ---------------'
   printf, ofd, ' '
   printf, ofd, '   Column Definitions:  '
   printf, ofd, ' '
   printf, ofd, '      UTC        : Universal Time Coordinate (corrected)'
   printf, ofd, '      Azi        : Azimuthal spin sector (0-44)'
   printf, ofd, '      tof        : Time of Flight (event data)'
   printf, ofd, '      plr        : Polar bin (event data)'
   printf, ofd, '      eng        : Energy bin (event data)'
   printf, ofd, '      evnts/sect : The total number of events per spin sector,'
   printf, ofd, '                   saturates at 64 events'
   printf, ofd, '      sum counts : The count summed over all masses, energy bins,'
   printf, ofd, '                   and polar bins. (image/binned data)
   printf, ofd, '      sec evnts  : The actual number of events per spin sector, '
   printf, ofd, '                   sectored events packet not available, column'
   printf, ofd, '                   is the same as sum counts (sum of binned events).'
   printf, ofd, '      sun pulse  : Sun pulse sector calculated using OA data'
   printf, ofd, '      steering   : The steering controller CMD level (housekeeping data)'
   printf, ofd, ' '
   printf, ofd, '-------------------------------------------------------------------------'
   printf, ofd, ' '
   printf, ofd, ' '
   printf, ofd, ' '

   ;-----------------------------------------
   ; event column headers
   ;-----------------------------------------
   printf, ofd,'UTC','Azi','tof','plr','eng','event cnts', 'sum cnts', $
             'sec evnts','sun pulse', 'steering (dec)',  $
             format='((A8,TR4),4(A6,TR1), 4(A10,TR1), (A14))'

   printf, ofd,'---','---','---','---','---','----------', '--------', $
             '---------','---------', '--------------', $
             format='((A8,TR4),4(A6,TR1), 4(A10,TR1), (A14))'


   ;--------------------------------------------------
   ; determine the number of records in each packet
   ;--------------------------------------------------
   n_oa    = n_elements ((*l_oa.startUdfTimePtr)[*])
   n_event = l_event.nitems
   n_binn  = n_elements ((*l_binned.startUdfTimePtr)[*])
   n_str   = n_binn
   nele    = l_event.nitems

   ;--------------------------------------------------
   ; variables used to print the packet number 
   ;--------------------------------------------------
   pcnt      = 0L   ;** count of number of packets (spins) **
   prev_sect = 44L  ;** previous sector number **
   prev_event_time =  Convert ((*l_event.startUdfTimePtr)[0])


   ;-----------------------------------------------------------
   ; define local arrays to store data synced to event data
   ;-----------------------------------------------------------
   binn_cnts = make_array (l_event.nitems, /INT, value = -9)
   event_cnts= make_array (l_event.nitems, /INT, value = -9)
   sunp      = make_array (l_event.nitems, /INT, value = -9)
   steering  = make_array (l_event.nitems, /INT, value = -9)

   SyncEvent2OA, l_oa, n_oa, l_event, n_event, sun_pulse, sunp
   SyncEvent2Binned, l_binned, l_event, n_binn, n_event, binn_cnts, steering


   for indx = 0L, n_event - 1L do begin

       start_event_time =  (*l_event.startUdfTimePtr)[indx] 
       sector = (*l_event.azimuthPtr)[indx] 
       curr_sect = sector

       ;-------------------------------------------------------
       ; if the spin sector has zero events, UDF puts in filler
       ; values, don't print them to the file 
       ;-------------------------------------------------------
       fstatus = CheckFiller ((*l_event.tofPtr)[indx], $
                              (*l_event.polarPtr)[indx], $
                              (*l_event.energyPtr)[indx] )
 

       if (fstatus eq 1L) then begin
 
          ;-------------------------------------------------------
          ; if a new spin print out a packet separator
          ;-------------------------------------------------------
          start = Convert (start_event_time)
          ev_time_diff = start - prev_event_time
          if (curr_sect lt prev_sect) OR (ev_time_diff gt 120) then begin
            str = '------------------------------ packet #' + $
                  strcompress(string(pcnt),/remove_all) + $
                  '-------------------------------------'
            printf, ofd, str
            pcnt = pcnt + 1L
          endif 
 
          prev_sect = curr_sect
          prev_event_time = Convert(start_event_time)

          ;-------------------------------------------------------
          ; write the data to the text file
          ;-------------------------------------------------------
          tmestr = STRING ((*l_event.startUdfTimePtr)[indx].doy, $
                           (*l_event.startUdfTimePtr)[indx].hour,$
                           (*l_event.startUdfTimePtr)[indx].min, $
                           (*l_event.startUdfTimePtr)[indx].sec, $
                           FORMAT="(i3,' ',i2.2,':',i2.2,':',i2.2)")
 

          printf, ofd, $
                  tmestr, $                      ;** UTC **
                  sector, $                      ;** spin sector **
                  (*l_event.tofPtr)[indx], $     ;** time of flight **
                  (*l_event.polarPtr)[indx], $   ;** polar bin **
                  (*l_event.energyPtr)[indx], $  ;** energy level **
                  (*l_event.sectCntsPtr)[indx],$ ;** summed event cnts **
                  binn_cnts[indx], $      ;** binned summed cnts **
                  binn_cnts[indx], $      ;** binned summed cnts (sect/evnts)**
                  sunp[indx], $           ;** sun pulse sector **
                  steering[indx],  $             ;** steering controller **
                  format='((A12,TR2),4(I4,TR3), 4(I5,TR6), (I4))'
       endif 
   endfor

   close, ofd
   free_lun, ofd


RETURN
END


;-------------------------------------------------------------------------------
;  Procedure: l_EventDmpSpectData
;
;  Description: Dump the event spectrogram data to a text file.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  filedes               long          I        output file descriptor
;  l_event               struct *      I        pointers to even data structure
;  l_event_setup         struct        I        event processing & plotting parameters
;  l_setup               struct        I        general setup parameters
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------

PRO l_EventDmpSpectData, filedes, l_event, l_event_setup, l_setup

   nitems = n_elements (*l_event.filledStartUdfTimePtr)

   if (l_event_setup.merge eq 'Y') then begin
       if (l_event_setup.hydrogen eq 'Y') AND (l_event_setup.oxygen eq 'Y') then begin
          title = 'Event'
       endif else begin
         if (l_event_setup.hydrogen eq 'Y') then begin
            title = 'Event H+'
            if (l_event_setup.energy0 eq 'Y') then title = title + ' E0'
            if (l_event_setup.energy1 eq 'Y') then title = title + ' E1'
            if (l_event_setup.energy2 eq 'Y') then title = title + ' E2'
         endif else begin
             if (l_event_setup.oxygen eq 'Y') then begin
                title = 'Event O+'
                if (l_event_setup.energy0 eq 'Y') then title = title + ' E0'
                if (l_event_setup.energy1 eq 'Y') then title = title + ' E1'
                if (l_event_setup.energy2 eq 'Y') then title = title + ' E2'
             endif
         endelse
       endelse

       l_DumpData, filedes, title, (*l_event.amDmpPtr), (*l_event.filledStartUdfTimePtr), $
                   nitems, l_setup.l_num_spinsec, l_setup
   endif else begin
       if (l_event_setup.hydrogen eq 'Y') then begin
          if (l_event_setup.energy0 eq 'Y') then begin
             title = 'Event H+ E0'
             l_DumpData, filedes, title, (*l_event.hydrogen_e0), (*l_event.filledStartUdfTimePtr), $
                   nitems, l_setup.l_num_spinsec, l_setup
          endif
          if (l_event_setup.energy1 eq 'Y') then begin
             title = 'Event H+ E1'
             l_DumpData, filedes, title, (*l_event.hydrogen_e1), (*l_event.filledStartUdfTimePtr), $
                   nitems, l_setup.l_num_spinsec, l_setup
          endif
          if (l_event_setup.energy2 eq 'Y') then begin
             title = 'Event H+ E2'
             l_DumpData, filedes, title, (*l_event.hydrogen_e2), (*l_event.filledStartUdfTimePtr), $
                   nitems, l_setup.l_num_spinsec, l_setup
          endif
       endif
       if (l_event_setup.oxygen eq 'Y') then begin
          if (l_event_setup.energy0 eq 'Y') then begin
             title = 'Event O+ E0'
             l_DumpData, filedes, title, (*l_event.oxygen_e0), (*l_event.filledStartUdfTimePtr), $
                   nitems, l_setup.l_num_spinsec, l_setup
          endif
          if (l_event_setup.energy1 eq 'Y') then begin
             title = 'Event O+ E1'
             l_DumpData, filedes, title, (*l_event.oxygen_e1), (*l_event.filledStartUdfTimePtr), $
                   nitems, l_setup.l_num_spinsec, l_setup
          endif
          if (l_event_setup.energy2 eq 'Y') then begin
             title = 'Event O+ E2'
             l_DumpData, filedes, title, (*l_event.oxygen_e2), (*l_event.filledStartUdfTimePtr), $
                   nitems, l_setup.l_num_spinsec, l_setup
          endif
       endif
   endelse ;--- merge ---


RETURN
END


;-------------------------------------------------------------------------------
;  Procedure : l_EventDump
;
;  Description: 
;      After July 31, 2000.  The new software packet SECTORED
;      EVENTS/SUN PULSE packets were available. Using data from this
;      packet to fill the sec_events and sun pulse column.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_setup               struct        I        generic setup parameters
;  l_event               struct        I        event data parameters
;  l_Xstruct             struct        I        pointer to input structure
;  l_binned              struct        I        image data parameters
;  l_pltpos              struct        I        generic plot parameters
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;     PLOTDEFS         MAX_PLOTS        int         G         max # plots per page
;                      MAX_WEB_SPINS    int         G         max # spins per plot
;                                                             when using web interface
;                      NO_CNTS          double      G         no counts indicator
;                      NO_DATA          double      G         no data indicator
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------

PRO l_EventDump, l_setup , l_event, l_Xstruct, l_binned, l_pltpos
COMMON PLOTDEFS, MAX_PLOTS, MAX_WEB_SPINS, NO_CNTS, NO_DATA

   ;-----------------------------------------
   ; generate the event file name
   ;-----------------------------------------
   l_GenFileName, l_setup, '0'
   outfile = l_setup.l_prod_dst + '/' + $
             l_setup.l_filename + '.dmp'

   ;-----------------------------------------
   ; open event file
   ;-----------------------------------------
   openw, ofd, outfile, /get_lun

   ;-----------------------------------------
   ; write standard dump header info
   ;-----------------------------------------
   l_pltpos.dmp_hdr_title = 'Event Dumps'
   l_DumpHdrInfo, ofd, l_setup, l_pltpos, outfile

   ;-----------------------------------------
   ; text documentation specific to data
   ; after FSW (7/31/00) 
   ;-----------------------------------------
   printf, ofd, ' '
   printf, ofd, '---------------- Data After Flight Software (FSW) Upload ----------------'
   printf, ofd, ' '
   printf, ofd, '    Column Definitions:  '
   printf, ofd, ' '
   printf, ofd, '       UTC        : Universal Time Coordinate (corrected)'
   printf, ofd, '       Azi        : Azimuthal spin sector (0-44)'
   printf, ofd, '       tof        : Time of Flight (event data)'
   printf, ofd, '       plr        : Polar bin (event data)'
   printf, ofd, '       eng        : Energy bin (event data)'
   printf, ofd, '       evnts/sect : The total number of events per spin sector,'
   printf, ofd, '                    saturates at 64 events'
   printf, ofd, '       sum counts : The count summed over all masses, energy bins,'
   printf, ofd, '                    and polar bins. (image/binned data)
   printf, ofd, '       sec evnts  : The actual number of events per spin sector, '
   printf, ofd, '                    does not saturate at 64 events (sectored events data)'
   printf, ofd, '       sun pulse  : Sun pulse sector (sectored events data)'
   printf, ofd, '       steering   : The steering controller CMD level (housekeeping data)'
   printf, ofd, ' '
   printf, ofd, '-------------------------------------------------------------------------'
   printf, ofd, ' '
   printf, ofd, ' '
   printf, ofd, ' '

   ;-----------------------------------------
   ; event column headers
   ;-----------------------------------------
   printf, ofd,'UTC','Azi','tof','plr','eng','event cnts', 'sum cnts', $
             'sec evnts','sun pulse', 'steering (dec)',  $
             format='((A8,TR4),4(A6,TR1), 4(A10,TR1), (A14))'

   printf, ofd,'---','---','---','---','---','----------', '--------', $
             '---------','---------', '--------------', $
             format='((A8,TR4),4(A6,TR1), 4(A10,TR1), (A14))'


   ;--------------------------------------------------
   ; determine the number of records in each packet
   ;--------------------------------------------------
   n_Xele  = n_elements ((*l_Xstruct.startUdfTimePtr)[*])
   n_event = l_event.nitems
   n_binn  = n_elements ((*l_binned.startUdfTimePtr)[*])
   n_str   = n_binn
   nele    = l_event.nitems

   ;--------------------------------------------------
   ; variables used to print the packet number 
   ;--------------------------------------------------
   pcnt      = 0L   ;** count of number of packets (spins) **
   prev_sect = 44L  ;** previous sector number **
   prev_event_time =  Convert ((*l_event.startUdfTimePtr)[0])


   ;-----------------------------------------------------------
   ; define local arrays to store data synced to event data
   ;-----------------------------------------------------------
   binn_cnts = make_array (l_event.nitems, /INT, value = -9)
   event_cnts= make_array (l_event.nitems, /INT, value = -9)
   sunp      = make_array (l_event.nitems, /INT, value = -9)
   steering  = make_array (l_event.nitems, /INT, value = -9)


   SyncEvent2SectEvnts, l_Xstruct, n_Xele, l_event, n_event, event_cnts, sunp
   SyncEvent2Binned, l_binned, l_event, n_binn, n_event, binn_cnts, steering


   for indx = 0L, n_event - 1L do begin

       start_event_time =  (*l_event.startUdfTimePtr)[indx] 
       sector = (*l_event.azimuthPtr)[indx] 
       curr_sect = sector

       ;-------------------------------------------------------
       ; if the spin sector has zero events, UDF puts in filler
       ; values, don't print them to the file 
       ;-------------------------------------------------------
       fstatus = CheckFiller ((*l_event.tofPtr)[indx], $
                              (*l_event.polarPtr)[indx], $
                              (*l_event.energyPtr)[indx] )
 

       if (fstatus eq 1L) then begin
 
          ;-------------------------------------------------------
          ; if a new spin print out a packet separator
          ;-------------------------------------------------------
          start = Convert (start_event_time)
          ev_time_diff = start - prev_event_time
          if (curr_sect lt prev_sect) OR (ev_time_diff gt 120) then begin
            str = '------------------------------ packet #' + $
                  strcompress(string(pcnt),/remove_all) + $
                  '-------------------------------------'
            printf, ofd, str
            pcnt = pcnt + 1L
          endif 
 
          prev_sect = curr_sect
          prev_event_time = Convert(start_event_time)

          ;-------------------------------------------------------
          ; write the data to the text file
          ;-------------------------------------------------------
          tmestr = STRING ((*l_event.startUdfTimePtr)[indx].doy, $
                           (*l_event.startUdfTimePtr)[indx].hour,$
                           (*l_event.startUdfTimePtr)[indx].min, $
                           (*l_event.startUdfTimePtr)[indx].sec, $
                           FORMAT="(i3,' ',i2.2,':',i2.2,':',i2.2)")
 

          printf, ofd, $
                  tmestr, $                      ;** UTC **
                  sector, $                      ;** spin sector **
                  (*l_event.tofPtr)[indx], $     ;** time of flight **
                  (*l_event.polarPtr)[indx], $   ;** polar bin **
                  (*l_event.energyPtr)[indx], $  ;** energy level **
                  (*l_event.sectCntsPtr)[indx],$ ;** summed event cnts **
                  binn_cnts[indx], $      ;** binned summed cnts **
                  event_cnts[indx],$      ;** evnt/sec(sp pkt)**
                  sunp[indx], $           ;** sun pulse sector **
                  steering[indx],  $             ;** steering controller **
                  format='((A12,TR2),4(I4,TR3), 4(I5,TR6), (I4))'
       endif 
   endfor

   close, ofd
   free_lun, ofd

RETURN
END


;-------------------------------------------------------------------------------
;  Procedure:  l_EventTerm
;
;  Description: Free allocated pointers.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_event               struct        I        event data parameters
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11            Original implementation
;-------------------------------------------------------------------------------

PRO l_EventTerm, l_event

   if (ptr_valid (l_event.timePtr)) then begin
       ptr_free, l_event.timePtr
   endif

   if (ptr_valid (l_event.spTimePtr)) then begin
       ptr_free, l_event.spTimePtr
   endif

   if (ptr_valid (l_event.filledTimePtr)) then begin
       ptr_free, l_event.filledTimePtr
   endif

   if (ptr_valid (l_event.bkcUdfTimePtr)) then begin
       ptr_free, l_event.bkcUdfTimePtr
   endif


   if (ptr_valid (l_event.azimuthPtr)) then begin
       ptr_free, l_event.azimuthPtr
   endif

   if (ptr_valid (l_event.tofPtr)) then begin
       ptr_free, l_event.tofPtr
   endif

   if (ptr_valid (l_event.polarPtr)) then begin
       ptr_free, l_event.polarPtr
   endif

   if (ptr_valid (l_event.energyPtr)) then begin
       ptr_free, l_event.energyPtr
   endif

   if (ptr_valid (l_event.sectCntsPtr)) then begin
       ptr_free, l_event.sectCntsPtr
   endif


   if (ptr_valid (l_event.startUdfTimePtr)) then begin
       ptr_free, l_event.startUdfTimePtr
   endif

   if (ptr_valid (l_event.filledStartUdfTimePtr)) then begin
       ptr_free, l_event.filledStartUdfTimePtr
   endif

   if (ptr_valid (l_event.orig_startUdfTimePtr)) then begin
       ptr_free, l_event.orig_startUdfTimePtr
   endif

   if (ptr_valid (l_event.stopUdfTimePtr)) then begin
       ptr_free, l_event.stopUdfTimePtr
   endif

   if (ptr_valid (l_event.orig_stopUdfTimePtr)) then begin
       ptr_free, l_event.orig_stopUdfTimePtr
   endif

   if (ptr_valid (l_event.filledStopUdfTimePtr)) then begin
       ptr_free, l_event.filledStopUdfTimePtr
   endif

   if (ptr_valid (l_event.background)) then begin
       ptr_free, l_event.background
   endif

   if (ptr_valid (l_event.background_e0)) then begin
       ptr_free, l_event.background_e0
   endif

   if (ptr_valid (l_event.background_e1)) then begin
       ptr_free, l_event.background_e1
   endif

   if (ptr_valid (l_event.background_e2)) then begin
       ptr_free, l_event.background_e2
   endif

   if (ptr_valid (l_event.background_raw_H)) then begin
       ptr_free, l_event.background_raw_H
   endif

   if (ptr_valid (l_event.background_raw_H_e0)) then begin
       ptr_free, l_event.background_raw_H_e0
   endif

   if (ptr_valid (l_event.background_raw_H_e1)) then begin
       ptr_free, l_event.background_raw_H_e1
   endif

   if (ptr_valid (l_event.background_raw_H_e2)) then begin
       ptr_free, l_event.background_raw_H_e2
   endif


   if (ptr_valid (l_event.background_raw_O)) then begin
       ptr_free, l_event.background_raw_O
   endif

   if (ptr_valid (l_event.background_raw_O_e0)) then begin
       ptr_free, l_event.background_raw_O_e0
   endif

   if (ptr_valid (l_event.background_raw_O_e1)) then begin
       ptr_free, l_event.background_raw_O_e1
   endif

   if (ptr_valid (l_event.background_raw_O_e2)) then begin
       ptr_free, l_event.background_raw_O_e2
   endif

   if (ptr_valid (l_event.background_raw_extra)) then begin
       ptr_free, l_event.background_raw_extra
   endif

   if (ptr_valid (l_event.background_raw_extra_e0)) then begin
       ptr_free, l_event.background_raw_extra_e0
   endif

   if (ptr_valid (l_event.background_raw_extra_e1)) then begin
       ptr_free, l_event.background_raw_extra_e1
   endif

   if (ptr_valid (l_event.background_raw_extra_e2)) then begin
       ptr_free, l_event.background_raw_extra_e2
   endif

   if (ptr_valid (l_event.background_H)) then begin
       ptr_free, l_event.background_H
   endif

   if (ptr_valid (l_event.background_H_e0)) then begin
       ptr_free, l_event.background_H_e0
   endif

   if (ptr_valid (l_event.background_H_e1)) then begin
       ptr_free, l_event.background_H_e1
   endif

   if (ptr_valid (l_event.background_H_e2)) then begin
       ptr_free, l_event.background_H_e2
   endif

   if (ptr_valid (l_event.background_H_extra)) then begin
       ptr_free, l_event.background_H_extra
   endif

   if (ptr_valid (l_event.background_H_extra_e0)) then begin
       ptr_free, l_event.background_H_extra_e0
   endif

   if (ptr_valid (l_event.background_H_extra_e1)) then begin
       ptr_free, l_event.background_H_extra_e1
   endif

   if (ptr_valid (l_event.background_H_extra_e2)) then begin
       ptr_free, l_event.background_H_extra_e2
   endif

   if (ptr_valid (l_event.background_O)) then begin
       ptr_free, l_event.background_O
   endif

   if (ptr_valid (l_event.background_O_e0)) then begin
       ptr_free, l_event.background_O_e0
   endif

   if (ptr_valid (l_event.background_O_e1)) then begin
       ptr_free, l_event.background_O_e1
   endif

   if (ptr_valid (l_event.background_O_e2)) then begin
       ptr_free, l_event.background_O_e2
   endif


   if (ptr_valid (l_event.background_O_extra)) then begin
       ptr_free, l_event.background_O_extra
   endif

   if (ptr_valid (l_event.background_O_extra_e0)) then begin
       ptr_free, l_event.background_O_extra_e0
   endif

   if (ptr_valid (l_event.background_O_extra_e1)) then begin
       ptr_free, l_event.background_O_extra_e1
   endif

   if (ptr_valid (l_event.background_O_extra_e2)) then begin
       ptr_free, l_event.background_O_extra_e2
   endif

   if (ptr_valid (l_event.hydrogen_raw)) then begin
       ptr_free, l_event.hydrogen_raw
   endif

   if (ptr_valid (l_event.hydrogen_raw_extra)) then begin
       ptr_free, l_event.hydrogen_raw_extra
   endif

   if (ptr_valid (l_event.hydrogen_e0)) then begin
       ptr_free, l_event.hydrogen_e0
   endif

   if (ptr_valid (l_event.hydrogen_e0_extra)) then begin
       ptr_free, l_event.hydrogen_e0_extra
   endif

   if (ptr_valid (l_event.hydrogen_e0_flux)) then begin
       ptr_free, l_event.hydrogen_e0_flux
   endif

   if (ptr_valid (l_event.hydrogen_e0_log_flux)) then begin
       ptr_free, l_event.hydrogen_e0_log_flux
   endif

   if (ptr_valid (l_event.hydrogen_e1)) then begin
       ptr_free, l_event.hydrogen_e1
   endif

   if (ptr_valid (l_event.hydrogen_e1_extra)) then begin
       ptr_free, l_event.hydrogen_e1_extra
   endif

   if (ptr_valid (l_event.hydrogen_e1_flux)) then begin
       ptr_free, l_event.hydrogen_e1_flux
   endif

   if (ptr_valid (l_event.hydrogen_e1_log_flux)) then begin
       ptr_free, l_event.hydrogen_e1_log_flux
   endif

   if (ptr_valid (l_event.hydrogen_e2)) then begin
       ptr_free, l_event.hydrogen_e2
   endif

   if (ptr_valid (l_event.hydrogen_e2_extra)) then begin
       ptr_free, l_event.hydrogen_e2_extra
   endif

   if (ptr_valid (l_event.hydrogen_e2_flux)) then begin
       ptr_free, l_event.hydrogen_e2_flux
   endif

   if (ptr_valid (l_event.hydrogen_e2_log_flux)) then begin
       ptr_free, l_event.hydrogen_e2_log_flux
   endif

   if (ptr_valid (l_event.H_sec_events)) then begin
       ptr_free, l_event.H_sec_events
   endif

   if (ptr_valid (l_event.H_sec_events_e0)) then begin
       ptr_free, l_event.H_sec_events_e0
   endif

   if (ptr_valid (l_event.H_sec_events_e1)) then begin
       ptr_free, l_event.H_sec_events_e1
   endif

   if (ptr_valid (l_event.H_sec_events_e2)) then begin
       ptr_free, l_event.H_sec_events_e2
   endif

   if (ptr_valid (l_event.H_sec_events_extra)) then begin
       ptr_free, l_event.H_sec_events_extra
   endif

   if (ptr_valid (l_event.H_sec_events_extra_e0)) then begin
       ptr_free, l_event.H_sec_events_extra_e0
   endif

   if (ptr_valid (l_event.H_sec_events_extra_e1)) then begin
       ptr_free, l_event.H_sec_events_extra_e1
   endif

   if (ptr_valid (l_event.H_sec_events_extra_e2)) then begin
       ptr_free, l_event.H_sec_events_extra_e2
   endif

   if (ptr_valid (l_event.H_n_in_peak)) then begin
       ptr_free, l_event.H_n_in_peak
   endif

   if (ptr_valid (l_event.H_n_in_peak_e0)) then begin
       ptr_free, l_event.H_n_in_peak_e0
   endif

   if (ptr_valid (l_event.H_n_in_peak_e1)) then begin
       ptr_free, l_event.H_n_in_peak_e1
   endif

   if (ptr_valid (l_event.H_n_in_peak_e2)) then begin
       ptr_free, l_event.H_n_in_peak_e2
   endif

   if (ptr_valid (l_event.H_n_in_peak_extra)) then begin
       ptr_free, l_event.H_n_in_peak_extra
   endif

   if (ptr_valid (l_event.H_n_in_peak_extra_e0)) then begin
       ptr_free, l_event.H_n_in_peak_extra_e0
   endif

   if (ptr_valid (l_event.H_n_in_peak_extra_e1)) then begin
       ptr_free, l_event.H_n_in_peak_extra_e1
   endif

   if (ptr_valid (l_event.H_n_in_peak_extra_e2)) then begin
       ptr_free, l_event.H_n_in_peak_extra_e2
   endif

   if (ptr_valid (l_event.H_scale_factor)) then begin
       ptr_free, l_event.H_scale_factor
   endif

   if (ptr_valid (l_event.H_scale_factor_e0)) then begin
       ptr_free, l_event.H_scale_factor_e0
   endif

   if (ptr_valid (l_event.H_scale_factor_e1)) then begin
       ptr_free, l_event.H_scale_factor_e1
   endif

   if (ptr_valid (l_event.H_scale_factor_e2)) then begin
       ptr_free, l_event.H_scale_factor_e2
   endif

   if (ptr_valid (l_event.H_scale_factor_extra)) then begin
       ptr_free, l_event.H_scale_factor_extra
   endif

   if (ptr_valid (l_event.H_scale_factor_extra_e0)) then begin
       ptr_free, l_event.H_scale_factor_extra_e0
   endif

   if (ptr_valid (l_event.H_scale_factor_extra_e1)) then begin
       ptr_free, l_event.H_scale_factor_extra_e1
   endif

   if (ptr_valid (l_event.H_scale_factor_extra_e2)) then begin
       ptr_free, l_event.H_scale_factor_extra_e2
   endif

   if (ptr_valid (l_event.corrected_H)) then begin
       ptr_free, l_event.corrected_H
   endif

   if (ptr_valid (l_event.corrected_H_e0)) then begin
       ptr_free, l_event.corrected_H_e0
   endif

   if (ptr_valid (l_event.corrected_H_e1)) then begin
       ptr_free, l_event.corrected_H_e1
   endif

   if (ptr_valid (l_event.corrected_H_e2)) then begin
       ptr_free, l_event.corrected_H_e2
   endif

   if (ptr_valid (l_event.corrected_H_extra)) then begin
       ptr_free, l_event.corrected_H_extra
   endif

   if (ptr_valid (l_event.corrected_H_extra_e0)) then begin
       ptr_free, l_event.corrected_H_extra_e0
   endif

   if (ptr_valid (l_event.corrected_H_extra_e1)) then begin
       ptr_free, l_event.corrected_H_extra_e1
   endif

   if (ptr_valid (l_event.corrected_H_extra_e2)) then begin
       ptr_free, l_event.corrected_H_extra_e2
   endif


   if (ptr_valid (l_event.corr_scl_H)) then begin
       ptr_free, l_event.corr_scl_H
   endif

   if (ptr_valid (l_event.corr_scl_H_e0)) then begin
       ptr_free, l_event.corr_scl_H_e0
   endif

   if (ptr_valid (l_event.corr_scl_H_e1)) then begin
       ptr_free, l_event.corr_scl_H_e1
   endif

   if (ptr_valid (l_event.corr_scl_H_e2)) then begin
       ptr_free, l_event.corr_scl_H_e2
   endif

   if (ptr_valid (l_event.scaled_H)) then begin
       ptr_free, l_event.scaled_H
   endif

   if (ptr_valid (l_event.scaled_H_e0)) then begin
       ptr_free, l_event.scaled_H_e0
   endif

   if (ptr_valid (l_event.scaled_H_e1)) then begin
       ptr_free, l_event.scaled_H_e1
   endif

   if (ptr_valid (l_event.scaled_H_e2)) then begin
       ptr_free, l_event.scaled_H_e2
   endif

   if (ptr_valid (l_event.oxygen_raw)) then begin
       ptr_free, l_event.oxygen_raw
   endif

   if (ptr_valid (l_event.oxygen_raw_extra)) then begin
       ptr_free, l_event.oxygen_raw_extra
   endif

   if (ptr_valid (l_event.oxygen_e0)) then begin
       ptr_free, l_event.oxygen_e0
   endif

   if (ptr_valid (l_event.oxygen_e0_extra)) then begin
       ptr_free, l_event.oxygen_e0_extra
   endif

   if (ptr_valid (l_event.oxygen_e0_flux)) then begin
       ptr_free, l_event.oxygen_e0_flux
   endif

   if (ptr_valid (l_event.oxygen_e0_log_flux)) then begin
       ptr_free, l_event.oxygen_e0_log_flux
   endif

   if (ptr_valid (l_event.oxygen_e1)) then begin
       ptr_free, l_event.oxygen_e1
   endif

   if (ptr_valid (l_event.oxygen_e1_extra)) then begin
       ptr_free, l_event.oxygen_e1_extra
   endif

   if (ptr_valid (l_event.oxygen_e1_flux)) then begin
       ptr_free, l_event.oxygen_e1_flux
   endif

   if (ptr_valid (l_event.oxygen_e1_log_flux)) then begin
       ptr_free, l_event.oxygen_e1_log_flux
   endif

   if (ptr_valid (l_event.oxygen_e2)) then begin
       ptr_free, l_event.oxygen_e2
   endif

   if (ptr_valid (l_event.oxygen_e2_extra)) then begin
       ptr_free, l_event.oxygen_e2_extra
   endif

   if (ptr_valid (l_event.oxygen_e2_flux)) then begin
       ptr_free, l_event.oxygen_e2_flux
   endif

   if (ptr_valid (l_event.oxygen_e2_log_flux)) then begin
       ptr_free, l_event.oxygen_e2_log_flux
   endif

   if (ptr_valid (l_event.O_sec_events)) then begin
       ptr_free, l_event.O_sec_events
   endif

   if (ptr_valid (l_event.O_sec_events_e0)) then begin
       ptr_free, l_event.O_sec_events_e0
   endif

   if (ptr_valid (l_event.O_sec_events_e1)) then begin
       ptr_free, l_event.O_sec_events_e1
   endif

   if (ptr_valid (l_event.O_sec_events_e2)) then begin
       ptr_free, l_event.O_sec_events_e2
   endif

   if (ptr_valid (l_event.O_sec_events_extra)) then begin
       ptr_free, l_event.O_sec_events_extra
   endif

   if (ptr_valid (l_event.O_sec_events_extra_e0)) then begin
       ptr_free, l_event.O_sec_events_extra_e0
   endif

   if (ptr_valid (l_event.O_sec_events_extra_e1)) then begin
       ptr_free, l_event.O_sec_events_extra_e1
   endif

   if (ptr_valid (l_event.O_sec_events_extra_e2)) then begin
       ptr_free, l_event.O_sec_events_extra_e2
   endif

   if (ptr_valid (l_event.O_n_in_peak)) then begin
       ptr_free, l_event.O_n_in_peak
   endif

   if (ptr_valid (l_event.O_n_in_peak_e0)) then begin
       ptr_free, l_event.O_n_in_peak_e0
   endif

   if (ptr_valid (l_event.O_n_in_peak_e1)) then begin
       ptr_free, l_event.O_n_in_peak_e1
   endif

   if (ptr_valid (l_event.O_n_in_peak_e2)) then begin
       ptr_free, l_event.O_n_in_peak_e2
   endif

   if (ptr_valid (l_event.O_n_in_peak_extra)) then begin
       ptr_free, l_event.O_n_in_peak_extra
   endif

   if (ptr_valid (l_event.O_n_in_peak_extra_e0)) then begin
       ptr_free, l_event.O_n_in_peak_extra_e0
   endif

   if (ptr_valid (l_event.O_n_in_peak_extra_e1)) then begin
       ptr_free, l_event.O_n_in_peak_extra_e1
   endif

   if (ptr_valid (l_event.O_n_in_peak_extra_e2)) then begin
       ptr_free, l_event.O_n_in_peak_extra_e2
   endif

   if (ptr_valid (l_event.O_scale_factor)) then begin
       ptr_free, l_event.O_scale_factor
   endif

   if (ptr_valid (l_event.O_scale_factor_e0)) then begin
       ptr_free, l_event.O_scale_factor_e0
   endif

   if (ptr_valid (l_event.O_scale_factor_e1)) then begin
       ptr_free, l_event.O_scale_factor_e1
   endif

   if (ptr_valid (l_event.O_scale_factor_e2)) then begin
       ptr_free, l_event.O_scale_factor_e2
   endif

   if (ptr_valid (l_event.O_scale_factor_extra)) then begin
       ptr_free, l_event.O_scale_factor_extra
   endif

   if (ptr_valid (l_event.O_scale_factor_extra_e0)) then begin
       ptr_free, l_event.O_scale_factor_extra_e0
   endif

   if (ptr_valid (l_event.O_scale_factor_extra_e1)) then begin
       ptr_free, l_event.O_scale_factor_extra_e1
   endif

   if (ptr_valid (l_event.O_scale_factor_extra_e2)) then begin
       ptr_free, l_event.O_scale_factor_extra_e2
   endif

   if (ptr_valid (l_event.corrected_O)) then begin
       ptr_free, l_event.corrected_O
   endif
   if (ptr_valid (l_event.corrected_O_e0)) then begin
       ptr_free, l_event.corrected_O_e0
   endif

   if (ptr_valid (l_event.corrected_O_e1)) then begin
       ptr_free, l_event.corrected_O_e1
   endif

   if (ptr_valid (l_event.corrected_O_e2)) then begin
       ptr_free, l_event.corrected_O_e2
   endif

   if (ptr_valid (l_event.corrected_O_extra)) then begin
       ptr_free, l_event.corrected_O_extra
   endif

   if (ptr_valid (l_event.corrected_O_extra_e0)) then begin
       ptr_free, l_event.corrected_O_extra_e0
   endif

   if (ptr_valid (l_event.corrected_O_extra_e1)) then begin
       ptr_free, l_event.corrected_O_extra_e1
   endif

   if (ptr_valid (l_event.corrected_O_extra_e2)) then begin
       ptr_free, l_event.corrected_O_extra_e2
   endif

   if (ptr_valid (l_event.corr_scl_O)) then begin
       ptr_free, l_event.corr_scl_O
   endif

   if (ptr_valid (l_event.corr_scl_O_e0)) then begin
       ptr_free, l_event.corr_scl_O_e0
   endif

   if (ptr_valid (l_event.corr_scl_O_e1)) then begin
       ptr_free, l_event.corr_scl_O_e1
   endif

   if (ptr_valid (l_event.corr_scl_O_e2)) then begin
       ptr_free, l_event.corr_scl_O_e2
   endif

   if (ptr_valid (l_event.scaled_O)) then begin
       ptr_free, l_event.scaled_O
   endif

   if (ptr_valid (l_event.scaled_O_e0)) then begin
       ptr_free, l_event.scaled_O_e0
   endif

   if (ptr_valid (l_event.scaled_O_e1)) then begin
       ptr_free, l_event.scaled_O_e1
   endif

   if (ptr_valid (l_event.scaled_O_e2)) then begin
       ptr_free, l_event.scaled_O_e2
   endif

   if (ptr_valid (l_event.all_masses)) then begin
       ptr_free, l_event.all_masses
   endif

   if (ptr_valid (l_event.amDmpPtr)) then begin
       ptr_free, l_event.amDmpPtr
   endif

   if (ptr_valid (l_event.imgHcorr)) then begin
       ptr_free, l_event.imgHcorr
   endif

   if (ptr_valid (l_event.imgOcorr)) then begin
       ptr_free, l_event.imgOcorr
   endif

   if (ptr_valid (l_event.imgSpTme)) then begin
       ptr_free, l_event.imgSpTme
   endif

   if (ptr_valid (l_event.H_elim_spins_time)) then begin
       ptr_free, l_event.H_elim_spins_time
   endif

   if (ptr_valid (l_event.H_elim_spins_nevents)) then begin
       ptr_free, l_event.H_elim_spins_nevents
   endif

   if (ptr_valid (l_event.O_elim_spins_time)) then begin
       ptr_free, l_event.O_elim_spins_time
   endif

   if (ptr_valid (l_event.O_elim_spins_nevents)) then begin
       ptr_free, l_event.O_elim_spins_nevents
   endif



END

;-------------------------------------------------------------------------------
;  Function:  CheckAvailData
;
;  Description:
;     Check the type of data that is available for that specified time range.
;
;  Return Value:  type = int 
;  Value                         Description
;  -------------------------     ------------------------------
;  1                             data before flight software upload, no sectored
;                                events packets
;  2                             data after FSW upload, sectored events data packet
;                                in memory dump 
;  3                             no sectored events packets data available in MEMD
;                                or as it's own packet 
;  4                             sectored events packets available
;  
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  start_time            struct        I        UDF start time [yr, doy, hh, mm, ss]
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;-------------------------------------------------------------------------------

FUNCTION CheckAvailData, start_time
  

    if (start_time[0] eq 2000) then begin
        if (start_time[1] le 213) then begin
           return, 1
        endif else begin
           if (start_time[1] gt 213) and (start_time[1] le 289) then begin
               return, 2
            endif else begin
                 if (start_time[1] gt 289) and (start_time[1] le 337) then begin
                    return, 3
                 endif else begin
                    return, 4
                 endelse
            endelse
        endelse
    endif else begin
        return, 4
    endelse

END


;-------------------------------------------------------------------------------
;  Procedure: EventBKHeader
;
;  Description: Writes a column description.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  type                  string        I        hydrogen or oxygen
;  filedes               long          I        output file descriptor
;  title                 string        I        title
;  zero_title            string        I        additional title data
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------


PRO l_EventBKHeader, type, filedes, title, zero_title


      printf, filedes, ' '
      printf, filedes, title
      printf, filedes, zero_title
      printf, filedes, ' '

      printf, filedes,'UTC','Azi', 'Raw', 'Bkgrnd', 'Bkgrnd', $ 
                      'Events in', '# Events', 'Scale', $
                      '  Corr/Scl', '  Corr/Scl' , $
                      format='((A8,TR4), (A6, TR7) , 5(A10, TR5), (A8,TR1), 2(A10, TR4))'

      type_str = 'in ' + type + ' Peak'
      printf, filedes,'   ','   ', 'Counts',  ' Raw ', 'Counts', $
                      ' Sector ',   type_str, 'Factor', $
                      ' Value', ' [Hz] ', $
                      format='((A8,TR4), (A6,TR7) , 5(A12, TR3), (A8,TR1), 2(A10, TR4))'

      printf, filedes,'---','---', '------',  '-----', '------', $
                      '--------', ' ----------', '------', $
                      ' ------','------', $
                      format='((A8,TR4), (A6, TR7) , 5(A12, TR3), (A8,TR1), 2(A10, TR4))'


RETURN
END

;-------------------------------------------------------------------------------
;  Procedure: EventWriteBKFile
;
;  Description: Write the event background data to a file.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  nspins                long          I        number of spins to write
;  bkcUdfTimePtr         int[]         I        UDF time corresponding to background
;                                               corrected data
;  raw_cnts              long[]        I        raw data counts
;  events_n_sec          long[]        I        # events in sector
;  binn_event            long[]        I        # of binned events
;  scale_factor          long[]        I        scaled data
;  corr_cnt              long[]        I        background corrected data
;  spinsec               long[]        I        spin sector
;  filedes               int           I        output file descriptor
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------

PRO l_EventWriteBK2File, nspins, background_flag, bkcUdfTimePtr, $
raw_cnts, bk_raw, bk_cnts,events_n_sect, nevents_n_peak, $
scale_factor, corr_cnts, corr_scl_val, spinsec, filedes

      if (background_flag eq 'Y') then begin

          for i = 0L, nspins - 1L do begin
              tmestr = STRING ((bkcUdfTimePtr)[i].doy, $
                               (bkcUdfTimePtr)[i].hour,$
                               (bkcUdfTimePtr)[i].min, $
                               (bkcUdfTimePtr)[i].sec, $
                               FORMAT="(i3,' ',i2.2,':',i2.2,':',i2.2)")

              for j = 0L, spinsec - 1L do begin
                 printf, filedes, tmestr, $
                                  j, $
                                  raw_cnts[j,i], $
                                  bk_raw[j,i], $
                                  bk_cnts[j,i], $
                                  events_n_sect[j,i], $
                                  nevents_n_peak[j,i], $
                                  scale_factor[j,i], $
                                  corr_cnts[j, i], $
                                  corr_scl_val[j,i], $
                                  format='((A12,TR2),(I4,TR8), 8(e12.3,TR2))'

              endfor
         endfor
    endif else begin
          for i = 0L, nspins - 1L do begin
              tmestr = STRING ((bkcUdfTimePtr)[i].doy, $
                               (bkcUdfTimePtr)[i].hour,$
                               (bkcUdfTimePtr)[i].min, $
                               (bkcUdfTimePtr)[i].sec, $
                               FORMAT="(i3,' ',i2.2,':',i2.2,':',i2.2)")

              for j = 0L, spinsec - 1L do begin
                 printf, filedes, tmestr, $
                                  j, $
                                  raw_cnts[j, i], $
                                  events_n_sect[j,i], $
                                  nevents_n_peak[j,i], $
                                  scale_factor[j,i], $
                                  corr_cnts[j, i], $
                                  corr_scl_val[j,i], $
                                  format='((A12,TR2),(I4,TR8), 6(e12.3,TR2))'

              endfor
         endfor

    endelse


RETURN
END

;-------------------------------------------------------------------------------
;  Function/Procedure: l_EventDumpBKData
;  Description:
;     Generate a text dump of the background plots.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_event               struct        I        event data pointers
;  l_event_setup         struct        I        general event data setup parms
;  l_setup               struct
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                08/01          v1.0.7          Original implementation
;-------------------------------------------------------------------------------

PRO l_EventDumpBKData, filedes, l_event, l_event_setup, l_setup


   nspins = n_elements ((*l_event.bkcUdfTimePtr)[*])
   
   zero_title = 'Spin Angle (-140 - 220)'
   if (l_setup.l_angle_zero eq 'E') then begin
      zero_title = 'Earth Centered Azimuth'
   endif else begin
      if (l_setup.l_angle_zero eq 'S') then begin
         zero_title = 'Sun Centered Azimuth'
      endif else begin
         if (l_setup.l_angle_zero eq 'M') then begin
            zero_title = 'Moon Centered Azimuth'
         endif
      endelse
   endelse 


   printf, filedes, '----------------------- Column Descriptions -----------------------------'
   printf, filedes, '                        EVENT DATA '
   printf, filedes, ' '
   printf, filedes, '      UTC             : Universal Time Coordinate (corrected)'
   printf, filedes, '      Azi             : Azimuthal spin sector (0-44)'
   printf, filedes, '      Raw Counts      : total number of binned events from sector '
   printf, filedes, '                        counter (>64)
   printf, filedes, '      Bkgrnd Raw      : counts in each spin sector falling in TOF '
   printf, filedes, '                        background range' 
   printf, filedes, '      Bkgrnd Counts   : background counts times scale factor' 
   printf, filedes, '      Events in Sector: events in the sector (0-64)'
   printf, filedes, '      # Events in Peak: the number of events falling between the '
   printf, filedes, '                        H/O channels'
   printf, filedes, '      Scale Factor    : scaling factor '
   printf, filedes, '      Corr/Scl Val    : background corrected scaled counts'
   printf, filedes, '      Corr/Scl [Hz]   : background corrected scaled counts '
   printf, filedes, '                        count rate [Hz]'
   printf, filedes, ' '
   printf, filedes, '-------------------------------------------------------------------------'
   printf, filedes, ' '
   printf, filedes, ' '
   printf, filedes, ' '

   
   if (l_event_setup.hydrogen eq 'Y') then begin

         title = 'Hydrogen'
         type  = 'H'
         if (l_event_setup.background eq 'Y') then begin
            title = title + ' Scaled Background Corrected'
         endif

         l_EventBKHeader, type, filedes, title, zero_title

         raw_cnts  = (*l_event.hydrogen_raw)

         if (l_event_setup.background eq 'Y') then begin
            bk_raw    = (*l_event.background_raw_H)
            bk_cnts   = (*l_event.background_H)
         endif

         corr_cnts = (*l_event.corrected_H)
         events_n_sect = (*l_event.H_sec_events)
         nevents_n_peak = (*l_event.H_n_in_peak)
         scale_factor = (*l_event.H_scale_factor)
         corr_scl_val = (*l_event.corr_scl_H)

         l_EventWriteBK2File, nspins, l_event_setup.background, (*l_event.bkcUdfTimePtr), $
         raw_cnts, bk_raw, bk_cnts,events_n_sect, nevents_n_peak, $
         scale_factor, corr_cnts, corr_scl_val, l_setup.l_num_spinsec, filedes

         if (l_event_setup.energy0 eq 'Y') then begin
            title = 'Hydrogen Energy 0'
            type  = 'H E0'
            if (l_event_setup.background eq 'Y') then begin
               title = title + ' Scaled Background Corrected'
            endif

            l_EventBKHeader, type, filedes, title, zero_title

            raw_cnts  = (*l_event.hydrogen_raw_e0)

            if (l_event_setup.background eq 'Y') then begin
               bk_raw    = (*l_event.background_raw_H_e0)
               bk_cnts   = (*l_event.background_H_e0)
            endif

            corr_cnts = (*l_event.corrected_H_e0)
            events_n_sect = (*l_event.H_sec_events_e0)
            nevents_n_peak = (*l_event.H_n_in_peak_e0)
            scale_factor = (*l_event.H_scale_factor_e0)
            corr_scl_val = (*l_event.corr_scl_H_e0)

            l_EventWriteBK2File, nspins, l_event_setup.background, (*l_event.bkcUdfTimePtr), $
            raw_cnts, bk_raw, bk_cnts,events_n_sect, nevents_n_peak, $
            scale_factor, corr_cnts, corr_scl_val, l_setup.l_num_spinsec, filedes
         endif
       
         if (l_event_setup.energy1 eq 'Y') then begin
            title = 'Hydrogen Energy 1'
            type  = 'H E1'
            if (l_event_setup.background eq 'Y') then begin
               title = title + ' Scaled Background Corrected'
            endif

            l_EventBKHeader, type, filedes, title, zero_title

            raw_cnts  = (*l_event.hydrogen_raw_e1)

            if (l_event_setup.background eq 'Y') then begin
               bk_raw    = (*l_event.background_raw_H_e1)
               bk_cnts   = (*l_event.background_H_e1)
            endif

            corr_cnts = (*l_event.corrected_H_e1)
            events_n_sect = (*l_event.H_sec_events_e1)
            nevents_n_peak = (*l_event.H_n_in_peak_e1)
            scale_factor = (*l_event.H_scale_factor_e1)
            corr_scl_val = (*l_event.corr_scl_H_e1)

            l_EventWriteBK2File, nspins, l_event_setup.background, (*l_event.bkcUdfTimePtr), $
            raw_cnts, bk_raw, bk_cnts,events_n_sect, nevents_n_peak, $
            scale_factor, corr_cnts, corr_scl_val, l_setup.l_num_spinsec, filedes
         endif

         if (l_event_setup.energy2 eq 'Y') then begin
            title = 'Hydrogen Energy 2'
            type  = 'H E2'
            if (l_event_setup.background eq 'Y') then begin
               title = title + ' Scaled Background Corrected'
            endif

            l_EventBKHeader, type, filedes, title, zero_title

            raw_cnts  = (*l_event.hydrogen_raw_e2)

            if (l_event_setup.background eq 'Y') then begin
               bk_raw    = (*l_event.background_raw_H_e2)
               bk_cnts   = (*l_event.background_H_e2)
            endif

            corr_cnts = (*l_event.corrected_H_e2)
            events_n_sect = (*l_event.H_sec_events_e2)
            nevents_n_peak = (*l_event.H_n_in_peak_e2)
            scale_factor = (*l_event.H_scale_factor_e2)
            corr_scl_val = (*l_event.corr_scl_H_e2)

            l_EventWriteBK2File, nspins, l_event_setup.background, (*l_event.bkcUdfTimePtr), $
            raw_cnts, bk_raw, bk_cnts,events_n_sect, nevents_n_peak, $
            scale_factor, corr_cnts, corr_scl_val, l_setup.l_num_spinsec, filedes
         endif


   endif


   if (l_event_setup.oxygen eq 'Y') then begin
      title = 'Oxygen'
      type = 'O' 
      if (l_event_setup.background eq 'Y') then begin
         title = title + ' Scaled Background Corrected'
      endif

      l_EventBKHeader, type, filedes, title, zero_title

      raw_cnts = (*l_event.oxygen_raw)

      if (l_event_setup.background eq 'Y') then begin
         bk_cnts = (*l_event.background_O)
         bk_raw  = (*l_event.background_raw_O)
      endif 

      corr_cnts = (*l_event.corrected_O)
      events_n_sect = (*l_event.O_sec_events)
      nevents_n_peak = (*l_event.O_n_in_peak)
      scale_factor = (*l_event.O_scale_factor)
      corr_scl_val = (*l_event.corr_scl_O)

      l_EventWriteBK2File, nspins, l_event_setup.background, (*l_event.bkcUdfTimePtr), $
      raw_cnts, bk_raw, bk_cnts,events_n_sect, nevents_n_peak, $
      scale_factor, corr_cnts, corr_scl_val, l_setup.l_num_spinsec, filedes


      if (l_event_setup.energy0 eq 'Y') then begin
         title = 'Oxygen Energy0'
         type = 'O E0' 
         if (l_event_setup.background eq 'Y') then begin
            title = title + ' Scaled Background Corrected'
         endif

         l_EventBKHeader, type, filedes, title, zero_title

         raw_cnts = (*l_event.oxygen_raw_e0)

         if (l_event_setup.background eq 'Y') then begin
            bk_cnts = (*l_event.background_O_e0)
            bk_raw  = (*l_event.background_raw_O_e0)
         endif 

         corr_cnts = (*l_event.corrected_O_e0)
         events_n_sect = (*l_event.O_sec_events_e0)
         nevents_n_peak = (*l_event.O_n_in_peak_e0)
         scale_factor = (*l_event.O_scale_factor_e0)
         corr_scl_val = (*l_event.corr_scl_O_e0)

         l_EventWriteBK2File, nspins, l_event_setup.background, (*l_event.bkcUdfTimePtr), $
         raw_cnts, bk_raw, bk_cnts,events_n_sect, nevents_n_peak, $
         scale_factor, corr_cnts, corr_scl_val, l_setup.l_num_spinsec, filedes

      endif

      if (l_event_setup.energy1 eq 'Y') then begin
         title = 'Oxygen Energy1'
         type = 'O E1' 
         if (l_event_setup.background eq 'Y') then begin
            title = title + ' Scaled Background Corrected'
         endif

         l_EventBKHeader, type, filedes, title, zero_title

         raw_cnts = (*l_event.oxygen_raw_e1)

         if (l_event_setup.background eq 'Y') then begin
            bk_cnts = (*l_event.background_O_e1)
            bk_raw  = (*l_event.background_raw_O_e1)
         endif 

         corr_cnts = (*l_event.corrected_O_e1)
         events_n_sect = (*l_event.O_sec_events_e1)
         nevents_n_peak = (*l_event.O_n_in_peak_e1)
         scale_factor = (*l_event.O_scale_factor_e1)
         corr_scl_val = (*l_event.corr_scl_O_e1)

         l_EventWriteBK2File, nspins, l_event_setup.background, (*l_event.bkcUdfTimePtr), $
         raw_cnts, bk_raw, bk_cnts,events_n_sect, nevents_n_peak, $
         scale_factor, corr_cnts, corr_scl_val, l_setup.l_num_spinsec, filedes

      endif

      if (l_event_setup.energy2 eq 'Y') then begin
         title = 'Oxygen Energy2'
         type = 'O E2' 
         if (l_event_setup.background eq 'Y') then begin
            title = title + ' Scaled Background Corrected'
         endif

         l_EventBKHeader, type, filedes, title, zero_title

         raw_cnts = (*l_event.oxygen_raw_e2)

         if (l_event_setup.background eq 'Y') then begin
            bk_cnts = (*l_event.background_O_e2)
            bk_raw  = (*l_event.background_raw_O_e2)
         endif 

         corr_cnts = (*l_event.corrected_O_e2)
         events_n_sect = (*l_event.O_sec_events_e2)
         nevents_n_peak = (*l_event.O_n_in_peak_e2)
         scale_factor = (*l_event.O_scale_factor_e2)
         corr_scl_val = (*l_event.corr_scl_O_e2)

         l_EventWriteBK2File, nspins, l_event_setup.background, (*l_event.bkcUdfTimePtr), $
         raw_cnts, bk_raw, bk_cnts,events_n_sect, nevents_n_peak, $
         scale_factor, corr_cnts, corr_scl_val, l_setup.l_num_spinsec, filedes

      endif

   endif


RETURN
END




;-------------------------------------------------------------------------------
;  Function : l_EventProc 
;  Description:  
;     This routine is for dumping the event data to a text
;     file. Does not do any plotting and will not do any plotting. A
;     separate Proc routine must be written for plotting. 
;
;  Return Value:  type = int
;  Value                         Description
;  -------------------------     ------------------------------
;  SUCCESS 
;  FAILURE
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_setup               struct        I        generic setup parameters
;  l_setup_file          string        I        set up file name
;  l_pltpos              struct        I        generic plot parameters
;  l_rgb_table           struct        I        generic color table, options
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;     PLOTDEFS         MAX_PLOTS        int         G         max # plots per page
;                      MAX_WEB_SPINS    int         G         max # spins per plot
;                                                             when using web interface
;                      NO_CNTS          double      G         no counts indicator
;                      NO_DATA          double      G         no data indicator
;
;      ERROR_TYPES     SUCCESS         int         G          successful completion
;                      FAILURE         int         G          unsuccessful completion
;                      WARNING         int         G          warning message
;                      INFO            int         G          informational message
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;-------------------------------------------------------------------------------

FUNCTION l_EventProc, l_setup, l_setup_file, l_pltpos, l_rgb_table
COMMON PLOTDEFS, MAX_PLOTS, MAX_WEB_SPINS, NO_CNTS, NO_DATA 
COMMON ERROR_TYPES, SUCCESS, FAILURE, WARNING, INFO

   ;-------------------------------------------------
   ; the maximum number of spins to plot if plotting
   ; via the web
   ;-------------------------------------------------

   status    = 0L
   plotted   = 0L
   pages     = 0L
   nplots    = 0L
   nele      = 0L
   year      = strmid(systime(0), 20, 23)
   l_pltpos.plot_hdr_title = 'LENA Event Data'


   ;-------------------------------------------------
   ; read the event data setup file
   ;-------------------------------------------------
   l_EventReadSetup, l_event_setup, l_setup_file

   ;-------------------------------------------------
   ; if via the web, need to determine the number
   ; of spins desired.  if the number of spins is
   ; greater than max_web_spins, then plot only
   ; max_web_spins
   ;-------------------------------------------------
   sp = ConvArrTimeToSecs (l_setup.l_stop_dtime)
   st = ConvArrTimeToSecs (l_setup.l_start_dtime)
   est_nspins = round((sp - st)/120.0)


   ;-------------------------------------------------
   ; get the event data
   ;-------------------------------------------------
   status = l_EventGetData  (l_setup, l_setup_file, $
                             l_pltpos, $
                             l_event_setup, l_event)

   ;-------------------------------------------------
   ; read setup paramters specific to the binned plots
   ;-------------------------------------------------
   l_BinnedReadSetup, l_binned_setup, l_setup_file

   ;-------------------------------------------------
   ; UDF returns data on a spin/record basis and not
   ; based on time desired.  Therefore, in order to
   ; get a complete set of binned and evsp data  to
   ; match the event set, 2 extra spins of binned
   ; and evsp data needs to be read.  This is ONLY 
   ; if the end time is less than 23:59:59
   ; if the start time is greater than 00:00:00, then
   ; 2 spins of binned and evsp before the desired
   ; start time needs to be added 
   ;-------------------------------------------------

   start_tme = l_setup.l_start_dtime
   stop_tme = l_setup.l_stop_dtime

   orig_start_tme = l_setup.l_start_dtime
   orig_stop_tme = l_setup.l_stop_dtime

   AddExtraTime2Start, start_tme, 6 
   AddExtraTime2Stop, stop_tme, 8 


   l_setup.l_start_dtime = start_tme
   l_setup.l_stop_dtime = stop_tme


   ;-------------------------------------------------
   ; get the binned data based on user start and
   ; stop time
   ;-------------------------------------------------
   status = l_BinnedGetData  (l_setup, $
                              l_setup_file, $
                              l_pltpos, $
                              l_binned_setup, $
                              l_binned, $
                              nele)
   l_setup.l_flux = 'N' 
   l_Binned_AdjExData, l_setup, l_pltpos, l_binned, l_binned_setup


   l_setup.l_start_dtime = orig_start_tme
   l_setup.l_stop_dtime = orig_stop_tme

   l_setup.l_filedesc = 'ev'

   davail = CheckAvailData (l_setup.l_start_dtime)
   case davail of
      1   : begin
               ;-------------------------------------------------
               ; no sectored events packets available, must use
               ; OA data to get sun pulse
               ;-------------------------------------------------
               status = l_OARead (l_oa, start_tme, $
                         stop_tme, $
                         l_pltpos)
               if (status eq FAILURE) then return, FAILURE
               l_CalcSunPulse, l_oa, sun_pulse, 'S'
               l_EventDump_Bef0731, l_setup, l_event, l_oa, sun_pulse, $
                                    l_binned, l_pltpos
               l_OATerm, l_oa
            end
      2   : begin
               ;-------------------------------------------------
               ; sectored events packets available in memory dump
               ; packets
               ;-------------------------------------------------
               status = l_MemdGetData (l_setup, l_pltpos, l_memd)
               if (status eq FAILURE) then return, FAILURE
               l_EventDump, l_setup, l_event, l_memd, l_binned, l_pltpos
               l_MemdTerm, l_memd

            end
      3   : begin 
               ;-------------------------------------------------
               ; between 10/16/00 - 12/02/00, sectored events 
               ; should be individual packets, but are not 
               ; available as individual packets or in the 
               ; memory dump, after reprocessing the packets
               ; SHOULD exist
               ;-------------------------------------------------
               ;-------------------------------------------------
               ; no sectored events packets available, must use
               ; OA data to get sun pulse
               ;-------------------------------------------------
               status = l_OARead (l_oa, start_tme, $
                                  stop_tme, $
                                  l_pltpos)
               if (status eq FAILURE) then return, FAILURE
               l_CalcSunPulse, l_oa, sun_pulse, 'S'
               l_EventDump_Bef0731, l_setup, l_event, l_oa, sun_pulse, $
                                    l_binned, l_pltpos
               l_OATerm, l_oa
            end
      4   : begin 
               ;-------------------------------------------------
               ; read sectored events/sun pulse packet
               ;-------------------------------------------------
               nele = 0L
               EvDmpFlg = 1L
               status = l_EvSpGetData (l_setup, l_pltpos, l_evsp, nele, EvDmpFlg)
               if (status eq FAILURE ) then RETURN, FAILURE
               l_EventDump, l_setup, l_event, l_evsp, l_binned, l_pltpos
               l_EvSpTerm, l_evsp
            end
      else : begin
               print, 'Invalid option.'
             end
   endcase 

   l_EventTerm, l_event
   l_BinnedTerm, l_binned

   RETURN, SUCCESS

end
