
;-------------------------------------------------------------------------------
;
;  Unit Name : l_EvSpRtns.pro
;
;  Purpose   : All routines related to reading, writing, generating plots
;              using the Event sectored/Sun pulse packets.
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;   ELH                               v1.0           Original implementation
;
;
;  File Revision Number:   %I%
;  File Last Modified  :   %E%  %T%
;
;-------------------------------------------------------------------------------


;-------------------------------------------------------------------------------
;  Function : l_EvSpReadData
;
;  Description: Reads the sectored events/sun pulse data record by record. 
;
;  Return Value:  type = int
;  Value                         Description
;  -------------------------     ------------------------------
;  SUCCESS                       successful completion
;  FAILURE                       unsuccessful completion
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  inst                  string        I        instrument (LENA)
;  expr                  string        I        experiment (LENASCI or LENAHSKP)
;  vinst                 string        I        virtual instrument (names found in
;                                               $UDF_DATA) in either the LENASCI or
;                                               LENAHSKP of the IMAGE1 or IMAGE1Q 
;                                               directories
;  btime                 long[]        I        UDF begin time [yr, doy, hh, mm, ss]
;  etime                 long[]        I        UDF end time [yr, doy, hh, mm, ss]
;  btme                  double[]      O        UDF time in seconds
;  l_setup               struct        I        general setup paremeters
;  l_evsp                struct *     I/O       pointers to sectored event data structures
;  numRecs               long          I        number of data records to store
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;      ERROR_TYPES     SUCCESS         int         G          successful completion
;                      FAILURE         int         G          unsuccessful completion
;                      WARNING         int         G          warning message
;                      INFO            int         G          informational message
;
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;-------------------------------------------------------------------------------

FUNCTION l_EvSpReadData, inst, expr, vinst, btime, etime, l_setup, l_evsp, numRecs, $
data_source
COMMON ERROR_TYPES, SUCCESS, FAILURE, WARNING, INFO



   data_source = ' '
   fh = l_UdfGetDataHandle (inst, expr, vinst, btime, etime, data_source, /GROUP)
   if (fh eq -1 ) then begin
      udf_close, fh 
      RETURN, FAILURE 
   endif


   indx = 0L
   sunPulse = fltarr (l_setup.l_num_spinsec, numRecs)
   eventCnts = fltarr (l_setup.l_num_spinsec, numRecs)
   sctime = dblarr (numRecs)
   start_array = replicate ({UDF_TIMES_STRUCT}, numRecs)
   stop_array  = replicate ({UDF_TIMES_STRUCT}, numRecs)


   while not udf_eof (fh) do begin
     d          = udf_read (fh)
     sctime[indx]= l_GetTime (d.btime, doy, hr, min, sec)

     sp         = d.sun_pulse_sector[*] + 1350
     sp1        = sp mod 3600
     sp2        = sp1/3600.
     sun_pulse  = sp2 * 45
     sunPulse [*, indx] = sun_pulse[*]
     eventCnts[*,indx]  = d.sun_sector[*]

     start_array[indx].year = d.btime.year
     start_array[indx].doy  = d.btime.doy
     start_array[indx].hour = d.btime.hour
     start_array[indx].min  = d.btime.min
     start_array[indx].sec  = d.btime.sec
     start_array[indx].msec = d.btime.msec

     stop_array[indx].year = d.etime.year
     stop_array[indx].doy  = d.etime.doy
     stop_array[indx].hour = d.etime.hour
     stop_array[indx].min  = d.etime.min
     stop_array[indx].sec  = d.etime.sec
     stop_array[indx].msec = d.etime.msec

     indx = indx + 1L

   endwhile 


   udf_close, fh


   ptr = ptr_new (sunPulse[*,0:indx-1L], /NO_COPY)
   l_evsp.sunPulsePtr = ptr

   ptr = ptr_new (eventCnts[*,0:indx-1L], /NO_COPY)
   l_evsp.eventCntsPtr  = ptr

   ptr = ptr_new (sctime[0:indx-1L], /NO_COPY)
   l_evsp.timePtr  = ptr

   ptr = ptr_new (start_array[0:indx-1L], /NO_COPY)
   l_evsp.startUdfTimePtr = ptr

   ptr = ptr_new (stop_array[0:indx-1L], /NO_COPY)
   l_evsp.stopUdfTimePtr = ptr


   RETURN, SUCCESS 
END

;-------------------------------------------------------------------------------
;  Function :  l_EvSpGetData
;
;  Description: Determines the number of records needed and reads the sectored
;    events/sun pulse data.
;
;  Return Value:  type = int 
;  Value                         Description
;  -------------------------     ------------------------------
;  SUCCESS                       successful completion
;  FAILURE                       failure to complete
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_setup               struct        I        generic setup parameters
;  l_pltpos              struct        I        generic plot parameters
;  l_evsp                struct        I        sectored events/sun pulse
;                                               parameters and attributes
;  nele                  long          I        total # of records for this read
;  EvDmpFlg              long          I        if dump to a file, read extra spins
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;     PLOTDEFS         MAX_PLOTS        int         G         max # plots per page
;                      MAX_WEB_SPINS    int         G         max # spins per plot
;                                                             when using web interface
;                      NO_CNTS          double      G         no counts indicator
;                      NO_DATA          double      G         no data indicator
;
;      ERROR_TYPES     SUCCESS         int         G          successful completion
;                      FAILURE         int         G          unsuccessful completion
;                      WARNING         int         G          warning message
;                      INFO            int         G          informational message
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;-------------------------------------------------------------------------------

FUNCTION l_EvSpGetData, l_setup, l_pltpos, l_evsp, nele, EvDmpFlg
COMMON PLOTDEFS, MAX_PLOTS, MAX_WEB_SPINS, NO_CNTS, NO_DATA
COMMON ERROR_TYPES, SUCCESS, FAILURE, WARNING, INFO


   inst    = 'LENA' 
   expr    = 'LENASCI'
   vinst   = 'IMLEVSP' 

   data_source = ' '
   numRecs     = 0L

   ptr= ptr_new()
   l_evsp = {          $
      sunPulsePtr       :ptr,$
      eventCntsPtr      :ptr,$
      startUdfTimePtr   :ptr,$
      stopUdfTimePtr    :ptr,$
      timePtr           :ptr $
   }


   ;----------------------------------------------------
   ; read the udf data
   ;----------------------------------------------------
   start_tme = l_setup.l_start_dtime
   stop_tme = l_setup.l_stop_dtime

   ;----------------------------------------------------
   ; if the start time is not 00:00:00, then add extra
   ; minutes to the begin time
   ;----------------------------------------------------
   addedStartFlag = 0L
   if (ChckStartTime(start_tme) eq FAILURE) then begin
       AddExtraTime2Start, start_tme, 2
       addedStartFlag = 1L
   endif

   ;----------------------------------------------------
   ; if the stop time is not 23:59:59, then add extra
   ; minutes to the end time
   ;----------------------------------------------------
   addedStopFlag = 0L
   if (ChckStopTime(stop_tme) eq FAILURE) then begin
       if (EvDmpFlg eq 0L) then begin
          AddExtraTime2Stop, stop_tme, 2
       endif else begin
          AddExtraTime2Stop, stop_tme, 8 
       endelse
       addedStopFlag = 1L
   endif


   ; get the number of records from T1 to T2
   est_spins = l_UdfGetNumRec (start_tme, $
                             stop_tme,  $
                             inst, expr, vinst, data_source)

   if (est_spins le 0) then RETURN, FAILURE 

   status = l_EvSpReadData (inst, expr, vinst,     $
                             start_tme, $
                             stop_tme,  $
                             l_setup, $
                             l_evsp, $
                             est_spins, $
                             data_source)

   l_pltpos.data_source = data_source

   plotted = 0L
   pages   = 1L


   RETURN, status 

END

;-------------------------------------------------------------------------------
;  Procedure :  l_EvSpTerm
;
;  Description: Frees pointers allocated.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_evsp                struct        I        sectored events/sun pulse
;                                               specific parameters and
;                                               attributes
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;-------------------------------------------------------------------------------

PRO l_EvSpTerm, l_evsp


   if (ptr_valid (l_evsp.sunPulsePtr)) then begin
       ptr_free, l_evsp.sunPulsePtr
   endif

   if (ptr_valid (l_evsp.eventCntsPtr)) then begin
       ptr_free, l_evsp.eventCntsPtr
   endif

   if (ptr_valid (l_evsp.timePtr)) then begin
       ptr_free, l_evsp.timePtr
   endif

   if (ptr_valid (l_evsp.startUdfTimePtr)) then begin
       ptr_free, l_evsp.startUdfTimePtr
   endif

   if (ptr_valid (l_evsp.stopUdfTimePtr)) then begin
       ptr_free, l_evsp.stopUdfTimePtr
   endif


END


;-------------------------------------------------------------------------------
;  Procedure:  l_EvSpProc
;
;  Description:  Main calling procedure to call routines to process the sectored
;     event data. 
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_setup               struct        I        generic setup parameters
;  l_pltpos              struct        I        generic plot setup parameters
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;     PLOTDEFS         MAX_PLOTS        int         G         max # plots per page
;                      MAX_WEB_SPINS    int         G         max # spins per plot
;                                                             when using web interface
;                      NO_CNTS          double      G         no counts indicator
;                      NO_DATA          double      G         no data indicator
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;-------------------------------------------------------------------------------

PRO l_EvSpProc, l_setup, l_pltpos
COMMON PLOTDEFS, MAX_PLOTS, MAX_WEB_SPINS, NO_CNTS, NO_DATA

   nele = 0L
   status = l_EvSpGetData (l_setup, l_pltpos, l_evsp, nele)

   ;-------------------------------------------------
   ; free pointers
   ;-------------------------------------------------
   l_EvSpTerm, l_evsp

   RETURN

END
