
;-------------------------------------------------------------------------------
;
;  Unit Name : l_EchoRtns.pro
;
;  Purpose   : Contains routines related to the processing of the ECHO data.
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;   ELH              04/25/03        v1.0.11         Original implementation
;
;
;  File Revision Number:   %I%
;  File Last Modified  :   %E%  %T%
;
;-------------------------------------------------------------------------------

;-------------------------------------------------------------------------------
;  Procedure: l_GenSysCmdTable
;
;  Description:  Generate the system command table.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
; l_sys_cmd              struct       I/O       system command table
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------

PRO l_GenSysCmdTable, l_sys_cmd

    opcode = 256L

    l_sys_cmd_table = { SYS_CMD_TABLE, $
           header : 0.0, $          ; System command stem
           cmd    : ' '   $
    } 


    l_sys_cmd = replicate ({SYS_CMD_TABLE}, opcode)
    l_sys_cmd[*].header = 154. 

    ;----------------------------------------------
    ; Enable LENA autonomous performance test at
    ; boot (default)
    ;----------------------------------------------
    l_sys_cmd[129].cmd    = 'L_SYS_PTEST_EN'

    ;---------------------------------------------
    ; Disable LENA autonomous performance test at
    ; boot 
    ;----------------------------------------------
    l_sys_cmd[65].cmd    = 'L_SYS_PTEST_DSBL'

    ;---------------------------------------------
    ; LENA operational mode set to program RAM 
    ; (default)
    ;----------------------------------------------
    l_sys_cmd[113].cmd    = 'L_SYS_MODE_PRAM'

    ;---------------------------------------------
    ; LENA operational mode set to program ROM 
    ;----------------------------------------------
    l_sys_cmd[68].cmd    = 'L_SYS_MODE_ROM'

    ;---------------------------------------------
    ; LENA operational mode set to HV DISABLED
    ;----------------------------------------------
    l_sys_cmd[69].cmd    = 'L_SYS_MODE_HVD'

    ;---------------------------------------------
    ; LENA operational mode set to HV SAFE 
    ;----------------------------------------------
    l_sys_cmd[70].cmd    = 'L_SYS_MODE_HVSF'

    ;---------------------------------------------
    ; LENA operational mode set to NORMAL 
    ;----------------------------------------------
    l_sys_cmd[71].cmd    = 'L_SYS_MODE_NRML'

    ;---------------------------------------------
    ; Enable NORMAL hk packet transmission
    ;----------------------------------------------
    l_sys_cmd[160].cmd    = 'L_SYS_HKNRM_EN'

    ;---------------------------------------------
    ; Disable NORMAL hk packet transmission
    ;----------------------------------------------
    l_sys_cmd[96].cmd    = 'L_SYS_HKNRM_DSBL'

    ;---------------------------------------------
    ; Enable quicklook hk packet transmission
    ;----------------------------------------------
    l_sys_cmd[137].cmd    = 'L_SYS_HKQL_EN'

    ;---------------------------------------------
    ; Disable quicklook hk packet transmission
    ;----------------------------------------------
    l_sys_cmd[73].cmd    = 'L_SYS_HKQL_DSBL'

    ;---------------------------------------------
    ; Enable Command Record hk packet transmission
    ;----------------------------------------------
    l_sys_cmd[138].cmd    = 'L_SYS_HKCM_EN'

    ;---------------------------------------------
    ; Disable Command Record hk packet transmission
    ;----------------------------------------------
    l_sys_cmd[74].cmd    = 'L_SYS_HKCM_DSBL'

    ;---------------------------------------------
    ; Enable Diagnostic hk packet transmission
    ;----------------------------------------------
    l_sys_cmd[139].cmd    = 'L_SYS_HKDG_EN'

    ;---------------------------------------------
    ; Disable Diagnostic hk packet transmission
    ;----------------------------------------------
    l_sys_cmd[75].cmd    = 'L_SYS_HKDG_DSBL'

    ;---------------------------------------------
    ; Enable MEMORY hk packet transmission 
    ;----------------------------------------------
    l_sys_cmd[162].cmd    = 'L_SYS_HKMEM_EN'

    ;---------------------------------------------
    ; Disable MEMORY hk packet transmission 
    ;----------------------------------------------
    l_sys_cmd[97].cmd    = 'L_SYS_HKMEM_DSBL'

    ;---------------------------------------------
    ; Reset LENA
    ;----------------------------------------------
    l_sys_cmd[168].cmd    = 'L_SYS_INST_RST'

    ;---------------------------------------------
    ; Reset all error counters, registers and flags
    ;----------------------------------------------
    l_sys_cmd[171].cmd    = 'L_SYS_ERR_RST'

    ;---------------------------------------------
    ; Enable transmission of binned science data
    ; packet
    ;----------------------------------------------
    l_sys_cmd[141].cmd    = 'L_SYS_BIN_EN'

    ;---------------------------------------------
    ; Disable transmission of binned science data
    ; packet
    ;----------------------------------------------
    l_sys_cmd[77].cmd    = 'L_SYS_BIN_DSBL'

    ;---------------------------------------------
    ; Enable transmission of event science data
    ; packet
    ;----------------------------------------------
    l_sys_cmd[142].cmd    = 'L_SYS_EVENT_EN'

    ;---------------------------------------------
    ; Disable transmission of event science data
    ; packet
    ;----------------------------------------------
    l_sys_cmd[78].cmd    = 'L_SYS_EVENT_DSBL'

    ;---------------------------------------------
    ; Enable transmission of singles science data
    ; packet
    ;----------------------------------------------
    l_sys_cmd[143].cmd    = 'L_SYS_SNGLS_EN'

    ;---------------------------------------------
    ; Disable transmission of singles science data
    ; packet
    ;----------------------------------------------
    l_sys_cmd[79].cmd    = 'L_SYS_SNGLS_DSBL'

    ;---------------------------------------------
    ; Enable transmission of selected ROI segments
    ; indicated by binary pattern [bn] "1" in selected
    ; field enables segment transfer. "0" disables
    ; segment transfer.
    ;   [0000 b3b2b1b0]
    ; b0 = ROI0 : default=1
    ; b1 = ROI1 : default=0
    ; b2 = ROI2 : default=0
    ; b3 = ROI3 : default=0
    ;----------------------------------------------
    l_sys_cmd[76].cmd    = 'L_SYS_ROI_SEL'

    ;---------------------------------------------
    ; Enable performance test hk packet transmission 
    ;----------------------------------------------
    l_sys_cmd[177].cmd    = 'L_SYS_HKPTS_EN'

    ;---------------------------------------------
    ; Disable performance test hk packet transmission 
    ;----------------------------------------------
    l_sys_cmd[178].cmd    = 'L_SYS_HKPTS_DSBL'

    ;---------------------------------------------
    ; Enable hvps stripchart hk packet transmission 
    ;----------------------------------------------
    l_sys_cmd[33].cmd    = 'L_SYS_HKSTR_EN'

    ;---------------------------------------------
    ; Disable hvps stripchart hk packet transmission 
    ;----------------------------------------------
    l_sys_cmd[34].cmd    = 'L_SYS_HKSTR_DSBL'

    ;---------------------------------------------
    ; Enable transmission of normal-a hk packet
    ;----------------------------------------------
    l_sys_cmd[39].cmd    = 'L_SYS_HKNMA_EN'

    ;---------------------------------------------
    ; Disable transmission of normal-a hk packet
    ;----------------------------------------------
    l_sys_cmd[45].cmd    = 'L_SYS_HKNMA_DSBL'

    ;---------------------------------------------
    ; Request upload from CIDP block P
    ; [0<=P<=4] 
    ;----------------------------------------------
    l_sys_cmd[99].cmd    = 'L_SYS_UPLD_REQ'

    ;---------------------------------------------
    ; Enable internal sync signal 
    ;----------------------------------------------
    l_sys_cmd[149].cmd    = 'L_SYS_ISNC_EN'

    ;---------------------------------------------
    ; Disable internal sync signal
    ;----------------------------------------------
    l_sys_cmd[85].cmd    = 'L_SYS_ISNC_DSBL'

    ;---------------------------------------------
    ; Set overvoltage thresholds for high voltage
    ; power supplies.  Overvoltage limits V for
    ; each supply are parameterized, where
    ; 0<V<=FFH
    ;   AA - MCP_Start
    ;   BB - MCP Stop
    ;   CC - Collimator+
    ;   DD - Collimator-
    ;   EE - Optics
    ;----------------------------------------------
    l_sys_cmd[153].cmd    = 'L_SYS_HVP_VTHR'

    ;---------------------------------------------
    ; NOP command 
    ;----------------------------------------------
    l_sys_cmd[0].cmd    = 'L_SYS_NOP'

    ;---------------------------------------------
    ; Set overcurrent thresholds for high voltage
    ; power supplies.  Overcurrent limits I for
    ; each supply are parameterized, where
    ; 0<I<=FFH
    ;    AA - MCP Start
    ;    BB - MCP Stop
    ;    CC - Collimator+
    ;    DD - Collimator-
    ;    EE - Optics
    ;----------------------------------------------
    l_sys_cmd[157].cmd    = 'L_SYS_HVP_ITHR'

    ;---------------------------------------------
    ; Set start and stop rate overcount threshold
    ; to 8-bit parameter.  Scale factor=8.45kHz/count
    ; (0xFF default)
    ;----------------------------------------------
    l_sys_cmd[19].cmd    = 'L_SYS_OVCT_THR'

    ;---------------------------------------------
    ; Enable LENA Stop MCP hvps auto-restore
    ;----------------------------------------------
    l_sys_cmd[196].cmd    = 'L_SYS_RSTR_EN'

    ;---------------------------------------------
    ; Disable LENA Stop MCP hvps auto-restore
    ;----------------------------------------------
    l_sys_cmd[200].cmd    = 'L_SYS_RSTR_DSBL'

    ;---------------------------------------------
    ; Enable Science data acquisition
    ;----------------------------------------------
    l_sys_cmd[90].cmd    = 'L_SYS_ACQR_EN'

    ;---------------------------------------------
    ; Disable Science data acquisition
    ;----------------------------------------------
    l_sys_cmd[92].cmd    = 'L_SYS_ACQR_DSBL'

    ;---------------------------------------------
    ; Save LENA configuration state 
    ;----------------------------------------------
    l_sys_cmd[193].cmd    = 'L_SYS_STAT_SAVE'

    ;---------------------------------------------
    ; Restore LENA configuration state 
    ;----------------------------------------------
    l_sys_cmd[199].cmd    = 'L_SYS_STAT_RSTR'

    ;---------------------------------------------
    ; Set 16-bit hex address [A3 A2 A1 A0] of data 
    ; read by diagnostic register
    ;----------------------------------------------
    l_sys_cmd[53].cmd    = 'L_SYS_DADR_SET'

    ;---------------------------------------------
    ; Enable transmission of ROI summation sci
    ; packet 
    ;----------------------------------------------
    l_sys_cmd[188].cmd    = 'L_SYS_RSUM_EN'

    ;---------------------------------------------
    ; Disable transmission of ROI summation sci
    ; packet 
    ;----------------------------------------------
    l_sys_cmd[189].cmd    = 'L_SYS_RSUM_DSBL'

    ;---------------------------------------------
    ; Enable transmission of valid event by
    ; sector sci packet 
    ;----------------------------------------------
    l_sys_cmd[184].cmd    = 'L_SYS_SECEV_EN'

    ;---------------------------------------------
    ; Disable transmission of valid event by
    ; sector sci packet 
    ;----------------------------------------------
    l_sys_cmd[186].cmd    = 'L_SYS_SECEV_DSBL'

    ;---------------------------------------------
    ; Enable transmission of monitor hk packet 
    ;----------------------------------------------
    l_sys_cmd[183].cmd    = 'L_SYS_MON_EN'

    ;---------------------------------------------
    ; Disable transmission of monitor hk packet 
    ;----------------------------------------------
    l_sys_cmd[190].cmd    = 'L_SYS_MON_DSBL'



RETURN
END


;-------------------------------------------------------------------------------
;  Procedure: l_GenROICmdTable
;
;  Description: Generates the ROI command table
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_roi_cmd             struct        I/O      ROI commmand table
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------

PRO l_GenROICmdTable, l_roi_cmd

    opcode = 256L

    l_roi_cmd_table = { ROI_CMD_TABLE, $
           header : 0.0, $          ; HVPS command stem
           cmd    : ' '   $
    } 

    l_roi_cmd = replicate ({ROI_CMD_TABLE}, opcode)
    l_roi_cmd[*].header = 210.0

    ;---------------------------------------------
    ; Define TOF, Polar, and Energy bin ranges for
    ; Region-of-interest 0
    ;    RAA = TOF start bin of 255 F.S.
    ;    RBB = TOF bin width (1,2,4,16,32,or64)
    ;    CCCC = Polar start bin
    ;    DDDD = Polar Stop Bin
    ;    EE   = Energy start bin
    ;    FF   = Energy stop bin
    ;    GG   = Start spin angle
    ;    HH   = Stop spin angle
    ;----------------------------------------------
    l_roi_cmd[84].cmd    = 'L_ROI_CNFG_PKG0'

    ;---------------------------------------------
    ; Define TOF, Polar, and Energy bin ranges for
    ; Region-of-interest 1 
    ;    RAA = TOF start bin of 255 F.S.
    ;    RBB = TOF bin width (1,2,4,16,32,or64)
    ;    CCCC = Polar start bin
    ;    DDDD = Polar Stop Bin
    ;    EE   = Energy start bin
    ;    FF   = Energy stop bin
    ;    GG   = Start spin angle
    ;    HH   = Stop spin angle
    ;----------------------------------------------
    l_roi_cmd[88].cmd    = 'L_ROI_CNFG_PKG1'

    ;---------------------------------------------
    ; Define TOF, Polar, and Energy bin ranges for
    ; Region-of-interest 2 
    ;    RAA = TOF start bin of 255 F.S.
    ;    RBB = TOF bin width (1,2,4,16,32,or64)
    ;    CCCC = Polar start bin
    ;    DDDD = Polar Stop Bin
    ;    EE   = Energy start bin
    ;    FF   = Energy stop bin
    ;    GG   = Start spin angle
    ;    HH   = Stop spin angle
    ;----------------------------------------------
    l_roi_cmd[102].cmd    = 'L_ROI_CNFG_PKG2'

    ;---------------------------------------------
    ; Define TOF, Polar, and Energy bin ranges for
    ; Region-of-interest 3 
    ;    RAA = TOF start bin of 255 F.S.
    ;    RBB = TOF bin width (1,2,4,16,32,or64)
    ;    CCCC = Polar start bin
    ;    DDDD = Polar Stop Bin
    ;    EE   = Energy start bin
    ;    FF   = Energy stop bin
    ;    GG   = Start spin angle
    ;    HH   = Stop spin angle
    ;----------------------------------------------
    l_roi_cmd[104].cmd    = 'L_ROI_CNFG_PKG3'

RETURN
END


;-------------------------------------------------------------------------------
;  Procedure: l_GenPerfCmdTable
;
;  Description: Generates the performance command table.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_perf_cmd            struct       I/O       performance command table
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------

PRO l_GenPerfCmdTable, l_perf_cmd

    opcode = 256L

    l_perf_cmd_table = { PERF_CMD_TABLE, $
           header : 0.0, $          
           cmd    : ' '   $
    } 

    l_perf_cmd = replicate ({PERF_CMD_TABLE}, opcode)
    l_perf_cmd[*].header = 86.0

    ;----------------------------------------------
    ; Enable low voltage performance test
    ;----------------------------------------------
    l_perf_cmd[129].cmd   = 'L_TST_LV_EN'

    ;----------------------------------------------
    ; Enable high voltage performance test
    ;----------------------------------------------
    l_perf_cmd[65].cmd   = 'L_TST_HV_EN'

    ;----------------------------------------------
    ; Enable MCP start HV power supply test.
    ; Parameter V0 is max test voltage
    ;----------------------------------------------
    l_perf_cmd[132].cmd   = 'L_TST_MSTR_EN'

    ;----------------------------------------------
    ; Disable MCP start HV power supply test.
    ; Parameter V0 is max test voltage
    ;----------------------------------------------
    l_perf_cmd[68].cmd   = 'L_TST_MSTR_DSBL'

    ;----------------------------------------------
    ; Enable MCP stop HV power supply test.
    ; Parameter V0 is max test voltage
    ;----------------------------------------------
    l_perf_cmd[149].cmd   = 'L_TST_MSTP_EN'

    ;----------------------------------------------
    ; Disable MCP stop HV power supply test.
    ;----------------------------------------------
    l_perf_cmd[85].cmd   = 'L_TST_MSTP_DSBL'

    ;----------------------------------------------
    ; Enable Collimator Positive HV power supply
    ; test. Parameter V0 is max test voltage.
    ;----------------------------------------------
    l_perf_cmd[150].cmd   = 'L_TST_COLP_EN'

    ;----------------------------------------------
    ; Disable Collimator Positive HV power supply test.
    ;----------------------------------------------
    l_perf_cmd[86].cmd   = 'L_TST_COLP_DSBL'

    ;----------------------------------------------
    ; Enable Collimator Negative HV power supply
    ; test. Parameter V0 is max test voltage.
    ;----------------------------------------------
    l_perf_cmd[135].cmd   = 'L_TST_COLN_EN'

    ;----------------------------------------------
    ; Disable Collimator Negative HV power supply test.
    ;----------------------------------------------
    l_perf_cmd[71].cmd   = 'L_TST_COLN_DSBL'

    ;----------------------------------------------
    ; Enable Optics HV power supply
    ; test. Parameter V0 is max test voltage.
    ;----------------------------------------------
    l_perf_cmd[136].cmd   = 'L_TST_OPT_EN'

    ;----------------------------------------------
    ; Disable Optics HV power supply test.
    ;----------------------------------------------
    l_perf_cmd[72].cmd   = 'L_TST_OPT_DSBL'

    ;----------------------------------------------
    ; Enable Optics Steering HV power supply
    ; test. Parameter V0 is max test voltage.
    ;----------------------------------------------
    l_perf_cmd[153].cmd   = 'L_TST_STR_EN'

    ;----------------------------------------------
    ; Disable Optics Steering HV power supply test.
    ;----------------------------------------------
    l_perf_cmd[89].cmd   = 'L_TST_STR_DSBL'


RETURN
END

;-------------------------------------------------------------------------------
;  Procedure: l_GenTOFCmdTable
;
;  Description: Generate the time of flight command table.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_tof_cmd             struct       I/O       time of flight command table
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------


PRO l_GenTOFCmdTable, l_tof_cmd

    opcode = 256L

    l_tof_cmd_table = { TOF_CMD_TABLE, $
           header : 0.0, $          
           cmd    : ' '   $
    } 

    l_tof_cmd = replicate ({TOF_CMD_TABLE}, opcode)
    l_tof_cmd[*].header = 182.0

    ;----------------------------------------------
    ; Enable TOF start-stop mode (default)
    ;----------------------------------------------
    l_tof_cmd[51].cmd   = 'L_TOF_MODE_STST'

    ;----------------------------------------------
    ; Enable TOF start-only mode (default)
    ;----------------------------------------------
    l_tof_cmd[3].cmd   = 'L_TOF_MODE_STRT'

    ;----------------------------------------------
    ; Define start TOF definition for species_1
    ; binning interval. [P1P0] - species1 start
    ; threshold 
    ; 0 < threshold <= FFH
    ; where threshold is 10bit TOF bin/4
    ;----------------------------------------------
    l_tof_cmd[33].cmd   = 'L_TOF_BIN_STRT1'

    ;----------------------------------------------
    ; Define start TOF definition for species_1
    ; binning interval. [P1P0] - species1 start
    ; threshold 
    ; 0 < threshold <= FFH
    ; where threshold is 10bit TOF bin/4
    ;----------------------------------------------
    l_tof_cmd[34].cmd   = 'L_TOF_BIN_STOP1'

    ;----------------------------------------------
    ; Define start TOF definition for species_2
    ; binning interval. [P1P0] - species2 start
    ; threshold 
    ; 0 < threshold <= FFH
    ; where threshold is 10bit TOF bin/4
    ;----------------------------------------------
    l_tof_cmd[35].cmd   = 'L_TOF_BIN_STRT2'

    ;----------------------------------------------
    ; Define start TOF definition for species_2
    ; binning interval. [P1P0] - species2 start
    ; threshold 
    ; 0 < threshold <= FFH
    ; where threshold is 10bit TOF bin/4
    ;----------------------------------------------
    l_tof_cmd[36].cmd   = 'L_TOF_BIN_STOP2'

    ;----------------------------------------------
    ; Enable TOF built-in-test mode
    ;----------------------------------------------
    l_tof_cmd[145].cmd   = 'L_TOF_BIT_ON'

    ;----------------------------------------------
    ; Disable TOF built-in-test mode
    ;----------------------------------------------
    l_tof_cmd[81].cmd   = 'L_TOF_BIT_OFF'

    ;----------------------------------------------
    ; Route single strobe to TOF system. Active in
    ; BIT mode only
    ;----------------------------------------------
    l_tof_cmd[82].cmd   = 'L_TOF_BIT_STRB'

    ;----------------------------------------------
    ; Define TOF BIT mode delay.  Active in BIT 
    ; mode only.  
    ; 0 < delay <= FFH
    ;----------------------------------------------
    l_tof_cmd[83].cmd   = 'L_TOF_BIT_DELAY'

    ;----------------------------------------------
    ; Define TOF CFD start threshold
    ; 0 < threshold <= FH
    ;----------------------------------------------
    l_tof_cmd[97].cmd   = 'L_TOF_TRSH_START'

    ;----------------------------------------------
    ; Define TOF CFD stop threshold
    ; 0 < threshold <= FH
    ;----------------------------------------------
    l_tof_cmd[98].cmd   = 'L_TOF_TRSH_STOP'

RETURN
END



;-------------------------------------------------------------------------------
;  Procedure: l_GenPSSCmdTable
;
;  Description: Generate the PSS command table.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_pss_cmd             struct       I/O       pss command table
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------

PRO l_GenPSSCmdTable, l_pss_cmd

    opcode = 256L

    l_pss_cmd_table = { PSS_CMD_TABLE, $
           header : 0.0, $ 
           cmd    : ' '   $
    } 

    l_pss_cmd = replicate ({PSS_CMD_TABLE}, opcode)
    l_pss_cmd[*].header = 199.0

    ;----------------------------------------------
    ; Set gain of position anoides to LOW gain (default) 
    ;----------------------------------------------
    l_pss_cmd[17].cmd   = 'L_PSS_GAIN_LOW'

    ;----------------------------------------------
    ; Set gain of position anoides to HIGH gain 
    ;----------------------------------------------
    l_pss_cmd[18].cmd   = 'L_PSS_GAIN_HIGH'

RETURN
END


;-------------------------------------------------------------------------------
;  Procedure: l_GenMemCmdTable
;
;  Description: Generate the memory command table.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_mem_cmd             struct       I/O       memory command table
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------

PRO l_GenMemCmdTable, l_mem_cmd

    opcode = 256L

    l_mem_cmd_table = { MEM_CMD_TABLE, $
           header : 0.0, $ 
           cmd    : ' '   $
    } 

    l_mem_cmd = replicate ({MEM_CMD_TABLE}, opcode)
    l_mem_cmd[*].header = 233.0

    ;----------------------------------------------
    ; Read LENA memory space. Transfer [N3N2N1N0] 
    ; (hex) 16-bit words beginning at 16 bit hex
    ; address [A3A2A1A0]
    ;----------------------------------------------
    l_mem_cmd[10].cmd   = 'L_MEM_WORD_RD'

    ;----------------------------------------------
    ; Write LENA memory space. Transfer [N3N2N1N0] 
    ; (hex) 16-bit words beginning at 16 bit hex
    ; address [A3A2A1A0]
    ;----------------------------------------------
    l_mem_cmd[9].cmd   = 'L_MEM_WORD_WR'

    ;----------------------------------------------
    ; Read LENA memory space. Transfer [N3N2N1N0]
    ; (hex) bytes beginning at 16 bit hex address
    ; [A3A2A1A0]
    ;----------------------------------------------
    l_mem_cmd[6].cmd   = 'L_MEM_BYTE_RD'

    ;----------------------------------------------
    ; Write LENA memory space. Transfer [N3N2N1N0]
    ; (hex) bytes beginning at 16 bit hex address
    ; [A3A2A1A0]
    ;----------------------------------------------
    l_mem_cmd[5].cmd   = 'L_MEM_BYTE_WR'

    ;----------------------------------------------
    ; Read program ram space. Transfer [N3N2N1N0]
    ; (hex) bytes beginning at 16 bit hex address
    ; [A3A2A1A0]
    ;----------------------------------------------
    l_mem_cmd[57].cmd   = 'L_MEM_PRAM_RD'

RETURN
END


;-------------------------------------------------------------------------------
;  Procedure: l_GenHvpsCmdTable
;
;  Description:  Generate the HVPS command table.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_hvps_cmd            struct       I/O       HVPS command table
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------

PRO l_GenHvpsCmdTable, l_hvps_cmd

    ;opcode = 256L
    opcode = 512L

    l_hvps_cmd_table = { HVPS_CMD_TABLE, $
           header : 0.0, $          ; HVPS command stem
           nbytes : 0L,   $
           cmd    : ' '   $
    } 


    l_hvps_cmd = replicate ({HVPS_CMD_TABLE}, opcode)
    l_hvps_cmd[*].header = 165.0


    ;----------------------------------------------
    ; Enable or disable MCP Start HV Power Supply 
    ;----------------------------------------------
    l_hvps_cmd[128].nbytes = 2
    l_hvps_cmd[128].cmd    = 'L_HVP_MSTR_EN'

    ;----------------------------------------------
    ; Disable MCP Start HV Power Supply
    ;----------------------------------------------
    l_hvps_cmd[64].nbytes = 2
    l_hvps_cmd[64].cmd    = 'L_HVP_MSTR_DSBL'

    ;----------------------------------------------
    ; Enable MCP Stop HV Power Supply
    ;----------------------------------------------
    l_hvps_cmd[129].nbytes = 2
    l_hvps_cmd[129].cmd    = 'L_HVP_MSTP_EN'

    ;----------------------------------------------
    ; Disable MCP Stop HV Power Supply
    ;----------------------------------------------
    l_hvps_cmd[65].nbytes = 2
    l_hvps_cmd[65].cmd    = 'L_HVP_MSTP_DSBL'

    ;----------------------------------------------
    ; Enable Collimator Positive HV Power Supply 
    ;----------------------------------------------
    l_hvps_cmd[130].nbytes = 2
    l_hvps_cmd[130].cmd    = 'L_HVP_COLP_EN'

    ;----------------------------------------------
    ; Disable Collimator Positive HV Power Supply 
    ;----------------------------------------------
    l_hvps_cmd[66].nbytes = 2
    l_hvps_cmd[66].cmd    = 'L_HVP_COLP_DSBL'

    ;----------------------------------------------
    ; Enable Collimator Negative HV Power Supply 
    ;----------------------------------------------
    l_hvps_cmd[131].nbytes = 2
    l_hvps_cmd[131].cmd    = 'L_HVP_COLN_EN'

    ;----------------------------------------------
    ; Disable Collimator Negative HV Power Supply 
    ;----------------------------------------------
    l_hvps_cmd[67].nbytes = 2
    l_hvps_cmd[67].cmd    = 'L_HVP_COLN_DSBL'

    ;----------------------------------------------
    ; Enable Ion Optics HV Power Supply 
    ;----------------------------------------------
    l_hvps_cmd[132].nbytes = 2
    l_hvps_cmd[132].cmd    = 'L_HVP_OPT_EN'

    ;----------------------------------------------
    ; Disable Ion Optics HV Power Supply 
    ;----------------------------------------------
    l_hvps_cmd[68].nbytes = 2
    l_hvps_cmd[68].cmd    = 'L_HVP_OPT_DSBL'

    ;----------------------------------------------
    ; Limit Max HVPS output to 20% of F.S.
    ;----------------------------------------------
    l_hvps_cmd[31].nbytes = 2
    l_hvps_cmd[31].cmd    = 'L_HVP_GBL_SAFE'

    ;----------------------------------------------
    ; Max HVPS outputs are limited to 100% of F.S.
    ;----------------------------------------------
    l_hvps_cmd[159].nbytes = 2
    l_hvps_cmd[159].cmd    = 'L_HVP_GBL_USAFE'

    ;----------------------------------------------
    ; Send voltage command [V1V0] hex to MCP Start
    ; HV power supply 0<V<=FFH 
    ;----------------------------------------------
    l_hvps_cmd[96].nbytes = 3 
    l_hvps_cmd[96].cmd    = 'L_HVP_MSTR_CMD'

    ;----------------------------------------------
    ; Send voltage command [V1V0] hex to MCP Stop
    ; HV power supply  0<V<=FFH
    ;----------------------------------------------
    l_hvps_cmd[97].nbytes = 3 
    l_hvps_cmd[97].cmd    = 'L_HVP_MSTP_CMD'

    ;----------------------------------------------
    ; Send voltage command [V1V0] hex to Collimator 
    ; Positive HV power supply  0<V<=FFH
    ;----------------------------------------------
    l_hvps_cmd[98].nbytes = 3 
    l_hvps_cmd[98].cmd    = 'L_HVP_COLP_CMD'

    ;----------------------------------------------
    ; Send voltage command [V1V0] hex to Collimator 
    ; Negative HV power supply  0<V<=FFH
    ;----------------------------------------------
    l_hvps_cmd[99].nbytes = 3 
    l_hvps_cmd[99].cmd    = 'L_HVP_COLN_CMD'

    ;----------------------------------------------
    ; Send voltage command [V1V0] hex to Ion 
    ; Optics HV power supply  0<V<=FFH
    ;----------------------------------------------
    L_hvps_cmd[100].nbytes = 3 
    l_hvps_cmd[100].cmd    = 'L_HVP_OPT_CMD'

    ;----------------------------------------------
    ; Send voltage command [V1V0] hex to Optics 
    ; steering HV power supply  0<V<=FFH
    ;----------------------------------------------
    l_hvps_cmd[101].nbytes = 3 
    l_hvps_cmd[101].cmd    = 'L_HVP_STR_CMD'

    ;----------------------------------------------
    ; Execute HVPS shutdown macro (slew to 0V,
    ; safe, disable)
    ;----------------------------------------------
    l_hvps_cmd[111].nbytes = 2 
    l_hvps_cmd[111].cmd    = 'L_HVP_GBL_OFF'

    ;----------------------------------------------
    ; Enable steering supply scan
    ;----------------------------------------------
    l_hvps_cmd[209].nbytes = 2 
    l_hvps_cmd[209].cmd    = 'L_HVP_SCN_EN'

    ;----------------------------------------------
    ; Disable steering supply scan
    ;----------------------------------------------
    l_hvps_cmd[210].nbytes = 2 
    l_hvps_cmd[210].cmd    = 'L_HVP_SCN_DSBL'

    ;----------------------------------------------
    ; Steering scan table is defined by array 
    ; P[15] - P[0].  Steering supply steps
    ; through 1 command P[n] per spin
    ;----------------------------------------------
    l_hvps_cmd[215].nbytes = 18 
    l_hvps_cmd[215].cmd    = 'L_HVP_SCN_LD'


RETURN
END


;-------------------------------------------------------------------------------
;  Function: l_EchoReadData
;
;  Description:  Read the ECHO data. 
;
;  Return Value:  type = int
;  Value                         Description
;  -------------------------     ------------------------------
;  SUCCESS                       successful completion
;  FAILURE                       unsuccessful completion
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  inst                  string        I        instrument
;  expr                  string        I        experiement
;  vinst                 string        I        virtual instrument
;  btime                 long[]        I        user specified begin time
;  etime                 long[]        I        user specified end time
;  l_echo                struct *     I/O       pointer to echo data structures 
;  numRecs               long          I        number of records to read
;  data_source           string       I/O       definitive or QL data used
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;      ERROR_TYPES     SUCCESS         int         G          successful completion
;                      FAILURE         int         G          unsuccessful completion
;                      WARNING         int         G          warning message
;                      INFO            int         G          informational message
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------

FUNCTION l_EchoReadData, inst, expr, vinst, btime, etime, l_echo, numRecs, data_source
COMMON ERROR_TYPES, SUCCESS, FAILURE, WARNING, INFO


    fh = l_UdfGetDataHandle (inst, expr, vinst, btime, etime, data_source, /GROUP)
    if (fh eq -1) then begin
       udf_close, fh
       RETURN, FAILURE
    endif

    sctime  = dblarr (numRecs)
    data    = fltarr (256, numRecs)
    data_recs_in_seq = fltarr (numRecs)
    data_recs_seq_num = fltarr (numRecs)
    num_cmds_exe     = fltarr (numRecs)

    start_time_arr = replicate ({UDF_TIMES_STRUCT}, numRecs)
    stop_time_arr  = replicate ({UDF_TIMES_STRUCT}, numRecs)


    indx = 0L

    while not udf_eof (fh) do begin
       d = udf_read (fh)

       bt = l_GetTime (d.btime, doy, hr, min, sec)
  
       sctime[indx] = bt

       start_time_arr[indx].year = d.btime.year
       start_time_arr[indx].doy  = d.btime.doy
       start_time_arr[indx].hour = d.btime.hour
       start_time_arr[indx].min  = d.btime.min
       start_time_arr[indx].sec  = d.btime.sec
       start_time_arr[indx].msec = d.btime.msec

       stop_time_arr[indx].year = d.etime.year
       stop_time_arr[indx].doy  = d.etime.doy
       stop_time_arr[indx].hour = d.etime.hour
       stop_time_arr[indx].min  = d.etime.min
       stop_time_arr[indx].sec  = d.etime.sec
       stop_time_arr[indx].msec = d.etime.msec

       data[*, indx] = d.telecommand_echo[*]
       data_recs_in_seq[indx] = d.data_recs_in_sequence[0]
       data_recs_seq_num[indx] = d.data_rec_sequence_num[0]
       num_cmds_exe [indx] = d.num_commands_executed[0]

       indx = indx + 1L

    endwhile

    udf_close, fh

    ptr = ptr_new (sctime[0:indx-1L], /NO_COPY)
    l_echo.timePtr = ptr

    ptr = ptr_new (start_time_arr[0:indx-1L], /NO_COPY)
    l_echo.startUdfTimePtr= ptr

    ptr = ptr_new (stop_time_arr[0:indx-1L], /NO_COPY)
    l_echo.stopUdfTimePtr= ptr

    ptr = ptr_new (data[*, 0:indx-1L], /NO_COPY)
    l_echo.dataPtr = ptr

    ptr = ptr_new (data_recs_in_seq[0:indx-1L], /NO_COPY)
    l_echo.data_recs_in_seq = ptr

    ptr = ptr_new (data_recs_seq_num[0:indx-1L], /NO_COPY)
    l_echo.data_recs_seq_num = ptr

    ptr = ptr_new (num_cmds_exe[0:indx-1L], /NO_COPY)
    l_echo.num_cmds_exe = ptr



RETURN, SUCCESS
END


;-------------------------------------------------------------------------------
;  Function: l_EchoGetData
;
;  Description:  Initialize and setup data structures for storage of the echo data.
;
;  Return Value:  type = int
;  Value                         Description
;  -------------------------     ------------------------------
;  SUCCESS                       successful completion
;  FAILURE                       unsuccessful completion
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_setup               struct        I        general setup paramaters
;  l_setup_file          string        I        setup file name
;  l_pltpos              struct        I        general plotting parameters
;  l_echo                struct *     I/O       pointers to echo data structures
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;     PLOTDEFS         MAX_PLOTS        int         G         max # plots per page
;                      MAX_WEB_SPINS    int         G         max # spins per plot
;                                                             when using web interface
;                      NO_CNTS          double      G         no counts indicator
;                      NO_DATA          double      G         no data indicator
;
;      ERROR_TYPES     SUCCESS         int         G          successful completion
;                      FAILURE         int         G          unsuccessful completion
;                      WARNING         int         G          warning message
;                      INFO            int         G          informational message
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------

FUNCTION l_EchoGetData, l_setup, l_setup_file, l_pltpos, l_echo
COMMON PLOTDEFS, MAX_PLOTS, MAX_WEB_SPINS, NO_CNTS, NO_DATA
COMMON ERROR_TYPES, SUCCESS, FAILURE, WARNING, INFO



   inst    = 'LENA'
   expr    = 'LENAHSKP'
   vinst   = 'IMLECHO'

   
   data_source = ' '
   numRecs = 0L
   status = SUCCESS 


   ptr = ptr_new ()

   l_echo = {                   $
       nitems            : 0L,  $
       timePtr           : ptr, $
       startUdfTimePtr   : ptr, $
       stopUdfTimePtr    : ptr, $
       data_recs_in_seq  : ptr, $
       data_recs_seq_num : ptr, $
       num_cmds_exe      : ptr, $
       dataPtr           : ptr, $ 
       sub_spinTime      : ptr, $
       sub_spinUdfTime   : ptr, $ 
       nevents_cmded     : ptr, $
       l_hvp_mstr_en     : ptr, $
       l_hvp_mstr_dsbl   : ptr, $
       l_hvp_mstp_en     : ptr, $
       l_hvp_mstp_dsbl   : ptr, $
       l_hvp_colp_en     : ptr, $
       l_hvp_colp_dsbl   : ptr, $
       l_hvp_coln_en     : ptr, $
       l_hvp_coln_dsbl   : ptr, $
       l_hvp_opt_en      : ptr, $
       l_hvp_opt_dsbl    : ptr, $
       l_hvp_gbl_safe    : ptr, $
       l_hvp_gbl_usafe   : ptr, $
       l_hvp_mstr_cmd    : ptr, $
       l_hvp_mstp_cmd    : ptr, $
       l_hvp_colp_cmd    : ptr, $
       l_hvp_coln_cmd    : ptr, $
       l_hvp_opt_cmd     : ptr, $
       l_hvp_str_cmd     : ptr, $
       l_hvp_gbl_off     : ptr, $
       l_hvp_scn_en      : ptr, $
       l_hvp_scn_dsbl    : ptr, $
       l_hvp_scn_ld      : ptr  $
   }

   
   start_time = l_setup.l_start_dtime
   stop_time = l_setup.l_stop_dtime

   numRecs = l_UdfGetNumRec (start_time, $
                             stop_time, $
                             inst, expr, $
                             vinst, $
                             data_source)
   if (numRecs le 0) then return, FAILURE


   l_echo.nitems = numRecs

   status = l_EchoReadData (inst, expr, vinst, $
                            start_time, $
                            stop_time, $
                            l_echo, numRecs, $
                            data_source)
   

   RETURN, status 
END

;-------------------------------------------------------------------------------
;  Procedure: l_EchoProcData
;
;  Description:  Call procedures to do the echo data processing.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_echo                struct *     I/O       pointers to echo data structures
;  l_hvps_cmd            struct       I/O       HVPS command table
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------


PRO l_EchoProcData, l_echo, l_hvps_cmd

    eitems = l_echo.nitems * 85


    ;-----------------------------------------------------------------
    ; initialize the arrays
    ;-----------------------------------------------------------------
    udfTime         = replicate ({UDF_TIMES_STRUCT}, eitems)
    time            = dblarr (eitems)
    nevents_cmded   = intarr (eitems)
    l_hvp_mstr_en   = intarr (eitems)
    l_hvp_mstr_dsbl = intarr (eitems)
    l_hvp_mstp_en   = intarr (eitems)
    l_hvp_mstp_dsbl = intarr (eitems)
    l_hvp_colp_en   = intarr (eitems)
    l_hvp_colp_dsbl = intarr (eitems)
    l_hvp_coln_en   = intarr (eitems)
    l_hvp_coln_dsbl = intarr (eitems)
    l_hvp_opt_en    = intarr (eitems)
    l_hvp_opt_dsbl  = intarr (eitems)
    l_hvp_gbl_safe  = intarr (eitems)
    l_hvp_gbl_usafe = intarr (eitems)
    l_hvp_mstr_cmd  = fltarr (eitems)
    l_hvp_mstp_cmd  = fltarr (eitems)
    l_hvp_colp_cmd  = fltarr (eitems)
    l_hvp_coln_cmd  = fltarr (eitems)
    l_hvp_opt_cmd   = fltarr (eitems)
    l_hvp_str_cmd   = fltarr (eitems)
    l_hvp_gbl_off   = intarr (eitems)
    l_hvp_scn_en    = intarr (eitems)
    l_hvp_scn_dsbl  = intarr (eitems)
    l_hvp_scn_ld    = fltarr (18L, eitems)

    indx = 0L

    ;-----------------------------------------------------------------
    ; determine which command  executed from the command table and its
    ; corresponding value
    ;    For example: from the HVPS command table (in hex)
    ;      sequence : A5 81 A5 60 11
    ;      A5 81 ==> HVPS command table, MCP stop HV power supply enabled 
    ;      A5 60 11 ==> HVPS command table, MCP stop voltage command sent
    ;                   is hex 11
    ;-----------------------------------------------------------------
    for nspins = 0L, l_echo.nitems - 1L do begin
       subset = (*l_echo.dataPtr)[0L:(*l_echo.num_cmds_exe)[nspins], nspins]
       nsub = n_elements (subset)
       ;header_indx = where (subset eq l_hvps_cmd[nspins].header, hcnt) 
       header_indx = where (subset eq l_hvps_cmd[0].header, hcnt) 
       if (hcnt gt 0) then begin
          nitems = n_elements (header_indx)
          for n = 0L, nitems - 1L do begin
           if ((header_indx[n] + 1L) lt nsub) then begin
              opcode = subset[header_indx[n]+1L] 
              time[indx] = (*l_echo.timePtr)[nspins]
              nevents_cmded[indx] = nitems
              udfTime[indx] = (*l_echo.startUdfTimePtr)[nspins]
              case l_hvps_cmd[opcode].cmd of 
                'L_HVP_MSTR_EN'   : begin
                                      l_hvp_mstr_en[indx] = 1L
                                    end
                'L_HVP_MSTR_DSBL' : begin
                                      l_hvp_mstr_dsbl[indx] = 1L
                                    end
                'L_HVP_MSTP_EN'   : begin
                                      l_hvp_mstp_en[indx] = 1L
                                    end
                'L_HVP_MSTP_DSBL' : begin
                                      l_hvp_mstp_dsbl[indx] = 1L
                                    end
                'L_HVP_COLP_EN'   : begin
                                      l_hvp_colp_en[indx] = 1L
                                    end
                'L_HVP_COLP_DSBL' : begin
                                      l_hvp_colp_dsbl[indx] = 1L
                                    end
                'L_HVP_COLN_EN'   : begin
                                      l_hvp_coln_en[indx] = 1L
                                    end
                'L_HVP_COLN_DSBL' : begin
                                      l_hvp_coln_dsbl[indx] = 1L
                                    end
                'L_HVP_OPT_EN'    : begin
                                      l_hvp_opt_en[indx] = 1L
                                    end
                'L_HVP_OPT_DSBL'  : begin
                                      l_hvp_opt_dsbl[indx] = 1L
                                    end
                'L_HVP_GBL_SAFE'  : begin
                                      l_hvp_gbl_safe[indx] = 1L
                                    end
                'L_HVP_GBL_USAFE' : begin
                                      l_hvp_gbl_usafe[indx] = 1L
                                    end
                'L_HVP_MSTR_CMD'  : begin
                                      l_hvp_mstr_cmd[indx] = subset[header_indx[n]+2L]  
                                    end
                'L_HVP_MSTP_CMD'  : begin
                                      l_hvp_mstp_cmd[indx] = subset[header_indx[n]+2L]  
                                    end
                'L_HVP_COLP_CMD'  : begin
                                      l_hvp_colp_cmd[indx] = subset[header_indx[n]+2L]  
                                    end
                'L_HVP_COLN_CMD'  : begin
                                      l_hvp_coln_cmd[indx] = subset[header_indx[n]+2L]  
                                    end
                'L_HVP_OPT_CMD'   : begin
                                      l_hvp_opt_cmd[indx] = subset[header_indx[n]+2L]  
                                    end
                'L_HVP_STR_CMD'   : begin
                                      l_hvp_str_cmd[indx] = subset[header_indx[n]+2L]  
                                    end
                'L_HVP_GBL_OFF'   : begin
                                      l_hvp_gbl_off[indx] = 1L  
                                    end
                'L_HVP_SCN_EN'    : begin
                                      l_hvp_scn_en[indx] = 1L  
                                    end
                'L_HVP_SCN_DSBL'  : begin
                                      l_hvp_scn_dsbl[indx] = 1L  
                                    end
                'L_HVP_SCN_LD'    : begin
                                      l_hvp_scn_ld[indx] = subset[header_indx[n]+18L]  
                                    end
              else : begin
                       print, 'Invalid HVPS command : ', l_hvps_cmd[opcode].cmd
                     end
              endcase

              indx = indx + 1L
            endif 
          endfor ;--- end n --- 
       endif
    endfor ;--- end nspins ---


    l_echo.nitems = indx - 1L

    ptr = ptr_new (time[0:indx-1L], /NO_COPY)
    l_echo.sub_spinTime = ptr

    ptr = ptr_new (udfTime[0:indx-1L], /NO_COPY)
    l_echo.sub_spinUdfTime = ptr

    ptr = ptr_new (nevents_cmded[0:indx-1L], /NO_COPY)
    l_echo.nevents_cmded = ptr

    ptr = ptr_new (l_hvp_mstr_en[0:indx-1L], /NO_COPY)
    l_echo.l_hvp_mstr_en = ptr

    ptr = ptr_new (l_hvp_mstr_dsbl[0:indx-1L], /NO_COPY)
    l_echo.l_hvp_mstr_dsbl = ptr

    ptr = ptr_new (l_hvp_mstp_en[0:indx-1L], /NO_COPY)
    l_echo.l_hvp_mstp_en = ptr

    ptr = ptr_new (l_hvp_mstp_dsbl[0:indx-1L], /NO_COPY)
    l_echo.l_hvp_mstp_dsbl = ptr

    ptr = ptr_new (l_hvp_colp_en[0:indx-1L], /NO_COPY)
    l_echo.l_hvp_colp_en = ptr

    ptr = ptr_new (l_hvp_colp_dsbl[0:indx-1L], /NO_COPY)
    l_echo.l_hvp_colp_dsbl = ptr

    ptr = ptr_new (l_hvp_coln_en[0:indx-1L], /NO_COPY)
    l_echo.l_hvp_coln_en = ptr

    ptr = ptr_new (l_hvp_coln_dsbl[0:indx-1L], /NO_COPY)
    l_echo.l_hvp_coln_dsbl = ptr

    ptr = ptr_new (l_hvp_opt_en[0:indx-1L], /NO_COPY)
    l_echo.l_hvp_opt_en = ptr

    ptr = ptr_new (l_hvp_opt_dsbl[0:indx-1L], /NO_COPY)
    l_echo.l_hvp_opt_dsbl = ptr

    ptr = ptr_new (l_hvp_gbl_safe[0:indx-1L], /NO_COPY)
    l_echo.l_hvp_gbl_safe = ptr

    ptr = ptr_new (l_hvp_gbl_usafe[0:indx-1L], /NO_COPY)
    l_echo.l_hvp_gbl_usafe = ptr

    ptr = ptr_new (l_hvp_mstr_cmd[0:indx-1L], /NO_COPY)
    l_echo.l_hvp_mstr_cmd = ptr

    ptr = ptr_new (l_hvp_mstp_cmd[0:indx-1L], /NO_COPY)
    l_echo.l_hvp_mstp_cmd = ptr

    ptr = ptr_new (l_hvp_colp_cmd[0:indx-1L], /NO_COPY)
    l_echo.l_hvp_colp_cmd = ptr

    ptr = ptr_new (l_hvp_coln_cmd[0:indx-1L], /NO_COPY)
    l_echo.l_hvp_coln_cmd = ptr

    ptr = ptr_new (l_hvp_opt_cmd[0:indx-1L], /NO_COPY)
    l_echo.l_hvp_opt_cmd = ptr

    ptr = ptr_new (l_hvp_str_cmd[0:indx-1L], /NO_COPY)
    l_echo.l_hvp_str_cmd = ptr

    ptr = ptr_new (l_hvp_gbl_off[0:indx-1L], /NO_COPY)
    l_echo.l_hvp_gbl_off = ptr

    ptr = ptr_new (l_hvp_scn_en[0:indx-1L], /NO_COPY)
    l_echo.l_hvp_scn_en = ptr

    ptr = ptr_new (l_hvp_scn_dsbl[0:indx-1L], /NO_COPY)
    l_echo.l_hvp_scn_dsbl = ptr

    ptr = ptr_new (l_hvp_scn_ld[0:indx-1L], /NO_COPY)
    l_echo.l_hvp_scn_ld = ptr


RETURN
END

;-------------------------------------------------------------------------------
;  Procedure: DetSysCmds
;
;  Description:  Determine which system command sent.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  ofd                   long          I        output file descriptor 
;  opcode                long          I        command code 
;  subset                long[]        I        list of commands
;  indx                  long          I        index to subset array
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------

PRO DetSysCmds, ofd, opcode, subset, indx

  case opcode of
     'L_SYS_PTEST_EN'  : begin
                           printf, ofd, '     LENA autonomous performance test at boot enabled.'
                           indx = indx + 1L
                         end
     'L_SYS_PTEST_DSBL'  : begin
                           printf, ofd, '     LENA autonomous performance test at boot disabled.'
                           indx = indx + 1L
                         end
     'L_SYS_MODE_PRAM'  : begin
                           printf, ofd, '     LENA operational mode set to RAM.'
                           indx = indx + 1L
                         end
     'L_SYS_MODE_ROM'  : begin
                           printf, ofd, '     LENA operational mode set to ROM.'
                           indx = indx + 1L
                         end
     'L_SYS_MODE_HVD'  : begin
                           printf, ofd, '     LENA operational mode set to HV disabled.'
                           indx = indx + 1L
                         end
     'L_SYS_MODE_HVSF'  : begin
                           printf, ofd, '     LENA operational mode set to HV safe.'
                           indx = indx + 1L
                         end
     'L_SYS_MODE_NRML'  : begin
                           printf, ofd, '     LENA operational mode set to normal.'
                           indx = indx + 1L
                         end
     'L_SYS_HKNRM_EN'  : begin
                           printf, ofd, '     NORMAL HK packet transmission enabled.'
                           indx = indx + 1L
                         end
     'L_SYS_HKNRM_DSBL'  : begin
                           printf, ofd, '     NORMAL HK packet transmission disabled.'
                           indx = indx + 1L
                         end
     'L_SYS_HKQL_EN'  : begin
                           printf, ofd, '     Quicklook packet transmission enabled.'
                           indx = indx + 1L
                         end
     'L_SYS_HKQL_DSBL'  : begin
                           printf, ofd, '     Quicklook packet transmission disabled.'
                           indx = indx + 1L
                         end
     'L_SYS_HKCM_EN'  : begin
                           printf, ofd, '     Command record HK packet transmission enabled.'
                           indx = indx + 1L
                         end
     'L_SYS_HKCM_DSBL'  : begin
                           printf, ofd, '     Command record HK packet transmission disabled.'
                           indx = indx + 1L
                         end
     'L_SYS_HKDG_EN'  : begin
                           printf, ofd, '     Diagnostic HK packet transmission enabled.'
                           indx = indx + 1L
                         end
     'L_SYS_HKDG_DSBL'  : begin
                           printf, ofd, '     Diagnostic record HK packet transmission disabled.'
                           indx = indx + 1L
                         end
     'L_SYS_HKMEM_EN'  : begin
                           printf, ofd, '     Memory HK packet transmission enabled.'
                           indx = indx + 1L
                         end
     'L_SYS_HKMEM_DSBL'  : begin
                           printf, ofd, '     Memory HK packet transmission disabled.'
                           indx = indx + 1L
                         end
     'L_SYS_INST_RST'  : begin
                           printf, ofd, '     LENA reset.'
                           indx = indx + 1L
                         end
     'L_SYS_ERR_RST'  : begin
                           printf, ofd, '     All error counters, registers, & flags reset.'
                           indx = indx + 1L
                         end
     'L_SYS_BIN_EN'  : begin
                           printf, ofd, '     Binned science packet transmission enabled.'
                           indx = indx + 1L
                         end
     'L_SYS_BIN_DSBL'  : begin
                           printf, ofd, '     Binned science packet transmission disabled.'
                           indx = indx + 1L
                         end
     'L_SYS_EVENT_EN'  : begin
                           printf, ofd, '     Event science packet transmission enabled.'
                           indx = indx + 1L
                         end
     'L_SYS_EVENT_DSBL'  : begin
                           printf, ofd, '     Event science packet transmission disabled.'
                           indx = indx + 1L
                         end
     'L_SYS_SNGLS_EN'  : begin
                           printf, ofd, '     Singles science packet transmission enabled.'
                           indx = indx + 1L
                         end
     'L_SYS_SNGLS_DSBL'  : begin
                           printf, ofd, '     Singles science packet transmission disabled.'
                           indx = indx + 1L
                         end
     'L_SYS_ROI_SEL'  : begin
                           printf, ofd, '     ROI segements tranmitted: ', $
                                                  subset[indx+2L]
                           indx = indx + 2L
                         end
     'L_SYS_HKPTS_EN'  : begin
                           printf, ofd, '     Performance test hk packet transmission enabled.'
                           indx = indx + 1L
                         end
     'L_SYS_HKPTS_DSBL'  : begin
                           printf, ofd, '     Performance test hk packet transmission disabled.'
                           indx = indx + 1L
                         end
     'L_SYS_HKSTR_EN'  : begin
                           printf, ofd, '     HVPS stripchart hk packet transmission enabled.'
                           indx = indx + 1L
                         end
     'L_SYS_HKSTR_DSBL'  : begin
                           printf, ofd, '     HVPS stripchart hk packet transmission disabled.'
                           indx = indx + 1L
                         end
     'L_SYS_HKNMA_EN'  : begin
                           printf, ofd, '     Normal-A hk packet transmission enabled.'
                           indx = indx + 1L
                         end
     'L_SYS_HKNMA_DSBL'  : begin
                           printf, ofd, '     Normal-A hk packet transmission disabled.'
                           indx = indx + 1L
                         end
     'L_SYS_UPLD_REQ'  : begin
                           printf, ofd, '     Request upload from CIDP block P : ', subset[indx+2L]
                           indx = indx + 2L
                         end
     'L_SYS_ISNC_EN'  : begin
                           printf, ofd, '     Internal sync signal enabled.'
                           indx = indx + 1L
                         end
     'L_SYS_ISNC_DSBL'  : begin
                           printf, ofd, '     Internal sync signal disabled.'
                           indx = indx + 1L
                         end
     'L_SYS_HVP_VTHR'  : begin
                           indx = indx + 2L
                           printf, ofd, '     Overvoltage thresholds for high voltage power supplies: ', $
                                              subset[indx:indx+4L]
                           indx = indx + 4L
                         end
     'L_SYS_NOP'  : begin
                           printf, ofd, '     NOP command.'
                         end
     'L_SYS_HVP_ITHR'  : begin
                           indx = indx + 2L
                           printf, ofd, '     Overcurrent thresholds for high voltage power supplies: ', $
                                              subset[indx:indx+4L]
                           indx = indx + 4L
                         end
     'L_SYS_OVCT_THR'  : begin
                           indx = indx + 2L
                           printf, ofd, '     Start/Stop Overcount threshold set to: ', $
                                              subset[indx:indx+1L]
                           indx = indx + 1L
                         end
     'L_SYS_RSTR_EN'  : begin
                           printf, ofd, '     LENA Stop MCP hvps auto-restore enabled.'
                           indx = indx + 1L
                         end
     'L_SYS_RSTR_DSBL'  : begin
                           printf, ofd, '     LENA Stop MCP hvps auto-restore disabled.'
                           indx = indx + 1L
                         end
     'L_SYS_ACQR_EN'  : begin
                           printf, ofd, '     Science data acquisition enabled.'
                           indx = indx + 1L
                         end
     'L_SYS_ACQR_DSBL'  : begin
                           printf, ofd, '     Science data acquisition disabled.'
                           indx = indx + 1L
                         end
     'L_SYS_STAT_SAVE'  : begin
                           printf, ofd, '     LENA configuration state saved.'
                         end
     'L_SYS_STAT_RSTR'  : begin
                           printf, ofd, '     LENA configuration state restored.'
                           indx = indx + 1L
                         end
     'L_SYS_DADR_SET'  : begin
                           indx = indx + 2L
                           printf, ofd, '     16-bit hex address of data read by diagnostic register set: ', $
                                              subset[indx:indx+1L]
                           indx = indx + 1L
                         end
     'L_SYS_STAT_SAVE'  : begin
                           printf, ofd, '     LENA configuration state saved.'
                         end
     'L_SYS_RSUM_EN'  : begin
                           printf, ofd, '     ROI summation science packet transmission enabled.'
                         end
     'L_SYS_RSUM_DSBL'  : begin
                           printf, ofd, '     ROI summation science packet transmission disabled.'
                         end
     'L_SYS_SECEV_EN'  : begin
                           printf, ofd, '     Valid event by sector science packet transmission enabled.'
                         end
     'L_SYS_SECEV_DSBL'  : begin
                           printf, ofd, '     Valid event by sector science packet transmission disabled.'
                         end
     'L_SYS_MON_EN'  : begin
                           printf, ofd, '     Monitor hk packet transmission enabled.'
                         end
     'L_SYS_MON_DSBL'  : begin
                           printf, ofd, '     Monitor hk packet transmission disabled.'
                         end
      else             : begin
                           printf, ofd, '     Invalid system command : ', opcode
                         end
  endcase

RETURN
END


;-------------------------------------------------------------------------------
;  Procedure: DetROICmds
;
;  Description: Determine which ROI commands sent.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  ofd                   long          I        output file descriptor 
;  opcode                long          I        command code 
;  subset                long[]        I        list of commands
;  indx                  long          I        index to subset array
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------

PRO DetROICmds, ofd, opcode, subset, indx
    case opcode of 
      'L_ROI_CNFG_PKG0'  : begin
                             indx = indx + 2L
                             printf, ofd, '     Definition of ROI 0: ', subset[indx:indx+5L]
                             indx = indx + 5L
                           end
      'L_ROI_CNFG_PKG1'  : begin
                             indx = indx + 2L
                             printf, ofd, '     Definition of ROI 1: ', subset[indx:indx+5L]
                             indx = indx + 5L
                           end
      'L_ROI_CNFG_PKG2'  : begin
                             indx = indx + 2L
                             printf, ofd, '     Definition of ROI 2: ', subset[indx:indx+5L]
                             indx = indx + 5L
                           end
      'L_ROI_CNFG_PKG3'  : begin
                             indx = indx + 2L
                             printf, ofd, '     Definition of ROI 3: ', subset[indx:indx+5L]
                             indx = indx + 5L
                           end
       else              : begin
                             printf, ofd, '     Invalid ROI command : ', opcode 
                           end
    endcase
RETURN
END

;-------------------------------------------------------------------------------
;  Procedure: DetPerfCmds
;
;  Description: Determine which performance commands sent.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  ofd                   long          I        output file descriptor 
;  opcode                long          I        command code 
;  subset                long[]        I        list of commands
;  indx                  long          I        index to subset array
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------

PRO DetPerfCmds, ofd, opcode, subset, indx
  
    case opcode of
      'L_TST_LV_EN'     : begin
                            printf, ofd, '     Low voltage performance test enabled.' 
                            indx = indx + 1L
                          end
      'L_TST_HV_EN'     : begin
                            printf, ofd, '     High voltage performance test enabled.' 
                            indx = indx + 1L
                          end
      'L_TST_MSTR_EN'   : begin
                            printf, ofd, '     MCP start HV power supply test enabled.'
                            indx = indx + 1L
                          end
      'L_TST_MSTR_DSBL' : begin
                            printf, ofd, '     MCP start HV power supply test disabled.'
                            indx = indx + 1L
                          end
      'L_TST_MSTP_EN'   : begin
                            printf, ofd, '     MCP stop HV power supply test enabled.'
                            indx = indx + 1L
                          end
      'L_TST_MSTP_DSBL' : begin
                            printf, ofd, '     MCP stop HV power supply test disabled.'
                            indx = indx + 1L
                          end
      'L_TST_COLP_EN'   : begin
                            printf, ofd, '     Collimator+ HV power supply test enabled.'
                            indx = indx + 1L
                          end
      'L_TST_COLP_DSBL' : begin
                            printf, ofd, '     Collimator+ HV power supply test disabled.'
                            indx = indx + 1L
                          end
      'L_TST_COLN_EN'   : begin
                            printf, ofd, '     Collimator- HV power supply test enabled.'
                            indx = indx + 1L
                          end
      'L_TST_COLN_DSBL' : begin
                            printf, ofd, '     Collimator- HV power supply test disabled.'
                            indx = indx + 1L
                          end
      'L_TST_OPT_EN'    : begin
                            printf, ofd, '     Ion Optics HV power supply test enabled.'
                            indx = indx + 1L
                          end
      'L_TST_OPT_DSBL'  : begin
                            printf, ofd, '     Ion Optics HV power supply test disabled.'
                            indx = indx + 1L
                          end
      'L_TST_STR_EN'    : begin
                            printf, ofd, '     Optics Steering power supply test enabled.'
                            indx = indx + 1L
                          end
      'L_TST_STR_DSBL'  : begin
                               printf, ofd, '     Optics Steering power supply test disabled.'
                               indx = indx + 1L
                          end
       else             : begin
                             printf, ofd, 'Invalid Performance Test command: ', opcode
                           end
    endcase
RETURN
END

;-------------------------------------------------------------------------------
;  Procedure: DetHvpsCmds
;
;  Description: Determine which HVPS commands sent.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  ofd                   long          I        output file descriptor 
;  opcode                long          I        command code 
;  subset                long[]        I        list of commands
;  indx                  long          I        index to subset array
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------

PRO DetHvpsCmds, ofd, opcode, subset, indx

   case opcode of 
        'L_HVP_MSTR_EN'   : begin
                               printf, ofd, '     MCP Start HV Enabled'
                               indx = indx + 1L
                            end
        'L_HVP_MSTR_DSBL' : begin
                               printf, ofd, '     MCP Start HV Disabled'
                               indx = indx + 1L
                            end
        'L_HVP_MSTP_EN'   : begin
                               printf, ofd, '     MCP Stop HV Enabled'
                               indx = indx + 1L
                            end
        'L_HVP_MSTP_DSBL' : begin
                               printf, ofd, '     MCP Stop HV Disabled'
                               indx = indx + 1L
                            end
        'L_HVP_COLP_EN'   : begin
                               printf, ofd, '     Collimator+ Enabled'
                               indx = indx + 1L
                            end
        'L_HVP_COLP_DSBL' : begin
                               printf, ofd, '     Collimator+ Disabled'
                               indx = indx + 1L
                            end
        'L_HVP_COLN_EN'   : begin
                               printf, ofd, '     Collimator- Enabled'
                               indx = indx + 1L
                            end
        'L_HVP_COLN_DSBL' : begin
                               printf, ofd, '     Collimator- Disabled'
                               indx = indx + 1L
                            end
        'L_HVP_OPT_EN'    : begin
                               printf, ofd, '     Ion Optics HV Enabled'
                               indx = indx + 1L
                            end
        'L_HVP_OPT_DSBL'  : begin
                               printf, ofd, '     Ion Optics HV Disabled'
                               indx = indx + 1L
                            end
        'L_HVP_GBL_SAFE'  : begin
                               printf, ofd, '     limit max HVPS output to 20% of F.S.'
                               indx = indx + 1L
                            end
        'L_HVP_GBL_USAFE' : begin
                               printf, ofd, '     max HVPS output to 100% of F.S.'
                               indx = indx + 1L
                            end
        'L_HVP_MSTR_CMD'  : begin
                               printf, ofd, '     MCP START HV Power Supply      : ', $
                                             subset[indx+2L]
                               indx = indx + 2L
                            end
        'L_HVP_MSTP_CMD'  : begin
                               printf, ofd, '     MCP STOP HV Power Supply       : ', $
                                             subset[indx+2L]
                               indx = indx + 2L
                            end
        'L_HVP_COLP_CMD'  : begin
                               printf, ofd, '     Collimator+ HV Power Supply: ', $
                                             subset[indx+2L]
                               indx = indx + 2L
                            end
        'L_HVP_COLN_CMD'  : begin
                               printf, ofd, '     Collimator- HV Power Supply: ', $
                                             subset[indx+2L] 
                               indx = indx + 2L
                            end
        'L_HVP_OPT_CMD'   : begin
                               printf, ofd, '     Ion Optics HV Power Supply: ', $
                                             subset[indx+2L] 
                               indx = indx + 2L
                            end
        'L_HVP_STR_CMD'   : begin
                               printf, ofd, '     Optics Steering HV Power Supply: ', $
                                             subset[indx+2L] 
                               indx = indx + 2L
                            end
        'L_HVP_GBL_OFF'   : begin
                               printf, ofd, '     Execute HVPS Shutdown Macro.'
                               indx = indx + 1L
                            end
        'L_HVP_SCN_EN'    : begin
                               printf, ofd, '     Steering supply scan enabled.'
                               indx = indx + 1L
                            end
        'L_HVP_SCN_DSBL'  : begin
                               printf, ofd, '     Steering supply scan disabled.'
                               indx = indx + 1L
                            end
        'L_HVP_SCN_LD'    : begin
                               indx = indx + 2L
                               printf, ofd, '     Steering supply steps: ', $
                                              subset[indx:indx+15]
                               indx = indx + 15L
                            end
        else : begin
                 printf, ofd, 'Invalid HVPS command : ', opcode
               end
   endcase


RETURN
END

;-------------------------------------------------------------------------------
;  Procedure: DetTOFCmds
;
;  Description: Determine which TOF commands sent.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  ofd                   long          I        output file descriptor 
;  opcode                long          I        command code 
;  subset                long[]        I        list of commands
;  indx                  long          I        index to subset array
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------

PRO DetTOFCmds, ofd, opcode, subset, indx
    case opcode of
       'L_TOF_MODE_STST'   : begin 
                               printf, ofd, '     TOF start-stop mode enabled.'
                               indx = indx + 1L
                             end
       'L_TOF_MODE_STRT'   : begin 
                               printf, ofd, '     TOF start-only mode enabled.'
                               indx = indx + 1L
                             end
       'L_TOF_BIN_STRT1'   : begin 
                               printf, ofd, $
                                     '     Start TOF definition for species1 binning interval: ', $
                                     subset[indx+2L]
                               indx = indx + 2L
                             end
       'L_TOF_BIN_STOP1'   : begin 
                               printf, ofd, $
                                     '     Stop TOF definition for species1 binning interval: ', $
                                     subset[indx+2L]
                               indx = indx + 2L
                             end
       'L_TOF_BIN_STRT2'   : begin 
                               printf, ofd, $
                                     '     Start TOF definition for species2 binning interval: ', $
                                     subset[indx+2L]
                               indx = indx + 2L
                             end
       'L_TOF_BIN_STOP2'   : begin 
                               printf, ofd, $
                                     '     Stop TOF definition for species2 binning interval: ', $
                                     subset[indx+2L]
                               indx = indx + 2L
                             end
       'L_TOF_BIT_ON'      : begin 
                               printf, ofd, '     TOF build-in-test mode enabled.'
                               indx = indx + 1L
                             end
       'L_TOF_BIT_OFF'     : begin 
                               printf, ofd, '     TOF build-in-test mode disabled.'
                               indx = indx + 1L
                             end
       'L_TOF_BIT_STRB'    : begin 
                               printf, ofd, '     Single strobe routied to TOF system.'
                               indx = indx + 1L
                             end
       'L_TOF_BIT_DELAY'   : begin 
                               printf, ofd, '     TOF bit mode delay: ', $ 
                                     subset[indx+2L] 
                               indx = indx + 2L
                             end
       'L_TOF_TRSH_START'  : begin 
                               printf, ofd, '     TOF CFD start threshold: ', $ 
                                     subset[indx+2L] 
                               indx = indx + 2L
                             end
       'L_TOF_TRSH_STOP'   : begin 
                               printf, ofd, '     TOF CFD stop threshold: ', $ 
                                     subset[indx+2L] 
                               indx = indx + 2L
                             end
        else               : begin
                                printf, ofd, 'Invalid TOF Command: ', opcode
                             end
   
    endcase
RETURN
END

;-------------------------------------------------------------------------------
;  Procedure: DetPSSCmds
;
;  Description: Determine which PSS commands sent.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  ofd                   long          I        output file descriptor 
;  opcode                long          I        command code 
;  subset                long[]        I        list of commands
;  indx                  long          I        index to subset array
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------

PRO DetPSSCmds, ofd, opcode, subset, indx
    case opcode of
       'L_PSS_GAIN_LOW'   : begin
                              printf, ofd, '     Position anodes to LOW gain set.' 
                              indx = indx + 1L
                            end
       'L_PSS_GAIN_HIGH'  : begin
                              printf, ofd, '     Position anodes to HIGH gain set.' 
                              indx = indx + 1L
                            end
        else              : begin
                               printf, ofd, 'Invalid PSS Command: ', opcode
                            end
    endcase
RETURN
END

;-------------------------------------------------------------------------------
;  Procedure: DetMemCmds
;
;  Description: Determine which memory commands sent.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  ofd                   long          I        output file descriptor 
;  opcode                long          I        command code 
;  subset                long[]        I        list of commands
;  indx                  long          I        index to subset array
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------

PRO DetMemCmds, ofd, opcode, subset, indx
    case opcode of
       'L_MEM_WORD_RD'  : begin
                            indx = indx + 2L
                            printf, ofd, '     Read word LENA memory space: ', subset[indx:indx+3L]
                            indx = indx + 3L
                          end
       'L_MEM_WORD_WR'  : begin
                            indx = indx + 2L
                            printf, ofd, '     Write word LENA memory space: ', subset[indx:indx+3L]
                            indx = indx + 3L
                          end
       'L_MEM_BYTE_RD'  : begin
                            indx = indx + 2L
                            printf, ofd, '     Read byte LENA memory space: ', subset[indx:indx+3L]
                            indx = indx + 3L
                          end
       'L_MEM_BYTE_WR'  : begin
                            indx = indx + 2L
                            printf, ofd, '     Write byte LENA memory space: ', subset[indx:indx+3L]
                            indx = indx + 3L
                          end
       'L_MEM_PRAM_RD'  : begin
                            indx = indx + 2L
                            printf, ofd, '     Read program ram space: ', subset[indx:indx+3L]
                            indx = indx + 3L
                          end
       else             : begin
                            printf, ofd, 'Invalid Memory Command: ', opcode
                          end
    endcase
RETURN
END

;-------------------------------------------------------------------------------
;  Function/Procedure: l_EchoDumpAllCmds
;
;  Description: Dump the commands to a text file.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_echo                struct *      I        pointers to echo data structures
;  l_sys_cmd             struct        I        system command table
;  l_roi_cmd             struct        I        roi command table
;  l_perf_cmd            struct        I        performance command table
;  l_hvps_cmd            struct        I        HVPS command table
;  l_tof_cmd             struct        I        time of flight command table
;  l_pss_cmd             struct        I        PSS command table
;  l_mem_cmd             struct        I        memory command table
;  ofd                   long          I        output file descriptor
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------

PRO l_EchoDumpAllCmds, l_echo, l_sys_cmd, l_roi_cmd, l_perf_cmd, $
l_hvps_cmd, l_tof_cmd, l_pss_cmd, l_mem_cmd, ofd


    sys_stem  =  l_sys_cmd[0].header
    roi_stem  =  l_roi_cmd[0].header
    perf_stem =  l_perf_cmd[0].header
    hvps_stem =  l_hvps_cmd[0].header
    tof_stem  =  l_tof_cmd[0].header
    pss_stem  =  l_pss_cmd[0].header
    mem_stem  =  l_mem_cmd[0].header



    printf, ofd, ' '
    printf, ofd, ' '
    printf, ofd, ' '
    printf, ofd, '============================================================'
    printf, ofd, '                   Commands Executed '
    printf, ofd, '  Each spin shows commands executed in the order it was '
    printf, ofd, '  received. Commands are from the Echo (Command Record '
    printf, ofd, '  Housekeeping Science Packet. '
    printf, ofd, ' '
    printf, ofd, '  Values are in decimal. '
    printf, ofd, '============================================================'
    printf, ofd, ' '
    printf, ofd, ' '
    printf, ofd, ' '
    printf, ofd, ' '


    for nspins = 0L, l_echo.nitems - 1L do begin
       indx = 0L
       subset = (*l_echo.dataPtr)[0L:(*l_echo.num_cmds_exe)[nspins]-1L, nspins]

       tmestr = STRING ((*l_echo.startUdfTimePtr)[nspins].doy, $
                        (*l_echo.startUdfTimePtr)[nspins].hour,$
                        (*l_echo.startUdfTimePtr)[nspins].min, $
                        (*l_echo.startUdfTimePtr)[nspins].sec, $
                        FORMAT="(i3,' ',i2.2,':',i2.2,':',i2.2)")

       printf, ofd, tmestr 

       while (indx lt (*l_echo.num_cmds_exe)[nspins] - 1L) do begin
           opcode = subset[indx+1]
           case subset[indx] of
               sys_stem : begin
                            DetSysCmds, ofd, l_hvps_cmd[opcode].cmd, subset, indx
                          end
               roi_stem : begin
                            DetROICmds, ofd, l_roi_cmd[opcode].cmd, subset, indx
                          end
               perf_stem : begin
                            DetPerfCmds, ofd, l_perf_cmd[opcode].cmd, subset, indx
                          end
               hvps_stem : begin
                            DetHvpsCmds, ofd, l_hvps_cmd[opcode].cmd, subset, indx
                          end
               tof_stem : begin
                            DetTOFCmds, ofd, l_tof_cmd[opcode].cmd, subset, indx
                          end
               pss_stem : begin
                            DetPSSCmds, ofd, l_pss_cmd[opcode].cmd, subset, indx
                          end
               mem_stem : begin
                            DetMemCmds, ofd, l_mem_cmd[opcode].cmd, subset, indx
                          end
               else     : begin
                            printf, ofd, 'Incorrect command stem found : ', subset[indx]
                          end
           endcase
           indx = indx + 1L
       endwhile
       printf, ofd, ' '
       printf, ofd, ' '
    endfor

RETURN
END


;-------------------------------------------------------------------------------
;  Procedure: l_EchoTerm
;
;  Description: Free memory allocated for echo data storage.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_echo                struct *      I        pointe to echo data structures
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------

;---------------------------------------------------------------------------
;---------------------------------------------------------------------------

PRO l_EchoTerm, l_echo

    if ptr_valid (l_echo.timePtr) then ptr_free, l_echo.timePtr 
    if ptr_valid (l_echo.startUdfTimePtr) then ptr_free, l_echo.startUdfTimePtr
    if ptr_valid (l_echo.stopUdfTimePtr) then ptr_free, l_echo.stopUdfTimePtr
    if ptr_valid (l_echo.dataPtr) then ptr_free, l_echo.dataPtr
    if ptr_valid (l_echo.data_recs_in_seq) then ptr_free, l_echo.data_recs_in_seq
    if ptr_valid (l_echo.data_recs_seq_num) then ptr_free, l_echo.data_recs_seq_num
    if ptr_valid (l_echo.num_cmds_exe) then ptr_free, l_echo.num_cmds_exe
    if ptr_valid (l_echo.sub_spinTime) then ptr_free, l_echo.sub_spinTime
    if ptr_valid (l_echo.sub_spinUdfTime) then ptr_free, l_echo.sub_spinUdfTime
    if ptr_valid (l_echo.nevents_cmded) then ptr_free, l_echo.nevents_cmded

    if ptr_valid (l_echo.l_hvp_mstr_en) then ptr_free, l_echo.l_hvp_mstr_en
    if ptr_valid (l_echo.l_hvp_mstr_dsbl) then ptr_free, l_echo.l_hvp_mstr_dsbl
    if ptr_valid (l_echo.l_hvp_mstp_en) then ptr_free, l_echo.l_hvp_mstp_en
    if ptr_valid (l_echo.l_hvp_mstp_dsbl) then ptr_free, l_echo.l_hvp_mstp_dsbl
    if ptr_valid (l_echo.l_hvp_colp_en) then ptr_free, l_echo.l_hvp_colp_en
    if ptr_valid (l_echo.l_hvp_colp_dsbl) then ptr_free, l_echo.l_hvp_colp_dsbl
    if ptr_valid (l_echo.l_hvp_coln_en) then ptr_free, l_echo.l_hvp_coln_en
    if ptr_valid (l_echo.l_hvp_coln_dsbl) then ptr_free, l_echo.l_hvp_coln_dsbl
    if ptr_valid (l_echo.l_hvp_opt_en) then ptr_free, l_echo.l_hvp_opt_en
    if ptr_valid (l_echo.l_hvp_opt_dsbl) then ptr_free, l_echo.l_hvp_opt_dsbl
    if ptr_valid (l_echo.l_hvp_gbl_safe) then ptr_free, l_echo.l_hvp_gbl_safe
    if ptr_valid (l_echo.l_hvp_gbl_usafe) then ptr_free, l_echo.l_hvp_gbl_usafe
    if ptr_valid (l_echo.l_hvp_mstr_cmd) then ptr_free, l_echo.l_hvp_mstr_cmd
    if ptr_valid (l_echo.l_hvp_mstp_cmd) then ptr_free, l_echo.l_hvp_mstp_cmd
    if ptr_valid (l_echo.l_hvp_colp_cmd) then ptr_free, l_echo.l_hvp_colp_cmd
    if ptr_valid (l_echo.l_hvp_coln_cmd) then ptr_free, l_echo.l_hvp_coln_cmd
    if ptr_valid (l_echo.l_hvp_opt_cmd) then ptr_free, l_echo.l_hvp_opt_cmd
    if ptr_valid (l_echo.l_hvp_str_cmd) then ptr_free, l_echo.l_hvp_str_cmd
    if ptr_valid (l_echo.l_hvp_gbl_off) then ptr_free, l_echo.l_hvp_gbl_off
    if ptr_valid (l_echo.l_hvp_scn_en) then ptr_free, l_echo.l_hvp_scn_en
    if ptr_valid (l_echo.l_hvp_scn_dsbl) then ptr_free, l_echo.l_hvp_scn_dsbl
    if ptr_valid (l_echo.l_hvp_scn_ld) then ptr_free, l_echo.l_hvp_scn_ld

RETURN
END


;-------------------------------------------------------------------------------
;  Function: l_EchoProc
;
;  Description: Main echo routine used to call the procedures to process
;        the echo data.
;
;  Return Value:  type = int
;  Value                         Description
;  -------------------------     ------------------------------
;  SUCCESS                       successful completion
;  FAILURE                       unsuccessful completion
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_setup               struct        I        general setup parameters
;  l_setup_file          string        I        runtime setup file name 
;  l_pltpos              struct        I        general plotting parameters
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;      ERROR_TYPES     SUCCESS         int         G          successful completion
;                      FAILURE         int         G          unsuccessful completion
;                      WARNING         int         G          warning message
;                      INFO            int         G          informational message
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------

FUNCTION l_EchoProc, l_setup, l_setup_file, l_pltpos
COMMON ERROR_TYPES, SUCCESS, FAILURE, WARNING, INFO


   l_pltpos.plot_hdr_title = 'Commands Executed'
   l_setup.l_filedesc = 'cmdexe'

   ;-----------------------------------------
   ; generate the state table log file name
   ;-----------------------------------------
   l_GenFileName, l_setup, '0'
   outfile = l_setup.l_prod_dst + '/' + $
             l_setup.l_filename + '.dmp'

   ;-----------------------------------------
   ; open state table log file
   ;-----------------------------------------
   openw, ofd, outfile, /get_lun


   l_GenSysCmdTable, l_sys_cmd
   l_GenROICmdTable, l_roi_cmd
   l_GenPerfCmdTable,l_perf_cmd
   l_GenHvpsCmdTable,l_hvps_cmd
   l_GenTOFCmdTable, l_tof_cmd
   l_GenPSSCmdTable, l_pss_cmd
   l_GenMemCmdTable, l_mem_cmd

   status = l_EchoGetData (l_setup, $
                  l_setup_file, $
                  l_pltpos, $
                  l_echo)
   if (status eq FAILURE) then return, FAILURE


   l_DumpHdrInfo, ofd, l_setup, l_pltpos, outfile
   l_EchoDumpAllCmds, l_echo, l_sys_cmd, l_roi_cmd, l_perf_cmd, $
                      l_hvps_cmd, l_tof_cmd, l_pss_cmd, l_mem_cmd, ofd

   close, ofd
   free_lun, ofd

   RETURN, SUCCESS

END
