;-------------------------------------------------------------------------------
;
;  Unit Name : l_EarthRtns.pro
;
;  Purpose   : Routines related to determining the earth distance and size. Draws
;         the markers on the spectrograms.
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;   ELH                               v0.0           Original implementation
;
;
;  File Revision Number:   %I%
;  File Last Modified  :   %E%  %T%
;
;-------------------------------------------------------------------------------

;-------------------------------------------------------------------------------
;  Procedure : l_EarthDirection
;  Description:
;      Draws the earth outline.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  X                     array         I        x values
;  Y                     array         I        y values
;  Z                     array         I        2nd set of y values
;  img_pos               long[]        I        image coordinates
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;     PLOTDEFS         MAX_PLOTS        int         G         max # plots per page
;                      MAX_WEB_SPINS    int         G         max # spins per plot
;                                                             when using web interface
;                      NO_CNTS          double      G         no counts indicator
;                      NO_DATA          double      G         no data indicator
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;-------------------------------------------------------------------------------

PRO l_DrawEarthDirection, X, Y, Z, img_pos, min_spin_ang, max_spin_ang 
COMMON PLOTDEFS, MAX_PLOTS, MAX_WEB_SPINS, NO_CNTS, NO_DATA

   blankticks = strarr (30)
   blankticks(*) = ' '


   PLOT, X, Y, /normal, /noerase, $
         xstyle=1, $
         xrange = [min(X), max(X)], $
         yrange = [min_spin_ang, max_spin_ang], $
         yticks = 4, $
         yminor = 6, $
         ticklen = -0.05, $
         xtickname = blankticks, $
         ytickname = blankticks, $
         position = img_pos, $
         linestyle = 0,  $             ; solid line
         color = 0                     ; white


   OPLOT, X, Z, $
          color = 0



RETURN
END

;-------------------------------------------------------------------------------
;  Procedure: l_DrawEarthCenter
;
;  Description: Draws the earth lines if earth is not centered at zero spin angle.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  X                     array         I        x values
;  Y                     array         I        y values
;  img_pos               array         I        position of image
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------

PRO l_DrawEarthCenter, X, Y, img_pos

   blankticks = strarr (30)
   blankticks(*) = ' '

   PLOT, X, Y, $
         /normal, $
         /noerase, $
         xstyle=1, $
         ystyle=1, $
         yrange = [0, 44], $
         yticks = 4, $
         yminor = 6, $
         ticklen = -0.05, $
         xtickname = blankticks, $
         ytickname = blankticks, $
         position = img_pos, $
         linestyle = 0,  $             ; solid line
         color = 0                     ; white

RETURN
END


;-------------------------------------------------------------------------------
;  Procedure: l_DetermineEarthDir
;
;  Description:  Determines the sectors the earth is in when the earth is not
;       centered at zero degree spin angle.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_setup               struct        I        general setup parameters    
;  l_oa                  struct *      I        pointers to OA data structures
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;      EPOCH_DATA      EPOCH_YR        int         G          launch year
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------


PRO l_DetermineEarthDir, earth_dir, l_setup, l_oa

  sectors = l_setup.l_num_spinsec - 1L

  if (l_setup.l_angle_zero eq 'S') then begin

     sun = (*l_oa.lena_sun_sectPtr)

     earth_dir = fltarr(n_elements(sun))

     ;-------------------------------------------
     ; determine how many sectors the sun shifted
     ; to zero spin angle
     ;-------------------------------------------
     sun_dist = sun - float(l_setup.l_spin_ang_zero)

     ;-------------------------------------------
     ; adjust earth based on sun shift
     ;-------------------------------------------
     earth_dir = l_setup.l_earth_center - sun_dist

     ;-------------------------------------------
     ; correct wrap arounds that are outside of
     ; the bounds (0,44)
     ;-------------------------------------------
     indx = where (earth_dir lt 0, zcnt)
     if (zcnt gt 0) then begin
       earth_dir[indx] = sectors + earth_dir[indx] 
     endif

     indx = where (earth_dir gt 44, zcnt)
     if (zcnt gt 0) then begin
       earth_dir[indx] = earth_dir[indx] - sectors 
     endif

   endif else begin
      if (l_setup.l_angle_zero eq 'M') then begin

         moon = (*l_oa.lena_moon_sectPtr)

         earth_dir = fltarr(n_elements(moon))
    
         ;-------------------------------------------
         ; determine how many sectors the moon shifted
         ; to zero spin angle
         ;-------------------------------------------
         moon_dist = moon - float(l_setup.l_spin_ang_zero)

         ;-------------------------------------------
         ; adjust earth based on moon shift
         ;-------------------------------------------
         earth_dir = l_setup.l_earth_center - moon_dist

         ;-------------------------------------------
         ; correct wrap arounds that are outside of
         ; the bounds (0,44)
         ;-------------------------------------------
         indx = where (earth_dir lt 0, zcnt)
         if (zcnt gt 0) then begin
           earth_dir[indx] = sectors + earth_dir[indx] 
         endif

         indx = where (earth_dir gt 44, zcnt)
         if (zcnt gt 0) then begin
           earth_dir[indx] = earth_dir[indx] - sectors 
         endif
      endif
   endelse

   RETURN

END


;-------------------------------------------------------------------------------
;  Procedure: l_CalcEarthDirection
;
;  Description: Routine which calls the necessary procedures to determine
;      the location of the earth.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_oa                  struct *      I        pointer to OA data structures
;  l_setup               struct        I        general setup parameters
;  img_pos               array         I        image position
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------

PRO l_CalcEarthDirection, l_oa, l_setup, img_pos


    l_OAEarthSize, l_oa, $
                   l_setup.l_start_dtime, $
                   top_markers, $
                   bot_markers, $
                   new_nspins, $
                   oa_time

    l_HndlMissOA, (*l_oa.tme_ptr), $
                  hmd_oa_tme, $
                  new_nspins, $
                  top_markers, $
                  top_hmd, $
                  l_setup.l_start_dtime, $
                  l_setup.l_stop_dtime, $
                  1


     bot_hmd = top_hmd * (-1)

     if (l_setup.l_angle_zero eq 'E') OR $
        (l_setup.l_angle_zero eq 'N') then begin
        l_DrawEarthDirection, hmd_oa_tme, $
                        top_hmd, $
                        bot_hmd, $
                        img_pos, $
                        l_setup.l_min_spin_angle, $
                        l_setup.l_max_spin_angle
      endif else begin

           ;------------------------------------------
           ; determine location of center of earth
           ; and adjust for sun wrap around
           ;------------------------------------------
           l_DetermineEarthDir, top_earth, l_setup, l_oa
           bot_earth = top_earth

           ;------------------------------------------
           ; convert earth center to degrees and add
           ; width of earth 
           ;------------------------------------------
           top = ((top_earth * 8.) + top_hmd)/8.
           l_DrawEarthCenter, hmd_oa_tme, top, img_pos

           ;------------------------------------------
           ; convert earth center to degrees and 
           ; subtract distance
           ;------------------------------------------
           bot = ((bot_earth * 8.) + bot_hmd)/8.
           l_DrawEarthCenter, hmd_oa_tme, bot , img_pos

      endelse


RETURN
END
