
;-------------------------------------------------------------------------------
;
;  Unit Name : l_DetMjMnTicks.pro
;
;  Purpose   : To determine the number of major and minor tick
;              marks and their labels.
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;  ELH              11/30/01         v1.0.8           Modified to handle general
;                                                     calculations of major and
;                                                     minor tick marks
;  ELH              07/14/04         v1.0.11          Modified condition which 
;                                                     handles DOY crossovers <
;                                                     24 hours.  Uses 86400 seconds
;                                                     as an entire day instead of
;                                                     using the day of year (which
;                                                     provided a difference in the
;                                                     # of actual days)
;
;
;
;  File Revision Number:   %I%
;  File Last Modified  :   %E%  %T%
;
;-------------------------------------------------------------------------------


;-------------------------------------------------------------------------------
;  Procedure: l_DetMjMnTicks
;  Description:
;    Determine the # of major and minor ticks on the axis.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  ustart_time           double        I        user start time
;  ustop_time            double        I        user stop time
;  dstart_time           double        I        data start time
;  dstop_time            double        I        data stop time
;  l_pltpos              struct        I        generic plot parameters
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Limitations:
;  ------------
;    Does not handle crossing year boundaries
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;  ELH              11/30/01         v1.0.8           Modified to handle general
;                                                     calculations of major and
;                                                     minor tick marks
;  ELH              04/06/06                          Correctly handle tick marks
;                                                     for requests greater than
;                                                     24 hours
;-------------------------------------------------------------------------------

PRO l_DetMjMnTicks, user_start_time, user_stop_time, dstart_time, dstop_time, l_pltpos


      ;---------------------------------------------
      ; the number of y axis minor ticks
      ;---------------------------------------------
      l_pltpos.ymn_tick = 6L 

      ;---------------------------------------------
      ; determine if the labels for the x axis will 
      ; be time or something else
      ;---------------------------------------------

      delta = 0
      flag = 0

      start_tick = dstart_time
      stop_tick = dstop_time
      ustart_time = ConvArrTimeToSecs(user_start_time)
      ustop_time = ConvArrTimeToSecs(user_stop_time)

      diff_doy = ustop_time - ustart_time
      if (diff_doy le 86400) then flag = 1L

      ;---------------------------------------------
      ; set the maximum number of x axis ticks 
      ; possible, if a 24 hour period, set to 7
      ;---------------------------------------------
      MAX_TICKS = 12L

      
      ;---------------------------------------------
      ; if more than one day
      ;---------------------------------------------
      if (diff_doy gt 86400)  then begin

         l_pltpos.X_axis_label = 'D'

         time_span_hours = ((ustop_time - ustart_time)/86400)*24 
         ;---------------------------------------------
         ; the number of days
         ;---------------------------------------------
         nontime_deltas = [1, 2, 5, 10, 15, 20, 30, 100, 1000]
         num_ntds = n_elements (nontime_deltas)

         ;---------------------------------------------
         ; set the start day of year label
         ;---------------------------------------------
         start_label = start_tick[1] 
         stop_label = stop_tick[1]
         if (start_tick[1] lt user_start_time[1]) then begin
             start_label = user_start_time[1]
             stop_label = user_stop_time[1]
         endif

         ;---------------------------------------------
         ; determine how many days
         ;---------------------------------------------
         ;diff = stop_label - start_label + 1L
         diff = stop_label - start_label

         ;---------------------------------------------
         ; determine the delta between major ticks
         ;---------------------------------------------
         mtest = -1
         for i = 0, num_ntds - 1 do begin
            if (i eq 7) or (i eq 8) then begin
               for j = 1, 9 do begin
                   nt_delta = diff/nontime_deltas[i] * j
                   if (mtest lt 0) AND (nt_delta lt MAX_TICKS) then begin
                      mtest = i
                   endif
               endfor
            endif else begin
               nt_delta = diff/nontime_deltas[i]
               if (mtest lt 0) AND (nt_delta lt MAX_TICKS) then begin
                   mtest = i
               endif
            endelse

         endfor

         ;---------------------------------------------
         ; generate the string label and determine the
         ; plotting location of the major X ticks
         ;---------------------------------------------
         delta = nontime_deltas[mtest]
         ;nticks = (diff/delta) + 1
         nticks = diff + 1


         maj_xtick_loc = dblarr (nticks)
         xtick_labels = strarr (nticks)

         for i = 0, nticks - 1L do begin
             label = start_label + i * delta
             xtick_labels[i] = strcompress (string(label), /remove_all)
             maj_xtick_loc[i] = i * 24. * delta 
         endfor


         ;---------------------------------------------
         ; store the X axis length
         ; 04/06/06 Modified to expand range to time
         ; specified and not confined by the day
         ;---------------------------------------------
         ;XaxisLen = diff * 24.  -- deleted 04/06/06 
         XaxisLen = time_span_hours

         l_pltpos.min_time_axis_len = 0
         l_pltpos.max_time_axis_len = XaxisLen

         ;---------------------------------------------------------
         ; determine the number of minor ticks
         ;---------------------------------------------------------

         min_time_deltas = [1, 5, 10, 15, 20, 30, 60, 120, 180, 240]
         nmtime_deltas = n_elements (min_time_deltas)

         tmin = start_label 
         tmax = start_label + delta 

         ;---------------------------------------------
         ; determine the delta between minor tick marks 
         ;---------------------------------------------
         if (delta gt 1439) then begin
            min_diff = fix(tmax - tmin)
            case min_diff of
                 5:  mn_tick = 5
                10:  mn_tick = 5
                30:  mn_tick = 3
               180:  mn_tick = 3
               360:  mn_tick = 3
              1080:  mn_tick = 3
              else:  mn_tick = 4
            endcase
         endif else begin

             MN_TICKS = 6 
             mn_diff = tmax - tmin
             mtest = -1
             for i = 0, nmtime_deltas - 1 do begin
                 mn_delta = mn_diff/min_time_deltas[i]
                 if (mtest lt 0) AND (mn_delta lt MN_TICKS) then begin
                    mtest = i
                 endif
             endfor

             mn_delta = min_time_deltas[mtest]
             mn_tick = mn_diff/mn_delta

          endelse

          ;---------------------------------------------------------
          ; store minor tick delta and X axis label info
          ;---------------------------------------------------------
          l_pltpos.mn_tick_delta = ((tmax - tmin)/mn_tick) * 24L

          l_pltpos.nticks = nticks
          l_pltpos.xtick_labels = xtick_labels
          l_pltpos.xtick_locs = maj_xtick_loc

      endif else begin
      ;---------------------------------------------
      ; if x axis labels are time
      ;---------------------------------------------
      if (flag eq 1) then begin

          ;---------------------------------------------------------
          ; determine the number of major ticks
          ;---------------------------------------------------------

          dstop_time[3] = dstop_time[3] + 2

          start_time = ConvArrTimeToSecs(dstart_time) 
          stop_time =  ConvArrTimeToSecs(dstop_time) 
          ustart_time = ConvArrTimeToSecs(user_start_time)
          ustop_time = ConvArrTimeToSecs(user_stop_time)


          ;---------------------------------------------------------
          ; beginning and end of the time axis used in calculating
          ; the position of the labels and tick markers 
          ;---------------------------------------------------------
          st_year = start_tick[0]   
          sp_year = stop_tick[0]

          orig_st_doy =  start_tick[1]
          orig_sp_doy =  stop_tick[1]

          ;---------------------------------------------------------
          ; find the day of year since launch
          ;---------------------------------------------------------
          DaysSinceLaunch, start_tick, new_doy
          start_tick[1] = new_doy

          DaysSinceLaunch, stop_tick, new_doy
          stop_tick[1] = new_doy

          st_doy = start_tick[1]
          sp_doy = stop_tick[1]
 
          st_hr = start_tick[2]
          sp_hr = stop_tick[2]
 
          st_min = start_tick[3]
          sp_min = stop_tick[3]

          st_sec = start_tick[4]
          sp_sec = stop_tick[4]
     

          monthly = 0   ;x-axis label in hh:mm format
    
          if (ustop_time - ustart_time le 86400) then begin 

             ;---------------------------------------------------------
             ; if less than 1 orbit of data
             ;---------------------------------------------------------
             if (ustop_time - ustart_time le 43200) then begin 

                st_year = dstart_time[0]
                st_doy = dstart_time[1]
                st_hr = dstart_time[2]
                st_min = dstart_time[3] 
                st_sec = 0L
                dstart_time[4] = 0L

                sp_year = dstop_time[0]
                sp_doy = dstop_time[1]
                sp_hr = dstop_time[2]
                sp_min = dstop_time[3] 
                sp_sec = 0L
                dstop_time[4] = 0L

             endif else begin

                st_year = user_start_time[0]
                st_doy = user_start_time[1]
                st_hr = user_start_time[2]
                st_min = user_start_time[3] 
                st_sec = user_start_time[4]

 
                sp_year = dstop_time[0]
                sp_doy = dstop_time[1]
                sp_hr = dstop_time[2]
                sp_min = dstop_time[3] 
                sp_sec = dstop_time[4]
             endelse

             start_time = ConvArrTimeToSecs([st_year, st_doy, st_hr, st_min, st_sec, 0]) 
             stop_time = ConvArrTimeToSecs([sp_year, sp_doy, sp_hr, sp_min, sp_sec, 0]) 
             MAX_TICKS = 7L

          endif else begin 

             MAX_TICKS = 9L
             diff_doy = dstop_time[1] - dstart_time[1] 

             if (diff_doy gt 1) then begin
                start_time = ConvArrTimeToSecs(start_tick) 
                monthly = 1  ;x-axis label in day of year format
                if (diff_doy lt MAX_TICKS) then MAX_TICKS = diff_doy + 1 
             endif

          endelse


          ;---------------------------------------------------------
          ; when data for a 24 hour period crosses over the day 
          ; boundary, if the hours for both the start and stop are
          ; zero, set the end time to 23:59:59
          ;---------------------------------------------------------
          if (st_hr eq 0) AND (sp_hr eq 0) then begin
             if (st_doy lt sp_doy) then begin
                sp_doy = sp_doy - 1
                sp_hr = 23
                sp_min = 59
                sp_sec = 59
             endif
         endif 


         l_pltpos.min_time_axis_len = ConvArrTimeToSecs (dstart_time) 
         l_pltpos.max_time_axis_len = ConvArrTimeToSecs (dstop_time) 

         start_time_in_min = 0.0D
         stop_time_in_min = 0.0D

         ;---------------------------------------------------------
         ; convert the user specified start/stop time into minutes 
         ;---------------------------------------------------------
         start_time_in_min = round ((st_doy * 1440.) + (st_hr * 60.) + $
                                   st_min + (st_sec/60.))
         stop_time_in_min = round ((sp_doy * 1440.) + (sp_hr * 60.) + $
                                   sp_min + (sp_sec/60.))


         ;---------------------------------------------------------
         ; find the length in time of the x axis 
         ;---------------------------------------------------------

         tim_ax_len = stop_time_in_min - start_time_in_min

      
         ;---------------------------------------------------------
         ; possible deltas between major tick marks
         ;---------------------------------------------------------
         time_deltas = [1, 2, 5, 10, 15, 20, 30, 60, 120, 180, 240, 360, 720, 1080, $
                        1440, 2880, 7200, 14400, 28800, 72000, 144000] 
         ntime_deltas = n_elements (time_deltas)

        
         ;---------------------------------------------------------
         ; determine the major tick interval
         ;---------------------------------------------------------
         mtest = -1
         for i = 0, ntime_deltas - 1 do begin
             mtst = tim_ax_len/time_deltas[i]
             if (mtest lt 0) AND (mtst lt MAX_TICKS) then begin
                mtest = i
             endif
         endfor

         if (mtest gt 0) then begin
            delta = time_deltas[mtest]
         endif else begin
            if (tim_ax_len lt MAX_TICKS) then begin
               delta = 1
            endif else begin
               print, 'Time period too long'
               RETURN
            endelse
         endelse



         ;---------------------------------------------------------
         ;  the number of tick marks
         ;---------------------------------------------------------
         tmp_nticks = round(double(tim_ax_len)/double(delta)) + 1 

         if (tmp_nticks gt MAX_TICKS) then tmp_nticks = MAX_TICKS 
         xtick_labels = strarr (tmp_nticks)
         maj_xtick_loc = dblarr(tmp_nticks)


         ;---------------------------------------------------------
         ; generate major labels as seconds in a day and as strings
         ;---------------------------------------------------------


         for i = 0, tmp_nticks - 1 do begin
             maj_xtick_loc[i] = start_time + double(i) * double(delta) * 60.
             xtick_labels[i] = ConvToUtc(maj_xtick_loc[i], monthly)
         endfor


         nticks = tmp_nticks
      
         l_pltpos.xtick_labels = xtick_labels 
         l_pltpos.xtick_locs = maj_xtick_loc 



         ;---------------------------------------------------------
         ; determine the number of minor ticks
         ;---------------------------------------------------------

         min_time_deltas = [1, 5, 10, 15, 20, 30, 60, 120, 180, 240]
         nmtime_deltas = n_elements (min_time_deltas)

         tmin = maj_xtick_loc[0]/60.
         tmax = maj_xtick_loc[1]/60.


         if (delta gt 1439) then begin
            min_diff = fix(tmax - tmin)
            case min_diff of
                 5:  mn_tick = 5
                10:  mn_tick = 5
                30:  mn_tick = 3
               180:  mn_tick = 3
               360:  mn_tick = 3
              1080:  mn_tick = 3
              else:  mn_tick = 4
            endcase
         endif else begin

             MN_TICKS = 6 
             mn_diff = tmax - tmin
             mtest = -1
             for i = 0, nmtime_deltas - 1 do begin
                 mn_delta = mn_diff/min_time_deltas[i]
                 if (mtest lt 0) AND (mn_delta lt MN_TICKS) then begin
                    mtest = i
                 endif
             endfor

             mn_delta = min_time_deltas[mtest]
             mn_tick = mn_diff/mn_delta

          endelse

          ;---------------------------------------------------------
          ; multiply by 60 to keep format the same with all the other
          ; x axis related values which is in seconds
          ;---------------------------------------------------------
          l_pltpos.mn_tick_delta = ((tmax - tmin)/mn_tick) * 60.

      endif else begin

         ;---------------------------------------------------------
         ; non-time x axis labels
         ;---------------------------------------------------------


         l_pltpos.min_time_axis_len = start_tick 
         l_pltpos.max_time_axis_len = stop_tick


         nontime_deltas = [1, 2, 5, 10, 15, 20, 30, 100, 1000]
         num_ntds = n_elements (nontime_deltas)

         diff = stop_tick - start_tick + 1L
        
         mtest = -1
         for i = 0, num_ntds - 1 do begin
             if (i eq 7) or (i eq 8) then begin
               for j = 1, 9 do begin
                  nt_delta = diff/nontime_deltas[i] * j
                  if (mtest lt 0) AND (nt_delta lt MAX_TICKS) then begin
                     mtest = i
                  endif
               endfor
             endif else begin
               nt_delta = diff/nontime_deltas[i]
               if (mtest lt 0) AND (nt_delta lt MAX_TICKS) then begin
                  mtest = i
               endif
             endelse
         endfor

         delta = nontime_deltas[mtest]
         nticks = (diff/delta) + 1 


         l_pltpos.nticks = nticks

         xtick_labels = strarr (nticks)

         maj_xtick_loc = intarr (nticks)
         for i = 0, nticks - 1 do begin
             maj_xtick_loc[i] = start_tick + i * delta
             xtick_labels[i] = STRCOMPRESS(STRING(maj_xtick_loc[i]), /REMOVE_ALL)
         endfor


         min_time_deltas = [1, 5, 10, 15, 20, 30, 60, 120, 180, 240]
         nmtime_deltas = n_elements (min_time_deltas)

         MN_TICKS = 6 
         tmin = maj_xtick_loc[0]
         tmax = maj_xtick_loc[1]
         mn_diff = tmax - tmin 
         mtest = -1
         for i = 0, nmtime_deltas - 1 do begin
             mn_delta = mn_diff/min_time_deltas[i]
             if (mtest lt 0) AND (mn_delta lt MN_TICKS) then begin
                mtest = i
             endif
         endfor

         mn_delta = min_time_deltas[mtest]
         mn_tick = mn_diff/mn_delta

         l_pltpos.xtick_labels = xtick_labels 
         l_pltpos.xtick_locs = maj_xtick_loc 
         l_pltpos.mn_tick_delta = ((tmax - tmin)/mn_tick)

 
      endelse
     endelse ;--- end else doy greater than one ---
        

      l_pltpos.xtick_delta = delta
      l_pltpos.nticks = nticks
      l_pltpos.xmn_tick = mn_tick


RETURN
END
