
;-------------------------------------------------------------------------------
;
;  Unit Name : l_ColorRtns.pro
;
;  Purpose   : Contains all of the routine related to color
;              tables and scaling.
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;   ELH               11/00           v1.0           Original implementation
;
;
;  File Revision Number:   %I%
;  File Last Modified  :   %E%  %T%
;
;-------------------------------------------------------------------------------


;-------------------------------------------------------------------------------
;  Function/Procedure: l_CreateColorBar
;  Description:
;      Create the color bar.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  colorbar              float[]       I        color table/bar values 
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;-------------------------------------------------------------------------------

PRO l_CreateColorBar, colorbar
COMMON MYCOLORS, N_COLOR_TABLE, N_GRAYS

  ncolors = N_COLOR_TABLE - N_GRAYS

  temp_cb = bytarr (2, ncolors)
  for i = 0, ncolors-1 do begin
     j = i + N_GRAYS
     temp_cb (0,i) = j
     temp_cb(1,i) = j
  endfor

  colorbar = rebin (temp_cb, 14, ncolors, /sample)

RETURN
END

;-------------------------------------------------------------------------------
;  Procedure: l_LoadRedTempCT
;
;  Description:  Generate the values for the red temperature color table.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  red                   array         O        red values
;  green                 array         O        green values
;  blue                  array         O        blue values
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;      MYCOLORS        N_COLOR_TABLE   int         G          # colors in color table
;                      N_GRAYS         int         G          # grays in color table
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------
PRO l_LoadRedTempCT, red, green, blue
COMMON MYCOLORS, N_COLOR_TABLE, N_GRAYS

   colors = indgen (221)
   red = intarr (N_COLOR_TABLE) 
   green = intarr (N_COLOR_TABLE) 
   blue = intarr (N_COLOR_TABLE) 

   red[0:4]   = [255., 0., 247., 200., 24]
   green[0:4] = [255., 0., 124., 200., 8.]
   blue[0:4]  = [255., 0., 248., 200., 127.]

   red[5:N_COLOR_TABLE-1] = 3*colors <220
   green[5:N_COLOR_TABLE-1] = 3*(colors-47) <220 >0
   blue[5:N_COLOR_TABLE-1] = 3*(colors-135) <220 >0

   openw, cfd, "red_temp.tbl", /get_lun
   for i=0, N_COLOR_TABLE-1 do begin
      printf, cfd, red[i], green[i], blue[i]
   endfor

   close, cfd
   free_lun, cfd 

RETURN
END


;-------------------------------------------------------------------------------
;  Procedure: l_LoadColor
;  Description:
;     Load a predefined color table
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  color_table           long          I        color table selected by user 
;  l_rgb_table           struct       I/0       selected color attributes
;  colorbar              float[]      I/O       color bar values
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;      MYCOLORS        N_COLOR_TABLE   int         G          # colors in color table
;                      N_GRAYS         int         G          # grays in color table
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
; 
;  Notes:
;  Possible color tables to load (predefined in files)
;     1.  color.tbl  => blue, green, yellow, red
;     2.  bproy.tbl  => blue, purple, red, orange, yellow
;     3.  black_white.tbl => black and white
;-------------------------------------------------------------------------------

PRO l_LoadColorTable, color_table, l_rgb_table , colorbar
COMMON MYCOLORS, N_COLOR_TABLE, N_GRAYS


  rgb = intarr (3, N_COLOR_TABLE)

  ;-------------------------------------------------
  ; read selected color table
  ;-------------------------------------------------
  case color_table of
    1: begin
         openr, ifd, 'color.tbl', /get_lun
       end
    2: begin
         openr, ifd, 'bproy.tbl', /get_lun
       end
    3: begin
         openr, ifd, 'black_white.tbl', /get_lun
       end
    4: begin
         openr, ifd, 'bpbcgyr.tbl', /get_lun
       end
    5: begin
         l_LoadRedTempCT, red, green, blue
       end
    else: begin
       end
  endcase


  ;-------------------------------------------------
  ; load selected color table
  ;-------------------------------------------------
     
  if (color_table ge 1) and (color_table le 4) then begin

     readf, ifd, rgb

     l_rgb_table.red   = rgb[0,*]
     l_rgb_table.green = rgb[1,*]
     l_rgb_table.blue  = rgb[2,*]

     tvlct, l_rgb_table.red, l_rgb_table.green, l_rgb_table.blue

     l_CreateColorBar, colorbar

     close, ifd
     free_lun, ifd
  endif else begin

     l_rgb_table.red   = red 
     l_rgb_table.green = green
     l_rgb_table.blue  = blue 

     tvlct, red, green, blue
     l_CreateColorBar, colorbar

  endelse



RETURN
END



;-------------------------------------------------------------------------------
;  Procedure:  l_DrawColorBar
;  Description:
;     Draw the color bar.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  colorbar              float[]       I        color bar values
;  l_setup               struct        I        generic setup parameters
;  bar_position          float[]       I        plot coordinates for color bar
;  l_pltpos              struct        I        generic plot parameters
;  LAXIS                 IDL keyword   I        plot axis on left OR
;  RAXIS                 IDL keyword   I        plot axis on right 
;
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;  ELH               05/04           v1.0.11          Modified if (ntcks lt 1)
;                                                     condition to have at least
;                                                     one tick mark
;-------------------------------------------------------------------------------


PRO l_DrawColorbar, colorbar, l_setup, $
bar_position, l_pltpos, LAXIS=laxis, RAXIS=raxis


  bar_min = l_setup.l_color_min
  bar_max = l_setup.l_color_max

  ntcks   = (bar_max - bar_min)
  if (ntcks lt 1) then begin
     bar_max = bar_min * bar_min/2.
     ntcks = 1
     if (bar_max eq 0) then begin
        bar_min = 0.0
        bar_max = 1.0
     endif
  endif
  if (ntcks gt 5) then begin
      ntcks = 5
  endif

  ncolors=!D.TABLE_SIZE

  ;--------------------------------------------------
  ; Get starting locations in NORMAL coordinates.
  ;--------------------------------------------------
  xstart = bar_position(0)
  ystart = bar_position(1)

  ;--------------------------------------------------
  ; Get the size of the bar in NORMAL coordinates.
  ;--------------------------------------------------
  xsize = bar_position(2) - bar_position(0)
  ysize = bar_position(3) - bar_position(1)

  bar_w = fix(xsize*l_pltpos.window_width)
  bar_h = fix(ysize*l_pltpos.window_height)

  ;--------------------------------------------------
  ; Display the color bar in the window. Sizing is
  ; different for PostScript and regular display.
  ; for device
  ;--------------------------------------------------
  bar = CONGRID(colorbar, bar_w, bar_h)

  TV, bar, xstart, ystart, XSIZE=xsize, YSIZE=ysize, /Normal

  PLOT, [0.0,1.0],                  $
        [bar_min,bar_max],          $
        /NODATA,                    $
        /NOERASE,                   $
        XTICKS = 1,                 $
        YTICKS = ntcks,             $
        XSTYLE = 1,                 $
        YSTYLE = 1,                 $
        POSITION = bar_position,    $
        COLOR = 1,                  $
        YTICKFORMAT = '(A1)',       $
        XTICKFORMAT = '(A1)',       $
        YTICKLEN = 0.2,             $
        FONT=-1,                    $
        YMINOR=2
  
  if Keyword_Set(laxis) then begin
      AXIS, /YAXIS,                      $
            YRANGE = [bar_min, bar_max], $
            YSTYLE = 1,                  $
            YTICKFORMAT = laxis.yformat, $
            YTICKS = ntcks,              $
            YTICKLEN = 0.2,              $
            COLOR = 1,                   $
            CHARSIZE = laxis.char_size,  $
            YCHARSIZE = laxis.char_size, $
            FONT = -1,                   $
            YMINOR = 2,                  $
            YTITLE = laxis.ytitle
  endif

  if Keyword_Set(raxis) then begin
      AXIS, /YAXIS,                      $
            YRANGE = raxis.yrange,       $
            YSTYLE = 1,                  $
            YTICKFORMAT = raxis.yformat, $
            YTICKS = ntcks,              $
            YTICKLEN = 0.2,              $
            COLOR = 1,                   $
            CHARSIZE = raxis.char_size,  $
            YCHARSIZE = raxis.char_size, $
            FONT = -1,                   $
            YMINOR = 2,                  $
            YTITLE = raxis.ytitle
  endif

END


;-------------------------------------------------------------------------------
;  Procedure: l_ScaleColor
;  Description:
;      Perform color scaling on the data.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  image                 float[]       I        image to be scaled
;  out_image             float[]       O        scaled image 
;  color_min             long          I        minimum color scale
;  color_max             long          I        maximum color scale
;  color_table           long          I        color table selected by user 
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;     PLOTDEFS         MAX_PLOTS        int         G         max # plots per page
;                      MAX_WEB_SPINS    int         G         max # spins per plot
;                                                             when using web interface
;                      NO_CNTS          double      G         no counts indicator
;                      NO_DATA          double      G         no data indicator
;
;      MYCOLORS        N_COLOR_TABLE   int         G          # colors in color table
;                      N_GRAYS         int         G          # grays in color table
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;-------------------------------------------------------------------------------

PRO l_ScaleColor, image, out_image, color_min, color_max, color_table
COMMON PLOTDEFS, MAX_PLOTS, MAX_WEB_SPINS, NO_CNTS, NO_DATA
COMMON MYCOLORS, N_COLOR_TABLE, N_GRAYS


  ncolors = float(N_COLOR_TABLE - N_GRAYS)

  ;---------------------------------------
  ; handles color bar range if all no data
  ; or all no counts
  ;---------------------------------------
  color_diff = color_max - color_min
  if (color_diff le 0.0) then begin
     color_max = color_min * color_min/2. 
     color_diff = color_max - color_min
     if (color_diff eq 0.0) then begin
        color_min = 0.0
        color_max = 1.0
        color_diff = 1.0
     endif
  endif

  new_image = fix(((image-color_min)/color_diff)* ncolors) + N_GRAYS 

  ;--------------------------------------
  ; value is below the requested minimum 
  ; make the lowest color value in table
  ;--------------------------------------
  less_min = where(image le color_min, wcnt)
  if (wcnt gt 0) then new_image[less_min] = 5

  ;--------------------------------------
  ; value is above requested maximum, make
  ; highest color value in table 
  ;--------------------------------------
  over_max = where(image gt color_max, wcnt)
  if (wcnt gt 0) then new_image[over_max] = 225

  ;--------------------------------------
  ; no counts, make dark blue
  ;--------------------------------------
  if (color_table lt 4) then begin
     if (color_table eq 3) then begin
       nocnt_color = 1 
     endif else begin      
       nocnt_color = 4 
     endelse
  endif else begin 
     nocnt_color = 5
  endelse

  nocnts  = where(image eq NO_CNTS, wcnt)
  if (wcnt gt 0) then new_image[nocnts] = nocnt_color 

  ;--------------------------------------
  ; no data, make gray 
  ;--------------------------------------
  nodata = where(image eq NO_DATA, wcnt)
  if (wcnt gt 0) then new_image[nodata] = 3 

  out_image = new_image

  RETURN

END
