
;-------------------------------------------------------------------------------
;
;  Unit Name : l_CoincRtns.pro
;
;  Purpose   : All routines related to retrieving and plotting the coincidence
;              rates (valid events).
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;   ELH               5/00           v1.0           Original implementation
;   ELH             07/05/01         v1.0           Modified for dumps to match
;                                                   spectrograms w/no conversions
;                                                   Modified y-axis labels for 
;                                                   sun, earth, moon, none 
;   ELH             09/18/01         v1.0.7         Draw sun direction
;   ELH             09/26/01         v1.0.7         Added l_DetermineSunDir 
;   ELH             01/02            v1.0.8         Added parm in l_HndlMissData to
;                                                   handle rolling data based on
;                                                   spin angle input
;
;
;  File Revision Number:   %I%
;  File Last Modified  :   %E%  %T%
;
;-------------------------------------------------------------------------------

;-------------------------------------------------------------------------------
;  Procedure: l_CoincReadSetup
;  Description:
;      Read parameters and user options specific to coincidences.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_plot_setup          struct       I/O       coincidence options/parameters
;  l_setup_file          string        I        setup filename
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;-------------------------------------------------------------------------------


PRO l_CoincReadSetup, l_plot_setup, l_setup_file


  l_plot_setup = {               $
       coinc           :  'N', $
       coinc_color_min :  0.0, $
       coinc_color_max :  0.0, $
       nplots          :  0L   $
   }

   openr, inlun, /get_lun, l_setup_file
   ;----------------------------------------
   ; read singles data setup parameters
   ;----------------------------------------

   l_plot_setup.nplots = 0L
   line = ' '

   ;----------------------------------------
   ;While there is text left, output it.
   ;----------------------------------------
   while not EOF(inlun) do begin
     readf, inlun, line
     IF (STRMID(line, 0, 1) ne ';') and $
          (STRLEN(line) gt 0) then begin

       parts = STR_SEP(line, ' ')

       if (n_elements(parts) gt 2) then begin
       case parts[0] of
         'l_coinc'             : begin
                                   l_plot_setup.coinc=STRUPCASE(parts[2])
                                   if l_plot_setup.coinc eq 'Y' then begin
                                      l_plot_setup.nplots = 1L
                                   endif
                                 end
         'l_coinc_color_min'   : begin
                                   l_plot_setup.coinc_color_min=parts[2]
                                 end
         'l_coinc_color_max'   : begin
                                   l_plot_setup.coinc_color_max=parts[2]
                                 end

         else: begin
               end
       endcase              ;End of CASE statement.
       endif
    endif
  endwhile

  close, inlun
  free_lun, inlun

RETURN
END

;-------------------------------------------------------------------------------
;  Procedure: l_CoincAdjExData
;
;  Description: This routine determines if extra spins were read at the beginning
;     or at the end of the user specified time period.   This is used when various
;     traces are centered at zero degree spin angle.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  addedStartFlag        long          I        spins added to start time
;  addedStopFlag         long          I        spins added to stop time
;  SPIN_SECTORS          long          I        number of spin sectors
;  bd                    struct *      I        poiner to binned/coincidence data
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;     PLOTDEFS         MAX_PLOTS        int         G         max # plots per page
;                      MAX_WEB_SPINS    int         G         max # spins per plot
;                                                             when using web interface
;                      NO_CNTS          double      G         no counts indicator
;                      NO_DATA          double      G         no data indicator
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------

PRO l_CoincAdjExData, addedStartFlag, addedStopFlag, SPIN_SECTORS, bd 
COMMON PLOTDEFS, MAX_PLOTS, MAX_WEB_SPINS, NO_CNTS, NO_DATA


     nspins = n_elements (*bd.startUdfTimePtr) - 1
     new_bd = fltarr (SPIN_SECTORS, nspins)
     extra_data = make_array (SPIN_SECTORS, 2L, /FLOAT, Value = NO_DATA)

     new_total = (*bd.m0RawPtr) + (*bd.m1RawPtr)
     n1        = total (new_total, 1)
     new_bd    = total (n1, 1)

     ptr = ptr_new (new_bd)
     bd.vePtr = ptr

     ptr = ptr_new (extra_data)
     bd.veExtraPtr = ptr

     if (nspins lt 4) then return

     nrec = n_elements ((*bd.timePtr)) - 1

     if (addedStartFlag eq 1L) AND (addedStopFlag eq 1L) then begin
          (*bd.timePtr) = (*bd.timePtr)[SPIN_SECTORS:nrec-SPIN_SECTORS]
          (*bd.spinTimePtr)     =  (*bd.spinTimePtr)[1:nspins-1]
          (*bd.startUdfTimePtr) =  (*bd.startUdfTimePtr)[1:nspins-1]
          (*bd.stopUdfTimePtr)  =  (*bd.stopUdfTimePtr)[1:nspins-1]
          extra_data[*,0] = new_bd[*, 0]
          extra_data[*,1] = new_bd[*, nspins]
          (*bd.veExtraPtr) = extra_data
          (*bd.vePtr) = (*bd.vePtr)[*, 1:nspins-1]
     endif else begin
        if (addedStartFlag eq 1L) AND (addedStopFlag eq 0L) then begin
           (*bd.timePtr) = (*bd.timePtr)[SPIN_SECTORS:nrec]
           (*bd.spinTimePtr)     =  (*bd.spinTimePtr)[1:nspins]
           (*bd.startUdfTimePtr) =  (*bd.startUdfTimePtr)[1:nspins]
           (*bd.stopUdfTimePtr)  =  (*bd.stopUdfTimePtr)[1:nspins]
           extra_data[*,0] = new_bd[*,0]
           extra_data[*,1] = NO_DATA
           (*bd.veExtraPtr) = extra_data
           (*bd.vePtr) = (*bd.vePtr)[*, 1:nspins]
        endif else begin
           if (addedStartFlag eq 0L) AND (addedStopFlag eq 1L) then begin
              (*bd.timePtr) = (*bd.timePtr)[0:nrec-SPIN_SECTORS]
              (*bd.spinTimePtr)     =  (*bd.spinTimePtr)[0:nspins-1]
              (*bd.startUdfTimePtr) =  (*bd.startUdfTimePtr)[0:nspins-1]
              (*bd.stopUdfTimePtr)  =  (*bd.stopUdfTimePtr)[0:nspins-1]
              extra_data[*,0] = NO_DATA
              extra_data[*,1] = new_bd[*, nspins]
              (*bd.veExtraPtr) = extra_data
              (*bd.vePtr) = (*bd.vePtr)[*, 0:nspins-1]
           endif
        endelse
     endelse

    nspins = n_elements (*bd.startUdfTimePtr) - 1 

RETURN
END

;-------------------------------------------------------------------------------
;  Function:  l_CoincRead
;
;  Description:
;     Read the binned data and sum over all masses, energies, and polar zones.
;
;  Return Value:  type = int 
;  Value                         Description
;  -------------------------     ------------------------------
;  SUCCESS                       successful completion
;  FAILURE                       failure to complete
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_binnedd_setup       struct        I        binned processing & plotting parameters
;  bd                    struct *      I        pointers to binned data
;  l_setup               struct        I        general setup parameters
;  nrec                  long          I        number of records
;  l_pltpos              struct        I        general plot parameters
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;     PLOTDEFS         MAX_PLOTS        int         G         max # plots per page
;                      MAX_WEB_SPINS    int         G         max # spins per plot
;                                                             when using web interface
;                      NO_CNTS          double      G         no counts indicator
;                      NO_DATA          double      G         no data indicator
;
;      ERROR_TYPES     SUCCESS         int         G          successful completion
;                      FAILURE         int         G          unsuccessful completion
;                      WARNING         int         G          warning message
;                      INFO            int         G          informational message
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;  ELH              11/02/01         v1.0.7           Removed UdfGetNumRec routine
;-------------------------------------------------------------------------------

FUNCTION l_CoincRead, l_binned_setup, bd, l_setup, nrec, l_pltpos
COMMON PLOTDEFS, MAX_PLOTS, MAX_WEB_SPINS, NO_CNTS, NO_DATA
COMMON ERROR_TYPES, SUCCESS, FAILURE, WARNING, INFO


  ;--------------------
  ;  get num image rec
  ;--------------------
  inst = 'LENA'
  expr = 'LENASCI'
  vinst = 'IMLIMAGE'

  nrec = 0L
  data_source = ' '

  start_tme = l_setup.l_start_dtime
  stop_tme = l_setup.l_stop_dtime


  ;----------------------------------------------------
  ; Add extra time to the user specified stop time
  ; to correct for the fact that UDF does not read up
  ; to and including the last spin for the binned data
  ; as it does for the other data, this is so that the
  ; time and data will be sync with the other data 
  ;----------------------------------------------------

  ;----------------------------------------------------
  ; if the start time is not 00:00:00, then add extra
  ; minutes to the begin time
  ;----------------------------------------------------
  addedStartFlag = 0L
  if (ChckStartTime(start_tme) eq FAILURE) then begin
      AddExtraTime2Start, start_tme, 2
      addedStartFlag = 1L
  endif

  ;----------------------------------------------------
  ; if the stop time is not 23:59:59, then add extra
  ; minutes to the end time
  ;----------------------------------------------------
  addedStopFlag = 0L
  if (ChckStopTime(stop_tme) eq FAILURE) then begin
      AddExtraTime2Stop, stop_tme, 4 
      addedStopFlag = 1L
  endif


  l_pltpos.data_source = data_source

  ;----------------------------------------------------
  ; read the binned udf data by time
  ;----------------------------------------------------
  l_binned_setup.l_binned_plot_type = 'T'
  status = l_BinnedReadData (l_setup, l_binned_setup, nrec, bd, $
                 l_pltpos, start_tme, stop_tme)
  if (status eq FAILURE) then return, FAILURE

  ;----------------------------------------------------
  ; determine the number of spins 
  ;----------------------------------------------------
  nspins = nrec/l_setup.l_num_spinsec
  l_setup.l_num_spin = nspins

  ;----------------------------------------------------
  ; sum binned data over energy, mass, and polar zone
  ; to obtain the sum value for each spin sector
  ; these are the coincidences  
  ;----------------------------------------------------
  if (status eq SUCCESS) then begin
     l_CoincAdjExData, addedStartFlag, addedStopFlag, l_setup.l_num_spinsec, bd 
  endif



  RETURN, SUCCESS 
END


;-------------------------------------------------------------------------------
;  Procedure: l_CoincHndlMissData
;
;  Description: Fill data gaps will fill data.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_binned              struct *      I        pointers to binned data
;  l_setup               struct        I        general setup parameters
;  l_oa                  struct *      I        pointers to OA data
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;     PLOTDEFS         MAX_PLOTS        int         G         max # plots per page
;                      MAX_WEB_SPINS    int         G         max # spins per plot
;                                                             when using web interface
;                      NO_CNTS          double      G         no counts indicator
;                      NO_DATA          double      G         no data indicator
;
;      ERROR_TYPES     SUCCESS         int         G          successful completion
;                      FAILURE         int         G          unsuccessful completion
;                      WARNING         int         G          warning message
;                      INFO            int         G          informational message
;
;      MYCOLORS        N_COLOR_TABLE   int         G          # colors in color table
;                      N_GRAYS         int         G          # grays in color table
;
;      EPOCH_DATA      EPOCH_YR        int         G          launch year
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------


PRO l_CoincHndlMissData, l_binned, l_setup, l_oa


  adjFlag = 'YES'
  if (l_setup.l_angle_zero eq 'N') then adjFlag = 'NO'


  if (l_setup.l_angle_zero eq 'S') then begin
     l_CalcSunPulse, l_oa, lena_sector, 'S'
     center_sector = fix(lena_sector)
     cs = fix (lena_sector/8.)
     if NOT (ptr_valid(l_oa.lena_sun_sectPtr)) then begin
        ptr = ptr_new (cs)
        l_oa.lena_sun_sectPtr = ptr
     endif
  endif else begin
     if (l_setup.l_angle_zero eq 'M') then begin
        l_CalcSunPulse, l_oa, lena_sector, 'M'
        center_sector = fix(lena_sector)
        cs = fix (center_sector/8.)
        if NOT (ptr_valid(l_oa.lena_moon_sectPtr)) then begin
           ptr = ptr_new (cs)
           l_oa.lena_moon_sectPtr = ptr
        endif
     endif else begin
        if (l_setup.l_angle_zero eq 'E') then begin
           center_sector = l_setup.l_earth_center
        endif else begin
           center_sector = 0
        endelse
     endelse
  endelse

  if (l_setup.l_sun_direction eq 'Y') AND (l_setup.l_angle_zero ne 'S') then begin
     l_CalcSunPulse, l_oa, lena_sector, 'S'
     sun_direction = fix (lena_sector/8)
     if NOT (ptr_valid(l_oa.lena_sun_sectPtr)) then begin
        ptr = ptr_new (sun_direction)
        l_oa.lena_sun_sectPtr = ptr
     endif
  endif

  nspins    = n_elements((*l_binned.startUdfTimePtr)) - 1

  l_HndlMissData, (*l_binned.timePtr), $
                  filled_time, $
                  nspins, $
                  (*l_binned.vePtr), $
                  filled_data, $
                  l_setup.l_num_spinsec , $ 
                  center_sector, $
                  l_setup, $
                  adjFlag, $
                  l_oa, $
                  (*l_binned.startUdfTimePtr),  $
                  filled_start_udf_time, $
                  (*l_binned.veExtraPtr) , $
                  l_setup.l_spin_ang_zero 


  ptr = ptr_new (filled_time)
  l_binned.filledTimePtr = ptr

  ptr = ptr_new (filled_data)
  l_binned.filledVePtr = ptr

  ptr = ptr_new(filled_start_udf_time)
  l_binned.filledStartUdfTimePtr = ptr


RETURN
END

;-------------------------------------------------------------------------------
;  Procedure : l_CoincPlot
;
;  Description:  Plot the coincidence data
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  nrec                  long          I        total number of records
;  l_pltpos              struct        I        plot parameters
;  l_oa                  struct*       I        OA parameters
;  plotted               long          I        number of plots currently plotted
;  nplots                long          I        total # of plots to be plotted
;  pages                 long          I        total # of plot pages needed
;  windx                 long[]        I        window index, if X windows selected
;  l_setup               struct        I        general setup parameters
;  l_rgb_table           struct        I        color table parameters
;  colorbar              float[]       I        color table values
;  l_coinc_setup         struct        I        coincidence plot specific options
;  l_hskp                struct*       I        housekeeping parameters
;  l_binned              struct*       I        pointers to binned data
;  
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;  ELH              07/05/01         v1.0             Modified dump to match 
;                                                     spectrograms
;                                                     Modified y-axis title for
;                                                     sun, earth, moon, none 
;  ELH              09/18/01         v1.0.7           Added capability to draw
;                                                     sun directions on plots if
;                                                     selected
;  ELH              09/26/01         v1.0.7           Added l_DetermineSunDir 
;-------------------------------------------------------------------------------

PRO l_CoincPlot, nrec, l_pltpos, l_oa, plotted, nplots, pages, $
windx, l_setup, l_rgb_table, colorbar, l_coinc_setup, l_hskp, l_binned


  ytitle      = '!5log count Rate[Hz]'


  if (l_setup.l_angle_zero eq 'S') then begin
     ytitle_frame= '!5Sun Cntrd Azi'
  endif else begin
     if (l_setup.l_angle_zero eq 'M') then begin
        ytitle_frame= '!5Moon Cntrd Azi'
     endif else begin
        if (l_setup.l_angle_zero eq 'E') then begin
           ytitle_frame= '!5Earth Cntrd Azi'
        endif else begin
           ytitle_frame= '!5Spin Angle'
        endelse
     endelse
  endelse


  ;SyncData2Time, (*l_binned.filledvePtr), $
  ;               (*l_binned.filledStartUdfTimePtr), $
  ;               l_setup

  ;-----------------------------------
  ; determine the number of major and
  ; minor tick marks
  ;-----------------------------------

   l_setup.l_d_start_dtime[0] = (*l_binned.filledStartUdfTimePtr)[0].year
   l_setup.l_d_start_dtime[1] = (*l_binned.filledStartUdfTimePtr)[0].doy
   l_setup.l_d_start_dtime[2] = (*l_binned.filledStartUdfTimePtr)[0].hour
   l_setup.l_d_start_dtime[3] = (*l_binned.filledStartUdfTimePtr)[0].min
   l_setup.l_d_start_dtime[4] = (*l_binned.filledStartUdfTimePtr)[0].sec
   l_setup.l_d_start_dtime[5] = (*l_binned.filledStartUdfTimePtr)[0].msec

   max_ele = n_elements ((*l_binned.filledStartUdfTimePtr)) - 1

   l_setup.l_d_stop_dtime[0] = (*l_binned.filledStartUdfTimePtr)[max_ele].year
   l_setup.l_d_stop_dtime[1] = (*l_binned.filledStartUdfTimePtr)[max_ele].doy
   l_setup.l_d_stop_dtime[2] = (*l_binned.filledStartUdfTimePtr)[max_ele].hour
   l_setup.l_d_stop_dtime[3] = (*l_binned.filledStartUdfTimePtr)[max_ele].min
   l_setup.l_d_stop_dtime[4] = (*l_binned.filledStartUdfTimePtr)[max_ele].sec
   l_setup.l_d_stop_dtime[5] = (*l_binned.filledStartUdfTimePtr)[max_ele].msec


   l_DetMjMnTicks, l_setup.l_start_dtime, $
                   l_setup.l_stop_dtime, $
                   l_setup.l_d_start_dtime, $
                   l_setup.l_d_stop_dtime, $
                   l_pltpos


 
  ;------------------------------------------------------
  ; find the indexes where values are not equal to zero
  ;------------------------------------------------------

  coinc_data = (*l_binned.filledVePtr)
  log_ztitle = ' '
  cnt_ztitle = ' '
  zaxis_title = 'linear cnts'

  if (l_setup.l_dumps eq 'Y') then begin
     ptr = ptr_new (coinc_data)
     l_binned.veDmpPtr = ptr
  endif

  if (l_setup.l_count_rate eq 'Y') then begin
     l_Conv2CntRate, coinc_data, coinc_cntRate, 'SECTOR'
     coinc_data = coinc_cntRate
     zaxis_title = ' '
     log_ztitle = '!5log '
  endif

  if (l_setup.l_log eq 'Y') then begin
     l_Conv2Log, coinc_data, coinc_log, coinc_min, coinc_max
     coinc_data = coinc_log
     zaxis_title = ' '
     cnt_ztitle = '!5Cnt Rate [Hz]'
  endif

  zaxis_title = log_ztitle + cnt_ztitle


  if (l_setup.l_auto_scale eq 'Y') then begin
      l_setup.l_color_min = coinc_min
      l_setup.l_color_max = coinc_max
  endif else begin
      if (l_setup.l_sep_plot_scale eq 'Y') then begin
          l_setup.l_color_min = l_coinc_setup.coinc_color_min
          l_setup.l_color_max = l_coinc_setup.coinc_color_max
      endif
  endelse


  if (l_setup.l_line_plots eq 'Y') then begin

     ;------------------------------------------------------
     ; generate a log plot
     ;------------------------------------------------------

     new_max = (coinc_max/4.) + coinc_max

     l_PlotStuff, nplots, $
                  (*l_binned.filledTimePtr), $
                  coinc_data,   $
                  coinc_min,   $
                  new_max, $
                  ytitle,  $
                  l_pltpos, $
                  plotted, $ 
                  l_oa, $
                  l_setup.l_start_dtime


  endif else begin

     ;------------------------------------------------------
     ; generate a spectrogram plot
     ;------------------------------------------------------

     l_ScaleColor, coinc_data, coinc_scaled, $
                   l_setup.l_color_min, $
                   l_setup.l_color_max, $
                   l_setup.l_color_tbl


     ;------------------------------------------------------------
     ; correct the image positioning, rotate 270 degrees
     ;------------------------------------------------------------
     coinc_rot = rotate(coinc_scaled, 3)

     yrng=[l_pltpos.min_spin_ang,l_pltpos.max_spin_ang]

     img_pos  = [l_pltpos.xpltmn,l_pltpos.ypltmn,$
                l_pltpos.xpltmx,l_pltpos.ypltmx]

     position = float(img_pos)

     xstart = position[0]
     ystart = position[1]
     ximage = l_pltpos.axis_size/ l_pltpos.xin
     yimage = l_pltpos.ypltmx - l_pltpos.ypltmn 
     xsize  = round(l_pltpos.window_width * ximage)
     ysize  = round(l_pltpos.window_height * yimage)

     coinc_spect = congrid (coinc_rot, $
                            xsize, $
                            ysize)

     if (l_setup.l_smooth_data eq 'Y') then begin
        TV, smooth (coinc_spect, 3, /edge_truncate), xstart, ystart, $
            XSIZE=ximage, YSIZE=yimage, /ORDER, /NORMAL
     endif else begin
        TV, coinc_spect, xstart, ystart, $
            XSIZE=ximage, YSIZE=yimage, /ORDER, /NORMAL
     endelse

     if (l_setup.l_earth_markers eq 'Y') then begin
         l_CalcEarthDirection, l_oa, l_setup, img_pos
     endif

     ;---------------------------------------------------
     ; draw the sun direction
     ;---------------------------------------------------
     if (l_setup.l_sun_direction eq 'Y') then begin
         l_CalcSunDirection, l_oa, l_setup, img_pos
     endif

     ;---------------------------------------------------
     ; calculate and draw the ram direction
     ;---------------------------------------------------
     if (l_setup.l_ram_direction eq 'Y') then begin
        l_CalcSunPulse, l_oa, ram_sector, 'R'
        ram_sector = ram_sector/8.
        if NOT (ptr_valid(l_oa.lena_ram_sectPtr)) then begin
           ptr = ptr_new (ram_sector)
           l_oa.lena_ram_sectPtr = ptr
        endif
        l_CalcRamDirection, l_oa, l_setup, img_pos
     endif


     ;---------------------------------------------------
     ; draw the B direction
     ;---------------------------------------------------
     if (l_setup.l_B_direction eq 'Y') then begin
        l_CallT96Model, l_oa, l_setup
        l_CalcSunPulse, l_oa, B_sector, 'B'
        B_sector = B_sector/8.
        if NOT (ptr_valid(l_oa.lena_mag_fieldPtr)) then begin
           ptr = ptr_new (B_sector)
           l_oa.lena_mag_fieldPtr = ptr
        endif
        l_CalcBDirection, l_oa, l_setup, img_pos
     endif


     bar_pos  = [l_pltpos.xpltmx+0.015, $
                l_pltpos.ypltmn, $
                l_pltpos.xpltmx+0.025, $
                l_pltpos.ypltmx]

     bar_info = {AXIS_DATA, char_size: l_pltpos.plot_font, $
                 ytitle:zaxis_title, $
                 yformat:'(F4.1)', $
                 yrange:[l_setup.l_color_min, $
                         l_setup.l_color_max ]}

     l_DrawColorbar, colorbar, l_setup, bar_pos, l_pltpos, $
                     RAXIS=bar_info

     l_PlotFrame, nplots, (*l_binned.filledTimePtr), yrng, img_pos, l_pltpos, $
                  plotted, l_oa, ytitle_frame, l_setup.l_start_dtime, $
                  l_hskp


  endelse

  yctr = (l_pltpos.ypltmn + l_pltpos.ypltmx)/2.0
  xyouts, 0.02, yctr, 'Binned VE', size=l_pltpos.sm_font, $
          /NORMAL, alignment = 0.5, orientation=90.0

  l_pltpos.ypltmx = l_pltpos.ypltmn - l_pltpos.yspace
  l_pltpos.ypltmn = l_pltpos.ypltmx - l_pltpos.ydel
  plotted = plotted + 1

  l_ChckCompPg, plotted, $
                nplots, $
                pages, $
                l_setup, $
                windx, $
                l_pltpos, $
                l_rgb_table


RETURN
END
