; Description: Contains all of the coincidence routines 
; FUNCTION: l_BrowseRtns
;
; Developer    Date     Build       Description
; ---------    ----     -----       -------------------------
; E. Lee       5/00      0.0        original implementation
;
;============================================================

;============================================================
; Function:  l_BrowseProc
; Description: read the binned data and sum over all mass,
;              energy, and polar zones
;
;============================================================
PRO l_BrowseProc, bd, mass0_bd, mass1_bd, l_setup, l_pltpos, l_rgb_table, $
                  nrec, data_during_request, req_record_start, $
                  req_record_stop, req_actual_num_spins, req_spins, $
                  req_spin_start_rec, req_spin_stop_rec
COMMON GLOBALS, g_report, g_log

;------------------------------------------------------------------------------
;  binned data structure
;------------------------------------------------------------------------------

ptr=ptr_new()       ; NULL ptr
bd = {                              $
      timePtr                : ptr, $    ; begin time for each spin sector
      filledTimePtr          : ptr, $    ; missing data included
      spinTimePtr            : ptr, $    ; begin time for each spin - 1 time/spin
      startUdfTimePtr        : ptr, $    ; udf start time
      filledStartUdfTimePtr  : ptr, $    ; missing data included
      stopUdfTimePtr         : ptr, $    ; udf start time
      m0RawPtr               : ptr, $    ; m0 telemetry, 3 energy levels
      filledM0RawPtr         : ptr, $   ; missing data included, m0 telemetry, 3 energy levels
      m0RawExtraPtr          : ptr, $    ; m0 telemetry, 3 energy levels
      m1RawPtr               : ptr, $    ; m1 telemetry, 3 energy levels
      filledM1RawPtr         : ptr, $    ; missing data included, m1 telemetry, 3 energy levels
      m1RawExtraPtr          : ptr, $    ; m1 telemetry, 3 energy levels
      m0CntPtr               : ptr, $    ; m0 count rate, 3 energy levels
      filledM0CntPtr         : ptr, $    ; missing data included, m0 count rate, 3 energy levels
      m0CntExtraPtr          : ptr, $    ; m0 count rate, 3 energy levels
      m1CntPtr               : ptr, $    ; m1 count rate, 3 energy levels
      filledM1CntPtr         : ptr, $    ; missing data included, m1 count rate, 3 energy levels
      m1CntExtraPtr          : ptr, $    ; m1 count rate, 3 energy levels
      m0FlxPtr               : ptr, $    ; m0 number flux, 3 energy levels
      m1FlxPtr               : ptr, $    ; m0 number flux, 3 energy levels
      flxPtr                 : ptr, $    ; collapsed number flux
      vePtr                  : ptr, $    ; all masses (valid events)
      filledVePtr            : ptr, $    ; missing data included, all masses (valid events)
      veExtraPtr             : ptr, $    ; all masses (valid events)
      veDmpPtr               : ptr, $    ; all masses (valid events) Dump
      m0Ptr                  : ptr, $    ; m0 (H) all energies, polar zone [sector, spin]
      filledM0Ptr            : ptr, $    ; missing data included, m0 (H) all energies, polar zone
                                         ; [sector, spin]
      m0ExtraPtr             : ptr, $    ; m0 (H) all energies, polar zone [sector, spin]
      m0DmpPtr               : ptr, $    ; m0 (H) all energies, polar zone [sector, spin], 
                                         ; text dump version
      m0lePtr                : ptr, $    ; m0 (H) lower energy
      filledM0lePtr          : ptr, $    ; missing data included, m0 (H) lower energy
      m0leExtraPtr           : ptr, $    ; m0 (H) lower energy
      m0leDmpPtr             : ptr, $    ; m0 (H) lower energy, text dump version
      m0cePtr                : ptr, $    ; m0 (H) center energy
      filledM0cePtr          : ptr, $    ; missing data included, m0 (H) center energy
      m0ceExtraPtr           : ptr, $    ; m0 (H) center energy
      m0ceDmpPtr             : ptr, $    ; m0 (H) center energy, text dump version
      m0uePtr                : ptr, $    ; m0 (H) upper energy
      filledM0uePtr          : ptr, $    ; missing data included, m0 (H) upper energy
      m0ueExtraPtr           : ptr, $    ; m0 (H) upper energy
      m0ueDmpPtr             : ptr, $    ; m0 (H) upper energy , text dump version
      m1Ptr                  : ptr, $    ; m1 (O)  all energies, polar zone
                                         ; [sector, spin]
      filledM1Ptr            : ptr, $    ; missing data included, m1 (O)  all energies, polar zone
                                         ; [sector, spin]
      m1ExtraPtr             : ptr, $    ; m1 (O)  all energies, polar zone
                                         ; [sector, spin]
      m1DmpPtr               : ptr, $    ; m1 (O)  all energies, polar zone
                                         ; [sector, spin], text dump version
      m1lePtr                : ptr, $    ; m1 (O) lower energy
      filledM1lePtr          : ptr, $    ; missing data included, m1 (O) lower energy
      m1leExtraPtr           : ptr, $    ; m1 (O) lower energy
      m1leDmpPtr             : ptr, $    ; m1 (O) lower energy, text dump version
      m1cePtr                : ptr, $    ; m1 (O) center energy
      filledM1cePtr          : ptr, $    ; missing data included, m1 (O) center energy
      m1ceExtraPtr           : ptr, $    ; m1 (O) center energy
      m1ceDmpPtr             : ptr, $    ; m1 (O) center energy, text dump version
      m1uePtr                : ptr, $    ; m1 (O) upper energy
      filledM1uePtr          : ptr, $    ; missing data included, m1 (O) upper energy
      m1ueExtraPtr           : ptr, $    ; m1 (O) upper energy
      m1ueDmpPtr             : ptr, $    ; m1 (O) upper energy, text dump version
      strStUdfTimePtr        : ptr, $    ; steering start UDF time
      filledStrStUdfTimePtr  : ptr, $    ; missing data included, steering start UDF time
      strSpUdfTimePtr        : ptr, $    ; steering stop UDF time
      filledStrSpUdfTimePtr  : ptr, $    ; missing data included, steering stop UDF time
      steervPtr              : ptr, $    ; steering voltage levels / spin
      filledSteervPtr        : ptr, $    ; missing data included, steering voltage levels / spin
      bkcUdfTimePtr          : ptr, $
      hydrogen_raw_cnts      : ptr, $
      hydrogen_raw_cnts_e0   : ptr, $
      hydrogen_raw_cnts_e1   : ptr, $
      hydrogen_raw_cnts_e2   : ptr, $
      corrected_hydrogen     : ptr, $
      corrected_hydrogen_e0  : ptr, $
      corrected_hydrogen_e1  : ptr, $
      corrected_hydrogen_e2  : ptr, $
      corr_scl_H             : ptr, $
      corr_scl_H_e0          : ptr, $
      corr_scl_H_e1          : ptr, $
      corr_scl_H_e2          : ptr, $
      oxygen_raw_cnts        : ptr, $
      oxygen_raw_cnts_e0     : ptr, $
      oxygen_raw_cnts_e1     : ptr, $
      oxygen_raw_cnts_e2     : ptr, $
      corrected_oxygen       : ptr, $
      corrected_oxygen_e0    : ptr, $
      corrected_oxygen_e1    : ptr, $
      corrected_oxygen_e2    : ptr, $
      corr_scl_O             : ptr, $
      corr_scl_O_e0          : ptr, $
      corr_scl_O_e1          : ptr, $
      corr_scl_O_e2          : ptr $
     }

;------------------------------------------------------------------------------
;  get num image rec
;------------------------------------------------------------------------------

inst = 'LENA'
expr = 'LENASCI'
vinst = 'IMLIMAGE'
nrec = 0L
data_source = ' '

data_during_request = 1

start_tme = l_setup.l_start_dtime
stop_tme = l_setup.l_stop_dtime
orig_start_tme = l_setup.l_start_dtime
orig_stop_tme = l_setup.l_stop_dtime

AddExtraTime2Start, start_tme, 6
AddExtraTime2Stop, stop_tme, 8

l_setup.l_start_dtime = start_tme
l_setup.l_stop_dtime = stop_tme

nrec = l_UdfGetNumRec(l_setup.l_start_dtime,  l_setup.l_stop_dtime,  $
                      inst, expr, vinst, data_source)
;print, 'l_BrowseProc: nrec: ', nrec

;------------------------------------------------------------------------------
; there is no data for the requested time period, so set up data to create a 
; single plot with null data
;------------------------------------------------------------------------------

if (nrec le 1) then begin
   data_during_request = -1
   l_setup.l_start_dtime = orig_start_tme
   l_setup.l_stop_dtime = orig_stop_tme
   mass0_bd  = fltarr (l_setup.l_num_polarzn, l_setup.l_num_spinsec)
   mass1_bd  = fltarr (l_setup.l_num_polarzn, l_setup.l_num_spinsec)
   for p = 0, 11 do begin
      for sec = 0, 44 do begin
         mass0_bd[p,sec] = -1
         mass1_bd[p,sec] = -1
      endfor
   endfor
   l_PlotNoDataPlot, l_setup, l_pltpos, l_rgb_table, $
                     orig_start_tme, orig_stop_tme, $
                     mass0_bd, mass1_bd, nadir
   return
endif

;------------------------------------------------------------------------------
; if there is data then get it
;------------------------------------------------------------------------------

nspins = nrec / l_setup.l_num_spinsec
l_setup.l_num_spin = long(nspins)

;------------------------------------------------------------------------------
; read the udf data
;------------------------------------------------------------------------------

start_time = l_setup.l_start_dtime
stop_time = l_setup.l_stop_dtime
start_sec = (start_time(2) * 3600) + (start_time(3) * 60) + start_time(4)
stop_sec = (stop_time(2) * 3600) + (stop_time(3) * 60) + stop_time(4)
if (stop_sec le start_sec) then stop_sec = stop_sec + 86400
del_sec = stop_sec - start_sec
est_num_spins = ceil(del_sec / 120+1)
;print, 'est_num_spins: ', est_num_spins

status = l_BinnedReadUdfDataByTime (l_setup, est_num_spins, bd, $
                                    start_time, stop_time, data_source)

l_setup.l_start_dtime = orig_start_tme
l_setup.l_stop_dtime = orig_stop_tme
nspins_read = n_elements(*bd.startUdfTimePtr)
;print, 'nspins_read: ', nspins_read
something = n_elements(*bd.timePtr)
;print, 'bd.timePtr: ', something

;------------------------------------------------------------------------------
; see if there is data during the requested time
;------------------------------------------------------------------------------

start = 0
if (nspins_read gt 0) then stop = (nspins_read * 44) + (nspins_read - 1)
;print, 'start/stop: ', start, stop

;----- get the requested start and stop time -----
;----- not needed for any reason other than for print statements, may be removed -----

;req_year_start  = 0L
;req_doy_start   = 0L
;req_hour_start  = 0L
;req_min_start   = 0L
;req_sec_start   = 0L
;req_secod_start = 0L
;req_year_stop   = 0L
;req_doy_stop    = 0L
;req_hour_stop   = 0L
;req_min_stop    = 0L
;req_sec_stop    = 0L
;req_secod_stop  = 0L
;reads, l_setup.l_start_dtime, req_year_start, req_doy_start, req_hour_start, req_min_start, $
;       req_sec_start
;req_secod_start = (req_hour_start * 3600) + (req_min_start * 60) + req_sec_start
;reads, l_setup.l_stop_dtime, req_year_stop, req_doy_stop, req_hour_stop, req_min_stop, req_sec_stop
;req_secod_stop = (req_hour_stop * 3600) + (req_min_stop * 60) + req_sec_stop

;print, 'rqst: ', req_year_start, req_doy_start, req_hour_start, req_min_start, req_sec_start
;print, 'rqst: ', req_year_stop, req_doy_stop, req_hour_stop, req_min_stop, req_sec_stop

;----- get the data returned start and stop time -----
;----- not needed for any reason other than for print statements, may be removed -----

;temp_time = (*bd.timePtr)[start]
;data_doy_start  = long(temp_time/86400.)
;data_secod_start = long(temp_time - (86400. * data_doy_start))
;data_hour_start = long((temp_time - (86400. * data_doy_start))/3600.)
;data_min_start  = long((temp_time - (86400. * data_doy_start) - (3600. * data_hour_start))/60.)
;data_sec_start  = long(temp_time mod 60.)
;temp_time = (*bd.timePtr)[stop]
;data_doy_stop  = long(temp_time/86400.)
;data_secod_stop = long(temp_time - (86400. * data_doy_stop))
;data_hour_stop = long((temp_time - (86400. * data_doy_stop))/3600.)
;data_min_stop  = long((temp_time - (86400. * data_doy_stop) - (3600. * data_hour_stop))/60.)
;data_sec_stop  = long(temp_time mod 60.)

;print, 'data: ', data_doy_start, data_hour_start, data_min_start, data_sec_start
;print, 'data: ', data_doy_stop, data_hour_stop, data_min_stop, data_sec_stop

;----- get only the spins that are within the requested start/stop times -----

nrec_min = nrec < n_elements((*bd.timePtr))
req_spin_start_rec = lonarr(est_num_spins+1)
req_spin_stop_rec = lonarr(est_num_spins+1)
req_spins = lonarr(est_num_spins+1)
req_spin_start_rec = lonarr(est_num_spins+1)
req_spin_stop_rec = lonarr(est_num_spins+1)
req_spins = lonarr(est_num_spins+1)
req_record_start = 0L
req_record_stop = 0L
l_GetRequestedSpins, l_setup.l_start_dtime, l_setup.l_stop_dtime, $
                     (*bd.timePtr)[*], nrec_min, req_record_start, req_record_stop, $
                     req_actual_num_spins, req_spins, $
                     req_spin_start_rec, req_spin_stop_rec
;print, req_actual_num_spins, req_record_start, req_record_stop
;for isp = 0, req_actual_num_spins-1 do begin
;   print, req_spins(isp), req_spin_start_rec(isp), req_spin_stop_rec(isp)
;endfor

;----- check to see any spins found -----

if (req_record_stop le req_record_start) then begin
   data_during_request = -1
   l_setup.l_start_dtime = orig_start_tme
   l_setup.l_stop_dtime = orig_stop_tme
   mass0_bd  = fltarr (l_setup.l_num_polarzn, l_setup.l_num_spinsec)
   mass1_bd  = fltarr (l_setup.l_num_polarzn, l_setup.l_num_spinsec)
   for p = 0, 11 do begin
      for sec = 0, 44 do begin
         mass0_bd[p,sec] = -1
         mass1_bd[p,sec] = -1
      endfor
   endfor
   l_PlotNoDataPlot, l_setup, l_pltpos, l_rgb_table, $
                     orig_start_tme, orig_stop_tme, $
                     mass0_bd, mass1_bd, nadir
   return
endif

;------------------------------------------------------------------------------
;------------------------------------------------------------------------------

l_BinnedConv2CountRate, l_setup, l_nplot_setup, rec, bd

;------------------------------------------------------------------------------
; sum binned data over energy, mass, and polar zone
; to obtain the sum value for each spin sector
; these are the coincidences  
;------------------------------------------------------------------------------

m0cnt     = (*bd.m0CntPtr)
m1cnt     = (*bd.m1CntPtr)

;------------------------------------------------------------------------------
;  generating plots on a spin basis
;------------------------------------------------------------------------------

if (l_setup.l_singles_sum eq 1) then begin

   mass0_bd  = fltarr (l_setup.l_num_polarzn, $
                       l_setup.l_num_spinsec, req_actual_num_spins)
   mass1_bd  = fltarr (l_setup.l_num_polarzn, $
                       l_setup.l_num_spinsec, req_actual_num_spins)

;----- collapse out energy -----

   for sp = 0, req_actual_num_spins-1 do begin
     isp = req_spins[sp]
     for sec = 0, 44 do begin
       for p = 0, 11 do begin
         for e = 0, 2 do begin
            mass0_bd[p,sec,sp] = mass0_bd[p,sec,sp] + m0cnt[e,p,sec,isp]
            mass1_bd[p,sec,sp] = mass1_bd[p,sec,sp] + m1cnt[e,p,sec,isp]
         endfor
       endfor
     endfor
   endfor

   if (l_setup.l_image_correction eq 'Y') then begin
      l_CorrectImages_PolSecSpn, mass0_bd, req_actual_num_spins
      l_CorrectImages_PolSecSpn, mass1_bd, req_actual_num_spins
   endif

endif else begin

     ;----------------------------------------------------
     ; plots are all spins summed over spins and energy to
     ; get one 45x12 image
     ;----------------------------------------------------
   nmass0_bd  = fltarr (l_setup.l_num_energylev, $
                        l_setup.l_num_polarzn, $
                        l_setup.l_num_spinsec)
   nmass1_bd  = fltarr (l_setup.l_num_energylev, $
                        l_setup.l_num_polarzn, $
                        l_setup.l_num_spinsec)
   mass0_bd  = fltarr (l_setup.l_num_polarzn, l_setup.l_num_spinsec)
   mass1_bd  = fltarr (l_setup.l_num_polarzn, l_setup.l_num_spinsec)

;----- collapse out spin -----

   for sp = 0, req_actual_num_spins-1 do begin
      isp = req_spins[sp]
      for sec = 0, 44 do begin
         for p = 0, 11 do begin
            for e = 0, 2 do begin
               nmass0_bd[e,p,sec] = nmass0_bd[e,p,sec] + m0cnt[e,p,sec,isp]
               nmass1_bd[e,p,sec] = nmass1_bd[e,p,sec] + m1cnt[e,p,sec,isp]
            endfor
         endfor
      endfor
   endfor
       
;----- collapse out energy -----

   for e = 0, 2 do begin
      for p = 0, 11 do begin
         for sec = 0, 44 do begin
            mass0_bd[p,sec] = mass0_bd[p,sec] + nmass0_bd[e,p,sec]
            mass1_bd[p,sec] = mass1_bd[p,sec] + nmass1_bd[e,p,sec]
         endfor
      endfor
   endfor

   if (l_setup.l_image_correction eq 'Y') then begin
      l_CorrectImages_PolSec, mass0_bd
      l_CorrectImages_PolSec, mass1_bd
   endif

endelse

RETURN
END

;============================================================
; FUNCTION: l_BrowsePlot
; Description:  Plot the coincidence data
;------------------------------------------------------------------------------
; Richard L. West      2001/02/08
; -----
; Modified so that when 1 spin time period is selected the same OA information
; is used if l_singles_sum is set to either 1 or 2
;============================================================
   
PRO l_BrowsePlot, nrec, btme, mass0_bd, mass1_bd, l_oa, l_setup, $
                  l_pltpos, l_rgb_table, req_record_start, req_record_stop, $
                  req_actual_num_spins, req_spins, req_spin_start_rec, $
                  req_spin_stop_rec

;
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;

common rlw_plot_parm, char_size, char_size_1, $
                      window_width, window_height, $
                      x_inches, y_inches

common chan_calib, cross_chan_calib

;
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;

cross_chan_calib = fltarr(12)
cross_chan_calib(0)  = 0.00
cross_chan_calib(1)  = 0.81
cross_chan_calib(2)  = 0.88
cross_chan_calib(3)  = 0.94
cross_chan_calib(4)  = 0.98
cross_chan_calib(5)  = 1.00
cross_chan_calib(6)  = 1.00
cross_chan_calib(7)  = 0.98
cross_chan_calib(8)  = 0.94
cross_chan_calib(9)  = 0.88
cross_chan_calib(10) = 0.81
cross_chan_calib(11) = 0.00

;
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;

nadir = 1         ; 1 = earthward view, -1 = anti-earthward view
spin_mode = -1    ; 1 = cort wheel,     -1 = opposite cart wheel
re = 6371.0       ; earth radius in km

;
;------------------------------------------------------------------------------
; plotting spin by spin
;------------------------------------------------------------------------------
;

if (l_setup.l_singles_sum eq 1) then begin ; process a single spin at a time

   l_PlotSpinBySpin, btme, l_setup, l_oa, re, spin_mode, mass0_bd, mass1_bd, $
                     l_pltpos, l_rgb_table, nadir, req_record_start, $
                     req_record_stop, req_actual_num_spins, req_spins, $
                     req_spin_start_rec, req_spin_stop_rec

;
;-----------------------------------------------------------------------------
; plotting the sum of the spins
;-----------------------------------------------------------------------------
;

endif else begin  ; if (l_setup.l_singles_sum eq 2) then begin ; sum over spins

   l_PlotSumOfSpins, l_setup, nrec, btme, req_record_start, req_record_stop, $
                     l_oa, re, spin_mode, mass0_bd, mass1_bd, l_pltpos, $
                     l_rgb_table, nadir

endelse

;
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;

RETURN
END

;==============================================================================
;==============================================================================

PRO l_PlotSpinBySpin, btme, l_setup, l_oa, re, spin_mode, mass0_bd, mass1_bd, $
                      l_pltpos, l_rgb_table, nadir, req_record_start, $
                      req_record_stop, req_actual_num_spins, req_spins, $
                      req_spin_start_rec, req_spin_stop_rec

;
;------------------------------------------------------------------------------
; this procedure plot the data when the option is plot each individual spin over
; specified time period
;------------------------------------------------------------------------------
;

common rlw_plot_parm, char_size, char_size_1, $
                      window_width, window_height, $
                      x_inches, y_inches

common chan_calib, cross_chan_calib
common oa_coord, gci_satellite_position, gci_satellite_velocity, $
                 gci_sat_spin, geo_satellite_position, $
                 gsm_satellite_position, gsm_satellite_velocity, $
                 sm_satellite_position, sm_sat_pos_curr, sm_satellite_velocity, $
                 sm_satellite_spin, geo_n_sm, sat_re, $
                 l_shell, mlt, mlat, invlat
common sm_coord, rs, rs2, e_rad, sm_im_x, sm_im_y, sm_im_z
common earth, npt, earth_x, earth_y
common re_circles, re_circles_x, re_circles_y, x_connect, y_connect, $
                   circle_pts, lt_lab
common dipole, dipole_x, dipole_y, dipole_pts
common auroral, auroral_x, auroral_y, auroral_pts

;
;
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;

npolar = 12

gci_satellite_position = fltarr(3)
gci_satellite_velocity = fltarr(3)
geo_satellite_position = fltarr(3)
geo_n_sm = fltarr(3)
gsm_satellite_position = fltarr(3)
gsm_satellite_velocity = fltarr(3)
sm_sat_pos_curr = fltarr(3)
sm_satellite_position = fltarr(3)
sm_satellite_velocity = fltarr(3)
sm_satellite_spin = fltarr(3)
gci_sat_spin = fltarr(3)

doy = 0L
hr = 0L
min = 0L
sec = 0L
req_start_secod = 0L
nodata_start_time = l_setup.l_start_dtime
nodata_stop_time = l_setup.l_start_dtime

;
;------------------------------------------------------------------------------
; see if we are missing any spins at the beginning, if so create no data plots
;------------------------------------------------------------------------------
;

first_time_seconds = btme[req_record_start] 
first_time_doy  = long(first_time_seconds/86400.)
first_time_secod = long(first_time_seconds - (86400. * first_time_doy))
req_start_secod = (long(l_setup.l_start_dtime[2]) * 3600) + $
                  (long(l_setup.l_start_dtime[3]) * 60) + $
                  long(l_setup.l_start_dtime[4])
diff_time = first_time_secod - req_start_secod

if (diff_time gt 0) then begin
   num_miss_spins = (diff_time / 120) + 1

   nodata_start_seconds = first_time_seconds - (124 * num_miss_spins)
   l_ConvSec2DOYTime, nodata_start_seconds, doy, hr, min, sec
   nodata_start_time[1] = doy
   nodata_start_time[2] = hr
   nodata_start_time[3] = min
   nodata_start_time[4] = sec
   nodata_start_time[5] = 0

   nodata_stop_seconds = nodata_start_seconds + 122
   l_ConvSec2DOYTime, nodata_stop_seconds, doy, hr, min, sec
   nodata_stop_time[1] = doy
   nodata_stop_time[2] = hr
   nodata_stop_time[3] = min
   nodata_stop_time[4] = sec
   nodata_stop_time[5] = 0

   for nspin = 0, num_miss_spins-1 do begin

      e_rad = 16.0
      l_PlotNoDataPlot, l_setup, l_pltpos, l_rgb_table, $
                        nodata_start_time, nodata_stop_time, $
                        mass0_bd, mass1_bd, nadir

      nodata_start_seconds = nodata_stop_seconds + 2
      l_ConvSec2DOYTime, nodata_start_seconds, doy, hr, min, sec
      nodata_start_time[1] = doy
      nodata_start_time[2] = hr
      nodata_start_time[3] = min
      nodata_start_time[4] = sec
      nodata_start_time[5] = 0

      nodata_stop_seconds = nodata_start_seconds + 122
      l_ConvSec2DOYTime, nodata_stop_seconds, doy, hr, min, sec
      nodata_stop_time[1] = doy
      nodata_stop_time[2] = hr
      nodata_stop_time[3] = min
      nodata_stop_time[4] = sec
      nodata_stop_time[5] = 0

   endfor
endif

;
;------------------------------------------------------------------------------
; loop thru each spin
;------------------------------------------------------------------------------
;

previous_stop_rec = req_spin_start_rec(0)

for nspins = 0, req_actual_num_spins-1 do begin

   start_rec = req_spin_start_rec(nspins)
   stop_rec = req_spin_stop_rec(nspins)

;
;------------------------------------------------------------------------------
; see if we are missing any spins, and if we are create the appropriate
; number of no data spins
;------------------------------------------------------------------------------
;

   previous_time = btme[previous_stop_rec] 
   previous_stop_doy  = long(previous_time/86400.)
   previous_stop_secod = long(previous_time - (86400. * previous_stop_doy))

   temp_time = btme[start_rec] 
   start_doy  = long(temp_time/86400.)
   start_secod = long(temp_time - (86400. * start_doy))

   time_dif = start_secod - previous_stop_secod

   if (time_dif ge 120) then begin

;
;------------------------------------------------------------------------------
;     we have missing spins, see how many and make no data plots for them
;------------------------------------------------------------------------------
;

      num_nodata_spins = time_dif / 120
      previous_time = previous_time + 2
      l_ConvSec2DOYTime, previous_time, doy, hr, min, sec
      nodata_start_time[1] = doy
      nodata_start_time[2] = hr
      nodata_start_time[3] = min
      nodata_start_time[4] = sec
      nodata_start_time[5] = previous_stop_secod

      nodata_stop_seconds = previous_time + 122
      nodata_stop_doy  = long(nodata_stop_seconds/86400.)
      nodata_stop_secod = long(nodata_stop_seconds - (86400. * nodata_stop_doy))
      l_ConvSec2DOYTime, nodata_stop_seconds, doy, hr, min, sec
      nodata_stop_time[1] = doy
      nodata_stop_time[2] = hr
      nodata_stop_time[3] = min
      nodata_stop_time[4] = sec
      nodata_stop_time[5] = nodata_stop_secod

      for nspns = 0, num_nodata_spins-1 do begin

         e_rad = 16.0
         l_PlotNoDataPlot, l_setup, l_pltpos, l_rgb_table, $
                           nodata_start_time, nodata_stop_time, $
                           mass0_bd, mass1_bd, nadir

         nodata_start_seconds = nodata_stop_seconds + 2
         nodata_start_doy  = long(nodata_start_seconds/86400.)
         nodata_start_secod = long(nodata_start_seconds - (86400. * nodata_start_doy))
         l_ConvSec2DOYTime, nodata_start_seconds, doy, hr, min, sec
         nodata_start_time[1] = doy
         nodata_start_time[2] = hr
         nodata_start_time[3] = min
         nodata_start_time[4] = sec
         nodata_start_time[5] = nodata_start_secod

         nodata_stop_seconds = nodata_start_seconds + 122
         nodata_stop_doy  = long(nodata_stop_seconds/86400.)
         nodata_stop_secod = long(nodata_stop_seconds - (86400. * nodata_stop_doy))
         l_ConvSec2DOYTime, nodata_stop_seconds, doy, hr, min, sec
         nodata_stop_time[1] = doy
         nodata_stop_time[2] = hr
         nodata_stop_time[3] = min
         nodata_stop_time[4] = sec
         nodata_stop_time[5] = nodata_stop_secod

      endfor

   endif    ;  missing data, no data plots
   previous_stop_rec = stop_rec

;
;------------------------------------------------------------------------------
; get the oa information to draw the earth, etc on the plot
;------------------------------------------------------------------------------
;

   oaindx = req_spins(nspins)
   l_GetOAForWorldMap, l_oa, oaindx, l_setup, re, spin_mode

;
;------------------------------------------------------------------------------
; Find the SM components of image frame in which:
;     z-axis is along satellite position
;     y-axis (spin angle) = z-axis x sm_satillite_spin
;     x-axis completes the right-hand rule
; For image looking outward from the Earth, z-axis is pointed from the
; satellite to the Earth.
;------------------------------------------------------------------------------
;

   calculate_sm_coord, nadir

;
;------------------------------------------------------------------------------
; calculate the earth
;------------------------------------------------------------------------------
;

   calculate_earth

;
;------------------------------------------------------------------------------
; calculate circles at 3 and 6.6 Re at the equator
;------------------------------------------------------------------------------
;

   calculate_re_circles, nadir, rs_tst

;
;------------------------------------------------------------------------------
; Calculate and dipole fieldlines at L=3, 6.6, MLT=0, 6, 12, 18
;------------------------------------------------------------------------------
;

   calculate_dipole, rs_tst, nadir

;
;------------------------------------------------------------------------------
; calculate the -75, -65, 65 and 75 deg latitudes on the surface
;------------------------------------------------------------------------------
;

   calculate_auroral, nadir, rs_tst

;
;------------------------------------------------------------------------------
; rearrange and calibrate the image data
; this is hard coded for now, but there is a routine which does this
;------------------------------------------------------------------------------
;

   image0_dmp=fltarr(npolar,45)
   image0_dmp[*,0:10]  = mass0_bd[*,6:16, nspins]
   image0_dmp[*,11]    = mass0_bd[*,17, nspins]
   image0_dmp[*,12:38] = mass0_bd[*,18:44, nspins]
   image0_dmp[*,39:44] = mass0_bd[*,0:5, nspins]
   image1_dmp=fltarr(npolar,45)
   image1_dmp[*,0:10]  = mass1_bd[*,6:16, nspins]
   image1_dmp[*,11]    = mass1_bd[*,17, nspins]
   image1_dmp[*,12:38] = mass1_bd[*,18:44, nspins]
   image1_dmp[*,39:44] = mass1_bd[*,0:5, nspins]

   if (l_setup.l_output_type eq 1) then begin
      naz_bins = 44
      image0=fltarr(npolar,45)
      image0[*,0:10]  = mass0_bd[*,6:16, nspins]
      image0[*,11]    = mass0_bd[*,17, nspins]
      image0[*,12:38] = mass0_bd[*,18:44, nspins]
      image0[*,39:44] = mass0_bd[*,0:5, nspins]
      image1=fltarr(npolar,45)
      image1[*,0:10]  = mass1_bd[*,6:16, nspins]
      image1[*,11]    = mass1_bd[*,17, nspins]
      image1[*,12:38] = mass1_bd[*,18:44, nspins]
      image1[*,39:44] = mass1_bd[*,0:5, nspins]
   endif else if(l_setup.l_output_type eq 2) then begin
      naz_bins = 44
      image0=fltarr(npolar,45)
      image0[*,0:5]  = mass0_bd[*,39:44,nspins]
      image0[*,6:44] = mass0_bd[*,0:38,nspins]
      image1=fltarr(npolar,45)
      image1[*,0:5]  = mass1_bd[*,39:44,nspins]
      image1[*,6:44] = mass1_bd[*,0:38,nspins]
   endif else if(l_setup.l_output_type eq 3) then begin
      naz_bins = 11
      image0=fltarr(npolar,12)
      image0[*,0:11] = mass0_bd[*,11:22, nspins]
      image1=fltarr(npolar,12)
      image1[*,0:11] = mass1_bd[*,11:22, nspins]
   endif

   ;-----------------------------------------
   ; convert to counts/s -- ELH
   ;-----------------------------------------
   image0 = image0/2.7
   image1 = image1/2.7

   for np = 0,11 do begin
      for na = 0,naz_bins do begin
         if (cross_chan_calib(np) eq 0.0) then begin
            image0(np,na) = -99.0
            image1(np,na) = -99.0
         endif else begin
            image0(np,na) = image0(np,na) / cross_chan_calib(np)
            image1(np,na) = image1(np,na) / cross_chan_calib(np)
         endelse
      endfor
   endfor
 
   image2 = image0 + image1 ; total = H + O

;
;------------------------------------------------------------------------------
; build the plotting file name
;------------------------------------------------------------------------------
;

   plot_title_start = ConvToUtc(btme[start_rec], 2)
   plot_title_stop = ConvToUtc(btme[stop_rec], 2)
   l_ConvSec2DOYTime, btme[start_rec], start_doy, start_hr, start_min, start_sec
   l_ConvSec2DOYTime, btme[stop_rec], stop_doy, stop_hr, stop_min, stop_sec
   start_year = l_setup.l_start_date[0]
   start_year_2 = start_year - (long(start_year / 100) * 100)
   stop_year = l_setup.l_stop_date[0]
   if (stop_doy lt start_doy) then start_year = start_year - 1
   l_ConvDOY2YMD, long(start_doy), start_year, start_mon, start_dom
   l_ConvDOY2YMD, long(stop_doy), stop_year, stop_mon, stop_dom

   mo = 0
   dy = 0
   l_filename_base = ''
   l_ConvDOY2YMD, l_setup.l_start_dtime[1], l_setup.l_start_dtime[0], mo, dy
   year = string(strcompress(l_setup.l_start_dtime[0], /REMOVE_ALL))
   yr = strmid(year, 2, 2)
   l_setup.l_filename = strcompress(('l' + $
                        string(format='(A2)', yr) +  $
                        string(format='(I2.2)',start_mon) + $
                        string(format='(I2.2)',start_dom) + $
                        string(format='(I2.2)',start_hr) + $
                        string(format='(I2.2)',start_min) + $
                        string(format='(I2.2)',stop_hr)  + $
                        string(format='(I2.2)',stop_min)), $
                        /REMOVE_ALL)
   l_filename_base = l_setup.l_filename

   if (l_setup.l_output_type eq 1) then begin
      l_setup.l_filedesc = 'image_3full'
      l_setup.l_filename = l_filename_base + '_3fl'
   endif else if (l_setup.l_output_type eq 2) then begin
      l_setup.l_filedesc = 'image_3cnst'
      l_setup.l_filename = l_filename_base + '_3cn'
   endif else begin
      l_setup.l_filedesc = 'image_3sq'
      l_setup.l_filename = l_filename_base + '_3sq'
   endelse

;
;------------------------------------------------------------------------------
; get the minimum and maximin scaling values from the setup file or
; determine minimum and maximum scaling in log of units based on data values
;------------------------------------------------------------------------------
;

   get_scale_minmax, l_setup, image0, image1, image2

;
;------------------------------------------------------------------------------
; load color table
;------------------------------------------------------------------------------
;

;   load_color_table

;
;------------------------------------------------------------------------------
; loop through and do each plot type
;------------------------------------------------------------------------------
;

   nplots = 0
   if (strupcase(l_setup.l_via_web) eq 'N') then $
      do_x_1 = 0 else $
      do_x_1 = 1
   if (l_setup.l_gen_movie eq 'Y') then $
      do_x_2 = 1 else $
      do_x_2 = 2
   for output_plot_type = do_x_1, do_x_2 do begin

;
;------------------------------------------------------------------------------
; set up plot type
;------------------------------------------------------------------------------
;

      if (output_plot_type eq 0) then begin
         l_SetUpDisplay, 'X', l_pltpos, l_rgb_table, $
                         l_setup.l_color_tbl, colorbar, nplots
      endif else if (output_plot_type eq 1) then begin
         l_SetUpDisplay, 'Z', l_pltpos, l_rgb_table, $
                         l_setup.l_color_tbl, colorbar, nplots
      endif else begin
         ps_filename = l_setup.l_prod_dst + '/' + l_setup.l_filename
         ps_filename = ps_filename + '1sp'
         l_pltpos.filename = ps_filename + l_setup.l_process_id + '.ps'
         l_SetUpDisplay, 'P', l_pltpos, l_rgb_table, $
                         l_setup.l_color_tbl, colorbar, nplots
      endelse

;
;------------------------------------------------------------------------------
; plot the 3 images and the global overlay and the color bar
;------------------------------------------------------------------------------
;

   if (l_setup.l_gen_movie eq 'Y') then begin
      plot_ImageMovie, image0, image1, image2, npolar, polar_angle, l_setup, $
                       nadir
   endif else begin
      plot_ENAflux, image0, image1, image2, l_setup, nadir

   endelse
;
;------------------------------------------------------------------------------
; put on the top left annotation
;------------------------------------------------------------------------------
;

      date_start = strcompress(( $
         string(format='(I4.4)', start_year) + '/' + $
         string(format='(I2.2)', start_mon)  + '/' + $
         string(format='(I2.2)', start_dom)),  /REMOVE_ALL)
      plot_start_time = strmid(plot_title_start, 4, 8)
      date_stop = strcompress(( $
         string(format='(I4.4)', stop_year) + '/' + $
         string(format='(I2.2)', stop_mon)  + '/' + $
         string(format='(I2.2)', stop_dom)),  /REMOVE_ALL)
      plot_stop_time = strmid(plot_title_stop, 4, 8)

      if (l_setup.l_gen_movie eq 'Y') then begin
         if (l_setup.l_output_type eq 1) then begin
            xpos=0.06
            ypos=0.96
         endif else begin
            xpos=0.11
            ypos=0.90
         endelse
         label = date_start + ' ' + plot_start_time + ' -> ' + plot_stop_time
         xyouts, xpos, ypos, /normal, label, charsize=char_size_1, align = 0.0
      endif else begin
         xyouts, 0.0, 0.0, '!5', /normal
         xpos=0.10
         ypos=0.97
         ydelta=0.022
         label = 'IMAGE LENA'
         xyouts, xpos, ypos, /normal, label, charsize=char_size_1, align = 0.0
         label = 'start time: ' + date_start + ' ' + plot_start_time
         xyouts, xpos, ypos-ydelta, /normal, label, $
                 charsize=char_size_1, align = 0.0
         label = 'stop time: ' + date_stop + ' ' + plot_stop_time
         xyouts, xpos, ypos-2*ydelta, /normal, label, $
                 charsize=char_size_1, align = 0.0
         xyouts, xpos, ypos-3*ydelta, /normal, '1 spin', $
                 charsize=char_size_1, align = 0.0
      endelse

;
;------------------------------------------------------------------------------
; put on the orbital annotation
;------------------------------------------------------------------------------
;

      if (l_setup.l_gen_movie eq 'Y') then begin
      endif else begin

         xpos = 0.50
         ypos = 0.97

         label = string (format = '(''Re = '', f7.4)', sat_re)
         xyouts, xpos, ypos, /normal, label, charsize=char_size_1, align = 0.0
         label = string (format = '(''Lshell = '', f7.3)', l_shell)
         xyouts, xpos, ypos-ydelta, /normal, label, charsize=char_size_1, $
                 align = 0.0
         label = string (format = '(''mlt = '', f4.1,''hr'')', mlt)
         xyouts, xpos, ypos-2*ydelta, /normal, label, charsize=char_size_1, $
                 align = 0.0
         label = string (format = '(''mlat = '', f7.3,''!m%!5'')', mlat)
         xyouts, xpos, ypos-3*ydelta, /normal, label, charsize=char_size_1, $
                 align = 0.0
         label = string (format = '(''invlat = '', f6.3,''!m%!5'')', invlat)
         xyouts, xpos, ypos-4*ydelta, /normal, label, charsize=char_size_1, $
                 align = 0.0
      endelse

;
;------------------------------------------------------------------------------
; put on version number and date
;------------------------------------------------------------------------------
;

      if (l_setup.l_gen_movie eq 'Y') then begin
      endif else begin
         xpos = 0.80
         ypos = 0.10

         label = 'Version 3.0.0'
         xyouts, xpos, ypos, /normal, label, charsize=char_size, align = 0.0

         time_string = systime(0)
         time = strmid(time_string, 11, 8)
         dom = strmid(time_string, 8, 2)
         mon = strmid(time_string, 4, 3)
         year = strmid(time_string, 20, 4)
         label = mon + '-' + dom + '-' + year
         xyouts, xpos, ypos-ydelta, /normal, label, charsize=char_size, $
                 align=0.0
         label = time
         xyouts, xpos, ypos-(2.0*ydelta), /normal, label, charsize=char_size, $
                 align=0.0
      endelse

;
;------------------------------------------------------------------------------
; write the gif file or close the postscript file
;------------------------------------------------------------------------------
;

      if (output_plot_type eq 1) then begin
         plot_image = TVRD()
         base_filename = l_setup.l_prod_dst + '/' + l_setup.l_filename
         base_filename = base_filename + '1sp'
         base_filename = base_filename + l_setup.l_process_id
         if (l_setup.l_output_plot_format eq 'PNG') then begin
            plot_image = reverse(plot_image, 2)
            png_filename = base_filename + '.png'
            write_png, png_filename, plot_image, $
                       l_rgb_table.red(0:225), l_rgb_table.green(0:225), $
                       l_rgb_table.blue(0:225)
         endif else begin
            gif_filename = base_filename + '.gif'
            write_gif, gif_filename, plot_image, $
                       l_rgb_table.red(0:225), l_rgb_table.green(0:225), $
                       l_rgb_table.blue(0:225)
         endelse
      endif else if (output_plot_type eq 2) then begin
         device, /close_file
      endif


;
;------------------------------------------------------------------------------
;  end of plotting loop
;------------------------------------------------------------------------------
;

   endfor     ; for output_plot_type = do_x_1, do_x_2 do begin

;
;------------------------------------------------------------------------------
; output to the dump file
;------------------------------------------------------------------------------
;

   if (l_setup.l_dumps eq 'Y') then begin
      num_spins = 1
      dump_filename = l_setup.l_prod_dst + '/' + l_filename_base + $
                      l_setup.l_process_id + '.image.dmp'
      openw, fp1, dump_filename, /get_lun
      l_BrowsePrintDump, fp1, dump_filename, l_pltpos, num_spins, $
                         date_start, start_doy, plot_start_time, $
                         date_stop, stop_doy, plot_stop_time, $
                         sat_re, sat_re, l_shell, l_shell, $
                         mlt, mlt, mlat, mlat, invlat, invlat, $
                         image0_dmp, image1_dmp
      close, fp1
   endif

;
;------------------------------------------------------------------------------
; end of spin loop
;------------------------------------------------------------------------------
;

endfor     ; for nspins = 0, num_of_spins-1 then begin

;
;------------------------------------------------------------------------------
; see if we are missing any spins at the end, if so create no data plots
;------------------------------------------------------------------------------
;

last_time_seconds = btme[req_record_stop] 
last_time_doy  = long(last_time_seconds/86400.)
last_time_secod = long(last_time_seconds - (86400. * first_time_doy))
req_stop_secod = (long(l_setup.l_stop_dtime[2]) * 3600) + $
                  (long(l_setup.l_stop_dtime[3]) * 60) + $
                  long(l_setup.l_stop_dtime[4])
diff_time = req_stop_secod - last_time_secod

if (diff_time gt 0) then begin
   num_miss_spins = (diff_time / 120) + 1

   nodata_start_seconds = last_time_seconds + 2
   l_ConvSec2DOYTime, nodata_start_seconds, doy, hr, min, sec
   nodata_start_time[1] = doy
   nodata_start_time[2] = hr
   nodata_start_time[3] = min
   nodata_start_time[4] = sec
   nodata_start_time[5] = 0

   nodata_stop_seconds = nodata_start_seconds + 122
   l_ConvSec2DOYTime, nodata_stop_seconds, doy, hr, min, sec
   nodata_stop_time[1] = doy
   nodata_stop_time[2] = hr
   nodata_stop_time[3] = min
   nodata_stop_time[4] = sec
   nodata_stop_time[5] = 0

   for nspin = 0, num_miss_spins-1 do begin

      e_rad = 16.0
      l_PlotNoDataPlot, l_setup, l_pltpos, l_rgb_table, $
                        nodata_start_time, nodata_stop_time, $
                        mass0_bd, mass1_bd, nadir

      nodata_start_seconds = nodata_stop_seconds + 2
      l_ConvSec2DOYTime, nodata_start_seconds, doy, hr, min, sec
      nodata_start_time[1] = doy
      nodata_start_time[2] = hr
      nodata_start_time[3] = min
      nodata_start_time[4] = sec
      nodata_start_time[5] = 0

      nodata_stop_seconds = nodata_start_seconds + 122
      l_ConvSec2DOYTime, nodata_stop_seconds, doy, hr, min, sec
      nodata_stop_time[1] = doy
      nodata_stop_time[2] = hr
      nodata_stop_time[3] = min
      nodata_stop_time[4] = sec
      nodata_stop_time[5] = 0

   endfor

endif

;
;------------------------------------------------------------------------------
; end of l_PlotSpinBySpin
;------------------------------------------------------------------------------
;

RETURN
END

;==============================================================================
;==============================================================================

PRO l_PlotSumOfSpins, l_setup, nrec, btme, req_record_start, req_record_stop, $
                      l_oa, re, spin_mode, mass0_bd, mass1_bd, l_pltpos, $
                      l_rgb_table, nadir

;------------------------------------------------------------------------------
;
; this procedure plot the data when the option is plot the sum of the spins over
; specified time period
;
;------------------------------------------------------------------------------

common rlw_plot_parm, char_size, char_size_1, $
                      window_width, window_height, $
                      x_inches, y_inches

common chan_calib, cross_chan_calib
common oa_coord, gci_satellite_position, gci_satellite_velocity, $
                 gci_sat_spin, geo_satellite_position, $
                 gsm_satellite_position, gsm_satellite_velocity, $
                 sm_satellite_position, sm_sat_pos_curr, sm_satellite_velocity, $
                 sm_satellite_spin, geo_n_sm, sat_re, $
                 l_shell, mlt, mlat, invlat
common sm_coord, rs, rs2, e_rad, sm_im_x, sm_im_y, sm_im_z
common earth, npt, earth_x, earth_y
common re_circles, re_circles_x, re_circles_y, x_connect, y_connect, $
                   circle_pts, lt_lab
common dipole, dipole_x, dipole_y, dipole_pts
common auroral, auroral_x, auroral_y, auroral_pts

;
;
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;

nspins = nrec / l_setup.l_num_spinsec
npolar = 12

gci_satellite_position = fltarr(3)
gci_satellite_velocity = fltarr(3)
geo_satellite_position = fltarr(3)
geo_n_sm = fltarr(3)
gsm_satellite_position = fltarr(3)
gsm_satellite_velocity = fltarr(3)
sm_sat_pos_curr = fltarr(3)
sm_satellite_position = fltarr(3)
sm_satellite_velocity = fltarr(3)
gci_sat_spin = fltarr(3)
sm_satellite_spin = fltarr(3)

;
;------------------------------------------------------------------------------
;  get the oa information to draw the earth, etc on the plot
;  since this is a sum over a time period use approximate middle spin
;------------------------------------------------------------------------------
;

if (nspins lt 2) then begin
   oaindx = 0
endif else begin ; get the middle spin if multiple spins
   oaindx = ceil(n_elements((*l_oa.gsm_x_pos_ptr)[*])/2)
endelse

l_GetOAForWorldMap, l_oa, oaindx, l_setup, re, spin_mode

;
;------------------------------------------------------------------------------
; Find the SM components of image frame in which:
;     z-axis is along satellite position
;     y-axis (spin angle) = z-axis x sm_satillite_spin
;     x-axis completes the right-hand rule
; For image looking outward from the Earth, z-axis is pointed from the
; satellite to the Earth.
;------------------------------------------------------------------------------
;

   calculate_sm_coord, nadir

;
;------------------------------------------------------------------------------
; calculate the earth
;------------------------------------------------------------------------------
;

   calculate_earth

;
;------------------------------------------------------------------------------
; calculate circles at 3 and 6.6 Re at the equator
;------------------------------------------------------------------------------
;

   calculate_re_circles, nadir, rs_tst

;
;------------------------------------------------------------------------------
; Calculate and dipole fieldlines at L=3, 6.6, MLT=0, 6, 12, 18
;------------------------------------------------------------------------------
;

   calculate_dipole, rs_tst, nadir

;
;------------------------------------------------------------------------------
; calculate the -75, -65, 65 and 75 deg latitudes on the surface
;------------------------------------------------------------------------------
;

   calculate_auroral, nadir, rs_tst
;
;------------------------------------------------------------------------------
; rearrange and calibrate the image data
; this is hard coded for now, but there is a routine which does this
;------------------------------------------------------------------------------
;

image0_dmp=fltarr(npolar,45)
image0_dmp[*,0:10]  = mass0_bd[*,6:16]
image0_dmp[*,11]    = mass0_bd[*,17]
image0_dmp[*,12:38] = mass0_bd[*,18:44]
image0_dmp[*,39:44] = mass0_bd[*,0:5]
image1_dmp=fltarr(npolar,45)
image1_dmp[*,0:10]  = mass1_bd[*,6:16]
image1_dmp[*,11]    = mass1_bd[*,17]
image1_dmp[*,12:38] = mass1_bd[*,18:44]
image1_dmp[*,39:44] = mass1_bd[*,0:5]
if (l_setup.l_output_type eq 1) then begin
   naz_bins = 44
   image0=fltarr(npolar,45)
   image0[*,0:10]  = mass0_bd[*,6:16]
   image0[*,11]    = mass0_bd[*,17]
   image0[*,12:38] = mass0_bd[*,18:44]
   image0[*,39:44] = mass0_bd[*,0:5]
   image1=fltarr(npolar,45)
   image1[*,0:10]  = mass1_bd[*,6:16]
   image1[*,11]    = mass1_bd[*,17]
   image1[*,12:38] = mass1_bd[*,18:44]
   image1[*,39:44] = mass1_bd[*,0:5]
endif else if(l_setup.l_output_type eq 2) then begin
   naz_bins = 44
   image0=fltarr(npolar,45)
   image0[*,0:5]  = mass0_bd[*,39:44]
   image0[*,6:44] = mass0_bd[*,0:38]
   image1=fltarr(npolar,45)
   image1[*,0:5]  = mass1_bd[*,39:44]
   image1[*,6:44] = mass1_bd[*,0:38]
endif else if (l_setup.l_output_type eq 3) then begin
   naz_bins = 11
   image0=fltarr(npolar,12)
   image0[*,0:11] = mass0_bd[*,11:22]
   image1=fltarr(npolar,12)
   image1[*,0:11] = mass1_bd[*,11:22]
endif
if (l_setup.l_avg_sum eq 1) then begin
   image0 = image0 / (2.7 * float(nspins))
   image1 = image1 / (2.7 * float(nspins))
endif else begin
   ;------------------------------------------------
   ; convert to counts/s -- ELH
   ;------------------------------------------------
   image0 = image0 / 2.7
   image1 = image1 / 2.7
endelse
for np = 0,11 do begin
   for na = 0,naz_bins do begin
      if (cross_chan_calib(np) eq 0.0) then begin
         image0(np,na) = -99.0
         image1(np,na) = -99.0
      endif else begin
         image0(np,na) = image0(np,na) / cross_chan_calib(np)
         image1(np,na) = image1(np,na) / cross_chan_calib(np)
      endelse
   endfor
endfor

image2 = image0 + image1 ; total = H + O

;
;------------------------------------------------------------------------------
; build the plotting file name
;------------------------------------------------------------------------------
;

plot_title_start = ConvToUtc(btme[req_record_start], 2)
plot_title_stop = ConvToUtc(btme[req_record_stop], 2)
l_ConvSec2DOYTime, btme[req_record_start], start_doy, hr, min, sec
l_ConvSec2DOYTime, btme[req_record_stop], stop_doy, hr, min, sec
start_year = l_setup.l_start_date[0]
start_year_2 = start_year - (long(start_year / 100) * 100)
stop_year = l_setup.l_stop_date[0]
if (stop_doy lt start_doy) then start_year = start_year - 1
l_ConvDOY2YMD, long(start_doy), start_year, start_mon, start_dom
l_ConvDOY2YMD, long(stop_doy), stop_year, stop_mon, stop_dom

mo = 0
dy = 0
l_filename_base = ''
l_ConvDOY2YMD, l_setup.l_start_dtime[1], l_setup.l_start_dtime[0], mo, dy
year = string(strcompress(l_setup.l_start_dtime[0], /REMOVE_ALL))
yr = strmid(year, 2, 2)
l_setup.l_filename = strcompress(('l' + $
                     string(format='(A2)', yr) +  $
                     string(format='(I2.2)',mo) + $
                     string(format='(I2.2)',dy) + $
                     string(format='(I2.2)',l_setup.l_start_dtime[2]) + $
                     string(format='(I2.2)',l_setup.l_start_dtime[3]) + $
                     string(format='(I2.2)',l_setup.l_stop_dtime[2])  + $
                     string(format='(I2.2)',l_setup.l_stop_dtime[3])), $
                     /REMOVE_ALL)
l_filename_base = l_setup.l_filename

if (l_setup.l_output_type eq 1) then begin
   l_setup.l_filedesc = 'image_3full'
   l_setup.l_filename = l_filename_base + '_3fl'
endif else if (l_setup.l_output_type eq 2) then begin
   l_setup.l_filedesc = 'image_3cnst'
   l_setup.l_filename = l_filename_base + '_3cn'
endif else begin
   l_setup.l_filedesc = 'image_3sq'
   l_setup.l_filename = l_filename_base + '_3sq'
endelse

nrec1 = (req_record_stop - req_record_start) + 1
nspins = ceil(float(nrec1) / float(l_setup.l_num_spinsec))

if (l_setup.l_avg_sum eq 1) then begin
   spinavgsum = string (format = '(i2.2,''spa'')', nspins)
endif else begin
   spinavgsum = string (format = '(i2.2,''sps'')', nspins)
endelse

;
;------------------------------------------------------------------------------
; get the minimum and maximin scaling values from the setup file or
; determine minimum and maximum scaling in log of units based on data values
;------------------------------------------------------------------------------
;

   get_scale_minmax, l_setup, image0, image1, image2

;
;------------------------------------------------------------------------------
; load color table
;------------------------------------------------------------------------------
;

;   load_color_table

;
;------------------------------------------------------------------------------
; loop through and do each plot type
;------------------------------------------------------------------------------
;

nplots = 0
if (strupcase(l_setup.l_via_web) eq 'N') then $
   do_x_1 = 0 else $
   do_x_1 = 1
if (l_setup.l_gen_movie eq 'Y') then $
   do_x_2 = 1 else $
   do_x_2 = 2

;
;------------------------------------------------------------------------------
; set up plot type
;------------------------------------------------------------------------------
;

for output_plot_type = do_x_1, do_x_2 do begin
   if (output_plot_type eq 0) then begin
      l_SetUpDisplay, 'X', l_pltpos, l_rgb_table, $
                      l_setup.l_color_tbl, colorbar, nplots
   endif else if (output_plot_type eq 1) then begin
      l_SetUpDisplay, 'Z',  l_pltpos, l_rgb_table, $
                      l_setup.l_color_tbl, colorbar, nplots
   endif else begin
      ps_filename = l_setup.l_prod_dst + '/' + l_setup.l_filename
      ps_filename = ps_filename + spinavgsum
      l_pltpos.filename = ps_filename + l_setup.l_process_id  + '.ps'
      l_SetUpDisplay, 'P', l_pltpos, l_rgb_table, $
                      l_setup.l_color_tbl, colorbar, nplots
   endelse
;
;------------------------------------------------------------------------------
; plot the 3 images and the global overlay and the color bar
;------------------------------------------------------------------------------
;

   if (l_setup.l_gen_movie eq 'Y') then begin
      plot_ImageMovie, image0, image1, image2, npolar, polar_angle, l_setup, $
                       nadir
   endif else begin
      plot_ENAflux, image0, image1, image2, l_setup, nadir
   endelse

;
;------------------------------------------------------------------------------
; put on the top left labeling
;------------------------------------------------------------------------------
;

   date_start = strcompress(( $
      string(format='(I4.4)', start_year) + '/' + $
      string(format='(I2.2)', start_mon)  + '/' + $
      string(format='(I2.2)', start_dom)),  /REMOVE_ALL)
   plot_start_time = strmid(plot_title_start, 4, 8)
   date_stop = strcompress(( $
      string(format='(I4.4)', stop_year) + '/' + $
      string(format='(I2.2)', stop_mon)  + '/' + $
      string(format='(I2.2)', stop_dom)),  /REMOVE_ALL)
   plot_stop_time = strmid(plot_title_stop, 4, 8)

   if (l_setup.l_gen_movie eq 'Y') then begin
      if (l_setup.l_output_type eq 1) then begin
         xpos=0.06
         ypos=0.96
      endif else begin
         xpos=0.11
         ypos=0.90
      endelse
      label = date_start + ' ' + plot_start_time + ' -> ' + plot_stop_time
      xyouts, xpos, ypos, /normal, label, charsize=char_size_1, align = 0.0
   endif else begin
      xyouts, 0.0, 0.0, '!5', /normal

      xpos=0.10
      ypos=0.97
      ydelta=0.022

      label = 'IMAGE LENA'
      xyouts, xpos, ypos, /normal, label, charsize=char_size_1, align = 0.0

      ; put on the data plotted start/stop times


      label = 'start time: ' + date_start + ' ' + plot_start_time
      xyouts, xpos, ypos-ydelta, /normal, label, $
              charsize=char_size_1, align = 0.0
      label = 'stop time: ' + date_stop + ' ' + plot_stop_time
      xyouts, xpos, ypos-2*ydelta, /normal, label, $
              charsize=char_size_1, align = 0.0

      ; put on the number of spins

      if (l_setup.l_avg_sum eq 1) then begin
         if (nspins lt 10) then begin
            label = string (format = '(i1,'' spin average'')', nspins)
         endif else begin
            label = string (format = '(i2,'' spin average'')', nspins)
         endelse
      endif else begin
         if (nspins lt 10) then begin
            label = string (format = '(i1,'' spin sum'')', nspins)
         endif else begin
            label = string (format = '(i2,'' spin sum'')', nspins)
         endelse
      endelse
      xyouts, xpos, ypos-3*ydelta, /normal, label, $
              charsize=char_size_1, align = 0.0
   endelse

;
;------------------------------------------------------------------------------
; put on the orbital annotation
;------------------------------------------------------------------------------
;

   if (l_setup.l_gen_movie eq 'Y') then begin
   endif else begin

      oa_start_spin = req_record_start / 45
      oa_stop_spin = req_record_stop / 45

      xgei = ((*l_oa.gci_x_pos_ptr)[oa_start_spin]) / re
      ygei = ((*l_oa.gci_y_pos_ptr)[oa_start_spin]) / re
      zgei = ((*l_oa.gci_z_pos_ptr)[oa_start_spin]) / re

      sat_re_1 = sqrt(xgei*xgei + ygei*ygei + zgei*zgei)
      l_shell_1 = (*l_oa.lsh_ptr)[oa_start_spin]
      invlat_1 = acos(sqrt(1.0/l_shell_1)) * 57.295779

      xgei = ((*l_oa.gci_x_pos_ptr)[oa_stop_spin]) / re
      ygei = ((*l_oa.gci_y_pos_ptr)[oa_stop_spin]) / re
      zgei = ((*l_oa.gci_z_pos_ptr)[oa_stop_spin]) / re

      sat_re_2 = sqrt(xgei*xgei + ygei*ygei + zgei*zgei)
      l_shell_2 = (*l_oa.lsh_ptr)[oa_stop_spin]
      invlat_2 = acos(sqrt(1.0/l_shell_2)) * 57.295779

      time_array = ConvSec2Arr ((*l_oa.tme_ptr)[oa_start_spin], $
                                  l_setup.l_start_dtime[0])

      RECALC, time_array[0], time_array[1], time_array[2], $
              time_array[3], time_array[4]

      GEOGSM, 0.0, 0.0, 1.0 ,xgsm, ygsm, zgsm, 1
      SMGSM, xsm, ysm, zsm, xgsm, ygsm, zgsm, -1

      mlt_1 = ((atan(ysm, xsm) * 12.0) / !pi ) + 12.0
      mlat_1 = (asin(zsm/sat_re_1) * 180.0) / !pi

      time_array = ConvSec2Arr ((*l_oa.tme_ptr)[oa_stop_spin], $
                                  l_setup.l_start_dtime[0])

      RECALC, time_array[0], time_array[1], time_array[2], $
              time_array[3], time_array[4]

      GEOGSM, 0.0, 0.0, 1.0 ,xgsm, ygsm, zgsm, 1
      SMGSM, xsm, ysm, zsm, xgsm, ygsm, zgsm, -1

      mlt_2 = ((atan(ysm, xsm) * 12.0) / !pi ) + 12.0
      mlat_2 = (asin(zsm/sat_re_2) * 180.0) / !pi


      xpos = 0.50
      ypos = 0.97

      label = string (format = '(''Re = '', f7.4, '' -> '', f7.4)', $
                      sat_re_1, sat_re_2)
      xyouts, xpos, ypos, /normal, label, charsize=char_size_1, align = 0.0
      label = string (format = '(''Lshell = '', f7.3, '' -> '', f7.3)', $
                      l_shell_1, l_shell_2)
      xyouts, xpos, ypos-ydelta, /normal, label, charsize=char_size_1, $
              align = 0.0
      label = string (format = '(''mlt = '', f4.1, '' -> '', f4.1, ''hr'')', $
              mlt_1, mlt_2)
      xyouts, xpos, ypos-2*ydelta, /normal, label, charsize=char_size_1, $
              align = 0.0
      label = string (format = $
              '(''mlat = '', f7.3, '' -> '', f7.3,''!m%!5'')', mlat_1, mlat_2)
      xyouts, xpos, ypos-3*ydelta, /normal, label, charsize=char_size_1, $
              align = 0.0
      label = string (format = $
              '(''invlat = '', f6.3, '' -> '', f6.3,''!m%!5'')', $
              invlat_1, invlat_2)
      xyouts, xpos, ypos-4*ydelta, /normal, label, charsize=char_size_1, $
              align = 0.0
      endelse

;
;------------------------------------------------------------------------------
; put on version number and date
;------------------------------------------------------------------------------
;

   if (l_setup.l_gen_movie eq 'Y') then begin
   endif else begin
      xpos = 0.80
      ypos = 0.10

      label = 'Version 3.0.0'
      xyouts, xpos, ypos, /normal, label, charsize=char_size, align = 0.0

      time_string = systime(0)
      time = strmid(time_string, 11, 8)
      dom = strmid(time_string, 8, 2)
      mon = strmid(time_string, 4, 3)
      year = strmid(time_string, 20, 4)
      label = mon + '-' + dom + '-' + year
      xyouts, xpos, ypos-ydelta, /normal, label, charsize=char_size, align=0.0
      label = time
      xyouts, xpos, ypos-(2.0*ydelta), /normal, label, charsize=char_size, $
              align=0.0
   endelse

;
;------------------------------------------------------------------------------
; write the gif file or close the postscript file
;------------------------------------------------------------------------------
;

   if (output_plot_type eq 1) then begin
      plot_image = TVRD()
      base_filename = l_setup.l_prod_dst + '/' + l_setup.l_filename
      base_filename = base_filename + spinavgsum
      base_filename = base_filename + l_setup.l_process_id
      if (l_setup.l_output_plot_format eq 'PNG') then begin
         plot_image = reverse(plot_image, 2)
         png_filename = base_filename + '.png'
         write_png, png_filename, plot_image, $
                    l_rgb_table.red(0:225), l_rgb_table.green(0:225), $
                    l_rgb_table.blue(0:225)
      endif else begin
         gif_filename = base_filename + '.gif'
         write_gif, gif_filename, plot_image, $
                    l_rgb_table.red(0:225), l_rgb_table.green(0:225), $
                    l_rgb_table.blue(0:225)
      endelse
   endif else if (output_plot_type eq 2) then begin
      device, /close_file
   endif

;
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;

endfor     ; for output_plot_type = do_x_1, do_x_2 do begin
;
;------------------------------------------------------------------------------
; output to the dump file
;------------------------------------------------------------------------------
;

if (l_setup.l_dumps eq 'Y') then begin
   dump_filename = l_setup.l_prod_dst + '/' + l_filename_base + $
                   l_setup.l_process_id + '.image.dmp'
   openw, fp1, dump_filename, /get_lun
   l_BrowsePrintDump, fp1, dump_filename, l_pltpos, nspins, $
                      date_start, start_doy, plot_start_time, $
                      date_stop, stop_doy, plot_stop_time, $
                      sat_re_1, sat_re_2, l_shell_1, l_shell_2, $
                      mlt_1, mlt_2, mlat_1, mlat_2, invlat_1, invlat_2, $
                      image0_dmp, image1_dmp
   close, fp1
endif

;
;------------------------------------------------------------------------------
; end of l_PlotSumOfSpins
;------------------------------------------------------------------------------
;

RETURN
END

;==============================================================================
;==============================================================================

PRO l_GetOAForWorldMap, l_oa, oaindx, l_setup, re, spin_mode

common oa_coord, gci_satellite_position, gci_satellite_velocity, $
                 gci_sat_spin, geo_satellite_position, $
                 gsm_satellite_position, gsm_satellite_velocity, $
                 sm_satellite_position, sm_sat_pos_curr, sm_satellite_velocity, $
                 sm_satellite_spin, geo_n_sm, sat_re, $
                 l_shell, mlt, mlat, invlat
;
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;


time_array = ConvSec2Arr ((*l_oa.tme_ptr)[oaindx], l_setup.l_start_dtime[0])

RECALC, time_array[0], time_array[1], time_array[2], time_array[3], $
        time_array[4]

xgei = ((*l_oa.gci_x_pos_ptr)[oaindx]) / re
ygei = ((*l_oa.gci_y_pos_ptr)[oaindx]) / re
zgei = ((*l_oa.gci_z_pos_ptr)[oaindx]) / re

sat_re = sqrt(xgei*xgei + ygei*ygei + zgei*zgei)
l_shell = (*l_oa.lsh_ptr)[oaindx]
invlat = acos(sqrt(1.0/l_shell)) * 57.295779

; gci satellite position and velocity values

gci_satellite_position[0] = xgei
gci_satellite_position[1] = ygei
gci_satellite_position[2] = zgei

gci_satellite_velocity[0] = ((*l_oa.gci_x_vel_ptr)[oaindx]) / re
gci_satellite_velocity[1] = ((*l_oa.gci_y_vel_ptr)[oaindx]) / re
gci_satellite_velocity[2] = ((*l_oa.gci_z_vel_ptr)[oaindx]) / re


gci_sat_spin[0] = gci_satellite_position[1] * gci_satellite_velocity[2] - $
                  gci_satellite_position[2] * gci_satellite_velocity[1]
gci_sat_spin[1] = gci_satellite_position[2] * gci_satellite_velocity[0] - $
                  gci_satellite_position[0] * gci_satellite_velocity[2]
gci_sat_spin[2] = gci_satellite_position[0] * gci_satellite_velocity[1] - $
                  gci_satellite_position[1] * gci_satellite_velocity[0]
gci_sat_spin[*] = spin_mode * gci_sat_spin[*]

geigeo, xgei, ygei, zgei, xgeo, ygeo, zgeo, 1, time_array

geo_satellite_position[0] = xgeo
geo_satellite_position[1] = ygeo
geo_satellite_position[2] = zgeo

GEOGSM, 0.0, 0.0, 1.0 ,xgsm, ygsm, zgsm, 1
SMGSM, xsm, ysm, zsm, xgsm, ygsm, zgsm, -1

mlt = ((atan(ysm, xsm) * 12.0) / !pi ) + 12.0
mlat = (asin(zsm/sat_re) * 180.0) / !pi

geo_n_sm[0] = xsm
geo_n_sm[1] = ysm
geo_n_sm[2] = zsm

gsm_satellite_position[0] = ((*l_oa.gsm_x_pos_ptr)[oaindx]) / re
gsm_satellite_position[1] = ((*l_oa.gsm_y_pos_ptr)[oaindx]) / re
gsm_satellite_position[2] = ((*l_oa.gsm_z_pos_ptr)[oaindx]) / re

; velocity in re/sec

gsm_satellite_velocity[0] = ((*l_oa.gsm_x_vel_ptr)[oaindx]) / re
gsm_satellite_velocity[1] = ((*l_oa.gsm_y_vel_ptr)[oaindx]) / re
gsm_satellite_velocity[2] = ((*l_oa.gsm_z_vel_ptr)[oaindx]) / re

; convert the gsm position coordinates to sm

SMGSM, xsm_pos, ysm_pos, zsm_pos, $
       (*l_oa.gsm_x_pos_ptr)[oaindx] / re, $
       (*l_oa.gsm_y_pos_ptr)[oaindx] / re, $
       (*l_oa.gsm_z_pos_ptr)[oaindx] / re, $
       -1

rsm = sqrt(xsm_pos*xsm_pos + ysm_pos*ysm_pos + zsm_pos*zsm_pos)


sm_sat_pos_curr[0] = xsm_pos
sm_sat_pos_curr[1] = ysm_pos
sm_sat_pos_curr[2] = zsm_pos

sm_satellite_position[*] = sm_sat_pos_curr[*]

; convert the gsm velocity coordinates to sm

SMGSM, xsm_vel, ysm_vel, zsm_vel, $
       (*l_oa.gsm_x_vel_ptr)[oaindx] / re, $
       (*l_oa.gsm_y_vel_ptr)[oaindx] / re, $
       (*l_oa.gsm_z_vel_ptr)[oaindx] / re, $
       -1

sm_satellite_velocity[0] = xsm_vel
sm_satellite_velocity[1] = ysm_vel
sm_satellite_velocity[2] = zsm_vel

; find the direction of spin axis in sm coordinates

xgci = gci_sat_spin[0]
ygci = gci_sat_spin[1]
zgci = gci_sat_spin[2]
epoch = 0
geigse, xgci, ygci, zgci, xgse, ygse, zgse, 1, epoch
gsmgse, xgsm, ygsm, zgsm, xgse, ygse, zgse, -1
smgsm, xsm, ysm, zsm, xgsm, ygsm, zgsm, -1
sm_satellite_spin[0] = xsm
sm_satellite_spin[1] = ysm
sm_satellite_spin[2] = zsm

;
;------------------------------------------------------------------------------
; end of l_GetOAForWorldMap
;------------------------------------------------------------------------------
;

RETURN
END

;============================================================
;============================================================
PRO l_BrowsePrintDump, fp1, dump_filename, l_pltpos, num_spins, $
                       date_start, start_doy, plot_start_time, $
                       date_stop, stop_doy, plot_stop_time, $
                       sat_re_1, sat_re_2, l_shell_1, l_shell_2, $
                       mlt_1, mlt_2, mlat_1, mlat_2, invlat_1, invlat_2, $
                       image0_dmp, image1_dmp
;
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;
no_cnts = 0.0
no_data = -99.0
;
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;
printf, fp1, '============================================================'
header = 'IMAGE/LENA Dump File: Images Text Dump'
printf, fp1, header
btime = string (format='(a10," (",i3.3,") ",a8)', date_start, start_doy, $
                plot_start_time)
printf, fp1, 'User specified data Start Time: ' + btime
etime = string (format='(a10," (",i3.3,") ",a8)', date_stop, stop_doy, $
                plot_stop_time)
printf, fp1, 'User specified data Stop Time:  ' + etime
temp = string (format = '(i5)', num_spins)
printf, fp1, 'Number of spins = ' + temp
printf, fp1, ' '
sw_ver= 'Software Version: ' + l_pltpos.sw_ver
printf, fp1, sw_ver
tme = 'Text Dump Creation Date: ' + systime()
printf, fp1, tme
if (l_pltpos.data_source eq 'D') then begin
   ds = 'UDF Definitive'
endif else begin
   ds = 'UDF Quicklook'
endelse
printf, fp1, 'Data Source: ' + ds
printf, fp1, 'Filename : ' + dump_filename
printf, fp1, '============================================================'
;
;------------------------------------------------------------------------------
; print the orbital information
;------------------------------------------------------------------------------
;
printf, fp1, format = '("Re         = ", f7.4, " -> ", f7.4)', $
                      sat_re_1, sat_re_2
printf, fp1, format = '("Lshell     = ", f7.3, " -> ", f7.3)', $
                      l_shell_1, l_shell_2
printf, fp1, format = '("mlt        = ", f4.1, " -> ", f4.1, " hr")', $
                      mlt_1, mlt_2
printf, fp1, format = '("mlat       = ", f7.3, " -> ", f7.3, " deg")', $
                      mlat_1, mlat_2
printf, fp1, format = '("invlat     = ", f6.3, " -> ", f6.3, " deg")', $
                      invlat_1, invlat_2
printf, fp1, '============================================================'
;
;------------------------------------------------------------------------------
; print mass data
;------------------------------------------------------------------------------
;
printf, fp1, ' '
printf, fp1, 'No Data Value   = ', no_data
printf, fp1, 'No Counts Value = ', no_cnts
printf, fp1, ' '
printf, fp1, ' '
printf, fp1, '****************** Hydrogen ******************'
printf, fp1, 'Values = Raw Data'
printf, fp1, ' '
printf, fp1, format='(a22, 4x, (A40))', 'Spin Sectors (degrees)', $
             'Polar Sectors 0->11 (-48 to +48 degrees)'
printf, fp1, format='(a22, 4x, (A40))', '----------------------', $
             '----------------------------------------'
spin_angle = -90
for ispin = 0,44 do begin
   printf, fp1, format = '(i4, ":", 1x, 12(e11.3))', $
                spin_angle, image0_dmp(*, ispin)
   spin_angle = spin_angle + 8
endfor
printf, fp1, ' '
printf, fp1, '******************* Oxygen *******************'
printf, fp1, 'Values = Raw Data'
printf, fp1, ' '
printf, fp1, format='(a22, 4x, (A39))', 'Spin Sectors (degrees)', $
             'Polar Sectors 0->11 (-48 to 48 degrees)'
printf, fp1, format='(a22, 4x, (A39))', '----------------------', $
             '---------------------------------------'
spin_angle = -90
for ispin = 0,44 do begin
   printf, fp1, format = '(i4, ":", 1x, 12(e11.3))', $
                spin_angle, image1_dmp(*, ispin)
   spin_angle = spin_angle + 8
endfor
printf, fp1, ' '
printf, fp1, '***************** All Masses *****************'
printf, fp1, 'Values = Raw Data'
printf, fp1, ' '
printf, fp1, format='(a22, 4x, (A39))', 'Spin Sectors (degrees)', $
             'Polar Sectors 0->11 (-48 to 48 degrees)'
printf, fp1, format='(a22, 4x, (A39))', '----------------------', $
             '---------------------------------------'
spin_angle = -90
image2_dmp = image0_dmp + image1_dmp
lt_zero = where (image2_dmp lt 0.0, lt_zero_counts)
if (lt_zero_counts gt 0) then image2_dmp(lt_zero) = no_data
for ispin = 0,44 do begin
   printf, fp1, format = '(i4, ":", 1x, 12(e11.3))', $
                spin_angle, image2_dmp(*, ispin)
   spin_angle = spin_angle + 8
endfor
;
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;

RETURN
END

;==============================================================================
;==============================================================================

PRO l_CompareTimeToSec, day_of_year1, seconds_of_day1, $
                        day_of_year2, seconds_of_day2, compare_flag

;------------------------------------------------------------------------------
;
; compares time1 to time2
;
; if time1 < time2 -> -1
; if time1 = time2 ->  0
; if time1 > time2 -> +1
;
;------------------------------------------------------------------------------

   compare_flag = 1

   if (day_of_year1 eq 365 or day_of_year1 eq 366) then begin
      if (day_of_year2 eq 1) then compare_flag = -1 ; crude year crossing check
   endif else if (day_of_year1 lt day_of_year2) then begin
      compare_flag = -1
   endif else if (day_of_year1 eq day_of_year2) then begin
      if (seconds_of_day1 lt seconds_of_day2) then begin
         compare_flag = -1
      endif else if (seconds_of_day1 eq seconds_of_day2) then begin
         compare_flag = 0
      endif
   endif

END

;==============================================================================
;==============================================================================

;------------------------------------------------------------------------------
;
; compares time1 to time2
;
; if time1 < time2 -> -1
; if time1 = time2 ->  0
; if time1 > time2 -> +1
;
;------------------------------------------------------------------------------

PRO l_CompareTimeToHMS, doy1, hour1, min1, sec1, $
                        doy2, hour2, min2, sec2, compare_flag

   sec_of_day_1 = (hour1 * 3600) + (min1 *60) + sec1
   sec_of_day_2 = (hour2 * 3600) + (min2 *60) + sec2

   l_CompareTimeToSec, doy1, sec_of_day_1, doy2, sec_of_day_2, compare_flag

END

;
;------------------------------------------------------------------------------
;
; l_GetRequestedSpins
;
; Due to the way that requested data is returned, ie returning data
; is outside the requested time period.  This routine descards these spins
; and keeps only the spin that fall within the requested time interval.
;
; A spin is within the requested time interval if its stop time is after the
; requested start time and/or its start time is before the requested stop time
;
; req_actual_num_spins = actual number of spin within requested time period
; req_spins = spin numbers (req_actual_num_spins)
; req_spin_start_rec = array of start records (req_actual_num_spins)
; req_spin_stop_rec = array of stop records (req_actual_num_spins)
; req_record_start = start record of first spin (req_actual_num_spins)
; req_record_stop = stop record of last spin (req_actual_num_spins)
;
;------------------------------------------------------------------------------
;

pro l_GetRequestedSpins, l_start_dtime, l_stop_dtime, record_time, nrec, $
                         req_record_start, req_record_stop, req_actual_num_spins, $
                         req_spins, req_spin_start_rec, req_spin_stop_rec

;------------------------------------------------------------------------------
; get requested start and stop times
;------------------------------------------------------------------------------

req_year_start  = 0L
req_doy_start   = 0L
req_hour_start  = 0L
req_min_start   = 0L
req_sec_start   = 0L
req_secod_start = 0L
req_year_stop   = 0L
req_doy_stop    = 0L
req_hour_stop   = 0L
req_min_stop    = 0L
req_sec_stop    = 0L
req_secod_stop  = 0L
reads, l_start_dtime, req_year_start, req_doy_start, req_hour_start, $
                      req_min_start, req_sec_start
req_secod_start = (req_hour_start * 3600) + (req_min_start * 60) + $
                  req_sec_start
reads, l_stop_dtime, req_year_stop, req_doy_stop, req_hour_stop, $
                     req_min_stop, req_sec_stop
req_secod_stop = (req_hour_stop * 3600) + (req_min_stop * 60) + $
                 req_sec_stop

;------------------------------------------------------------------------------
; initialize the start/stop records 
;------------------------------------------------------------------------------

spin_start_record = 0
spin_stop_record = spin_start_record + 44
spin_num = 0
req_actual_num_spins = 0

;------------------------------------------------------------------------------
; get the time of the 1st record and see we have data during
; requested time interval
;------------------------------------------------------------------------------

start_spin_doy  = long(record_time[0]/86400.)
start_spin_secod = long(record_time[0] - (86400. * start_spin_doy))

l_CompareTimeToSec, start_spin_doy, start_spin_secod, $
                    req_doy_stop, req_secod_stop, compare_flag

if (compare_flag gt 0) then begin     ; data starts after requested stop
   GOTO, END_OF_PRO
endif

;------------------------------------------------------------------------------
;------------------------------------------------------------------------------

while spin_start_record lt nrec do begin

;---- get the start and stop time of current spin -----

   current_spin_start_doy  = long(record_time[spin_start_record] / 86400.)
   current_spin_start_secod = long(record_time[spin_start_record] - $
                                  (86400. * current_spin_start_doy))
   current_spin_start_hour = long(current_spin_start_secod / 3600)
   current_spin_start_min  = long((current_spin_start_secod - (current_spin_start_hour * 3600)) /60)
   current_spin_start_sec  = long(current_spin_start_secod mod 60)

   current_spin_stop_doy  = long(record_time[spin_stop_record] / 86400.)
   current_spin_stop_secod = long(record_time[spin_stop_record] - $
                                 (86400. * current_spin_stop_doy))
   current_spin_stop_hour = long(current_spin_stop_secod / 3600)
   current_spin_stop_min  = long((current_spin_stop_secod - (current_spin_stop_hour * 3600)) /60)
   current_spin_stop_sec  = long(current_spin_stop_secod mod 60)
;   print, current_spin_start_hour, current_spin_start_min, current_spin_start_sec
;   print, current_spin_stop_hour, current_spin_stop_min, current_spin_stop_sec

;---- see if current spin prior to requested start time  -----

   l_CompareTimeToSec, current_spin_stop_doy, current_spin_stop_secod, $
                       req_doy_start, req_secod_start, compare_flag
   if (compare_flag lt 0) then begin    ; current spin prior to requested start time
;      print, 'spin prior to requested start'
      spin_num = spin_num + 1
      spin_start_record = spin_stop_record + 1
      spin_stop_record = spin_start_record + 44

;---- see if current spin after to requested stop time  -----

   endif else begin
      l_CompareTimeToSec, current_spin_start_doy, current_spin_start_secod, $
                          req_doy_stop, req_secod_stop, compare_flag
      if (compare_flag gt 0) then begin
;         print, 'spin after requested stop'
         GOTO, END_OF_PRO
      endif else begin
;         print, spin_start_record, spin_stop_record, $
;                current_spin_start_hour, current_spin_start_min, current_spin_start_sec, $
;                current_spin_stop_hour, current_spin_stop_min, current_spin_stop_sec
         req_spins(req_actual_num_spins) = spin_num
         req_spin_start_rec(req_actual_num_spins) = spin_start_record
         req_spin_stop_rec(req_actual_num_spins) = spin_stop_record
         req_actual_num_spins = req_actual_num_spins + 1
         spin_num = spin_num + 1
         spin_start_record = spin_stop_record + 1
         spin_stop_record = spin_start_record + 44
      endelse
   endelse

endwhile

END_OF_PRO:

if (req_actual_num_spins gt 0) then begin
   req_record_start = req_spin_start_rec(0)
   req_record_stop = req_spin_stop_rec(req_actual_num_spins-1)
endif

; 
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
; 

END

;
;------------------------------------------------------------------------------
;
; l_PlotNoDataPlot
;
;------------------------------------------------------------------------------
;

pro l_PlotNoDataPlot, l_setup, l_pltpos, l_rgb_table, $
                      plot_start_time, plot_stop_time, $
                      mass0_bd, mass1_bd, nadir

;
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;

common rlw_plot_parm, char_size, char_size_1, $
                      window_width, window_height, $
                      x_inches, y_inches

;
;------------------------------------------------------------------------------
; rearrange and calibrate the image data
; this is hard coded for now, but there is a routine which does this
;------------------------------------------------------------------------------
;

nspins = 0
npolar = 12

image0_dmp=fltarr(npolar,45)
image0_dmp[*,0:10]  = mass0_bd[*,6:16]
image0_dmp[*,11]    = mass0_bd[*,17]
image0_dmp[*,12:38] = mass0_bd[*,18:44]
image0_dmp[*,39:44] = mass0_bd[*,0:5]
image1_dmp=fltarr(npolar,45)
image1_dmp[*,0:10]  = mass1_bd[*,6:16]
image1_dmp[*,11]    = mass1_bd[*,17]
image1_dmp[*,12:38] = mass1_bd[*,18:44]
image1_dmp[*,39:44] = mass1_bd[*,0:5]
if (l_setup.l_output_type eq 1) then begin
   naz_bins = 44
   image0=fltarr(npolar,45)
   image0[*,0:10]  = mass0_bd[*,6:16]
   image0[*,11]    = mass0_bd[*,17]
   image0[*,12:38] = mass0_bd[*,18:44]
   image0[*,39:44] = mass0_bd[*,0:5]
   image1=fltarr(npolar,45)
   image1[*,0:10]  = mass1_bd[*,6:16]
   image1[*,11]    = mass1_bd[*,17]
   image1[*,12:38] = mass1_bd[*,18:44]
   image1[*,39:44] = mass1_bd[*,0:5]
endif else if(l_setup.l_output_type eq 2) then begin
   naz_bins = 44
   image0=fltarr(npolar,45)
   image0[*,0:5]  = mass0_bd[*,39:44]
   image0[*,6:44] = mass0_bd[*,0:38]
   image1=fltarr(npolar,45)
   image1[*,0:5]  = mass1_bd[*,39:44]
   image1[*,6:44] = mass1_bd[*,0:38]
endif else if (l_setup.l_output_type eq 3) then begin
   naz_bins = 11
   image0=fltarr(npolar,12)
   image0[*,0:11] = mass0_bd[*,11:22]
   image1=fltarr(npolar,12)
   image1[*,0:11] = mass1_bd[*,11:22]
endif
for np = 0,11 do begin
   for na = 0,naz_bins do begin
      image0(np,na) = -99.0
      image1(np,na) = -99.0
   endfor
endfor

image2 = image0 + image1     ; total = H + O

;
;------------------------------------------------------------------------------
; create the filename
;------------------------------------------------------------------------------
;

mo = 0
dy = 0
l_filename_base = ''
l_ConvDOY2YMD, plot_start_time[1], plot_start_time[0], mo, dy
year = string(strcompress(plot_start_time[0], /REMOVE_ALL))
yr = strmid(year, 2, 2)
l_setup.l_filename = strcompress(('l' + $
                     string(format='(A2)', yr) +  $
                     string(format='(I2.2)',mo) + $
                     string(format='(I2.2)',dy) + $
                     string(format='(I2.2)',plot_start_time[2]) + $
                     string(format='(I2.2)',plot_start_time[3]) + $
                     string(format='(I2.2)',plot_stop_time[2])  + $
                     string(format='(I2.2)',plot_stop_time[3])), $
                     /REMOVE_ALL)
l_filename_base = l_setup.l_filename

if (l_setup.l_output_type eq 1) then begin
   l_setup.l_filedesc = 'image_3full'
   l_setup.l_filename = l_filename_base + '_3fl'
endif else if (l_setup.l_output_type eq 2) then begin
   l_setup.l_filedesc = 'image_3cnst'
   l_setup.l_filename = l_filename_base + '_3cn'
endif else begin
   l_setup.l_filedesc = 'image_3sq'
   l_setup.l_filename = l_filename_base + '_3sq'
endelse

if (l_setup.l_singles_sum eq 1) then begin
   spinavgsum = '0sp'
endif else begin
   if (l_setup.l_avg_sum eq 1) then begin
      spinavgsum = '00spa'
   endif else begin
      spinavgsum = '00sps'
   endelse
endelse

;
;------------------------------------------------------------------------------
; load color table
;------------------------------------------------------------------------------
;

;   load_color_table

;
;------------------------------------------------------------------------------
; loop through and do each plot type
;------------------------------------------------------------------------------
;

nplots = 0
if (strupcase(l_setup.l_via_web) eq 'N') then $
   do_x_1 = 0 else $
   do_x_1 = 1
if (l_setup.l_gen_movie eq 'Y') then $
   do_x_2 = 1 else $
   do_x_2 = 2

;
;------------------------------------------------------------------------------
; set up plot type
;------------------------------------------------------------------------------
;

for output_plot_type = do_x_1, do_x_2 do begin
   if (output_plot_type eq 0) then begin
      l_SetUpDisplay, 'X', l_pltpos, l_rgb_table, $
                      l_setup.l_color_tbl, colorbar, nplots
   endif else if (output_plot_type eq 1) then begin
      l_SetUpDisplay, 'Z',  l_pltpos, l_rgb_table, $
                      l_setup.l_color_tbl, colorbar, nplots
   endif else begin
      ps_filename = l_setup.l_prod_dst + '/' + l_setup.l_filename
      ps_filename = ps_filename + spinavgsum
      l_pltpos.filename = ps_filename + l_setup.l_process_id  + '.ps'
      l_SetUpDisplay, 'P', l_pltpos, l_rgb_table, $
                      l_setup.l_color_tbl, colorbar, nplots
   endelse
;
;------------------------------------------------------------------------------
; plot the 3 images and the global overlay and the color bar
;------------------------------------------------------------------------------
;

   if (l_setup.l_gen_movie eq 'Y') then begin
      plot_ImageMovie, image0, image1, image2, npolar, polar_angle, l_setup, $
                       nadir
   endif else begin
      if (l_setup.l_output_type eq 1) then begin
         plot_3full_nodata, l_setup, image0, image1, image2
      endif else begin
         plot_3consqr_nodata, l_setup, image0, image1, image2 ; constant & square
      endelse
   endelse

;
;------------------------------------------------------------------------------
; put on the top left labeling
;------------------------------------------------------------------------------
;

   l_ConvDOY2YMD, plot_start_time[1], plot_start_time[0], $
                  start_mo, start_dy
   start_yr = plot_start_time[0]
   date_start = strcompress(( $
      string(format='(I4.4)', start_yr) + '/' + $
      string(format='(I2.2)', start_mo)  + '/' + $
      string(format='(I2.2)', start_dy)),  /REMOVE_ALL)
   plot_start_time_string = string(format='(I2.2, '':'', I2.2, '':'', I2.2)', $
                            plot_start_time[2], $
                            plot_start_time[3], $
                            plot_start_time[4])
   l_ConvDOY2YMD, plot_stop_time[1], plot_stop_time[0], $
                  stop_mo, stop_dy
   stop_yr = plot_stop_time[0]
   date_stop = strcompress(( $
      string(format='(I4.4)', stop_yr) + '/' + $
      string(format='(I2.2)', stop_mo)  + '/' + $
      string(format='(I2.2)', stop_dy)),  /REMOVE_ALL)
   plot_stop_time_string = string(format='(I2.2, '':'', I2.2, '':'', I2.2)', $
                           plot_stop_time[2], $
                           plot_stop_time[3], $
                           plot_stop_time[4])
;   print, plot_start_time_string
;   print, plot_stop_time_string

   if (l_setup.l_gen_movie eq 'Y') then begin
      if (l_setup.l_output_type eq 1) then begin
         xpos=0.06
         ypos=0.96
      endif else begin
         xpos=0.11
         ypos=0.90
      endelse
      label = date_start + ' ' + plot_start_time_string + ' -> ' + plot_stop_time_string
      xyouts, xpos, ypos, /normal, label, charsize=char_size_1, align = 0.0
   endif else begin
      xyouts, 0.0, 0.0, '!5', /normal

      xpos=0.10
      ypos=0.97
      ydelta=0.022

      label = 'IMAGE LENA'
      xyouts, xpos, ypos, /normal, label, charsize=char_size_1, align = 0.0

      ; put on the data plotted start/stop times


      label = 'start time: ' + date_start + ' ' + plot_start_time_string
      xyouts, xpos, ypos-ydelta, /normal, label, $
              charsize=char_size_1, align = 0.0
      label = 'stop time: ' + date_stop + ' ' + plot_stop_time_string
      xyouts, xpos, ypos-2*ydelta, /normal, label, $
              charsize=char_size_1, align = 0.0

      ; put on the number of spins

      if (l_setup.l_singles_sum eq 1) then begin
         xyouts, xpos, ypos-3*ydelta, /normal, '0 spin', $
                 charsize=char_size_1, align = 0.0
      endif else begin
         if (l_setup.l_avg_sum eq 1) then begin
            if (nspins lt 10) then begin
               label = string (format = '(i1,'' spin average'')', nspins)
            endif else begin
               label = string (format = '(i2,'' spin average'')', nspins)
            endelse
         endif else begin
            if (nspins lt 10) then begin
               label = string (format = '(i1,'' spin sum'')', nspins)
            endif else begin
               label = string (format = '(i2,'' spin sum'')', nspins)
            endelse
         endelse
         xyouts, xpos, ypos-3*ydelta, /normal, label, $
                 charsize=char_size_1, align = 0.0
      endelse
   endelse

;
;------------------------------------------------------------------------------
; put on the orbital annotation
;------------------------------------------------------------------------------
;

   if (l_setup.l_gen_movie eq 'Y') then begin
   endif else begin
      sat_re_1 = 0.0
      sat_re_2 = 0.0
      l_shell_1 = 0.0
      l_shell_2 = 0.0
      mlt_1 = 0.0
      mlt_2 = 0.0
      mlat_1 = 0.0
      mlat_2 = 0.0
      invlat_1 = 0.0
      invlat_2 = 0.0

      xpos = 0.50
      ypos = 0.97

      if (l_setup.l_singles_sum eq 1) then begin
         label = string (format = '(''Re = '', f7.4)', sat_re_1)
         xyouts, xpos, ypos, /normal, label, charsize=char_size_1, align = 0.0
         label = string (format = '(''Lshell = '', f7.3)', l_shell_1)
         xyouts, xpos, ypos-ydelta, /normal, label, charsize=char_size_1, $
                 align = 0.0
         label = string (format = '(''mlt = '', f4.1)', mlt_1)
         xyouts, xpos, ypos-2*ydelta, /normal, label, charsize=char_size_1, $
                 align = 0.0
         label = string (format = '(''mlat = '', f7.3, ''!m%!5'')', mlat_1)
         xyouts, xpos, ypos-3*ydelta, /normal, label, charsize=char_size_1, $
                 align = 0.0
         label = string (format = '(''invlat = '', f6.3, ''!m%!5'')', invlat_1)
         xyouts, xpos, ypos-4*ydelta, /normal, label, charsize=char_size_1, $
                 align = 0.0
      endif else begin
         label = string (format = '(''Re = '', f7.4, '' -> '', f7.4)', $
                         sat_re_1, sat_re_1)
         xyouts, xpos, ypos, /normal, label, charsize=char_size_1, align = 0.0
         label = string (format = '(''Lshell = '', f7.3, '' -> '', f7.3)', $
                         l_shell_1, l_shell_2)
         xyouts, xpos, ypos-ydelta, /normal, label, charsize=char_size_1, $
                 align = 0.0
         label = string (format = '(''mlt = '', f4.1, '' -> '', f4.1, ''hr'')', $
                 mlt_1, mlt_2)
         xyouts, xpos, ypos-2*ydelta, /normal, label, charsize=char_size_1, $
                 align = 0.0
         label = string (format = $
                 '(''mlat = '', f7.3, '' -> '', f7.3,''!m%!5'')', mlat_1, mlat_2)
         xyouts, xpos, ypos-3*ydelta, /normal, label, charsize=char_size_1, $
                 align = 0.0
         label = string (format = $
                 '(''invlat = '', f6.3, '' -> '', f6.3,''!m%!5'')', $
                 invlat_1, invlat_2)
         xyouts, xpos, ypos-4*ydelta, /normal, label, charsize=char_size_1, $
                 align = 0.0
      endelse
   endelse

;
;------------------------------------------------------------------------------
; put on version number and date
;------------------------------------------------------------------------------
;

   if (l_setup.l_gen_movie eq 'Y') then begin
   endif else begin
      xpos = 0.80
      ypos = 0.10

      label = 'Version 3.0.0'
      xyouts, xpos, ypos, /normal, label, charsize=char_size, align = 0.0

      time_string = systime(0)
      time = strmid(time_string, 11, 8)
      dom = strmid(time_string, 8, 2)
      mon = strmid(time_string, 4, 3)
      year = strmid(time_string, 20, 4)
      label = mon + '-' + dom + '-' + year
      xyouts, xpos, ypos-ydelta, /normal, label, charsize=char_size, align=0.0
      label = time
      xyouts, xpos, ypos-(2.0*ydelta), /normal, label, charsize=char_size, $
              align=0.0
   endelse


;
;------------------------------------------------------------------------------
; write the gif file or close the postscript file
;------------------------------------------------------------------------------
;

   if (output_plot_type eq 1) then begin
      plot_image = TVRD()
      base_filename = l_setup.l_prod_dst + '/' + l_setup.l_filename
      base_filename = base_filename + spinavgsum
      base_filename = base_filename + l_setup.l_process_id
      if (l_setup.l_output_plot_format eq 'PNG') then begin
         plot_image = reverse(plot_image, 2)
         png_filename = base_filename + '.png'
         write_png, png_filename, plot_image, $
                    l_rgb_table.red(0:225), l_rgb_table.green(0:225), $
                    l_rgb_table.blue(0:225)
      endif else begin
         gif_filename = base_filename + '.gif'
         write_gif, gif_filename, plot_image, $
                    l_rgb_table.red(0:225), l_rgb_table.green(0:225), $
                    l_rgb_table.blue(0:225)
      endelse
   endif else if (output_plot_type eq 2) then begin
      device, /close_file
   endif

;
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;

endfor     ; for output_plot_type = do_x_1, do_x_2 do begin
;
;------------------------------------------------------------------------------
; output to the dump file
;------------------------------------------------------------------------------
;

if (l_setup.l_dumps eq 'Y') then begin
   dump_filename = l_setup.l_prod_dst + '/' + l_filename_base + $
                   l_setup.l_process_id + '.image.dmp'
   openw, fp1, dump_filename, /get_lun
   l_BrowsePrintDump, fp1, dump_filename, l_pltpos, nspins, $
                      date_start, start_doy, plot_start_time, $
                      date_stop, stop_doy, plot_stop_time, $
                      sat_re_1, sat_re_2, l_shell_1, l_shell_2, $
                      mlt_1, mlt_2, mlat_1, mlat_2, invlat_1, invlat_2, $
                      image0_dmp, image1_dmp
   close, fp1
endif

;
;------------------------------------------------------------------------------
; end of l_PlotNoDataPlot
;------------------------------------------------------------------------------
;

RETURN
END
