;-------------------------------------------------------------------------------
;  Procedure:  l_BinnedEventProc
;  Description:
;       Main.  Provides the option of selecting to use binned or event data.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_setup               struct        I        general setup options/parameters
;  l_setup_file          string        I        setup filename
;  l_pltpos              struct        I        general plot parameters
;  l_rgb_table           struct        I        color table parameters
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;     PLOTDEFS         MAX_PLOTS        int         G         max # plots per page
;                      MAX_WEB_SPINS    int         G         max # spins per plot
;                                                             when using web interface
;                      NO_CNTS          double      G         no counts indicator
;                      NO_DATA          double      G         no data indicator
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;  ELH               01/02           v1.0.8           Add to flow binned/event 
;-------------------------------------------------------------------------------


;-------------------------------------------------------------------------------
;  Procedure: l_BinnedEventTerm
;  Description: Free allocated memory.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_roi0                struct*       I        ROI data ptr structure
;  l_binned              struct*       I        binned data ptr structure
;  l_binned_setup        struct        I        binned setup structure
;  l_event               struct*       I        event data ptr structure
;  l_event_setup         struct        I        event setup structure
;  l_memd                struct*       I        memory dump data ptr structure
;  l_evsp                struct*       I        sectored events data ptr structure
;  l_hskp                struct*       I        housekeeping data ptr structure
;  l_oa                  struct*       I        orbit & attitude data ptr structure
;  avail                 int           I        lena data packet availability 
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                01/02          v1.0.8            Original implementation
;-------------------------------------------------------------------------------

PRO l_BinnedEventTerm, l_roi0, l_binned, l_binned_setup, l_event, l_event_setup, $
l_memd, l_evsp, l_hskp, l_oa, l_setup, davail

   ;-------------------------------------------------
   ; free pointers, if memory allocated
   ;-------------------------------------------------

   rel_bin = 0L
   if (l_binned_setup.use_binned eq 'Y') then begin
       l_BinnedTerm, l_binned
       rel_bin = 1L
   endif

   if (l_event_setup.use_event eq 'Y') OR ((l_binned_setup.use_binned eq 'Y') AND $
         (l_setup.l_flux eq 'Y')) then begin
       l_EventTerm, l_event
       case davail of
          1    : begin
                    if (rel_bin eq 0L) then begin
                       l_BinnedTerm, l_binned
                    endif
                 end
          2    : begin
                    l_MemdTerm, l_memd
                 end
          3    : begin
                 end
          4    : begin
                   l_EvSpTerm, l_evsp
                 end
          else : begin
                  print, 'Invalid option.'
                 end
       endcase
   endif


   l_HskpTerm, l_hskp
   l_OATerm, l_oa

RETURN
END


;-------------------------------------------------------------------------------
;  Function :  l_BinnedEventSelect
;
;  Description:  Select whether to read binned or event data based on
;    user specifications. Read the appropriate data packets to be used
;    with selected data type depending on data availability.
;   Before FSW upload 7/31/00 = no sectored event data packets
;   7/31/00 - 10/15/00 = sectored event packets in memory dump
;   10/16/00 - 12/02/00 = no sectored event packets 
;   12/02/00 - present = sectored event packets 
;
;  Return Value:  type = int
;  Value                         Description
;  -------------------------     ------------------------------
;  SUCCESS                        successful completion
;  FAILURE                        unsuccessful completion
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_setup               struct        I        general processing parameters 
;  l_setup_file          string        I        runtime file name
;  l_pltpos              struct        I        general plotting parameters
;  l_roi0                struct *      I        pointers to ROI data structures
;  l_binned              struct *      I        pointers to binned data structures
;  l_binned_setup        struct        I        binned processing & plotting parameters
;  l_event               struct *      I        pointers to event data structures
;  l_event_setup         struct        I        event processing & plotting parameters
;  l_memd                struct *      I        pointers to memory dump data structures
;  l_memd_setup          struct        I        memory dump processing & plotting parameters
;  l_evsp                struct *      I        pointers to sectored event data structures
;  l_evsp_setup          struct        I        sectored events processing & plotting parameters
;  l_hskp                struct *      I        pointers to housekeeping data structures
;  l_oa                  struct *      I        pointers to orbit and attitude data structures 
;  davail                int           I        data availability flag
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;     PLOTDEFS         MAX_PLOTS        int         G         max # plots per page
;                      MAX_WEB_SPINS    int         G         max # spins per plot
;                                                             when using web interface
;                      NO_CNTS          double      G         no counts indicator
;                      NO_DATA          double      G         no data indicator
;     ERROR_TYPES      SUCCESS          int         G         successful completion
;                      FAILURE          int         G         unsuccessful completion
;                      WARNING          int         G         warning message
;                      INFO             int         G         informational message
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                01/02          v1.0.8            Original implementation
;-------------------------------------------------------------------------------


FUNCTION l_BinnedEventSelect, l_setup, l_setup_file, l_pltpos, $
l_roi0, l_binned, l_binned_setup, l_event, l_event_setup, $
l_memd, l_memd_setup, l_evsp, l_evsp_setup, l_hskp, l_oa, davail
COMMON PLOTDEFS, MAX_PLOTS, MAX_WEB_SPINS, NO_CNTS, NO_DATA
COMMON ERROR_TYPES, SUCCESS, FAILURE, WARNING, INFO

   ;-------------------------------------------------
   ; read setup paramters specific to the data 
   ;-------------------------------------------------
   l_BinnedReadSetup, l_binned_setup, l_setup_file
   l_EventReadSetup, l_event_setup, l_setup_file

   ;-------------------------------------------------
   ; Can not select to use both binned and event data
   ; if both are selected, binned data is used 
   ;-------------------------------------------------
   if (l_binned_setup.use_binned eq 'Y') AND (l_event_setup.use_event eq 'Y') then begin
      l_pltpos.plot_hdr_title = l_pltpos.plot_hdr_title + ' Binned'
      if (l_setup.l_flux eq 'Y') then begin 
           l_pltpos.plot_hdr_title = l_pltpos.plot_hdr_title + ' Bk Cor'
      endif
      print, 'Unable to select both binned & event. Defaulting to using binned data.'
      l_event_setup.use_event = 'N'
   endif else begin
     if (l_binned_setup.use_binned eq 'Y') AND (l_event_setup.use_event eq 'N') then begin
        l_pltpos.plot_hdr_title = l_pltpos.plot_hdr_title + ' Binned'
        if (l_setup.l_flux eq 'Y') then begin 
           l_pltpos.plot_hdr_title = l_pltpos.plot_hdr_title + ' Bk Cor'
        endif 
     endif else begin
        if (l_binned_setup.use_binned eq 'N') AND (l_event_setup.use_event eq 'Y') then begin
           l_pltpos.plot_hdr_title = l_pltpos.plot_hdr_title + ' Event'
           if (l_event_setup.background eq 'Y') then begin
              l_pltpos.plot_hdr_title = l_pltpos.plot_hdr_title + ' Bk Cor'
           endif
        endif else begin
           print, 'No binned/event data selected.'
           return, INFO
        endelse
      endelse
   endelse


   binned_read = 0L


   ;----------------------------------------------------------------------
   ; read the event data, if selected
   ;----------------------------------------------------------------------
   if (l_event_setup.use_event eq 'Y') OR ((l_setup.l_flux eq 'Y') AND $
        (l_binned_setup.use_binned eq 'Y')) then begin

      use_binned_data = 0L

      status = l_EventGetData  (l_setup, $
                                l_setup_file, $
                                l_pltpos, $
                                l_event_setup, $
                                l_event)
      if (status eq FAILURE) then return, FAILURE


      n_event    = l_event.nitems
      event_cnts = make_array(l_event.nitems, /INT, value = NO_DATA)
      sunp       = make_array(l_event.nitems, /INT, value = NO_DATA)
      binned_read= 0

      ;----------------------------------------------------------------------
      ; determine if and where the sectored events data packets are available 
      ;----------------------------------------------------------------------
      davail = CheckAvailData (l_setup.l_start_dtime)
      case davail of
         1   : begin
                  ;-------------------------------------------------
                  ; no sectored events packets available, must use
                  ; sum of the binned counts
                  ;-------------------------------------------------

                  status = l_BinnedGetData  (l_setup, $
                                             l_setup_file, $
                                             l_pltpos, $
                                             l_binned_setup, $
                                             l_binned, $
                                             nele)
                  if (status eq FAILURE) then return, FAILURE
                  binned_read = 1

                  n_binn  = n_elements ((*l_binned.startUdfTimePtr)[*])
                  steering  = make_array(l_event.nitems, /INT, value = NO_DATA)
                  SyncEvent2Binned, l_binned, l_event, n_binn, n_event, $
                                    event_cnts, steering


               end
         2   : begin
                  ;-------------------------------------------------
                  ; sectored events packets available in memory dump
                  ; packets
                  ;-------------------------------------------------
                  status = l_MemdGetData (l_setup, l_pltpos, l_memd)
                  if (status eq FAILURE) then return, FAILURE
                  n_memd = n_elements ((*l_memd.startUdfTimePtr)[*])
                  SyncEvent2SectEvnts, l_memd, n_memd, l_event, $
                                       n_event, event_cnts, sunp

               end
         3   : begin
                  ;-------------------------------------------------
                  ; between 10/16/00 - 12/02/00, sectored events
                  ; should be individual packets, but are not
                  ; available as individual packets or in the
                  ; memory dump, after reprocessing the packets
                  ; SHOULD exist, therefore, use binned data instead
                  ;-------------------------------------------------
                  ;-------------------------------------------------
                  ; no sectored events packets available, must use
                  ; sum of the binned counts
                  ;-------------------------------------------------

                  status = l_BinnedGetData  (l_setup, $
                                             l_setup_file, $
                                             l_pltpos, $
                                             l_binned_setup, $
                                             l_binned, $
                                             nele)
                  if (status eq FAILURE) then return, FAILURE
                  binned_read = 1

                  n_binn  = n_elements ((*l_binned.startUdfTimePtr)[*])
                  steering  = make_array(l_event.nitems, /INT, value = NO_DATA)
                  SyncEvent2Binned, l_binned, l_event, n_binn, n_event, $
                                    event_cnts, steering
               end
         4   : begin
                  ;-------------------------------------------------
                  ; read sectored events/sun pulse packet
                  ;-------------------------------------------------
                  nele = 0L
                  EvDmpFlg = 0L
                  status = l_EvSpGetData (l_setup, l_pltpos, l_evsp, nele, EvDmpFlg)
                  if (status eq FAILURE) then return, FAILURE
                  n_evsp = n_elements ((*l_evsp.startUdfTimePtr)[*])
                  SyncEvent2SectEvnts, l_evsp, n_evsp, l_event, $
                                       n_event, event_cnts, sunp
               end
         else : begin
                  print, 'Invalid option.'
                end
      endcase   ;--- end data avail case ---


      ;-------------------------------------------------
      ; handle missing data and do background correction
      ; of event data
      ;-------------------------------------------------
      if (l_event_setup.use_event eq 'Y') then begin
         l_Event_CorrBkgrnd, l_event_setup, l_event, l_setup, $
                             event_cnts

         l_Event_Adj4ExtraData, l_event, l_event_setup, l_setup

         l_EventHndlMissData, l_setup, l_event, l_event_setup, l_oa, $
                              l_pltpos

         l_Event_Conv2Units, l_setup, l_event_setup, l_event, l_pltpos
      endif

   endif        ;--- end event data ---


   ;-------------------------------------------------------------
   ; if binned data selected & background correction, need to 
   ; find the background data from the event data; therefore, need 
   ; to read the just the background data from the event
   ; data in order to apply this to the binned data
   ;-------------------------------------------------------------
   if (l_binned_setup.use_binned eq 'Y') AND (l_setup.l_flux eq 'Y') then begin
       l_EventJustBkgrndData, l_event, l_event_setup, l_setup, event_cnts
   endif


   ;-------------------------------------------------
   ; get the binned data based on user start and
   ; stop time
   ;-------------------------------------------------
   if (l_binned_setup.use_binned eq 'Y') or (l_setup.l_dumps_bkdata eq 'Y') then begin
      if (binned_read eq 0) then begin
         status = l_BinnedGetData  (l_setup, $
                                    l_setup_file, $
                                    l_pltpos, $
                                    l_binned_setup, $
                                    l_binned, $
                                    nele)
         if (status eq FAILURE) then return, FAILURE
      endif


      if (l_binned_setup.use_binned eq 'Y') then begin
         if (l_setup.l_flux eq 'Y') then begin
            l_Binned_AdjExData, l_setup, l_pltpos, l_binned, l_binned_setup, l_event
         endif else begin
            l_setup.l_flux = 'N'
            l_Binned_AdjExData, l_setup, l_pltpos, l_binned, l_binned_setup
         endelse
      endif ;--- use binned ---

   endif ;--- use binned or dump background data ---


   RETURN, SUCCESS

END


;-------------------------------------------------------------------------------
;  Procedure: l_BinnedEventPlot
;
;  Description:
;     Plot the selected data based on the user selection.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_setup               struct        I        general setup parameters
;  l_binned_setup        struct        I        binned processing & plotting parameters
;  l_event_setup         struct        I        event processing & plotting parameters
;  l_binned              struct *      I        pointers to binned data structures 
;  l_event               struct *      I        pointers to event data structures 
;  l_pltpos              struct        I        general plotting parameters
;  l_roi0                struct *      I        pointers to ROI 0 data structures
;  l_oa                  struct *      I        pointers to OA data structures
;  l_hskp                struct *      I        pointers to housekeeping data structures
;  l_rgb_table           struct        I        red, green, blue color values
;  nplots                int           I        number of plots to be plotted
;  windx                 array         I        window index number
;  plotted               int           I        number of plots plotted
;  pages                 int           I        number of plotting pages needed
;  colorbar              array         I        color bar values
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;     PLOTDEFS         MAX_PLOTS        int         G         max # plots per page
;                      MAX_WEB_SPINS    int         G         max # spins per plot
;                                                             when using web interface
;                      NO_CNTS          double      G         no counts indicator
;                      NO_DATA          double      G         no data indicator
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               01/02          v1.0.8             Original implementation
;-------------------------------------------------------------------------------

PRO l_BinnedEventPlot, l_setup, l_binned_setup, l_event_setup, l_binned, l_event, $
l_pltpos, l_roi0, l_oa, l_hskp, l_rgb_table, nplots, windx, plotted, pages, colorbar
COMMON PLOTDEFS, MAX_PLOTS, MAX_WEB_SPINS, NO_CNTS, NO_DATA


    data_format = l_setup.l_num_spinsec

    ;------------------------------------------------------------
    ; plot the binned data
    ;------------------------------------------------------------
    if (l_binned_setup.use_binned eq 'Y') then begin
       l_BinnedPlot, l_binned_setup, $
                     l_binned, $
                     l_pltpos, $
                     nplots, $
                     windx, $
                     pages, $
                     plotted, $
                     l_setup, $
                     l_oa , $
                     l_rgb_table, $
                     colorbar, $
                     l_hskp
    endif

    ;------------------------------------------------------------
    ; plot the event data based on user selection of which
    ; masses or energies to plot
    ;------------------------------------------------------------
    if (l_event_setup.use_event eq 'Y') then begin

       ;------------------------------------------------------------
       ; determine which masses and energy combinations
       ;------------------------------------------------------------
       if (l_event_setup.merge eq 'Y') then begin
          merge_cnt = 0L
          if (l_event_setup.hydrogen eq 'Y') then merge_cnt = merge_cnt + 1L
          if (l_event_setup.oxygen eq 'Y') then merge_cnt = merge_cnt + 1L
          if (l_event_setup.energy0 eq 'Y') then merge_cnt = merge_cnt + 1L
          if (l_event_setup.energy1 eq 'Y') then merge_cnt = merge_cnt + 1L
          if (l_event_setup.energy2 eq 'Y') then merge_cnt = merge_cnt + 1L
       
          if (merge_cnt lt 3) then l_event_setup.merge = 'N'
       endif

       ;------------------------------------------------------------
       ; plot hydrogen data
       ;------------------------------------------------------------
       if (l_event_setup.hydrogen eq 'Y') then begin

          if (l_event_setup.merge eq 'Y') then begin
             if (l_event_setup.energy0 eq 'Y') then begin
                he0_nc_indx = where ((*l_event.hydrogen_e0_flux) eq NO_CNTS, he0_nc_cnt)
                he0_nd_indx = where ((*l_event.hydrogen_e0_flux) eq NO_DATA, he0_nd_cnt)
                if (he0_nc_cnt gt 0) then (*l_event.hydrogen_e0_flux)[he0_nc_indx] = 0.0
                if (he0_nd_cnt gt 0) then (*l_event.hydrogen_e0_flux)[he0_nd_indx] = 0.0
             endif

             if (l_event_setup.energy1 eq 'Y') then begin
                he1_nc_indx = where ((*l_event.hydrogen_e1_flux) eq NO_CNTS, he1_nc_cnt)
                he1_nd_indx = where ((*l_event.hydrogen_e1_flux) eq NO_DATA, he1_nd_cnt)
                if (he1_nc_cnt gt 0) then (*l_event.hydrogen_e1_flux)[he1_nc_indx] = 0.0
                if (he1_nd_cnt gt 0) then (*l_event.hydrogen_e1_flux)[he1_nd_indx] = 0.0
             endif

             if (l_event_setup.energy2 eq 'Y') then begin
                he2_nc_indx = where ((*l_event.hydrogen_e2_flux) eq NO_CNTS, he2_nc_cnt)
                he2_nd_indx = where ((*l_event.hydrogen_e2_flux) eq NO_DATA, he2_nd_cnt)
                if (he2_nc_cnt gt 0) then (*l_event.hydrogen_e2_flux)[he2_nc_indx] = 0.0
                if (he2_nd_cnt gt 0) then (*l_event.hydrogen_e2_flux)[he2_nd_indx] = 0.0
             endif
          endif
       endif

       ;------------------------------------------------------------
       ; plot oxygen data
       ;------------------------------------------------------------
       if (l_event_setup.oxygen eq 'Y') then begin

          if (l_event_setup.merge eq 'Y') then begin
             if (l_event_setup.energy0 eq 'Y') then begin
                oe0_nc_indx = where ((*l_event.oxygen_e0_flux) eq NO_CNTS, oe0_nc_cnt)
                oe0_nd_indx = where ((*l_event.oxygen_e0_flux) eq NO_DATA, oe0_nd_cnt)
                if (oe0_nc_cnt gt 0) then (*l_event.oxygen_e0_flux)[oe0_nc_indx] = 0.0
                if (oe0_nd_cnt gt 0) then (*l_event.oxygen_e0_flux)[oe0_nd_indx] = 0.0
             endif

             if (l_event_setup.energy1 eq 'Y') then begin
                oe1_nc_indx = where ((*l_event.oxygen_e1_flux) eq NO_CNTS, oe1_nc_cnt)
                oe1_nd_indx = where ((*l_event.oxygen_e1_flux) eq NO_DATA, oe1_nd_cnt)
                if (oe1_nc_cnt gt 0) then (*l_event.oxygen_e1_flux)[oe1_nc_indx] = 0.0
                if (oe1_nd_cnt gt 0) then (*l_event.oxygen_e1_flux)[oe1_nd_indx] = 0.0
             endif

             if (l_event_setup.energy2 eq 'Y') then begin
                oe2_nc_indx = where ((*l_event.oxygen_e2_flux) eq NO_CNTS, oe2_nc_cnt)
                oe2_nd_indx = where ((*l_event.oxygen_e2_flux) eq NO_DATA, oe2_nd_cnt)
                if (oe2_nc_cnt gt 0) then (*l_event.oxygen_e2_flux)[oe2_nc_indx] = 0.0
                if (oe2_nd_cnt gt 0) then (*l_event.oxygen_e2_flux)[oe2_nd_indx] = 0.0
             endif
           endif
       endif


    ;------------------------------------------------------------
    ; plotting both hydrogen and oxygen data
    ;------------------------------------------------------------
    title = ' '
    if (l_event_setup.merge eq 'Y') then begin 
       ;------------------------------------------------------------
       ; both hydrogen and oxygen data using all energies
       ;------------------------------------------------------------
       if (l_event_setup.hydrogen eq 'Y') AND (l_event_setup.oxygen eq 'Y') then begin
          title = 'Event'
          if (l_event_setup.background eq 'Y') then begin
             title = 'Event Sc/Bk Corr'
          endif

          if (l_event_setup.energy0 eq 'Y') AND (l_event_setup.energy1 eq 'Y') AND $
             (l_event_setup.energy2 eq 'Y') then begin
             title1 = '10eV-300eV'

             Hsum = (*l_event.hydrogen_e0_flux) + $
                    (*l_event.hydrogen_e1_flux) + $
                    (*l_event.hydrogen_e2_flux) 


             Osum = (*l_event.oxygen_e0_flux) + $
                    (*l_event.oxygen_e1_flux) + $
                    (*l_event.oxygen_e2_flux) 

             summed = Hsum + Osum 

             if (oe0_nd_cnt gt 0) then summed[oe0_nd_indx] = NO_DATA 
             if (oe1_nd_cnt gt 0) then summed[oe1_nd_indx] = NO_DATA 
             if (oe2_nd_cnt gt 0) then summed[oe2_nd_indx] = NO_DATA 

             if (he0_nd_cnt gt 0) then summed[he0_nd_indx] = NO_DATA 
             if (he1_nd_cnt gt 0) then summed[he1_nd_indx] = NO_DATA 
             if (he2_nd_cnt gt 0) then summed[he2_nd_indx] = NO_DATA 

             if (oe0_nc_cnt gt 0) then summed[oe0_nc_indx] = NO_CNTS
             if (oe1_nc_cnt gt 0) then summed[oe1_nc_indx] = NO_CNTS
             if (oe2_nc_cnt gt 0) then summed[oe2_nc_indx] = NO_CNTS

             if (he0_nc_cnt gt 0) then summed[he0_nc_indx] = NO_CNTS
             if (he1_nc_cnt gt 0) then summed[he1_nc_indx] = NO_CNTS
             if (he2_nc_cnt gt 0) then summed[he2_nc_indx] = NO_CNTS

             ptr = ptr_new (summed)
             l_event.amDmpPtr = ptr

             if (l_setup.l_log eq 'Y') then begin
                l_Conv2Log, summed, event_log, event_min, event_max
                summed = event_log
             endif


             ptr = ptr_new (summed)
             l_event.all_masses = ptr
          endif else begin  ;--- all energies ---
             ;------------------------------------------------------------
             ; hydrogen and oxygen data using just energy 0 and 1
             ;------------------------------------------------------------
             if (l_event_setup.energy0 eq 'Y') AND (l_event_setup.energy1 eq 'Y') then begin
                title1 = '10eV-100eV'

                Hsum = (*l_event.hydrogen_e0_flux) + (*l_event.hydrogen_e1_flux)
                Osum = (*l_event.oxygen_e0_flux) + (*l_event.oxygen_e1_flux)
                summed = Hsum + Osum

                if (oe0_nd_cnt gt 0) then summed[oe0_nd_indx] = NO_DATA 
                if (oe1_nd_cnt gt 0) then summed[oe1_nd_indx] = NO_DATA 
   
                if (he0_nd_cnt gt 0) then summed[he0_nd_indx] = NO_DATA 
                if (he1_nd_cnt gt 0) then summed[he1_nd_indx] = NO_DATA 

                if (oe0_nc_cnt gt 0) then summed[oe0_nc_indx] = NO_CNTS
                if (oe1_nc_cnt gt 0) then summed[oe1_nc_indx] = NO_CNTS

                if (he0_nc_cnt gt 0) then summed[he0_nc_indx] = NO_CNTS
                if (he1_nc_cnt gt 0) then summed[he1_nc_indx] = NO_CNTS

                ptr = ptr_new (summed)
                l_event.amDmpPtr = ptr

                if (l_setup.l_log eq 'Y') then begin
                   l_Conv2Log, summed, event_log, event_min, event_max
                   summed = event_log
                endif

                ptr = ptr_new (summed)
                l_event.all_masses = ptr

             endif else begin
               ;------------------------------------------------------------
               ; hydrogen and oxygen using just energy 0 and 2
               ;------------------------------------------------------------
               if (l_event_setup.energy0 eq 'Y') AND (l_event_setup.energy2 eq 'Y') then begin    
                  title1 = '10eV-300eV'

                  Hsum = (*l_event.hydrogen_e0_flux) + (*l_event.hydrogen_e2_flux) 
                  Osum = (*l_event.oxygen_e0_flux) + (*l_event.oxygen_e2_flux) 
                  summed = Hsum + Osum 

                  if (oe0_nd_cnt gt 0) then summed[oe0_nd_indx] = NO_DATA 
                  if (oe2_nd_cnt gt 0) then summed[oe2_nd_indx] = NO_DATA 

                  if (he0_nd_cnt gt 0) then summed[he0_nd_indx] = NO_DATA 
                  if (he2_nd_cnt gt 0) then summed[he2_nd_indx] = NO_DATA 

                  if (oe0_nc_cnt gt 0) then summed[oe0_nc_indx] = NO_CNTS
                  if (oe2_nc_cnt gt 0) then summed[oe2_nc_indx] = NO_CNTS

                  if (he0_nc_cnt gt 0) then summed[he0_nc_indx] = NO_CNTS
                  if (he2_nc_cnt gt 0) then summed[he2_nc_indx] = NO_CNTS

                  ptr = ptr_new (summed)
                  l_event.amDmpPtr = ptr

                  if (l_setup.l_log eq 'Y') then begin
                     l_Conv2Log, summed, event_log, event_min, event_max
                     summed = event_log
                  endif

                  ptr = ptr_new (summed)
                  l_event.all_masses = ptr
               endif else begin
                   ;------------------------------------------------------------
                   ; hydrogen and oxygen using just energy 1 and 2
                   ;------------------------------------------------------------
                   if (l_event_setup.energy1 eq 'Y') AND (l_event_setup.energy2 eq 'Y') then begin
                      title1 = '30eV-300eV'

                      Hsum = (*l_event.hydrogen_e1_flux) + (*l_event.hydrogen_e2_flux) 
                      Osum = (*l_event.oxygen_e1_flux) + (*l_event.oxygen_e2_flux) 
                      summed = Hsum + Osum 

                      if (oe1_nd_cnt gt 0) then summed[oe1_nd_indx] = NO_DATA 
                      if (oe2_nd_cnt gt 0) then summed[oe2_nd_indx] = NO_DATA 

                      if (he1_nd_cnt gt 0) then summed[he1_nd_indx] = NO_DATA 
                      if (he2_nd_cnt gt 0) then summed[he2_nd_indx] = NO_DATA 

                      if (oe1_nc_cnt gt 0) then summed[oe1_nc_indx] = NO_CNTS
                      if (oe2_nc_cnt gt 0) then summed[oe2_nc_indx] = NO_CNTS

                      if (he1_nc_cnt gt 0) then summed[he1_nc_indx] = NO_CNTS
                      if (he2_nc_cnt gt 0) then summed[he2_nc_indx] = NO_CNTS

                      ptr = ptr_new (summed)
                      l_event.amDmpPtr = ptr

                      if (l_setup.l_log eq 'Y') then begin
                         l_Conv2Log, summed, event_log, event_min, event_max
                         summed = event_log
                      endif

                      ptr = ptr_new (summed)
                      l_event.all_masses = ptr
                   endif else begin ;--- E1,E2 ---
                      ;------------------------------------------------------------
                      ; hydrogen and oxygen just energy 0 
                      ;------------------------------------------------------------
                      if (l_event_setup.energy0 eq 'Y') then begin
                         title1 = '10eV-30eV'

                         Hsum = (*l_event.hydrogen_e0_flux)
                         Osum = (*l_event.oxygen_e0_flux)
                         summed = Hsum + Osum 

                         if (oe0_nd_cnt gt 0) then summed[oe0_nd_indx] = NO_DATA 
                         if (he0_nd_cnt gt 0) then summed[he0_nd_indx] = NO_DATA 
                         if (oe0_nc_cnt gt 0) then summed[oe0_nc_indx] = NO_CNTS
                         if (he0_nc_cnt gt 0) then summed[he0_nc_indx] = NO_CNTS

                         ptr = ptr_new (summed)
                         l_event.amDmpPtr = ptr

                         if (l_setup.l_log eq 'Y') then begin
                            l_Conv2Log, summed, event_log, event_min, event_max
                            summed = event_log
                         endif


                         ptr = ptr_new (summed)
                         l_event.all_masses = ptr
                      endif else begin
                         ;------------------------------------------------------------
                         ; hydrogen and oxygen just energy 1 
                         ;------------------------------------------------------------
                         if (l_event_setup.energy1 eq 'Y') then begin
                            title1 = '30eV-100eV'

                            Hsum = (*l_event.hydrogen_e1_flux)
                            Osum = (*l_event.oxygen_e1_flux)
                            summed = Hsum + Osum 

                            if (oe1_nd_cnt gt 0) then summed[oe1_nd_indx] = NO_DATA 
                            if (he1_nd_cnt gt 0) then summed[he1_nd_indx] = NO_DATA 
                            if (oe1_nc_cnt gt 0) then summed[oe1_nc_indx] = NO_CNTS
                            if (he1_nc_cnt gt 0) then summed[he1_nc_indx] = NO_CNTS

                            ptr = ptr_new (summed)
                            l_event.amDmpPtr = ptr

                            if (l_setup.l_log eq 'Y') then begin
                               l_Conv2Log, summed, event_log, event_min, event_max
                               summed = event_log
                            endif

                            ptr = ptr_new (summed)
                            l_event.all_masses = ptr
                         endif else begin
                            ;------------------------------------------------------------
                            ; hydrogen and oxygen just energy 2 
                            ;------------------------------------------------------------
                            if (l_event_setup.energy2 eq 'Y') then begin
                              title1 = '100eV-300eV'

                              Hsum = (*l_event.hydrogen_e2_flux) 
                              Osum = (*l_event.oxygen_e2_flux) 
                              summed = Hsum + Osum 

                              if (oe2_nd_cnt gt 0) then summed[oe2_nd_indx] = NO_DATA 
                              if (he2_nd_cnt gt 0) then summed[he2_nd_indx] = NO_DATA 
                              if (oe2_nc_cnt gt 0) then summed[oe2_nc_indx] = NO_CNTS
                              if (he2_nc_cnt gt 0) then summed[he2_nc_indx] = NO_CNTS

                              ptr = ptr_new (summed)
                              l_event.amDmpPtr = ptr

                              if (l_setup.l_log eq 'Y') then begin
                                 l_Conv2Log, summed, event_log, event_min, event_max
                                 summed = event_log
                              endif

                              ptr = ptr_new (summed)
                              l_event.all_masses = ptr
                            endif ;--- E2 only ---
                         endelse ;--- E1 only ---
                      endelse ;--- E0 only ---
                  endelse ;---E1,E2 ---
               endelse   ;--- E0,E2 ---
             endelse  ;--- E0,E1 ---
           endelse  ;--- all energies ---

      endif else begin

         ;------------------------------------------------------------
         ; just hydrogen
         ;------------------------------------------------------------
         if (l_event_setup.hydrogen eq 'Y') then begin
            title = 'Ev H '
            if (l_event_setup.background eq 'Y') then begin
                title = 'Ev H Sc/Bk Corr'
             endif
          
            ;------------------------------------------------------------
            ; just hydrogen, all energies
            ;------------------------------------------------------------
            if (l_event_setup.energy0 eq 'Y') AND (l_event_setup.energy1 eq 'Y') AND $
               (l_event_setup.energy2 eq 'Y') then begin

               summed = (*l_event.hydrogen_e0_flux) + (*l_event.hydrogen_e1_flux) + $
                        (*l_event.hydrogen_e2_flux)
               title1 = '10eV-300eV'

               if (he0_nd_cnt gt 0) then summed[he0_nd_indx] = NO_DATA
               if (he0_nc_cnt gt 0) then summed[he0_nc_indx] = NO_CNTS

               if (he1_nd_cnt gt 0) then summed[he1_nd_indx] = NO_DATA
               if (he1_nc_cnt gt 0) then summed[he1_nc_indx] = NO_CNTS

               if (he2_nd_cnt gt 0) then summed[he2_nd_indx] = NO_DATA
               if (he2_nc_cnt gt 0) then summed[he2_nc_indx] = NO_CNTS

               ptr = ptr_new (summed)
               l_event.amDmpPtr = ptr

               if (l_setup.l_log eq 'Y') then begin
                  l_Conv2Log, summed, event_log, event_min, event_max
                  summed = event_log
               endif

               ptr = ptr_new (summed)
               l_event.all_masses = ptr

             endif else begin
                ;------------------------------------------------------------
                ; just hydrogen, energies 0 and 1
                ;------------------------------------------------------------
                if (l_event_setup.energy0 eq 'Y') AND (l_event_setup.energy1 eq 'Y') then begin 
                   summed = (*l_event.hydrogen_e0_flux) + (*l_event.hydrogen_e1_flux)
                   title1 = '10eV-100eV'

                   if (he0_nd_cnt gt 0) then summed[he0_nd_indx] = NO_DATA
                   if (he0_nc_cnt gt 0) then summed[he0_nc_indx] = NO_CNTS

                   if (he1_nd_cnt gt 0) then summed[he1_nd_indx] = NO_DATA
                   if (he1_nc_cnt gt 0) then summed[he1_nc_indx] = NO_CNTS

                   ptr = ptr_new (summed)
                   l_event.amDmpPtr = ptr

                   if (l_setup.l_log eq 'Y') then begin
                      l_Conv2Log, summed, event_log, event_min, event_max
                      summed = event_log
                   endif

                   ptr = ptr_new (summed)
                   l_event.all_masses = ptr
 
                 endif else begin
                     ;------------------------------------------------------------
                     ; just hydrogen, energies 0 and 2 
                     ;------------------------------------------------------------
                     if (l_event_setup.energy0 eq 'Y') AND (l_event_setup.energy2 eq 'Y') then begin
                        summed = (*l_event.hydrogen_e0_flux) + (*l_event.hydrogen_e2_flux)
                        title1 = '10eV-300eV'

                        if (he0_nd_cnt gt 0) then summed[he0_nd_indx] = NO_DATA
                        if (he0_nc_cnt gt 0) then summed[he0_nc_indx] = NO_CNTS

                        if (he2_nd_cnt gt 0) then summed[he2_nd_indx] = NO_DATA
                        if (he2_nc_cnt gt 0) then summed[he2_nc_indx] = NO_CNTS
 
                        ptr = ptr_new (summed)
                        l_event.amDmpPtr = ptr

                        if (l_setup.l_log eq 'Y') then begin
                           l_Conv2Log, summed, event_log, event_min, event_max
                           summed = event_log
                        endif

                        ptr = ptr_new (summed)
                        l_event.all_masses = ptr

                     endif else begin
                           ;------------------------------------------------------------
                           ; just hydrogen, energies 1 and 2 
                           ;------------------------------------------------------------
                           if (l_event_setup.energy1 eq 'Y') AND $
                              (l_event_setup.energy2 eq 'Y') then begin 
                              summed = (*l_event.hydrogen_e1_flux) + (*l_event.hydrogen_e2_flux)
                              title1 = '30eV-300eV'

                              if (he1_nd_cnt gt 0) then summed[he1_nd_indx] = NO_DATA
                              if (he1_nc_cnt gt 0) then summed[he1_nc_indx] = NO_CNTS

                              if (he2_nd_cnt gt 0) then summed[he2_nd_indx] = NO_DATA
                              if (he2_nc_cnt gt 0) then summed[he2_nc_indx] = NO_CNTS

                              ptr = ptr_new (summed)
                              l_event.amDmpPtr = ptr

                              if (l_setup.l_log eq 'Y') then begin
                                 l_Conv2Log, summed, event_log, event_min, event_max
                                 summed = event_log
                              endif

                              ptr = ptr_new (summed)
                              l_event.all_masses = ptr

                           endif ;--- end He1, He2 ---
                     endelse ;--- end He0, He2 ---
                 endelse ;--- end He0, He1 ---
              endelse ;--- He0,He1,He2 ---

         endif else begin ;--- end hydrogen --- 

            ;------------------------------------------------------------
            ; just oxygen
            ;------------------------------------------------------------
            if (l_event_setup.oxygen eq 'Y') then begin

               title = 'Ev O '
               if (l_event_setup.background eq 'Y') then begin
                   title = 'Ev O Sc/Bk Corr'
               endif
          
               ;------------------------------------------------------------
               ; just oxygen, all energies
               ;------------------------------------------------------------
               if (l_event_setup.energy0 eq 'Y') AND (l_event_setup.energy1 eq 'Y') AND $
                  (l_event_setup.energy2 eq 'Y') then begin

                  summed = (*l_event.oxygen_e0_flux) + (*l_event.oxygen_e1_flux) + $
                           (*l_event.oxygen_e2_flux)
                  title1 = '10eV-300ev'

                  if (oe0_nd_cnt gt 0) then summed[oe0_nd_indx] = NO_DATA
                  if (oe0_nc_cnt gt 0) then summed[oe0_nc_indx] = NO_CNTS

                  if (oe1_nd_cnt gt 0) then summed[oe1_nd_indx] = NO_DATA
                  if (oe1_nc_cnt gt 0) then summed[oe1_nc_indx] = NO_CNTS

                  if (oe2_nd_cnt gt 0) then summed[oe2_nd_indx] = NO_DATA
                  if (oe2_nc_cnt gt 0) then summed[oe2_nc_indx] = NO_CNTS

                  ptr = ptr_new (summed)
                  l_event.amDmpPtr = ptr

                  if (l_setup.l_log eq 'Y') then begin
                      l_Conv2Log, summed, event_log, event_min, event_max
                      summed = event_log
                  endif

                  ptr = ptr_new (summed)
                  l_event.all_masses = ptr

               endif else begin
                 ;------------------------------------------------------------
                 ; just oxygen, energies 0 and 1
                 ;------------------------------------------------------------
                 if (l_event_setup.energy0 eq 'Y') AND (l_event_setup.energy1 eq 'Y') then begin 
                     summed = (*l_event.oxygen_e0_flux) + (*l_event.oxygen_e1_flux)
                     title1 = '10eV-100eV'

                     if (oe0_nd_cnt gt 0) then summed[oe0_nd_indx] = NO_DATA
                     if (oe0_nc_cnt gt 0) then summed[oe0_nc_indx] = NO_CNTS

                     if (oe1_nd_cnt gt 0) then summed[oe1_nd_indx] = NO_DATA
                     if (oe1_nc_cnt gt 0) then summed[oe1_nc_indx] = NO_CNTS

                     ptr = ptr_new (summed)
                     l_event.amDmpPtr = ptr

                     if (l_setup.l_log eq 'Y') then begin
                         l_Conv2Log, summed, event_log, event_min, event_max
                         summed = event_log
                     endif

                     ptr = ptr_new (summed)
                     l_event.all_masses = ptr
 
                 endif else begin
                     ;------------------------------------------------------------
                     ; just oxygen, energies 0 and 2 
                     ;------------------------------------------------------------
                     if (l_event_setup.energy0 eq 'Y') AND (l_event_setup.energy2 eq 'Y') then begin
                        summed = (*l_event.oxygen_e0_flux) + (*l_event.oxygen_e2_flux)
                        title1 = '10eV-300eV'

                        if (oe0_nd_cnt gt 0) then summed[oe0_nd_indx] = NO_DATA
                        if (oe0_nc_cnt gt 0) then summed[oe0_nc_indx] = NO_CNTS

                        if (oe2_nd_cnt gt 0) then summed[oe2_nd_indx] = NO_DATA
                        if (oe2_nc_cnt gt 0) then summed[oe2_nc_indx] = NO_CNTS

                        ptr = ptr_new (summed)
                        l_event.amDmpPtr = ptr

                        if (l_setup.l_log eq 'Y') then begin
                            l_Conv2Log, summed, event_log, event_min, event_max
                            summed = event_log
                        endif

                        ptr = ptr_new (summed)
                        l_event.all_masses = ptr

                     endif else begin
                        ;------------------------------------------------------------
                        ; just oxygen, energies 1 and 2 
                        ;------------------------------------------------------------
                        if (l_event_setup.energy1 eq 'Y') AND $
                           (l_event_setup.energy2 eq 'Y') then begin 
                           summed = (*l_event.oxygen_e1_flux) + (*l_event.oxygen_e2_flux)
                           title1 = '30eV-300eV'

                              if (oe1_nd_cnt gt 0) then summed[oe1_nd_indx] = NO_DATA
                              if (oe1_nc_cnt gt 0) then summed[oe1_nc_indx] = NO_CNTS

                              if (oe2_nd_cnt gt 0) then summed[oe2_nd_indx] = NO_DATA
                              if (oe2_nc_cnt gt 0) then summed[oe2_nc_indx] = NO_CNTS

                              ptr = ptr_new (summed)
                              l_event.amDmpPtr = ptr

                              if (l_setup.l_log eq 'Y') then begin
                                 l_Conv2Log, summed, event_log, event_min, event_max
                                 summed = event_log
                              endif

                              ptr = ptr_new (summed)
                              l_event.all_masses = ptr

                         endif ;--- end Oe1,Oe2 ---
                     endelse ;--- end Oe0,Oe2 ---
                 endelse ;--- end Oe0,Oe1 ---
               endelse ;--- end Oe0,Oe1,Oe2 ---
          endif ;--- end oxygen ---
      endelse ;--- end else hydrogen ---
      endelse ;--- hydrogen & oxygen combo ---

      ;------------------------------------------------------------
      ; plot the event data
      ;------------------------------------------------------------
      l_EventPlot, l_event_setup, $
                   summed, $
                   l_pltpos, $
                   nplots, $
                   windx, $
                   pages, $
                   plotted, $
                   l_setup, $
                   l_oa, $
                   data_format, $
                   l_rgb_table, $
                   colorbar, $
                   l_hskp, $
                   (*l_event.filledTimePtr), $
                   event_min, $
                   event_max, $
                   title, title1

   endif else begin ;--- end merge ---
      ;------------------------------------------------------------
      ; No merging of masses or energies, separate plots 
      ;------------------------------------------------------------
      if (l_event_setup.hydrogen eq 'Y') then begin

         if (l_event_setup.energy0 eq 'Y') then begin
             title = 'Ev H E0'
             title1 = '10eV-30eV'
             if (l_setup.l_log eq 'Y') then begin
                l_Conv2Log, (*l_event.hydrogen_e0_flux), evData, event_min, event_max
             endif else begin
                evData = (*l_event.hydrogen_e0_flux)
                event_min = (min (*l_event.hydrogen_e0_flux) > 0.0)
                event_max = max (*l_event.hydrogen_e0_flux)
             endelse 
             l_EventPlot, l_event_setup, $
                          evData, $
                          l_pltpos, $
                          nplots, $
                          windx, $
                          pages, $
                          plotted, $
                          l_setup, $
                          l_oa, $
                          data_format, $
                          l_rgb_table, $
                          colorbar, $
                          l_hskp, $
                          (*l_event.filledTimePtr), $
                          event_min, $
                          event_max, $
                          title, title1
         endif

         if (l_event_setup.energy1 eq 'Y') then begin
             title = 'Ev H E1'
             title1 = '30eV-100eV'
             if (l_setup.l_log eq 'Y') then begin
                l_Conv2Log, (*l_event.hydrogen_e1_flux), evData, event_min, event_max
             endif else begin
                evData = (*l_event.hydrogen_e1_flux)
                event_min = (min (*l_event.hydrogen_e1_flux) > 0.0)
                event_max = max (*l_event.hydrogen_e1_flux)
             endelse 
             l_EventPlot, l_event_setup, $
                          evData, $
                          l_pltpos, $
                          nplots, $
                          windx, $
                          pages, $
                          plotted, $
                          l_setup, $
                          l_oa, $
                          data_format, $
                          l_rgb_table, $
                          colorbar, $
                          l_hskp, $
                          (*l_event.filledTimePtr), $
                          event_min, $
                          event_max, $
                          title, title1
         endif

         if (l_event_setup.energy2 eq 'Y') then begin
             title = 'Ev H E2'
             title1 = '100eV-300eV'
             if (l_setup.l_log eq 'Y') then begin
                l_Conv2Log, (*l_event.hydrogen_e2_flux), evData, event_min, event_max
             endif else begin
                evData = (*l_event.hydrogen_e2_flux)
                event_min = (min (*l_event.hydrogen_e2_flux) > 0.0)
                event_max = max (*l_event.hydrogen_e2_flux)
             endelse 
             l_EventPlot, l_event_setup, $
                          evData, $
                          l_pltpos, $
                          nplots, $
                          windx, $
                          pages, $
                          plotted, $
                          l_setup, $
                          l_oa, $
                          data_format, $
                          l_rgb_table, $
                          colorbar, $
                          l_hskp, $
                          (*l_event.filledTimePtr), $
                          event_min, $
                          event_max, $
                          title, title1
         endif

      endif ;--- end hydrogen ---

      if (l_event_setup.oxygen eq 'Y') then begin
         if (l_event_setup.energy0 eq 'Y') then begin
             title = 'Ev O E0'
             title1 = '10eV-30eV'
             if (l_setup.l_log eq 'Y') then begin
                l_Conv2Log, (*l_event.oxygen_e0_flux), evData, event_min, event_max
             endif else begin
                evData = (*l_event.oxygen_e0_flux)
                event_min = (min (*l_event.oxygen_e0_flux) > 0.0)
                event_max = max (*l_event.oxygen_e0_flux)
             endelse 
             l_EventPlot, l_event_setup, $
                          evData, $
                          l_pltpos, $
                          nplots, $
                          windx, $
                          pages, $
                          plotted, $
                          l_setup, $
                          l_oa, $
                          data_format, $
                          l_rgb_table, $
                          colorbar, $
                          l_hskp, $
                          (*l_event.filledTimePtr), $
                          event_min, $
                          event_max, $
                          title, title1
         endif

         if (l_event_setup.energy1 eq 'Y') then begin
             title = 'Ev O E1'
             title1 = '30eV-100eV'
             if (l_setup.l_log eq 'Y') then begin
                l_Conv2Log, (*l_event.oxygen_e1_flux), evData, event_min, event_max
             endif else begin
                evData = (*l_event.oxygen_e1_flux)
                event_min = (min (*l_event.oxygen_e1_flux) > 0.0)
                event_max = max (*l_event.oxygen_e1_flux)
             endelse 
             l_EventPlot, l_event_setup, $
                          evData, $
                          l_pltpos, $
                          nplots, $
                          windx, $
                          pages, $
                          plotted, $
                          l_setup, $
                          l_oa, $
                          data_format, $
                          l_rgb_table, $
                          colorbar, $
                          l_hskp, $
                          (*l_event.filledTimePtr), $
                          event_min, $
                          event_max, $
                          title, title1
         endif

         if (l_event_setup.energy2 eq 'Y') then begin
             title = 'Ev O E2'
             title1 = '100eV-300eV'
             if (l_setup.l_log eq 'Y') then begin
                l_Conv2Log, (*l_event.oxygen_e2_flux), evData, event_min, event_max
             endif else begin
                evData = (*l_event.oxygen_e2_flux)
                event_min = (min (*l_event.oxygen_e2_flux) > 0.0)
                event_max = max (*l_event.oxygen_e2_flux)
             endelse 
             l_EventPlot, l_event_setup, $
                          evData, $
                          l_pltpos, $
                          nplots, $
                          windx, $
                          pages, $
                          plotted, $
                          l_setup, $
                          l_oa, $
                          data_format, $
                          l_rgb_table, $
                          colorbar, $
                          l_hskp, $
                          (*l_event.filledTimePtr), $
                          event_min, $
                          event_max, $
                          title, title1
         endif
           
       endif ;--- oxygen ---
   endelse ;--- merge ---

   endif ;--- use event ---


RETURN
END


;-------------------------------------------------------------------------------
;  Function : l_BinnedEventProc
;
;  Description:  Main controlling routine to processing binned or event data
;    based on user selection.
;
;  Return Value:  type = int
;  Value                         Description
;  -------------------------     ------------------------------
;  SUCCESS                       successful completion
;  FAILURE                       unsuccessful completion
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_setup               struct        I        general setup parameters
;  l_setup_file          string        I        runtime setup file name
;  l_pltpos              struct        I        general plotting parameters
;  l_rgb_table           struct        I        red, green, blue color table values
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;     PLOTDEFS         MAX_PLOTS        int         G         max # plots per page
;                      MAX_WEB_SPINS    int         G         max # spins per plot
;                                                             when using web interface
;                      NO_CNTS          double      G         no counts indicator
;                      NO_DATA          double      G         no data indicator
;     ERROR_TYPES      SUCCESS          int         G         successful completion
;                      FAILURE          int         G         unsuccessful completion
;                      WARNING          int         G         warning message
;                      INFO             int         G         informational message
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               01/02           v1.0.8          Original implementation
;-------------------------------------------------------------------------------

FUNCTION l_BinnedEventProc, l_setup, l_setup_file, l_pltpos, l_rgb_table
COMMON PLOTDEFS, MAX_PLOTS, MAX_WEB_SPINS, NO_CNTS, NO_DATA
COMMON ERROR_TYPES, SUCCESS, FAILURE, WARNING, INFO


   ;-------------------------------------------------
   ; the maximum number of spins to plot if plotting
   ; via the web
   ;-------------------------------------------------

   nele      = 0L
   year      = strmid(systime(0), 20, 23)

   l_setup.l_filedesc = 'sp'

   ;-------------------------------------------------
   ; read the OA data
   ;-------------------------------------------------
   status = l_OARead (l_oa, $
                      l_setup.l_start_dtime, $
                      l_setup.l_stop_dtime, $
                      l_pltpos)
   if (status eq FAILURE) then return, FAILURE


   ;---------------------------------------------------
   ; read the housekeeping data
   ;---------------------------------------------------
   status = l_HskpGetData (l_setup, $
                  l_setup_file, $
                  l_pltpos, $
                  l_hskp_setup, $
                  l_hskp)
   if (status eq FAILURE) then return, FAILURE


   ;---------------------------------------------------
   ; determine if binned or event data selected
   ;---------------------------------------------------
   status = l_BinnedEventSelect (l_setup, l_setup_file, l_pltpos, $
                        l_roi0, l_binned, l_binned_setup, l_event, l_event_setup, $
                        l_memd, l_memd_setup, l_evsp, l_evsp_setup, l_hskp, l_oa, $
                        davail)
   if (status eq FAILURE) then return, FAILURE



   ;---------------------------------------------------
   ; set loop for postscript
   ;---------------------------------------------------
   nloops = 1L
   if (l_setup.l_postscript eq 'Y') then nloops = 2L

   ;---------------------------------------------------
   ; generate a spectrogram
   ;---------------------------------------------------
   l_setup.l_line_plots = 'N'
   if (l_setup.l_line_plots eq 'N') then begin
      for i=0L, nloops-1L do begin

         l_pltpos.ps = 'N'
         pages     = 0L
         nplots    = 0L
         plotted   = 0L

        ;---------------------------------------------------
        ; power supply bars
        ;---------------------------------------------------
         if (l_setup.l_pwrsup_bars eq 'Y') then begin
            nplots    = 1L
         endif

         ;---------------------------------------------------
         ; set plot title
         ;---------------------------------------------------
         if (i eq 1) and (l_setup.l_postscript eq 'Y') then begin
            if (l_binned_setup.use_binned eq 'Y') then begin
               l_pltpos.plot_hdr_title = '!5Binned Data'
            endif else begin
               l_pltpos.plot_hdr_title = '!5Event Data'
            endelse

            l_pltpos.ps = 'Y'
            l_GenFileName, l_setup, '0'
            l_pltpos.filename = l_setup.l_prod_dst + '/' + $
                                l_setup.l_filename + '.ps'
         endif

         if (l_binned_setup.use_binned eq 'Y') then begin
            nplots    = nplots + l_binned_setup.nplots 
         endif else begin
            if (l_event_setup.use_event eq 'Y') then begin
               nplots = nplots + l_event_setup.nplots
            endif
         endelse

         ;---------------------------------------------------
         ; setup display and plotting environment
         ;---------------------------------------------------
         l_SetUpZBuffer, l_pltpos, $
                         l_setup, $
                         l_rgb_table, $
                         colorbar, $
                         nplots


         l_pltpos.linePlots = l_setup.l_line_plots


         npages = CEIL(nplots/MAX_PLOTS)
         if (nplots - max_plots gt 0) then npages = npages + 1
         if (npages eq 0) then npages = 1
         if (npages gt 0) then windx = intarr (npages)

         ;-------------------------------------------------
         ; do a spectrogram plot
         ; determine the number of plots and calculate the
         ; size of each plot to fit the page
         ;-------------------------------------------------
         if (npages gt 1) then begin
            plots_per_page = 5
            if (l_setup.l_pwrsup_bars eq 'Y') then plots_per_page = 6
         endif else begin
            plots_per_page = nplots
         endelse


         ;-------------------------------------------------
         ; determine the size of each plot 
         ;-------------------------------------------------
         l_AdjPlotSize, l_pltpos, l_setup, plots_per_page

         ;-------------------------------------------------
         ; draw power supply bars
         ;-------------------------------------------------
         if (l_setup.l_pwrsup_bars eq 'Y') then begin
            l_DrawOpBars, l_hskp, l_setup, l_pltpos, l_rgb_table, l_oa
            nplots = nplots - 1L
            plotted   = 1L
         endif


         ;-------------------------------------------------
         ; plot the selected binned or event data 
         ;-------------------------------------------------
         l_BinnedEventPlot, l_setup, l_binned_setup, $ 
                            l_event_setup, l_binned, l_event, $
                            l_pltpos, l_roi0, l_oa, l_hskp, $
                            l_rgb_table, nplots, windx, plotted, $
                            pages, colorbar


         ;-------------------------------------------------
         ; close all opened devices
         ;-------------------------------------------------
         if (i eq 1) and (l_setup.l_postscript eq 'Y') then begin
             device, /close_file
         endif

         if (!D.NAME eq 'Z') then begin
            if (l_setup.l_via_web eq 'Y') or (l_setup.l_gen_sum eq 'Y') then begin
               device, /close
            endif
         endif

     endfor
  endif


  ;-------------------------------------------------
  ; create an ascii dump of the binned data
  ;-------------------------------------------------
  if (l_setup.l_dumps eq 'Y') then begin

     ;-------------------------------
     ; create text dump filename
     ;-------------------------------
     if (l_binned_setup.use_binned eq 'Y') then begin
        l_setup.l_filedesc = 'bin'
     endif else begin
        l_setup.l_filedesc = 'ev'
     endelse

     l_GenFileName, l_setup, '0'
     filename = l_setup.l_prod_dst + '/' + l_setup.l_filename + '.dmp'

     ;-------------------------------
     ; open text dump file
     ;-------------------------------

     openw, ofd, filename, /get_lun
     
     l_DumpHdrInfo, ofd, l_setup, l_pltpos, filename

     if (l_binned_setup.use_binned eq 'Y') then begin
        l_BinnedDmpSpectData, ofd, l_binned, l_binned_setup, l_roi0, l_setup
     endif

     if (l_event_setup.use_event eq 'Y') then begin
        l_EventDmpSpectData, ofd, l_event, l_event_setup, l_setup

        if (l_event_setup.merge eq 'Y') AND (l_event_setup.background eq 'Y') then begin
           l_GenFileName, l_setup, '0'
           fname = l_setup.l_prod_dst + '/' + l_setup.l_filename + 'bk' + '.dmp'
           openw, ofd1, fname, /get_lun
           l_EventDumpBKData, ofd1, l_event, l_event_setup, l_setup
           close, ofd1
           free_lun, ofd1
        endif
     endif

     close, ofd
     free_lun, ofd

   endif

   ;----------------------------------------------------
   ; free memory allocated for either the binned or
   ; event data
   ;----------------------------------------------------
   l_BinnedEventTerm, l_roi0, l_binned, l_binned_setup, $ 
                      l_event, l_event_setup, l_memd, $
                      l_evsp, l_hskp, l_oa, l_setup, davail


   RETURN, SUCCESS

END


