
;-------------------------------------------------------------------------------
;
;  Unit Name : l_ApoPerTimes.pro
;
;  Purpose   : Reads the apogee.txt and perigee.txt files that are generated
;              by David Simpson.  These files are used for long term trending
;              plots.  These files need to be updated each time a new long term
;              trending plot is requested and the apogee/perigee times are 
;              are not listed or updated with the most recent data.
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;   ELH             04/23/03         v1.0.11         Original implementation
;
;
;  File Revision Number:   %I%
;  File Last Modified  :   %E%  %T%
;
;------------------------------------------------------------ 

;-------------------------------------------------------------------------------
;  Procedure: l_ReadApogeeTimes
; 
;  Description:  Reads the apogee times from the apogee.txt file and parses the
;                data in each line into separate variables.
;
;  Return Value:  type = None
;  Value                         Description
;  -------------------------     ------------------------------
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  start_time            array         I        requested start time of plot
;  stop_time             array         I        requested stop time of plot
;  apogee_times          array         O        list of apogee times
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;     EPOCH_DATA       EPOCH_YR        int          G         launch year
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03        v1.0.11         Original implementation
;-------------------------------------------------------------------------------

PRO l_ReadApogeeTimes, start_time, stop_time, apogee_times
COMMON EPOCH_DATA, EPOCH_YR

   time_array = intarr (5)

   ;-------------------------------------------
   ; convert the start/stop time into seconds
   ;-------------------------------------------
   start_secs = ConvArrTimeToSecs(start_time)
   stop_secs = ConvArrTimeToSecs(stop_time)

   ;-------------------------------------------
   ; determine the number of days between
   ; requested time period
   ;-------------------------------------------
   days = round (((stop_secs - start_secs)/86400) * 3L)
   
   apogee_times = replicate ({DATE_TIME_STRUCT}, days)

   ;-------------------------------------------
   ; open the apogee.txt file
   ;-------------------------------------------
   openr, inlun, 'apogee.txt', /get_lun

   line = ' ' 

   header = strarr (9)
   readf, inlun, header
 
   idx = 0L

   ;-------------------------------------------
   ; read the file, line by line
   ;-------------------------------------------
   while not EOF (inlun) do begin
       readf, inlun, line
       if (line ne '') then begin
          ;-------------------------------------------
          ; parse the line, separate values into 
          ; different variables 
          ;-------------------------------------------
          parts = str_sep (line, ' ')
          date_parts = str_sep (parts[5], '/')
          time_parts = str_sep (parts[7], ':')
          doy_parts = parts [n_elements (parts) -1L]
          yr = date_parts[0]

          ;-------------------------------------------
          ; select the apogee times within the user
          ; requested time period
          ;-------------------------------------------
          if (yr ge start_time[0]) AND (yr le stop_time[0]) then begin
             time_array = [yr, doy_parts, time_parts[0], time_parts[1], time_parts[2]]
             current_time = ConvArrTimeToSecs(time_array)
             if (current_time ge start_secs) AND (current_time le stop_secs) then begin
;                print, 'time_array = ', time_array
                apogee_times[idx].year  = yr
                apogee_times[idx].doy   = doy_parts
                apogee_times[idx].month = date_parts[1] 
                apogee_times[idx].day   = date_parts[2]
                apogee_times[idx].hour  = time_parts[0] 
                apogee_times[idx].min   = time_parts[1]
                apogee_times[idx].sec   = time_parts[2]
                apogee_times[idx].time_n_secs = current_time 
;                print, 'apogee_times SECS = ', apogee_times[idx]
                idx = idx + 1L
             endif
          endif
       endif

   endwhile


   apogee_times = apogee_times[0:idx-1L]

   close, inlun
   free_lun, inlun

RETURN
END


;-------------------------------------------------------------------------------
;  Procedure: l_ReadPerigeeTimes
;
;  Description: Read the perigee times from the perigee.txt file and parse the
;               data line from the file into separate variables.
;
;  Return Value:  type = None
;  Value                         Description
;  -------------------------     ------------------------------
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  start_time            array         I        requested start time of plot
;  stop_time             array         I        requested stop time of plot
;  perigee_times         array         O        list of perigee times
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;     EPOCH_DATA       EPOCH_YR        int          G         launch year
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03        v1.0.11          Original implementation
;---------------------------------------------------------------------------------- 

PRO l_ReadPerigeeTimes, start_time, stop_time, perigee_times
COMMON EPOCH_DATA, EPOCH_YR

   time_array = intarr (5)

   ;-------------------------------------------
   ; convert user requested time into seconds
   ;-------------------------------------------
   start_secs = ConvArrTimeToSecs(start_time)
   stop_secs = ConvArrTimeToSecs(stop_time)

   ;-------------------------------------------
   ; determine the number of days/items within
   ; the user specified request time
   ;-------------------------------------------
   days = round (((stop_secs - start_secs)/86400) * 3L)
   
   perigee_times = replicate ({DATE_TIME_STRUCT}, days)

   ;-------------------------------------------
   ; open the perigee file
   ;-------------------------------------------
   openr, inlun, 'perigee.txt', /get_lun

   line = ' ' 

   header = strarr (9)
   readf, inlun, header
 
   idx = 0L

   ;-------------------------------------------
   ; read the perigee file line by line
   ;-------------------------------------------
   while not EOF (inlun) do begin
       readf, inlun, line
       if (line ne '') then begin
          ;-------------------------------------------
          ; parse each line read from file into separate
          ; variables
          ;-------------------------------------------
          parts = str_sep (line, ' ')
          date_parts = str_sep (parts[5], '/')
          time_parts = str_sep (parts[7], ':')
          doy_parts = parts[n_elements(parts) - 1L] 
          yr = date_parts[0]

          ;-------------------------------------------
          ; read perigee times that are within the user
          ; specified time range
          ;-------------------------------------------
          if (yr ge start_time[0]) AND (yr le stop_time[0]) then begin
             time_array = [yr, doy_parts, time_parts[0], time_parts[1], time_parts[2]]
             current_time = ConvArrTimeToSecs(time_array)
             if (current_time ge start_secs) AND (current_time le stop_secs) then begin
;                print, 'time_array = ', time_array
                perigee_times[idx].year  = yr
                perigee_times[idx].doy   = doy_parts
                perigee_times[idx].month = date_parts[1] 
                perigee_times[idx].day   = date_parts[2]
                perigee_times[idx].hour  = time_parts[0] 
                perigee_times[idx].min   = time_parts[1]
                perigee_times[idx].sec   = time_parts[2]
                perigee_times[idx].time_n_secs = current_time 
;                print, 'perigee_times SECS = ', perigee_times[idx]
                idx = idx + 1L
             endif
          endif
       endif

   endwhile

   perigee_times = perigee_times[0:idx-1L]

   close, inlun
   free_lun, inlun

RETURN
END
