/**************************************************************************/
/*  This routine prefilters the data in several stages.                   */
/*                                                                        */
/*  1) If data is to be given as log scaled then take the log of the data */
/*     and tread any data <= 0 as below tolerances                        */
/*                                                                        */
/*  2) If the data is pregridded then we can't remove data outside the    */
/*     the tolerances so just mark it bad in the grid mesh.               */
/*                                                                        */
/*  3) If this is a carteographic plot then any data which has latitudes  */
/*     outside the range -90.0 to 90.0 or longitudes outside the range    */
/*     -360.0 to 360.0. are marked as bad                                 */
/*                                                                        */
/*  4) The data is scanned for any data outside the tolerance ranges and  */
/*     this data is marked as bad                                         */
/*                                                                        */
/*  The routine may modify the input data arrays.  The intensity could    */
/*  have the log taken of it and the X values in cateographic plots could */
/*  be range corrected.                                                   */
/**************************************************************************/

#include <math.h>
#include "gph_str.h"
#include "gph_ansi.h"
#include "cont_str.h"
#include "gph_shortcut_defs.h"

static ByTe_1 rcsid[] = "$Id: PreFilter.c,v 1.1 1999/11/21 08:39:31 chris.gurgiolo.b2r Stab chrisg $";

void PreFilter (ReaL_4 *v, ReaL_4 *x, ReaL_4 *y, ReaL_4 *z)
{
   extern struct cont_info cinfo;
   extern struct cont_alloc cal;
   extern struct cont_disp cdisp;

   register ReaL_4 *f1, *f2, *fend;
   register ByTe_1  *c1, *cend;

   ReaL_4 LOW;

/**************************************************************************/
/* Set up data quality flag                                               */
/**************************************************************************/

   f1 = v;                                          /* ptr to intensities    */
   c1 = cal.dQ;                                     /* pointer to data qual  */
   cend = c1 + cinfo.N;                             /* end of data quality   */
   for ( ; c1 < cend; ++f1)                         /* loop over data qual   */
      *c1++ = (*f1 > cinfo.Utol || *f1 < cinfo.Ltol) ? 0 : 1;

/**************************************************************************/
/* Do tolerances                                                          */
/**************************************************************************/

   f1 = v;                                          /* ptr to intensities    */
   fend = f1 + cinfo.N;                             /* end of intensities    */
   switch (cinfo.data_type)
   {
       case 3:
          c1 = cal.bad;
          for ( ; f1 < fend; ++f1)                  /* loop over intensities */
              *c1++ = (*f1 > cinfo.Utol || *f1 < cinfo.Ltol) ? 1 : 2;
       break;
       case 4:
          c1 = cal.bad;
          for ( ; f1 < fend; ++f1)                 /* loop over intensities */
             *c1++ = (*f1 > cinfo.Utol || *f1 < cinfo.Ltol) ? 1 : 0;
       break;
   }

/**************************************************************************/
/*  Take log if necessary                                                 */
/**************************************************************************/

   if (cdisp.Cs == 1)                               /* log plot?             */
   {                                                /* yes                   */
      fend = v + cinfo.N;                           /* end of intensity      */
      f1 = v;                                       /* ptr to intensity      */
      c1 = cal.dQ;
      if (cdisp.IgnZero == 2) {                     /* BEG MARK <= 0 BAD     */
         for ( ; f1 < fend; ++f1, ++c1) {           /* BEG DATA LOOP         */
            if (*f1 > 0.0)                          /* data can have log?    */
               *f1 = log10(*f1);                    /* yes - take it         */
            else {                                  /* BEG NO LOG            */
               *f1 = -60;                           /* off end of world      */
               *c1 = 0;                             /* mark data bad         */
            }                                       /* END NO LOG            */
         }                                          /* END DATA LOOP         */
      } else {                                      /* BEG SET <= 0 TO LOW   */
         LOW = (cdisp.IgnZero == 0) ? -60.0 : log10 (cdisp.colmin / 5.0); 
         for ( ; f1 < fend; ++f1)                   /* loop over data        */
           *f1 = (*f1 > 0.0) ? log10(*f1) : LOW;    /* take log if can       */
      }                                             /* END SET <= 0 TO LOW   */
   }

   if (cdisp.Cs == 2)                              /* used to override log   */
      cdisp.Cs = 1;                                /* but data is log scaled */

/**************************************************************************/
/* Do carteographic data                                                  */
/**************************************************************************/

   if (cinfo.plot_fmt == 1)                         /* map plot?             */
   {                                                /* you got it            */
      if (cinfo.data_type != 3)                     /* not hollow grid       */
      {                                             /* BEG CHECK ALL VALS    */
         f1 = x;                                    /* X position pointer    */
         f2 = y;                                    /* Y position pointer    */
         c1 = cal.dQ;                               /* data quality pointer  */
         fend = x + cinfo.N;                        /* end of data           */
         for ( ; f1 < fend; ++f1, ++f2, ++c1)       /* loop over data        */
         {
            if ((*f2 >= -90. && *f2 <= 90.) &&      /* latitude or longitude */
                  (*f1 >= -360.0 && *f1 <= 360.0))  /* in range              */
            {                                       /* yes!                  */
               if (*f1 > 180.0)                     /* > upper long limit?   */
                  *f1 -= 360.0;                     /* yes - make it good    */
               if (*f1 < -180.0)                    /* < lower limit?        */
                  *f1 += 360.0;                     /* yes - make it good    */
            }
            else
               *c1 = 0;                             /* mark data as bad      */
         }
      }                                             /* END CHECK ALL VALS    */
      else                                          /* hollow grid           */
      {                                             /* BEG EDGE VALS         */
      }                                             /* END EDGE VALS         */
   }
}
