#include "gph_str.h"
#include "cont_str.h"
#include "gph_ansi.h"

static ByTe_1 rcsid[] = "$Id: get_map_cell.c,v 1.1 1999/11/21 08:39:31 chris.gurgiolo.b2r Stab chrisg $";

/************************************************************************/
/*  This subroutine returns the coordinates of one subcell in a larger  */
/*  cell when using map coordinates.  The largest cell coordinates are  */
/*  given by r such that                                                */
/*                                                                      */
/*                r[6],r[7]  x-------------x r[4],r[5]                  */
/*                           .             .                            */
/*                           .             .                            */
/*                           .             .                            */
/*                           .             .                            */
/*                           .             .                            */
/*                r[0],r[1]  x-------------x r[2],r[3]                  */
/*                                                                      */
/*  The position of the subcell is given by bot, side.  The subcell     */
/*  bot = 0, side = 0 is in the lower lefthand corner of the full cell. */
/*  Checks are made before subdividing the cell to make sure that it    */
/*  it doesn't straddle the 180,-180 discontinuity.  If it does, then   */
/*  longitude is corrected to 0-360 for processing and then returned to */
/*  the original range                                                  */
/*                                                                      */
/*  A cell straddles the discontinuity if one longitude is > 90.0 and   */
/*  another is found to be < -90.0.                                     */
/************************************************************************/


void get_map_cell(ByTe_4 bot, ByTe_4 side, ReaL_4 *r, ReaL_4 *ce)
{
   extern struct cont_disp cdisp;
   register ReaL_4 *f1, *f2, *fend;
   ReaL_4 sm;
   ReaL_4 S1, S2;
   ReaL_4 x[4];
   ByTe_1 edge = 0;

   sm = cdisp.smooth;

   /******************************************************************/
   /* Check to see if there are any longitudes which are > 90.0      */
   /******************************************************************/

   f1 = r;
   fend = r + 8;

   while (edge == 0 && f1 < fend)
   {
      if (*f1 > 90.0)
         edge = 1;
     f1 += 2;
   }

   /******************************************************************/
   /* If the cell may straddle the discontinuity then we must see if */
   /* any of the longitudes are < -90.0 and if so correct them.  If  */
   /* the cell does not straddle the boundary then no correction,    */ 
   /******************************************************************/

   if (edge)
   {
      for (f1 = r, f2 = x; f1 < fend; f1 += 2)
         *f2++ = (*f1 < -90.0) ? *f1 + 360.0 : *f1;
   }
   else
   {
      for (f1 = r, f2 = x; f1 < fend; f1 += 2)
         *f2++ = *f1;
   }

   /******************************************************************/
   /* Now find the x coordinates for the corners of this subcell.    */
   /*                                                                */
   /* First find the slopes of the lower and upper edges of the main */
   /* cell then use linear interpolation to find the edges of the    */
   /* subcell.  Cordinates are stored in the same order as for the   */
   /* parent cell but all x's are stored in first 4 array slots      */
   /******************************************************************/

   f1 = x; 
   f2 = ce;

   S1 = (*(f1+1) - *f1)/sm;
   S2 = (*(f1+2) - *(f1+3))/sm;

   *f2++ = S1 * bot + *f1; 
   *f2++ = S1 * (bot + 1) + *f1; 

   *f2++ = S2 * (bot + 1) + *(f1+3); 
   *f2++ = S2 * bot + *(f1+3); 

   /******************************************************************/
   /* Now find the y coordinates for the corners of this subcell.    */
   /*                                                                */
   /* First find the slopes of the left  and right edges of the main */
   /* cell then use linear interpolation to find the edges of the    */
   /* subcell.  Cordinates are stored in the same order as for the   */
   /* parent cell but all y's are stored in last  4 array slots      */
   /******************************************************************/

   S1 = (*(r+7) - *(r+1))/sm;
   S2 = (*(r+5) - *(r+3))/sm;

   *f2++ = S1 * side + *(r+1); 
   *f2++ = S2 * side + *(r+3); 

   *f2++ = S2 * (side + 1) + *(r+3); 
   *f2   = S1 * (side + 1) + *(r+1); 

   /******************************************************************/
   /* Put longitudes back into the  -180 to 180 range                */
   /******************************************************************/

   fend = ce + 4;
   for (f1 = ce; f1 < fend; ++f1)
      if (*f1 > 180.0)
         *f1 -= 360.0;
}
