/******************************************************************************
                           scsdmp.cpp  -  description                        
                             -------------------                              
    begin                : June 07 15:30 CET 2001
                                           
    author               : Hans Vaith
    email                : hav@mpe.mpg.de
    copyright            : (C) 2001 by Max-Planck-Institut fr extra-
                           terrestrische Physik, D-85740 Garching 

    Library for reading of Cluster-II SCS DMP files

*****************************************************************************
$Id: scsdmp.cpp,v 1.2 2003/08/27 09:23:42 hav Exp $
*****************************************************************************
$Log: scsdmp.cpp,v $
Revision 1.2  2003/08/27 09:23:42  hav
set correct values for data type and data source

Revision 1.1.1.1  2002/10/14 17:08:19  hav
c/c++ sources

Revision 1.5  2001/11/12 17:32:34  hav
?
/

Revision 1.4  2001/08/10 11:29:10  hav
no change

Revision 1.3  2001/07/25 16:03:32  hav
fixed missing $ at rcsid

Revision 1.2  2001/07/25 15:22:52  hav
added rcsid

Revision 1.1  2001/07/25 14:56:36  hav
Initial revision

******************************************************************************/

/******************** INCLUDE FILES ******************************************/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "scsdmp.h"
#include "openfile.h"
#include "filesize.h"

/******************** CONSTANTS AND MACROS ***********************************/

#define SCSDMP_HDR_SZ  64

/******************** TYPE DEFINITIONS **************************************/

typedef struct {
   FILE *Fp;
   long FileSize;
   long MaxPacket, PktLen;
   int Datasource, DataType;
   char FileName[MAXLEN];
} scsdmp_t;


/******************** FUNCTION PROTOTYPES ***********************************/

static void ProcessSCSDMPheader(char *buffer, tmheader *hp, long pktno);
static void PrintSCSDMPheader(tmheader hdr);

// ****************** GLOBAL STATIC DATA *************************************

// ****************** LOCAL STATIC DATA **************************************

static scsdmp_t scsdmp;
static char HdrBuf[SCSDMP_HDR_SZ];
//static char *DataTypeStr[] = { "NSD", "BSD", "BSD", "BSD", "HKD" };
static char *DataTypeStr[] = { "???", "NSD", "BSD", "???", "???" };
static tmheader FirstHeader;

static char rcsid[] = "$Id: scsdmp.cpp,v 1.2 2003/08/27 09:23:42 hav Exp $";

/******************** FUNCTION DEFINITIONS ***********************************/


// ===========================================================================
   void SCSDMP_OpenFile(const char *fn)
// ===========================================================================
// open file and set all data elements in static variable 'cdds'
// ***************************************************************************
{
   if (0) printf("%s\n", rcsid); // dummy instruction

   char *fname = "SCSDMP_OpenFile()";

   if (scsdmp.Fp != NULL) {
      fprintf(stderr, "%s: file already open\n", fname);
      exit(1);
   }

   // Open File, check file type and data offset
   // ==========================================
   strncpy(scsdmp.FileName, fn, sizeof(scsdmp.FileName)-1);
   scsdmp.FileName[sizeof(scsdmp.FileName)-1] = '\0';

   scsdmp.Fp = OpenFile(fn, OPF_RB, OPF_EXIT);

   // determine file size, number of tm packets and science tm mode (NSD/BSD)
   scsdmp.FileSize = FileSize(scsdmp.Fp);

   fread(HdrBuf, SCSDMP_HDR_SZ, 1, scsdmp.Fp);
   ProcessSCSDMPheader(HdrBuf, &FirstHeader, -1);

   //   printf("filesize : %ld\n", scsdmp.FileSize);
   //   printf("data packet size: %ld\n", FirstHeader.packet_length);

   scsdmp.PktLen    = FirstHeader.packet_length + SCSDMP_HDR_SZ;
   scsdmp.MaxPacket = scsdmp.FileSize / scsdmp.PktLen;

   //   printf("Number of packets : %ld\n", scsdmp.MaxPacket);

   switch (FirstHeader.data_type) {
   case 0:
   case 1:
   case 2: scsdmp.DataType = CL2TM_NSD_DATA; break;
   case 3:
   case 4:
   case 5: scsdmp.DataType = CL2TM_BSD_DATA; break;
   default : fprintf(stderr, "%s : unknown tm mode identifier "
		     "(0-2=NMx,3-5=BMx) : %d\n", fname, (int)FirstHeader.data_type);
             exit(1);
   } // endswitch

   if (HdrBuf[27] != 0) {
      fprintf(stderr, "%s: non-EDI data source in SCS DMP file\n", fname);
      exit(1);
   }

   rewind(scsdmp.Fp);
   //   fprintf(stderr, "temporary exit in %s\n", fname);
   //   exit(1);
}


// ===========================================================================
   void SCSDMP_CloseFile(void)
// ===========================================================================
{
   if (scsdmp.Fp!=NULL) {
      fclose(scsdmp.Fp);
      scsdmp.Fp = NULL;
   }
}


// ===========================================================================
   int SCSDMP_GetScId(void)
// ===========================================================================
// ***************************************************************************
{
   return 0;
}


// ===========================================================================
   int SCSDMP_GetDataSource(void)
// ===========================================================================
// ***************************************************************************
{
   if (!scsdmp.Fp) {
      fprintf(stderr, "SCSDMP_GetDataSource(): no file open yet!\n");
      exit(1);
   }

   return 0;
}

// ===========================================================================
   int SCSDMP_GetDataType(void)
// ===========================================================================
// ***************************************************************************
{
   if (!scsdmp.Fp) {
      fprintf(stderr, "SCSDMP_GetDataType(): no file open yet!\n");
      exit(1);
   }

   return scsdmp.DataType;
}


// ===========================================================================
   void SCSDMP_ReadReq(FILE *ofp)
// ===========================================================================
// Write requested data to ofp (must be open for writing)
// ***************************************************************************
{
   char *fname = "SCSDMP_ReadReq()";
   tmpacket tmpkt;

   // Check if file has been opened
   // =============================
   if (!scsdmp.Fp) {
      fprintf(stderr, "%s: no file open yet!\n", fname);
      exit(1);
   }

   // Read if there are data
   // ======================
   if (scsdmp.MaxPacket==0) {
      fprintf(stderr, "%s: no Data\n", fname);
      return;
   }
   
   // check if output is defined
   // ==========================
   if (ofp == NULL) {
      fprintf(stderr, "%s: ofp is NULL\n", fname);
      return;
   }

   //   fprintf(ofp, "%s\nEDI%s0\n", scsdmp.FileName, DataTypeStr[scsdmp.DataType]);           

   rewind(scsdmp.Fp);
   tmpkt.ofp = ofp;
   tmpkt.pktno = 0;

   while (1) {

      // Read a packet
      if (SCSDMP_ReadPacket(&tmpkt) == -1)  break ; // end of file

      // Write data to tmpkt.ofp
      SCSDMP_WritePacket(tmpkt);

      ++tmpkt.pktno;
   }

}


// ===========================================================================
   int SCSDMP_ReadPacket(tmpacket *tmptr)
// ===========================================================================
// returns -1 if EOF
// ***************************************************************************
{
   char hdrbuf[SCSDMP_HDR_SZ];
   static char tmbuf[BM3_SIZE + HK_SIZE];
   char *fname = "SCSDMP_ReadPacket()";
   int sz;

   tmptr->data = (short *)tmbuf;

   if (fseek(scsdmp.Fp, (tmptr->pktno)*(scsdmp.PktLen), SEEK_SET) != 0) {
      fprintf(stderr, "%s: fseek() error\n", fname);
      perror("");
      exit(1);
   }

   //  Read and process SCS DMP header
   if (fread(hdrbuf, SCSDMP_HDR_SZ, 1, scsdmp.Fp) != 1) {
      if (feof(scsdmp.Fp)) return -1;
      else {
         fprintf(stderr, "%s: SCS DMP header fread() error\n", fname);
	 exit(1);
      }
   }
   ProcessSCSDMPheader(hdrbuf, &(tmptr->hdr), tmptr->pktno);

   // Check size of following data packet
   if ((unsigned long)tmptr->hdr.packet_length > sizeof(tmbuf)) {
      fprintf(stderr, "%s: current packet (%ld) is too big: %ld bytes.\n"
                      "File pointer position after SCS DMP header: %ld\n"
		      "SCS DMP packet length (hdr+data) : %ld\n",
                       fname, tmptr->pktno, (long)(tmptr->hdr.packet_length),
		       (long)ftell(scsdmp.Fp),
		       scsdmp.PktLen);
      PrintSCSDMPheader(tmptr->hdr);
      exit(1);
   }

   // Check data packet size consistency
   if (tmptr->hdr.packet_length != (scsdmp.PktLen-SCSDMP_HDR_SZ) ) {
       fprintf(stderr, "unexpected data packet length: %ld\n"
                       "packet number                : %ld\n",
               tmptr->hdr.packet_length, tmptr->pktno);
       PrintSCSDMPheader(tmptr->hdr);
       exit(1);
   }

   // Read data packet
   if (fread(tmbuf, tmptr->hdr.packet_length, 1, scsdmp.Fp) != 1) {
      if (feof(scsdmp.Fp)) return -1;
      else {
         fprintf(stderr, "%s: SCS DMP data fread() error\n", fname);
         exit(1);
      }
   }

   // reorder by removing HK data
   //   fprintf(stderr, "need to remove HK data in SCSDMP_ReadPacket().\n");
   //   exit(1);
   if (scsdmp.DataType == CL2TM_NSD_DATA) {
      memmove(tmbuf+2*98, tmbuf+2*98+HK_SIZE, 8*98);
   } else if (scsdmp.DataType == CL2TM_BSD_DATA) {
      switch (tmptr->hdr.packet_length-HK_SIZE) {
      case BM1_SIZE : sz = 112; break;
      case BM2_SIZE : sz = 16; break;
      case BM3_SIZE : sz = 374; break;
      default : fprintf(stderr, "%s : unknown packet size : %ld. Skipping this packet...\n",
                        fname, (long)tmptr->hdr.packet_length);
                return 0;

      }
      memmove(tmbuf+9*sz, tmbuf+9*sz+34, 2*sz);
      memmove(tmbuf+11*sz, tmbuf+11*sz+HK_SIZE, 51*sz);
   }


   // adjust packet length
   tmptr->hdr.packet_length -= HK_SIZE;

   return 0;
}


// ===========================================================================
   void SCSDMP_WritePacket(tmpacket tmpkt)
// ===========================================================================
{
   scet *tp = &(tmpkt.hdr.sc_time);
   short *sp = tmpkt.data;

   if (tmpkt.ofp==NULL) {
      fprintf(stderr, "SCSDMP_WritePacket(): tmpkt.ofp is NULL\n");
      return;
   }

   if (tmpkt.pktno==0) {
      fprintf(tmpkt.ofp, "%s\nEDI%s0\n", scsdmp.FileName, DataTypeStr[scsdmp.DataType]);
   }

   // Write out packet time and data packet size (in bytes)
   fprintf(tmpkt.ofp, "%04hd.%03hd.%02hd.%02hd.%02hd.%03hd.%03hd\n",
                   tp->year, tp->doy, tp->hr, tp->min,
                   tp->sec, tp->msec, tp->usec);
   fprintf(tmpkt.ofp, "%8ld\n", tmpkt.hdr.packet_length);

   // Write out data
   for (long i=0; i<tmpkt.hdr.packet_length/2; ++i) {
      fprintf(tmpkt.ofp, "%04hx ", sp[i]);
      if (i%16==15) fputc('\n', tmpkt.ofp);
   }
   fprintf(tmpkt.ofp, "\n");


}



// *********************** STATIC FUNCTIONS **********************************


// ===========================================================================
   static void ProcessSCSDMPheader(char *buffer, tmheader *hp, long pktno)
// ***************************************************************************
{
   int i;
   scet *tp = &(hp->sc_time);
   short days_in_month[] = { 31, 28, 31, 30,  31, 30,
                             31, 31,  30, 31, 30, 31};


   // process packet time
   if (pktno==-1) { // first packet

      sscanf(buffer+2,  "%hd/%hd/%hd", &tp->month, &tp->day, &tp->year);
      sscanf(buffer+12, "%hd:%hd:%hd", &tp->hr, &tp->min, &tp->sec);
      tp->msec = 0;
      tp->usec = 0;
      tp->nsec = 0;

      if (tp->year < 50) tp->year += 2000;
      else               tp->year += 1900;

      // calculate day of year from day,month,year
      if (tp->year % 4 == 0) days_in_month[1] = 29;
      tp->doy = 0;
      for (i=1; i<tp->month; ++i)
          tp->doy += days_in_month[i-1];
      tp->doy += tp->day; // add days of current month

      hp->data_type = buffer[26];

   } else {
      hp->data_type   = scsdmp.DataType;
      
      long nsec = FirstHeader.sc_time.nsec + pktno*TMR_NSEC;

      long usec = nsec/1000 + FirstHeader.sc_time.usec + pktno*TMR_USEC;
      tp->nsec = nsec % 1000;

      long msec = usec/1000 + FirstHeader.sc_time.msec + pktno*TMR_MSEC;
      tp->usec = usec % 1000;

      long sec = msec/1000 + FirstHeader.sc_time.sec + pktno*TMR_SEC;
      tp->msec = msec % 1000;

      long min = sec/60 + FirstHeader.sc_time.min;
      tp->sec  = sec % 60;

      long hr  = min/60 + FirstHeader.sc_time.hr;
      tp->min = min % 60;

      long doy = hr/24 + FirstHeader.sc_time.doy;
      tp->hr = hr % 24;

      long year = FirstHeader.sc_time.year;

      short doy_max = 365; if (year%4 == 0) doy_max = 366;
      while (doy > doy_max) {
 	 doy -= doy_max;
         ++year;
         doy_max = 365;
         if (year%4 == 0) doy_max = 366;
      }
      tp->doy = doy;
      tp->year = year;

      // calculate day and month from day of year (doy)
      if (tp->year %4 == 0) days_in_month[1] = 29;
      tp->month=0;
      short day = tp->doy;
      while (day > days_in_month[tp->month]) {
         day -= days_in_month[tp->month];
         ++tp->month;
         if (tp->month > 11) {
            fprintf(stderr, "ProcessCDDSheader(): date conversion error: "
                            "month > Dec\n");
	    exit(1);
         }
      }
      tp->day = day;
      ++tp->month; //  1 - 12
   }

   hp->packet_length=            (0x00ff&buffer[22])  +
                            256L*(0x00ff&buffer[23]) +
                          65536L*(0x00ff&buffer[24]) +
                     256L*65536L*(0x00ff&buffer[25]);

/*
   if (pktno == -1) {
      printf("first header packet length: %ld\n", (long)hp->packet_length);
      printf("bytes 22-25 : %02hx %02hx %02hx %02hx\n",
                  (short)buffer[22], (short)buffer[23],
                  (short)buffer[24], (short)buffer[25]);
   }
*/

   hp->data_source = CL2TM_DSRC_EDI;

}

// ============================================================================
   static void PrintSCSDMPheader(tmheader hdr)
// ============================================================================
{
   scet *sctp = &hdr.sc_time;
   fprintf(stderr, "partial SCS DMP header dump\n"
                   "   pkt time (ydhms): %04hd %03hd %02hd %02hd %02hd\n"
                   "   pkt len  : %ld\n",
                   sctp->year, sctp->doy, sctp->hr, sctp->min, sctp->sec,
	           hdr.packet_length);
}






















