FUNCTION rdm_get_sun_data, key_time
; CLUSTER II IDL Function
;
; Abstract: extract sun event data from a RDM packet
;
; Created by: Mark Chutter, UNH 
; Mar. 13, 2000
;
; Modified:
; Apr. 20, 2000 MWC no longer print EOF message
; July 18, 2000 MWC if there is no SC_HK then set the seconds part of
;                   the rdm packet time to 99
; Aug. 16, 2000 MWC now worries about matching packet time with
;                   science data for FTP or RDM files; if the packet
;                   time is too high, then the file is rewound so that 
;                   the sun data is not lost
; Oct.  3, 2000 MWC fixed check for time low
; Oct.  4, 2000 MWC replaced RDM_HEADER_LENGTH with RDM_HEADER_SIZE
; Oct. 17, 2000 MWC changed status in error packet to -1
; Nov. 10, 2000 MWC added SC_ID to rdm
; Jan. 28, 2002 MWC added text to print statement
;
; Calling Interface:
; key_time   lonarr(2) (i) packet time search key
;
; Return Value:
; A rdm structure is returned.  If the error field is
; less than zero, then the structure is not valid.
;
; Affected Variables:
;
; Description:
; This function reads the header of a RDM packet to determine the
; source (which instrument and what kind of data) and the length of
; the packet.  It then reads the packet itself.  If the packet is a
; Spacecraft Housekeeping packet, it sets the packet number and packet 
; time and compares
; these to the key_time.  If the packet satifies the
; requirements from the keys, the data is swapped and a the sun event
; portion of the packet is returned.
; For rdm files, the packet_number is the same as the second element of
; the packet time array.  The packet_time array is a two element array whose
; values come from the Spacecraft Event Time (SCET) from the first eight
; bytes in the DDS header (the packet header).  The first 2 bytes contain
; the number of days since January 1, 1958.  The next four bytes contain the
; number of milliseconds since the start of the current day.  The next 2
; bytes contain the number of microseconds within the current millisecond.
COMMON shk, pick_shk_lun

@pick_const
RDM_HEADER_SIZE = 15
rdm_header = bytarr(RDM_HEADER_SIZE)
packet_length = 0l
sun_data_length = 36
sun_data_offset = 234
source_table = [51, 91, 131, 171]
source_id = -1
source = -1
pos = 0l

on_ioerror, eof
IF pick_shk_lun NE -3 THEN BEGIN
    readu, pick_shk_lun, rdm_header
    packet_length = long(rdm_header[9]) * '10000'xL + $
      long(rdm_header[10]) * '100'xL + $
      long(rdm_header[11])
ENDIF ELSE packet_length = 1

;Thanks for not telling me this before commissioning Cluster
IF packet_length EQ 604 THEN sun_data_offset = sun_data_offset + 84

data = bytarr(packet_length)
; I could have used a sc_pick_pck_struct here, but it seemed easier to
; make another structure that exactly fit the needs here.
rdm = {$
        sc_id: -1, $
        source: -1, $
        packet_number: 0L, $
        packet_time: lonarr(2), $
        packet_length: sun_data_length, $
        data: bytarr(sun_data_length), $
        status: 1 $
      }

;if there is no SC_HK data available
IF pick_shk_lun EQ -3 THEN BEGIN
    rdm.packet_time[0] = 99
    rdm.status = 1
    return, rdm
ENDIF

FOR i = 0, n_elements(source_table)-1 DO BEGIN
    source_id = rdm_header[8]
    IF (source_id EQ source_table[i]) THEN source = PICK_SHK_PACKET
ENDFOR

readu, pick_shk_lun, data

IF (source NE PICK_SHK_PACKET) THEN BEGIN
    rdm.status = NOT_SHK_PACKET
    return, rdm
ENDIF

IF source_id EQ 51 THEN rdm.sc_id = 1 $
ELSE IF source_id EQ 51 THEN rdm.sc_id = 2 $
ELSE IF source_id EQ 91 THEN rdm.sc_id = 3 $
ELSE IF source_id EQ 131 THEN rdm.sc_id = 4

rdm.packet_number = long(rdm_header[2]) * '1000000'xL + $
  long(rdm_header[3]) * '10000'xL + $
  long(rdm_header[4]) * '100'xL + $
  long(rdm_header[5])

rdm_get_packet_time, rdm_header, sec, msec
rdm.packet_time[0] = sec
rdm.packet_time[1] = msec

; this checks the primary 28V voltage
IF data[83] LT 215 OR data[83] GT 217 THEN BEGIN
    pick_cvt_sec_vax, rdm.packet_time, bad_data_time
    print, 'Primary Voltage out of range ', bad_data_time, data[83]
ENDIF

; if key_time[0] EQ 0, then the next shk packet is sought,
; otherwise an shk packet with a matching packet_time is sought
IF (key_time[0] NE 0) THEN BEGIN
    IF (rdm.packet_time[0] LT key_time[0]) THEN BEGIN
        rdm.status = PACKET_TIME_LOW
        return, rdm
    ENDIF ELSE IF (rdm.packet_time[0] GT key_time[0]) THEN BEGIN
        rdm.status = PACKET_TIME_HIGH
;BUGGER, this might not work for VMS
        point_lun, -1 * pick_shk_lun, pos ;rewind the file by one packet
        point_lun, pick_shk_lun, pos - RDM_HEADER_SIZE - packet_length
        rdm.packet_time[0] = 99
        return, rdm
    ENDIF
ENDIF

rdm.data = data[sun_data_offset:sun_data_offset+sun_data_length-1]

return, rdm
eof: rdm_error = {$
                   status: -1 $
                 }
rdm_error.status = EOF
return, rdm_error
END


