FUNCTION rdm_get_packet, key_number, key_time, file_type, pick_lun, $
            write_file = write_file
; CLUSTER II IDL Function
;
; Abstract: read a RDM packet
;
; Created by: Mark Chutter, UNH 
; June 29, 1999
;
; Modified:
; Aug. 11, 1999 MWC added the write_file keyword which causes the
;                   packet to be written to a file if the packet time
;                   is greater than the key time
; Sep. 15, 1999 MWC a rdm structure is returned regardless of whether 
;                   a valid packet was found or not (with status set)
; Mar. 15, 2000 MWC changed times to atime (seconds and microseconds)
; Apr. 20, 2000 MWC no longer print EOF message; now also reads
;                   spacecraft housekeeping for MSF files
; June  5, 2000 MWC added SC_ID to RDM structure
; July 13, 2000 MWC fixed code for determining SC_ID
; Oct.  3, 2000 MWC fixed check for time low
; Nov. 28, 2000 MWC pass file_type and pick_lun rather than use COMMON
; Jan.  5, 2001 MWC added data_stream
;
; Calling Interface:
; key_number long      (i) packet number search key
; key_time   lonarr(2) (i) packet time search key
;
; Return Value:
; A rdm structure is returned.  If the error field is
; less than zero, then the structure is not valid.
;
; Affected Variables:
;
; Description:
; This function reads the header of a RDM packet to determine the
; source (which instrument and what kind of data) and the length of
; the packet.  It then reads the packet itself.  If the packet is an
; EDI packet, it sets the packet number and packet time and compares
; these to the key_number and key_time.  If the packet satifies the
; requirements from the keys, the data is returned.
; For rdm files, the packet_number is the same as the second element of
; the packet time array.  The packet_time array is a two element array whose
; values come from the Spacecraft Event Time (SCET) from the first eight
; bytes in the DDS header (the packet header).  The first 2 bytes contain
; the number of days since January 1, 1958.  The next four bytes contain the
; number of milliseconds since the start of the current day.  The next 2
; bytes contain the number of microseconds within the current millisecond.
COMMON data_out, out_lun

@pick_const
RDM_HEADER_SIZE = 15
rdm_header = bytarr(RDM_HEADER_SIZE)
packet_length = 0l
data_length = 0l
packet_id = 0
sun_data_length = 36
sun_data_offset = 0
SC_HK_source_table = [51, 91, 131, 171]
EDI_source_table = [30, 37, 44, $
                    70, 77, 84, $
                    110, 117, 124, $
                    150, 157, 164]
source_id = -1
source = -1

on_ioerror, eof
readu, pick_lun, rdm_header
packet_length = long(rdm_header[9]) * '10000'xL + $
  long(rdm_header[10]) * '100'xL + $
  long(rdm_header[11])

;There were some packets that had messed up RDM headers
;size_check = size(packet_length)
IF packet_length EQ 0 THEN GOTO, bad_packet

data = bytarr(packet_length)

IF file_type EQ 'msf' THEN BEGIN
    FOR i = 0, n_elements(SC_HK_source_table)-1 DO BEGIN
        source_id = rdm_header[8]
        IF (source_id EQ SC_HK_source_table[i]) THEN BEGIN
            source = PICK_SHK_PACKET
            data_length = sun_data_length
            packet_id = SC_HK_source_table[i]
        ENDIF
    ENDFOR
ENDIF

FOR i = 0, n_elements(EDI_source_table)-1 DO BEGIN
    source_id = rdm_header[8]
    IF (source_id EQ EDI_source_table[i]) THEN BEGIN
        source = PICK_EDI_PACKET
        data_length = packet_length
        packet_id = EDI_source_table[i]
    ENDIF
ENDFOR

IF data_length EQ 0 THEN GOTO, eof ;this means the file is messed up

;There were some packets that had messed up RDM headers
;size_check = size(data_length)
;IF size_check[2] EQ 0 THEN GOTO, bad_packet

rdm = {$
        sc_id: -1, $
        data_stream: 0B, $
        source: -1, $
        packet_number: 0L, $
        packet_time: lonarr(2), $
        packet_length: data_length, $
        data: bytarr(data_length), $
        status: 1 $
      }

readu, pick_lun, data

IF file_type EQ 'rdm' THEN BEGIN
    IF (source NE PICK_EDI_PACKET) THEN BEGIN
        rdm.status = NOT_EDI_PACKET
        return, rdm
    ENDIF
ENDIF ELSE BEGIN
    IF (source NE PICK_EDI_PACKET AND $
        source NE PICK_SHK_PACKET) THEN BEGIN
        rdm.status = NOT_EDI_PACKET
        return, rdm
    ENDIF
ENDELSE

IF packet_id LE 51 THEN rdm.sc_id = 1 $
ELSE IF packet_id GT 51 AND packet_id LE 91 THEN rdm.sc_id = 2 $
ELSE IF packet_id GT 91 AND packet_id LE 131 THEN rdm.sc_id = 3 $
ELSE IF packet_id GT 131 AND packet_id LE 171 THEN rdm.sc_id = 4

rdm.packet_number = long(rdm_header[2]) * '1000000'xL + $
  long(rdm_header[3]) * '10000'xL + $
  long(rdm_header[4]) * '100'xL + $
  long(rdm_header[5])

IF (rdm.packet_number LT key_number) THEN BEGIN
    rdm.status = PACKET_NUMBER_LOW
    return, rdm
ENDIF

rdm.data_stream = rdm_header[13]

rdm_get_packet_time, rdm_header, sec, usec
rdm.packet_time[0] = sec
rdm.packet_time[1] = usec

IF (rdm.packet_time[0] LT key_time[0]) THEN BEGIN
    rdm.status = PACKET_TIME_LOW
    return, rdm
ENDIF

IF keyword_set(write_file) THEN BEGIN
    writeu, out_lun, rdm_header
    writeu, out_lun, data
    return, rdm
ENDIF

source = source_id - (source_id/10)*10
IF (source EQ 1) THEN rdm.source = PICK_SHK_PACKET
IF (source EQ 0) THEN rdm.source = PICK_EDI_PACKET
IF (source EQ 7) THEN rdm.source = PICK_EDI_PACKET
IF (source EQ 4) THEN rdm.source = PICK_HK_PACKET

IF source EQ 1 THEN $
  rdm.data = data[sun_data_offset:sun_data_offset+sun_data_length-1] $
ELSE rdm.data = data

return, rdm

bad_packet: rdm_error = {$
                          status: 0 $
                        }
print, rdm_header
rdm_error.status = INVALID_EDI_DATA

return, rdm_error
eof: rdm_error = {$
                   status: 0 $
                 }
rdm_error.status = EOF
return, rdm_error
END


