FUNCTION ctm_get_packet, key_number, key_time, write_file = write_file
; CLUSTER II IDL Function
;
; Abstract: read a CTM packet
;
; Created by: Mark Chutter, UNH 
; Sep. 15, 1999
;
; Modified:
; Apr. 20, 2000 MWC no longer print EOF message
; July  6, 2000 MWC added spacecraft ID
; July 11, 2000 MWC this might actually work regardless of operating
;                   system.  The trick is to use /stream in
;                   pick_init.pro when opening the file and being sure 
;                   that the VMS file information is not confused
;                   during transfer (ftp seemed to change the variable 
;                   record length to fixed).  Before assuming this
;                   routine is broken on VMS, make sure that the file
;                   has variable record length.
; July 12, 2000 MWC OK, now it should work
; Aug. 26, 2000 MWC the packet length can also be off by one for
;                   science data, added kludge_flag
; Aug. 28, 2000 MWC fixed the fractions of a second calculation 
; Sep. 22, 2000 MWC arbitrarily added 6 to BM1 packet lengths
; Oct.  4, 2000 MWC fixed low time check
;
; Calling Interface:
; key_number long      (i) packet number search key
; key_time   lonarr(2) (i) packet time search key
;
; Return Value:
; A ctm structure is returned.  If the error field is
; less than zero, then the structure is not valid.
;
; Affected Variables:
;
; Description:
; This code was written without any referencing actual docmentation
; and is based on the old FORTRAN routine written by Kosta.  I hope
; that is is fairly close to what is required or that I get some
; documentation which describes the CTM format.
; This function reads the header of a CTM packet to determine the
; source (which instrument and what kind of data) and the length of
; the packet.  It then reads the packet itself.  If the packet is an
; EDI packet, it sets the packet number and packet time and compares
; these to the key_number and key_time.  If the packet satifies the
; requirements from the keys, the data is swapped and a the packet is
; returned.  This routine has not yet been tested with segmented records.
;

COMMON pick, file_type, pick_lun
@pick_const
rec_length = bytarr(2) ;2 bytes for VMS record length
VMS_SEG_SIZE = 2042
seg_rec = bytarr(VMS_SEG_SIZE)
CTM_HEADER_SIZE = 16
ctm_header = bytarr(CTM_HEADER_SIZE)
packet_length = 0l
source = -1
segmented = 0

on_ioerror, eof
; the record length is handled by VMS; on other systems, the following 
; takes care of it
;IF !version.os_family NE 'vms' THEN readu, pick_lun, rec_length
;readu, pick_lun, rec_length; This did not work 8-26-2000

readu, pick_lun, ctm_header
; packet_length includes the time data and whatever byte 6 stores
; which I read into the ctm_header so packet data is actually 8 bytes less
packet_length = long(ctm_header[6]) * '100'xL + $
  long(ctm_header[7]) - 8
source = ishft((ctm_header[2] AND '07'x), 1) + $
  ishft((ctm_header[3] AND '80'x), -7)
segmented = ishft((ctm_header[4] AND 'C0'x), -6)

; In Archive Files sometime the lengths given in the headers will not
; match with reality, but are by 1 less.
IF (packet_length MOD 2 EQ 1) THEN packet_length = packet_length + 1

;MWC don't know why this helps Sep. 22, 2000
IF packet_length EQ 6944 THEN packet_length = 6950
data = bytarr(packet_length)
ctm = {$
        source: 0, $
        sc_id: 0, $
        packet_number: 0L, $
        packet_time: lonarr(2), $
        packet_length: packet_length, $
        data: bytarr(packet_length), $
        status: 1 $
      }

kludge_flag = 1;I am not sure what is up with these files 8-25-2000
IF kludge_flag EQ 1 OR ctm_header[0] NE 1 THEN BEGIN
    readu, pick_lun, data
ENDIF ELSE BEGIN
readu, pick_lun, rec_length
    rec_l = fix(rec_length[1]) * '100'x + fix(rec_length[0])
    seg_rec = bytarr(rec_l-16);ctm_header is part of this record
    readu, pick_lun, seg_rec
    data[0:rec_l-17] = seg_rec
    offset = rec_l-16
    done_seg = 0
    WHILE done_seg NE 1 DO BEGIN
        readu, pick_lun, rec_length
        rec_l = fix(rec_length[1]) * '100'x + fix(rec_length[0])
        seg_rec = bytarr(rec_l)
        readu, pick_lun, seg_rec
        data[offset:offset+rec_l-3] = seg_rec[2:rec_l-1]
        offset = offset + rec_l - 2
        IF seg_rec[0] EQ 2 THEN done_seg = 1
    ENDWHILE
ENDELSE

IF (source NE 10) THEN BEGIN
    ctm.status = NOT_EDI_PACKET
    return, ctm
ENDIF

data_type = (ctm_header[3] AND '0F'x)
IF (data_type EQ 3) THEN ctm.source = PICK_EDI_PACKET $
ELSE IF (data_type EQ 4) THEN ctm.source = PICK_EDI_PACKET $
ELSE IF (data_type EQ 1) THEN ctm.source = PICK_HK_PACKET $
ELSE BEGIN 
    ctm.status = INVALID_EDI_DATA
    return, ctm
END

ctm.packet_number = long((ctm_header[4]) AND '3F'x) * '100'xL + $
  long(ctm_header[5])

IF (ctm.packet_number LT key_number) THEN BEGIN
    ctm.status = PACKET_NUMBER_LOW
    return, ctm
ENDIF

ctm.packet_time[0] = long(ctm_header[9]) * '1000000'x + $
  long(ctm_header[10]) * '10000'x + $
  long(ctm_header[11]) * '100'x + $
  long(ctm_header[12])
ctm.packet_time[1] = long(float(long(ctm_header[13]) * '1000'x + $
                                long(ctm_header[14]) * '10'x + $
                                long(ctm_header[15])) * 2.^(-20) * 1000000.)

IF (ctm.packet_time[0] LT key_time[0]) THEN BEGIN
    ctm.status = PACKET_TIME_LOW
    return, ctm
ENDIF

IF keyword_set(write_file) THEN BEGIN
    writeu, out_lun, ctm_header
    writeu, out_lun, data
    return, cdm
ENDIF

;store data in ctm record
ctm.data = data

return, ctm
eof: ctm_error = {$
                   status: 0 $
                 }
ctm_error.status = EOF
return, ctm_error
END


