FUNCTION cdm_get_packet, key_number, key_time, write_file = write_file, $
            headers_only = headers_only
; CLUSTER II IDL Function
;
; Abstract: read a CDM packet
;
; Created by: Mark Chutter, UNH 
; July 7, 1999
;
; Modified:
; Aug.  6, 1999 MWC tried to check the dump file for its
;                   source, but the source does not seem to be set
;                   also fixed the key_time check which was
;                   incorrect in one case added cdm_dummy structure
; Aug. 11, 1999 MWC added the write_file keyword which causes the
;                   packet to be written to a file if the packet time
;                   is greater than the key time
; Nov. 18, 1999 MWC updated the description
; June  5, 2000 MWC added SC_ID to CDM structure
;
; Version: 0.1
;
; Calling Interface:
; key_number   long      (i) packet number search key
; key_time     lonarr(2) (i) packet time search key
; write_file   int       (i) if set, output packet
; headers_only int       (i) if set, only read header and secondary header
;
; Return Value:
; A cdm structure is returned.  If the error field is
; less than zero, then the structure is not valid.
;
; Affected Variables:
;
; Description:
; cdm_get_packet actually gets 2 packets: an EDI science packet and an
; EDI housekeeping packet.  There are 2 IDL keywords to this
; function.  If the write_file keyword is set, then the packet is
; written to file.  If the header_only keyword is set, then only the
; header portion of the packet is read and the remaining portion of
; the packet is skipped.  This is not a user level function.
COMMON pick, file_type, pick_lun
COMMON pick_buffer, pick_buffer, pick_buf_ptr
COMMON data_out, out_lun

@pick_const
CDM_HEADER_SIZE = 64
header = bytarr(CDM_HEADER_SIZE)
hk1_loc = 0
hk1_length = 0
hk2_loc = 0
hk2_length = 0
src_offset = 0
dest_offset = 0
length1 = 0
length2 = 0
length3 = 0
atime = lonarr(2)

on_ioerror, eof

pick_buffer_size = size(pick_buffer)
IF pick_buffer_size[0] EQ 0 THEN readu, pick_lun, header $
ELSE BEGIN
    IF pick_buf_ptr EQ pick_buffer_size[1] THEN BEGIN
        cdm_error = {$
                      status: 0 $
                    }
        cdm_error.status = EOF
        return, cdm_error
    ENDIF
    header = pick_buffer[pick_buf_ptr:pick_buf_ptr+CDM_HEADER_SIZE-1]
    pick_buf_ptr = pick_buf_ptr+CDM_HEADER_SIZE
ENDELSE

packet_length = long(header[25]) * '1000000'xL + $
  long(header[24]) * '10000'xL + $
  long(header[23]) * '100'xL + $
  long(header[22])

tmp_data = bytarr(packet_length)

IF keyword_set(headers_only) THEN BEGIN
;only allocate space for the header plus secondary header
    cdm = {$
            sc_id: 0, $
            source: 0, $
            packet_number: 0L, $
            packet_time: lonarr(2), $
            packet_length: 40, $
            data: bytarr(40), $
            status: 1 $
          }
ENDIF ELSE BEGIN
    cdm = {$
            sc_id: 0, $
            source: 0, $
            packet_number: 0L, $
            packet_time: lonarr(2), $
            packet_length: packet_length, $
            data: bytarr(packet_length-HK_SIZE), $
            hk: bytarr(HK_SIZE), $
            status: 1 $
          }
ENDELSE

; read the temporary packet which includes both EDI science and EDI hk
pick_buffer_size = size(pick_buffer)
IF pick_buffer_size[0] EQ 0 THEN readu, pick_lun, tmp_data $
ELSE BEGIN
    tmp_data = pick_buffer[pick_buf_ptr:pick_buf_ptr+packet_length-1]
    pick_buf_ptr = pick_buf_ptr+packet_length
ENDELSE

; determine the source, except that the source does not seem to get set
; in cdm files, so I will just set it to PICK_EDI_PACKET
;cdm.source = header[27]
;IF (cdm.source EQ 0) THEN cdm.status = INVALID_EDI_DATA
cdm.source = PICK_EDI_PACKET

; determine the packet number and check against key_number
cdm.packet_number = $
  3600 * ((long(header[12]) - 48) * 10 + long(header[13]) - 48) + $
  60 * ((long(header[15]) - 48) * 10 + long(header[16]) - 48) + $
  (long(header[18]) - 48) * 10 + long(header[19]) - 48

IF (cdm.packet_number LT key_number) THEN BEGIN
    cdm.status = PACKET_NUMBER_LOW
    return, cdm
ENDIF

; determine the packet time and check against key_time
time_stamp = cdm_parse_time_stamp(header)
pick_cvt_str_sec, time_stamp, atime
cdm.packet_time = atime

IF (cdm.packet_time[0] LT key_time[0]) THEN BEGIN
    cdm.status = PACKET_TIME_LOW
    return, cdm
ENDIF ELSE IF (cdm.packet_time[1] LT key_time[1]) THEN BEGIN
    cdm.status = PACKET_TIME_LOW
    return, cdm
ENDIF

IF keyword_set(write_file) THEN BEGIN
    writeu, out_lun, header
    writeu, out_lun, tmp_data
    return, cdm
ENDIF

; check the packet type, if the packet type is 0, then the packet is invalid
type = (tmp_data[0] AND '7'x)
IF (type EQ 0) THEN BEGIN
    cdm.status = INVALID_EDI_DATA
    return, cdm
END

; If you are only interested in the headers, than save a little processing
IF keyword_set(headers_only) THEN BEGIN

    ;swap bytes of data
    FOR i=0, 39, 2 DO BEGIN
        dummy = tmp_data[i+1]
        tmp_data[i+1] = tmp_data[i]
        tmp_data[i] = dummy
    ENDFOR

    cdm.data = tmp_data[0:39]
    return, cdm
END

; HK is mixed in dump files rather than being in a separate packet
; for other file types like RDM.  Bytes are swapped because the
; HK locations and sizes were written as shorts.  hk1_length and hk2_length
; are the sizes in words. */
hk1_loc = FIX(header[47]) * '100'x + FIX(header[46])
hk1_length = (FIX(header[49]) * '100'x + FIX(header[48])) * 2
hk2_loc = FIX(header[51]) * '100'x + FIX(header[50])
hk2_length = (FIX(header[53]) * '100'x + FIX(header[52])) * 2

IF ((hk1_length + hk2_length) NE HK_SIZE) THEN BEGIN
    print, 'missing hk data'
    cdm.status = INVALID_EDI_DATA
    return, cdm
ENDIF

;swap bytes of data
FOR i=0, cdm.packet_length-1, 2 DO BEGIN
    dummy = tmp_data[i+1]
    tmp_data[i+1] = tmp_data[i]
    tmp_data[i] = dummy
ENDFOR

; copy the science data from tmp_packet to cdm.packet
dest_offset = 0
src_offset = 0
length1 = hk1_loc
IF (length1 GT 0) THEN $
  cdm.data[dest_offset:dest_offset+length1-1] = $
  tmp_data[src_offset:src_offset+length1-1]
dest_offset = length1
src_offset = hk1_loc + hk1_length
length2 = hk2_loc - src_offset
IF (length2 GT 0) THEN $
  cdm.data[dest_offset:dest_offset+length2-1] = $
  tmp_data[src_offset:src_offset+length2-1]
dest_offset = length1 + length2
src_offset = hk2_loc + hk2_length
length3 = packet_length - src_offset
IF (length3 GT 0) THEN $
  cdm.data[dest_offset:dest_offset+length3-1] = $
  tmp_data[src_offset:src_offset+length3-1]
cdm.packet_length = length1 + length2 + length3

FOR i=0, HK_SIZE-1 DO cdm.hk[i] = 0
; copy the hk data from tmp_packet to cdm.hk
src_offset = hk1_loc
dest_offset = 0
length1 = hk1_length
cdm.hk[dest_offset:dest_offset+length1-1] = $
  tmp_data[src_offset:src_offset+length1-1]
dest_offset = length1
src_offset = hk1_loc + hk1_length
length2 = hk2_length
cdm.hk[dest_offset:dest_offset+length2-1] = $
  tmp_data[src_offset:src_offset+length2-1]

return, cdm
eof:cdm_error = {$
                  status: 0 $
                }
cdm_error.status = EOF
return, cdm_error
END
