function read_edidsi_data, filename, $
                           t_ssm, e_dsi, $
                           trange=trange, gc=gc

;+
; NAME: read_edidsi_data.pro
;
; PURPOSE: Read EDI electric field data from a special file made for the EDI/EFW
;          comparison and merging project.  This file has the
;          original "IFF" format defined by the Cluster Data Centers.
;          The electric field is in the DSI (despun satellite inverted)
;          coordinate system.  Data files exist that are in both
;          inertial and satellite frames.
;
; CATEGORY: File reading
;
; CALLING SEQUENCE:    istat = read_edidsi_data(filename, t_ssm, e_dsi)
;
; INPUTS:  filename    String, name of file to be read
;
; OUTPUTS: istat       Status flag: 0=failure, 1=success
;          t_ssm       Time in seconds since midnight
;          e_dsi       EDI Electric field in DSI (despun satellite
;                      inverted) coordinate system.  The frame
;                      (satellite versus inertial) depends on which
;                      file you're using (both exist).
;                      Units are mV/m
;                      Dimensions are [3,ntime]
;
; MODIFICATION HISTORY: Written March 20th, 2004, Pamela A. Puhl-Quinn
;                       Documentation modified 31-Mar-2004 (both
;                       inertial and satellite frame data files exist now)
;-

gc = keyword_set(gc)

f = findfile(filename)
if (f(0) eq '') then begin
    message, 'File not found: '+filename, /cont
    return, 0
endif
openr, id, f(0), /get_lun

;R 000167 14:11:26.144 51086144 002 069 255 008 017 003 000 -7.485e+00 -5.894e+00  6.723e+00 -9.419e-01  1.087e+00 -9.615e-02  1.503e+00
dum = ''
d = ''
n = 0L
t = ''
ms = 0L
s0 = 0L
s1 = 0L
s2 = 0L
s3 = 0L
s4 = 0L
s5 = 0L
s6 = 0L
r1 = 0.
r2 = 0.
r3 = 0.
r4 = 0.
r5 = 0.
r6 = 0.
r7 = 0.

t_ssm = [0.]
ex = [0.]
ey = [0.]
ez = [0.]

while not eof(id) do begin

    readf, id, dum
    if (strmid(dum,0,1) eq 'R') then begin
        
        reads, dum, d,n,t,ms,s0,s1,s2,s3,s4,s5,s6,r1,r2,r3,r4,r5,r6,r7, $
          format='(a1,1x,i6.6,1x,a12,1x,i8.8,1x,7(i3.3,1x),6(e10.3,1x),e10.3)'
        
        if (n_elements(trange) ne 0) then begin
            if (ms*1.e-3/3600. gt trange(1)) then goto, out_early
        endif
        
        if (s0 lt 255) then begin
            
            if (gc) then begin
                if (s0 ge 1) then begin ; Good and Caution only
                    t_ssm = [t_ssm,ms*1.e-3]
                    ex = [ex,r4]
                    ey = [ey,r5]
                    ez = [ez,r6]
                endif
            endif else begin
                t_ssm = [t_ssm,ms*1.e-3]
                ex = [ex,r4]
                ey = [ey,r5]
                ez = [ez,r6]
            endelse
            
        endif
    endif
endwhile
out_early:
free_lun, id
nrec = n_elements(t_ssm)

if (nrec eq 1) then begin
    message, 'No valid EDI data within file: '+f(0), /cont
    return, 0
endif

t_ssm = t_ssm(1:nrec-1)
e_dsi = fltarr(3,nrec-1)
e_dsi(0,*) = ex(1:nrec-1)
e_dsi(1,*) = ey(1:nrec-1)
e_dsi(2,*) = ez(1:nrec-1)

return, 1
end
