;	plot_bestarg.pro,v 1.14 2003/07/25 17:25:59 ppq Exp	
;========================================================================
;+
; NAME: plot_bestarg.pro
;
; PURPOSE: Using data from EDI_PISO auxilliary files, plot the
; triangulation results spin-by-spin.
;
; CATEGORY: Plotting
;
; CALLING SEQUENCE:
;     plot_bestarg, date, scnum, siz=siz, survey=survey, errors=errors, $
;                   aux_path=aux_path
;
; CALLING EXAMPLE:
;     plot_bestarg, '20010223', '1', siz=5, /errors
;
; INPUTS:
;     date                     'YYYYMMDD'
;     scnum                    '1', '2', '3', or '4'
;
;
; KEYWORD PARAMETERS:
;     siz=var_in               Range in meters of x and y    IN
;                              Default: range is made big enough
;                              to see the single runner target and
;                              it's error.
;     /survey                  9 plots per page instead of 1
;     /errors                  Will overplot errors in target
;                                 magnitude and angle
;     /error2                  For TRI results, will oplot the scaled
;                              error, which is a better indication
;     aux_path=var_in          String, path to the edi_piso aux files
;                                 you want to use.  In the default
;                                 case, the environment variable
;                                 EDI_PISO_AUX_PATH is checked and
;                                 used if it has been assigned.
;                                 Otherwise, the following path is
;                                 used:
;                                   aux_path = './aux_data/'
; OUTPUTS: NONE
;
; COMMON BLOCKS: NONE
;
; SIDE EFFECTS: Uses some routines in the EDI_PISO code distribution
;
; RESTRICTIONS:
;
; MODIFICATION HISTORY: Written, 2/2001 by Pamela A. Puhl-Quinn, ppq@mpe.mpg.de
;-

pro plot_bestarg, date, scnum, siz=siz_in, survey=survey, errors=errors, $
                  hans=hans, aux_path=aux_path_in, $
                  ordmax=ordmax, $
                  charsize=charsize_in, notitle=notitle, $
                  xrange=xrange, yrange=yrange, thick=thick, $
                  error2=error2, parallel=parallel, nbeam_max=nbeam_max, $
                  ntime_max=ntime_max, trange=trange, energy=energy

if (n_elements(aux_path_in) eq 0) then begin
    aux_path = getenv('EDI_PISO_AUX_PATH')
    if (aux_path eq '') then aux_path = './aux_data/'
endif else aux_path = aux_path_in

para = keyword_set(parallel)
errors = keyword_set(errors)
error2 = keyword_set(error2)
if (errors or error2) then plot_error=1 else plot_error=0
hans = keyword_set(hans)
survey = keyword_set(survey)
if (n_elements(energy) eq 0) then energy=1.0

if (para and survey) then message, 'survey and para keywords incompatible'

rfill = -1.e+31

date = string(date,'(i8)')
sc_number = string(scnum,'(i1)')

extract_aux_data, $
  date, sc_number, $            ; IN
  /nostatfilter, $              ; IN
  aux_path=aux_path, $          ; IN
  /search_for_all, $            ; IN
  ntime_max=ntime_max, $        ; IN
  trange=trange, $              ; IN
  energy=energy, $              ; IN
  t_ssm=t_ssm, bfield_scs=bfield_scs, $ ; OUT
  flip=flip, scs2bpp_refvec=scs2bpp_refvec, /rotmat, bpp2scs=bpp2scs, $ ; OUT
  nbeam_tot=nbeam_tot, $        ; OUT
  target_mag=target_mag,target_azi=target_azi, $ ; OUT
  error_target_mag=error_target_mag,error_target_azi=error_target_azi, $ ; OUT
  bestord_beam=bestord_beam,out_beam=out_beam,maxchan_beam=maxchan_beam, $ ; OUT
  gunid_beam=gunid_beam,xg_beam=xg_beam,yg_beam=yg_beam, $ ; OUT
  alpha_beam=alpha_beam,tof_beam=tof_beam,btime_ssm_beam=btime_ssm_beam, $ ; OUT
  tmid=tmid,rchi2_cg=rchi2_cg,code_type=code_type, $ ; OUT
  nbeam_usd=nbeam_usd, version_out=version, gdotb_beam=gdotb_beam, $ ; OUT
  fname_out=fname, clevel_cg=clevel_cg, rchi2_fg=rchi2_fg, $
  clevel_fg=clevel_fg, efield_gse=egse, $
  error2_target_mag=error2_target_mag,error2_target_azi=error2_target_azi, $
  methstat=methstat, meth_usd=meth_usd, ptest_stdev_angle = ptest_stdev_angle,$
  nbeam_max=nbeam_max, class_init_beam=class_init_beam, qual_beam=qual_beam, $
  ambig_180=ambig_180

if (version eq 'devel') then version = 'v9_99'
v1 = long(strmid(version,1,1))
v2 = long(strmid(version,3,2))
vers = v1 + v2/100.
;===================BESTARG PICTURES=======================

if (not para) then begin
    if (survey) then !p.multi=[0,3,3] else !p.multi=[0,1,1]
endif else begin
    !p.multi=[0,1,2]
endelse

code_str = ['Short','Long']
fit_str = ['TRI', 'PMT', 'SMT', 'RMT']

fit_exit_str = strarr(40)

if (vers le 1.38) then begin
    fit_exit_str(0) = 'Success'
    fit_exit_str(1) = 'Too few beams after order prep'
    fit_exit_str(2) = 'Robust/Coarse solution grid edge'
    fit_exit_str(3) = 'Too few beams after outlier removal'
    fit_exit_str(4) = 'Non-Robust/Coarse solution on grid edge'
    fit_exit_str(5) = 'Poormans ToF attempted but failed'
    fit_exit_str(6) = 'Forced Poormans ToF attempted but failed'
    fit_exit_str(7) = 'Too few "good" beams for Poormans ToF'
    fit_exit_str(8) = 'Value of pp_method not valid'
    fit_exit_str(9) = 'Coarse grid rchi2 larger than pp_rchi2_abortlim'
    fit_exit_str(10) = 'Non-Robust/Fine solution on grid edge'
endif else if (vers gt 1.38 and vers le 2.02) then begin
    fit_exit_str(0) = 'Success'
    fit_exit_str(1) = 'Too few beams after order prep'
    fit_exit_str(2) = 'Coarse grid solution on grid edge'
    fit_exit_str(3) = 'Too few beams after outlier removal'
    fit_exit_str(4) = 'PMT/SMT Branch: Both attempted, both failed'
    fit_exit_str(5) = 'PMT Branch: Attempted but failed'
    fit_exit_str(6) = 'SMT Branch: Not enough beams to start with'
    fit_exit_str(7) = 'PMT Branch: Not enough beams to start with'
    fit_exit_str(8) = 'Value of pp_method not valid'
    fit_exit_str(9) = 'TRI Branch: Coarse grid rchi2 > pp_rchi2_abortlim'
    fit_exit_str(10) = 'TRI Branch: Fine grid solution on grid edge'
    fit_exit_str(11) = 'TRI Branch: Fine grid rchi2 > pp_rchi2_outlim'
    fit_exit_str(12) = 'SMT Branch: Attempted but failed'
endif else begin
    fit_exit_str = $
      ['Success', $             ; 0
       'Success', $             ; 1
       'Success', $             ; 2
       'Success', $             ; 3
       '', $                    ; 4
       '', $                    ; 5
       '', $                    ; 6
       '', $                    ; 7
       '', $                    ; 8
       '', $                    ; 9
       $
       'TRI Failure: Coarse grid target on grid edge', $ ; 10
       'TRI Failure: Coarse grid rchi2 > pp_rchi2_abortlim', $ ; 11
       'TRI Failure: Fine grid target on grid edge', $ ; 12
       'TRI Failure: Fine grid rchi2 > pp_rchi2_outlim', $ ; 13
       'TRI/RMT Failure: Not enough beams in each to/away class for RMT, too parallel for TRI', $ ; 14
       'TRI Failure: Bmag lt pp_bmag_trilim', $ ; 15
       '', $                    ; 16
       '', $                    ; 17
       '', $                    ; 18
       '', $                    ; 19
       $
       'PMT Failure: Not enough beams to start with', $ ; 20
       'PMT Failure: TofError.pro failure', $ ; 21
       'PMT Failure: Early return, only SMT desired', $ ; 22
       '', $                    ; 23
       '', $                    ; 24
       'RMT Failure: Not enough beams to start with', $ ; 25
       'RMT Failure: Not enough beams in each to/away class', $ ; 26
       'RMT Failure: TofError.pro failure', $ ; 27
       'RMT Failure: Towards/Away Indexing Problem', $ ; 28
       '', $                    ; 29
       $
       'PMT/SMT Failure: Not enough beams to start with', $ ; 30
       'PMT/SMT Failure: Not enough beams in each to/away class', $ ; 31
       'PMT/SMT Failure: Both attempted, both failed', $ ; 32
       '', $                    ; 33
       '', $                    ; 34
       '', $                    ; 35
       '', $                    ; 36
       '', $                    ; 37
       '', $                    ; 38
       '', $                    ; 39
       $
       'SMT Failure: Not enough beams to start with', $ ; 40
       'SMT Failure: Not enough simultaneous pairs', $ ; 41
       'SMT Failure: TofError.pro failure', $ ; 42
       '', $                    ; 43
       '', $                    ; 44
       '', $                    ; 45
       '', $                    ; 46
       '', $                    ; 47
       '', $                    ; 48
       '', $                    ; 49
       $
       'Generic Failure: Not enough beams after order/class prep', $ ; 50
       'Generic Failure: Not enough beams after outlier removal', $ ; 51
       'Generic Failure: Value of pp_method not valid', $ ; 52
       'Generic Failure: Mean/Stdev Angle=NaN Error', $ ; 53
       '', $                    ; 54
       '', $                    ; 55
       '', $                    ; 56
       '', $                    ; 57
       '', $                    ; 58
       '']                      ; 59
endelse


inds = lindgen(n_elements(t_ssm))

ii = 0
if (survey) then window, 6, xsize=800, ysize=800

while ii le n_elements(inds)-1 do begin
    
    ispin = inds(ii)
    
; Project orbit onto plane
    phio = findgen(1000)*2.*!pi/999.
    orbitx_scs = 3.2*cos(phio)
    orbity_scs = 3.2*sin(phio)
    orbitz_scs = make_array(1000,/float,value=0.)
    
; SCS-->BPP rotation matrix
    s2b = transpose(bpp2scs(*,*,ispin))
    
    orbitx_bpp = s2b(0,0)*orbitx_scs + s2b(0,1)*orbity_scs + s2b(0,2)*orbitz_scs
    orbity_bpp = s2b(1,0)*orbitx_scs + s2b(1,1)*orbity_scs + s2b(1,2)*orbitz_scs
    orbitz_bpp = s2b(2,0)*orbitx_scs + s2b(2,1)*orbity_scs + s2b(2,2)*orbitz_scs
    
    mb = nbeam_tot(ispin)
    
    if (methstat(ispin) eq 0) then begin
        
        bestarg_ok = 1
        
        rd = target_mag(ispin)
        az = target_azi(ispin)/!radeg ; radians
        yd = rd*sin(az)
        xd = rd*cos(az)
        
        if (not error2) then begin
            rd_error = error_target_mag(ispin)
            az_error = error_target_azi(ispin)*!pi/180. ; radians
        endif else begin
            if (meth_usd(ispin) eq 0) then begin
                rd_error = error2_target_mag(ispin)
                az_error = error2_target_azi(ispin)*!pi/180. ; radians
            endif else begin
                rd_error = error_target_mag(ispin)
                az_error = error_target_azi(ispin)*!pi/180. ; radians
            endelse
        endelse

        rmin = rd - rd_error > 0.
        rmax = rd + rd_error
        amin = az - az_error
        amax = az + az_error
        
; Create the [x], [y] footprints for polyfilling the
; target error swath
        n = 100
        
        slp = (rmax - rmin) / float(n-1)
        r1 = slp*findgen(n) + rmin
        a1 = make_array(n,/float,value=amax)
        
        slp = (amin - amax) / float(n-1)
        a2 = slp*findgen(n) + amax
        r2 = make_array(n,/float,value=rmax)
        
        slp = (rmin - rmax) / float(n-1)
        r3 = slp*findgen(n) + rmax
        a3 = make_array(n,/float,value=amin)
        
        slp = (amax - amin) / float(n-1)
        a4 = slp*findgen(n) + amin
        r4 = make_array(n,/float,value=rmin)
        
        eswathx_bpp = [r1*cos(a1),r2*cos(a2),r3*cos(a3),r4*cos(a4)]
        eswathy_bpp = [r1*sin(a1),r2*sin(a2),r3*sin(a3),r4*sin(a4)]
        
        bestord = bestord_beam(0:mb-1,ispin)

;================================================================
; "out" definitions:
; 0,1 means bestarg used them
; 2 means this beam was determined (statistically) to be a
; triangulation outlier and wasn't used in the analysis
; 3 means this beam wasn't used in the analysis because it was class
; "C" and these weren't allowed
; 4 means this beam was unsuitable for PMT analysis
; 5 means this beam was unsuitable for SMT analysis
; 6 means this beam was unsuitable for RMT analysis
        out = out_beam(0:mb-1,ispin) 
;================================================================
        
    endif else begin
        bestarg_ok = 0
        bestord = intarr(mb)+1  ; All set to ones (single runners)
        out = out_beam(0:mb-1,ispin)        
        xd = 1
        yd = 1
    endelse
    
    mxch = maxchan_beam(0:mb-1,ispin)
    class_init = class_init_beam(0:mb-1,ispin)
    qual = qual_beam(0:mb-1,ispin)
    gunid = gunid_beam(0:mb-1,ispin)
    xg = xg_beam(0:mb-1,ispin)
    yg = yg_beam(0:mb-1,ispin)
    alpha = alpha_beam(0:mb-1,ispin)
    tof = tof_beam(0:mb-1,ispin)
    btime_ssm = btime_ssm_beam(0:mb-1,ispin)
    gdotb = gdotb_beam(0:mb-1,ispin)

    gdotb_avg = mean(gdotb)
    gdotb_var = variance(gdotb)

    plot_msg = ''
    if (tmid(ispin) eq '23-Feb-2001 19:12:18.335937' and hans) then begin
        ddtt = btime_ssm(1:mb-1) - btime_ssm(0:mb-2)
        if (mb eq 14 and min(ddtt) gt 0) then begin
            bestord = [1,1,1,1,2,3,3,3,3,1,1,1,1,1]
            out = intarr(mb)    ; All set to 0 (plot them all)
            bestarg_ok = 0      ; Don't oplot the best target
            plot_msg = "Hans' order assignment"
        endif else stop
    endif
    
    fqstr = string(rchi2_cg(ispin),format='(e9.2)')+'/'+string(clevel_cg(ispin),'(f6.2)')+', '+ string(rchi2_fg(ispin),format='(e9.2)')+'/'+string(clevel_fg(ispin),'(f6.2)')
    
    if (n_elements(siz_in) eq 0) then siz=5. else siz=siz_in
    
;    print, 'ptest: ',ptest_stdev_angle (ispin)*!radeg
    
    if (not survey) then begin
        
; Do some of your own beam stats book-keeping here...
        ntot = n_elements(out)
        iusd = where(out le 1)
        if (iusd(0) ne -1) then nusd = n_elements(iusd) else nusd = 0
        if (methstat(ispin) ne 0) then nusd = 0
        
        if (nusd ne nbeam_usd(ispin)) then message, $
              'Something wrong with beam stats....'

        if (ntot ne nbeam_tot(ispin)) then message, $
          'Something wrong with beam stats....'

        bstr = '!c# Beams TOT: '+string(ntot,'(i4)')+', USED: '+ $
          string(nusd,'(i4)')
        
        if (meth_usd(ispin) ne 255) then $
          fstr = '!cFit: '+fit_str(meth_usd(ispin)) else $
          fstr = '!cFit: NONE'
        if (fit_exit_str(abs(methstat(ispin))) eq 'Success' and $
            meth_usd(ispin) eq 0) then $
          fstr=fstr+', '+fit_exit_str(abs(methstat(ispin))) + $
          ', Rchi2/Clevel (coarse,fine)='+fqstr else $
          fstr=fstr+', '+fit_exit_str(abs(methstat(ispin)))
        
        plot_title =  'C'+sc_number+', Center: '+tmid(ispin)+ $
          '!cCode Type: '+code_str(code_type(ispin))+ $
          ', GdotB: '+string(gdotb_avg,'(f5.2)')+'!9+!x'+string(gdotb_var,'(f5.2)') + $
          fstr + bstr
        if keyword_set(notitle) then plot_title = ''

        if (not para) then window, 6, xsize=800, ysize=800 else $
          window, 6, xsize=475, ysize=950

        if (n_elements(charsize_in) eq 0) then charsize = 1.0 else charsize=charsize_in
        plot_ba, bestord, siz, plot_title, orbitx_bpp, orbity_bpp, $
          bestarg_ok, plot_error, mb, out, xg, yg, gunid, alpha, xd, yd, $
          mxch, plot_msg, eswathx_bpp, eswathy_bpp, font, survey, fname, $
          ispin, charsize, ordmax=ordmax, xrange=xrange, yrange=yrange, $
          thick=thick, cs=cs

        if (para) then begin
            id_pmt = where(mxch eq 7 and $
                           qual ge 2 and $
                           class_init ne 'C')
            
            if (n_elements(id_pmt) ge 4) then begin
                asend = alpha(id_pmt)
                ep_calc_para, xd, yd, asend, cs, sdev_out, $
                  ptest_stdev_angle(ispin)
            endif else begin
                !p.multi(0) = 0
                message, 'No beams to send to ep_calc_para.pro', /cont
            endelse
        endif
        
        time_stamp_edi, label=fname+$
          '!cINDEX: '+strtrim(ispin,2)

;        print, 'INDEX: ',ispin, flip(ispin)
;        print, tmid(ispin)
;        print, 'd: ', target_mag(ispin)
;        print, 'Efield: ', sqrt(total(egse(*,ispin)^2))
        
;        print, class_init
;        print, bestord

        res=''
        read, res, prompt='n(ext), b(ack), q(uit), p(ost-script) or index: '
        if (res eq 'b' and ii ne 0) then ii=ii-1 else $
          if (res eq 'n') then ii = ii+1 else $
          if (res eq 'q') then goto, quitit else $
          if (res eq 'p') then begin
            res2=''
            read, res2, prompt='m(y own name) or d(efault name): '
            if (res2 eq 'm') then begin
                res3=''
                read, res3, prompt='Please input name for ps-file (no quotes): '
                psname = res3
            endif else begin
                ipos = strpos(fname,'_edi_piso_aux_')
                if (ipos(0) ne -1) then begin
;c1_20010223_17_00_00_edi_piso_aux_1_v1_20.mpecl_msf
                    psname = 'pb_'+strmid(fname,ipos-20, ipos+20 - (ipos-20))+$
                      '_'+strtrim(ispin,2)+'.ps'
                endif else begin
;20010223_1_190000_devel_DM00_0_aux.mpecl_ftp
                    ipos1 = strpos(fname,'_DM00_')
                    ipos2 = strpos(fname,'_aux.')
                    psname = 'pb_'+strmid(fname,ipos1-23,ipos2-(ipos1-23))+$
                      '_'+strtrim(ispin,2)+'.ps'
                endelse
            endelse
            set_plot, 'PS'
            device, file=psname
            device, /color, bits=8
            device, /portrait
            device ,xoff=2.0,yoff=5.0,xs=18.2,ys=21.0
            plot_ba, bestord, siz, plot_title, orbitx_bpp, orbity_bpp, $
              bestarg_ok, plot_error, mb, out, xg, yg, gunid, alpha, xd, yd, $
              mxch, plot_msg, eswathx_bpp, eswathy_bpp, font, survey, fname, $
              ispin, charsize, ordmax=ordmax, xrange=xrange, yrange=yrange, $
              thick=thick

            time_stamp_edi, label='Code: plot_bestarg.pro, '+ $
              '!cFilename: '+psname+'!c'+fname+$
              '!cINDEX: '+strtrim(ispin,2)
            device, /close_file
            set_plot, 'X'
            ii = ii+1
        endif else ii = long(strtrim(res,2))
        
    endif else begin

;        sdev_out = -1e+31
;        id_pmt = where(mxch eq 7 and $
;                       qual ge 2 and $
;                       class_init ne 'C')
;        if (n_elements(id_pmt) ge 4) then begin
;            asend = alpha(id_pmt)
;            ep_calc_para, xd, yd, asend, cs, sdev_out, $
;              ptest_stdev_angle(ispin), /noplot
;        endif        
;        if (bestarg_ok and sdev_out ne -1e+31 and sdev_out le 10. and nbeam_usd(ispin) ge 6 and ambig_180(ispin) ne 0) then begin

            plot_title =  'C'+sc_number+', Center: '+tmid(ispin)        
            plot_ba, bestord, siz, plot_title, orbitx_bpp, orbity_bpp, $
              bestarg_ok, plot_error, mb, out, xg, yg, gunid, alpha, xd, yd, $
              mxch, plot_msg, eswathx_bpp, eswathy_bpp, font, survey, fname, $
              ispin, charsize, ordmax=ordmax, xrange=xrange, yrange=yrange, $
              thick=thick
            ii = ii+1
;            print, 'INDEX, RD, stdev: ',ispin, rd, sdev_out
            if (!p.multi(0) eq 0) then begin
                res=''
                read, res, prompt='n(ext), q(uit), index: '
                if (res eq 'n') then goto, next_page else $
                  if (res eq 'q') then goto, quitit else $
                  ii = long(strtrim(res,2))
            endif
            next_page:

;        endif else begin
;            ii = ii+1
;        endelse
        
    endelse
    
endwhile

quitit:

return
end
