; ============================================================================
  FUNCTION EquivSrToF, mtof, order, tgs, tcode
; ============================================================================
; Project: Cluster-II EDI
; Author : Hans Vaith, MPE Garching
; Creation Date: 01-Aug-2001
; 
; EquivSrToF: calculate equivalent single runner time of flight
;
; ============================================================================
; Id: equivsrtof.pro,v 1.1 2001/08/07 11:33:44 hav Exp hav 
; Log: equivsrtof.pro,v 
; Revision 1.1  2001/08/07 11:33:44  hav
; Initial revision
;
; ============================================================================
; Parameters:
;    mtof	measured time of flight as reported in telemetry
;    order      assumed runner order
;    tgs        single gyration gyro time
;    tcode      code length
;
; Return value:
;    a structure is returned which contains at least the following
;    two elements:
;          - status : 0 = ok, 1 = error
;          - msg    : contains an error message if status=1
;
;    if status is ok (0) then there is a third element:
;         -  estof  : equivalent single runner time of flight
;                     this is an array of the same dimension as parameter 
;                     mtof
;
; Notes:
; - mtof can be a scalar or an array. If it is an array, the returned 
;   equivalent single runner time of flight will also be an array.
; - order, tgs and tocde may either be scalars or they may be of the same 
;   dimension as mtof
; ============================================================================

; define preliminary return structure
; -----------------------------------
ret = { status:0, msg:'' }

; check parameters
; ----------------
if n_params() ne 4 then begin
   ret.status = 1
   ret.msg = 'EquivSrTof(): need 4 parameters, but there are ' + $
             strtrim(n_params(), 2)
   return, ret
endif

ndim = n_elements(mtof)

if n_elements(order) eq 1 then begin
   l_order = intarr(ndim) + order
endif else if n_elements(order) ne ndim then begin
   ret.status = 1
   ret.msg = 'EquivSrToF(): bad array size of parameter <order>'
   return, ret
endif else $
   l_order = order
   

if n_elements(tgs) eq 1 then begin
   l_tgs = intarr(ndim) + tgs
endif else if n_elements(tgs) ne ndim then begin
   ret.status = 1
   ret.msg = 'EquivSrToF(): bad array size of parameter <tgs>'
   return, ret
endif else $
   l_tgs = tgs


if n_elements(tcode) eq 1 then begin
   l_tcode = intarr(ndim) + tcode
endif else if n_elements(tcode) ne ndim then begin
   ret.status = 1
   ret.msg = 'EquivSrToF(): bad array size of parameter <tcode>'
   return, ret
endif else $
   l_tcode = tcode

ret = { status:0, msg:'', estof:fltarr(ndim) }


; ---------------------------------------------------
; find a suitable multiplier i that fits the equation
; (I) mtof + i * tcode ~= order * tgs
; left side : true multi-runner time of flight
; right side: n-fold gyro-time
; 
; The equation can more exactly be written as
; (1) mtof + i * tcode < order * tgs + 0.5 * tcode
; (2) mtof + i * tcode > order * tgs - 0.5 * tcode
;
;                           order * tgs - mtof
; this yields : i = floor ( ------------------  + 0.5 )
;                                 tcode
; ---------------------------------------------------

i = floor( (l_order*l_tgs - mtof)/l_tcode  + 0.5 )

ret.estof = (mtof + i * l_tcode) / l_order

return, ret

END





