;+ 
; NAME: ep_time_handler.pro
; TYPE: IDL function
; PURPOSE: Conversion between time formats
; AUTHOR: Pamela A. Puhl-Quinn, ppq@mpe.mpg.de, 04.2001
; USAGE EXAMPLE:
;
;         time_out = ep_time_handler(time_in, 't70', 'epoch')
;
; ARGUMENTS:     time_in     Scalar or 1-D array of times in one  IN
;                            of four formats described below
;                ttype_in    Format of time_in                    IN
;                ttype_out   Format of time_out                   IN
;
; KEYWORD: date='YYYYMMDD'   Either mandatory input, or optional output:
;  For ttype_in='ssm' :  date='YYYYMMDD'  is MANDATORY input
;  For ttype_out='ssm':  date='YYYYMMDD'  is optional output,
;                        and in the case where time_in crosses day 
;                        boundaries, it will contain the date
;                        associated with time_in(0).
;
; ttype_in and ttype_out must be one of the following strings:
;
;         't70'          time is double type, in units of
;                            decimal seconds since 
;                            01-Jan-1970 00:00:00.000
;         'vax_string'   time is of string type, with format
;                           'DD-MMM-YYYY hh:mm:ss.mmm', e.g.
;                           '09-Dec-2000 02:59:42.583'
;         'epoch'        time is double type, in units of
;                            milliseconds since 
;                            01-Jan-0000 00:00:00.000
;         'ssm'          time is decimal seconds since midnight
;                            The 'date' keyword MUST be set if this is
;                            the format for time_in. If this is the
;                            desired format for time_out, then the
;                            date keyword will return on output the
;                            date associated with time_in(0).
;-

function ep_time_handler, time_in, ttype_in, ttype_out, date=date_in

if (ttype_in eq 'ssm' and n_elements(date_in) eq 0) then begin
    message, "You must assign the 'date' keyword if ttype_in='ssm'"
endif

if (ttype_in eq 'ssm') then begin
    over = where(time_in gt 86399.99)
    if (over(0) ne -1) then begin
        nover = n_elements(over)
        message, '============Overflow control==============', /cont
        message, 'time_in contains values exceeding 86399.99', /cont
        message, 'Setting these/this '+strtrim(nover,2)+' value(s) to 86399.99', /cont
        time_in(over) = 86399.99d0
    endif
endif

if (ttype_out eq ttype_in) then return, time_in ; Safety
ntime = n_elements(time_in)
if ( (size(time_in))(0) eq 0 ) then scalar=1 else scalar=0
errmsg = 'Case for this ttype_in and ttype_out combo not found: '+ $
  ttype_in+', '+ttype_out

case ttype_in of
    'vax_string':begin
        if (scalar) then begin
            pick_cvt_str_sec, time_in, t70
            t70_out = double(t70(0)) + double(t70(1))/1d6 ; decimal t70
            t_mssst = 0d0       ; millisecs since start time
            ts0 = time_in
        endif else begin
            t70_out = dblarr(ntime)
            for ii=0,ntime-1 do begin
                pick_cvt_str_sec, time_in(ii), t70
                t70_out(ii) = double(t70(0)) + double(t70(1))/1d6 ; decimal t70
            endfor
            t_mssst = (t70_out - t70_out(0))*1d3 ; millisecs since start time
            ts0 = time_in(0)
        endelse
        case ttype_out of
            't70':return, t70_out
            'epoch':begin
                date = date2datevax(strmid(ts0,0,11),/reverse)
                year = long(strmid(date,0,4))
                month = long(strmid(date,4,2))
                day = long(strmid(date,6,2))
                hour = long(strmid(ts0,12,2))
                minute = long(strmid(ts0,15,2))
                second = long(strmid(ts0,18,2))
                msec = long(strmid(ts0,21,3))
                cdf_epoch, epoch0, year, month, day, $
                  hour, minute, second, msec, /compute_epoch
                return, epoch0 + t_mssst
            end
            'ssm':begin
                pick_cvt_str_sec, strmid(ts0,0,11)+' 00:00:00.000' , t70_mid 
                t70_mid = double(t70_mid(0)) + double(t70_mid(1))/1d6
                date_in = date2datevax(strmid(ts0,0,11),/reverse)
                return, t70_out - t70_mid
            end
            else:message, errmsg
        endcase
    end
    't70':begin
        if (scalar) then begin
            t0 = long(time_in)
            t1 = long((time_in - long(time_in))*1d6)
            t_mssst = 0d0       ; millisecs since start time
        endif else begin
            t0 = long(time_in(0))
            t1 = long((time_in(0) - long(time_in(0)))*1d6)
            t_mssst = (time_in - time_in(0))*1d3 ; millisecs since start time
        endelse
        pick_cvt_sec_vax, [t0,t1], ts0
        case ttype_out of
            'epoch':begin
                date = date2datevax(strmid(ts0,0,11),/reverse)
                year = long(strmid(date,0,4))
                month = long(strmid(date,4,2))
                day = long(strmid(date,6,2))
                hour = long(strmid(ts0,12,2))
                minute = long(strmid(ts0,15,2))
                second = long(strmid(ts0,18,2))
                msec = long(strmid(ts0,21,3))
                cdf_epoch, epoch0, year, month, day, $
                  hour, minute, second, msec, /compute_epoch
                return, epoch0 + t_mssst
            end
            'vax_string':begin
                if (scalar) then begin
                    return, ts0
                endif else begin
                    tstring_out = strarr(ntime)
                    for ii=0,ntime-1 do begin
                        t0 = long(time_in(ii))
                        t1 = long((time_in(ii) - long(time_in(ii)))*1d6)
                        pick_cvt_sec_vax, [t0,t1], ts
                        tstring_out(ii) = ts
                    endfor
                    return, tstring_out
                endelse
            end
            'ssm':begin
                pick_cvt_str_sec, strmid(ts0,0,11)+' 00:00:00.000' , t70_mid 
                t70_mid = double(t70_mid(0)) + double(t70_mid(1))/1d6
                date_in = date2datevax(strmid(ts0,0,11),/reverse)
                return, time_in - t70_mid
            end
            else:message, errmsg
        endcase
    end
    'epoch':begin
        if (scalar) then begin
            te0 = time_in
            t_ssst = 0d0        ; seconds since start time
        endif else begin
            te0 = time_in(0)
            t_ssst = (time_in - time_in(0))/1d3 ; seconds since start time
        endelse
        cdf_epoch, te0, year, month, day, $
          hour, minute, second, msec, /break
        ts0 = date2datevax(string(year,'(i4.4)')+ $
                           string(month,'(i2.2)')+ $
                           string(day,'(i2.2)'))+ ' '+ $
          string(hour,'(i2.2)')+':'+string(minute,'(i2.2)')+':'+ $
          string(second,'(i2.2)')+'.'+string(msec,'(i3.3)')
        pick_cvt_str_sec, ts0, t70
        t70 = double(t70(0)) + double(t70(1))/1d6
        case ttype_out of
            't70':return, t70 + t_ssst
            'vax_string':begin
                if (scalar) then begin
                    return, ts0
                endif else begin
                    tstring_out = strarr(ntime)
                    for ii=0,ntime-1 do begin
                        cdf_epoch, time_in(ii), year, month, day, $
                          hour, minute, second, msec, /break
                        tstring_out(ii) = date2datevax(string(year,'(i4.4)')+$
                                                       string(month,'(i2.2)')+$
                                                       string(day,'(i2.2)'))+$
                          ' '+string(hour,'(i2.2)')+':'+$
                          string(minute,'(i2.2)')+':'+$
                          string(second,'(i2.2)')+'.'+string(msec,'(i3.3)')
                    endfor
                    return, tstring_out
                endelse
            end
            'ssm':begin
                date = date2datevax(strmid(ts0,0,11),/reverse)
                year = long(strmid(date,0,4))
                month = long(strmid(date,4,2))
                day = long(strmid(date,6,2))
                cdf_epoch, epoch_mid, year, month, day, $
                  0,0,0,0,/compute
                date_in = date
                return, (time_in - epoch_mid)/1d3
            end
            else:message, errmsg
        endcase
    end
    'ssm':begin
        date_vax = date2datevax(date_in)
        if (scalar) then begin
            fh0 = double(time_in)/3600d0
            t_ssst = 0d0
        endif else begin
            fh0 = double(time_in(0))/3600d0
            t_ssst = double(time_in) - double(time_in(0))
        endelse
        h0 =  long(   fh0                                )
        m0 =  long(  (fh0 - h0)*60d0                     )
        s0 =  long( ((fh0 - h0)*60d0 - m0)*60d0          )
        ms0 = long((((fh0 - h0)*60d0 - m0)*60d0 - s0)*1d3)
        ts0 = date_vax+ ' '+ $
          string(h0,'(i2.2)')+':'+string(m0,'(i2.2)')+':'+ $
          string(s0,'(i2.2)')+'.'+string(ms0,'(i3.3)')
        
        case ttype_out of
            't70':begin
                pick_cvt_str_sec, ts0, t70
                t70 = double(t70(0)) + double(t70(1))/1d6
                return, t70 + t_ssst
            end
            'vax_string':begin
                if (scalar) then begin
                    return, ts0
                endif else begin
                    tstring_out = strarr(ntime)
                    for ii=0,ntime-1 do begin
                        fh = double(time_in(ii))/3600d0
                        hour =  long(   fh   )
                        minute =long(  (fh-hour)*60d0)
                        second = long( ((fh-hour)*60d0-minute)*60d0)
                        msec = long((((fh-hour)*60d0-minute)*60d0-second)*1d3)
                        tstring_out(ii) = date_vax+' '+ $
                          string(hour,'(i2.2)')+':'+$
                          string(minute,'(i2.2)')+':'+$
                          string(second,'(i2.2)')+'.'+$
                          string(msec,'(i3.3)')
                    endfor
                    return, tstring_out
                endelse
            end
            'epoch':begin
                cdf_epoch, te0, long(strmid(date_in,0,4)), $
                  long(strmid(date_in,4,2)), $
                  long(strmid(date_in,6,2)), $
                  h0, m0, s0, ms0, /compute_epoch
                return, te0 + t_ssst*1d3
            end
            else:message, errmsg
        endcase
    end
    else:message, errmsg
endcase

return, ''
end
