PRO EP_ORBIT,MJD2000,KODE,FILE,IERROR,NSAT,X,REVNUM
;
;P ORBIT: RETRIEVAL ROUTINE FOR COMPRESSED CLUSTER ORBIT
;
;I MJD2000 (DBL) = MODIFIED JULIAN DAY, FROM 2000, FOR THE STATE VECTOR
;I KODE    (INT) = NUMBER OF COMPONENTS OF STATE VECTOR = DIM. OF ARRAY X()
;                    = 3 FOR S/C POSITION, = 6 FOR POSITION & VELOCITY
;I FILE    (STR) = INPUT DATA FILE of Short Term Orbit File (STOF)
;                  this FILE can contain the 15-Byte-headers
;                   (at "mpecl":/nfs/cl1/rdda/clusterN/aux_N/yymmddba.1aN)
;                  or not -> for so-called history files
;                   (at "mpecl":/nfs/cl1/support/att_orb/stof.clN with "N"=S/C-No.)
;                  For short, the alias FILE='S/C-number' (1,2,3, or 4) can
;                  be used; then the 'history files' are accessed by default.
;
;O IERROR  (INT) = RETURN CODE: 0 = NO ERROR
;                               1 = 'MJD2000' TOO EARLY
;                               2 = TOO LATE
;                               3 = TIME GAP IN DATA
;                               4 = WRONG VALUE OF 'KODE'
;                               5 = FILE CONTENT INCONSISTENT
;                               6 = READ ERROR FROM DATA FILE
;O NSAT    (INT) = SATELLITE NUMBER: 1, 2, 3, or 4
;O X(KODE) (DBL) = SPACECRAFT POSITION, KM (AND VELOCITY, KM/S)
;O REVNUM  (DBL) = REVOLUTION NUMBER
;
;F READS A SEQUENTIAL FORMATTED Short Term Orbit File (STOF) 'FILE'
;
; History:
; ========
; 07-Dec-2000  Adapted from the FORTRAN program "orbit.f"; (c) mfo@mpe.mpg.de
; 04-Jan-2001  Handling of input file (FILE) both with 15-Byte-header and w/o
; 05-Jan-2001  Data gaps (err3) are ignored if MJD2000 not inside the gap
; 07-Jan-2001  FILE='S/C-No.' as short alias introduced (with default files)
; 10-Jul-2002  save-file replaced by COMMON orbit_idlsave_data -> runs faster
; 08-Oct-2002  fixed bug which caused non-closed files during read NULL parms
;============================================================================

  COMMON orbit_idlsave_data, stored,stored_data

  Y      = DBLARR(6)
  X      = DBLARR(KODE)
  COEFF  = DBLARR(10,6) ; MATRIX WITH UP TO 10 COEFFICIENTS OF THE CHEBYSHEV
  COEFF_K= DBLARR(6)    ; POLYNOMIAL FOR EACH OF THE 6 COMP.OF THE STATE VECTOR
  header = BYTARR(15)
  MFILE  = 'NO_FILE'    ; INITIALISE FILE TO FORCE FILE READING AT 1st CALL
  FILE   = strtrim(string(FILE),2)

  if n_elements(stored) EQ 0 then stored = 0B
  if stored then begin
    MFILE  = stored_data.MFILE  & COEFF    = stored_data.COEFF 
    EPOCH  = stored_data.EPOCH  & err      = stored_data.err
    header = stored_data.header & ID_PACKET= stored_data.ID_PACKET
    KOEFF  = stored_data.KOEFF  & MJDBEG   = stored_data.MJDBEG
    MJDEND = stored_data.MJDEND & MJDFIR   = stored_data.MJDFIR
    MJDLAS = stored_data.MJDLAS & NREC     = stored_data.NREC
    NSAT   = stored_data.NSAT   & OMOTIN   = stored_data.OMOTIN
    RDIST  = stored_data.RDIST  & REVEPO   = stored_data.REVEPO
    SMAXIS = stored_data.SMAXIS & Y        = stored_data.Y
  endif ; stored

  IERROR = 0            ; INITIALISE ERROR CODES

  if (KODE LE 0) OR (KODE GT 6) then goto,err4
 
; ALWAYS REWIND IF A NEW FILE IS USED
  if (FILE NE MFILE) then goto,INITIALISE
 
; CHECK IF 'MJD2000' IS INSIDE LAST READ RECORD BLOCK
  if (MJD2000 GT MJDEND+1.D-4) then goto,INITIALISE
  if (MJD2000 GE MJDBEG-1.D-4) then goto,NO_COEFF
 
INITIALISE:             ; INITIALISE THE READING FROM THE FILE

  if (FILE EQ '1') OR (FILE EQ '2') OR (FILE EQ '3') OR (FILE EQ '4') $
    then LFILE = '/nfs/cl1/support/att_orb/stof.cl'+FILE              $
    else LFILE = FILE
  openr,lun,LFILE,error=err,/get_lun
  if (err NE 0) then begin
    print,'"orbit.pro": FILE = ',lfile,'  provokes an ERROR!'
    print,!error_state.msg
    goto,err6
  end
  readu,lun,header
  free_lun,lun
  ID_PACKET = header(8)
  openr,lun,LFILE,error=err,/get_lun

  MJDFIR = 99.D9
  MJDLAS = 99.D9
  MFILE = FILE

FIRST_REC:              ; READ 1ST RECORD IN A BLOCK

  if ID_PACKET EQ 3 then begin
    READU,lun,header
    on_ioerror,err6
  end
 
  NSAT = 0
  readf,lun, NSAT, FORMAT="(I3)"
  on_ioerror,err6

; IF: NSAT = A SATELLITE NUMBER; THEN THIS IS 1ST RECORD IN A BLOCK
  if (NSAT LE 0) OR (NSAT GT 4) then goto,FIRST_REC
 
; READ 2ND RECORD IN THE BLOCK
  NREC   = 0            ; RECORD IDENTIFICATION, SHALL BE = 200 + NSAT
  MJDBEG = 9.99D99      ; BEGIN TIME OF THE RECORD (MJD2000)
  MJDEND = 9.99D99      ; END TIME OF THE RECORD (MJD2000)
  EPOCH  = 9.99D99      ; EPOCH OF REFERENCE STATE VECTOR (MJD2000)
  REVEPO = 9.99D99      ; REVOLUTION NUMBER AT EPOCH
  SMAXIS = 9.99D99      ; SEMIMAJOR AXIS FOR THE KEPLER ORBIT
  OMOTIN = 9.99D99      ; INVERSE MEAN MOTION FOR THE KEPLER ORBIT
  readf,lun, NREC,MJDBEG,MJDEND,EPOCH,REVEPO,SMAXIS,OMOTIN, $
               FORMAT="(I3,2F12.6,F15.9,F11.3,2F13.5)"
  on_ioerror,err6
; print, NREC,MJDBEG,MJDEND,EPOCH,REVEPO,SMAXIS,OMOTIN, $
;              FORMAT="(I3,2F12.6,F15.9,F11.3,2F13.5)"
; CHECK CONSISTENCY OF FILE
  if (NREC   NE 200 + NSAT) then goto,err5
  if (MJDBEG GT MJDEND    ) then goto,err5
 
; MJDFIR = START TIME OF 1ST RECORD ON 1ST BLOCK ON THE FILE - MARGIN
  MJDFIR = MIN([MJDFIR,MJDBEG-1.D-4])
; ERROR RETURN IF 'MJD2000' IS BEFORE START OF FILE (WITH MARGIN)
  if (MJD2000 LT MJDFIR) then goto,err1
; ERROR RETURN IF THERE IS A GAP FROM LAST BLOCK (WITH MARGIN)
; if (MJDBEG GT MJDLAS) then goto,err3
  if (MJDBEG GT MJDLAS) AND (MJD2000 LT MJDBEG-1.D-4) then goto,err3
; MJDLAS = END TIME OF LAST READ RECORD BLOCK + MARGIN
  MJDLAS = MJDEND + 2.D-4
 
; CONTINUE READ IF 'MJD2000' IS AFTER END OF THIS RECORD BLOCK
  if (MJD2000 GT MJDEND+1.D-4) then goto,FIRST_REC
; REWIND WHEN 'MJD2000' IS EARLIER THAN START OF PRESENT RECORD BLOCK
  if (MJD2000 LT MJDBEG-1.D-4) then goto,INITIALISE
 
; READ 3RD RECORD IN THE BLOCK
  NREC   = 0            ; RECORD IDENTIFICATION, SHALL BE =300+NR.OF POL.COEFF.
  Y      = DBLARR(6)    ; REFERENCE STATE VECTOR FOR KEPLER ORBIT (KM, KM/S)
  RDIST  = 9.99D9       ; S/C EARTH CENTRE DISTANCE AT EPOCH
  readf,lun, NREC,Y,RDIST, FORMAT="(I3,3F11.3,3F11.7,F11.3)"
  on_ioerror,err6
; print, NREC,Y,RDIST, FORMAT="(I3,3F11.3,3F11.7,F11.3)"

; CHECK CONSISTENCY OF FILE
  if (NREC GT 310) then goto,err5
  if (NREC LT 300) then goto,err5
; KOEFF = NUMBER OF POLYNOMIAL COEFFICIENTS, BETWEEN 0 AND 10
  KOEFF = NREC - 300
 
; IF THERE ARE NO COEFFICIENTS IN THIS BLOCK
  if (KOEFF LE 0) then begin
    free_lun,lun
    goto,NO_COEFF
  endif

  NREC   = 0            ;  NREC = RECORD IDENT. = KOEFF + 11*K
  for K = 1,KOEFF do begin
    readf,lun, NREC,COEFF_K, FORMAT="(I3,3F11.3,3F11.7)"
    on_ioerror,err6
    COEFF(K-1,*)=COEFF_K
;   print, NREC,COEFF(K-1,*), FORMAT="(I3,3F11.3,3F11.7)"
    if (11*K+KOEFF NE NREC) then goto,err5 ; CHECK CONSISTENCY OF FILE
  end

; END OF BLOCK READING SEQUENCE
  free_lun,lun
  stored_data = {MFILE  : MFILE ,  COEFF    : COEFF    , $
                 EPOCH  : EPOCH ,  err      : err      , $
                 header : header,  ID_PACKET: ID_PACKET, $
                 KOEFF  : KOEFF ,  MJDBEG   : MJDBEG   , $
                 MJDEND : MJDEND,  MJDFIR   : MJDFIR   , $
                 MJDLAS : MJDLAS,  NREC     : NREC     , $
                 NSAT   : NSAT  ,  OMOTIN   : OMOTIN   , $
                 RDIST  : RDIST ,  REVEPO   : REVEPO   , $
                 SMAXIS : SMAXIS,  Y        : Y           }
  stored = 1B

NO_COEFF:

; TIME CONVERTED TO DIFF. IN MEAN ANOMALY:
  DMANOM = (MJD2000 - EPOCH)*864.D2/OMOTIN

  REVNUM = REVEPO+DMANOM/6.2831853072D0 ; ORBIT NUMBER
 
  ARIN   = SMAXIS/RDIST                 ; START MODELLING KEPLER ORBIT
  ARM    = (RDIST - SMAXIS)/SMAXIS
  RVWAM  = (Y(0)*Y(3) + Y(1)*Y(4) + Y(2)*Y(5))*OMOTIN/SMAXIS/SMAXIS

  TAM    = DMANOM - RVWAM               ; CALC. OF ECC. ANOMALY
  COMP   = 1.D-7 + 1.D-10*ABS(TAM)      ;  (BY NEWTON'S ITERATION)
  B      = TAM

  for ITER = 1,15 do begin              ; ITERATIONS TO SOLVE KEPLER'S EQUATION:
    GO  = COS(B)
    G1  = SIN(B)
    BET = TAM - ARM*G1 + RVWAM*GO
    D   = (BET - B)/(1.D0 + ARM*GO + RVWAM*G1)
    B   = B + D
    if (ABS(D) LE COMP) then goto,OK    ; THIS GIVES THE ACCURACY
  end                                   ;  (1.D-14 IN B & THE G'S)

  goto,err5             ;  NO CONVERGENCE, ERROR RETURN

OK:

  GO = GO - D*G1
  G1 = G1 + D*GO
  G2 = 1.D0 - GO
  G3 = B - G1
  FX = 1.D0  - G2*ARIN
  GX = (DMANOM - G3)*OMOTIN
 
  K = MIN([KODE,3])-1
  for J = 0,K do    X(J) = FX*Y(J) + GX*Y(J+3)
 
  if (KODE LE 3) then goto,END_OF_KEPLER

  RX = SQRT(X(0)*X(0) + X(1)*X(1) + X(2)*X(2))
  FH = -G1*SMAXIS*ARIN/(OMOTIN*RX)
  GH = 1.D0 - G2*SMAXIS/RX
  for J = 3,KODE-1 do X(J) = FH*Y(J-3) + GH*Y(J)

END_OF_KEPLER:          ;  END OF MODELLING KEPLER ORBIT
 
; CHECK IF POLYNOMIAL COEFFICIENTS ARE REQUIRED (1 IS NOT WORTH WHILE)
  if (KOEFF LE 1) then return

; MID-POINT & SCALE FACTOR FOR CHEBYSHEV POLYNOMIAL
  MJDMID = 0.5D0*(MJDBEG + MJDEND)
  SCALE  = 4.D0/(MJDEND - MJDBEG)
; ADD CHEBYSHEV POLYNOMIAL TO KEPLER STATE VECTOR
  S      = SCALE*(MJD2000 - MJDMID)
  PA     = 1.D0
  P      = S*0.5D0
 
; 'KODE' = NUMBER OF COMPONENTS OF THE STATE VECTOR
  for J = 0,KODE-1 do X(J) = X(J) + COEFF(0,J) + COEFF(1,J)*P
 
  if (KOEFF LE 2) then return
  for L = 2,KOEFF-1 do begin
    PB = PA
    PA = P
    P = S*PA - PB
    for J=0,KODE-1 do X(J) = X(J) + COEFF(L,J)*P
  end

  return

err9: ; ERROR RETURNS; IERROR = 5, 6, 4, 3, 2 OR 1
      ; END-OF-FILE ONLY IF AT LEAST ONE RECORD HAS BEEN READ

  if (MJDLAS LT 1.D9) then goto,err2
  err5:  IERROR = -1
  err6:  IERROR = IERROR + 2
  err4:  IERROR = IERROR + 1
  err3:  IERROR = IERROR + 1
  err2:  IERROR = IERROR + 1
  err1:  IERROR = IERROR + 1

  free_lun,lun
; FORCE A RE-INITIALISATION OF READ AT NEXT CALL AFTER AN ERROR:
  MFILE = 'NO_FILE'

RETURN
END
