pro ep_errors_inertial, v_bpp, vsc_gse, gse2scs, scs2bpp, $ ; IN
                        tgyro, magd_error, phid_error, ambig_180, $ ; IN
                        magd_error_inertial, phid_error_inertial ; OUT
                        
; Non-inertial (SC frame) drift velocity magnitude error (magv_error) is
; proportional to the drift step magnitude error (magd_error):
magv_error = (magd_error*1.e-3)/(tgyro*1.e-6) ; km/s

; ambig_180=1 for RMT means derror/d > 1
; ambig_180=1 for TRI means that the part of the chi-squared surface
; having values lying between the minimum and a certain level up
; from the minimum (depending on the confidence level) has an angular
; extent greater than 90 degrees from the minimum, so that the target
; location is ill-defined to within 180 degrees.  This doesn't apply,
; however, to very small drift step cases where the target is very
; close to the origin.
;
; HERE:  If the non-inertial angle error is >= 90 degrees, then reset
; it to 89 degrees.  Go through the calculation below for the inertial
; magnitude and angle error.  Then if ambig_180=1 for the non-inertial
; angle, then reset the inertial angle error to 180 degrees.

if (phid_error ge 90.) then phiv_error = 89.*!dtor else $
  phiv_error = phid_error*!dtor ; radians, v and d are parallel to one-another

; Rotate the spacecraft velocity into the BPP system
vsc_scs = [total(gse2scs(0,*)*vsc_gse), $
           total(gse2scs(1,*)*vsc_gse), $
           total(gse2scs(2,*)*vsc_gse)] ; km/s
vsc_bpp = [total(scs2bpp(0,*)*vsc_scs), $
           total(scs2bpp(1,*)*vsc_scs), $
           total(scs2bpp(2,*)*vsc_scs)] ; km/s

; Only working IN the BPP (B-perp-plane)
; Not concerned with the component out of the plane
vsc_perp_bpp = [vsc_bpp(0), vsc_bpp(1), 0.] ; km/s

; Construct the drift velocity WRT the Earth rest frame (v0)
; Construct the 4 vectors which define the error box (v1,v2,v3,v4)
; These 4 are all with respect to the Earth's rest frame
v0 = vsc_perp_bpp + v_bpp       ; Drift velocity WRT Earth rest frame
magv0 = sqrt(total(v0^2))

magv = sqrt(total(v_bpp^2))     ; km/s
vhat_parl = v_bpp/magv          ; v_bpp lies in the x-y BPP plane
vhat_perp = crossp(vhat_parl,[0.,0.,1.]) ; perp to v_bpp AND in the x-y BPP plane

dv_parl = magv_error*vhat_parl  ; km/s
v1 = vsc_perp_bpp + v_bpp + dv_parl ; km/s
v2 = vsc_perp_bpp + v_bpp - dv_parl ; km/s

dv_perp = magv*tan(phiv_error)*vhat_perp ; km/s
v3 = vsc_perp_bpp + v_bpp + dv_perp ; km/s
v4 = vsc_perp_bpp + v_bpp - dv_perp ; km/s

d1 = sqrt(total((v0-v1)^2))
d2 = sqrt(total((v0-v2)^2))
d3 = sqrt(total((v0-v3)^2))
d4 = sqrt(total((v0-v4)^2))
err = (max([d1,d2,d3,d4]) + min([d1,d2,d3,d4]))/2. ; km/s
magd_error_inertial = (err*1.e+3)*(tgyro*1.e-6) ; m

a1 = acos(total(v0*v1)/magv0/sqrt(total(v1^2)))
a2 = acos(total(v0*v2)/magv0/sqrt(total(v2^2)))
a3 = acos(total(v0*v3)/magv0/sqrt(total(v3^2)))
a4 = acos(total(v0*v4)/magv0/sqrt(total(v4^2)))
err = max([a1,a2,a3,a4])        ; radians
if (err ge !pi/2. or ambig_180 eq 1) then err = !pi
phid_error_inertial = err*!radeg ; degrees

; Check
;message, 'Check the inertial errors', /cont
;stop
;IDL> plot, [0,vsc_perp_bpp(0)],[0,vsc_perp_bpp(1)], xrange=[-4,0], yrange=[-1,1], /nodata                           
;IDL> arrow, 0,0,vsc_perp_bpp(0),vsc_perp_bpp(1), /data, color=cs.red                                                
;IDL> arrow, vsc_perp_bpp(0),vsc_perp_bpp(1),vsc_perp_bpp(0)+v_bpp(0),vsc_perp_bpp(1)+v_bpp(1), /data, color=cs.green
;IDL> arrow, 0,0,v0(0),v0(1),/data,color=cs.magenta                                                                  
;IDL> arrow, 0,0,v1(0),v1(1),/data,color=cs.orange                                                                   
;IDL> arrow, 0,0,v2(0),v2(1),/data,color=cs.orange                                                                   
;IDL> arrow, 0,0,v3(0),v3(1),/data,color=cs.cyan                                                                     
;IDL> arrow, 0,0,v4(0),v4(1),/data,color=cs.cyan                                                                     
;IDL> print, (max([d1,d2,d3,d4]) + min([d1,d2,d3,d4]))/2.
;      0.24731432
;IDL> print, phid_error_inertial
;       6.1539528

return
end
