; This routine is used by plot_bestarg.pro
; It is not a part of the EDI Prime Parameters production code.

pro ep_calc_para, xd, yd, $     ; Target
                  alpha,  $     ; Beams
                  cs, $         ; Color structure
                  sdev_out, $
                  sdev, noplot=noplot

; xd and yd are scalars: X and Y of the target, in meters
; alpha is a 1-D vectors:
;    alpha(i) = firing angle of gun for beam i, in radians

; Calculate towards and away AS IF THE BEAMS ARE FIRED FROM THE
; SPACECRAFT CENTER, NOT THE GUN LOCATION!!!  IMPORTANT DIFFERENCE
; FROM THE REAL TOWARDS/AWAY DISTINCTION!!!
nbeam = n_elements(alpha)
that = [xd,yd]                  ; [2]
tmag = sqrt(total(that^2))      ; scalar
that = that/tmag                ; [2], Unit length
that = make_array(nbeam,/float,value=1.)#that ; [nbeam,2], Unit length
ghat = [[cos(alpha)],[sin(alpha)]] ; [nbeam,2], Unit length
dot = total(that*ghat,2)        ; [nbeam], dot>0 towards, dot<0 away

to = where(dot ge 0)
aw = where(dot lt 0)

; Start with the original firing directions
alp = alpha

; Flip all of the away beams
if (aw(0) ne -1) then alp(aw) = alp(aw) + !pi

; All of the beams should now be within 180 degrees of each other, but the
; edge of this 180 degree fan can have any numerical value.
; Need to rotate this fan until all of the angular values are between
; 0 and 180 so that an average can be performed on the angles.  This
; rotation is one that puts the target vector at 90 degrees.
tangle = atan(yd,xd)            ; [-180,180]
rotangle = !pi/2. - tangle
alp = alp + rotangle

; Could at this point have some angles greater than 360 (say alp is
; 340 and rotangle is 50, for instance).  So make sure they're all
; between 0 and 180.
x = cos(alp)
y = sin(alp)
alp = atan(y,x)

; OK, NOW all beams are within 180 degrees of each other, AND they
; have numerical values ranging from 0 to 180.  We can now take the
; average and standard deviation:

mean_angle = mean(alp) - rotangle
stdev_angle = stddev(alp)
alp = alp - rotangle            ; Go back to actual system for plot
sdev_out = stdev_angle*!radeg

if (keyword_set(noplot)) then return

; Create the [x], [y] footprints for polyfilling the
; target error swath
rmin = 0.
rmax = 1.
amin = mean_angle - stdev_angle
amax = mean_angle + stdev_angle

n = 100

slp = (rmax - rmin) / float(n-1)
r1 = slp*findgen(n) + rmin
a1 = make_array(n,/float,value=amax)

slp = (amin - amax) / float(n-1)
a2 = slp*findgen(n) + amax
r2 = make_array(n,/float,value=rmax)

slp = (rmin - rmax) / float(n-1)
r3 = slp*findgen(n) + rmax
a3 = make_array(n,/float,value=amin)

slp = (amax - amin) / float(n-1)
a4 = slp*findgen(n) + amin
r4 = make_array(n,/float,value=rmin)

eswathx_bpp = [r1*cos(a1),r2*cos(a2),r3*cos(a3),r4*cos(a4)]
eswathy_bpp = [r1*sin(a1),r2*sin(a2),r3*sin(a3),r4*sin(a4)]

plot, [0], [0], /nodata, xrange=[-1,1], yrange=[-1,1]
polyfill, eswathx_bpp, eswathy_bpp, color=cs.grey60, /data
for i=0,nbeam-1 do begin
    arrow, 0., 0., cos(alp(i)), sin(alp(i)), /data
endfor
arrow, 0.,0.,xd/sqrt(xd^2+yd^2), yd/sqrt(xd^2+yd^2), color=cs.red, /data, thick=2
arrow, 0., 0., cos(mean_angle), sin(mean_angle), color=cs.green, /data, thick=2

outs = norm_axis([.1,.4])
xyouts, outs(0), outs(1), 'Mean Angle: '+strtrim(mean_angle*!radeg,2)+ $
  '!cSTDEV Angle: '+strtrim(stdev_angle*!radeg,2)+ $
  '!cSTDEV Angle in aux file: '+ strtrim(sdev*!radeg,2)

return
end
