;	beamwidth.pro,v 1.3 2001/05/25 17:00:03 ppq Exp	
; ============================================================================
  FUNCTION BeamWidth, gx, gy, gz, $
                      bx, by, bz, $
                      dph=dph, dth=dth, $
                      debug=debug 
; ============================================================================
; Created 15-May-2001  Hans Vaith, MPE Garching
; ============================================================================
; ****************************************************************************
; This function calculates the projection of the beam width into the plane 
; perpendicular to the magnetic field, assuming an elliptically shaped beam
; profile with either predefined polar angle dependence of the semi-axes or
; setable semi-axes values via keywords.
; 
; Parameters:
;   gx, gy, gz : arrays of cartesian components of the gun firing unit vector 
;                in the respective spinning GDU frame of reference, as 
;                calculated directly from the telemetered analytical gun 
;                voltages. 
;                The GDU1 and GDU2 coordinate systems are related by the 
;                coordinate transformation (x1=x2, y1=-y2, z1=-z2)
;   bx,by,bz   : arrays of cartesian components of the magnetic field in the 
;                respective GDU frame of reference (diffferent for GDU1 and 
;                GDU2 !!!)
;                Needed to determine the orientation of the Bperp plane in
;                the respective frame of reference
;   dph, dth   : optional keywords to allow setting of the semi-axes of the 
;                beam profile ellipse
;
; Return value:
;   An array of angular widths of the beam profile in the Bperp plane 
;   (in degrees)
; ****************************************************************************

fname = 'BeamWidth(): '

if not keyword_set(debug) then debug = 0

; check for complete set of parameters
; ------------------------------------
if n_params() ne 6 then begin
   print, fname + 'expecting 6 parameters (gx,gy,gz, bx,by,bz)'
   retall ; force program exit
endif

; make sure all arrays have the same dimension
; --------------------------------------------
NN = n_elements(gx)
if n_elements(gy) ne NN or n_elements(gz) ne NN or n_elements(bx) ne NN or $
   n_elements(by) ne NN or n_elements(bz) ne NN then begin
   print, fname + 'array dimensions are different.'
   print, 'n_elements(gx) : ', n_elements(gx)
   print, 'n_elements(gy) : ', n_elements(gy)
   print, 'n_elements(gz) : ', n_elements(gz)
   print, 'n_elements(bx) : ', n_elements(bx)
   print, 'n_elements(by) : ', n_elements(by)
   print, 'n_elements(bz) : ', n_elements(bz)
   retall
endif

; make sure that the gun vector is normalized (for calculation of angles)
; -----------------------------------------------------------------------
gn = sqrt(double(gx)^2 + double(gy)^2 + double(gz)^2)
gxn = gx / gn
gyn = gy / gn
gzn = gz / gn

; calculate azimuth and polar angle of gun firing direction
; ---------------------------------------------------------
phi = dblarr(NN)
x = where( abs(gxn) ge 1.0d-8 or abs(gyn) ge 1.0d-8)
if x(0) ne -1 then phi(x) = atan(gyn(x), gxn(x))

theta = acos(gzn)

; set semi-axes in phi direction from keyword or let it vary from 1.0/2 
; to 0.5/2 as theta goes from 0 to 90 degrees 
; ---------------------------------------------------------------------
if keyword_set(dph) then dph_2 = dblarr(NN) + 0.5 * dph $
else                     dph_2 = 0.5 * ( 1 - theta/!pi )

; set semi-axis in theta direction from keyword, or let it vary from 1.0/2
; to 4.0/2 as theta goes from 0 to 90 degrees
; ------------------------------------------------------------------------
if keyword_set(dth) then dth_2 = dblarr(NN) + 0.5 * dth $
else                     dth_2 = 0.5 * ( 1 + 6.0 * theta/!pi )


; calculate ep = (g x b); this is a tangent vector along the Bperp line
; (gives the orientation of the dth semi-axis of the beam profile ellipse)  
; -----------------------------------------------------------------------
epx = gyn*bz - gzn*by
epy = gzn*bx - gxn*bz
epz = gxn*by - gyn*bx
epn = sqrt(epx^2 + epy^2 + epz^2)
epx = epx/epn & epy = epy/epn & epz = epz/epn

; calculate eth; this is the unit vector which is locally oriented along theta
; ----------------------------------------------------------------------------
ethx = cos(phi)*cos(theta)
ethy = sin(phi)*cos(theta)
ethz = -sin(theta)

; calculate angle between ep and eth
; ----------------------------------
alpha = acos(epx*ethx + epy*ethy + epz*ethz)

sigma = 2*sqrt( (1+tan(alpha)^2) / $
                ( (1/dth_2^2) + (tan(alpha)^2/dph_2^2) ) )

if debug then begin
   help, alpha
   help, sigma
   print, 'alpha = ', alpha*180/!pi
   print, 'sigma = ', sigma
endif

return, sigma

end



