;++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
; $Id: sc2scs.pro,v 1.5 2003/10/28 19:24:32 hav Exp $
;++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

@vts_util.pro
@vtstriang.pro

; ------------------------------------------------------------------------
; despin gun firing directions and gun positions
; At sun reference pulse time the position of GDU1 has rotated past
; the sun direction by 63.8 degrees in the counterclockwise direction
; when viewing the SC from the top (+x)
; ------------------------------------------------------------------------


; ===========================================================================
  FUNCTION SC2SCS_OLD, invec, ct, sd, srp=srp, tspin=tspin, $
                   reverse=reverse, $
                   debug=debug
; ===========================================================================
;
; +++ NOTE +++ NOTE +++ NOTE +++ NOTE +++ NOTE +++ NOTE +++ NOTE +++
;
;     This function uses a single sun reference pulse to despin
;     the vectors. This is ok for short time periods close to
;     sun reference pulse, but for longer periods this gives
;     BAD results (as can be seen from calculating g dot B)
;     Therefore use the newer function SC2SCS() instead,
;     which attempts to use, for each vector, the closest sun
;     reference pulse
;
; +++ NOTE +++ NOTE +++ NOTE +++ NOTE +++ NOTE +++ NOTE +++ NOTE +++
;
;
; This function despins a vector array (invec) from spinning GDU1 coordinates
; (=SC coordinates) to non-spinning SC-Sun (SCS) coordinates. This involves
; a swapping of x and z directions as a final step.
;
; There are two ways to call this function
;
;    outvec = SC2SCS( invec, ct, sd )
;    outvec = SC2SCS( invec, ct, srp=srp, tspin=tspin)
;
; where
;    invec is an array of structures containing elements x,y,z
;          ({x:0.0,y:0.0,z:0.0})
;    ct    is an array of associated time tags in t70 format
;    sd    is a structure as returned by GetSrpData() in readlib.pro
;          it contains spin period and sun reference pulse data
;
;    srp   is an array of sun reference pulses in t70 format
;    tspin is an array of spin periods
;
; Other keywords
;
;    reverse  perform the inverse transformation
;    debug    print out debugging information
; ****************************************************************************

  message, 'This function is deprecated and should no longer be used!', /cont
  retall

  ; check keywords
  ; --------------
  if keyword_set(reverse) then reverse=1 else reverse=0
  if keyword_set(debug)   then debug=1   else debug=0

  ; check which passing mechanism was used
  ; --------------------------------------
  if n_params() eq 3 then begin ; sd is present
     tspin = total(sd.data.tspin) / n_elements(sd.data)

     x = where(sd.data.ct lt ct(0))
     if x(0) eq -1 then srp = sd.data(0).ct $
     else srp = sd.data(n_elements(x)-1).ct
  endif else if n_params() eq 2 then begin
     if not keyword_set(srp) then message, 'missing srp (sun ref pulse)'
     if not keyword_set(tspin) then message, 'missing tspin'
  endif else begin
      message, 'need either three parameters or two parameters and keywords srp,tspin'
      retall
  endelse

  ; some more consistency checks
  ; ----------------------------
  if n_elements(invec) ne n_elements(ct) then begin
     message, '(invec,ct):  incompatible dimensions'
  endif

  ; copy invec
  ; ----------
  myinvec = invec

  ; first switch axes if going from scs to sc
  ; -----------------------------------------
  if reverse eq 1 then begin
     tmp = myinvec.x
     myinvec.x = myinvec.z
     myinvec.z = tmp
     myinvec.y = - myinvec.y
  endif

  ; do calculations
  ; ---------------
  tref = srp - (63.8/360)*tspin

  wt = 2*!PI/tspin * (ct - tref)
  sinwt =  sin(wt)
  coswt =  cos(wt)

  ; despin the firing directions and positions
  ; ------------------------------------------
  outvec = vDeSpin_(myinvec, coswt, sinwt, reverse=reverse)

  ; finally switch axes if going from sc to scs
  ; -------------------------------------------
  if reverse eq 0 then begin
     tmp = outvec.x
     outvec.x = outvec.z
     outvec.z = tmp
     outvec.y = - outvec.y
  endif

  return, outvec

END


; ===========================================================================
  FUNCTION SC2SCS, vts_in, $
                   srp=srp, tspin=tspin, sc_id=sc_id, $
                   reverse=reverse, fgm=fgm, $
                   debug=debug
; ===========================================================================
; Despins the vector invec from rotating GDU1 coordinates (SC) to
; spacecraft-sun coordinates (SCS). This includes a swapping of the
; x and z axis (and inversion of y to maintain a right-handed coordinate
; system)
;
;    vts_in  array(VTS_T)
;
; Other keywords
;
;    srp      is an array of sun reference pulses in t70 format
;    tspin    is an array of spin periods
;
;             if srp or tspin keyword is not present then both will be read automatically
;
;    sc_id    needed if srp or tspin is missing
;
;    reverse  perform the inverse transformation
;    debug    print out debugging information
;    fgm      if set the transformation is done between rotating FGM system
;             and SCS (rotating FGM is 6.5 degrees off from rotating GDU1)
;
; ***************************************************************************

compile_opt strictarr

  ; check keywords
  ; --------------
  if keyword_set(fgm) then fgm=6.5 else fgm = 0
  if keyword_set(reverse) then reverse=1 else reverse=0
  if keyword_set(debug)   then debug=1   else debug=0

  if n_params() ne 1 then begin
     message, 'need exactly one parameter: (VTS_T)', /cont
     retall
  endif

  vts_debug, 1
  dummy=vts_check(vts_in, /err_exit)

  NN = n_elements(vts_in)

  ct = vts_in.ct

  ; some more consistency checks
  ; ----------------------------
  if not keyword_set(srp) or not keyword_set(tspin) then begin
     if not keyword_set(sc_id) then begin
        message, 'need keyword sc_id', /cont
        retall
     endif
     sd    = GetSrpData(sc_id, ctr=[min(ct),max(ct)],/quiet)
     if sd.hdr.status eq 1 then begin
        message, sd.hdr.msg, /cont
        retall
     endif
     srp   = sd.data.ct
     tspin = sd.data.tspin
  endif

  N_SD = n_elements(srp)
  if N_SD ne n_elements(tspin) then begin
     message, '(srp,tspin):  incompatible dimensions', /cont
     retall
  endif

  if debug then help, NN, N_SD

  ; get largest sun ref pulse lower than lowest time tag (srp0)
  ; and smallest sun ref pulse higher than highest time tag (srp1)
  ; --------------------------------------------------------------
  x = where(srp lt ct[0])
  if x[0] eq -1 then srp0 = srp[0] $
  else srp0 = srp[x[n_elements(x)-1]]

  x = where(srp gt ct[NN-1])
  if x[0] eq -1 then srp1 = srp[N_SD-1] $
  else srp1 = srp[x[0]]

  if debug then begin
     print, 'srp0 and srp1 : '
     print, clutimeval2str(srp0)
     print, clutimeval2str(srp1)
  endif

  if debug then begin
     x = where(ct lt srp0, cnt)
     print, '#data < srp0 : ', cnt
     x = where(ct gt srp1, cnt)
     print, '#data > srp1 : ', cnt
  endif

  ; extract all sun ref pulses inbetween srp0 and srp1
  ; --------------------------------------------------
  x = where(srp gt (srp0-1.d-3) and srp lt (srp1+1.d-3))
  srp_arr = srp[x]
  tspin_arr = tspin[x]
  tspin_avg = mean(tspin_arr)
  N_SRP = n_elements(srp_arr)

  if debug then help, N_SRP

  ; calculate the indices of the sun ref pulses nearest to each time tag
  ; --------------------------------------------------------------------
  srp_idx = long( interpol(lindgen(N_SRP), srp_arr-srp0, ct-srp0) + 0.5 )

  x = where(srp_idx lt 0, cnt)
  if cnt gt 0 then begin
     if debug then begin
        print, 'detected ' + strtrim(cnt,2) + ' indices below zero'
        print, 'average of offending indices : ', mean(srp_idx[x])
        print, 'highest                      : ', max(srp_idx[x])
        print, 'lowest                       : ', min(srp_idx[x])
     endif

     srp_idx[x] = 0
  endif

  x = where(srp_idx ge N_SRP, cnt)
  if cnt gt 0 then begin
     if debug then begin
        print, 'detected ' + strtrim(cnt,2) + ' indices above limit (' $
               + strtrim(N_SRP-1,2) + ')'
        print, 'average of offending indices : ', mean(srp_idx[x])
        print, 'highest                      : ', max(srp_idx[x])
        print, 'lowest                       : ', min(srp_idx[x])
     endif

     srp_idx[x] = N_SRP-1
  endif



  ; create array of nearest neighbor sun ref pulses
  ; -----------------------------------------------
  srp_int   = srp_arr[srp_idx]
  tspin_int = tspin_arr[srp_idx]

  if debug then begin
     x = where(abs(ct-srp_int) gt 1.5*tspin_avg, cnt)
     if cnt ne 0 then begin
        print, '#data with distance srp<->time-tag > 1.5*tspin : ', cnt
        print, 'average distance of these : ', mean(abs(ct[x]-srp_int[x]))
        print, 'median                    : ', median(abs(ct[x]-srp_int[x]))
     endif else begin
        print, 'all sun ref pulses chosen close to time tags. good!'
     endelse
  endif

  ; calculate reference time for despinning
  ; ---------------------------------------
  tref = srp_int - (63.8-fgm)/360 * tspin_int

  ;  ... and phase array
  ; --------------------
  wt = ( (ct-tref) * 2*!PI / tspin_int ) mod (2*!PI)


  ; copy invec
  ; ----------
  myinvec = vts_in.val

  ; first switch axes if going from scs to sc
  ; -----------------------------------------
  if reverse eq 1 then begin
     tmp = myinvec[0,*]
     myinvec[0,*] = myinvec[2,*]
     myinvec[2,*] = tmp
     myinvec[1,*] = - myinvec[1,*]
  endif


  ; despin the firing directions and positions
  ; ------------------------------------------
  myoutvec = vDeSpin_(myinvec, wt, reverse=reverse)

  ; finally switch axes if going from sc to scs
  ; -------------------------------------------
  if reverse eq 0 then begin
     tmp = myoutvec[0,*]
     myoutvec[0,*] = myoutvec[2,*]
     myoutvec[2,*] = tmp
     myoutvec[1,*] = - myoutvec[1,*]
  endif

  vts_out = replicate( {VTS_T}, NN )

  vts_out.val = myoutvec
  vts_out.ct  = ct

  return, vts_out

END
