/******************************************************************************
                          wwconv.cpp  -  description
                             -------------------
    begin                : Thu Dec 14 10:57:00 CET 2000

    author               : Hans Vaith
    email                : hav@mpe.mpg.de
    copyright            : (C) 2000 by Max-Planck-Institut fr extra-
                           terrestrische Physik, D-85740 Garching

    Cluster-II EDI: print WW pacmo1/5 converted data

    modifications
         18-Dec-2000  use popen() to call cl2dmp
******************************************************************************
$Id: wwconv.cpp,v 1.8 2004/11/02 11:31:05 hav Exp $
$Log: wwconv.cpp,v $
Revision 1.8  2004/11/02 11:31:05  hav
modified DeAlias() to support amGE pacmo==0 and the programs that call it.

Revision 1.7  2004/05/06 09:55:39  hav
fixed packet counter bug when checking for correct telemetry mode

Revision 1.6  2004/01/23 15:58:22  hav
added option -m {nm|bm} to select tm mode

Revision 1.5  2003/08/01 11:01:30  hav
added process id in binary output file name for uniqueness

Revision 1.4  2003/04/12 16:36:36  hav
removed gd12 error message

Revision 1.3  2003/04/12 16:26:47  hav
added tm extraction for one_gd_pair == 12 (SC2)

Revision 1.2  2002/10/15 12:42:20  hav

added check for NMx PM1 (not yet supported)

Revision 1.1.1.1  2002/10/14 17:08:19  hav
c/c++ sources

Revision 1.13  2002/10/11 17:42:53  hav
changed to structure ww_conv2_t

Revision 1.12  2002/10/10 13:06:43  hav
check for single gd pair tm mode and terminate

Revision 1.11  2001/11/12 17:36:19  hav
removed some debugging messages

Revision 1.10  2001/09/12 12:02:49  hav
added rcsid

Revision 1.9  2001/09/12 12:00:01  hav
streamlined, added option of binary output for easy & fast input in IDL

Revision 1.8  2001/08/10 11:32:10  hav
new command line options to extract only part of the data

Revision 1.7  2001/07/25 15:23:57  hav
*** empty log message ***

Revision 1.6  2001/07/10 15:11:36  hav
extended precision of tof,dtof to three subdecimal digits

Revision 1.5  2001/04/02 13:04:40  hav
*** empty log message ***

Revision 1.4  2001/02/20 15:52:30  hav
new cdds ASCII output format: filename in first line

Revision 1.3  2001/02/14 12:32:30  hav
common rcs format

Revision 1.2  2001/02/14 11:28:11  hav
location of cl2dmp binary now in  a header file

******************************************************************************/


/******************** INCLUDE FILES ******************************************/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>
#include <sys/types.h>
#include <unistd.h>
#include "openfile.h"
#include "editm.h"
#include "cl2tm.h"
#include "dealias.h"

/******************** CONSTANTS AND MACROS **********************************/

/* these are for all modes */
#define FMT1  "%8.1f%8.1f%8.1f%8.1f%10.3f%10.3f%10.3f"  /*th/ph1 th/ph2 tof1/2, dtof */
#define FMT2  "%3hd%3hd%2hd%2hd"        /* e n q1 q2 */

/* only for NM1 PM1 */
#define FMT3  "%4hd%3hd%3hd%3hd"        /* ix_lo1 ix_lo2 ix_hi1 ix_hi2 */

/* only for NM1 PM5 and BM1 PM1 */
#define FMT4  "%6hd%5hd%5hd%5hd"        /* d14_1 d15_1 d14_2 d15_2 */
#define FMT5  "%4hd%3hd"        /* maxaddr11 maxaddr2 */
#define FMT6  "%4hd%3hd%3hd%3hd"        /* bci_idx1 frm_idx1 bci_idx2 frm_idx2 */

/* for new tof format: add m and overflow flags */
#define FMT7  "%5hd%4hd%2hd%2hd"        /* m ovfl_t1 ovfl_t2 ovfl_dt */

#define INVALID_TT 0x800

#define TM_SEL_NM 1
#define TM_SEL_BM 2

/******************** TYPE DEFINITIONS **************************************/


/******************** FUNCTION PROTOTYPES ***********************************/

void ConvertWWData (sci_t sbuf, short i, short stlen,
                    ww_conv2_t *p1, ww_conv2_t *p2);
void PrintWWData (FILE * outfp, ww_conv2_t d);

void ConvertToF (short tof1_int, short dtof_int, float *tof1, float *tof2,
                 int n);
void ConvertVxVy (float vx, float vy, float *php, float *thp);

static void Usage (const char *exename);
void AddCorrection(ww_conv2_t d1, ww_conv2_t d2, short i, short submo,
                   short pacmo, short fr_per_st, short *dt1, short *dt2);

/******************** GLOBAL STATIC DATA ************************************/

/******************** LOCAL STATIC DATA *************************************/

static int oldtof;
static short shdr_m;
static short shdr_codetype;
static short nconv[] = { 1, 2, 4, 8, 8, 16, 32, 64 };
static short mconv[] = { 16, 8, 4, 2 };
static int one_gd_pair;
static int tm_sel;

static char rcsid[] = "$Id: wwconv.cpp,v 1.8 2004/11/02 11:31:05 hav Exp $";

/******************** MAIN PROGRAM ******************************************/

// ===========================================================================
int
main (int argc, char *argv[])
// ===========================================================================
{
   FILE *ofp = stdout;
   char bin_name[256];

   tmpacket tmpkt;
   scet *tp = &(tmpkt.hdr.sc_time);

   short scimo, submo, pacmo, numst, stlen;
   short tmr_tt, data_tt, delta_tt;
   short fr_per_st, dt1, dt2;

   int begin_spec = 0;
   int end_spec = 0;
   int t_spec = 0;
   int debug = 0;
   int binary = 0;

   scet begin, end, current;

   char tch;
   int delta;

   ww_conv2_t d1, d2;

   sprintf(bin_name, "wwconv%d.bin", (int)getpid()) ;

   tmpkt.pktno = 0;             //

   if (0) printf("%s\n", rcsid);

   // Process command line arguments and options
   // ------------------------------------------
   int ch;
   while (1) {
      if ((ch = getopt (argc, argv, "hdbs:e:t:m:")) == -1)
         break;

      switch (ch) {

      case '?':
      case 'h':
         Usage (argv[0]);
         exit (1);

      case 'd':
         debug = 1;
         break;

      case 'b':
         binary = 1;
         break;

      case 's':
         begin_spec = 1;
         if (sscanf(optarg, "%4hd-%2hd-%2hd%*[tT]%2hd:%2hd:%2hd",
                            &begin.year, &begin.month, &begin.day,
                            &begin.hr, &begin.min, &begin.sec) != 6) {
            fprintf(stderr, "could not read begin date and time from <%s>\n",
                            optarg);
            Usage(argv[0]);
            exit(1);
         } else {
            TL_Set(&begin, begin.year, begin.month, begin.day,
                           begin.hr, begin.min, begin.sec, 0, 0, 0);
         }
         break;

      case 'e':
         if (t_spec) {
            fprintf(stderr, "Must not specify both -e and -t\n");
            Usage(argv[0]);
            exit(1);
         }

         end_spec = 1;
         if (sscanf(optarg, "%4hd-%2hd-%2hd%*[tT]%2hd:%2hd:%2hd",
                            &end.year, &end.month, &end.day,
                            &end.hr, &end.min, &end.sec) != 6) {
            fprintf(stderr, "could not read end date and time from <%s>\n",
                            optarg);
            Usage(argv[0]);
            exit(1);
         } else {
            TL_Set(&end, end.year, end.month, end.day,
                         end.hr, end.min, end.sec, 0, 0, 0);
         }
         break;

      case 't':
         if (end_spec) {
            fprintf(stderr, "Must not specify both -e and -t\n");
            Usage(argv[0]);
            exit(1);
         }

         t_spec = 1;
         if (sscanf(optarg, "%d%c", &delta, &tch) != 2) {
            fprintf(stderr, "could not read delta time unit: <%s>\n",
            optarg);
            Usage(argv[0]);
            exit(1);
         }
         break;
      case 'm' :
         if (strcmp(optarg,"nm") == 0) {
            tm_sel = TM_SEL_NM;
         } else if (strcmp(optarg,"bm") == 0) {
            tm_sel = TM_SEL_BM;
         } else {
            fprintf(stderr, "unknown tm mode option : %s\n", optarg);
            Usage(argv[0]);
            exit(1);
         }
         break;
      }
   }


   if (argc <= optind) {
      fprintf(stderr, "Missing input file name\n");
      Usage(argv[0]);
      exit(1);
   }


   // Open the data file
   // ------------------
   CL2TM_OpenFile (argv[optind]);

   if (binary) ofp = OpenFile(bin_name, OPF_WB, OPF_EXIT);

   // if begin time has not been specified on the command line,
   // determine it from the time of the first packet
   if (!begin_spec) {
      if (CL2TM_ReadPacket(&tmpkt) == -1) {
         fprintf(stderr, "EOF encountered reading first packet!\n");
         exit(1);
      }
      begin = tmpkt.hdr.sc_time;
      TL_AddCt(&begin, -5.152221); // subtract one TMR
   }

   if (debug) {
      fprintf(stderr, "Begin time: ");
      TL_PrintDateTime(&begin, stderr);
   }

   if (t_spec) {

      end = begin;

      switch (tch) {
      case 's':
      case 'S': TL_Add(&end, TL_SEC, delta); break;
      case 'm':
      case 'M': TL_Add(&end, TL_MIN, delta); break;
      case 'h':
      case 'H': TL_Add(&end, TL_HOUR, delta); break;
      default:
         fprintf(stderr, "option -t: bad time specification character <%c>\n",
                        tch);
         Usage(argv[0]);
         exit(1);
      }

   }

   if (debug && (end_spec || t_spec)) {
      fprintf(stderr, "End time  : ");
      TL_PrintDateTime(&end, stderr);
   }

   if (!binary)
      fprintf (ofp, "%s\n", argv[optind]);

   // read all data packets in a loop
   // ===============================
   while (CL2TM_ReadPacket (&tmpkt) != -1) {

      // make sure it's EDI science data
      if ((tmpkt.hdr.data_source != CL2TM_DSRC_EDI) ||
          (tmpkt.hdr.data_type != CL2TM_NSD_DATA &&
           tmpkt.hdr.data_type != CL2TM_BSD_DATA)) {
         ++tmpkt.pktno;
         continue;
      }

      if (tm_sel == TM_SEL_NM && tmpkt.hdr.data_type != CL2TM_NSD_DATA) {
         ++tmpkt.pktno;
         continue;
      }
      if (tm_sel == TM_SEL_BM && tmpkt.hdr.data_type != CL2TM_BSD_DATA) {
         ++tmpkt.pktno;
         continue;
      }


      // check if packet time is within specified interval
      current = tmpkt.hdr.sc_time;
      TL_AddCt(&current, -5.152221); // subtract one TMR
      if (current.ct < begin.ct) {
         ++tmpkt.pktno;
         continue;
      }

      if (t_spec || end_spec) {
         if (current.ct > end.ct)
            break;
      }

      // extract mode information
      scimo = (tmpkt.data[0] >> 3) & 0x1f;
      submo = tmpkt.data[0] & 0x7;
      pacmo = (tmpkt.data[2] >> 8) & 0xf;

      // extract and time tags
      tmr_tt = tmpkt.data[4];
      data_tt = tmpkt.data[20];

      // see if it is WW and correct submo / pacmo
      if (scimo != 5) {
         ++tmpkt.pktno;
         continue;
      }

      if (submo >= 1 && submo <= 3 && pacmo == 1) {
         fprintf(stderr, "NMx PM1 currently not supported "
                             "--> dropping TM packet\n");
         ++tmpkt.pktno;
         continue;
      }

      if ((submo >= 1 && submo <= 3 && pacmo == 5) ||
          (submo == 5 && pacmo == 1)) {

         if (submo == 5 || pacmo == 5) {
            stlen = 10;
            if (submo==5) fr_per_st = 64;
            else          fr_per_st = 512;
         }
         else {
            stlen = 5;
            fr_per_st = 256;
         }

         // make sure that we are not in the one gun-detector pair tm mode

         if ((tmpkt.data[0] & 0x0100) == 0x0100) {
            if ((tmpkt.data[0] & 0x0200) == 0) {
               one_gd_pair = 21; // gun 2 / detector 1
            } else {
               one_gd_pair = 12; // gun 1 / detector 2
            }
         } else {
            one_gd_pair = 0;
         }

         delta_tt = DeAlias (tmr_tt, data_tt, scimo, submo, pacmo);

         if (delta_tt == INVALID_TT) {
            numst = 0;
         }
         else {
            // read number of pacmo 1/5 data structures in this (TMR) packet
            numst = tmpkt.data[21];
         }

         // add time tag delta to packet base time
         TL_AddCt(&current, delta_tt/4096.0);

         if (!binary) {
            // print packet time
            fprintf (ofp, "%04hd.%03hd.%02hd.%02hd.%02hd.%03d.%03d\n",
                     tp->year, tp->doy,
                     tp->hr, tp->min, tp->sec, tp->msec, tp->usec);

            // print mode information
            fprintf (ofp, "%02hd.%02hd.%02hd\n", scimo, submo, pacmo);

            // print second header
            for (int j = 4; j < 20; ++j)
               fprintf (ofp, "%04hx ", tmpkt.data[j]);

            fprintf (ofp, "\n");


            // print number of data structures
            fprintf (ofp, "%5hd%8hd\n", numst, delta_tt);
         }

         // select ToF conversion algorithm
         if ((tmpkt.hdr.sc_id < 4 && (tp->year < 2001
                                      || (tp->year == 2001
                                           && tp->doy < 57)))
              || (tmpkt.hdr.sc_id == 4
                   && (tp->year < 2001
                      || (tp->year == 2001 && tp->doy < 95)))) {
            oldtof = 1;
         }
         else {
            oldtof = 0;
         }

         shdr_m = (tmpkt.data[5]>>4) & 0x3; // m from second header
         shdr_codetype = (tmpkt.data[8]>>7) & 0x1; // code type from second header

         // convert and write pacmo 1/5 data
         for (int j = 0; j < numst; ++j) {
            ConvertWWData (*(sci_t *) tmpkt.data, j, stlen, &d1, &d2);
            AddCorrection(d1, d2, j, submo, pacmo, fr_per_st, &dt1, &dt2);
            d1.ct = current.ct + dt1/4096.0;
            d2.ct = current.ct + dt2/4096.0;
            if (binary) {
               fwrite(&d1, sizeof(d1), 1, ofp);
               fwrite(&d2, sizeof(d2), 1, ofp);
            }
            else {
               PrintWWData (ofp, d1);
               PrintWWData (ofp, d2);
            }
         }

      }                         // endif (WW and correct submo/pacmo)

      ++tmpkt.pktno;
   }                            // endwhile packet-loop

   CL2TM_CloseFile ();

   if (binary) {
      printf("%s\n", bin_name);
      fclose(ofp);
   }

   return 0;
}


/******************** FUNCTION DEFINITIONS **********************************/

/*=========================================================================*/
  void AddCorrection(ww_conv2_t d1, ww_conv2_t d2, short i, short submo,
                   short pacmo, short fr_per_st, short *dt1, short *dt2)
/*=========================================================================*/
/* calculate BCI and GDU-frame correction to time tags
 ***************************************************************************/
{
   const char *fname = "wwconv->AddCorrection()";

   *dt1 = i * fr_per_st;
   *dt2 = *dt1;

   if (d1.q==0)   *dt1 -= fr_per_st+16;
   else           *dt1 -= 16*(d1.bci_idx+1);

   if (d2.q==0)   *dt2 -= fr_per_st+16;
   else           *dt2 -= 16*(d2.bci_idx+1);

   if (submo <=3 && pacmo==1) { // NM PM1
      fprintf(stderr, "%s: NM PM1 not supported\n", fname);
      exit(1);
      // are the formulas ok, or do we need to first have to set
      // bci_idx and frm_idx correctly (NM PM1 delivers lo and hi indices!!)?
      *dt1 += 3 + 8*d1.frm_idx;
      *dt2 += 3 + 8*d2.frm_idx;
   }
   else {
      *dt1 += 4*d1.frm_idx;
      *dt2 += 4*d2.frm_idx;
   }

}


/*=========================================================================*/
  void
  ConvertWWData (sci_t sbuf, short i, short stlen,
                 ww_conv2_t *p1, ww_conv2_t *p2)
/*=========================================================================*/
/* This function is used to extract the data from WW PM1/PM5 packets       */
/***************************************************************************/
{
   short tof1_int, dtof_int;
   float vx1, vy1, vx2, vy2;
   short stbase = i * stlen;
   static int shft[4] = { 4, 0, 12, 8 };

   if (stlen==5) {
      p1->d14 = 0;
      p1->d15 = 0;
      p2->d14 = 0;
      p2->d15 = 0;
      p1->mch = 0;
      p2->mch = 0;
   }

   p1->stcnt = i;
   p2->stcnt = i;

   p1->ovfl = 0;
   p2->ovfl = 0;

   // get raw time of flight (gun2/detector1)
   tof1_int = sbuf.bm1.pm1.data[stbase + 0] & 0x0fff;

   // get delta time of flight;
   dtof_int = (sbuf.bm1.pm1.data[stbase + 0] >> 4) & 0x0f00 |
      (sbuf.bm1.pm1.data[stbase + 1] >> 8) & 0x00f0 |
      (sbuf.bm1.pm1.data[stbase + 2] >> 12) & 0x000f;


   if (oldtof == 0) {

      dtof_int &= 0x3ff;        // 10 bits only

      if (dtof_int == 0x1ff || dtof_int == 0x200)
         p2->ovfl = 1;

      if (tof1_int == 0xfff) {
         p1->ovfl = 1;
         p2->ovfl = 1;
      }


      p1->m = mconv[(sbuf.bm1.pm1.data[stbase + 0] >> 14) & 0x3];

   } else {
      p1->m = mconv[shdr_m];
   }


   p2->m = p1->m;

   vx1 = 8.0 * (sbuf.bm1.pm1.data[stbase + 1] & 0x0fff) - 12800;
   vy1 = 8.0 * (sbuf.bm1.pm1.data[stbase + 2] & 0x0fff) - 12800;
   vx2 = 8.0 * (sbuf.bm1.pm1.data[stbase + 3] & 0x0fff) - 12800;
   vy2 = 8.0 * (sbuf.bm1.pm1.data[stbase + 4] & 0x0fff) - 12800;

   p1->e = (sbuf.bm1.pm1.data[stbase + 3] >> 15) & 0x0001;
   p1->n = nconv[(sbuf.bm1.pm1.data[stbase + 3] >> 12) & 0x0007];
   p2->e = p1->e;
   p2->n = p1->n;

   p1->ctype = shdr_codetype;
   p2->ctype = p1->ctype;

   ConvertToF (tof1_int, dtof_int, &p1->tof, &p2->tof, p1->n);

   ConvertVxVy (vx1, vy1, &p2->ph, &p2->th);
   ConvertVxVy (vx2, vy2, &p1->ph, &p1->th);

   p1->q = (sbuf.bm1.pm1.data[stbase + 4] >> 14) & 0x0003;
   p2->q = (sbuf.bm1.pm1.data[stbase + 4] >> 12) & 0x0003;


   if (stlen==5) {

     if (one_gd_pair == 12 || one_gd_pair == 21) {
         fprintf(stderr, "NM1 PM1 not yet checked with one gd-pair tm mode\n");
         exit(1);
      }

      /* low index bits: BCI within 16BCI structure (low byte first in each word) */
      p1->frm_idx = (sbuf.nmx.pm1.ixlo[i / 2] >> ((i % 2) * 8 + 4)) & 0x000f;        /* detector 1 */
      p2->frm_idx = (sbuf.nmx.pm1.ixlo[i / 2] >> ((i % 2) * 8)) & 0x000f;    /* detector 2 */

      /* high index bits: sub-BCI resolution */
      if (i < 80) {
         p1->bci_idx = (sbuf.nmx.pm1.ixhi[i / 4] >> (shft[i % 4] + 2)) & 0x0003;     // detector 1
         p2->bci_idx = (sbuf.nmx.pm1.ixhi[i / 4] >> (shft[i % 4])) & 0x0003; // detector 2
      }
      else {                       /* last nibble needs special treatment */
         p1->bci_idx = (sbuf.nmx.pm1.ixhi[i / 4] >> 2) & 0x0003;     /* detector 1 */
         p2->bci_idx = (sbuf.nmx.pm1.ixhi[i / 4]) & 0x0003;  /* detector 2 */
      }
   }
   else {
      p1->d14 = (short) ((sbuf.bm1.pm1.data[stbase + 5] & 0x3fff) << 2) >> 2;
      p1->d15 = sbuf.bm1.pm1.data[stbase + 6] & 0x3fff;
      p2->d14 = (short) ((sbuf.bm1.pm1.data[stbase + 7] & 0x3fff) << 2) >> 2;
      p2->d15 = sbuf.bm1.pm1.data[stbase + 8] & 0x3fff;

      p1->mch = ((sbuf.bm1.pm1.data[stbase + 5] >> 12) & 0xc) |
         ((sbuf.bm1.pm1.data[stbase + 6] >> 14) & 0x3);

      p2->mch = ((sbuf.bm1.pm1.data[stbase + 7] >> 12) & 0xc) |
         ((sbuf.bm1.pm1.data[stbase + 8] >> 14) & 0x3);


      /* Index bits */
      /* ========== */

      /* BCI index: BM1 PM1: bits 10-8 and 2-0 for detector 1 and 2 */
      /*            NM1 PM5: bits 13-8 and 5-0 for detector 1 and 2 */
      p1->bci_idx = (sbuf.bm1.pm1.data[stbase + 9] >> 8) & 0x3f;
      p2->bci_idx = (sbuf.bm1.pm1.data[stbase + 9]) & 0x3f;


      /* Frame index */
      /* bits 15,14 (detector1) and 7,6 (detector 2) of word 9 */
      p1->frm_idx = (sbuf.bm1.pm1.data[stbase + 9] >> 14) & 0x3;
      p2->frm_idx = (sbuf.bm1.pm1.data[stbase + 9] >> 6) & 0x3;

   }


   if (one_gd_pair == 0) {

      p1->gd = 21; /* gun2, detector 1 */
      p2->gd = 12; /* gun1, detector 2 */

   } else if (one_gd_pair == 21) {

      p1->gd = 21;
      p2->gd = 21;

   } else if (one_gd_pair == 12) {

      p1->gd = 12;
      p2->gd = 12;

      {
         float tmp_float;
         short tmp_short;

         tmp_float = p1->ph; p1->ph = p2->ph; p2->ph = tmp_float;
         tmp_float = p1->th; p1->th = p2->th; p2->th = tmp_float;

         tmp_short = p1->q;   p1->q   = p2->q;   p2->q   = tmp_short;
         tmp_short = p1->d14; p1->d14 = p2->d14; p2->d14 = tmp_short;
         tmp_short = p1->d15; p1->d15 = p2->d15; p2->d15 = tmp_short;
         tmp_short = p1->mch; p1->mch = p2->mch; p2->mch = tmp_short;

         tmp_short = p1->bci_idx; p1->bci_idx = p2->bci_idx; p2->bci_idx = tmp_short;
         tmp_short = p1->frm_idx; p1->frm_idx = p2->frm_idx; p2->frm_idx = tmp_short;
      }

   }

}

/*=========================================================================*/
void
PrintWWData (FILE * outfp, ww_conv2_t d)
/*=========================================================================*/
{
   fprintf (outfp, "%5hd  %2hd  %14.3f  "
                   "%7.1f %7.1f %10.3f   "
                   "%hd %2hd %2hd %hd   "
                   "%2hd %5hd %5hd   "
                   "%hd %hd  %2hd %2hd\n",
            d.stcnt, d.gd, d.ct,
            d.th, d.ph, d.tof,
            d.e, d.n, d.m, d.q,
            d.mch, d.d14, d.d15,
            d.ctype, d.ovfl, d.bci_idx, d.frm_idx);
}


/*==========================================================================*/
void
ConvertToF (short tof1_int, short dtof_int, float *tof1, float *tof2, int n)
/*==========================================================================*/
//
{
   float dtof;
   double DT2US = 1 / 16.777216;        // convert from 59.6 ns to micro-s

   if (oldtof == 1) {

      // convert Time-of-Flight
      *tof1 = (tof1_int & 0x07ff) * 4 * DT2US;

      // check for range bit
      if (tof1_int & 0x0800)
         *tof1 *= 16;

      // convert delta Time-of-Flight
      dtof = ((short) ((dtof_int & 0x07ff) << 5) / 32) * DT2US;

      // check for range bit
      if (dtof_int & 0x0800)
         dtof *= 16;

   }
   else {                       // new conversion algorithm

      *tof1 = DT2US * tof1_int * n;
//      dtof = DT2US * ((short) (dtof_int << 6) / 64) * n;
// need to split the previous line into several ones below, because
// gcc 2.95.3 on ibmcl1 does otherwise not recognize the sign extension
// to 16 bits !!! Oh well.....
      dtof_int <<= 6;
      dtof_int /= 64;
      dtof = DT2US * dtof_int * n;

   }

   *tof2 = *tof1 - dtof;
}

/*==========================================================================*/
void
ConvertVxVy (float vx, float vy, float *php, float *thp)
/*==========================================================================*/
{
   float sq;

   *php = 180 / M_PI * atan2 (vy, vx);
   *php = (*php < 0) ? ((*php) * 10 - 0.5) / 10 : ((*php) * 10 + 0.5) / 10;
   sq = sqrt (vx * vx + vy * vy) / (8 * 1667.0);
   if (sq > (1.0 + 1.0e-4))
      *thp = -1.0;
   else if (sq > 1.0)
      *thp = 124.1;
   else {
      *thp = 79.0 * asin (sq);
      *thp = ((*thp) * 10 + 0.5) / 10;
   }

}

/*===========================================================================*/
static void
Usage (const char *exename)
/*===========================================================================*/
{
   fprintf (stdout,
            "Usage: %s [-h] [-d] [-b] [-s starttime]\n"
            "          [-e endtime | -t delta]\n"
            "           data-file-name\n"
            "\n"
            "  -h  print this help\n"
            "  -d  print debugging info\n"
            "  -b  binary output\n"
            "  -s  specify start time (yyyy-mm-ssThh:mm:ss)\n"
            "  -e  specify end time   (yyyy-mm-ssThh:mm:ss)\n"
            "  -t  specify time interval (delta=nX)\n"
            "      n = integer number\n"
            "      X = h (hours), m (minutes) or s (seconds)\n"
            "  -m {nm|bm}   select telemetry mode\n"
            "\n", exename);

}


