; ***************************************************************************
; CLUTIME.PRO
; ***************************************************************************
; Procedures and functions to handle structured time and time in seconds
; since 1.Jan.1970
; ===========================================================================
; $Id: clutime.pro,v 1.7 2005/03/19 16:21:54 hav Exp $
; $Log: clutime.pro,v $
; Revision 1.7  2005/03/19 16:21:54  hav
; more dev
;
; Revision 1.6  2004/11/03 13:06:55  hav
; shortcut function names in clutime.pro; fgm_cal path for pcedism in paths.inc;
; uncalb (all 3 axes) in uncalspinaxis.pro
;
; Revision 1.5  2004/06/25 15:30:07  hav
; more dev
;
; Revision 1.3  2003/11/14 18:05:05  hav
; more dev
;
; Revision 1.2  2003/07/04 16:45:20  hav
; more development
;
; Revision 1.1.1.1  2003/03/28 18:54:35  hav
; initial import
;
; Revision 1.1.1.1  2002/03/25 08:48:19  hav
; Initial Import
;
; Revision 1.1  2001/05/15  16:17:41  hav
; Initial revision
;
; ***************************************************************************

; ==========================================================================
  FUNCTION CluTimeOk, str_time, info=info
; ==========================================================================
; checks for correct format of ISO time string
; yyyy-mm-ddThh:mm:ss.mss.uss.nss
;
; returns 1 if ok 0 otherwise
; if 0, then more detailed information is returned in the named variable
; 'info'
; **************************************************************************

  pos   = strsplit(str_time, '-tT:.')
  elems = strsplit(str_time, '-tT:.', /extract)


  nominal_pos = [ 0, 5, 8, 11, 14, 17, 20, 24, 28 ]
  x = where((nominal_pos - pos) ne 0, dpos_sum)
  if dpos_sum gt 0 then begin
     info = 'bad string format'
     return, 0
  endif

  NN = n_elements(elems)
  if NN lt 3 then begin
     info = 'Not enough elements in <'+str_time+'>'
     return, 0
  endif
  if NN gt 9 then begin
     info = 'Too many elements in <'+str_time+'>'
     return, 0
  endif

  low  = [ 2001,  1,  1,  0,  0,  0,   0,   0,   0 ]
  high = [ 2005, 12, 31, 23, 59, 59, 999, 999, 999 ]
  name = [ 'year', 'month', 'day', 'hour', 'minute', 'second', $
           'millisecond', 'microsecond', 'nanosecond' ]

  for i=0,NN-1 do begin
     val = fix(elems[i])
     if val lt low[i] or val gt high[i] then begin
        info = 'Value for field "' + name[i] + '" is out of bounds'
        return, 0
     endif
  endfor


  info = 'Ok'
  return, 1

END

; ==========================================================================
  FUNCTION CluTimeAbsYdm, at
; ==========================================================================

   mo_days = [31, 28, 31, 30,  31, 30, 31, 31, 30, 31, 30, 31]

   if at.year mod 4 eq 0 then mo_days(1) = 29
   day = at.doy & mo = 0
   while day gt mo_days(mo) do begin
      day = day - mo_days(mo)
      mo = mo + 1
   endwhile
   mo=mo+1

   ydm = { year:at.year, month:mo, day:day }

   return, ydm

END

; ===========================================================================
  FUNCTION CluTimeGetDoy,  ymd_str
; ===========================================================================
; ymd_str has to be in format 'yyyy-mm-dd'
; This routine is vectorized !
; ***************************************************************************

   NN = n_elements(ymd_str)
   days_per_month = [31, 28, 31, 30,  31, 30, 31, 31, 30, 31, 30, 31]

   year = fix(strmid(ymd_str, 0, 4))
   month = fix(strmid(ymd_str,  5, 2))
   day   = fix(strmid(ymd_str, 8, 2))

   doy = intarr(NN)

   for cur_month = 1, 12 do begin
      x = where(month gt cur_month)
      if x(0) ne -1 then doy(x) =  doy(x) + days_per_month(cur_month-1)
   endfor

   x = where( (year mod 4 eq 0) and (month gt 2) )
   if x(0) ne -1 then doy(x) = doy(x) + 1

   doy = doy + day

   return, doy

 END



; ===========================================================================
  FUNCTION CluTimeAbs2Ct, at
; ===========================================================================
; This routine is vectorized! ('at' may be an array)
; ***************************************************************************

  NN = n_elements(at)

  days = at.doy - 1 ; full days in current year

  year=at.year
  x = where( year gt 2000 )
  while x(0) ne -1 do begin
     year(x)=year(x)-1
     days(x) = days(x) + 365

     xleap = where( year(x) mod 4 eq 0 )
     if xleap(0) ne -1 then days(x(xleap)) = days(x(xleap)) + 1

     x = where( year gt 2000 )
  endwhile

;  ct = 978307200.0d + $ ; seconds from 1-Jan-1970 to 1-Jan-2001
  ct = 946684800.0d + $ ; seconds from 1-Jan-1970 to 1-Jan-2000
       (double(days)*24 + at.hr) * 3600 + at.min*60 + at.sec + $
       at.msec*1.0d-3 + at.usec*1.0d-6 + at.nsec*1.0d-9

  return, ct

END

; ==========================================================================
  FUNCTION CluTimeCt2Abs, ct
; ==========================================================================

@stime.inc

;  ct_2001 = ct - 978307200.0d ; seconds since 1-Jan-2001
  ct_2000 = ct - 946684800.0d ; seconds since 1-Jan-2000

  if ct_2000 lt 0 then begin
     message, 'too lazy to handle times prior to 1-Jan-2000 !', /cont
     message, '(I was called with a value of '+string(format='(F20.3)',ct), /cont
     help, /traceback
     retall
;     return, at
  endif

  at.year = 2000
  at.doy  = long(ct_2000 / 86400L) ; zero-based! corrected further down
  at.hr   = (ct_2000 - 86400L*at.doy) / 3600L
  at.min  = (ct_2000 - 86400L*at.doy - 3600L*at.hr) / 60L
  at.sec  = (ct_2000 - 86400L*at.doy - 3600L*at.hr - 60L*at.min)
  subsec = ct_2000 - long(ct_2000)
  at.msec = long(1000L*subsec)
  at.usec = long(1000L* (1000L*subsec - at.msec))
  at.nsec = long(1000L* (1000000L*subsec - 1000L*at.msec - at.usec))

  at.doy = at.doy + 1

  endloop = 0
  while not endloop do begin
     if at.year mod 4 eq 0 then daymax = 366 else daymax = 365
     if at.doy gt daymax then begin
        at.year = at.year + 1
 at.doy = at.doy - daymax
     endif else $
        endloop = 1
  endwhile

  return, at

END


; ===========================================================================
  FUNCTION CluTimeVal2Str, val, doy=doy
; ===========================================================================
; Takes a date/time value and converts it into a readable string
; in format 'yyyy-mm-ddThh:mm:ss.msc.usc.nsc'
; (or 'yyyy.doy.hh.mm.ss.msc.usc.nsc' if keyword doy is present)
; The date/time value may be specified as either ct or at
; ***************************************************************************

  type = size(val, /type)
  if type eq 5 then $ ; double (-->ct)
     st = CluTimeCt2Abs(val) $
  else $
     st = val

  tagnames = tag_names(st)

  absolute = 0
  for i=0, n_elements(tagnames)-1 do begin
     if tagnames(i) eq 'YEAR' then absolute = 1
  endfor

  if absolute then begin
     if keyword_set(doy) then begin
        fmt = '(i4.4,".",i3.3,".",i2.2,".",i2.2' + $
                   ',".",i2.2,".",i3.3,".",i3.3,".",i3.3)'
           st_str = string(format = fmt, st.year, st.doy, st.hr, $
                           st.min, st.sec, st.msec, st.usec, st.nsec)
     endif else begin
        ymd = CluTimeAbsYdm(st)
        fmt = '(i4.4,"-",i2.2,"-",i2.2,"T",i2.2,":",i2.2' + $
                   ',":",i2.2,".",i3.3,".",i3.3,".",i3.3)'
           st_str = string(format = fmt, ymd.year, ymd.month, ymd.day, $
                           st.hr, st.min, st.sec, st.msec, st.usec, st.nsec)
     endelse
  endif else begin
     fmt = '("         ",i2.2,".",i2.2,".",i2.2' + $
                   ',".",i3.3,".",i3.3,".",i3.3)'
        st_str = string(format = fmt, st.hr, st.min, st.sec, $
                                      st.msec, st.usec, st.nsec)
  endelse

  return, st_str

END


; ===========================================================================
  FUNCTION CluTimeStr2Val,  time_str, abst = abst
; ===========================================================================
; Description:
;    Extracts elements from a time string given in one of two formats:
;
;       'yyyy-mm-ddThh:mm:ss.msc.usc.nsc' or
;       'yyyy.doy.hh.mm.ss.msc.usc.nsc'
;
;    and returns the corresponding time as ct (time in seconds since
;    1-Jan-1970) or as 'at' structure (if keyword 'abst' set)
;
;    The separation characters ('-','T',':','.') in the time strings
;    are arbitrary with the exception of the first separation
;    character (after yyyy). If it is a dot ('.') it will be assumed
;    that the second format is being used (doy instead of month/day)
;    The time string must at least contain yyyy-mm-dd (or yyyy.doy).
;    If the subsequent elements are not available they will be set to
;    zero.
;
;    This routine is vectorized (time_str may be an array)
; ***************************************************************************

@stime.inc

  NN = n_elements(time_str)

  length = strlen(time_str)
  x =  where(length lt 5)
  if x(0) ne -1 then $
     message, 'passed string is too short: ' + time_str(x(0))

  sep_char = strmid(time_str(0), 4, 1)
  x = where(strmid(time_str, 4, 1) ne sep_char)
  if x(0) ne -1 then $
     message, 'not all strings are given in identical format.'

  if sep_char eq '.' then begin
      doy_format = 1
      offs = 2
  endif else begin
      doy_format = 0
      offs = 0
  endelse

  x = where(length lt 10 - offs)
  if x(0) ne -1 then $
     message, 'passed string too short : ' + time_str(x(0))

  at_arr = replicate(at, NN)
  at_arr.year = fix( strmid(time_str, 0, 4) )

  if doy_format eq 0 then at_arr.doy = CluTimeGetDoy(time_str) $
  else                    at_arr.doy = fix( strmid(time_str, 5, 3) )

;  print, ' doy_format = ', doy_format

  at_arr.hr = 0
  x = where(length ge 13-offs)
;  print, time_str(x(0))
;  print, strmid(time_str(x(0)), 11-offs, 2)
;  stop
  if x(0) ne -1 then at_arr(x).hr =  fix( strmid(time_str(x), 11-offs, 2) )

  at_arr.min = 0
  x = where(length ge 16-offs)
  if x(0) ne -1 then at_arr(x).min = fix( strmid(time_str(x), 14-offs, 2) )

  at_arr.sec = 0
  x = where(length ge 19-offs)
  if x(0) ne -1 then at_arr(x).sec = fix( strmid(time_str(x), 17-offs, 2) )

  at_arr.msec = 0
  x = where(length ge 23-offs)
;  print, time_str(x(0))
;  print, strmid(time_str(x(0)), 20-offs, 3)
;  stop
  if x(0) ne -1 then at_arr(x).msec = fix( strmid(time_str(x), 20-offs, 3) )

  at_arr.usec = 0
  x = where(length ge 27-offs)
  if x(0) ne -1 then at_arr(x).usec = fix( strmid(time_str(x), 24-offs, 3) )

  at_arr.nsec = 0
  x = where(length ge 31-offs)
  if x(0) ne -1 then at_arr(x).nsec = fix( strmid(time_str(x), 28-offs, 3) )

  ; Return the time
  ; ---------------
  if keyword_set(abst) then return, at_arr $
  else return, CluTimeAbs2Ct(at_arr)

END


; ==========================================================================
  FUNCTION GetCtr, ymd_hms, dsec
; ==========================================================================
; Parameters
;   ymd_hms  string   start time in format 'yyyy-mm-ddThh:mm:ss.sss.mmm.nnn'
;                     At least 'yyyy-mm-dd' must be present
;   dsec     number   the interval length in seconds
;
; Return
;   a dblarr(2) containing the interval start/end times in t70 format
;
; **************************************************************************

  return, CluTimeStr2Val(ymd_hms) + [0.0d, dsec]

END



; ===========================================================================
  PRO PrintTimeInterval, ctr, fp=fp, leading=leading, $
                         trailing = trailing, length=length
; ===========================================================================

  if not keyword_set(fp) then fp = -1 ; stdout
  if not keyword_set(leading) then leading=''
  if not keyword_set(trailing) then trailing = ''
  if not keyword_set(length) then length = 23 ; millisecond resolution

  t1 = strmid(CluTimeVal2Str(ctr[0]),0,length)
  NN = n_elements(ctr)
  if NN gt 2 then begin
     message, 'parameter ctr must have one or two elements', /cont
     retall
  endif else if NN eq 2 then begin
     t2 = '  -  ' + strmid(CluTimeVal2Str(ctr[1]),0,length)
  endif else t2 = ''

  printf, fp, leading + t1 + t2 + trailing

END

; ++++++++++++++ SOME HANDY SHORTCUTS BELOW +++++++++++++++++++++++

FUNCTION Ct2Str, ct
   return, ClutimeVal2Str(ct)
END

FUNCTION Str2Ct, str
   return, CluTimeStr2Val(str)
END

FUNCTION Ct2Ymd, ct
   return, strmid(CluTimeVal2Str(ct),0,10)
END


























